#  Copyright (C) 1999-2012
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc SAODef {} {
    global sao
    global isao

    set isao(top) .sao
    set isao(mb) .saomb

    set sao(sky) fk5
    set sao(rformat) arcmin
    set sao(width) 15
    set sao(height) 15
    set sao(mode) new
    set sao(save) 0
    set sao(valid) 0
    set sao(survey) dss
}

proc SAODialog {} {
    global sao
    global isao
    global wcs

    if [winfo exists $isao(top)] {
	raise $isao(top)
	return
    }

    set varname dsao
    upvar #0 $varname var
    global $varname

    set var(top) $isao(top)
    set var(mb) $isao(mb)
    set var(sky) $sao(sky)
    set var(skyformat) $wcs(skyformat)
    set var(rformat) $sao(rformat)
    set var(width) $sao(width)
    set var(height) $sao(height)
    set var(mode) $sao(mode)
    set var(save) $sao(save)
    set var(valid) $sao(valid)
    set var(survey) $sao(survey)

    set w $var(top)
    IMGSVRInit $varname "SAO-DSS [msgcat::mc {Server}]" SAOExec SAOAck

    IMGSVRUpdate $varname 1
}

proc SAOExec {varname} {
    upvar #0 $varname var
    global $varname

    if {$var(save)} {
	set compress no
	set var(fn) [SaveFileDialog savefitsfbox]
	if {$var(fn) == {}} {
	    return
	}

    } else {
	set compress gzip
	set var(fn) [tmpnam ds9sao ".fits.gz"]
    }

    # skyformat
    switch -- $var(skyformat) {
	degrees {
	    set xx [uformat d h: $var(x)]
	    set yy [uformat d d: $var(y)]
	}
	sexagesimal {
	    set xx $var(x)
	    set yy $var(y)
	}
    }

    # size - convert to arcmin
    switch -- $var(rformat) {
	degrees {
	    set ww [expr $var(width)*60.]
	    set hh [expr $var(height)*60.]
	}
	arcmin {
	    set ww $var(width)
	    set hh $var(height)
	}
	arcsec {
	    set ww [expr $var(width)/60.]
	    set hh [expr $var(height)/60.]
	}
    }
    if {$ww>60} {
	set ww 60
    }
    if {$hh>60} {
	set hh 60
    }

    # query
    set var(query) [http::formatQuery r $xx d $yy e J2000 w $ww h $ww c $compress]
    set var(url) "http://www.cfa.harvard.edu/archive/dss"
    IMGSVRLoad $varname
}

proc SAOAck {varname} {
    upvar #0 $varname var
    global $varname

set msg {Acknowledgments for the DSS-SAO

The Digitized Sky Surveys were produced at the Space Telescope Science
Institute under U.S.  Government grant NAG W-2166. The images of these
surveys are based on photographic data obtained using the Oschin
Schmidt Telescope on Palomar Mountain and the UK Schmidt Telescope.
The plates were processed into the present compressed digital form
with the permission of these institutions.

The Oschin Schmidt Telescope is operated by the California Institute
of Technology and Palomar Observatory.

The UK Schmidt Telescope was operated by the Royal Observatory
Edinburgh, with funding from the UK Science and Engineering Research
Council (later the UK Particle Physics and Astronomy Research
Council), until 1988 June, and thereafter by the Anglo-Australian
Observatory. The blue plates of the southern Sky Atlas and its
Equatorial Extension (together known as the SERC-J), as well as the
Equatorial Red (ER), and the Second Epoch [red] Survey (SES) were all
taken with the UK Schmidt.
}

SimpleTextDialog ${varname}ack [msgcat::mc {Acknowledgment}] 80 40 insert top $msg
}

# Process Cmds

proc ProcessSAOCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    SAODialog
    IMGSVRProcessCmd $varname $iname dsao
}

proc ProcessSendSAOCmd {proc id param} {
    SAODialog
    IMGSVRProcessSendCmd $proc $id $param dsao
}

