// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __context_h__
#define __context_h__

#include "tcl.h"
#include "base.h"
#include "coord.h"
#include "frscale.h"
#include "list.h"
#include "fitsmask.h"

class Base;
class FVContour;

class Context {
 public:
  FitsImage* fits;
  FitsImage* cfits;

  List <FitsMask> mask;

  FVContour* contour;
  List<Contour> auxcontours;

  FrScale frScale;

 private:
  Base* parent_;

  int mosaicCount_;
  int naxis_[FTY_MAXAXES]; // the first two are ignored
  int slice_[FTY_MAXAXES]; // the first two are ignored

  Base::MosaicType mosaicType;
  Coord::CoordSystem mosaicSystem;

  int nhdu();
  void updateClip(FrScale*, int);

 public:
  Context();
  ~Context();
  Context(const Context&);
  Context& operator=(const Context&);

  void analysis();

  void bltHist(char*, char*, int);

  int calcSlice();
  void clearContour();
  void contourAuxHead() {auxcontours.head();}
  void contourAuxNext() {auxcontours.next();}

  int fitsCount();

  Vector getClip(FrScale::ClipMode, float);
  Vector getClip();

  int hasContour() {return contour ? 1 : 0;}
  int hasContourAux() {return auxcontours.current() ? 1 : 0;}
  double* histequ() {return frScale.histequ(fits);}

  int isMosaic() {return mosaicCount_>1 ? 1 : 0;}
  int isCube() {return nhdu()>1 ? 1 : 0;}

  int load(Base*, Base::MemType, const char*, FitsImage*, Base::LayerType);
  int loadExtCube(Base*, Base::MemType, const char*, FitsImage*);
  int loadSlice(Base*, Base::MemType, const char*, FitsImage*);
  int loadMosaic(Base*, Base::MemType, const char*, FitsImage*, 
		 Base::LayerType, Base::MosaicType, Coord::CoordSystem);
  int loadMosaicImage(Base*, Base::MemType, const char*, FitsImage*, 
		      Base::LayerType, Base::MosaicType, Coord::CoordSystem);
  int loadMosaicWFPC2(Base*, Base::MemType, const char*, FitsImage*);
  void loadInit(Base::MosaicType, Coord::CoordSystem);
  void loadFinish();
  void loadFinishMask();

  int naxis(int ii) {return naxis_[ii];}
  int naxes();

  void parent(Base* pp) {parent_ = pp;}

  void setContour(FVContour*);
  int slice(int ii) {return slice_[ii];}

  void unload();
  void updateBin(Base*);
  void updateBinFileNames();
  void updateClip();
  void updateContours();
  void updateContours(const Matrix&);
  void updateSlice(int, int);
};

#endif
