// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>

#include <iostream>
#include <sstream>
using namespace std;

#include <tcl.h>
#include <tk.h>

extern "C" {
  int SAOAppInit(Tcl_Interp *interp);
  int SAOLocalMainHook(int* argc, char*** argv);

  void TclSetStartupScriptFileName(const char*);

  int Zvfs_Init(Tcl_Interp*);
  int Zvfs_Mount(Tcl_Interp*, char*, char *);

  int Blt_core_Init(Tcl_Interp*);
  int Blt_x_Init(Tcl_Interp*);
  int Tktable_Init(Tcl_Interp*);
  int Checkdns_Init(Tcl_Interp*);
  int Saotk_Init(Tcl_Interp*);
  int Tkhtml_Init(Tcl_Interp*);
  int Tkmpeg_Init(Tcl_Interp*);

  int Tkimg_Init(Tcl_Interp*);
  int Zlibtcl_Init(Tcl_Interp*);
  int Jpegtcl_Init(Tcl_Interp*);
  int Tkimgjpeg_Init(Tcl_Interp*);
  int Tifftcl_Init(Tcl_Interp*);
  int Tkimgtiff_Init(Tcl_Interp*);
  int Pngtcl_Init(Tcl_Interp*);
  int Tkimgpng_Init(Tcl_Interp*);
  int Tkimggif_Init(Tcl_Interp*);
  int Tkimgppm_Init(Tcl_Interp*);
  int Tkimgwindow_Init(Tcl_Interp*);

  int Tclxpa_Init(Tcl_Interp*);
  int Iis_Init(Tcl_Interp*);

  int Tclxml_Init(Tcl_Interp*);
  int Tclxml_libxml2_Init(Tcl_Interp*);

#ifndef _WIN32
  int Signal_ext_Init(Tcl_Interp*);
#endif

#ifdef _MACOSX
  int Tkmacosx_Init(Tcl_Interp*);
#endif
#ifdef _WIN32
  int Tkwin32_Init(Tcl_Interp*);

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#undef WIN32_LEAN_AND_MEAN

#endif
}

static char* appname = NULL;
char* dupstr(const char* str);
Tcl_Interp *global_interp;

void internalError(const char* msg)
{
  Tcl_SetVar2(global_interp, "ds9", "msg", msg, TCL_GLOBAL_ONLY);
  Tcl_SetVar2(global_interp, "ds9", "msg,level", "error", TCL_GLOBAL_ONLY);
}

// currently use relative path
// using full path with spaces causes problems 
// with htmwidget and tcl/tk

int SAOLocalMainHook(int* argcPtr, char*** argvPtr)
{
  // sync C++ io calls with C io calls
  ios::sync_with_stdio();

  // save real application name
  int argc = *argcPtr;
  char** argv = *argvPtr;
  appname = dupstr(argv[0]);

  // so that tcl and tk know where to find their libs
  // we do it here before InitLibraryPath is called
  putenv((char*)"TCL_LIBRARY=./zvfsmntpt/tcl8.5");
  putenv((char*)"TK_LIBRARY=./zvfsmntpt/tk8.5");

  // invoke startup script
  TclSetStartupScriptFileName("./zvfsmntpt/src/ds9.tcl");
}

int SAOAppInit(Tcl_Interp *interp)
{
  // reset argv0
  if (appname) {
    Tcl_SetVar(interp, "argv0", appname, TCL_GLOBAL_ONLY);
    delete [] appname;
  }

  // save interp for cputs function
  global_interp = interp;

  // We have to initialize the virtual filesystem before calling
  // Tcl_Init().  Otherwise, Tcl_Init() will not be able to find
  // its startup script files.
  if (Zvfs_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
    Tcl_StaticPackage (interp, "zvfs", Zvfs_Init, 
		     (Tcl_PackageInitProc*)NULL);


  // find current working directory, and set as mount point
  {
    Tcl_DString pwd;
    Tcl_DStringInit(&pwd);
    Tcl_GetCwd(interp, &pwd);
#ifdef ZIPFILE
    ostringstream str;
#ifndef _WIN32
    str << (char *)Tcl_GetNameOfExecutable() 
	<< ".zip" 
	<<  ends;
#else
    str << (char *)Tcl_GetNameOfExecutable() 
	<< "/../ds9.zip" 
	<<  ends;
#endif
    if( Zvfs_Mount(interp, (char*)str.str().c_str(), Tcl_DStringValue(&pwd)) != TCL_OK ){
      char str[] = "ERROR: Unable to open the auxiliary ds9 file 'ds9.zip'. If you moved the ds9 program from its original location, please also move the zip file to the same place.";

#ifndef _WIN32
      cerr << str << endl;
#else
      MessageBox(NULL, str, "SAOImage DS9", MB_ICONSTOP | MB_OK | MB_TASKMODAL | MB_SETFOREGROUND);
#endif
      exit(1);
    }
#else
    Zvfs_Mount(interp, (char *)Tcl_GetNameOfExecutable(), 
	       Tcl_DStringValue(&pwd));
#endif
    Tcl_DStringFree(&pwd);
  }

  // Initialize Tcl and Tk
  if (Tcl_Init(interp))
    return TCL_ERROR;

  // Tk
  if (Tk_Init(interp))
    return TCL_ERROR;
  Tcl_StaticPackage(interp,"Tk", Tk_Init, Tk_SafeInit);

  {
    Tcl_DString pwd;
    Tcl_DStringInit(&pwd);
    Tcl_GetCwd(interp, &pwd);
    Tcl_DStringFree(&pwd);
  }

  // Blt
  if (Blt_core_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage(interp, "blt_core", Blt_core_Init, 
		    (Tcl_PackageInitProc*)NULL);
  if (Blt_x_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage(interp, "blt_extra", Blt_x_Init,
		    (Tcl_PackageInitProc*)NULL);

  // Tktable
  if (Tktable_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "Tktable", Tktable_Init, 
		     (Tcl_PackageInitProc*)NULL);

  // Checkdns
  if (Checkdns_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "checkdns", Checkdns_Init, 
		     (Tcl_PackageInitProc*)NULL);

  // Saotk
  if (Saotk_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  //Tcl_StaticPackage (interp, "saotk", Saotk_Init,
  //		     (Tcl_PackageInitProc*)NULL);

  // Tkhtml
  if (Tkhtml_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "tkhtml", Tkhtml_Init,
  		     (Tcl_PackageInitProc*)NULL);

  // Tclxpa
  if (Tclxpa_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "Tclxpa", Tclxpa_Init,
		     (Tcl_PackageInitProc*)NULL);

  // IIS
  if (Iis_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "iis", Iis_Init, 
		     (Tcl_PackageInitProc*)NULL);

  // Tkmpeg
  if (Tkmpeg_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "tkmpeg", Tkmpeg_Init, 
		     (Tcl_PackageInitProc*)NULL);

  // Tclxml
  if (Tclxml_Init(interp) == TCL_ERROR)
    return TCL_ERROR;

  // Tkimg
  if (Tkimg_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "img", Tkimg_Init,
		     (Tcl_PackageInitProc*)NULL);

  // zlibtcl
  if (Zlibtcl_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "zlibtcl", Zlibtcl_Init,
		     (Tcl_PackageInitProc*)NULL);

  // jpegtcl
  if (Jpegtcl_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "jpegtcl", Jpegtcl_Init,
		     (Tcl_PackageInitProc*)NULL);

  // Tkimgjpeg
  if (Tkimgjpeg_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "jpeg", Tkimgjpeg_Init,
		     (Tcl_PackageInitProc*)NULL);

  // Tifftcl
  if (Tifftcl_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "tifftcl", Tifftcl_Init,
		     (Tcl_PackageInitProc*)NULL);

  // Tkimgtiff
  if (Tkimgtiff_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "tiff", Tkimgtiff_Init,
		     (Tcl_PackageInitProc*)NULL);

  // Pngtcl
  if (Pngtcl_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "pngtcl", Pngtcl_Init,
		     (Tcl_PackageInitProc*)NULL);

  // Tkimgpng
  if (Tkimgpng_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "png", Tkimgpng_Init,
		     (Tcl_PackageInitProc*)NULL);

  // Tkimggif
  if (Tkimggif_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "gif", Tkimggif_Init,
		     (Tcl_PackageInitProc*)NULL);

  // Tkimgppm
  if (Tkimgppm_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "ppm", Tkimgppm_Init,
		     (Tcl_PackageInitProc*)NULL);

  // Tkimgwindow
  if (Tkimgwindow_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "window", Tkimgwindow_Init,
		     (Tcl_PackageInitProc*)NULL);

  // Signal_Ext
#ifndef _WIN32
  if (Signal_ext_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "signal", Signal_ext_Init, 
		     (Tcl_PackageInitProc*)NULL);
#endif

#ifdef _MACOSX
  if (Tkmacosx_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "macosx", Tkmacosx_Init,
		     (Tcl_PackageInitProc*)NULL);
#endif

#ifdef _WIN32
  if (Tkwin32_Init(interp) == TCL_ERROR)
    return TCL_ERROR;
  Tcl_StaticPackage (interp, "win32", Tkwin32_Init,
		     (Tcl_PackageInitProc*)NULL);
#endif

  // Variables
  Tcl_SetVar(interp, "auto_path", "./zvfsmntpt/tcl8.5 ./zvfsmntpt/tcl8/8.3 ./zvfsmntpt/tcl8/8.4 ./zvfsmntpt/tcl8/8.5 ./zvfsmntpt/tk8.5 ./zvfsmntpt/blt3.0 ./zvfsmntpt/tcllib1.10 ./zvfsmntpt/src", TCL_GLOBAL_ONLY); 
  Tcl_SetVar(interp, "tcl_libPath", "./zvfsmntpt/tcl8.5", TCL_GLOBAL_ONLY);
  Tcl_SetVar(interp, "blt_library", "./zvfsmntpt/blt3.0", TCL_GLOBAL_ONLY);
  Tcl_SetVar(interp, "blt_libPath", "./zvfsmntpt/blt3.0", TCL_GLOBAL_ONLY);
  Tcl_SetVar2(interp, "env", "TK_TABLE_LIBRARY", "", TCL_GLOBAL_ONLY);

  //Tcl_SetVar(interp, "tcl_rcFileName", "~/.wishrc", TCL_GLOBAL_ONLY);

  return TCL_OK;
}

