require 'test_helper'

class SfTest < GSL::TestCase

  TEST_TOL = [2.0, 16.0, 256.0, 2048.0, 16384.0, 131072.0, 1048576.0]

  TEST_SQRT_TOL0 = 2.0 * GSL::SQRT_DBL_EPSILON

  TEST_SNGL = 1.0e-06

  TEST_FACTOR = 100.0
  TEST_SIGMA = 1.5

  ERR_INCONS = 'value / expected not consistent within reported error'

  def test_mode
    z = GSL::Complex[1, 0]
    m = GSL::Matrix::Complex.eye(2, z)

    assert_equal z, m[0, 0]
    assert_equal GSL::Complex[0, 0], m[0, 1]
    assert_equal GSL::Complex[0, 0], m[1, 0]
    assert_equal z, m[1, 1]
  end

  def _test_sf(func, args, val, tol)
    r, = GSL::Sf.send(func, *args)

    desc = '%s(%p): %20.16g %22.18g %22.18g %g: ' % [
      func, args, val, r.val, r.err, r.err / (r.val.abs + r.err)
    ]

    if GSL.isnan?(r.val) || GSL.isnan?(val)
      assert GSL.isnan(r.val) == GSL.isnan(val), desc + ERR_INCONS
    elsif GSL.isinf?(r.val) || GSL.isinf?(val)
      assert GSL.isinf(r.val) == GSL.isinf(val), desc + ERR_INCONS
    else
      refute((val - r.val).abs > 2.0 * TEST_SIGMA * r.err, desc + ERR_INCONS)
      refute(r.err < 0.0, desc + 'reported error negative')

      tol = TEST_TOL[tol] * GSL::DBL_EPSILON if tol.integer?

      refute(((val.zero? && r.val.zero?) ? 0.0 :
        (val <= GSL::DBL_MAX && r.val < GSL::DBL_MAX && val + r.val != 0) ?
        ((val - r.val) / (val + r.val)).abs : 1.0) > TEST_FACTOR * tol,
        desc + 'value not within tolerance of expected value')
    end
  end

  {
    :airy => [
      [:airy_Ai_e, [-500.0, GSL::MODE_DEFAULT],              0.0725901201040411396, 4],
      [:airy_Ai_e, [-5.0, GSL::MODE_DEFAULT],                0.3507610090241142,    0],
      [:airy_Ai_e, [-0.3000000000000094, GSL::MODE_DEFAULT], 0.4309030952855831,    0],
      [:airy_Ai_e, [0.6999999999999907, GSL::MODE_DEFAULT],  0.1891624003981519,    0],
      [:airy_Ai_e, [1.649999999999991, GSL::MODE_DEFAULT],   0.05831058618720882,   0],
      [:airy_Ai_e, [1.649999999999991, GSL::MODE_DEFAULT],   0.0583105861872088521,   0],
      [:airy_Ai_e, [2.54999999999999, GSL::MODE_DEFAULT],    0.01446149513295428,   0],
      [:airy_Ai_e, [3.499999999999987, GSL::MODE_DEFAULT],   0.002584098786989702,  1],
      [:airy_Ai_e, [5.39999999999998, GSL::MODE_DEFAULT],    4.272986169411866e-05, 0],
      [:airy_Ai_scaled_e, [-5.0, GSL::MODE_DEFAULT],                  0.3507610090241142, 0],
      [:airy_Ai_scaled_e, [0.6999999999999907, GSL::MODE_DEFAULT], 0.2795125667681217, 0],
      [:airy_Ai_scaled_e, [1.649999999999991, GSL::MODE_DEFAULT],  0.2395493001442741, 0],
      [:airy_Ai_scaled_e, [2.54999999999999, GSL::MODE_DEFAULT],   0.2183658595899388, 0],
      [:airy_Ai_scaled_e, [3.499999999999987, GSL::MODE_DEFAULT],  0.2032920808163519, 0],
      [:airy_Ai_scaled_e, [5.39999999999998, GSL::MODE_DEFAULT],   0.1836050093282229, 0],
      [:airy_Bi_e, [-500.0, GSL::MODE_DEFAULT],             -0.094688570132991028, 4],
      [:airy_Bi_e, [-5.0, GSL::MODE_DEFAULT],               -0.1383691349016005,   1],
      [:airy_Bi_e, [0.6999999999999907, GSL::MODE_DEFAULT],  0.9733286558781599,   0],
      [:airy_Bi_e, [1.649999999999991, GSL::MODE_DEFAULT],   2.196407956850028,    0],
      [:airy_Bi_e, [2.54999999999999, GSL::MODE_DEFAULT],    6.973628612493443,    0],
      [:airy_Bi_e, [3.499999999999987, GSL::MODE_DEFAULT],   33.05550675461069,    1],
      [:airy_Bi_e, [5.39999999999998, GSL::MODE_DEFAULT],    1604.476078241272,    1],
      [:airy_Bi_scaled_e, [-5.0, GSL::MODE_DEFAULT],                  -0.1383691349016005, 1],
      [:airy_Bi_scaled_e, [0.6999999999999907, GSL::MODE_DEFAULT],  0.6587080754582302, 0],
      [:airy_Bi_scaled_e, [1.649999999999991, GSL::MODE_DEFAULT],   0.5346449995597539, 0],
      [:airy_Bi_scaled_e, [2.54999999999999, GSL::MODE_DEFAULT],    0.461835455542297,  0],
      [:airy_Bi_scaled_e, [3.499999999999987, GSL::MODE_DEFAULT],   0.4201771882353061, 1],
      [:airy_Bi_scaled_e, [5.39999999999998, GSL::MODE_DEFAULT],    0.3734050675720473, 0],
      [:airy_Ai_deriv_e, [-5.0, GSL::MODE_DEFAULT],                 0.3271928185544435,       1],
      [:airy_Ai_deriv_e, [-0.5500000000000094, GSL::MODE_DEFAULT], -0.1914604987143629,    0],
      [:airy_Ai_deriv_e, [0.4999999999999906, GSL::MODE_DEFAULT],  -0.2249105326646850,    0],
      [:airy_Ai_deriv_e, [1.899999999999992, GSL::MODE_DEFAULT],   -0.06043678178575718,   0],
      [:airy_Ai_deriv_e, [3.249999999999988, GSL::MODE_DEFAULT],   -0.007792687926790889,  0],
      [:airy_Ai_deriv_e, [5.199999999999981, GSL::MODE_DEFAULT],   -0.0001589434526459543, 1],
      [:airy_Ai_deriv_scaled_e, [-5.0, GSL::MODE_DEFAULT],                0.3271928185544435, 1],
      [:airy_Ai_deriv_scaled_e, [0.5499999999999906, GSL::MODE_DEFAULT], -0.2874057279170166, 0],
      [:airy_Ai_deriv_scaled_e, [1.499999999999991, GSL::MODE_DEFAULT],  -0.3314199796863637, 0],
      [:airy_Ai_deriv_scaled_e, [2.49999999999999, GSL::MODE_DEFAULT],   -0.3661089384751620, 0],
      [:airy_Ai_deriv_scaled_e, [3.649999999999986, GSL::MODE_DEFAULT],  -0.3974033831453963, 0],
      [:airy_Ai_deriv_scaled_e, [6.299999999999977, GSL::MODE_DEFAULT],  -0.4508799189585947, 0],
      [:airy_Bi_deriv_e, [-5.0, GSL::MODE_DEFAULT],                0.778411773001899,  0],
      [:airy_Bi_deriv_e, [-0.5500000000000094, GSL::MODE_DEFAULT], 0.5155785358765014, 0],
      [:airy_Bi_deriv_e, [0.4999999999999906, GSL::MODE_DEFAULT],  0.5445725641405883, 0],
      [:airy_Bi_deriv_e, [1.899999999999992, GSL::MODE_DEFAULT],   3.495165862891568,  0],
      [:airy_Bi_deriv_e, [3.249999999999988, GSL::MODE_DEFAULT],   36.55485149250338,  0],
      [:airy_Bi_deriv_e, [5.199999999999981, GSL::MODE_DEFAULT],   2279.748293583233,  1],
      [:airy_Bi_deriv_scaled_e, [-5.0, GSL::MODE_DEFAULT],               0.778411773001899,  0],
      [:airy_Bi_deriv_scaled_e, [0.5499999999999906, GSL::MODE_DEFAULT], 0.4322811281817566, 0],
      [:airy_Bi_deriv_scaled_e, [1.499999999999991, GSL::MODE_DEFAULT],  0.5542307563918037, 0],
      [:airy_Bi_deriv_scaled_e, [2.49999999999999, GSL::MODE_DEFAULT],   0.6755384441644985, 0],
      [:airy_Bi_deriv_scaled_e, [3.649999999999986, GSL::MODE_DEFAULT],  0.7613959373000228, 0],
      [:airy_Bi_deriv_scaled_e, [6.299999999999977, GSL::MODE_DEFAULT],  0.8852064139737571, 0],
      [:airy_zero_Ai_e, [2],  -4.087949444130970617, 0],
      [:airy_zero_Ai_e, [50], -38.02100867725525443, 0],
      [:airy_zero_Ai_e, [100],  -60.45555727411669871, 0],
      [:airy_zero_Ai_e, [110],  -64.43135670991324811, 0],
      [:airy_zero_Bi_e, [2], -3.271093302836352716, 0],
      [:airy_zero_Bi_e, [50], -37.76583438165180116, 0],
      [:airy_zero_Bi_e, [100], -60.25336482580837088, 0],
      [:airy_zero_Bi_e, [110], -64.2355167606561537,  0],
      [:airy_zero_Bi_e, [111], -64.6268994819519378,  0],
      [:airy_zero_Bi_e, [200], -95.88699147356682665, 0],
      [:airy_zero_Ai_deriv_e, [2], -3.248197582179836561, 0],
      [:airy_zero_Ai_deriv_e, [50], -37.76565910053887108, 0],
      [:airy_zero_Ai_deriv_e, [100], -60.25329596442479317, 0],
      [:airy_zero_Ai_deriv_e, [110], -64.23545617243546956, 0],
      [:airy_zero_Ai_deriv_e, [1000], -280.9378080358935071, 0],
      [:airy_zero_Bi_deriv_e, [2], -4.073155089071828216, 0],
      [:airy_zero_Bi_deriv_e, [50], -38.02083574095788210, 0],
      [:airy_zero_Bi_deriv_e, [100], -60.45548887257140819, 0],
      [:airy_zero_Bi_deriv_e, [110], -64.43129648944845060, 0],
      [:airy_zero_Bi_deriv_e, [111], -64.82208737584206093, 0],
      [:airy_zero_Bi_deriv_e, [200], -96.04731050310324450, 0],
      [:airy_zero_Bi_deriv_e, [1000], -281.0315164471118527, 0]
    ],
    :bessel => [
      [:bessel_J0_e, [0.1],     0.99750156206604003230,    0],
      [:bessel_J0_e, [2.0],     0.22389077914123566805,    0],
      [:bessel_J0_e, [100.0],   0.019985850304223122424,   0],
      [:bessel_J0_e, [1.0e+10], 2.1755917502468917269e-06, TEST_SQRT_TOL0],
      [:bessel_J1_e, [0.1],      0.04993752603624199756,   0],
      [:bessel_J1_e, [2.0],      0.57672480775687338720,   0],
      [:bessel_J1_e, [100.0],   -0.07714535201411215803,   0],
      [:bessel_J1_e, [1.0e+10], -7.676508175684157103e-06, 4],
      [:bessel_Jn_e, [4, 0.1],     2.6028648545684032338e-07, 0],
      [:bessel_Jn_e, [5, 2.0],     0.007039629755871685484,   0],
      [:bessel_Jn_e, [10, 20.0],   0.18648255802394508321,    0],
      [:bessel_Jn_e, [100, 100.0], 0.09636667329586155967,    0],
      [:bessel_Jn_e, [2, 900.0], -0.019974345269680646400, 4],
      [:bessel_Jn_e, [2, 15000.0], -0.0020455820181216382666, 4],
      [:bessel_Jn_e, [0, 1.0e+10], 2.1755917502468917269e-06, TEST_SQRT_TOL0],
      [:bessel_Jn_e, [1, 1.0e+10], -7.676508175684157103e-06, 4],
#     [:bessel_Jn_e, [0, 20000], -7.676508175684157103e-06, 4],
      [:bessel_Y0_e, [0.1],         -1.5342386513503668441,    0],
      [:bessel_Y0_e, [2],            0.5103756726497451196,    0],
      [:bessel_Y0_e, [256.0],       -0.03381290171792454909,  0],
      [:bessel_Y0_e, [4294967296.0], 3.657903190017678681e-06, TEST_SQRT_TOL0],
      [:bessel_Y1_e, [0.1],         -6.45895109470202698800,     0],
      [:bessel_Y1_e, [2],           -0.10703243154093754689,     0],
      [:bessel_Y1_e, [100.0],       -0.020372312002759793305,    0],
      [:bessel_Y1_e, [4294967296.0], 0.000011612249378370766284, 4],
      [:bessel_Yn_e, [4, 0.1],            -305832.29793353160319,    1],
      [:bessel_Yn_e, [5, 2],              -9.935989128481974981,     0],
      [:bessel_Yn_e, [100, 100.0],        -0.16692141141757650654,   0],
      [:bessel_Yn_e, [100, 4294967296.0],  3.657889671577715808e-06, TEST_SQRT_TOL0],
      [:bessel_Yn_e, [1000, 4294967296.0], 3.656551321485397501e-06, 2.0e-05],
      [:bessel_Yn_e, [2, 15000.0], -0.006185217273358617849, 4],
      [:bessel_I0_scaled_e, [1e-10],   0.99999999990000000001,   0],
      [:bessel_I0_scaled_e, [0.1],     0.90710092578230109640,   0],
      [:bessel_I0_scaled_e, [2],       0.30850832255367103953,   0],
      [:bessel_I0_scaled_e, [100.0],   0.03994437929909668265,   0],
      [:bessel_I0_scaled_e, [65536.0], 0.0015583712551952223537, 0],
      [:bessel_I1_scaled_e, [0.1],     0.04529844680880932501,   0],
      [:bessel_I1_scaled_e, [2],       0.21526928924893765916,   0],
      [:bessel_I1_scaled_e, [100.0],   0.03974415302513025267,   0],
      [:bessel_I1_scaled_e, [65536.0], 0.0015583593657207350452, 0],
      [:bessel_In_scaled_e, [  -4,    0.1], 2.3575258620054605307e-07, 0],
      [:bessel_In_scaled_e, [   4,    0.1], 2.3575258620054605307e-07, 0],
      [:bessel_In_scaled_e, [   5,    2.0], 0.0013297610941881578142, 0],
      [:bessel_In_scaled_e, [ 100,  100.0], 1.7266862628167695785e-22, 0],
      [:bessel_I0_e, [0.1], 1.0025015629340956014, 0],
      [:bessel_I0_e, [2.0], 2.2795853023360672674, 0],
      [:bessel_I0_e, [100.0], 1.0737517071310738235e+42, 2],
      [:bessel_I1_e, [0.1], 0.05006252604709269211,      0],
      [:bessel_I1_e, [2.0], 1.59063685463732906340,      0],
      [:bessel_I1_e, [100.0], 1.0683693903381624812e+42, 2],
      [:bessel_In_e, [   4,    0.1], 2.6054690212996573677e-07, 0],
      [:bessel_In_e, [   5,    2.0], 0.009825679323131702321,   0],
      [:bessel_In_e, [ 100,  100.0], 4.641534941616199114e+21,  2],
      [:bessel_K0_scaled_e, [0.1], 2.6823261022628943831, 0],
      [:bessel_K0_scaled_e, [2.0], 0.8415682150707714179, 0],
      [:bessel_K0_scaled_e, [100.0], 0.1251756216591265789, 0],
      [:bessel_K1_scaled_e, [0.1], 10.890182683049696574, 0],
      [:bessel_K1_scaled_e, [2.0], 1.0334768470686885732, 0],
      [:bessel_K1_scaled_e, [100.0], 0.1257999504795785293, 0],
      [:bessel_Kn_scaled_e, [   4,    0.1], 530040.2483725626207, 1],
      [:bessel_Kn_scaled_e, [   5,    2.0], 69.68655087607675118, 0],
      [:bessel_Kn_scaled_e, [ 100,  100.0], 2.0475736731166756813e+19, 1],
      [:bessel_K0_e, [0.1], 2.4270690247020166125, 0],
      [:bessel_K0_e, [2.0], 0.11389387274953343565, 0],
      [:bessel_K0_e, [100.0], 4.656628229175902019e-45, 2],
      [:bessel_K1_e, [0.1], 9.853844780870606135,       0],
      [:bessel_K1_e, [2.0], 0.13986588181652242728,     0],
      [:bessel_K1_e, [100.0], 4.679853735636909287e-45, 2],
      [:bessel_Kn_e, [   4,    0.1], 479600.2497925682849,     1],
      [:bessel_Kn_e, [   5,    2.0], 9.431049100596467443,     0],
      [:bessel_Kn_e, [ 100,  100.0], 7.617129630494085416e-25, 2],
      [:bessel_j0_e, [-10.0], -0.05440211108893698134, 0],
      [:bessel_j0_e, [0.001], 0.9999998333333416667, 0],
      [:bessel_j0_e, [  1.0], 0.84147098480789650670, 0],
      [:bessel_j0_e, [ 10.0], -0.05440211108893698134, 0],
      [:bessel_j0_e, [100.0], -0.005063656411097587937, 1],
      [:bessel_j0_e, [1048576.0], 3.1518281938718287624e-07, 2],
      [:bessel_j1_e, [-10.0], -0.07846694179875154709, 0],
      [:bessel_j1_e, [0.01], 0.003333300000119047399, 0],
      [:bessel_j1_e, [  1.0], 0.30116867893975678925, 0],
      [:bessel_j1_e, [ 10.0], 0.07846694179875154709, 0],
      [:bessel_j1_e, [100.0], -0.008673825286987815220, 0],
      [:bessel_j1_e, [1048576.0], -9.000855242905546158e-07, 0],
      [:bessel_j2_e, [-10.0], 0.07794219362856244547, 0],
      [:bessel_j2_e, [0.01], 6.666619047751322551e-06, 0],
      [:bessel_j2_e, [  1.0], 0.06203505201137386110, 0],
      [:bessel_j2_e, [ 10.0], 0.07794219362856244547, 0],
      [:bessel_j2_e, [100.0], 0.004803441652487953480, 1],
      [:bessel_j2_e, [1048576.0], -3.1518539455252413111e-07, 2],
      [:bessel_jl_e, [0, 0.0], 1.0, 0],
      [:bessel_jl_e, [1,       10.0],   0.07846694179875154709000, 0],
      [:bessel_jl_e, [5,        1.0],   0.00009256115861125816357, 0],
      [:bessel_jl_e, [10,      10.0],   0.06460515449256426427,    0],
      [:bessel_jl_e, [100,    100.0],   0.010880477011438336539,   1],
      [:bessel_jl_e, [2000, 1048576.0], 7.449384239168568534e-07,  TEST_SQRT_TOL0],
      [:bessel_jl_e, [2, 900.0],   -0.0011089115568832940086,  4],
      [:bessel_jl_e, [2, 15000.0], -0.00005955592033075750554, 4],
      [:bessel_y0_e, [0.001], -999.99950000004166670, 0],
      [:bessel_y0_e, [  1.0], -0.5403023058681397174, 0],
      [:bessel_y0_e, [ 10.0], 0.08390715290764524523, 0],
      [:bessel_y0_e, [100.0], -0.008623188722876839341, 0],
      [:bessel_y0_e, [65536.0], 0.000011014324202158573930, 0],
      [:bessel_y0_e, [4294967296.0], 2.0649445131370357007e-10, TEST_SQRT_TOL0],
      [:bessel_y1_e, [ 0.01], -10000.499987500069444, 0],
      [:bessel_y1_e, [  1.0], -1.3817732906760362241, 0],
      [:bessel_y1_e, [ 10.0], 0.06279282637970150586, 0],
      [:bessel_y1_e, [100.0], 0.004977424523868819543, 0],
      [:bessel_y1_e, [4294967296.0], 1.0756463271573404688e-10, TEST_SQRT_TOL0],
      [:bessel_y2_e, [ 0.01], -3.0000500012499791668e+06, 0],
      [:bessel_y2_e, [  1.0], -3.605017566159968955, 0],
      [:bessel_y2_e, [ 10.0], -0.06506930499373479347, 0],
      [:bessel_y2_e, [100.0], 0.008772511458592903927, 0],
      [:bessel_y2_e, [4294967296.0], -2.0649445123857054207e-10, TEST_SQRT_TOL0],
      [:bessel_yl_e, [0,        0.01], -99.995000041666528,    0],
      [:bessel_yl_e, [0,        1.0],  -0.54030230586813972,   0],
      [:bessel_yl_e, [1,       10.0],   0.062792826379701506,   0],
      [:bessel_yl_e, [5,        1.0],  -999.44034339223641,     0],
      [:bessel_yl_e, [10,       0.01], -6.5473079797378378e+30, 0],
      [:bessel_yl_e, [10,      10.0],  -0.172453672088057849,    0],
      [:bessel_yl_e, [100,      1.0],  -6.6830794632586775e+186, 1],
      [:bessel_yl_e, [100,    100.0],  -0.0229838504915622811,   1],
      [:bessel_yl_e, [2000, 1048576.0], 5.9545201447146155e-07,  TEST_SQRT_TOL0],
      [:bessel_i0_scaled_e, [0.1], 0.9063462346100907067, 0],
      [:bessel_i0_scaled_e, [2.0], 0.24542109027781645493, 0],
      [:bessel_i0_scaled_e, [100.0], 0.005000000000000000000, 0],
      [:bessel_i1_scaled_e, [0.1], 0.030191419289002226846, 0],
      [:bessel_i1_scaled_e, [2.0], 0.131868364583275317610, 0],
      [:bessel_i1_scaled_e, [100.0], 0.004950000000000000000, 0],
      [:bessel_i2_scaled_e, [0.1], 0.0006036559400239012567, 0],
      [:bessel_i2_scaled_e, [2.0], 0.0476185434029034785100, 0],
      [:bessel_i2_scaled_e, [100.0], 0.0048515000000000000000, 0],
      [:bessel_il_scaled_e, [   4, 0.001], 1.0571434341190365013e-15, 0],
      [:bessel_il_scaled_e, [   4,   0.1], 9.579352242057134927e-08,  1],
      [:bessel_il_scaled_e, [   5,   2.0], 0.0004851564602127540059,  0],
      [:bessel_il_scaled_e, [   5, 100.0], 0.004300446777500000000,   1],
      [:bessel_il_scaled_e, [ 100, 100.0], 1.3898161964299132789e-23, 0],
      [:bessel_k0_scaled_e, [0.1], 15.707963267948966192, 0],
      [:bessel_k0_scaled_e, [2.0], 0.7853981633974483096, 0],
      [:bessel_k0_scaled_e, [100.0], 0.015707963267948966192, 0],
      [:bessel_k1_scaled_e, [0.1], 172.78759594743862812, 0],
      [:bessel_k1_scaled_e, [2.0], 1.1780972450961724644, 0],
      [:bessel_k1_scaled_e, [100.0], 0.015865042900628455854, 0],
      [:bessel_k2_scaled_e, [0.1], 5199.335841691107810, 0],
      [:bessel_k2_scaled_e, [2.0], 2.5525440310417070063, 0],
      [:bessel_k2_scaled_e, [100.0], 0.016183914554967819868, 0],
      [:bessel_kl_scaled_e, [   4, 1.0 / 256.0], 1.8205599816961954439e+14, 0],
      [:bessel_kl_scaled_e, [   4, 1.0 / 8.0],   6.1173217814406597530e+06, 0],
      [:bessel_kl_scaled_e, [   5,   2.0],     138.10735829492005119,     0],
      [:bessel_kl_scaled_e, [ 100, 100.0],     3.985930768060258219e+18,  1],
      [:bessel_Jnu_e, [0.0001, 1.0],         0.7652115411876708497,  2],
      [:bessel_Jnu_e, [0.0001, 10.0],       -0.2459270166445205,     2],
      [:bessel_Jnu_e, [0.0009765625, 10.0], -0.2458500798634692,     2],
      [:bessel_Jnu_e, [0.75, 1.0],           0.5586524932048917478,  2],
      [:bessel_Jnu_e, [0.75, 10.0],         -0.04968928974751508135, 2],
      [:bessel_Jnu_e, [ 1.0, 0.001], 0.0004999999375000026,     0],
      [:bessel_Jnu_e, [ 1.0,   1.0], 0.4400505857449335160,     0],
      [:bessel_Jnu_e, [ 1.75,  1.0], 0.168593922545763170103,     1],
      [:bessel_Jnu_e, [30.0,   1.0], 3.482869794251482902e-42,  0],
      [:bessel_Jnu_e, [30.0, 100.0], 0.08146012958117222297,    1],
      [:bessel_Jnu_e, [10.0,   1.0], 2.6306151236874532070e-10, 0],
      [:bessel_Jnu_e, [10.0, 100.0], -0.05473217693547201474,   2],
      [:bessel_Jnu_e, [10.2, 100.0], -0.03548919161046526864,   2],
      [:bessel_Jnu_e, [2.0, 900.0],   -0.019974345269680646400,  4],
      [:bessel_Jnu_e, [2.0, 15000.0], -0.0020455820181216382666, 4],
      [:bessel_Ynu_e, [0.0001, 1.0],  0.08813676933044478439,    2],
      [:bessel_Ynu_e, [0.0001,10.0],  0.05570979797521875261,    2],
      [:bessel_Ynu_e, [ 0.75,  1.0], -0.6218694174429746383,     0],
      [:bessel_Ynu_e, [ 0.75, 10.0],  0.24757063446760384953,    0],
      [:bessel_Ynu_e, [ 1.0, 0.001], -636.6221672311394281,      0],
      [:bessel_Ynu_e, [ 1.0,   1.0], -0.7812128213002887165,     0],
      [:bessel_Ynu_e, [30.0,   1.0], -3.0481287832256432162e+39, 0],
      [:bessel_Ynu_e, [30.0, 100.0],  0.006138839212010033452,   2],
      [:bessel_Ynu_e, [10.0,   1.0], -1.2161801427868918929e+08, 0],
      [:bessel_Ynu_e, [10.0, 100.0],  0.05833157423641492875,    2],
      [:bessel_Ynu_e, [10.2, 100.0],  0.07169383985546287091,    2],
      [:bessel_Inu_scaled_e, [0.0001,10.0], 0.12783333709581669672,    0],
      [:bessel_Inu_scaled_e, [ 1.0, 0.001], 0.0004995003123542213370,  0],
      [:bessel_Inu_scaled_e, [ 1.0,   1.0], 0.20791041534970844887,    0],
      [:bessel_Inu_scaled_e, [30.0,   1.0], 1.3021094983785914437e-42, 0],
      [:bessel_Inu_scaled_e, [30.0, 100.0], 0.0004486987756920986146,  3],
      [:bessel_Inu_scaled_e, [10.0,   1.0], 1.0127529864692066036e-10, 0],
      [:bessel_Inu_scaled_e, [10.0, 100.0], 0.024176682718258828365,   3],
      [:bessel_Inu_scaled_e, [10.2, 100.0], 0.023691628843913810043,   3],
      [:bessel_Inu_e, [0.0001,10.0], 2815.7166269770030352,     0],
      [:bessel_Inu_e, [ 1.0, 0.001], 0.0005000000625000026042,  0],
      [:bessel_Inu_e, [ 1.0,   1.0], 0.5651591039924850272,     0],
      [:bessel_Inu_e, [30.0,   1.0], 3.539500588106447747e-42,  0],
      [:bessel_Inu_e, [30.0, 100.0], 1.2061548704498434006e+40, 2],
      [:bessel_Inu_e, [10.0,   1.0], 2.7529480398368736252e-10, 0],
      [:bessel_Inu_e, [10.0, 100.0], 6.498975524720147799e+41,  2],
      [:bessel_Inu_e, [10.2, 100.0], 6.368587361287030443e+41,  2],
      [:bessel_Knu_scaled_e, [0.0001,10.0], 0.3916319346235421817, 0],
      [:bessel_Knu_scaled_e, [ 1.0, 0.001], 1000.9967345590684524, 0],
      [:bessel_Knu_scaled_e, [ 1.0,   1.0], 1.6361534862632582465, 0],
      [:bessel_Knu_scaled_e, [30.0,   1.0], 1.2792629867539753925e+40, 0],
      [:bessel_Knu_scaled_e, [30.0, 100.0], 10.673443449954850040, 0],
      [:bessel_Knu_scaled_e, [10.0,   1.0], 4.912296520990198599e+08, 0],
      [:bessel_Knu_scaled_e, [10.0, 100.0], 0.20578687173955779807, 0],
      [:bessel_Knu_scaled_e, [10.0, 1000.0], 0.04165905142800565788, 0],
      [:bessel_Knu_scaled_e, [10.0, 1.0e+8], 0.00012533147624060789938, 0],
      [:bessel_Knu_scaled_e, [10.2, 100.0], 0.20995808355244385075, 0],
      [:bessel_Knu_e, [0.0001,0.001], 7.023689431812884141,      0],
      [:bessel_Knu_e, [0.0001,10.0], 0.000017780062324654874306, 0],
      [:bessel_Knu_e, [ 1.0, 0.001], 999.9962381560855743,      0],
      [:bessel_Knu_e, [ 1.0,   1.0], 0.6019072301972345747,     0],
      [:bessel_Knu_e, [10.0, 0.001], 1.8579455483904008064e+38, 0],
      [:bessel_Knu_e, [10.0,   1.0], 1.8071328990102945469e+08, 0],
      [:bessel_Knu_e, [10.0, 100.0], 7.655427977388100611e-45,  2],
      [:bessel_Knu_e, [10.2, 100.0], 7.810600225948217841e-45,  2],
      [:bessel_Knu_e, [30.0,   1.0], 4.706145526783626883e+39,  1],
      [:bessel_Knu_e, [30.0, 100.0], 3.970602055959398739e-43,  2],
      [:bessel_lnKnu_e, [0.0001,1.0e-100], 5.439794449319847, 0],
      [:bessel_lnKnu_e, [0.0001,0.0001], 2.232835507214331, 0],
      [:bessel_lnKnu_e, [0.0001,10.0], -10.93743282256098, 0],
      [:bessel_lnKnu_e, [ 1.0, 1.0e-100], 230.2585092994045, 0],
      [:bessel_lnKnu_e, [ 1.0, 1.0e-10], 23.025850929940456840, 0],
      [:bessel_lnKnu_e, [ 1.0, 0.001], 6.907751517131146, 0],
      [:bessel_lnKnu_e, [ 1.0,   1.0], -0.5076519482107523309, 0],
      [:bessel_lnKnu_e, [30.0, 1.0e-100], 6999.113586185543475, 0],
      [:bessel_lnKnu_e, [30.0,   1.0], 91.34968784026325464, 0],
      [:bessel_lnKnu_e, [30.0, 100.0], -97.63224126416760932, 0],
      [:bessel_lnKnu_e, [100.0, 1.0e-100], 23453.606706185466825, 0],
      [:bessel_lnKnu_e, [100.0, 1.0], 427.7532510250188083, 0],
      [:bessel_lnKnu_e, [100.0, 100.0], -55.53422771502921431, 0],
      [:bessel_lnKnu_e, [1000.0, 1.0e-100], 236856.183755993135, 0],
      [:bessel_lnKnu_e, [10000.0, 1.0e-100], 2.39161558914890695e+06, 0],
      [:bessel_zero_J0_e, [ 1],  2.404825557695771, 1],
      [:bessel_zero_J0_e, [ 2],  5.520078110286304, 1],
      [:bessel_zero_J0_e, [20], 62.048469190227081, 1],
      [:bessel_zero_J0_e, [25], 77.756025630388058, 1],
      [:bessel_zero_J0_e, [100], 313.37426607752784, 1],
      [:bessel_zero_J1_e, [ 1], 3.831705970207512, 2],
      [:bessel_zero_J1_e, [ 2], 7.015586669815619, 2],
      [:bessel_zero_J1_e, [20], 63.61135669848124, 2],
      [:bessel_zero_J1_e, [25], 79.32048717547630, 2],
      [:bessel_zero_J1_e, [100], 314.9434728377672, 2],
      [:bessel_zero_Jnu_e, [0.0,  1],  2.404825557695771, 1],
      [:bessel_zero_Jnu_e, [0.0,  2],  5.520078110286304, 1],
      [:bessel_zero_Jnu_e, [0.0, 20], 62.048469190227081, 1],
      [:bessel_zero_Jnu_e, [0.0, 25], 77.756025630388058, 1],
      [:bessel_zero_Jnu_e, [0.0, 100], 313.37426607752784, 1],
      [:bessel_zero_Jnu_e, [ 1.5, 1],  4.4934094579090641, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 1],  8.7714838159599540, 1],
      [:bessel_zero_Jnu_e, [ 1.5, 2],  7.7252518369377072, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 2],  12.338604197466944, 1],
      [:bessel_zero_Jnu_e, [ 1.5, 3],  10.904121659428900, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 3],  15.700174079711671, 1],
      [:bessel_zero_Jnu_e, [ 1.5, 4],  14.066193912831473, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 4],  18.980133875179921, 1],
      [:bessel_zero_Jnu_e, [ 1.5, 5],  17.220755271930768, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 5],  22.217799896561268, TEST_SQRT_TOL0],
      [:bessel_zero_Jnu_e, [ 8.0, 5],  26.266814641176644, TEST_SQRT_TOL0],
      [:bessel_zero_Jnu_e, [20.0, 5],  41.413065513892636, TEST_SQRT_TOL0],
      [:bessel_zero_Jnu_e, [ 1.5, 6],  20.371302959287563, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 6],  25.430341154222704, 1],
      [:bessel_zero_Jnu_e, [ 8.0, 6],  29.545659670998550, 1],
      [:bessel_zero_Jnu_e, [ 1.5, 7],  23.519452498689007, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 7],  28.626618307291138, 1],
      [:bessel_zero_Jnu_e, [ 8.0, 7],  32.795800037341462, 1],
      [:bessel_zero_Jnu_e, [ 1.5, 8],  26.666054258812674, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 8],  31.811716724047763, 1],
      [:bessel_zero_Jnu_e, [10.0, 8],  38.761807017881651, 1],
      [:bessel_zero_Jnu_e, [ 1.5, 9],  29.811598790892959, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 9],  34.988781294559295, 1],
      [:bessel_zero_Jnu_e, [10.0, 9],  42.004190236671805, 1],
      [:bessel_zero_Jnu_e, [ 1.5, 10],  32.956389039822477, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 10],  38.159868561967132, 1],
      [:bessel_zero_Jnu_e, [15.0, 10],  52.017241278881633, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 11], 41.326383254047406, 1],
      [:bessel_zero_Jnu_e, [15.0, 11], 55.289204146560061, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 12], 44.4893191232197314, 1],
      [:bessel_zero_Jnu_e, [15.0, 12], 58.5458289043850856, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 13], 47.6493998066970948, 1],
      [:bessel_zero_Jnu_e, [15.0, 13], 61.7897598959450550, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 14], 50.8071652030063595, 1],
      [:bessel_zero_Jnu_e, [15.0, 14], 65.0230502510422545, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 15], 53.9630265583781707, 1],
      [:bessel_zero_Jnu_e, [15.0, 15], 68.2473219964207837, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 16], 57.1173027815042647, 1],
      [:bessel_zero_Jnu_e, [15.0, 16], 71.4638758850226630, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 17], 60.2702450729428077, 1],
      [:bessel_zero_Jnu_e, [15.0, 17], 74.6737687121404241, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 18], 63.4220540458757799, 1],
      [:bessel_zero_Jnu_e, [15.0, 18], 77.8778689734863729, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 19], 66.5728918871182703, 1],
      [:bessel_zero_Jnu_e, [15.0, 19], 81.0768977206328326, 1],
      [:bessel_zero_Jnu_e, [ 5.0, 20], 69.722891161716742,  1],
      [:bessel_zero_Jnu_e, [15.0, 20], 84.271459069716442,  1],
      [:bessel_zero_Jnu_e, [ 23.0, 11], 65.843393469524653, 6],
      [:bessel_zero_Jnu_e, [ 30.0, 11], 74.797306585175426, 6],
      [:bessel_zero_Jnu_e, [ 32.0, 15], 90.913637691861741, 6],
      [:bessel_zero_Jnu_e, [ 50.0, 15], 113.69747988073942, 6],
      [:bessel_zero_Jnu_e, [  5.0, 22], 76.020793430591605, 2],
      [:bessel_zero_Jnu_e, [ 10.0, 22], 83.439189796105756, 3],
      [:bessel_zero_Jnu_e, [ 12.0, 22], 86.345496520534055, 6],
      [:bessel_zero_Jnu_e, [100.0, 22], 199.82150220122519, 4],
      [:bessel_zero_Jnu_e, [500.0, 22], 649.34132440891735, 2]
    ],
    :coulomb => [
      [:hydrogenicR_1_e, [3.0, 2.0],  0.025759948256148471036,  0],
      [:hydrogenicR_1_e, [3.0, 10.0], 9.724727052062819704e-13, 1],
      [:hydrogenicR_e, [4, 0, 3.0, 2.0], -0.03623182256981820062,  2],
      [:hydrogenicR_e, [4, 1, 3.0, 2.0], -0.028065049083129581005, 2],
      [:hydrogenicR_e, [4, 2, 3.0, 2.0],  0.14583027278668431009,  0],
      [:hydrogenicR_e, [100,  0, 3.0, 2.0], -0.00007938950980052281367, 3],
      [:hydrogenicR_e, [100, 10, 3.0, 2.0],  7.112823375353605977e-12,  2],
      [:hydrogenicR_e, [100, 90, 3.0, 2.0],  5.845231751418131548e-245, 2]
    ],
    :dilog => [
      [:dilog_e, [-3.0],   -1.9393754207667089531,     0],
      [:dilog_e, [-0.5],   -0.4484142069236462024,     0],
      [:dilog_e, [-0.001], -0.0009997501110486510834,  0],
      [:dilog_e, [0.1],     0.1026177910993911,        0],
      [:dilog_e, [0.7],     0.8893776242860387386,     0],
      [:dilog_e, [1.0],     1.6449340668482260,        0],
      [:dilog_e, [1.5],     2.3743952702724802007,     0],
      [:dilog_e, [2.0],     2.4674011002723397,        0],
      [:dilog_e, [ 5.0],    1.7837191612666306277,     0],
      [:dilog_e, [ 11.0],   0.3218540439999117111,     1],
      [:dilog_e, [12.59],   0.0010060918167266208634,  3],
      [:dilog_e, [12.595],  0.00003314826006436236810, 5],
      [:dilog_e, [13.0],   -0.07806971248458575855,    2],
      [:dilog_e, [20.0],   -1.2479770861745251168,     0],
      [:dilog_e, [150.0],  -9.270042702348657270,      0],
      [:dilog_e, [1100.0], -21.232504073931749553,     0]
    ],
    :gamma => [
      [:lngamma_e, [-0.1], 2.368961332728788655, 0],
      [:lngamma_e, [-1.0 / 256.0], 5.547444766967471595, 0],
      [:lngamma_e, [1.0e-08], 18.420680738180208905, 0],
      [:lngamma_e, [0.1], 2.252712651734205, 0],
      [:lngamma_e, [1.0 + 1.0 / 256.0], -0.0022422226599611501448, 0],
      [:lngamma_e, [2.0 + 1.0 / 256.0], 0.0016564177556961728692, 0],
      [:lngamma_e, [100.0], 359.1342053695753, 0],
      [:lngamma_e, [-1.0-1.0 / 65536.0], 11.090348438090047844, 0],
      [:lngamma_e, [-1.0-1.0 / 268435456.0], 19.408121054103474300, 0],
      [:lngamma_e, [-100.5], -364.9009683094273518, 0],
      [:lngamma_e, [-100-1.0 / 65536.0], -352.6490910117097874, 0],
      [:gamma_e, [1.0 + 1.0 / 4096.0], 0.9998591371459403421, 0],
      [:gamma_e, [1.0 + 1.0 / 32.0], 0.9829010992836269148, 0],
      [:gamma_e, [2.0 + 1.0 / 256.0], 1.0016577903733583299, 0],
      [:gamma_e, [9.0], 40320.0, 0],
      [:gamma_e, [10.0], 362880.0, 0],
      [:gamma_e, [100.0], 9.332621544394415268e+155, 2],
      [:gamma_e, [170.0], 4.269068009004705275e+304, 2],
      [:gamma_e, [171.0], 7.257415615307998967e+306, 2],
      [:gamma_e, [-10.5], -2.640121820547716316e-07, 0],
      [:gamma_e, [-1.0 + 1.0 / 65536.0], -65536.42280587818970, 0],
      [:gammastar_e, [1.0e-08], 3989.423555759890865, 1],
      [:gammastar_e, [1.0e-05], 126.17168469882690233, 0],
      [:gammastar_e, [0.001], 12.708492464364073506, 0],
      [:gammastar_e, [1.5], 1.0563442442685598666, 0],
      [:gammastar_e, [3.0], 1.0280645179187893045, 0],
      [:gammastar_e, [9.0], 1.0092984264218189715, 0],
      [:gammastar_e, [11.0], 1.0076024283104962850, 0],
      [:gammastar_e, [100.0], 1.0008336778720121418, 0],
      [:gammastar_e, [1.0e+05], 1.0000008333336805529, 0],
      [:gammastar_e, [1.0e+20], 1.0, 0],
      [:gammainv_e, [1.0], 1.0, 0],
      [:gammainv_e, [2.0], 1.0, 0],
      [:gammainv_e, [3.0], 0.5, 0],
      [:gammainv_e, [4.0], 1.0 / 6.0, 0],
      [:gammainv_e, [10.0], 1.0 / 362880.0, 0],
      [:gammainv_e, [100.0], 1.0715102881254669232e-156, 2],
      [:gammainv_e, [0.0], 0.0, 0],
      [:gammainv_e, [-1.0], 0.0, 0],
      [:gammainv_e, [-2.0], 0.0, 0],
      [:gammainv_e, [-3.0], 0.0, 0],
      [:gammainv_e, [-4.0], 0.0, 0],
      [:gammainv_e, [-10.5], -1.0 / 2.640121820547716316e-07, 2],
      [:gammainv_e, [-11.25], 1.0 / 6.027393816261931672e-08, 1],
      [:gammainv_e, [-1.0 + 1.0 / 65536.0], -1.0 / 65536.42280587818970, 1],
      [:taylorcoeff_e, [10,   1.0 / 1048576.0], 1.7148961854776073928e-67, 0],
      [:taylorcoeff_e, [10,   1.0 / 1024.0], 2.1738891788497900281e-37, 0],
      [:taylorcoeff_e, [10,   1.0], 2.7557319223985890653e-07, 0],
      [:taylorcoeff_e, [10,   5.0], 2.6911444554673721340, 0],
      [:taylorcoeff_e, [10,   500.0], 2.6911444554673721340e+20, 0],
      [:taylorcoeff_e, [100,  100.0], 1.0715102881254669232e+42, 1],
      [:taylorcoeff_e, [1000, 200.0], 2.6628790558154746898e-267, 1],
      [:taylorcoeff_e, [1000, 500.0], 2.3193170139740855074e+131, 1],
      [:fact_e, [0], 1.0, 0],
      [:fact_e, [1], 1.0, 0],
      [:fact_e, [7], 5040.0, 0],
      [:fact_e, [33], 8.683317618811886496e+36, 0],
      [:doublefact_e, [0], 1.0, 0],
      [:doublefact_e, [1], 1.0, 0],
      [:doublefact_e, [7], 105.0, 0],
      [:doublefact_e, [33], 6.332659870762850625e+18, 0],
      [:lnfact_e, [0], 0.0, 0],
      [:lnfact_e, [1], 0.0, 0],
      [:lnfact_e, [7], 8.525161361065414300, 0],
      [:lnfact_e, [33], 85.05446701758151741, 0],
      [:lndoublefact_e, [0], 0.0, 0],
      [:lndoublefact_e, [7], 4.653960350157523371, 0],
      [:lndoublefact_e, [33], 43.292252022541719660, 0],
      [:lndoublefact_e, [34], 45.288575519655959140, 0],
      [:lndoublefact_e, [1034], 3075.6383796271197707, 0],
      [:lndoublefact_e, [1035], 3078.8839081731809169, 0],
      [:lnchoose_e, [7,3], 3.555348061489413680, 0],
      [:lnchoose_e, [5,2], 2.302585092994045684, 0],
      [:choose_e, [7,3], 35.0, 0],
      [:choose_e, [7,4], 35.0, 0],
      [:choose_e, [5,2], 10.0, 0],
      [:choose_e, [5,3], 10.0, 0],
      [:choose_e, [500,495], 255244687600.0, 0],
      [:choose_e, [500,5], 255244687600.0, 0],
      [:choose_e, [500,200], 5.054949849935532221e+144, 5],
      [:choose_e, [500,300], 5.054949849935532221e+144, 5],
      [:lnpoch_e, [5, 1.0 / 65536.0], 0.000022981557571259389129, 0],
      [:lnpoch_e, [5, 1.0 / 256.0],   0.005884960217985189004,    2],
      [:lnpoch_e, [7,3], 6.222576268071368616, 0],
      [:lnpoch_e, [5,2], 3.401197381662155375, 0],
      [:poch_e, [7,3], 504.0, 0],
      [:poch_e, [5,2], 30.0, 1],
      [:poch_e, [5,1.0 / 256.0], 1.0059023106151364982, 0],
      [:pochrel_e, [7,3], 503.0 / 3.0, 0],
      [:pochrel_e, [5,2], 29.0 / 2.0, 1],
      [:pochrel_e, [5,0.01], 1.5186393661368275330, 2],
      [:pochrel_e, [-5.5,0.01], 1.8584945633829063516, 1],
      [:pochrel_e, [-5.5,-1.0 / 8.0], 1.0883319303552135488, 1],
      [:pochrel_e, [-5.5,-1.0 / 256.0], 1.7678268037726177453, 1],
      [:pochrel_e, [-5.5,-11.0], 0.09090909090939652475, 0],
      [:gamma_inc_P_e, [0.001, 1.0], 0.9997803916424144436, 0],
      [:gamma_inc_P_e, [0.001, 10.0], 0.9999999958306921828, 0],
      [:gamma_inc_P_e, [1.0, 0.001], 0.0009995001666250083319, 0],
      [:gamma_inc_P_e, [1.0, 1.01], 0.6357810204284766802, 0],
      [:gamma_inc_P_e, [1.0, 10.0], 0.9999546000702375151, 0],
      [:gamma_inc_P_e, [10.0, 10.01], 0.5433207586693410570, 0],
      [:gamma_inc_P_e, [10.0, 20.0], 0.9950045876916924128, 0],
      [:gamma_inc_P_e, [1000.0, 1000.1], 0.5054666401440661753, 2],
      [:gamma_inc_P_e, [1000.0, 2000.0], 1.0, 0],
      [:gamma_inc_P_e, [34.0, 32.0], 0.3849626436463866776322932129, 2],
      [:gamma_inc_P_e, [37.0, 3.499999999999999289e+01], 0.3898035054195570860969333039, 2],
      [:gamma_inc_P_e, [10, 1e-16], 2.755731922398588814734648067e-167, 2],
      [:gamma_inc_Q_e, [0.001, 0.001], 0.006312353291139709793, 0],
      [:gamma_inc_Q_e, [0.001, 1.0], 0.00021960835758555639171, 1],
      [:gamma_inc_Q_e, [0.001, 2.0], 0.00004897691783098147880, 2],
      [:gamma_inc_Q_e, [0.001, 5.0], 1.1509813397308608541e-06, 1],
      [:gamma_inc_Q_e, [1.0, 0.001], 0.9990004998333749917, 0],
      [:gamma_inc_Q_e, [1.0, 1.01], 0.3642189795715233198, 0],
      [:gamma_inc_Q_e, [1.0, 10.0], 0.00004539992976248485154, 0],
      [:gamma_inc_Q_e, [10.0, 10.01], 0.4566792413306589430, 0],
      [:gamma_inc_Q_e, [10.0, 100.0], 1.1253473960842733885e-31, 2],
      [:gamma_inc_Q_e, [1000.0, 1000.1], 0.4945333598559338247, 2],
      [:gamma_inc_Q_e, [1000.0, 2000.0], 6.847349459614753180e-136, 2],
      [:gamma_inc_Q_e, [100,  99.0], 0.5266956696005394, 2],
      [:gamma_inc_Q_e, [200, 199.0], 0.5188414119121281, 2],
      [:gamma_inc_P_e, [100,  99.0], 0.4733043303994607, 2],
      [:gamma_inc_P_e, [200, 199.0], 0.4811585880878718, 2],
      [:gamma_inc_e, [-1.0 / 1048576.0, 1.0 / 1048576.0], 13.285819596290624271, 0],
      [:gamma_inc_e, [-0.001, 1.0 / 1048576.0], 13.381275128625328858, 0],
      [:gamma_inc_e, [-1.0,   1.0 / 1048576.0], 1.0485617142715768655e+06, 0],
      [:gamma_inc_e, [-0.00001,0.001], 6.3317681434563592142, 0],
      [:gamma_inc_e, [-0.0001,0.001], 6.3338276439767189385, 0],
      [:gamma_inc_e, [-0.001, 0.001], 6.3544709102510843793, 0],
      [:gamma_inc_e, [-0.5,   0.001], 59.763880515942196981, 0],
      [:gamma_inc_e, [-1.0,   0.001], 992.66896046923884234, 0],
      [:gamma_inc_e, [-3.5,   0.001], 9.0224404490639003706e+09, 1],
      [:gamma_inc_e, [-10.5,  0.001], 3.0083661558184815656e+30, 2],
      [:gamma_inc_e, [-0.001, 0.1], 1.8249109609418620068, 0],
      [:gamma_inc_e, [-0.5,   0.1], 3.4017693366916154163, 0],
      [:gamma_inc_e, [-10.0,  0.1], 8.9490757483586989181e+08, 1],
      [:gamma_inc_e, [-10.5,  0.1], 2.6967403834226421766e+09, 1],
      [:gamma_inc_e, [-0.001, 1.0], 0.21928612679072766340, 1],
      [:gamma_inc_e, [-0.5,   1.0], 0.17814771178156069019, 1],
      [:gamma_inc_e, [-1.0,   1.0], 0.14849550677592204792, 1],
      [:gamma_inc_e, [-2.5,   1.0], 0.096556648631275160264, 1],
      [:gamma_inc_e, [-1.0,   10.0], 3.8302404656316087616e-07, 1],
      [:gamma_inc_e, [-0.001, 10.0], 4.1470562324807320961e-06, 1],
      [:gamma_inc_e, [-0.5,   10.0], 1.2609042613241570681e-06, 0],
      [:gamma_inc_e, [-1.0,   10.0], 3.8302404656316087616e-07, 1],
      [:gamma_inc_e, [-10.5,  10.0], 6.8404927328441566785e-17, 1],
      [:gamma_inc_e, [-100.0, 10.0], 4.1238327669858313997e-107, 2],
      [:gamma_inc_e, [-200.0, 10.0], 2.1614091830529343423e-207, 2],
      [:gamma_inc_e, [  0.0,     0.001], 6.3315393641361493320, 0],
      [:gamma_inc_e, [  0.001,   0.001], 6.3087159394864007261, 0],
      [:gamma_inc_e, [  1.0,     0.001], 0.99900049983337499167, 0],
      [:gamma_inc_e, [ 10.0,     0.001], 362880.0, 0],
      [:gamma_inc_e, [  0.0,     1.0], 0.21938393439552027368, 0],
      [:gamma_inc_e, [  0.001,   1.0], 0.21948181320730279613, 1],
      [:gamma_inc_e, [  1.0,     1.0], 0.36787944117144232160, 0],
      [:gamma_inc_e, [ 10.0,     1.0], 362879.95956592242045, 0],
      [:gamma_inc_e, [100.0,     1.0], 9.3326215443944152682e+155, 0],
      [:gamma_inc_e, [  0.0,   100.0], 3.6835977616820321802e-46, 2],
      [:gamma_inc_e, [  0.001, 100.0], 3.7006367674063550631e-46, 2],
      [:gamma_inc_e, [  1.0,   100.0], 3.7200759760208359630e-44, 2],
      [:gamma_inc_e, [ 10.0,   100.0], 4.0836606309106112723e-26, 2],
      [:gamma_inc_e, [100.0,   100.0], 4.5421981208626694294e+155, 1],
      [:lnbeta_e, [1.0e-8, 1.0e-8],  19.113827924512310617, 0],
      [:lnbeta_e, [1.0e-8, 0.01],  18.420681743788563403, 0],
      [:lnbeta_e, [1.0e-8, 1.0],  18.420680743952365472, 0],
      [:lnbeta_e, [1.0e-8, 10.0],  18.420680715662683009, 0],
      [:lnbeta_e, [1.0e-8, 1000.0],  18.420680669107656949, 0],
      [:lnbeta_e, [0.1, 0.1], 2.9813614810376273949, 1],
      [:lnbeta_e, [0.1, 1.0],  2.3025850929940456840, 1],
      [:lnbeta_e, [0.1, 100.0],  1.7926462324527931217, 0],
      [:lnbeta_e, [0.1, 1000],  1.5619821298353164928, 0],
      [:lnbeta_e, [1.0, 1.00025],  -0.0002499687552073570, 4],
      [:lnbeta_e, [1.0, 1.01],  -0.009950330853168082848, 3],
      [:lnbeta_e, [1.0, 1000.0],  -6.907755278982137052, 0],
      [:lnbeta_e, [100.0, 100.0],  -139.66525908670663927, 2],
      [:lnbeta_e, [100.0, 1000.0],  -336.4348576477366051, 0],
      [:lnbeta_e, [100.0, 1.0e+8],  -1482.9339185256447309, 0],
      [:beta_e, [1.0,   1.0], 1.0, 0],
      [:beta_e, [1.0, 1.001], 0.9990009990009990010, 0],
      [:beta_e, [1.0,   5.0], 0.2, 1],
      [:beta_e, [1.0,  100.0], 0.01, 1],
      [:beta_e, [10.0, 100.0], 2.3455339739604649879e-15, 2],
      [:beta_inc_e, [1.0, 1.0, 0.0], 0.0, 2],
      [:beta_inc_e, [1.0, 1.0, 1.0], 1.0, 2],
      [:beta_inc_e, [0.1, 0.1, 1.0], 1.0, 2],
      [:beta_inc_e, [ 1.0,  1.0, 0.5], 0.5, 2],
      [:beta_inc_e, [ 0.1,  1.0, 0.5], 0.9330329915368074160, 2],
      [:beta_inc_e, [10.0,  1.0, 0.5], 0.0009765625000000000000, 2],
      [:beta_inc_e, [50.0,  1.0, 0.5], 8.881784197001252323e-16, 2],
      [:beta_inc_e, [ 1.0,  0.1, 0.5], 0.06696700846319258402, 2],
      [:beta_inc_e, [ 1.0, 10.0, 0.5], 0.99902343750000000000, 2],
      [:beta_inc_e, [ 1.0, 50.0, 0.5], 0.99999999999999911180, 2],
      [:beta_inc_e, [ 1.0,  1.0, 0.1], 0.10, 2],
      [:beta_inc_e, [ 1.0,  2.0, 0.1], 0.19, 2],
      [:beta_inc_e, [ 1.0,  2.0, 0.9], 0.99, 2],
      [:beta_inc_e, [50.0, 60.0, 0.5], 0.8309072939016694143, 2],
      [:beta_inc_e, [90.0, 90.0, 0.5], 0.5, 2],
      [:beta_inc_e, [ 500.0,  500.0, 0.6], 0.9999999999157549630, 2],
      [:beta_inc_e, [5000.0, 5000.0, 0.4], 4.518543727260666383e-91, 5],
      [:beta_inc_e, [5000.0, 5000.0, 0.6], 1.0, 2],
      [:beta_inc_e, [5000.0, 2000.0, 0.6], 8.445388773903332659e-89, 5]
    ],
    :hyperg => [
      [:hyperg_0F1_e, [1, 0.5],     1.5660829297563505373, 0],
      [:hyperg_0F1_e, [5, 0.5],     1.1042674404828684574, 1],
      [:hyperg_0F1_e, [100, 30],    1.3492598639485110176, 2],
      [:hyperg_0F1_e, [-0.5, 3],   -39.29137997543434276,  1],
      [:hyperg_0F1_e, [-100.5, 50], 0.6087930289227538496, 3],
      [:hyperg_0F1_e, [1, -5.0],   -0.3268752818235339109, 0],
      [:hyperg_0F1_e, [-0.5, -5.0],-4.581634759005381184,  1],
      [:hyperg_1F1_int_e, [1, 1, 0.5], 1.6487212707001281468, 0],
      [:hyperg_1F1_int_e, [1, 2, 500.0], 2.8071844357056748215e+214, 2],
      [:hyperg_1F1_int_e, [1, 2, -500.0], 0.002, 0],
      [:hyperg_1F1_int_e, [8, 1, 0.5], 13.108875178030540372, 0],
      [:hyperg_1F1_int_e, [10, 1, 1.0],  131.63017574352619931, 0],
      [:hyperg_1F1_int_e, [10, 1, 10.0], 8.514625476546280796e+09, 0],
      [:hyperg_1F1_int_e, [10, 1, 100.0],  1.5671363646800353320e+56, 2],
      [:hyperg_1F1_int_e, [10, 20, 1.0],  1.6585618002669675465, 2],
      [:hyperg_1F1_int_e, [10, 20, 10.0],  265.26686430340188871, 2],
      [:hyperg_1F1_int_e, [10, 20, 100.0], 3.640477355063227129e+34, 2],
      [:hyperg_1F1_int_e, [10, 100, 1.0],  1.1056660194025527099, 0],
      [:hyperg_1F1_int_e, [10, 100, 10.0],  2.8491063634727594206, 0],
      [:hyperg_1F1_int_e, [10, 100, 40.0],  133.85880835831230986, 0],
      [:hyperg_1F1_int_e, [10, 100, 80.0],  310361.16228011433406, 0],
      [:hyperg_1F1_int_e, [10, 100, 100.0],  8.032171336754168282e+07, 2],
      [:hyperg_1F1_int_e, [10, 100, 500.0],  7.633961202528731426e+123, 2],
      [:hyperg_1F1_int_e, [100, 1, 1.0],  6.892842729046469965e+07, 1],
      [:hyperg_1F1_int_e, [100, 1, 10.0],  2.4175917112200409098e+28, 1],
      [:hyperg_1F1_int_e, [100, 1, 100.0],  1.9303216896309102993e+110, 2],
      [:hyperg_1F1_int_e, [100, 200, 1.0],  1.6497469106162459226, 2],
      [:hyperg_1F1_int_e, [100, 200, 10.0],  157.93286197349321981, 2],
      [:hyperg_1F1_int_e, [100, 200, 100.0],  2.1819577501255075240e+24, 2],
      [:hyperg_1F1_int_e, [100, 200, 400.0],  3.728975529926573300e+119, 2],
      [:hyperg_1F1_int_e, [100, 400, 10.0],  12.473087623658878813, 0],
      [:hyperg_1F1_int_e, [100, 400, 100.0],  9.071230376818550241e+11, 1],
      [:hyperg_1F1_int_e, [100, 400, 150.0],  7.160949515742170775e+18, 0],
      [:hyperg_1F1_int_e, [100, 400, 200.0],   2.7406690412731576823e+26, 2],
      [:hyperg_1F1_int_e, [100, 400, 300.0],  6.175110613473276193e+43, 2],
      [:hyperg_1F1_int_e, [100, 400, 400.0],  1.1807417662711371440e+64, 3],
      [:hyperg_1F1_int_e, [100, 400, 600.0],  2.4076076354888886030e+112, 3],
      [:hyperg_1F1_int_e, [10, 1, -1.0],      0.11394854824644542810,   0],
      [:hyperg_1F1_int_e, [10, 1, -10.0],     0.0006715506365396127863, 1],
      [:hyperg_1F1_int_e, [10, 1, -100.0],   -4.208138537480269868e-32, 2],
      [:hyperg_1F1_int_e, [10, 50, -1.0],     0.820006196079380,        0],
      [:hyperg_1F1_int_e, [10, 100, -10.0],   0.38378859043466243,      0],
      [:hyperg_1F1_int_e, [10, 100, -100.0],  0.0008460143401464189061, 0],
      [:hyperg_1F1_int_e, [10, 100, -500.0],  1.1090822141973655929e-08, 0],
      [:hyperg_1F1_int_e, [10, 100, -10000.0], 5.173783508088272292e-21, 2],
      [:hyperg_1F1_int_e, [50, 1, -90.0],    -1.6624258547648311554e-21, 2],
      [:hyperg_1F1_int_e, [50, 1, -100.0],    4.069661775122048204e-24, 2],
      [:hyperg_1F1_int_e, [50, 1, -110.0],    1.0072444993946236025e-25, 2],
      [:hyperg_1F1_int_e, [100, 10, -100.0], -2.7819353611733941962e-37, 2],
      [:hyperg_1F1_int_e, [100, 1, -90.0],  7.501705041159802854e-22, 2],
      [:hyperg_1F1_int_e, [100, 1, -100.0],  6.305128893152291187e-25, 3],
      [:hyperg_1F1_int_e, [100, 1, -110.0],  -7.007122115422439755e-26, 3],
      [:hyperg_1F1_int_e, [100, 10, -100.0],  -2.7819353611733941962e-37, 2],
      [:hyperg_1F1_int_e, [200, 50, -1.0],  0.016087060191732290813, 1],
      [:hyperg_1F1_int_e, [200, 50, -300.0],  -4.294975979706421471e-121, 2],
      [:hyperg_1F1_int_e, [200, 100, -1.0],  0.13397521083325179687, 0],
      [:hyperg_1F1_int_e, [200, 100, -10.0],  5.835134393749807387e-10, 1],
      [:hyperg_1F1_int_e, [200, 100, -100.0],  4.888460453078914804e-74, 2],
      [:hyperg_1F1_int_e, [200, 100, -500.0],  -1.4478509059582015053e-195, 2],
      [:hyperg_1F1_int_e, [-1, 1, 2.0],  -1.0, 0],
      [:hyperg_1F1_int_e, [-1, -2, 2.0],  2.0, 0],
      [:hyperg_1F1_int_e, [-2, -3, 2.0],  3.0, 0],
      [:hyperg_1F1_int_e, [-10, 1, 1.0],  0.4189459325396825397, 0],
      [:hyperg_1F1_int_e, [-10, 1, 10.0],  27.984126984126984127, 0],
      [:hyperg_1F1_int_e, [-10, 1, 100.0],  9.051283795429571429e+12, 0],
      [:hyperg_1F1_int_e, [-100, 20, 1.0],  0.0020203016320697069566, 0],
      [:hyperg_1F1_int_e, [-10, -20, 1.0],  1.6379141878548080173, 0],
      [:hyperg_1F1_int_e, [-10, -20, 10.0],  78.65202404521289970, 0],
      [:hyperg_1F1_int_e, [-10, -20, 100.0],  4.416169713262624315e+08, 0],
      [:hyperg_1F1_int_e, [-10, -100, 1.0],  1.1046713999681950919, 0],
      [:hyperg_1F1_int_e, [-10, -100, 10.0],  2.6035952191039006838, 0],
      [:hyperg_1F1_int_e, [-10, -100, 100.0],  1151.6852040836932392, 0],
      [:hyperg_1F1_int_e, [-100, -200, 1.0],  1.6476859702535324743, 0],
      [:hyperg_1F1_int_e, [-100, -200, 10.0],  139.38026829540687270, 0],
      [:hyperg_1F1_int_e, [-100, -200, 100.0],  1.1669433576237933752e+19, 1],
      [:hyperg_1F1_int_e, [-10, -20, -1.0],  0.6025549561148035735, 0],
      [:hyperg_1F1_int_e, [-10, -20, -10.0],  0.00357079636732993491, 1],
      [:hyperg_1F1_int_e, [-10, -20, -100.0],  1.64284868563391159e-35, 2],
      [:hyperg_1F1_int_e, [-10, -100, -1.0],  0.90442397250313899, 0],
      [:hyperg_1F1_int_e, [-10, -100, -10.0],  0.35061515251367215, 1],
      [:hyperg_1F1_int_e, [-10, -100, -100.0],  8.19512187960476424e-09, 2],
      [:hyperg_1F1_int_e, [-100, -200, -1.0],  0.6061497939628952629, 0],
      [:hyperg_1F1_int_e, [-100, -200, -10.0],  0.0063278543908877674, 1],
      [:hyperg_1F1_int_e, [-100, -200, -100.0],  4.34111795007336552e-25, 2],
      [:hyperg_1F1_e, [1, 1.5, 1], 2.0300784692787049755, 0],
      [:hyperg_1F1_e, [1, 1.5, 10],  6172.859561078406855, 0],
      [:hyperg_1F1_e, [1, 1.5, 100],  2.3822817898485692114e+42, 1],
      [:hyperg_1F1_e, [1, 1.5, 500],  5.562895351723513581e+215, 2],
      [:hyperg_1F1_e, [1.5, 2.5, 1], 1.8834451238277954398, 0],
      [:hyperg_1F1_e, [1.5, 2.5, 10],  3128.7352996840916381, 1],
      [:hyperg_1F1_e, [10, 1.1, 1],  110.17623733873889579, 1],
      [:hyperg_1F1_e, [10, 1.1, 10],  6.146657975268385438e+09, 1],
      [:hyperg_1F1_e, [10, 1.1, 100], 9.331833897230312331e+55, 2],
      [:hyperg_1F1_e, [10, 1.1, 500],  4.519403368795715843e+235, 2],
      [:hyperg_1F1_e, [10, 50.1, 2],  1.5001295507968071788, 0],
      [:hyperg_1F1_e, [10, 50.1, 10],  8.713385849265044908, 0],
      [:hyperg_1F1_e, [10, 50.1, 100],  5.909423932273380330e+18, 2],
      [:hyperg_1F1_e, [10, 50.1, 500],  9.740060618457198900e+165, 2],
      [:hyperg_1F1_e, [100, 1.1, 1],  5.183531067116809033e+07, 2],
      [:hyperg_1F1_e, [100, 1.1, 10],  1.6032649110096979462e+28, 2],
      [:hyperg_1F1_e, [100, 1.1, 100],  1.1045151213192280064e+110, 2],
      [:hyperg_1F1_e, [100, 50.1, 1],  7.222953133216603757, 1],
      [:hyperg_1F1_e, [100, 50.1, 10],  1.0998696410887171538e+08, 1],
      [:hyperg_1F1_e, [100, 50.1, 100],  7.235304862322283251e+63, 2],
      [:hyperg_1F1_e, [1, 1.5, -1], 0.5380795069127684191, 0],
      [:hyperg_1F1_e, [1, 1.5, -10],  0.05303758099290164485, 1],
      [:hyperg_1F1_e, [1, 1.5, -100], 0.005025384718759852803, 1],
      [:hyperg_1F1_e, [1, 1.5, -500], 0.0010010030151059555322, 1],
      [:hyperg_1F1_e, [1, 1.1, -500], 0.00020036137599690208265, 1],
      [:hyperg_1F1_e, [10, 1.1, -1],     0.07227645648935938168, 1],
      [:hyperg_1F1_e, [10, 1.1, -10],    0.0003192415409695588126, 1],
      [:hyperg_1F1_e, [10, 1.1, -500],  -3.400379216707701408e-23, 2],
      [:hyperg_1F1_e, [50, 1.1, -100],   4.632883869540640460e-24, TEST_SQRT_TOL0],
      [:hyperg_1F1_e, [50, 1.1, -110.0], 5.642684651305310023e-26, 0.03],
      [:hyperg_1F1_e, [100, 1.1, -1],    0.0811637344096042096, 2],
      [:hyperg_1F1_e, [100, 1.1, -10],   0.00025945610092231574387, 2],
      [:hyperg_1F1_e, [100, 1.1, -50],   2.4284830988994084452e-13, 2],
      [:hyperg_1F1_e, [100, 1.1, -90],   2.4468224638378426461e-22, 2],
      [:hyperg_1F1_e, [100, 1.1, -99],   1.0507096272617608461e-23, 2],
      [:hyperg_1F1_e, [100, 1.1, -100],  1.8315497474210138602e-24, 2],
#     [:hyperg_1F1_e, [100, 1.1, -101], -2.3916306291344452490e-24, 0.04],
#     [:hyperg_1F1_e, [100, 1.1, -110], -4.517581986037732280e-26, 0],
#     [:hyperg_1F1_e, [100, 10.1, -220], -4.296130300021696573e-64, 1],
      [:hyperg_1F1_e, [-10, -10.1, 10.0], 10959.603204633058116, 1],
      [:hyperg_1F1_e, [-10, -10.1, 1000.0], 2.0942691895502242831e+23, 2],
      [:hyperg_1F1_e, [-10, -100.1, 10.0],  2.6012036337980078062, 1],
      [:hyperg_1F1_e, [-1000, -1000.1, 10.0],  22004.341698908631636, 3],
      [:hyperg_1F1_e, [-1000, -1000.1, 200.0], 7.066514294896245043e+86, 3],
      [:hyperg_1F1_e, [-8.1, -10.1, -10.0],    0.00018469685276347199258, 0],
      [:hyperg_1F1_e, [-10, -5.1, 1],  16.936141866089601635, 2],
      [:hyperg_1F1_e, [-10, -5.1, 10],  771534.0349543820541, 2],
      [:hyperg_1F1_e, [-10, -5.1, 100],  2.2733956505084964469e+17, 2],
      [:hyperg_1F1_e, [-100, -50.1, -1],  0.13854540373629275583, 3],
      [:hyperg_1F1_e, [-100, -50.1, -10],  -9.142260314353376284e+19, 3],
      [:hyperg_1F1_e, [-100, -50.1, -100],  -1.7437371339223929259e+87, 3],
      [:hyperg_1F1_e, [-100, -50.1, 1],  7.516831748170351173, 3],
      [:hyperg_1F1_e, [-100, -50.1, 10],  1.0551632286359671976e+11, TEST_SQRT_TOL0],
#     [:hyperg_1F1_e, [-100, -50.1, 50],  -7.564755600940346649e+36, 3],
#     [:hyperg_1F1_e, [-100, -50.1, 100],  4.218776962675977e+55, 3],
      [:hyperg_1F1_e, [-10.5, -8.1, 0.1],  1.1387201443786421724, 0],
      [:hyperg_1F1_e, [-10.5, -11.1, 1],  2.5682766147138452362, 1],
      [:hyperg_1F1_e, [-100.5, -80.1, 10],  355145.4517305220603, 3],
      [:hyperg_1F1_e, [-100.5, -102.1, 10],  18678.558725244365016, 1],
      [:hyperg_1F1_e, [-100.5, -500.1, 10],  7.342209011101454, 0],
      [:hyperg_1F1_e, [-100.5, -500.1, 100],  1.2077443075367177662e+8, 1],
      [:hyperg_1F1_e, [-500.5, -80.1, 2],  774057.8541325341699, 4],
      [:hyperg_1F1_e, [100, -10.1, 1],  -2.1213846338338567395e+12, 0],
      [:hyperg_1F1_e, [100, -10.1, 10],  -6.624849346145112398e+39, 0],
      [:hyperg_1F1_e, [100, -10.1, 100],  -1.2413466759089171904e+129, 0],
#     [:hyperg_1F1_e, [100, -10.1, -1],  34456.29405305551691, 0],
#     [:hyperg_1F1_e, [100, -10.1, -10],  -7.809224251467710833e+07, 0],
#     [:hyperg_1F1_e, [100, -10.1, -100],   -5.214065452753988395e-07, 0],
      [:hyperg_1F1_e, [-100, 1.1, 1],  0.21519810496314438414, 2],
      [:hyperg_1F1_e, [-100, 1.1, 10],  8.196123715597869948, 1],
      [:hyperg_1F1_e, [-100, 1.1, 100],  -1.4612966715976530293e+20, 1],
      [:hyperg_1F1_e, [-100, 20.1, 1],  0.0021267655527278456412, 2],
      [:hyperg_1F1_e, [-100, 20.1, 10],   2.0908665169032186979e-11, 2],
      [:hyperg_1F1_e, [-100, 20.1, 100],  -0.04159447537001340412, 2],
      [:hyperg_1F1_e, [-100, 1.1, -1],  2.1214770215694685282e+07, 3],
      [:hyperg_1F1_e, [-100, 1.1, -10],  1.0258848879387572642e+24, 3],
      [:hyperg_1F1_e, [-100, 1.1, -100],  1.1811367147091759910e+67, 3],
      [:hyperg_1F1_e, [-100, 50.1, -1],  6.965259317271427390, 3],
      [:hyperg_1F1_e, [-100, 50.1, -10],  1.0690052487716998389e+07, 3],
      [:hyperg_1F1_e, [-100, 50.1, -100],  6.889644435777096248e+36, 3],
      [:hyperg_1F1_e, [-2.05, 1.0, 5.05], 3.79393389516785e+00, 3],
      [:hyperg_U_int_e, [1, 1, 0.0001],  8.634088070212725330, 0],
      [:hyperg_U_int_e, [1, 1, 0.01],  4.078511443456425847, 0],
      [:hyperg_U_int_e, [1, 1, 0.5],  0.9229106324837304688, 0],
      [:hyperg_U_int_e, [1, 1, 2.0],  0.3613286168882225847, 0],
      [:hyperg_U_int_e, [1, 1, 100],  0.009901942286733018406, 0],
      [:hyperg_U_int_e, [1, 1, 1000],  0.0009990019940238807150, 0],
      [:hyperg_U_int_e, [1, 8, 0.01],  7.272361203006010000e+16, 0],
      [:hyperg_U_int_e, [1, 8, 1],  1957.0, 0],
      [:hyperg_U_int_e, [1, 8, 5],  1.042496, 1],
      [:hyperg_U_int_e, [1, 8, 8],  0.3207168579101562500, 0],
      [:hyperg_U_int_e, [1, 8, 50],  0.022660399001600000000, 0],
      [:hyperg_U_int_e, [1, 8, 100],  0.010631236727200000000, 0],
      [:hyperg_U_int_e, [1, 8, 1000],  0.0010060301203607207200, 0],
      [:hyperg_U_int_e, [1, 20, 1],  1.7403456103284421000e+16, 0],
      [:hyperg_U_int_e, [1, 20, 20],  0.22597813610531052969, 0],
      [:hyperg_U_int_e, [1, 50, 1],  3.374452117521520758e+61, 0],
      [:hyperg_U_int_e, [1, 50, 50],  0.15394136814987651785, 0],
      [:hyperg_U_int_e, [1, 100, 0.1],  1.0418325171990852858e+253, 2],
      [:hyperg_U_int_e, [1, 100, 1],  2.5624945006073464385e+154, 2],
      [:hyperg_U_int_e, [1, 100, 50],  3.0978624160896431391e+07, 2],
      [:hyperg_U_int_e, [1, 100, 100],  0.11323192555773717475, 0],
      [:hyperg_U_int_e, [1, 100, 200],  0.009715680951406713589, 0],
      [:hyperg_U_int_e, [1, 100, 1000],  0.0011085142546061528661, 0],
      [:hyperg_U_int_e, [1, 1000, 2000],  0.0009970168547036318206, 0],
      [:hyperg_U_int_e, [1, -1, 1],  0.29817368116159703717, 1],
      [:hyperg_U_int_e, [1, -1, 10],  0.07816669698940409380, 1],
      [:hyperg_U_int_e, [1, -10, 1],  0.08271753756946041959, 1],
      [:hyperg_U_int_e, [1, -10, 5],  0.06127757419425055261, 2],
      [:hyperg_U_int_e, [1, -10, 10],  0.04656199948873187212, 2],
      [:hyperg_U_int_e, [1, -10, 20],  0.031606421847946077709, 1],
      [:hyperg_U_int_e, [1, -100, 0.01],  0.009900000099999796950, 2],
      [:hyperg_U_int_e, [1, -100, 1],  0.009802970197050404429, 2],
      [:hyperg_U_int_e, [1, -100, 10],  0.009001648897173103447, 2],
      [:hyperg_U_int_e, [1, -100, 20],  0.008253126487166557546, 2],
      [:hyperg_U_int_e, [1, -100, 50],  0.006607993916432051008, 2],
      [:hyperg_U_int_e, [1, -100, 90],  0.005222713769726871937, 2],
      [:hyperg_U_int_e, [1, -100, 110],  0.004727658137692606210, 2],
      [:hyperg_U_int_e, [1, -1000, 1010],  0.0004971408839859245170, 4],
      [:hyperg_U_int_e, [8, 1, 0.001],  0.0007505359326875706975, 0],
      [:hyperg_U_int_e, [8, 1, 0.5],  6.449509938973479986e-06, 3],
      [:hyperg_U_int_e, [8, 1, 8],  6.190694573035761284e-10, 0],
      [:hyperg_U_int_e, [8, 1, 20],  3.647213845460374016e-12, 0],
      [:hyperg_U_int_e, [8, 8, 1],  0.12289755012652317578, 1],
      [:hyperg_U_int_e, [8, 8, 10],  5.687710359507564272e-09, 1],
      [:hyperg_U_int_e, [8, 8, 20],  2.8175404594901039724e-11, 1],
      [:hyperg_U_int_e, [100, 100, 0.01],  1.0099979491941914867e+196, 2],
      [:hyperg_U_int_e, [100, 100, 0.1],  1.0090713562719862833e+97, 2],
      [:hyperg_U_int_e, [100, 100, 1],  0.009998990209084729106, 2],
      [:hyperg_U_int_e, [100, 100, 20],  1.3239363905866130603e-131, 2],
      [:hyperg_U_int_e, [-10, 1, 0.01],  3.274012540759009536e+06, 0],
      [:hyperg_U_int_e, [-10, 1, 1],  1.5202710000000000000e+06, 0],
      [:hyperg_U_int_e, [-10, 1, 10],  1.0154880000000000000e+08, 0],
      [:hyperg_U_int_e, [-10, 1, 100],  3.284529863685482880e+19, 0],
      [:hyperg_U_int_e, [-10, 10, 1],  1.1043089864100000000e+11, 0],
      [:hyperg_U_int_e, [-10, 100, 1],  1.3991152402448957897e+20, 0],
      [:hyperg_U_int_e, [-10, 100, 10],  5.364469916567136000e+19, 0],
      [:hyperg_U_int_e, [-10, 100, 100],  3.909797568000000000e+12, 0],
      [:hyperg_U_int_e, [-10, 100, 500],  8.082625576697984130e+25, 0],
      [:hyperg_U_int_e, [-50, 1, 0.01],  1.6973422555823855798e+64, 2],
      [:hyperg_U_int_e, [-50, 1, 1],  7.086160198304780325e+63, 1],
      [:hyperg_U_int_e, [-50, 1, 10],  5.332862895528712200e+65, 1],
      [:hyperg_U_int_e, [-50, 10, 1],  -7.106713471565790573e+71, 1],
      [:hyperg_U_int_e, [-50, 100, 1],  2.4661377199407186476e+104, 1],
      [:hyperg_U_int_e, [-50, 10, 10],  5.687538583671241287e+68, 1],
      [:hyperg_U_int_e, [-50, 100, 10],  1.7880761664553373445e+102, 1],
      [:hyperg_U_int_e, [-90, 1, 0.01],  4.185245354032917715e+137, 2],
      [:hyperg_U_int_e, [-90, 1, 0.1],  2.4234043408007841358e+137, 3],
      [:hyperg_U_int_e, [-90, 1, 10],  -1.8987677149221888807e+139, 1],
      [:hyperg_U_int_e, [-90, 10, 10],  -5.682999988842066677e+143, 1],
      [:hyperg_U_int_e, [-90, 100, 10],  2.3410029853990624280e+189, 2],
      [:hyperg_U_int_e, [-90, 1000, 10],  1.9799451517572225316e+271, 3],
      [:hyperg_U_int_e, [-50, -1, 10],  -9.083195466262584149e+64, 1],
      [:hyperg_U_int_e, [-50, -10, 10],  -1.4418257327071634407e+62, 1],
      [:hyperg_U_int_e, [-50, -100, 0.01],  3.0838993811468983931e+93, 2],
      [:hyperg_U_int_e, [-50, -100, 10],  4.014552630378340665e+95, 2],
      [:hyperg_U_int_e, [-100, -100, 10],  2.0556466922347982030e+162, 2],
      [:hyperg_U_int_e, [-100, -200, 10],  1.1778399522973555582e+219, 2],
      [:hyperg_U_int_e, [-100, -200, 100],  9.861313408898201873e+235, 3],
      [:hyperg_U_e, [0.0001, 0.0001, 0.0001], 1.0000576350699863577, 1],
      [:hyperg_U_e, [0.0001, 0.0001, 1.0], 0.9999403679233247536, 0],
      [:hyperg_U_e, [0.0001, 0.0001, 100.0], 0.9995385992657260887, 0],
      [:hyperg_U_e, [0.0001, 1, 0.0001], 1.0009210608660065989, 2],
      [:hyperg_U_e, [0.0001, 1.0, 1.0], 0.9999999925484179084, 2],
      [:hyperg_U_e, [0.0001, 10, 1], 13.567851006281412726, 3],
      [:hyperg_U_e, [0.0001, 10, 10], 0.9999244381454633265, 0],
      [:hyperg_U_e, [0.0001, 100, 98], 0.9998517867411840044, 2],
      [:hyperg_U_e, [0.0001, 1000, 999], 0.9997195294193261604, 2],
      [:hyperg_U_e, [0.0001, 1000, 1100],  0.9995342990014584713, 1],
      [:hyperg_U_e, [0.5, 1000, 800], 9.103916020464797207e+08, 2],
      [:hyperg_U_e, [0.5, 1000, 998], 0.21970269691801966806, 2],
      [:hyperg_U_e, [0.5, 0.5, 1.0], 0.7578721561413121060, 2],
      [:hyperg_U_e, [1, 0.0001, 0.0001], 0.9992361337764090785, 1],
      [:hyperg_U_e, [1, 0.0001, 1], 0.4036664068111504538, 2],
      [:hyperg_U_e, [1, 0.0001, 100], 0.009805780851264329587, 1],
      [:hyperg_U_e, [1, 1.2, 2.0], 0.3835044780075602550, 1],
      [:hyperg_U_e, [1, -0.0001, 1], 0.4036388693605999482, 1],
      [:hyperg_U_e, [8, 10.5, 1],  27.981926466707438538, 1],
      [:hyperg_U_e, [8, 10.5, 10],  2.4370135607662056809e-8, 0],
      [:hyperg_U_e, [8, 10.5, 100],  1.1226567526311488330e-16, 2],
      [:hyperg_U_e, [10, -2.5, 10],  6.734690720346560349e-14, 1],
      [:hyperg_U_e, [10, 2.5, 10],  6.787780794037971638e-13, 0],
      [:hyperg_U_e, [10, 2.5, 50],  2.4098720076596087125e-18, 0],
      [:hyperg_U_e, [-10.5, 1.1, 1],  -3.990841457734147e+6, 2],
      [:hyperg_U_e, [-10.5, 1.1, 10],  1.307472052129343e+8, 1],
      [:hyperg_U_e, [-10.5, 1.1, 50],  3.661978424114088e+16, 0],
      [:hyperg_U_e, [-10.5, 1.1, 90],  8.09469542130868e+19, 1],
      [:hyperg_U_e, [-10.5, 1.1, 99],  2.546328328942063e+20, 1],
      [:hyperg_U_e, [-10.5, 1.1, 100],  2.870463201832814e+20, 1],
      [:hyperg_U_e, [-10.5, 1.1, 200],  8.05143453769373e+23, 2],
      [:hyperg_U_e, [-10.5, 10.1, 0.1],  -3.043016255306515e+20, 2],
      [:hyperg_U_e, [-10.5, 10.1, 1],  -3.194745265896115e+12, 1],
      [:hyperg_U_e, [-10.5, 10.1, 4],  -6.764203430361954e+07, 2],
      [:hyperg_U_e, [-10.5, 10.1, 10],  -2.067399425480545e+09, 1],
      [:hyperg_U_e, [-10.5, 10.1, 50],  4.661837330822824e+14, 1],
      [:hyperg_U_e, [-10.5, 100.4, 10],  -6.805460513724838e+66, 1],
      [:hyperg_U_e, [-10.5, 100.4, 50],  -2.081052558162805e+18, 1],
      [:hyperg_U_e, [-10.5, 100.4, 80],  2.034113191014443e+14, 2],
      [:hyperg_U_e, [-10.5, 100.4, 100],  6.85047268436107e+13, 1],
      [:hyperg_U_e, [-10.5, 100.4, 200],  1.430815706105649e+20, 2],
      [:hyperg_U_e, [-19.5, 82.1, 10],  5.464313196201917432e+60, 2],
      [:hyperg_U_e, [-50.5, 100.1, 10],  -5.5740216266953e+126, 1],
      [:hyperg_U_e, [-50.5, 100.1, 40],  5.937463786613894e+91, 1],
      [:hyperg_U_e, [-50.5, 100.1, 50],  -1.631898534447233e+89, 1],
      [:hyperg_U_e, [-50.5, 100.1, 70],  3.249026971618851e+84, 2],
      [:hyperg_U_e, [-50.5, 100.1, 100],  1.003401902126641e+85, 1],
      [:hyperg_2F1_e, [1, 1, 1, 0.5], 2.0, 0],
      [:hyperg_2F1_e, [8, 8, 1, 0.5], 12451584.0, 0],
      [:hyperg_2F1_e, [8, -8, 1, 0.5], 0.13671875, 0],
#     [:hyperg_2F1_e, [8, -8.1, 1, 0.5], 0.14147385378899930422, 4],
      [:hyperg_2F1_e, [8, -8, 1, -0.5], 4945.136718750000000, 0],
      [:hyperg_2F1_e, [8, -8, -5.5, 0.5],  -906.6363636363636364, 0],
      [:hyperg_2F1_e, [8, -8, -5.5, -0.5], 24565.363636363636364, 0],
      [:hyperg_2F1_e, [8, 8, 1, -0.5], -0.006476312098196747669, 2],
      [:hyperg_2F1_e, [8, 8, 5, 0.5], 4205.714285714285714, 0],
      [:hyperg_2F1_e, [8, 8, 5, -0.5], 0.0028489656290296436616, 2],
      [:hyperg_2F1_e, [9, 9, 1, 0.99], 1.2363536673577259280e+38, 2],
      [:hyperg_2F1_e, [9, 9, -1.5, 0.99], 3.796186436458346579e+46, 2],
      [:hyperg_2F1_e, [9, 9, -1.5, -0.99], 0.14733409946001025146, 1],
      [:hyperg_2F1_e, [9, 9, -8.5, 0.99], -1.1301780432998743440e+65, 2],
      [:hyperg_2F1_e, [9, 9, -8.5, -0.99], -8.856462606575344483, 1],
      [:hyperg_2F1_e, [9, 9, -21.5, 0.99], 2.0712920991876073253e+95, 3],
      [:hyperg_2F1_e, [9, 9, -21.5, -0.99], -74.30517015382249216, 2],
      [:hyperg_2F1_e, [9, 9, -100.5, 0.99],  -3.186778061428268980e+262, 3],
      [:hyperg_2F1_e, [9, 9, -100.5, -0.99],  2.4454358338375677520, 1],
      [:hyperg_2F1_e, [25, 25, 1, -0.5], -2.9995530823639545027e-06, TEST_SQRT_TOL0],
      [:hyperg_2F1_e, [1.5, 0.5, 2.0, 1.0-1.0 / 64.0], 3.17175539044729373926, 3],
      [:hyperg_2F1_e, [1.5, 0.5, 2.0, 1.0-1.0 / 128.0], 3.59937243502024563424, 2],
      [:hyperg_2F1_e, [1.5, 0.5, 2.0, 1.0-1.0 / 256.0], 4.03259299524392504369, 1],
      [:hyperg_2F1_e, [1.5, 0.5, 2.0, 1.0-1.0 / 1024.0], 4.90784159359675398250, 1],
      [:hyperg_2F1_e, [1.5, 0.5, 2.0, 1.0-1.0 / 65536.0], 7.552266033399683914, 1],
      [:hyperg_2F1_e, [1.5, 0.5, 2.0, 1.0-1.0 / 16777216.0], 11.08235454026043830363, 1],
      [:hyperg_2F1_e, [1.5, 0.5, 2.0, -1.0 + 1.0 / 1024.0], 0.762910940909954974527, 0],
      [:hyperg_2F1_e, [1.5, 0.5, 2.0, -1.0 + 1.0 / 65536.0], 0.762762124908845424449, 0],
      [:hyperg_2F1_e, [1.5, 0.5, 2.0, -1.0 + 1.0 / 1048576.0], 0.762759911089064738044, 0],
      [:hyperg_2F1_conj_e, [1, 1, 1, 0.5], 3.352857095662929028, 0],
      [:hyperg_2F1_conj_e, [8, 8, 1, 0.5], 1.7078067538891293983e+09, 0],
      [:hyperg_2F1_conj_e, [8, 8, 5, 0.5], 285767.15696901140627, 1],
      [:hyperg_2F1_conj_e, [8, 8, 1, -0.5], 0.007248196261471276276, 3],
      [:hyperg_2F1_conj_e, [8, 8, 5, -0.5], 0.00023301916814505902809, 3],
      [:hyperg_2F1_conj_e, [25, 25, 1, -0.5], 5.1696944096e-06, TEST_SQRT_TOL0],
#     [:hyperg_2F0_e, [0.01, 1.0, -0.02], 0.999803886708565, 0],
#     [:hyperg_2F0_e, [0.1,  0.5, -0.02], 0.999015947934831, 0],
#     [:hyperg_2F0_e, [1,   1, -0.02], 0.980755496569062, 0],
#     [:hyperg_2F0_e, [8,   8, -0.02], 0.3299059284994299, 0],
#     [:hyperg_2F0_e, [50, 50, -0.02], 2.688995263773233e-13, 0],
      [:hyperg_2F1_renorm_e, [1, 1, 1, 0.5], 2.0, 0],
      [:hyperg_2F1_renorm_e, [8, 8, 1, 0.5], 12451584.0, 0],
      [:hyperg_2F1_renorm_e, [8, -8, 1, 0.5], 0.13671875, 0],
      [:hyperg_2F1_renorm_e, [8, -8, 1, -0.5], 4945.13671875, 0],
      [:hyperg_2F1_renorm_e, [8, -8, -5.5, 0.5], -83081.19167659493609, 2],
      [:hyperg_2F1_renorm_e, [8, -8, -5.5, -0.5], 2.2510895952730178518e+06, 2],
      [:hyperg_2F1_renorm_e, [8, 8, 5, 0.5], 175.2380952380952381, 1],
      [:hyperg_2F1_renorm_e, [9, 9, -1.5, 0.99], 1.6063266334913066551e+46, 2],
      [:hyperg_2F1_renorm_e, [9, 9, -1.5, -0.99], 0.06234327316254516616, 2],
      [:hyperg_2F1_renorm_e, [5, 5, -1, 0.5], 4949760.0, 1],
      [:hyperg_2F1_renorm_e, [5, 5, -10, 0.5], 139408493229637632000.0, 2],
      [:hyperg_2F1_renorm_e, [5, 5, -100, 0.5], 3.0200107544594411315e+206, 3],
      [:hyperg_2F1_conj_renorm_e, [9, 9, -1.5, 0.99], 5.912269095984229412e+49, 2],
      [:hyperg_2F1_conj_renorm_e, [9, 9, -1.5, -0.99], 0.10834020229476124874, 2],
      [:hyperg_2F1_conj_renorm_e, [5, 5, -1, 0.5], 1.4885106335357933625e+08, 2],
      [:hyperg_2F1_conj_renorm_e, [5, 5, -10, 0.5], 7.968479361426355095e+21, 2],
      [:hyperg_2F1_conj_renorm_e, [5, 5, -100, 0.5], 3.1113180227052313057e+208, 3]
    ],
    :legendre => [
      [:legendre_P1_e, [-0.5], -0.5, 0],
      [:legendre_P1_e, [ 0.5], 0.5, 0],
      [:legendre_P2_e, [0.0], -0.5, 0],
      [:legendre_P2_e, [0.5], -0.125, 0],
      [:legendre_P2_e, [1.0], 1.0, 0],
      [:legendre_P2_e, [100.0], 14999.5, 0],
      [:legendre_P3_e, [ -0.5], 0.4375, 0],
      [:legendre_P3_e, [  0.5], -0.4375, 0],
      [:legendre_P3_e, [  1.0], 1.0, 0],
      [:legendre_P3_e, [100.0], 2.49985e+06, 0],
      [:legendre_Pl_e, [1, -0.5], -0.5, 0],
      [:legendre_Pl_e, [1,  1.0e-8], 1.0e-08, 0],
      [:legendre_Pl_e, [1,  0.5], 0.5, 0],
      [:legendre_Pl_e, [1,  1.0], 1.0, 0],
      [:legendre_Pl_e, [10, -0.5], -0.1882286071777345, 0],
      [:legendre_Pl_e, [10,  1.0e-8], -0.24609374999999864648, 0],
      [:legendre_Pl_e, [10,  0.5], -0.18822860717773437500, 0],
      [:legendre_Pl_e, [10,  1.0], 1.0, 0],
      [:legendre_Pl_e, [99, -0.5], 0.08300778172138770477, 0],
      [:legendre_Pl_e, [99,  1.0e-8], -7.958923738716563193e-08, 0],
      [:legendre_Pl_e, [99,  0.5], -0.08300778172138770477, 0],
      [:legendre_Pl_e, [99,  0.999], -0.3317727359254778874, 2],
      [:legendre_Pl_e, [99,  1.0], 1.0, 0],
      [:legendre_Pl_e, [1000, -0.5],   -0.019168251091650277878, 2],
      [:legendre_Pl_e, [1000,  1.0e-8], 0.02522501817709828,     2],
      [:legendre_Pl_e, [1000,  0.5],   -0.019168251091650277878, 2],
      [:legendre_Pl_e, [1000,  1.0],    1.0,                     0],
      [:legendre_Pl_e, [4000, -0.5], -0.009585404456573080972, 2],
      [:legendre_Pl_e, [4000,  0.5], -0.009585404456573080972, 2],
      [:legendre_Pl_e, [4000,  1.0], 1.0, 0],
      [:legendre_Plm_e, [10, 0, -0.5], -0.18822860717773437500, 0],
      [:legendre_Plm_e, [10, 0, 1.0e-08], -0.24609374999999864648, 0],
      [:legendre_Plm_e, [10, 0, 0.5], -0.18822860717773437500, 0],
      [:legendre_Plm_e, [10, 1, -0.5], -2.0066877394361256516, 0],
      [:legendre_Plm_e, [10, 1, 1.0e-08], -2.7070312499999951725e-07, 0],
      [:legendre_Plm_e, [10, 1, 0.5], 2.0066877394361256516, 0],
      [:legendre_Plm_e, [10, 5, -0.5],    -30086.169706116174977,    0],
      [:legendre_Plm_e, [10, 5, 1.0e-08], -0.0025337812499999964949, 0],
      [:legendre_Plm_e, [10, 5, 0.5],      30086.169706116174977,    0],
      [:legendre_Plm_e, [10, 5, 0.999],   -0.5036411489013270406,    1],
      [:legendre_Plm_e, [100, 5, -0.5], -6.617107444248382171e+08, 0],
      [:legendre_Plm_e, [100, 5, 1.0e-08], 817.8987598063712851, 0],
      [:legendre_Plm_e, [100, 5, 0.5], 6.617107444248382171e+08, 0],
      [:legendre_Plm_e, [100, 5, 0.999], -1.9831610803806212189e+09, 2],
      [:legendre_sphPlm_e, [10, 0, -0.5], -0.24332702369300133776, 0],
      [:legendre_sphPlm_e, [10, 0, 0.5], -0.24332702369300133776, 0],
      [:legendre_sphPlm_e, [10, 0, 0.999], 1.2225754122797385990, 1],
      [:legendre_sphPlm_e, [10, 5, -0.5],    -0.3725739049803293972,     0],
      [:legendre_sphPlm_e, [10, 5, 1.0e-08], -3.1377233589376792243e-08, 0],
      [:legendre_sphPlm_e, [10, 5, 0.5],      0.3725739049803293972,     0],
      [:legendre_sphPlm_e, [10, 5, 0.999],   -6.236870674727370094e-06,  2],
      [:legendre_sphPlm_e, [10, 10, -0.5], 0.12876871185785724117, 1],
      [:legendre_sphPlm_e, [10, 10, 0.5], 0.12876871185785724117,  1],
      [:legendre_sphPlm_e, [10, 10, 0.999], 1.7320802307583118647e-14, 2],
      [:legendre_sphPlm_e, [200, 1, -0.5],   0.3302975570099492931, 1],
      [:legendre_sphPlm_e, [200, 1, 0.5],   -0.3302975570099492931, 1],
      [:legendre_sphPlm_e, [200, 1, 0.999], -1.4069792055546256912, 2],
      [:conicalP_half_e, [0.0, -0.5],   0.8573827581049917129, 0],
      [:conicalP_half_e, [0.0,  0.5],   0.8573827581049917129, 0],
      [:conicalP_half_e, [0.0,  2.0],   0.6062611623284649811, 0],
      [:conicalP_half_e, [0.0,  100.0], 0.07979045091636735635, 0],
      [:conicalP_half_e, [10.0, -0.5],    5.345484922591867188e+08, 1],
      [:conicalP_half_e, [10.0,  0.5],    15137.910380385258370, 1],
      [:conicalP_half_e, [10.0,  2.0],    0.4992680691891618544, 1],
      [:conicalP_half_e, [10.0,  100.0], -0.07272008163718195685, 2],
      [:conicalP_half_e, [200.0, -1.0e-3],  1.3347639529084185010e+136, 2],
      [:conicalP_half_e, [200.0,  1.0e-8],  1.0928098010940058507e+136, 2],
      [:conicalP_half_e, [200.0,  0.5],     3.895546021611205442e+90,   2],
      [:conicalP_half_e, [200.0,  10.0],   -0.04308567180833581268,     3],
      [:conicalP_half_e, [200.0,  100.0],  -0.04694669186576399194,     3],
      [:conicalP_half_e, [200.0,  1000.0],  0.023698140704121273277,    3],
      [:conicalP_half_e, [200.0,  1.0e+8], -0.00006790983312124277891,  3],
      [:conicalP_half_e, [1.0e+8,  1.1],   1.1599311133054742944,  TEST_SQRT_TOL0],
      [:conicalP_half_e, [1.0e+8,  100.0], 0.07971967557381557875, TEST_SQRT_TOL0],
      [:conicalP_mhalf_e, [0.0, -0.5],  1.7956982494514644808, 0],
      [:conicalP_mhalf_e, [0.0,  0.5],  0.8978491247257322404, 0],
      [:conicalP_mhalf_e, [0.0,  2.0],  0.7984204253272901551, 0],
      [:conicalP_mhalf_e, [0.0,  100.0],  0.4227531369388072584, 0],
      [:conicalP_mhalf_e, [10.0, -0.5],  5.345484922591867181e+07, 1],
      [:conicalP_mhalf_e, [10.0,  0.5],  1513.7910356104985334, 1],
      [:conicalP_mhalf_e, [10.0,  2.0],  0.03439243987215615642, 1],
      [:conicalP_mhalf_e, [10.0,  100.0],  0.003283756665952609624, 2],
      [:conicalP_mhalf_e, [200.0, -0.5],  1.7699538115312304280e+179, 2],
      [:conicalP_mhalf_e, [200.0,  1.0e-8],  5.464049005470029253e+133, 2],
      [:conicalP_mhalf_e, [200.0,  0.5],  1.9477730108056027211e+88, 2],
      [:conicalP_mhalf_e, [200.0,  10.0],  0.0012462575917716355362, 2],
      [:conicalP_mhalf_e, [200.0,  100.0],  -0.0003225881344802625149, 2],
      [:conicalP_mhalf_e, [200.0,  1000.0], -0.00004330652890886567623, 3],
      [:conicalP_mhalf_e, [200.0,  1.0e+8],  2.0943091278037078483e-07, 3],
      [:conicalP_mhalf_e, [1.0e+8,  1.1], 2.092320445620989618e-09, 16.0 * TEST_SQRT_TOL0],
      [:conicalP_mhalf_e, [1.0e+8,  100.0],  -3.359967833599016923e-11, 256.0 * TEST_SQRT_TOL0],
      [:conicalP_0_e, [0.0, -0.5],  1.3728805006183501647, 0],
      [:conicalP_0_e, [0.0,  0.5],  1.0731820071493643751, 0],
      [:conicalP_0_e, [0.0,  2.0],  0.9012862993604472987, 0],
      [:conicalP_0_e, [0.0,  100.0],  0.30091748588199264556, 0],
      [:conicalP_0_e, [10.0, -0.5],  1.6795592815421804669e+08, 1],
      [:conicalP_0_e, [10.0,  0.5],  4826.034132009618240,      1],
      [:conicalP_0_e, [10.0,  2.0],  0.18798468917758716146,    2],
      [:conicalP_0_e, [10.0,  100.0], -0.008622130749987962529, 2],
      [:conicalP_0_e, [200.0,  -0.5], 2.502194818646823e+180, 4],
      [:conicalP_0_e, [1000.0,  100.0],   0.0017908817653497715844, 3],
      [:conicalP_0_e, [1000.0,  1000.0], -0.0006566893804926284301, 3],
      [:conicalP_0_e, [1000.0,  1.0e+8],  2.3167213561756390068e-06, 4],
      [:conicalP_1_e, [0.0, -0.5],    0.4939371126656998499,  1],
      [:conicalP_1_e, [0.0,  0.5],    0.14933621085538265636, 1],
      [:conicalP_1_e, [0.0,  2.0],   -0.13666874968871549533, 1],
      [:conicalP_1_e, [0.0,  100.0], -0.10544528203156629098, 2],
      [:conicalP_1_e, [10.0, -0.5],    1.7253802958788312520e+09, 2],
      [:conicalP_1_e, [10.0,  0.5],    46781.02294059967988,      1],
      [:conicalP_1_e, [10.0,  2.0],    0.26613342643657444400,    2],
      [:conicalP_1_e, [10.0,  100.0], -0.23281959695501029796,    2],
      [:conicalP_1_e, [200.0, -0.999], 2.71635193199341135e+270, 2],
      [:conicalP_1_e, [200.0, -0.9],   4.2952493176812905e+234,  2],
      [:conicalP_1_e, [200.0, -0.5],   5.01159205956053439e+182, 3],
      [:conicalP_1_e, [200.0,  0.999], 195733.0396081538,        2],
      [:conicalP_1_e, [200.0,  10.0], -2.9272610662414349553,    2],
      [:conicalP_1_e, [1000.0, 100.0],  -1.7783258105862399857,    6],
      [:conicalP_1_e, [1000.0, 1000.0],  0.4535161075156427179,    4],
      [:conicalP_1_e, [1000.0, 1.0e+8],  0.0009983414549874888478, TEST_SQRT_TOL0],
      [:conicalP_sph_reg_e, [2,  1.0, -0.5],  1.6406279287008789526,      0],
      [:conicalP_sph_reg_e, [10, 1.0, -0.5],  0.000029315266725049129448, 1],
      [:conicalP_sph_reg_e, [20, 1.0, -0.5],  7.335769429462034431e-15,   1],
      [:conicalP_sph_reg_e, [30, 1.0, -0.5],  1.3235612394267378871e-26,  2],
      [:conicalP_sph_reg_e, [10, 1.0, 0.5],  2.7016087199857873954e-10, 1],
      [:conicalP_sph_reg_e, [20, 1.0, 0.5],  1.1782569701435933399e-24, 1],
      [:conicalP_sph_reg_e, [30, 1.0, 0.5],  3.636240588303797919e-41,  1],
      [:conicalP_sph_reg_e, [10, 1.0, 2.0],  2.4934929626284934483e-10, 1],
      [:conicalP_sph_reg_e, [20, 1.0, 2.0],  1.1284762488012616191e-24, 2],
      [:conicalP_sph_reg_e, [30, 100.0, 100.0],  -1.6757772087159526048e-64, 6],
      [:conicalP_cyl_reg_e, [2, 1.0, -0.5],   2.2048510472375258708,       0],
      [:conicalP_cyl_reg_e, [10, 1.0, -0.5],  0.00007335034531618655690,   1],
      [:conicalP_cyl_reg_e, [20, 1.0, -0.5],  2.5419860619212164696e-14,   1],
      [:conicalP_cyl_reg_e, [30, 1.0, -0.5],  5.579714972260536827e-26,    2],
      [:conicalP_cyl_reg_e, [10, 1.0, 0.5],  1.1674078819646475282e-09,    0],
      [:conicalP_cyl_reg_e, [20, 1.0, 0.5],  7.066408031229072207e-24,     1],
      [:conicalP_cyl_reg_e, [30, 1.0, 0.5],  2.6541973286862588488e-40,    1],
      [:conicalP_cyl_reg_e, [10, 1.0, 2.0],  1.0736109751890863051e-09,    2],
      [:conicalP_cyl_reg_e, [20, 1.0, 2.0],  6.760965304863386741e-24,     2],
      [:conicalP_cyl_reg_e, [30, 100.0, 100.0], -4.268753482520651007e-63, 4],
      [:legendre_H3d_0_e, [1.0e-06, 1.0e-06], 0.9999999999998333333, 0],
      [:legendre_H3d_0_e, [1.0, 0.0], 1.0, 0],
      [:legendre_H3d_0_e, [1.0, 1.0], 0.7160229153604338713, 0],
      [:legendre_H3d_0_e, [1.0, 100.0], -3.767437313149604566e-44, 2],
      [:legendre_H3d_0_e, [1.0, 500.0], -6.665351935878582205e-218, 2],
      [:legendre_H3d_0_e, [100.0, 1.0], -0.004308757035378200029, 0],
      [:legendre_H3d_0_e, [100.0, 10.0], 7.508054627912986427e-07, 0],
      [:legendre_H3d_0_e, [1000.0, 1.0], 0.0007036067909088818319, 0],
      [:legendre_H3d_0_e, [1.0e+08, 1.0], 7.927485371429105968e-09, 3],
      [:legendre_H3d_0_e, [1.0e+08, 100.0], -3.627118904186918957e-52, 32.0 * TEST_SQRT_TOL0],
      [:legendre_H3d_1_e, [1.0e-06, 1.0e-06], 3.333333333334222222e-07, 0],
      [:legendre_H3d_1_e, [1.0, 1.0e-10], 4.714045207910316829e-11,     0],
      [:legendre_H3d_1_e, [1.0, 1.0], 0.3397013994799344639,           0],
      [:legendre_H3d_1_e, [1.0, 100.0], -7.200624449531811272e-44,     2],
      [:legendre_H3d_1_e, [1.0, 500.0], 4.192260336821728677e-218,     2],
      [:legendre_H3d_1_e, [100.0, 0.01], 0.30117664944267412324, 1],
      [:legendre_H3d_1_e, [100.0, 1.0], -0.007393833425336299309, 0],
      [:legendre_H3d_1_e, [100.0, 10.0], -5.031062029821254982e-07, 0],
      [:legendre_H3d_1_e, [1000.0, 0.001], 0.30116875865090396421, 0],
      [:legendre_H3d_1_e, [1000.0, 1.0], -0.0004776144516074971885, 0],
      [:legendre_H3d_1_e, [1.0e+08, 1.0e-08], 0.30116867893975679722, 1],
      [:legendre_H3d_1_e, [1.0e+08, 1.0], 3.0921097047369081582e-09, 4],
      [:legendre_H3d_1_e, [1.0e+08, 100.0], -6.496142701296286936e-52, 32.0 * TEST_SQRT_TOL0],
      [:legendre_H3d_e, [5, 1.0e-06, 1.0e-06],  1.1544011544013627977e-32, 2],
      [:legendre_H3d_e, [5, 1.0, 1.0e-10],      2.0224912016958766992e-52, 2],
      [:legendre_H3d_e, [5, 1.0, 1.0],          0.011498635037491577728,   1],
      [:legendre_H3d_e, [5, 1.0, 5.0],          0.0020696945662545205776,  4],
      [:legendre_H3d_e, [5, 1.0, 7.0],     -0.0017555303787488993676,   4],
      [:legendre_H3d_e, [5, 1.0, 10.0],     0.00008999979724504887101,  2],
      [:legendre_H3d_e, [5, 1.0, 100.0],   -4.185397793298567945e-44,   2],
      [:legendre_H3d_e, [5, 1.0, 500.0],    1.4235113901091961263e-217, 3],
      [:legendre_H3d_e, [5, 100.0, 0.001],  9.642762597222417946e-10,   2],
      [:legendre_H3d_e, [5, 100.0, 0.002],  3.0821201254308036109e-08,  2],
      [:legendre_H3d_e, [5, 100.0, 0.01],   0.00009281069019005840532,  1],
      [:legendre_H3d_e, [5, 100.0, 1.0],   -0.008043100696178624653,    2],
      [:legendre_H3d_e, [5, 100.0, 10.0],  -3.927678432813974207e-07,   3],
      [:legendre_H3d_e, [5, 1000.0, 0.001],  0.00009256365284253254503, 1],
      [:legendre_H3d_e, [5, 1000.0, 0.01],  -0.05553733815473079983, 0],
      [:legendre_H3d_e, [5, 1.0e+08, 1.0e-08],   0.00009256115861125841299, 2],
      [:legendre_H3d_e, [5, 1.0e+08, 100.0],    -6.496143209092860765e-52, 128.0 * TEST_SQRT_TOL0],
      [:legendre_Q0_e, [-0.9999847412109375], -5.8917472200477175158028143531855, 4],
      [:legendre_Q0_e, [-0.5], -0.5493061443340548457, 0],
      [:legendre_Q0_e, [-1e-10], -1.000000000000000000e-10, 0],
      [:legendre_Q0_e, [0.0], 0.0, 0],
      [:legendre_Q0_e, [1e-10], 1.000000000000000000e-10, 0],
      [:legendre_Q0_e, [0.9999847412109375], 5.8917472200477175158028143531855, 4],
      [:legendre_Q0_e, [ 1.0000152587890625], 5.8917548494422489138325509750429, 4],
      [:legendre_Q0_e, [ 1.5], 0.8047189562170501873, 0],
      [:legendre_Q0_e, [ 9.99], 0.1004364599660005447, 0],
      [:legendre_Q0_e, [ 10.0], 0.1003353477310755806, 0],
      [:legendre_Q0_e, [ 10.01], 0.1002344395571710243, 0],
      [:legendre_Q0_e, [ 100], 0.010000333353334762015, 0],
      [:legendre_Q0_e, [ 1e10], 1.000000000000000000e-10, 0],
      [:legendre_Q1_e, [-0.9999847412109375], 4.8916573191196772369, 4],
      [:legendre_Q1_e, [-0.5], -0.7253469278329725772, 0],
      [:legendre_Q1_e, [-0.01], -0.9998999966664666524, 0],
      [:legendre_Q1_e, [-1e-10], -0.999999999999999999, 0],
      [:legendre_Q1_e, [0.0], -1.0, 0],
      [:legendre_Q1_e, [1e-10], -0.999999999999999999, 0],
      [:legendre_Q1_e, [0.0001], -0.9999999899999999667, 0],
      [:legendre_Q1_e, [0.01], -0.9998999966664666524, 0],
      [:legendre_Q1_e, [0.5], -0.7253469278329725772, 0],
      [:legendre_Q1_e, [0.9999847412109375], 4.8916573191196772369, 4],
      [:legendre_Q1_e, [1.0000152587890625], 4.8918447504867045145, 4],
      [:legendre_Q1_e, [ 1.5], 0.20707843432557528095, 0],
      [:legendre_Q1_e, [ 9.99], 3.360235060345441639e-3, 0],
      [:legendre_Q1_e, [ 10.0], 3.353477310755806357e-3, 0],
      [:legendre_Q1_e, [ 10.01], 3.346739967281953346e-3, 0],
      [:legendre_Q1_e, [ 100.0], 3.333533347620158821e-5, 0],
      [:legendre_Q1_e, [ 1e10], 3.333333333333333333e-21, 0],
      [:legendre_Ql_e, [10, -0.5], -0.29165813966586752393,    0],
      [:legendre_Ql_e, [10,  0.5], 0.29165813966586752393,     0],
      [:legendre_Ql_e, [10,  1.5], 0.000014714232718207477406, 0],
      [:legendre_Ql_e, [100, -0.5], -0.09492507395207282096,   1],
      [:legendre_Ql_e, [100,  0.5], 0.09492507395207282096,    1],
      [:legendre_Ql_e, [100,  1.5], 1.1628163435044121988e-43, 2],
      [:legendre_Ql_e, [1000, -0.5], -0.030105074974005303500, 1],
      [:legendre_Ql_e, [1000,  0.5], 0.030105074974005303500,  1],
      [:legendre_Ql_e, [1000,  1.1], 1.0757258447825356443e-194, 3]
    ],
    :mathieu => [
      [:mathieu_ce_e, [0, 0.0, 0.0], 0.7071067811865475, TEST_SNGL],
      [:mathieu_ce_e, [0, 0.0, GSL::M_PI_2], 0.7071067811865475, TEST_SNGL],
      [:mathieu_ce_e, [0, 5.0, 0.0], 0.04480018165188902, TEST_SNGL],
      [:mathieu_ce_e, [0, 5.0, GSL::M_PI_2], 1.334848674698019, TEST_SNGL],
      [:mathieu_ce_e, [0, 10.0, 0.0], 0.007626517570935782, TEST_SNGL],
      [:mathieu_ce_e, [0, 10.0, GSL::M_PI_2], 1.468660470712856, TEST_SNGL],
      [:mathieu_ce_e, [0, 15.0, 0.0], 0.001932508315204592, TEST_SNGL],
      [:mathieu_ce_e, [0, 15.0, GSL::M_PI_2], 1.550108146686649, TEST_SNGL],
      [:mathieu_ce_e, [0, 20.0, 0.0], 0.0006037438292242197, TEST_SNGL],
      [:mathieu_ce_e, [0, 20.0, GSL::M_PI_2], 1.609890857395926, TEST_SNGL],
      [:mathieu_ce_e, [0, 25.0, 0.0], 0.0002158630184146612, TEST_SNGL],
      [:mathieu_ce_e, [0, 25.0, GSL::M_PI_2], 1.657510298323475, TEST_SNGL],
      [:mathieu_ce_e, [1, 0.0, 0.0], 1.00000000, TEST_SNGL],
      [:mathieu_ce_e, [1, 5.0, 0.0], 0.2565428793223637, TEST_SNGL],
      [:mathieu_ce_e, [1, 10.0, 0.0], 0.05359874774717657, TEST_SNGL],
      [:mathieu_ce_e, [1, 15.0, 0.0], 0.01504006645382623, TEST_SNGL],
      [:mathieu_se_e, [5, 0.0, GSL::M_PI_2], 1.0000000, TEST_SNGL],
      [:mathieu_se_e, [5, 5.0, GSL::M_PI_2], 0.9060779302023551, TEST_SNGL],
      [:mathieu_se_e, [5, 10.0, GSL::M_PI_2], 0.8460384335355106, TEST_SNGL],
      [:mathieu_se_e, [5, 15.0, GSL::M_PI_2], 0.837949340012484, TEST_SNGL],
      [:mathieu_se_e, [5, 20.0, GSL::M_PI_2], 0.8635431218533667, TEST_SNGL],
      [:mathieu_se_e, [5, 25.0, GSL::M_PI_2], 0.8992683245108413, TEST_SNGL],
      [:mathieu_ce_e, [10, 0.0, 0.0], 1.00000000, TEST_SNGL],
      [:mathieu_ce_e, [10, 0.0, GSL::M_PI_2], -1.00000000, TEST_SNGL]
    ],
    :sf => [
      [:clausen_e, [GSL::M_PI / 20.0], 0.4478882448133546, 0],
      [:clausen_e, [GSL::M_PI / 6.0], 0.8643791310538927, 0],
      [:clausen_e, [GSL::M_PI / 3.0], 1.0149416064096535, 0],
      [:clausen_e, [  2.0 * GSL::M_PI + GSL::M_PI / 3.0], 1.0149416064096535, 0],
      [:clausen_e, [100.0 * GSL::M_PI + GSL::M_PI / 3.0], 1.0149416064096535, 0],
      [:coupling_3j_e, [0, 1, 1, 0,  1, -1], Math.sqrt(1.0 / 2.0), 0],
      [:coupling_3j_e, [1, 1, 2, 1, -1,  0], Math.sqrt(1.0 / 6.0), 0],
      [:coupling_3j_e, [2, 4, 6, 0,  2, -2], Math.sqrt(8.0 / 105.0), 0],
      [:coupling_3j_e, [4, 4, 8, 0,  0,  0], Math.sqrt(2.0 / 35.0), 0],
      [:coupling_3j_e, [4, 4, 8, 2, -2,  0], 2.0 / 3.0 * Math.sqrt(2.0 / 35.0), 2],
      [:coupling_3j_e, [4, 4, 8, 4, -4,  0], 1.0 / (3.0 * Math.sqrt(70.0)), 2],
#     [:coupling_3j_e, [-1, 1, 2, 1, -1, 0], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_3j_e, [1, -1, 2, 1, -1, 0], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_3j_e, [1, 1, -2, 1, -1, 0], GSL::NAN, GSL::NAN, GSL::EDOM],
      [:coupling_3j_e, [1, 1, 2, 2, -1, 0], 0, 0.0],
      [:coupling_3j_e, [1, 1, 2, 1, -2, 0], 0, 0.0],
      [:coupling_3j_e, [1, 1, 2, 1, -1, 3], 0, 0.0],
      [:coupling_3j_e, [1, 1, 3, 1, -1, 0], 0, 0.0],
      [:coupling_3j_e, [1, 4, 2, 1, -1, 0], 0, 0.0],
      [:coupling_3j_e, [4, 1, 2, 1, -1, 0], 0, 0.0],
      [:coupling_6j_e, [2, 2, 4, 2, 2, 2],  1.0 / 6.0, 0],
      [:coupling_6j_e, [4, 4, 2, 4, 4, 4], -1.0 / 10.0, 0],
      [:coupling_6j_e, [4, 4, 2, 4, 4, 2],  1.0 / 6.0, 0],
      [:coupling_6j_e, [4, 4, 2, 2, 2, 2], -0.5 / Math.sqrt(5.0), 0],
      [:coupling_6j_e, [4, 4, 4, 2, 2, 2],  Math.sqrt(7.0 / 3.0) / 10.0, 0],
      [:coupling_6j_e, [6, 6, 6, 4, 4, 4], -Math.sqrt(3.0 / 5.0) / 14.0, 0],
      [:coupling_6j_e, [6, 6, 6, 4, 4, 2], -Math.sqrt(3.0 / 5.0) / 7.0, 0],
#     [:coupling_6j_e, [-2, 2, 4, 2, 2, 2], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_6j_e, [2, -2, 4, 2, 2, 2], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_6j_e, [2, 2, -4, 2, 2, 2], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_6j_e, [2, 2, 4, -2, 2, 2], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_6j_e, [2, 2, 4, 2, -2, 2], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_6j_e, [2, 2, 4, 2, 2, -2], GSL::NAN, GSL::NAN, GSL::EDOM],
      [:coupling_6j_e, [2, 2, 4, 2, 2, 7], 0, 0.0],
      [:coupling_6j_e, [2, 2, 4, 2, 7, 2], 0, 0.0],
      [:coupling_6j_e, [2, 2, 4, 7, 2, 2], 0, 0.0],
      [:coupling_6j_e, [2, 2, 7, 2, 2, 2], 0, 0.0],
      [:coupling_6j_e, [2, 7, 4, 2, 2, 2], 0, 0.0],
      [:coupling_6j_e, [7, 2, 4, 2, 2, 2], 0, 0.0],
      [:coupling_9j_e, [4, 2,  4, 3, 3, 2, 1, 1, 2], -Math.sqrt(1.0 / 6.0) / 10.0, 2],
      [:coupling_9j_e, [8, 4, 10, 7, 3, 8, 1, 1, 2],  Math.sqrt(7.0 / 3.0) / 60.0, 2],
#     [:coupling_9j_e, [-4, 2, 4, 3, 3, 2, 1, 1, 2], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_9j_e, [4, -2, 4, 3, 3, 2, 1, 1, 2], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_9j_e, [4, 2, -4, 3, 3, 2, 1, 1, 2], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_9j_e, [4, 2, 4, -3, 3, 2, 1, 1, 2], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_9j_e, [4, 2, 4, 3, -3, 2, 1, 1, 2], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_9j_e, [4, 2, 4, 3, 3, -2, 1, 1, 2], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_9j_e, [4, 2, 4, 3, 3, 2, -1, 1, 2], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_9j_e, [4, 2, 4, 3, 3, 2, 1, -1, 2], GSL::NAN, GSL::NAN, GSL::EDOM],
#     [:coupling_9j_e, [4, 2, 4, 3, 3, 2, 1, 1, -2], GSL::NAN, GSL::NAN, GSL::EDOM],
      [:coupling_9j_e, [10, 2, 4, 3, 3, 2, 1, 1, 2], 0, 0.0],
      [:coupling_9j_e, [4, 10, 4, 3, 3, 2, 1, 1, 2], 0, 0.0],
      [:coupling_9j_e, [4, 2, 10, 3, 3, 2, 1, 1, 2], 0, 0.0],
      [:coupling_9j_e, [4, 2, 4, 10, 3, 2, 1, 1, 2], 0, 0.0],
      [:coupling_9j_e, [4, 2, 4, 3, 10, 2, 1, 1, 2], 0, 0.0],
      [:coupling_9j_e, [4, 2, 4, 3, 3, 10, 1, 1, 2], 0, 0.0],
      [:coupling_9j_e, [4, 2, 4, 3, 3, 2, 10, 1, 2], 0, 0.0],
      [:coupling_9j_e, [4, 2, 4, 3, 3, 2, 1, 10, 2], 0, 0.0],
      [:coupling_9j_e, [4, 2, 4, 3, 3, 2, 1, 1, 10], 0, 0.0],
      [:dawson_e, [1.0e-15], 1.0e-15, 0],
      [:dawson_e, [0.5], 0.4244363835020222959, 0],
      [:dawson_e, [2.0], 0.30134038892379196603, 0],
      [:dawson_e, [1000.0], 0.0005000002500003750009, 0],
      [:debye_1_e, [0.1],  0.975277750004723276, 0],
      [:debye_1_e, [1.0],  0.777504634112248239, 0],
      [:debye_1_e, [10.0], 0.164443465679946027, 0],
      [:debye_2_e, [0.1],  0.967083287045302664,  0],
      [:debye_2_e, [1.0],  0.70787847562782924,   0],
      [:debye_2_e, [10.0], 0.0479714980201218708, 0],
      [:debye_3_e, [0.1],  0.962999940487211048,  0],
      [:debye_3_e, [1.0],  0.674415564077814667,  0],
      [:debye_3_e, [10.0], 0.0192957656903454886, 0],
      [:debye_4_e, [0.1],  0.960555486124335944,   0],
      [:debye_4_e, [1.0],  0.654874068886737049,   0],
      [:debye_4_e, [10.0], 0.00967367556027115896, 0],
      [:multiply_e, [-3.0,2.0], -6.0,          0],
      [:multiply_e, [0.2 * GSL::DBL_MAX, 1.0 / (0.2 * GSL::DBL_MAX)],  1.0,          0],
      [:multiply_e, [0.2 * GSL::DBL_MAX, 0.2],    0.04 * GSL::DBL_MAX, 1],
      [:multiply_e, [0.2 * GSL::DBL_MAX, 4.0],    0.8 * GSL::DBL_MAX,  1],
      [:ellint_Kcomp_e, [ 0.99, GSL::MODE_DEFAULT], 3.3566005233611923760, 0],
      [:ellint_Kcomp_e, [ 0.50, GSL::MODE_DEFAULT], 1.6857503548125960429, 0],
      [:ellint_Kcomp_e, [0.010, GSL::MODE_DEFAULT], 1.5708355989121522360, 0],
      [:ellint_Ecomp_e, [0.99, GSL::MODE_DEFAULT], 1.0284758090288040010, 0],
      [:ellint_Ecomp_e, [0.50, GSL::MODE_DEFAULT], 1.4674622093394271555, 0],
      [:ellint_Ecomp_e, [0.01, GSL::MODE_DEFAULT], 1.5707570561503852873, 0],
      [:ellint_F_e, [GSL::M_PI / 3.0, 0.99, GSL::MODE_DEFAULT], 1.3065333392738766762, 0],
      [:ellint_F_e, [GSL::M_PI / 3.0, 0.50, GSL::MODE_DEFAULT], 1.0895506700518854093, 0],
      [:ellint_F_e, [GSL::M_PI / 3.0, 0.01, GSL::MODE_DEFAULT], 1.0472129063770918952, 0],
      [:ellint_E_e, [GSL::M_PI / 3.0, 0.99, GSL::MODE_DEFAULT], 0.8704819220377943536, 0],
      [:ellint_E_e, [GSL::M_PI / 3.0, 0.50, GSL::MODE_DEFAULT], 1.0075555551444720293, 0],
      [:ellint_E_e, [GSL::M_PI / 3.0, 0.01, GSL::MODE_DEFAULT], 1.0471821963889481104, 0],
      [:ellint_P_e, [GSL::M_PI / 3.0, 0.99, 0.5, GSL::MODE_DEFAULT], 1.1288726598764099882, 0],
      [:ellint_P_e, [GSL::M_PI / 3.0, 0.50, 0.5, GSL::MODE_DEFAULT], 0.9570574331323584890, 0],
      [:ellint_P_e, [GSL::M_PI / 3.0, 0.01, 0.5, GSL::MODE_DEFAULT], 0.9228868127118118465, 0],
      [:ellint_RF_e, [5.0e-11, 1.0e-10, 1.0, GSL::MODE_DEFAULT], 12.36441982979439, 0],
      [:ellint_RF_e, [1.0, 2.0, 3.0, GSL::MODE_DEFAULT], 0.7269459354689082, 0],
      [:ellint_RD_e, [5.0e-11, 1.0e-10, 1.0, GSL::MODE_DEFAULT], 34.0932594919337362, 0],
      [:ellint_RD_e, [1.0, 2.0, 3.0, GSL::MODE_DEFAULT], 0.2904602810289906, 0],
      [:ellint_RC_e, [1.0, 2.0, GSL::MODE_DEFAULT], 0.7853981633974482, 0],
      [:ellint_RJ_e, [2.0, 3.0, 4.0, 5.0, GSL::MODE_DEFAULT], 0.1429757966715675, 0],
      [:erfc_e, [-10.0], 2.0, 0],
      [:erfc_e, [-5.0000002], 1.9999999999984625433, 0],
      [:erfc_e, [-5.0], 1.9999999999984625402, 0],
      [:erfc_e, [-1.0], 1.8427007929497148693, 0],
      [:erfc_e, [-0.5], 1.5204998778130465377, 0],
      [:erfc_e, [1.0], 0.15729920705028513066, 0],
      [:erfc_e, [3.0], 0.000022090496998585441373, 1],
      [:erfc_e, [7.0], 4.183825607779414399e-23, 2],
      [:erfc_e, [10.0], 2.0884875837625447570e-45, 2],
      [:log_erfc_e, [-1.0], Math.log(1.842700792949714869), 0],
      [:log_erfc_e, [-0.1], 0.106576400586522485015, 0],
      [:log_erfc_e, [-1e-10],  1.1283791670318505967e-10, 0],
      [:log_erfc_e, [0.0], Math.log(1.0), 0],
      [:log_erfc_e, [1e-10], -1.128379167159174551e-10, 0],
      [:log_erfc_e, [0.001], -0.0011290158896213548027, 0],
      [:log_erfc_e, [0.1], -0.119304973737395598329, 0],
      [:log_erfc_e, [1.0], Math.log(0.15729920705028513066), 0],
      [:log_erfc_e, [10.0], Math.log(2.0884875837625447570e-45), 0],
      [:erf_e, [-10.0], -1.0000000000000000000, 0],
      [:erf_e, [0.5], 0.5204998778130465377, 0],
      [:erf_e, [1.0], 0.8427007929497148693, 0],
      [:erf_e, [10.0], 1.0000000000000000000, 0],
      [:erf_Z_e, [1.0],  0.24197072451914334980,   0],
      [:erf_Q_e, [10.0], 7.619853024160526066e-24, 2],
      [:hazard_e, [-20.0], 5.5209483621597631896e-88, 2],
      [:hazard_e, [-10.0], 7.6945986267064193463e-23, 2],
      [:hazard_e, [-1.0], 0.28759997093917836123, 0],
      [:hazard_e, [ 0.0], 0.79788456080286535588, 0],
      [:hazard_e, [ 1.0], 1.5251352761609812091, 0],
      [:hazard_e, [10.0], 10.098093233962511963, 2],
      [:hazard_e, [20.0], 20.049753068527850542, 2],
      [:hazard_e, [30.0], 30.033259667433677037, 2],
      [:hazard_e, [50.0], 50.019984031905639809, 0],
      [:hazard_e, [80.0], 80.012496096798234468, 0],
      [:hazard_e, [150.0], 150.00666607420571802, 0],
      [:hazard_e, [300.0], 300.00333325926337415, 0],
      [:hazard_e, [900.0], 900.00111110836764382, 0],
      [:hazard_e, [1001.0], 1001.0009989990049990, 0],
      [:hazard_e, [2000.0], 2000.0004999997500003, 0],
      [:exp_e, [-10.0], Math.exp(-10.0), 0],
      [:exp_e, [ 10.0], Math.exp( 10.0), 0],
      [:exp_err_e, [-10.0, TEST_TOL[1]], Math.exp(-10.0), 1],
      [:exp_err_e, [ 10.0, TEST_TOL[1]], Math.exp( 10.0), 1],
      [:exp_mult_e, [-10.0,  1.0e-06], 1.0e-06 * Math.exp(-10.0), 0],
      [:exp_mult_e, [-10.0,  2.0],     2.0 * Math.exp(-10.0),     0],
      [:exp_mult_e, [-10.0, -2.0],    -2.0 * Math.exp(-10.0),     0],
      [:exp_mult_e, [ 10.0,  1.0e-06], 1.0e-06 * Math.exp( 10.0), 0],
      [:exp_mult_e, [ 10.0, -2.0],    -2.0 * Math.exp( 10.0),     0],
      [:exp_mult_e, [  0.0, 1.00001],      1.00001 * Math.exp(0),     3],
      [:exp_mult_e, [  0.0, 1.000001],     1.000001 * Math.exp(0),    3],
      [:exp_mult_e, [  0.0, 1.000000001],  1.000000001 * Math.exp(0), 3],
      [:exp_mult_e, [  0.0, 100.0],        100.0 * Math.exp(0),       3],
      [:exp_mult_e, [  0.0, 1.0e+20],      1.0e+20 * Math.exp(0),     3],
      [:exp_mult_e, [  0.0, Math.exp(0) * Math.exp(GSL::M_LN2)],  2.0, 4],
      [:exp_mult_err_e, [-10.0, TEST_SQRT_TOL0, 2.0, TEST_SQRT_TOL0], 2.0 * Math.exp(-10.0), TEST_SQRT_TOL0],
      [:exp_mult_err_e, [  0.0, 0, Math.exp(0) * Math.exp(GSL::M_LN2), 0],  2.0, TEST_SQRT_TOL0],
      [:expm1_e, [-10.0], Math.exp(-10.0)-1.0, 0],
      [:expm1_e, [-0.001], -0.00099950016662500845, 0],
      [:expm1_e, [-1.0e-8], -1.0e-08 + 0.5e-16, 0],
      [:expm1_e, [ 1.0e-8], 1.0e-08 + 0.5e-16, 0],
      [:expm1_e, [ 0.001], 0.0010005001667083417, 0],
      [:expm1_e, [ 10.0], Math.exp(10.0)-1.0, 0],
      [:exprel_e, [-10.0], 0.0999954600070237515, 0],
      [:exprel_e, [-0.001], 0.9995001666250084, 0],
      [:exprel_e, [-1.0e-8], 1.0 - 0.5e-08, 0],
      [:exprel_e, [ 1.0e-8], 1.0 + 0.5e-08, 0],
      [:exprel_e, [ 0.001], 1.0005001667083417, 0],
      [:exprel_e, [ 10.0], 2202.5465794806716517, 0],
      [:exprel_2_e, [-10.0], 0.18000090799859524970, 0],
      [:exprel_2_e, [-0.001], 0.9996667499833361107, 0],
      [:exprel_2_e, [-1.0e-8], 0.9999999966666666750, 0],
      [:exprel_2_e, [ 1.0e-8], 1.0000000033333333417, 0],
      [:exprel_2_e, [ 0.001], 1.0003334166833361115, 0],
      [:exprel_2_e, [ 10.0], 440.3093158961343303, 0],
      [:exprel_n_e, [3, -1000.0], 0.00299400600000000000, 0],
      [:exprel_n_e, [3, -100.0], 0.02940600000000000000, 0],
      [:exprel_n_e, [3, -10.0], 0.24599972760042142509, 0],
      [:exprel_n_e, [3, -3.0], 0.5444917625849191238, 0],
      [:exprel_n_e, [3, -0.001], 0.9997500499916678570, 0],
      [:exprel_n_e, [3, -1.0e-8], 0.9999999975000000050, 0],
      [:exprel_n_e, [3,  1.0e-8], 1.0000000025000000050, 0],
      [:exprel_n_e, [3,  0.001], 1.0002500500083345240, 0],
      [:exprel_n_e, [3,  3.0], 2.5745637607083706091, 0],
      [:exprel_n_e, [3,  3.1], 2.6772417068460206247, 0],
      [:exprel_n_e, [3,  10.0], 131.79279476884029910, 1],
      [:exprel_n_e, [3,  100.0], 1.6128702850896812690e+38, 2],
      [:exprel_n_e, [50, -1000.0], 0.04766231609253975959, 0],
      [:exprel_n_e, [50, -100.0], 0.3348247572345889317, 0],
      [:exprel_n_e, [50, -10.0], 0.8356287051853286482, 0],
      [:exprel_n_e, [50, -3.0], 0.9443881609152163615, 0],
      [:exprel_n_e, [50, -1.0], 0.980762245565660617, 0],
      [:exprel_n_e, [50, -1.0e-8], 1.0 - 1.0e-8 / 51.0, 0],
      [:exprel_n_e, [50,  1.0e-8], 1.0 + 1.0e-8 / 51.0, 0],
      [:exprel_n_e, [50,  1.0], 1.01999216583666790, 0],
      [:exprel_n_e, [50,  3.0], 1.0624205757460368307, 0],
      [:exprel_n_e, [50,  48.0], 7.499573876877194416, 0],
      [:exprel_n_e, [50,  50.1], 9.311803306230992272, 4],
      [:exprel_n_e, [50,  100.0], 8.175664432485807634e+07, 4],
      [:exprel_n_e, [50,  500.0], 4.806352370663185330e+146, 3],
      [:exprel_n_e, [500, -1000.0], 0.3334815803127619256, 0],
      [:exprel_n_e, [500, -100.0], 0.8335646217536183909, 0],
      [:exprel_n_e, [500, -10.0], 0.9804297803131823066, 0],
      [:exprel_n_e, [500, -3.0], 0.9940475488850672997, 0],
      [:exprel_n_e, [500, -1.0], 0.9980079602383488808, 0],
      [:exprel_n_e, [500, -1.0e-8], 1.0 - 1.0e-8 / 501.0, 0],
      [:exprel_n_e, [500,  1.0e-8], 1.0 + 1.0e-8 / 501.0, 0],
      [:exprel_n_e, [500,  1.0], 1.0019999920160634252, 0],
      [:exprel_n_e, [500,  3.0], 1.0060240236632444934, 0],
      [:exprel_n_e, [500,  48.0], 1.1059355517981272174, 0],
      [:exprel_n_e, [500,  100.0], 1.2492221464878287204, 1],
      [:exprel_n_e, [500,  500.0], 28.363019877927630858, 2],
      [:exprel_n_e, [500,  1000.0], 2.4037563160335300322e+68, 4],
      [:exprel_n_e, [500,  1600.0], 7.899293535320607403e+226, 4],
      [:expint_E1_e, [-1.0], -1.8951178163559367555, 0],
      [:expint_E1_e, [1.0e-10], 22.448635265138923980, 0],
      [:expint_E1_e, [1.0e-05], 10.935719800043695615, 0],
      [:expint_E1_e, [0.1], 1.82292395841939066610, 0],
      [:expint_E1_e, [1.0], 0.21938393439552027368, 0],
      [:expint_E1_e, [10.0], 4.156968929685324277e-06, 1],
      [:expint_E1_e, [50.0], 3.783264029550459019e-24, 2],
      [:expint_E1_e, [300.0], 1.710384276804510115e-133, 2],
      [:expint_E2_e, [-1.0], 0.8231640121031084799, 1],
      [:expint_E2_e, [1.0 / 4294967296.0], 0.9999999947372139168, 0],
      [:expint_E2_e, [1.0 / 65536.0], 0.9998243233207178845, 0],
      [:expint_E2_e, [0.1], 0.7225450221940205066, 0],
      [:expint_E2_e, [1.0], 0.14849550677592204792, 0],
      [:expint_E2_e, [10.0], 3.830240465631608762e-06, 1],
      [:expint_E2_e, [50.0], 3.711783318868827367e-24, 2],
      [:expint_E2_e, [300.0], 1.7047391998483433998e-133, 2],
    #if GSL::GSL_VERSION >= '1.9.90'
      [:expint_En_e, [1, -1.0], -1.8951178163559367555, 2],
      [:expint_En_e, [1, 1.0e-10],  22.448635265138923980, 2],
      [:expint_En_e, [1, 1.0e-5], 10.93571980004369561, 2],
      [:expint_En_e, [1,0.1], 1.82292395841939066610, 0],
      [:expint_En_e, [1,1.0], 0.21938393439552027368, 0],
      [:expint_En_e, [1,10.0], 4.156968929685324277e-06, 1],
      [:expint_En_e, [1,50.0], 3.783264029550459019e-24, 2],
      [:expint_En_e, [1,300.0], 1.710384276804510115e-133, 2],

      [:expint_En_e, [2,-1.0], 0.8231640121031084799, 1],
      [:expint_En_e, [2,0.0], 1.0, 0],
      [:expint_En_e, [2,1.0 / 4294967296.0], 0.9999999947372139168, 0],
      [:expint_En_e, [2,1.0 / 65536.0], 0.9998243233207178845, 0],
      [:expint_En_e, [2,0.1], 0.7225450221940205066, 0],
      [:expint_En_e, [2,1.0], 0.14849550677592204792, 0],
      [:expint_En_e, [2,10.0], 3.830240465631608762e-06, 1],
      [:expint_En_e, [2,50.0], 3.711783318868827367e-24, 2],
      [:expint_En_e, [2,300.0], 1.7047391998483433998e-133, 2],

      [:expint_En_e, [3,0.0], 0.5, 0],
      [:expint_En_e, [3,1.0 / 4294967296.0], 0.499999999767169356972, 1],
      [:expint_En_e, [3,1.0 / 65536.0], 0.4999847426094515610, 0],
      [:expint_En_e, [3,0.1], 0.4162914579082787612543, 0],
      [:expint_En_e, [3,1.0], 0.10969196719776013683858, 1],
      [:expint_En_e, [3,10.0],0.000003548762553084381959981, 1],
      [:expint_En_e, [3,50.0], 3.6429094264752049812e-24, 2],
      [:expint_En_e, [3,300.0],1.699131143349179084e-133, 2],

      [:expint_En_e, [10,0.0], 0.111111111111111111, 0],
      [:expint_En_e, [10,1.0 / 4294967296.0], 0.111111111082007280658, 2],
      [:expint_En_e, [10,1.0 / 65536.0], 0.11110920377910896018606, 1],
      [:expint_En_e, [10,0.1], 0.099298432000896813567905, 1],
      [:expint_En_e, [10,1.0], 0.036393994031416401634164534, 1],
      [:expint_En_e, [10,10.0], 0.00000232530265702821081778968, 1],
      [:expint_En_e, [10,50.0], 3.223296586749110919572e-24, 2],
      [:expint_En_e, [10,300.0], 1.6608815083360041367294736e-133, 2],
    #end
      [:expint_Ei_e, [-1.0],   -0.21938393439552027368, 0],
      [:expint_Ei_e, [1.0 / 4294967296.0], -21.603494112783886397, 0],
      [:expint_Ei_e, [1.0], 1.8951178163559367555, 0],
      [:expint_E1_scaled_e, [-10000.0], -0.00010001000200060024012, 0],
      [:expint_E1_scaled_e, [-1000.0], -0.0010010020060241207251, 0],
      [:expint_E1_scaled_e, [-10.0], -0.11314702047341077803, 0],
      [:expint_E1_scaled_e, [-1.0], -0.69717488323506606877, 0],
      [:expint_E1_scaled_e, [1.0e-10], 22.448635267383787506, 0],
      [:expint_E1_scaled_e, [1.0e-05], 10.935829157788483865, 0],
      [:expint_E1_scaled_e, [0.1], 2.0146425447084516791, 0],
      [:expint_E1_scaled_e, [1.0], 0.59634736232319407434, 0],
      [:expint_E1_scaled_e, [10.0], 0.091563333939788081876, 0],
      [:expint_E1_scaled_e, [50.0], 0.019615109930114870365, 0],
      [:expint_E1_scaled_e, [300.0], 0.0033222955652707070644, 0],
      [:expint_E1_scaled_e, [1000.0], 0.00099900199402388071500, 0],
      [:expint_E1_scaled_e, [10000.0], 0.000099990001999400239880, 0],
      [:expint_E2_scaled_e, [-10000.0], -0.00010002000600240120072, 3],
      [:expint_E2_scaled_e, [-1000.0], -0.0010020060241207250807, 3],
      [:expint_E2_scaled_e, [-10.0], -0.13147020473410778034, 1],
      [:expint_E2_scaled_e, [-1.0], 0.30282511676493393123, 1],
      [:expint_E2_scaled_e, [1.0 / 4294967296.0], 0.99999999497004455927, 0],
      [:expint_E2_scaled_e, [1.0 / 65536.0], 0.99983957954556245453, 0],
      [:expint_E2_scaled_e, [0.1], 0.79853574552915483209, 0],
      [:expint_E2_scaled_e, [1.0], 0.40365263767680592566, 0],
      [:expint_E2_scaled_e, [10.0], 0.084366660602119181239, 1],
      [:expint_E2_scaled_e, [50.0], 0.019244503494256481735, 2],
      [:expint_E2_scaled_e, [300.0], 0.0033113304187878806691, 0],
      [:expint_E2_scaled_e, [1000.0], 0.00099800597611928500004, 0],
      [:expint_E2_scaled_e, [10000.0], 0.000099980005997601199281, 0],
      [:expint_Ei_scaled_e, [-1000.0], -0.00099900199402388071500, 0],
      [:expint_Ei_scaled_e, [-1.0], -0.59634736232319407434, 0],
      [:expint_Ei_scaled_e, [1.0 / 4294967296.0], -21.603494107753930958, 0],
      [:expint_Ei_scaled_e, [1.0], 0.69717488323506606877, 0],
      [:expint_Ei_scaled_e, [1000.0], 0.0010010020060241207251, 0],
      [:Shi_e, [-1.0], -1.0572508753757285146, 0],
      [:Shi_e, [1.0 / 4294967296.0], 2.3283064365386962891e-10, 0],
      [:Shi_e, [1.0 / 65536.0], 0.00001525878906269737298, 0],
      [:Shi_e, [0.1], 0.1000555722250569955, 0],
      [:Shi_e, [1.0], 1.0572508753757285146, 0],
      [:Shi_e, [10.0], 1246.1144901994233444, 1],
      [:Shi_e, [50.0], 5.292818448565845482e+19, 2],
      [:Shi_e, [300.0], 3.248241254044332895e+127, 2],
      [:Chi_e, [-1.0], 0.8378669409802082409, 0],
      [:Chi_e, [1.0 / 4294967296.0], -21.603494113016717041, 0],
      [:Chi_e, [1.0 / 65536.0], -10.513139223999384429, 0],
      [:Chi_e, [1.0 / 8.0], -1.4983170827635760646, 0],
      [:Chi_e, [1.0], 0.8378669409802082409, 0],
      [:Chi_e, [10.0], 1246.1144860424544147, 1],
      [:Chi_e, [50.0], 5.292818448565845482e+19, 2],
      [:Chi_e, [300.0], 3.248241254044332895e+127, 2],
      [:expint_3_e, [1.0e-10], 1.0e-10, 0],
      [:expint_3_e, [1.0e-05], 9.9999999999999975e-06, 0],
      [:expint_3_e, [0.1], 0.09997500714119079665122, 0],
      [:expint_3_e, [0.5], 0.48491714311363971332427, 0],
      [:expint_3_e, [1.0], 0.80751118213967145285833, 0],
      [:expint_3_e, [2.0], 0.89295351429387631138208, 0],
      [:expint_3_e, [5.0], 0.89297951156924921121856, 0],
      [:expint_3_e, [10.0], 0.89297951156924921121856, 0],
      [:expint_3_e, [100.0], 0.89297951156924921121856, 0],
      [:Si_e, [-1.0], -0.9460830703671830149, 0],
      [:Si_e, [1.0e-10], 1.0e-10, 0],
      [:Si_e, [1.0e-05], 9.999999999944444444e-06, 0],
      [:Si_e, [0.1], 0.09994446110827695016, 0],
      [:Si_e, [1.0], 0.9460830703671830149, 0],
      [:Si_e, [10.0], 1.6583475942188740493, 0],
      [:Si_e, [50.0], 1.5516170724859358947, 0],
      [:Si_e, [300.0], 1.5708810882137495193, 0],
      [:Si_e, [1.0e+20], 1.5707963267948966192, 0],
      [:Ci_e, [1.0 / 4294967296.0], -21.603494113016717041, 0],
      [:Ci_e, [1.0 / 65536.0], -10.513139224115799751, 0],
      [:Ci_e, [1.0 / 8.0], -1.5061295845296396649, 0],
      [:Ci_e, [1.0], 0.3374039229009681347, 0],
      [:Ci_e, [10.0], -0.04545643300445537263, 0],
      [:Ci_e, [50.0], -0.005628386324116305440, 0],
      [:Ci_e, [300.0], -0.003332199918592111780, 0],
      [:Ci_e, [65536.0], 0.000010560248837656279453, 0],
      [:Ci_e, [4294967296.0], -1.0756463261957757485e-10, TEST_SQRT_TOL0],
      [:Ci_e, [1099511627776.0], -3.689865584710764214e-13, 1024.0 * TEST_SQRT_TOL0],
      [:atanint_e, [1.0e-10], 1.0e-10, 0],
      [:atanint_e, [1.0e-05], 9.99999999988888888889e-06, 0],
      [:atanint_e, [0.1], 0.09988928686033618404, 0],
      [:atanint_e, [1.0], 0.91596559417721901505, 0],
      [:atanint_e, [2.0], 1.57601540344632342236, 0],
      [:atanint_e, [10.0], 3.71678149306806859029, 0],
      [:atanint_e, [50.0], 6.16499047850274874222, 0],
      [:atanint_e, [300.0], 8.96281388924518959990, 0],
      [:atanint_e, [1.0e+5], 18.084471031038661920, 0],
      [:fermi_dirac_m1_e, [-10.0], 0.00004539786870243439450, 0],
      [:fermi_dirac_m1_e, [ -1.0], 0.26894142136999512075, 0],
      [:fermi_dirac_m1_e, [  1.0], 0.7310585786300048793, 0],
      [:fermi_dirac_m1_e, [ 10.0], 0.9999546021312975656, 0],
      [:fermi_dirac_0_e, [-10.0], 0.00004539889921686464677, 0],
      [:fermi_dirac_0_e, [ -1.0], 0.31326168751822283405, 0],
      [:fermi_dirac_0_e, [  1.0], 1.3132616875182228340, 0],
      [:fermi_dirac_0_e, [ 10.0], 10.000045398899216865, 0],
      [:fermi_dirac_1_e, [-10.0], 0.00004539941448447633524, 0],
      [:fermi_dirac_1_e, [ -2.0], 0.13101248471442377127, 0],
      [:fermi_dirac_1_e, [ -1.0], 0.3386479964034521798, 0],
      [:fermi_dirac_1_e, [ -0.4], 0.5825520806897909028, 0],
      [:fermi_dirac_1_e, [  0.4], 1.1423819861584355337, 0],
      [:fermi_dirac_1_e, [  1.0], 1.8062860704447742567, 0],
      [:fermi_dirac_1_e, [  1.5], 2.5581520872227806402, 0],
      [:fermi_dirac_1_e, [  2.5], 4.689474797599761667, 0],
      [:fermi_dirac_1_e, [ 10.0], 51.64488866743374196, 0],
      [:fermi_dirac_1_e, [ 12.0], 73.64492792264531092, 0],
      [:fermi_dirac_1_e, [ 20.0], 201.64493406478707282, 0],
      [:fermi_dirac_1_e, [ 50.0], 1251.6449340668482264, 0],
      [:fermi_dirac_2_e, [-10.0], 0.00004539967212174776662, 0],
      [:fermi_dirac_2_e, [ -2.0], 0.13313272938565030508, 0],
      [:fermi_dirac_2_e, [ -1.0], 0.3525648792978077590, 0],
      [:fermi_dirac_2_e, [ -0.4], 0.6229402647001272120, 0],
      [:fermi_dirac_2_e, [  0.4], 1.2915805581060844533, 0],
      [:fermi_dirac_2_e, [  1.0], 2.1641656128127008622, 0],
      [:fermi_dirac_2_e, [  1.5], 3.247184513920792475, 0],
      [:fermi_dirac_2_e, [  2.5], 6.797764392735056317, 0],
      [:fermi_dirac_2_e, [ 10.0], 183.11605273482105278, 0],
      [:fermi_dirac_2_e, [ 12.0], 307.73921494638635166, 0],
      [:fermi_dirac_2_e, [ 20.0], 1366.2320146723590157, 0],
      [:fermi_dirac_2_e, [ 50.0], 20915.580036675744655, 0],
      [:fermi_dirac_2_e, [200.0], 1.3336623201467029786e+06, 0],
      [:fermi_dirac_mhalf_e, [-10.0], 0.00004539847236080549532, 0],
      [:fermi_dirac_mhalf_e, [ -2.0], 0.12366562180120994266, 0],
      [:fermi_dirac_mhalf_e, [ -1.0], 0.29402761761145122022, 0],
      [:fermi_dirac_mhalf_e, [ -0.4], 0.4631755336886027800, 0],
      [:fermi_dirac_mhalf_e, [  0.4], 0.7654084737661656915, 0],
      [:fermi_dirac_mhalf_e, [  1.0], 1.0270571254743506890, 0],
      [:fermi_dirac_mhalf_e, [  1.5], 1.2493233478527122008, 0],
      [:fermi_dirac_mhalf_e, [  2.5], 1.6663128834358313625, 0],
      [:fermi_dirac_mhalf_e, [ 10.0], 3.552779239536617160, 0],
      [:fermi_dirac_mhalf_e, [ 12.0], 3.897268231925439359, 0],
      [:fermi_dirac_mhalf_e, [ 20.0], 5.041018507535328603, 0],
      [:fermi_dirac_mhalf_e, [ 50.0], 7.977530858581869960, 1],
      [:fermi_dirac_half_e, [-10.0], 0.00004539920105264132755, 1],
      [:fermi_dirac_half_e, [ -2.0], 0.12929851332007559106, 0],
      [:fermi_dirac_half_e, [ -1.0], 0.3277951592607115477, 0],
      [:fermi_dirac_half_e, [ -0.4], 0.5522452153690688947, 0],
      [:fermi_dirac_half_e, [  0.4], 1.0386797503389389277, 0],
      [:fermi_dirac_half_e, [  1.0], 1.5756407761513002308, 0],
      [:fermi_dirac_half_e, [  1.5], 2.1448608775831140360, 0],
      [:fermi_dirac_half_e, [  2.5], 3.606975377950373251, 0],
      [:fermi_dirac_half_e, [ 10.0], 24.084656964637653615, 0],
      [:fermi_dirac_half_e, [ 12.0], 31.540203287044242593, 0],
      [:fermi_dirac_half_e, [ 20.0], 67.49151222165892049, 0],
      [:fermi_dirac_half_e, [ 50.0], 266.09281252136259343, 1],
      [:fermi_dirac_3half_e, [-10.0], 0.00004539956540456176333, 0],
      [:fermi_dirac_3half_e, [ -2.0], 0.13224678225177236685, 0],
      [:fermi_dirac_3half_e, [ -1.0], 0.3466747947990574170, 0],
      [:fermi_dirac_3half_e, [ -0.4], 0.6056120213305040910, 0],
      [:fermi_dirac_3half_e, [  0.4], 1.2258236403963668282, 0],
      [:fermi_dirac_3half_e, [  1.0], 2.0022581487784644573, 0],
      [:fermi_dirac_3half_e, [  1.5], 2.9277494127932173068, 0],
      [:fermi_dirac_3half_e, [  2.5], 5.768879312210516582, 0],
      [:fermi_dirac_3half_e, [ 10.0], 101.00510084332600020, 2],
      [:fermi_dirac_3half_e, [ 12.0], 156.51518642795728036, 1],
      [:fermi_dirac_3half_e, [ 20.0], 546.5630100657601959, 1],
      [:fermi_dirac_3half_e, [ 50.0], 5332.353566687145552, 1],
      [:fermi_dirac_int_e, [3,  -2.0], 0.1342199155038680215, 0],
      [:fermi_dirac_int_e, [3,   0.0], 0.9470328294972459176, 0],
      [:fermi_dirac_int_e, [3,   0.1], 1.0414170610956165759, 0],
      [:fermi_dirac_int_e, [3,   1.0], 2.3982260822489407070, 0],
      [:fermi_dirac_int_e, [3,   3.0], 12.621635313399690724, 1],
      [:fermi_dirac_int_e, [3, 100.0], 4.174893231066566793e+06, 1],
      [:fermi_dirac_int_e, [3, 500.0], 2.604372285319088354e+09, 1],
      [:fermi_dirac_int_e, [5,  -2.0], 0.13505242246823676478, 0],
      [:fermi_dirac_int_e, [5,   0.0], 0.9855510912974351041, 0],
      [:fermi_dirac_int_e, [5,   0.1], 1.0876519750101492782, 0],
      [:fermi_dirac_int_e, [5,   1.0], 2.6222337848692390539, 0],
      [:fermi_dirac_int_e, [5,   3.0], 17.008801618012113022, 1],
      [:fermi_dirac_int_e, [5, 100.0], 1.3957522531334869874e+09, 1],
      [:fermi_dirac_int_e, [5, 500.0], 2.1705672808114817955e+13, 2],
      [:fermi_dirac_int_e, [7,  -2.0], 0.1352641105671255851, 0],
      [:fermi_dirac_int_e, [7,   0.0], 0.9962330018526478992, 0],
      [:fermi_dirac_int_e, [7,   0.1], 1.1005861815180315485, 0],
      [:fermi_dirac_int_e, [7,   1.0], 2.6918878172003129203, 0],
      [:fermi_dirac_int_e, [7,   3.0], 19.033338976999367642, 2],
      [:fermi_dirac_int_e, [7,  10.0], 5654.530932873610014, 1],
      [:fermi_dirac_int_e, [7,  50.0], 1.005005069985066278e+09, 2],
      [:fermi_dirac_int_e, [7, 500.0], 9.691690268341569514e+16, 3],
      [:fermi_dirac_int_e, [9,  -2.0], 0.1353174385330242691, 0],
      [:fermi_dirac_int_e, [9,   0.0], 0.9990395075982715656, 0],
      [:fermi_dirac_int_e, [9,   0.1], 1.1039997234712941212, 0],
      [:fermi_dirac_int_e, [9,   1.0], 2.7113648898129249947, 0],
      [:fermi_dirac_int_e, [9,   3.0], 19.768544008138602223, 2],
      [:fermi_dirac_int_e, [9,  10.0], 10388.990167312912478, 2],
      [:fermi_dirac_int_e, [9,  50.0], 2.85466960802601649e+10, 1],
      [:fermi_dirac_int_e, [9, 500.0], 2.69273849842695876e+20, 2 * 1],
      [:fermi_dirac_int_e, [10,  -2.0], 0.13532635396712288092, 0],
      [:fermi_dirac_int_e, [10,   0.0], 0.9995171434980607541, 0],
      [:fermi_dirac_int_e, [10,   0.1], 1.1045818238852612296, 0],
      [:fermi_dirac_int_e, [10,   1.0], 2.7147765350346120647, 0],
      [:fermi_dirac_int_e, [10,   3.0], 19.917151938411675171, 1],
      [:fermi_dirac_int_e, [10,  10.0], 12790.918595516495955, 2],
      [:fermi_dirac_int_e, [10,  50.0], 1.3147703201869657654e+11, 2],
      [:fermi_dirac_int_e, [10, 500.0], 1.2241331244469204398e+22, 2],
      [:fermi_dirac_int_e, [11,  -2.0], 0.1353308162894847149, 0],
      [:fermi_dirac_int_e, [11,   0.0], 0.9997576851438581909, 0],
      [:fermi_dirac_int_e, [11,   0.1], 1.1048751811565850418, 0],
      [:fermi_dirac_int_e, [11,   1.0], 2.7165128749007313436, 0],
      [:fermi_dirac_int_e, [11,   3.0], 19.997483022044603065, 2],
      [:fermi_dirac_int_e, [11,  10.0], 14987.996005901818036, 2],
      [:fermi_dirac_int_e, [11,  50.0], 5.558322924078990628e+11, 2],
      [:fermi_dirac_int_e, [11, 500.0], 5.101293089606198280e+23, 2],
      [:fermi_dirac_int_e, [20,  -2.0], 0.13533527450327238373, 0],
      [:fermi_dirac_int_e, [20,   0.0], 0.9999995232582155428, 0],
      [:fermi_dirac_int_e, [20,   0.1], 1.1051703357941368203, 0],
      [:fermi_dirac_int_e, [20,   1.0], 2.7182783069905721654, 0],
      [:fermi_dirac_int_e, [20,   3.0], 20.085345296028242734, 2],
      [:fermi_dirac_int_e, [20,  10.0], 21898.072920149606475, 2],
      [:fermi_dirac_int_e, [20,  50.0], 1.236873256595717618e+16, 2],
      [:fermi_dirac_int_e, [20, 500.0], 9.358938204369557277e+36, 2],
      [:gegenpoly_1_e, [-0.2,   1.0], -0.4, 0],
      [:gegenpoly_1_e, [ 0.0,   1.0], 2.0, 0],
      [:gegenpoly_1_e, [ 1.0,   1.0], 2.0, 0],
      [:gegenpoly_1_e, [ 1.0,   0.5], 1.0, 0],
      [:gegenpoly_1_e, [ 5.0,   1.0], 10.0, 0],
      [:gegenpoly_1_e, [ 100.0, 0.5], 100.0, 0],
      [:gegenpoly_2_e, [-0.2,   0.5], 0.12, 0],
      [:gegenpoly_2_e, [ 0.0,   1.0], 1.00, 0],
      [:gegenpoly_2_e, [ 1.0,   1.0], 3.00, 0],
      [:gegenpoly_2_e, [ 1.0,   0.1], -0.96, 0],
      [:gegenpoly_2_e, [ 5.0,   1.0], 55.0, 0],
      [:gegenpoly_2_e, [ 100.0, 0.5], 4950.0, 0],
      [:gegenpoly_3_e, [-0.2,   0.5], 0.112, 0],
      [:gegenpoly_3_e, [ 0.0,   1.0], -2.0 / 3.0, 0],
      [:gegenpoly_3_e, [ 1.0,   1.0], 4.000, 0],
      [:gegenpoly_3_e, [ 1.0,   0.1], -0.392, 0],
      [:gegenpoly_3_e, [ 5.0,   1.0], 220.000, 0],
      [:gegenpoly_3_e, [ 100.0, 0.5], 161600.000, 0],
      [:gegenpoly_n_e, [1,       1.0, 1.0], 2.000, 0],
      [:gegenpoly_n_e, [10,      1.0, 1.0], 11.000, 0],
      [:gegenpoly_n_e, [10,      1.0, 0.1], -0.4542309376, 0],
      [:gegenpoly_n_e, [10,      5.0, 1.0], 9.23780e+4, 0],
      [:gegenpoly_n_e, [10,    100.0, 0.5], 1.5729338392690000e+13, 0],
      [:gegenpoly_n_e, [1000,  100.0, 1.0], 3.3353666135627322e+232, 1],
      [:gegenpoly_n_e, [100,  2000.0, 1.0], 5.8753432034937579e+202, 0],
      [:gegenpoly_n_e, [103,   207.0, 2.0], 1.4210272202235983e+145, 0],
      [:gegenpoly_n_e, [103,    -0.4, 0.3], -1.64527498094522e-04, 1],
      [:laguerre_1_e, [0.5, -1.0], 2.5, 0],
      [:laguerre_1_e, [0.5,  1.0], 0.5, 0],
      [:laguerre_1_e, [1.0,  1.0], 1.0, 0],
      [:laguerre_2_e, [ 0.5, -1.0], 4.875,  0],
      [:laguerre_2_e, [ 0.5,  1.0], -0.125, 0],
      [:laguerre_2_e, [ 1.0,  1.0],  0.5, 0],
      [:laguerre_2_e, [-1.0,  1.0], -0.5, 0],
      [:laguerre_2_e, [-2.0,  1.0],  0.5, 0],
      [:laguerre_2_e, [-3.0,  1.0],  2.5, 0],
      [:laguerre_3_e, [0.5, -1.0], 8.479166666666666667,    0],
      [:laguerre_3_e, [0.5,  1.0], -0.6041666666666666667,  0],
      [:laguerre_3_e, [1.0,  1.0], -0.16666666666666666667, 1],
      [:laguerre_3_e, [ 2.0,  1.0], 2.3333333333333333333,  0],
      [:laguerre_3_e, [-2.0,  1.0], 1.0 / 3.0,  0],
      [:laguerre_3_e, [-3.0,  1.0], -1.0 / 6.0, 0],
      [:laguerre_3_e, [-4.0,  1.0], -8.0 / 3.0, 0],
      [:laguerre_n_e, [1, 0.5, 1.0], 0.5, 0],
      [:laguerre_n_e, [2, 1.0, 1.0], 0.5, 1],
      [:laguerre_n_e, [3, 2.0, 1.0], 2.3333333333333333333,   1],
      [:laguerre_n_e, [4, 2.0, 0.5], 6.752604166666666667,    1],
      [:laguerre_n_e, [90, 2.0,  0.5], -48.79047157201507897, 1],
      [:laguerre_n_e, [90, 2.0, -100.0], 2.5295879275042410902e+63, 2],
      [:laguerre_n_e, [90, 2.0,  100.0], -2.0929042259546928670e+20, 1],
      [:laguerre_n_e, [100, 2.0, -0.5], 2.2521795545919391405e+07,  2],
      [:laguerre_n_e, [100, 2.0,  0.5], -28.764832945909097418,     2],
      [:laguerre_n_e, [1000, 2.0, -0.5], 2.4399915170947549589e+21, 3],
      [:laguerre_n_e, [100000, 2.0, 1.0], 5107.73491348319,         4],
      [:laguerre_n_e, [1, -2.0, 1.0],  -2.0,     0],
      [:laguerre_n_e, [2, -2.0, 1.0],   0.5,     0],
      [:laguerre_n_e, [3, -2.0, 1.0],   1.0 / 3.0, 0],
      [:laguerre_n_e, [10, -2.0, 1.0], -0.04654954805996472663,   2],
      [:laguerre_n_e, [10, -5.0, 1.0], -0.0031385030864197530864, 2],
      [:laguerre_n_e, [10, -9.0, 1.0], -2.480158730158730159e-06, 5],
      [:laguerre_n_e, [10, -11.0,  1.0], 2.7182818011463844797,    2],
      [:laguerre_n_e, [10, -11.0, -1.0], 0.3678794642857142857,    2],
      [:laguerre_n_e, [100, -2.0,  1.0],  -0.0027339992019526273866,  TEST_SQRT_TOL0],
      [:laguerre_n_e, [100, -2.0, -1.0],   229923.09193402028290,     5],
      [:laguerre_n_e, [100, -10.0,  1.0],  3.25966665871244092e-11,   6],
      [:laguerre_n_e, [100, -10.0, -1.0],  0.00016484365618205810025, 6],
      [:laguerre_n_e, [100, -20.0, 1.0],  5.09567630343671251e-21,  3],
      [:laguerre_n_e, [100, -30.0, 1.0],  3.46063150272466192e-34,  1],
      [:laguerre_n_e, [100, -50.0,  1.0],  1.20981872933162889e-65,  1],
      [:laguerre_n_e, [100, -50.0, -1.0],  8.60763477742332922e-65,  1],
      [:laguerre_n_e, [100, -50.5,  1.0],  4.84021010426688393e-31,  1],
      [:laguerre_n_e, [100, -50.5, -1.0],  8.49861345212160618e-33,  1],
      [:laguerre_n_e, [100, -101.0,  1.0], 2.7182818284590452354,    1],
      [:laguerre_n_e, [100, -101.0, -1.0], 0.3678794411714423216,    1],
      [:laguerre_n_e, [100, -102.0,  1.0], 271.8281828459045235,    1],
      [:laguerre_n_e, [100, -102.0, -1.0], 37.52370299948711680,    1],
      [:laguerre_n_e, [100, -110.0,  1.0], 1.0666955248998831554e+13, 1],
      [:laguerre_n_e, [100, -110.0, -1.0], 1.7028306108058225871e+12, 1],
      [:laguerre_n_e, [100, -200.0,  1.0], 7.47851889721356628e+58,  1],
      [:laguerre_n_e, [100, -200.0, -1.0], 2.73740299754732273e+58,  1],
      [:laguerre_n_e, [100, -50.0,  10.0], 4.504712811317745591e-21,  TEST_SQRT_TOL0],
      [:laguerre_n_e, [100, -50.0, -10.0], 1.475165520610679937e-11,  1],
      [:lambert_W0_e, [0.0],  0.0,  0],
      [:lambert_W0_e, [1.0],  0.567143290409783872999969,  0],
      [:lambert_W0_e, [2.0],  0.852605502013725491346472,  0],
      [:lambert_W0_e, [20.0], 2.205003278024059970493066,  0],
      [:lambert_W0_e, [1000.0], 5.24960285240159622712606,  0],
      [:lambert_W0_e, [1.0e+6], 11.38335808614005262200016,  0],
      [:lambert_W0_e, [1.0e+12], 24.43500440493491313826305,  0],
      [:lambert_W0_e, [1.0e+308], 702.641362034106812081125,  0],
#     [:lambert_W0_e, [-1.0 / GSL::M_E - GSL::DBL_EPSILON], -1.0,  0, GSL::EDOM],
      [:lambert_W0_e, [-1.0 / GSL::M_E + 1.0 / (1024.0 * 1024.0 * 1024.0)], -0.999928845560308370714970, 0],
      [:lambert_W0_e, [-1.0 / GSL::M_E + 1.0 / (1024.0 * 1024.0)], -0.997724730359774141620354, 0],
      [:lambert_W0_e, [-1.0 / GSL::M_E + 1.0 / 512.0], -0.900335676696088773044678, 0],
      [:lambert_W0_e, [-1.0 / GSL::M_E + 0.25], -0.1349044682661213545487599, 0],
      [:lambert_Wm1_e, [0.0],  0.0,  0],
      [:lambert_Wm1_e, [1.0],  0.567143290409783872999969,  0],
      [:lambert_Wm1_e, [2.0],  0.852605502013725491346472,  0],
      [:lambert_Wm1_e, [20.0], 2.205003278024059970493066,  0],
#     [:lambert_Wm1_e, [-1.0 / GSL::M_E - GSL::DBL_EPSILON], -1.0,  0, GSL::EDOM],
      [:lambert_Wm1_e, [-1.0 / GSL::M_E + 1.0 / (1024.0 * 1024.0 * 1024.0)], -1.000071157815154608049055, 1],
      [:lambert_Wm1_e, [-1.0 / GSL::M_E + 1.0 / (1024.0 * 1024.0)], -1.002278726118593023934693, 1],
      [:lambert_Wm1_e, [-1.0 / GSL::M_E + 1.0 / 512.0], -1.106761200865743124599130, 1],
      [:lambert_Wm1_e, [-1.0 / GSL::M_E + 1.0 / 64.0], -1.324240940341812125489772, 1],
      [:lambert_Wm1_e, [-1.0 / GSL::M_E + 0.25], -3.345798131120112, 1],
      [:log_e, [0.1], -2.3025850929940456840,  0],
      [:log_e, [1.1], 0.09531017980432486004,  1],
      [:log_e, [1000.0], 6.907755278982137052, 0],
      [:log_abs_e, [-0.1], -2.3025850929940456840,  0],
      [:log_abs_e, [-1.1], 0.09531017980432486004,  1],
      [:log_abs_e, [-1000.0], 6.907755278982137052, 0],
      [:log_abs_e, [0.1], -2.3025850929940456840,  0],
      [:log_abs_e, [1.1], 0.09531017980432486004,  1],
      [:log_abs_e, [1000.0], 6.907755278982137052, 0],
      [:log_1plusx_e, [1.0e-10], 9.999999999500000000e-11, 0],
      [:log_1plusx_e, [1.0e-8], 9.999999950000000333e-09, 0],
      [:log_1plusx_e, [1.0e-4], 0.00009999500033330833533, 0],
      [:log_1plusx_e, [0.1], 0.09531017980432486004, 0],
      [:log_1plusx_e, [0.49], 0.3987761199573677730, 0],
      [:log_1plusx_e, [-0.49], -0.6733445532637655964, 0],
      [:log_1plusx_e, [1.0], GSL::M_LN2, 0],
      [:log_1plusx_e, [-0.99], -4.605170185988091368, 0],
      [:log_1plusx_mx_e, [1.0e-10], -4.999999999666666667e-21, 0],
      [:log_1plusx_mx_e, [1.0e-8], -4.999999966666666917e-17, 0],
      [:log_1plusx_mx_e, [1.0e-4], -4.999666691664666833e-09, 0],
      [:log_1plusx_mx_e, [0.1], -0.004689820195675139956, 0],
      [:log_1plusx_mx_e, [0.49], -0.09122388004263222704, 0],
      [:log_1plusx_mx_e, [-0.49], -0.18334455326376559639, 0],
      [:log_1plusx_mx_e, [1.0], GSL::M_LN2 - 1.0, 0],
      [:log_1plusx_mx_e, [-0.99], -3.615170185988091368, 0],
      [:pow_int_e, [2.0, 3], 8.0, 0],
      [:pow_int_e, [-2.0, 3], -8.0, 0],
      [:pow_int_e, [2.0, -3], 1.0 / 8.0, 0],
      [:pow_int_e, [-2.0, -3], -1.0 / 8.0, 0],
      [:pow_int_e, [10.0, 4], 1.0e+4, 0],
      [:pow_int_e, [10.0, -4], 1.0e-4, 0],
      [:pow_int_e, [-10.0, 4], 1.0e+4, 0],
      [:pow_int_e, [-10.0, -4], 1.0e-4, 0],
      [:pow_int_e, [10.0, 40], 1.0e+40, 0],
      [:pow_int_e, [8.0, -40], 7.523163845262640051e-37, 0],
      [:pow_int_e, [-10.0, 40], 1.0e+40, 0],
      [:pow_int_e, [-8.0, -40], 7.523163845262640051e-37, 0],
      [:pow_int_e, [10.0, 41], 1.0e+41, 0],
      [:pow_int_e, [8.0, -41], 9.403954806578300064e-38, 0],
      [:pow_int_e, [-10.0, 41], -1.0e+41, 0],
      [:pow_int_e, [-8.0, -41], -9.403954806578300064e-38, 0],
      [:psi_int_e, [1], -0.57721566490153286060, 0],
      [:psi_int_e, [2], 0.42278433509846713939, 0],
      [:psi_int_e, [3], 0.92278433509846713939, 0],
      [:psi_int_e, [4], 1.2561176684318004727, 0],
      [:psi_int_e, [5], 1.5061176684318004727, 0],
      [:psi_int_e, [100], 4.600161852738087400, 0],
      [:psi_int_e, [110], 4.695928024251535633, 0],
      [:psi_int_e, [5000], 8.517093188082904107, 0],
      [:psi_e, [5000.0], 8.517093188082904107, 0],
      [:psi_e, [5.0], 1.5061176684318004727, 0],
      [:psi_e, [-10.5],       2.3982391295357816134,  0],
      [:psi_e, [-100.5],      4.615124601338064117,  2],
      [:psi_e, [-1.0e+5-0.5], 11.512935464924395337, 4.0 * 4],
      [:psi_e, [-262144.0-0.5], 12.476653064769611581, 4.0 * 4],
      [:psi_1piy_e, [0.8], -0.07088340212750589223, 1],
      [:psi_1piy_e, [1.0],  0.09465032062247697727, 0],
      [:psi_1piy_e, [5.0],  1.6127848446157465854, 2],
      [:psi_1piy_e, [100.0],  4.605178519404762003, 0],
      [:psi_1piy_e, [2000.0], 7.600902480375416216, 0],
      [:psi_1piy_e, [-0.8], -0.07088340212750589223, 1],
      [:psi_1piy_e, [-1.0],  0.09465032062247697727, 0],
      [:psi_1piy_e, [-5.0],  1.6127848446157465854, 2],
      [:psi_1piy_e, [-100.0],  4.605178519404762003, 0],
      [:psi_1piy_e, [-2000.0], 7.600902480375416216, 0],
      [:psi_1_int_e, [1], 1.6449340668482264364,  0],
      [:psi_1_int_e, [2], 0.64493406684822643647, 0],
      [:psi_1_int_e, [3], 0.39493406684822643647, 0],
      [:psi_1_int_e, [4], 0.28382295573711532536, 0],
      [:psi_1_int_e, [1], 1.6449340668482264365,      0],
      [:psi_1_int_e, [5], 0.22132295573711532536,     0],
      [:psi_1_int_e, [100], 0.010050166663333571395,  0],
      [:psi_1_int_e, [110], 0.009132356622022545705,  0],
      [:psi_1_int_e, [500], 0.0020020013333322666697, 0],
    #if GSL::RB_GSL_VERSION >= '1.5'
      [:psi_1_e, [1.0 / 32.0], 1025.5728544782377089,  0],
      [:psi_1_e, [1.0], 1.6449340668482264365,       0],
      [:psi_1_e, [5.0], 0.22132295573711532536,      0],
      [:psi_1_e, [100.0], 0.010050166663333571395,   0],
      [:psi_1_e, [110.0], 0.009132356622022545705,   0],
      [:psi_1_e, [500.0], 0.0020020013333322666697,  0],
      [:psi_1_e, [-1.0 - 1.0 / 128.0], 16386.648472598746587, 0],
      [:psi_1_e, [-1.50], 9.3792466449891237539, 0],
      [:psi_1_e, [-10.5], 9.7787577398148123845, 0],
      [:psi_1_e, [-15.5], 9.8071247184113896201, 0],
      [:psi_1_e, [-50.5], 9.8499971860824842274, 0],
      [:psi_1_e, [-1000.5], 9.8686054001734414233, 0],
    #end
      [:psi_n_e, [1, 1], 1.6449340668482264364,   0],
      [:psi_n_e, [1, 2], 0.64493406684822643647,  0],
      [:psi_n_e, [1, 3], 0.39493406684822643647,  0],
      [:psi_n_e, [1, 4], 0.28382295573711532536,  0],
      [:psi_n_e, [1, 5], 0.22132295573711532536,     0],
      [:psi_n_e, [1, 100], 0.010050166663333571395,  0],
      [:psi_n_e, [1, 110], 0.009132356622022545705,  0],
      [:psi_n_e, [1, 500], 0.0020020013333322666697, 0],
      [:psi_n_e, [3, 5.0], 0.021427828192755075022,     0],
      [:psi_n_e, [3, 500.0], 1.6048063999872000683e-08, 0],
      [:psi_n_e, [10, 5.0], -0.08675107579196581317,    1],
      [:psi_n_e, [10, 50.0], -4.101091112731268288e-12, 0],
      [:psi_n_e, [0, -1.5], 0.70315664064524318723,  0],
      [:psi_n_e, [1, -1.5], 9.3792466449891237539,   0],
      [:synchrotron_1_e, [0.01],  0.444972504114210632,    0],
      [:synchrotron_1_e, [1.0],   0.651422815355364504,    1],
      [:synchrotron_1_e, [10.0],  0.000192238264300868882, 1],
      [:synchrotron_1_e, [100.0], 4.69759366592220221e-43, 1],
      [:synchrotron_2_e, [0.01],  0.23098077342226277732, 2],
      [:synchrotron_2_e, [1.0],   0.4944750621042082670,  1],
      [:synchrotron_2_e, [10.0],  0.00018161187569530204281,  1],
      [:transport_2_e, [1.0e-10], 9.9999999999999999999e-11, 0],
      [:transport_2_e, [1.0],     0.97303256135517012845, 0],
      [:transport_2_e, [3.0],     2.41105004901695346199, 0],
      [:transport_2_e, [10.0],    3.28432911449795173575, 0],
      [:transport_2_e, [100.0],   3.28986813369645287294, 0],
      [:transport_2_e, [1.0e+05], 3.28986813369645287294, 0],
      [:transport_3_e, [1.0e-10], 4.999999999999999999997e-21, 0],
      [:transport_3_e, [1.0],     0.479841006572417499939, 0],
      [:transport_3_e, [3.0],     3.210604662942246772338, 0],
      [:transport_3_e, [5.0],     5.614386613842273228585, 0],
      [:transport_3_e, [10.0],    7.150322712008592975030, 0],
      [:transport_3_e, [30.0],    7.212341416160946511930, 0],
      [:transport_3_e, [100.0],   7.212341418957565712398, 0],
      [:transport_3_e, [1.0e+05], 7.212341418957565712398, 0],
      [:transport_4_e, [1.0e-10], 3.33333333333333333333e-31, 0],
      [:transport_4_e, [1.0e-07], 3.33333333333333166666e-22, 0],
      [:transport_4_e, [1.0e-04], 3.33333333166666666726e-13, 0],
      [:transport_4_e, [0.1], 0.000333166726172109903824, 0],
      [:transport_4_e, [1.0], 0.31724404523442648241, 0],
      [:transport_4_e, [3.0], 5.96482239737147652446, 0],
      [:transport_4_e, [5.0], 15.3597843168821829816, 0],
      [:transport_4_e, [10.0], 25.2736676770304417334, 0],
      [:transport_4_e, [30.0], 25.9757575220840937469, 0],
      [:transport_4_e, [100.0], 25.9757576090673165963, 1],
      [:transport_4_e, [1.0e+05], 25.9757576090673165963, 2],
      [:transport_5_e, [1.0e-10], 2.49999999999999999999e-41, 0],
      [:transport_5_e, [1.0e-07], 2.49999999999999861111e-29, 0],
      [:transport_5_e, [1.0e-04], 2.49999999861111111163e-17, 0],
      [:transport_5_e, [0.1], 0.000024986116317791487410, 0],
      [:transport_5_e, [1.0], 0.236615879239094789259153, 0],
      [:transport_5_e, [3.0], 12.77055769104415951115760, 0],
      [:transport_5_e, [5.0], 50.26309221817518778543615, 0],
      [:transport_5_e, [10.0], 116.3807454024207107698556, 0],
      [:transport_5_e, [30.0], 124.4313279083858954839911, 0],
      [:transport_5_e, [100.0], 124.4313306172043911597639, 0],
      [:transport_5_e, [1.0e+05], 124.43133061720439115976, 0],
      [:sin_e, [-10.0],       0.5440211108893698134,    0],
      [:sin_e, [1.0],         0.8414709848078965067,    0],
      [:sin_e, [1000.0],      0.8268795405320025603,    0],
      [:sin_e, [1048576.75],  0.8851545351115651914,    1],
      [:sin_e, [62831853.75], 0.6273955953485000827,    3],
      [:sin_e, [1073741822.5], -0.8284043541754465988,  TEST_SQRT_TOL0],
      [:sin_e, [1073741824.0], -0.6173264150460421708,  TEST_SQRT_TOL0],
      [:sin_e, [1073741825.5],  0.7410684679436226926,  TEST_SQRT_TOL0],
      [:sin_e, [1099511627776.0], -0.4057050115328287198, 32.0 * TEST_SQRT_TOL0],
      [:cos_e, [-10.0],      -0.8390715290764524523,    0],
      [:cos_e, [1.0],         0.5403023058681397174,    0],
      [:cos_e, [1000.0],      0.5623790762907029911,    1],
      [:cos_e, [1048576.75],  0.4652971620066351799,    2],
      [:cos_e, [62831853.75], 0.7787006914966116436,    2],
      [:cos_e, [1073741822.5],   -0.5601305436977716102,  TEST_SQRT_TOL0],
      [:cos_e, [1073741824.0],    0.7867071229411881196,  TEST_SQRT_TOL0],
      [:cos_e, [1099511627776.0], -0.9140040719915570023, 128.0 * TEST_SQRT_TOL0],
      [:sinc_e, [1.0 / 1024.0], 0.9999984312693665404, 0],
      [:sinc_e, [1.0 / 2.0], 2.0 / GSL::M_PI,              0],
      [:sinc_e, [80.5], 0.0039541600768172754, 0],
      [:sinc_e, [100.5], 0.0031672625490924445, 0],
      [:sinc_e, [1.0e+06 + 0.5], 3.18309727028927157e-07, 0],
#     [:sin_pi_x_e, [1000.5], 1.0, 0],
#     [:sin_pi_x_e, [10000.0 + 1.0 / 65536.0], 0.00004793689960306688455, 0],
#     [:sin_pi_x_e, [1099511627776.0 + 1 + 0.125], -0.3826834323650897717, 0],
      [:lnsinh_e, [0.1],  -2.3009189815304652235,  0],
      [:lnsinh_e, [1.0],   0.16143936157119563361, 0],
      [:lnsinh_e, [5.0],   4.306807418479684201,   0],
      [:lnsinh_e, [100.0], 99.30685281944005469,   0],
      [:lncosh_e, [0.125], 0.007792239318898252791, 0],
      [:lncosh_e, [1.0],   0.4337808304830271870,   0],
      [:lncosh_e, [5.0],   4.306898218339271555, 0],
      [:lncosh_e, [100.0], 99.30685281944005469, 0],
      [:zeta_int_e, [-61.0], -3.30660898765775767257e+34, 0],
      [:zeta_int_e, [-5.0],  -0.003968253968253968253968, 0],
      [:zeta_int_e, [ 5.0], 1.0369277551433699263313655, 0],
      [:zeta_int_e, [31.0], 1.0000000004656629065033784, 0],
      [:zetam1_int_e, [-61.0], -3.30660898765775767257e+34, 0],
      [:zetam1_int_e, [-5.0],  -1.003968253968253968253968, 0],
      [:zetam1_int_e, [ 5.0], 0.0369277551433699263313655, 0],
      [:zetam1_int_e, [31.0], 0.0000000004656629065033784, 0],
      [:zeta_e, [-151], 8.195215221831378294e+143, 2],
      [:zeta_e, [-51], 9.68995788746359406565e+24, 1],
      [:zeta_e, [-5], -0.003968253968253968253968, 1],
      [:zeta_e, [-0.5], -0.207886224977354566017307, 1],
      [:zeta_e, [-1e-10], -0.49999999990810614668948, 1],
      [:zeta_e, [0.0],    -0.5, 0],
      [:zeta_e, [1e-10],  -0.50000000009189385333058, 0],
      [:zeta_e, [0.5], -1.460354508809586812889499, 0],
      [:zeta_e, [1.0-1.0 / 1024.0], -1023.4228554489429787, 0],
      [:zeta_e, [1.0 + 1.0 / 1048576], 1.0485765772157343441e+06, 0],
      [:zeta_e, [5.0], 1.036927755143369926331365, 0],
      [:zeta_e, [25.5], 1.000000021074106110269959, 0],
      [:zetam1_e, [0.5], -2.460354508809586812889499, 0],
      [:zetam1_e, [2.0],  0.64493406684822643647,     1],
      [:zetam1_e, [3.0],  0.20205690315959428540,     1],
      [:zetam1_e, [5.0],  0.0369277551433699263314,   1],
      [:zetam1_e, [9.5],  0.0014125906121736622712,   1],
      [:zetam1_e, [10.5], 0.000700842641736155219500, 1],
      [:zetam1_e, [12.5], 0.000173751733643178193390, 1],
      [:zetam1_e, [13.5], 0.000086686727462338155188, 1],
      [:zetam1_e, [15.5], 0.000021619904246069108133, 1],
      [:zetam1_e, [16.5], 0.000010803124900178547671, 0],
      [:zetam1_e, [25.5], 0.000000021074106110269959, 0],
      [:hzeta_e, [2,  1.0],  1.6449340668482264365, 0],
      [:hzeta_e, [2, 10.0],  0.1051663356816857461, 0],
      [:hzeta_e, [5,  1.0],  1.0369277551433699263, 0],
      [:hzeta_e, [5, 10.0],  0.000030413798676470276, 0],
      [:hzeta_e, [9,  0.1],  1.0000000004253980e+09, 0],
      [:hzeta_e, [30, 0.5],  1.0737418240000053e+09, 0],
      [:hzeta_e, [30, 0.9],  2.3589824880264765e+01, 1],
      [:hzeta_e, [75, 0.25], 1.4272476927059599e+45, 1],
      [:eta_int_e, [-91], -4.945598888750002040e+94, 0],
      [:eta_int_e, [-51], -4.363969073121683116e+40, 0],
      [:eta_int_e, [-5], 0.25, 0],
      [:eta_int_e, [-1], 0.25, 0],
      [:eta_int_e, [ 0], 0.5, 0],
      [:eta_int_e, [ 5], 0.9721197704469093059, 0],
      [:eta_int_e, [ 6], 0.9855510912974351041, 0],
      [:eta_int_e, [ 20], 0.9999990466115815221, 0],
      [:eta_int_e, [ 1000], 1.0, 0],
      [:eta_e, [-51.5], -1.2524184036924703656e+41, 2],
      [:eta_e, [-5], 0.25, 0],
      [:eta_e, [0.5], 0.6048986434216303702, 0],
      [:eta_e, [0.999], 0.6929872789683383574, 0],
      [:eta_e, [1.0], 0.6931471805599453094, 0],
      [:eta_e, [1.0 + 1.0e-10], 0.6931471805759321998, 0],
      [:eta_e, [ 5], 0.9721197704469093059, 0],
      [:eta_e, [ 5.2], 0.9755278712546684682, 0],
      [:eta_e, [ 6], 0.9855510912974351041, 0],
      [:eta_e, [ 20], 0.9999990466115815221, 0]
    ]
  }.each { |k, v|
    define_method("test_#{k}") {
      GSL.set_error_handler_off if k == :sf
      v.each { |a| _test_sf(*a) }
      GSL.set_error_handler if k == :sf
    }
  }

end
