/****************************************************************************
**
** Copyright (C) 2016 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of Qbs.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 3 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL3 included in the
** packaging of this file. Please review the following information to
** ensure the GNU Lesser General Public License version 3 requirements
** will be met: https://www.gnu.org/licenses/lgpl-3.0.html.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 2.0 or (at your option) the GNU General
** Public license version 3 or any later version approved by the KDE Free
** Qt Foundation. The licenses are as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-2.0.html and
** https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/
#include "astpropertiesitemhandler.h"

#include <language/builtindeclarations.h>
#include <language/item.h>
#include <language/itempool.h>
#include <language/value.h>

#include <logging/translator.h>
#include <tools/error.h>
#include <tools/qbsassert.h>
#include <tools/stringconstants.h>

namespace qbs {
namespace Internal {

static const QString &autoGeneratedId()
{
    static const QString autoGenerated = QLatin1String("__autoGenerated");
    return autoGenerated;
}

ASTPropertiesItemHandler::ASTPropertiesItemHandler(
    Item *parentItem, ItemPool &itemPool, Logger &logger)
    : m_parentItem(parentItem)
    , m_itemPool(itemPool)
    , m_logger(logger)
{
    prepareGroup();
}

void ASTPropertiesItemHandler::handlePropertiesItems()
{
    // TODO: Simply forbid Properties items to have child items and get rid of this check.
    if (m_parentItem->type() != ItemType::Properties)
        setupAlternatives();
}

void ASTPropertiesItemHandler::prepareGroup()
{
    if (m_parentItem->type() != ItemType::Group)
        return;

    // Syntactic sugar: If we encounter property bindings of the form "product.x.y: z"
    // (or "module.x.y: z" for groups in modules), we transform them into "x.y: z"
    // and put them in a newly created Properties item, thus making sure they will be
    // applied "globally".
    Item::PropertyMap &props = m_parentItem->properties();
    for (auto it = props.begin(); it != props.end(); ++it) {
        if (it.value()->type() != Value::ItemValueType)
            continue;

        const bool hasProductPrefix = it.key() == StringConstants::productVar();
        const bool hasModulePrefix = !hasProductPrefix && it.key() == StringConstants::moduleVar();
        if (!hasProductPrefix && !hasModulePrefix)
            continue;

        Item * const valueItem = static_cast<ItemValue *>(it.value().get())->item();
        Item * const propertiesItem = m_itemPool.allocateItem(ItemType::Properties);
        propertiesItem->setFile(m_parentItem->file());
        propertiesItem->setProperties(valueItem->properties());
        propertiesItem->setupForBuiltinType(DeprecationWarningMode::Off, m_logger);
        propertiesItem->setId(autoGeneratedId());
        Item::addChild(m_parentItem, propertiesItem);

        if (hasModulePrefix) {
            Item *moduleItem = m_parentItem;
            while (moduleItem->type() == ItemType::Group)
                moduleItem = moduleItem->parent();
            if (moduleItem->type() != ItemType::Module) {
                const QString errMsg = Tr::tr("Use of '%1' is only allowed in %2 items.")
                                           .arg(
                                               it.key(),
                                               BuiltinDeclarations::instance().nameForType(
                                                   ItemType::Module));
                const CodeLocation loc = valueItem->properties().first()->location();
                throw ErrorInfo(errMsg, loc);
            }
        }

        props.erase(it);
        break;
    }
}

void ASTPropertiesItemHandler::setupAlternatives()
{
    auto it = m_parentItem->m_children.begin();
    while (it != m_parentItem->m_children.end()) {
        Item * const child = *it;
        bool remove = false;
        if (child->type() == ItemType::Properties) {
            handlePropertiesBlock(child);
            remove = m_parentItem->type() != ItemType::Export;
        }
        if (remove)
            it = m_parentItem->m_children.erase(it);
        else
            ++it;
    }
}

class PropertiesBlockConverter
{
public:
    PropertiesBlockConverter(
        const JSSourceValue::AltProperty &condition,
        const JSSourceValue::AltProperty &overrideListProperties,
        Item *propertiesBlockContainer,
        const Item *propertiesBlock,
        ItemPool &pool)
        : m_propertiesBlockContainer(propertiesBlockContainer)
        , m_propertiesBlock(propertiesBlock)
        , m_itemPool(pool)
        , m_autoGenerated(propertiesBlock->id() == autoGeneratedId())
    {
        m_alternative.condition = condition;
        m_alternative.overrideListProperties = overrideListProperties;
        while (m_targetItem->type() == ItemType::Group)
            m_targetItem = m_targetItem->parent();
    }

    void apply()
    {
        doApply(m_propertiesBlockContainer, m_propertiesBlock);
    }

private:
    JSSourceValue::Alternative m_alternative;
    Item * const m_propertiesBlockContainer;
    Item *m_targetItem = m_propertiesBlockContainer;
    const Item * const m_propertiesBlock;
    ItemPool &m_itemPool;
    const bool m_autoGenerated;

    void doApply(Item *outer, const Item *inner)
    {
        for (auto it = inner->properties().constBegin();
                it != inner->properties().constEnd(); ++it) {
            if (inner == m_propertiesBlock
                    && (it.key() == StringConstants::conditionProperty()
                        || it.key() == StringConstants::overrideListPropertiesProperty())) {
                continue;
            }
            if (it.value()->type() == Value::ItemValueType) {
                Item * const innerVal = std::static_pointer_cast<ItemValue>(it.value())->item();
                Item * const targetItem = outer == m_propertiesBlockContainer ? m_targetItem
                                                                              : outer;
                ItemValuePtr outerVal = targetItem->itemProperty(it.key(), m_itemPool);
                if (!outerVal) {
                    outerVal = ItemValue::create(Item::create(&m_itemPool, innerVal->type()),
                                                 true);
                    targetItem->setProperty(it.key(), outerVal);
                }
                doApply(outerVal->item(), innerVal);
            } else if (it.value()->type() == Value::JSSourceValueType) {
                Item * const targetItem = outer == m_propertiesBlockContainer && m_autoGenerated
                                              ? m_targetItem
                                              : outer;
                const ValuePtr outerVal = targetItem->property(it.key());
                if (Q_UNLIKELY(outerVal && outerVal->type() != Value::JSSourceValueType)) {
                    throw ErrorInfo(Tr::tr("Incompatible value type in unconditional value at %1.")
                                    .arg(outerVal->location().toString()));
                }
                const JSSourceValuePtr value = std::static_pointer_cast<JSSourceValue>(it.value());
                doApply(
                    it.key(), targetItem, std::static_pointer_cast<JSSourceValue>(outerVal), value);
                if ((outer != m_propertiesBlockContainer || m_autoGenerated)
                    && m_targetItem != m_propertiesBlockContainer) {
                    QBS_CHECK(m_propertiesBlockContainer->type() == ItemType::Group);
                    value->setScope(m_propertiesBlockContainer, {});
                }
            } else {
                QBS_CHECK(!"Unexpected value type in conditional value.");
            }
        }
    }

    void doApply(const QString &propertyName, Item *item, JSSourceValuePtr value,
               const JSSourceValuePtr &conditionalValue)
    {
        if (!value) {
            value = JSSourceValue::create(true);
            value->setFile(conditionalValue->file());
            item->setProperty(propertyName, value);
            value->setSourceCode(StringConstants::baseVar());
            value->setSourceUsesBase();
        }
        m_alternative.value = conditionalValue;
        if (m_alternative.condition.value == QLatin1String("undefined"))
            conditionalValue->setFallback();
        value->addAlternative(m_alternative);
    }
};

static JSSourceValue::AltProperty getPropertyData(const Item *propertiesItem, const QString &name)
{
    const ValuePtr value = propertiesItem->property(name);
    if (!value)
        return {StringConstants::falseValue(), propertiesItem->location()};
    if (Q_UNLIKELY(value->type() != Value::JSSourceValueType)) {
        throw ErrorInfo(Tr::tr("Properties.%1 must be a value binding.").arg(name),
                    propertiesItem->location());
    }
    if (name == StringConstants::overrideListPropertiesProperty()) {
        const Item *parent = propertiesItem->parent();
        while (parent) {
            if (parent->type() == ItemType::Product)
                break;
            parent = parent->parent();
        }
        if (!parent) {
            throw ErrorInfo(Tr::tr("Properties.overrideListProperties can only be set "
                                   "in a Product item."));
        }

    }
    const JSSourceValuePtr srcval = std::static_pointer_cast<JSSourceValue>(value);
    return {srcval->sourceCodeForEvaluation(), srcval->location()};
}

void ASTPropertiesItemHandler::handlePropertiesBlock(const Item *propertiesItem)
{
    const auto condition = getPropertyData(propertiesItem, StringConstants::conditionProperty());
    const auto overrideListProperties = getPropertyData(propertiesItem,
            StringConstants::overrideListPropertiesProperty());
    PropertiesBlockConverter(condition, overrideListProperties, m_parentItem,
                             propertiesItem, m_itemPool).apply();
}

} // namespace Internal
} // namespace qbs
