///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/OpenGLShader.h>
#include <core/viewport/Window3D.h>

namespace Core {

/******************************************************************************
* Constructor - The shader object will be added to the
* container's list of shader object.
******************************************************************************/
OpenGLShader::OpenGLShader(Window3D* container, const QString& name)
	: QObject(container), ogl(*container),
	programObject(0), vertexShaderObject(0), fragmentShaderObject(0),
	_isValid(false)
{
	setObjectName(name);
}


/******************************************************************************
* Loads the source files for the vertex and the fragment shader and compiles them.
* Throws an exception when an error occurs.
******************************************************************************/
void OpenGLShader::loadShader(const QString& vertexShaderPath, const QString& fragmentShaderPath)
{
	OVITO_ASSERT(vertexShaderObject == 0);
	OVITO_ASSERT(fragmentShaderObject == 0);
	OVITO_ASSERT(!vertexShaderPath.isEmpty() || !fragmentShaderPath.isEmpty());
	OVITO_ASSERT(ogl.hasShaderObjectsExtension());
	OVITO_ASSERT(ogl.hasVertexShaderExtension() || vertexShaderPath.isEmpty());
	OVITO_ASSERT(ogl.hasFragmentShaderExtension() || fragmentShaderPath.isEmpty());
	_isValid = false;

	// Notify user that we are using OpenGL shading language programs.
	static bool firstFunctionCall = true;
	if(firstFunctionCall) {
		VerboseLogger() << logdate << "Found OpenGL shading language version:" << (const char*)glGetString(GL_SHADING_LANGUAGE_VERSION) << endl;
		firstFunctionCall = false;
	}

	// Load vertex shader source file.
	if(!vertexShaderPath.isEmpty()) {
		//VerboseLogger() << logdate << "Loading and compiling OpenGL vertex shader source file:" << vertexShaderPath << endl;
		QFile vertexFile(vertexShaderPath);
		if(!vertexFile.open(QIODevice::ReadOnly|QIODevice::Text))
			throw Exception(tr("Could not open the vertex shader source file %1: %2").arg(vertexShaderPath, vertexFile.errorString()));
		QByteArray vertexShaderSource = vertexFile.readAll();
		if(vertexFile.error() != QFile::NoError)
			throw Exception(tr("Could not read the vertex shader source file %1: %2").arg(vertexShaderPath, vertexFile.errorString()));
		// Allocate shader object.
		CHECK_OPENGL(vertexShaderObject = ogl.glCreateShaderObjectARB(GL_VERTEX_SHADER_ARB));
		// Set shader source.
		const GLcharARB* sourcePointer = vertexShaderSource.constData();
		CHECK_OPENGL(ogl.glShaderSourceARB(vertexShaderObject, 1, &sourcePointer, NULL));
		// Compile shader.
		GLint compileStatus = 0;
		CHECK_OPENGL(ogl.glCompileShaderARB(vertexShaderObject));
		CHECK_OPENGL(ogl.glGetObjectParameterivARB(vertexShaderObject, GL_OBJECT_COMPILE_STATUS_ARB, &compileStatus));
		if(!compileStatus) {
			printInfoLog(vertexShaderObject);
			throw Exception(tr("The vertex shader source code %1 failed to compile. See log for details.").arg(vertexShaderPath));
		}
	}

	// Load fragment shader source file.
	if(!fragmentShaderPath.isEmpty()) {
		//VerboseLogger() << logdate << "Loading and compiling OpenGL fragment shader source file:" << fragmentShaderPath << endl;
		QFile fragmentFile(fragmentShaderPath);
		if(!fragmentFile.open(QIODevice::ReadOnly|QIODevice::Text))
			throw Exception(tr("Could not open the fragment shader source file %1: %2").arg(fragmentShaderPath, fragmentFile.errorString()));
		QByteArray fragmentShaderSource = fragmentFile.readAll();
		if(fragmentFile.error() != QFile::NoError)
			throw Exception(tr("Could not read the fragment shader source file %1: %2").arg(fragmentShaderPath, fragmentFile.errorString()));
		// Allocate shader object.
		CHECK_OPENGL(fragmentShaderObject = ogl.glCreateShaderObjectARB(GL_FRAGMENT_SHADER_ARB));
		// Set shader source.
		const GLcharARB* sourcePointer = fragmentShaderSource.constData();
		CHECK_OPENGL(ogl.glShaderSourceARB(fragmentShaderObject, 1, &sourcePointer, NULL));
		// Compile shader.
		GLint compileStatus = 0;
		CHECK_OPENGL(ogl.glCompileShaderARB(fragmentShaderObject));
		CHECK_OPENGL(ogl.glGetObjectParameterivARB(fragmentShaderObject, GL_OBJECT_COMPILE_STATUS_ARB, &compileStatus));
		if(!compileStatus) {
			printInfoLog(fragmentShaderObject);
			throw Exception(tr("The fragment shader source code %1 failed to compile. See log for details.").arg(fragmentShaderPath));
		}
	}

	// Allocate program object and populate it with the compiled shaders.
	CHECK_OPENGL(programObject = ogl.glCreateProgramObjectARB());
	if(vertexShaderObject)
		CHECK_OPENGL(ogl.glAttachObjectARB(programObject, vertexShaderObject));
	if(fragmentShaderObject)
		CHECK_OPENGL(ogl.glAttachObjectARB(programObject, fragmentShaderObject));

	// We want the shader object to go away as soon as it is detached
	// from the program object it is attached to. We can call delete now
	// to achieve that. Note that calling delete on a program object
	// will result in all shaders attached to that program object to be
	// detached. If delete has been called for the shader object,
	// calling delete on the program object will result in the shader
	// object being deleted as well.
	if(vertexShaderObject)
		CHECK_OPENGL(ogl.glDeleteObjectARB(vertexShaderObject));
	if(fragmentShaderObject)
		CHECK_OPENGL(ogl.glDeleteObjectARB(fragmentShaderObject));

	// Link
	GLint linkStatus = 0;
	CHECK_OPENGL(ogl.glLinkProgramARB(programObject));
	CHECK_OPENGL(ogl.glGetObjectParameterivARB(programObject, GL_OBJECT_LINK_STATUS_ARB, &linkStatus));
	if(!linkStatus) {
		printInfoLog(programObject);
		throw Exception(tr("The OpenGL shader program %1 failed to link. See log for details.").arg(objectName()));
	}

	// Everything's okay.
	_isValid = true;
}

/******************************************************************************
* Prints the compilation log to the console.
******************************************************************************/
void OpenGLShader::printInfoLog(GLhandleARB handle)
{
	GLint logLength;
	ogl.glGetObjectParameterivARB(handle, GL_OBJECT_INFO_LOG_LENGTH_ARB, &logLength);
	if(logLength > 1) {
		QByteArray buffer(logLength, ' ');
		GLint len;
		ogl.glGetInfoLogARB(handle, buffer.size(), &len, buffer.data());
		MsgLogger() << "OpenGL log:" << endl;
		MsgLogger() << buffer << endl;
	}
}

/******************************************************************************
* Activates or deactivates the shader.
******************************************************************************/
void OpenGLShader::setEnabled(bool on)
{
	if(!_isValid || !programObject) return;
	if(on) {
		CHECK_OPENGL(ogl.glUseProgramObjectARB(programObject));
	}
	else {
		ogl.glUseProgramObjectARB(0);
	}
}

/******************************************************************************
* Sets the value of a uniform variable in the shader program.
******************************************************************************/
void OpenGLShader::sendUniform1i(const char* name, int value)
{
	if(!_isValid || !programObject) return;

	CHECK_OPENGL(int location = ogl.glGetUniformLocationARB(programObject, name));
	OVITO_ASSERT_MSG(location >= 0, "OpenGLShader::sendUniform", QString("This uniform name '%1' is not defined in the shader program (%2).").arg(name, objectName()).toLocal8Bit().constData());
	OVITO_ASSERT_MSG(ogl.glGetHandleARB(GL_PROGRAM_OBJECT_ARB) == programObject, "OpenGLShader::sendUniform", "The shader has not been activated.");

	CHECK_OPENGL(ogl.glUniform1iARB(location, value));
}

/******************************************************************************
* Sets the value of a uniform variable in the shader program.
******************************************************************************/
void OpenGLShader::sendUniform1f(const char* name, float value)
{
	if(!_isValid || !programObject) return;

	OVITO_ASSERT_MSG(ogl.glGetHandleARB(GL_PROGRAM_OBJECT_ARB) == programObject, "OpenGLShader::sendUniform", "The shader has not been activated.");
	CHECK_OPENGL(int location = ogl.glGetUniformLocationARB(programObject, name));
	OVITO_ASSERT_MSG(location >= 0, "OpenGLShader::sendUniform", QString("This uniform name '%1' is not defined in the shader program (%2).").arg(name, objectName()).toLocal8Bit().constData());

	CHECK_OPENGL(ogl.glUniform1fARB(location, value));
}

/******************************************************************************
* Sets the value of a uniform variable in the shader program.
******************************************************************************/
void OpenGLShader::sendUniform2f(const char* name, float value1, float value2)
{
	if(!_isValid || !programObject) return;

	OVITO_ASSERT_MSG(ogl.glGetHandleARB(GL_PROGRAM_OBJECT_ARB) == programObject, "OpenGLShader::sendUniform", "The shader has not been activated.");
	CHECK_OPENGL(int location = ogl.glGetUniformLocationARB(programObject, name));
	OVITO_ASSERT_MSG(location >= 0, "OpenGLShader::sendUniform", QString("This uniform name '%1' is not defined in the shader program (%2).").arg(name, objectName()).toLocal8Bit().constData());

	CHECK_OPENGL(ogl.glUniform2fARB(location, value1, value2));
}

/******************************************************************************
* Sets the value of a uniform variable in the shader program.
******************************************************************************/
void OpenGLShader::sendUniform3f(const char* name, const Point3& value)
{
	if(!_isValid || !programObject) return;

	OVITO_ASSERT_MSG(ogl.glGetHandleARB(GL_PROGRAM_OBJECT_ARB) == programObject, "OpenGLShader::sendUniform", "The shader has not been activated.");
	CHECK_OPENGL(int location = ogl.glGetUniformLocationARB(programObject, name));
	OVITO_ASSERT_MSG(location >= 0, "OpenGLShader::sendUniform", QString("This uniform name '%1' is not defined in the shader program (%2).").arg(name, objectName()).toLocal8Bit().constData());

	CHECK_OPENGL(ogl.glUniform3fARB(location, (float)value.X, (float)value.Y, (float)value.Z));
}

/******************************************************************************
* Sets the value of a uniform variable in the shader program.
******************************************************************************/
void OpenGLShader::sendUniform3fv(const char* name, const float* array)
{
	if(!_isValid || !programObject) return;

	OVITO_ASSERT_MSG(ogl.glGetHandleARB(GL_PROGRAM_OBJECT_ARB) == programObject, "OpenGLShader::sendUniform", "The shader has not been activated.");
	CHECK_OPENGL(int location = ogl.glGetUniformLocationARB(programObject, name));
	OVITO_ASSERT_MSG(location >= 0, "OpenGLShader::sendUniform", QString("This uniform name '%1' is not defined in the shader program (%2).").arg(name, objectName()).toLocal8Bit().constData());

	CHECK_OPENGL(ogl.glUniform3fvARB(location, 1, array));
}

/******************************************************************************
* Returns storage location of an vertex attribute used by the vertex shader.
******************************************************************************/
GLint OpenGLShader::getAttribLocation(const char* name)
{
	OVITO_ASSERT_MSG(_isValid && ogl.glGetHandleARB(GL_PROGRAM_OBJECT_ARB) == programObject, "OpenGLShader::sendUniform", "The shader has not been activated.");

	GLint location;
	CHECK_OPENGL(location = ogl.glGetAttribLocationARB(programObject, name));
	OVITO_ASSERT_MSG(location >= 0, "OpenGLShader::getAttribLocation", QString("This attribute name '%1' is not defined in the shader program (%2).").arg(name, objectName()).toLocal8Bit().constData());

	return location;
}


};
