///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/RefTargetListParameterUI.h>
#include "AtomTypeDataChannel.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(AtomTypeDataChannel, DataChannel)
DEFINE_VECTOR_REFERENCE_FIELD(AtomTypeDataChannel, AtomType, "AtomTypes", _atomTypes)
SET_PROPERTY_FIELD_LABEL(AtomTypeDataChannel, _atomTypes, "Atom Types")

/******************************************************************************
* Serialization constructor.
******************************************************************************/
AtomTypeDataChannel::AtomTypeDataChannel(bool isLoading) : DataChannel(isLoading)
{
	INIT_PROPERTY_FIELD(AtomTypeDataChannel, _atomTypes);
}

/******************************************************************************
* Special constructor to create a standard data channel.
******************************************************************************/
AtomTypeDataChannel::AtomTypeDataChannel(DataChannelIdentifier which) : DataChannel(which)
{
	INIT_PROPERTY_FIELD(AtomTypeDataChannel, _atomTypes);
}

/******************************************************************************
* Creates an AtomType at the given index or returns the existing one.
******************************************************************************/
AtomType* AtomTypeDataChannel::createAtomType(int index)
{
	if(index < 0) index = atomTypes().size();

	OVITO_ASSERT_MSG(type() == qMetaTypeId<int>(), "AtomTypeDataChannel::createAtomType()", "The data channel should have the data type integer.");
	if(_atomTypes.size() > index && _atomTypes[index] != NULL)
		return _atomTypes[index];

	AtomType::SmartPtr type;
	{
		UndoSuspender undoSuspender;	// Do not create undo records for this.
		type = new AtomType();
		// Assign initial standard color to new atom type.
		static Color defaultAtomTypeColors[] = {
			Color(0.4f,1.0f,0.4f),
			Color(1.0f,0.4f,0.4f),
			Color(0.4f,0.4f,1.0f),
			Color(1.0f,1.0f,0.7f),
			Color(1.0f,1.0f,1.0f),
			Color(1.0f,1.0f,0.0f),
			Color(1.0f,0.4f,1.0f),
			Color(0.7f,0.0f,1.0f),
			Color(0.2f,1.0f,1.0f),
			Color(1.0f,0.4f,0.4f),
		};
		size_t colorIndex = index % (sizeof(defaultAtomTypeColors) / sizeof(defaultAtomTypeColors[0]));
		type->colorController()->setValue(0, defaultAtomTypeColors[colorIndex]);
		type->setName(tr("Atom type %1").arg(index));
	}

	if(index >= _atomTypes.size()) {
		while(index > _atomTypes.size())
			_atomTypes.push_back(NULL);
		_atomTypes.push_back(type);
	}
	else _atomTypes.set(index, type);

	return type.get();
}

/******************************************************************************
* Inserts an atom type into this object. The atom type
* must not be owned by another AtomsObject.
******************************************************************************/
void AtomTypeDataChannel::insertAtomType(const AtomType::SmartPtr& atype, int index)
{
	OVITO_ASSERT_MSG(type() == qMetaTypeId<int>(), "DataChannel::createAtomType()", "The data channel should be have the data type integer.");

	// Insert into array.
	_atomTypes.insert(index, atype);
}

/******************************************************************************
* Returns the first atom type with the given name or NULL if there is
* no such atom type with the given name.
******************************************************************************/
AtomType* AtomTypeDataChannel::findAtomTypeByName(const QString& name) const
{
	Q_FOREACH(AtomType* atype, atomTypes()) {
		if(atype && name == atype->name())
			return atype;
	}
	return NULL;
}

/******************************************************************************
* Returns the index of the first atom type with the given name or -1 if there is
* no such atom type with the given name.
******************************************************************************/
int AtomTypeDataChannel::findAtomTypeIndexByName(const QString& name) const
{
	for(int index = 0; index < atomTypes().size(); index++) {
		if(atomTypes()[index] && name == atomTypes()[index]->name())
			return index;
	}
	return -1;
}

/******************************************************************************
* Removed a single atom type from this object.
******************************************************************************/
void AtomTypeDataChannel::removeAtomType(AtomType* atomType)
{
	int index = _atomTypes.indexOf(atomType);
	if(index != -1)
		_atomTypes.remove(index);
}

IMPLEMENT_PLUGIN_CLASS(AtomTypeDataChannelEditor, PropertiesEditor)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void AtomTypeDataChannelEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Atom Type Data Channel"), rolloutParams, "atomviz.data_channels.atomtype.parameters");

    // Create the rollout contents.
	QVBoxLayout* layout = new QVBoxLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);

	// Atom types

	// Derive a custom class from the list parameter UI to display the atom type colors.
	class CustomRefTargetListParameterUI : public RefTargetListParameterUI {
	public:
		CustomRefTargetListParameterUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField, const RolloutInsertionParameters& rolloutParams)
			: RefTargetListParameterUI(parentEditor, refField, rolloutParams, PLUGINCLASSINFO(AtomTypeEditor)) {}
	protected:
		virtual QVariant getItemData(RefTarget* target, const QModelIndex& index, int role) {
			if(role == Qt::DecorationRole && target != NULL) {
				return (QColor)static_object_cast<AtomType>(target)->color();
			}
			else return RefTargetListParameterUI::getItemData(target, index, role);
		}
	};

	QWidget* subEditorContainer = new QWidget(rollout);
	QVBoxLayout* sublayout = new QVBoxLayout(subEditorContainer);
	sublayout->setContentsMargins(0,0,0,0);
	layout->addWidget(subEditorContainer);

	RefTargetListParameterUI* atomTypesListUI = new CustomRefTargetListParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomTypeDataChannel, _atomTypes), RolloutInsertionParameters().insertInto(subEditorContainer));
	layout->insertWidget(0, atomTypesListUI->listWidget());
}

};	// End of namespace AtomViz
