/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */

// Google Tests
#include <gtest/gtest.h>

// C++ Standard Library
#include <string>
#include <vector>

// Media Scanner Library
#include "mediascanner/glibutils.h"
#include "mediascanner/logging.h"
#include "mediascanner/taskmanager.h"

// Test Suite
#include "testlib/testutils.h"

namespace mediascanner {

// Standard Library
using std::string;

// Context specific logging domains
static const logging::Domain kDebug("debug/tests", logging::debug());

class WordPlay {
public:
    explicit WordPlay(const string &expected_words)
        : expected_words_(expected_words) {
        g_mutex_init(&mutex_);
        g_cond_init(&cond_);
    }

    string expected_words() const {
        return expected_words_;
    }

    string actual_words() const {
        return actual_words_;
    }

    void append_word(const string &word) {
        lock("append word");

        kDebug("Appending \"{1}\".") % word;

        if (actual_words_.length())
            actual_words_ += " ";

        actual_words_ += word;

        if (actual_words_.size() >= expected_words_.size())
            notify_unlocked("completion of word play");

        unlock();
    }

    void lock(const string &goal) {
        kDebug("Waiting for lock to {1}.") % goal;
        g_mutex_lock(&mutex_);
    }

    void unlock() {
        g_mutex_unlock(&mutex_);
    }

    void notify(const string &event) {
        lock("notify about " + event);
        notify_unlocked(event);
        unlock();
    }

    void notify_unlocked(const string &event) {
        kDebug("Sending notification about {1}.") % event;
        g_cond_signal(&cond_);
    }

    bool wait(const string &goal, double seconds) {
        lock("wait for " + goal);
        const bool signal_received = wait_unlocked(goal, seconds);
        unlock();
        return signal_received;
    }

    bool wait_unlocked(const string &goal, double seconds) {
        bool signal_received = true;

        if (seconds > 0) {
            const int64_t end_time =
                    g_get_monotonic_time() +
                    seconds * G_TIME_SPAN_SECOND;

            kDebug("Waiting up to {2,p3} s for {1}.") % goal % seconds;
            signal_received = g_cond_wait_until(&cond_, &mutex_, end_time);
        } else {
            kDebug("Waiting infinitely for {1}.") % goal;
            g_cond_wait(&cond_, &mutex_);
        }

        EXPECT_TRUE(signal_received) << ("...while waiting for " + goal);

        return signal_received;
    }

    bool notify_and_wait(const string &event, const string &goal,
                         double seconds) {
        lock("notify about " + event + ", and then wait for " + goal);
        notify_unlocked(event);

        const bool signal_received = wait_unlocked(goal, seconds);
        unlock();

        return signal_received;
    }

    TaskManager::TaskFunction bind_append_word(const string &word) {
        return std::bind(&WordPlay::append_word, this, word);
    }

    TaskManager::TaskFunction bind_wait(const string &goal, double seconds) {
        return std::bind(&WordPlay::wait, this, goal, seconds);
    }

    TaskManager::TaskFunction bind_notify_and_wait(const string &event,
                                                   const string &goal,
                                                   double seconds) {
        return std::bind(&WordPlay::notify_and_wait, this,
                           event, goal, seconds);
    }

private:
    const string expected_words_;
    string actual_words_;
    GMutex mutex_;
    GCond cond_;
};

TEST(TaskManagerTest, AppendTask) {
    WordPlay test("this are some famous last words");
    TaskManager manager(test_info_->name());

    test.lock("append tasks");
    manager.AppendTask(test.bind_notify_and_wait("task manager running",
                                                 "all tasks being queued", 5));
    test.wait_unlocked("task manager running", 1);

    manager.AppendTask(test.bind_append_word("famous"),  4);
    manager.AppendTask(test.bind_append_word("some"),   3);
    manager.AppendTask(test.bind_append_word("this"),   1);
    manager.AppendTask(test.bind_append_word("are"), 2);
    manager.AppendTask(test.bind_append_word("last"));
    manager.AppendTask(test.bind_append_word("words"));

    test.notify_unlocked("all tasks being queued");
    test.wait_unlocked("all tasks being run", 1);
    test.unlock();

    for (int i = 1; i < 5; ++i)
        manager.CancelByGroupId(i);

    ASSERT_EQ(test.expected_words(), test.actual_words());
}

TEST(TaskManagerTest, PrependTask) {
    WordPlay test("this are some famous last words");
    TaskManager manager(test_info_->name());

    test.lock("append tasks");
    manager.PrependTask(test.bind_notify_and_wait("task manager running",
                                                  "all tasks being queued", 5));
    test.wait_unlocked("task manager running", 1);

    manager.PrependTask(test.bind_append_word("famous"), 2);
    manager.PrependTask(test.bind_append_word("some"),   1);
    manager.PrependTask(test.bind_append_word("last"),   3);
    manager.PrependTask(test.bind_append_word("words"),  4);
    manager.PrependTask(test.bind_append_word("are"));
    manager.PrependTask(test.bind_append_word("this"));

    test.notify_unlocked("all tasks being queued");
    test.wait_unlocked("all tasks being run", 1);
    test.unlock();

    for (int i = 1; i < 5; ++i)
        manager.CancelByGroupId(i);

    ASSERT_EQ(test.expected_words(), test.actual_words());
}

TEST(TaskManagerTest, CancelTask) {
    WordPlay test("this is good");
    TaskManager manager(test_info_->name());

    test.lock("append tasks");
    manager.AppendTask(test.bind_notify_and_wait("task manager running",
                                                 "all tasks being queued", 5));
    test.wait_unlocked("task manager running", 1);

    manager.AppendTask(test.bind_append_word("this"), 1);
    manager.AppendTask(test.bind_append_word("is"),   2);
    manager.AppendTask(test.bind_append_word("not"),  3);
    manager.AppendTask(test.bind_append_word("good"), 4);
    manager.CancelByGroupId(3);

    test.notify_unlocked("all tasks being queued");
    test.wait_unlocked("all tasks being run", 1);
    test.unlock();

    for (int i = 1; i < 5; ++i)
        manager.CancelByGroupId(i);

    ASSERT_EQ(test.expected_words(), test.actual_words());
}

} // namespace mediascanner

int main(int argc, char *argv[]) {
    mediascanner::InitTests(&argc, argv);
    return RUN_ALL_TESTS();
}
