/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/dbusservice.h"

// C++ Standard Library
#include <string>

// Media Scanner Library
#include "mediascanner/dbustypes.h"

namespace mediascanner {
namespace dbus {

void MediaScannerProxy::connect(Wrapper<GCancellable> cancellable) {
    InterfaceProxy::connect
            (G_BUS_TYPE_SESSION, MediaScannerSkeleton::service_name(),
             MediaScannerSkeleton::object_path(), cancellable);
}

void MediaScannerProxy::connect(Wrapper<GDBusConnection> connection,
                                Wrapper<GCancellable> cancellable) {
    InterfaceProxy::connect
            (connection, MediaScannerSkeleton::service_name(),
             MediaScannerSkeleton::object_path(), cancellable);
}

bool MediaScannerProxy::ConnectAndWait(Wrapper<GCancellable> cancellable,
                                       GError **error) {
    const char *const address = g_getenv("DBUS_SESSION_BUS_ADDRESS");
    Wrapper<GDBusConnection> connection;

    if (address) {
        connection = take(g_dbus_connection_new_for_address_sync
                (address, G_DBUS_CONNECTION_FLAGS_MESSAGE_BUS_CONNECTION |
                 G_DBUS_CONNECTION_FLAGS_AUTHENTICATION_CLIENT, nullptr,
                 nullptr, error));
    } else {
        connection =
            take(g_bus_get_sync(G_BUS_TYPE_SESSION, cancellable.get(), error));
    }

    return connection && ConnectAndWait(connection, cancellable, error);
}

bool MediaScannerProxy::ConnectAndWait(Wrapper<GDBusConnection> connection,
                                       Wrapper<GCancellable> cancellable,
                                       GError **error) {
    return InterfaceProxy::ConnectAndWait
            (connection, MediaScannerSkeleton::service_name(),
             MediaScannerSkeleton::object_path(), cancellable, error);
}

const std::string& MediaScannerSkeleton::object_path() {
    static const std::string object_path = "/com/canonical/MediaScanner";
    return object_path;
}

const std::string& MediaScannerSkeleton::service_name() {
    static const std::string service_name = "com.canonical.MediaScanner";
    return service_name;
}

template<>
const Signature& Type<MediaChangeType>::signature() {
    return Type<uint32_t>::signature();
}

template<>
GVariant* Type<MediaChangeType>::make_variant(MediaChangeType value) {
    return Type<uint32_t>::make_variant(static_cast<uint32_t>(value));
}

template<>
MediaChangeType Type<MediaChangeType>::make_value(GVariant *variant) {
    return static_cast<MediaChangeType>(Type<uint32_t>::make_value(variant));
}

std::string to_string(MediaChangeType change_type) {
    switch (change_type) {
    case MEDIA_INFO_CREATED:
        return "media-info-created";
    case MEDIA_INFO_UPDATED:
        return "media-info-updated";
    case MEDIA_INFO_REMOVED:
        return "media-info-removed";
    }

    std::ostringstream oss;
    oss << "unknown-media-change-" << change_type;
    return oss.str();
}

} // namespace dbus
} // namespace mediascanner

