/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_localaccount.mod localaccount plugin

    \section localaccountsection1 SYNOPSIS
    \b lcmaps_localaccount.mod
        [-gridmapfile|-GRIDMAPFILE|-gridmap|-GRIDMAP \<location gridmapfile\>]

    \section localaccountsection2 DESCRIPTION
 
    This plugin is an Acquisition Plugin and will provide the LCMAPS system with Local Account
    credential information.
    To do this it needs to look up the Distinghuished Name (DN) from a user's certificate in the gridmapfile.
    If this DN is found in the gridmapfile the plugin knows the mapped local (system) account username.
    By knowing the username of the local account the plugin can gather additional information about this account.
    The plugin will resolve the UID, GID and all the secondary GIDs.
    When this has been done and there weren't any problems detected,
    the plugin will add this information to a datastructure in the Plugin Manager.
    The plugin will finish its run with a LCMAPS_MOD_SUCCESS.
    This result will be reported to the Plugin Manager which started this plugin and
    it will forward this result to the Evaluation Manager, which will take appropriate actions
    for the next plugin to run.
    Normally this plugin would be followed by an Enforcement plugin that can apply these gathered
    credentials in a way that is appropriate to a system administration's needs.
                                      

    \section localaccountoptions OPTIONS
    \subsection localaccountoptie1 -GRIDMAPFILE <gridmapfile>
        See \ref localaccountoptie4 "-gridmap"

    \subsection localaccountoptie2 -gridmapfile <gridmapfile>
        See \ref localaccountoptie4 "-gridmap"

    \subsection localaccountoptie3 -GRIDMAP <gridmapfile>
        See \ref localaccountoptie4 "-gridmap"

    \subsection localaccountoptie4 -gridmap <gridmapfile>
        When this option is set it will override the default path of the gridmapfile.
        It is advised to use an absolute path to the gridmapfile to avoid usage of the wrong file(path).

\section localaccountReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure


\section localaccountErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/) 

\section localaccountSeeAlso SEE ALSO
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_poolaccount.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod"
*/

/*!
    \file   lcmaps_localaccount.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code for localaccount plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

/* Define both _XOPEN_SOURCE for strdup and _GNU_SOURCE, this makes Solaris
 * happier */
#define _XOPEN_SOURCE	500
/* Try to provide RTLD_DEFAULT */
/* MacOS needs no POSIX or _DARWIN_C_SOURCE to have RTLD_DEFAULT */
#ifdef __APPLE__
# define _DARWIN_C_SOURCE
#else
# define _GNU_SOURCE
#endif
#include <dlfcn.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>

#include "lcmaps_plugins_basic_config.h"

#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>

#if defined(HAVE_LCMAPS_LCMAPS_PLUGIN_PROTOTYPES_H)
#   include <lcmaps/lcmaps_plugin_prototypes.h>
#else
#   include "lcmaps_plugin_prototypes.h"
#endif

#include "lcmaps_gridmapfile.h"


/************************************************************************
 * defines
 ************************************************************************/

#define PLUGIN_PREFIX	"lcmaps_localaccount"

#define PLUGIN_RUN	0   /* full run mode */
#define PLUGIN_VERIFY	1   /* verify-only mode */


/************************************************************************
 * global variables
 ************************************************************************/

static char *gridmapfile = NULL; /* filename of gridmapfile */


/************************************************************************
 * private prototypes
 ************************************************************************/

/* called by plugin_run() and plugin_verify() */
static int plugin_run_or_verify(int argc, lcmaps_argument_t *argv,
				int lcmaps_mode);


/************************************************************************
 * public functions
 ************************************************************************/

/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(int argc, char **argv) {
    const char * logstr = PLUGIN_PREFIX"-plugin_initialize()";
    int i;

    /* Log commandline parameters on debug */
    lcmaps_log(LOG_DEBUG,"%s: passed arguments:\n",logstr);
    for (i=0; i < argc; i++)
	lcmaps_log(LOG_DEBUG,"%s: arg %d is %s\n", logstr, i, argv[i]);

    /* Parse arguments, argv[0] = name of plugin, so start with i = 1 */
    for (i = 1; i < argc; i++) {
	/* check grid-mapfile option (or similar variations) */
        if ( strcmp(argv[i], "-gridmapfile") == 0 ||
             strcmp(argv[i], "-GRIDMAPFILE") == 0 ||
             strcmp(argv[i], "-gridmap") == 0 ||
             strcmp(argv[i], "-GRIDMAP") == 0 )
	{
	    /* check valid filename argument */
            if (argv[i + 1] == NULL || argv[i + 1][0]=='\0') {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by valid filename\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* free existing one and copy new one */
	    free(gridmapfile); gridmapfile=NULL;
	    if (argv[i+1][0]=='/')  { /* absolute path */
		if ( (gridmapfile = strdup(argv[i + 1])) == NULL)	{
		    lcmaps_log(LOG_ERR, "%s: out of memory\n", logstr);
		    goto fail_init_module;
		}
	    } else { /* relative path */
		if (lcmaps_get_prefixed_file(argv[i + 1], &gridmapfile)==-1)
		    goto fail_init_module;
	    }
	    /* log the name and increase arg counter */
	    lcmaps_log(LOG_DEBUG, "%s: Using grid-mapfile \"%s\".\n",
		    logstr, gridmapfile);
            i++;
        }
	/* any other argument is an error */
	else
	{
            lcmaps_log(LOG_ERR,
		    "%s: Unknown argument for plugin: %s (failure)\n",
		    logstr, argv[i]);
            goto fail_init_module;
        }
    }

    /* Check we have a grid-mapfile */
    if (gridmapfile==NULL || gridmapfile[0]=='\0')  {
	free(gridmapfile); gridmapfile=NULL;
        lcmaps_log(LOG_INFO,
	    "%s: No grid-mapfile was provided, will use default.\n", logstr);
    }

    return LCMAPS_MOD_SUCCESS;

fail_init_module:
    free(gridmapfile);
    gridmapfile = NULL;

    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(int *argc, lcmaps_argument_t **argv) {
    const char * logstr = PLUGIN_PREFIX"-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
	{"user_dn","char *", 1,NULL},
	{NULL	  ,NULL	   ,-1,NULL},
	{NULL     ,NULL	   ,-1,NULL}
    };

    /* Get the version of LCMAPS being used: we need at least 1.5.8 to be able
     * to demand "requested_username" in the argList */
    int major=0,minor=0,patch=0;
    /* Most UNIX now support RTLD_DEFAULT (POSIX reserved) */
#ifdef RTLD_DEFAULT
    int (*lcmaps_major)(void),(*lcmaps_minor)(void),(*lcmaps_patch)(void);
    dlerror();
    lcmaps_major=dlsym(RTLD_DEFAULT,"lcmaps_get_major_version");
    lcmaps_minor=dlsym(RTLD_DEFAULT,"lcmaps_get_minor_version");
    lcmaps_patch=dlsym(RTLD_DEFAULT,"lcmaps_get_patch_version");
    if (dlerror()==NULL)    {
	major=lcmaps_major();
	minor=lcmaps_minor();
	patch=lcmaps_patch();
    }
#else
    /* No RTLD_DEFAULT, just hope the symbol exists in LCMAPS */
    major=lcmaps_get_major_version();
    minor=lcmaps_get_minor_version();
    patch=lcmaps_get_patch_version();
#endif

    /* Too old when older than 1.5.8 */
    if (major<1 || (major==1 && (minor<5 || (minor==5 && patch<8))))	{
	lcmaps_log(LOG_DEBUG,
	    "%s: Old LCMAPS found (%d.%d.%d), not using requested_username\n",
	    logstr,major,minor,patch);
    } else {
	lcmaps_log(LOG_DEBUG,
	    "%s LCMAPS (%d.%d.%d) supports using requested_username\n",
	    logstr,major,minor,patch);
	argList[1].argName="requested_username";
	argList[1].argType="char *";
	argList[1].argInOut=1;
	argList[1].value=NULL;
    }

    lcmaps_log(LOG_DEBUG,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log(LOG_DEBUG,"%s: address first argument: %p\n",
	    logstr, (void*)argList);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(int argc, lcmaps_argument_t *argv) {
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().
    In this particular case "plugin_verify()" is identical to "plugin_run()"

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(int argc, lcmaps_argument_t *argv) {
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate(void) {
    const char * logstr = PLUGIN_PREFIX"-plugin_terminate()";

    lcmaps_log(LOG_DEBUG,"%s: terminating\n", logstr);

    free(gridmapfile);
    gridmapfile=NULL;

    return LCMAPS_MOD_SUCCESS;
}


/************************************************************************
 * private functions
 ************************************************************************/

/**
 * Actual run/verify function. Called by both plugin_run and plugin_verify
 * with different lcmaps_mode.
 */
static int plugin_run_or_verify(int argc, lcmaps_argument_t *argv,
				int lcmaps_mode) {
    const char *        logstr       = NULL;
    const char *        logmapfile   = gridmapfile ? gridmapfile
						   : "default grid-mapfile";
    void *              value        = NULL;
    char **             user_dn_list = NULL;
    char *              dn           = NULL;
    int                 dn_cnt       = 0;
    char *              req_username = NULL;
    int                 req_username_needs_free=0;
    uid_t               req_uid      = (uid_t)(-1);
    struct passwd *     pw           = NULL;
    unsigned short      options      = 0;
    char *              searchstr    = NULL;
    int                 rc           = 0;
    char *              username     = NULL;
    struct passwd       *user_info   = NULL;
    int                 cnt_sec_gid  = 0;
    gid_t *             sec_gid      = NULL;
    int                 i            = 0;

    /* Set suitable logstr */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = PLUGIN_PREFIX"-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = PLUGIN_PREFIX"-plugin_verify()";
    else {
        lcmaps_log(LOG_ERR, PLUGIN_PREFIX"-plugin_run_or_verify(): "
		"attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_plugin;
    }

    /* Try to get DN from LCMAPS */
    /* First try to obtain DN from the credential data (i.e. stored by other
     * plugins */
    user_dn_list = getCredentialData(DN, &dn_cnt);
    if (dn_cnt>0)   {
	/* Log already registered DNs */
	for (i=0; i<dn_cnt; i++)
	    lcmaps_log(LOG_DEBUG,"%s: found registered DN[%d/%d]: %s\n",
		    logstr, i+1, dn_cnt, user_dn_list[i]);
	dn=user_dn_list[0];
    } else {
	/* No DNs are registered, use the introspect/run arguments */
	value=lcmaps_getArgValue("user_dn", "char *", argc, argv);
	if (value == NULL || (dn = *(char **)value) == NULL ) {
	    lcmaps_log(LOG_WARNING,"%s: could not get value of dn !\n", logstr);
	    return -1;
	}
	
	/* push it to the end-result registry */
	lcmaps_log(LOG_DEBUG, "%s: Adding DN to LCMAPS framework: %s\n",
		logstr, dn);
	addCredentialData(DN, &dn);
    }

    /* Log the found DN */
    lcmaps_log(LOG_DEBUG,"%s: found dn: %s\n", logstr, dn);

    /* In verify mode, requested account is typically in requested_uid, in run
     * mode, it can be provided as requested_username. Let's just see if either
     * is set. */

    /* Get requested_username value */
    value = lcmaps_getArgValue("requested_username", "char *", argc, argv);
    if ( value != NULL && *(char **)value !=NULL ) {
	/* copy and log resulting account name */
	req_username=*(char **)value;
	lcmaps_log(LOG_DEBUG,"%s: the requested user is %s\n",
		logstr, req_username);
    } else { /* No (valid) requested_username, try requested_uid */
	/* Get requested_uid value */
	value = lcmaps_getArgValue("requested_uid", "uid_t", argc, argv);
	if ( value!=NULL && *(int *)value != -1)  { /* undefined value -> -1 */
	    req_uid = *(uid_t *)value;
	    /* Basic sanity check */
	    if (req_uid == 0) {
		lcmaps_log(LOG_ERR,
			"%s: illegal request for uid == 0 (failure)\n", logstr);
		goto fail_plugin;
	    }
	    /* Get passwd info */
	    if ( (pw = getpwuid(req_uid )) == NULL ||
		 pw->pw_name == NULL )
	    {
		lcmaps_log(LOG_WARNING,
			"%s: the requested uid %d is illegal.\n",
			logstr, (int) req_uid);
		goto fail_plugin;
	    }
	    /* copy resulting account name */
	    if ( (req_username=strdup(pw->pw_name))==NULL ) {
		lcmaps_log(LOG_ERR,"%s: out of memory\n",logstr);
		goto fail_plugin;
	    }
	    /* keep track whether it needs free */
	    req_username_needs_free=1;
	    /* Log the result */
	    lcmaps_log(LOG_DEBUG,"%s: the requested user is %s(%d)\n",
		    logstr, req_username, (int)req_uid);
	} else if (lcmaps_mode == PLUGIN_VERIFY)    {
	    lcmaps_log(LOG_WARNING,
		    "%s: both requested_username and requested_uid are unset, "
		    "need at least one in PLUGIN_VERIFY mode\n", logstr);
	    goto fail_plugin;
	}
    }

    /* Set the matching options */
    if (req_username)	{
	options = MATCH_EXACT;
	searchstr=req_username;
    } else {
	options = 0;
	searchstr=".";
    }

    /* Do lookup for DN in grid-mapfile */
    rc = lcmaps_gridmapfile(gridmapfile, dn, searchstr, options, &username);

    /* parse return value of lcmaps_gridmapfile */
    if (rc==-1)
	/* error */
	goto fail_plugin;
    if (rc==0 && username==NULL)    { /* default also returns rc==0 */
	/* no match, this should be at most a NOTICE */
	if (req_username)
	    lcmaps_log(LOG_NOTICE,
		"%s: No match for requested localaccount %s for user \"%s\", "
		"in %s\n", logstr, req_username, dn, logmapfile);
	else
	    lcmaps_log(LOG_NOTICE,
		"%s: No match for a localaccount for user \"%s\", "
		"in %s\n", logstr, dn, logmapfile);
	goto fail_plugin;
    }

    /* found match: log */
    lcmaps_log(LOG_DEBUG,"%s: found %susername: %s\n",
	    logstr, req_username ? "requested " : "", username);

    /* Convert username to uid, pgid and sgids */

    /* Get account info for found username */
    if ( (user_info = getpwnam(username)) == NULL ) {
	lcmaps_log(LOG_WARNING,
		"%s: no user account found with the name \"%s\"\n",
		logstr, username);
	goto fail_plugin;
    }

    /* Log resulting account */
    lcmaps_log(LOG_DEBUG,"%s: username : %s\n", logstr, user_info->pw_name);
    lcmaps_log(LOG_DEBUG,"%s: user_id  : %d\n", logstr, user_info->pw_uid);
    lcmaps_log(LOG_DEBUG,"%s: group_id : %d\n", logstr, user_info->pw_gid);
    lcmaps_log(LOG_DEBUG,"%s: home dir : %s\n", logstr, user_info->pw_dir);

    /* Add this credential data to the credential data repository in the plugin
     * manager */
    addCredentialData(UID, &(user_info->pw_uid));
    addCredentialData(PRI_GID, &(user_info->pw_gid));

    /* Retrieve secondary group id's */
    if (lcmaps_get_gidlist(username, &cnt_sec_gid, &sec_gid)==0) {
	for (i = 0; i < cnt_sec_gid; i++)
	    addCredentialData(SEC_GID, &(sec_gid[i]));
	free(sec_gid);
    }

    /* success */
    if (req_username_needs_free)
	free(req_username);
    free(username);

    lcmaps_log(LOG_INFO,"%s: localaccount plugin succeeded\n", logstr);

    return LCMAPS_MOD_SUCCESS;

fail_plugin:
    if (req_username_needs_free)
	free(req_username);
    free(username);

    lcmaps_log(LOG_INFO,"%s: localaccount plugin failed\n", logstr);

    return LCMAPS_MOD_FAIL;
}
