#ifndef MULTIREADER_H
#define MULTIREADER_H

#include <Exception.h>
#include <vector>
#include <map>

/// Read data from multiple filehandles
class MultiReader
{
protected:
	struct fddata
	{
		int fd;
		int old_flags;
		fddata(int fd, int old_flags) throw () : fd(fd), old_flags(old_flags) {}
	};
	std::vector<fddata> fds;

	virtual void handleData(int fd, char* buf, unsigned int size) = 0;
	virtual void handleEOF(int fd) = 0;

public:
	virtual ~MultiReader() {}

	/// Add another fd and listener to the listening set
	void addFD(int fd) throw (SystemException);

	/// Do the read loop
	void readLoop() throw (SystemException, InterruptedException);
};

class LineMultiReader : public MultiReader
{
public:
	/// Interface for data listeners
	class Listener
	{
	public:
		/// Notify that there is a new line available from `fd'.  Note: newline
		/// is stripped away from `line'.
		virtual void haveLine(int fd, const std::string& line) throw () = 0;

		/// Notify the end-of-file condition on `fd'
		virtual void haveEOF(int fd) throw () {}
	};

protected:
	std::string linebuf;
	std::map<int, Listener*> listeners;
	
	virtual void handleData(int fd, char* buf, unsigned int size) throw ();
	virtual void handleEOF(int fd) throw ();

public:
	void addFD(int fd, Listener* l) throw (SystemException)
	{
		MultiReader::addFD(fd);
		listeners.insert(std::pair<int, Listener*>(fd, l));
	}
};

// vim:set ts=4 sw=4:
#endif
