package latexDraw.ui.components;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;

import latexDraw.util.LaTeXDrawPoint2D;

/**
 * A delimiter delimits figures in the draw panel.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 * <br>
 * 06/26/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class Delimitor implements Serializable, Cloneable
{
	private static final long serialVersionUID = 1L;
	
	public static final int DEFAULT_OPACITY = 130;

	/** The coordinates of the centre of the delimiter */
	protected LaTeXDrawPoint2D center;
	
	/** The thickness of the delimiters */
	public static final int THICKNESS = 2;
	
	/** The dimension of the delimiter */
	protected double dim  = 8.;
	
	/** The opacity of the delimiters. Can be changed. */
	private static int opacity = DEFAULT_OPACITY;
	
	/** A possible colour for the interior of the delimiter */
	public static final Color INTERIOR  = new Color(153, 153, 204, opacity);	
	
	/** A possible colour for the interior of the delimiter */
	public static final Color INTERIOR_2  = new Color(204, 133, 133, opacity);	
		
	/** A possible colour for the interior of the delimiter */
	public static final Color INTERIOR_3  = new Color(150, 98, 144, opacity);	
	
	/** A possible colour for the interior of the delimiter */
	public static final Color INTERIOR_4  = new Color(60, 160, 115, opacity);	

	private static final BasicStroke STROKE = new BasicStroke(2, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_MITER);
	
	
	/** The colour of the interior of the delimiter */
	private Color interior;

	
	/**
	 * The constructor by default
	 */
	public Delimitor()
	{
		this(new LaTeXDrawPoint2D());
	}
	
	
	
	/**
	 * The constructor
	 * @param p The location of the delimiter
	 */
	public Delimitor(LaTeXDrawPoint2D p)
	{
		center = p==null ? new LaTeXDrawPoint2D() : p;
		interior = INTERIOR;
	}
	
	
	
	
	/**
	 * Allows to change the centre
	 * @param c The new centre
	 */
	public synchronized  void setCenter(LaTeXDrawPoint2D c)
	{
		if(c==null)
			throw new IllegalArgumentException();
		
		center = c;			
	}
	
	
	
	/**
	 * Allows to change to change the colours of the delimiter
	 */
	public synchronized void setColorSet1()
	{
		interior = new Color(INTERIOR.getRed(), INTERIOR.getGreen(), INTERIOR.getBlue(), opacity);
	}
	
	
	/**
	 * Allows to change to change the colours of the delimiter
	 */
	public synchronized void setColorSet2()
	{
		interior = new Color(INTERIOR_2.getRed(), INTERIOR_2.getGreen(), INTERIOR_2.getBlue(), opacity);
	}
	
	
	
	/**
	 * Allows to change to change the colours of the delimiter
	 */
	public synchronized void setColorSet3()
	{
		interior = new Color(INTERIOR_3.getRed(), INTERIOR_3.getGreen(), INTERIOR_3.getBlue(), opacity);
	}
	

	
	/**
	 * Allows to change to change the colours of the delimiter
	 */
	public synchronized void setColorSet4()
	{
		interior = new Color(INTERIOR_4.getRed(), INTERIOR_4.getGreen(), INTERIOR_4.getBlue(), opacity);
	}
	
	
	
	
	/**
	 * Allows to set the width of the delimiter
	 * @param dim Its new width
	 */
	public synchronized void setDim(double dim)
	{
		if(dim<=0)
			throw new IllegalArgumentException();
		
		this.dim = dim;
	}

	
	
	/**
	 * Allows to get the X-coordinate of the delimiter
	 * @return The X-coordinate
	 */
	public double getX()
	{
		return center.getX();
	}
	
	
	
	
	/**
	 * Allows to access to the centre of the delimiter
	 * @return The centre of the delimiter
	 */
	public synchronized LaTeXDrawPoint2D getCenter()
	{
		return center;
	}
	
	
	
	/**
	 * Allows to get the Y-coordinate of the delimiter
	 * @return The Y-coordinate
	 */
	public double getY()
	{		
		return center.getY();
	}
	
	
	

	/**
	 * This method allows to draw the delimiter.
	 * @param g The Graphics of the panel where the delimiter must be drawn
	 */
	public void draw(Graphics2D g)
	{
		double demiDim = dim/2.;
		
		if(opacity!=interior.getTransparency())
			interior = new Color(interior.getRed(), interior.getGreen(),interior.getBlue(), opacity);
		
		g.setStroke(STROKE);
		g.setColor(interior);
		g.fill(new Rectangle2D.Double(center.x-demiDim, center.y-demiDim, dim, dim));
	}
	
	
	
	/**
	 * Allows the change the coordinate of the delimiter
	 * @param p The new centre of the delimiter
	 */	
	public synchronized void setCoordinates(LaTeXDrawPoint2D p)
	{
		if(p==null)
			throw new IllegalArgumentException();
		
		setCoordinates(p.x, p.y);
	}
	
	
	
	
	/**
	 * Allows the change the coordinate of the delimiter
	 * @param x The X-coordinate of the new position
	 * @param y The Y-coordinate of the new position
	 */
	public synchronized void setCoordinates(double x, double y) 
	{
		center.setLocation(x, y);		
	}
	
	
	
	
	/**
	 * Allows to know if the point pt is in the delimiter
	 * @param pt The point which we want to know if it's in the delimiter
	 * @return true if it is in the delimiter and else false
	 */
	public boolean isIn(LaTeXDrawPoint2D pt)
	{
		if(pt==null)
			return false;
		
		return pt.x>=center.x-dim/2. && pt.x<=center.x+dim/2. 
			&& pt.y>=center.y-dim/2. && pt.y<=center.y+dim/2.;
	}
	
	

	
	/**
	 * Allows to know if the point pt is in the delimiter
	 * @param pt The point which we want to know if it's in the delimiter
	 * @return true if it is in the delimiter and else false
	 */
	public boolean isIn(Point pt)
	{
		if(pt==null)
			return false;
		
		return isIn(new LaTeXDrawPoint2D(pt.x, pt.y));
	}
	
	
	
	@Override
	public Object clone() throws CloneNotSupportedException
	{
		Delimitor d =(Delimitor) super.clone();
		d.center = (LaTeXDrawPoint2D)center.clone();
		d.dim = dim;
		d.interior = interior;
		
		return d;
	}
	
	
	
	private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException
	{
		ois.readObject();// The two other colours that are no more used.
		ois.readObject();
		interior = (Color) ois.readObject();
		center = (LaTeXDrawPoint2D) ois.readObject();
		dim = ois.readDouble();
	}



	/**
	 * @return the dim
	 */
	public synchronized double getDim()
	{
		return dim;
	}



	/**
	 * @return the opacity.
	 */
	public static synchronized int getOpacity()
	{
		return opacity;
	}



	/**
	 * @param opacity the opacity to set.
	 */
	public static void setOpacity(int opacity)
	{
		if(opacity<0 || opacity>255)
			throw new IllegalArgumentException();
		
		Delimitor.opacity = opacity;
	}



	/**
	 * @return the interior.
	 * @since 2.0.0
	 */
	public synchronized Color getInterior()
	{
		return interior;
	}
}
