package latexDraw.generators.svg;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Vector;

import javax.swing.JProgressBar;
import javax.swing.SwingUtilities;

import latexDraw.figures.Draw;
import latexDraw.figures.Figure;
import latexDraw.parsers.svg.MalformedSVGDocument;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.SVGElements;
import latexDraw.parsers.svg.elements.*;
import latexDraw.ui.DrawContainer;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.util.LaTeXDrawNamespace;

import com.sun.org.apache.xml.internal.serialize.OutputFormat;
import com.sun.org.apache.xml.internal.serialize.XMLSerializer;

/**
 * Defines a generator that creates SVG documents from drawings.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class SVGDocumentGenerator
{	
	public static Draw toLatexdraw(File file, LaTeXDrawFrame frame, final JProgressBar progressBar) throws MalformedSVGDocument
	{
		if(file==null || !file.canRead())
			return null;
		
		Draw drawing;
		Figure f;
		
		SVGDocument svgDoc = new SVGDocument(file.toURI());
		
		if(progressBar!=null)
			try { SwingUtilities.invokeAndWait(new Runnable() { public void run() { progressBar.setValue(20); } }); }
			catch(InterruptedException e) 		{ e.printStackTrace(); }
			catch(InvocationTargetException e) 	{ e.printStackTrace(); }
		
		f = svgDoc.toLaTeXDraw(progressBar);
		
		if(f==null)
			return null;
		
		if(f instanceof Draw)
			drawing = (Draw)f;
		else
		{
			drawing = new Draw(false, false);
			drawing.addFigure(f);
		}

		if(frame!=null)
			frame.getDrawPanel().getDraw().setXMLMetadata(svgDoc.getDocumentElement().getMeta());
		
		return drawing;
	}
	
	
	
	public static SVGDocument toSVG(Draw drawing)
	{
		if(drawing==null)
			return null;
		
		SVGDocument doc 		= new SVGDocument();
		SVGSVGElement root 		= doc.getFirstChild();
		SVGGElement g 			= new SVGGElement(doc);
		SVGDefsElement defs		= new SVGDefsElement(doc);
		Vector<Figure> figures  = drawing.getFigures();
		
		root.appendChild(defs);
		root.appendChild(g);
		root.setAttribute("xmlns:"+LaTeXDrawNamespace.LATEXDRAW_NAMESPACE, LaTeXDrawNamespace.LATEXDRAW_NAMESPACE_URI);//$NON-NLS-1$
		
        for(Figure f : figures)
        	if(f!=null)
	        {
        		SVGElement elt = SVGShapesFactory.createSVGElement(f, doc);
	        	
	        	if(elt!=null)
	        		g.appendChild(elt);
	        }
        
        return doc;
	}
	
	
	
	
	public static SVGDocument toSVG(DrawContainer drawing)
	{
		if(drawing==null)
			return null;
        
		SVGDocument doc			= toSVG(new Draw(drawing.getFigures(), false, false, true));
		SVGSVGElement root 		= doc.getFirstChild();
		SVGMetadataElement meta = new SVGMetadataElement(doc);
		SVGElement	metaLTD		= (SVGElement)doc.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+SVGElements.SVG_METADATA);
		
		meta.appendChild(metaLTD);
		root.appendChild(meta);

		drawing.getXMLMetadata(doc, metaLTD);
		drawing.getParentPanel().getXMLMetadata(doc, metaLTD);
		drawing.getParentPanel().getCodePanel().getXMLMetadata(doc, metaLTD);
		drawing.getParentPanel().getFrameParent().getXMLMetadata(doc, metaLTD);
		
		root.setAttribute(SVGAttributes.SVG_VERSION, "1.1");//$NON-NLS-1$
		root.setAttribute(SVGAttributes.SVG_BASE_PROFILE, "full");//$NON-NLS-1$
        
		return doc;
	}
	
	
	
	public static void saveSVGDocument(String path, SVGDocument doc) throws IOException
	{
		if(doc==null || path==null)
			return ;
		
		FileOutputStream fos = new FileOutputStream(path);
        OutputFormat of;
        XMLSerializer xmls;
        
        of = new OutputFormat(doc);
        of.setIndenting(true);
        xmls = new XMLSerializer(fos, of);
        xmls.serialize(doc.getDocumentElement());
		fos.close();
	}
	
	
	
	public static void exportAsSVG(String path, Draw drawing) throws IOException
	{
		saveSVGDocument(path, SVGDocumentGenerator.toSVG(drawing));
	}
	
	
	
	
	/**
	 * Exports the drawing in SVG.
	 * @param path The path of the file.
	 * @throws IOException 
	 * @since 2.0.0
	 */
	public static void exportAsSVG(String path, DrawContainer drawing) throws IOException
	{
		saveSVGDocument(path, SVGDocumentGenerator.toSVG(drawing));
	}
}
