package latexDraw.generators.svg;

import java.awt.geom.Line2D;

import latexDraw.figures.ArrowHead;
import latexDraw.figures.Line;
import latexDraw.figures.properties.Arrowable;
import latexDraw.parsers.svg.MalformedSVGDocument;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.elements.*;
import latexDraw.parsers.svg.elements.path.SVGPathSegLineto;
import latexDraw.parsers.svg.elements.path.SVGPathSegList;
import latexDraw.parsers.svg.elements.path.SVGPathSegMoveto;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines a SVG generator for a line.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LLineSVGGenerator extends LPolygonSVGGenerator
{
	public LLineSVGGenerator(Line f)
	{
		super(f);
	}

	
	
	/**
	 * Creates a line using a SVG path.
	 * @param elt The SVG path.
	 */
	public LLineSVGGenerator(SVGPathElement elt)
	{
		super(new Line(true));
		
		if(elt==null || !elt.isLine())
			throw new IllegalArgumentException();
		
		SVGPathSegList segs = elt.getSegList();
		Line l = (Line)getShape();
		
		l.setFirstPoint(new LaTeXDrawPoint2D(((SVGPathSegMoveto)segs.firstElement()).getPoint()));
		l.setLastPoint(new LaTeXDrawPoint2D(((SVGPathSegLineto)segs.lastElement()).getPoint()));
		l.updateShape();
		setSVGParameters(elt);
		applyTransformations(elt);
	}
	
	
	
	public LLineSVGGenerator(SVGLineElement e)
	{
		this(new Line(e.getX1(), e.getY1(), e.getX2(), e.getY2(), true));

		setSVGParameters(e);
		applyTransformations(e);
	}
	
	
	
	
	public LLineSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	
	public LLineSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new Line(false));
		
		setNumber(elt);
		SVGElement elt2 = getLaTeXDrawElement(elt, null);
		Line l = (Line)getShape();
		ArrowHead arrowHead1 = l.getArrowHead1();
		ArrowHead arrowHead2 = l.getArrowHead2();
		
		if(elt==null || elt2==null || !(elt2 instanceof SVGLineElement))
			throw new IllegalArgumentException();
		
		SVGLineElement main = (SVGLineElement)elt2;
		setSVGLatexdrawParameters(elt);
		setSVGParameters(main);
		
		l.getPt1().setLocation(main.getX1(), main.getY1());
		l.getPt2().setLocation(main.getX2(), main.getY2());
		
		arrowHead1.setPosition(l.getPoint(0));
		arrowHead1.setLine(l);
		arrowHead1.setFigure(l);
		arrowHead2.setPosition(l.getPoint(1));
		arrowHead2.setLine(l);
		arrowHead2.setFigure(l);
		l.updateAandB();
		l.updateShape();
		
		setSVGArrow(arrowHead1, main.getAttribute(main.getUsablePrefix()+SVGAttributes.SVG_MARKER_START), main);
		setSVGArrow(arrowHead2, main.getAttribute(main.getUsablePrefix()+SVGAttributes.SVG_MARKER_END), main);
		homogeniseArrows(arrowHead1, arrowHead2);
		setSVGShadowParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_SHADOW));
		setSVGDbleBordersParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_DBLE_BORDERS));
		
		if(withTransformation)
			applyTransformations(elt);
	}
	
	
	
	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		SVGElement root = new SVGGElement(doc), elt;
		int number = shape.getNumber();
		ArrowHead arrowHeadL = ((Arrowable)shape).getArrowHead1();
		ArrowHead arrowHeadR = ((Arrowable)shape).getArrowHead2();
		SVGDefsElement defs = doc.getFirstChild().getDefs();
		Line l = (Line)shape;
		SVGElement arrow1 = null, arrow2 = null, arrow1Shad = null, arrow2Shad = null;
		
		root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE + ':' + LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_LINE);
		root.setAttribute(SVGAttributes.SVG_ID, getSVGID());
		String arrow1Name = "arrow1-" + number;//$NON-NLS-1$
		String arrow2Name = "arrow2-" + number;//$NON-NLS-1$
		String arrow1ShadName = "arrow1Shad-" + number;//$NON-NLS-1$
		String arrow2ShadName = "arrow2Shad-" + number;//$NON-NLS-1$
		LaTeXDrawPoint2D pt1 = l.getPt1(), pt2 = l.getPt2();

		if(!arrowHeadL.isWithoutStyle())
		{
			arrow1 = new LArrowHeadSVGGenerator(arrowHeadL).toSVG(doc, false);

			arrow1.setAttribute(SVGAttributes.SVG_ID, arrow1Name);
			defs.appendChild(arrow1);

			if(l.hasShadow())
			{
				arrow1Shad = new LArrowHeadSVGGenerator(arrowHeadL).toSVG(doc, true);
				arrow1Shad.setAttribute(SVGAttributes.SVG_ID, arrow1ShadName);
				defs.appendChild(arrow1Shad);
			}
		}

		if(!arrowHeadR.isWithoutStyle())
		{
			arrow2 = new LArrowHeadSVGGenerator(arrowHeadR).toSVG(doc, false);

			arrow2.setAttribute(SVGAttributes.SVG_ID, arrow2Name);
			defs.appendChild(arrow2);

			if(l.hasShadow())
			{
				arrow2Shad = new LArrowHeadSVGGenerator(arrowHeadR).toSVG(doc, true);
				arrow2Shad.setAttribute(SVGAttributes.SVG_ID, arrow2ShadName);
				defs.appendChild(arrow2Shad);
			}
		}

		if(l.hasShadow())
		{
			SVGElement shad = new SVGLineElement(doc);

			shad.setAttribute(SVGAttributes.SVG_X1, String.valueOf(pt1.x));
			shad.setAttribute(SVGAttributes.SVG_X2, String.valueOf(pt2.x));
			shad.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(pt1.y));
			shad.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(pt2.y));
			setSVGShadowAttributes(shad, false);
			root.appendChild(shad);

			if(arrow1Shad != null)
				shad.setAttribute(SVGAttributes.SVG_MARKER_START, "url(#" + arrow1ShadName + ")");//$NON-NLS-1$//$NON-NLS-2$

			if(arrow2Shad != null)
				shad.setAttribute(SVGAttributes.SVG_MARKER_END, "url(#" + arrow2ShadName + ")");//$NON-NLS-1$//$NON-NLS-2$
		}

		elt = new SVGLineElement(doc);
		elt.setAttribute(SVGAttributes.SVG_X1, String.valueOf(pt1.x));
		elt.setAttribute(SVGAttributes.SVG_X2, String.valueOf(pt2.x));
		elt.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(pt1.y));
		elt.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(pt2.y));
		root.appendChild(elt);

		if(l.hasDoubleBoundary())
		{
			SVGElement dblBord = new SVGLineElement(doc);

			dblBord.setAttribute(SVGAttributes.SVG_X1, String.valueOf(pt1.x));
			dblBord.setAttribute(SVGAttributes.SVG_X2, String.valueOf(pt2.x));
			dblBord.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(pt1.y));
			dblBord.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(pt2.y));
			setSVGDoubleBordersAttributes(dblBord);
			root.appendChild(dblBord);
		}

		setSVGAttributes(doc, elt, false);
		elt.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE + ':' + LaTeXDrawNamespace.XML_ROTATION, String.valueOf(l.getRotationAngle()));

		if(arrow1 != null)
			elt.setAttribute(SVGAttributes.SVG_MARKER_START, "url(#" + arrow1Name + ")");//$NON-NLS-1$//$NON-NLS-2$

		if(arrow2 != null)
			elt.setAttribute(SVGAttributes.SVG_MARKER_END, "url(#" + arrow2Name + ")");//$NON-NLS-1$//$NON-NLS-2$

		return root;
	}
	
	
	
	public static SVGLineElement toSVG(Line2D l, SVGDocument doc)
	{
		if(l==null || doc==null)
			return null;
		
		SVGLineElement le;
		
		try { le = new SVGLineElement(l.getX1(), l.getY1(), l.getX2(), l.getY2(), doc); }
		catch(MalformedSVGDocument e) { le = null; }
		
		return le;
	}
}
