package latexDraw.generators.svg;

import latexDraw.figures.Circle;
import latexDraw.figures.LaTeXDrawRectangle;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.elements.SVGCircleElement;
import latexDraw.parsers.svg.elements.SVGElement;
import latexDraw.parsers.svg.elements.SVGGElement;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines a SVG generator for a circle.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LCircleSVGGenerator extends LEllipseSVGGenerator
{
	public LCircleSVGGenerator(Circle f)
	{
		super(f);
	}


	
	/**
	 * Creates a circle from an SVG circle element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LCircleSVGGenerator(SVGCircleElement elt)
	{
		this(new Circle(true));
		
		Circle c = (Circle)getShape();
		setSVGParameters(elt);
		LaTeXDrawRectangle borders = c.getBorders(); 
		borders.getPoint(0).setLocation(elt.getCx()-elt.getR(), elt.getCy()-elt.getR());
		borders.getPoint(1).setLocation(elt.getCx()+elt.getR(), elt.getCy()-elt.getR());
		borders.getPoint(2).setLocation(elt.getCx()-elt.getR(), elt.getCy()+elt.getR());
		borders.getPoint(3).setLocation(elt.getCx()+elt.getR(), elt.getCy()+elt.getR());
		c.updateShape();
		c.updateGravityCenter();
		c.setBordersPosition(PSTricksConstants.BORDERS_MIDDLE);
		applyTransformations(elt);
	}
	
	
	
	
	public LCircleSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	
	/**
	 * Creates a circle from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LCircleSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new Circle(false));

		setNumber(elt);
		SVGElement elt2 = getLaTeXDrawElement(elt, null);
		
		if(elt==null || elt2==null || !(elt2 instanceof SVGCircleElement))
			throw new IllegalArgumentException();
		
		SVGCircleElement main = (SVGCircleElement)elt2;
		LaTeXDrawRectangle borders = shape.getBorders(); 
		setSVGLatexdrawParameters(elt);
		setSVGParameters(main);
		double gap = getPositionGap()/2.;
		
		borders.getPoint(0).setLocation(main.getCx()-main.getR()+gap, main.getCy()-main.getR()+gap);
		borders.getPoint(1).setLocation(main.getCx()+main.getR()-gap, main.getCy()-main.getR()+gap);
		borders.getPoint(2).setLocation(main.getCx()-main.getR()+gap, main.getCy()+main.getR()-gap);
		borders.getPoint(3).setLocation(main.getCx()+main.getR()-gap, main.getCy()+main.getR()-gap);
		shape.updateShape();
		shape.updateGravityCenter();
		setSVGShadowParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_SHADOW));
		setSVGDbleBordersParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_DBLE_BORDERS));
		
		if(withTransformation)
			applyTransformations(elt);
	}
	
	
	
	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		if(doc==null || doc.getFirstChild().getDefs()==null)
			return null;

		LaTeXDrawPoint2D nw = shape.getTheNWPoint();
		LaTeXDrawPoint2D se = shape.getTheSEPoint();
		SVGElement elt, shad = null, dblBord = null;
		SVGElement root = new SVGGElement(doc);
        root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_CIRCLE);
        root.setAttribute(SVGAttributes.SVG_ID, getSVGID());
        double gap = getPositionGap();

        if(shape.hasShadow())
        {
        	shad = new SVGCircleElement((se.x+nw.x)/2., (se.y+nw.y)/2., (se.x-nw.x+gap)/2., doc);
        	setSVGShadowAttributes(shad, true);
        	root.appendChild(shad);
        }
        
        if(shape.hasShadow() && !shape.getLineStyle().equals(PSTricksConstants.LINE_NONE_STYLE))
        {// The background of the borders must be filled is there is a shadow.
	        elt = new SVGCircleElement((se.x+nw.x)/2., (se.y+nw.y)/2., (se.x-nw.x+gap)/2., doc);
	        setSVGBorderBackground(elt, root);
        }
        
        elt = new SVGCircleElement((se.x+nw.x)/2., (se.y+nw.y)/2., (se.x-nw.x+gap)/2., doc);
        root.appendChild(elt);
        
        if(shape.hasDoubleBoundary())
        {
        	dblBord = new SVGCircleElement((se.x+nw.x)/2., (se.y+nw.y)/2., (se.x-nw.x+gap)/2., doc);
        	setSVGDoubleBordersAttributes(dblBord);
        	root.appendChild(dblBord);
        }

        setSVGAttributes(doc, elt, true);
        setSVGRotationAttribute(root);
        
		return root;
	}
}
