package latexDraw.figures;

import java.awt.*;
import java.awt.geom.GeneralPath;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.util.Vector;

import latexDraw.figures.properties.Arrowable;
import latexDraw.psTricks.DviPsColors;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.ui.components.Delimitor;
import latexDraw.util.LaTeXDrawNumber;
import latexDraw.util.LaTeXDrawPoint2D;


/**
 * This class defines a kind of figure: joined lines.<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 * <br>
 * LaTeXDraw is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or any later version.<br>
 * <br>
 * LaTeXDraw is distributed without any warranty; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.<br>
 * <br>
 * 01/28/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class JoinedLines extends LaTeXDrawPolygon implements Arrowable
{
	private static final long serialVersionUID = 1L;

	/** the arrowhead of the first line. */
	private ArrowHead arrowHead1;

	/** The arrowhead of the last line. */
	private ArrowHead arrowHead2;



	 /**
	 * The constructor using two points
	 * @param pt1 The first point of the polygon
	 * @param pt2 The second point of the polygon
	 */
	public JoinedLines(LaTeXDrawPoint2D pt1, LaTeXDrawPoint2D pt2, boolean increaseMeter)
	{
		super(pt1, pt2, increaseMeter);
		
		canBeFilled = true;
		isBordersMovable = false;
		isDoubleBoundaryable = true;
		canHaveArrow = true;
		shape = getInsideOutsideOrMiddleBorders();
		arrowHead1 = new ArrowHead(pt1, new Line(pt1, pt2, false), this);
		arrowHead2 = new ArrowHead(pt2, new Line(pt2, pt1, false), this);
	} 



	/**
	 * Allows to get the outer or the inner borders following a distance
	 * @param gap The distance of the borders
	 * @param into True, you will get the borders inside the real borders
	 */
	@Override
	protected GeneralPath getBorders(double gap, boolean into)
	{
		GeneralPath gp = new GeneralPath();
		
		gp.moveTo((float)pts.firstElement().x, (float)pts.firstElement().y);
		
		if(pts.size()==1)
			gp.lineTo((float)pts.firstElement().x-1, (float)pts.firstElement().y);
		else
		if(pts.size()==2)
			gp.lineTo(((float)pts.firstElement().x)==((float)pts.lastElement().x) && 
						((float)pts.firstElement().y)==((float)pts.lastElement().y) ? (float)pts.lastElement().x-1 : 
						(float)pts.lastElement().x, (float)pts.lastElement().y);
		else
			for(int i=1, size=pts.size(); i<size; i++)
				gp.lineTo((float)pts.elementAt(i).x, (float)pts.elementAt(i).y);
		
		return gp;
	}





	@Override
	public void draw(Graphics2D g, Object antiAlias, Object rendering, Object alphaInter, Object colorRendering)
	{ 
		double dx=0, dy=0;
		LaTeXDrawPoint2D first = (LaTeXDrawPoint2D)pts.firstElement().clone();
		LaTeXDrawPoint2D last =  (LaTeXDrawPoint2D)pts.lastElement().clone();
		Color formerCol = g.getColor();
		String arrowHead1Style = arrowHead1.getArrowStyle();
		String arrowHead2Style = arrowHead2.getArrowStyle();
		boolean arrow1Drawable = arrowHead1.isDrawable() && !arrowHead1Style.equals(PSTricksConstants.NONEARROW_STYLE) && pts.size()>1;
		boolean arrow2Drawable = arrowHead2.isDrawable() && !arrowHead2Style.equals(PSTricksConstants.NONEARROW_STYLE) && pts.size()>1;
		boolean update = false;
		LaTeXDrawPoint2D pFormer0 = pts.firstElement();
		LaTeXDrawPoint2D pFormer1 = pts.lastElement();
		BasicStroke wideline = new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER);
		float thick = thickness;
		
		if(hasShadow)
		{
			LaTeXDrawPoint2D cg = getGravityCenter();
			LaTeXDrawPoint2D shadowCg = (LaTeXDrawPoint2D)cg.clone();
			shadowCg.setLocation(cg.x+shadowSize, cg.y);
			shadowCg = Figure.rotatePoint(shadowCg, cg, shadowAngle);
			dx = shadowCg.x-cg.x;
			dy = cg.y-shadowCg.y;
		}
		
		if(shape==null)
			shape = getInsideOutsideOrMiddleBorders();
			
		if(arrow1Drawable)
		{
			LaTeXDrawPoint2D pt0 = pts.firstElement();
			LaTeXDrawPoint2D pt1 = pts.elementAt(1);
			Line l = arrowHead1.getLine();
			l.updateAandB();
			double lineAngle = Math.atan(l.getA());
			double xRot,yRot, b = l.getB();
			double lgth = arrowHead1.getArrowHeadLength();
	
			if(Math.abs(lineAngle)==(Math.PI/2.))
			{
				first.y = pt0.y<pt1.y ? pt0.y+lgth : pt0.y-lgth;
				first.x = pt0.x;
			}
			else
			{
				xRot = Math.cos(-lineAngle)*pt0.x-Math.sin(-lineAngle)*(pt0.y-b); 
				yRot = Math.sin(-lineAngle)*pt0.x+Math.cos(-lineAngle)*(pt0.y-b)+b;
				if(pt0.x<pt1.x) xRot+=lgth;
				else xRot-=lgth;
				first.x = Math.cos(lineAngle)*xRot-Math.sin(lineAngle)*(yRot-b);
				first.y = Math.sin(lineAngle)*xRot+Math.cos(lineAngle)*(yRot-b)+b;	
			}	
			
			pFormer0 = pts.remove(0);
			pts.add(0, first);
			update = true;
		}
			
		if(arrow2Drawable)
		{
			LaTeXDrawPoint2D pt0 = pts.lastElement();
			LaTeXDrawPoint2D pt1 = pts.elementAt(pts.size()-2);
			Line l = arrowHead2.getLine();
			l.updateAandB();
			double lineAngle = Math.atan(l.getA());
			double xRot,yRot, b = l.getB();
			double lgth = arrowHead2.getArrowHeadLength();
	
			if(Math.abs(lineAngle)==(Math.PI/2.))
			{
				last.y = pt0.y<pt1.y ? pt0.y+lgth : pt0.y-lgth;
				last.x = pt0.x;
			}
			else
			{
				xRot = Math.cos(-lineAngle)*pt0.x-Math.sin(-lineAngle)*(pt0.y-b); 
				yRot = Math.sin(-lineAngle)*pt0.x+Math.cos(-lineAngle)*(pt0.y-b)+b;
				if(pt0.x<pt1.x) xRot+=lgth;
				else xRot-=lgth;
				last.x = Math.cos(lineAngle)*xRot-Math.sin(lineAngle)*(yRot-b);
				last.y = Math.sin(lineAngle)*xRot+Math.cos(lineAngle)*(yRot-b)+b;	
			}	
			
			pFormer1 = pts.remove(pts.size()-1);
			pts.add(pts.size(), last);
			update = true;
		}
			
		if(update)
			shape = getBorders(0, true);
		
		if(hasDoubleBoundary)
		{
			thick = (float)(2*thickness+doubleSep);
			
			if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
				wideline = new BasicStroke(thick, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER);
			else 
			if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
				wideline = new BasicStroke(thick, BasicStroke.CAP_ROUND,
						BasicStroke.JOIN_MITER, 1.f, new float[] { 0, thick + dotSep }, 0);
			else
				wideline = new BasicStroke(thick, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER,
						1.f, new float[]{blackDashLength, whiteDashLength}, 0);
		}
		
		if(hasShadow)
		{
			g.setStroke(new BasicStroke(thick, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
			g.translate(dx, dy);
			g.setColor(shadowColor);
			
			if(isFilled() || isHatched() || hasGradient()) g.fill(shape);
				g.draw(shape);
				
			g.translate(-dx, -dy);
			
			if(isFilled() || isHatched() || hasGradient()) 
			{
				g.setColor(interiorColor);
				g.draw(shape);
			}
		}

		if(arrow1Drawable && hasShadow)
		{
			g.translate(dx, dy);
			arrowHead1.draw(g, interiorColor, true);
			g.translate(-dx, -dy);
		}
			
		if(arrow2Drawable && hasShadow)
		{
			g.translate(dx, dy);
			arrowHead2.draw(g, interiorColor, true);
			g.translate(-dx, -dy);
		}
			
		if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
			g.setStroke(new BasicStroke(thick, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
		else 
		if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
			g.setStroke(new BasicStroke(thick, BasicStroke.CAP_ROUND, BasicStroke.JOIN_MITER,
					1.f, new float[]{0,thick+dotSep}, 0));
		else
		if(lineStyle.equals(PSTricksConstants.LINE_DASHED_STYLE))
			g.setStroke(new BasicStroke(thick, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER,
					1.f, new float[]{blackDashLength, whiteDashLength}, 0));
			
		fillFigure(g, antiAlias, rendering, alphaInter, colorRendering, shape);
		g.setColor(linesColor);
		
		if(hasDoubleBoundary())
		{
			Shape outline = wideline.createStrokedShape(shape);
			g.fill(outline);
			g.setColor(doubleColor);
			wideline = new BasicStroke((float)doubleSep, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER);
	        outline = wideline.createStrokedShape(shape);
			g.fill(outline);
		}
		else
			g.draw(shape);
		
		if(update)
		{
			pts.remove(0);
			pts.remove(pts.size()-1);
			pts.add(0, pFormer0);
			pts.add(pts.size(), pFormer1);
			updateShape();
		}
	
		g.setStroke(new BasicStroke(thick, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
		
		if(arrow1Drawable)
			arrowHead1.draw(g, interiorColor, false);
		
		if(arrow2Drawable)
			arrowHead2.draw(g, interiorColor, false);

		g.setColor(formerCol);

		if(isSelected)
		{
			int sizeD = delimiters.size();
			for(int i=0; i<sizeD; i++)
				delimiters.elementAt(i).draw(g);
			
			if(borders!=null)
				borders.draw(g, false, antiAlias, rendering, alphaInter, colorRendering);
		}
	}





	@Override
	public synchronized String getCodePSTricks(DrawBorders drawBorders, float ppc)
	{
		LaTeXDrawPoint2D d = drawBorders.getOriginPoint(), p;
		String coord = "", add = "", fillType = "", arrowsCode = ""; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$//$NON-NLS-4$
		int i, size = getNbPoints();
		double threshold = 0.001;
		
		if(size<2)
			return null;

		for(i = 0; i<size; i++)
		{
			p = pts.elementAt(i);
			coord += "("+LaTeXDrawNumber.getCutNumber((float)((p.x-d.x)/ppc),threshold)+","+ //$NON-NLS-1$ //$NON-NLS-2$
					LaTeXDrawNumber.getCutNumber((float)((d.y-p.y)/ppc),threshold)+")"; //$NON-NLS-1$
		}

		if(hasShadow)
		{
			fillType += ",shadow=true";//$NON-NLS-1$
			if(Math.toDegrees(shadowAngle)!=PSTricksConstants.DEFAULT_SHADOW_ANGLE)
				fillType += ",shadowangle="+(float)Math.toDegrees(shadowAngle);//$NON-NLS-1$

			if(((float)shadowSize)!=((float)DEFAULT_SHADOW_SIZE))
				fillType += ",shadowsize="+LaTeXDrawNumber.getCutNumber((float)(shadowSize/PPC),threshold);//$NON-NLS-1$

			if(!shadowColor.equals(PSTricksConstants.DEFAULT_SHADOW_COLOR))
			{
				String name = DviPsColors.getColourName(shadowColor);
				if(name==null)
				{
					name = "color"+number+'e';//$NON-NLS-1$
					DviPsColors.addUserColour(shadowColor, name);
				}
				add += ",shadowcolor="+name; //$NON-NLS-1$
			}
		}

		String str = getPSTricksCodeFilling(ppc);
		if(str.length()>0)
			fillType = fillType+','+str;

		str = getPSTricksCodeLine(ppc);
		if(str.length()>0)
			add = add+','+str;

		String arrowHead1Style = arrowHead1.getArrowStyle();
		String arrowHead2Style = arrowHead2.getArrowStyle();

		if(!arrowHead1Style.equals(PSTricksConstants.NONEARROW_STYLE) ||
			!arrowHead2Style.equals(PSTricksConstants.NONEARROW_STYLE))
		{
			if(arrowHead2Style.equals(PSTricksConstants.DLARROW_STYLE))
				arrowHead2Style = PSTricksConstants.DRARROW_STYLE;
			else
				if(arrowHead2Style.equals(PSTricksConstants.DRARROW_STYLE))
					arrowHead2Style = PSTricksConstants.DLARROW_STYLE;
				else
					if(arrowHead2Style.equals(PSTricksConstants.RARROW_STYLE))
						arrowHead2Style = PSTricksConstants.LARROW_STYLE;
					else
						if(arrowHead2Style.equals(PSTricksConstants.LARROW_STYLE))
							arrowHead2Style = PSTricksConstants.RARROW_STYLE;
						else
							if(arrowHead2Style.equals(PSTricksConstants.DLARROW_STYLE))
								arrowHead2Style = PSTricksConstants.DRARROW_STYLE;
							else
								if(arrowHead2Style.equals(PSTricksConstants.LRBRACKET_STYLE))
									arrowHead2Style = PSTricksConstants.RRBRACKET_STYLE;
								else
									if(arrowHead2Style.equals(PSTricksConstants.RRBRACKET_STYLE))
										arrowHead2Style = PSTricksConstants.LRBRACKET_STYLE;
									else
										if(arrowHead2Style.equals(PSTricksConstants.RSBRACKET_STYLE))
											arrowHead2Style = PSTricksConstants.LSBRACKET_STYLE;
										else
											if(arrowHead2Style.equals(PSTricksConstants.LSBRACKET_STYLE))
												arrowHead2Style = PSTricksConstants.RSBRACKET_STYLE;

			String paramsR = ","+arrowHead1.getParametersCode(); //$NON-NLS-1$
			String paramsL = ","+arrowHead2.getParametersCode(); //$NON-NLS-1$
			if(paramsR.equals(","))paramsR = ""; //$NON-NLS-1$ //$NON-NLS-2$
			if(paramsL.equals(","))paramsL = ""; //$NON-NLS-1$ //$NON-NLS-2$

			if(!arrowHead1Style.equals(PSTricksConstants.NONEARROW_STYLE))
			{
				arrowsCode = "{"+arrowHead1Style+'-'; //$NON-NLS-1$

				if(!arrowHead2Style.equals(PSTricksConstants.NONEARROW_STYLE))
					arrowsCode += arrowHead2Style;
				arrowsCode += '}';

				if(!arrowHead1.isOfTheSameTypeAs(arrowHead2))
				{
					if((arrowHead2Style.equals(PSTricksConstants.LRBRACKET_STYLE) ||
						arrowHead2Style.equals(PSTricksConstants.RRBRACKET_STYLE) ||
						arrowHead2Style.equals(PSTricksConstants.RSBRACKET_STYLE)||
						arrowHead2Style.equals(PSTricksConstants.RSBRACKET_STYLE)) &&
						(arrowHead1Style.equals(PSTricksConstants.BAREND_STYLE)
						||arrowHead1Style.equals(PSTricksConstants.BARIN_STYLE)
						||arrowHead1Style.equals(PSTricksConstants.LRBRACKET_STYLE)
						||arrowHead1Style.equals(PSTricksConstants.RRBRACKET_STYLE)
						||arrowHead1Style.equals(PSTricksConstants.RSBRACKET_STYLE)||
						arrowHead1Style.equals(PSTricksConstants.RSBRACKET_STYLE)))
						add = paramsR;
					else
						add += paramsL+paramsR;
				}
				else
					add += paramsR;
			}
			else
				if(!arrowHead2Style.equals(PSTricksConstants.NONEARROW_STYLE))
				{
					add += paramsL;
					arrowsCode = "{-"+arrowHead2Style+'}'; //$NON-NLS-1$
				}
		}

		if(hasDoubleBoundary)
		{
			add += ",doubleline=true,doublesep="+LaTeXDrawNumber.getCutNumber((float)(doubleSep/ppc),threshold); //$NON-NLS-1$

			if(doubleColor!=PSTricksConstants.DEFAULT_DOUBLE_COLOR)
			{
				String name = DviPsColors.getColourName(doubleColor);
				if(name==null)
				{
					name = "color"+number+'d';//$NON-NLS-1$
					DviPsColors.addUserColour(doubleColor, name);
				}
				add += ",doublecolor="+name; //$NON-NLS-1$
			}
		}

		if(!isFilled
				&&!interiorColor.equals(PSTricksConstants.DEFAULT_INTERIOR_COLOR))
		{
			String name = DviPsColors.getColourName(interiorColor);
			if(name==null)
			{
				name = "color"+number+'b';//$NON-NLS-1$
				DviPsColors.addUserColour(interiorColor, name);
			}
			add += ",fillcolor="+name; //$NON-NLS-1$
		}

		return "\\psline[linewidth="+LaTeXDrawNumber.getCutNumber((thickness/ppc),threshold) //$NON-NLS-1$
				+add+fillType+"]"+arrowsCode+coord; //$NON-NLS-1$
	}





	/**
	 * @return the arrowHead1Style
	 */
	public synchronized String getArrow1Style()
	{
		return arrowHead1.getArrowStyle();
	}





	/**
	 * @param arrowHeadStyle the arrowHead1Style to set
	 */
	public synchronized void setArrow1Style(String arrowHeadStyle)
	{
		if(PSTricksConstants.isValidArrowStyle(arrowHeadStyle))
			arrowHead1.setArrowStyle(arrowHeadStyle);
	}





	/**
	 * @param arrowHeadStyle the arrowHead2Style to set
	 */
	public synchronized void setArrow2Style(String arrowHeadStyle)
	{
		if(PSTricksConstants.isValidArrowStyle(arrowHeadStyle))
			arrowHead2.setArrowStyle(arrowHeadStyle);
	}





	/**
	 * @return the arrowHead2Style
	 */
	public synchronized String getArrow2Style()
	{
		return arrowHead2.getArrowStyle();
	}





	@Override
	public synchronized void setThickness(float val)
	{
		super.setThickness(val);

		if(arrowHead1!=null)
		{
			arrowHead1.getLine().setThickness(val);
			arrowHead2.getLine().setThickness(val);
		}
	}





	@Override
	public boolean addPoint(LaTeXDrawPoint2D pt)
	{
		boolean ok = super.addPoint(pt);

		if(ok&&pts.size()>1)
		{
			Line l = new Line(pt, pts.elementAt(pts.size()-2), false);
			l.setThickness(thickness);
			l.setLinesColor(linesColor);

			if(arrowHead2!=null)
			{
				arrowHead2.setLine(l);
				arrowHead2.setPosition(pt);
			}
		}

		return ok;
	}





	@Override
	public void removePointAt(int id)
	{
		super.removePointAt(id);

		if(id==0&&pts.size()>1)
		{
			Line l = new Line(pts.firstElement(), pts.elementAt(1), false);
			l.setThickness(thickness);
			l.setLinesColor(linesColor);
			arrowHead1.setLine(l);
			arrowHead1.setPosition(pts.firstElement());
		}
		else
			if((id==-1||id>pts.size())&&pts.size()>1)
			{
				Line l = new Line(pts.lastElement(), pts
						.elementAt(pts.size()-2), false);
				l.setThickness(thickness);
				l.setLinesColor(linesColor);
				arrowHead2.setLine(l);
				arrowHead2.setPosition(pts.lastElement());
			}

	}





	@Override
	public Object clone() throws CloneNotSupportedException
	{
		JoinedLines f = (JoinedLines)super.clone();

		f.arrowHead1 = (ArrowHead)arrowHead1.clone();
		f.arrowHead2 = (ArrowHead)arrowHead2.clone();
		Line l1 = new Line(f.pts.lastElement(), f.pts.elementAt(f.pts.size()-2), false);
		l1.setThickness(thickness);
		l1.setLinesColor(linesColor);
		f.arrowHead2.setLine(l1);
		f.arrowHead2.setPosition(f.pts.lastElement());
		f.arrowHead2.setFigure(f);
		Line l2 = new Line(f.pts.firstElement(), f.pts.elementAt(1), false);
		l2.setThickness(thickness);
		l2.setLinesColor(linesColor);
		f.arrowHead1.setLine(l2);
		f.arrowHead1.setPosition(f.pts.firstElement());
		f.arrowHead1.setFigure(f);

		return f;
	}





	@SuppressWarnings("unchecked")
	private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException
	{
		canHaveShadow = true;
		canHaveArrow = true;
		isDoubleBoundaryable = true;

		interiorColor = (Color)ois.readObject();
		lineStyle = (String)ois.readObject();
		rotationAngle = ois.readDouble();
		thickness = ois.readFloat();
		isFilled = ois.readBoolean();
		isSelected = ois.readBoolean();
		isOnRotation = ois.readBoolean();
		linesColor = (Color)ois.readObject();
		blackDashLength = ois.readFloat();
		dotSep = ois.readFloat();
		whiteDashLength = ois.readFloat();
		pts = (Vector<LaTeXDrawPoint2D>)ois.readObject();

		delimiters = new Vector<Delimitor>();
		for(int i = 0, size = pts.size(); i<size; i++)
			delimiters.add(new Delimitor(pts.elementAt(i)));

		hasDoubleBoundary = ois.readBoolean();
		doubleColor = (Color)ois.readObject();
		doubleSep = ois.readDouble();
		hatchingAngle = ois.readDouble();
		hatchingColor = (Color)ois.readObject();
		hatchingStyle = (String)ois.readObject();
		hatchingWidth = ois.readFloat();

		arrowHead1 = (ArrowHead)ois.readObject();
		arrowHead2 = (ArrowHead)ois.readObject();

		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.7")>=0) //$NON-NLS-1$
		{
			hasShadow = ois.readBoolean();
			shadowAngle = ois.readDouble();
			shadowSize = ois.readDouble();
			shadowColor = (Color)ois.readObject();
			gradientEndColor = (Color)ois.readObject();
			gradientStartColor = (Color)ois.readObject();
			gradientAngle = ois.readDouble();
			gradientMidPoint = ois.readDouble();
		}
		else
		{
			hasShadow = DEFAULT_SHADOW_HAS;
			shadowAngle = DEFAULT_SHADOW_ANGLE;
			shadowSize = DEFAULT_SHADOW_SIZE;
			shadowColor = DEFAULT_SHADOW_COLOR;
			gradientEndColor = PSTricksConstants.DEFAULT_GRADIENT_END_COLOR;
			gradientStartColor = PSTricksConstants.DEFAULT_GRADIENT_START_COLOR;
			gradientAngle = DEFAULT_GRADIENT_ANGLE;
			gradientMidPoint = DEFAULT_GRADIENT_MID_POINT;
		}

		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.8")>=0) //$NON-NLS-1$
			hatchingSep = ois.readDouble();
		else
			hatchingSep = DEFAULT_HATCH_SEP;

		shape = getInsideOutsideOrMiddleBorders();
		setThickness(thickness);
	}



	/**
	 * @return the arrowHead1
	 */
	public ArrowHead getArrowHead1()
	{
		return arrowHead1;
	}





	/**
	 * @return the arrowHead2
	 */
	public ArrowHead getArrowHead2()
	{
		return arrowHead2;
	}





//	@Override
//	public boolean intersects(Rectangle2D.Double r)
//	{
//		if(r==null)
//			return false;
//
//		boolean again = true;
//		int i = 0, size = pts.size()-1;
//		Line2D.Double l = new Line2D.Double();
//		BasicStroke wideline = new BasicStroke((thickness));
//
//		while(again&&i<size)
//		{
//			l.setLine(pts.elementAt(i), pts.elementAt(i+1));
//			Shape outline = wideline.createStrokedShape(l);
//
//			if(outline.intersects(r))
//				again = false;
//			else
//				i++;
//		}
//
//		if(again && (isFilled() || isHatched() || hasGradient()))
//		{
//			l = new Line2D.Double(pts.lastElement(), pts.firstElement());
//
//			if(l.intersects(r))
//				again = false;
//		}
//
//		return !again;
//	}




	@Override
	public int hashCode()
	{
		return super.hashCode()^getNbPoints();
	}




	public boolean hasTwoLeftArrows()
	{
		return true;
	}
	
	
	
	@Override
	public boolean isIn(LaTeXDrawPoint2D pt)
	{
		if(isSelected && (borders.dNE.isIn(pt) || borders.dNW.isIn(pt) || borders.dSE.isIn(pt) || 
			borders.dSW.isIn(pt) || borders.dS.isIn(pt)  || borders.dN.isIn(pt) || borders.dE.isIn(pt)  || borders.dW.isIn(pt)))
			return true;
		
		boolean in = false;
		int i, sizeD = delimiters.size();
		
		// If the point is in a delimiter
		for(i=0; i<sizeD && !in; i++)
			if(delimiters.elementAt(i).isIn(pt))
				in = true;
		
		if(in) return true;
		
		Stroke wideline = new BasicStroke(hasDoubleBoundary ?(float)(thickness*2+doubleSep) : thickness, 
											BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER);
		Shape s = wideline.createStrokedShape(shape);
		
		if(s.contains(pt))
			return true;
		
		if(isFilled || hasGradient() || isHatched())
			return shape.contains(pt);

		return false;
	}

	
	
	@Override
	public boolean intersected(Rectangle2D.Double r)
	{
		if(r==null)
			return false;
		
		BasicStroke wideline = new BasicStroke(hasDoubleBoundary() ? (float)(thickness*2+doubleSep) : thickness);
        Shape outline = wideline.createStrokedShape(shape);
        
        boolean ok = outline.intersects(r);
        
        if(!ok && (isFilled() || isHatched() || hasGradient()))
        {
        	Line2D.Double l = new Line2D.Double(pts.firstElement(), pts.lastElement());
        	
        	if(l.intersects(r))
        		ok = true;
        }
        
        return ok;
	}
	
	
	@Override
	public boolean shadowFillsShape()
	{
		return false;
	}
}
