/*
 * TestAbstractPstricksParser.java
 */
package junit.test;
import java.awt.Color;
import java.awt.geom.Point2D;

import junit.framework.TestCase;
import latexDraw.parsers.pstricks.PSTricksParserActionsManager;
import latexDraw.psTricks.DviPsColors;
import latexDraw.psTricks.PSTricksConstants;

import org.junit.Test;

/** 
 * This class contains tests to PSTricks parser.<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 * 
 * <br>
 * 12/18/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
@SuppressWarnings("nls")
public class TestAbstractPstricksParser extends TestCase
{
	@Test
	public void testGoToNextChar() throws Exception
	{
		int[] i = {0};
		String str = "\n \t         \n \t .dsq";//$NON-NLS-1$
		PSTricksParserActionsManager.goToNextChar(str, i);
		assertEquals(str.charAt(i[0]), '.');
	}
	
	
	
	@Test
	public void testConvertCm() throws Exception
	{
		assertEquals(PSTricksParserActionsManager.convertInCm(10., "cm"),10.);//$NON-NLS-1$
		assertEquals(PSTricksParserActionsManager.convertInCm(10., "in"),10.*PSTricksConstants.INCH_VAL_CM);//$NON-NLS-1$
		assertEquals(PSTricksParserActionsManager.convertInCm(10., "pt"),10./PSTricksConstants.CM_VAL_PT);//$NON-NLS-1$
		assertEquals(PSTricksParserActionsManager.convertInCm(10., "mm"),1.);//$NON-NLS-1$
		assertEquals(PSTricksParserActionsManager.convertInCm(10., null),10.);
		assertEquals(PSTricksParserActionsManager.convertInCm(10., ""),10.);//$NON-NLS-1$
		try
		{ 
			PSTricksParserActionsManager.convertInCm(10., "sddsq"); //$NON-NLS-1$
			fail("testConvertCm");//$NON-NLS-1$
		}
		catch(Exception e) { /**/}
	}
	
	
	@Test
	public void testReadOneCoord() throws Exception
	{
		PSTricksParserActionsManager.readOneCoordinate("(1,2)", 0, 0, true);//$NON-NLS-1$
		PSTricksParserActionsManager.readOneCoordinate("(1,)", 0, 0, true); //$NON-NLS-1$
		PSTricksParserActionsManager.readOneCoordinate("(,2)", 0, 0, true); //$NON-NLS-1$
		PSTricksParserActionsManager.readOneCoordinate("(,)", 0, 0, true); //$NON-NLS-1$
		PSTricksParserActionsManager.readOneCoordinate("(\n	,	\n	2)", 0, 0, true);//$NON-NLS-1$
		PSTricksParserActionsManager.readOneCoordinate("(-0.1,0.2)", 0, 0, true); //$NON-NLS-1$
		Point2D.Double pt = PSTricksParserActionsManager.readOneCoordinate("(-0.1,0.2)", 0, 0, true);//$NON-NLS-1$
		assertEquals(-0.1, pt.x);
		assertEquals(0.2 , pt.y);
		pt = PSTricksParserActionsManager.readOneCoordinate("(+1,)", 0, 0, true);//$NON-NLS-1$
		assertEquals(1., pt.x);
		assertEquals(Double.NaN, pt.y);
	}
	
	
	
	@Test
	public void testReadEqualInteger() throws Exception 
	{
		int id[] = {0};
		assertEquals(PSTricksParserActionsManager.readEqualInteger(id, "=10"), 10);//$NON-NLS-1$
		id[0] = 0;
		assertEquals(PSTricksParserActionsManager.readEqualInteger(id, "=+10"), 10);//$NON-NLS-1$
		id[0] = 0;
		assertEquals(PSTricksParserActionsManager.readEqualInteger(id, "=+++++10"), 10);//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualInteger(id, "=10");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualInteger(id, "	=	\n	2");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualInteger(id, "     = 	2");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualInteger(id, "=20erzrfd");//$NON-NLS-1$
		try 
		{ 
			int id2[] = {-1};
			PSTricksParserActionsManager.readEqualInteger(id2, "=10");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		try 
		{ 
			int id2[] = {3};
			PSTricksParserActionsManager.readEqualInteger(id2, "=10");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualInteger(id, "10");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualInteger(id, "==10");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualInteger(id, "==-10");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualInteger(id, "=--10");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualInteger(id, "=-1-0");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualInteger(id, "= ");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualInteger(id, null);
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualInteger(id, "");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualInteger(id, "= ezrezr");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
	}
	
	
	
	@Test
	public void testReadEqualReal() throws Exception
	{
		int id[] = {0};
		assertEquals(PSTricksParserActionsManager.readEqualReal(id, "=++-+--+-10"), 10.);//$NON-NLS-1$
		id[0] = 0;
		assertEquals(PSTricksParserActionsManager.readEqualReal(id, "=10"), 10.);//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualReal(id, "=10");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualReal(id, "=0.2");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualReal(id, "=.2");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualReal(id, "	=	\n	0.2");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualReal(id, "     = 	.2");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualReal(id, "=-.2");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualReal(id, "=-2");//$NON-NLS-1$
	
		try 
		{ 
			int id2[] = {-1};
			PSTricksParserActionsManager.readEqualReal(id2, "=10");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		try 
		{ 
			int id2[] = {3};
			PSTricksParserActionsManager.readEqualReal(id2, "=10");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualReal(id, "10");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualReal(id, "==10");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualReal(id, "==-10");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualReal(id, "=-1-0");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualReal(id, "=..10");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualReal(id, "=.1.0");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualReal(id, "= ");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualReal(id, null);
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualReal(id, "");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
	}
	
	@Test
	public void testReadEqualRealUnit() throws Exception
	{
		int id[] = {0};
		assertEquals(PSTricksParserActionsManager.readEqualRealUnit(id, "=10"), 10.);//$NON-NLS-1$
		id[0] = 0;
		assertEquals(PSTricksParserActionsManager.readEqualRealUnit(id, "=10cm"), 10.);//$NON-NLS-1$
		id[0] = 0;
		assertEquals(PSTricksParserActionsManager.readEqualRealUnit(id, "=10mm"), 1.);//$NON-NLS-1$
		id[0] = 0;
		assertEquals(PSTricksParserActionsManager.readEqualRealUnit(id, "=10 pt"), 10./PSTricksConstants.CM_VAL_PT);//$NON-NLS-1$
		id[0] = 0;
		assertEquals(PSTricksParserActionsManager.readEqualRealUnit(id, "=10in"), 10.*PSTricksConstants.INCH_VAL_CM);//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualRealUnit(id, "=10 pt");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualRealUnit(id, "=10 cm");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualRealUnit(id, "=10 mm");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualRealUnit(id, "=10 in");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualRealUnit(id, "=0.2 pt");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualRealUnit(id, "=.2 pt");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualRealUnit(id, "	=		0.2\n pt");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualRealUnit(id, "     = 	.2");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualRealUnit(id, "=.2");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualRealUnit(id, "=-.2");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualRealUnit(id, "=-2");//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readEqualRealUnit(id, "=4");//$NON-NLS-1$
	
		try 
		{ 
			int id2[] = {-1};
			PSTricksParserActionsManager.readEqualRealUnit(id2, "=10 pt");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		try 
		{ 
			int id2[] = {3};
			PSTricksParserActionsManager.readEqualRealUnit(id2, "=10 pt");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualRealUnit(id, "10 pt");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualRealUnit(id, "==10 pt");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualRealUnit(id, "==-10 pt");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualRealUnit(id, "=-1-0 pt");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualRealUnit(id, "=..10 pt");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
	
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualRealUnit(id, "=.1.0 pt");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualRealUnit(id, "= cm ");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualRealUnit(id, "==10 dsd");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualRealUnit(id, null);
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readEqualRealUnit(id, "");//$NON-NLS-1$
			fail("testReadEqualRealUnit");//$NON-NLS-1$
		}catch(Exception e) { /* */}
	}

	
	
	@Test
	public void testReadRealUnit() throws Exception
	{
		int id[] = {0};
		assertEquals(PSTricksParserActionsManager.readRealUnit(id, ".5", true), .5);//$NON-NLS-1$
		id[0] = 0;
		assertEquals(PSTricksParserActionsManager.readRealUnit(id, "10", true), 10.);//$NON-NLS-1$
		id[0] = 0;
		assertEquals(PSTricksParserActionsManager.readRealUnit(id, "10cm", true), 10.);//$NON-NLS-1$
		id[0] = 0;
		assertEquals(PSTricksParserActionsManager.readRealUnit(id, "10mm", false), 1.);//$NON-NLS-1$
		id[0] = 0;
		assertEquals(PSTricksParserActionsManager.readRealUnit(id, "10 pt", true), 10./PSTricksConstants.CM_VAL_PT);//$NON-NLS-1$
		id[0] = 0;
		assertEquals(PSTricksParserActionsManager.readRealUnit(id, "10in", true), 10.*PSTricksConstants.INCH_VAL_CM);//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, "10 pt", true);//$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, "10 cm", false); //$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, "10 mm", true); //$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, "=0 in", false); //$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, "0.2 pt", true); //$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, ".2 pt", true); //$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, "		0.2\n pt", true); //$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, "     	.2", true); //$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, ".2", true); //$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, "-.2", true); //$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, "-2", true); //$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, "4", true); //$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, "++--+++---+4", true); //$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, "+++4", true); //$NON-NLS-1$
		id[0] = 0;
		PSTricksParserActionsManager.readRealUnit(id, "---4", true); //$NON-NLS-1$
	
		try 
		{ 
			int id2[] = {-1};
			PSTricksParserActionsManager.readRealUnit(id2, "10 pt", true); //$NON-NLS-1$
			fail("testReadEqualRealUnit"); //$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		try 
		{ 
			int id2[] = {3};
			PSTricksParserActionsManager.readRealUnit(id2, "10 pt", true); //$NON-NLS-1$
			fail("testReadEqualRealUnit"); //$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readRealUnit(id, "-1-0 pt", true); //$NON-NLS-1$
			fail("testReadEqualRealUnit"); //$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readRealUnit(id, "..10 pt", true); //$NON-NLS-1$
			fail("testReadEqualRealUnit"); //$NON-NLS-1$
		}catch(Exception e) { /* */}
	
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readRealUnit(id, ".1.0 pt", true); //$NON-NLS-1$
			fail("testReadEqualRealUnit"); //$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readRealUnit(id, " cm ", true); //$NON-NLS-1$
			fail("testReadEqualRealUnit"); //$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readRealUnit(id, null, true);
			fail("testReadEqualRealUnit"); //$NON-NLS-1$
		}catch(Exception e) { /* */}
		
		id[0] = 0;
		try 
		{ 
			PSTricksParserActionsManager.readRealUnit(id, "", true); //$NON-NLS-1$
			fail("testReadEqualRealUnit"); //$NON-NLS-1$
		}catch(Exception e) { /* */}
	}
	
	
	
	
	@Test
	public void testParseNewrgbcolorParameters()
	{
		try
		{
			PSTricksParserActionsManager.parseNewrgbcolorParameters(null);
			fail();
		}
		catch(Exception e) { /* Good */ }
		
		try
		{
			PSTricksParserActionsManager.parseNewrgbcolorParameters(" ");
			fail();
		}
		catch(Exception e) { /* Good */ }
		
		try
		{
			PSTricksParserActionsManager.parseNewrgbcolorParameters("newrgbcolor{name}{1 1 1}");
			fail();
		}
		catch(Exception e) { /* Good */ }
		
		try
		{
			PSTricksParserActionsManager.parseNewrgbcolorParameters("{rgbColourTest}{1 1 0}");
			assertEquals(DviPsColors.getColour("rgbColourTest"), new Color(255, 255, 0));
		}
		catch(Exception e) { fail(); }
		
		try
		{
			PSTricksParserActionsManager.parseNewrgbcolorParameters("{  \nrgbColourTest\n }\n  { 1 \n \t 1 \n0 \n  }");
			assertEquals(DviPsColors.getColour("rgbColourTest"), new Color(255, 255, 0));
		}
		catch(Exception e) { fail(); }
		
		try
		{
			PSTricksParserActionsManager.parseNewrgbcolorParameters("{rgbColourTest}{0.5, 0.5, 0.5}");
			fail();
		}
		catch(Exception e) { /* Good */  }
		
		try
		{
			PSTricksParserActionsManager.parseNewrgbcolorParameters("{test rgbColourTest}{0.5 0.5 0.5}");
			fail();
		}
		catch(Exception e) { /* Good */  }
		
		try
		{
			PSTricksParserActionsManager.parseNewrgbcolorParameters("{0.1234}{0.5 0.5 0.5}");
			fail();
		}
		catch(Exception e) { /* Good */  }
	}
	
	
	
	
	
	@Test
	public void testParseNewcmykcolorParameters()
	{
		try
		{
			PSTricksParserActionsManager.parseNewcmykcolorParameters(null);
			fail();
		}
		catch(Exception e) { /* Good */ }
		
		try
		{
			PSTricksParserActionsManager.parseNewcmykcolorParameters(" ");
			fail();
		}
		catch(Exception e) { /* Good */ }
		
		try
		{
			PSTricksParserActionsManager.parseNewcmykcolorParameters("newcmykcolor{name}{1 1 1 1}");
			fail();
		}
		catch(Exception e) { /* Good */ }
		
		try
		{
			PSTricksParserActionsManager.parseNewcmykcolorParameters("{cmykColourTest}{0.5 0.1 0.3 0.6}");
			assertEquals(DviPsColors.getColour("cmykColourTest"), new Color(51, 92, 71));
		}
		catch(Exception e) { fail(); }
		
		try
		{
			PSTricksParserActionsManager.parseNewcmykcolorParameters("{  \ncmykColourTest\n }\n  { 0.5 \n \t 0.1 \n0.3 \n  0.6}");
			assertEquals(DviPsColors.getColour("cmykColourTest"), new Color(51, 92, 71));
		}
		catch(Exception e) { fail(); }
		
		try
		{
			PSTricksParserActionsManager.parseNewcmykcolorParameters("{cmykColourTest}{0.5, 0.5, 0.5cm 1t}");
			fail();
		}
		catch(Exception e) { /* Good */  }
		
		try
		{
			PSTricksParserActionsManager.parseNewcmykcolorParameters("{test cmykColourTest}{0.5 0.5 0.5 1}");
			fail();
		}
		catch(Exception e) { /* Good */  }
		
		try
		{
			PSTricksParserActionsManager.parseNewcmykcolorParameters("{0.1234}{0.5 0.5 0.5 1}");
			fail();
		}
		catch(Exception e) { /* Good */  }
	}
	
	
	
	
	@Test
	public void testParseNewhsbcolorParameters()
	{
		try
		{
			PSTricksParserActionsManager.parseNewhsbcolorParameters(null);
			fail();
		}
		catch(Exception e) { /* Good */ }
		
		try
		{
			PSTricksParserActionsManager.parseNewhsbcolorParameters(" ");
			fail();
		}
		catch(Exception e) { /* Good */ }
		
		try
		{
			PSTricksParserActionsManager.parseNewhsbcolorParameters("newhsbcolor{name}{1 1 1}");
			fail();
		}
		catch(Exception e) { /* Good */ }
		
		try
		{
			PSTricksParserActionsManager.parseNewhsbcolorParameters("{hsbColourTest}{1 1 0}");
			assertEquals(DviPsColors.getColour("hsbColourTest"), Color.getHSBColor(1, 1, 0));
		}
		catch(Exception e) { fail(); }
		
		try
		{
			PSTricksParserActionsManager.parseNewhsbcolorParameters("{  \nhsbColourTest\n }\n  { 1 \n \t 1 \n0 \n  }");
			assertEquals(DviPsColors.getColour("hsbColourTest"), Color.getHSBColor(1, 1, 0));
		}
		catch(Exception e) { fail(); }
		
		try
		{
			PSTricksParserActionsManager.parseNewhsbcolorParameters("{hsbColourTest}{0.5, 0.5, 0.5}");
			fail();
		}
		catch(Exception e) { /* Good */  }
		
		try
		{
			PSTricksParserActionsManager.parseNewhsbcolorParameters("{test hsbColourTest}{0.5 0.5 0.5}");
			fail();
		}
		catch(Exception e) { /* Good */  }
		
		try
		{
			PSTricksParserActionsManager.parseNewhsbcolorParameters("{0.1234}{0.5 0.5 0.5}");
			fail();
		}
		catch(Exception e) { /* Good */  }
	}
	
	
	
	
	@Test
	public void testParseNewgrayParameters()
	{
		try
		{
			PSTricksParserActionsManager.parseNewgrayParameters(null);
			fail();
		}
		catch(Exception e) { /* Good */ }
		
		try
		{
			PSTricksParserActionsManager.parseNewgrayParameters(" ");
			fail();
		}
		catch(Exception e) { /* Good */ }
		
		try
		{
			PSTricksParserActionsManager.parseNewgrayParameters("newgray{name}{1 1 1}");
			fail();
		}
		catch(Exception e) { /* Good */ }
		
		try
		{
			PSTricksParserActionsManager.parseNewgrayParameters("{grayColourTest}{0.5}");
			assertEquals(DviPsColors.getColour("grayColourTest"), new Color(128, 128, 128));
		}
		catch(Exception e) { fail(); }
		
		try
		{
			PSTricksParserActionsManager.parseNewgrayParameters("{  \ngrayColourTest\n }\n  { \t \n 0.5 \r}");
			assertEquals(DviPsColors.getColour("grayColourTest"), new Color(128, 128, 128));
		}
		catch(Exception e) { fail(); }
		
		try
		{
			PSTricksParserActionsManager.parseNewgrayParameters("{grayColourTest}{0.5cm}");
			fail();
		}
		catch(Exception e) { /* Good */  }
		
		try
		{
			PSTricksParserActionsManager.parseNewgrayParameters("{test grayColourTest}{0.5}");
			fail();
		}
		catch(Exception e) { /* Good */  }
		
		try
		{
			PSTricksParserActionsManager.parseNewgrayParameters("{0.1234}{0.5}");
			fail();
		}
		catch(Exception e) { /* Good */  }
	}
	
	
	
	@Test
	public void testIsCharToIgnore()
	{
		assertTrue(PSTricksParserActionsManager.isCharToIgnore(' '));
		assertTrue(PSTricksParserActionsManager.isCharToIgnore('\t'));
		assertTrue(PSTricksParserActionsManager.isCharToIgnore('\n'));
		assertTrue(PSTricksParserActionsManager.isCharToIgnore('\r'));
		assertFalse(PSTricksParserActionsManager.isCharToIgnore('1'));
		assertFalse(PSTricksParserActionsManager.isCharToIgnore('a'));
		assertFalse(PSTricksParserActionsManager.isCharToIgnore('&'));
		assertFalse(PSTricksParserActionsManager.isCharToIgnore('M'));
	}
}





