;;; julia-mode.el --- Major mode for editing Julia source code

;; Copyright (C) 2009-2014 Julia contributors
;; URL: https://github.com/JuliaLang/julia
;; Version: 0.3
;; Keywords: languages

;;; Usage:
;; Put the following code in your .emacs, site-load.el, or other relevant file
;; (add-to-list 'load-path "path-to-julia-mode")
;; (require 'julia-mode)

;;; Commentary:
;; This is the official Emacs mode for editing Julia programs.

;;; License:
;; Permission is hereby granted, free of charge, to any person obtaining
;; a copy of this software and associated documentation files (the
;; "Software"), to deal in the Software without restriction, including
;; without limitation the rights to use, copy, modify, merge, publish,
;; distribute, sublicense, and/or sell copies of the Software, and to
;; permit persons to whom the Software is furnished to do so, subject to
;; the following conditions:
;;
;; The above copyright notice and this permission notice shall be
;; included in all copies or substantial portions of the Software.
;;
;; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
;; EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
;; MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
;; NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
;; LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
;; OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
;; WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

;;; Code:

;; We can't use cl-lib whilst supporting Emacs 23 users who don't use
;; ELPA.
(with-no-warnings
  (require 'cl)) ;; incf, decf, plusp

(defvar julia-mode-hook nil)

(defgroup julia ()
  "Major mode for the julia programming language."
  :group 'languages
  :prefix "julia-")

(defcustom julia-indent-offset 4
  "Number of spaces per indentation level."
  :type 'integer
  :group 'julia)

(defface julia-macro-face
  '((t :inherit font-lock-preprocessor-face))
  "Face for Julia macro invocations."
  :group 'julia-mode)

(defface julia-quoted-symbol-face
  '((t :inherit font-lock-preprocessor-face))
  "Face for quoted Julia symbols, e.g. :foo."
  :group 'julia-mode)


;;;###autoload
(add-to-list 'auto-mode-alist '("\\.jl\\'" . julia-mode))

;; define ignore-errors macro if it isn't present
;; (necessary for emacs 22 compatibility)
(when (not (fboundp 'ignore-errors))
  (defmacro ignore-errors (body) `(condition-case nil ,body (error nil))))

(defun julia--regexp-opt (strings &optional paren)
  "Emacs 23 provides `regexp-opt', but it does not support PAREN taking the value 'symbols.
This function provides equivalent functionality, but makes no efforts to optimise the regexp."
  (cond
   ((>= emacs-major-version 24)
    (regexp-opt strings paren))
   ((not (eq paren 'symbols))
    (regexp-opt strings paren))
   ((null strings)
    "")
   ('t
    (rx-to-string `(seq symbol-start (or ,@strings) symbol-end)))))

(defvar julia-mode-syntax-table
  (let ((table (make-syntax-table)))
    (modify-syntax-entry ?_ "_" table)
    (modify-syntax-entry ?@ "_" table)
    (modify-syntax-entry ?! "_" table)
    (modify-syntax-entry ?# "< 14" table)  ; # single-line and multiline start
    (modify-syntax-entry ?= ". 23bn" table)
    (modify-syntax-entry ?\n ">" table)  ; \n single-line comment end
    (modify-syntax-entry ?\{ "(} " table)
    (modify-syntax-entry ?\} "){ " table)
    (modify-syntax-entry ?\[ "(] " table)
    (modify-syntax-entry ?\] ")[ " table)
    (modify-syntax-entry ?\( "() " table)
    (modify-syntax-entry ?\) ")( " table)
    ;; Here, we treat ' as punctuation (when it's used for transpose),
    ;; see our use of `julia-char-regex' for handling ' as a character
    ;; delimeter
    (modify-syntax-entry ?'  "." table)
    (modify-syntax-entry ?\" "\"" table)
    (modify-syntax-entry ?` "\"" table)

    (modify-syntax-entry ?. "." table)
    (modify-syntax-entry ?? "." table)
    (modify-syntax-entry ?$ "." table)
    (modify-syntax-entry ?& "." table)
    (modify-syntax-entry ?* "." table)
    (modify-syntax-entry ?/ "." table)
    (modify-syntax-entry ?+ "." table)
    (modify-syntax-entry ?- "." table)
    (modify-syntax-entry ?< "." table)
    (modify-syntax-entry ?> "." table)
    (modify-syntax-entry ?% "." table)
    table)
  "Syntax table for `julia-mode'.")

(eval-when-compile
  (defconst julia-char-regex
    (rx (or (any "-" ";" "\\" "^" "!" "|" "?" "*" "<" "%" "," "=" ">" "+" "/" "&" "$" "~" ":")
            (syntax open-parenthesis)
            (syntax whitespace)
            bol)
        (group "'")
        (group
         (or (repeat 0 8 (not (any "'"))) (not (any "\\"))
             "\\\\"))
        (group "'"))))

(defconst julia-triple-quoted-string-regex
  ;; We deliberately put a group on the first and last delimiter, so
  ;; we can mark these as string delimiters for font-lock.
  (rx (group "\"")
      (group "\"\""
             ;; After the delimiter, we're a sequence of
             ;; non-backslashes or blackslashes paired with something.
             (*? (or (not (any "\\"))
                     (seq "\\" anything)))
             "\"\"")
      (group "\"")))

(defconst julia-unquote-regex
  "\\(\\s(\\|\\s-\\|-\\|[,%=<>\\+*/?&|!\\^~\\\\;:]\\|^\\)\\($[a-zA-Z0-9_]+\\)")

(defconst julia-forloop-in-regex
  "for +.*[^
].* \\(in\\)\\(\\s-\\|$\\)+")

(defconst julia-function-regex
  (rx line-start (* (or space "@inline" "@noinline")) symbol-start
      "function"
      (1+ space)
      ;; Don't highlight module names in function declarations:
      (* (seq (1+ (or word (syntax symbol))) "."))
      ;; The function name itself
      (group (1+ (or word (syntax symbol))))))

(defconst julia-function-assignment-regex
  (rx line-start (* (or space "@inline" "@noinline")) symbol-start
      (* (seq (1+ (or word (syntax symbol))) ".")) ; module name
      (group (1+ (or word (syntax symbol))))
      (* space)
      (? "{" (* (not (any "}"))) "}")
      (* space)
      "(" (* (or
              (seq "(" (* (not (any "(" ")"))) ")")
              (not (any "(" ")"))))
      ")"
      (* space)
      "="
      (not (any "="))))

(defconst julia-type-regex
  (rx symbol-start (or "immutable" "type" "abstract") (1+ space) (group (1+ (or word (syntax symbol))))))

(defconst julia-type-annotation-regex
  (rx "::" (0+ space) (group (1+ (or word (syntax symbol))))))

;;(defconst julia-type-parameter-regex
;;  (rx symbol-start (1+ (or (or word (syntax symbol)) ?_)) "{" (group (1+ (or (or word (syntax symbol)) ?_))) "}"))

(defconst julia-subtype-regex
  (rx "<:" (0+ space) (group (1+ (or word (syntax symbol)))) (0+ space) (or "\n" "{" "}" "end")))

(defconst julia-macro-regex
  (rx symbol-start (group "@" (1+ (or word (syntax symbol))))))

(defconst julia-keyword-regex
  (julia--regexp-opt
   '("if" "else" "elseif" "while" "for" "begin" "end" "quote"
     "try" "catch" "return" "local" "abstract" "function" "macro" "ccall"
     "finally" "typealias" "break" "continue" "type" "global"
     "module" "using" "import" "export" "const" "let" "bitstype" "do" "in"
     "baremodule" "importall" "immutable")
   'symbols))

(defconst julia-builtin-regex
  (julia--regexp-opt
   ;;'("error" "throw")
   '()
   'symbols))

(defconst julia-builtin-types-regex
  (julia--regexp-opt
   '("Number" "Real" "BigInt" "Integer"
     "UInt" "UInt8" "UInt16" "UInt32" "UInt64" "UInt128"
     "Int" "Int8" "Int16" "Int32" "Int64" "Int128"
     "BigFloat" "AbstractFloat" "Float16" "Float32" "Float64"
     "Complex128" "Complex64"
     "Bool"
     "Cuchar" "Cshort" "Cushort" "Cint" "Cuint" "Clonglong" "Culonglong" "Cintmax_t" "Cuintmax_t"
     "Cfloat" "Cdouble" "Cptrdiff_t" "Cssize_t" "Csize_t"
     "Cchar" "Clong" "Culong" "Cwchar_t"
     "Char" "ASCIIString" "UTF8String" "ByteString" "SubString"
     "Array" "DArray" "AbstractArray" "AbstractVector" "AbstractMatrix" "AbstractSparseMatrix" "SubArray" "StridedArray" "StridedVector" "StridedMatrix" "VecOrMat" "StridedVecOrMat" "DenseArray" "SparseMatrixCSC" "BitArray"
     "Range" "OrdinalRange" "StepRange" "UnitRange" "FloatRange"
     "Tuple" "NTuple" "Vararg"
     "DataType" "Symbol" "Function" "Vector" "Matrix" "Union" "Type" "Any" "Complex" "AbstractString" "Ptr" "Void" "Exception" "Task" "Signed" "Unsigned" "Associative" "Dict" "IO" "IOStream" "Rational" "Regex" "RegexMatch" "Set" "IntSet" "Expr" "WeakRef" "ObjectIdDict"
     "AbstractRNG" "MersenneTwister"
     )
   'symbols))

(defconst julia-quoted-symbol-regex
  ;; :foo and :foo2 are valid, but :123 is not.
  (rx (or whitespace "(" "[" "," "=")
      (group ":" (or letter (syntax symbol)) (0+ (or word (syntax symbol))))))

(defconst julia-font-lock-keywords
  (list
   ;; Ensure :: and <: aren't highlighted, so we don't confuse ::Foo with :foo.
   ;; (in Emacs, keywords don't overlap).
   (cons (rx (or "::" "<:")) ''default)
   ;; Highlight quoted symbols before keywords, so :function is not
   ;; highlighted as a keyword.
   (list julia-quoted-symbol-regex 1 ''julia-quoted-symbol-face)
   (cons julia-builtin-types-regex 'font-lock-type-face)
   (cons julia-keyword-regex 'font-lock-keyword-face)
   (cons julia-macro-regex ''julia-macro-face)
   (cons
    (julia--regexp-opt
     '("true" "false" "C_NULL" "Inf" "NaN" "Inf32" "NaN32" "nothing")
     'symbols)
    'font-lock-constant-face)
   (list julia-unquote-regex 2 'font-lock-constant-face)
   (list julia-forloop-in-regex 1 'font-lock-keyword-face)
   (list julia-function-regex 1 'font-lock-function-name-face)
   (list julia-function-assignment-regex 1 'font-lock-function-name-face)
   (list julia-type-regex 1 'font-lock-type-face)
   (list julia-type-annotation-regex 1 'font-lock-type-face)
   ;;(list julia-type-parameter-regex 1 'font-lock-type-face)
   (list julia-subtype-regex 1 'font-lock-type-face)
   (list julia-builtin-regex 1 'font-lock-builtin-face)
   ))

(defconst julia-block-start-keywords
  (list "if" "while" "for" "begin" "try" "function" "type" "let" "macro"
        "quote" "do" "immutable"))

(defconst julia-block-end-keywords
  (list "end" "else" "elseif" "catch" "finally"))

(defun julia-stringify-triple-quote ()
  "Put `syntax-table' property on triple-quoted string delimeters.

Based on `python-syntax-stringify'."
  (let* ((string-start-pos (- (point) 3))
         (string-end-pos (point))
         (ppss (prog2
                   (backward-char 3)
                   (syntax-ppss)
                 (forward-char 3)))
         (in-comment (nth 4 ppss))
         (in-string (nth 8 ppss)))
    (unless in-comment
      (if in-string
          ;; We're in a string, so this must be the closing triple-quote.
          ;; Put | on the last " character.
          (put-text-property (1- string-end-pos) string-end-pos
                             'syntax-table (string-to-syntax "|"))
        ;; We're not in a string, so this is the opening triple-quote.
        ;; Put | on the first " character.
        (put-text-property string-start-pos (1+ string-start-pos)
                           'syntax-table (string-to-syntax "|"))))))

(unless (< emacs-major-version 24)
  (defconst julia-syntax-propertize-function
    (syntax-propertize-rules
     ("\"\"\""
      (0 (ignore (julia-stringify-triple-quote))))
     (julia-char-regex
      (1 "\"") ; Treat ' as a string delimiter.
      (2 ".") ; Don't highlight anything between.
      (3 "\""))))) ; Treat the last " in """ as a string delimiter.

(defun julia-in-comment ()
  "Return non-nil if point is inside a comment.
Handles both single-line and multi-line comments."
  (nth 4 (syntax-ppss)))

(defun julia-in-string ()
  "Return non-nil if point is inside a string.
Note this is Emacs' notion of what is highlighted as a string.
As a result, it is true inside \"foo\", `foo` and 'f'."
  (nth 3 (syntax-ppss)))

(defun julia-in-brackets ()
  "Return non-nil if point is inside square brackets."
  (let ((start-pos (point))
        (open-count 0))
    ;; Count all the [ and ] characters on the current line.
    (save-excursion
      (beginning-of-line)

      (while (< (point) start-pos)
        ;; Don't count [ or ] inside strings, characters or comments.
        (unless (or (julia-in-string) (julia-in-comment))

          (when (looking-at (rx "["))
            (incf open-count))
          (when (looking-at (rx "]"))
            (decf open-count)))

        (forward-char 1)))

    ;; If we've opened more than we've closed, we're inside brackets.
    (plusp open-count)))

(defun julia-at-keyword (kw-list)
  "Return the word at point if it matches any keyword in KW-LIST.
KW-LIST is a list of strings.  The word at point is not considered
a keyword if used as a field name, X.word, or quoted, :word."
  (and (or (= (point) 1)
	   (and (not (equal (char-before (point)) ?.))
		(not (equal (char-before (point)) ?:))))
       (member (current-word t) kw-list)
       (not (julia-in-comment))
       ;; 'end' is not a keyword when used for indexing, e.g. foo[end-2]
       (or (not (equal (current-word t) "end"))
           (not (julia-in-brackets)))))

;; if backward-sexp gives an error, move back 1 char to move over the '('
(defun julia-safe-backward-sexp ()
  (if (condition-case nil (backward-sexp) (error t))
      (ignore-errors (backward-char))))

(defun julia-last-open-block-pos (min)
  "Return the position of the last open block, if one found.
Do not move back beyond position MIN."
  (save-excursion
    (let ((count 0))
      (while (not (or (> count 0) (<= (point) min)))
        (julia-safe-backward-sexp)
        (setq count
              (cond ((julia-at-keyword julia-block-start-keywords)
                     (+ count 1))
                    ;; fixme: breaks on strings
                    ((and (equal (current-word t) "end")
                          (not (julia-in-comment)) (not (julia-in-brackets)))
                     (- count 1))
                    (t count))))
      (if (> count 0)
          (point)
        nil))))

(defun julia-last-open-block (min)
  "Move back and return indentation level for last open block.
Do not move back beyond MIN."
  ;; Ensure MIN is not before the start of the buffer.
  (setq min (max min (point-min)))
  (let ((pos (julia-last-open-block-pos min)))
    (and pos
	 (progn
	   (goto-char pos)
	   (+ julia-indent-offset (current-indentation))))))

(defsubst julia--safe-backward-char ()
  "Move back one character, but don't error if we're at the
beginning of the buffer."
  (unless (eq (point) (point-min))
    (backward-char)))

(defvar julia-max-paren-lookback 400
  "When indenting, don't look back more than this
many characters to see if there are unclosed parens.

This variable has a major effect on indent performance if set too
high.")

(defvar julia-max-block-lookback 5000
  "When indenting, don't look back more than this
many characters to see if there are unclosed blocks.

This variable has a moderate effect on indent performance if set too
high.")

(defun julia-paren-indent ()
  "Return the column of the text following the innermost
containing paren before point, so we can align succeeding code
with it. Returns nil if we're not within nested parens."
  (save-excursion
    ;; Back up to previous line (beginning-of-line was already called)
    (backward-char)
    (let ((min-pos (max (- (point) julia-max-paren-lookback)
                        (point-min)))
          (open-count 0))
      (while (and (> (point) min-pos)
                  (not (plusp open-count)))

        (when (looking-at (rx (any "[" "]" "(" ")")))
          (unless (or (julia-in-string) (julia-in-comment))
            (cond ((looking-at (rx (any "[" "(")))
                   (incf open-count))
                  ((looking-at (rx (any "]" ")")))
                   (decf open-count)))))

        (julia--safe-backward-char))

      (if (plusp open-count)
          (progn (forward-char 2)
                 (while (looking-at (rx blank))
                   (forward-char))
                 (current-column))
        nil))))

(defun julia-indent-line ()
  "Indent current line of julia code."
  (interactive)
  (let* ((point-offset (- (current-column) (current-indentation))))
    (end-of-line)
    (indent-line-to
     (or
      ;; If we're inside an open paren, indent to line up arguments.
      (save-excursion
        (beginning-of-line)
        (ignore-errors (julia-paren-indent)))
      ;; If the previous line ends in =, increase the indent.
      (ignore-errors ; if previous line is (point-min)
        (save-excursion
          (if (and (not (equal (point-min) (line-beginning-position)))
                   (progn
                     (forward-line -1)
                     (end-of-line) (backward-char 1)
                     (and (equal (char-after (point)) ?=)
                          (not (julia-in-comment)))))
              (+ julia-indent-offset (current-indentation))
            nil)))
      ;; Indent according to how many nested blocks we are in.
      (save-excursion
        (beginning-of-line)
        (forward-to-indentation 0)
        (let ((endtok (julia-at-keyword julia-block-end-keywords))
              (last-open-block (julia-last-open-block (- (point) julia-max-block-lookback))))
          (max 0 (+ (or last-open-block 0)
                    (if endtok (- julia-indent-offset) 0)))))))
    ;; Point is now at the beginning of indentation, restore it
    ;; to its original position (relative to indentation).
    (when (>= point-offset 0)
      (move-to-column (+ (current-indentation) point-offset)))))

(defmacro julia--should-indent (from to)
  "Assert that we indent text FROM producing text TO in `julia-mode'."
  `(with-temp-buffer
     (let ((julia-indent-offset 4))
       (julia-mode)
       (insert ,from)
       (indent-region (point-min) (point-max))
       (should (equal (buffer-substring-no-properties (point-min) (point-max))
                      ,to)))))

;; Emacs 23.X doesn't include ert, so we ignore any errors that occur
;; when we define tests.
(ignore-errors
  (require 'ert)

  (ert-deftest julia--test-indent-if ()
    "We should indent inside if bodies."
    (julia--should-indent
     "
if foo
bar
end"
     "
if foo
    bar
end"))

  (ert-deftest julia--test-indent-else ()
    "We should indent inside else bodies."
    (julia--should-indent
     "
if foo
    bar
else
baz
end"
     "
if foo
    bar
else
    baz
end"))

  (ert-deftest julia--test-indent-toplevel ()
    "We should not indent toplevel expressions. "
    (julia--should-indent
     "
foo()
bar()"
     "
foo()
bar()"))

  (ert-deftest julia--test-indent-nested-if ()
    "We should indent for each level of indentation."
    (julia--should-indent
     "
if foo
    if bar
bar
    end
end"
     "
if foo
    if bar
        bar
    end
end"))

  (ert-deftest julia--test-indent-function ()
    "We should indent function bodies."
    (julia--should-indent
     "
function foo()
bar
end"
     "
function foo()
    bar
end"))

  (ert-deftest julia--test-indent-begin ()
    "We should indent after a begin keyword."
    (julia--should-indent
     "
@async begin
bar
end"
     "
@async begin
    bar
end"))

  (ert-deftest julia--test-indent-paren ()
    "We should indent to line up with the text after an open paren."
    (julia--should-indent
     "
foobar(bar,
baz)"
     "
foobar(bar,
       baz)"))

  (ert-deftest julia--test-indent-paren-space ()
    "We should indent to line up with the text after an open
paren, even if there are additional spaces."
    (julia--should-indent
     "
foobar( bar,
baz )"
     "
foobar( bar,
        baz )"))

  (ert-deftest julia--test-indent-equals ()
    "We should increase indent on a trailing =."
    (julia--should-indent
     "
foo() =
bar"
     "
foo() =
    bar"))

  (ert-deftest julia--test-indent-ignores-blank-lines ()
    "Blank lines should not affect indentation of non-blank lines."
    (julia--should-indent
     "
if foo
        
bar
end"
     "
if foo
    
    bar
end"))

  (ert-deftest julia--test-indent-comment-equal ()
    "`=` at the end of comment should not increase indent level."
    (julia--should-indent
     "
# a =
# b =
c"
     "
# a =
# b =
c"))

  (ert-deftest julia--test-indent-leading-paren ()
    "`(` at the beginning of a line should not affect indentation."
    (julia--should-indent
     "
(1)"
     "
(1)"))

  (ert-deftest julia--test-top-level-following-paren-indent ()
    "`At the top level, a previous line indented due to parens should not affect indentation."
    (julia--should-indent
     "y1 = f(x,
       z)
y2 = g(x)"
     "y1 = f(x,
       z)
y2 = g(x)"))

  (defun julia--run-tests ()
    (interactive)
    (ert-run-tests-interactively "julia--test")))

(defalias 'julia-mode-prog-mode
  (if (fboundp 'prog-mode)
      'prog-mode
    'fundamental-mode))

;;; IMENU
(defvar julia-imenu-generic-expression
  ;; don't use syntax classes, screws egrep
  '(("Function (_)" "[ \t]*function[ \t]+\\(_[^ \t\n]*\\)" 1)
    ("Function" "^[ \t]*function[ \t]+\\([^_][^\t\n]*\\)" 1)
    ("Const" "[ \t]*const \\([^ \t\n]*\\)" 1)
    ("Type"  "^[ \t]*[a-zA-Z0-9_]*type[a-zA-Z0-9_]* \\([^ \t\n]*\\)" 1)
    ("Require"      " *\\(\\brequire\\)(\\([^ \t\n)]*\\)" 2)
    ("Include"      " *\\(\\binclude\\)(\\([^ \t\n)]*\\)" 2)
    ;; ("Classes" "^.*setClass(\\(.*\\)," 1)
    ;; ("Coercions" "^.*setAs(\\([^,]+,[^,]*\\)," 1) ; show from and to
    ;; ("Generics" "^.*setGeneric(\\([^,]*\\)," 1)
    ;; ("Methods" "^.*set\\(Group\\|Replace\\)?Method(\"\\(.+\\)\"," 2)
    ;; ;;[ ]*\\(signature=\\)?(\\(.*,?\\)*\\)," 1)
    ;; ;;
    ;; ;;("Other" "^\\(.+\\)\\s-*<-[ \t\n]*[^\\(function\\|read\\|.*data\.frame\\)]" 1)
    ;; ("Package" "^.*\\(library\\|require\\)(\\(.*\\)," 2)
    ;; ("Data" "^\\(.+\\)\\s-*<-[ \t\n]*\\(read\\|.*data\.frame\\).*(" 1)))
    ))

;;;###autoload
(define-derived-mode julia-mode julia-mode-prog-mode "Julia"
  "Major mode for editing julia code."
  (set-syntax-table julia-mode-syntax-table)
  (set (make-local-variable 'comment-start) "# ")
  (set (make-local-variable 'comment-start-skip) "#+\\s-*")
  (set (make-local-variable 'font-lock-defaults) '(julia-font-lock-keywords))
  (if (< emacs-major-version 24)
      ;; Emacs 23 doesn't have syntax-propertize-function
      (set (make-local-variable 'font-lock-syntactic-keywords)
           (list
            `(,julia-char-regex
              (1 "\"") ; Treat ' as a string delimiter.
              (2 ".") ; Don't highlight anything between the open and close '.
              (3 "\"")); Treat the close ' as a string delimiter.
            `(,julia-triple-quoted-string-regex
              (1 "\"") ; Treat the first " in """ as a string delimiter.
              (2 ".") ; Don't highlight anything between.
              (3 "\"")))) ; Treat the last " in """ as a string delimiter.
    ;; Emacs 24 and later has syntax-propertize-function, so use that instead.
    (set (make-local-variable 'syntax-propertize-function)
         julia-syntax-propertize-function))
  (set (make-local-variable 'indent-line-function) 'julia-indent-line)
  (setq indent-tabs-mode nil)
  (setq imenu-generic-expression julia-imenu-generic-expression)
  (imenu-add-to-menubar "Imenu"))

(defvar julia-latexsubs (make-hash-table :test 'equal))

(defun julia-latexsub ()
  "Perform a LaTeX-like Unicode symbol substitution."
  (interactive "*i")
  (let ((orig-pt (point)))
    (while (not (or (bobp) (= ?\\ (char-before))
		    (= ?\s (char-syntax (char-before)))))
      (backward-char))
    (if (and (not (bobp)) (= ?\\ (char-before)))
        (progn
          (backward-char)
          (let ((sub (gethash (buffer-substring (point) orig-pt) julia-latexsubs)))
            (if sub
                (progn
                  (delete-region (point) orig-pt)
                  (insert sub))
              (goto-char orig-pt))))
      (goto-char orig-pt))))

(defalias 'latexsub 'julia-latexsub)

(defun julia-latexsub-or-indent (arg)
  "Either indent according to mode or perform a LaTeX-like symbol substution"
  (interactive "*i")
  (if (latexsub)
      (indent-for-tab-command arg)))
(define-key julia-mode-map (kbd "TAB") 'julia-latexsub-or-indent)

(defalias 'latexsub-or-indent 'julia-latexsub-or-indent)

; LaTeX-like symbol substitutions, equivalent to those in the Julia REPL,
; generated by:
;for (k,v) in sort!(collect(Base.REPLCompletions.latex_symbols), by=x->x[2])
;    ks = escape_string(k)
;    vs = escape_string(v)
;    if ismatch(r"^\\U[0-9A-Fa-f]+$", vs)
;        # codepoints outside the BMP can be problematic in older Emacsen
;        cp = vs[3:end]
;        println("(let ((c (decode-char 'ucs #x$cp)))\n",
;                "  (if c (puthash \"$ks\" (char-to-string c) julia-latexsubs)))")
;    else
;        println("(puthash \"$ks\" \"$vs\" julia-latexsubs)")
;    end
;end
; (See Julia issue #8947 for why we don't use the Emacs tex input mode.)
(puthash "\\textexclamdown" "¡" julia-latexsubs)
(puthash "\\sterling" "£" julia-latexsubs)
(puthash "\\yen" "¥" julia-latexsubs)
(puthash "\\textbrokenbar" "¦" julia-latexsubs)
(puthash "\\S" "§" julia-latexsubs)
(puthash "\\textasciidieresis" "¨" julia-latexsubs)
(puthash "\\copyright" "©" julia-latexsubs)
(puthash "\\textordfeminine" "ª" julia-latexsubs)
(puthash "\\neg" "¬" julia-latexsubs)
(puthash "\\circledR" "®" julia-latexsubs)
(puthash "\\textasciimacron" "¯" julia-latexsubs)
(puthash "\\degree" "°" julia-latexsubs)
(puthash "\\pm" "±" julia-latexsubs)
(puthash "\\^2" "²" julia-latexsubs)
(puthash "\\^3" "³" julia-latexsubs)
(puthash "\\textasciiacute" "´" julia-latexsubs)
(puthash "\\P" "¶" julia-latexsubs)
(puthash "\\cdotp" "·" julia-latexsubs)
(puthash "\\^1" "¹" julia-latexsubs)
(puthash "\\textordmasculine" "º" julia-latexsubs)
(puthash "\\textonequarter" "¼" julia-latexsubs)
(puthash "\\textonehalf" "½" julia-latexsubs)
(puthash "\\textthreequarters" "¾" julia-latexsubs)
(puthash "\\textquestiondown" "¿" julia-latexsubs)
(puthash "\\AA" "Å" julia-latexsubs)
(puthash "\\AE" "Æ" julia-latexsubs)
(puthash "\\DH" "Ð" julia-latexsubs)
(puthash "\\times" "×" julia-latexsubs)
(puthash "\\O" "Ø" julia-latexsubs)
(puthash "\\TH" "Þ" julia-latexsubs)
(puthash "\\ss" "ß" julia-latexsubs)
(puthash "\\aa" "å" julia-latexsubs)
(puthash "\\ae" "æ" julia-latexsubs)
(puthash "\\eth" "ð" julia-latexsubs)
(puthash "\\div" "÷" julia-latexsubs)
(puthash "\\o" "ø" julia-latexsubs)
(puthash "\\th" "þ" julia-latexsubs)
(puthash "\\DJ" "Đ" julia-latexsubs)
(puthash "\\dj" "đ" julia-latexsubs)
(puthash "\\Elzxh" "ħ" julia-latexsubs)
(puthash "\\hbar" "ħ" julia-latexsubs)
(puthash "\\L" "Ł" julia-latexsubs)
(puthash "\\l" "ł" julia-latexsubs)
(puthash "\\NG" "Ŋ" julia-latexsubs)
(puthash "\\ng" "ŋ" julia-latexsubs)
(puthash "\\OE" "Œ" julia-latexsubs)
(puthash "\\oe" "œ" julia-latexsubs)
(puthash "\\texthvlig" "ƕ" julia-latexsubs)
(puthash "\\textnrleg" "ƞ" julia-latexsubs)
(puthash "\\Zbar" "Ƶ" julia-latexsubs)
(puthash "\\textdoublepipe" "ǂ" julia-latexsubs)
(puthash "\\Elztrna" "ɐ" julia-latexsubs)
(puthash "\\Elztrnsa" "ɒ" julia-latexsubs)
(puthash "\\Elzopeno" "ɔ" julia-latexsubs)
(puthash "\\Elzrtld" "ɖ" julia-latexsubs)
(puthash "\\Elzschwa" "ə" julia-latexsubs)
(puthash "\\varepsilon" "ɛ" julia-latexsubs)
(puthash "\\Elzpgamma" "ɣ" julia-latexsubs)
(puthash "\\Elzpbgam" "ɤ" julia-latexsubs)
(puthash "\\Elztrnh" "ɥ" julia-latexsubs)
(puthash "\\Elzbtdl" "ɬ" julia-latexsubs)
(puthash "\\Elzrtll" "ɭ" julia-latexsubs)
(puthash "\\Elztrnm" "ɯ" julia-latexsubs)
(puthash "\\Elztrnmlr" "ɰ" julia-latexsubs)
(puthash "\\Elzltlmr" "ɱ" julia-latexsubs)
(puthash "\\Elzltln" "ɲ" julia-latexsubs)
(puthash "\\Elzrtln" "ɳ" julia-latexsubs)
(puthash "\\Elzclomeg" "ɷ" julia-latexsubs)
(puthash "\\textphi" "ɸ" julia-latexsubs)
(puthash "\\Elztrnr" "ɹ" julia-latexsubs)
(puthash "\\Elztrnrl" "ɺ" julia-latexsubs)
(puthash "\\Elzrttrnr" "ɻ" julia-latexsubs)
(puthash "\\Elzrl" "ɼ" julia-latexsubs)
(puthash "\\Elzrtlr" "ɽ" julia-latexsubs)
(puthash "\\Elzfhr" "ɾ" julia-latexsubs)
(puthash "\\Elzrtls" "ʂ" julia-latexsubs)
(puthash "\\Elzesh" "ʃ" julia-latexsubs)
(puthash "\\Elztrnt" "ʇ" julia-latexsubs)
(puthash "\\Elzrtlt" "ʈ" julia-latexsubs)
(puthash "\\Elzpupsil" "ʊ" julia-latexsubs)
(puthash "\\Elzpscrv" "ʋ" julia-latexsubs)
(puthash "\\Elzinvv" "ʌ" julia-latexsubs)
(puthash "\\Elzinvw" "ʍ" julia-latexsubs)
(puthash "\\Elztrny" "ʎ" julia-latexsubs)
(puthash "\\Elzrtlz" "ʐ" julia-latexsubs)
(puthash "\\Elzyogh" "ʒ" julia-latexsubs)
(puthash "\\Elzglst" "ʔ" julia-latexsubs)
(puthash "\\Elzreglst" "ʕ" julia-latexsubs)
(puthash "\\Elzinglst" "ʖ" julia-latexsubs)
(puthash "\\textturnk" "ʞ" julia-latexsubs)
(puthash "\\Elzdyogh" "ʤ" julia-latexsubs)
(puthash "\\Elztesh" "ʧ" julia-latexsubs)
(puthash "\\^h" "ʰ" julia-latexsubs)
(puthash "\\^j" "ʲ" julia-latexsubs)
(puthash "\\^r" "ʳ" julia-latexsubs)
(puthash "\\^w" "ʷ" julia-latexsubs)
(puthash "\\^y" "ʸ" julia-latexsubs)
(puthash "\\rasp" "ʼ" julia-latexsubs)
(puthash "\\textasciicaron" "ˇ" julia-latexsubs)
(puthash "\\Elzverts" "ˈ" julia-latexsubs)
(puthash "\\Elzverti" "ˌ" julia-latexsubs)
(puthash "\\Elzlmrk" "ː" julia-latexsubs)
(puthash "\\Elzhlmrk" "ˑ" julia-latexsubs)
(puthash "\\Elzsbrhr" "˒" julia-latexsubs)
(puthash "\\Elzsblhr" "˓" julia-latexsubs)
(puthash "\\Elzrais" "˔" julia-latexsubs)
(puthash "\\Elzlow" "˕" julia-latexsubs)
(puthash "\\u" "˘" julia-latexsubs)
(puthash "\\texttildelow" "˜" julia-latexsubs)
(puthash "\\^l" "ˡ" julia-latexsubs)
(puthash "\\^s" "ˢ" julia-latexsubs)
(puthash "\\^x" "ˣ" julia-latexsubs)
(puthash "\\grave" "̀" julia-latexsubs)
(puthash "\\acute" "́" julia-latexsubs)
(puthash "\\hat" "̂" julia-latexsubs)
(puthash "\\tilde" "̃" julia-latexsubs)
(puthash "\\bar" "̄" julia-latexsubs)
(puthash "\\overbar" "̅" julia-latexsubs)
(puthash "\\breve" "̆" julia-latexsubs)
(puthash "\\dot" "̇" julia-latexsubs)
(puthash "\\ddot" "̈" julia-latexsubs)
(puthash "\\ovhook" "̉" julia-latexsubs)
(puthash "\\ocirc" "̊" julia-latexsubs)
(puthash "\\H" "̋" julia-latexsubs)
(puthash "\\check" "̌" julia-latexsubs)
(puthash "\\candra" "̐" julia-latexsubs)
(puthash "\\oturnedcomma" "̒" julia-latexsubs)
(puthash "\\ocommatopright" "̕" julia-latexsubs)
(puthash "\\droang" "̚" julia-latexsubs)
(puthash "\\Elzpalh" "̡" julia-latexsubs)
(puthash "\\Elzrh" "̢" julia-latexsubs)
(puthash "\\c" "̧" julia-latexsubs)
(puthash "\\k" "̨" julia-latexsubs)
(puthash "\\Elzsbbrg" "̪" julia-latexsubs)
(puthash "\\wideutilde" "̰" julia-latexsubs)
(puthash "\\underbar" "̱" julia-latexsubs)
(puthash "\\Elzxl" "̵" julia-latexsubs)
(puthash "\\Elzbar" "̶" julia-latexsubs)
(puthash "\\sout" "̶" julia-latexsubs)
(puthash "\\not" "̸" julia-latexsubs)
(puthash "\\Alpha" "Α" julia-latexsubs)
(puthash "\\Beta" "Β" julia-latexsubs)
(puthash "\\Gamma" "Γ" julia-latexsubs)
(puthash "\\Delta" "Δ" julia-latexsubs)
(puthash "\\Epsilon" "Ε" julia-latexsubs)
(puthash "\\Zeta" "Ζ" julia-latexsubs)
(puthash "\\Eta" "Η" julia-latexsubs)
(puthash "\\Theta" "Θ" julia-latexsubs)
(puthash "\\Iota" "Ι" julia-latexsubs)
(puthash "\\Kappa" "Κ" julia-latexsubs)
(puthash "\\Lambda" "Λ" julia-latexsubs)
(puthash "\\upMu" "Μ" julia-latexsubs)
(puthash "\\upNu" "Ν" julia-latexsubs)
(puthash "\\Xi" "Ξ" julia-latexsubs)
(puthash "\\upOmicron" "Ο" julia-latexsubs)
(puthash "\\Pi" "Π" julia-latexsubs)
(puthash "\\Rho" "Ρ" julia-latexsubs)
(puthash "\\Sigma" "Σ" julia-latexsubs)
(puthash "\\Tau" "Τ" julia-latexsubs)
(puthash "\\Upsilon" "Υ" julia-latexsubs)
(puthash "\\Phi" "Φ" julia-latexsubs)
(puthash "\\Chi" "Χ" julia-latexsubs)
(puthash "\\Psi" "Ψ" julia-latexsubs)
(puthash "\\Omega" "Ω" julia-latexsubs)
(puthash "\\alpha" "α" julia-latexsubs)
(puthash "\\beta" "β" julia-latexsubs)
(puthash "\\gamma" "γ" julia-latexsubs)
(puthash "\\delta" "δ" julia-latexsubs)
(puthash "\\upepsilon" "ε" julia-latexsubs)
(puthash "\\zeta" "ζ" julia-latexsubs)
(puthash "\\eta" "η" julia-latexsubs)
(puthash "\\theta" "θ" julia-latexsubs)
(puthash "\\iota" "ι" julia-latexsubs)
(puthash "\\kappa" "κ" julia-latexsubs)
(puthash "\\lambda" "λ" julia-latexsubs)
(puthash "\\mu" "μ" julia-latexsubs)
(puthash "\\nu" "ν" julia-latexsubs)
(puthash "\\xi" "ξ" julia-latexsubs)
(puthash "\\upomicron" "ο" julia-latexsubs)
(puthash "\\pi" "π" julia-latexsubs)
(puthash "\\rho" "ρ" julia-latexsubs)
(puthash "\\varsigma" "ς" julia-latexsubs)
(puthash "\\sigma" "σ" julia-latexsubs)
(puthash "\\tau" "τ" julia-latexsubs)
(puthash "\\upsilon" "υ" julia-latexsubs)
(puthash "\\varphi" "φ" julia-latexsubs)
(puthash "\\chi" "χ" julia-latexsubs)
(puthash "\\psi" "ψ" julia-latexsubs)
(puthash "\\omega" "ω" julia-latexsubs)
(puthash "\\upvarbeta" "ϐ" julia-latexsubs)
(puthash "\\vartheta" "ϑ" julia-latexsubs)
(puthash "\\phi" "ϕ" julia-latexsubs)
(puthash "\\varpi" "ϖ" julia-latexsubs)
(puthash "\\upoldKoppa" "Ϙ" julia-latexsubs)
(puthash "\\upoldkoppa" "ϙ" julia-latexsubs)
(puthash "\\Stigma" "Ϛ" julia-latexsubs)
(puthash "\\upstigma" "ϛ" julia-latexsubs)
(puthash "\\Digamma" "Ϝ" julia-latexsubs)
(puthash "\\digamma" "ϝ" julia-latexsubs)
(puthash "\\Koppa" "Ϟ" julia-latexsubs)
(puthash "\\upkoppa" "ϟ" julia-latexsubs)
(puthash "\\Sampi" "Ϡ" julia-latexsubs)
(puthash "\\upsampi" "ϡ" julia-latexsubs)
(puthash "\\varkappa" "ϰ" julia-latexsubs)
(puthash "\\varrho" "ϱ" julia-latexsubs)
(puthash "\\textTheta" "ϴ" julia-latexsubs)
(puthash "\\epsilon" "ϵ" julia-latexsubs)
(puthash "\\backepsilon" "϶" julia-latexsubs)
(puthash "\\^A" "\u1d2c" julia-latexsubs)
(puthash "\\^B" "\u1d2e" julia-latexsubs)
(puthash "\\^D" "\u1d30" julia-latexsubs)
(puthash "\\^E" "\u1d31" julia-latexsubs)
(puthash "\\^G" "\u1d33" julia-latexsubs)
(puthash "\\^H" "\u1d34" julia-latexsubs)
(puthash "\\^I" "\u1d35" julia-latexsubs)
(puthash "\\^J" "\u1d36" julia-latexsubs)
(puthash "\\^K" "\u1d37" julia-latexsubs)
(puthash "\\^L" "\u1d38" julia-latexsubs)
(puthash "\\^M" "\u1d39" julia-latexsubs)
(puthash "\\^N" "\u1d3a" julia-latexsubs)
(puthash "\\^O" "\u1d3c" julia-latexsubs)
(puthash "\\^P" "\u1d3e" julia-latexsubs)
(puthash "\\^R" "\u1d3f" julia-latexsubs)
(puthash "\\^T" "\u1d40" julia-latexsubs)
(puthash "\\^U" "\u1d41" julia-latexsubs)
(puthash "\\^W" "\u1d42" julia-latexsubs)
(puthash "\\^a" "\u1d43" julia-latexsubs)
(puthash "\\^alpha" "\u1d45" julia-latexsubs)
(puthash "\\^b" "\u1d47" julia-latexsubs)
(puthash "\\^d" "\u1d48" julia-latexsubs)
(puthash "\\^e" "\u1d49" julia-latexsubs)
(puthash "\\^epsilon" "\u1d4b" julia-latexsubs)
(puthash "\\^g" "\u1d4d" julia-latexsubs)
(puthash "\\^k" "\u1d4f" julia-latexsubs)
(puthash "\\^m" "\u1d50" julia-latexsubs)
(puthash "\\^o" "\u1d52" julia-latexsubs)
(puthash "\\^p" "\u1d56" julia-latexsubs)
(puthash "\\^t" "\u1d57" julia-latexsubs)
(puthash "\\^u" "\u1d58" julia-latexsubs)
(puthash "\\^v" "\u1d5b" julia-latexsubs)
(puthash "\\^beta" "\u1d5d" julia-latexsubs)
(puthash "\\^gamma" "\u1d5e" julia-latexsubs)
(puthash "\\^delta" "\u1d5f" julia-latexsubs)
(puthash "\\^phi" "\u1d60" julia-latexsubs)
(puthash "\\^chi" "\u1d61" julia-latexsubs)
(puthash "\\_i" "\u1d62" julia-latexsubs)
(puthash "\\_r" "\u1d63" julia-latexsubs)
(puthash "\\_u" "\u1d64" julia-latexsubs)
(puthash "\\_v" "\u1d65" julia-latexsubs)
(puthash "\\_beta" "\u1d66" julia-latexsubs)
(puthash "\\_gamma" "\u1d67" julia-latexsubs)
(puthash "\\_rho" "\u1d68" julia-latexsubs)
(puthash "\\_phi" "\u1d69" julia-latexsubs)
(puthash "\\_chi" "\u1d6a" julia-latexsubs)
(puthash "\\^c" "\u1d9c" julia-latexsubs)
(puthash "\\^f" "\u1da0" julia-latexsubs)
(puthash "\\^iota" "\u1da5" julia-latexsubs)
(puthash "\\^Phi" "\u1db2" julia-latexsubs)
(puthash "\\^z" "\u1dbb" julia-latexsubs)
(puthash "\\^theta" "\u1dbf" julia-latexsubs)
(puthash "\\enspace" " " julia-latexsubs)
(puthash "\\quad" " " julia-latexsubs)
(puthash "\\thickspace" " " julia-latexsubs)
(puthash "\\thinspace" " " julia-latexsubs)
(puthash "\\hspace" " " julia-latexsubs)
(puthash "\\endash" "–" julia-latexsubs)
(puthash "\\emdash" "—" julia-latexsubs)
(puthash "\\Vert" "‖" julia-latexsubs)
(puthash "\\lq" "‘" julia-latexsubs)
(puthash "\\rq" "’" julia-latexsubs)
(puthash "\\Elzreapos" "‛" julia-latexsubs)
(puthash "\\textquotedblleft" "“" julia-latexsubs)
(puthash "\\textquotedblright" "”" julia-latexsubs)
(puthash "\\dagger" "†" julia-latexsubs)
(puthash "\\ddagger" "‡" julia-latexsubs)
(puthash "\\bullet" "•" julia-latexsubs)
(puthash "\\dots" "…" julia-latexsubs)
(puthash "\\textperthousand" "‰" julia-latexsubs)
(puthash "\\textpertenthousand" "‱" julia-latexsubs)
(puthash "\\prime" "′" julia-latexsubs)
(puthash "\\pprime" "″" julia-latexsubs)
(puthash "\\ppprime" "‴" julia-latexsubs)
(puthash "\\backprime" "‵" julia-latexsubs)
(puthash "\\backpprime" "‶" julia-latexsubs)
(puthash "\\backppprime" "‷" julia-latexsubs)
(puthash "\\guilsinglleft" "‹" julia-latexsubs)
(puthash "\\guilsinglright" "›" julia-latexsubs)
(puthash "\\tieconcat" "⁀" julia-latexsubs)
(puthash "\\pppprime" "⁗" julia-latexsubs)
(puthash "\\nolinebreak" "\u2060" julia-latexsubs)
(puthash "\\^0" "⁰" julia-latexsubs)
(puthash "\\^i" "ⁱ" julia-latexsubs)
(puthash "\\^4" "⁴" julia-latexsubs)
(puthash "\\^5" "⁵" julia-latexsubs)
(puthash "\\^6" "⁶" julia-latexsubs)
(puthash "\\^7" "⁷" julia-latexsubs)
(puthash "\\^8" "⁸" julia-latexsubs)
(puthash "\\^9" "⁹" julia-latexsubs)
(puthash "\\^+" "⁺" julia-latexsubs)
(puthash "\\^-" "⁻" julia-latexsubs)
(puthash "\\^=" "⁼" julia-latexsubs)
(puthash "\\^(" "⁽" julia-latexsubs)
(puthash "\\^)" "⁾" julia-latexsubs)
(puthash "\\^n" "ⁿ" julia-latexsubs)
(puthash "\\_0" "₀" julia-latexsubs)
(puthash "\\_1" "₁" julia-latexsubs)
(puthash "\\_2" "₂" julia-latexsubs)
(puthash "\\_3" "₃" julia-latexsubs)
(puthash "\\_4" "₄" julia-latexsubs)
(puthash "\\_5" "₅" julia-latexsubs)
(puthash "\\_6" "₆" julia-latexsubs)
(puthash "\\_7" "₇" julia-latexsubs)
(puthash "\\_8" "₈" julia-latexsubs)
(puthash "\\_9" "₉" julia-latexsubs)
(puthash "\\_+" "₊" julia-latexsubs)
(puthash "\\_-" "₋" julia-latexsubs)
(puthash "\\_=" "₌" julia-latexsubs)
(puthash "\\_(" "₍" julia-latexsubs)
(puthash "\\_)" "₎" julia-latexsubs)
(puthash "\\_a" "\u2090" julia-latexsubs)
(puthash "\\_e" "\u2091" julia-latexsubs)
(puthash "\\_o" "\u2092" julia-latexsubs)
(puthash "\\_x" "\u2093" julia-latexsubs)
(puthash "\\_schwa" "\u2094" julia-latexsubs)
(puthash "\\_h" "\u2095" julia-latexsubs)
(puthash "\\_k" "\u2096" julia-latexsubs)
(puthash "\\_l" "\u2097" julia-latexsubs)
(puthash "\\_m" "\u2098" julia-latexsubs)
(puthash "\\_n" "\u2099" julia-latexsubs)
(puthash "\\_p" "\u209a" julia-latexsubs)
(puthash "\\_s" "\u209b" julia-latexsubs)
(puthash "\\_t" "\u209c" julia-latexsubs)
(puthash "\\Elzpes" "₧" julia-latexsubs)
(puthash "\\euro" "€" julia-latexsubs)
(puthash "\\leftharpoonaccent" "⃐" julia-latexsubs)
(puthash "\\rightharpoonaccent" "⃑" julia-latexsubs)
(puthash "\\vertoverlay" "⃒" julia-latexsubs)
(puthash "\\overleftarrow" "⃖" julia-latexsubs)
(puthash "\\vec" "⃗" julia-latexsubs)
(puthash "\\dddot" "⃛" julia-latexsubs)
(puthash "\\ddddot" "⃜" julia-latexsubs)
(puthash "\\enclosecircle" "⃝" julia-latexsubs)
(puthash "\\enclosesquare" "⃞" julia-latexsubs)
(puthash "\\enclosediamond" "⃟" julia-latexsubs)
(puthash "\\overleftrightarrow" "⃡" julia-latexsubs)
(puthash "\\enclosetriangle" "⃤" julia-latexsubs)
(puthash "\\annuity" "⃧" julia-latexsubs)
(puthash "\\threeunderdot" "⃨" julia-latexsubs)
(puthash "\\widebridgeabove" "⃩" julia-latexsubs)
(puthash "\\underrightharpoondown" "\u20ec" julia-latexsubs)
(puthash "\\underleftharpoondown" "\u20ed" julia-latexsubs)
(puthash "\\underleftarrow" "\u20ee" julia-latexsubs)
(puthash "\\underrightarrow" "\u20ef" julia-latexsubs)
(puthash "\\asteraccent" "\u20f0" julia-latexsubs)
(puthash "\\BbbC" "ℂ" julia-latexsubs)
(puthash "\\Eulerconst" "ℇ" julia-latexsubs)
(puthash "\\mscrg" "ℊ" julia-latexsubs)
(puthash "\\mscrH" "ℋ" julia-latexsubs)
(puthash "\\mfrakH" "ℌ" julia-latexsubs)
(puthash "\\BbbH" "ℍ" julia-latexsubs)
(puthash "\\Planckconst" "ℎ" julia-latexsubs)
(puthash "\\hslash" "ℏ" julia-latexsubs)
(puthash "\\mscrI" "ℐ" julia-latexsubs)
(puthash "\\Im" "ℑ" julia-latexsubs)
(puthash "\\mscrL" "ℒ" julia-latexsubs)
(puthash "\\ell" "ℓ" julia-latexsubs)
(puthash "\\BbbN" "ℕ" julia-latexsubs)
(puthash "\\textnumero" "№" julia-latexsubs)
(puthash "\\wp" "℘" julia-latexsubs)
(puthash "\\BbbP" "ℙ" julia-latexsubs)
(puthash "\\BbbQ" "ℚ" julia-latexsubs)
(puthash "\\mscrR" "ℛ" julia-latexsubs)
(puthash "\\Re" "ℜ" julia-latexsubs)
(puthash "\\BbbR" "ℝ" julia-latexsubs)
(puthash "\\Elzxrat" "℞" julia-latexsubs)
(puthash "\\texttrademark" "™" julia-latexsubs)
(puthash "\\BbbZ" "ℤ" julia-latexsubs)
(puthash "\\mho" "℧" julia-latexsubs)
(puthash "\\mfrakZ" "ℨ" julia-latexsubs)
(puthash "\\turnediota" "℩" julia-latexsubs)
(puthash "\\Angstrom" "Å" julia-latexsubs)
(puthash "\\mscrB" "ℬ" julia-latexsubs)
(puthash "\\mfrakC" "ℭ" julia-latexsubs)
(puthash "\\mscre" "ℯ" julia-latexsubs)
(puthash "\\mscrE" "ℰ" julia-latexsubs)
(puthash "\\mscrF" "ℱ" julia-latexsubs)
(puthash "\\Finv" "Ⅎ" julia-latexsubs)
(puthash "\\mscrM" "ℳ" julia-latexsubs)
(puthash "\\mscro" "ℴ" julia-latexsubs)
(puthash "\\aleph" "ℵ" julia-latexsubs)
(puthash "\\beth" "ℶ" julia-latexsubs)
(puthash "\\gimel" "ℷ" julia-latexsubs)
(puthash "\\daleth" "ℸ" julia-latexsubs)
(puthash "\\Bbbpi" "\u213c" julia-latexsubs)
(puthash "\\Bbbgamma" "ℽ" julia-latexsubs)
(puthash "\\BbbGamma" "ℾ" julia-latexsubs)
(puthash "\\BbbPi" "ℿ" julia-latexsubs)
(puthash "\\bbsum" "⅀" julia-latexsubs)
(puthash "\\Game" "⅁" julia-latexsubs)
(puthash "\\sansLturned" "⅂" julia-latexsubs)
(puthash "\\sansLmirrored" "⅃" julia-latexsubs)
(puthash "\\Yup" "⅄" julia-latexsubs)
(puthash "\\mitBbbD" "ⅅ" julia-latexsubs)
(puthash "\\mitBbbd" "ⅆ" julia-latexsubs)
(puthash "\\mitBbbe" "ⅇ" julia-latexsubs)
(puthash "\\mitBbbi" "ⅈ" julia-latexsubs)
(puthash "\\mitBbbj" "ⅉ" julia-latexsubs)
(puthash "\\PropertyLine" "⅊" julia-latexsubs)
(puthash "\\upand" "⅋" julia-latexsubs)
(puthash "\\leftarrow" "←" julia-latexsubs)
(puthash "\\uparrow" "↑" julia-latexsubs)
(puthash "\\rightarrow" "→" julia-latexsubs)
(puthash "\\downarrow" "↓" julia-latexsubs)
(puthash "\\leftrightarrow" "↔" julia-latexsubs)
(puthash "\\updownarrow" "↕" julia-latexsubs)
(puthash "\\nwarrow" "↖" julia-latexsubs)
(puthash "\\nearrow" "↗" julia-latexsubs)
(puthash "\\searrow" "↘" julia-latexsubs)
(puthash "\\swarrow" "↙" julia-latexsubs)
(puthash "\\nleftarrow" "↚" julia-latexsubs)
(puthash "\\nrightarrow" "↛" julia-latexsubs)
(puthash "\\twoheadleftarrow" "↞" julia-latexsubs)
(puthash "\\twoheaduparrow" "↟" julia-latexsubs)
(puthash "\\twoheadrightarrow" "↠" julia-latexsubs)
(puthash "\\twoheaddownarrow" "↡" julia-latexsubs)
(puthash "\\leftarrowtail" "↢" julia-latexsubs)
(puthash "\\rightarrowtail" "↣" julia-latexsubs)
(puthash "\\mapsfrom" "↤" julia-latexsubs)
(puthash "\\mapsup" "↥" julia-latexsubs)
(puthash "\\mapsto" "↦" julia-latexsubs)
(puthash "\\mapsdown" "↧" julia-latexsubs)
(puthash "\\updownarrowbar" "↨" julia-latexsubs)
(puthash "\\hookleftarrow" "↩" julia-latexsubs)
(puthash "\\hookrightarrow" "↪" julia-latexsubs)
(puthash "\\looparrowleft" "↫" julia-latexsubs)
(puthash "\\looparrowright" "↬" julia-latexsubs)
(puthash "\\leftrightsquigarrow" "↭" julia-latexsubs)
(puthash "\\nleftrightarrow" "↮" julia-latexsubs)
(puthash "\\downzigzagarrow" "↯" julia-latexsubs)
(puthash "\\Lsh" "↰" julia-latexsubs)
(puthash "\\Rsh" "↱" julia-latexsubs)
(puthash "\\Ldsh" "↲" julia-latexsubs)
(puthash "\\Rdsh" "↳" julia-latexsubs)
(puthash "\\linefeed" "↴" julia-latexsubs)
(puthash "\\carriagereturn" "↵" julia-latexsubs)
(puthash "\\curvearrowleft" "↶" julia-latexsubs)
(puthash "\\curvearrowright" "↷" julia-latexsubs)
(puthash "\\barovernorthwestarrow" "↸" julia-latexsubs)
(puthash "\\barleftarrowrightarrowbar" "↹" julia-latexsubs)
(puthash "\\circlearrowleft" "↺" julia-latexsubs)
(puthash "\\circlearrowright" "↻" julia-latexsubs)
(puthash "\\leftharpoonup" "↼" julia-latexsubs)
(puthash "\\leftharpoondown" "↽" julia-latexsubs)
(puthash "\\upharpoonleft" "↾" julia-latexsubs)
(puthash "\\upharpoonright" "↿" julia-latexsubs)
(puthash "\\rightharpoonup" "⇀" julia-latexsubs)
(puthash "\\rightharpoondown" "⇁" julia-latexsubs)
(puthash "\\downharpoonright" "⇂" julia-latexsubs)
(puthash "\\downharpoonleft" "⇃" julia-latexsubs)
(puthash "\\rightleftarrows" "⇄" julia-latexsubs)
(puthash "\\dblarrowupdown" "⇅" julia-latexsubs)
(puthash "\\leftrightarrows" "⇆" julia-latexsubs)
(puthash "\\leftleftarrows" "⇇" julia-latexsubs)
(puthash "\\upuparrows" "⇈" julia-latexsubs)
(puthash "\\rightrightarrows" "⇉" julia-latexsubs)
(puthash "\\downdownarrows" "⇊" julia-latexsubs)
(puthash "\\leftrightharpoons" "⇋" julia-latexsubs)
(puthash "\\rightleftharpoons" "⇌" julia-latexsubs)
(puthash "\\nLeftarrow" "⇍" julia-latexsubs)
(puthash "\\nLeftrightarrow" "⇎" julia-latexsubs)
(puthash "\\nRightarrow" "⇏" julia-latexsubs)
(puthash "\\Leftarrow" "⇐" julia-latexsubs)
(puthash "\\Uparrow" "⇑" julia-latexsubs)
(puthash "\\Rightarrow" "⇒" julia-latexsubs)
(puthash "\\Downarrow" "⇓" julia-latexsubs)
(puthash "\\Leftrightarrow" "⇔" julia-latexsubs)
(puthash "\\Updownarrow" "⇕" julia-latexsubs)
(puthash "\\Nwarrow" "⇖" julia-latexsubs)
(puthash "\\Nearrow" "⇗" julia-latexsubs)
(puthash "\\Searrow" "⇘" julia-latexsubs)
(puthash "\\Swarrow" "⇙" julia-latexsubs)
(puthash "\\Lleftarrow" "⇚" julia-latexsubs)
(puthash "\\Rrightarrow" "⇛" julia-latexsubs)
(puthash "\\leftsquigarrow" "⇜" julia-latexsubs)
(puthash "\\rightsquigarrow" "⇝" julia-latexsubs)
(puthash "\\nHuparrow" "⇞" julia-latexsubs)
(puthash "\\nHdownarrow" "⇟" julia-latexsubs)
(puthash "\\leftdasharrow" "⇠" julia-latexsubs)
(puthash "\\updasharrow" "⇡" julia-latexsubs)
(puthash "\\rightdasharrow" "⇢" julia-latexsubs)
(puthash "\\downdasharrow" "⇣" julia-latexsubs)
(puthash "\\barleftarrow" "⇤" julia-latexsubs)
(puthash "\\rightarrowbar" "⇥" julia-latexsubs)
(puthash "\\leftwhitearrow" "⇦" julia-latexsubs)
(puthash "\\upwhitearrow" "⇧" julia-latexsubs)
(puthash "\\rightwhitearrow" "⇨" julia-latexsubs)
(puthash "\\downwhitearrow" "⇩" julia-latexsubs)
(puthash "\\whitearrowupfrombar" "⇪" julia-latexsubs)
(puthash "\\circleonrightarrow" "⇴" julia-latexsubs)
(puthash "\\DownArrowUpArrow" "⇵" julia-latexsubs)
(puthash "\\rightthreearrows" "⇶" julia-latexsubs)
(puthash "\\nvleftarrow" "⇷" julia-latexsubs)
(puthash "\\nvrightarrow" "⇸" julia-latexsubs)
(puthash "\\nvleftrightarrow" "⇹" julia-latexsubs)
(puthash "\\nVleftarrow" "⇺" julia-latexsubs)
(puthash "\\nVrightarrow" "⇻" julia-latexsubs)
(puthash "\\nVleftrightarrow" "⇼" julia-latexsubs)
(puthash "\\leftarrowtriangle" "⇽" julia-latexsubs)
(puthash "\\rightarrowtriangle" "⇾" julia-latexsubs)
(puthash "\\leftrightarrowtriangle" "⇿" julia-latexsubs)
(puthash "\\forall" "∀" julia-latexsubs)
(puthash "\\complement" "∁" julia-latexsubs)
(puthash "\\partial" "∂" julia-latexsubs)
(puthash "\\exists" "∃" julia-latexsubs)
(puthash "\\nexists" "∄" julia-latexsubs)
(puthash "\\varnothing" "∅" julia-latexsubs)
(puthash "\\increment" "∆" julia-latexsubs)
(puthash "\\del" "∇" julia-latexsubs)
(puthash "\\nabla" "∇" julia-latexsubs)
(puthash "\\in" "∈" julia-latexsubs)
(puthash "\\notin" "∉" julia-latexsubs)
(puthash "\\smallin" "∊" julia-latexsubs)
(puthash "\\ni" "∋" julia-latexsubs)
(puthash "\\nni" "∌" julia-latexsubs)
(puthash "\\smallni" "∍" julia-latexsubs)
(puthash "\\QED" "∎" julia-latexsubs)
(puthash "\\prod" "∏" julia-latexsubs)
(puthash "\\coprod" "∐" julia-latexsubs)
(puthash "\\sum" "∑" julia-latexsubs)
(puthash "\\minus" "−" julia-latexsubs)
(puthash "\\mp" "∓" julia-latexsubs)
(puthash "\\dotplus" "∔" julia-latexsubs)
(puthash "\\setminus" "∖" julia-latexsubs)
(puthash "\\ast" "∗" julia-latexsubs)
(puthash "\\circ" "∘" julia-latexsubs)
(puthash "\\vysmblkcircle" "∙" julia-latexsubs)
(puthash "\\surd" "√" julia-latexsubs)
(puthash "\\sqrt" "√" julia-latexsubs)
(puthash "\\cbrt" "∛" julia-latexsubs)
(puthash "\\fourthroot" "∜" julia-latexsubs)
(puthash "\\propto" "∝" julia-latexsubs)
(puthash "\\infty" "∞" julia-latexsubs)
(puthash "\\rightangle" "∟" julia-latexsubs)
(puthash "\\angle" "∠" julia-latexsubs)
(puthash "\\measuredangle" "∡" julia-latexsubs)
(puthash "\\sphericalangle" "∢" julia-latexsubs)
(puthash "\\mid" "∣" julia-latexsubs)
(puthash "\\nmid" "∤" julia-latexsubs)
(puthash "\\parallel" "∥" julia-latexsubs)
(puthash "\\nparallel" "∦" julia-latexsubs)
(puthash "\\wedge" "∧" julia-latexsubs)
(puthash "\\vee" "∨" julia-latexsubs)
(puthash "\\cap" "∩" julia-latexsubs)
(puthash "\\cup" "∪" julia-latexsubs)
(puthash "\\int" "∫" julia-latexsubs)
(puthash "\\iint" "∬" julia-latexsubs)
(puthash "\\iiint" "∭" julia-latexsubs)
(puthash "\\oint" "∮" julia-latexsubs)
(puthash "\\oiint" "∯" julia-latexsubs)
(puthash "\\oiiint" "∰" julia-latexsubs)
(puthash "\\clwintegral" "∱" julia-latexsubs)
(puthash "\\varointclockwise" "∲" julia-latexsubs)
(puthash "\\ointctrclockwise" "∳" julia-latexsubs)
(puthash "\\therefore" "∴" julia-latexsubs)
(puthash "\\because" "∵" julia-latexsubs)
(puthash "\\Colon" "∷" julia-latexsubs)
(puthash "\\dotminus" "∸" julia-latexsubs)
(puthash "\\dotsminusdots" "∺" julia-latexsubs)
(puthash "\\kernelcontraction" "∻" julia-latexsubs)
(puthash "\\sim" "∼" julia-latexsubs)
(puthash "\\backsim" "∽" julia-latexsubs)
(puthash "\\lazysinv" "∾" julia-latexsubs)
(puthash "\\sinewave" "∿" julia-latexsubs)
(puthash "\\wr" "≀" julia-latexsubs)
(puthash "\\nsim" "≁" julia-latexsubs)
(puthash "\\eqsim" "≂" julia-latexsubs)
(puthash "\\neqsim" "≂̸" julia-latexsubs)
(puthash "\\simeq" "≃" julia-latexsubs)
(puthash "\\nsime" "≄" julia-latexsubs)
(puthash "\\cong" "≅" julia-latexsubs)
(puthash "\\approxnotequal" "≆" julia-latexsubs)
(puthash "\\ncong" "≇" julia-latexsubs)
(puthash "\\approx" "≈" julia-latexsubs)
(puthash "\\napprox" "≉" julia-latexsubs)
(puthash "\\approxeq" "≊" julia-latexsubs)
(puthash "\\tildetrpl" "≋" julia-latexsubs)
(puthash "\\allequal" "≌" julia-latexsubs)
(puthash "\\asymp" "≍" julia-latexsubs)
(puthash "\\Bumpeq" "≎" julia-latexsubs)
(puthash "\\nBumpeq" "≎̸" julia-latexsubs)
(puthash "\\bumpeq" "≏" julia-latexsubs)
(puthash "\\nbumpeq" "≏̸" julia-latexsubs)
(puthash "\\doteq" "≐" julia-latexsubs)
(puthash "\\Doteq" "≑" julia-latexsubs)
(puthash "\\fallingdotseq" "≒" julia-latexsubs)
(puthash "\\risingdotseq" "≓" julia-latexsubs)
(puthash "\\coloneq" "≔" julia-latexsubs)
(puthash "\\eqcolon" "≕" julia-latexsubs)
(puthash "\\eqcirc" "≖" julia-latexsubs)
(puthash "\\circeq" "≗" julia-latexsubs)
(puthash "\\arceq" "≘" julia-latexsubs)
(puthash "\\wedgeq" "≙" julia-latexsubs)
(puthash "\\veeeq" "≚" julia-latexsubs)
(puthash "\\starequal" "≛" julia-latexsubs)
(puthash "\\triangleq" "≜" julia-latexsubs)
(puthash "\\eqdef" "≝" julia-latexsubs)
(puthash "\\measeq" "≞" julia-latexsubs)
(puthash "\\questeq" "≟" julia-latexsubs)
(puthash "\\ne" "≠" julia-latexsubs)
(puthash "\\equiv" "≡" julia-latexsubs)
(puthash "\\nequiv" "≢" julia-latexsubs)
(puthash "\\Equiv" "≣" julia-latexsubs)
(puthash "\\le" "≤" julia-latexsubs)
(puthash "\\ge" "≥" julia-latexsubs)
(puthash "\\leqq" "≦" julia-latexsubs)
(puthash "\\geqq" "≧" julia-latexsubs)
(puthash "\\lneqq" "≨" julia-latexsubs)
(puthash "\\lvertneqq" "≨︀" julia-latexsubs)
(puthash "\\gneqq" "≩" julia-latexsubs)
(puthash "\\gvertneqq" "≩︀" julia-latexsubs)
(puthash "\\ll" "≪" julia-latexsubs)
(puthash "\\NotLessLess" "≪̸" julia-latexsubs)
(puthash "\\gg" "≫" julia-latexsubs)
(puthash "\\NotGreaterGreater" "≫̸" julia-latexsubs)
(puthash "\\between" "≬" julia-latexsubs)
(puthash "\\nasymp" "≭" julia-latexsubs)
(puthash "\\nless" "≮" julia-latexsubs)
(puthash "\\ngtr" "≯" julia-latexsubs)
(puthash "\\nleq" "≰" julia-latexsubs)
(puthash "\\ngeq" "≱" julia-latexsubs)
(puthash "\\lesssim" "≲" julia-latexsubs)
(puthash "\\gtrsim" "≳" julia-latexsubs)
(puthash "\\nlesssim" "≴" julia-latexsubs)
(puthash "\\ngtrsim" "≵" julia-latexsubs)
(puthash "\\lessgtr" "≶" julia-latexsubs)
(puthash "\\gtrless" "≷" julia-latexsubs)
(puthash "\\notlessgreater" "≸" julia-latexsubs)
(puthash "\\notgreaterless" "≹" julia-latexsubs)
(puthash "\\prec" "≺" julia-latexsubs)
(puthash "\\succ" "≻" julia-latexsubs)
(puthash "\\preccurlyeq" "≼" julia-latexsubs)
(puthash "\\succcurlyeq" "≽" julia-latexsubs)
(puthash "\\precsim" "≾" julia-latexsubs)
(puthash "\\nprecsim" "≾̸" julia-latexsubs)
(puthash "\\succsim" "≿" julia-latexsubs)
(puthash "\\nsuccsim" "≿̸" julia-latexsubs)
(puthash "\\nprec" "⊀" julia-latexsubs)
(puthash "\\nsucc" "⊁" julia-latexsubs)
(puthash "\\subset" "⊂" julia-latexsubs)
(puthash "\\supset" "⊃" julia-latexsubs)
(puthash "\\nsubset" "⊄" julia-latexsubs)
(puthash "\\nsupset" "⊅" julia-latexsubs)
(puthash "\\subseteq" "⊆" julia-latexsubs)
(puthash "\\supseteq" "⊇" julia-latexsubs)
(puthash "\\nsubseteq" "⊈" julia-latexsubs)
(puthash "\\nsupseteq" "⊉" julia-latexsubs)
(puthash "\\subsetneq" "⊊" julia-latexsubs)
(puthash "\\varsubsetneqq" "⊊︀" julia-latexsubs)
(puthash "\\supsetneq" "⊋" julia-latexsubs)
(puthash "\\varsupsetneq" "⊋︀" julia-latexsubs)
(puthash "\\cupdot" "⊍" julia-latexsubs)
(puthash "\\uplus" "⊎" julia-latexsubs)
(puthash "\\sqsubset" "⊏" julia-latexsubs)
(puthash "\\NotSquareSubset" "⊏̸" julia-latexsubs)
(puthash "\\sqsupset" "⊐" julia-latexsubs)
(puthash "\\NotSquareSuperset" "⊐̸" julia-latexsubs)
(puthash "\\sqsubseteq" "⊑" julia-latexsubs)
(puthash "\\sqsupseteq" "⊒" julia-latexsubs)
(puthash "\\sqcap" "⊓" julia-latexsubs)
(puthash "\\sqcup" "⊔" julia-latexsubs)
(puthash "\\oplus" "⊕" julia-latexsubs)
(puthash "\\ominus" "⊖" julia-latexsubs)
(puthash "\\otimes" "⊗" julia-latexsubs)
(puthash "\\oslash" "⊘" julia-latexsubs)
(puthash "\\odot" "⊙" julia-latexsubs)
(puthash "\\circledcirc" "⊚" julia-latexsubs)
(puthash "\\circledast" "⊛" julia-latexsubs)
(puthash "\\circledequal" "⊜" julia-latexsubs)
(puthash "\\circleddash" "⊝" julia-latexsubs)
(puthash "\\boxplus" "⊞" julia-latexsubs)
(puthash "\\boxminus" "⊟" julia-latexsubs)
(puthash "\\boxtimes" "⊠" julia-latexsubs)
(puthash "\\boxdot" "⊡" julia-latexsubs)
(puthash "\\vdash" "⊢" julia-latexsubs)
(puthash "\\dashv" "⊣" julia-latexsubs)
(puthash "\\top" "⊤" julia-latexsubs)
(puthash "\\bot" "⊥" julia-latexsubs)
(puthash "\\models" "⊧" julia-latexsubs)
(puthash "\\vDash" "⊨" julia-latexsubs)
(puthash "\\Vdash" "⊩" julia-latexsubs)
(puthash "\\Vvdash" "⊪" julia-latexsubs)
(puthash "\\VDash" "⊫" julia-latexsubs)
(puthash "\\nvdash" "⊬" julia-latexsubs)
(puthash "\\nvDash" "⊭" julia-latexsubs)
(puthash "\\nVdash" "⊮" julia-latexsubs)
(puthash "\\nVDash" "⊯" julia-latexsubs)
(puthash "\\prurel" "⊰" julia-latexsubs)
(puthash "\\scurel" "⊱" julia-latexsubs)
(puthash "\\vartriangleleft" "⊲" julia-latexsubs)
(puthash "\\vartriangleright" "⊳" julia-latexsubs)
(puthash "\\trianglelefteq" "⊴" julia-latexsubs)
(puthash "\\trianglerighteq" "⊵" julia-latexsubs)
(puthash "\\original" "⊶" julia-latexsubs)
(puthash "\\image" "⊷" julia-latexsubs)
(puthash "\\multimap" "⊸" julia-latexsubs)
(puthash "\\hermitconjmatrix" "⊹" julia-latexsubs)
(puthash "\\intercal" "⊺" julia-latexsubs)
(puthash "\\veebar" "⊻" julia-latexsubs)
(puthash "\\barwedge" "⊼" julia-latexsubs)
(puthash "\\barvee" "⊽" julia-latexsubs)
(puthash "\\rightanglearc" "⊾" julia-latexsubs)
(puthash "\\varlrtriangle" "⊿" julia-latexsubs)
(puthash "\\bigwedge" "⋀" julia-latexsubs)
(puthash "\\bigvee" "⋁" julia-latexsubs)
(puthash "\\bigcap" "⋂" julia-latexsubs)
(puthash "\\bigcup" "⋃" julia-latexsubs)
(puthash "\\diamond" "⋄" julia-latexsubs)
(puthash "\\cdot" "⋅" julia-latexsubs)
(puthash "\\star" "⋆" julia-latexsubs)
(puthash "\\divideontimes" "⋇" julia-latexsubs)
(puthash "\\bowtie" "⋈" julia-latexsubs)
(puthash "\\ltimes" "⋉" julia-latexsubs)
(puthash "\\rtimes" "⋊" julia-latexsubs)
(puthash "\\leftthreetimes" "⋋" julia-latexsubs)
(puthash "\\rightthreetimes" "⋌" julia-latexsubs)
(puthash "\\backsimeq" "⋍" julia-latexsubs)
(puthash "\\curlyvee" "⋎" julia-latexsubs)
(puthash "\\curlywedge" "⋏" julia-latexsubs)
(puthash "\\Subset" "⋐" julia-latexsubs)
(puthash "\\Supset" "⋑" julia-latexsubs)
(puthash "\\Cap" "⋒" julia-latexsubs)
(puthash "\\Cup" "⋓" julia-latexsubs)
(puthash "\\pitchfork" "⋔" julia-latexsubs)
(puthash "\\equalparallel" "⋕" julia-latexsubs)
(puthash "\\lessdot" "⋖" julia-latexsubs)
(puthash "\\gtrdot" "⋗" julia-latexsubs)
(puthash "\\verymuchless" "⋘" julia-latexsubs)
(puthash "\\ggg" "⋙" julia-latexsubs)
(puthash "\\lesseqgtr" "⋚" julia-latexsubs)
(puthash "\\gtreqless" "⋛" julia-latexsubs)
(puthash "\\eqless" "⋜" julia-latexsubs)
(puthash "\\eqgtr" "⋝" julia-latexsubs)
(puthash "\\curlyeqprec" "⋞" julia-latexsubs)
(puthash "\\curlyeqsucc" "⋟" julia-latexsubs)
(puthash "\\npreccurlyeq" "⋠" julia-latexsubs)
(puthash "\\nsucccurlyeq" "⋡" julia-latexsubs)
(puthash "\\nsqsubseteq" "⋢" julia-latexsubs)
(puthash "\\nsqsupseteq" "⋣" julia-latexsubs)
(puthash "\\sqsubsetneq" "⋤" julia-latexsubs)
(puthash "\\Elzsqspne" "⋥" julia-latexsubs)
(puthash "\\lnsim" "⋦" julia-latexsubs)
(puthash "\\gnsim" "⋧" julia-latexsubs)
(puthash "\\precnsim" "⋨" julia-latexsubs)
(puthash "\\succnsim" "⋩" julia-latexsubs)
(puthash "\\ntriangleleft" "⋪" julia-latexsubs)
(puthash "\\ntriangleright" "⋫" julia-latexsubs)
(puthash "\\ntrianglelefteq" "⋬" julia-latexsubs)
(puthash "\\ntrianglerighteq" "⋭" julia-latexsubs)
(puthash "\\vdots" "⋮" julia-latexsubs)
(puthash "\\cdots" "⋯" julia-latexsubs)
(puthash "\\adots" "⋰" julia-latexsubs)
(puthash "\\ddots" "⋱" julia-latexsubs)
(puthash "\\disin" "⋲" julia-latexsubs)
(puthash "\\varisins" "⋳" julia-latexsubs)
(puthash "\\isins" "⋴" julia-latexsubs)
(puthash "\\isindot" "⋵" julia-latexsubs)
(puthash "\\varisinobar" "⋶" julia-latexsubs)
(puthash "\\isinobar" "⋷" julia-latexsubs)
(puthash "\\isinvb" "⋸" julia-latexsubs)
(puthash "\\isinE" "⋹" julia-latexsubs)
(puthash "\\nisd" "⋺" julia-latexsubs)
(puthash "\\varnis" "⋻" julia-latexsubs)
(puthash "\\nis" "⋼" julia-latexsubs)
(puthash "\\varniobar" "⋽" julia-latexsubs)
(puthash "\\niobar" "⋾" julia-latexsubs)
(puthash "\\bagmember" "⋿" julia-latexsubs)
(puthash "\\diameter" "⌀" julia-latexsubs)
(puthash "\\house" "⌂" julia-latexsubs)
(puthash "\\vardoublebarwedge" "⌆" julia-latexsubs)
(puthash "\\lceil" "⌈" julia-latexsubs)
(puthash "\\rceil" "⌉" julia-latexsubs)
(puthash "\\lfloor" "⌊" julia-latexsubs)
(puthash "\\rfloor" "⌋" julia-latexsubs)
(puthash "\\invnot" "⌐" julia-latexsubs)
(puthash "\\sqlozenge" "⌑" julia-latexsubs)
(puthash "\\profline" "⌒" julia-latexsubs)
(puthash "\\profsurf" "⌓" julia-latexsubs)
(puthash "\\recorder" "⌕" julia-latexsubs)
(puthash "\\viewdata" "⌗" julia-latexsubs)
(puthash "\\turnednot" "⌙" julia-latexsubs)
(puthash "\\ulcorner" "⌜" julia-latexsubs)
(puthash "\\urcorner" "⌝" julia-latexsubs)
(puthash "\\llcorner" "⌞" julia-latexsubs)
(puthash "\\lrcorner" "⌟" julia-latexsubs)
(puthash "\\frown" "⌢" julia-latexsubs)
(puthash "\\smile" "⌣" julia-latexsubs)
(puthash "\\varhexagonlrbonds" "⌬" julia-latexsubs)
(puthash "\\conictaper" "⌲" julia-latexsubs)
(puthash "\\topbot" "⌶" julia-latexsubs)
(puthash "\\obar" "⌽" julia-latexsubs)
(puthash "\\APLnotslash" "⌿" julia-latexsubs)
(puthash "\\APLnotbackslash" "⍀" julia-latexsubs)
(puthash "\\APLboxupcaret" "⍓" julia-latexsubs)
(puthash "\\APLboxquestion" "⍰" julia-latexsubs)
(puthash "\\hexagon" "⎔" julia-latexsubs)
(puthash "\\Elzdlcorn" "⎣" julia-latexsubs)
(puthash "\\lmoustache" "⎰" julia-latexsubs)
(puthash "\\rmoustache" "⎱" julia-latexsubs)
(puthash "\\overbracket" "⎴" julia-latexsubs)
(puthash "\\underbracket" "⎵" julia-latexsubs)
(puthash "\\bbrktbrk" "⎶" julia-latexsubs)
(puthash "\\sqrtbottom" "⎷" julia-latexsubs)
(puthash "\\lvboxline" "⎸" julia-latexsubs)
(puthash "\\rvboxline" "⎹" julia-latexsubs)
(puthash "\\varcarriagereturn" "⏎" julia-latexsubs)
(puthash "\\trapezium" "\u23e2" julia-latexsubs)
(puthash "\\benzenr" "\u23e3" julia-latexsubs)
(puthash "\\strns" "\u23e4" julia-latexsubs)
(puthash "\\fltns" "\u23e5" julia-latexsubs)
(puthash "\\accurrent" "\u23e6" julia-latexsubs)
(puthash "\\elinters" "\u23e7" julia-latexsubs)
(puthash "\\blanksymbol" "␢" julia-latexsubs)
(puthash "\\textvisiblespace" "␣" julia-latexsubs)
(puthash "\\circledS" "Ⓢ" julia-latexsubs)
(puthash "\\Elzdshfnc" "┆" julia-latexsubs)
(puthash "\\Elzsqfnw" "┙" julia-latexsubs)
(puthash "\\diagup" "╱" julia-latexsubs)
(puthash "\\diagdown" "╲" julia-latexsubs)
(puthash "\\blockuphalf" "▀" julia-latexsubs)
(puthash "\\blocklowhalf" "▄" julia-latexsubs)
(puthash "\\blockfull" "█" julia-latexsubs)
(puthash "\\blocklefthalf" "▌" julia-latexsubs)
(puthash "\\blockrighthalf" "▐" julia-latexsubs)
(puthash "\\blockqtrshaded" "░" julia-latexsubs)
(puthash "\\blockhalfshaded" "▒" julia-latexsubs)
(puthash "\\blockthreeqtrshaded" "▓" julia-latexsubs)
(puthash "\\blacksquare" "■" julia-latexsubs)
(puthash "\\square" "□" julia-latexsubs)
(puthash "\\squoval" "▢" julia-latexsubs)
(puthash "\\blackinwhitesquare" "▣" julia-latexsubs)
(puthash "\\squarehfill" "▤" julia-latexsubs)
(puthash "\\squarevfill" "▥" julia-latexsubs)
(puthash "\\squarehvfill" "▦" julia-latexsubs)
(puthash "\\squarenwsefill" "▧" julia-latexsubs)
(puthash "\\squareneswfill" "▨" julia-latexsubs)
(puthash "\\squarecrossfill" "▩" julia-latexsubs)
(puthash "\\smblksquare" "▪" julia-latexsubs)
(puthash "\\smwhtsquare" "▫" julia-latexsubs)
(puthash "\\hrectangleblack" "▬" julia-latexsubs)
(puthash "\\hrectangle" "▭" julia-latexsubs)
(puthash "\\vrectangleblack" "▮" julia-latexsubs)
(puthash "\\Elzvrecto" "▯" julia-latexsubs)
(puthash "\\parallelogramblack" "▰" julia-latexsubs)
(puthash "\\parallelogram" "▱" julia-latexsubs)
(puthash "\\bigblacktriangleup" "▲" julia-latexsubs)
(puthash "\\bigtriangleup" "△" julia-latexsubs)
(puthash "\\blacktriangle" "▴" julia-latexsubs)
(puthash "\\vartriangle" "▵" julia-latexsubs)
(puthash "\\blacktriangleright" "▶" julia-latexsubs)
(puthash "\\triangleright" "▹" julia-latexsubs)
(puthash "\\blackpointerright" "►" julia-latexsubs)
(puthash "\\whitepointerright" "▻" julia-latexsubs)
(puthash "\\bigblacktriangledown" "▼" julia-latexsubs)
(puthash "\\bigtriangledown" "▽" julia-latexsubs)
(puthash "\\blacktriangledown" "▾" julia-latexsubs)
(puthash "\\triangledown" "▿" julia-latexsubs)
(puthash "\\blacktriangleleft" "◀" julia-latexsubs)
(puthash "\\triangleleft" "◃" julia-latexsubs)
(puthash "\\blackpointerleft" "◄" julia-latexsubs)
(puthash "\\whitepointerleft" "◅" julia-latexsubs)
(puthash "\\mdlgblkdiamond" "◆" julia-latexsubs)
(puthash "\\mdlgwhtdiamond" "◇" julia-latexsubs)
(puthash "\\blackinwhitediamond" "◈" julia-latexsubs)
(puthash "\\fisheye" "◉" julia-latexsubs)
(puthash "\\lozenge" "◊" julia-latexsubs)
(puthash "\\bigcirc" "○" julia-latexsubs)
(puthash "\\dottedcircle" "◌" julia-latexsubs)
(puthash "\\circlevertfill" "◍" julia-latexsubs)
(puthash "\\bullseye" "◎" julia-latexsubs)
(puthash "\\mdlgblkcircle" "●" julia-latexsubs)
(puthash "\\Elzcirfl" "◐" julia-latexsubs)
(puthash "\\Elzcirfr" "◑" julia-latexsubs)
(puthash "\\Elzcirfb" "◒" julia-latexsubs)
(puthash "\\circletophalfblack" "◓" julia-latexsubs)
(puthash "\\circleurquadblack" "◔" julia-latexsubs)
(puthash "\\blackcircleulquadwhite" "◕" julia-latexsubs)
(puthash "\\blacklefthalfcircle" "◖" julia-latexsubs)
(puthash "\\blackrighthalfcircle" "◗" julia-latexsubs)
(puthash "\\Elzrvbull" "◘" julia-latexsubs)
(puthash "\\inversewhitecircle" "◙" julia-latexsubs)
(puthash "\\invwhiteupperhalfcircle" "◚" julia-latexsubs)
(puthash "\\invwhitelowerhalfcircle" "◛" julia-latexsubs)
(puthash "\\ularc" "◜" julia-latexsubs)
(puthash "\\urarc" "◝" julia-latexsubs)
(puthash "\\lrarc" "◞" julia-latexsubs)
(puthash "\\llarc" "◟" julia-latexsubs)
(puthash "\\topsemicircle" "◠" julia-latexsubs)
(puthash "\\botsemicircle" "◡" julia-latexsubs)
(puthash "\\lrblacktriangle" "◢" julia-latexsubs)
(puthash "\\llblacktriangle" "◣" julia-latexsubs)
(puthash "\\ulblacktriangle" "◤" julia-latexsubs)
(puthash "\\urblacktriangle" "◥" julia-latexsubs)
(puthash "\\smwhtcircle" "◦" julia-latexsubs)
(puthash "\\Elzsqfl" "◧" julia-latexsubs)
(puthash "\\Elzsqfr" "◨" julia-latexsubs)
(puthash "\\squareulblack" "◩" julia-latexsubs)
(puthash "\\Elzsqfse" "◪" julia-latexsubs)
(puthash "\\boxbar" "◫" julia-latexsubs)
(puthash "\\trianglecdot" "◬" julia-latexsubs)
(puthash "\\triangleleftblack" "◭" julia-latexsubs)
(puthash "\\trianglerightblack" "◮" julia-latexsubs)
(puthash "\\lgwhtcircle" "◯" julia-latexsubs)
(puthash "\\squareulquad" "◰" julia-latexsubs)
(puthash "\\squarellquad" "◱" julia-latexsubs)
(puthash "\\squarelrquad" "◲" julia-latexsubs)
(puthash "\\squareurquad" "◳" julia-latexsubs)
(puthash "\\circleulquad" "◴" julia-latexsubs)
(puthash "\\circlellquad" "◵" julia-latexsubs)
(puthash "\\circlelrquad" "◶" julia-latexsubs)
(puthash "\\circleurquad" "◷" julia-latexsubs)
(puthash "\\ultriangle" "◸" julia-latexsubs)
(puthash "\\urtriangle" "◹" julia-latexsubs)
(puthash "\\lltriangle" "◺" julia-latexsubs)
(puthash "\\mdwhtsquare" "◻" julia-latexsubs)
(puthash "\\mdblksquare" "◼" julia-latexsubs)
(puthash "\\mdsmwhtsquare" "◽" julia-latexsubs)
(puthash "\\mdsmblksquare" "◾" julia-latexsubs)
(puthash "\\lrtriangle" "◿" julia-latexsubs)
(puthash "\\bigstar" "★" julia-latexsubs)
(puthash "\\bigwhitestar" "☆" julia-latexsubs)
(puthash "\\astrosun" "☉" julia-latexsubs)
(puthash "\\danger" "☡" julia-latexsubs)
(puthash "\\blacksmiley" "☻" julia-latexsubs)
(puthash "\\sun" "☼" julia-latexsubs)
(puthash "\\rightmoon" "☽" julia-latexsubs)
(puthash "\\mercury" "☿" julia-latexsubs)
(puthash "\\venus" "♀" julia-latexsubs)
(puthash "\\female" "♀" julia-latexsubs)
(puthash "\\male" "♂" julia-latexsubs)
(puthash "\\mars" "♂" julia-latexsubs)
(puthash "\\jupiter" "♃" julia-latexsubs)
(puthash "\\saturn" "♄" julia-latexsubs)
(puthash "\\uranus" "♅" julia-latexsubs)
(puthash "\\neptune" "♆" julia-latexsubs)
(puthash "\\pluto" "♇" julia-latexsubs)
(puthash "\\aries" "♈" julia-latexsubs)
(puthash "\\taurus" "♉" julia-latexsubs)
(puthash "\\gemini" "♊" julia-latexsubs)
(puthash "\\cancer" "♋" julia-latexsubs)
(puthash "\\leo" "♌" julia-latexsubs)
(puthash "\\virgo" "♍" julia-latexsubs)
(puthash "\\libra" "♎" julia-latexsubs)
(puthash "\\scorpio" "♏" julia-latexsubs)
(puthash "\\sagittarius" "♐" julia-latexsubs)
(puthash "\\capricornus" "♑" julia-latexsubs)
(puthash "\\aquarius" "♒" julia-latexsubs)
(puthash "\\pisces" "♓" julia-latexsubs)
(puthash "\\spadesuit" "♠" julia-latexsubs)
(puthash "\\heartsuit" "♡" julia-latexsubs)
(puthash "\\diamondsuit" "♢" julia-latexsubs)
(puthash "\\clubsuit" "♣" julia-latexsubs)
(puthash "\\varspadesuit" "♤" julia-latexsubs)
(puthash "\\varheartsuit" "♥" julia-latexsubs)
(puthash "\\vardiamondsuit" "♦" julia-latexsubs)
(puthash "\\varclubsuit" "♧" julia-latexsubs)
(puthash "\\quarternote" "♩" julia-latexsubs)
(puthash "\\eighthnote" "♪" julia-latexsubs)
(puthash "\\twonotes" "♫" julia-latexsubs)
(puthash "\\flat" "♭" julia-latexsubs)
(puthash "\\natural" "♮" julia-latexsubs)
(puthash "\\sharp" "♯" julia-latexsubs)
(puthash "\\acidfree" "\u267e" julia-latexsubs)
(puthash "\\dicei" "⚀" julia-latexsubs)
(puthash "\\diceii" "⚁" julia-latexsubs)
(puthash "\\diceiii" "⚂" julia-latexsubs)
(puthash "\\diceiv" "⚃" julia-latexsubs)
(puthash "\\dicev" "⚄" julia-latexsubs)
(puthash "\\dicevi" "⚅" julia-latexsubs)
(puthash "\\circledrightdot" "⚆" julia-latexsubs)
(puthash "\\circledtwodots" "⚇" julia-latexsubs)
(puthash "\\blackcircledrightdot" "⚈" julia-latexsubs)
(puthash "\\blackcircledtwodots" "⚉" julia-latexsubs)
(puthash "\\Hermaphrodite" "\u26a5" julia-latexsubs)
(puthash "\\mdwhtcircle" "\u26aa" julia-latexsubs)
(puthash "\\mdblkcircle" "\u26ab" julia-latexsubs)
(puthash "\\mdsmwhtcircle" "\u26ac" julia-latexsubs)
(puthash "\\neuter" "\u26b2" julia-latexsubs)
(puthash "\\checkmark" "✓" julia-latexsubs)
(puthash "\\maltese" "✠" julia-latexsubs)
(puthash "\\circledstar" "✪" julia-latexsubs)
(puthash "\\varstar" "✶" julia-latexsubs)
(puthash "\\dingasterisk" "✽" julia-latexsubs)
(puthash "\\draftingarrow" "➛" julia-latexsubs)
(puthash "\\threedangle" "\u27c0" julia-latexsubs)
(puthash "\\whiteinwhitetriangle" "\u27c1" julia-latexsubs)
(puthash "\\perp" "\u27c2" julia-latexsubs)
(puthash "\\bsolhsub" "\u27c8" julia-latexsubs)
(puthash "\\suphsol" "\u27c9" julia-latexsubs)
(puthash "\\wedgedot" "⟑" julia-latexsubs)
(puthash "\\upin" "⟒" julia-latexsubs)
(puthash "\\bigbot" "⟘" julia-latexsubs)
(puthash "\\bigtop" "⟙" julia-latexsubs)
(puthash "\\langle" "⟨" julia-latexsubs)
(puthash "\\rangle" "⟩" julia-latexsubs)
(puthash "\\UUparrow" "⟰" julia-latexsubs)
(puthash "\\DDownarrow" "⟱" julia-latexsubs)
(puthash "\\longleftarrow" "⟵" julia-latexsubs)
(puthash "\\longrightarrow" "⟶" julia-latexsubs)
(puthash "\\longleftrightarrow" "⟷" julia-latexsubs)
(puthash "\\Longleftarrow" "⟸" julia-latexsubs)
(puthash "\\Longrightarrow" "⟹" julia-latexsubs)
(puthash "\\Longleftrightarrow" "⟺" julia-latexsubs)
(puthash "\\longmapsfrom" "⟻" julia-latexsubs)
(puthash "\\longmapsto" "⟼" julia-latexsubs)
(puthash "\\Longmapsfrom" "⟽" julia-latexsubs)
(puthash "\\Longmapsto" "⟾" julia-latexsubs)
(puthash "\\longrightsquigarrow" "⟿" julia-latexsubs)
(puthash "\\nvtwoheadrightarrow" "⤀" julia-latexsubs)
(puthash "\\nVtwoheadrightarrow" "⤁" julia-latexsubs)
(puthash "\\nvLeftarrow" "⤂" julia-latexsubs)
(puthash "\\nvRightarrow" "⤃" julia-latexsubs)
(puthash "\\nvLeftrightarrow" "⤄" julia-latexsubs)
(puthash "\\twoheadmapsto" "⤅" julia-latexsubs)
(puthash "\\Mapsfrom" "⤆" julia-latexsubs)
(puthash "\\Mapsto" "⤇" julia-latexsubs)
(puthash "\\downarrowbarred" "⤈" julia-latexsubs)
(puthash "\\uparrowbarred" "⤉" julia-latexsubs)
(puthash "\\Uuparrow" "⤊" julia-latexsubs)
(puthash "\\Ddownarrow" "⤋" julia-latexsubs)
(puthash "\\leftbkarrow" "⤌" julia-latexsubs)
(puthash "\\bkarow" "⤍" julia-latexsubs)
(puthash "\\leftdbkarrow" "⤎" julia-latexsubs)
(puthash "\\dbkarow" "⤏" julia-latexsubs)
(puthash "\\drbkarrow" "⤐" julia-latexsubs)
(puthash "\\rightdotarrow" "⤑" julia-latexsubs)
(puthash "\\UpArrowBar" "⤒" julia-latexsubs)
(puthash "\\DownArrowBar" "⤓" julia-latexsubs)
(puthash "\\nvrightarrowtail" "⤔" julia-latexsubs)
(puthash "\\nVrightarrowtail" "⤕" julia-latexsubs)
(puthash "\\twoheadrightarrowtail" "⤖" julia-latexsubs)
(puthash "\\nvtwoheadrightarrowtail" "⤗" julia-latexsubs)
(puthash "\\nVtwoheadrightarrowtail" "⤘" julia-latexsubs)
(puthash "\\diamondleftarrow" "⤝" julia-latexsubs)
(puthash "\\rightarrowdiamond" "⤞" julia-latexsubs)
(puthash "\\diamondleftarrowbar" "⤟" julia-latexsubs)
(puthash "\\barrightarrowdiamond" "⤠" julia-latexsubs)
(puthash "\\hksearow" "⤥" julia-latexsubs)
(puthash "\\hkswarow" "⤦" julia-latexsubs)
(puthash "\\tona" "⤧" julia-latexsubs)
(puthash "\\toea" "⤨" julia-latexsubs)
(puthash "\\tosa" "⤩" julia-latexsubs)
(puthash "\\towa" "⤪" julia-latexsubs)
(puthash "\\rdiagovfdiag" "⤫" julia-latexsubs)
(puthash "\\fdiagovrdiag" "⤬" julia-latexsubs)
(puthash "\\seovnearrow" "⤭" julia-latexsubs)
(puthash "\\neovsearrow" "⤮" julia-latexsubs)
(puthash "\\fdiagovnearrow" "⤯" julia-latexsubs)
(puthash "\\rdiagovsearrow" "⤰" julia-latexsubs)
(puthash "\\neovnwarrow" "⤱" julia-latexsubs)
(puthash "\\nwovnearrow" "⤲" julia-latexsubs)
(puthash "\\ElzRlarr" "⥂" julia-latexsubs)
(puthash "\\ElzrLarr" "⥄" julia-latexsubs)
(puthash "\\rightarrowplus" "⥅" julia-latexsubs)
(puthash "\\leftarrowplus" "⥆" julia-latexsubs)
(puthash "\\Elzrarrx" "⥇" julia-latexsubs)
(puthash "\\leftrightarrowcircle" "⥈" julia-latexsubs)
(puthash "\\twoheaduparrowcircle" "⥉" julia-latexsubs)
(puthash "\\leftrightharpoonupdown" "⥊" julia-latexsubs)
(puthash "\\leftrightharpoondownup" "⥋" julia-latexsubs)
(puthash "\\updownharpoonrightleft" "⥌" julia-latexsubs)
(puthash "\\updownharpoonleftright" "⥍" julia-latexsubs)
(puthash "\\LeftRightVector" "⥎" julia-latexsubs)
(puthash "\\RightUpDownVector" "⥏" julia-latexsubs)
(puthash "\\DownLeftRightVector" "⥐" julia-latexsubs)
(puthash "\\LeftUpDownVector" "⥑" julia-latexsubs)
(puthash "\\LeftVectorBar" "⥒" julia-latexsubs)
(puthash "\\RightVectorBar" "⥓" julia-latexsubs)
(puthash "\\RightUpVectorBar" "⥔" julia-latexsubs)
(puthash "\\RightDownVectorBar" "⥕" julia-latexsubs)
(puthash "\\DownLeftVectorBar" "⥖" julia-latexsubs)
(puthash "\\DownRightVectorBar" "⥗" julia-latexsubs)
(puthash "\\LeftUpVectorBar" "⥘" julia-latexsubs)
(puthash "\\LeftDownVectorBar" "⥙" julia-latexsubs)
(puthash "\\LeftTeeVector" "⥚" julia-latexsubs)
(puthash "\\RightTeeVector" "⥛" julia-latexsubs)
(puthash "\\RightUpTeeVector" "⥜" julia-latexsubs)
(puthash "\\RightDownTeeVector" "⥝" julia-latexsubs)
(puthash "\\DownLeftTeeVector" "⥞" julia-latexsubs)
(puthash "\\DownRightTeeVector" "⥟" julia-latexsubs)
(puthash "\\LeftUpTeeVector" "⥠" julia-latexsubs)
(puthash "\\LeftDownTeeVector" "⥡" julia-latexsubs)
(puthash "\\leftharpoonsupdown" "⥢" julia-latexsubs)
(puthash "\\upharpoonsleftright" "⥣" julia-latexsubs)
(puthash "\\rightharpoonsupdown" "⥤" julia-latexsubs)
(puthash "\\downharpoonsleftright" "⥥" julia-latexsubs)
(puthash "\\leftrightharpoonsup" "⥦" julia-latexsubs)
(puthash "\\leftrightharpoonsdown" "⥧" julia-latexsubs)
(puthash "\\rightleftharpoonsup" "⥨" julia-latexsubs)
(puthash "\\rightleftharpoonsdown" "⥩" julia-latexsubs)
(puthash "\\leftharpoonupdash" "⥪" julia-latexsubs)
(puthash "\\dashleftharpoondown" "⥫" julia-latexsubs)
(puthash "\\rightharpoonupdash" "⥬" julia-latexsubs)
(puthash "\\dashrightharpoondown" "⥭" julia-latexsubs)
(puthash "\\UpEquilibrium" "⥮" julia-latexsubs)
(puthash "\\ReverseUpEquilibrium" "⥯" julia-latexsubs)
(puthash "\\RoundImplies" "⥰" julia-latexsubs)
(puthash "\\Vvert" "⦀" julia-latexsubs)
(puthash "\\Elroang" "⦆" julia-latexsubs)
(puthash "\\Elzddfnc" "⦙" julia-latexsubs)
(puthash "\\measuredangleleft" "⦛" julia-latexsubs)
(puthash "\\Angle" "⦜" julia-latexsubs)
(puthash "\\rightanglemdot" "⦝" julia-latexsubs)
(puthash "\\angles" "⦞" julia-latexsubs)
(puthash "\\angdnr" "⦟" julia-latexsubs)
(puthash "\\Elzlpargt" "⦠" julia-latexsubs)
(puthash "\\sphericalangleup" "⦡" julia-latexsubs)
(puthash "\\turnangle" "⦢" julia-latexsubs)
(puthash "\\revangle" "⦣" julia-latexsubs)
(puthash "\\angleubar" "⦤" julia-latexsubs)
(puthash "\\revangleubar" "⦥" julia-latexsubs)
(puthash "\\wideangledown" "⦦" julia-latexsubs)
(puthash "\\wideangleup" "⦧" julia-latexsubs)
(puthash "\\measanglerutone" "⦨" julia-latexsubs)
(puthash "\\measanglelutonw" "⦩" julia-latexsubs)
(puthash "\\measanglerdtose" "⦪" julia-latexsubs)
(puthash "\\measangleldtosw" "⦫" julia-latexsubs)
(puthash "\\measangleurtone" "⦬" julia-latexsubs)
(puthash "\\measangleultonw" "⦭" julia-latexsubs)
(puthash "\\measangledrtose" "⦮" julia-latexsubs)
(puthash "\\measangledltosw" "⦯" julia-latexsubs)
(puthash "\\revemptyset" "⦰" julia-latexsubs)
(puthash "\\emptysetobar" "⦱" julia-latexsubs)
(puthash "\\emptysetocirc" "⦲" julia-latexsubs)
(puthash "\\emptysetoarr" "⦳" julia-latexsubs)
(puthash "\\emptysetoarrl" "⦴" julia-latexsubs)
(puthash "\\circledparallel" "⦷" julia-latexsubs)
(puthash "\\obslash" "⦸" julia-latexsubs)
(puthash "\\odotslashdot" "⦼" julia-latexsubs)
(puthash "\\circledwhitebullet" "⦾" julia-latexsubs)
(puthash "\\circledbullet" "⦿" julia-latexsubs)
(puthash "\\olessthan" "⧀" julia-latexsubs)
(puthash "\\ogreaterthan" "⧁" julia-latexsubs)
(puthash "\\boxdiag" "⧄" julia-latexsubs)
(puthash "\\boxbslash" "⧅" julia-latexsubs)
(puthash "\\boxast" "⧆" julia-latexsubs)
(puthash "\\boxcircle" "⧇" julia-latexsubs)
(puthash "\\ElzLap" "⧊" julia-latexsubs)
(puthash "\\Elzdefas" "⧋" julia-latexsubs)
(puthash "\\LeftTriangleBar" "⧏" julia-latexsubs)
(puthash "\\NotLeftTriangleBar" "⧏̸" julia-latexsubs)
(puthash "\\RightTriangleBar" "⧐" julia-latexsubs)
(puthash "\\NotRightTriangleBar" "⧐̸" julia-latexsubs)
(puthash "\\dualmap" "⧟" julia-latexsubs)
(puthash "\\lrtriangleeq" "⧡" julia-latexsubs)
(puthash "\\shuffle" "⧢" julia-latexsubs)
(puthash "\\eparsl" "⧣" julia-latexsubs)
(puthash "\\smeparsl" "⧤" julia-latexsubs)
(puthash "\\eqvparsl" "⧥" julia-latexsubs)
(puthash "\\blacklozenge" "⧫" julia-latexsubs)
(puthash "\\RuleDelayed" "⧴" julia-latexsubs)
(puthash "\\dsol" "⧶" julia-latexsubs)
(puthash "\\rsolbar" "⧷" julia-latexsubs)
(puthash "\\doubleplus" "⧺" julia-latexsubs)
(puthash "\\tripleplus" "⧻" julia-latexsubs)
(puthash "\\bigodot" "⨀" julia-latexsubs)
(puthash "\\bigoplus" "⨁" julia-latexsubs)
(puthash "\\bigotimes" "⨂" julia-latexsubs)
(puthash "\\bigcupdot" "⨃" julia-latexsubs)
(puthash "\\biguplus" "⨄" julia-latexsubs)
(puthash "\\bigsqcap" "⨅" julia-latexsubs)
(puthash "\\bigsqcup" "⨆" julia-latexsubs)
(puthash "\\conjquant" "⨇" julia-latexsubs)
(puthash "\\disjquant" "⨈" julia-latexsubs)
(puthash "\\bigtimes" "⨉" julia-latexsubs)
(puthash "\\modtwosum" "⨊" julia-latexsubs)
(puthash "\\sumint" "⨋" julia-latexsubs)
(puthash "\\iiiint" "⨌" julia-latexsubs)
(puthash "\\intbar" "⨍" julia-latexsubs)
(puthash "\\intBar" "⨎" julia-latexsubs)
(puthash "\\clockoint" "⨏" julia-latexsubs)
(puthash "\\cirfnint" "⨐" julia-latexsubs)
(puthash "\\awint" "⨑" julia-latexsubs)
(puthash "\\rppolint" "⨒" julia-latexsubs)
(puthash "\\scpolint" "⨓" julia-latexsubs)
(puthash "\\npolint" "⨔" julia-latexsubs)
(puthash "\\pointint" "⨕" julia-latexsubs)
(puthash "\\sqrint" "⨖" julia-latexsubs)
(puthash "\\intx" "⨘" julia-latexsubs)
(puthash "\\intcap" "⨙" julia-latexsubs)
(puthash "\\intcup" "⨚" julia-latexsubs)
(puthash "\\upint" "⨛" julia-latexsubs)
(puthash "\\lowint" "⨜" julia-latexsubs)
(puthash "\\ringplus" "⨢" julia-latexsubs)
(puthash "\\plushat" "⨣" julia-latexsubs)
(puthash "\\simplus" "⨤" julia-latexsubs)
(puthash "\\plusdot" "⨥" julia-latexsubs)
(puthash "\\plussim" "⨦" julia-latexsubs)
(puthash "\\plussubtwo" "⨧" julia-latexsubs)
(puthash "\\plustrif" "⨨" julia-latexsubs)
(puthash "\\commaminus" "⨩" julia-latexsubs)
(puthash "\\minusdot" "⨪" julia-latexsubs)
(puthash "\\minusfdots" "⨫" julia-latexsubs)
(puthash "\\minusrdots" "⨬" julia-latexsubs)
(puthash "\\opluslhrim" "⨭" julia-latexsubs)
(puthash "\\oplusrhrim" "⨮" julia-latexsubs)
(puthash "\\ElzTimes" "⨯" julia-latexsubs)
(puthash "\\dottimes" "⨰" julia-latexsubs)
(puthash "\\timesbar" "⨱" julia-latexsubs)
(puthash "\\btimes" "⨲" julia-latexsubs)
(puthash "\\smashtimes" "⨳" julia-latexsubs)
(puthash "\\otimeslhrim" "⨴" julia-latexsubs)
(puthash "\\otimesrhrim" "⨵" julia-latexsubs)
(puthash "\\otimeshat" "⨶" julia-latexsubs)
(puthash "\\Otimes" "⨷" julia-latexsubs)
(puthash "\\odiv" "⨸" julia-latexsubs)
(puthash "\\triangleplus" "⨹" julia-latexsubs)
(puthash "\\triangleminus" "⨺" julia-latexsubs)
(puthash "\\triangletimes" "⨻" julia-latexsubs)
(puthash "\\intprod" "⨼" julia-latexsubs)
(puthash "\\intprodr" "⨽" julia-latexsubs)
(puthash "\\amalg" "⨿" julia-latexsubs)
(puthash "\\capdot" "⩀" julia-latexsubs)
(puthash "\\uminus" "⩁" julia-latexsubs)
(puthash "\\barcup" "⩂" julia-latexsubs)
(puthash "\\barcap" "⩃" julia-latexsubs)
(puthash "\\capwedge" "⩄" julia-latexsubs)
(puthash "\\cupvee" "⩅" julia-latexsubs)
(puthash "\\twocups" "⩊" julia-latexsubs)
(puthash "\\twocaps" "⩋" julia-latexsubs)
(puthash "\\closedvarcup" "⩌" julia-latexsubs)
(puthash "\\closedvarcap" "⩍" julia-latexsubs)
(puthash "\\Sqcap" "⩎" julia-latexsubs)
(puthash "\\Sqcup" "⩏" julia-latexsubs)
(puthash "\\closedvarcupsmashprod" "⩐" julia-latexsubs)
(puthash "\\wedgeodot" "⩑" julia-latexsubs)
(puthash "\\veeodot" "⩒" julia-latexsubs)
(puthash "\\ElzAnd" "⩓" julia-latexsubs)
(puthash "\\ElzOr" "⩔" julia-latexsubs)
(puthash "\\wedgeonwedge" "⩕" julia-latexsubs)
(puthash "\\ElOr" "⩖" julia-latexsubs)
(puthash "\\bigslopedvee" "⩗" julia-latexsubs)
(puthash "\\bigslopedwedge" "⩘" julia-latexsubs)
(puthash "\\wedgemidvert" "⩚" julia-latexsubs)
(puthash "\\veemidvert" "⩛" julia-latexsubs)
(puthash "\\midbarwedge" "⩜" julia-latexsubs)
(puthash "\\midbarvee" "⩝" julia-latexsubs)
(puthash "\\perspcorrespond" "⩞" julia-latexsubs)
(puthash "\\Elzminhat" "⩟" julia-latexsubs)
(puthash "\\wedgedoublebar" "⩠" julia-latexsubs)
(puthash "\\varveebar" "⩡" julia-latexsubs)
(puthash "\\doublebarvee" "⩢" julia-latexsubs)
(puthash "\\veedoublebar" "⩣" julia-latexsubs)
(puthash "\\eqdot" "⩦" julia-latexsubs)
(puthash "\\dotequiv" "⩧" julia-latexsubs)
(puthash "\\dotsim" "⩪" julia-latexsubs)
(puthash "\\simrdots" "⩫" julia-latexsubs)
(puthash "\\simminussim" "⩬" julia-latexsubs)
(puthash "\\congdot" "⩭" julia-latexsubs)
(puthash "\\asteq" "⩮" julia-latexsubs)
(puthash "\\hatapprox" "⩯" julia-latexsubs)
(puthash "\\approxeqq" "⩰" julia-latexsubs)
(puthash "\\eqqplus" "⩱" julia-latexsubs)
(puthash "\\pluseqq" "⩲" julia-latexsubs)
(puthash "\\eqqsim" "⩳" julia-latexsubs)
(puthash "\\Coloneq" "⩴" julia-latexsubs)
(puthash "\\Equal" "⩵" julia-latexsubs)
(puthash "\\eqeqeq" "⩶" julia-latexsubs)
(puthash "\\ddotseq" "⩷" julia-latexsubs)
(puthash "\\equivDD" "⩸" julia-latexsubs)
(puthash "\\ltcir" "⩹" julia-latexsubs)
(puthash "\\gtcir" "⩺" julia-latexsubs)
(puthash "\\ltquest" "⩻" julia-latexsubs)
(puthash "\\gtquest" "⩼" julia-latexsubs)
(puthash "\\leqslant" "⩽" julia-latexsubs)
(puthash "\\nleqslant" "⩽̸" julia-latexsubs)
(puthash "\\geqslant" "⩾" julia-latexsubs)
(puthash "\\ngeqslant" "⩾̸" julia-latexsubs)
(puthash "\\lesdot" "⩿" julia-latexsubs)
(puthash "\\gesdot" "⪀" julia-latexsubs)
(puthash "\\lesdoto" "⪁" julia-latexsubs)
(puthash "\\gesdoto" "⪂" julia-latexsubs)
(puthash "\\lesdotor" "⪃" julia-latexsubs)
(puthash "\\gesdotol" "⪄" julia-latexsubs)
(puthash "\\lessapprox" "⪅" julia-latexsubs)
(puthash "\\gtrapprox" "⪆" julia-latexsubs)
(puthash "\\lneq" "⪇" julia-latexsubs)
(puthash "\\gneq" "⪈" julia-latexsubs)
(puthash "\\lnapprox" "⪉" julia-latexsubs)
(puthash "\\gnapprox" "⪊" julia-latexsubs)
(puthash "\\lesseqqgtr" "⪋" julia-latexsubs)
(puthash "\\gtreqqless" "⪌" julia-latexsubs)
(puthash "\\lsime" "⪍" julia-latexsubs)
(puthash "\\gsime" "⪎" julia-latexsubs)
(puthash "\\lsimg" "⪏" julia-latexsubs)
(puthash "\\gsiml" "⪐" julia-latexsubs)
(puthash "\\lgE" "⪑" julia-latexsubs)
(puthash "\\glE" "⪒" julia-latexsubs)
(puthash "\\lesges" "⪓" julia-latexsubs)
(puthash "\\gesles" "⪔" julia-latexsubs)
(puthash "\\eqslantless" "⪕" julia-latexsubs)
(puthash "\\eqslantgtr" "⪖" julia-latexsubs)
(puthash "\\elsdot" "⪗" julia-latexsubs)
(puthash "\\egsdot" "⪘" julia-latexsubs)
(puthash "\\eqqless" "⪙" julia-latexsubs)
(puthash "\\eqqgtr" "⪚" julia-latexsubs)
(puthash "\\eqqslantless" "⪛" julia-latexsubs)
(puthash "\\eqqslantgtr" "⪜" julia-latexsubs)
(puthash "\\simless" "⪝" julia-latexsubs)
(puthash "\\simgtr" "⪞" julia-latexsubs)
(puthash "\\simlE" "⪟" julia-latexsubs)
(puthash "\\simgE" "⪠" julia-latexsubs)
(puthash "\\NestedLessLess" "⪡" julia-latexsubs)
(puthash "\\NotNestedLessLess" "⪡̸" julia-latexsubs)
(puthash "\\NestedGreaterGreater" "⪢" julia-latexsubs)
(puthash "\\NotNestedGreaterGreater" "⪢̸" julia-latexsubs)
(puthash "\\partialmeetcontraction" "⪣" julia-latexsubs)
(puthash "\\glj" "⪤" julia-latexsubs)
(puthash "\\gla" "⪥" julia-latexsubs)
(puthash "\\ltcc" "⪦" julia-latexsubs)
(puthash "\\gtcc" "⪧" julia-latexsubs)
(puthash "\\lescc" "⪨" julia-latexsubs)
(puthash "\\gescc" "⪩" julia-latexsubs)
(puthash "\\smt" "⪪" julia-latexsubs)
(puthash "\\lat" "⪫" julia-latexsubs)
(puthash "\\smte" "⪬" julia-latexsubs)
(puthash "\\late" "⪭" julia-latexsubs)
(puthash "\\bumpeqq" "⪮" julia-latexsubs)
(puthash "\\preceq" "⪯" julia-latexsubs)
(puthash "\\npreceq" "⪯̸" julia-latexsubs)
(puthash "\\succeq" "⪰" julia-latexsubs)
(puthash "\\nsucceq" "⪰̸" julia-latexsubs)
(puthash "\\precneq" "⪱" julia-latexsubs)
(puthash "\\succneq" "⪲" julia-latexsubs)
(puthash "\\preceqq" "⪳" julia-latexsubs)
(puthash "\\succeqq" "⪴" julia-latexsubs)
(puthash "\\precneqq" "⪵" julia-latexsubs)
(puthash "\\succneqq" "⪶" julia-latexsubs)
(puthash "\\precapprox" "⪷" julia-latexsubs)
(puthash "\\succapprox" "⪸" julia-latexsubs)
(puthash "\\precnapprox" "⪹" julia-latexsubs)
(puthash "\\succnapprox" "⪺" julia-latexsubs)
(puthash "\\Prec" "⪻" julia-latexsubs)
(puthash "\\Succ" "⪼" julia-latexsubs)
(puthash "\\subsetdot" "⪽" julia-latexsubs)
(puthash "\\supsetdot" "⪾" julia-latexsubs)
(puthash "\\subsetplus" "⪿" julia-latexsubs)
(puthash "\\supsetplus" "⫀" julia-latexsubs)
(puthash "\\submult" "⫁" julia-latexsubs)
(puthash "\\supmult" "⫂" julia-latexsubs)
(puthash "\\subedot" "⫃" julia-latexsubs)
(puthash "\\supedot" "⫄" julia-latexsubs)
(puthash "\\subseteqq" "⫅" julia-latexsubs)
(puthash "\\nsubseteqq" "⫅̸" julia-latexsubs)
(puthash "\\supseteqq" "⫆" julia-latexsubs)
(puthash "\\nsupseteqq" "⫆̸" julia-latexsubs)
(puthash "\\subsim" "⫇" julia-latexsubs)
(puthash "\\supsim" "⫈" julia-latexsubs)
(puthash "\\subsetapprox" "⫉" julia-latexsubs)
(puthash "\\supsetapprox" "⫊" julia-latexsubs)
(puthash "\\subsetneqq" "⫋" julia-latexsubs)
(puthash "\\supsetneqq" "⫌" julia-latexsubs)
(puthash "\\lsqhook" "⫍" julia-latexsubs)
(puthash "\\rsqhook" "⫎" julia-latexsubs)
(puthash "\\csub" "⫏" julia-latexsubs)
(puthash "\\csup" "⫐" julia-latexsubs)
(puthash "\\csube" "⫑" julia-latexsubs)
(puthash "\\csupe" "⫒" julia-latexsubs)
(puthash "\\subsup" "⫓" julia-latexsubs)
(puthash "\\supsub" "⫔" julia-latexsubs)
(puthash "\\subsub" "⫕" julia-latexsubs)
(puthash "\\supsup" "⫖" julia-latexsubs)
(puthash "\\suphsub" "⫗" julia-latexsubs)
(puthash "\\supdsub" "⫘" julia-latexsubs)
(puthash "\\forkv" "⫙" julia-latexsubs)
(puthash "\\mlcp" "⫛" julia-latexsubs)
(puthash "\\forks" "⫝̸" julia-latexsubs)
(puthash "\\forksnot" "⫝" julia-latexsubs)
(puthash "\\dashV" "⫣" julia-latexsubs)
(puthash "\\Dashv" "⫤" julia-latexsubs)
(puthash "\\interleave" "⫴" julia-latexsubs)
(puthash "\\Elztdcol" "⫶" julia-latexsubs)
(puthash "\\lllnest" "⫷" julia-latexsubs)
(puthash "\\gggnest" "⫸" julia-latexsubs)
(puthash "\\leqqslant" "⫹" julia-latexsubs)
(puthash "\\geqqslant" "⫺" julia-latexsubs)
(puthash "\\squaretopblack" "\u2b12" julia-latexsubs)
(puthash "\\squarebotblack" "\u2b13" julia-latexsubs)
(puthash "\\squareurblack" "\u2b14" julia-latexsubs)
(puthash "\\squarellblack" "\u2b15" julia-latexsubs)
(puthash "\\diamondleftblack" "\u2b16" julia-latexsubs)
(puthash "\\diamondrightblack" "\u2b17" julia-latexsubs)
(puthash "\\diamondtopblack" "\u2b18" julia-latexsubs)
(puthash "\\diamondbotblack" "\u2b19" julia-latexsubs)
(puthash "\\dottedsquare" "\u2b1a" julia-latexsubs)
(puthash "\\lgblksquare" "\u2b1b" julia-latexsubs)
(puthash "\\lgwhtsquare" "\u2b1c" julia-latexsubs)
(puthash "\\vysmblksquare" "\u2b1d" julia-latexsubs)
(puthash "\\vysmwhtsquare" "\u2b1e" julia-latexsubs)
(puthash "\\pentagonblack" "\u2b1f" julia-latexsubs)
(puthash "\\pentagon" "\u2b20" julia-latexsubs)
(puthash "\\varhexagon" "\u2b21" julia-latexsubs)
(puthash "\\varhexagonblack" "\u2b22" julia-latexsubs)
(puthash "\\hexagonblack" "\u2b23" julia-latexsubs)
(puthash "\\lgblkcircle" "\u2b24" julia-latexsubs)
(puthash "\\mdblkdiamond" "\u2b25" julia-latexsubs)
(puthash "\\mdwhtdiamond" "\u2b26" julia-latexsubs)
(puthash "\\mdblklozenge" "\u2b27" julia-latexsubs)
(puthash "\\mdwhtlozenge" "\u2b28" julia-latexsubs)
(puthash "\\smblkdiamond" "\u2b29" julia-latexsubs)
(puthash "\\smblklozenge" "\u2b2a" julia-latexsubs)
(puthash "\\smwhtlozenge" "\u2b2b" julia-latexsubs)
(puthash "\\blkhorzoval" "\u2b2c" julia-latexsubs)
(puthash "\\whthorzoval" "\u2b2d" julia-latexsubs)
(puthash "\\blkvertoval" "\u2b2e" julia-latexsubs)
(puthash "\\whtvertoval" "\u2b2f" julia-latexsubs)
(puthash "\\circleonleftarrow" "\u2b30" julia-latexsubs)
(puthash "\\leftthreearrows" "\u2b31" julia-latexsubs)
(puthash "\\leftarrowonoplus" "\u2b32" julia-latexsubs)
(puthash "\\longleftsquigarrow" "\u2b33" julia-latexsubs)
(puthash "\\nvtwoheadleftarrow" "\u2b34" julia-latexsubs)
(puthash "\\nVtwoheadleftarrow" "\u2b35" julia-latexsubs)
(puthash "\\twoheadmapsfrom" "\u2b36" julia-latexsubs)
(puthash "\\twoheadleftdbkarrow" "\u2b37" julia-latexsubs)
(puthash "\\leftdotarrow" "\u2b38" julia-latexsubs)
(puthash "\\nvleftarrowtail" "\u2b39" julia-latexsubs)
(puthash "\\nVleftarrowtail" "\u2b3a" julia-latexsubs)
(puthash "\\twoheadleftarrowtail" "\u2b3b" julia-latexsubs)
(puthash "\\nvtwoheadleftarrowtail" "\u2b3c" julia-latexsubs)
(puthash "\\nVtwoheadleftarrowtail" "\u2b3d" julia-latexsubs)
(puthash "\\leftarrowx" "\u2b3e" julia-latexsubs)
(puthash "\\leftcurvedarrow" "\u2b3f" julia-latexsubs)
(puthash "\\equalleftarrow" "\u2b40" julia-latexsubs)
(puthash "\\bsimilarleftarrow" "\u2b41" julia-latexsubs)
(puthash "\\leftarrowbackapprox" "\u2b42" julia-latexsubs)
(puthash "\\rightarrowgtr" "\u2b43" julia-latexsubs)
(puthash "\\rightarrowsupset" "\u2b44" julia-latexsubs)
(puthash "\\LLeftarrow" "\u2b45" julia-latexsubs)
(puthash "\\RRightarrow" "\u2b46" julia-latexsubs)
(puthash "\\bsimilarrightarrow" "\u2b47" julia-latexsubs)
(puthash "\\rightarrowbackapprox" "\u2b48" julia-latexsubs)
(puthash "\\similarleftarrow" "\u2b49" julia-latexsubs)
(puthash "\\leftarrowapprox" "\u2b4a" julia-latexsubs)
(puthash "\\leftarrowbsimilar" "\u2b4b" julia-latexsubs)
(puthash "\\rightarrowbsimilar" "\u2b4c" julia-latexsubs)
(puthash "\\medwhitestar" "\u2b50" julia-latexsubs)
(puthash "\\medblackstar" "\u2b51" julia-latexsubs)
(puthash "\\smwhitestar" "\u2b52" julia-latexsubs)
(puthash "\\rightpentagonblack" "\u2b53" julia-latexsubs)
(puthash "\\rightpentagon" "\u2b54" julia-latexsubs)
(puthash "\\_j" "\u2c7c" julia-latexsubs)
(puthash "\\^V" "\u2c7d" julia-latexsubs)
(puthash "\\postalmark" "〒" julia-latexsubs)
(puthash "\\openbracketleft" "〚" julia-latexsubs)
(puthash "\\openbracketright" "〛" julia-latexsubs)
(puthash "\\overbrace" "︷" julia-latexsubs)
(puthash "\\underbrace" "︸" julia-latexsubs)
(puthash "\\mbfA" "𝐀" julia-latexsubs)
(puthash "\\mbfB" "𝐁" julia-latexsubs)
(puthash "\\mbfC" "𝐂" julia-latexsubs)
(puthash "\\mbfD" "𝐃" julia-latexsubs)
(puthash "\\mbfE" "𝐄" julia-latexsubs)
(puthash "\\mbfF" "𝐅" julia-latexsubs)
(puthash "\\mbfG" "𝐆" julia-latexsubs)
(puthash "\\mbfH" "𝐇" julia-latexsubs)
(puthash "\\mbfI" "𝐈" julia-latexsubs)
(puthash "\\mbfJ" "𝐉" julia-latexsubs)
(puthash "\\mbfK" "𝐊" julia-latexsubs)
(puthash "\\mbfL" "𝐋" julia-latexsubs)
(puthash "\\mbfM" "𝐌" julia-latexsubs)
(puthash "\\mbfN" "𝐍" julia-latexsubs)
(puthash "\\mbfO" "𝐎" julia-latexsubs)
(puthash "\\mbfP" "𝐏" julia-latexsubs)
(puthash "\\mbfQ" "𝐐" julia-latexsubs)
(puthash "\\mbfR" "𝐑" julia-latexsubs)
(puthash "\\mbfS" "𝐒" julia-latexsubs)
(puthash "\\mbfT" "𝐓" julia-latexsubs)
(puthash "\\mbfU" "𝐔" julia-latexsubs)
(puthash "\\mbfV" "𝐕" julia-latexsubs)
(puthash "\\mbfW" "𝐖" julia-latexsubs)
(puthash "\\mbfX" "𝐗" julia-latexsubs)
(puthash "\\mbfY" "𝐘" julia-latexsubs)
(puthash "\\mbfZ" "𝐙" julia-latexsubs)
(puthash "\\mbfa" "𝐚" julia-latexsubs)
(puthash "\\mbfb" "𝐛" julia-latexsubs)
(puthash "\\mbfc" "𝐜" julia-latexsubs)
(puthash "\\mbfd" "𝐝" julia-latexsubs)
(puthash "\\mbfe" "𝐞" julia-latexsubs)
(puthash "\\mbff" "𝐟" julia-latexsubs)
(puthash "\\mbfg" "𝐠" julia-latexsubs)
(puthash "\\mbfh" "𝐡" julia-latexsubs)
(puthash "\\mbfi" "𝐢" julia-latexsubs)
(puthash "\\mbfj" "𝐣" julia-latexsubs)
(puthash "\\mbfk" "𝐤" julia-latexsubs)
(puthash "\\mbfl" "𝐥" julia-latexsubs)
(puthash "\\mbfm" "𝐦" julia-latexsubs)
(puthash "\\mbfn" "𝐧" julia-latexsubs)
(puthash "\\mbfo" "𝐨" julia-latexsubs)
(puthash "\\mbfp" "𝐩" julia-latexsubs)
(puthash "\\mbfq" "𝐪" julia-latexsubs)
(puthash "\\mbfr" "𝐫" julia-latexsubs)
(puthash "\\mbfs" "𝐬" julia-latexsubs)
(puthash "\\mbft" "𝐭" julia-latexsubs)
(puthash "\\mbfu" "𝐮" julia-latexsubs)
(puthash "\\mbfv" "𝐯" julia-latexsubs)
(puthash "\\mbfw" "𝐰" julia-latexsubs)
(puthash "\\mbfx" "𝐱" julia-latexsubs)
(puthash "\\mbfy" "𝐲" julia-latexsubs)
(puthash "\\mbfz" "𝐳" julia-latexsubs)
(puthash "\\mitA" "𝐴" julia-latexsubs)
(puthash "\\mitB" "𝐵" julia-latexsubs)
(puthash "\\mitC" "𝐶" julia-latexsubs)
(puthash "\\mitD" "𝐷" julia-latexsubs)
(puthash "\\mitE" "𝐸" julia-latexsubs)
(puthash "\\mitF" "𝐹" julia-latexsubs)
(puthash "\\mitG" "𝐺" julia-latexsubs)
(puthash "\\mitH" "𝐻" julia-latexsubs)
(puthash "\\mitI" "𝐼" julia-latexsubs)
(puthash "\\mitJ" "𝐽" julia-latexsubs)
(puthash "\\mitK" "𝐾" julia-latexsubs)
(puthash "\\mitL" "𝐿" julia-latexsubs)
(puthash "\\mitM" "𝑀" julia-latexsubs)
(puthash "\\mitN" "𝑁" julia-latexsubs)
(puthash "\\mitO" "𝑂" julia-latexsubs)
(puthash "\\mitP" "𝑃" julia-latexsubs)
(puthash "\\mitQ" "𝑄" julia-latexsubs)
(puthash "\\mitR" "𝑅" julia-latexsubs)
(puthash "\\mitS" "𝑆" julia-latexsubs)
(puthash "\\mitT" "𝑇" julia-latexsubs)
(puthash "\\mitU" "𝑈" julia-latexsubs)
(puthash "\\mitV" "𝑉" julia-latexsubs)
(puthash "\\mitW" "𝑊" julia-latexsubs)
(puthash "\\mitX" "𝑋" julia-latexsubs)
(puthash "\\mitY" "𝑌" julia-latexsubs)
(puthash "\\mitZ" "𝑍" julia-latexsubs)
(puthash "\\mita" "𝑎" julia-latexsubs)
(puthash "\\mitb" "𝑏" julia-latexsubs)
(puthash "\\mitc" "𝑐" julia-latexsubs)
(puthash "\\mitd" "𝑑" julia-latexsubs)
(puthash "\\mite" "𝑒" julia-latexsubs)
(puthash "\\mitf" "𝑓" julia-latexsubs)
(puthash "\\mitg" "𝑔" julia-latexsubs)
(puthash "\\miti" "𝑖" julia-latexsubs)
(puthash "\\mitj" "𝑗" julia-latexsubs)
(puthash "\\mitk" "𝑘" julia-latexsubs)
(puthash "\\mitl" "𝑙" julia-latexsubs)
(puthash "\\mitm" "𝑚" julia-latexsubs)
(puthash "\\mitn" "𝑛" julia-latexsubs)
(puthash "\\mito" "𝑜" julia-latexsubs)
(puthash "\\mitp" "𝑝" julia-latexsubs)
(puthash "\\mitq" "𝑞" julia-latexsubs)
(puthash "\\mitr" "𝑟" julia-latexsubs)
(puthash "\\mits" "𝑠" julia-latexsubs)
(puthash "\\mitt" "𝑡" julia-latexsubs)
(puthash "\\mitu" "𝑢" julia-latexsubs)
(puthash "\\mitv" "𝑣" julia-latexsubs)
(puthash "\\mitw" "𝑤" julia-latexsubs)
(puthash "\\mitx" "𝑥" julia-latexsubs)
(puthash "\\mity" "𝑦" julia-latexsubs)
(puthash "\\mitz" "𝑧" julia-latexsubs)
(puthash "\\mbfitA" "𝑨" julia-latexsubs)
(puthash "\\mbfitB" "𝑩" julia-latexsubs)
(puthash "\\mbfitC" "𝑪" julia-latexsubs)
(puthash "\\mbfitD" "𝑫" julia-latexsubs)
(puthash "\\mbfitE" "𝑬" julia-latexsubs)
(puthash "\\mbfitF" "𝑭" julia-latexsubs)
(puthash "\\mbfitG" "𝑮" julia-latexsubs)
(puthash "\\mbfitH" "𝑯" julia-latexsubs)
(puthash "\\mbfitI" "𝑰" julia-latexsubs)
(puthash "\\mbfitJ" "𝑱" julia-latexsubs)
(puthash "\\mbfitK" "𝑲" julia-latexsubs)
(puthash "\\mbfitL" "𝑳" julia-latexsubs)
(puthash "\\mbfitM" "𝑴" julia-latexsubs)
(puthash "\\mbfitN" "𝑵" julia-latexsubs)
(puthash "\\mbfitO" "𝑶" julia-latexsubs)
(puthash "\\mbfitP" "𝑷" julia-latexsubs)
(puthash "\\mbfitQ" "𝑸" julia-latexsubs)
(puthash "\\mbfitR" "𝑹" julia-latexsubs)
(puthash "\\mbfitS" "𝑺" julia-latexsubs)
(puthash "\\mbfitT" "𝑻" julia-latexsubs)
(puthash "\\mbfitU" "𝑼" julia-latexsubs)
(puthash "\\mbfitV" "𝑽" julia-latexsubs)
(puthash "\\mbfitW" "𝑾" julia-latexsubs)
(puthash "\\mbfitX" "𝑿" julia-latexsubs)
(puthash "\\mbfitY" "𝒀" julia-latexsubs)
(puthash "\\mbfitZ" "𝒁" julia-latexsubs)
(puthash "\\mbfita" "𝒂" julia-latexsubs)
(puthash "\\mbfitb" "𝒃" julia-latexsubs)
(puthash "\\mbfitc" "𝒄" julia-latexsubs)
(puthash "\\mbfitd" "𝒅" julia-latexsubs)
(puthash "\\mbfite" "𝒆" julia-latexsubs)
(puthash "\\mbfitf" "𝒇" julia-latexsubs)
(puthash "\\mbfitg" "𝒈" julia-latexsubs)
(puthash "\\mbfith" "𝒉" julia-latexsubs)
(puthash "\\mbfiti" "𝒊" julia-latexsubs)
(puthash "\\mbfitj" "𝒋" julia-latexsubs)
(puthash "\\mbfitk" "𝒌" julia-latexsubs)
(puthash "\\mbfitl" "𝒍" julia-latexsubs)
(puthash "\\mbfitm" "𝒎" julia-latexsubs)
(puthash "\\mbfitn" "𝒏" julia-latexsubs)
(puthash "\\mbfito" "𝒐" julia-latexsubs)
(puthash "\\mbfitp" "𝒑" julia-latexsubs)
(puthash "\\mbfitq" "𝒒" julia-latexsubs)
(puthash "\\mbfitr" "𝒓" julia-latexsubs)
(puthash "\\mbfits" "𝒔" julia-latexsubs)
(puthash "\\mbfitt" "𝒕" julia-latexsubs)
(puthash "\\mbfitu" "𝒖" julia-latexsubs)
(puthash "\\mbfitv" "𝒗" julia-latexsubs)
(puthash "\\mbfitw" "𝒘" julia-latexsubs)
(puthash "\\mbfitx" "𝒙" julia-latexsubs)
(puthash "\\mbfity" "𝒚" julia-latexsubs)
(puthash "\\mbfitz" "𝒛" julia-latexsubs)
(puthash "\\mscrA" "𝒜" julia-latexsubs)
(puthash "\\mscrC" "𝒞" julia-latexsubs)
(puthash "\\mscrD" "𝒟" julia-latexsubs)
(puthash "\\mscrG" "𝒢" julia-latexsubs)
(puthash "\\mscrJ" "𝒥" julia-latexsubs)
(puthash "\\mscrK" "𝒦" julia-latexsubs)
(puthash "\\mscrN" "𝒩" julia-latexsubs)
(puthash "\\mscrO" "𝒪" julia-latexsubs)
(puthash "\\mscrP" "𝒫" julia-latexsubs)
(puthash "\\mscrQ" "𝒬" julia-latexsubs)
(puthash "\\mscrS" "𝒮" julia-latexsubs)
(puthash "\\mscrT" "𝒯" julia-latexsubs)
(puthash "\\mscrU" "𝒰" julia-latexsubs)
(puthash "\\mscrV" "𝒱" julia-latexsubs)
(puthash "\\mscrW" "𝒲" julia-latexsubs)
(puthash "\\mscrX" "𝒳" julia-latexsubs)
(puthash "\\mscrY" "𝒴" julia-latexsubs)
(puthash "\\mscrZ" "𝒵" julia-latexsubs)
(puthash "\\mscra" "𝒶" julia-latexsubs)
(puthash "\\mscrb" "𝒷" julia-latexsubs)
(puthash "\\mscrc" "𝒸" julia-latexsubs)
(puthash "\\mscrd" "𝒹" julia-latexsubs)
(puthash "\\mscrf" "𝒻" julia-latexsubs)
(puthash "\\mscrh" "𝒽" julia-latexsubs)
(puthash "\\mscri" "𝒾" julia-latexsubs)
(puthash "\\mscrj" "𝒿" julia-latexsubs)
(puthash "\\mscrk" "𝓀" julia-latexsubs)
(let ((c (decode-char 'ucs #x1d4c1)))
  (if c (puthash "\\mscrl" (char-to-string c) julia-latexsubs)))
(puthash "\\mscrm" "𝓂" julia-latexsubs)
(puthash "\\mscrn" "𝓃" julia-latexsubs)
(puthash "\\mscrp" "𝓅" julia-latexsubs)
(puthash "\\mscrq" "𝓆" julia-latexsubs)
(puthash "\\mscrr" "𝓇" julia-latexsubs)
(puthash "\\mscrs" "𝓈" julia-latexsubs)
(puthash "\\mscrt" "𝓉" julia-latexsubs)
(puthash "\\mscru" "𝓊" julia-latexsubs)
(puthash "\\mscrv" "𝓋" julia-latexsubs)
(puthash "\\mscrw" "𝓌" julia-latexsubs)
(puthash "\\mscrx" "𝓍" julia-latexsubs)
(puthash "\\mscry" "𝓎" julia-latexsubs)
(puthash "\\mscrz" "𝓏" julia-latexsubs)
(puthash "\\mbfscrA" "𝓐" julia-latexsubs)
(puthash "\\mbfscrB" "𝓑" julia-latexsubs)
(puthash "\\mbfscrC" "𝓒" julia-latexsubs)
(puthash "\\mbfscrD" "𝓓" julia-latexsubs)
(puthash "\\mbfscrE" "𝓔" julia-latexsubs)
(puthash "\\mbfscrF" "𝓕" julia-latexsubs)
(puthash "\\mbfscrG" "𝓖" julia-latexsubs)
(puthash "\\mbfscrH" "𝓗" julia-latexsubs)
(puthash "\\mbfscrI" "𝓘" julia-latexsubs)
(puthash "\\mbfscrJ" "𝓙" julia-latexsubs)
(puthash "\\mbfscrK" "𝓚" julia-latexsubs)
(puthash "\\mbfscrL" "𝓛" julia-latexsubs)
(puthash "\\mbfscrM" "𝓜" julia-latexsubs)
(puthash "\\mbfscrN" "𝓝" julia-latexsubs)
(puthash "\\mbfscrO" "𝓞" julia-latexsubs)
(puthash "\\mbfscrP" "𝓟" julia-latexsubs)
(puthash "\\mbfscrQ" "𝓠" julia-latexsubs)
(puthash "\\mbfscrR" "𝓡" julia-latexsubs)
(puthash "\\mbfscrS" "𝓢" julia-latexsubs)
(puthash "\\mbfscrT" "𝓣" julia-latexsubs)
(puthash "\\mbfscrU" "𝓤" julia-latexsubs)
(puthash "\\mbfscrV" "𝓥" julia-latexsubs)
(puthash "\\mbfscrW" "𝓦" julia-latexsubs)
(puthash "\\mbfscrX" "𝓧" julia-latexsubs)
(puthash "\\mbfscrY" "𝓨" julia-latexsubs)
(puthash "\\mbfscrZ" "𝓩" julia-latexsubs)
(puthash "\\mbfscra" "𝓪" julia-latexsubs)
(puthash "\\mbfscrb" "𝓫" julia-latexsubs)
(puthash "\\mbfscrc" "𝓬" julia-latexsubs)
(puthash "\\mbfscrd" "𝓭" julia-latexsubs)
(puthash "\\mbfscre" "𝓮" julia-latexsubs)
(puthash "\\mbfscrf" "𝓯" julia-latexsubs)
(puthash "\\mbfscrg" "𝓰" julia-latexsubs)
(puthash "\\mbfscrh" "𝓱" julia-latexsubs)
(puthash "\\mbfscri" "𝓲" julia-latexsubs)
(puthash "\\mbfscrj" "𝓳" julia-latexsubs)
(puthash "\\mbfscrk" "𝓴" julia-latexsubs)
(puthash "\\mbfscrl" "𝓵" julia-latexsubs)
(puthash "\\mbfscrm" "𝓶" julia-latexsubs)
(puthash "\\mbfscrn" "𝓷" julia-latexsubs)
(puthash "\\mbfscro" "𝓸" julia-latexsubs)
(puthash "\\mbfscrp" "𝓹" julia-latexsubs)
(puthash "\\mbfscrq" "𝓺" julia-latexsubs)
(puthash "\\mbfscrr" "𝓻" julia-latexsubs)
(puthash "\\mbfscrs" "𝓼" julia-latexsubs)
(puthash "\\mbfscrt" "𝓽" julia-latexsubs)
(puthash "\\mbfscru" "𝓾" julia-latexsubs)
(puthash "\\mbfscrv" "𝓿" julia-latexsubs)
(puthash "\\mbfscrw" "𝔀" julia-latexsubs)
(puthash "\\mbfscrx" "𝔁" julia-latexsubs)
(puthash "\\mbfscry" "𝔂" julia-latexsubs)
(puthash "\\mbfscrz" "𝔃" julia-latexsubs)
(puthash "\\mfrakA" "𝔄" julia-latexsubs)
(puthash "\\mfrakB" "𝔅" julia-latexsubs)
(puthash "\\mfrakD" "𝔇" julia-latexsubs)
(puthash "\\mfrakE" "𝔈" julia-latexsubs)
(puthash "\\mfrakF" "𝔉" julia-latexsubs)
(puthash "\\mfrakG" "𝔊" julia-latexsubs)
(puthash "\\mfrakJ" "𝔍" julia-latexsubs)
(puthash "\\mfrakK" "𝔎" julia-latexsubs)
(puthash "\\mfrakL" "𝔏" julia-latexsubs)
(puthash "\\mfrakM" "𝔐" julia-latexsubs)
(puthash "\\mfrakN" "𝔑" julia-latexsubs)
(puthash "\\mfrakO" "𝔒" julia-latexsubs)
(puthash "\\mfrakP" "𝔓" julia-latexsubs)
(puthash "\\mfrakQ" "𝔔" julia-latexsubs)
(puthash "\\mfrakS" "𝔖" julia-latexsubs)
(puthash "\\mfrakT" "𝔗" julia-latexsubs)
(puthash "\\mfrakU" "𝔘" julia-latexsubs)
(puthash "\\mfrakV" "𝔙" julia-latexsubs)
(puthash "\\mfrakW" "𝔚" julia-latexsubs)
(puthash "\\mfrakX" "𝔛" julia-latexsubs)
(puthash "\\mfrakY" "𝔜" julia-latexsubs)
(puthash "\\mfraka" "𝔞" julia-latexsubs)
(puthash "\\mfrakb" "𝔟" julia-latexsubs)
(puthash "\\mfrakc" "𝔠" julia-latexsubs)
(puthash "\\mfrakd" "𝔡" julia-latexsubs)
(puthash "\\mfrake" "𝔢" julia-latexsubs)
(puthash "\\mfrakf" "𝔣" julia-latexsubs)
(puthash "\\mfrakg" "𝔤" julia-latexsubs)
(puthash "\\mfrakh" "𝔥" julia-latexsubs)
(puthash "\\mfraki" "𝔦" julia-latexsubs)
(puthash "\\mfrakj" "𝔧" julia-latexsubs)
(puthash "\\mfrakk" "𝔨" julia-latexsubs)
(puthash "\\mfrakl" "𝔩" julia-latexsubs)
(puthash "\\mfrakm" "𝔪" julia-latexsubs)
(puthash "\\mfrakn" "𝔫" julia-latexsubs)
(puthash "\\mfrako" "𝔬" julia-latexsubs)
(puthash "\\mfrakp" "𝔭" julia-latexsubs)
(puthash "\\mfrakq" "𝔮" julia-latexsubs)
(puthash "\\mfrakr" "𝔯" julia-latexsubs)
(puthash "\\mfraks" "𝔰" julia-latexsubs)
(puthash "\\mfrakt" "𝔱" julia-latexsubs)
(puthash "\\mfraku" "𝔲" julia-latexsubs)
(puthash "\\mfrakv" "𝔳" julia-latexsubs)
(puthash "\\mfrakw" "𝔴" julia-latexsubs)
(puthash "\\mfrakx" "𝔵" julia-latexsubs)
(puthash "\\mfraky" "𝔶" julia-latexsubs)
(puthash "\\mfrakz" "𝔷" julia-latexsubs)
(puthash "\\BbbA" "𝔸" julia-latexsubs)
(puthash "\\BbbB" "𝔹" julia-latexsubs)
(puthash "\\BbbD" "𝔻" julia-latexsubs)
(puthash "\\BbbE" "𝔼" julia-latexsubs)
(puthash "\\BbbF" "𝔽" julia-latexsubs)
(puthash "\\BbbG" "𝔾" julia-latexsubs)
(puthash "\\BbbI" "𝕀" julia-latexsubs)
(puthash "\\BbbJ" "𝕁" julia-latexsubs)
(puthash "\\BbbK" "𝕂" julia-latexsubs)
(puthash "\\BbbL" "𝕃" julia-latexsubs)
(puthash "\\BbbM" "𝕄" julia-latexsubs)
(puthash "\\BbbO" "𝕆" julia-latexsubs)
(puthash "\\BbbS" "𝕊" julia-latexsubs)
(puthash "\\BbbT" "𝕋" julia-latexsubs)
(puthash "\\BbbU" "𝕌" julia-latexsubs)
(puthash "\\BbbV" "𝕍" julia-latexsubs)
(puthash "\\BbbW" "𝕎" julia-latexsubs)
(puthash "\\BbbX" "𝕏" julia-latexsubs)
(puthash "\\BbbY" "𝕐" julia-latexsubs)
(puthash "\\Bbba" "𝕒" julia-latexsubs)
(puthash "\\Bbbb" "𝕓" julia-latexsubs)
(puthash "\\Bbbc" "𝕔" julia-latexsubs)
(puthash "\\Bbbd" "𝕕" julia-latexsubs)
(puthash "\\Bbbe" "𝕖" julia-latexsubs)
(puthash "\\Bbbf" "𝕗" julia-latexsubs)
(puthash "\\Bbbg" "𝕘" julia-latexsubs)
(puthash "\\Bbbh" "𝕙" julia-latexsubs)
(puthash "\\Bbbi" "𝕚" julia-latexsubs)
(puthash "\\Bbbj" "𝕛" julia-latexsubs)
(puthash "\\Bbbk" "𝕜" julia-latexsubs)
(puthash "\\Bbbl" "𝕝" julia-latexsubs)
(puthash "\\Bbbm" "𝕞" julia-latexsubs)
(puthash "\\Bbbn" "𝕟" julia-latexsubs)
(puthash "\\Bbbo" "𝕠" julia-latexsubs)
(puthash "\\Bbbp" "𝕡" julia-latexsubs)
(puthash "\\Bbbq" "𝕢" julia-latexsubs)
(puthash "\\Bbbr" "𝕣" julia-latexsubs)
(puthash "\\Bbbs" "𝕤" julia-latexsubs)
(puthash "\\Bbbt" "𝕥" julia-latexsubs)
(puthash "\\Bbbu" "𝕦" julia-latexsubs)
(puthash "\\Bbbv" "𝕧" julia-latexsubs)
(puthash "\\Bbbw" "𝕨" julia-latexsubs)
(puthash "\\Bbbx" "𝕩" julia-latexsubs)
(puthash "\\Bbby" "𝕪" julia-latexsubs)
(puthash "\\Bbbz" "𝕫" julia-latexsubs)
(puthash "\\mbffrakA" "𝕬" julia-latexsubs)
(puthash "\\mbffrakB" "𝕭" julia-latexsubs)
(puthash "\\mbffrakC" "𝕮" julia-latexsubs)
(puthash "\\mbffrakD" "𝕯" julia-latexsubs)
(puthash "\\mbffrakE" "𝕰" julia-latexsubs)
(puthash "\\mbffrakF" "𝕱" julia-latexsubs)
(puthash "\\mbffrakG" "𝕲" julia-latexsubs)
(puthash "\\mbffrakH" "𝕳" julia-latexsubs)
(puthash "\\mbffrakI" "𝕴" julia-latexsubs)
(puthash "\\mbffrakJ" "𝕵" julia-latexsubs)
(puthash "\\mbffrakK" "𝕶" julia-latexsubs)
(puthash "\\mbffrakL" "𝕷" julia-latexsubs)
(puthash "\\mbffrakM" "𝕸" julia-latexsubs)
(puthash "\\mbffrakN" "𝕹" julia-latexsubs)
(puthash "\\mbffrakO" "𝕺" julia-latexsubs)
(puthash "\\mbffrakP" "𝕻" julia-latexsubs)
(puthash "\\mbffrakQ" "𝕼" julia-latexsubs)
(puthash "\\mbffrakR" "𝕽" julia-latexsubs)
(puthash "\\mbffrakS" "𝕾" julia-latexsubs)
(puthash "\\mbffrakT" "𝕿" julia-latexsubs)
(puthash "\\mbffrakU" "𝖀" julia-latexsubs)
(puthash "\\mbffrakV" "𝖁" julia-latexsubs)
(puthash "\\mbffrakW" "𝖂" julia-latexsubs)
(puthash "\\mbffrakX" "𝖃" julia-latexsubs)
(puthash "\\mbffrakY" "𝖄" julia-latexsubs)
(puthash "\\mbffrakZ" "𝖅" julia-latexsubs)
(puthash "\\mbffraka" "𝖆" julia-latexsubs)
(puthash "\\mbffrakb" "𝖇" julia-latexsubs)
(puthash "\\mbffrakc" "𝖈" julia-latexsubs)
(puthash "\\mbffrakd" "𝖉" julia-latexsubs)
(puthash "\\mbffrake" "𝖊" julia-latexsubs)
(puthash "\\mbffrakf" "𝖋" julia-latexsubs)
(puthash "\\mbffrakg" "𝖌" julia-latexsubs)
(puthash "\\mbffrakh" "𝖍" julia-latexsubs)
(puthash "\\mbffraki" "𝖎" julia-latexsubs)
(puthash "\\mbffrakj" "𝖏" julia-latexsubs)
(puthash "\\mbffrakk" "𝖐" julia-latexsubs)
(puthash "\\mbffrakl" "𝖑" julia-latexsubs)
(puthash "\\mbffrakm" "𝖒" julia-latexsubs)
(puthash "\\mbffrakn" "𝖓" julia-latexsubs)
(puthash "\\mbffrako" "𝖔" julia-latexsubs)
(puthash "\\mbffrakp" "𝖕" julia-latexsubs)
(puthash "\\mbffrakq" "𝖖" julia-latexsubs)
(puthash "\\mbffrakr" "𝖗" julia-latexsubs)
(puthash "\\mbffraks" "𝖘" julia-latexsubs)
(puthash "\\mbffrakt" "𝖙" julia-latexsubs)
(puthash "\\mbffraku" "𝖚" julia-latexsubs)
(puthash "\\mbffrakv" "𝖛" julia-latexsubs)
(puthash "\\mbffrakw" "𝖜" julia-latexsubs)
(puthash "\\mbffrakx" "𝖝" julia-latexsubs)
(puthash "\\mbffraky" "𝖞" julia-latexsubs)
(puthash "\\mbffrakz" "𝖟" julia-latexsubs)
(puthash "\\msansA" "𝖠" julia-latexsubs)
(puthash "\\msansB" "𝖡" julia-latexsubs)
(puthash "\\msansC" "𝖢" julia-latexsubs)
(puthash "\\msansD" "𝖣" julia-latexsubs)
(puthash "\\msansE" "𝖤" julia-latexsubs)
(puthash "\\msansF" "𝖥" julia-latexsubs)
(puthash "\\msansG" "𝖦" julia-latexsubs)
(puthash "\\msansH" "𝖧" julia-latexsubs)
(puthash "\\msansI" "𝖨" julia-latexsubs)
(puthash "\\msansJ" "𝖩" julia-latexsubs)
(puthash "\\msansK" "𝖪" julia-latexsubs)
(puthash "\\msansL" "𝖫" julia-latexsubs)
(puthash "\\msansM" "𝖬" julia-latexsubs)
(puthash "\\msansN" "𝖭" julia-latexsubs)
(puthash "\\msansO" "𝖮" julia-latexsubs)
(puthash "\\msansP" "𝖯" julia-latexsubs)
(puthash "\\msansQ" "𝖰" julia-latexsubs)
(puthash "\\msansR" "𝖱" julia-latexsubs)
(puthash "\\msansS" "𝖲" julia-latexsubs)
(puthash "\\msansT" "𝖳" julia-latexsubs)
(puthash "\\msansU" "𝖴" julia-latexsubs)
(puthash "\\msansV" "𝖵" julia-latexsubs)
(puthash "\\msansW" "𝖶" julia-latexsubs)
(puthash "\\msansX" "𝖷" julia-latexsubs)
(puthash "\\msansY" "𝖸" julia-latexsubs)
(puthash "\\msansZ" "𝖹" julia-latexsubs)
(puthash "\\msansa" "𝖺" julia-latexsubs)
(puthash "\\msansb" "𝖻" julia-latexsubs)
(puthash "\\msansc" "𝖼" julia-latexsubs)
(puthash "\\msansd" "𝖽" julia-latexsubs)
(puthash "\\msanse" "𝖾" julia-latexsubs)
(puthash "\\msansf" "𝖿" julia-latexsubs)
(puthash "\\msansg" "𝗀" julia-latexsubs)
(puthash "\\msansh" "𝗁" julia-latexsubs)
(puthash "\\msansi" "𝗂" julia-latexsubs)
(puthash "\\msansj" "𝗃" julia-latexsubs)
(puthash "\\msansk" "𝗄" julia-latexsubs)
(puthash "\\msansl" "𝗅" julia-latexsubs)
(puthash "\\msansm" "𝗆" julia-latexsubs)
(puthash "\\msansn" "𝗇" julia-latexsubs)
(puthash "\\msanso" "𝗈" julia-latexsubs)
(puthash "\\msansp" "𝗉" julia-latexsubs)
(puthash "\\msansq" "𝗊" julia-latexsubs)
(puthash "\\msansr" "𝗋" julia-latexsubs)
(puthash "\\msanss" "𝗌" julia-latexsubs)
(puthash "\\msanst" "𝗍" julia-latexsubs)
(puthash "\\msansu" "𝗎" julia-latexsubs)
(puthash "\\msansv" "𝗏" julia-latexsubs)
(puthash "\\msansw" "𝗐" julia-latexsubs)
(puthash "\\msansx" "𝗑" julia-latexsubs)
(puthash "\\msansy" "𝗒" julia-latexsubs)
(puthash "\\msansz" "𝗓" julia-latexsubs)
(puthash "\\mbfsansA" "𝗔" julia-latexsubs)
(puthash "\\mbfsansB" "𝗕" julia-latexsubs)
(puthash "\\mbfsansC" "𝗖" julia-latexsubs)
(puthash "\\mbfsansD" "𝗗" julia-latexsubs)
(puthash "\\mbfsansE" "𝗘" julia-latexsubs)
(puthash "\\mbfsansF" "𝗙" julia-latexsubs)
(puthash "\\mbfsansG" "𝗚" julia-latexsubs)
(puthash "\\mbfsansH" "𝗛" julia-latexsubs)
(puthash "\\mbfsansI" "𝗜" julia-latexsubs)
(puthash "\\mbfsansJ" "𝗝" julia-latexsubs)
(puthash "\\mbfsansK" "𝗞" julia-latexsubs)
(puthash "\\mbfsansL" "𝗟" julia-latexsubs)
(puthash "\\mbfsansM" "𝗠" julia-latexsubs)
(puthash "\\mbfsansN" "𝗡" julia-latexsubs)
(puthash "\\mbfsansO" "𝗢" julia-latexsubs)
(puthash "\\mbfsansP" "𝗣" julia-latexsubs)
(puthash "\\mbfsansQ" "𝗤" julia-latexsubs)
(puthash "\\mbfsansR" "𝗥" julia-latexsubs)
(puthash "\\mbfsansS" "𝗦" julia-latexsubs)
(puthash "\\mbfsansT" "𝗧" julia-latexsubs)
(puthash "\\mbfsansU" "𝗨" julia-latexsubs)
(puthash "\\mbfsansV" "𝗩" julia-latexsubs)
(puthash "\\mbfsansW" "𝗪" julia-latexsubs)
(puthash "\\mbfsansX" "𝗫" julia-latexsubs)
(puthash "\\mbfsansY" "𝗬" julia-latexsubs)
(puthash "\\mbfsansZ" "𝗭" julia-latexsubs)
(puthash "\\mbfsansa" "𝗮" julia-latexsubs)
(puthash "\\mbfsansb" "𝗯" julia-latexsubs)
(puthash "\\mbfsansc" "𝗰" julia-latexsubs)
(puthash "\\mbfsansd" "𝗱" julia-latexsubs)
(puthash "\\mbfsanse" "𝗲" julia-latexsubs)
(puthash "\\mbfsansf" "𝗳" julia-latexsubs)
(puthash "\\mbfsansg" "𝗴" julia-latexsubs)
(puthash "\\mbfsansh" "𝗵" julia-latexsubs)
(puthash "\\mbfsansi" "𝗶" julia-latexsubs)
(puthash "\\mbfsansj" "𝗷" julia-latexsubs)
(puthash "\\mbfsansk" "𝗸" julia-latexsubs)
(puthash "\\mbfsansl" "𝗹" julia-latexsubs)
(puthash "\\mbfsansm" "𝗺" julia-latexsubs)
(puthash "\\mbfsansn" "𝗻" julia-latexsubs)
(puthash "\\mbfsanso" "𝗼" julia-latexsubs)
(puthash "\\mbfsansp" "𝗽" julia-latexsubs)
(puthash "\\mbfsansq" "𝗾" julia-latexsubs)
(puthash "\\mbfsansr" "𝗿" julia-latexsubs)
(puthash "\\mbfsanss" "𝘀" julia-latexsubs)
(puthash "\\mbfsanst" "𝘁" julia-latexsubs)
(puthash "\\mbfsansu" "𝘂" julia-latexsubs)
(puthash "\\mbfsansv" "𝘃" julia-latexsubs)
(puthash "\\mbfsansw" "𝘄" julia-latexsubs)
(puthash "\\mbfsansx" "𝘅" julia-latexsubs)
(puthash "\\mbfsansy" "𝘆" julia-latexsubs)
(puthash "\\mbfsansz" "𝘇" julia-latexsubs)
(puthash "\\mitsansA" "𝘈" julia-latexsubs)
(puthash "\\mitsansB" "𝘉" julia-latexsubs)
(puthash "\\mitsansC" "𝘊" julia-latexsubs)
(puthash "\\mitsansD" "𝘋" julia-latexsubs)
(puthash "\\mitsansE" "𝘌" julia-latexsubs)
(puthash "\\mitsansF" "𝘍" julia-latexsubs)
(puthash "\\mitsansG" "𝘎" julia-latexsubs)
(puthash "\\mitsansH" "𝘏" julia-latexsubs)
(puthash "\\mitsansI" "𝘐" julia-latexsubs)
(puthash "\\mitsansJ" "𝘑" julia-latexsubs)
(puthash "\\mitsansK" "𝘒" julia-latexsubs)
(puthash "\\mitsansL" "𝘓" julia-latexsubs)
(puthash "\\mitsansM" "𝘔" julia-latexsubs)
(puthash "\\mitsansN" "𝘕" julia-latexsubs)
(puthash "\\mitsansO" "𝘖" julia-latexsubs)
(puthash "\\mitsansP" "𝘗" julia-latexsubs)
(puthash "\\mitsansQ" "𝘘" julia-latexsubs)
(puthash "\\mitsansR" "𝘙" julia-latexsubs)
(puthash "\\mitsansS" "𝘚" julia-latexsubs)
(puthash "\\mitsansT" "𝘛" julia-latexsubs)
(puthash "\\mitsansU" "𝘜" julia-latexsubs)
(puthash "\\mitsansV" "𝘝" julia-latexsubs)
(puthash "\\mitsansW" "𝘞" julia-latexsubs)
(puthash "\\mitsansX" "𝘟" julia-latexsubs)
(puthash "\\mitsansY" "𝘠" julia-latexsubs)
(puthash "\\mitsansZ" "𝘡" julia-latexsubs)
(puthash "\\mitsansa" "𝘢" julia-latexsubs)
(puthash "\\mitsansb" "𝘣" julia-latexsubs)
(puthash "\\mitsansc" "𝘤" julia-latexsubs)
(puthash "\\mitsansd" "𝘥" julia-latexsubs)
(puthash "\\mitsanse" "𝘦" julia-latexsubs)
(puthash "\\mitsansf" "𝘧" julia-latexsubs)
(puthash "\\mitsansg" "𝘨" julia-latexsubs)
(puthash "\\mitsansh" "𝘩" julia-latexsubs)
(puthash "\\mitsansi" "𝘪" julia-latexsubs)
(puthash "\\mitsansj" "𝘫" julia-latexsubs)
(puthash "\\mitsansk" "𝘬" julia-latexsubs)
(puthash "\\mitsansl" "𝘭" julia-latexsubs)
(puthash "\\mitsansm" "𝘮" julia-latexsubs)
(puthash "\\mitsansn" "𝘯" julia-latexsubs)
(puthash "\\mitsanso" "𝘰" julia-latexsubs)
(puthash "\\mitsansp" "𝘱" julia-latexsubs)
(puthash "\\mitsansq" "𝘲" julia-latexsubs)
(puthash "\\mitsansr" "𝘳" julia-latexsubs)
(puthash "\\mitsanss" "𝘴" julia-latexsubs)
(puthash "\\mitsanst" "𝘵" julia-latexsubs)
(puthash "\\mitsansu" "𝘶" julia-latexsubs)
(puthash "\\mitsansv" "𝘷" julia-latexsubs)
(puthash "\\mitsansw" "𝘸" julia-latexsubs)
(puthash "\\mitsansx" "𝘹" julia-latexsubs)
(puthash "\\mitsansy" "𝘺" julia-latexsubs)
(puthash "\\mitsansz" "𝘻" julia-latexsubs)
(puthash "\\mbfitsansA" "𝘼" julia-latexsubs)
(puthash "\\mbfitsansB" "𝘽" julia-latexsubs)
(puthash "\\mbfitsansC" "𝘾" julia-latexsubs)
(puthash "\\mbfitsansD" "𝘿" julia-latexsubs)
(puthash "\\mbfitsansE" "𝙀" julia-latexsubs)
(puthash "\\mbfitsansF" "𝙁" julia-latexsubs)
(puthash "\\mbfitsansG" "𝙂" julia-latexsubs)
(puthash "\\mbfitsansH" "𝙃" julia-latexsubs)
(puthash "\\mbfitsansI" "𝙄" julia-latexsubs)
(puthash "\\mbfitsansJ" "𝙅" julia-latexsubs)
(puthash "\\mbfitsansK" "𝙆" julia-latexsubs)
(puthash "\\mbfitsansL" "𝙇" julia-latexsubs)
(puthash "\\mbfitsansM" "𝙈" julia-latexsubs)
(puthash "\\mbfitsansN" "𝙉" julia-latexsubs)
(puthash "\\mbfitsansO" "𝙊" julia-latexsubs)
(puthash "\\mbfitsansP" "𝙋" julia-latexsubs)
(puthash "\\mbfitsansQ" "𝙌" julia-latexsubs)
(puthash "\\mbfitsansR" "𝙍" julia-latexsubs)
(puthash "\\mbfitsansS" "𝙎" julia-latexsubs)
(puthash "\\mbfitsansT" "𝙏" julia-latexsubs)
(puthash "\\mbfitsansU" "𝙐" julia-latexsubs)
(puthash "\\mbfitsansV" "𝙑" julia-latexsubs)
(puthash "\\mbfitsansW" "𝙒" julia-latexsubs)
(puthash "\\mbfitsansX" "𝙓" julia-latexsubs)
(puthash "\\mbfitsansY" "𝙔" julia-latexsubs)
(puthash "\\mbfitsansZ" "𝙕" julia-latexsubs)
(puthash "\\mbfitsansa" "𝙖" julia-latexsubs)
(puthash "\\mbfitsansb" "𝙗" julia-latexsubs)
(puthash "\\mbfitsansc" "𝙘" julia-latexsubs)
(puthash "\\mbfitsansd" "𝙙" julia-latexsubs)
(puthash "\\mbfitsanse" "𝙚" julia-latexsubs)
(puthash "\\mbfitsansf" "𝙛" julia-latexsubs)
(puthash "\\mbfitsansg" "𝙜" julia-latexsubs)
(puthash "\\mbfitsansh" "𝙝" julia-latexsubs)
(puthash "\\mbfitsansi" "𝙞" julia-latexsubs)
(puthash "\\mbfitsansj" "𝙟" julia-latexsubs)
(puthash "\\mbfitsansk" "𝙠" julia-latexsubs)
(puthash "\\mbfitsansl" "𝙡" julia-latexsubs)
(puthash "\\mbfitsansm" "𝙢" julia-latexsubs)
(puthash "\\mbfitsansn" "𝙣" julia-latexsubs)
(puthash "\\mbfitsanso" "𝙤" julia-latexsubs)
(puthash "\\mbfitsansp" "𝙥" julia-latexsubs)
(puthash "\\mbfitsansq" "𝙦" julia-latexsubs)
(puthash "\\mbfitsansr" "𝙧" julia-latexsubs)
(puthash "\\mbfitsanss" "𝙨" julia-latexsubs)
(puthash "\\mbfitsanst" "𝙩" julia-latexsubs)
(puthash "\\mbfitsansu" "𝙪" julia-latexsubs)
(puthash "\\mbfitsansv" "𝙫" julia-latexsubs)
(puthash "\\mbfitsansw" "𝙬" julia-latexsubs)
(puthash "\\mbfitsansx" "𝙭" julia-latexsubs)
(puthash "\\mbfitsansy" "𝙮" julia-latexsubs)
(puthash "\\mbfitsansz" "𝙯" julia-latexsubs)
(puthash "\\mttA" "𝙰" julia-latexsubs)
(puthash "\\mttB" "𝙱" julia-latexsubs)
(puthash "\\mttC" "𝙲" julia-latexsubs)
(puthash "\\mttD" "𝙳" julia-latexsubs)
(puthash "\\mttE" "𝙴" julia-latexsubs)
(puthash "\\mttF" "𝙵" julia-latexsubs)
(puthash "\\mttG" "𝙶" julia-latexsubs)
(puthash "\\mttH" "𝙷" julia-latexsubs)
(puthash "\\mttI" "𝙸" julia-latexsubs)
(puthash "\\mttJ" "𝙹" julia-latexsubs)
(puthash "\\mttK" "𝙺" julia-latexsubs)
(puthash "\\mttL" "𝙻" julia-latexsubs)
(puthash "\\mttM" "𝙼" julia-latexsubs)
(puthash "\\mttN" "𝙽" julia-latexsubs)
(puthash "\\mttO" "𝙾" julia-latexsubs)
(puthash "\\mttP" "𝙿" julia-latexsubs)
(puthash "\\mttQ" "𝚀" julia-latexsubs)
(puthash "\\mttR" "𝚁" julia-latexsubs)
(puthash "\\mttS" "𝚂" julia-latexsubs)
(puthash "\\mttT" "𝚃" julia-latexsubs)
(puthash "\\mttU" "𝚄" julia-latexsubs)
(puthash "\\mttV" "𝚅" julia-latexsubs)
(puthash "\\mttW" "𝚆" julia-latexsubs)
(puthash "\\mttX" "𝚇" julia-latexsubs)
(puthash "\\mttY" "𝚈" julia-latexsubs)
(puthash "\\mttZ" "𝚉" julia-latexsubs)
(puthash "\\mtta" "𝚊" julia-latexsubs)
(puthash "\\mttb" "𝚋" julia-latexsubs)
(puthash "\\mttc" "𝚌" julia-latexsubs)
(puthash "\\mttd" "𝚍" julia-latexsubs)
(puthash "\\mtte" "𝚎" julia-latexsubs)
(puthash "\\mttf" "𝚏" julia-latexsubs)
(puthash "\\mttg" "𝚐" julia-latexsubs)
(puthash "\\mtth" "𝚑" julia-latexsubs)
(puthash "\\mtti" "𝚒" julia-latexsubs)
(puthash "\\mttj" "𝚓" julia-latexsubs)
(puthash "\\mttk" "𝚔" julia-latexsubs)
(puthash "\\mttl" "𝚕" julia-latexsubs)
(puthash "\\mttm" "𝚖" julia-latexsubs)
(puthash "\\mttn" "𝚗" julia-latexsubs)
(puthash "\\mtto" "𝚘" julia-latexsubs)
(puthash "\\mttp" "𝚙" julia-latexsubs)
(puthash "\\mttq" "𝚚" julia-latexsubs)
(puthash "\\mttr" "𝚛" julia-latexsubs)
(puthash "\\mtts" "𝚜" julia-latexsubs)
(puthash "\\mttt" "𝚝" julia-latexsubs)
(puthash "\\mttu" "𝚞" julia-latexsubs)
(puthash "\\mttv" "𝚟" julia-latexsubs)
(puthash "\\mttw" "𝚠" julia-latexsubs)
(puthash "\\mttx" "𝚡" julia-latexsubs)
(puthash "\\mtty" "𝚢" julia-latexsubs)
(puthash "\\mttz" "𝚣" julia-latexsubs)
(let ((c (decode-char 'ucs #x1d6a4)))
  (if c (puthash "\\imath" (char-to-string c) julia-latexsubs)))
(let ((c (decode-char 'ucs #x1d6a5)))
  (if c (puthash "\\jmath" (char-to-string c) julia-latexsubs)))
(puthash "\\mbfAlpha" "𝚨" julia-latexsubs)
(puthash "\\mbfBeta" "𝚩" julia-latexsubs)
(puthash "\\mbfGamma" "𝚪" julia-latexsubs)
(puthash "\\mbfDelta" "𝚫" julia-latexsubs)
(puthash "\\mbfEpsilon" "𝚬" julia-latexsubs)
(puthash "\\mbfZeta" "𝚭" julia-latexsubs)
(puthash "\\mbfEta" "𝚮" julia-latexsubs)
(puthash "\\mbfTheta" "𝚯" julia-latexsubs)
(puthash "\\mbfIota" "𝚰" julia-latexsubs)
(puthash "\\mbfKappa" "𝚱" julia-latexsubs)
(puthash "\\mbfLambda" "𝚲" julia-latexsubs)
(puthash "\\mbfMu" "𝚳" julia-latexsubs)
(puthash "\\mbfNu" "𝚴" julia-latexsubs)
(puthash "\\mbfXi" "𝚵" julia-latexsubs)
(puthash "\\mbfOmicron" "𝚶" julia-latexsubs)
(puthash "\\mbfPi" "𝚷" julia-latexsubs)
(puthash "\\mbfRho" "𝚸" julia-latexsubs)
(puthash "\\mbfvarTheta" "𝚹" julia-latexsubs)
(puthash "\\mbfSigma" "𝚺" julia-latexsubs)
(puthash "\\mbfTau" "𝚻" julia-latexsubs)
(puthash "\\mbfUpsilon" "𝚼" julia-latexsubs)
(puthash "\\mbfPhi" "𝚽" julia-latexsubs)
(puthash "\\mbfChi" "𝚾" julia-latexsubs)
(puthash "\\mbfPsi" "𝚿" julia-latexsubs)
(puthash "\\mbfOmega" "𝛀" julia-latexsubs)
(puthash "\\mbfnabla" "𝛁" julia-latexsubs)
(puthash "\\mbfalpha" "𝛂" julia-latexsubs)
(puthash "\\mbfbeta" "𝛃" julia-latexsubs)
(puthash "\\mbfgamma" "𝛄" julia-latexsubs)
(puthash "\\mbfdelta" "𝛅" julia-latexsubs)
(puthash "\\mbfepsilon" "𝛆" julia-latexsubs)
(puthash "\\mbfzeta" "𝛇" julia-latexsubs)
(puthash "\\mbfeta" "𝛈" julia-latexsubs)
(puthash "\\mbftheta" "𝛉" julia-latexsubs)
(puthash "\\mbfiota" "𝛊" julia-latexsubs)
(puthash "\\mbfkappa" "𝛋" julia-latexsubs)
(puthash "\\mbflambda" "𝛌" julia-latexsubs)
(puthash "\\mbfmu" "𝛍" julia-latexsubs)
(puthash "\\mbfnu" "𝛎" julia-latexsubs)
(puthash "\\mbfxi" "𝛏" julia-latexsubs)
(puthash "\\mbfomicron" "𝛐" julia-latexsubs)
(puthash "\\mbfpi" "𝛑" julia-latexsubs)
(puthash "\\mbfrho" "𝛒" julia-latexsubs)
(puthash "\\mbfvarsigma" "𝛓" julia-latexsubs)
(puthash "\\mbfsigma" "𝛔" julia-latexsubs)
(puthash "\\mbftau" "𝛕" julia-latexsubs)
(puthash "\\mbfupsilon" "𝛖" julia-latexsubs)
(puthash "\\mbfvarphi" "𝛗" julia-latexsubs)
(puthash "\\mbfchi" "𝛘" julia-latexsubs)
(puthash "\\mbfpsi" "𝛙" julia-latexsubs)
(puthash "\\mbfomega" "𝛚" julia-latexsubs)
(puthash "\\mbfpartial" "𝛛" julia-latexsubs)
(puthash "\\mbfvarepsilon" "𝛜" julia-latexsubs)
(puthash "\\mbfvartheta" "𝛝" julia-latexsubs)
(puthash "\\mbfvarkappa" "𝛞" julia-latexsubs)
(puthash "\\mbfphi" "𝛟" julia-latexsubs)
(puthash "\\mbfvarrho" "𝛠" julia-latexsubs)
(puthash "\\mbfvarpi" "𝛡" julia-latexsubs)
(puthash "\\mitAlpha" "𝛢" julia-latexsubs)
(puthash "\\mitBeta" "𝛣" julia-latexsubs)
(puthash "\\mitGamma" "𝛤" julia-latexsubs)
(puthash "\\mitDelta" "𝛥" julia-latexsubs)
(puthash "\\mitEpsilon" "𝛦" julia-latexsubs)
(puthash "\\mitZeta" "𝛧" julia-latexsubs)
(puthash "\\mitEta" "𝛨" julia-latexsubs)
(puthash "\\mitTheta" "𝛩" julia-latexsubs)
(puthash "\\mitIota" "𝛪" julia-latexsubs)
(puthash "\\mitKappa" "𝛫" julia-latexsubs)
(puthash "\\mitLambda" "𝛬" julia-latexsubs)
(puthash "\\mitMu" "𝛭" julia-latexsubs)
(puthash "\\mitNu" "𝛮" julia-latexsubs)
(puthash "\\mitXi" "𝛯" julia-latexsubs)
(puthash "\\mitOmicron" "𝛰" julia-latexsubs)
(puthash "\\mitPi" "𝛱" julia-latexsubs)
(puthash "\\mitRho" "𝛲" julia-latexsubs)
(puthash "\\mitvarTheta" "𝛳" julia-latexsubs)
(puthash "\\mitSigma" "𝛴" julia-latexsubs)
(puthash "\\mitTau" "𝛵" julia-latexsubs)
(puthash "\\mitUpsilon" "𝛶" julia-latexsubs)
(puthash "\\mitPhi" "𝛷" julia-latexsubs)
(puthash "\\mitChi" "𝛸" julia-latexsubs)
(puthash "\\mitPsi" "𝛹" julia-latexsubs)
(puthash "\\mitOmega" "𝛺" julia-latexsubs)
(puthash "\\mitnabla" "𝛻" julia-latexsubs)
(puthash "\\mitalpha" "𝛼" julia-latexsubs)
(puthash "\\mitbeta" "𝛽" julia-latexsubs)
(puthash "\\mitgamma" "𝛾" julia-latexsubs)
(puthash "\\mitdelta" "𝛿" julia-latexsubs)
(puthash "\\mitepsilon" "𝜀" julia-latexsubs)
(puthash "\\mitzeta" "𝜁" julia-latexsubs)
(puthash "\\miteta" "𝜂" julia-latexsubs)
(puthash "\\mittheta" "𝜃" julia-latexsubs)
(puthash "\\mitiota" "𝜄" julia-latexsubs)
(puthash "\\mitkappa" "𝜅" julia-latexsubs)
(puthash "\\mitlambda" "𝜆" julia-latexsubs)
(puthash "\\mitmu" "𝜇" julia-latexsubs)
(puthash "\\mitnu" "𝜈" julia-latexsubs)
(puthash "\\mitxi" "𝜉" julia-latexsubs)
(puthash "\\mitomicron" "𝜊" julia-latexsubs)
(puthash "\\mitpi" "𝜋" julia-latexsubs)
(puthash "\\mitrho" "𝜌" julia-latexsubs)
(puthash "\\mitvarsigma" "𝜍" julia-latexsubs)
(puthash "\\mitsigma" "𝜎" julia-latexsubs)
(puthash "\\mittau" "𝜏" julia-latexsubs)
(puthash "\\mitupsilon" "𝜐" julia-latexsubs)
(puthash "\\mitphi" "𝜑" julia-latexsubs)
(puthash "\\mitchi" "𝜒" julia-latexsubs)
(puthash "\\mitpsi" "𝜓" julia-latexsubs)
(puthash "\\mitomega" "𝜔" julia-latexsubs)
(puthash "\\mitpartial" "𝜕" julia-latexsubs)
(puthash "\\mitvarepsilon" "𝜖" julia-latexsubs)
(puthash "\\mitvartheta" "𝜗" julia-latexsubs)
(puthash "\\mitvarkappa" "𝜘" julia-latexsubs)
(puthash "\\mitvarphi" "𝜙" julia-latexsubs)
(puthash "\\mitvarrho" "𝜚" julia-latexsubs)
(puthash "\\mitvarpi" "𝜛" julia-latexsubs)
(puthash "\\mbfitAlpha" "𝜜" julia-latexsubs)
(puthash "\\mbfitBeta" "𝜝" julia-latexsubs)
(puthash "\\mbfitGamma" "𝜞" julia-latexsubs)
(puthash "\\mbfitDelta" "𝜟" julia-latexsubs)
(puthash "\\mbfitEpsilon" "𝜠" julia-latexsubs)
(puthash "\\mbfitZeta" "𝜡" julia-latexsubs)
(puthash "\\mbfitEta" "𝜢" julia-latexsubs)
(puthash "\\mbfitTheta" "𝜣" julia-latexsubs)
(puthash "\\mbfitIota" "𝜤" julia-latexsubs)
(puthash "\\mbfitKappa" "𝜥" julia-latexsubs)
(puthash "\\mbfitLambda" "𝜦" julia-latexsubs)
(puthash "\\mbfitMu" "𝜧" julia-latexsubs)
(puthash "\\mbfitNu" "𝜨" julia-latexsubs)
(puthash "\\mbfitXi" "𝜩" julia-latexsubs)
(puthash "\\mbfitOmicron" "𝜪" julia-latexsubs)
(puthash "\\mbfitPi" "𝜫" julia-latexsubs)
(puthash "\\mbfitRho" "𝜬" julia-latexsubs)
(puthash "\\mbfitvarTheta" "𝜭" julia-latexsubs)
(puthash "\\mbfitSigma" "𝜮" julia-latexsubs)
(puthash "\\mbfitTau" "𝜯" julia-latexsubs)
(puthash "\\mbfitUpsilon" "𝜰" julia-latexsubs)
(puthash "\\mbfitPhi" "𝜱" julia-latexsubs)
(puthash "\\mbfitChi" "𝜲" julia-latexsubs)
(puthash "\\mbfitPsi" "𝜳" julia-latexsubs)
(puthash "\\mbfitOmega" "𝜴" julia-latexsubs)
(puthash "\\mbfitnabla" "𝜵" julia-latexsubs)
(puthash "\\mbfitalpha" "𝜶" julia-latexsubs)
(puthash "\\mbfitbeta" "𝜷" julia-latexsubs)
(puthash "\\mbfitgamma" "𝜸" julia-latexsubs)
(puthash "\\mbfitdelta" "𝜹" julia-latexsubs)
(puthash "\\mbfitepsilon" "𝜺" julia-latexsubs)
(puthash "\\mbfitzeta" "𝜻" julia-latexsubs)
(puthash "\\mbfiteta" "𝜼" julia-latexsubs)
(puthash "\\mbfittheta" "𝜽" julia-latexsubs)
(puthash "\\mbfitiota" "𝜾" julia-latexsubs)
(puthash "\\mbfitkappa" "𝜿" julia-latexsubs)
(puthash "\\mbfitlambda" "𝝀" julia-latexsubs)
(puthash "\\mbfitmu" "𝝁" julia-latexsubs)
(puthash "\\mbfitnu" "𝝂" julia-latexsubs)
(puthash "\\mbfitxi" "𝝃" julia-latexsubs)
(puthash "\\mbfitomicron" "𝝄" julia-latexsubs)
(puthash "\\mbfitpi" "𝝅" julia-latexsubs)
(puthash "\\mbfitrho" "𝝆" julia-latexsubs)
(puthash "\\mbfitvarsigma" "𝝇" julia-latexsubs)
(puthash "\\mbfitsigma" "𝝈" julia-latexsubs)
(puthash "\\mbfittau" "𝝉" julia-latexsubs)
(puthash "\\mbfitupsilon" "𝝊" julia-latexsubs)
(puthash "\\mbfitphi" "𝝋" julia-latexsubs)
(puthash "\\mbfitchi" "𝝌" julia-latexsubs)
(puthash "\\mbfitpsi" "𝝍" julia-latexsubs)
(puthash "\\mbfitomega" "𝝎" julia-latexsubs)
(puthash "\\mbfitpartial" "𝝏" julia-latexsubs)
(puthash "\\mbfitvarepsilon" "𝝐" julia-latexsubs)
(puthash "\\mbfitvartheta" "𝝑" julia-latexsubs)
(puthash "\\mbfitvarkappa" "𝝒" julia-latexsubs)
(puthash "\\mbfitvarphi" "𝝓" julia-latexsubs)
(puthash "\\mbfitvarrho" "𝝔" julia-latexsubs)
(puthash "\\mbfitvarpi" "𝝕" julia-latexsubs)
(puthash "\\mbfsansAlpha" "𝝖" julia-latexsubs)
(puthash "\\mbfsansBeta" "𝝗" julia-latexsubs)
(puthash "\\mbfsansGamma" "𝝘" julia-latexsubs)
(puthash "\\mbfsansDelta" "𝝙" julia-latexsubs)
(puthash "\\mbfsansEpsilon" "𝝚" julia-latexsubs)
(puthash "\\mbfsansZeta" "𝝛" julia-latexsubs)
(puthash "\\mbfsansEta" "𝝜" julia-latexsubs)
(puthash "\\mbfsansTheta" "𝝝" julia-latexsubs)
(puthash "\\mbfsansIota" "𝝞" julia-latexsubs)
(puthash "\\mbfsansKappa" "𝝟" julia-latexsubs)
(puthash "\\mbfsansLambda" "𝝠" julia-latexsubs)
(puthash "\\mbfsansMu" "𝝡" julia-latexsubs)
(puthash "\\mbfsansNu" "𝝢" julia-latexsubs)
(puthash "\\mbfsansXi" "𝝣" julia-latexsubs)
(puthash "\\mbfsansOmicron" "𝝤" julia-latexsubs)
(puthash "\\mbfsansPi" "𝝥" julia-latexsubs)
(puthash "\\mbfsansRho" "𝝦" julia-latexsubs)
(puthash "\\mbfsansvarTheta" "𝝧" julia-latexsubs)
(puthash "\\mbfsansSigma" "𝝨" julia-latexsubs)
(puthash "\\mbfsansTau" "𝝩" julia-latexsubs)
(puthash "\\mbfsansUpsilon" "𝝪" julia-latexsubs)
(puthash "\\mbfsansPhi" "𝝫" julia-latexsubs)
(puthash "\\mbfsansChi" "𝝬" julia-latexsubs)
(puthash "\\mbfsansPsi" "𝝭" julia-latexsubs)
(puthash "\\mbfsansOmega" "𝝮" julia-latexsubs)
(puthash "\\mbfsansnabla" "𝝯" julia-latexsubs)
(puthash "\\mbfsansalpha" "𝝰" julia-latexsubs)
(puthash "\\mbfsansbeta" "𝝱" julia-latexsubs)
(puthash "\\mbfsansgamma" "𝝲" julia-latexsubs)
(puthash "\\mbfsansdelta" "𝝳" julia-latexsubs)
(puthash "\\mbfsansepsilon" "𝝴" julia-latexsubs)
(puthash "\\mbfsanszeta" "𝝵" julia-latexsubs)
(puthash "\\mbfsanseta" "𝝶" julia-latexsubs)
(puthash "\\mbfsanstheta" "𝝷" julia-latexsubs)
(puthash "\\mbfsansiota" "𝝸" julia-latexsubs)
(puthash "\\mbfsanskappa" "𝝹" julia-latexsubs)
(puthash "\\mbfsanslambda" "𝝺" julia-latexsubs)
(puthash "\\mbfsansmu" "𝝻" julia-latexsubs)
(puthash "\\mbfsansnu" "𝝼" julia-latexsubs)
(puthash "\\mbfsansxi" "𝝽" julia-latexsubs)
(puthash "\\mbfsansomicron" "𝝾" julia-latexsubs)
(puthash "\\mbfsanspi" "𝝿" julia-latexsubs)
(puthash "\\mbfsansrho" "𝞀" julia-latexsubs)
(puthash "\\mbfsansvarsigma" "𝞁" julia-latexsubs)
(puthash "\\mbfsanssigma" "𝞂" julia-latexsubs)
(puthash "\\mbfsanstau" "𝞃" julia-latexsubs)
(puthash "\\mbfsansupsilon" "𝞄" julia-latexsubs)
(puthash "\\mbfsansphi" "𝞅" julia-latexsubs)
(puthash "\\mbfsanschi" "𝞆" julia-latexsubs)
(puthash "\\mbfsanspsi" "𝞇" julia-latexsubs)
(puthash "\\mbfsansomega" "𝞈" julia-latexsubs)
(puthash "\\mbfsanspartial" "𝞉" julia-latexsubs)
(puthash "\\mbfsansvarepsilon" "𝞊" julia-latexsubs)
(puthash "\\mbfsansvartheta" "𝞋" julia-latexsubs)
(puthash "\\mbfsansvarkappa" "𝞌" julia-latexsubs)
(puthash "\\mbfsansvarphi" "𝞍" julia-latexsubs)
(puthash "\\mbfsansvarrho" "𝞎" julia-latexsubs)
(puthash "\\mbfsansvarpi" "𝞏" julia-latexsubs)
(puthash "\\mbfitsansAlpha" "𝞐" julia-latexsubs)
(puthash "\\mbfitsansBeta" "𝞑" julia-latexsubs)
(puthash "\\mbfitsansGamma" "𝞒" julia-latexsubs)
(puthash "\\mbfitsansDelta" "𝞓" julia-latexsubs)
(puthash "\\mbfitsansEpsilon" "𝞔" julia-latexsubs)
(puthash "\\mbfitsansZeta" "𝞕" julia-latexsubs)
(puthash "\\mbfitsansEta" "𝞖" julia-latexsubs)
(puthash "\\mbfitsansTheta" "𝞗" julia-latexsubs)
(puthash "\\mbfitsansIota" "𝞘" julia-latexsubs)
(puthash "\\mbfitsansKappa" "𝞙" julia-latexsubs)
(puthash "\\mbfitsansLambda" "𝞚" julia-latexsubs)
(puthash "\\mbfitsansMu" "𝞛" julia-latexsubs)
(puthash "\\mbfitsansNu" "𝞜" julia-latexsubs)
(puthash "\\mbfitsansXi" "𝞝" julia-latexsubs)
(puthash "\\mbfitsansOmicron" "𝞞" julia-latexsubs)
(puthash "\\mbfitsansPi" "𝞟" julia-latexsubs)
(puthash "\\mbfitsansRho" "𝞠" julia-latexsubs)
(puthash "\\mbfitsansvarTheta" "𝞡" julia-latexsubs)
(puthash "\\mbfitsansSigma" "𝞢" julia-latexsubs)
(puthash "\\mbfitsansTau" "𝞣" julia-latexsubs)
(puthash "\\mbfitsansUpsilon" "𝞤" julia-latexsubs)
(puthash "\\mbfitsansPhi" "𝞥" julia-latexsubs)
(puthash "\\mbfitsansChi" "𝞦" julia-latexsubs)
(puthash "\\mbfitsansPsi" "𝞧" julia-latexsubs)
(puthash "\\mbfitsansOmega" "𝞨" julia-latexsubs)
(puthash "\\mbfitsansnabla" "𝞩" julia-latexsubs)
(puthash "\\mbfitsansalpha" "𝞪" julia-latexsubs)
(puthash "\\mbfitsansbeta" "𝞫" julia-latexsubs)
(puthash "\\mbfitsansgamma" "𝞬" julia-latexsubs)
(puthash "\\mbfitsansdelta" "𝞭" julia-latexsubs)
(puthash "\\mbfitsansepsilon" "𝞮" julia-latexsubs)
(puthash "\\mbfitsanszeta" "𝞯" julia-latexsubs)
(puthash "\\mbfitsanseta" "𝞰" julia-latexsubs)
(puthash "\\mbfitsanstheta" "𝞱" julia-latexsubs)
(puthash "\\mbfitsansiota" "𝞲" julia-latexsubs)
(puthash "\\mbfitsanskappa" "𝞳" julia-latexsubs)
(puthash "\\mbfitsanslambda" "𝞴" julia-latexsubs)
(puthash "\\mbfitsansmu" "𝞵" julia-latexsubs)
(puthash "\\mbfitsansnu" "𝞶" julia-latexsubs)
(puthash "\\mbfitsansxi" "𝞷" julia-latexsubs)
(puthash "\\mbfitsansomicron" "𝞸" julia-latexsubs)
(puthash "\\mbfitsanspi" "𝞹" julia-latexsubs)
(puthash "\\mbfitsansrho" "𝞺" julia-latexsubs)
(puthash "\\mbfitsansvarsigma" "𝞻" julia-latexsubs)
(puthash "\\mbfitsanssigma" "𝞼" julia-latexsubs)
(puthash "\\mbfitsanstau" "𝞽" julia-latexsubs)
(puthash "\\mbfitsansupsilon" "𝞾" julia-latexsubs)
(puthash "\\mbfitsansphi" "𝞿" julia-latexsubs)
(puthash "\\mbfitsanschi" "𝟀" julia-latexsubs)
(puthash "\\mbfitsanspsi" "𝟁" julia-latexsubs)
(puthash "\\mbfitsansomega" "𝟂" julia-latexsubs)
(puthash "\\mbfitsanspartial" "𝟃" julia-latexsubs)
(puthash "\\mbfitsansvarepsilon" "𝟄" julia-latexsubs)
(puthash "\\mbfitsansvartheta" "𝟅" julia-latexsubs)
(puthash "\\mbfitsansvarkappa" "𝟆" julia-latexsubs)
(puthash "\\mbfitsansvarphi" "𝟇" julia-latexsubs)
(puthash "\\mbfitsansvarrho" "𝟈" julia-latexsubs)
(puthash "\\mbfitsansvarpi" "𝟉" julia-latexsubs)
(let ((c (decode-char 'ucs #x1d7ca)))
  (if c (puthash "\\mbfDigamma" (char-to-string c) julia-latexsubs)))
(let ((c (decode-char 'ucs #x1d7cb)))
  (if c (puthash "\\mbfdigamma" (char-to-string c) julia-latexsubs)))
(puthash "\\mbfzero" "𝟎" julia-latexsubs)
(puthash "\\mbfone" "𝟏" julia-latexsubs)
(puthash "\\mbftwo" "𝟐" julia-latexsubs)
(puthash "\\mbfthree" "𝟑" julia-latexsubs)
(puthash "\\mbffour" "𝟒" julia-latexsubs)
(puthash "\\mbffive" "𝟓" julia-latexsubs)
(puthash "\\mbfsix" "𝟔" julia-latexsubs)
(puthash "\\mbfseven" "𝟕" julia-latexsubs)
(puthash "\\mbfeight" "𝟖" julia-latexsubs)
(puthash "\\mbfnine" "𝟗" julia-latexsubs)
(puthash "\\Bbbzero" "𝟘" julia-latexsubs)
(puthash "\\Bbbone" "𝟙" julia-latexsubs)
(puthash "\\Bbbtwo" "𝟚" julia-latexsubs)
(puthash "\\Bbbthree" "𝟛" julia-latexsubs)
(puthash "\\Bbbfour" "𝟜" julia-latexsubs)
(puthash "\\Bbbfive" "𝟝" julia-latexsubs)
(puthash "\\Bbbsix" "𝟞" julia-latexsubs)
(puthash "\\Bbbseven" "𝟟" julia-latexsubs)
(puthash "\\Bbbeight" "𝟠" julia-latexsubs)
(puthash "\\Bbbnine" "𝟡" julia-latexsubs)
(puthash "\\msanszero" "𝟢" julia-latexsubs)
(puthash "\\msansone" "𝟣" julia-latexsubs)
(puthash "\\msanstwo" "𝟤" julia-latexsubs)
(puthash "\\msansthree" "𝟥" julia-latexsubs)
(puthash "\\msansfour" "𝟦" julia-latexsubs)
(puthash "\\msansfive" "𝟧" julia-latexsubs)
(puthash "\\msanssix" "𝟨" julia-latexsubs)
(puthash "\\msansseven" "𝟩" julia-latexsubs)
(puthash "\\msanseight" "𝟪" julia-latexsubs)
(puthash "\\msansnine" "𝟫" julia-latexsubs)
(puthash "\\mbfsanszero" "𝟬" julia-latexsubs)
(puthash "\\mbfsansone" "𝟭" julia-latexsubs)
(puthash "\\mbfsanstwo" "𝟮" julia-latexsubs)
(puthash "\\mbfsansthree" "𝟯" julia-latexsubs)
(puthash "\\mbfsansfour" "𝟰" julia-latexsubs)
(puthash "\\mbfsansfive" "𝟱" julia-latexsubs)
(puthash "\\mbfsanssix" "𝟲" julia-latexsubs)
(puthash "\\mbfsansseven" "𝟳" julia-latexsubs)
(puthash "\\mbfsanseight" "𝟴" julia-latexsubs)
(puthash "\\mbfsansnine" "𝟵" julia-latexsubs)
(puthash "\\mttzero" "𝟶" julia-latexsubs)
(puthash "\\mttone" "𝟷" julia-latexsubs)
(puthash "\\mtttwo" "𝟸" julia-latexsubs)
(puthash "\\mttthree" "𝟹" julia-latexsubs)
(puthash "\\mttfour" "𝟺" julia-latexsubs)
(puthash "\\mttfive" "𝟻" julia-latexsubs)
(puthash "\\mttsix" "𝟼" julia-latexsubs)
(puthash "\\mttseven" "𝟽" julia-latexsubs)
(puthash "\\mtteight" "𝟾" julia-latexsubs)
(puthash "\\mttnine" "𝟿" julia-latexsubs)

;; Code for `inferior-julia-mode'
(require 'comint)

(defcustom julia-program "julia"
  "Path to the program used by `inferior-julia'."
  :type 'string
  :group 'julia)

(defcustom julia-arguments '()
  "Commandline arguments to pass to `julia-program'."
  :type 'string
  :group 'julia)

(defvar julia-prompt-regexp "julia>"
  "Regexp for matching `inferior-julia' prompt.")

(defvar inferior-julia-mode-map
  (let ((map (nconc (make-sparse-keymap) comint-mode-map)))
    ;; example definition
    (define-key map (kbd "TAB") 'julia-latexsub-or-indent)
    map)
  "Basic mode map for `inferior-julia-mode'.")

;;;###autoload
(defun inferior-julia ()
    "Run an inferior instance of `julia' inside Emacs."
    (interactive)
    (let ((julia-program julia-program)
          (buffer (get-buffer-create "*Julia*")))
      (when (not (comint-check-proc "*Julia*"))
            (apply #'make-comint-in-buffer "Julia" "*Julia*" julia-program julia-arguments))
      (pop-to-buffer-same-window "*Julia*")
      (inferior-julia-mode)))

(defun inferior-julia--initialize ()
    "Helper function to initialize `inferior-julia'."
    (setq comint-use-prompt-regexp t))

(define-derived-mode inferior-julia-mode comint-mode "Julia"
  "Major mode for `inferior-julia'.

\\<inferior-julia-mode-map>"
  nil "Julia"
  (setq comint-prompt-regexp julia-prompt-regexp)
  (setq comint-prompt-read-only t)
  (set (make-local-variable 'font-lock-defaults) '(julia-font-lock-keywords t))
  (set (make-local-variable 'paragraph-start) julia-prompt-regexp)
  (set (make-local-variable 'indent-line-function) 'julia-indent-line))

(add-hook 'inferior-julia-mode-hook 'inferior-julia--initialize)

(provide 'julia-mode)

;; Local Variables:
;; coding: utf-8
;; byte-compile-warnings: (not obsolete)
;; End:
;;; julia-mode.el ends here
