/// Hard-coded Huffman codes used regardless of the input.
///
/// These values work well for PNGs with some form of filtering enabled, but will likely make most
/// other inputs worse.
pub(crate) const HUFFMAN_LENGTHS: [u8; 286] = [
    2, 3, 4, 5, 5, 6, 6, 7, 7, 7, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 10, 11, 10, 10, 10, 10, 10, 10, 10, 10, 10, 9, 9, 9, 9, 9, 8, 9, 8, 8, 8, 8, 8, 7,
    7, 7, 6, 6, 6, 5, 4, 3, 12, 12, 12, 9, 9, 11, 10, 11, 11, 10, 11, 11, 11, 11, 11, 11, 12, 11,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 9,
];

pub(crate) const HUFFMAN_CODES: [u16; 286] = match crate::compute_codes(&HUFFMAN_LENGTHS) {
    Some(codes) => codes,
    None => panic!("HUFFMAN_LENGTHS is invalid"),
};

/// Length code for length values (derived from deflate spec).
pub(crate) const LENGTH_TO_SYMBOL: [u16; 256] = [
    257, 258, 259, 260, 261, 262, 263, 264, 265, 265, 266, 266, 267, 267, 268, 268, 269, 269, 269,
    269, 270, 270, 270, 270, 271, 271, 271, 271, 272, 272, 272, 272, 273, 273, 273, 273, 273, 273,
    273, 273, 274, 274, 274, 274, 274, 274, 274, 274, 275, 275, 275, 275, 275, 275, 275, 275, 276,
    276, 276, 276, 276, 276, 276, 276, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277,
    277, 277, 277, 277, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278,
    278, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 280, 280,
    280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 281, 281, 281, 281, 281,
    281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281,
    281, 281, 281, 281, 281, 281, 281, 281, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282,
    282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282,
    282, 282, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283,
    283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283, 284, 284, 284, 284,
    284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284,
    284, 284, 284, 284, 284, 284, 284, 284, 285,
];

/// Number of extra bits for length values (derived from deflate spec).
pub(crate) const LENGTH_TO_LEN_EXTRA: [u8; 256] = [
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 0,
];

pub(crate) const BITMASKS: [u32; 17] = [
    0x0000, 0x0001, 0x0003, 0x0007, 0x000F, 0x001F, 0x003F, 0x007F, 0x00FF, 0x01FF, 0x03FF, 0x07FF,
    0x0FFF, 0x1FFF, 0x3FFF, 0x7FFF, 0xFFFF,
];

/// Order of the length code length alphabet (derived from deflate spec).
pub(crate) const CLCL_ORDER: [usize; 19] = [
    16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15,
];

/// Number of extra bits for each length code (derived from deflate spec).
pub(crate) const LEN_SYM_TO_LEN_EXTRA: [u8; 29] = [
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 5, 5, 5, 5, 0,
];

/// The base length for each length code (derived from deflate spec).
pub(crate) const LEN_SYM_TO_LEN_BASE: [usize; 29] = [
    3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31, 35, 43, 51, 59, 67, 83, 99, 115, 131,
    163, 195, 227, 258,
];

/// Number of extra bits for each distance code (derived from deflate spec.)
pub(crate) const DIST_SYM_TO_DIST_EXTRA: [u8; 30] = [
    0, 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13,
    13,
];

/// The base distance for each distance code (derived from deflate spec).
pub(crate) const DIST_SYM_TO_DIST_BASE: [u16; 30] = [
    1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193, 257, 385, 513, 769, 1025, 1537,
    2049, 3073, 4097, 6145, 8193, 12289, 16385, 24577,
];

pub(crate) const FDEFLATE_LITLEN_DECODE_TABLE: [u32; 4096] = [
    0x8204, 0x28206, 0x18205, 0xfa8208, 0x2008206, 0x38207, 0xff8205, 0xf4820a, 0x1008205,
    0xfe8206, 0x2018207, 0x98209, 0xfa008208, 0xfd8207, 0x2ff8207, 0xb010a, 0x8204, 0x2028208,
    0x1018206, 0xfc8208, 0x3008207, 0x48207, 0x1ff8206, 0xee820b, 0xff008205, 0x2fe8208,
    0xfa018209, 0xa820a, 0xf400820a, 0x58208, 0xfaff8209, 0x65810c, 0x8204, 0x1028207, 0x18205,
    0xfb8208, 0xfe008206, 0x2038209, 0xff8205, 0x11820b, 0x1008205, 0x1fe8207, 0x3018208, 0xf88209,
    0x9008209, 0x2fd8209, 0x3ff8208, 0x17020b, 0x8204, 0xfa02820a, 0xff018206, 0x78209, 0xfd008207,
    0x2048209, 0xffff8206, 0x1a820c, 0xff008205, 0xfafe820a, 0xf401820b, 0xe820a, 0x8102, 0x68208,
    0xf4ff820b, 0xa5810c, 0x8204, 0x28206, 0x18205, 0x2fa820a, 0x2008206, 0x1038208, 0xff8205,
    0xf6820a, 0x1008205, 0xfe8206, 0xfe018207, 0xf78209, 0xfc008208, 0x1fd8208, 0xfeff8207,
    0xd8810b, 0x8204, 0x3028209, 0x1018206, 0x2fc820a, 0x4008207, 0x1048208, 0x1ff8206, 0x60009,
    0xff008205, 0x3fe8209, 0x901820a, 0xc820a, 0xee00820b, 0x205820a, 0x9ff820a, 0x85810c, 0x8204,
    0xff028207, 0x18205, 0x2fb820a, 0xfe008206, 0xfa03820b, 0xff8205, 0x15820b, 0x1008205,
    0xfffe8207, 0xfd018208, 0xf98209, 0xa00820a, 0xfafd820b, 0xfdff8208, 0x45810c, 0x8204,
    0xf402820c, 0xff018206, 0x88209, 0x5008208, 0xfa04820b, 0xffff8206, 0xe4820c, 0xff008205,
    0xf4fe820c, 0x18103, 0xf2820a, 0x8102, 0x206820a, 0xff8103, 0xc5810c, 0x8204, 0x28206, 0x18205,
    0x1fa8209, 0x2008206, 0x38207, 0xff8205, 0xf5820a, 0x1008205, 0xfe8206, 0x2018207, 0x209820b,
    0xfb008208, 0xfd8207, 0x2ff8207, 0x25810b, 0x8204, 0xfe028208, 0x1018206, 0x1fc8209, 0x3008207,
    0x48207, 0x1ff8206, 0xf1820b, 0xff008205, 0xfefe8208, 0xfc018209, 0xb820a, 0x1100820b,
    0x1058209, 0xfcff8209, 0x75810c, 0x8204, 0x1028207, 0x18205, 0x1fb8209, 0xfe008206, 0x303820a,
    0xff8205, 0x13820b, 0x1008205, 0x1fe8207, 0x4018208, 0x2f8820b, 0xf8008209, 0x3fd820a,
    0x4ff8208, 0x35810c, 0x8204, 0x902820b, 0xff018206, 0x207820b, 0xfd008207, 0x304820a,
    0xffff8206, 0x1e820c, 0xff008205, 0x9fe820b, 0xee01820c, 0xf0820a, 0x8102, 0x1068209,
    0xeeff820c, 0xb5810c, 0x8204, 0x28206, 0x18205, 0xfafa820c, 0x2008206, 0xff038208, 0xff8205,
    0xf820b, 0x1008205, 0xfe8206, 0xfe018207, 0x2f7820b, 0x7008209, 0xfffd8208, 0xfeff8207,
    0xe1810b, 0x8204, 0xfd028209, 0x1018206, 0xfafc820c, 0x4008207, 0xff048208, 0x1ff8206,
    0x16820c, 0xff008205, 0xfdfe8209, 0xa01820b, 0xd820a, 0x1a00820c, 0xfa05820c, 0xaff820b,
    0x95810c, 0x8204, 0xff028207, 0x18205, 0xfafb820c, 0xfe008206, 0x38105, 0xff8205, 0xec820b,
    0x1008205, 0xfffe8207, 0x5018209, 0x2f9820b, 0xe00820a, 0xfd8105, 0x5ff8209, 0x55810c, 0x8204,
    0x28104, 0xff018206, 0x208820b, 0x6008208, 0x48105, 0xffff8206, 0xe8820c, 0xff008205, 0xfe8104,
    0x18103, 0xf3820a, 0x8102, 0xfa06820c, 0xff8103, 0xd5810c, 0x8204, 0x28206, 0x18205, 0xfa8208,
    0x2008206, 0x38207, 0xff8205, 0x2f4820c, 0x1008205, 0xfe8206, 0x2018207, 0x109820a, 0xfa008208,
    0xfd8207, 0x2ff8207, 0x21810b, 0x8204, 0x2028208, 0x1018206, 0xfc8208, 0x3008207, 0x48207,
    0x1ff8206, 0xef820b, 0xff008205, 0x2fe8208, 0xfb018209, 0x20a820c, 0xf600820a, 0x58208,
    0xfbff8209, 0x6d810c, 0x8204, 0x1028207, 0x18205, 0xfb8208, 0xfe008206, 0xfe038209, 0xff8205,
    0x12820b, 0x1008205, 0x1fe8207, 0x3018208, 0x1f8820a, 0xf7008209, 0xfefd8209, 0x3ff8208,
    0x2d810c, 0x8204, 0xfc02820a, 0xff018206, 0x107820a, 0xfd008207, 0xfe048209, 0xffff8206,
    0x1c820c, 0xff008205, 0xfcfe820a, 0x1101820c, 0x20e820c, 0x8102, 0x68208, 0x11ff820c, 0xad810c,
    0x8204, 0x28206, 0x18205, 0x3fa820b, 0x2008206, 0x1038208, 0xff8205, 0x2f6820c, 0x1008205,
    0xfe8206, 0xfe018207, 0x1f7820a, 0xfc008208, 0x1fd8208, 0xfeff8207, 0xdc810b, 0x8204,
    0x4028209, 0x1018206, 0x3fc820b, 0x4008207, 0x1048208, 0x1ff8206, 0x1020009, 0xff008205,
    0x4fe8209, 0xf801820a, 0x20c820c, 0x8102, 0x305820b, 0xf8ff820a, 0x8d810c, 0x8204, 0xff028207,
    0x18205, 0x3fb820b, 0xfe008206, 0x903820c, 0xff8205, 0xeb820b, 0x1008205, 0xfffe8207,
    0xfd018208, 0x1f9820a, 0xc00820a, 0x9fd820c, 0xfdff8208, 0x4d810c, 0x8204, 0x28104, 0xff018206,
    0x108820a, 0x5008208, 0x904820c, 0xffff8206, 0xe6820c, 0xff008205, 0xfe8104, 0x18103,
    0x2f2820c, 0x8102, 0x306820b, 0xff8103, 0xcd810c, 0x8204, 0x28206, 0x18205, 0xfffa8209,
    0x2008206, 0x38207, 0xff8205, 0x2f5820c, 0x1008205, 0xfe8206, 0x2018207, 0x98107, 0xfb008208,
    0xfd8207, 0x2ff8207, 0x29810b, 0x8204, 0xfe028208, 0x1018206, 0xfffc8209, 0x3008207, 0x48207,
    0x1ff8206, 0x50009, 0xff008205, 0xfefe8208, 0x701820a, 0x20b820c, 0x1500820b, 0xff058209,
    0x7ff820a, 0x7d810c, 0x8204, 0x1028207, 0x18205, 0xfffb8209, 0xfe008206, 0xfd03820a, 0xff8205,
    0x14820b, 0x1008205, 0x1fe8207, 0x4018208, 0xf88107, 0xf9008209, 0xfdfd820a, 0x4ff8208,
    0x3d810c, 0x8204, 0xa02820c, 0xff018206, 0x78107, 0xfd008207, 0xfd04820a, 0xffff8206, 0xe2820c,
    0xff008205, 0xafe820c, 0x18103, 0x2f0820c, 0x8102, 0xff068209, 0xff8103, 0xbd810c, 0x8204,
    0x28206, 0x18205, 0xfa8106, 0x2008206, 0xff038208, 0xff8205, 0x10820b, 0x1008205, 0xfe8206,
    0xfe018207, 0xf78107, 0x8008209, 0xfffd8208, 0xfeff8207, 0xd010b, 0x8204, 0x502820a, 0x1018206,
    0xfc8106, 0x4008207, 0xff048208, 0x1ff8206, 0x18820c, 0xff008205, 0x5fe820a, 0xe01820b,
    0x20d820c, 0xe400820c, 0x58106, 0xeff820b, 0x9d810c, 0x8204, 0xff028207, 0x18205, 0xfb8106,
    0xfe008206, 0x38105, 0xff8205, 0xed820b, 0x1008205, 0xfffe8207, 0x6018209, 0xf98107,
    0xf200820a, 0xfd8105, 0x6ff8209, 0x5d810c, 0x8204, 0x28104, 0xff018206, 0x88107, 0x6008208,
    0x48105, 0xffff8206, 0xea820c, 0xff008205, 0xfe8104, 0x18103, 0x2f3820c, 0x8102, 0x68106,
    0xff8103, 0x43040c, 0x8204, 0x28206, 0x18205, 0xfa8208, 0x2008206, 0x38207, 0xff8205,
    0x1f4820b, 0x1008205, 0xfe8206, 0x2018207, 0x98209, 0xfa008208, 0xfd8207, 0x2ff8207, 0x1f810b,
    0x8204, 0x2028208, 0x1018206, 0xfc8208, 0x3008207, 0x48207, 0x1ff8206, 0xee8109, 0xff008205,
    0x2fe8208, 0xfa018209, 0x10a820b, 0xf500820a, 0x58208, 0xfaff8209, 0x69810c, 0x8204, 0x1028207,
    0x18205, 0xfb8208, 0xfe008206, 0x2038209, 0xff8205, 0x118109, 0x1008205, 0x1fe8207, 0x3018208,
    0xf88209, 0x9008209, 0x2fd8209, 0x3ff8208, 0x23030b, 0x8204, 0xfb02820a, 0xff018206, 0x78209,
    0xfd008207, 0x2048209, 0xffff8206, 0x1b820c, 0xff008205, 0xfbfe820a, 0xf601820b, 0x10e820b,
    0x8102, 0x68208, 0xf6ff820b, 0xa9810c, 0x8204, 0x28206, 0x18205, 0xfefa820a, 0x2008206,
    0x1038208, 0xff8205, 0x1f6820b, 0x1008205, 0xfe8206, 0xfe018207, 0xf78209, 0xfc008208,
    0x1fd8208, 0xfeff8207, 0xda810b, 0x8204, 0x3028209, 0x1018206, 0xfefc820a, 0x4008207,
    0x1048208, 0x1ff8206, 0x60009, 0xff008205, 0x3fe8209, 0xf701820a, 0x10c820b, 0xf100820b,
    0xfe05820a, 0xf7ff820a, 0x89810c, 0x8204, 0xff028207, 0x18205, 0xfefb820a, 0xfe008206,
    0xfc03820b, 0xff8205, 0x158109, 0x1008205, 0xfffe8207, 0xfd018208, 0xf98209, 0xb00820a,
    0xfcfd820b, 0xfdff8208, 0x49810c, 0x8204, 0x28104, 0xff018206, 0x88209, 0x5008208, 0xfc04820b,
    0xffff8206, 0xe5820c, 0xff008205, 0xfe8104, 0x18103, 0x1f2820b, 0x8102, 0xfe06820a, 0xff8103,
    0xc9810c, 0x8204, 0x28206, 0x18205, 0x1fa8209, 0x2008206, 0x38207, 0xff8205, 0x1f5820b,
    0x1008205, 0xfe8206, 0x2018207, 0x309820c, 0xfb008208, 0xfd8207, 0x2ff8207, 0x27810b, 0x8204,
    0xfe028208, 0x1018206, 0x1fc8209, 0x3008207, 0x48207, 0x1ff8206, 0xf18109, 0xff008205,
    0xfefe8208, 0xfc018209, 0x10b820b, 0x1300820b, 0x1058209, 0xfcff8209, 0x79810c, 0x8204,
    0x1028207, 0x18205, 0x1fb8209, 0xfe008206, 0x403820a, 0xff8205, 0x138109, 0x1008205, 0x1fe8207,
    0x4018208, 0x3f8820c, 0xf8008209, 0x4fd820a, 0x4ff8208, 0x39810c, 0x8204, 0xf802820b,
    0xff018206, 0x307820c, 0xfd008207, 0x404820a, 0xffff8206, 0xe0820c, 0xff008205, 0xf8fe820b,
    0x18103, 0x1f0820b, 0x8102, 0x1068209, 0xff8103, 0xb9810c, 0x8204, 0x28206, 0x18205, 0xfa8106,
    0x2008206, 0xff038208, 0xff8205, 0xf8109, 0x1008205, 0xfe8206, 0xfe018207, 0x3f7820c,
    0x7008209, 0xfffd8208, 0xfeff8207, 0x9000b, 0x8204, 0xfd028209, 0x1018206, 0xfc8106, 0x4008207,
    0xff048208, 0x1ff8206, 0x17820c, 0xff008205, 0xfdfe8209, 0xc01820b, 0x10d820b, 0x1e00820c,
    0x58106, 0xcff820b, 0x99810c, 0x8204, 0xff028207, 0x18205, 0xfb8106, 0xfe008206, 0x38105,
    0xff8205, 0xec8109, 0x1008205, 0xfffe8207, 0x5018209, 0x3f9820c, 0xf000820a, 0xfd8105,
    0x5ff8209, 0x59810c, 0x8204, 0x28104, 0xff018206, 0x308820c, 0x6008208, 0x48105, 0xffff8206,
    0xe9820c, 0xff008205, 0xfe8104, 0x18103, 0x1f3820b, 0x8102, 0x68106, 0xff8103, 0x1f020c,
    0x8204, 0x28206, 0x18205, 0xfa8208, 0x2008206, 0x38207, 0xff8205, 0xf48108, 0x1008205,
    0xfe8206, 0x2018207, 0xff09820a, 0xfa008208, 0xfd8207, 0x2ff8207, 0x23810b, 0x8204, 0x2028208,
    0x1018206, 0xfc8208, 0x3008207, 0x48207, 0x1ff8206, 0xef8109, 0xff008205, 0x2fe8208,
    0xfb018209, 0xa8108, 0xf00820b, 0x58208, 0xfbff8209, 0x71810c, 0x8204, 0x1028207, 0x18205,
    0xfb8208, 0xfe008206, 0xfe038209, 0xff8205, 0x128109, 0x1008205, 0x1fe8207, 0x3018208,
    0xfff8820a, 0xf7008209, 0xfefd8209, 0x3ff8208, 0x31810c, 0x8204, 0x702820b, 0xff018206,
    0xff07820a, 0xfd008207, 0xfe048209, 0xffff8206, 0x1d820c, 0xff008205, 0x7fe820b, 0x1501820c,
    0xe8108, 0x8102, 0x68208, 0x15ff820c, 0xb1810c, 0x8204, 0x28206, 0x18205, 0xfdfa820b,
    0x2008206, 0x1038208, 0xff8205, 0xf68108, 0x1008205, 0xfe8206, 0xfe018207, 0xfff7820a,
    0xfc008208, 0x1fd8208, 0xfeff8207, 0xde810b, 0x8204, 0x4028209, 0x1018206, 0xfdfc820b,
    0x4008207, 0x1048208, 0x1ff8206, 0x1020009, 0xff008205, 0x4fe8209, 0xf901820a, 0xc8108,
    0x1600820c, 0xfd05820b, 0xf9ff820a, 0x91810c, 0x8204, 0xff028207, 0x18205, 0xfdfb820b,
    0xfe008206, 0x38105, 0xff8205, 0xeb8109, 0x1008205, 0xfffe8207, 0xfd018208, 0xfff9820a,
    0xd00820a, 0xfd8105, 0xfdff8208, 0x51810c, 0x8204, 0x28104, 0xff018206, 0xff08820a, 0x5008208,
    0x48105, 0xffff8206, 0xe7820c, 0xff008205, 0xfe8104, 0x18103, 0xf28108, 0x8102, 0xfd06820b,
    0xff8103, 0xd1810c, 0x8204, 0x28206, 0x18205, 0xfffa8209, 0x2008206, 0x38207, 0xff8205,
    0xf58108, 0x1008205, 0xfe8206, 0x2018207, 0x98107, 0xfb008208, 0xfd8207, 0x2ff8207, 0xd6810b,
    0x8204, 0xfe028208, 0x1018206, 0xfffc8209, 0x3008207, 0x48207, 0x1ff8206, 0x50009, 0xff008205,
    0xfefe8208, 0x801820a, 0xb8108, 0xec00820b, 0xff058209, 0x8ff820a, 0x81810c, 0x8204, 0x1028207,
    0x18205, 0xfffb8209, 0xfe008206, 0x503820b, 0xff8205, 0x148109, 0x1008205, 0x1fe8207,
    0x4018208, 0xf88107, 0xf9008209, 0x5fd820b, 0x4ff8208, 0x41810c, 0x8204, 0xe02820c, 0xff018206,
    0x78107, 0xfd008207, 0x504820b, 0xffff8206, 0xe3820c, 0xff008205, 0xefe820c, 0x18103, 0xf08108,
    0x8102, 0xff068209, 0xff8103, 0xc1810c, 0x8204, 0x28206, 0x18205, 0xfa8106, 0x2008206,
    0xff038208, 0xff8205, 0x108109, 0x1008205, 0xfe8206, 0xfe018207, 0xf78107, 0x8008209,
    0xfffd8208, 0xfeff8207, 0x11010b, 0x8204, 0x602820a, 0x1018206, 0xfc8106, 0x4008207,
    0xff048208, 0x1ff8206, 0x19820c, 0xff008205, 0x6fe820a, 0xf201820b, 0xd8108, 0xe800820c,
    0x58106, 0xf2ff820b, 0xa1810c, 0x8204, 0xff028207, 0x18205, 0xfb8106, 0xfe008206, 0x38105,
    0xff8205, 0xed8109, 0x1008205, 0xfffe8207, 0x6018209, 0xf98107, 0xf300820a, 0xfd8105,
    0x6ff8209, 0x61810c, 0x8204, 0x28104, 0xff018206, 0x88107, 0x6008208, 0x48105, 0xffff8206,
    0x8000a, 0xff008205, 0xfe8104, 0x18103, 0xf38108, 0x8102, 0x68106, 0xff8103, 0x83050c, 0x8204,
    0x28206, 0x18205, 0xfa8208, 0x2008206, 0x38207, 0xff8205, 0xf4820a, 0x1008205, 0xfe8206,
    0x2018207, 0x98209, 0xfa008208, 0xfd8207, 0x2ff8207, 0xb010a, 0x8204, 0x2028208, 0x1018206,
    0xfc8208, 0x3008207, 0x48207, 0x1ff8206, 0x1ee820c, 0xff008205, 0x2fe8208, 0xfa018209, 0xa820a,
    0xf400820a, 0x58208, 0xfaff8209, 0x67810c, 0x8204, 0x1028207, 0x18205, 0xfb8208, 0xfe008206,
    0x2038209, 0xff8205, 0x111820c, 0x1008205, 0x1fe8207, 0x3018208, 0xf88209, 0x9008209,
    0x2fd8209, 0x3ff8208, 0x1b020b, 0x8204, 0xfa02820a, 0xff018206, 0x78209, 0xfd008207, 0x2048209,
    0xffff8206, 0x1a810a, 0xff008205, 0xfafe820a, 0xf501820b, 0xe820a, 0x8102, 0x68208, 0xf5ff820b,
    0xa7810c, 0x8204, 0x28206, 0x18205, 0x2fa820a, 0x2008206, 0x1038208, 0xff8205, 0xf6820a,
    0x1008205, 0xfe8206, 0xfe018207, 0xf78209, 0xfc008208, 0x1fd8208, 0xfeff8207, 0xd9810b, 0x8204,
    0x3028209, 0x1018206, 0x2fc820a, 0x4008207, 0x1048208, 0x1ff8206, 0x60009, 0xff008205,
    0x3fe8209, 0x901820a, 0xc820a, 0xef00820b, 0x205820a, 0x9ff820a, 0x87810c, 0x8204, 0xff028207,
    0x18205, 0x2fb820a, 0xfe008206, 0xfb03820b, 0xff8205, 0x115820c, 0x1008205, 0xfffe8207,
    0xfd018208, 0xf98209, 0xa00820a, 0xfbfd820b, 0xfdff8208, 0x47810c, 0x8204, 0xf602820c,
    0xff018206, 0x88209, 0x5008208, 0xfb04820b, 0xffff8206, 0xe4810a, 0xff008205, 0xf6fe820c,
    0x18103, 0xf2820a, 0x8102, 0x206820a, 0xff8103, 0xc7810c, 0x8204, 0x28206, 0x18205, 0x1fa8209,
    0x2008206, 0x38207, 0xff8205, 0xf5820a, 0x1008205, 0xfe8206, 0x2018207, 0xfe09820b, 0xfb008208,
    0xfd8207, 0x2ff8207, 0x26810b, 0x8204, 0xfe028208, 0x1018206, 0x1fc8209, 0x3008207, 0x48207,
    0x1ff8206, 0x1f1820c, 0xff008205, 0xfefe8208, 0xfc018209, 0xb820a, 0x1200820b, 0x1058209,
    0xfcff8209, 0x77810c, 0x8204, 0x1028207, 0x18205, 0x1fb8209, 0xfe008206, 0x303820a, 0xff8205,
    0x113820c, 0x1008205, 0x1fe8207, 0x4018208, 0xfef8820b, 0xf8008209, 0x3fd820a, 0x4ff8208,
    0x37810c, 0x8204, 0xf702820b, 0xff018206, 0xfe07820b, 0xfd008207, 0x304820a, 0xffff8206,
    0x1e810a, 0xff008205, 0xf7fe820b, 0xf101820c, 0xf0820a, 0x8102, 0x1068209, 0xf1ff820c,
    0xb7810c, 0x8204, 0x28206, 0x18205, 0xfcfa820c, 0x2008206, 0xff038208, 0xff8205, 0x10f820c,
    0x1008205, 0xfe8206, 0xfe018207, 0xfef7820b, 0x7008209, 0xfffd8208, 0xfeff8207, 0x7000b,
    0x8204, 0xfd028209, 0x1018206, 0xfcfc820c, 0x4008207, 0xff048208, 0x1ff8206, 0x16810a,
    0xff008205, 0xfdfe8209, 0xb01820b, 0xd820a, 0x1c00820c, 0xfc05820c, 0xbff820b, 0x97810c,
    0x8204, 0xff028207, 0x18205, 0xfcfb820c, 0xfe008206, 0x38105, 0xff8205, 0x1ec820c, 0x1008205,
    0xfffe8207, 0x5018209, 0xfef9820b, 0xe00820a, 0xfd8105, 0x5ff8209, 0x57810c, 0x8204, 0x28104,
    0xff018206, 0xfe08820b, 0x6008208, 0x48105, 0xffff8206, 0xe8810a, 0xff008205, 0xfe8104,
    0x18103, 0xf3820a, 0x8102, 0xfc06820c, 0xff8103, 0x3000c, 0x8204, 0x28206, 0x18205, 0xfa8208,
    0x2008206, 0x38207, 0xff8205, 0xf48108, 0x1008205, 0xfe8206, 0x2018207, 0x109820a, 0xfa008208,
    0xfd8207, 0x2ff8207, 0x22810b, 0x8204, 0x2028208, 0x1018206, 0xfc8208, 0x3008207, 0x48207,
    0x1ff8206, 0x1ef820c, 0xff008205, 0x2fe8208, 0xfb018209, 0xa8108, 0xf600820a, 0x58208,
    0xfbff8209, 0x6f810c, 0x8204, 0x1028207, 0x18205, 0xfb8208, 0xfe008206, 0xfe038209, 0xff8205,
    0x112820c, 0x1008205, 0x1fe8207, 0x3018208, 0x1f8820a, 0xf7008209, 0xfefd8209, 0x3ff8208,
    0x2f810c, 0x8204, 0xfc02820a, 0xff018206, 0x107820a, 0xfd008207, 0xfe048209, 0xffff8206,
    0x1c810a, 0xff008205, 0xfcfe820a, 0x1301820c, 0xe8108, 0x8102, 0x68208, 0x13ff820c, 0xaf810c,
    0x8204, 0x28206, 0x18205, 0x4fa820b, 0x2008206, 0x1038208, 0xff8205, 0xf68108, 0x1008205,
    0xfe8206, 0xfe018207, 0x1f7820a, 0xfc008208, 0x1fd8208, 0xfeff8207, 0xdd810b, 0x8204,
    0x4028209, 0x1018206, 0x4fc820b, 0x4008207, 0x1048208, 0x1ff8206, 0x1020009, 0xff008205,
    0x4fe8209, 0xf801820a, 0xc8108, 0x8102, 0x405820b, 0xf8ff820a, 0x8f810c, 0x8204, 0xff028207,
    0x18205, 0x4fb820b, 0xfe008206, 0xf803820c, 0xff8205, 0x1eb820c, 0x1008205, 0xfffe8207,
    0xfd018208, 0x1f9820a, 0xc00820a, 0xf8fd820c, 0xfdff8208, 0x4f810c, 0x8204, 0x28104,
    0xff018206, 0x108820a, 0x5008208, 0xf804820c, 0xffff8206, 0xe6810a, 0xff008205, 0xfe8104,
    0x18103, 0xf28108, 0x8102, 0x406820b, 0xff8103, 0xcf810c, 0x8204, 0x28206, 0x18205, 0xfffa8209,
    0x2008206, 0x38207, 0xff8205, 0xf58108, 0x1008205, 0xfe8206, 0x2018207, 0x98107, 0xfb008208,
    0xfd8207, 0x2ff8207, 0x2a810b, 0x8204, 0xfe028208, 0x1018206, 0xfffc8209, 0x3008207, 0x48207,
    0x1ff8206, 0x50009, 0xff008205, 0xfefe8208, 0x701820a, 0xb8108, 0xeb00820b, 0xff058209,
    0x7ff820a, 0x7f810c, 0x8204, 0x1028207, 0x18205, 0xfffb8209, 0xfe008206, 0xfd03820a, 0xff8205,
    0x114820c, 0x1008205, 0x1fe8207, 0x4018208, 0xf88107, 0xf9008209, 0xfdfd820a, 0x4ff8208,
    0x3f810c, 0x8204, 0xc02820c, 0xff018206, 0x78107, 0xfd008207, 0xfd04820a, 0xffff8206, 0xe2810a,
    0xff008205, 0xcfe820c, 0x18103, 0xf08108, 0x8102, 0xff068209, 0xff8103, 0xbf810c, 0x8204,
    0x28206, 0x18205, 0xfa8106, 0x2008206, 0xff038208, 0xff8205, 0x110820c, 0x1008205, 0xfe8206,
    0xfe018207, 0xf78107, 0x8008209, 0xfffd8208, 0xfeff8207, 0xf010b, 0x8204, 0x502820a, 0x1018206,
    0xfc8106, 0x4008207, 0xff048208, 0x1ff8206, 0x18810a, 0xff008205, 0x5fe820a, 0xf001820b,
    0xd8108, 0xe600820c, 0x58106, 0xf0ff820b, 0x9f810c, 0x8204, 0xff028207, 0x18205, 0xfb8106,
    0xfe008206, 0x38105, 0xff8205, 0x1ed820c, 0x1008205, 0xfffe8207, 0x6018209, 0xf98107,
    0xf200820a, 0xfd8105, 0x6ff8209, 0x5f810c, 0x8204, 0x28104, 0xff018206, 0x88107, 0x6008208,
    0x48105, 0xffff8206, 0xea810a, 0xff008205, 0xfe8104, 0x18103, 0xf38108, 0x8102, 0x68106,
    0xff8103, 0x63040c, 0x8204, 0x28206, 0x18205, 0xfa8208, 0x2008206, 0x38207, 0xff8205,
    0xfff4820b, 0x1008205, 0xfe8206, 0x2018207, 0x98209, 0xfa008208, 0xfd8207, 0x2ff8207, 0x20810b,
    0x8204, 0x2028208, 0x1018206, 0xfc8208, 0x3008207, 0x48207, 0x1ff8206, 0xee8109, 0xff008205,
    0x2fe8208, 0xfa018209, 0xff0a820b, 0xf500820a, 0x58208, 0xfaff8209, 0x6b810c, 0x8204,
    0x1028207, 0x18205, 0xfb8208, 0xfe008206, 0x2038209, 0xff8205, 0x118109, 0x1008205, 0x1fe8207,
    0x3018208, 0xf88209, 0x9008209, 0x2fd8209, 0x3ff8208, 0x2b810c, 0x8204, 0xfb02820a, 0xff018206,
    0x78209, 0xfd008207, 0x2048209, 0xffff8206, 0x1b810a, 0xff008205, 0xfbfe820a, 0xf01820c,
    0xff0e820b, 0x8102, 0x68208, 0xfff820c, 0xab810c, 0x8204, 0x28206, 0x18205, 0xfefa820a,
    0x2008206, 0x1038208, 0xff8205, 0xfff6820b, 0x1008205, 0xfe8206, 0xfe018207, 0xf78209,
    0xfc008208, 0x1fd8208, 0xfeff8207, 0xdb810b, 0x8204, 0x3028209, 0x1018206, 0xfefc820a,
    0x4008207, 0x1048208, 0x1ff8206, 0x60009, 0xff008205, 0x3fe8209, 0xf701820a, 0xff0c820b,
    0x8102, 0xfe05820a, 0xf7ff820a, 0x8b810c, 0x8204, 0xff028207, 0x18205, 0xfefb820a, 0xfe008206,
    0x703820c, 0xff8205, 0x158109, 0x1008205, 0xfffe8207, 0xfd018208, 0xf98209, 0xb00820a,
    0x7fd820c, 0xfdff8208, 0x4b810c, 0x8204, 0x28104, 0xff018206, 0x88209, 0x5008208, 0x704820c,
    0xffff8206, 0xe5810a, 0xff008205, 0xfe8104, 0x18103, 0xfff2820b, 0x8102, 0xfe06820a, 0xff8103,
    0xcb810c, 0x8204, 0x28206, 0x18205, 0x1fa8209, 0x2008206, 0x38207, 0xff8205, 0xfff5820b,
    0x1008205, 0xfe8206, 0x2018207, 0xfd09820c, 0xfb008208, 0xfd8207, 0x2ff8207, 0x28810b, 0x8204,
    0xfe028208, 0x1018206, 0x1fc8209, 0x3008207, 0x48207, 0x1ff8206, 0xf18109, 0xff008205,
    0xfefe8208, 0xfc018209, 0xff0b820b, 0x1400820b, 0x1058209, 0xfcff8209, 0x7b810c, 0x8204,
    0x1028207, 0x18205, 0x1fb8209, 0xfe008206, 0x403820a, 0xff8205, 0x138109, 0x1008205, 0x1fe8207,
    0x4018208, 0xfdf8820c, 0xf8008209, 0x4fd820a, 0x4ff8208, 0x3b810c, 0x8204, 0xf902820b,
    0xff018206, 0xfd07820c, 0xfd008207, 0x404820a, 0xffff8206, 0xe0810a, 0xff008205, 0xf9fe820b,
    0x18103, 0xfff0820b, 0x8102, 0x1068209, 0xff8103, 0xbb810c, 0x8204, 0x28206, 0x18205, 0xfa8106,
    0x2008206, 0xff038208, 0xff8205, 0xf8109, 0x1008205, 0xfe8206, 0xfe018207, 0xfdf7820c,
    0x7008209, 0xfffd8208, 0xfeff8207, 0xa000b, 0x8204, 0xfd028209, 0x1018206, 0xfc8106, 0x4008207,
    0xff048208, 0x1ff8206, 0x17810a, 0xff008205, 0xfdfe8209, 0xd01820b, 0xff0d820b, 0xe200820c,
    0x58106, 0xdff820b, 0x9b810c, 0x8204, 0xff028207, 0x18205, 0xfb8106, 0xfe008206, 0x38105,
    0xff8205, 0xec8109, 0x1008205, 0xfffe8207, 0x5018209, 0xfdf9820c, 0xf000820a, 0xfd8105,
    0x5ff8209, 0x5b810c, 0x8204, 0x28104, 0xff018206, 0xfd08820c, 0x6008208, 0x48105, 0xffff8206,
    0xe9810a, 0xff008205, 0xfe8104, 0x18103, 0xfff3820b, 0x8102, 0x68106, 0xff8103, 0x33030c,
    0x8204, 0x28206, 0x18205, 0xfa8208, 0x2008206, 0x38207, 0xff8205, 0xf48108, 0x1008205,
    0xfe8206, 0x2018207, 0xff09820a, 0xfa008208, 0xfd8207, 0x2ff8207, 0x24810b, 0x8204, 0x2028208,
    0x1018206, 0xfc8208, 0x3008207, 0x48207, 0x1ff8206, 0xef8109, 0xff008205, 0x2fe8208,
    0xfb018209, 0xa8108, 0x1000820b, 0x58208, 0xfbff8209, 0x73810c, 0x8204, 0x1028207, 0x18205,
    0xfb8208, 0xfe008206, 0xfe038209, 0xff8205, 0x128109, 0x1008205, 0x1fe8207, 0x3018208,
    0xfff8820a, 0xf7008209, 0xfefd8209, 0x3ff8208, 0x33810c, 0x8204, 0x802820b, 0xff018206,
    0xff07820a, 0xfd008207, 0xfe048209, 0xffff8206, 0x1d810a, 0xff008205, 0x8fe820b, 0xec01820c,
    0xe8108, 0x8102, 0x68208, 0xecff820c, 0xb3810c, 0x8204, 0x28206, 0x18205, 0x5fa820c, 0x2008206,
    0x1038208, 0xff8205, 0xf68108, 0x1008205, 0xfe8206, 0xfe018207, 0xfff7820a, 0xfc008208,
    0x1fd8208, 0xfeff8207, 0xdf810b, 0x8204, 0x4028209, 0x1018206, 0x5fc820c, 0x4008207, 0x1048208,
    0x1ff8206, 0x1020009, 0xff008205, 0x4fe8209, 0xf901820a, 0xc8108, 0x1800820c, 0x505820c,
    0xf9ff820a, 0x93810c, 0x8204, 0xff028207, 0x18205, 0x5fb820c, 0xfe008206, 0x38105, 0xff8205,
    0xeb8109, 0x1008205, 0xfffe8207, 0xfd018208, 0xfff9820a, 0xd00820a, 0xfd8105, 0xfdff8208,
    0x53810c, 0x8204, 0x28104, 0xff018206, 0xff08820a, 0x5008208, 0x48105, 0xffff8206, 0xe7810a,
    0xff008205, 0xfe8104, 0x18103, 0xf28108, 0x8102, 0x506820c, 0xff8103, 0xd3810c, 0x8204,
    0x28206, 0x18205, 0xfffa8209, 0x2008206, 0x38207, 0xff8205, 0xf58108, 0x1008205, 0xfe8206,
    0x2018207, 0x98107, 0xfb008208, 0xfd8207, 0x2ff8207, 0xd7810b, 0x8204, 0xfe028208, 0x1018206,
    0xfffc8209, 0x3008207, 0x48207, 0x1ff8206, 0x50009, 0xff008205, 0xfefe8208, 0x801820a, 0xb8108,
    0xed00820b, 0xff058209, 0x8ff820a, 0x83810c, 0x8204, 0x1028207, 0x18205, 0xfffb8209,
    0xfe008206, 0x603820b, 0xff8205, 0x148109, 0x1008205, 0x1fe8207, 0x4018208, 0xf88107,
    0xf9008209, 0x6fd820b, 0x4ff8208, 0x43810c, 0x8204, 0xf202820c, 0xff018206, 0x78107,
    0xfd008207, 0x604820b, 0xffff8206, 0xe3810a, 0xff008205, 0xf2fe820c, 0x18103, 0xf08108, 0x8102,
    0xff068209, 0xff8103, 0xc3810c, 0x8204, 0x28206, 0x18205, 0xfa8106, 0x2008206, 0xff038208,
    0xff8205, 0x108109, 0x1008205, 0xfe8206, 0xfe018207, 0xf78107, 0x8008209, 0xfffd8208,
    0xfeff8207, 0x13020b, 0x8204, 0x602820a, 0x1018206, 0xfc8106, 0x4008207, 0xff048208, 0x1ff8206,
    0x19810a, 0xff008205, 0x6fe820a, 0xf301820b, 0xd8108, 0xea00820c, 0x58106, 0xf3ff820b,
    0xa3810c, 0x8204, 0xff028207, 0x18205, 0xfb8106, 0xfe008206, 0x38105, 0xff8205, 0xed8109,
    0x1008205, 0xfffe8207, 0x6018209, 0xf98107, 0xf300820a, 0xfd8105, 0x6ff8209, 0x63810c, 0x8204,
    0x28104, 0xff018206, 0x88107, 0x6008208, 0x48105, 0xffff8206, 0x8000a, 0xff008205, 0xfe8104,
    0x18103, 0xf38108, 0x8102, 0x68106, 0xff8103, 0xc3050c, 0x8204, 0x28206, 0x18205, 0xfa8208,
    0x2008206, 0x38207, 0xff8205, 0xf4820a, 0x1008205, 0xfe8206, 0x2018207, 0x98209, 0xfa008208,
    0xfd8207, 0x2ff8207, 0xb010a, 0x8204, 0x2028208, 0x1018206, 0xfc8208, 0x3008207, 0x48207,
    0x1ff8206, 0xee820b, 0xff008205, 0x2fe8208, 0xfa018209, 0xa820a, 0xf400820a, 0x58208,
    0xfaff8209, 0x66810c, 0x8204, 0x1028207, 0x18205, 0xfb8208, 0xfe008206, 0x2038209, 0xff8205,
    0x11820b, 0x1008205, 0x1fe8207, 0x3018208, 0xf88209, 0x9008209, 0x2fd8209, 0x3ff8208, 0x17020b,
    0x8204, 0xfa02820a, 0xff018206, 0x78209, 0xfd008207, 0x2048209, 0xffff8206, 0x1a810a,
    0xff008205, 0xfafe820a, 0xf401820b, 0xe820a, 0x8102, 0x68208, 0xf4ff820b, 0xa6810c, 0x8204,
    0x28206, 0x18205, 0x2fa820a, 0x2008206, 0x1038208, 0xff8205, 0xf6820a, 0x1008205, 0xfe8206,
    0xfe018207, 0xf78209, 0xfc008208, 0x1fd8208, 0xfeff8207, 0xd8810b, 0x8204, 0x3028209,
    0x1018206, 0x2fc820a, 0x4008207, 0x1048208, 0x1ff8206, 0x60009, 0xff008205, 0x3fe8209,
    0x901820a, 0xc820a, 0xee00820b, 0x205820a, 0x9ff820a, 0x86810c, 0x8204, 0xff028207, 0x18205,
    0x2fb820a, 0xfe008206, 0xfa03820b, 0xff8205, 0x15820b, 0x1008205, 0xfffe8207, 0xfd018208,
    0xf98209, 0xa00820a, 0xfafd820b, 0xfdff8208, 0x46810c, 0x8204, 0xf502820c, 0xff018206, 0x88209,
    0x5008208, 0xfa04820b, 0xffff8206, 0xe4810a, 0xff008205, 0xf5fe820c, 0x18103, 0xf2820a, 0x8102,
    0x206820a, 0xff8103, 0xc6810c, 0x8204, 0x28206, 0x18205, 0x1fa8209, 0x2008206, 0x38207,
    0xff8205, 0xf5820a, 0x1008205, 0xfe8206, 0x2018207, 0x209820b, 0xfb008208, 0xfd8207, 0x2ff8207,
    0x25810b, 0x8204, 0xfe028208, 0x1018206, 0x1fc8209, 0x3008207, 0x48207, 0x1ff8206, 0xf1820b,
    0xff008205, 0xfefe8208, 0xfc018209, 0xb820a, 0x1100820b, 0x1058209, 0xfcff8209, 0x76810c,
    0x8204, 0x1028207, 0x18205, 0x1fb8209, 0xfe008206, 0x303820a, 0xff8205, 0x13820b, 0x1008205,
    0x1fe8207, 0x4018208, 0x2f8820b, 0xf8008209, 0x3fd820a, 0x4ff8208, 0x36810c, 0x8204, 0x902820b,
    0xff018206, 0x207820b, 0xfd008207, 0x304820a, 0xffff8206, 0x1e810a, 0xff008205, 0x9fe820b,
    0xef01820c, 0xf0820a, 0x8102, 0x1068209, 0xefff820c, 0xb6810c, 0x8204, 0x28206, 0x18205,
    0xfbfa820c, 0x2008206, 0xff038208, 0xff8205, 0xf820b, 0x1008205, 0xfe8206, 0xfe018207,
    0x2f7820b, 0x7008209, 0xfffd8208, 0xfeff8207, 0xe1810b, 0x8204, 0xfd028209, 0x1018206,
    0xfbfc820c, 0x4008207, 0xff048208, 0x1ff8206, 0x16810a, 0xff008205, 0xfdfe8209, 0xa01820b,
    0xd820a, 0x1b00820c, 0xfb05820c, 0xaff820b, 0x96810c, 0x8204, 0xff028207, 0x18205, 0xfbfb820c,
    0xfe008206, 0x38105, 0xff8205, 0xec820b, 0x1008205, 0xfffe8207, 0x5018209, 0x2f9820b,
    0xe00820a, 0xfd8105, 0x5ff8209, 0x56810c, 0x8204, 0x28104, 0xff018206, 0x208820b, 0x6008208,
    0x48105, 0xffff8206, 0xe8810a, 0xff008205, 0xfe8104, 0x18103, 0xf3820a, 0x8102, 0xfb06820c,
    0xff8103, 0x400c, 0x8204, 0x28206, 0x18205, 0xfa8208, 0x2008206, 0x38207, 0xff8205, 0xfef4820c,
    0x1008205, 0xfe8206, 0x2018207, 0x109820a, 0xfa008208, 0xfd8207, 0x2ff8207, 0x21810b, 0x8204,
    0x2028208, 0x1018206, 0xfc8208, 0x3008207, 0x48207, 0x1ff8206, 0xef820b, 0xff008205, 0x2fe8208,
    0xfb018209, 0xfe0a820c, 0xf600820a, 0x58208, 0xfbff8209, 0x6e810c, 0x8204, 0x1028207, 0x18205,
    0xfb8208, 0xfe008206, 0xfe038209, 0xff8205, 0x12820b, 0x1008205, 0x1fe8207, 0x3018208,
    0x1f8820a, 0xf7008209, 0xfefd8209, 0x3ff8208, 0x2e810c, 0x8204, 0xfc02820a, 0xff018206,
    0x107820a, 0xfd008207, 0xfe048209, 0xffff8206, 0x1c810a, 0xff008205, 0xfcfe820a, 0x1201820c,
    0xfe0e820c, 0x8102, 0x68208, 0x12ff820c, 0xae810c, 0x8204, 0x28206, 0x18205, 0x3fa820b,
    0x2008206, 0x1038208, 0xff8205, 0xfef6820c, 0x1008205, 0xfe8206, 0xfe018207, 0x1f7820a,
    0xfc008208, 0x1fd8208, 0xfeff8207, 0xdc810b, 0x8204, 0x4028209, 0x1018206, 0x3fc820b,
    0x4008207, 0x1048208, 0x1ff8206, 0x1020009, 0xff008205, 0x4fe8209, 0xf801820a, 0xfe0c820c,
    0x8102, 0x305820b, 0xf8ff820a, 0x8e810c, 0x8204, 0xff028207, 0x18205, 0x3fb820b, 0xfe008206,
    0xf703820c, 0xff8205, 0xeb820b, 0x1008205, 0xfffe8207, 0xfd018208, 0x1f9820a, 0xc00820a,
    0xf7fd820c, 0xfdff8208, 0x4e810c, 0x8204, 0x28104, 0xff018206, 0x108820a, 0x5008208,
    0xf704820c, 0xffff8206, 0xe6810a, 0xff008205, 0xfe8104, 0x18103, 0xfef2820c, 0x8102, 0x306820b,
    0xff8103, 0xce810c, 0x8204, 0x28206, 0x18205, 0xfffa8209, 0x2008206, 0x38207, 0xff8205,
    0xfef5820c, 0x1008205, 0xfe8206, 0x2018207, 0x98107, 0xfb008208, 0xfd8207, 0x2ff8207, 0x29810b,
    0x8204, 0xfe028208, 0x1018206, 0xfffc8209, 0x3008207, 0x48207, 0x1ff8206, 0x50009, 0xff008205,
    0xfefe8208, 0x701820a, 0xfe0b820c, 0x1500820b, 0xff058209, 0x7ff820a, 0x7e810c, 0x8204,
    0x1028207, 0x18205, 0xfffb8209, 0xfe008206, 0xfd03820a, 0xff8205, 0x14820b, 0x1008205,
    0x1fe8207, 0x4018208, 0xf88107, 0xf9008209, 0xfdfd820a, 0x4ff8208, 0x3e810c, 0x8204, 0xb02820c,
    0xff018206, 0x78107, 0xfd008207, 0xfd04820a, 0xffff8206, 0xe2810a, 0xff008205, 0xbfe820c,
    0x18103, 0xfef0820c, 0x8102, 0xff068209, 0xff8103, 0xbe810c, 0x8204, 0x28206, 0x18205,
    0xfa8106, 0x2008206, 0xff038208, 0xff8205, 0x10820b, 0x1008205, 0xfe8206, 0xfe018207, 0xf78107,
    0x8008209, 0xfffd8208, 0xfeff8207, 0xd010b, 0x8204, 0x502820a, 0x1018206, 0xfc8106, 0x4008207,
    0xff048208, 0x1ff8206, 0x18810a, 0xff008205, 0x5fe820a, 0xe01820b, 0xfe0d820c, 0xe500820c,
    0x58106, 0xeff820b, 0x9e810c, 0x8204, 0xff028207, 0x18205, 0xfb8106, 0xfe008206, 0x38105,
    0xff8205, 0xed820b, 0x1008205, 0xfffe8207, 0x6018209, 0xf98107, 0xf200820a, 0xfd8105,
    0x6ff8209, 0x5e810c, 0x8204, 0x28104, 0xff018206, 0x88107, 0x6008208, 0x48105, 0xffff8206,
    0xea810a, 0xff008205, 0xfe8104, 0x18103, 0xfef3820c, 0x8102, 0x68106, 0xff8103, 0x53040c,
    0x8204, 0x28206, 0x18205, 0xfa8208, 0x2008206, 0x38207, 0xff8205, 0x1f4820b, 0x1008205,
    0xfe8206, 0x2018207, 0x98209, 0xfa008208, 0xfd8207, 0x2ff8207, 0x1f810b, 0x8204, 0x2028208,
    0x1018206, 0xfc8208, 0x3008207, 0x48207, 0x1ff8206, 0xee8109, 0xff008205, 0x2fe8208,
    0xfa018209, 0x10a820b, 0xf500820a, 0x58208, 0xfaff8209, 0x6a810c, 0x8204, 0x1028207, 0x18205,
    0xfb8208, 0xfe008206, 0x2038209, 0xff8205, 0x118109, 0x1008205, 0x1fe8207, 0x3018208, 0xf88209,
    0x9008209, 0x2fd8209, 0x3ff8208, 0x23030b, 0x8204, 0xfb02820a, 0xff018206, 0x78209, 0xfd008207,
    0x2048209, 0xffff8206, 0x1b810a, 0xff008205, 0xfbfe820a, 0xf601820b, 0x10e820b, 0x8102,
    0x68208, 0xf6ff820b, 0xaa810c, 0x8204, 0x28206, 0x18205, 0xfefa820a, 0x2008206, 0x1038208,
    0xff8205, 0x1f6820b, 0x1008205, 0xfe8206, 0xfe018207, 0xf78209, 0xfc008208, 0x1fd8208,
    0xfeff8207, 0xda810b, 0x8204, 0x3028209, 0x1018206, 0xfefc820a, 0x4008207, 0x1048208,
    0x1ff8206, 0x60009, 0xff008205, 0x3fe8209, 0xf701820a, 0x10c820b, 0xf100820b, 0xfe05820a,
    0xf7ff820a, 0x8a810c, 0x8204, 0xff028207, 0x18205, 0xfefb820a, 0xfe008206, 0xfc03820b,
    0xff8205, 0x158109, 0x1008205, 0xfffe8207, 0xfd018208, 0xf98209, 0xb00820a, 0xfcfd820b,
    0xfdff8208, 0x4a810c, 0x8204, 0x28104, 0xff018206, 0x88209, 0x5008208, 0xfc04820b, 0xffff8206,
    0xe5810a, 0xff008205, 0xfe8104, 0x18103, 0x1f2820b, 0x8102, 0xfe06820a, 0xff8103, 0xca810c,
    0x8204, 0x28206, 0x18205, 0x1fa8209, 0x2008206, 0x38207, 0xff8205, 0x1f5820b, 0x1008205,
    0xfe8206, 0x2018207, 0x409820c, 0xfb008208, 0xfd8207, 0x2ff8207, 0x27810b, 0x8204, 0xfe028208,
    0x1018206, 0x1fc8209, 0x3008207, 0x48207, 0x1ff8206, 0xf18109, 0xff008205, 0xfefe8208,
    0xfc018209, 0x10b820b, 0x1300820b, 0x1058209, 0xfcff8209, 0x7a810c, 0x8204, 0x1028207, 0x18205,
    0x1fb8209, 0xfe008206, 0x403820a, 0xff8205, 0x138109, 0x1008205, 0x1fe8207, 0x4018208,
    0x4f8820c, 0xf8008209, 0x4fd820a, 0x4ff8208, 0x3a810c, 0x8204, 0xf802820b, 0xff018206,
    0x407820c, 0xfd008207, 0x404820a, 0xffff8206, 0xe0810a, 0xff008205, 0xf8fe820b, 0x18103,
    0x1f0820b, 0x8102, 0x1068209, 0xff8103, 0xba810c, 0x8204, 0x28206, 0x18205, 0xfa8106,
    0x2008206, 0xff038208, 0xff8205, 0xf8109, 0x1008205, 0xfe8206, 0xfe018207, 0x4f7820c,
    0x7008209, 0xfffd8208, 0xfeff8207, 0x9000b, 0x8204, 0xfd028209, 0x1018206, 0xfc8106, 0x4008207,
    0xff048208, 0x1ff8206, 0x17810a, 0xff008205, 0xfdfe8209, 0xc01820b, 0x10d820b, 0xe000820c,
    0x58106, 0xcff820b, 0x9a810c, 0x8204, 0xff028207, 0x18205, 0xfb8106, 0xfe008206, 0x38105,
    0xff8205, 0xec8109, 0x1008205, 0xfffe8207, 0x5018209, 0x4f9820c, 0xf000820a, 0xfd8105,
    0x5ff8209, 0x5a810c, 0x8204, 0x28104, 0xff018206, 0x408820c, 0x6008208, 0x48105, 0xffff8206,
    0xe9810a, 0xff008205, 0xfe8104, 0x18103, 0x1f3820b, 0x8102, 0x68106, 0xff8103, 0x2b030c,
    0x8204, 0x28206, 0x18205, 0xfa8208, 0x2008206, 0x38207, 0xff8205, 0xf48108, 0x1008205,
    0xfe8206, 0x2018207, 0xff09820a, 0xfa008208, 0xfd8207, 0x2ff8207, 0x23810b, 0x8204, 0x2028208,
    0x1018206, 0xfc8208, 0x3008207, 0x48207, 0x1ff8206, 0xef8109, 0xff008205, 0x2fe8208,
    0xfb018209, 0xa8108, 0xf00820b, 0x58208, 0xfbff8209, 0x72810c, 0x8204, 0x1028207, 0x18205,
    0xfb8208, 0xfe008206, 0xfe038209, 0xff8205, 0x128109, 0x1008205, 0x1fe8207, 0x3018208,
    0xfff8820a, 0xf7008209, 0xfefd8209, 0x3ff8208, 0x32810c, 0x8204, 0x702820b, 0xff018206,
    0xff07820a, 0xfd008207, 0xfe048209, 0xffff8206, 0x1d810a, 0xff008205, 0x7fe820b, 0xeb01820c,
    0xe8108, 0x8102, 0x68208, 0xebff820c, 0xb2810c, 0x8204, 0x28206, 0x18205, 0xfdfa820b,
    0x2008206, 0x1038208, 0xff8205, 0xf68108, 0x1008205, 0xfe8206, 0xfe018207, 0xfff7820a,
    0xfc008208, 0x1fd8208, 0xfeff8207, 0xde810b, 0x8204, 0x4028209, 0x1018206, 0xfdfc820b,
    0x4008207, 0x1048208, 0x1ff8206, 0x1020009, 0xff008205, 0x4fe8209, 0xf901820a, 0xc8108,
    0x1700820c, 0xfd05820b, 0xf9ff820a, 0x92810c, 0x8204, 0xff028207, 0x18205, 0xfdfb820b,
    0xfe008206, 0x38105, 0xff8205, 0xeb8109, 0x1008205, 0xfffe8207, 0xfd018208, 0xfff9820a,
    0xd00820a, 0xfd8105, 0xfdff8208, 0x52810c, 0x8204, 0x28104, 0xff018206, 0xff08820a, 0x5008208,
    0x48105, 0xffff8206, 0xe7810a, 0xff008205, 0xfe8104, 0x18103, 0xf28108, 0x8102, 0xfd06820b,
    0xff8103, 0xd2810c, 0x8204, 0x28206, 0x18205, 0xfffa8209, 0x2008206, 0x38207, 0xff8205,
    0xf58108, 0x1008205, 0xfe8206, 0x2018207, 0x98107, 0xfb008208, 0xfd8207, 0x2ff8207, 0xd6810b,
    0x8204, 0xfe028208, 0x1018206, 0xfffc8209, 0x3008207, 0x48207, 0x1ff8206, 0x50009, 0xff008205,
    0xfefe8208, 0x801820a, 0xb8108, 0xec00820b, 0xff058209, 0x8ff820a, 0x82810c, 0x8204, 0x1028207,
    0x18205, 0xfffb8209, 0xfe008206, 0x503820b, 0xff8205, 0x148109, 0x1008205, 0x1fe8207,
    0x4018208, 0xf88107, 0xf9008209, 0x5fd820b, 0x4ff8208, 0x42810c, 0x8204, 0xf002820c,
    0xff018206, 0x78107, 0xfd008207, 0x504820b, 0xffff8206, 0xe3810a, 0xff008205, 0xf0fe820c,
    0x18103, 0xf08108, 0x8102, 0xff068209, 0xff8103, 0xc2810c, 0x8204, 0x28206, 0x18205, 0xfa8106,
    0x2008206, 0xff038208, 0xff8205, 0x108109, 0x1008205, 0xfe8206, 0xfe018207, 0xf78107,
    0x8008209, 0xfffd8208, 0xfeff8207, 0x11010b, 0x8204, 0x602820a, 0x1018206, 0xfc8106, 0x4008207,
    0xff048208, 0x1ff8206, 0x19810a, 0xff008205, 0x6fe820a, 0xf201820b, 0xd8108, 0xe900820c,
    0x58106, 0xf2ff820b, 0xa2810c, 0x8204, 0xff028207, 0x18205, 0xfb8106, 0xfe008206, 0x38105,
    0xff8205, 0xed8109, 0x1008205, 0xfffe8207, 0x6018209, 0xf98107, 0xf300820a, 0xfd8105,
    0x6ff8209, 0x62810c, 0x8204, 0x28104, 0xff018206, 0x88107, 0x6008208, 0x48105, 0xffff8206,
    0x8000a, 0xff008205, 0xfe8104, 0x18103, 0xf38108, 0x8102, 0x68106, 0xff8103, 0xa3050c, 0x8204,
    0x28206, 0x18205, 0xfa8208, 0x2008206, 0x38207, 0xff8205, 0xf4820a, 0x1008205, 0xfe8206,
    0x2018207, 0x98209, 0xfa008208, 0xfd8207, 0x2ff8207, 0xb010a, 0x8204, 0x2028208, 0x1018206,
    0xfc8208, 0x3008207, 0x48207, 0x1ff8206, 0xffee820c, 0xff008205, 0x2fe8208, 0xfa018209,
    0xa820a, 0xf400820a, 0x58208, 0xfaff8209, 0x68810c, 0x8204, 0x1028207, 0x18205, 0xfb8208,
    0xfe008206, 0x2038209, 0xff8205, 0xff11820c, 0x1008205, 0x1fe8207, 0x3018208, 0xf88209,
    0x9008209, 0x2fd8209, 0x3ff8208, 0x1b020b, 0x8204, 0xfa02820a, 0xff018206, 0x78209, 0xfd008207,
    0x2048209, 0xffff8206, 0x1a810a, 0xff008205, 0xfafe820a, 0xf501820b, 0xe820a, 0x8102, 0x68208,
    0xf5ff820b, 0xa8810c, 0x8204, 0x28206, 0x18205, 0x2fa820a, 0x2008206, 0x1038208, 0xff8205,
    0xf6820a, 0x1008205, 0xfe8206, 0xfe018207, 0xf78209, 0xfc008208, 0x1fd8208, 0xfeff8207,
    0xd9810b, 0x8204, 0x3028209, 0x1018206, 0x2fc820a, 0x4008207, 0x1048208, 0x1ff8206, 0x60009,
    0xff008205, 0x3fe8209, 0x901820a, 0xc820a, 0xef00820b, 0x205820a, 0x9ff820a, 0x88810c, 0x8204,
    0xff028207, 0x18205, 0x2fb820a, 0xfe008206, 0xfb03820b, 0xff8205, 0xff15820c, 0x1008205,
    0xfffe8207, 0xfd018208, 0xf98209, 0xa00820a, 0xfbfd820b, 0xfdff8208, 0x48810c, 0x8204, 0x28104,
    0xff018206, 0x88209, 0x5008208, 0xfb04820b, 0xffff8206, 0xe4810a, 0xff008205, 0xfe8104,
    0x18103, 0xf2820a, 0x8102, 0x206820a, 0xff8103, 0xc8810c, 0x8204, 0x28206, 0x18205, 0x1fa8209,
    0x2008206, 0x38207, 0xff8205, 0xf5820a, 0x1008205, 0xfe8206, 0x2018207, 0xfe09820b, 0xfb008208,
    0xfd8207, 0x2ff8207, 0x26810b, 0x8204, 0xfe028208, 0x1018206, 0x1fc8209, 0x3008207, 0x48207,
    0x1ff8206, 0xfff1820c, 0xff008205, 0xfefe8208, 0xfc018209, 0xb820a, 0x1200820b, 0x1058209,
    0xfcff8209, 0x78810c, 0x8204, 0x1028207, 0x18205, 0x1fb8209, 0xfe008206, 0x303820a, 0xff8205,
    0xff13820c, 0x1008205, 0x1fe8207, 0x4018208, 0xfef8820b, 0xf8008209, 0x3fd820a, 0x4ff8208,
    0x38810c, 0x8204, 0xf702820b, 0xff018206, 0xfe07820b, 0xfd008207, 0x304820a, 0xffff8206,
    0x1e810a, 0xff008205, 0xf7fe820b, 0x18103, 0xf0820a, 0x8102, 0x1068209, 0xff8103, 0xb8810c,
    0x8204, 0x28206, 0x18205, 0xfa8106, 0x2008206, 0xff038208, 0xff8205, 0xff0f820c, 0x1008205,
    0xfe8206, 0xfe018207, 0xfef7820b, 0x7008209, 0xfffd8208, 0xfeff8207, 0x7000b, 0x8204,
    0xfd028209, 0x1018206, 0xfc8106, 0x4008207, 0xff048208, 0x1ff8206, 0x16810a, 0xff008205,
    0xfdfe8209, 0xb01820b, 0xd820a, 0x1d00820c, 0x58106, 0xbff820b, 0x98810c, 0x8204, 0xff028207,
    0x18205, 0xfb8106, 0xfe008206, 0x38105, 0xff8205, 0xffec820c, 0x1008205, 0xfffe8207, 0x5018209,
    0xfef9820b, 0xe00820a, 0xfd8105, 0x5ff8209, 0x58810c, 0x8204, 0x28104, 0xff018206, 0xfe08820b,
    0x6008208, 0x48105, 0xffff8206, 0xe8810a, 0xff008205, 0xfe8104, 0x18103, 0xf3820a, 0x8102,
    0x68106, 0xff8103, 0x4000c, 0x8204, 0x28206, 0x18205, 0xfa8208, 0x2008206, 0x38207, 0xff8205,
    0xf48108, 0x1008205, 0xfe8206, 0x2018207, 0x109820a, 0xfa008208, 0xfd8207, 0x2ff8207, 0x22810b,
    0x8204, 0x2028208, 0x1018206, 0xfc8208, 0x3008207, 0x48207, 0x1ff8206, 0xffef820c, 0xff008205,
    0x2fe8208, 0xfb018209, 0xa8108, 0xf600820a, 0x58208, 0xfbff8209, 0x70810c, 0x8204, 0x1028207,
    0x18205, 0xfb8208, 0xfe008206, 0xfe038209, 0xff8205, 0xff12820c, 0x1008205, 0x1fe8207,
    0x3018208, 0x1f8820a, 0xf7008209, 0xfefd8209, 0x3ff8208, 0x30810c, 0x8204, 0xfc02820a,
    0xff018206, 0x107820a, 0xfd008207, 0xfe048209, 0xffff8206, 0x1c810a, 0xff008205, 0xfcfe820a,
    0x1401820c, 0xe8108, 0x8102, 0x68208, 0x14ff820c, 0xb0810c, 0x8204, 0x28206, 0x18205,
    0x4fa820b, 0x2008206, 0x1038208, 0xff8205, 0xf68108, 0x1008205, 0xfe8206, 0xfe018207,
    0x1f7820a, 0xfc008208, 0x1fd8208, 0xfeff8207, 0xdd810b, 0x8204, 0x4028209, 0x1018206,
    0x4fc820b, 0x4008207, 0x1048208, 0x1ff8206, 0x1020009, 0xff008205, 0x4fe8209, 0xf801820a,
    0xc8108, 0x8102, 0x405820b, 0xf8ff820a, 0x90810c, 0x8204, 0xff028207, 0x18205, 0x4fb820b,
    0xfe008206, 0xf903820c, 0xff8205, 0xffeb820c, 0x1008205, 0xfffe8207, 0xfd018208, 0x1f9820a,
    0xc00820a, 0xf9fd820c, 0xfdff8208, 0x50810c, 0x8204, 0x28104, 0xff018206, 0x108820a, 0x5008208,
    0xf904820c, 0xffff8206, 0xe6810a, 0xff008205, 0xfe8104, 0x18103, 0xf28108, 0x8102, 0x406820b,
    0xff8103, 0xd0810c, 0x8204, 0x28206, 0x18205, 0xfffa8209, 0x2008206, 0x38207, 0xff8205,
    0xf58108, 0x1008205, 0xfe8206, 0x2018207, 0x98107, 0xfb008208, 0xfd8207, 0x2ff8207, 0x2a810b,
    0x8204, 0xfe028208, 0x1018206, 0xfffc8209, 0x3008207, 0x48207, 0x1ff8206, 0x50009, 0xff008205,
    0xfefe8208, 0x701820a, 0xb8108, 0xeb00820b, 0xff058209, 0x7ff820a, 0x80810c, 0x8204, 0x1028207,
    0x18205, 0xfffb8209, 0xfe008206, 0xfd03820a, 0xff8205, 0xff14820c, 0x1008205, 0x1fe8207,
    0x4018208, 0xf88107, 0xf9008209, 0xfdfd820a, 0x4ff8208, 0x40810c, 0x8204, 0xd02820c,
    0xff018206, 0x78107, 0xfd008207, 0xfd04820a, 0xffff8206, 0xe2810a, 0xff008205, 0xdfe820c,
    0x18103, 0xf08108, 0x8102, 0xff068209, 0xff8103, 0xc0810c, 0x8204, 0x28206, 0x18205, 0xfa8106,
    0x2008206, 0xff038208, 0xff8205, 0xff10820c, 0x1008205, 0xfe8206, 0xfe018207, 0xf78107,
    0x8008209, 0xfffd8208, 0xfeff8207, 0xf010b, 0x8204, 0x502820a, 0x1018206, 0xfc8106, 0x4008207,
    0xff048208, 0x1ff8206, 0x18810a, 0xff008205, 0x5fe820a, 0xf001820b, 0xd8108, 0xe700820c,
    0x58106, 0xf0ff820b, 0xa0810c, 0x8204, 0xff028207, 0x18205, 0xfb8106, 0xfe008206, 0x38105,
    0xff8205, 0xffed820c, 0x1008205, 0xfffe8207, 0x6018209, 0xf98107, 0xf200820a, 0xfd8105,
    0x6ff8209, 0x60810c, 0x8204, 0x28104, 0xff018206, 0x88107, 0x6008208, 0x48105, 0xffff8206,
    0xea810a, 0xff008205, 0xfe8104, 0x18103, 0xf38108, 0x8102, 0x68106, 0xff8103, 0x73040c, 0x8204,
    0x28206, 0x18205, 0xfa8208, 0x2008206, 0x38207, 0xff8205, 0xfff4820b, 0x1008205, 0xfe8206,
    0x2018207, 0x98209, 0xfa008208, 0xfd8207, 0x2ff8207, 0x20810b, 0x8204, 0x2028208, 0x1018206,
    0xfc8208, 0x3008207, 0x48207, 0x1ff8206, 0xee8109, 0xff008205, 0x2fe8208, 0xfa018209,
    0xff0a820b, 0xf500820a, 0x58208, 0xfaff8209, 0x6c810c, 0x8204, 0x1028207, 0x18205, 0xfb8208,
    0xfe008206, 0x2038209, 0xff8205, 0x118109, 0x1008205, 0x1fe8207, 0x3018208, 0xf88209,
    0x9008209, 0x2fd8209, 0x3ff8208, 0x2c810c, 0x8204, 0xfb02820a, 0xff018206, 0x78209, 0xfd008207,
    0x2048209, 0xffff8206, 0x1b810a, 0xff008205, 0xfbfe820a, 0x1001820c, 0xff0e820b, 0x8102,
    0x68208, 0x10ff820c, 0xac810c, 0x8204, 0x28206, 0x18205, 0xfefa820a, 0x2008206, 0x1038208,
    0xff8205, 0xfff6820b, 0x1008205, 0xfe8206, 0xfe018207, 0xf78209, 0xfc008208, 0x1fd8208,
    0xfeff8207, 0xdb810b, 0x8204, 0x3028209, 0x1018206, 0xfefc820a, 0x4008207, 0x1048208,
    0x1ff8206, 0x60009, 0xff008205, 0x3fe8209, 0xf701820a, 0xff0c820b, 0x8102, 0xfe05820a,
    0xf7ff820a, 0x8c810c, 0x8204, 0xff028207, 0x18205, 0xfefb820a, 0xfe008206, 0x803820c, 0xff8205,
    0x158109, 0x1008205, 0xfffe8207, 0xfd018208, 0xf98209, 0xb00820a, 0x8fd820c, 0xfdff8208,
    0x4c810c, 0x8204, 0x28104, 0xff018206, 0x88209, 0x5008208, 0x804820c, 0xffff8206, 0xe5810a,
    0xff008205, 0xfe8104, 0x18103, 0xfff2820b, 0x8102, 0xfe06820a, 0xff8103, 0xcc810c, 0x8204,
    0x28206, 0x18205, 0x1fa8209, 0x2008206, 0x38207, 0xff8205, 0xfff5820b, 0x1008205, 0xfe8206,
    0x2018207, 0x98107, 0xfb008208, 0xfd8207, 0x2ff8207, 0x28810b, 0x8204, 0xfe028208, 0x1018206,
    0x1fc8209, 0x3008207, 0x48207, 0x1ff8206, 0xf18109, 0xff008205, 0xfefe8208, 0xfc018209,
    0xff0b820b, 0x1400820b, 0x1058209, 0xfcff8209, 0x7c810c, 0x8204, 0x1028207, 0x18205, 0x1fb8209,
    0xfe008206, 0x403820a, 0xff8205, 0x138109, 0x1008205, 0x1fe8207, 0x4018208, 0xf88107,
    0xf8008209, 0x4fd820a, 0x4ff8208, 0x3c810c, 0x8204, 0xf902820b, 0xff018206, 0x78107,
    0xfd008207, 0x404820a, 0xffff8206, 0xe0810a, 0xff008205, 0xf9fe820b, 0x18103, 0xfff0820b,
    0x8102, 0x1068209, 0xff8103, 0xbc810c, 0x8204, 0x28206, 0x18205, 0xfa8106, 0x2008206,
    0xff038208, 0xff8205, 0xf8109, 0x1008205, 0xfe8206, 0xfe018207, 0xf78107, 0x7008209,
    0xfffd8208, 0xfeff8207, 0xa000b, 0x8204, 0xfd028209, 0x1018206, 0xfc8106, 0x4008207,
    0xff048208, 0x1ff8206, 0x17810a, 0xff008205, 0xfdfe8209, 0xd01820b, 0xff0d820b, 0xe300820c,
    0x58106, 0xdff820b, 0x9c810c, 0x8204, 0xff028207, 0x18205, 0xfb8106, 0xfe008206, 0x38105,
    0xff8205, 0xec8109, 0x1008205, 0xfffe8207, 0x5018209, 0xf98107, 0xf000820a, 0xfd8105,
    0x5ff8209, 0x5c810c, 0x8204, 0x28104, 0xff018206, 0x88107, 0x6008208, 0x48105, 0xffff8206,
    0xe9810a, 0xff008205, 0xfe8104, 0x18103, 0xfff3820b, 0x8102, 0x68106, 0xff8103, 0x3b030c,
    0x8204, 0x28206, 0x18205, 0xfa8208, 0x2008206, 0x38207, 0xff8205, 0xf48108, 0x1008205,
    0xfe8206, 0x2018207, 0xff09820a, 0xfa008208, 0xfd8207, 0x2ff8207, 0x24810b, 0x8204, 0x2028208,
    0x1018206, 0xfc8208, 0x3008207, 0x48207, 0x1ff8206, 0xef8109, 0xff008205, 0x2fe8208,
    0xfb018209, 0xa8108, 0x1000820b, 0x58208, 0xfbff8209, 0x74810c, 0x8204, 0x1028207, 0x18205,
    0xfb8208, 0xfe008206, 0xfe038209, 0xff8205, 0x128109, 0x1008205, 0x1fe8207, 0x3018208,
    0xfff8820a, 0xf7008209, 0xfefd8209, 0x3ff8208, 0x34810c, 0x8204, 0x802820b, 0xff018206,
    0xff07820a, 0xfd008207, 0xfe048209, 0xffff8206, 0x1d810a, 0xff008205, 0x8fe820b, 0xed01820c,
    0xe8108, 0x8102, 0x68208, 0xedff820c, 0xb4810c, 0x8204, 0x28206, 0x18205, 0x6fa820c, 0x2008206,
    0x1038208, 0xff8205, 0xf68108, 0x1008205, 0xfe8206, 0xfe018207, 0xfff7820a, 0xfc008208,
    0x1fd8208, 0xfeff8207, 0xdf810b, 0x8204, 0x4028209, 0x1018206, 0x6fc820c, 0x4008207, 0x1048208,
    0x1ff8206, 0x1020009, 0xff008205, 0x4fe8209, 0xf901820a, 0xc8108, 0x1900820c, 0x605820c,
    0xf9ff820a, 0x94810c, 0x8204, 0xff028207, 0x18205, 0x6fb820c, 0xfe008206, 0x38105, 0xff8205,
    0xeb8109, 0x1008205, 0xfffe8207, 0xfd018208, 0xfff9820a, 0xd00820a, 0xfd8105, 0xfdff8208,
    0x54810c, 0x8204, 0x28104, 0xff018206, 0xff08820a, 0x5008208, 0x48105, 0xffff8206, 0xe7810a,
    0xff008205, 0xfe8104, 0x18103, 0xf28108, 0x8102, 0x606820c, 0xff8103, 0xd4810c, 0x8204,
    0x28206, 0x18205, 0xfffa8209, 0x2008206, 0x38207, 0xff8205, 0xf58108, 0x1008205, 0xfe8206,
    0x2018207, 0x98107, 0xfb008208, 0xfd8207, 0x2ff8207, 0xd7810b, 0x8204, 0xfe028208, 0x1018206,
    0xfffc8209, 0x3008207, 0x48207, 0x1ff8206, 0x50009, 0xff008205, 0xfefe8208, 0x801820a, 0xb8108,
    0xed00820b, 0xff058209, 0x8ff820a, 0x84810c, 0x8204, 0x1028207, 0x18205, 0xfffb8209,
    0xfe008206, 0x603820b, 0xff8205, 0x148109, 0x1008205, 0x1fe8207, 0x4018208, 0xf88107,
    0xf9008209, 0x6fd820b, 0x4ff8208, 0x44810c, 0x8204, 0xf302820c, 0xff018206, 0x78107,
    0xfd008207, 0x604820b, 0xffff8206, 0xe3810a, 0xff008205, 0xf3fe820c, 0x18103, 0xf08108, 0x8102,
    0xff068209, 0xff8103, 0xc4810c, 0x8204, 0x28206, 0x18205, 0xfa8106, 0x2008206, 0xff038208,
    0xff8205, 0x108109, 0x1008205, 0xfe8206, 0xfe018207, 0xf78107, 0x8008209, 0xfffd8208,
    0xfeff8207, 0x13020b, 0x8204, 0x602820a, 0x1018206, 0xfc8106, 0x4008207, 0xff048208, 0x1ff8206,
    0x19810a, 0xff008205, 0x6fe820a, 0xf301820b, 0xd8108, 0x8102, 0x58106, 0xf3ff820b, 0xa4810c,
    0x8204, 0xff028207, 0x18205, 0xfb8106, 0xfe008206, 0x38105, 0xff8205, 0xed8109, 0x1008205,
    0xfffe8207, 0x6018209, 0xf98107, 0xf300820a, 0xfd8105, 0x6ff8209, 0x64810c, 0x8204, 0x28104,
    0xff018206, 0x88107, 0x6008208, 0x48105, 0xffff8206, 0x8000a, 0xff008205, 0xfe8104, 0x18103,
    0xf38108, 0x8102, 0x68106, 0xff8103, 0xe3050c,
];

pub(crate) const FDEFLATE_DIST_DECODE_TABLE: [u32; 512] = [
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0,
    0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001, 0x0, 0x10001,
    0x0, 0x10001, 0x0, 0x10001, 0x0,
];

pub(crate) const FIXED_CODE_LENGTHS: [u8; 320] = make_fixed_code_lengths();
const fn make_fixed_code_lengths() -> [u8; 320] {
    let mut i = 0;
    let mut lengths = [0; 320];
    while i < 144 {
        lengths[i] = 8;
        i += 1;
    }
    while i < 256 {
        lengths[i] = 9;
        i += 1;
    }
    while i < 280 {
        lengths[i] = 7;
        i += 1;
    }
    while i < 288 {
        lengths[i] = 8;
        i += 1;
    }
    while i < 320 {
        lengths[i] = 5;
        i += 1;
    }
    lengths
}
