/*
    options.c - Options dialog
    Copyright (c) 2006  Joey Schulze <joey@infodrom.org>

    This file is part of the Gerstensaft package.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111, USA.
*/

#include <gtk/gtk.h>
#include <stdlib.h>
#include <string.h>
#include "beer.h"
#include "nls.h"

struct s_options_data {
  GtkWidget *dialog;
  GtkWidget *send_dotfiles;
  GtkWidget *send_closewindow;
  GtkWidget *rec_keepfiles;
  GtkWidget *proto_auto;
  GtkWidget *proto_ipv4;
  GtkWidget *proto_ipv6;
};

void preferences_init (struct s_options_data *data)
{
  if (config.dotfiles)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->send_dotfiles), TRUE);
  else
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->send_dotfiles), FALSE);

  if (config.closewindow)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->send_closewindow), TRUE);
  else
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->send_closewindow), FALSE);

  if (config.keep)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->rec_keepfiles), TRUE);
  else
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->rec_keepfiles), FALSE);

  switch (config.protocol) {
  case PROTOCOL_AUTO:
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->proto_auto), TRUE);
    break;
  case PROTOCOL_IPV4:
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->proto_ipv4), TRUE);
    break;
  case PROTOCOL_IPV6:
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->proto_ipv6), TRUE);
    break;
  }
}

void preferences_accept (struct s_options_data *data)
{
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->rec_keepfiles)))
    config.keep = TRUE;
  else
    config.keep = FALSE;

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->send_dotfiles)))
    config.dotfiles = TRUE;
  else
    config.dotfiles = FALSE;

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->send_closewindow)))
    config.closewindow = TRUE;
  else
    config.closewindow = FALSE;

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->proto_auto)))
    config.protocol = PROTOCOL_AUTO;
  else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->proto_ipv4)))
    config.protocol = PROTOCOL_IPV4;
  else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->proto_ipv6)))
    config.protocol = PROTOCOL_IPV6;
}

gint callback_options_delete (GtkWidget *widget, GdkEvent *event, gpointer data)
{
  return FALSE;
}

void callback_options_destroy (GtkWidget *widget, gpointer data)
{
  if (((struct s_options_data *) data) != NULL)
    free (((struct s_options_data *) data));
}

void callback_options_ok (GtkButton *button, gpointer data)
{
  preferences_accept ((struct s_options_data *) data);
  gtk_signal_emit_by_name (GTK_OBJECT (((struct s_options_data *) data)->dialog), "destroy");
}

void options_dialog (GtkWidget *widget, gpointer dummy)
{
  GtkWidget *mainbox;
  GtkWidget *frame;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *button;
  GtkTooltips *tooltips;
  GSList *protogroup;
  struct s_options_data *data;

  if ((data = (struct s_options_data *) malloc (sizeof(struct s_options_data))) == NULL) {
    perror ("malloc");
    return;
  }
  memset (data, 0, sizeof (struct s_options_data));

  /* Create a new dialog box */
  data->dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  
  gtk_grab_add (data->dialog);
  gtk_window_set_title (GTK_WINDOW (data->dialog), _("Preferences"));
  gtk_window_set_resizable (GTK_WINDOW (data->dialog), FALSE);

  /* Connect the appropriate signal handlers */
  g_signal_connect (G_OBJECT (data->dialog), "delete_event",
		    G_CALLBACK (callback_options_delete), (gpointer) data);
  g_signal_connect (G_OBJECT (data->dialog), "destroy",
		    G_CALLBACK (callback_options_destroy), (gpointer) data);

  mainbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (mainbox);
  gtk_container_add (GTK_CONTAINER (data->dialog), mainbox);

  frame = gtk_frame_new (_("Send Options"));
  gtk_widget_show (frame);
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_container_add (GTK_CONTAINER (mainbox), frame);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  data->send_dotfiles = gtk_check_button_new_with_mnemonic (_("_Show Dot Files"));
  gtk_widget_show (data->send_dotfiles);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, data->send_dotfiles, _("Display files and directories starting with a dot"), NULL);
  gtk_container_add (GTK_CONTAINER (vbox), data->send_dotfiles);

  data->send_closewindow = gtk_check_button_new_with_mnemonic (_("Close Send _Window"));
  gtk_widget_show (data->send_closewindow);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, data->send_closewindow, _("Close the send window after successful transmission"), NULL);
  gtk_container_add (GTK_CONTAINER (vbox), data->send_closewindow);

  frame = gtk_frame_new (_("Receive Options"));
  gtk_widget_show (frame);
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_container_add (GTK_CONTAINER (mainbox), frame);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
  gtk_container_add (GTK_CONTAINER (frame), hbox);

  data->rec_keepfiles = gtk_check_button_new_with_mnemonic (_("_Keep Files"));
  gtk_widget_show (data->rec_keepfiles);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, data->rec_keepfiles, _("Keep files in the receive queue after receiving"), NULL);
  gtk_container_add (GTK_CONTAINER (hbox), data->rec_keepfiles);

  frame = gtk_frame_new (_("Protocol Options"));
  gtk_widget_show (frame);
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_container_add (GTK_CONTAINER (mainbox), frame);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  data->proto_auto = gtk_radio_button_new_with_mnemonic (NULL, _("_Automatic"));
  gtk_widget_show (data->proto_auto);
  gtk_container_add (GTK_CONTAINER (vbox), data->proto_auto);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, data->proto_auto, _("Let sendfile and sendmsg decide which version of IP to use"), NULL);
  protogroup = gtk_radio_button_get_group (GTK_RADIO_BUTTON (data->proto_auto));
  
  data->proto_ipv4 = gtk_radio_button_new_with_mnemonic (protogroup, _("Always IPv_4"));
  gtk_widget_show (data->proto_ipv4);
  gtk_container_add (GTK_CONTAINER (vbox), data->proto_ipv4);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, data->proto_ipv4, _("Lock sendfile and sendmsg to IP version 4"), NULL);
  protogroup = gtk_radio_button_get_group (GTK_RADIO_BUTTON (data->proto_auto));
  
  data->proto_ipv6 = gtk_radio_button_new_with_mnemonic (protogroup, _("Always IPv_6"));
  gtk_widget_show (data->proto_ipv6);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, data->proto_ipv6, _("Lock sendfile and sendmsg to IP version 6"), NULL);
  gtk_container_add (GTK_CONTAINER (vbox), data->proto_ipv6);

  preferences_init (data);

  /* horizontal box containing the buttons */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER(hbox), 5);
  gtk_box_pack_end (GTK_BOX(mainbox), hbox, TRUE, TRUE, 0);
  gtk_box_set_homogeneous (GTK_BOX(hbox), TRUE);
  gtk_box_set_spacing (GTK_BOX(hbox), 5);
  gtk_widget_show (hbox);

  button = gtk_button_new_from_stock ("gtk-cancel");
  g_signal_connect_swapped (G_OBJECT (button), "clicked",
			   G_CALLBACK (gtk_widget_destroy), (gpointer) data->dialog);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, button, _("Cancel previously made settings and return to main program"), NULL);
  gtk_container_add (GTK_CONTAINER(hbox), button);
  gtk_widget_show (button);

  button = gtk_button_new_from_stock ("gtk-ok");
  g_signal_connect (G_OBJECT (button), "clicked",
		    G_CALLBACK (callback_options_ok), (gpointer) data);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, button, _("Accept settings and return to main program"), NULL);
  gtk_container_add (GTK_CONTAINER(hbox), button);
  gtk_widget_show (button);

  gtk_widget_show (data->dialog);
}
