/*
 * Copyright (c) 2022-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include "arm_gemm.hpp"
#include "../../utils.hpp"
#include "../../bfloat.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void sme2_gemv_bf16fp32_dot_16VL (
    const bfloat16 *A_ptr, const bfloat16 *B_ptr, float *output_ptr,
    size_t N, size_t K,
    const float *bias, Activation act, bool
)
{
    struct KernelArgs {
        float maxval = static_cast<float>(std::numeric_limits<float>::infinity());
        float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
        const bfloat16 *B_ptr = {};
        size_t output_offset = {};
        unsigned int input_initial_col = {};
    } ka;

    unsigned long flags=0;
    ka.B_ptr = B_ptr;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "ptrue p8.b\n"
      ".inst 0xd503477f  // SMSTART ZA\n"
      "mov x9, #0x0\n"
      "cntw x28, ALL, MUL #4\n"
      "mov x27, %x[B_ptr]\n"
      "add x26, %x[N], x28\n"
      "mov x25, %x[output_ptr]\n"
      "sub x26, x26, #0x1\n"
      "ptrue p1.b\n"
      "udiv x26, x26, x28\n"
      ".inst 0x25207811  // ptrue pn9.b\n"
      "add x22, x26, #0x3\n"
      "mov x21, #0x1\n"
      "and x22, x22, #0xfffffffffffffffc\n"
      "mul x22, x22, x28\n"
      "mul x22, x22, %x[K]\n"
      "lsl x22, x22, #0x1\n"
      "1:"  // RHS size check loop
      "cmp x22, #0x200000\n"
      "blt 2f\n"
      "tbnz x22, #0, 3f\n"
      "lsr x22, x22, #0x1\n"
      "lsl x21, x21, #0x1\n"
      "b 1b\n"
      "2:"  // RHS do prefetch
      "lsl x20, x22, #0x26\n"
      "sub x21, x21, #0x1\n"
      "lsl x21, x21, #0x16\n"
      "orr x22, x22, x20\n"
      "orr x22, x22, x21\n"
      ".inst 0xf8b64b7a  // rprfm pldonce, x22, [x27]\n"
      "3:"  // RHS prefetch exit
      "mov x24, %x[bias]\n"
      "4:"  // Column loop
      "cmp x26, #0x4\n"
      "bge 28f\n"
      "cmp x26, #0x2\n"
      "bgt 20f\n"
      "beq 12f\n"
      "mov x23, %x[A_ptr]\n"
      "lsl x21, %x[K], #0x1\n"
      "mov x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      ".inst 0x25b467f0  // whilelt p8.s, XZR, x20, VLx4\n"
      "cbz x24, 5f\n"
      ".inst 0xa040c714  // ld1w { z20.s-z23.s }, pn9.b/Z, [x24]\n"
      ".inst 0xc0042e80  // mova za.d[x9, #0], { z20.d-z23.d }\n"
      "b 6f\n"
      "5:"  // Width 1: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "6:"  // Width 1: setup done
      "cmp x22, #0x8\n"
      "ble 8f\n"
      "7:"  // Width 1: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      "ld1rqh { z2.h }, p0/Z, [x23]\n"
      "sub x22, x22, #0x8\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa040a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      "cmp x22, #0x8\n"
      ".inst 0xa040a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc152b398  // bfdot za.s[x9, 0], { z28.h-z31.h }, z2.h[0]\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc152b618  // bfdot za.s[x9, 0], { z16.h-z19.h }, z2.h[1]\n"
      ".inst 0xc152bb18  // bfdot za.s[x9, 0], { z24.h-z27.h }, z2.h[2]\n"
      ".inst 0xc152bd98  // bfdot za.s[x9, 0], { z12.h-z15.h }, z2.h[3]\n"
      "bgt 7b\n"
      "8:"  // Width 1: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x2\n"
      "ld1rqh { z3.h }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b198  // bfdot za.s[x9, 0], { z12.h-z15.h }, z3.h[0]\n"
      "ble 9f\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x2\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b498  // bfdot za.s[x9, 0], { z4.h-z7.h }, z3.h[1]\n"
      "ble 9f\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x2\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b998  // bfdot za.s[x9, 0], { z12.h-z15.h }, z3.h[2]\n"
      "ble 9f\n"
      ".inst 0xa040a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153bd18  // bfdot za.s[x9, 0], { z8.h-z11.h }, z3.h[3]\n"
      "9:"  // Width 1: Multiply loop: multiply skip
      "tbz %x[flags], #1, 10f\n"
      "add x21, %x[args_ptr], %[offset_min]\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      ".inst 0xc0062c00  // mova { z0.d-z3.d }, za.d[x9, #0]\n"
      "ld1rw { z23.s }, p1/Z, [x21]\n"
      "ld1rw { z22.s }, p1/Z, [x20]\n"
      ".inst 0xc1b6cae0  // fclamp { z0.s-z3.s }, z23.s, z22.s\n"
      ".inst 0xa060c320  // st1w { z0.s-z3.s }, p8, [x25]\n"
      "addvl x25, x25, #4\n"
      "b 11f\n"
      "10:"  // Width 1: No activation
      ".inst 0xc0062c00  // mova { z0.d-z3.d }, za.d[x9, #0]\n"
      ".inst 0xa060c320  // st1w { z0.s-z3.s }, p8, [x25]\n"
      "addvl x25, x25, #4\n"
      "11:"  // Width 1: Output done
      "b 36f\n"
      "12:"  // Width 2
      "mov x23, %x[A_ptr]\n"
      "lsl x21, %x[K], #0x1\n"
      "sub x20, %x[N], x28\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      ".inst 0x25b467f0  // whilelt p8.s, XZR, x20, VLx4\n"
      "cbz x24, 13f\n"
      ".inst 0xa040c71c  // ld1w { z28.s-z31.s }, pn9.b/Z, [x24]\n"
      ".inst 0xa041c704  // ld1w { z4.s-z7.s }, pn9.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xc0042f80  // mova za.d[x9, #0], { z28.d-z31.d }\n"
      ".inst 0xc0042c81  // mova za.d[x9, #1], { z4.d-z7.d }\n"
      "b 14f\n"
      "13:"  // Width 2: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "14:"  // Width 2: setup done
      "cmp x22, #0x8\n"
      "ble 16f\n"
      "15:"  // Width 2: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqh { z1.h }, p0/Z, [x23]\n"
      "cmp x22, #0x8\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa041a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xa040a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc151b398  // bfdot za.s[x9, 0], { z28.h-z31.h }, z1.h[0]\n"
      ".inst 0xa041a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc151b199  // bfdot za.s[x9, 1], { z12.h-z15.h }, z1.h[0]\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      ".inst 0xa041a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xa040a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc151b618  // bfdot za.s[x9, 0], { z16.h-z19.h }, z1.h[1]\n"
      ".inst 0xa041a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc151b719  // bfdot za.s[x9, 1], { z24.h-z27.h }, z1.h[1]\n"
      ".inst 0xc151bb98  // bfdot za.s[x9, 0], { z28.h-z31.h }, z1.h[2]\n"
      ".inst 0xc151b999  // bfdot za.s[x9, 1], { z12.h-z15.h }, z1.h[2]\n"
      ".inst 0xc151bd18  // bfdot za.s[x9, 0], { z8.h-z11.h }, z1.h[3]\n"
      ".inst 0xc151be99  // bfdot za.s[x9, 1], { z20.h-z23.h }, z1.h[3]\n"
      "bgt 15b\n"
      "16:"  // Width 2: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x2\n"
      "ld1rqh { z3.h }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa041a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b218  // bfdot za.s[x9, 0], { z16.h-z19.h }, z3.h[0]\n"
      ".inst 0xc153b399  // bfdot za.s[x9, 1], { z28.h-z31.h }, z3.h[0]\n"
      "ble 17f\n"
      ".inst 0xa040a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x2\n"
      ".inst 0xa041a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b698  // bfdot za.s[x9, 0], { z20.h-z23.h }, z3.h[1]\n"
      ".inst 0xc153b619  // bfdot za.s[x9, 1], { z16.h-z19.h }, z3.h[1]\n"
      "ble 17f\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x2\n"
      ".inst 0xa041a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b898  // bfdot za.s[x9, 0], { z4.h-z7.h }, z3.h[2]\n"
      ".inst 0xc153ba19  // bfdot za.s[x9, 1], { z16.h-z19.h }, z3.h[2]\n"
      "ble 17f\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      ".inst 0xa041a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153bf98  // bfdot za.s[x9, 0], { z28.h-z31.h }, z3.h[3]\n"
      ".inst 0xc153bd99  // bfdot za.s[x9, 1], { z12.h-z15.h }, z3.h[3]\n"
      "17:"  // Width 2: Multiply loop: multiply skip
      "tbz %x[flags], #1, 18f\n"
      "add x21, %x[args_ptr], %[offset_min]\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      ".inst 0xc0062c04  // mova { z4.d-z7.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c28  // mova { z8.d-z11.d }, za.d[x9, #1]\n"
      "ld1rw { z17.s }, p1/Z, [x21]\n"
      "ld1rw { z23.s }, p1/Z, [x20]\n"
      ".inst 0xc1b7ca24  // fclamp { z4.s-z7.s }, z17.s, z23.s\n"
      ".inst 0xc1b7ca28  // fclamp { z8.s-z11.s }, z17.s, z23.s\n"
      ".inst 0xa060c724  // st1w { z4.s-z7.s }, pn9.b, [x25]\n"
      ".inst 0xa061c328  // st1w { z8.s-z11.s }, p8, [x25, #0x4, MUL VL]\n"
      "addvl x25, x25, #8\n"
      "b 19f\n"
      "18:"  // Width 2: No activation
      ".inst 0xc0062c08  // mova { z8.d-z11.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c30  // mova { z16.d-z19.d }, za.d[x9, #1]\n"
      ".inst 0xa060c728  // st1w { z8.s-z11.s }, pn9.b, [x25]\n"
      ".inst 0xa061c330  // st1w { z16.s-z19.s }, p8, [x25, #0x4, MUL VL]\n"
      "addvl x25, x25, #8\n"
      "19:"  // Width 2: Output done
      "b 36f\n"
      "20:"  // Width 3
      "mov x20, #0x2\n"
      "mov x23, %x[A_ptr]\n"
      "lsl x21, %x[K], #0x1\n"
      "msub x20, x28, x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      ".inst 0x25b467f0  // whilelt p8.s, XZR, x20, VLx4\n"
      "cbz x24, 21f\n"
      ".inst 0xa040c708  // ld1w { z8.s-z11.s }, pn9.b/Z, [x24]\n"
      ".inst 0xa041c700  // ld1w { z0.s-z3.s }, pn9.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xa042c704  // ld1w { z4.s-z7.s }, pn9.b/Z, [x24, #0x8, MUL VL]\n"
      ".inst 0xc0042d00  // mova za.d[x9, #0], { z8.d-z11.d }\n"
      ".inst 0xc0042c01  // mova za.d[x9, #1], { z0.d-z3.d }\n"
      ".inst 0xc0042c82  // mova za.d[x9, #2], { z4.d-z7.d }\n"
      "b 22f\n"
      "21:"  // Width 3: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "22:"  // Width 3: setup done
      "cmp x22, #0x8\n"
      "ble 24f\n"
      "23:"  // Width 3: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqh { z3.h }, p0/Z, [x23]\n"
      "cmp x22, #0x8\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b198  // bfdot za.s[x9, 0], { z12.h-z15.h }, z3.h[0]\n"
      ".inst 0xa040a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc153b119  // bfdot za.s[x9, 1], { z8.h-z11.h }, z3.h[0]\n"
      ".inst 0xa041a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc153b09a  // bfdot za.s[x9, 2], { z4.h-z7.h }, z3.h[0]\n"
      ".inst 0xa042a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc153b618  // bfdot za.s[x9, 0], { z16.h-z19.h }, z3.h[1]\n"
      ".inst 0xa041a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc153b699  // bfdot za.s[x9, 1], { z20.h-z23.h }, z3.h[1]\n"
      ".inst 0xa042a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b71a  // bfdot za.s[x9, 2], { z24.h-z27.h }, z3.h[1]\n"
      ".inst 0xa040a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27]\n"
      ".inst 0xa041a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc153b998  // bfdot za.s[x9, 0], { z12.h-z15.h }, z3.h[2]\n"
      ".inst 0xa042a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153bb99  // bfdot za.s[x9, 1], { z28.h-z31.h }, z3.h[2]\n"
      ".inst 0xc153b91a  // bfdot za.s[x9, 2], { z8.h-z11.h }, z3.h[2]\n"
      ".inst 0xc153bc98  // bfdot za.s[x9, 0], { z4.h-z7.h }, z3.h[3]\n"
      ".inst 0xc153be99  // bfdot za.s[x9, 1], { z20.h-z23.h }, z3.h[3]\n"
      ".inst 0xc153bf1a  // bfdot za.s[x9, 2], { z24.h-z27.h }, z3.h[3]\n"
      "bgt 23b\n"
      "24:"  // Width 3: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x2\n"
      "ld1rqh { z3.h }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa041a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b298  // bfdot za.s[x9, 0], { z20.h-z23.h }, z3.h[0]\n"
      ".inst 0xc153b199  // bfdot za.s[x9, 1], { z12.h-z15.h }, z3.h[0]\n"
      ".inst 0xc153b09a  // bfdot za.s[x9, 2], { z4.h-z7.h }, z3.h[0]\n"
      "ble 25f\n"
      ".inst 0xa040a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x2\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b698  // bfdot za.s[x9, 0], { z20.h-z23.h }, z3.h[1]\n"
      ".inst 0xc153b519  // bfdot za.s[x9, 1], { z8.h-z11.h }, z3.h[1]\n"
      ".inst 0xc153b61a  // bfdot za.s[x9, 2], { z16.h-z19.h }, z3.h[1]\n"
      "ble 25f\n"
      ".inst 0xa040a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x2\n"
      ".inst 0xa041a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153bb98  // bfdot za.s[x9, 0], { z28.h-z31.h }, z3.h[2]\n"
      ".inst 0xc153bb19  // bfdot za.s[x9, 1], { z24.h-z27.h }, z3.h[2]\n"
      ".inst 0xc153b99a  // bfdot za.s[x9, 2], { z12.h-z15.h }, z3.h[2]\n"
      "ble 25f\n"
      ".inst 0xa040a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27]\n"
      ".inst 0xa041a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153bd18  // bfdot za.s[x9, 0], { z8.h-z11.h }, z3.h[3]\n"
      ".inst 0xc153bf99  // bfdot za.s[x9, 1], { z28.h-z31.h }, z3.h[3]\n"
      ".inst 0xc153bd9a  // bfdot za.s[x9, 2], { z12.h-z15.h }, z3.h[3]\n"
      "25:"  // Width 3: Multiply loop: multiply skip
      "tbz %x[flags], #1, 26f\n"
      "add x21, %x[args_ptr], %[offset_min]\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      ".inst 0xc0062c08  // mova { z8.d-z11.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c2c  // mova { z12.d-z15.d }, za.d[x9, #1]\n"
      "ld1rw { z21.s }, p1/Z, [x21]\n"
      ".inst 0xc0062c50  // mova { z16.d-z19.d }, za.d[x9, #2]\n"
      "ld1rw { z20.s }, p1/Z, [x20]\n"
      ".inst 0xc1b4caa8  // fclamp { z8.s-z11.s }, z21.s, z20.s\n"
      ".inst 0xc1b4caac  // fclamp { z12.s-z15.s }, z21.s, z20.s\n"
      ".inst 0xc1b4cab0  // fclamp { z16.s-z19.s }, z21.s, z20.s\n"
      ".inst 0xa060c728  // st1w { z8.s-z11.s }, pn9.b, [x25]\n"
      ".inst 0xa061c72c  // st1w { z12.s-z15.s }, pn9.b, [x25, #0x4, MUL VL]\n"
      ".inst 0xa062c330  // st1w { z16.s-z19.s }, p8, [x25, #0x8, MUL VL]\n"
      "addvl x25, x25, #12\n"
      "b 27f\n"
      "26:"  // Width 3: No activation
      ".inst 0xc0062c04  // mova { z4.d-z7.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c2c  // mova { z12.d-z15.d }, za.d[x9, #1]\n"
      ".inst 0xc0062c5c  // mova { z28.d-z31.d }, za.d[x9, #2]\n"
      ".inst 0xa060c724  // st1w { z4.s-z7.s }, pn9.b, [x25]\n"
      ".inst 0xa061c72c  // st1w { z12.s-z15.s }, pn9.b, [x25, #0x4, MUL VL]\n"
      ".inst 0xa062c33c  // st1w { z28.s-z31.s }, p8, [x25, #0x8, MUL VL]\n"
      "addvl x25, x25, #12\n"
      "27:"  // Width 3: Output done
      "b 36f\n"
      "28:"  // Width 4
      "mov x20, #0x3\n"
      "mov x23, %x[A_ptr]\n"
      "lsl x21, %x[K], #0x1\n"
      "msub x20, x28, x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      ".inst 0x25b467f0  // whilelt p8.s, XZR, x20, VLx4\n"
      "cbz x24, 29f\n"
      ".inst 0xa040c704  // ld1w { z4.s-z7.s }, pn9.b/Z, [x24]\n"
      ".inst 0xa041c70c  // ld1w { z12.s-z15.s }, pn9.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xa042c71c  // ld1w { z28.s-z31.s }, pn9.b/Z, [x24, #0x8, MUL VL]\n"
      ".inst 0xa043c710  // ld1w { z16.s-z19.s }, pn9.b/Z, [x24, #0xc, MUL VL]\n"
      ".inst 0xc0042c80  // mova za.d[x9, #0], { z4.d-z7.d }\n"
      "addvl x24, x24, #16\n"
      ".inst 0xc0042d81  // mova za.d[x9, #1], { z12.d-z15.d }\n"
      ".inst 0xc0042f82  // mova za.d[x9, #2], { z28.d-z31.d }\n"
      ".inst 0xc0042e03  // mova za.d[x9, #3], { z16.d-z19.d }\n"
      "b 30f\n"
      "29:"  // Width 4: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "30:"  // Width 4: setup done
      "cmp x22, #0x8\n"
      "ble 32f\n"
      "31:"  // Width 4: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqh { z3.h }, p0/Z, [x23]\n"
      "cmp x22, #0x8\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa041a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc153b198  // bfdot za.s[x9, 0], { z12.h-z15.h }, z3.h[0]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b299  // bfdot za.s[x9, 1], { z20.h-z23.h }, z3.h[0]\n"
      ".inst 0xa040a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc153b21a  // bfdot za.s[x9, 2], { z16.h-z19.h }, z3.h[0]\n"
      ".inst 0xa041a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc153b11b  // bfdot za.s[x9, 3], { z8.h-z11.h }, z3.h[0]\n"
      ".inst 0xa042a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc153b718  // bfdot za.s[x9, 0], { z24.h-z27.h }, z3.h[1]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b599  // bfdot za.s[x9, 1], { z12.h-z15.h }, z3.h[1]\n"
      ".inst 0xa040a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc153b51a  // bfdot za.s[x9, 2], { z8.h-z11.h }, z3.h[1]\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc153b49b  // bfdot za.s[x9, 3], { z4.h-z7.h }, z3.h[1]\n"
      ".inst 0xa042a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc153bb18  // bfdot za.s[x9, 0], { z24.h-z27.h }, z3.h[2]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b919  // bfdot za.s[x9, 1], { z8.h-z11.h }, z3.h[2]\n"
      ".inst 0xa040a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27]\n"
      ".inst 0xc153ba1a  // bfdot za.s[x9, 2], { z16.h-z19.h }, z3.h[2]\n"
      ".inst 0xa041a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xc153b89b  // bfdot za.s[x9, 3], { z4.h-z7.h }, z3.h[2]\n"
      ".inst 0xa042a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc153bd18  // bfdot za.s[x9, 0], { z8.h-z11.h }, z3.h[3]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153bf99  // bfdot za.s[x9, 1], { z28.h-z31.h }, z3.h[3]\n"
      ".inst 0xc153bd9a  // bfdot za.s[x9, 2], { z12.h-z15.h }, z3.h[3]\n"
      ".inst 0xc153be9b  // bfdot za.s[x9, 3], { z20.h-z23.h }, z3.h[3]\n"
      "bgt 31b\n"
      "32:"  // Width 4: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      ".inst 0xa040a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x2\n"
      "ld1rqh { z3.h }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      ".inst 0xa041a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a77d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc153b218  // bfdot za.s[x9, 0], { z16.h-z19.h }, z3.h[0]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b199  // bfdot za.s[x9, 1], { z12.h-z15.h }, z3.h[0]\n"
      ".inst 0xc153b39a  // bfdot za.s[x9, 2], { z28.h-z31.h }, z3.h[0]\n"
      ".inst 0xc153b29b  // bfdot za.s[x9, 3], { z20.h-z23.h }, z3.h[0]\n"
      "ble 33f\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x2\n"
      ".inst 0xa041a765  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a779  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc153b598  // bfdot za.s[x9, 0], { z12.h-z15.h }, z3.h[1]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b499  // bfdot za.s[x9, 1], { z4.h-z7.h }, z3.h[1]\n"
      ".inst 0xc153b71a  // bfdot za.s[x9, 2], { z24.h-z27.h }, z3.h[1]\n"
      ".inst 0xc153b69b  // bfdot za.s[x9, 3], { z20.h-z23.h }, z3.h[1]\n"
      "ble 33f\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      "subs x22, x22, #0x2\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc153b998  // bfdot za.s[x9, 0], { z12.h-z15.h }, z3.h[2]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153b919  // bfdot za.s[x9, 1], { z8.h-z11.h }, z3.h[2]\n"
      ".inst 0xc153ba9a  // bfdot za.s[x9, 2], { z20.h-z23.h }, z3.h[2]\n"
      ".inst 0xc153ba1b  // bfdot za.s[x9, 3], { z16.h-z19.h }, z3.h[2]\n"
      "ble 33f\n"
      ".inst 0xa040a76d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x27]\n"
      ".inst 0xa041a769  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x27, #0x4, MUL VL]\n"
      ".inst 0xa042a775  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x27, #0x8, MUL VL]\n"
      ".inst 0xa043a771  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x27, #0xc, MUL VL]\n"
      ".inst 0xc153bd98  // bfdot za.s[x9, 0], { z12.h-z15.h }, z3.h[3]\n"
      "addvl x27, x27, #16\n"
      ".inst 0xc153bd19  // bfdot za.s[x9, 1], { z8.h-z11.h }, z3.h[3]\n"
      ".inst 0xc153be9a  // bfdot za.s[x9, 2], { z20.h-z23.h }, z3.h[3]\n"
      ".inst 0xc153be1b  // bfdot za.s[x9, 3], { z16.h-z19.h }, z3.h[3]\n"
      "33:"  // Width 4: Multiply loop: multiply skip
      "tbz %x[flags], #1, 34f\n"
      "add x21, %x[args_ptr], %[offset_min]\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      ".inst 0xc0062c04  // mova { z4.d-z7.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c20  // mova { z0.d-z3.d }, za.d[x9, #1]\n"
      "ld1rw { z21.s }, p1/Z, [x21]\n"
      ".inst 0xc0062c4c  // mova { z12.d-z15.d }, za.d[x9, #2]\n"
      "ld1rw { z20.s }, p1/Z, [x20]\n"
      ".inst 0xc0062c70  // mova { z16.d-z19.d }, za.d[x9, #3]\n"
      ".inst 0xc1b4caa4  // fclamp { z4.s-z7.s }, z21.s, z20.s\n"
      ".inst 0xc1b4caa0  // fclamp { z0.s-z3.s }, z21.s, z20.s\n"
      ".inst 0xc1b4caac  // fclamp { z12.s-z15.s }, z21.s, z20.s\n"
      ".inst 0xc1b4cab0  // fclamp { z16.s-z19.s }, z21.s, z20.s\n"
      ".inst 0xa060c724  // st1w { z4.s-z7.s }, pn9.b, [x25]\n"
      ".inst 0xa061c720  // st1w { z0.s-z3.s }, pn9.b, [x25, #0x4, MUL VL]\n"
      ".inst 0xa062c72c  // st1w { z12.s-z15.s }, pn9.b, [x25, #0x8, MUL VL]\n"
      ".inst 0xa063c330  // st1w { z16.s-z19.s }, p8, [x25, #0xc, MUL VL]\n"
      "addvl x25, x25, #16\n"
      "b 35f\n"
      "34:"  // Width 4: No activation
      ".inst 0xc0062c0c  // mova { z12.d-z15.d }, za.d[x9, #0]\n"
      ".inst 0xc0062c20  // mova { z0.d-z3.d }, za.d[x9, #1]\n"
      ".inst 0xc0062c50  // mova { z16.d-z19.d }, za.d[x9, #2]\n"
      ".inst 0xc0062c64  // mova { z4.d-z7.d }, za.d[x9, #3]\n"
      ".inst 0xa060c72c  // st1w { z12.s-z15.s }, pn9.b, [x25]\n"
      ".inst 0xa061c720  // st1w { z0.s-z3.s }, pn9.b, [x25, #0x4, MUL VL]\n"
      ".inst 0xa062c730  // st1w { z16.s-z19.s }, pn9.b, [x25, #0x8, MUL VL]\n"
      ".inst 0xa063c324  // st1w { z4.s-z7.s }, p8, [x25, #0xc, MUL VL]\n"
      "addvl x25, x25, #16\n"
      "35:"  // Width 4: Output done
      "subs x26, x26, #0x4\n"
      "sub %x[N], %x[N], x28, LSL #2\n"
      "bgt 4b\n"
      "36:"  // Exit
      ".inst 0xd503467f  // SMSTOP\n"
      "ptrue p8.b\n"
      : [N] "+&r" (N)
      : [A_ptr] "r" (A_ptr), [B_ptr] "r" (B_ptr), [K] "r" (K), [args_ptr] "r" (&ka), [bias] "r" (bias), [flags] "r" (flags), [offset_max] "I" (offsetof(KernelArgs, maxval)), [offset_min] "I" (offsetof(KernelArgs, minval)), [output_ptr] "r" (output_ptr)
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x9", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
