#!/bin/bash
# Creates several USA zones and charts from the FAA airports data base.

# FAA data base directory.
FAA_DB=../../../FAA-airports-database

# Destination dir. of the creates sceneries:
ZONES_DIR=../../objects/zones

# Destination dir. of the created charts:
CHARTS_DIR=../../doc/charts

# WMM coefficients:
WMM_COF=../../objects/WMM.COF


# Generates PDF of a navigation chart.
# Parameters:
# $1: scenery file.
# $2: destination PDF file name without extension.
# $3: title.
# $4: central latitude.
# $5: central longitude.
# $6: scale.
# $7, $8, $9: optional further parameters for the chart.exe program.
function doChart
{
	local dst
	dst=$2
	echo "Generating chart $dst.pdf ..."
	../chart/chart.exe --wmm-cof $WMM_COF \
		--scene "$1" \
		--title "$3" \
		--olat $4 \
		--olon $5 \
		--scale $6 \
		$7 $8 $9 > $dst.ps || exit 1
	ps2pdf -sPAPERSIZE=a4 $dst.ps $dst.pdf || exit 1
	rm $dst.ps
}


function newyork()
{
	local lat_min lat_max lon_min lon_max scenery dst
	lat_min=40N
	lat_max=45N
	lon_min=75W
	lon_max=70W
	scenery=$ZONES_DIR/usa/newyork.txt
	{
		echo "# New York area scenery."
		echo "# Created: " $(date +%Y-%m-%d)
		echo "# Latitude range: [$lat_min, $lat_max["
		echo "# Longitude range: [$lon_min, $lon_max["
		echo "# 2017-10-04 Checked against https://nfdc.faa.gov/xwiki/bin/view/NFDC/56DaySub-2017-03-02"
		echo "# \"56 Day NASR Subscription - Effective March 2, 2017\"."
		echo
		echo "# New York, JFK 04L:"
		echo "TEAM1_LOC 40-37-19.2759N 073-47-08.1083W 000 031"
		echo "# Boston, BOS 22R:"
		echo "TEAM2_LOC 42-22-41.8759N 071-00-16.2499W 000 199"
		echo "GROUND_COLOR #305030"
		./faaairports.exe \
			--lat-min $lat_min \
			--lat-max $lat_max \
			--lon-min $lon_min \
			--lon-max $lon_max \
			--src-dir $FAA_DB
	} > $scenery

	dst=$CHARTS_DIR/usa/newyork
	if [ ! -d $dst ]; then
		mkdir $dst || exit 1
	fi
	doChart $scenery $dst/new-york "New York area" 40-40N 73-45W 250000
	doChart $scenery $dst/boston "Boston area" 42-22N 71-05W 250000
}


function dallas()
{
	local lat_min lat_max lon_min lon_max scenery dst
	lat_min=30N
	lat_max=35N
	lon_min=100W
	lon_max=95W
	scenery=../../objects/zones/usa/dallas.txt
	{
		echo "# Dallas area scenery."
		echo "# Created: " $(date +%Y-%m-%d)
		echo "# Latitude range: [$lat_min, $lat_max["
		echo "# Longitude range: [$lon_min, $lon_max["
		echo "# 2017-10-04 Checked against https://nfdc.faa.gov/xwiki/bin/view/NFDC/56DaySub-2017-03-02"
		echo "# \"56 Day NASR Subscription - Effective March 2, 2017\"."
		echo
		echo "TEAM1_LOC 32-58-40.4N 096-50-25.75W 645 160"
		echo "TEAM2_LOC 34-17-36.089N 097-01-25.366W 762.0 040"
		echo "GROUND_COLOR #305030"
		echo
		echo "# Addison:"
		echo "FEATURE ../../features/tower.obv 32-58-04.800N 096-50-16.800W 644 0"
		echo "FEATURE ../../features/man.obv 32-58-29.000N 096-50-20.000W 644 90"
		echo
		./faaairports.exe \
			--lat-min $lat_min \
			--lat-max $lat_max \
			--lon-min $lon_min \
			--lon-max $lon_max \
			--src-dir $FAA_DB

		echo
		echo "#"
		echo "# To visit the ACM air museum at the Addison airport, uncomment these lines:"
		echo "#"
		echo "#FEATURE ../../missiles/aim9.obv    32-57-10.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../missiles/sa10.obv    32-57-11.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../missiles/sa11.obv    32-57-12.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../missiles/sa2.obv     32-57-13.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../features/telar1.obv  32-57-14.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../tracer.obv           32-57-15.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../features/hangar.obv  32-57-17.000N 096-49-59.000W 644 290"
		echo "#FEATURE ../../aircraft/c172.obv   32-57-19.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../aircraft/f16.obv    32-57-21.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../aircraft/f18.obv    32-57-23.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../aircraft/il78.obv   32-57-25.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../aircraft/kc135.obv  32-57-27.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../aircraft/mig23.obv  32-57-29.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../aircraft/mig25.obv  32-57-31.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../aircraft/mig29.obv  32-57-33.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../aircraft/su30.obv   32-57-35.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../aircraft/ufo.obv    32-57-37.000N 096-49-59.000W 660 290"
		echo "#FEATURE ../../aircraft/b-747.obv  32-57-39.000N 096-49-59.000W 680 290"
	} > $scenery

	dst=$CHARTS_DIR/usa/dallas
	if [ ! -d $dst ]; then
		mkdir $dst || exit 1
	fi
	# Navigation charts:
	###doChart $scenery $dst/O O 32-30-00.000N 096-40-00.000W 2500000 --no-ils
	doChart $scenery $dst/A A 33-25-00.000N 097-55-00.000W 1000000 --no-ils
	doChart $scenery $dst/B B 33-25-00.000N 095-55-00.000W 1000000 --no-ils
	doChart $scenery $dst/C C 32-15-00.000N 097-55-00.000W 1000000 --no-ils
	doChart $scenery $dst/D D 32-15-00.000N 095-55-00.000W 1000000 --no-ils
	# Instrumental landing:
	doChart $scenery $dst/AA "Wichita Falls Area" 33-59-00N 98-31-30W 250000
	doChart $scenery $dst/AB "Ardmore (KADM, ADM)" 34-15-00N 97-00-00W 250000
	doChart $scenery $dst/BA "Dallas, Addison, Love Field" 32-52-00.000N 096-56-00.000W 250000
	doChart $scenery $dst/BB "Dallas, Ft. Worth (KDFW, DFW)" 32-54-00.000N 097-02-20.000W 100000t
	doChart $scenery $dst/BC "Texas, Paris (KPRX, PRX)" 33-37-00.000N 095-27-20.000W 100000
	doChart $scenery $dst/CA "Waco Regional Airport (KACT, ACT)" 31-40-00N 97-10-00W 250000
	doChart $scenery $dst/DA "Tyler Pounds Regional Airport (KTYR, TYR)" 32-22-00N 95-24-00W 100000hart
	doChart $scenery $dst/CB "Temple, Draughon-Miller (KTPL, TPL)" 31-10-00N 97-25-00W 250000
}


function sfrancisco()
{
	local lat_min lat_max lon_min lon_max scenery
	lat_min=35N
	lat_max=40N
	lon_min=125W
	lon_max=120W
	scenery=$ZONES_DIR/usa/sfrancisco.txt
	{
		echo "# S. Francisco area scenery."
		echo "# Created: " $(date +%Y-%m-%d)
		echo "# Latitude range: [$lat_min, $lat_max["
		echo "# Longitude range: [$lon_min, $lon_max["
		echo "# 2017-10-04 Checked against https://nfdc.faa.gov/xwiki/bin/view/NFDC/56DaySub-2017-03-02"
		echo "# \"56 Day NASR Subscription - Effective March 2, 2017\"."
		echo
		echo "# ?????:"
		echo "TEAM1_LOC 35N 125W 000 031"
		echo "# ?????:"
		echo "TEAM1_LOC 35N 125W 000 031"
		echo "GROUND_COLOR #305030"
		./faaairports.exe \
			--lat-min $lat_min \
			--lat-max $lat_max \
			--lon-min $lon_min \
			--lon-max $lon_max \
			--src-dir $FAA_DB
	} > $scenery
}


function losangeles()
{
	local lat_min lat_max lon_min lon_max scenery dst
	lat_min=30N
	lat_max=35N
	lon_min=120W
	lon_max=115W
	scenery=$ZONES_DIR/usa/losangeles.txt
	{
		echo "# Los Angeles area scenery."
		echo "# Created: " $(date +%Y-%m-%d)
		echo "# Latitude range: [$lat_min, $lat_max["
		echo "# Longitude range: [$lon_min, $lon_max["
		echo "# 2017-10-04 Checked against https://nfdc.faa.gov/xwiki/bin/view/NFDC/56DaySub-2017-03-02"
		echo "# \"56 Day NASR Subscription - Effective March 2, 2017\"."
		echo
		echo "# ?????:"
		echo "TEAM1_LOC 30N 120W 000 031"
		echo "# ?????:"
		echo "TEAM1_LOC 30N 120W 000 031"
		echo "GROUND_COLOR #305030"
		./faaairports.exe \
			--lat-min $lat_min \
			--lat-max $lat_max \
			--lon-min $lon_min \
			--lon-max $lon_max \
			--src-dir $FAA_DB
	} > $scenery

	dst=$CHARTS_DIR/usa/losangeles
	if [ ! -d $dst ]; then
		mkdir $dst || exit 1
	fi
	doChart $scenery $dst/los-angeles-area "Los Angeles area" 34-01N 118-18W 250000
	doChart $scenery $dst/los-angeles "Los Angeles airport" 33-57N 118-24W 100000
}


function lasvegaswest()
{
	local lat_min lat_max lon_min lon_max scenery
	lat_min=35N
	lat_max=40N
	lon_min=120W
	lon_max=115W
	scenery=$ZONES_DIR/usa/lasvegaswest.txt
	{
		echo "# Las Vegas west area scenery."
		echo "# Created: " $(date +%Y-%m-%d)
		echo "# Latitude range: [$lat_min, $lat_max["
		echo "# Longitude range: [$lon_min, $lon_max["
		echo "# 2017-10-04 Checked against https://nfdc.faa.gov/xwiki/bin/view/NFDC/56DaySub-2017-03-02"
		echo "# \"56 Day NASR Subscription - Effective March 2, 2017\"."
		echo
		echo "# ?????:"
		echo "TEAM1_LOC 35N 120W 000 031"
		echo "# ?????:"
		echo "TEAM1_LOC 35N 120W 000 031"
		echo "GROUND_COLOR #305030"
		./faaairports.exe \
			--lat-min $lat_min \
			--lat-max $lat_max \
			--lon-min $lon_min \
			--lon-max $lon_max \
			--src-dir $FAA_DB
	} > $scenery
}


function lasvegaseast()
{
	local lat_min lat_max lon_min lon_max scenery
	lat_min=35N
	lat_max=40N
	lon_min=115W
	lon_max=110W
	scenery=$ZONES_DIR/usa/lasvegaseast.txt
	{
		echo "# Las Vegas east area scenery."
		echo "# Created: " $(date +%Y-%m-%d)
		echo "# Latitude range: [$lat_min, $lat_max["
		echo "# Longitude range: [$lon_min, $lon_max["
		echo "# 2017-10-04 Checked against https://nfdc.faa.gov/xwiki/bin/view/NFDC/56DaySub-2017-03-02"
		echo "# \"56 Day NASR Subscription - Effective March 2, 2017\"."
		echo
		echo "# ?????:"
		echo "TEAM1_LOC 35N 115W 000 031"
		echo "# ?????:"
		echo "TEAM1_LOC 35N 115W 000 031"
		echo "GROUND_COLOR #305030"
		./faaairports.exe \
			--lat-min $lat_min \
			--lat-max $lat_max \
			--lon-min $lon_min \
			--lon-max $lon_max \
			--src-dir $FAA_DB
	} > $scenery
}


function tucson()
{
	local lat_min lat_max lon_min lon_max scenery
	lat_min=30N
	lat_max=35N
	lon_min=115W
	lon_max=110W
	scenery=$ZONES_DIR/usa/tucson.txt
	{
		echo "# Tucson area scenery."
		echo "# Created: " $(date +%Y-%m-%d)
		echo "# Latitude range: [$lat_min, $lat_max["
		echo "# Longitude range: [$lon_min, $lon_max["
		echo "# 2017-10-04 Checked against https://nfdc.faa.gov/xwiki/bin/view/NFDC/56DaySub-2017-03-02"
		echo "# \"56 Day NASR Subscription - Effective March 2, 2017\"."
		echo
		echo "# ?????:"
		echo "TEAM1_LOC 30N 115W 000 031"
		echo "# ?????:"
		echo "TEAM1_LOC 30N 115W 000 031"
		echo "GROUND_COLOR #305030"
		./faaairports.exe \
			--lat-min $lat_min \
			--lat-max $lat_max \
			--lon-min $lon_min \
			--lon-max $lon_max \
			--src-dir $FAA_DB
	} > $scenery
}

if [ ! -d $ZONES_DIR/usa ]; then
	mkdir $ZONES_DIR/usa || exit 1
fi

if [ ! -d $CHARTS_DIR/usa ]; then
	mkdir $CHARTS_DIR/usa || exit 1
fi

newyork
dallas
sfrancisco
losangeles
lasvegaswest
lasvegaseast
tucson
