/*
 *  acm : an aerial combat simulator for X
 *  Copyright (C) 1991,1992  Riley Rainey
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 dated June, 1991.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program;  if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave., Cambridge, MA 02139, USA.
 */


#ifndef _damage_h
#define _damage_h

#include "pm.h"

#ifdef damage_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

#define damage_isFunctioning(c, sys)	((c->damageBits & sys) == 0)

/*
 *  Each of these bits is true if the corresponding system is
 *  damaged (== not functioning).
 *  Some flags (for ex. FLAG_LOWFUEL) aren't proper "damages"
 *  are set to warn the pilot.
 */

#define SYS_RADAR   0x0001      /* set if radar damaged */
#define SYS_TEWS    0x0002      /* set if TEWS damaged */
#define SYS_FLAPS   0x0004      /* set if flaps damaged */
#define SYS_SPEEDBRAKE  0x0008
#define SYS_HYD1    0x0010
#define SYS_HYD2    0x0020
#define SYS_GEN1    0x0040
#define SYS_GEN2    0x0080
#define SYS_ENGINE1     0x0100  /* set if engine #1 damaged */
//#define SYS_ENGINE2   0x0200  /* set if engine #2 damaged (FIXME: NOT USED) */
#define SYS_HUD     0x0400      /* set if HUD damaged */
#define SYS_FUEL    0x0800      /* set if there is a fuel leak */
#define SYS_NOSEGEAR    0x1000
#define SYS_LEFTMAIN    0x2000
#define SYS_RIGHTMAIN   0x4000
#define SYS_WINGS       0x08000 /* if wings broken, no lift! */
#define FLAG_LOWFUEL    0x10000 /* set when fuel < 1000 lb */
#define FLAG_SPEEDBRAKE 0x20000 /* set when speedbrake is deployed */
#define FLAG_WHEELBRAKE 0x40000 /* set when wheel brakes are applied */
#define FLAG_MAX_G_LOAD 0x80000 /* near the max positive/neg. wings load */
#define FLAG_STALL_WARN 0x100000 /* stall warning */

/**
 * Repairs any damage, making a brand new aircraft :-).
 */
EXTERN void damage_reset(craft * c);

/**
 * Craft c is hit, assesses damage points. The points of damage depends on
 * the type of the warhead, the velocity of impact (for "kinetic"
 * warheads) and distance of the blast (for "blast" warhead).
 * 
 * 	- For kinetic warhead, points are proportional to the product
 * 	between 0.5 * warhead damage factor and the square of the
 * 	velocity of impact.
 * 
 * 	- For blast warhead, point are calculated as product between
 * 	warhead damage factor / (distance^2 + 1.0).
 * 
 * Actual damage sustained is adjusted by a damage factor that forms a
 * bell curve centered around 0.75 * d.
 * 
 * For each damage point absorbed, there is a 15 percent chance that it
 * will be absorbed by some subsystem other than the actual airframe.
 * 
 * For each point absorbed by the airframe, there is a 20% chance that
 * it'll be absorbed by the wing and induce a rolling moment or a 10
 * percent chance that it will hit a horizontal stabilizer and induce a
 * pitching and rolling moment.
 * 
 * @param target The craft hit.
 * @param warhead_dis_type DIS type of the warhead.
 * @param warhead_type Type of warhead.
 * @param fuze_type Ignored.
 * @param distance_meters Distance from the blast (blast warhead only, m).
 * @param velocity_meters_per_sec Velocity of impact (meaningful for kinetic
 *        warhead only, m/s).
 * @param explosion_diameter_meters Here returns the explosion diameter for the
 *        warhead (m).
 * @return False if the damage cannot be absorbed and the craft must be
 *         destroyed.
 */
EXTERN int damage_absorbDISDamage(craft * target,
	dis_entity_type *warhead_dis_type, 
	u_short warhead_type,
	u_short fuze_type,
	double distance_meters,
	double velocity_meters_per_sec,
	double *explosion_diameter_meters);

#undef EXTERN
#endif
