// This magma script takes a filename in the format of
// Cremona's elliptic curve tables (a text file - not a gzipped text file)
// as a command line argument filename

// Each line of the output has 8 columns. 
// N class # [a1,a2,a3,a4,a6] index level matrix list (possibly empty) label

// For example
// 11 a 1 [0,-1,1,-10,-20] 1 1 [] X1
// 15 a 1 [1,1,1,-10,-10] 96 8 [[5,4,2,3],[1,0,0,5],[1,4,0,5],[1,0,4,5]] X187d
// 27 a 1 [0,0,1,0,-7] inf inf [] CM

// The index is the index of the 2-adic image in GL_2(Z_2)

// The level is the smallest 2^r so that the 2-adic image contains
// all matrices congruent to I mod 2^r.

// The matrix list is a list of generators of the image in GL_2(Z/2^r Z).
// The 2-adic image is conjugate (in GL_2(Z_2)) 
// to the inverse image in GL_2(Z_2) of the subgroup generated by the list
// of matrices.
// NOTE: These matrices act on the right! That is, we represent elements
// of E[2^r] as row vectors v, and the action of M in GL_2(Z_2) on these row
// vectors is by v -> v*M.

// The label is something like "X1" or "X7" or "X441" or "X243p". These are 
// the names we've given to the modular curves that parametrize these entries. 
// They all have the form Xn for some positive integer n, or Xnm, where n is a 
// positive integer and m is a lowercase letter. If there is no letter, the 
// 2-adic image contains -I. If there is a letter, the 2-adic image does not 
// contain -I. For each label, there is a human-readable website about that 
// image. The URL is something like http://users.wfu.edu/rouseja/2adic/X441.html

// A function field in one variable - used for maps to the j-line
F<t> := FunctionField(Rationals());
// A polynomial ring - used for computing preimages of maps to the j-line
R<x> := PolynomialRing(Rationals());
// A polynomial ring in 3 variables - used for constructing maps from
// E -> P^1
PPP<a,b,c> := PolynomialRing(Rationals(),3);

// An abbreviated collection of data about the 727 subgroups we consider
// The three things in the tuples are (i) the index, (ii) the level
// (for techical reasons, we consider GL_2(Z_2) to have level 2), and
// (iii) the genus of the corresponding modular curve.

subdata := [ <1, 2, 0>, <2, 2, 0>, <2, 4, 0>, <2, 8, 0>, <2, 8, 0>, <3, 2, 0>, 
<4, 4, 0>, <6, 2, 0>, <6, 4, 0>, <6, 4, 0>, <6, 4, 0>, <6, 4, 0>, <6, 4, 0>, <6,
8, 0>, <6, 8, 0>, <6, 8, 0>, <6, 8, 0>, <6, 8, 0>, <6, 8, 0>, <8, 4, 0>, <8, 4, 
0>, <8, 8, 0>, <12, 4, 0>, <12, 4, 0>, <12, 4, 0>, <12, 4, 0>, <12, 4, 0>, <12, 
8, 0>, <12, 8, 0>, <12, 8, 0>, <12, 8, 0>, <12, 8, 0>, <12, 8, 0>, <12, 8, 0>, 
<12, 8, 0>, <12, 8, 0>, <12, 8, 0>, <12, 8, 0>, <12, 8, 0>, <12, 8, 0>, <12, 8, 
0>, <12, 8, 0>, <12, 8, 0>, <12, 8, 0>, <12, 8, 0>, <12, 8, 0>, <12, 8, 0>, <12,
8, 0>, <12, 8, 0>, <12, 8, 0>, <12, 8, 1>, <12, 8, 1>, <12, 8, 1>, <12, 8, 1>, 
<16, 8, 0>, <16, 16, 0>, <16, 16, 0>, <24, 4, 0>, <24, 4, 0>, <24, 4, 0>, <24, 
8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, 
<24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 
0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24,
8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, 
<24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 
0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24, 8, 0>, <24,
8, 0>, <24, 8, 0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 
0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 
0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 
0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 0>, <24, 16, 0>, <24, 8, 1>,
<24, 8, 1>, <24, 8, 1>, <24, 8, 1>, <24, 8, 1>, <24, 8, 1>, <24, 8, 1>, <24, 8, 
1>, <24, 8, 1>, <24, 8, 1>, <24, 8, 1>, <24, 8, 1>, <24, 8, 1>, <24, 8, 1>, <24,
8, 1>, <24, 8, 1>, <24, 8, 1>, <24, 8, 1>, <24, 8, 1>, <24, 8, 1>, <24, 8, 1>, 
<24, 8, 1>, <24, 8, 1>, <24, 8, 1>, <24, 16, 1>, <24, 16, 1>, <24, 16, 1>, <24, 
16, 1>, <24, 16, 1>, <24, 16, 1>, <24, 16, 1>, <24, 16, 1>, <24, 16, 1>, <24, 
16, 1>, <24, 16, 1>, <24, 16, 1>, <24, 16, 1>, <24, 16, 1>, <24, 16, 1>, <24, 
16, 1>, <24, 16, 1>, <24, 16, 1>, <24, 16, 1>, <24, 16, 1>, <24, 16, 2>, <24, 
16, 2>, <24, 16, 2>, <24, 16, 2>, <24, 16, 2>, <24, 16, 2>, <24, 16, 2>, <24, 
16, 2>, <32, 32, 0>, <32, 32, 0>, <32, 8, 1>, <32, 8, 1>, <48, 8, 0>, <48, 8, 
0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 0>, <48,
8, 0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 0>, 
<48, 8, 0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 
0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 0>, <48, 8, 0>, <48, 16, 0>, <48, 16, 0>, 
<48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, 
<48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, 
<48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, 
<48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, 
<48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 16, 0>, <48, 32, 0>, 
<48, 32, 0>, <48, 32, 0>, <48, 32, 0>, <48, 32, 0>, <48, 32, 0>, <48, 8, 1>, 
<48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 
1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48,
8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, 
<48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 
1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48, 8, 1>, <48,
8, 1>, <48, 8, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 
1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 
1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 
1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 
1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 
1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 
1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 
1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 
1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 
1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 
1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 
1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 1>, <48, 16, 
1>, <48, 16, 1>, <48, 16, 1>, <48, 32, 1>, <48, 32, 1>, <48, 32, 1>, <48, 32, 
1>, <48, 8, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>,
<48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, 
<48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, 
<48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, 
<48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, 
<48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, 
<48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, 
<48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, <48, 16, 2>, 
<48, 32, 2>, <48, 32, 2>, <48, 32, 2>, <48, 32, 2>, <48, 16, 3>, <48, 16, 3>, 
<48, 16, 3>, <48, 16, 3>, <48, 16, 3>, <48, 16, 3>, <48, 16, 3>, <48, 16, 3>, 
<48, 16, 3>, <48, 16, 3>, <48, 16, 3>, <48, 16, 3>, <48, 16, 3>, <48, 16, 3>, 
<48, 16, 3>, <48, 16, 3>, <48, 16, 3>, <48, 16, 3>, <48, 32, 3>, <48, 32, 3>, 
<48, 32, 3>, <48, 32, 3>, <48, 32, 3>, <48, 32, 3>, <48, 32, 3>, <48, 32, 3>, 
<48, 32, 3>, <48, 32, 3>, <48, 32, 3>, <48, 32, 3>, <64, 16, 1>, <64, 16, 1>, 
<64, 16, 2>, <96, 8, 1>, <96, 8, 1>, <96, 8, 1>, <96, 8, 1>, <96, 8, 1>, <96, 8,
1>, <96, 8, 1>, <96, 8, 1>, <96, 8, 1>, <96, 8, 1>, <96, 8, 1>, <96, 8, 1>, <96,
8, 1>, <96, 8, 1>, <96, 8, 1>, <96, 8, 1>, <96, 8, 1>, <96, 8, 1>, <96, 8, 1>, 
<96, 8, 1>, <96, 8, 1>, <96, 8, 1>, <96, 8, 1>, <96, 8, 1>, <96, 16, 1>, <96, 
16, 1>, <96, 16, 1>, <96, 16, 1>, <96, 16, 1>, <96, 16, 1>, <96, 16, 1>, <96, 
16, 1>, <96, 16, 1>, <96, 16, 1>, <96, 16, 1>, <96, 16, 1>, <96, 16, 1>, <96, 
16, 1>, <96, 16, 1>, <96, 16, 1>, <96, 16, 1>, <96, 16, 1>, <96, 16, 1>, <96, 
16, 1>, <96, 16, 1>, <96, 16, 1>, <96, 32, 1>, <96, 32, 1>, <96, 32, 1>, <96, 
32, 1>, <96, 32, 1>, <96, 32, 1>, <96, 32, 1>, <96, 32, 1>, <96, 32, 1>, <96, 
32, 1>, <96, 32, 1>, <96, 32, 1>, <96, 16, 2>, <96, 16, 2>, <96, 16, 2>, <96, 
16, 2>, <96, 16, 2>, <96, 16, 2>, <96, 16, 2>, <96, 16, 2>, <96, 16, 2>, <96, 
16, 2>, <96, 16, 2>, <96, 16, 2>, <96, 16, 2>, <96, 16, 2>, <96, 16, 2>, <96, 
16, 2>, <96, 16, 2>, <96, 16, 2>, <96, 16, 2>, <96, 16, 2>, <96, 32, 2>, <96, 
32, 2>, <96, 32, 2>, <96, 32, 2>, <96, 32, 2>, <96, 32, 2>, <96, 32, 2>, <96, 
32, 2>, <96, 32, 2>, <96, 32, 2>, <96, 32, 2>, <96, 32, 2>, <96, 32, 2>, <96, 
32, 2>, <96, 32, 2>, <96, 32, 2>, <96, 8, 3>, <96, 8, 3>, <96, 8, 3>, <96, 8, 
3>, <96, 8, 3>, <96, 8, 3>, <96, 8, 3>, <96, 8, 3>, <96, 16, 3>, <96, 16, 3>, 
<96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, 
<96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, 
<96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, 
<96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, 
<96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, 
<96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, 
<96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, 
<96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, 
<96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, 
<96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, 
<96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, 
<96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 16, 3>, <96, 32, 3>, <96, 32, 3>, 
<96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, 
<96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, 
<96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, 
<96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, 
<96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, 
<96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, 
<96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, <96, 32, 3>, 
<96, 32, 3>, <96, 32, 3>, <96, 64, 3>, <96, 64, 3>, <96, 64, 3>, <96, 64, 3>, 
<96, 64, 3>, <96, 64, 3>, <96, 64, 3>, <96, 64, 3>, <96, 16, 5>, <96, 16, 5>, 
<96, 32, 5>, <96, 32, 5>, <96, 32, 5>, <96, 32, 5>, <96, 32, 5>, <96, 32, 5>, 
<96, 32, 5>, <96, 32, 5>, <96, 32, 5>, <96, 32, 5>, <96, 32, 5>, <96, 32, 5>, 
<96, 32, 5>, <96, 32, 5>, <96, 32, 5>, <96, 32, 5>, <96, 32, 5>, <96, 32, 5>, 
<96, 32, 5>, <96, 32, 5>, <96, 32, 5>, <96, 32, 5>, <96, 32, 5>, <96, 32, 5>, 
<96, 64, 5>, <96, 64, 5>, <96, 64, 5>, <96, 64, 5>, <96, 64, 5>, <96, 64, 5>, 
<96, 64, 5>, <96, 64, 5>, <96, 32, 7>, <96, 32, 7>, <96, 32, 7>, <96, 32, 7>, 
<96, 32, 7>, <96, 32, 7>, <96, 32, 7>, <96, 32, 7>, <96, 32, 7>, <96, 32, 7>, 
<96, 32, 7>, <96, 32, 7>, <96, 32, 7>, <96, 32, 7>, <128, 32, 5>, <128, 32, 5>, 
<192, 64, 5>, <192, 64, 5>, <192, 64, 5>, <192, 64, 5>, <192, 64, 5>, <192, 64, 
5>, <192, 64, 5>, <192, 64, 5> ];

// Data about the 1006 subgroups that can occur as a 2-adic image and
// don't contain -I. Each of these is an index 2 subgroup of one of the
// subgroups referenced in the first list.

finesubdata := [ <2, 4, 4, "X2a">, <2, 4, 8, "X2b">, <3, 4, 4, "X3a">,
<3, 4, 8, "X3b">, <8, 12, 8, "X8a">, <8, 12, 4, "X8b">, <8, 12, 8,
"X8c">, <8, 12, 4, "X8d">, <9, 12, 8, "X9a">, <9, 12, 8, "X9b">, <9,
12, 8, "X9c">, <9, 12, 8, "X9d">, <10, 12, 8, "X10a">, <10, 12, 4,
"X10b">, <10, 12, 8, "X10c">, <10, 12, 4, "X10d">, <13, 12, 8,
"X13a">, <13, 12, 8, "X13b">, <13, 12, 8, "X13c">, <13, 12, 8,
"X13d">, <13, 12, 8, "X13e">, <13, 12, 4, "X13f">, <13, 12, 8,
"X13g">, <13, 12, 4, "X13h">, <20, 16, 8, "X20a">, <20, 16, 4,
"X20b">, <24, 24, 8, "X24a">, <24, 24, 8, "X24b">, <24, 24, 8,
"X24c">, <24, 24, 4, "X24d">, <24, 24, 4, "X24e">, <24, 24, 8,
"X24f">, <24, 24, 8, "X24g">, <24, 24, 8, "X24h">, <25, 24, 8,
"X25a">, <25, 24, 8, "X25b">, <25, 24, 8, "X25c">, <25, 24, 8,
"X25d">, <25, 24, 8, "X25e">, <25, 24, 8, "X25f">, <25, 24, 8,
"X25g">, <25, 24, 4, "X25h">, <25, 24, 4, "X25i">, <25, 24, 8,
"X25j">, <25, 24, 8, "X25k">, <25, 24, 8, "X25l">, <25, 24, 8,
"X25m">, <25, 24, 4, "X25n">, <27, 24, 8, "X27a">, <27, 24, 8,
"X27b">, <27, 24, 8, "X27c">, <27, 24, 8, "X27d">, <27, 24, 8,
"X27e">, <27, 24, 4, "X27f">, <27, 24, 8, "X27g">, <27, 24, 4,
"X27h">, <32, 24, 16, "X32a">, <32, 24, 16, "X32b">, <32, 24, 8,
"X32c">, <32, 24, 8, "X32d">, <32, 24, 8, "X32e">, <32, 24, 8,
"X32f">, <32, 24, 16, "X32g">, <32, 24, 16, "X32h">, <32, 24, 8,
"X32i">, <32, 24, 8, "X32j">, <32, 24, 8, "X32k">, <32, 24, 8,
"X32l">, <33, 24, 8, "X33a">, <33, 24, 8, "X33b">, <33, 24, 8,
"X33c">, <33, 24, 8, "X33d">, <33, 24, 8, "X33e">, <33, 24, 8,
"X33f">, <33, 24, 8, "X33g">, <33, 24, 8, "X33h">, <34, 24, 8,
"X34a">, <34, 24, 8, "X34b">, <34, 24, 8, "X34c">, <34, 24, 8,
"X34d">, <34, 24, 8, "X34e">, <34, 24, 8, "X34f">, <34, 24, 8,
"X34g">, <34, 24, 8, "X34h">, <36, 24, 8, "X36a">, <36, 24, 16,
"X36b">, <36, 24, 8, "X36c">, <36, 24, 16, "X36d">, <36, 24, 8,
"X36e">, <36, 24, 8, "X36f">, <36, 24, 16, "X36g">, <36, 24, 8,
"X36h">, <36, 24, 8, "X36i">, <36, 24, 16, "X36j">, <36, 24, 8,
"X36k">, <36, 24, 8, "X36l">, <36, 24, 16, "X36m">, <36, 24, 8,
"X36n">, <36, 24, 8, "X36o">, <36, 24, 16, "X36p">, <36, 24, 8,
"X36q">, <36, 24, 16, "X36r">, <36, 24, 8, "X36s">, <36, 24, 16,
"X36t">, <37, 24, 16, "X37a">, <37, 24, 8, "X37b">, <37, 24, 16,
"X37c">, <37, 24, 8, "X37d">, <38, 24, 8, "X38a">, <38, 24, 8,
"X38b">, <38, 24, 8, "X38c">, <38, 24, 8, "X38d">, <40, 24, 16,
"X40a">, <40, 24, 16, "X40b">, <40, 24, 16, "X40c">, <40, 24, 16,
"X40d">, <42, 24, 16, "X42a">, <42, 24, 8, "X42b">, <42, 24, 16,
"X42c">, <42, 24, 8, "X42d">, <44, 24, 8, "X44a">, <44, 24, 8,
"X44b">, <44, 24, 8, "X44c">, <44, 24, 8, "X44d">, <46, 24, 8,
"X46a">, <46, 24, 8, "X46b">, <46, 24, 8, "X46c">, <46, 24, 8,
"X46d">, <48, 24, 8, "X48a">, <48, 24, 8, "X48b">, <48, 24, 8,
"X48c">, <48, 24, 8, "X48d">, <57, 32, 32, "X57a">, <57, 32, 32,
"X57b">, <58, 48, 8, "X58a">, <58, 48, 8, "X58b">, <58, 48, 8,
"X58c">, <58, 48, 8, "X58d">, <58, 48, 8, "X58e">, <58, 48, 8,
"X58f">, <58, 48, 8, "X58g">, <58, 48, 8, "X58h">, <58, 48, 4,
"X58i">, <58, 48, 8, "X58j">, <58, 48, 8, "X58k">, <60, 48, 8,
"X60a">, <60, 48, 8, "X60b">, <60, 48, 8, "X60c">, <60, 48, 4,
"X60d">, <61, 48, 16, "X61a">, <61, 48, 16, "X61b">, <61, 48, 8,
"X61c">, <61, 48, 8, "X61d">, <61, 48, 16, "X61e">, <61, 48, 16,
"X61f">, <61, 48, 8, "X61g">, <61, 48, 8, "X61h">, <62, 48, 8,
"X62a">, <62, 48, 8, "X62b">, <62, 48, 8, "X62c">, <62, 48, 8,
"X62d">, <62, 48, 8, "X62e">, <62, 48, 8, "X62f">, <62, 48, 8,
"X62g">, <62, 48, 8, "X62h">, <62, 48, 8, "X62i">, <62, 48, 8,
"X62j">, <66, 48, 16, "X66a">, <66, 48, 8, "X66b">, <66, 48, 16,
"X66c">, <66, 48, 8, "X66d">, <66, 48, 8, "X66e">, <66, 48, 8,
"X66f">, <67, 48, 8, "X67a">, <67, 48, 8, "X67b">, <67, 48, 8,
"X67c">, <67, 48, 8, "X67d">, <75, 48, 16, "X75a">, <75, 48, 16,
"X75b">, <75, 48, 16, "X75c">, <75, 48, 16, "X75d">, <75, 48, 8,
"X75e">, <75, 48, 8, "X75f">, <75, 48, 8, "X75g">, <75, 48, 8,
"X75h">, <75, 48, 8, "X75i">, <75, 48, 8, "X75j">, <76, 48, 8,
"X76a">, <76, 48, 8, "X76b">, <76, 48, 8, "X76c">, <76, 48, 8,
"X76d">, <78, 48, 16, "X78a">, <78, 48, 16, "X78b">, <78, 48, 16,
"X78c">, <78, 48, 16, "X78d">, <78, 48, 8, "X78e">, <78, 48, 8,
"X78f">, <78, 48, 8, "X78g">, <78, 48, 8, "X78h">, <78, 48, 8,
"X78i">, <78, 48, 8, "X78j">, <79, 48, 16, "X79a">, <79, 48, 8,
"X79b">, <79, 48, 16, "X79c">, <79, 48, 8, "X79d">, <79, 48, 16,
"X79e">, <79, 48, 16, "X79f">, <79, 48, 16, "X79g">, <79, 48, 8,
"X79h">, <79, 48, 8, "X79i">, <79, 48, 16, "X79j">, <82, 48, 16,
"X82a">, <82, 48, 16, "X82b">, <82, 48, 16, "X82c">, <82, 48, 16,
"X82d">, <84, 48, 16, "X84a">, <84, 48, 16, "X84b">, <84, 48, 16,
"X84c">, <84, 48, 16, "X84d">, <84, 48, 16, "X84e">, <84, 48, 16,
"X84f">, <84, 48, 16, "X84g">, <84, 48, 16, "X84h">, <84, 48, 8,
"X84i">, <84, 48, 8, "X84j">, <84, 48, 8, "X84k">, <84, 48, 8,
"X84l">, <84, 48, 8, "X84m">, <84, 48, 8, "X84n">, <84, 48, 8,
"X84o">, <84, 48, 8, "X84p">, <85, 48, 16, "X85a">, <85, 48, 16,
"X85b">, <85, 48, 16, "X85c">, <85, 48, 16, "X85d">, <85, 48, 16,
"X85e">, <85, 48, 16, "X85f">, <85, 48, 16, "X85g">, <85, 48, 16,
"X85h">, <85, 48, 8, "X85i">, <85, 48, 8, "X85j">, <85, 48, 8,
"X85k">, <85, 48, 8, "X85l">, <85, 48, 8, "X85m">, <85, 48, 8,
"X85n">, <85, 48, 8, "X85o">, <85, 48, 8, "X85p">, <86, 48, 16,
"X86a">, <86, 48, 16, "X86b">, <86, 48, 16, "X86c">, <86, 48, 16,
"X86d">, <86, 48, 16, "X86e">, <86, 48, 16, "X86f">, <86, 48, 16,
"X86g">, <86, 48, 16, "X86h">, <86, 48, 8, "X86i">, <86, 48, 8,
"X86j">, <86, 48, 8, "X86k">, <86, 48, 8, "X86l">, <86, 48, 8,
"X86m">, <86, 48, 8, "X86n">, <86, 48, 8, "X86o">, <86, 48, 8,
"X86p">, <87, 48, 8, "X87a">, <87, 48, 8, "X87b">, <87, 48, 8,
"X87c">, <87, 48, 8, "X87d">, <87, 48, 8, "X87e">, <87, 48, 8,
"X87f">, <87, 48, 8, "X87g">, <87, 48, 8, "X87h">, <87, 48, 8,
"X87i">, <87, 48, 8, "X87j">, <87, 48, 8, "X87k">, <87, 48, 8,
"X87l">, <87, 48, 8, "X87m">, <87, 48, 8, "X87n">, <87, 48, 8,
"X87o">, <87, 48, 8, "X87p">, <92, 48, 8, "X92a">, <92, 48, 16,
"X92b">, <92, 48, 8, "X92c">, <92, 48, 16, "X92d">, <92, 48, 8,
"X92e">, <92, 48, 16, "X92f">, <92, 48, 8, "X92g">, <92, 48, 8,
"X92h">, <92, 48, 8, "X92i">, <92, 48, 16, "X92j">, <92, 48, 16,
"X92k">, <94, 48, 16, "X94a">, <94, 48, 8, "X94b">, <94, 48, 16,
"X94c">, <94, 48, 8, "X94d">, <94, 48, 8, "X94e">, <94, 48, 8,
"X94f">, <95, 48, 8, "X95a">, <95, 48, 8, "X95b">, <95, 48, 16,
"X95c">, <95, 48, 16, "X95d">, <95, 48, 8, "X95e">, <95, 48, 8,
"X95f">, <95, 48, 16, "X95g">, <95, 48, 16, "X95h">, <96, 48, 8,
"X96a">, <96, 48, 16, "X96b">, <96, 48, 8, "X96c">, <96, 48, 16,
"X96d">, <96, 48, 8, "X96e">, <96, 48, 16, "X96f">, <96, 48, 8,
"X96g">, <96, 48, 16, "X96h">, <96, 48, 8, "X96i">, <96, 48, 16,
"X96j">, <96, 48, 8, "X96k">, <96, 48, 8, "X96l">, <96, 48, 16,
"X96m">, <96, 48, 8, "X96n">, <96, 48, 8, "X96o">, <96, 48, 16,
"X96p">, <96, 48, 8, "X96q">, <96, 48, 16, "X96r">, <96, 48, 8,
"X96s">, <96, 48, 8, "X96t">, <98, 48, 8, "X98a">, <98, 48, 8,
"X98b">, <98, 48, 8, "X98c">, <98, 48, 8, "X98d">, <98, 48, 8,
"X98e">, <98, 48, 8, "X98f">, <98, 48, 8, "X98g">, <98, 48, 8,
"X98h">, <98, 48, 8, "X98i">, <98, 48, 8, "X98j">, <98, 48, 8,
"X98k">, <98, 48, 8, "X98l">, <98, 48, 8, "X98m">, <98, 48, 8,
"X98n">, <98, 48, 8, "X98o">, <98, 48, 8, "X98p">, <99, 48, 8,
"X99a">, <99, 48, 8, "X99b">, <99, 48, 8, "X99c">, <99, 48, 8,
"X99d">, <99, 48, 8, "X99e">, <99, 48, 8, "X99f">, <99, 48, 8,
"X99g">, <99, 48, 8, "X99h">, <99, 48, 8, "X99i">, <99, 48, 8,
"X99j">, <99, 48, 8, "X99k">, <99, 48, 8, "X99l">, <99, 48, 8,
"X99m">, <99, 48, 8, "X99n">, <99, 48, 8, "X99o">, <99, 48, 8,
"X99p">, <100, 48, 8, "X100a">, <100, 48, 8, "X100b">, <100, 48, 8,
"X100c">, <100, 48, 8, "X100d">, <100, 48, 8, "X100e">, <100, 48, 8,
"X100f">, <100, 48, 8, "X100g">, <100, 48, 8, "X100h">, <100, 48, 8,
"X100i">, <100, 48, 8, "X100j">, <101, 48, 8, "X101a">, <101, 48, 8,
"X101b">, <101, 48, 8, "X101c">, <101, 48, 8, "X101d">, <101, 48, 8,
"X101e">, <101, 48, 8, "X101f">, <101, 48, 8, "X101g">, <101, 48, 8,
"X101h">, <101, 48, 8, "X101i">, <101, 48, 8, "X101j">, <101, 48, 8,
"X101k">, <101, 48, 8, "X101l">, <101, 48, 8, "X101m">, <101, 48, 8,
"X101n">, <101, 48, 8, "X101o">, <101, 48, 8, "X101p">, <102, 48, 16,
"X102a">, <102, 48, 16, "X102b">, <102, 48, 16, "X102c">, <102, 48,
16, "X102d">, <102, 48, 16, "X102e">, <102, 48, 16, "X102f">, <102,
48, 16, "X102g">, <102, 48, 16, "X102h">, <102, 48, 8, "X102i">, <102,
48, 8, "X102j">, <102, 48, 8, "X102k">, <102, 48, 8, "X102l">, <102,
48, 8, "X102m">, <102, 48, 8, "X102n">, <102, 48, 8, "X102o">, <102,
48, 8, "X102p">, <108, 48, 32, "X108a">, <108, 48, 16, "X108b">, <108,
48, 32, "X108c">, <108, 48, 16, "X108d">, <108, 48, 16, "X108e">,
<108, 48, 16, "X108f">, <115, 48, 16, "X115a">, <115, 48, 16,
"X115b">, <115, 48, 16, "X115c">, <115, 48, 16, "X115d">, <115, 48,
16, "X115e">, <115, 48, 16, "X115f">, <115, 48, 16, "X115g">, <115,
48, 16, "X115h">, <116, 48, 32, "X116a">, <116, 48, 32, "X116b">,
<116, 48, 16, "X116c">, <116, 48, 16, "X116d">, <116, 48, 16,
"X116e">, <116, 48, 16, "X116f">, <116, 48, 32, "X116g">, <116, 48,
32, "X116h">, <116, 48, 16, "X116i">, <116, 48, 16, "X116j">, <116,
48, 16, "X116k">, <116, 48, 16, "X116l">, <117, 48, 16, "X117a">,
<117, 48, 16, "X117b">, <117, 48, 16, "X117c">, <117, 48, 16,
"X117d">, <117, 48, 16, "X117e">, <117, 48, 16, "X117f">, <117, 48,
16, "X117g">, <117, 48, 16, "X117h">, <117, 48, 16, "X117i">, <117,
48, 16, "X117j">, <117, 48, 16, "X117k">, <117, 48, 16, "X117l">,
<117, 48, 16, "X117m">, <117, 48, 16, "X117n">, <117, 48, 16,
"X117o">, <117, 48, 16, "X117p">, <118, 48, 16, "X118a">, <118, 48,
32, "X118b">, <118, 48, 16, "X118c">, <118, 48, 32, "X118d">, <118,
48, 16, "X118e">, <118, 48, 16, "X118f">, <118, 48, 16, "X118g">,
<118, 48, 32, "X118h">, <118, 48, 16, "X118i">, <118, 48, 32,
"X118j">, <118, 48, 16, "X118k">, <118, 48, 16, "X118l">, <118, 48,
16, "X118m">, <118, 48, 32, "X118n">, <118, 48, 16, "X118o">, <118,
48, 32, "X118p">, <118, 48, 16, "X118q">, <118, 48, 32, "X118r">,
<118, 48, 16, "X118s">, <118, 48, 16, "X118t">, <118, 48, 16,
"X118u">, <118, 48, 32, "X118v">, <118, 48, 16, "X118w">, <118, 48,
16, "X118x">, <119, 48, 16, "X119a">, <119, 48, 16, "X119b">, <119,
48, 16, "X119c">, <119, 48, 16, "X119d">, <119, 48, 16, "X119e">,
<119, 48, 16, "X119f">, <119, 48, 16, "X119g">, <119, 48, 16,
"X119h">, <119, 48, 16, "X119i">, <119, 48, 16, "X119j">, <119, 48,
16, "X119k">, <119, 48, 16, "X119l">, <119, 48, 16, "X119m">, <119,
48, 16, "X119n">, <119, 48, 16, "X119o">, <119, 48, 16, "X119p">,
<120, 48, 16, "X120a">, <120, 48, 16, "X120b">, <120, 48, 16,
"X120c">, <120, 48, 16, "X120d">, <120, 48, 16, "X120e">, <120, 48,
16, "X120f">, <120, 48, 16, "X120g">, <120, 48, 16, "X120h">, <120,
48, 16, "X120i">, <120, 48, 16, "X120j">, <120, 48, 16, "X120k">,
<120, 48, 16, "X120l">, <120, 48, 16, "X120m">, <120, 48, 16,
"X120n">, <120, 48, 16, "X120o">, <120, 48, 16, "X120p">, <121, 48,
16, "X121a">, <121, 48, 16, "X121b">, <121, 48, 16, "X121c">, <121,
48, 16, "X121d">, <121, 48, 16, "X121e">, <121, 48, 16, "X121f">,
<121, 48, 16, "X121g">, <121, 48, 16, "X121h">, <121, 48, 16,
"X121i">, <121, 48, 16, "X121j">, <121, 48, 16, "X121k">, <121, 48,
16, "X121l">, <121, 48, 16, "X121m">, <121, 48, 16, "X121n">, <121,
48, 16, "X121o">, <121, 48, 16, "X121p">, <122, 48, 16, "X122a">,
<122, 48, 16, "X122b">, <122, 48, 16, "X122c">, <122, 48, 16,
"X122d">, <122, 48, 16, "X122e">, <122, 48, 16, "X122f">, <122, 48,
16, "X122g">, <122, 48, 16, "X122h">, <122, 48, 16, "X122i">, <122,
48, 16, "X122j">, <122, 48, 16, "X122k">, <122, 48, 16, "X122l">,
<122, 48, 16, "X122m">, <122, 48, 16, "X122n">, <122, 48, 16,
"X122o">, <122, 48, 16, "X122p">, <123, 48, 32, "X123a">, <123, 48,
16, "X123b">, <123, 48, 16, "X123c">, <123, 48, 16, "X123d">, <123,
48, 32, "X123e">, <123, 48, 16, "X123f">, <181, 96, 8, "X181a">, <181,
96, 8, "X181b">, <181, 96, 8, "X181c">, <181, 96, 8, "X181d">, <181,
96, 8, "X181e">, <181, 96, 8, "X181f">, <181, 96, 8, "X181g">, <181,
96, 8, "X181h">, <183, 96, 8, "X183a">, <183, 96, 8, "X183b">, <183,
96, 8, "X183c">, <183, 96, 8, "X183d">, <183, 96, 8, "X183e">, <183,
96, 8, "X183f">, <183, 96, 8, "X183g">, <183, 96, 8, "X183h">, <183,
96, 8, "X183i">, <183, 96, 8, "X183j">, <183, 96, 8, "X183k">, <183,
96, 8, "X183l">, <185, 96, 16, "X185a">, <185, 96, 16, "X185b">, <185,
96, 16, "X185c">, <185, 96, 16, "X185d">, <185, 96, 8, "X185e">, <185,
96, 8, "X185f">, <185, 96, 8, "X185g">, <185, 96, 8, "X185h">, <185,
96, 8, "X185i">, <185, 96, 8, "X185j">, <185, 96, 8, "X185k">, <185,
96, 8, "X185l">, <187, 96, 8, "X187a">, <187, 96, 8, "X187b">, <187,
96, 8, "X187c">, <187, 96, 8, "X187d">, <187, 96, 8, "X187e">, <187,
96, 8, "X187f">, <187, 96, 8, "X187g">, <187, 96, 8, "X187h">, <187,
96, 8, "X187i">, <187, 96, 8, "X187j">, <187, 96, 8, "X187k">, <187,
96, 8, "X187l">, <188, 96, 8, "X188a">, <188, 96, 8, "X188b">, <188,
96, 8, "X188c">, <188, 96, 8, "X188d">, <188, 96, 8, "X188e">, <188,
96, 8, "X188f">, <188, 96, 8, "X188g">, <188, 96, 8, "X188h">, <189,
96, 8, "X189a">, <189, 96, 8, "X189b">, <189, 96, 8, "X189c">, <189,
96, 8, "X189d">, <189, 96, 8, "X189e">, <189, 96, 16, "X189f">, <189,
96, 16, "X189g">, <189, 96, 8, "X189h">, <189, 96, 8, "X189i">, <189,
96, 8, "X189j">, <189, 96, 8, "X189k">, <189, 96, 8, "X189l">, <189,
96, 16, "X189m">, <189, 96, 16, "X189n">, <190, 96, 8, "X190a">, <190,
96, 8, "X190b">, <190, 96, 8, "X190c">, <190, 96, 8, "X190d">, <190,
96, 8, "X190e">, <190, 96, 8, "X190f">, <190, 96, 8, "X190g">, <190,
96, 8, "X190h">, <192, 96, 16, "X192a">, <192, 96, 16, "X192b">, <192,
96, 16, "X192c">, <192, 96, 16, "X192d">, <192, 96, 8, "X192e">, <192,
96, 16, "X192f">, <192, 96, 8, "X192g">, <192, 96, 16, "X192h">, <192,
96, 8, "X192i">, <192, 96, 8, "X192j">, <192, 96, 16, "X192k">, <192,
96, 8, "X192l">, <192, 96, 16, "X192m">, <192, 96, 8, "X192n">, <193,
96, 8, "X193a">, <193, 96, 16, "X193b">, <193, 96, 8, "X193c">, <193,
96, 16, "X193d">, <193, 96, 16, "X193e">, <193, 96, 16, "X193f">,
<193, 96, 8, "X193g">, <193, 96, 16, "X193h">, <193, 96, 8, "X193i">,
<193, 96, 16, "X193j">, <193, 96, 16, "X193k">, <193, 96, 16,
"X193l">, <193, 96, 8, "X193m">, <193, 96, 8, "X193n">, <194, 96, 16,
"X194a">, <194, 96, 16, "X194b">, <194, 96, 16, "X194c">, <194, 96,
16, "X194d">, <194, 96, 8, "X194e">, <194, 96, 8, "X194f">, <194, 96,
8, "X194g">, <194, 96, 8, "X194h">, <194, 96, 8, "X194i">, <194, 96,
8, "X194j">, <194, 96, 8, "X194k">, <194, 96, 8, "X194l">, <195, 96,
16, "X195a">, <195, 96, 16, "X195b">, <195, 96, 16, "X195c">, <195,
96, 16, "X195d">, <195, 96, 16, "X195e">, <195, 96, 16, "X195f">,
<195, 96, 8, "X195g">, <195, 96, 8, "X195h">, <195, 96, 8, "X195i">,
<195, 96, 8, "X195j">, <195, 96, 8, "X195k">, <195, 96, 8, "X195l">,
<197, 96, 16, "X197a">, <197, 96, 16, "X197b">, <197, 96, 16,
"X197c">, <197, 96, 16, "X197d">, <197, 96, 8, "X197e">, <197, 96, 8,
"X197f">, <197, 96, 8, "X197g">, <197, 96, 8, "X197h">, <199, 96, 16,
"X199a">, <199, 96, 16, "X199b">, <199, 96, 16, "X199c">, <199, 96,
16, "X199d">, <199, 96, 8, "X199e">, <199, 96, 8, "X199f">, <199, 96,
8, "X199g">, <199, 96, 8, "X199h">, <200, 96, 8, "X200a">, <200, 96,
8, "X200b">, <200, 96, 8, "X200c">, <200, 96, 8, "X200d">, <200, 96,
8, "X200e">, <200, 96, 8, "X200f">, <200, 96, 8, "X200g">, <200, 96,
8, "X200h">, <202, 96, 16, "X202a">, <202, 96, 16, "X202b">, <202, 96,
16, "X202c">, <202, 96, 16, "X202d">, <202, 96, 8, "X202e">, <202, 96,
8, "X202f">, <202, 96, 8, "X202g">, <202, 96, 8, "X202h">, <203, 96,
8, "X203a">, <203, 96, 8, "X203b">, <203, 96, 8, "X203c">, <203, 96,
8, "X203d">, <203, 96, 8, "X203e">, <203, 96, 8, "X203f">, <203, 96,
8, "X203g">, <203, 96, 8, "X203h">, <204, 96, 8, "X204a">, <204, 96,
8, "X204b">, <204, 96, 8, "X204c">, <204, 96, 8, "X204d">, <204, 96,
8, "X204e">, <204, 96, 8, "X204f">, <204, 96, 8, "X204g">, <204, 96,
8, "X204h">, <205, 96, 16, "X205a">, <205, 96, 16, "X205b">, <205, 96,
16, "X205c">, <205, 96, 16, "X205d">, <205, 96, 16, "X205e">, <205,
96, 16, "X205f">, <205, 96, 8, "X205g">, <205, 96, 8, "X205h">, <205,
96, 8, "X205i">, <205, 96, 8, "X205j">, <205, 96, 8, "X205k">, <205,
96, 8, "X205l">, <207, 96, 32, "X207a">, <207, 96, 32, "X207b">, <207,
96, 16, "X207c">, <207, 96, 16, "X207d">, <207, 96, 32, "X207e">,
<207, 96, 32, "X207f">, <207, 96, 16, "X207g">, <207, 96, 16,
"X207h">, <207, 96, 16, "X207i">, <207, 96, 16, "X207j">, <207, 96,
16, "X207k">, <207, 96, 16, "X207l">, <207, 96, 16, "X207m">, <207,
96, 16, "X207n">, <208, 96, 16, "X208a">, <208, 96, 16, "X208b">,
<208, 96, 16, "X208c">, <208, 96, 16, "X208d">, <208, 96, 16,
"X208e">, <208, 96, 16, "X208f">, <208, 96, 16, "X208g">, <208, 96,
16, "X208h">, <208, 96, 16, "X208i">, <208, 96, 16, "X208j">, <208,
96, 16, "X208k">, <208, 96, 16, "X208l">, <209, 96, 16, "X209a">,
<209, 96, 16, "X209b">, <209, 96, 16, "X209c">, <209, 96, 16,
"X209d">, <210, 96, 16, "X210a">, <210, 96, 16, "X210b">, <210, 96,
16, "X210c">, <210, 96, 16, "X210d">, <211, 96, 16, "X211a">, <211,
96, 32, "X211b">, <211, 96, 16, "X211c">, <211, 96, 32, "X211d">,
<211, 96, 16, "X211e">, <211, 96, 32, "X211f">, <211, 96, 16,
"X211g">, <211, 96, 32, "X211h">, <211, 96, 32, "X211i">, <211, 96,
16, "X211j">, <211, 96, 32, "X211k">, <211, 96, 32, "X211l">, <211,
96, 16, "X211m">, <211, 96, 32, "X211n">, <211, 96, 16, "X211o">,
<211, 96, 16, "X211p">, <211, 96, 16, "X211q">, <211, 96, 16,
"X211r">, <211, 96, 16, "X211s">, <211, 96, 16, "X211t">, <212, 96,
32, "X212a">, <212, 96, 32, "X212b">, <212, 96, 32, "X212c">, <212,
96, 32, "X212d">, <212, 96, 16, "X212e">, <212, 96, 16, "X212f">,
<212, 96, 16, "X212g">, <212, 96, 16, "X212h">, <212, 96, 16,
"X212i">, <212, 96, 16, "X212j">, <212, 96, 16, "X212k">, <212, 96,
16, "X212l">, <213, 96, 32, "X213a">, <213, 96, 32, "X213b">, <213,
96, 32, "X213c">, <213, 96, 32, "X213d">, <213, 96, 16, "X213e">,
<213, 96, 16, "X213f">, <213, 96, 16, "X213g">, <213, 96, 16,
"X213h">, <213, 96, 16, "X213i">, <213, 96, 16, "X213j">, <213, 96,
16, "X213k">, <213, 96, 16, "X213l">, <214, 96, 16, "X214a">, <214,
96, 16, "X214b">, <214, 96, 16, "X214c">, <214, 96, 16, "X214d">,
<215, 96, 16, "X215a">, <215, 96, 16, "X215b">, <215, 96, 16,
"X215c">, <215, 96, 16, "X215d">, <215, 96, 16, "X215e">, <215, 96,
16, "X215f">, <215, 96, 16, "X215g">, <215, 96, 16, "X215h">, <215,
96, 16, "X215i">, <215, 96, 16, "X215j">, <215, 96, 16, "X215k">,
<215, 96, 16, "X215l">, <217, 96, 16, "X217a">, <217, 96, 16,
"X217b">, <217, 96, 16, "X217c">, <217, 96, 16, "X217d">, <217, 96,
16, "X217e">, <217, 96, 16, "X217f">, <217, 96, 16, "X217g">, <217,
96, 16, "X217h">, <219, 96, 16, "X219a">, <219, 96, 16, "X219b">,
<219, 96, 16, "X219c">, <219, 96, 16, "X219d">, <219, 96, 16,
"X219e">, <219, 96, 16, "X219f">, <219, 96, 16, "X219g">, <219, 96,
16, "X219h">, <221, 96, 16, "X221a">, <221, 96, 16, "X221b">, <221,
96, 16, "X221c">, <221, 96, 16, "X221d">, <222, 96, 16, "X222a">,
<222, 96, 16, "X222b">, <222, 96, 16, "X222c">, <222, 96, 16,
"X222d">, <223, 96, 16, "X223a">, <223, 96, 16, "X223b">, <223, 96,
16, "X223c">, <223, 96, 16, "X223d">, <223, 96, 16, "X223e">, <223,
96, 16, "X223f">, <223, 96, 16, "X223g">, <223, 96, 16, "X223h">,
<225, 96, 32, "X225a">, <225, 96, 32, "X225b">, <225, 96, 32,
"X225c">, <225, 96, 32, "X225d">, <225, 96, 16, "X225e">, <225, 96,
16, "X225f">, <225, 96, 16, "X225g">, <225, 96, 16, "X225h">, <225,
96, 16, "X225i">, <225, 96, 16, "X225j">, <225, 96, 16, "X225k">,
<225, 96, 16, "X225l">, <226, 96, 16, "X226a">, <226, 96, 16,
"X226b">, <226, 96, 16, "X226c">, <226, 96, 16, "X226d">, <226, 96,
16, "X226e">, <226, 96, 16, "X226f">, <226, 96, 16, "X226g">, <226,
96, 16, "X226h">, <227, 96, 32, "X227a">, <227, 96, 32, "X227b">,
<227, 96, 32, "X227c">, <227, 96, 32, "X227d">, <227, 96, 16,
"X227e">, <227, 96, 16, "X227f">, <227, 96, 16, "X227g">, <227, 96,
16, "X227h">, <227, 96, 16, "X227i">, <227, 96, 16, "X227j">, <227,
96, 16, "X227k">, <227, 96, 16, "X227l">, <228, 96, 16, "X228a">,
<228, 96, 16, "X228b">, <228, 96, 16, "X228c">, <228, 96, 16,
"X228d">, <228, 96, 16, "X228e">, <228, 96, 16, "X228f">, <228, 96,
16, "X228g">, <228, 96, 16, "X228h">, <229, 96, 16, "X229a">, <229,
96, 16, "X229b">, <229, 96, 16, "X229c">, <229, 96, 16, "X229d">,
<229, 96, 16, "X229e">, <229, 96, 16, "X229f">, <229, 96, 16,
"X229g">, <229, 96, 16, "X229h">, <230, 96, 16, "X230a">, <230, 96,
16, "X230b">, <230, 96, 16, "X230c">, <230, 96, 16, "X230d">, <230,
96, 16, "X230e">, <230, 96, 16, "X230f">, <230, 96, 16, "X230g">,
<230, 96, 16, "X230h">, <231, 96, 16, "X231a">, <231, 96, 16,
"X231b">, <231, 96, 16, "X231c">, <231, 96, 16, "X231d">, <233, 96,
16, "X233a">, <233, 96, 16, "X233b">, <233, 96, 16, "X233c">, <233,
96, 16, "X233d">, <233, 96, 16, "X233e">, <233, 96, 16, "X233f">,
<233, 96, 16, "X233g">, <233, 96, 16, "X233h">, <234, 96, 16,
"X234a">, <234, 96, 16, "X234b">, <234, 96, 16, "X234c">, <234, 96,
16, "X234d">, <234, 96, 16, "X234e">, <234, 96, 16, "X234f">, <234,
96, 16, "X234g">, <234, 96, 16, "X234h">, <235, 96, 16, "X235a">,
<235, 96, 32, "X235b">, <235, 96, 16, "X235c">, <235, 96, 32,
"X235d">, <235, 96, 16, "X235e">, <235, 96, 16, "X235f">, <235, 96,
16, "X235g">, <235, 96, 32, "X235h">, <235, 96, 16, "X235i">, <235,
96, 32, "X235j">, <235, 96, 16, "X235k">, <235, 96, 16, "X235l">,
<235, 96, 16, "X235m">, <235, 96, 32, "X235n">, <235, 96, 16,
"X235o">, <235, 96, 32, "X235p">, <235, 96, 16, "X235q">, <235, 96,
16, "X235r">, <235, 96, 32, "X235s">, <235, 96, 32, "X235t">, <236,
96, 16, "X236a">, <236, 96, 16, "X236b">, <236, 96, 16, "X236c">,
<236, 96, 16, "X236d">, <236, 96, 16, "X236e">, <236, 96, 16,
"X236f">, <236, 96, 16, "X236g">, <236, 96, 16, "X236h">, <238, 96,
32, "X238a">, <238, 96, 32, "X238b">, <238, 96, 32, "X238c">, <238,
96, 32, "X238d">, <239, 96, 32, "X239a">, <239, 96, 32, "X239b">,
<239, 96, 32, "X239c">, <239, 96, 32, "X239d">, <240, 96, 32,
"X240a">, <240, 96, 32, "X240b">, <240, 96, 32, "X240c">, <240, 96,
32, "X240d">, <240, 96, 32, "X240e">, <240, 96, 32, "X240f">, <240,
96, 32, "X240g">, <240, 96, 32, "X240h">, <240, 96, 32, "X240i">,
<240, 96, 32, "X240j">, <240, 96, 32, "X240k">, <240, 96, 32,
"X240l">, <240, 96, 32, "X240m">, <240, 96, 32, "X240n">, <240, 96,
32, "X240o">, <240, 96, 32, "X240p">, <241, 96, 32, "X241a">, <241,
96, 32, "X241b">, <241, 96, 32, "X241c">, <241, 96, 32, "X241d">,
<241, 96, 32, "X241e">, <241, 96, 32, "X241f">, <241, 96, 32,
"X241g">, <241, 96, 32, "X241h">, <242, 96, 32, "X242a">, <242, 96,
32, "X242b">, <242, 96, 32, "X242c">, <242, 96, 32, "X242d">, <242,
96, 32, "X242e">, <242, 96, 32, "X242f">, <242, 96, 32, "X242g">,
<242, 96, 32, "X242h">, <243, 96, 32, "X243a">, <243, 96, 32,
"X243b">, <243, 96, 32, "X243c">, <243, 96, 32, "X243d">, <243, 96,
32, "X243e">, <243, 96, 32, "X243f">, <243, 96, 32, "X243g">, <243,
96, 32, "X243h">, <243, 96, 32, "X243i">, <243, 96, 32, "X243j">,
<243, 96, 32, "X243k">, <243, 96, 32, "X243l">, <243, 96, 32,
"X243m">, <243, 96, 32, "X243n">, <243, 96, 32, "X243o">, <243, 96,
32, "X243p"> ]; 
// Lists of generating matrices.  
genlists := [ "[]",
"[[0,1,1,1]]", "[[3,3,0,1],[0,1,3,1]]", "[[7,7,0,1],[5,0,1,1]]",
"[[5,5,0,1],[0,1,1,1]]", "[[1,1,0,1]]", "[[2,1,3,1],[3,1,2,1]]", "[]",
"[[1,1,0,1],[3,0,2,1]]", "[[1,0,2,1],[3,3,0,1],[1,2,0,1]]",
"[[1,0,2,3],[1,2,0,1],[3,1,2,1]]", "[[1,0,0,3],[1,1,2,3]]",
"[[3,0,0,1],[1,1,0,1],[3,0,0,3]]", "[[1,0,2,5],[1,0,0,7],[1,1,6,7]]",
"[[1,1,0,1],[7,0,2,1],[5,0,2,1]]", "[[1,1,0,1],[7,0,0,1],[5,0,2,1]]",
"[[1,0,2,1],[1,0,0,7],[1,1,0,5]]", "[[1,0,2,5],[1,0,6,7],[1,1,6,7]]",
"[[1,0,2,1],[1,0,0,3],[1,1,0,5]]", "[[3,3,0,1],[1,1,3,0]]",
"[[1,1,3,0],[1,2,2,3]]", "[[2,3,1,7],[1,7,6,7]]",
"[[1,0,2,3],[1,1,2,3]]", "[[3,0,0,1],[3,2,2,3],[3,0,0,3]]",
"[[3,0,0,1],[3,0,0,3],[1,2,0,1]]", "[[1,1,0,3],[1,2,2,3]]",
"[[3,3,0,1],[3,0,0,3],[1,2,0,1]]",
"[[1,0,0,5],[1,2,0,1],[1,3,4,3],[1,3,6,7]]",
"[[1,2,4,1],[1,2,0,1],[1,0,6,3],[1,3,6,7]]",
"[[3,6,0,1],[3,0,0,7],[3,0,0,3],[3,3,2,1]]",
"[[1,1,2,1],[1,3,6,7],[1,0,0,7]]",
"[[7,0,0,1],[7,7,4,3],[5,0,0,1],[3,0,0,3]]",
"[[7,0,0,1],[7,7,4,3],[3,0,0,3],[7,0,4,3]]",
"[[5,0,4,3],[7,7,4,3],[3,0,0,3],[7,0,4,3]]",
"[[1,2,0,1],[1,3,4,3],[1,3,6,7],[1,0,4,5]]",
"[[1,1,0,1],[7,0,0,7],[7,0,0,1],[5,0,0,1],[3,0,0,3]]",
"[[1,0,6,7],[1,1,0,5],[3,0,0,3]]",
"[[1,0,6,7],[3,0,6,7],[3,0,0,3],[3,6,6,7]]",
"[[1,1,2,7],[1,3,0,7],[1,3,2,7],[1,0,4,1]]",
"[[1,2,0,1],[1,3,6,7],[1,0,0,7]]",
"[[1,2,4,1],[1,2,0,1],[1,1,2,7],[1,2,2,3]]",
"[[7,7,0,1],[1,2,0,1],[5,0,2,1]]",
"[[1,2,0,1],[1,3,2,3],[5,0,0,1],[7,5,0,1]]",
"[[3,0,0,5],[1,1,0,5],[3,0,0,3],[1,0,4,1]]",
"[[1,2,0,1],[5,0,4,1],[7,5,2,1],[7,5,0,1]]",
"[[3,0,0,5],[3,0,6,7],[3,0,0,3],[3,6,6,7]]",
"[[1,2,4,1],[1,0,0,5],[1,3,4,3],[1,3,2,3]]",
"[[1,1,0,5],[3,0,0,1],[3,0,0,3],[1,0,4,1]]",
"[[1,1,2,7],[1,0,2,3],[1,3,2,7],[1,0,4,1]]",
"[[1,2,4,1],[1,3,4,3],[1,3,6,7],[1,0,4,5]]",
"[[5,0,0,5],[1,3,6,7],[5,5,2,5],[1,1,2,5]]",
"[[1,2,0,3],[1,0,0,7],[1,1,6,3]]",
"[[1,2,4,1],[1,3,4,3],[1,0,4,5],[1,3,2,3]]",
"[[1,2,4,1],[1,0,0,5],[1,3,4,3],[1,3,6,7]]",
"[[7,5,7,2],[6,7,5,7],[3,2,7,5]]", "[[13,11,1,6],[7,3,0,1]]",
"[[1,7,1,2],[15,11,0,1]]", "[[3,0,0,3],[3,0,2,1]]",
"[[3,0,0,3],[1,2,2,3]]", "[[3,3,0,1],[3,0,0,3]]",
"[[1,0,0,5],[1,2,6,7],[3,6,2,7],[3,0,0,3]]",
"[[3,6,0,7],[7,0,0,7],[7,0,0,1],[7,0,4,7],[3,0,0,3]]",
"[[1,2,2,7],[1,3,2,7],[1,1,6,3]]",
"[[5,0,0,1],[3,0,0,3],[7,7,2,1],[3,3,0,1]]",
"[[5,5,2,3],[7,7,2,1],[3,3,0,1]]",
"[[3,0,0,5],[1,4,0,1],[3,0,0,3],[3,6,6,3]]",
"[[1,4,0,5],[3,0,0,5],[3,0,0,3],[3,6,6,3]]",
"[[1,0,0,5],[1,0,2,3],[1,1,6,7]]",
"[[1,1,2,7],[1,0,2,3],[1,4,0,1],[1,1,6,7]]",
"[[1,1,0,7],[1,0,6,3],[1,2,4,5]]", "[[1,1,0,7],[1,0,2,3],[1,2,0,5]]",
"[[3,6,0,1],[1,1,6,7],[1,1,6,3]]",
"[[1,0,4,5],[1,2,2,3],[1,5,4,7],[1,4,0,1]]",
"[[1,2,2,3],[5,0,0,1],[3,3,0,1]]",
"[[7,0,0,7],[5,5,0,1],[3,0,0,1],[3,0,0,3]]",
"[[1,1,2,7],[1,0,6,3],[5,0,0,5],[1,1,6,3]]",
"[[1,2,0,1],[1,3,0,7],[1,1,2,7]]",
"[[7,0,0,7],[7,0,0,1],[5,5,0,1],[3,0,0,3]]",
"[[3,0,0,5],[3,3,4,3],[3,0,0,3]]", "[[1,0,6,7],[1,1,2,7],[1,0,2,7]]",
"[[1,2,0,1],[1,3,6,7],[1,0,2,7]]", "[[1,2,0,1],[1,0,6,7],[1,3,0,3]]",
"[[7,7,0,1],[7,6,2,1],[1,0,4,1]]",
"[[3,3,0,3],[7,0,0,7],[5,0,0,1],[5,0,0,3]]",
"[[3,3,0,3],[7,0,0,7],[7,0,0,1],[7,0,0,3]]",
"[[3,3,0,3],[7,0,0,7],[7,0,0,3],[5,0,0,3]]",
"[[7,6,4,7],[7,0,0,7],[7,0,0,1],[7,0,0,3],[3,0,4,7]]",
"[[3,1,6,7],[3,0,0,7],[1,1,6,3]]", "[[3,1,6,7],[1,1,6,7],[1,1,6,3]]",
"[[1,3,6,7],[1,1,4,3],[1,2,0,5]]", "[[1,3,0,7],[1,1,2,7],[1,2,0,5]]",
"[[1,2,0,1],[7,0,0,7],[5,0,0,1],[3,0,0,3],[3,3,0,1]]",
"[[3,6,0,1],[3,0,0,7],[1,1,6,3]]",
"[[1,2,0,1],[3,0,0,3],[7,0,4,3],[5,5,4,3]]",
"[[7,7,0,1],[3,0,4,3],[3,1,0,1],[3,0,0,3]]",
"[[1,2,0,1],[7,0,0,7],[7,0,0,1],[7,0,0,3],[5,0,0,1]]",
"[[1,3,0,7],[1,3,2,7],[1,1,6,3]]",
"[[7,6,4,7],[7,0,0,7],[5,0,0,1],[7,0,4,3],[1,0,4,7]]",
"[[7,6,4,7],[7,0,0,7],[7,0,0,1],[5,0,0,1],[7,0,4,3]]",
"[[1,2,0,1],[7,0,0,7],[3,0,4,7],[7,0,4,3],[1,0,4,7]]",
"[[7,6,4,7],[7,0,0,7],[7,0,0,3],[3,0,4,7],[1,0,4,7]]",
"[[1,1,0,1],[7,0,0,7],[7,0,0,1],[5,0,0,1]]",
"[[1,0,0,5],[3,15,2,1],[3,12,0,3],[1,6,0,3]]",
"[[3,9,14,7],[3,0,0,7],[3,3,2,1],[1,5,6,3]]",
"[[13,10,0,1],[13,13,2,3],[15,13,0,1],[1,0,4,1]]",
"[[13,10,0,1],[13,13,2,3],[5,10,2,3],[1,0,4,1]]",
"[[5,15,2,3],[13,13,2,3],[5,10,2,3],[1,0,4,1]]",
"[[3,0,0,11],[1,1,10,5],[3,0,14,7]]",
"[[1,2,0,1],[1,3,0,7],[1,3,14,7],[1,0,4,5]]",
"[[1,2,0,1],[1,3,14,7],[1,0,4,5],[1,0,6,11]]",
"[[1,2,0,1],[1,3,0,7],[1,3,10,7],[1,0,4,13]]",
"[[1,2,0,1],[1,3,10,7],[1,0,4,13],[1,3,12,15]]",
"[[1,3,12,3],[1,1,12,7],[1,0,4,5],[1,3,14,7]]",
"[[1,3,12,3],[1,3,0,3],[1,0,2,3],[1,2,0,13]]",
"[[3,6,0,7],[1,2,0,7],[3,0,0,3],[3,9,12,7]]",
"[[1,0,0,5],[3,0,0,5],[3,0,0,3],[3,9,12,7]]",
"[[1,1,0,1],[3,0,8,7],[7,0,0,7],[1,0,8,7],[3,0,0,3]]",
"[[1,1,0,1],[7,0,0,7],[7,0,0,1],[5,0,0,1],[3,0,0,3]]",
"[[3,3,0,3],[3,0,8,3],[7,0,0,7],[7,0,0,1],[5,0,0,1]]",
"[[3,3,0,3],[3,0,8,3],[7,0,0,7],[5,0,0,1],[5,0,0,3]]",
"[[3,3,0,3],[3,0,8,3],[7,0,0,7],[7,0,0,1],[7,0,0,3]]",
"[[3,3,0,3],[3,0,8,3],[7,0,0,7],[7,0,0,3],[5,0,0,3]]",
"[[11,0,0,3],[13,13,0,1],[5,0,6,3]]",
"[[5,15,2,3],[13,13,2,3],[15,13,0,1],[1,0,4,1]]",
"[[1,2,4,1],[1,1,0,7],[1,3,6,7]]", "[[3,1,6,3],[3,0,0,3],[1,1,6,3]]",
"[[5,0,0,5],[1,1,2,3],[5,0,0,1],[3,3,0,1]]",
"[[1,2,2,7],[1,1,6,3],[1,2,0,5]]",
"[[7,6,4,7],[7,0,0,7],[3,0,4,7],[7,0,4,3],[1,0,4,7]]",
"[[1,3,0,7],[1,0,6,3],[1,2,4,5]]", "[[1,2,4,1],[1,1,2,7],[1,0,2,7]]",
"[[1,2,6,7],[1,4,0,5],[3,0,0,3],[3,6,6,7]]",
"[[1,2,4,1],[1,0,6,3],[1,1,6,3]]",
"[[3,6,4,3],[1,1,4,7],[3,0,4,7],[3,0,0,3]]",
"[[3,6,4,3],[1,0,0,5],[1,1,4,7],[3,0,0,3]]",
"[[1,1,2,3],[1,1,4,7],[3,0,4,7],[3,0,0,3]]",
"[[1,3,0,7],[1,0,2,3],[1,2,0,5]]", "[[1,2,4,1],[1,0,2,7],[1,3,0,3]]",
"[[3,6,2,1],[5,0,0,1],[3,3,0,1]]",
"[[3,1,6,1],[1,2,6,3],[1,2,2,7],[3,0,0,3]]",
"[[1,0,0,5],[3,0,0,5],[3,0,0,3],[3,6,6,3]]",
"[[7,6,4,7],[7,0,0,7],[7,0,0,3],[5,0,0,1],[1,0,4,7]]",
"[[1,3,0,7],[1,1,2,7],[1,2,4,5]]", "[[1,3,0,7],[1,1,6,3],[1,2,0,5]]",
"[[1,3,6,7],[5,0,0,5],[5,0,0,3]]", "[[1,0,0,5],[1,1,2,3],[1,1,4,7]]",
"[[1,3,6,7],[1,1,4,3],[1,2,4,5]]", "[[5,0,4,1],[1,1,2,3],[3,3,0,1]]",
"[[1,3,0,7],[1,3,10,7],[1,0,4,13],[1,2,4,13]]",
"[[1,3,10,7],[1,0,4,13],[1,2,4,13],[1,3,12,15]]",
"[[1,0,4,5],[1,3,14,7],[1,0,6,11],[1,1,6,3]]",
"[[1,3,12,3],[1,1,12,7],[1,3,0,3],[1,3,14,7]]",
"[[1,3,0,7],[1,0,4,5],[1,3,14,7],[1,1,6,3]]",
"[[1,3,12,3],[1,0,4,5],[1,3,14,7],[1,2,0,13]]",
"[[1,3,12,3],[1,1,12,7],[1,3,0,3],[1,0,2,3]]",
"[[1,3,12,3],[1,0,2,3],[1,0,4,5],[1,2,0,13]]",
"[[1,3,12,3],[1,1,4,5],[1,0,0,5],[3,0,0,3]]",
"[[1,3,12,3],[1,1,4,5],[3,0,0,3],[3,9,12,3]]",
"[[7,7,8,7],[3,0,8,7],[7,0,0,7],[1,0,8,7],[3,0,0,3]]",
"[[13,10,0,1],[13,13,2,3],[11,11,2,1],[5,10,2,3]]",
"[[5,15,2,3],[13,13,2,3],[11,11,2,1],[5,10,2,3]]",
"[[1,5,14,7],[3,9,14,7],[3,0,0,3],[3,3,2,1]]",
"[[3,9,14,7],[3,0,0,7],[3,3,2,1],[3,0,0,3]]",
"[[1,5,14,7],[3,0,0,3],[3,3,2,1],[3,2,0,1]]",
"[[13,10,0,1],[13,13,2,3],[15,13,0,1],[11,11,2,1]]",
"[[5,15,2,3],[13,13,2,3],[15,13,0,1],[11,11,2,1]]",
"[[3,0,0,7],[3,0,0,3],[3,3,2,1],[3,2,0,1]]",
"[[7,7,8,7],[7,0,0,7],[1,0,8,7],[5,0,0,1],[3,0,0,3]]",
"[[7,1,2,1],[5,0,0,5],[3,2,0,5],[5,5,10,5]]",
"[[1,3,12,3],[1,1,12,7],[1,0,2,3],[1,0,4,5]]",
"[[1,3,12,3],[1,3,0,3],[1,3,14,7],[1,2,0,13]]",
"[[3,15,2,1],[3,12,0,3],[3,0,0,7],[1,6,0,3]]",
"[[7,1,2,1],[1,1,10,5],[5,0,0,5],[5,5,10,5]]",
"[[1,5,14,7],[3,9,14,7],[3,3,2,1],[1,5,6,3]]",
"[[7,1,2,1],[1,1,10,5],[5,0,0,5],[7,10,0,5]]",
"[[7,1,2,1],[5,0,0,5],[3,2,0,5],[7,10,0,5]]",
"[[7,3,0,1],[25,15,1,2]]", "[[9,31,1,2],[23,19,0,1]]",
"[[1,5,0,7],[1,3,1,6]]", "[[5,0,3,3],[0,3,1,5],[7,0,1,1]]",
"[[7,0,0,7],[7,0,0,1],[5,2,0,5],[3,0,4,7]]",
"[[7,0,0,7],[5,2,0,5],[1,0,0,3],[3,0,4,7]]",
"[[1,0,6,3],[7,0,0,7],[3,4,6,5],[1,0,2,3],[3,0,4,7]]",
"[[7,0,0,7],[7,0,0,3],[7,6,0,3],[5,0,0,7]]",
"[[3,6,0,7],[7,0,0,7],[7,0,0,1],[7,0,0,3]]",
"[[1,0,6,3],[7,0,0,7],[1,0,2,3],[3,0,4,7],[3,4,4,3]]",
"[[1,0,0,5],[7,0,0,7],[3,4,6,5],[1,0,4,5],[3,0,6,1]]",
"[[3,6,0,1],[7,0,0,7],[5,2,0,5],[7,0,4,3]]",
"[[1,0,6,7],[7,0,0,7],[3,0,4,7],[3,4,4,3],[1,0,4,1]]",
"[[7,0,0,7],[5,2,0,5],[7,0,4,3],[5,2,0,3]]",
"[[5,0,0,1],[7,7,2,1],[3,3,0,1]]",
"[[7,0,0,7],[7,0,0,3],[1,0,0,7],[7,6,0,3]]",
"[[3,6,0,1],[7,0,0,7],[1,0,0,7],[5,0,0,1]]",
"[[7,0,0,7],[7,0,0,1],[5,0,0,1],[1,2,0,5]]",
"[[7,0,0,7],[7,6,0,7],[5,0,0,1],[3,3,0,1]]",
"[[1,1,0,3],[1,0,0,5],[3,3,2,5]]", "[[7,0,0,7],[5,5,0,1],[3,0,0,1]]",
"[[7,0,0,7],[7,0,4,7],[1,2,0,5],[5,0,0,3]]",
"[[7,7,0,3],[7,0,0,7],[1,0,0,3]]",
"[[3,6,0,7],[7,0,0,7],[7,0,0,1],[7,0,4,7]]",
"[[7,7,0,3],[7,0,0,7],[5,0,0,3]]", "[[7,0,0,7],[7,0,0,1],[5,5,0,1]]",
"[[7,0,0,7],[5,2,4,1],[7,0,0,1],[3,0,4,3]]",
"[[5,2,4,1],[7,0,0,7],[3,0,4,3],[1,0,4,7]]",
"[[1,2,0,1],[7,0,0,7],[7,0,0,3],[3,1,0,1]]",
"[[7,0,0,7],[3,6,0,5],[5,0,0,1],[5,2,0,1]]",
"[[1,1,0,3],[3,0,8,3],[7,14,0,7],[7,0,0,7],[7,0,0,3]]",
"[[3,0,8,7],[7,14,0,7],[7,0,0,7],[7,0,0,3],[1,0,8,7]]",
"[[1,2,14,5],[1,0,0,5],[3,6,10,5],[3,0,0,3]]",
"[[1,2,14,5],[3,6,10,5],[1,2,14,1],[3,0,0,3]]",
"[[3,3,0,3],[11,11,8,5],[15,0,8,5],[13,13,8,3],[11,11,0,7]]",
"[[3,3,0,3],[7,0,0,7],[7,0,0,3],[5,0,0,3]]",
"[[7,7,0,7],[7,0,0,7],[7,0,8,1],[5,0,0,1]]",
"[[3,0,0,13],[1,2,2,9],[1,4,0,1],[3,0,0,3]]",
"[[7,14,0,7],[7,0,0,7],[7,0,8,3],[1,0,8,7],[5,0,0,1]]",
"[[1,2,10,15],[1,2,0,1],[1,1,10,15]]",
"[[7,7,0,3],[3,0,8,3],[7,0,0,7],[1,0,0,3]]",
"[[1,2,0,1],[1,3,14,7],[1,0,10,15]]",
"[[5,5,0,5],[7,0,0,7],[1,0,0,7],[1,0,8,3]]",
"[[1,2,0,1],[1,3,12,15],[1,0,10,15]]",
"[[15,14,0,3],[3,0,12,3],[9,11,0,3],[3,0,0,3]]",
"[[7,7,0,1],[3,0,12,3],[9,11,0,3],[3,0,0,3]]",
"[[1,1,0,1],[7,0,0,7],[1,0,8,7],[3,0,0,1]]",
"[[15,0,2,1],[3,3,2,1],[1,0,4,1]]",
"[[3,3,0,3],[7,0,0,7],[7,0,0,1],[7,0,0,3]]",
"[[3,6,0,3],[9,9,4,1],[15,0,0,1],[3,0,0,3]]",
"[[3,3,0,3],[7,0,0,7],[5,0,0,1],[5,0,0,3]]",
"[[5,5,0,5],[7,0,0,7],[1,0,8,3],[5,0,0,3]]",
"[[5,5,0,5],[3,0,8,1],[7,0,0,7],[5,0,0,3]]",
"[[7,7,0,7],[3,0,8,1],[7,0,0,7],[1,0,0,7]]",
"[[1,1,12,15],[3,6,0,3],[3,0,0,3],[3,0,12,15]]",
"[[11,11,4,1],[1,0,4,1],[5,10,2,3]]",
"[[7,7,0,3],[3,0,8,3],[7,0,0,7],[7,0,0,1]]",
"[[7,7,0,3],[3,0,8,3],[7,0,0,7],[5,0,0,3]]",
"[[1,1,0,1],[7,0,0,15],[7,0,0,7],[7,0,0,1],[5,0,0,1]]",
"[[3,3,0,3],[7,0,0,7],[1,0,8,7],[1,0,0,3]]",
"[[11,11,4,1],[7,14,2,1],[1,0,4,1]]",
"[[21,0,2,1],[1,2,0,1],[17,19,2,1]]",
"[[27,0,0,3],[29,29,2,1],[31,0,2,1]]",
"[[1,1,0,1],[7,0,0,7],[3,0,16,3],[5,0,0,1],[7,0,16,1]]",
"[[7,28,0,1],[5,15,4,1],[5,20,0,1],[3,0,0,3]]",
"[[7,0,0,1],[5,15,4,1],[5,0,0,1],[3,0,0,3]]",
"[[1,1,0,1],[7,0,0,7],[5,0,16,1],[3,0,16,3],[7,0,16,1]]",
"[[1,2,0,7],[7,0,0,7],[5,0,0,1],[1,0,4,7]]",
"[[1,2,0,7],[7,0,0,7],[7,0,0,3],[1,0,4,7]]",
"[[3,6,0,7],[7,0,0,7],[7,0,0,1],[3,0,0,3]]",
"[[7,0,0,7],[5,2,4,1],[3,0,0,3],[1,0,4,7]]",
"[[1,0,6,3],[7,0,0,7],[3,4,6,5],[1,0,2,3],[3,0,6,1]]",
"[[7,0,0,7],[1,0,2,3],[3,0,4,7],[1,0,4,5],[3,4,4,3]]",
"[[1,5,2,7],[1,0,2,3],[1,1,6,7]]",
"[[7,0,6,1],[3,4,0,3],[5,0,4,1],[7,0,0,7],[7,4,0,7]]",
"[[1,5,2,7],[1,0,2,3],[1,0,4,5]]", "[[1,5,0,7],[1,1,4,7],[1,6,2,3]]",
"[[3,6,2,5],[1,2,6,1],[3,0,0,3]]",
"[[7,0,6,1],[3,4,0,3],[7,0,0,7],[1,4,0,5],[7,4,0,7]]",
"[[3,0,0,5],[1,2,2,1],[3,0,0,3]]",
"[[3,4,0,3],[7,3,0,1],[1,4,0,1],[3,3,2,1]]",
"[[1,1,2,7],[5,1,0,3],[5,0,0,5],[1,4,0,1]]",
"[[3,6,0,7],[7,0,0,7],[5,7,0,7],[3,0,0,3]]",
"[[1,2,4,5],[3,1,4,5],[3,0,0,3]]", "[[5,0,0,5],[5,0,2,3],[3,3,0,1]]",
"[[1,1,4,7],[1,4,6,7],[1,4,0,1]]", "[[1,0,6,7],[3,0,0,3],[3,3,0,1]]",
"[[1,0,2,7],[1,1,4,7],[3,0,0,3]]", "[[1,2,0,7],[3,1,6,3]]",
"[[1,3,2,1],[3,6,0,5]]", "[[1,3,6,7],[1,3,2,7],[1,6,2,3]]",
"[[1,2,4,7],[7,0,0,7],[7,6,0,3],[7,0,4,3]]",
"[[3,6,0,7],[7,0,0,7],[7,0,0,1],[3,0,4,7]]",
"[[3,6,0,7],[7,0,0,7],[1,0,0,3],[3,0,4,7]]",
"[[1,0,4,3],[1,2,4,1],[7,0,0,7],[5,0,0,1]]",
"[[7,0,0,7],[5,2,4,3],[1,0,0,3],[5,0,0,1]]",
"[[1,2,4,1],[5,0,4,7],[7,0,0,7],[7,0,0,3]]",
"[[1,2,4,3],[7,0,0,7],[7,0,0,3],[1,0,0,7]]",
"[[7,3,0,1],[5,6,2,3],[1,4,0,1]]", "[[7,3,0,1],[7,2,2,1],[1,4,0,1]]",
"[[3,4,0,3],[3,0,4,7],[1,5,4,7],[3,0,0,3]]",
"[[7,0,0,7],[5,0,0,1],[3,0,0,3],[3,3,0,1]]",
"[[7,0,0,7],[7,6,0,3],[7,6,4,5],[7,0,4,3]]",
"[[3,3,4,1],[9,9,4,3],[5,15,4,3],[7,5,2,1]]",
"[[9,2,2,7],[1,0,8,1],[9,9,4,3],[5,15,4,3]]",
"[[1,0,8,1],[9,9,4,3],[5,10,0,1],[7,5,2,1]]",
"[[9,0,0,1],[9,2,2,7],[3,3,4,1],[5,10,0,1]]",
"[[9,2,2,7],[3,3,4,1],[9,9,4,3],[5,10,0,1]]",
"[[9,0,0,1],[9,2,2,7],[3,3,4,1],[5,15,4,3]]",
"[[13,10,0,1],[1,0,8,1],[15,13,0,1],[9,9,2,7]]",
"[[1,0,8,1],[1,3,2,7],[15,13,0,1],[9,9,2,7]]",
"[[9,2,2,7],[1,0,8,1],[13,10,0,1],[9,9,2,7]]",
"[[9,2,2,7],[1,0,8,1],[1,3,2,7],[9,9,2,7]]",
"[[9,0,8,1],[1,1,2,7],[13,10,8,1],[7,5,8,1]]",
"[[9,0,8,1],[13,10,0,1],[13,13,2,3],[7,5,8,1]]",
"[[7,14,0,1],[3,5,6,3],[5,15,2,1]]",
"[[7,0,0,7],[7,5,6,3],[5,0,0,1],[3,2,0,1]]",
"[[3,9,6,3],[1,5,6,3],[7,7,2,1]]",
"[[7,14,0,1],[7,0,0,7],[5,0,0,1],[1,5,6,3]]",
"[[7,10,0,1],[3,6,0,1],[5,9,6,3]]",
"[[7,10,0,1],[9,0,0,1],[7,0,0,3],[5,3,2,1]]",
"[[7,14,0,1],[3,2,0,1],[1,1,6,3]]",
"[[9,0,0,1],[7,14,0,1],[7,0,0,3],[1,1,6,3]]",
"[[9,0,0,1],[1,1,2,7],[13,10,0,1],[15,13,0,1]]",
"[[3,15,2,1],[9,4,0,1],[7,1,0,1],[13,11,2,3]]",
"[[13,7,2,3],[3,3,2,1],[13,10,2,3]]",
"[[5,15,2,3],[9,4,0,1],[13,10,4,1],[7,3,4,1]]",
"[[5,10,4,1],[9,4,0,1],[7,5,0,1],[15,14,2,1]]",
"[[7,14,0,7],[7,0,0,7],[5,0,0,1],[3,0,0,3],[3,3,0,1]]",
"[[7,0,0,7],[5,5,8,7],[1,2,8,1],[5,0,0,1],[3,0,0,3]]",
"[[3,9,4,1],[5,10,2,3],[13,10,4,1]]",
"[[9,4,0,1],[5,14,4,1],[15,9,0,1],[15,14,2,1]]",
"[[13,7,2,3],[11,11,2,1],[1,4,0,1],[7,1,0,1]]",
"[[5,4,2,3],[13,7,2,3],[11,11,2,1],[1,4,0,1]]",
"[[5,10,4,1],[13,7,2,3],[1,4,0,1],[7,1,0,1]]",
"[[5,10,4,1],[5,4,2,3],[13,7,2,3],[1,4,0,1]]",
"[[7,14,0,7],[7,0,0,7],[7,0,0,1],[7,0,0,3],[5,0,0,1]]",
"[[3,0,8,7],[7,0,0,7],[7,0,8,3],[1,0,8,7],[1,2,8,1]]",
"[[3,0,8,7],[7,14,0,7],[7,0,0,7],[7,0,8,3],[1,0,8,7]]",
"[[7,0,0,7],[7,0,0,3],[1,0,8,7],[1,2,8,1],[5,0,0,1]]",
"[[1,0,6,11],[1,1,10,7],[1,1,14,15]]",
"[[1,1,10,15],[1,0,4,5],[1,0,10,15]]",
"[[1,0,14,15],[1,1,10,7],[1,1,14,15],[1,5,14,15]]",
"[[7,7,0,1],[13,13,2,3],[1,4,0,1],[7,7,2,1]]",
"[[1,0,2,3],[1,1,10,15],[1,4,0,1],[1,1,14,7]]",
"[[7,7,0,1],[13,0,4,1],[13,13,2,3],[1,4,0,1]]",
"[[1,0,2,3],[1,1,10,15],[1,0,4,5],[1,4,0,1]]",
"[[1,0,2,3],[1,1,10,15],[1,5,10,15],[1,0,4,5]]",
"[[3,0,2,1],[5,10,4,1],[3,3,2,1]]",
"[[15,13,4,1],[1,2,2,3],[13,10,4,1]]",
"[[9,0,2,3],[15,15,0,1],[1,2,2,3]]",
"[[9,0,2,3],[13,10,0,1],[13,13,2,3]]",
"[[7,0,0,7],[5,5,0,1],[3,0,0,1],[3,0,0,3]]",
"[[7,0,0,7],[5,0,8,7],[3,0,0,3],[3,3,8,7]]",
"[[7,0,0,7],[7,0,0,1],[5,5,0,1],[3,0,0,3]]",
"[[7,0,0,7],[1,0,8,7],[3,0,0,3],[3,3,8,7]]",
"[[9,0,0,1],[11,1,12,3],[3,0,0,3],[15,15,4,1]]",
"[[9,0,0,1],[1,1,4,1],[15,0,0,1],[3,0,0,3]]",
"[[7,0,0,7],[11,11,8,5],[15,0,0,5],[1,0,0,7],[11,11,8,7]]",
"[[7,7,0,1],[3,3,8,3],[7,0,0,7],[5,0,0,1]]",
"[[7,7,0,1],[3,3,8,3],[7,0,0,7],[7,0,0,3]]",
"[[1,1,8,7],[7,7,8,7],[7,0,0,7],[5,0,0,1]]",
"[[3,3,8,3],[7,0,0,7],[7,0,0,3],[5,5,0,3]]",
"[[7,0,0,7],[7,0,8,1],[3,9,8,7],[3,0,0,7],[3,3,8,7]]",
"[[7,0,0,7],[1,0,0,7],[1,0,8,3],[3,3,8,5]]",
"[[1,1,8,7],[3,0,8,1],[7,0,0,7],[1,0,0,7]]",
"[[5,0,0,11],[5,0,0,5],[1,3,14,15],[1,4,0,1]]",
"[[7,0,0,7],[1,0,8,7],[5,5,0,3],[1,0,0,3]]",
"[[7,7,0,1],[7,0,0,7],[3,0,0,1],[5,0,8,3]]",
"[[7,7,0,1],[7,0,0,7],[1,0,0,3],[5,0,8,3]]",
"[[7,7,0,1],[7,0,0,7],[1,0,8,7],[3,0,0,1]]",
"[[1,3,10,15],[1,6,2,11],[1,3,14,7],[1,2,2,11]]",
"[[1,3,10,15],[1,3,14,7],[1,4,0,1],[1,2,2,11]]",
"[[1,3,10,15],[1,6,2,11],[1,5,12,7],[1,2,2,11]]",
"[[1,3,10,15],[1,5,12,7],[1,4,0,1],[1,2,2,11]]",
"[[9,0,0,1],[13,10,8,1],[13,13,2,3],[7,5,8,1]]",
"[[7,0,0,7],[5,0,16,1],[1,1,16,1],[7,0,16,1],[3,0,0,3]]",
"[[1,1,0,1],[7,0,0,7],[7,0,0,1],[5,0,0,1],[3,0,0,3]]",
"[[7,0,0,7],[1,1,16,1],[5,0,0,1],[7,0,16,1],[3,0,0,3]]",
"[[1,1,0,1],[7,0,0,7],[5,0,16,1],[7,0,16,1],[3,0,0,3]]",
"[[7,4,2,5],[7,0,2,1],[7,0,0,7],[1,0,4,5],[5,4,2,7]]",
"[[1,2,10,15],[1,1,10,15],[1,5,10,15],[1,6,4,13]]",
"[[13,10,0,1],[3,0,0,3],[15,14,2,1],[7,5,0,1]]",
"[[1,3,0,7],[1,3,6,3],[1,2,4,13],[1,6,4,5]]",
"[[13,14,4,1],[1,4,0,1],[7,5,2,1],[15,14,2,1]]",
"[[1,3,0,7],[1,1,10,15],[1,5,10,15],[1,6,4,13]]",
"[[1,3,14,7],[1,2,2,3],[1,0,10,15]]",
"[[1,2,2,3],[1,3,12,15],[1,0,10,15]]",
"[[3,9,2,1],[5,10,0,1],[11,11,0,1],[3,0,0,3]]",
"[[15,0,2,1],[5,0,2,3],[3,3,2,1]]",
"[[7,5,4,1],[5,14,4,1],[1,4,0,1],[9,9,2,3]]",
"[[9,9,4,1],[11,1,12,3],[15,0,0,1],[3,0,0,3]]",
"[[9,9,4,1],[11,1,12,3],[3,0,0,3],[15,15,4,1]]",
"[[7,7,0,1],[7,14,2,1],[9,0,4,1]]",
"[[1,1,0,3],[5,10,0,5],[3,0,8,3],[7,0,0,7],[7,0,0,3]]",
"[[5,10,0,1],[9,11,12,3],[3,0,0,3],[5,10,2,3]]",
"[[1,3,2,11],[1,3,0,7],[1,1,10,15]]",
"[[5,10,0,1],[3,0,0,3],[7,5,0,1],[15,14,2,1]]",
"[[7,3,0,1],[13,14,4,1],[1,4,0,1],[7,5,2,1]]",
"[[1,1,12,15],[3,9,4,13],[3,0,0,3],[3,0,12,15]]",
"[[1,4,10,7],[1,7,10,15],[1,0,0,9],[1,1,6,3]]",
"[[5,15,0,3],[15,15,0,1],[5,5,12,3],[3,0,0,3]]",
"[[7,5,4,1],[11,0,12,3],[3,0,0,3],[13,10,4,1]]",
"[[1,4,10,7],[1,2,4,13],[1,7,10,15],[1,0,0,9]]",
"[[15,14,0,3],[7,5,4,1],[11,0,12,3],[3,0,0,3]]",
"[[3,3,0,5],[3,3,6,1],[3,0,0,3],[3,0,12,15]]",
"[[3,0,14,13],[3,3,6,1],[3,0,0,3],[3,0,12,15]]",
"[[7,3,0,1],[1,1,2,3],[1,4,0,1],[3,3,2,1]]",
"[[3,0,14,5],[1,1,2,3],[3,0,0,3],[3,3,2,1]]",
"[[7,0,0,7],[7,0,8,3],[1,0,8,7],[1,2,8,1],[5,0,0,1]]",
"[[9,2,2,3],[7,1,4,1],[15,13,0,1],[1,4,0,1]]",
"[[13,10,0,1],[15,13,0,1],[3,3,2,1],[3,0,0,3]]",
"[[1,1,12,7],[1,3,6,3],[1,2,4,13],[1,6,4,5]]",
"[[13,10,0,1],[3,0,0,3],[3,3,2,1],[13,10,2,3]]",
"[[1,2,10,15],[1,3,2,11],[1,1,10,15]]",
"[[3,4,0,1],[5,6,2,1],[3,0,0,3],[7,12,0,1]]",
"[[3,4,0,1],[1,0,0,5],[1,14,10,5],[3,0,0,3]]",
"[[11,1,2,1],[5,10,4,1],[11,1,4,1]]",
"[[1,6,2,9],[3,6,10,13],[1,2,2,9],[3,0,0,3]]",
"[[1,3,14,7],[1,0,6,11],[1,2,4,5]]",
"[[1,1,12,15],[1,2,4,13],[1,2,2,11]]",
"[[1,6,2,9],[3,0,0,13],[1,2,2,9],[3,0,0,3]]",
"[[13,13,2,3],[13,0,2,3],[13,10,4,1]]",
"[[7,0,2,1],[13,13,2,3],[13,0,2,3]]",
"[[7,3,0,1],[5,14,4,1],[1,4,0,1],[9,9,2,3]]",
"[[15,15,0,1],[9,0,4,1],[15,14,2,1]]",
"[[1,2,2,3],[15,9,0,1],[1,4,0,1],[3,3,2,1]]",
"[[3,0,8,7],[7,0,0,7],[7,0,0,3],[1,0,8,7],[1,2,8,1]]",
"[[7,0,0,7],[3,0,16,3],[1,1,16,1],[5,0,0,1],[7,0,16,1]]",
"[[7,0,0,7],[5,0,16,1],[3,0,16,3],[1,1,16,1],[7,0,16,1]]",
"[[3,9,4,1],[5,3,4,1],[5,0,0,1],[3,0,0,3]]",
"[[3,9,4,1],[1,3,4,1],[5,3,4,1],[3,0,0,3]]",
"[[1,4,12,3],[7,0,0,7],[1,0,12,7],[5,0,0,1],[1,6,0,7]]",
"[[1,0,4,3],[5,0,4,7],[7,0,0,7],[1,0,12,7],[1,2,12,1]]",
"[[15,15,0,1],[11,1,4,1],[5,5,12,3],[3,0,0,3]]",
"[[3,6,0,7],[1,1,12,7],[3,0,0,3],[3,0,12,15]]",
"[[1,0,0,5],[3,0,0,5],[1,2,2,1],[3,0,0,3]]",
"[[1,2,4,9],[1,3,12,11],[1,3,14,7]]",
"[[1,5,12,7],[1,2,4,13],[1,0,0,9],[1,3,6,11]]",
"[[15,13,4,1],[11,6,2,1],[9,2,4,1]]",
"[[1,6,10,15],[1,0,0,9],[1,5,10,7],[1,6,4,13]]",
"[[11,6,2,1],[9,2,4,1],[11,11,0,1]]",
"[[3,5,6,3],[1,0,0,9],[1,1,6,3],[3,0,0,3]]",
"[[1,6,0,15],[1,0,0,9],[3,9,6,11],[3,0,0,3]]",
"[[5,15,10,13],[5,0,0,5],[1,3,14,15],[1,7,14,15]]",
"[[15,12,0,1],[5,0,0,5],[9,4,0,1],[3,9,10,5]]",
"[[1,6,4,1],[7,0,0,7],[7,0,0,3],[1,0,12,7],[5,0,0,1]]",
"[[5,4,12,7],[7,0,0,7],[7,0,0,3],[1,0,12,7],[1,6,0,7]]",
"[[1,2,10,1],[3,6,10,5],[3,0,0,3],[1,2,10,5]]",
"[[1,3,14,15],[1,2,14,7],[1,1,10,15]]",
"[[5,15,0,3],[3,6,0,3],[7,0,0,7],[7,0,0,3],[3,9,24,3]]",
"[[5,15,0,3],[3,6,0,3],[7,0,0,7],[3,9,24,3],[5,0,0,1]]",
"[[5,15,0,3],[3,6,0,3],[7,21,24,7],[7,0,0,7],[5,0,0,1]]",
"[[1,2,24,7],[7,0,0,7],[3,9,24,7],[3,0,0,3],[7,7,24,7]]",
"[[1,2,24,7],[7,0,0,7],[5,0,0,1],[3,0,0,3],[7,7,24,7]]",
"[[7,7,0,1],[7,21,24,7],[7,0,0,7],[3,9,0,1],[3,0,0,3]]",
"[[7,7,0,1],[7,21,24,7],[7,0,0,7],[5,0,0,1],[3,0,0,3]]",
"[[3,3,28,7],[1,7,28,7],[3,15,12,3],[3,0,0,3]]",
"[[1,0,0,5],[3,3,28,7],[1,7,28,7],[3,0,0,3]]",
"[[7,14,0,1],[3,18,0,1],[3,0,0,3],[7,3,12,3]]",
"[[3,18,0,1],[5,0,0,1],[3,0,0,3],[7,3,12,3]]",
"[[5,15,0,3],[3,6,0,3],[7,21,24,7],[7,0,0,7],[7,0,0,3]]",
"[[4,3,3,4],[1,3,1,14]]", "[[2,7,13,3],[4,1,9,12]]",
"[[4,7,15,12],[7,14,7,9],[2,1,11,9]]",
"[[7,0,0,7],[5,2,0,1],[5,2,0,3]]", "[[7,3,0,1],[1,4,0,1],[3,3,2,1]]",
"[[1,1,0,3],[7,0,0,7],[7,0,0,3]]", "[[7,0,0,7],[5,0,0,1],[3,3,0,1]]",
"[[3,0,4,5],[7,0,0,7],[1,0,0,3],[7,4,0,7]]",
"[[5,4,2,3],[7,0,0,7],[7,4,0,7],[3,0,0,7]]",
"[[5,0,4,7],[7,0,0,7],[5,2,0,3]]",
"[[7,0,0,7],[3,4,6,5],[7,4,0,3],[1,0,4,1]]",
"[[7,0,0,7],[3,0,4,1],[3,0,4,7],[7,4,0,7]]",
"[[7,0,0,7],[1,0,0,7],[7,6,0,3]]",
"[[5,0,4,1],[7,0,0,7],[5,0,6,3],[7,0,4,7]]",
"[[3,6,0,7],[7,0,0,7],[5,7,0,7]]",
"[[5,0,4,1],[7,0,0,7],[3,4,6,5],[7,4,0,7]]",
"[[7,0,0,7],[1,0,0,7],[1,2,4,5]]", "[[3,6,0,7],[7,0,0,7],[7,0,0,1]]",
"[[5,4,2,3],[1,0,0,5],[7,0,0,7],[7,4,0,7]]",
"[[5,4,2,3],[7,0,0,7],[3,0,4,7],[1,0,4,1]]",
"[[1,2,4,3],[7,0,0,7],[1,6,4,3],[3,0,4,7]]",
"[[5,0,4,1],[7,0,0,7],[1,4,0,5],[1,0,2,3]]", "[[1,0,6,7],[3,3,0,1]]",
"[[7,0,0,7],[5,2,4,1],[3,6,4,5]]", "[[1,2,4,7],[7,0,0,7],[7,0,0,5]]",
"[[7,0,0,7],[5,2,0,7],[5,0,0,3]]",
"[[7,0,0,7],[3,4,6,5],[7,0,4,7],[7,0,4,3]]",
"[[7,7,8,3],[9,0,0,1],[1,0,8,3],[7,0,8,5]]",
"[[1,1,0,7],[7,7,0,1],[5,15,8,1],[1,3,0,7]]",
"[[3,0,8,7],[7,0,0,7],[1,0,8,7],[1,2,0,5]]",
"[[1,2,8,5],[7,0,0,7],[5,0,0,1],[7,14,0,5]]",
"[[7,0,0,15],[3,6,0,9],[1,0,0,7],[5,0,0,7],[1,0,0,9]]",
"[[7,0,0,15],[7,0,0,7],[7,0,0,1],[5,5,0,1]]",
"[[9,0,8,7],[1,0,8,7],[15,0,8,1],[3,3,8,1]]",
"[[9,0,0,1],[9,0,0,7],[15,0,0,7],[3,3,8,7]]",
"[[7,14,0,7],[1,2,0,1],[3,9,0,1],[3,9,8,7]]",
"[[7,0,0,15],[1,0,0,7],[5,10,8,1],[1,0,0,9],[5,0,8,1]]",
"[[7,0,0,7],[7,0,8,3],[3,9,0,1],[3,3,0,1]]",
"[[1,0,0,3],[1,1,0,5],[15,0,0,7],[7,0,0,5]]",
"[[15,14,0,3],[11,11,8,5],[15,0,0,7],[11,11,8,1],[3,3,8,1]]",
"[[7,0,0,15],[7,0,0,7],[5,10,0,1],[5,0,0,1],[3,3,0,1]]",
"[[1,0,8,1],[15,12,2,1],[1,4,0,1],[9,13,2,7]]",
"[[7,0,0,15],[7,0,0,7],[5,5,0,1],[3,0,0,1]]",
"[[9,0,0,1],[15,14,0,3],[9,0,0,3],[1,0,0,7],[15,0,0,7]]",
"[[9,0,0,1],[7,0,8,3],[1,0,0,7],[15,14,8,5],[15,0,0,7]]",
"[[1,7,14,7],[1,0,8,1],[1,6,14,15],[1,4,0,1]]",
"[[3,6,0,7],[7,0,0,7],[7,0,8,3],[1,0,8,7]]",
"[[7,0,0,7],[7,0,0,3],[1,0,8,7],[5,10,8,1]]",
"[[7,0,0,7],[5,5,8,7],[3,9,8,1],[5,0,8,1]]",
"[[7,0,0,7],[5,5,16,5],[5,0,0,1],[5,0,0,3]]",
"[[9,0,0,1],[13,13,16,5],[15,0,0,7],[15,0,8,1],[15,0,0,3]]",
"[[9,0,0,1],[15,0,16,3],[15,0,24,1],[13,13,16,5],[15,0,0,7]]",
"[[1,1,0,1],[7,0,0,7],[5,0,16,1],[1,1,24,7]]",
"[[1,1,0,1],[7,0,0,7],[1,1,24,7],[3,3,0,7]]",
"[[7,0,0,7],[3,0,16,1],[7,7,16,1],[7,0,16,3]]",
"[[7,7,0,3],[7,0,0,7],[5,5,16,5],[5,0,0,3]]",
"[[7,0,0,7],[7,0,0,1],[7,0,0,3],[3,3,16,1]]",
"[[1,1,0,1],[7,0,0,15],[7,0,0,1],[7,0,0,7],[5,0,0,1]]",
"[[7,0,0,7],[3,3,16,1],[7,0,16,1],[7,0,16,3]]",
"[[7,0,0,7],[7,0,0,3],[7,7,16,1],[3,0,0,1]]",
"[[1,1,0,1],[7,0,0,15],[7,0,0,7],[5,0,16,1],[7,0,16,1]]",
"[[7,0,0,15],[7,12,0,11],[3,12,12,11],[3,12,0,3],[3,12,10,5]]",
"[[7,0,0,15],[7,12,0,11],[3,12,12,11],[3,12,0,3],[1,0,10,15]]",
"[[3,0,12,7],[7,0,0,15],[3,12,12,11],[3,12,0,3],[3,12,10,5]]",
"[[3,0,12,7],[7,0,0,15],[3,12,12,11],[3,12,0,3],[1,0,10,15]]",
"[[9,0,8,7],[15,14,0,3],[7,0,8,3],[1,0,0,7],[15,0,0,7]]",
"[[9,0,8,7],[7,0,8,3],[1,0,0,7],[15,14,8,5],[15,0,0,7]]",
"[[9,0,8,7],[9,0,0,3],[1,0,0,7],[15,14,8,5],[15,0,0,7]]",
"[[7,0,0,15],[1,0,8,15],[3,6,0,9],[1,0,0,7],[5,0,0,7]]",
"[[7,0,0,15],[7,0,8,15],[1,0,8,15],[5,10,8,1],[5,0,0,7]]",
"[[7,0,0,15],[1,0,8,15],[1,0,0,7],[5,10,8,1],[5,0,8,1]]",
"[[7,0,0,15],[1,0,8,15],[1,0,0,7],[5,10,8,1],[5,0,0,7]]",
"[[5,5,0,5],[15,0,0,7],[15,0,8,1],[15,0,0,3]]",
"[[5,5,0,5],[3,3,8,5],[15,0,0,7],[15,0,0,3]]",
"[[7,0,0,3],[15,0,0,7],[15,0,8,1],[3,3,0,7]]",
"[[7,0,0,3],[1,0,8,3],[15,0,0,7],[3,3,0,7]]",
"[[7,7,0,7],[7,0,0,15],[1,0,0,7],[5,0,0,1]]",
"[[7,7,0,7],[7,0,0,15],[7,0,0,1],[3,0,0,7]]",
"[[7,7,0,7],[7,0,0,15],[1,0,0,7],[3,0,0,7]]",
"[[1,1,0,1],[7,0,0,15],[7,0,0,1],[5,0,0,1]]",
"[[9,0,8,7],[15,14,0,3],[9,0,0,3],[1,0,0,7],[15,0,0,7]]",
"[[1,2,0,1],[29,26,2,3],[29,29,2,3]]",
"[[11,0,24,7],[7,7,0,1],[13,0,8,5],[13,26,8,5],[15,0,0,7]]",
"[[13,13,0,5],[15,0,16,3],[15,0,24,1],[9,0,16,1],[15,0,0,7]]",
"[[7,7,0,7],[7,0,0,15],[7,0,0,1],[1,0,16,9],[5,0,0,1]]",
"[[7,7,0,7],[7,0,0,15],[1,0,16,9],[1,0,0,7],[5,0,0,1]]",
"[[19,19,2,1],[15,0,2,1],[1,0,4,1]]",
"[[19,19,2,1],[15,0,2,1],[29,0,2,3]]",
"[[11,0,24,7],[9,18,0,1],[13,0,8,5],[11,11,8,5],[15,0,0,7]]",
"[[13,13,0,5],[15,0,16,3],[9,0,16,1],[15,0,0,7],[15,0,8,1]]",
"[[11,0,24,7],[13,0,8,5],[13,26,8,5],[11,11,8,5],[15,0,0,7]]",
"[[13,13,0,5],[15,0,24,1],[9,0,16,1],[15,0,0,7],[15,0,0,3]]",
"[[11,0,24,7],[7,7,0,1],[9,18,0,1],[13,0,8,5],[15,0,0,7]]",
"[[13,13,0,5],[9,0,16,1],[15,0,0,7],[15,0,8,1],[15,0,0,3]]",
"[[1,3,2,19],[1,2,10,31],[1,1,10,31]]",
"[[7,7,0,7],[7,0,0,15],[7,0,0,1],[1,0,16,9],[3,0,0,7]]",
"[[7,7,0,7],[7,0,0,15],[1,0,16,9],[1,0,0,7],[3,0,0,7]]",
"[[5,4,2,3],[5,0,4,1],[7,0,0,7],[3,0,0,3]]",
"[[3,7,0,5],[3,4,4,7],[3,0,0,3]]", "[[1,5,0,7],[1,2,6,7]]",
"[[1,5,2,7],[1,4,6,7]]", "[[5,4,2,3],[7,4,4,3],[7,0,0,7],[3,0,0,3]]",
"[[3,4,0,7],[7,0,0,7],[3,4,6,5],[3,0,0,3]]",
"[[5,0,0,5],[5,1,2,3],[5,0,2,7]]", "[[1,5,4,7],[3,0,4,7],[3,0,0,3]]",
"[[9,0,0,1],[11,0,0,3],[7,0,0,7],[13,7,0,7],[11,6,0,7]]",
"[[1,0,10,7],[9,0,0,1],[5,0,0,5],[11,11,0,1]]",
"[[9,0,0,1],[3,12,2,5],[1,4,0,1],[7,3,4,1]]",
"[[11,0,0,3],[7,0,0,7],[13,13,14,7],[15,0,2,1]]",
"[[7,14,0,1],[7,0,0,3],[9,9,6,3]]",
"[[7,7,10,5],[5,15,10,5],[7,0,0,3]]",
"[[7,14,0,1],[9,9,6,3],[7,7,2,1]]",
"[[7,7,10,5],[5,15,10,5],[7,7,2,1]]",
"[[7,7,8,7],[7,14,0,7],[1,2,8,7],[3,9,8,7]]",
"[[7,7,8,7],[7,14,0,7],[1,2,8,7],[5,0,0,1]]",
"[[7,7,8,7],[7,14,0,7],[3,9,8,7],[7,5,0,1]]",
"[[7,10,0,1],[7,0,0,3],[13,11,2,1]]",
"[[7,0,0,3],[3,5,14,7],[7,7,2,1]]",
"[[7,10,0,1],[3,5,6,3],[13,11,2,1]]",
"[[3,5,6,3],[3,5,14,7],[7,7,2,1]]",
"[[7,7,8,7],[7,14,0,7],[5,0,0,1],[7,5,0,1]]",
"[[9,0,0,7],[15,0,0,7],[11,11,8,3],[3,3,8,7]]",
"[[11,11,8,5],[15,0,0,7],[11,11,8,3],[3,3,8,7]]",
"[[9,0,0,7],[15,0,0,7],[11,11,8,3],[15,0,0,3]]",
"[[7,14,0,1],[1,5,6,11],[3,0,0,7]]",
"[[1,2,0,7],[1,5,6,11],[3,0,0,15]]",
"[[1,2,0,7],[1,5,6,11],[3,0,0,7]]",
"[[7,14,0,1],[5,0,0,1],[1,5,6,3]]",
"[[11,11,8,5],[15,0,0,7],[11,11,8,3],[15,0,0,3]]",
"[[1,8,4,5],[1,13,10,15],[1,8,0,1],[1,5,0,15]]",
"[[1,5,0,7],[1,5,10,15],[1,0,4,5]]",
"[[1,5,10,15],[1,4,14,7],[1,1,14,15]]",
"[[7,5,6,11],[5,0,0,1],[3,2,0,1]]",
"[[7,5,6,11],[5,0,0,1],[1,5,6,3]]",
"[[7,5,6,11],[3,9,6,3],[3,2,0,1]]",
"[[7,5,6,11],[3,9,6,3],[1,5,6,3]]",
"[[1,4,14,15],[1,5,10,15],[1,1,14,15]]",
"[[3,0,6,9],[1,5,12,7],[1,5,10,7],[3,0,0,3]]",
"[[1,5,0,7],[1,1,12,7],[1,5,10,7]]",
"[[1,0,2,3],[1,4,14,7],[1,5,10,7]]",
"[[1,2,8,13],[1,3,14,7],[1,1,4,11]]",
"[[11,11,4,1],[5,10,0,1],[11,1,2,5]]",
"[[1,5,0,7],[1,0,14,15],[1,5,10,7]]",
"[[5,5,2,3],[5,10,0,1],[11,11,8,1]]",
"[[13,10,0,1],[5,15,4,3],[13,10,2,3]]",
"[[13,10,8,1],[5,15,4,3],[13,10,2,3]]",
"[[5,10,8,1],[11,11,8,1],[9,9,2,7]]",
"[[3,12,12,3],[7,0,0,7],[3,0,10,13],[3,0,12,15],[3,0,0,3]]",
"[[5,5,2,3],[13,10,8,1],[15,13,8,1]]",
"[[5,5,2,3],[5,10,0,1],[3,3,8,1]]",
"[[5,10,8,1],[9,9,2,7],[3,3,8,1]]",
"[[5,5,2,3],[13,10,8,1],[7,5,8,1]]",
"[[13,10,0,1],[15,13,0,1],[9,9,2,7]]",
"[[9,0,0,1],[11,0,0,3],[7,0,0,7],[13,10,8,1],[3,9,8,1]]",
"[[5,15,0,7],[11,0,0,3],[7,0,0,7],[7,0,8,7],[15,15,0,1]]",
"[[1,3,10,15],[1,3,14,7],[1,1,0,15]]",
"[[1,3,10,15],[1,3,14,15],[1,6,2,3]]",
"[[1,6,2,11],[1,1,12,7],[1,7,10,15]]",
"[[1,15,10,15],[1,6,2,11],[1,11,14,15],[1,8,0,1]]",
"[[1,5,0,7],[1,0,0,9],[1,6,14,7],[1,4,0,1]]",
"[[5,5,0,3],[1,4,0,1],[1,1,6,3],[11,12,0,3]]",
"[[1,8,0,9],[3,0,0,13],[1,2,2,1],[3,0,0,3]]",
"[[1,8,0,9],[3,6,10,5],[3,0,0,13],[3,0,0,3]]",
"[[7,12,12,3],[7,0,0,7],[3,0,14,5],[3,12,0,11],[7,12,0,7]]",
"[[7,0,0,7],[5,0,0,1],[3,0,0,3],[3,3,0,1]]",
"[[7,0,0,15],[1,2,12,9],[5,0,4,7],[1,0,12,15],[1,0,4,15]]",
"[[7,0,0,15],[1,2,12,9],[1,0,12,15],[1,0,4,15],[5,0,0,1]]",
"[[7,0,0,15],[1,2,12,9],[5,0,4,7],[1,0,4,15],[1,0,8,9]]",
"[[1,4,12,7],[7,0,0,15],[1,6,4,1],[1,4,0,1],[5,0,0,1]]",
"[[9,0,12,7],[9,0,4,7],[9,0,4,3],[15,0,0,7],[1,2,12,1]]",
"[[9,0,8,1],[9,0,4,7],[9,0,4,3],[15,0,0,7],[1,2,12,1]]",
"[[7,0,0,3],[9,0,12,7],[9,0,4,7],[15,0,0,7],[1,2,12,1]]",
"[[1,4,12,7],[7,0,0,3],[15,0,0,7],[1,4,0,1],[9,6,4,1]]",
"[[3,6,0,7],[7,0,0,7],[7,0,0,1],[3,0,0,3]]",
"[[7,0,0,7],[1,0,8,7],[5,10,8,1],[3,0,0,3]]",
"[[3,0,12,7],[3,12,12,3],[7,0,0,7],[1,0,4,13],[1,0,2,11]]",
"[[13,10,0,1],[9,9,2,7],[7,5,0,1]]",
"[[11,2,0,1],[15,30,0,1],[1,1,6,3]]",
"[[7,0,0,7],[5,5,0,1],[3,0,0,1],[3,0,0,3]]",
"[[25,18,2,7],[25,25,2,7],[1,0,8,1],[29,26,0,1]]",
"[[25,18,2,7],[25,25,2,7],[1,0,8,1],[25,11,2,7]]",
"[[7,0,0,7],[3,0,16,1],[3,0,0,3],[5,5,16,1]]",
"[[7,0,0,7],[7,0,0,1],[5,5,0,1],[3,0,0,3]]",
"[[7,0,0,7],[7,0,16,1],[3,0,0,3],[5,5,16,1]]",
"[[1,1,28,15],[3,3,12,11],[3,15,12,3],[3,0,0,3]]",
"[[3,6,0,1],[7,0,0,7],[1,0,24,7],[7,0,0,3],[5,20,24,7]]",
"[[7,0,0,7],[1,0,24,7],[1,4,24,3],[5,0,0,1],[3,6,24,7]]",
"[[23,19,4,1],[25,4,0,1],[15,9,2,1],[21,0,4,1]]",
"[[7,0,0,7],[5,0,16,1],[7,14,16,1],[7,0,16,1],[7,0,16,3]]",
"[[29,26,0,1],[9,9,2,7],[29,29,2,3],[5,10,2,3]]",
"[[7,14,0,1],[7,0,0,1],[7,0,0,7],[7,0,0,3],[5,0,0,1]]",
"[[21,10,2,3],[19,19,2,1],[19,19,0,1]]",
"[[15,28,0,1],[3,3,12,3],[9,4,0,1],[3,0,0,3]]",
"[[1,2,20,5],[1,6,20,29],[1,2,30,15],[1,5,28,15]]",
"[[29,0,4,1],[31,27,0,1],[13,7,2,3],[1,4,0,1]]",
"[[1,2,20,5],[1,6,20,29],[1,0,10,31],[1,5,28,15]]",
"[[21,20,2,3],[15,13,0,1],[1,4,0,1],[31,30,2,1]]",
"[[5,10,16,1],[7,0,0,7],[5,0,16,1],[3,3,16,1],[3,0,0,3]]",
"[[21,20,2,3],[1,4,0,1],[31,30,2,1],[5,3,2,3]]",
"[[7,0,0,7],[5,10,0,1],[5,0,0,1],[3,0,0,3],[3,3,0,1]]",
"[[7,0,0,15],[1,2,0,1],[7,21,24,7],[7,21,0,1],[5,0,0,1]]",
"[[27,27,4,1],[21,21,2,3],[5,0,2,3]]",
"[[25,18,2,7],[1,0,8,1],[13,26,0,1],[7,7,2,1]]",
"[[7,0,0,15],[1,2,0,1],[7,7,0,9],[7,7,24,15],[5,10,0,9]]",
"[[7,28,0,3],[7,6,0,1],[5,15,2,1]]",
"[[7,0,0,15],[1,2,0,1],[7,7,0,9],[7,7,24,15],[5,0,0,1]]",
"[[3,21,6,3],[7,6,0,1],[5,15,2,1]]",
"[[7,20,0,3],[5,3,2,1],[7,7,2,1]]", "[[5,3,2,1],[7,7,2,1],[1,5,6,3]]",
"[[7,0,0,15],[1,2,0,1],[7,21,0,1],[7,7,24,15],[5,10,0,9]]",
"[[29,0,4,1],[31,27,0,1],[1,4,0,1],[31,31,2,1]]",
"[[25,0,8,1],[9,9,4,3],[15,13,2,1],[29,23,4,3]]",
"[[1,0,20,21],[1,1,30,31],[1,5,28,31],[1,4,0,1]]",
"[[11,2,0,1],[3,9,6,3],[9,9,6,3]]",
"[[3,6,0,1],[5,25,6,3],[15,26,0,1]]",
"[[1,0,20,5],[1,1,10,15],[1,5,28,7],[1,4,20,13]]",
"[[7,10,0,1],[3,6,0,1],[5,25,6,3]]",
"[[9,18,0,1],[9,0,0,7],[3,9,24,7],[11,11,24,3],[15,0,0,7]]",
"[[5,10,4,1],[31,31,0,1],[25,18,2,3]]",
"[[25,0,2,3],[21,21,2,3],[21,10,0,1]]",
"[[9,18,0,1],[3,9,24,7],[11,11,8,5],[11,11,24,3],[15,0,0,7]]",
"[[9,18,0,1],[11,11,8,5],[15,30,0,3],[11,11,24,3],[15,0,0,7]]",
"[[9,18,0,1],[9,0,0,7],[15,30,0,3],[11,11,24,3],[15,0,0,7]]",
"[[1,0,0,5],[5,10,0,5],[7,0,0,7],[3,9,0,1],[3,0,16,5]]",
"[[1,1,0,1],[7,0,0,7],[3,0,0,5],[5,0,0,1],[3,0,32,3]]",
"[[7,0,0,7],[3,0,16,1],[5,0,0,5],[7,14,16,1],[5,5,16,1]]",
"[[1,1,0,1],[1,0,0,5],[7,0,0,7],[3,0,0,5],[3,0,32,3]]",
"[[7,0,0,7],[5,0,0,5],[3,9,0,1],[5,0,0,1],[7,14,16,1]]",
"[[1,1,0,1],[7,0,0,7],[5,0,0,5],[1,0,32,7],[3,0,32,7]]",
"[[1,1,0,1],[7,0,0,1],[7,0,0,7],[5,0,0,5],[5,0,0,1]]",
"[[7,21,0,5],[5,10,0,5],[7,0,0,7],[5,0,0,1],[3,0,16,5]]",
"[[7,0,0,7],[7,0,6,9],[3,12,0,3],[7,12,0,15],[3,12,12,15]]",
"[[11,0,0,3],[7,0,0,7],[13,10,8,1],[3,9,8,1],[15,0,8,7]]",
"[[21,30,4,1],[9,4,0,1],[29,0,2,3],[15,11,4,1]]",
"[[21,30,4,1],[9,4,0,1],[15,30,2,1],[15,11,4,1]]",
"[[19,19,4,1],[9,9,4,3],[15,13,2,1],[29,23,4,3]]",
"[[25,0,8,1],[13,7,2,3],[29,29,2,3],[5,10,2,3]]",
"[[1,7,30,15],[1,5,28,7],[1,1,28,31],[1,0,20,29]]",
"[[25,0,8,1],[23,5,0,1],[13,13,2,3],[11,11,0,1]]",
"[[21,21,2,3],[17,0,8,1],[15,13,8,1],[13,26,8,1]]",
"[[1,6,18,27],[1,5,28,7],[1,1,28,31],[1,0,20,29]]",
"[[9,18,2,7],[15,13,4,1],[17,0,8,1],[29,26,8,1]]",
"[[3,6,0,1],[7,0,0,7],[1,0,24,7],[1,4,24,3],[5,0,0,1]]",
"[[7,0,0,7],[1,0,24,7],[7,0,0,3],[5,20,24,7],[3,6,24,7]]",
"[[15,13,4,1],[17,0,8,1],[29,26,2,3],[29,26,8,1]]",
"[[1,3,4,9],[1,7,4,1],[3,3,20,5],[3,0,0,3]]",
"[[7,10,0,1],[9,31,4,1],[15,30,0,1],[3,0,0,3]]",
"[[21,20,2,3],[15,13,0,1],[13,0,2,3],[31,30,2,1]]",
"[[29,0,4,1],[31,27,0,1],[13,7,2,3],[5,3,2,3]]",
"[[1,2,18,11],[1,3,10,15],[1,0,20,21],[1,7,26,7]]",
"[[21,20,2,3],[13,0,2,3],[31,30,2,1],[5,3,2,3]]",
"[[23,5,0,1],[21,21,2,3],[17,0,8,1],[13,26,8,1]]",
"[[1,5,0,31],[1,0,20,21],[1,1,30,31],[1,4,20,21]]",
"[[1,0,20,21],[1,1,30,31],[1,4,20,21],[1,5,28,31]]",
"[[23,19,4,1],[13,0,4,1],[25,4,0,1],[13,13,2,3]]",
"[[23,19,4,1],[13,0,4,1],[15,15,2,1],[25,4,0,1]]",
"[[19,19,4,1],[9,18,2,7],[9,9,4,3],[29,23,4,3]]",
"[[7,21,0,5],[1,0,0,5],[5,10,0,5],[7,0,0,7],[3,0,16,5]]",
"[[3,3,0,3],[1,0,0,5],[7,0,0,7],[3,0,0,5],[3,0,32,3]]",
"[[3,3,0,3],[7,0,0,7],[3,0,0,5],[5,0,0,1],[3,0,32,3]]",
"[[7,7,32,7],[7,0,0,7],[5,0,0,5],[1,0,32,7],[3,0,32,7]]",
"[[7,7,32,7],[7,0,0,7],[5,0,0,5],[1,0,32,7],[5,0,0,1]]",
"[[5,10,0,5],[7,0,0,7],[3,9,0,1],[5,0,0,1],[3,0,16,5]]",
"[[7,0,0,7],[5,0,0,5],[5,0,0,1],[7,14,16,1],[5,5,16,1]]",
"[[7,0,0,7],[3,0,16,1],[5,0,0,5],[3,9,0,1],[7,14,16,1]]",
"[[3,6,0,1],[13,1,6,3],[15,13,6,3]]",
"[[19,19,0,1],[27,27,4,1],[21,21,2,3]]",
"[[25,0,8,1],[9,9,4,3],[15,13,2,1],[21,10,0,1]]",
"[[3,21,6,3],[7,31,2,1],[5,15,2,1]]",
"[[7,14,0,1],[11,2,0,1],[1,1,6,3]]",
"[[1,13,6,3],[3,1,6,3],[7,31,2,1]]",
"[[3,9,6,3],[9,9,6,3],[7,17,6,3]]",
"[[7,2,0,1],[7,31,2,1],[1,5,6,3]]",
"[[1,1,30,15],[1,0,20,5],[1,5,28,7],[1,4,20,13]]",
"[[13,7,2,3],[9,9,2,7],[29,29,2,3],[5,10,2,3]]",
"[[23,19,4,1],[31,27,0,1],[25,4,0,1],[15,9,2,1]]",
"[[7,28,0,3],[7,31,2,1],[5,15,2,1]]",
"[[25,0,2,3],[1,2,2,3],[21,21,2,3]]",
"[[13,1,6,3],[15,13,6,3],[11,13,6,3]]", "[[4,1,17,4],[2,3,17,23]]",
"[[4,1,1,4],[2,3,1,7]]",
"[[7,0,0,15],[1,0,40,7],[15,0,0,15],[1,0,16,13],[3,3,48,11]]",
"[[15,0,40,13],[9,0,16,5],[13,13,0,1],[15,0,0,15],[1,0,16,5]]",
"[[1,1,0,1],[7,0,0,15],[1,0,0,7],[5,0,0,1],[1,0,32,9]]",
"[[1,1,0,1],[7,0,0,15],[1,0,0,7],[3,0,0,7],[1,0,32,9]]",
"[[7,7,0,7],[7,0,0,15],[1,0,0,7],[5,0,0,1],[1,0,32,9]]",
"[[7,7,0,7],[7,0,0,15],[1,0,0,7],[3,0,0,7],[1,0,32,9]]",
"[[7,0,0,15],[1,0,40,7],[15,0,0,15],[1,0,16,13],[13,13,16,13]]",
"[[15,0,40,13],[9,0,16,5],[15,0,0,15],[1,0,16,5],[13,13,16,13]]"];
finegenlists := [ "[[1,1,1,2],[1,0,2,3]]", "[[1,1,3,0],[1,0,6,7]]",
"[[2,3,3,1],[1,0,3,3]]", "[[0,1,3,1],[1,0,3,3]]",
"[[1,2,2,5],[1,0,0,7],[3,6,0,5],[1,2,0,5]]",
"[[1,0,0,3],[3,2,0,1],[1,2,2,1]]",
"[[1,0,2,1],[1,0,0,7],[3,6,0,5],[1,2,0,5]]",
"[[1,0,0,3],[1,0,2,1],[3,2,0,1]]", "[[1,0,6,3],[1,0,2,7],[1,1,2,5]]",
"[[1,0,0,5],[1,0,6,3],[1,1,2,5]]", "[[1,1,0,1],[7,0,2,1],[1,0,6,3]]",
"[[1,1,0,1],[1,0,6,3],[5,0,0,1]]", "[[1,0,2,1],[3,3,6,3],[3,0,6,7]]",
"[[3,0,2,3],[3,3,2,3]]", "[[1,0,2,5],[1,1,0,3],[3,3,0,5]]",
"[[1,1,0,3],[1,0,2,1]]", "[[1,1,4,5],[1,0,4,5],[1,0,0,3],[3,0,4,1]]",
"[[1,0,0,5],[1,1,4,5],[1,0,0,3],[3,0,4,1]]",
"[[1,1,0,1],[5,0,4,1],[1,0,0,3],[3,0,4,1]]",
"[[1,1,0,1],[1,0,0,5],[1,0,0,3],[3,0,4,1]]",
"[[5,5,0,1],[1,0,0,3],[3,3,0,1],[1,0,4,1]]", "[[1,0,0,3],[3,3,0,1]]",
"[[1,1,0,3],[5,5,0,1],[3,0,0,1],[1,0,4,1]]", "[[3,0,0,1],[1,1,0,3]]",
"[[1,1,3,0],[1,0,3,7]]", "[[1,0,3,3],[3,3,1,0]]",
"[[1,2,6,3],[3,6,2,3],[1,0,0,3],[3,6,6,7]]",
"[[1,0,0,5],[1,2,6,3],[3,6,2,3],[1,0,0,3]]",
"[[1,2,6,3],[1,0,0,3],[3,6,6,7],[1,0,4,1]]", "[[1,0,0,3],[1,2,2,3]]",
"[[1,0,0,3],[3,2,2,3]]", "[[3,0,0,5],[1,0,0,3],[3,0,4,1],[3,6,6,3]]",
"[[1,0,0,5],[1,0,0,3],[3,0,4,1],[3,6,6,3]]",
"[[3,0,0,5],[1,0,0,3],[1,0,4,1],[3,6,6,3]]",
"[[1,0,0,5],[1,2,0,1],[1,0,0,3],[3,0,0,1],[1,0,4,1]]",
"[[1,0,0,5],[1,2,0,1],[5,0,4,1],[7,0,0,1],[5,0,0,3]]",
"[[1,2,0,1],[5,0,4,1],[7,0,0,1],[5,0,0,5],[1,0,0,3]]",
"[[5,0,4,1],[7,0,0,1],[5,0,0,5],[1,0,0,3],[1,2,0,5]]",
"[[1,2,0,1],[7,0,0,1],[5,0,0,5],[1,0,0,3],[1,0,4,1]]",
"[[7,0,0,1],[5,0,0,5],[1,0,0,3],[1,2,0,5],[1,0,4,1]]",
"[[1,2,0,1],[5,0,0,5],[3,0,0,1],[5,0,0,3],[1,0,4,1]]",
"[[1,0,0,3],[1,2,0,1]]", "[[1,0,0,3],[3,2,0,1]]",
"[[1,2,0,1],[3,0,0,5],[3,0,4,1],[5,0,0,1],[3,0,0,3]]",
"[[3,0,0,5],[3,0,4,1],[5,0,0,1],[1,2,0,5],[3,0,0,3]]",
"[[1,0,0,5],[1,2,0,1],[1,0,0,3],[3,0,4,1],[5,0,0,1]]",
"[[3,6,0,1],[1,0,0,5],[1,0,0,3],[3,0,4,1],[5,0,0,1]]",
"[[3,0,0,1],[1,2,0,1]]", "[[3,3,4,5],[1,2,4,1],[1,0,4,5],[3,0,4,3]]",
"[[3,3,4,5],[1,2,4,1],[1,0,0,5],[3,0,4,3]]",
"[[5,0,4,1],[1,2,0,1],[7,7,4,1],[3,0,4,3]]",
"[[3,3,4,5],[1,0,0,5],[1,2,0,1],[3,0,4,3]]",
"[[3,6,0,3],[3,1,0,1],[5,2,0,1],[1,0,4,1]]", "[[3,1,0,1],[3,2,0,3]]",
"[[7,7,0,1],[1,1,0,3],[1,2,0,1],[1,0,4,1]]", "[[1,1,0,3],[1,2,0,1]]",
"[[3,9,4,1],[3,3,12,7],[1,3,4,5],[3,9,12,3]]",
"[[3,9,4,1],[1,0,0,5],[3,3,12,7],[1,3,4,5]]",
"[[3,3,4,1],[3,3,4,3],[5,5,4,1]]", "[[3,3,4,3],[5,0,0,3],[5,5,4,1]]",
"[[3,3,4,1],[5,0,0,1],[5,5,4,1]]", "[[1,0,0,5],[1,1,4,5],[1,0,0,7]]",
"[[1,2,0,7],[3,6,0,3],[1,2,0,5],[3,9,12,7]]",
"[[1,0,0,5],[1,2,0,7],[3,6,0,3],[3,9,12,7]]",
"[[1,0,0,5],[3,3,4,7],[1,0,0,7]]", "[[7,0,0,1],[7,0,0,3],[7,7,4,3]]",
"[[3,3,4,7],[1,0,0,7],[3,0,0,7]]", "[[7,0,0,1],[7,7,4,3],[5,0,0,1]]",
"[[3,3,4,1],[5,5,4,1],[7,0,4,3]]", "[[5,0,0,3],[5,5,4,1],[7,0,4,3]]",
"[[3,3,4,1],[5,5,0,1],[5,5,4,1]]", "[[5,5,0,1],[5,0,0,3],[5,5,4,1]]",
"[[1,1,0,3],[7,7,0,3],[7,7,4,3]]", "[[7,7,0,3],[7,7,4,3],[5,0,0,3]]",
"[[5,5,0,1],[7,7,4,3],[5,0,0,3]]", "[[7,0,0,1],[5,5,0,1],[7,7,4,3]]",
"[[3,3,4,5],[1,1,4,5],[3,3,0,5]]", "[[1,1,4,5],[1,0,0,3],[3,3,0,5]]",
"[[3,3,4,5],[1,1,4,5],[1,0,4,7]]", "[[1,1,4,5],[1,0,0,3],[1,0,4,7]]",
"[[3,3,4,7],[1,0,0,3],[1,0,4,7]]", "[[3,0,4,5],[3,3,4,7],[3,0,0,1]]",
"[[3,0,4,5],[3,3,4,7],[1,0,0,3]]", "[[3,3,4,7],[3,0,0,1],[1,0,4,7]]",
"[[1,1,0,1],[7,0,0,1],[5,0,0,1],[3,0,0,3]]",
"[[1,1,0,1],[7,0,8,7],[7,0,0,1],[5,0,0,1],[3,0,0,3]]",
"[[7,7,0,7],[5,0,0,5],[1,0,0,7],[3,0,0,7]]",
"[[7,7,0,7],[7,0,8,1],[5,0,0,5],[1,0,0,7],[3,0,0,7]]",
"[[7,7,0,7],[7,0,0,1],[5,0,0,5],[3,0,0,7]]",
"[[7,7,0,7],[5,0,0,5],[1,0,0,7],[5,0,0,1]]",
"[[7,7,0,7],[7,0,8,1],[5,0,0,5],[1,0,0,7],[5,0,0,1]]",
"[[7,7,0,7],[7,0,0,1],[5,0,0,5],[5,0,0,1]]",
"[[1,1,0,1],[5,0,0,5],[1,0,0,7],[3,0,0,7]]",
"[[1,1,0,1],[7,0,8,1],[5,0,0,5],[1,0,0,7],[3,0,0,7]]",
"[[1,1,0,1],[7,0,0,1],[5,0,0,5],[3,0,0,7]]",
"[[1,1,0,1],[5,0,0,5],[1,0,0,7],[5,0,0,1]]",
"[[1,1,0,1],[7,0,8,1],[5,0,0,5],[1,0,0,7],[5,0,0,1]]",
"[[1,1,0,1],[7,0,0,1],[5,0,0,5],[5,0,0,1]]",
"[[7,7,0,7],[1,0,0,7],[3,0,0,7],[3,0,0,3]]",
"[[7,7,0,7],[7,0,8,1],[1,0,0,7],[3,0,0,7],[3,0,0,3]]",
"[[7,7,0,7],[7,0,0,1],[3,0,0,7],[3,0,0,3]]",
"[[7,7,0,7],[7,0,8,7],[7,0,0,1],[3,0,0,7],[3,0,0,3]]",
"[[7,7,0,7],[1,0,0,7],[5,0,0,1],[3,0,0,3]]",
"[[7,7,0,7],[7,0,8,7],[7,0,0,1],[5,0,0,1],[3,0,0,3]]",
"[[1,0,14,7],[3,0,10,5],[3,3,14,7]]", "[[3,0,2,5],[3,3,6,7]]",
"[[1,0,14,7],[3,0,10,5],[1,1,0,5]]", "[[7,0,2,1],[5,5,0,1]]",
"[[3,6,2,5],[3,6,0,5],[1,2,0,5]]", "[[1,0,0,3],[3,6,0,5],[3,0,6,7]]",
"[[1,0,6,7],[1,0,0,3],[3,6,6,7]]", "[[3,0,2,5],[3,0,0,1],[3,6,6,7]]",
"[[1,1,6,11],[1,3,14,15],[1,3,2,1]]",
"[[1,1,6,11],[1,3,14,15],[1,0,0,7]]",
"[[1,3,14,15],[1,2,0,9],[1,3,2,1]]",
"[[1,3,14,15],[1,0,0,7],[1,2,0,9]]",
"[[3,0,14,15],[1,0,10,5],[3,3,14,7]]", "[[3,3,6,7],[3,0,6,7]]",
"[[7,7,0,1],[1,2,2,1],[5,0,2,1]]", "[[1,1,0,7],[1,0,2,5]]",
"[[5,0,4,3],[1,1,4,1],[3,0,4,3]]", "[[5,0,4,3],[5,5,0,1],[3,0,4,3]]",
"[[1,1,0,3],[7,7,0,3],[1,0,4,1]]", "[[1,0,0,7],[3,3,0,7],[1,0,4,1]]",
"[[1,0,2,5],[3,6,2,1],[1,2,0,5]]", "[[3,6,0,7],[3,6,2,1],[3,0,6,7]]",
"[[1,0,2,5],[1,0,0,7],[3,6,6,7]]", "[[3,0,2,1],[1,0,2,5],[3,6,6,7]]",
"[[1,0,4,3],[1,1,4,1],[3,0,4,3]]", "[[1,0,4,3],[5,5,0,1],[3,0,4,3]]",
"[[7,7,0,3],[5,5,0,3],[1,0,4,1]]", "[[1,0,0,3],[3,3,0,7],[1,0,4,1]]",
"[[1,7,1,2],[15,11,0,1]]", "[[31,27,0,1],[17,23,1,2]]",
"[[5,4,4,5],[1,0,0,5],[1,0,6,3],[3,0,0,3],[1,0,4,1]]",
"[[1,0,0,5],[1,0,6,3],[1,4,4,5],[3,0,0,3],[1,0,4,1]]",
"[[5,4,4,5],[5,0,4,1],[5,0,0,5],[7,0,0,3],[5,0,6,3]]",
"[[5,0,4,1],[5,0,0,5],[7,0,0,3],[5,4,4,1],[5,0,6,3]]",
"[[5,4,4,5],[5,0,0,5],[7,0,0,3],[5,0,6,3],[1,0,4,1]]",
"[[5,0,0,5],[7,0,0,3],[5,4,4,1],[5,0,6,3],[1,0,4,1]]",
"[[1,0,0,5],[5,0,4,1],[5,0,6,3],[1,4,4,5],[3,0,0,3]]",
"[[5,4,4,5],[1,0,0,5],[5,0,6,3],[3,0,0,3],[1,0,4,1]]", "[[1,0,2,3]]",
"[[7,4,4,7],[3,0,2,1],[1,0,0,5],[3,0,4,3],[5,0,0,1]]",
"[[5,4,4,5],[1,0,6,7],[1,0,2,3],[5,0,0,1],[3,0,0,3]]",
"[[3,3,4,5],[1,1,0,3],[3,3,0,5]]", "[[3,3,4,5],[1,0,0,5],[1,1,0,3]]",
"[[1,1,0,3],[3,3,0,5],[1,0,4,1]]", "[[1,1,0,3]]",
"[[1,2,6,7],[1,2,6,3],[3,6,10,5],[1,2,6,5]]",
"[[1,0,0,5],[1,2,6,7],[3,6,10,5],[1,2,6,5]]",
"[[3,6,2,5],[1,2,6,3],[1,2,6,5]]", "[[3,6,2,5],[1,0,0,5],[1,2,6,5]]",
"[[1,2,6,7],[1,2,6,3],[3,6,10,5],[1,0,12,3]]",
"[[1,0,0,5],[1,2,6,7],[3,6,10,5],[1,0,12,3]]",
"[[1,0,4,3],[3,6,2,5],[1,2,6,3]]", "[[1,0,4,3],[3,6,2,5],[1,0,0,5]]",
"[[5,2,4,1],[7,0,4,1],[5,0,4,5],[1,0,4,1]]",
"[[5,2,4,1],[1,0,0,7],[5,0,4,5],[1,0,4,1]]",
"[[5,2,4,1],[7,0,4,1],[3,0,0,3],[1,0,4,1]]",
"[[5,2,4,1],[1,0,0,7],[3,0,0,3],[1,0,4,1]]",
"[[3,6,0,7],[7,0,4,1],[5,0,4,5],[1,0,4,1]]",
"[[3,6,0,7],[1,0,0,7],[5,0,4,5],[1,0,4,1]]",
"[[3,6,0,1],[7,0,4,7],[5,2,0,1],[1,2,0,5]]",
"[[3,6,0,1],[7,0,4,7],[5,2,0,1],[3,0,0,3]]",
"[[1,0,0,7],[7,0,4,7],[5,2,0,1],[3,0,0,3]]",
"[[3,6,0,7],[3,0,0,5],[5,0,0,5],[7,0,4,7]]",
"[[3,6,6,11],[3,6,10,5],[1,2,2,1],[3,6,2,11]]",
"[[3,6,2,5],[3,6,2,3],[1,2,2,1]]",
"[[3,6,6,11],[3,6,10,5],[1,2,2,1],[1,0,4,1]]",
"[[3,6,2,5],[1,2,2,1],[1,0,4,1]]", "[[1,0,0,7],[1,0,4,1],[3,6,6,3]]",
"[[3,0,0,5],[3,0,4,3],[3,6,6,3]]", "[[3,6,2,5],[3,6,2,7],[1,2,2,1]]",
"[[3,6,2,5],[1,2,2,1],[1,0,4,5]]", "[[1,0,0,7],[1,0,4,5],[3,6,6,3]]",
"[[3,0,0,5],[3,0,4,7],[3,6,6,3]]",
"[[3,0,8,1],[7,0,8,7],[5,5,8,1],[1,0,8,3]]",
"[[3,0,8,1],[7,0,8,7],[5,5,8,1],[1,0,0,3]]",
"[[7,0,8,7],[5,5,8,1],[1,0,0,3],[3,0,0,1]]",
"[[3,0,8,1],[7,0,8,7],[5,5,0,1],[1,0,0,3]]",
"[[7,7,0,1],[5,5,0,3],[3,3,0,7]]", "[[7,7,0,1],[1,0,0,3],[3,3,0,7]]",
"[[5,5,0,3],[3,0,0,1],[3,3,0,7]]", "[[1,0,0,3],[3,0,0,1],[3,3,0,7]]",
"[[5,5,0,1],[1,0,0,3],[5,0,0,7]]", "[[5,5,0,1],[3,0,0,1],[7,0,0,5]]",
"[[5,5,2,3],[1,1,4,3],[1,1,6,3]]", "[[5,5,2,3],[1,1,4,3],[1,0,4,1]]",
"[[1,0,6,3],[5,5,2,3],[1,1,6,3]]", "[[1,0,6,3],[5,5,2,3],[1,0,4,1]]",
"[[7,0,8,7],[5,0,8,5],[5,5,8,1],[1,0,8,7]]",
"[[7,0,8,7],[5,0,8,5],[5,5,8,1],[1,0,0,7]]",
"[[7,0,8,7],[5,0,0,5],[5,5,8,1],[1,0,8,7]]",
"[[7,0,8,7],[5,0,0,5],[5,5,0,1],[1,0,8,7]]",
"[[5,5,0,7],[1,1,0,5],[3,3,0,7]]", "[[1,0,0,7],[1,1,0,5],[3,3,0,7]]",
"[[5,5,0,7],[5,0,0,5],[3,3,0,7]]", "[[5,0,0,5],[1,0,0,7],[3,3,0,7]]",
"[[5,0,0,5],[5,5,0,1],[1,0,0,7]]", "[[7,0,0,1],[5,0,0,5],[5,5,0,1]]",
"[[3,3,4,5],[1,3,4,9],[1,1,4,1],[3,3,12,11]]",
"[[7,7,4,1],[1,1,4,1]]",
"[[1,3,4,9],[1,0,0,7],[1,1,4,1],[3,3,12,11]]",
"[[1,0,0,7],[1,1,4,1]]",
"[[3,3,4,5],[3,6,0,3],[1,1,4,1],[3,3,12,11]]",
"[[3,3,4,5],[3,6,0,3],[1,1,4,1],[1,0,0,9]]",
"[[3,6,0,3],[1,0,0,7],[1,1,4,1],[3,3,12,11]]",
"[[7,0,0,1],[3,3,4,3]]", "[[3,3,4,3],[5,0,0,3]]",
"[[3,6,0,3],[3,3,12,3],[1,0,0,7],[1,0,0,9]]",
"[[1,0,14,7],[5,10,14,7],[1,3,14,7]]",
"[[5,10,0,5],[1,0,14,7],[1,3,14,7]]",
"[[5,15,0,7],[1,0,14,7],[5,10,14,7]]",
"[[5,15,0,7],[5,10,0,5],[1,0,14,7]]",
"[[3,3,8,3],[7,0,8,7],[1,0,8,3],[5,0,8,1]]",
"[[3,3,8,3],[7,0,8,7],[1,0,0,3],[5,0,8,1]]",
"[[3,3,8,3],[7,0,8,7],[1,0,8,3],[5,0,0,1]]",
"[[3,3,8,3],[7,0,8,7],[1,0,0,3],[5,0,0,1]]",
"[[3,3,0,3],[7,0,8,7],[1,0,8,3],[5,0,8,1]]",
"[[3,3,0,3],[7,0,8,7],[1,0,0,3],[5,0,8,1]]",
"[[3,3,0,3],[7,0,8,7],[1,0,8,3],[5,0,0,1]]",
"[[3,3,0,3],[7,0,8,7],[1,0,0,3],[5,0,0,1]]",
"[[5,5,0,5],[7,7,0,3],[3,3,0,1]]", "[[5,5,0,5],[7,7,0,3],[1,0,0,3]]",
"[[5,5,0,5],[5,0,0,1],[3,3,0,1]]", "[[5,5,0,5],[1,0,0,3],[5,0,0,1]]",
"[[3,3,0,3],[3,0,0,7],[5,0,0,3]]", "[[3,3,0,3],[1,0,0,3],[3,0,0,7]]",
"[[3,3,0,3],[1,0,0,3],[5,0,0,1]]", "[[3,3,0,3],[5,0,0,1],[7,0,0,5]]",
"[[3,3,8,3],[1,0,8,5],[7,0,8,7],[1,0,8,7]]",
"[[3,3,8,3],[1,0,8,5],[7,0,8,7],[1,0,0,7]]",
"[[3,3,8,3],[1,0,0,5],[7,0,8,7],[1,0,8,7]]",
"[[3,3,8,3],[1,0,0,5],[7,0,8,7],[1,0,0,7]]",
"[[3,3,0,3],[1,0,8,5],[7,0,8,7],[1,0,8,7]]",
"[[3,3,0,3],[1,0,8,5],[7,0,8,7],[1,0,0,7]]",
"[[3,3,0,3],[1,0,0,5],[7,0,8,7],[1,0,8,7]]",
"[[3,3,0,3],[1,0,0,5],[7,0,8,7],[1,0,0,7]]",
"[[5,5,0,5],[3,3,0,5],[3,3,0,7]]", "[[5,5,0,5],[1,0,0,7],[3,3,0,7]]",
"[[5,5,0,5],[1,0,0,5],[3,3,0,5]]", "[[5,5,0,5],[1,0,0,5],[1,0,0,7]]",
"[[3,3,0,3],[1,0,0,5],[1,0,0,7]]", "[[3,3,0,3],[7,0,0,1],[7,0,0,3]]",
"[[3,3,0,3],[7,0,0,3],[1,0,0,7]]", "[[3,3,0,3],[1,0,0,5],[7,0,0,1]]",
"[[3,3,8,3],[1,0,8,5],[3,0,8,1],[7,0,8,7]]",
"[[3,3,8,3],[1,0,0,5],[3,0,8,1],[7,0,8,7]]",
"[[3,3,8,3],[1,0,8,5],[7,0,8,7],[3,0,0,1]]",
"[[3,3,8,3],[1,0,0,5],[7,0,8,7],[3,0,0,1]]",
"[[3,3,0,3],[1,0,8,5],[3,0,8,1],[7,0,8,7]]",
"[[3,3,0,3],[1,0,0,5],[3,0,8,1],[7,0,8,7]]",
"[[3,3,0,3],[1,0,8,5],[7,0,8,7],[3,0,0,1]]",
"[[3,3,0,3],[1,0,0,5],[7,0,8,7],[3,0,0,1]]",
"[[5,5,0,5],[1,1,0,3],[3,3,0,7]]", "[[5,5,0,5],[1,0,0,5],[1,1,0,3]]",
"[[5,5,0,5],[3,0,0,1],[3,3,0,7]]", "[[5,5,0,5],[1,0,0,5],[3,0,0,1]]",
"[[3,3,0,3],[3,0,0,5],[5,0,0,7]]", "[[3,3,0,3],[1,0,0,5],[5,0,0,7]]",
"[[3,3,0,3],[1,0,0,5],[3,0,0,1]]", "[[3,3,0,3],[7,0,0,3],[3,0,0,1]]",
"[[1,2,4,1],[7,6,4,1],[7,6,4,3],[5,2,0,1]]",
"[[1,2,4,1],[7,6,4,3],[1,0,0,7],[5,2,0,1]]",
"[[1,2,4,1],[7,6,4,1],[1,0,0,5],[5,2,0,1]]",
"[[1,2,4,1],[1,0,0,5],[1,0,0,7],[5,2,0,1]]",
"[[1,2,4,1],[7,6,4,1],[7,6,4,3],[3,0,4,7]]",
"[[1,2,4,1],[7,6,4,3],[1,0,0,7],[3,0,4,7]]",
"[[1,2,4,1],[7,6,4,1],[1,0,0,5],[3,0,4,7]]",
"[[1,2,4,1],[1,0,0,5],[1,0,0,7],[3,0,4,7]]",
"[[7,6,4,7],[5,0,4,1],[3,0,4,1],[3,0,4,3]]",
"[[7,6,4,7],[5,0,4,1],[1,0,0,7],[3,0,4,3]]",
"[[1,0,0,5],[7,6,4,7],[5,0,4,1],[3,0,4,1]]",
"[[1,0,0,5],[7,6,4,7],[5,0,4,1],[1,0,0,7]]",
"[[1,0,0,5],[7,6,4,7],[1,0,0,7],[3,0,4,7]]",
"[[7,6,4,7],[7,0,0,1],[7,0,0,3],[3,0,4,7]]",
"[[7,6,4,7],[7,0,0,3],[1,0,0,7],[3,0,4,7]]",
"[[1,0,0,5],[7,6,4,7],[7,0,0,1],[3,0,4,7]]",
"[[1,2,0,1],[5,0,0,1],[3,0,0,3],[3,3,0,1]]",
"[[1,2,0,1],[7,0,8,7],[5,0,0,1],[3,0,0,3],[3,3,0,1]]",
"[[5,5,0,7],[5,0,0,5],[7,6,0,7],[3,0,0,7]]",
"[[7,14,0,7],[5,5,0,7],[5,0,0,5],[5,0,8,1],[3,0,0,7]]",
"[[5,5,0,7],[5,0,0,5],[7,6,0,7],[5,0,0,1]]",
"[[7,14,0,7],[7,0,8,7],[5,5,0,7],[5,0,0,5],[5,0,0,1]]",
"[[1,2,0,1],[5,5,0,7],[5,0,0,5],[3,0,0,7]]",
"[[1,2,0,1],[5,5,0,7],[5,0,0,5],[5,0,0,1]]",
"[[5,5,0,7],[7,6,0,7],[3,0,0,7],[3,0,0,3]]",
"[[7,14,0,7],[5,5,0,7],[5,0,8,1],[3,0,0,7],[3,0,0,3]]",
"[[1,2,0,1],[5,5,0,7],[5,0,8,1],[3,0,0,7],[3,0,0,3]]",
"[[3,0,12,7],[1,1,12,7],[3,6,12,7],[1,0,4,5]]",
"[[1,1,4,7],[1,0,4,5],[3,6,4,7]]",
"[[3,0,12,7],[1,2,0,1],[1,1,12,7],[1,0,4,5]]",
"[[5,0,4,1],[1,2,0,1],[5,5,4,3]]", "[[5,5,0,3],[7,0,4,3],[7,5,0,1]]",
"[[1,2,0,1],[5,5,0,3],[7,0,4,3]]", "[[5,7,4,3],[1,1,4,3],[1,0,4,1]]",
"[[7,7,0,1],[5,7,4,3],[1,0,4,1]]",
"[[11,0,0,3],[1,1,12,3],[5,15,12,3],[11,0,12,3]]",
"[[7,7,0,1],[11,0,0,3],[5,15,12,3],[11,0,12,3]]",
"[[7,7,0,1],[3,1,0,1],[3,0,4,3]]", "[[7,7,0,1],[1,3,0,3],[3,0,4,3]]",
"[[3,0,12,3],[11,0,0,3],[3,9,0,1],[5,10,0,1]]",
"[[7,7,0,1],[3,0,12,3],[11,0,0,3],[3,9,0,1]]",
"[[1,2,0,7],[7,6,0,7],[1,2,0,5],[5,2,0,1]]",
"[[1,2,0,7],[7,14,0,7],[1,0,8,7],[5,10,0,1],[1,2,0,5]]",
"[[1,0,0,5],[1,2,0,7],[7,6,0,7],[5,2,0,1]]",
"[[1,0,0,5],[1,2,0,7],[7,14,0,7],[1,0,8,7],[5,10,0,1]]",
"[[1,2,0,7],[7,6,0,7],[5,0,0,1],[1,2,0,5]]",
"[[1,2,0,7],[7,14,0,7],[1,0,8,7],[5,0,0,1],[1,2,0,5]]",
"[[1,0,0,5],[1,2,0,7],[7,6,0,7],[5,0,0,1]]",
"[[1,0,0,5],[1,2,0,7],[7,14,0,7],[1,0,8,7],[5,0,0,1]]",
"[[1,2,0,1],[5,0,0,5],[5,0,0,7],[3,0,0,7]]",
"[[1,2,0,1],[5,0,0,5],[1,0,8,7],[5,0,0,7],[3,0,0,7]]",
"[[1,2,0,1],[5,0,0,5],[1,0,0,7],[3,0,0,7]]",
"[[1,0,0,5],[1,2,0,1],[5,0,0,7],[3,0,0,7]]",
"[[1,0,0,5],[1,2,0,1],[1,0,8,7],[5,0,0,7],[3,0,0,7]]",
"[[1,0,0,5],[1,2,0,1],[1,0,0,7],[3,0,0,7]]",
"[[1,0,0,5],[1,2,0,1],[1,0,0,7],[5,0,0,1]]",
"[[1,0,0,5],[1,2,0,1],[7,0,8,7],[1,0,0,7],[5,0,0,1]]",
"[[1,2,0,1],[7,0,0,1],[7,0,0,3],[5,0,0,1]]",
"[[1,2,0,1],[7,0,0,1],[7,0,0,3],[1,0,8,7],[5,0,0,1]]",
"[[1,2,0,1],[7,0,0,3],[1,0,0,7],[5,0,0,1]]",
"[[1,0,0,5],[1,2,0,1],[7,0,0,1],[5,0,0,1]]",
"[[1,2,4,1],[7,6,0,1],[7,6,4,5],[3,6,4,7]]",
"[[1,2,4,1],[1,0,0,3],[7,6,0,1],[3,6,4,7]]",
"[[1,2,4,1],[7,6,4,5],[1,0,4,7],[3,6,4,7]]",
"[[1,2,4,1],[1,0,0,3],[1,0,4,7],[3,6,4,7]]",
"[[1,2,4,1],[7,6,0,1],[7,6,4,5],[5,0,0,1]]",
"[[1,2,4,1],[1,0,0,3],[7,6,0,1],[5,0,0,1]]",
"[[1,2,4,1],[7,6,4,5],[5,0,0,1],[1,0,4,7]]",
"[[1,2,4,1],[1,0,0,3],[5,0,0,1],[1,0,4,7]]",
"[[7,6,4,7],[5,0,4,7],[3,0,0,7],[5,0,0,3]]",
"[[7,6,4,7],[5,0,4,7],[1,0,0,3],[3,0,0,7]]",
"[[7,6,4,7],[3,0,0,7],[5,0,0,3],[1,0,4,7]]",
"[[7,6,4,7],[1,0,0,3],[3,0,0,7],[1,0,4,7]]",
"[[7,6,4,7],[1,0,0,3],[5,0,0,1],[1,0,4,7]]",
"[[7,6,4,7],[7,0,4,1],[5,0,0,1],[7,0,0,5]]",
"[[7,6,4,7],[1,0,0,3],[7,0,4,1],[5,0,0,1]]",
"[[7,6,4,7],[5,0,0,1],[7,0,0,5],[1,0,4,7]]",
"[[1,2,4,1],[7,6,4,1],[7,6,0,3],[3,6,4,7]]",
"[[1,2,4,1],[1,0,0,7],[7,6,0,3],[3,6,4,7]]",
"[[1,2,4,1],[7,6,4,1],[1,0,4,5],[3,6,4,7]]",
"[[1,2,4,1],[1,0,0,7],[1,0,4,5],[3,6,4,7]]",
"[[1,2,4,1],[7,6,4,1],[7,6,0,3],[5,0,0,1]]",
"[[1,2,4,1],[1,0,0,7],[7,6,0,3],[5,0,0,1]]",
"[[1,2,4,1],[7,6,4,1],[1,0,4,5],[5,0,0,1]]",
"[[1,2,4,1],[1,0,0,7],[1,0,4,5],[5,0,0,1]]",
"[[7,6,4,7],[5,0,0,7],[3,0,0,7],[5,0,4,5]]",
"[[7,6,4,7],[1,0,0,7],[3,0,0,7],[5,0,4,5]]",
"[[7,6,4,7],[1,0,4,5],[5,0,0,7],[3,0,0,7]]",
"[[7,6,4,7],[1,0,0,7],[1,0,4,5],[3,0,0,7]]",
"[[7,6,4,7],[1,0,0,7],[1,0,4,5],[5,0,0,1]]",
"[[7,6,4,7],[7,0,0,1],[5,0,0,1],[7,0,4,3]]",
"[[7,6,4,7],[1,0,0,7],[5,0,0,1],[7,0,4,3]]",
"[[7,6,4,7],[7,0,0,1],[1,0,4,5],[5,0,0,1]]",
"[[1,2,4,7],[1,2,0,3],[7,6,0,7],[3,6,4,7]]",
"[[1,2,4,7],[1,0,0,3],[7,6,0,7],[3,6,4,7]]",
"[[1,2,0,3],[7,6,0,7],[1,0,4,7],[3,6,4,7]]",
"[[1,2,4,7],[1,0,0,3],[3,0,4,7],[7,6,0,7]]",
"[[3,0,4,5],[5,0,4,1],[1,2,0,1],[3,0,0,1]]",
"[[5,0,4,1],[1,2,0,1],[1,0,0,3],[3,0,0,1]]",
"[[3,0,4,5],[5,0,4,1],[1,2,0,1],[1,0,4,7]]",
"[[5,0,4,1],[1,2,0,1],[1,0,0,3],[1,0,4,7]]",
"[[1,2,0,1],[1,0,0,3],[3,0,4,7],[7,0,4,1]]",
"[[1,2,0,1],[3,0,4,7],[7,0,0,5],[1,0,4,7]]",
"[[1,2,4,1],[7,6,4,3],[7,6,0,1],[5,2,0,1]]",
"[[1,2,4,1],[1,0,0,5],[7,6,0,1],[5,2,0,1]]",
"[[1,2,4,1],[7,6,4,3],[5,2,0,1],[1,0,4,7]]",
"[[1,2,4,1],[1,0,0,5],[5,2,0,1],[1,0,4,7]]",
"[[1,2,4,1],[7,6,4,3],[3,0,4,7],[7,6,0,1]]",
"[[1,2,4,1],[1,0,0,5],[3,0,4,7],[7,6,0,1]]",
"[[1,2,4,1],[7,6,4,3],[3,0,4,7],[1,0,4,7]]",
"[[1,2,4,1],[1,0,0,5],[3,0,4,7],[1,0,4,7]]",
"[[7,6,4,7],[5,0,4,1],[3,0,4,3],[3,0,0,1]]",
"[[1,0,0,5],[7,6,4,7],[5,0,4,1],[3,0,0,1]]",
"[[7,6,4,7],[5,0,4,1],[3,0,4,3],[1,0,4,7]]",
"[[1,0,0,5],[7,6,4,7],[5,0,4,1],[1,0,4,7]]",
"[[1,0,0,5],[7,6,4,7],[3,0,4,7],[1,0,4,7]]",
"[[7,6,4,7],[7,0,0,3],[3,0,4,7],[7,0,4,1]]",
"[[1,0,0,5],[7,6,4,7],[3,0,4,7],[7,0,4,1]]",
"[[7,6,4,7],[7,0,0,3],[3,0,4,7],[1,0,4,7]]",
"[[1,1,8,1],[7,0,8,7],[1,0,8,7],[5,0,8,1]]",
"[[1,1,8,1],[7,0,8,7],[1,0,0,7],[5,0,8,1]]",
"[[1,1,8,1],[7,0,8,7],[1,0,8,7],[5,0,0,1]]",
"[[1,1,8,1],[7,0,8,7],[1,0,0,7],[5,0,0,1]]",
"[[1,1,0,1],[7,0,8,7],[1,0,8,7],[5,0,8,1]]",
"[[1,1,0,1],[7,0,8,7],[1,0,0,7],[5,0,8,1]]",
"[[1,1,0,1],[7,0,8,7],[1,0,8,7],[5,0,0,1]]",
"[[1,1,0,1],[7,0,8,7],[1,0,0,7],[5,0,0,1]]",
"[[1,1,0,7],[7,7,0,7],[5,5,0,1]]", "[[7,7,0,7],[5,5,0,1],[1,0,0,7]]",
"[[1,1,0,7],[7,7,0,7],[5,0,0,1]]", "[[7,7,0,7],[1,0,0,7],[5,0,0,1]]",
"[[1,1,0,1],[5,0,0,7],[3,0,0,7]]", "[[1,1,0,1],[1,0,0,7],[3,0,0,7]]",
"[[1,1,0,1],[1,0,0,7],[5,0,0,1]]", "[[1,1,0,1],[7,0,0,1],[5,0,0,1]]",
"[[1,1,10,5],[3,0,14,7],[1,0,10,21]]", "[[1,1,10,5],[1,0,10,5]]",
"[[3,3,0,21],[3,0,14,7],[1,0,10,21]]", "[[15,15,0,1],[5,0,2,1]]",
"[[3,0,14,7],[1,1,0,15]]", "[[9,2,2,1],[15,15,0,1]]",
"[[3,3,4,1],[3,9,4,5],[1,3,4,5]]", "[[3,3,4,1],[1,0,0,3],[1,3,4,5]]",
"[[1,2,0,7],[3,9,4,5],[1,3,4,5]]", "[[1,2,0,7],[1,0,0,3],[1,3,4,5]]",
"[[3,6,0,5],[1,2,0,5],[3,9,12,7]]",
"[[1,0,0,3],[3,6,0,5],[3,9,12,7]]",
"[[1,2,0,7],[1,0,0,3],[3,9,12,7]]",
"[[1,2,0,7],[3,0,0,1],[3,9,12,7]]",
"[[3,9,4,1],[1,3,20,5],[3,21,28,7],[3,9,12,3]]",
"[[3,9,4,1],[1,0,0,5],[1,3,20,5],[3,21,28,7]]",
"[[3,9,4,1],[1,3,4,5],[3,9,12,3]]",
"[[1,0,0,7],[1,3,4,5],[3,9,12,3]]", "[[3,9,4,1],[1,0,0,5],[1,3,4,5]]",
"[[1,0,0,5],[1,0,0,7],[1,3,4,5]]",
"[[3,9,28,7],[1,4,0,5],[3,12,0,3],[1,4,0,7]]",
"[[3,9,28,7],[1,0,0,5],[3,12,0,3],[1,4,0,7]]",
"[[1,0,0,5],[1,0,0,7],[3,9,12,7]]",
"[[3,0,0,5],[3,0,0,7],[3,9,12,7]]",
"[[1,0,0,7],[3,0,0,7],[3,9,12,7]]",
"[[1,0,0,5],[3,0,0,5],[3,9,12,7]]",
"[[1,1,8,7],[3,0,8,7],[7,0,8,1],[3,0,8,5]]",
"[[1,1,8,7],[7,0,8,1],[3,0,0,1],[3,0,8,5]]",
"[[1,1,0,1],[3,0,8,7],[7,0,8,1],[3,0,8,5]]",
"[[1,1,0,1],[7,0,8,1],[3,0,0,1],[3,0,8,5]]",
"[[1,1,8,7],[3,0,8,7],[7,0,8,1],[3,0,0,3]]",
"[[1,1,8,7],[7,0,8,1],[3,0,0,1],[3,0,0,3]]",
"[[1,1,0,1],[3,0,8,7],[7,0,8,1],[3,0,0,3]]",
"[[1,1,0,1],[7,0,8,1],[3,0,0,1],[3,0,0,3]]",
"[[3,3,0,3],[5,0,0,5],[1,0,8,7],[1,0,0,3]]",
"[[3,3,0,3],[5,0,0,5],[1,0,8,7],[3,0,0,1]]",
"[[1,1,0,1],[5,0,0,5],[1,0,8,7],[1,0,0,3]]",
"[[1,1,0,1],[5,0,0,5],[1,0,8,7],[3,0,0,1]]",
"[[1,1,0,1],[1,0,8,7],[3,0,0,1],[3,0,0,3]]",
"[[7,7,0,7],[1,0,8,7],[5,0,0,7],[3,0,0,3]]",
"[[7,7,0,7],[1,0,8,7],[3,0,0,1],[3,0,0,3]]",
"[[1,1,0,1],[1,0,8,7],[5,0,0,7],[3,0,0,3]]",
"[[3,3,0,3],[7,0,0,3],[5,0,0,5],[5,0,0,3]]",
"[[3,3,0,3],[5,0,16,1],[7,0,0,3],[5,0,0,5],[5,0,0,3]]",
"[[3,3,0,3],[5,0,0,5],[5,0,0,1],[5,0,0,3]]",
"[[3,3,0,3],[3,0,16,3],[5,0,0,5],[5,0,0,1],[5,0,0,3]]",
"[[3,3,0,3],[7,0,0,1],[7,0,0,3],[5,0,0,5]]",
"[[3,3,0,3],[7,0,0,1],[5,0,0,5],[5,0,0,1]]",
"[[1,1,0,1],[7,0,0,3],[5,0,0,5],[5,0,0,3]]",
"[[1,1,0,1],[5,0,16,1],[7,0,0,3],[5,0,0,5],[5,0,0,3]]",
"[[1,1,0,1],[5,0,0,5],[5,0,0,1],[5,0,0,3]]",
"[[1,1,0,1],[3,0,16,3],[5,0,0,5],[5,0,0,1],[5,0,0,3]]",
"[[1,1,0,1],[7,0,0,1],[7,0,0,3],[5,0,0,5]]",
"[[1,1,0,1],[7,0,0,1],[5,0,0,5],[5,0,0,1]]",
"[[1,1,0,1],[7,0,0,1],[5,0,0,1],[3,0,0,3]]",
"[[1,1,0,1],[7,0,0,1],[5,0,0,1],[7,0,16,7],[3,0,0,3]]",
"[[7,7,0,7],[1,0,0,7],[3,0,0,7],[3,0,0,3]]",
"[[7,7,0,7],[5,0,16,1],[1,0,0,7],[3,0,0,7],[3,0,0,3]]",
"[[7,7,0,7],[1,0,0,7],[5,0,0,1],[3,0,0,3]]",
"[[7,7,0,7],[1,0,0,7],[5,0,0,1],[7,0,16,7],[3,0,0,3]]",
"[[7,7,0,7],[7,0,0,1],[3,0,0,7],[3,0,0,3]]",
"[[7,7,0,7],[7,0,0,1],[5,0,0,1],[3,0,0,3]]",
"[[1,1,0,1],[1,0,0,7],[3,0,0,7],[3,0,0,3]]",
"[[1,1,0,1],[5,0,16,1],[1,0,0,7],[3,0,0,7],[3,0,0,3]]",
"[[1,1,0,1],[1,0,0,7],[5,0,0,1],[3,0,0,3]]",
"[[1,1,0,1],[7,0,0,1],[3,0,0,7],[3,0,0,3]]",
"[[1,1,8,1],[7,0,8,3],[5,0,8,5],[5,0,8,3]]",
"[[1,1,8,1],[5,0,8,5],[5,0,0,1],[5,0,8,3]]",
"[[1,1,8,1],[7,0,0,1],[7,0,8,3],[5,0,8,5]]",
"[[1,1,8,1],[7,0,0,1],[5,0,8,5],[5,0,0,1]]",
"[[3,3,0,3],[7,0,8,3],[5,0,8,5],[5,0,8,3]]",
"[[3,3,0,3],[5,0,8,5],[5,0,0,1],[5,0,8,3]]",
"[[3,3,0,3],[7,0,0,1],[7,0,8,3],[5,0,8,5]]",
"[[3,3,0,3],[7,0,0,1],[5,0,8,5],[5,0,0,1]]",
"[[5,5,0,5],[7,7,0,3],[3,0,8,3],[5,5,0,3]]",
"[[5,5,0,5],[3,0,8,3],[5,5,0,3],[5,0,0,1]]",
"[[5,5,0,5],[7,7,0,3],[3,0,8,3],[7,0,0,1]]",
"[[5,5,0,5],[3,0,8,3],[7,0,0,1],[5,0,0,1]]",
"[[3,3,0,3],[3,0,8,3],[7,0,0,1],[5,0,0,1]]",
"[[3,3,0,3],[3,0,8,3],[1,0,0,7],[3,0,0,7]]",
"[[3,3,0,3],[3,0,8,3],[1,0,0,7],[5,0,0,1]]",
"[[3,3,0,3],[3,0,8,3],[7,0,0,1],[3,0,0,7]]",
"[[1,1,8,1],[7,0,8,3],[5,0,8,5],[5,5,8,3]]",
"[[1,1,8,1],[5,0,8,5],[5,5,8,3],[5,0,0,1]]",
"[[7,7,0,1],[1,1,8,1],[7,0,8,3],[5,0,8,5]]",
"[[7,7,0,1],[1,1,8,1],[5,0,8,5],[5,0,0,1]]",
"[[3,3,0,3],[7,0,8,3],[5,0,8,5],[5,5,8,3]]",
"[[3,3,0,3],[5,0,8,5],[5,5,8,3],[5,0,0,1]]",
"[[7,7,0,1],[3,3,0,3],[7,0,8,3],[5,0,8,5]]",
"[[7,7,0,1],[3,3,0,3],[5,0,8,5],[5,0,0,1]]",
"[[5,5,0,5],[7,7,0,3],[3,0,8,3],[5,0,0,3]]",
"[[5,5,0,5],[3,0,8,3],[5,0,0,1],[5,0,0,3]]",
"[[7,7,0,1],[5,5,0,5],[7,7,0,3],[3,0,8,3]]",
"[[7,7,0,1],[5,5,0,5],[3,0,8,3],[5,0,0,1]]",
"[[7,7,0,1],[3,3,0,3],[3,0,8,3],[5,0,0,1]]",
"[[1,1,0,7],[3,3,0,3],[3,0,8,3],[3,0,0,7]]",
"[[1,1,0,7],[3,3,0,3],[3,0,8,3],[5,0,0,1]]",
"[[7,7,0,1],[3,3,0,3],[3,0,8,3],[3,0,0,7]]",
"[[1,1,8,1],[7,7,8,3],[5,0,8,5],[5,0,8,3]]",
"[[1,1,8,1],[7,7,8,3],[7,0,0,1],[5,0,8,5]]",
"[[1,1,8,1],[5,5,0,1],[5,0,8,5],[5,0,8,3]]",
"[[1,1,8,1],[7,0,0,1],[5,5,0,1],[5,0,8,5]]",
"[[3,3,0,3],[7,7,8,3],[5,0,8,5],[5,0,8,3]]",
"[[3,3,0,3],[7,7,8,3],[7,0,0,1],[5,0,8,5]]",
"[[3,3,0,3],[5,5,0,1],[5,0,8,5],[5,0,8,3]]",
"[[3,3,0,3],[7,0,0,1],[5,5,0,1],[5,0,8,5]]",
"[[5,5,0,5],[3,0,8,3],[7,0,0,3],[5,5,0,3]]",
"[[5,5,0,5],[3,0,8,3],[7,0,0,1],[7,0,0,3]]",
"[[5,5,0,5],[3,0,8,3],[5,5,0,1],[5,5,0,3]]",
"[[5,5,0,5],[3,0,8,3],[7,0,0,1],[5,5,0,1]]",
"[[3,3,0,3],[3,0,8,3],[7,0,0,1],[5,5,0,1]]",
"[[3,3,0,3],[3,0,8,3],[1,0,0,7],[3,3,0,7]]",
"[[3,3,0,3],[3,0,8,3],[7,0,0,1],[3,3,0,7]]",
"[[3,3,0,3],[3,0,8,3],[5,5,0,1],[1,0,0,7]]",
"[[1,1,8,1],[7,7,8,3],[5,0,8,5],[1,0,8,3]]",
"[[1,1,8,1],[7,7,8,3],[5,0,8,5],[3,0,0,1]]",
"[[1,1,8,1],[5,5,0,1],[5,0,8,5],[1,0,8,3]]",
"[[1,1,8,1],[5,5,0,1],[5,0,8,5],[3,0,0,1]]",
"[[3,3,0,3],[7,7,8,3],[5,0,8,5],[1,0,8,3]]",
"[[3,3,0,3],[7,7,8,3],[5,0,8,5],[3,0,0,1]]",
"[[3,3,0,3],[5,5,0,1],[5,0,8,5],[1,0,8,3]]",
"[[3,3,0,3],[5,5,0,1],[5,0,8,5],[3,0,0,1]]",
"[[5,5,0,5],[1,1,0,3],[3,0,8,3],[7,0,0,3]]",
"[[5,5,0,5],[3,0,8,3],[7,0,0,3],[3,0,0,1]]",
"[[5,5,0,5],[1,1,0,3],[3,0,8,3],[5,5,0,1]]",
"[[5,5,0,5],[3,0,8,3],[5,5,0,1],[3,0,0,1]]",
"[[3,3,0,3],[3,0,8,3],[5,5,0,1],[3,0,0,1]]",
"[[3,3,0,3],[3,0,8,3],[5,0,0,7],[3,3,0,7]]",
"[[3,3,0,3],[3,0,8,3],[3,0,0,1],[3,3,0,7]]",
"[[3,3,0,3],[3,0,8,3],[5,5,0,1],[5,0,0,7]]",
"[[15,15,6,3],[5,0,6,3],[15,0,2,1]]", "[[15,15,6,3],[15,0,2,1]]",
"[[13,13,0,1],[15,0,2,1]]", "[[13,13,0,1],[5,0,6,3]]",
"[[31,31,0,3],[11,0,0,3],[21,0,6,3]]", "[[15,15,0,3],[5,0,6,3]]",
"[[3,6,0,3],[7,6,4,3],[5,2,0,3]]", "[[3,6,0,3],[7,6,4,3],[1,0,0,7]]",
"[[3,6,0,3],[3,0,4,7],[5,2,0,3]]", "[[3,6,0,3],[1,0,0,7],[3,0,4,7]]",
"[[5,0,4,1],[5,2,0,5],[3,0,4,1]]", "[[5,0,4,1],[1,0,0,7],[5,2,0,5]]",
"[[1,0,0,7],[5,2,0,5],[3,0,4,7]]", "[[7,0,0,1],[5,2,0,5],[3,0,4,7]]",
"[[3,4,0,7],[5,4,2,3],[3,4,4,7],[1,4,6,3]]",
"[[3,4,0,7],[5,4,2,3],[1,0,6,3],[1,4,6,3]]",
"[[5,4,2,3],[3,4,4,7],[1,0,2,3],[1,4,6,3]]",
"[[5,4,2,3],[1,0,6,3],[1,0,2,3],[1,4,6,3]]",
"[[3,4,0,7],[5,4,2,3],[3,4,4,7],[3,0,4,7]]",
"[[3,4,0,7],[5,4,2,3],[1,0,6,3],[3,0,4,7]]",
"[[5,4,2,3],[3,4,4,7],[1,0,2,3],[3,0,4,7]]",
"[[5,4,2,3],[1,0,6,3],[1,0,2,3],[3,0,4,7]]",
"[[5,0,4,1],[3,4,6,5],[3,0,6,5],[3,0,2,5]]",
"[[5,0,4,1],[1,0,6,3],[3,4,6,5],[1,0,2,3]]",
"[[1,0,6,3],[3,4,6,5],[1,0,2,3],[3,0,4,7]]",
"[[3,4,6,5],[3,0,4,7],[7,0,6,5],[7,0,2,5]]",
"[[1,0,8,5],[3,6,8,7],[7,0,8,7],[1,0,8,7]]",
"[[1,0,0,5],[3,6,8,7],[7,0,8,7],[1,0,8,7]]",
"[[1,0,8,5],[3,6,0,7],[7,0,8,7],[1,0,8,7]]",
"[[1,0,0,5],[3,6,0,7],[7,0,8,7],[1,0,8,7]]",
"[[3,6,0,1],[3,6,0,3],[5,2,0,1]]", "[[3,6,0,3],[1,0,0,7],[5,2,0,1]]",
"[[3,6,0,1],[1,0,0,5],[5,2,0,1]]", "[[1,0,0,5],[1,0,0,7],[5,2,0,1]]",
"[[1,0,0,5],[3,6,0,7],[1,0,0,7]]", "[[3,6,0,7],[7,0,0,1],[7,0,0,3]]",
"[[3,6,0,7],[7,0,0,3],[1,0,0,7]]", "[[1,0,0,5],[3,6,0,7],[7,0,0,1]]",
"[[5,4,2,3],[3,4,2,1],[1,4,0,5],[3,4,6,1]]",
"[[5,4,2,3],[1,0,0,5],[3,4,2,1],[1,4,0,5]]",
"[[5,4,2,3],[1,4,0,5],[1,0,4,5],[3,4,6,1]]",
"[[5,4,2,3],[1,0,0,5],[1,4,0,5],[1,0,4,5]]",
"[[5,4,2,3],[3,4,2,1],[3,4,6,1],[3,0,6,1]]",
"[[5,4,2,3],[1,0,0,5],[3,4,2,1],[3,0,6,1]]",
"[[5,4,2,3],[1,0,4,5],[3,4,6,1],[3,0,6,1]]",
"[[5,4,2,3],[1,0,0,5],[1,0,4,5],[3,0,6,1]]",
"[[3,4,6,5],[3,0,6,5],[3,0,2,5],[5,0,2,7]]",
"[[1,0,0,5],[3,4,6,5],[1,0,4,5],[5,0,2,7]]",
"[[1,0,0,5],[3,4,6,5],[1,0,4,5],[3,0,6,1]]",
"[[7,0,0,3],[3,4,6,5],[7,0,4,3],[3,0,6,1]]",
"[[1,0,4,5],[7,0,4,1],[3,6,4,7]]", "[[1,0,0,3],[1,0,4,5],[3,6,4,7]]",
"[[5,2,0,5],[1,0,4,5],[7,0,4,1]]", "[[5,2,0,5],[1,0,0,3],[1,0,4,5]]",
"[[3,6,0,3],[5,2,0,7],[7,0,4,3]]", "[[3,6,0,3],[1,0,0,3],[7,0,4,3]]",
"[[5,2,0,5],[1,0,0,3],[7,0,4,3]]", "[[5,2,0,5],[7,0,4,3],[7,0,0,5]]",
"[[5,0,4,1],[3,4,4,3],[3,0,0,7],[3,0,6,1]]",
"[[5,0,4,1],[1,4,0,5],[3,0,0,7],[3,0,6,1]]",
"[[5,0,4,1],[3,4,4,3],[3,0,6,1],[1,0,4,1]]",
"[[5,0,4,1],[1,4,0,5],[3,0,6,1],[1,0,4,1]]",
"[[1,0,6,7],[5,0,4,1],[1,4,0,5],[1,0,4,1]]",
"[[7,0,0,7],[3,12,12,11],[3,0,0,7],[3,0,12,15],[3,0,6,1]]",
"[[3,12,12,3],[7,0,0,7],[3,0,0,7],[3,0,12,15],[3,0,6,1]]",
"[[1,0,6,7],[3,4,4,3],[3,0,4,7],[1,0,4,1]]",
"[[5,4,4,5],[1,0,6,7],[3,0,4,7],[1,0,4,1]]",
"[[7,0,2,1],[3,4,4,3],[3,0,4,7],[7,0,4,7]]",
"[[5,4,4,5],[7,0,2,1],[3,0,4,7],[7,0,4,7]]",
"[[5,4,4,5],[7,0,2,1],[3,0,4,7],[1,0,4,1]]",
"[[3,0,12,7],[1,0,10,7],[1,0,4,9],[7,0,0,7],[3,12,12,11]]",
"[[3,12,12,3],[3,0,12,7],[1,0,10,7],[1,0,4,9],[7,0,0,7]]",
"[[7,0,4,5],[1,0,4,5],[3,6,4,7]]", "[[1,0,0,7],[1,0,4,5],[3,6,4,7]]",
"[[7,0,4,5],[5,2,0,5],[1,0,4,5]]", "[[1,0,0,7],[5,2,0,5],[1,0,4,5]]",
"[[3,6,0,3],[7,0,4,3],[5,2,0,3]]", "[[3,6,0,3],[1,0,0,7],[7,0,4,3]]",
"[[1,0,0,7],[5,2,0,5],[7,0,4,3]]", "[[7,0,0,1],[5,2,0,5],[7,0,4,3]]",
"[[9,0,8,1],[15,14,0,3],[7,14,0,7],[7,0,0,7],[7,14,0,5]]",
"[[9,0,8,1],[15,14,0,3],[7,14,0,7],[7,0,0,7],[1,0,0,7]]",
"[[9,0,8,1],[1,0,0,5],[15,14,0,3],[7,0,0,7],[7,14,0,5]]",
"[[9,0,8,7],[1,0,0,5],[15,14,0,3],[7,0,0,7],[7,14,0,5]]",
"[[7,6,0,5],[7,6,0,7],[1,2,0,5]]",
"[[9,0,8,7],[9,0,0,1],[7,14,0,7],[7,14,0,5],[1,2,0,5]]",
"[[1,0,0,5],[7,6,0,5],[1,2,0,5]]",
"[[9,0,0,1],[7,14,0,5],[1,2,0,5],[9,0,8,7],[1,0,0,5]]",
"[[1,0,0,5],[1,0,0,7],[1,2,0,5]]", "[[1,0,0,5],[1,0,0,7],[7,6,0,3]]",
"[[1,0,0,5],[9,0,0,1],[7,14,0,3],[1,0,0,7],[15,0,8,7]]",
"[[7,0,0,1],[7,0,0,3],[7,6,0,3]]",
"[[9,0,8,7],[9,0,0,1],[7,0,0,1],[7,14,0,3],[7,0,0,3]]",
"[[7,0,0,3],[1,0,0,7],[7,6,0,3]]", "[[3,6,0,7],[7,6,0,1],[5,2,0,7]]",
"[[3,6,0,7],[7,14,0,1],[1,0,8,7],[5,10,0,7],[1,0,0,9]]",
"[[1,0,0,7],[7,6,0,1],[5,2,0,7]]",
"[[3,6,8,1],[7,14,0,1],[1,0,0,7],[5,10,0,7],[1,0,0,9]]",
"[[3,6,0,7],[7,14,0,1],[7,0,0,7],[3,6,0,9],[1,0,8,1]]",
"[[3,6,0,7],[7,14,0,1],[7,0,0,7],[3,6,0,9],[1,0,8,7]]",
"[[3,6,0,7],[5,0,0,1],[5,2,0,7]]",
"[[3,6,0,7],[1,0,8,7],[5,10,0,7],[1,0,0,9],[5,0,0,1]]",
"[[1,0,0,7],[5,0,0,1],[5,2,0,7]]",
"[[3,6,8,1],[1,0,0,7],[5,10,0,7],[1,0,0,9],[5,0,0,1]]",
"[[3,6,0,7],[7,0,0,7],[3,6,0,9],[1,0,8,1],[5,0,0,1]]",
"[[7,0,0,7],[3,6,0,9],[1,0,8,1],[1,0,0,7],[5,0,0,1]]",
"[[3,6,0,1],[1,0,0,7],[3,0,0,7]]", "[[3,6,0,1],[7,0,0,1],[5,0,0,1]]",
"[[1,2,8,5],[7,0,8,7],[1,0,8,7],[5,0,8,1]]",
"[[1,2,8,5],[7,0,8,7],[1,0,8,7],[5,0,0,1]]",
"[[7,0,8,7],[1,0,8,7],[5,0,8,1],[1,2,0,5]]",
"[[7,0,8,7],[1,0,8,7],[5,0,0,1],[1,2,0,5]]",
"[[1,2,0,3],[5,2,0,5],[7,6,0,3]]", "[[1,0,0,7],[5,2,0,5],[7,6,0,3]]",
"[[1,2,0,3],[7,6,0,3],[5,0,0,1]]", "[[1,0,0,7],[7,6,0,3],[5,0,0,1]]",
"[[5,0,0,7],[3,0,0,7],[1,2,0,5]]", "[[1,0,0,7],[3,0,0,7],[1,2,0,5]]",
"[[1,0,0,7],[5,0,0,1],[1,2,0,5]]", "[[7,0,0,1],[5,0,0,1],[1,2,0,5]]",
"[[7,14,8,7],[7,0,8,7],[5,0,8,1],[3,3,8,1]]",
"[[7,14,8,7],[7,0,8,7],[5,0,0,1],[3,3,8,1]]",
"[[7,14,0,7],[7,0,8,7],[5,0,8,1],[3,3,8,1]]",
"[[7,14,0,7],[7,0,8,7],[5,0,0,1],[3,3,8,1]]",
"[[7,14,8,7],[7,0,8,7],[5,0,8,1],[3,3,0,1]]",
"[[7,14,8,7],[7,0,8,7],[5,0,0,1],[3,3,0,1]]",
"[[7,7,0,1],[5,5,0,7],[5,7,0,7]]", "[[5,5,0,7],[5,7,0,7],[5,0,0,1]]",
"[[7,7,0,1],[5,5,0,7],[7,6,0,7]]", "[[5,5,0,7],[7,6,0,7],[5,0,0,1]]",
"[[3,6,0,7],[1,2,0,1],[3,3,0,1]]", "[[1,2,0,1],[5,0,0,1],[3,3,0,1]]",
"[[3,0,8,1],[7,0,8,7],[5,5,8,1]]", "[[7,0,8,7],[5,5,8,1],[3,0,0,1]]",
"[[3,0,8,1],[7,0,8,7],[5,5,0,1]]", "[[7,0,8,7],[5,5,0,1],[3,0,0,1]]",
"[[7,7,0,1],[3,3,0,7]]", "[[3,0,0,1],[3,3,0,7]]",
"[[5,5,0,1],[5,0,0,7]]", "[[5,5,0,1],[3,0,0,1]]",
"[[7,7,8,3],[7,0,8,7],[1,0,8,3]]", "[[7,7,8,3],[7,0,8,7],[1,0,0,3]]",
"[[7,7,0,3],[7,0,8,7],[1,0,8,3]]", "[[7,7,0,3],[7,0,8,7],[1,0,0,3]]",
"[[7,7,0,1],[1,1,0,5]]", "[[1,0,0,3],[1,1,0,5]]",
"[[7,7,0,3],[1,0,0,3]]", "[[7,7,0,3],[7,0,0,5]]",
"[[5,2,4,1],[7,0,4,1],[1,0,4,1]]", "[[5,2,4,1],[1,0,0,7],[1,0,4,1]]",
"[[3,6,0,7],[7,0,4,1],[1,0,4,1]]", "[[3,6,0,7],[1,0,0,7],[1,0,4,1]]",
"[[3,6,0,1],[7,0,4,7],[5,2,0,1]]", "[[1,0,0,7],[7,0,4,7],[5,2,0,1]]",
"[[3,6,0,7],[1,0,0,7],[7,0,4,7]]", "[[3,6,0,7],[7,0,0,1],[7,0,4,7]]",
"[[7,0,8,7],[5,5,8,1],[1,0,8,7]]", "[[7,0,8,7],[5,5,8,1],[1,0,0,7]]",
"[[7,0,8,7],[5,5,0,1],[1,0,8,7]]", "[[7,0,8,7],[5,5,0,1],[1,0,0,7]]",
"[[5,5,0,7],[3,3,0,7]]", "[[1,0,0,7],[3,3,0,7]]",
"[[5,5,0,1],[1,0,0,7]]", "[[7,0,0,1],[5,5,0,1]]",
"[[5,2,4,7],[7,6,0,3],[3,6,4,7]]", "[[1,0,0,7],[7,6,0,3],[3,6,4,7]]",
"[[5,2,4,7],[3,0,4,3],[3,6,4,7]]", "[[1,0,0,7],[3,0,4,3],[3,6,4,7]]",
"[[3,0,4,5],[5,2,4,1],[5,0,4,5]]", "[[5,2,4,1],[1,0,0,7],[5,0,4,5]]",
"[[5,2,4,1],[1,0,0,7],[3,0,4,3]]", "[[7,0,0,1],[5,2,4,1],[3,0,4,3]]",
"[[7,6,4,5],[5,2,0,7],[3,6,4,7]]", "[[7,6,4,5],[1,0,4,7],[3,6,4,7]]",
"[[3,0,0,5],[5,2,0,7],[3,6,4,7]]", "[[3,0,0,5],[1,0,4,7],[3,6,4,7]]",
"[[5,2,4,1],[3,0,4,3],[5,0,0,3]]", "[[5,2,4,1],[5,0,0,3],[1,0,4,7]]",
"[[5,2,4,1],[3,0,0,5],[1,0,4,7]]", "[[5,2,4,1],[3,0,0,5],[7,0,4,1]]",
"[[7,0,8,7],[7,0,8,3],[3,9,8,1],[1,2,8,1]]",
"[[1,2,0,1],[7,0,8,7],[7,0,8,3],[3,9,8,1]]",
"[[1,2,0,1],[7,0,8,7],[7,0,8,3],[3,9,0,1]]",
"[[7,0,8,7],[7,0,0,3],[3,9,8,1],[1,2,8,1]]",
"[[1,2,0,1],[7,0,8,7],[7,0,0,3],[3,9,8,1]]",
"[[1,2,0,1],[7,0,8,7],[7,0,0,3],[3,9,0,1]]",
"[[1,0,0,5],[5,7,0,3],[7,6,0,3]]", "[[1,0,0,5],[1,2,0,1],[5,7,0,3]]",
"[[1,0,0,5],[1,2,0,1],[3,1,0,1]]", "[[1,2,0,1],[7,0,0,3],[3,1,0,1]]",
"[[7,0,0,3],[5,7,0,7],[7,6,0,7]]", "[[1,2,0,1],[7,0,0,3],[5,7,0,7]]",
"[[11,0,24,3],[1,1,0,3],[9,27,0,3],[7,7,0,5],[15,0,0,3]]",
"[[1,1,0,3],[9,27,0,3],[3,0,24,3],[7,7,0,5],[15,0,0,3]]",
"[[3,0,8,3],[7,0,0,3],[1,3,0,3],[7,7,0,5]]",
"[[7,0,0,3],[1,3,0,3],[7,7,0,5],[3,3,8,1]]",
"[[7,7,0,1],[11,0,24,3],[1,1,0,3],[9,27,0,3],[7,7,0,5]]",
"[[7,7,0,1],[1,1,0,3],[9,27,0,3],[3,0,24,3],[7,7,0,5]]",
"[[7,7,0,1],[3,0,8,3],[1,3,0,3],[7,7,0,5]]",
"[[7,7,0,1],[1,3,0,3],[7,7,0,5],[3,3,8,1]]",
"[[3,0,8,3],[1,2,0,1],[7,0,0,3],[7,7,0,5]]",
"[[1,2,0,1],[7,0,0,3],[7,7,0,5],[3,3,8,1]]",
"[[7,7,0,1],[3,0,8,3],[1,2,0,1],[7,7,0,5]]",
"[[7,7,0,1],[1,2,0,1],[7,7,0,5],[3,3,8,1]]",
"[[7,7,0,1],[1,1,0,3],[1,2,0,1],[5,0,8,5]]",
"[[1,1,0,7],[1,1,0,3],[1,2,0,1],[5,0,8,5]]",
"[[1,0,0,5],[5,0,8,5],[1,0,8,7],[5,10,8,1]]",
"[[3,0,8,3],[7,0,0,3],[1,0,8,7],[5,10,8,1]]",
"[[1,0,0,5],[5,0,8,5],[5,10,8,1],[3,0,0,1]]",
"[[3,0,8,3],[7,0,0,3],[5,10,8,1],[3,0,0,1]]",
"[[1,0,0,5],[7,14,0,7],[5,0,8,5],[1,0,8,7]]",
"[[3,0,8,3],[7,14,0,7],[7,0,0,3],[1,0,8,7]]",
"[[1,0,0,5],[3,0,8,7],[5,10,0,7],[1,2,0,5]]",
"[[3,0,8,7],[7,14,0,3],[7,0,0,3],[5,10,0,7]]",
"[[1,0,0,5],[3,0,8,7],[3,0,0,1],[1,2,0,5]]",
"[[3,0,8,7],[7,14,0,3],[7,0,0,3],[3,0,0,1]]",
"[[1,0,0,5],[3,0,8,7],[7,14,0,7],[5,0,0,3]]",
"[[3,0,8,7],[7,14,0,7],[3,0,0,5],[7,0,0,3]]",
"[[1,2,6,1],[1,2,6,3],[3,6,10,7]]",
"[[1,0,0,5],[1,2,6,3],[3,6,10,7]]",
"[[1,0,0,5],[1,2,14,5],[1,0,4,7]]",
"[[1,2,14,5],[3,0,12,5],[3,0,0,7]]",
"[[1,2,14,1],[1,2,14,3],[3,6,10,7]]",
"[[1,0,8,5],[1,2,14,3],[3,6,10,7]]",
"[[1,0,8,5],[1,2,14,5],[1,0,12,7]]",
"[[3,0,8,7],[1,2,14,5],[3,0,4,5]]",
"[[7,0,8,1],[11,11,8,5],[1,0,8,7],[11,11,8,1]]",
"[[7,0,8,1],[1,0,24,7],[11,11,8,5],[9,0,16,1],[11,11,8,1]]",
"[[9,0,0,1],[11,11,8,5],[1,0,8,7],[11,11,8,1]]",
"[[9,0,0,1],[1,0,24,7],[11,11,8,5],[15,0,24,1],[11,11,8,1]]",
"[[7,0,8,1],[5,5,0,1],[11,11,8,5],[1,0,8,7]]",
"[[7,0,8,1],[1,0,24,7],[5,5,0,1],[11,11,8,5],[9,0,16,1]]",
"[[9,0,0,1],[5,5,0,1],[11,11,8,5],[1,0,8,7]]",
"[[9,0,0,1],[1,0,24,7],[5,5,0,1],[11,11,8,5],[15,0,24,1]]",
"[[5,5,0,5],[7,0,8,1],[1,0,24,7],[9,0,16,1],[11,11,8,1]]",
"[[5,5,0,5],[9,0,0,1],[1,0,8,7],[11,11,8,1]]",
"[[5,5,0,5],[9,0,0,1],[1,0,24,7],[15,0,24,1],[11,11,8,1]]",
"[[5,5,0,5],[7,0,8,1],[1,0,24,7],[5,5,0,1],[9,0,16,1]]",
"[[5,5,0,5],[9,0,0,1],[5,5,0,1],[1,0,8,7]]",
"[[5,5,0,5],[9,0,0,1],[1,0,24,7],[5,5,0,1],[15,0,24,1]]",
"[[1,0,0,5],[13,13,0,5],[15,0,8,1],[11,11,0,3]]",
"[[9,0,0,1],[1,0,0,5],[15,0,8,1],[11,11,0,3]]",
"[[13,13,0,5],[5,5,0,1],[15,0,8,1],[11,11,0,3]]",
"[[9,0,0,1],[5,5,0,1],[15,0,8,1],[11,11,0,3]]",
"[[5,5,0,5],[9,0,0,1],[5,5,0,1],[15,0,8,1]]",
"[[5,5,0,5],[9,0,0,1],[11,11,0,7],[15,0,8,1]]",
"[[3,0,16,1],[7,0,16,7],[5,5,16,1],[7,0,16,3]]",
"[[3,0,16,1],[5,5,0,1],[7,0,16,7],[7,0,16,3]]",
"[[3,0,16,1],[7,0,0,3],[7,0,16,7],[5,5,16,1]]",
"[[3,0,16,1],[5,5,0,1],[7,0,0,3],[7,0,16,7]]",
"[[5,5,0,1],[7,0,0,3],[3,0,0,1]]", "[[1,0,0,5],[5,0,0,7],[3,3,0,7]]",
"[[1,0,0,5],[3,0,0,1],[3,3,0,7]]", "[[1,0,0,5],[5,5,0,1],[5,0,0,7]]",
"[[1,0,0,5],[5,5,0,1],[3,0,0,1]]", "[[7,0,0,3],[5,0,0,7],[3,3,0,7]]",
"[[7,0,0,3],[3,0,0,1],[3,3,0,7]]", "[[5,5,0,1],[7,0,0,3],[5,0,0,7]]",
"[[7,0,24,1],[5,0,16,1],[1,1,16,1],[7,0,16,7]]",
"[[7,0,24,1],[1,1,16,1],[5,0,0,1],[7,0,16,7]]",
"[[1,1,0,1],[7,0,24,1],[5,0,16,1],[7,0,16,7]]",
"[[1,1,0,1],[7,0,24,1],[5,0,0,1],[7,0,16,7]]",
"[[7,7,8,1],[3,0,8,1],[1,0,8,7]]", "[[7,7,8,1],[1,0,8,7],[5,0,0,1]]",
"[[1,1,0,1],[3,0,8,1],[1,0,8,7]]", "[[1,1,0,1],[1,0,8,7],[5,0,0,1]]",
"[[1,1,0,1],[7,0,8,1],[5,0,0,1]]", "[[7,7,0,7],[7,0,8,1],[3,0,0,7]]",
"[[7,7,0,7],[7,0,8,1],[5,0,0,1]]", "[[1,1,0,1],[7,0,8,1],[3,0,0,7]]",
"[[3,6,6,11],[1,2,14,7],[1,6,2,1]]",
"[[3,6,6,11],[1,2,14,7],[1,4,0,1]]",
"[[3,12,0,3],[1,4,0,15],[1,2,2,9]]",
"[[1,0,0,15],[1,4,0,1],[1,2,2,9]]",
"[[1,0,8,5],[7,14,8,1],[7,0,8,1],[5,0,8,7]]",
"[[7,14,8,1],[7,0,8,1],[1,0,0,3],[5,0,8,7]]",
"[[1,0,8,5],[7,14,8,1],[7,0,8,1],[5,0,0,1]]",
"[[7,14,8,1],[7,0,8,1],[1,0,0,3],[5,0,0,1]]",
"[[1,0,8,5],[7,14,0,7],[7,0,8,1],[5,0,8,7]]",
"[[7,14,0,7],[7,0,8,1],[1,0,0,3],[5,0,8,7]]",
"[[1,0,8,5],[7,14,0,7],[7,0,8,1],[5,0,0,1]]",
"[[7,14,0,7],[7,0,8,1],[1,0,0,3],[5,0,0,1]]",
"[[3,6,0,7],[1,2,0,1],[1,0,8,7],[7,14,0,5]]",
"[[3,6,0,7],[1,2,0,1],[1,0,8,7],[1,0,0,3]]",
"[[1,2,0,1],[1,0,8,7],[5,0,0,1],[7,14,0,5]]",
"[[1,2,0,1],[1,0,8,7],[1,0,0,3],[5,0,0,1]]",
"[[3,0,8,1],[5,0,8,5],[5,5,8,3]]", "[[7,7,0,1],[3,0,8,1],[5,0,8,5]]",
"[[5,0,8,5],[5,5,8,3],[1,0,0,3]]", "[[7,7,0,1],[5,0,8,5],[1,0,0,3]]",
"[[7,7,0,3],[3,0,8,3],[7,0,0,5]]", "[[7,7,0,1],[3,0,8,3],[7,0,0,5]]",
"[[7,7,0,1],[3,0,8,3],[1,0,0,3]]", "[[1,1,0,7],[3,0,8,3],[1,0,0,3]]",
"[[1,0,8,5],[5,5,8,7],[7,0,8,5]]", "[[5,5,8,7],[1,0,0,7],[7,0,8,5]]",
"[[5,5,0,5],[1,0,8,5],[7,0,8,5]]", "[[5,5,0,5],[1,0,0,7],[7,0,8,5]]",
"[[3,3,0,3],[5,5,0,3],[1,0,8,3]]", "[[3,3,0,3],[1,0,0,7],[1,0,8,3]]",
"[[5,5,0,5],[1,0,0,7],[1,0,8,3]]", "[[5,5,0,5],[7,0,0,1],[1,0,8,3]]",
"[[9,9,12,3],[5,10,0,1],[13,10,4,1]]",
"[[15,14,0,3],[9,9,12,3],[7,14,12,3]]",
"[[3,0,12,3],[15,15,0,1],[5,10,0,1]]",
"[[15,14,0,3],[3,0,12,3],[3,9,0,1]]",
"[[1,1,12,3],[13,7,12,3],[1,0,4,1]]",
"[[7,7,0,1],[13,7,12,3],[1,0,4,1]]",
"[[7,7,0,1],[3,0,12,3],[3,9,0,1]]",
"[[7,7,0,1],[3,0,12,3],[9,11,0,3]]",
"[[1,1,8,7],[3,0,8,7],[7,0,8,1]]", "[[1,1,8,7],[7,0,8,1],[3,0,0,1]]",
"[[1,1,0,1],[3,0,8,7],[7,0,8,1]]", "[[1,1,0,1],[7,0,8,1],[3,0,0,1]]",
"[[1,1,0,1],[1,0,8,7],[3,0,0,1]]", "[[7,7,0,7],[1,0,8,7],[5,0,0,7]]",
"[[7,7,0,7],[1,0,8,7],[3,0,0,1]]", "[[1,1,0,1],[1,0,8,7],[5,0,0,7]]",
"[[3,3,16,3],[7,0,16,1],[7,0,16,7],[5,5,16,1]]",
"[[5,5,0,1],[3,3,16,3],[7,0,16,1],[7,0,16,7]]",
"[[3,3,0,3],[7,0,16,1],[7,0,16,7],[5,5,16,1]]",
"[[3,3,0,3],[5,5,0,1],[7,0,16,1],[7,0,16,7]]",
"[[5,5,0,5],[7,0,0,3],[5,5,0,3]]", "[[5,5,0,5],[7,0,0,1],[7,0,0,3]]",
"[[5,5,0,5],[5,5,0,1],[5,5,0,3]]", "[[5,5,0,5],[7,0,0,1],[5,5,0,1]]",
"[[3,3,0,3],[7,0,0,1],[5,5,0,1]]", "[[3,3,0,3],[1,0,0,7],[3,3,0,7]]",
"[[3,3,0,3],[7,0,0,1],[3,3,0,7]]", "[[3,3,0,3],[5,5,0,1],[1,0,0,7]]",
"[[1,3,4,1],[11,11,12,3],[7,7,4,1]]",
"[[1,3,4,1],[11,11,12,3],[15,0,0,1]]",
"[[11,11,12,3],[1,2,0,1],[7,7,4,1]]",
"[[11,11,12,3],[1,2,0,1],[15,0,0,1]]",
"[[1,2,0,1],[9,9,4,1],[15,0,0,1]]",
"[[3,6,0,3],[9,9,4,1],[13,0,0,3]]",
"[[3,6,0,3],[9,9,4,1],[15,0,0,1]]",
"[[1,2,0,1],[9,9,4,1],[13,0,0,3]]",
"[[5,0,16,1],[5,0,16,3],[7,7,16,1],[7,0,16,7]]",
"[[5,0,16,3],[7,7,16,1],[5,0,0,1],[7,0,16,7]]",
"[[7,7,0,1],[5,0,16,1],[5,0,16,3],[7,0,16,7]]",
"[[7,7,0,1],[5,0,16,3],[5,0,0,1],[7,0,16,7]]",
"[[7,7,0,1],[5,0,0,1],[5,0,0,3]]", "[[1,1,0,7],[3,0,0,5],[3,0,0,7]]",
"[[1,1,0,7],[3,0,0,5],[5,0,0,1]]", "[[7,7,0,1],[3,0,0,5],[3,0,0,7]]",
"[[7,7,0,1],[3,0,0,5],[5,0,0,1]]", "[[1,1,0,7],[3,0,0,7],[5,0,0,3]]",
"[[1,1,0,7],[5,0,0,1],[5,0,0,3]]", "[[7,7,0,1],[3,0,0,7],[5,0,0,3]]",
"[[7,7,8,3],[5,0,8,1],[7,0,8,5]]", "[[7,7,0,1],[5,0,8,1],[7,0,8,5]]",
"[[7,7,8,3],[5,0,0,3],[7,0,8,5]]", "[[7,7,0,1],[5,0,0,3],[7,0,8,5]]",
"[[7,7,0,1],[1,0,8,3],[5,0,0,3]]", "[[1,1,0,7],[3,0,0,5],[1,0,8,3]]",
"[[7,7,0,1],[3,0,0,5],[1,0,8,3]]", "[[1,1,0,7],[1,0,8,3],[5,0,0,3]]",
"[[5,5,8,1],[7,7,8,5],[5,0,8,7]]", "[[7,7,0,1],[7,7,8,5],[5,0,8,7]]",
"[[5,5,0,5],[5,5,8,1],[5,0,8,7]]", "[[7,7,0,1],[5,5,0,5],[5,0,8,7]]",
"[[3,3,0,3],[3,0,8,1],[3,0,0,5]]", "[[7,7,0,1],[3,3,0,3],[3,0,8,1]]",
"[[7,7,0,1],[5,5,0,5],[3,0,8,1]]", "[[1,1,0,7],[5,5,0,5],[3,0,8,1]]",
"[[3,0,8,7],[5,5,8,7],[5,0,8,7]]", "[[5,5,8,7],[1,0,0,7],[5,0,8,7]]",
"[[7,7,0,7],[3,0,8,7],[5,0,8,7]]", "[[7,7,0,7],[1,0,0,7],[5,0,8,7]]",
"[[7,7,0,1],[1,1,0,1],[3,0,8,1]]", "[[1,1,0,1],[3,0,8,1],[1,0,0,7]]",
"[[7,7,0,7],[3,0,8,1],[1,0,0,7]]", "[[7,7,0,7],[3,0,8,1],[7,0,0,1]]",
"[[3,3,4,13],[3,9,4,5],[3,0,12,15]]",
"[[3,3,4,13],[3,6,0,3],[3,0,12,15]]",
"[[3,3,4,13],[3,3,0,13],[3,9,4,5]]",
"[[3,3,4,13],[3,6,0,3],[3,3,0,13]]",
"[[5,0,8,5],[5,5,8,1],[5,0,8,3]]", "[[7,0,0,1],[5,0,8,5],[5,5,8,1]]",
"[[7,7,0,3],[5,0,8,5],[5,0,8,3]]", "[[7,7,0,3],[7,0,0,1],[5,0,8,5]]",
"[[1,1,0,3],[3,0,8,3],[1,1,0,5]]", "[[3,0,8,3],[7,0,0,1],[1,1,0,5]]",
"[[7,7,0,3],[3,0,8,3],[7,0,0,1]]", "[[7,7,0,3],[3,0,8,3],[1,0,0,7]]",
"[[1,1,8,3],[5,5,8,1],[5,0,8,5]]", "[[5,5,8,1],[5,0,8,5],[3,3,0,1]]",
"[[1,1,8,3],[7,7,0,3],[5,0,8,5]]", "[[7,7,0,3],[5,0,8,5],[3,3,0,1]]",
"[[7,7,0,3],[3,0,8,3],[3,3,0,1]]", "[[3,0,8,3],[5,5,0,7],[1,1,0,5]]",
"[[3,0,8,3],[1,1,0,5],[3,3,0,1]]", "[[7,7,0,3],[3,0,8,3],[5,5,0,7]]",
"[[7,7,0,7],[1,0,0,7],[3,0,0,7],[1,0,0,9]]",
"[[7,7,0,7],[5,0,16,1],[1,0,0,7],[3,0,0,7],[1,0,0,9]]",
"[[7,7,0,7],[1,0,0,7],[5,0,0,1],[1,0,0,9]]",
"[[7,7,0,7],[1,0,0,7],[5,0,0,1],[1,0,0,9],[7,0,16,7]]",
"[[7,7,0,7],[7,0,0,1],[3,0,0,7],[1,0,0,9]]",
"[[7,7,0,7],[7,0,0,1],[5,0,0,1],[1,0,0,9]]",
"[[1,1,0,1],[1,0,0,7],[3,0,0,7],[1,0,0,9]]",
"[[1,1,0,1],[5,0,16,1],[1,0,0,7],[3,0,0,7],[1,0,0,9]]",
"[[1,1,0,1],[1,0,0,7],[5,0,0,1],[1,0,0,9]]",
"[[1,1,0,1],[1,0,0,7],[5,0,0,1],[1,0,0,9],[7,0,16,7]]",
"[[1,1,0,1],[7,0,0,1],[3,0,0,7],[1,0,0,9]]",
"[[1,1,0,1],[7,0,0,1],[5,0,0,1],[1,0,0,9]]",
"[[1,1,0,1],[7,0,0,1],[7,0,0,7],[5,0,0,1]]",
"[[1,1,0,1],[7,0,0,1],[7,0,0,7],[7,0,16,15],[5,0,0,1]]",
"[[1,0,0,15],[7,0,0,7],[3,0,0,15],[7,7,0,15]]",
"[[5,0,16,1],[1,0,0,15],[7,0,0,7],[3,0,0,15],[7,7,0,15]]",
"[[1,0,0,15],[7,0,0,7],[5,0,0,1],[7,7,0,15]]",
"[[7,0,0,1],[7,0,0,7],[3,0,0,15],[7,7,0,15]]",
"[[5,0,16,1],[7,0,0,1],[7,0,0,7],[3,0,0,15],[7,7,0,15]]",
"[[7,0,0,1],[7,0,0,7],[7,0,16,15],[5,0,0,1],[7,7,0,15]]",
"[[1,0,8,5],[7,0,8,1],[3,3,8,7]]", "[[1,0,8,5],[7,0,8,1],[3,3,0,1]]",
"[[7,0,8,1],[1,0,0,3],[3,3,8,7]]", "[[7,0,8,1],[1,0,0,3],[3,3,0,1]]",
"[[3,3,0,3],[1,0,8,7],[7,0,0,5]]", "[[1,0,8,7],[3,3,0,1],[7,0,0,5]]",
"[[1,0,8,7],[1,0,0,3],[3,3,0,1]]", "[[5,5,0,7],[1,0,8,7],[1,0,0,3]]",
"[[21,0,2,1],[21,21,2,1]]", "[[31,31,0,1],[21,21,2,1]]",
"[[31,29,0,1],[17,19,2,1]]", "[[31,31,0,1],[17,19,2,1]]",
"[[15,15,6,3],[31,0,2,1]]", "[[15,15,6,3],[29,29,0,1]]",
"[[29,29,2,1],[29,29,0,1]]", "[[29,29,2,1],[15,15,0,3]]",
"[[1,0,16,7],[3,0,16,1],[3,0,16,3],[7,7,16,1]]",
"[[1,0,16,7],[3,0,16,3],[7,7,16,1],[5,0,0,1]]",
"[[1,1,0,1],[1,0,16,7],[3,0,16,1],[3,0,16,3]]",
"[[1,1,0,1],[1,0,16,7],[3,0,16,3],[5,0,0,1]]",
"[[1,0,16,7],[3,0,16,1],[7,7,16,1],[5,0,0,3]]",
"[[1,0,16,7],[7,7,16,1],[5,0,0,1],[5,0,0,3]]",
"[[1,1,0,1],[1,0,16,7],[3,0,16,1],[5,0,0,3]]",
"[[1,1,0,1],[1,0,16,7],[5,0,0,1],[5,0,0,3]]",
"[[3,0,0,5],[5,5,0,3],[1,0,0,3],[7,0,16,1]]",
"[[3,0,0,5],[5,5,0,3],[5,0,0,1],[7,0,16,1]]",
"[[1,1,0,1],[3,0,0,5],[1,0,0,3],[7,0,16,1]]",
"[[1,1,0,1],[3,0,0,5],[5,0,0,1],[7,0,16,1]]",
"[[1,1,0,1],[5,0,0,1],[5,0,0,3],[7,0,16,1]]",
"[[7,7,0,7],[3,0,0,7],[5,0,0,3],[7,0,16,1]]",
"[[7,7,0,7],[5,0,0,1],[5,0,0,3],[7,0,16,1]]",
"[[1,1,0,1],[3,0,0,7],[5,0,0,3],[7,0,16,1]]",
"[[3,9,28,7],[3,9,20,5],[1,7,4,1]]",
"[[3,9,28,7],[1,7,4,1],[3,0,0,1]]",
"[[5,20,0,1],[7,21,12,3],[7,21,4,1]]",
"[[3,0,0,1],[5,20,0,1],[7,21,12,3]]",
"[[5,15,4,1],[3,0,0,1],[5,20,0,1]]",
"[[3,12,0,7],[1,3,20,5],[1,0,0,3]]",
"[[3,12,0,7],[1,3,20,5],[3,0,0,1]]",
"[[1,4,0,5],[1,3,20,5],[1,0,0,3]]", "[[3,9,4,1],[1,3,4,1],[1,0,0,7]]",
"[[3,9,4,1],[1,0,0,5],[1,0,0,7]]", "[[1,3,12,3],[1,3,4,1],[3,0,0,5]]",
"[[1,3,12,3],[1,0,0,5],[3,0,0,5]]",
"[[1,0,0,5],[1,0,0,7],[1,3,20,5]]",
"[[1,0,0,7],[1,3,20,5],[3,0,0,1]]",
"[[3,0,0,5],[1,3,20,5],[1,0,0,3]]",
"[[1,0,0,5],[3,0,0,5],[1,3,20,5]]",
"[[1,0,16,3],[3,3,16,3],[7,0,16,1],[5,0,16,5]]",
"[[3,3,16,3],[3,0,0,1],[7,0,16,1],[5,0,16,5]]",
"[[1,1,0,1],[1,0,16,3],[7,0,16,1],[5,0,16,5]]",
"[[1,1,0,1],[3,0,0,1],[7,0,16,1],[5,0,16,5]]",
"[[1,0,16,3],[3,3,16,3],[5,0,0,3],[5,0,16,5]]",
"[[3,3,16,3],[3,0,0,1],[5,0,0,3],[5,0,16,5]]",
"[[1,1,0,1],[1,0,16,3],[5,0,0,3],[5,0,16,5]]",
"[[1,1,0,1],[3,0,0,1],[5,0,0,3],[5,0,16,5]]",
"[[7,0,0,3],[3,0,0,5],[3,0,16,3],[5,5,0,3]]",
"[[3,0,0,5],[3,0,16,3],[5,5,0,3],[3,0,0,1]]",
"[[1,1,0,1],[7,0,0,3],[3,0,0,5],[3,0,16,3]]",
"[[1,1,0,1],[3,0,0,5],[3,0,16,3],[3,0,0,1]]",
"[[1,1,0,1],[3,0,16,3],[3,0,0,1],[5,0,0,3]]",
"[[7,7,0,7],[3,0,16,3],[5,0,0,7],[5,0,0,3]]",
"[[7,7,0,7],[3,0,16,3],[3,0,0,1],[5,0,0,3]]",
"[[1,1,0,1],[3,0,16,3],[5,0,0,7],[5,0,0,3]]"]; 
// The nth entry in this list is the set of all m so that the subgroup 
// H_m is a maximal subgroup of H_n.  
subdatamaxsub := [ [ 2, 3, 4, 5, 6, 7 ], [ 8, 21 ],
[ 10, 20 ], [ 19, 22 ], [ 17 ], [ 8, 9, 10, 11, 12, 13, 14, 15, 16,
17, 18, 19 ], [ 20, 21, 22, 26, 55 ], [ 24, 25, 38, 46 ], [ 24, 37 ],
[ 27, 42 ], [ 23, 24, 26, 27, 28, 29, 35, 39, 41, 43, 45, 47, 49, 50,
53, 54 ], [ 25, 30, 31, 40, 51, 52 ], [ 25, 27, 32, 33, 34, 36, 44, 48
], [ 38, 44 ], [ 41, 46 ], [ 29, 38, 40, 42 ], [ 37, 40, 41, 44 ], [
37, 42, 46, 48 ], [ 29, 48 ], [ 60, 180 ], [ 59 ], [ 56, 57, 83, 179
], [ 58, 60, 64, 65, 68, 69, 76, 80, 127, 136, 146, 148 ], [ 58, 59,
61, 66, 67, 132, 141 ], [ 58, 62, 87, 96, 98, 99, 100, 101, 129, 142
], [ 59, 60, 73, 74, 83, 139, 140 ], [ 60, 92, 94, 95, 134, 135 ], [
67, 68, 81, 90, 128, 146 ], [ 76, 77, 83, 125, 133 ], [ 72, 88, 89,
93, 103, 104, 126, 142, 162, 163, 164, 167, 172, 174 ], [ 129, 145 ],
[ 79, 96, 115, 116, 157, 158 ], [ 78, 100 ], [ 75, 79, 94, 100 ], [
61, 63, 65, 82, 147, 148 ], [ 75, 78, 84, 85, 86, 92, 96, 102, 117,
118, 119, 120, 121, 122, 159, 168 ], [ 123 ], [ 62, 66 ], [ 63, 66,
80, 83, 90, 91, 95, 97, 105, 106, 107, 124, 128, 143, 144, 147, 160,
161, 165, 166 ], [ 126, 145 ], [ 80, 81, 82, 131, 138 ], [ 108 ], [
64, 67, 71, 74, 77, 82, 91, 92, 127, 137, 139, 144 ], [ 78, 79 ], [
61, 69, 70, 73, 77, 81, 94, 97, 109, 110, 111, 112, 130, 136, 140,
143, 149, 150, 151, 153 ], [ 62 ], [ 63, 64, 68, 70, 74, 97, 132, 133,
135, 138 ], [ 75 ], [ 70, 71, 76, 95, 130, 137 ], [ 65, 69, 71, 73,
90, 91, 113, 114, 125, 131, 134, 141, 152, 154, 155, 156, 170, 171 ],
[ 142, 145, 169, 173, 175, 176 ], [ 126, 129 ], [ 128, 133, 134, 136,
137, 138, 140, 141, 144, 148 ], [ 125, 127, 130, 131, 132, 135, 139,
143, 146, 147 ], [ 179, 180, 253, 441 ], [ 177, 178, 402, 439 ], [
370, 440 ], [ 183, 186, 187, 189, 248, 249, 251, 255, 357 ], [], [
277, 278 ], [ 209, 210, 251, 254, 392, 393 ], [ 198, 200, 203, 204,
246, 247 ], [ 254, 263, 302, 307 ], [ 191, 196, 255, 257, 263, 278 ],
[ 251, 261, 317, 318 ], [ 214, 254, 256, 395, 398 ], [ 255, 256 ], [
255, 264 ], [ 250, 251, 252, 258, 262, 277, 319, 320, 321, 322, 323,
324 ], [ 257, 260, 325, 326 ], [ 258, 259, 327, 328 ], [ 266, 273 ], [
253, 267, 276, 277, 348, 349, 350, 351 ], [ 275, 278 ], [ 197, 199,
217, 329, 330 ], [ 257, 258 ], [ 216, 257, 258, 275, 276, 373, 391 ],
[ 201, 202, 233, 234, 331, 332 ], [ 226, 333, 334, 368, 369 ], [ 224,
261, 262, 263, 264, 366, 400 ], [ 218, 220, 262, 264, 363, 364 ], [
261, 263 ], [ 232, 237, 275, 276, 370, 402 ], [ 195, 199, 201, 206,
227, 228, 336, 346 ], [ 192, 199, 202, 205, 211, 219, 225, 236, 335,
339, 341, 344 ], [ 184, 197, 201, 205, 212, 229, 337, 345 ], [ 181,
182, 183, 184, 185, 186, 198, 204, 269, 270, 273, 274 ], [ 266, 271 ],
[ 245, 265, 292, 294, 297, 299 ], [ 256, 262, 280, 281, 282, 283, 284,
285 ], [ 256, 259, 261, 276, 286, 287, 288, 289, 290, 291, 300, 352 ],
[ 195, 205, 207, 259, 278, 305, 306, 371 ], [ 244, 265, 293, 295, 296,
298 ], [ 231, 260, 277, 376, 378 ], [ 221, 222, 259, 260, 379, 381 ],
[ 184, 185, 192, 193, 194, 206, 208, 215, 246, 249, 313, 314, 315,
316, 386, 404 ], [ 254, 260, 264, 275, 301, 303, 304, 308, 309, 310,
311, 312, 377, 380 ], [ 181, 187, 188, 189, 193, 194, 200, 204, 244,
269, 272, 279 ], [ 182, 186, 187, 190, 194, 198, 203, 206, 268, 270,
271, 272 ], [ 181, 182, 188, 190, 247, 248 ], [ 183, 185, 188, 189,
190, 192, 200, 203, 245, 268, 274, 279 ], [ 193, 195, 197, 202, 213,
223, 230, 235, 338, 340, 342, 347 ], [ 296, 297, 298, 299 ], [ 292,
293, 294, 295 ], [ 214, 221, 224, 232, 282, 288, 301, 302, 388, 389,
390, 394 ], [ 221, 237, 286, 304, 359, 360 ], [ 214, 222, 224, 237,
281, 287, 307, 308, 362, 372, 374, 399 ], [ 238 ], [ 209, 220, 311,
320, 382, 401 ], [ 209, 216, 218, 231, 312, 321, 325, 349, 367, 383,
387, 397 ], [ 210, 218, 309, 322, 375, 384 ], [ 210, 216, 220, 231,
310, 323, 326, 351, 361, 385, 396, 403 ], [ 284, 290, 317, 324, 328,
348 ], [ 280, 291, 318, 319, 327, 350 ], [ 314, 334 ], [ 241, 242,
316, 333, 407, 408 ], [ 219, 223, 228, 229, 230, 236, 306, 315, 330,
332 ], [ 211, 212, 213, 225, 227, 235, 240, 243, 305, 313, 329, 331,
353, 354, 355, 356, 405, 406 ], [ 213, 215, 219, 233, 338, 344 ], [
207, 215, 217, 227, 229, 234, 235, 236, 336, 340, 341, 345 ], [ 208,
223, 225, 234, 339, 342 ], [ 207, 208, 211, 212, 217, 228, 230, 233,
335, 337, 346, 347 ], [ 239 ], [ 222, 232, 289, 303, 358, 365 ], [
258, 276, 414, 426 ], [ 265, 266, 419, 420 ], [ 255, 258, 261, 278 ],
[ 256, 264, 372, 390 ], [ 246, 248, 268, 269, 270, 279 ], [ 258, 260,
396, 397 ], [ 261, 262, 416, 418 ], [ 254, 255 ], [ 257, 275 ], [ 259,
277, 411, 412 ], [ 260, 278 ], [ 251, 257, 264, 277, 382, 383, 384,
385 ], [ 257, 259 ], [ 263, 264 ], [ 276, 278 ], [ 275, 277, 387, 403
], [ 251, 256, 413, 425 ], [ 244, 245, 247, 249, 271, 272, 273, 274,
409, 410, 423, 424 ], [ 254, 260, 262, 276, 358, 360, 361, 362, 367,
375, 389, 401, 415, 417 ], [ 256, 259, 263, 275, 359, 365, 374, 388 ],
[ 343, 421, 422 ], [ 255, 262 ], [ 254, 261, 394, 399 ], [ 251, 263 ],
[ 311, 322, 325, 351, 363, 367, 378, 385, 391, 392 ], [ 312, 323, 364,
373, 376, 384, 392, 397, 401, 403 ], [ 310, 321, 363, 373, 375, 378,
382, 387, 393, 396 ], [ 290, 319, 411, 414, 416, 425 ], [ 309, 320,
326, 349, 361, 364, 376, 383, 391, 393 ], [ 280, 317, 327, 348, 412,
413, 414, 416 ], [ 284, 318, 328, 350, 411, 418, 425, 426 ], [ 291,
324, 412, 413, 418, 426 ], [ 316, 334, 434, 435, 436, 437 ], [ 314,
333 ], [ 306, 315, 329, 331, 341, 342, 344, 345, 346, 347 ], [ 282,
286, 303, 307, 362, 366, 370, 381, 388, 398 ], [ 287, 301, 360, 365,
370, 372, 379, 394, 398, 400 ], [ 292, 296, 410, 419 ], [ 293, 299,
420, 423 ], [ 294, 298, 410, 420 ], [ 288, 308, 358, 359, 381, 390,
395, 399, 400, 402 ], [ 281, 289, 302, 304, 366, 374, 379, 389, 395,
402 ], [ 295, 297, 419, 423 ], [ 305, 313, 330, 332, 335, 336, 337,
338, 339, 340, 427, 428, 429, 430, 431, 432, 433, 438 ], [], [], [],
[], [], [], [], [], [ 719 ], [ 718 ], [ 439, 440, 538 ], [ 537 ], [
448, 450, 455, 459 ], [ 446, 448, 459, 463 ], [ 446, 447, 450, 452,
454, 458 ], [ 452, 464 ], [ 442, 452, 456, 458, 485, 486 ], [ 446,
452, 459, 460 ], [ 449, 454, 457, 459, 460, 465 ], [ 449, 450, 455,
462 ], [ 449, 450, 458, 465, 500, 501, 502, 503 ], [ 446, 449, 462,
463 ], [ 443, 445, 447, 461 ], [ 451, 458, 482, 483, 504, 505, 506,
519 ], [ 456, 465, 470, 475, 507, 508, 509, 510 ], [ 451, 460, 464,
465, 468, 469 ], [ 445, 453, 479, 487 ], [ 443, 444, 457, 461 ], [
467, 481 ], [ 442, 448, 463, 464 ], [ 466, 477 ], [ 451, 455, 456, 462
], [], [ 471, 472, 473, 474 ], [ 442, 451, 462, 463 ], [ 448, 455,
456, 464 ], [ 444, 453, 476, 478 ], [ 442, 460 ], [ 476, 478, 479,
487, 521, 527, 529, 531 ], [ 468, 475, 482, 486 ], [], [], [ 472, 477,
478, 482, 489, 490, 511, 512, 513, 514, 522, 528, 530, 532 ], [ 478,
481, 493, 498 ], [ 470, 472, 491, 492 ], [], [ 469, 470, 483, 485 ], [
520, 533 ], [ 466, 467, 477, 481 ], [], [ 473, 483 ], [], [], [], [
474, 475 ], [ 525, 526 ], [ 471, 482, 495, 497 ], [], [ 477, 479, 488,
494 ], [ 466, 487 ], [ 467, 476 ], [ 467, 473, 475, 487 ], [], [], [
472, 473 ], [ 471, 474 ], [ 470, 471, 479, 481, 496, 499, 515, 516,
517, 518, 523, 524, 534, 535 ], [ 466, 474, 476, 483 ], [], [], [], [
488, 489, 491, 493, 496, 497 ], [], [], [ 490, 492, 494, 495, 498, 499
], [ 455, 465, 604, 605, 606, 607 ], [ 458, 462, 608, 609, 610, 611 ],
[ 442, 451, 456, 464, 612, 613 ], [ 448, 455, 462, 463 ], [ 446, 449,
450, 459, 540, 541 ], [ 452, 458, 460, 465, 536, 541, 586, 614 ], [
537, 539, 540, 542, 570, 575, 577, 578 ], [ 454, 536, 540, 602, 670 ],
[ 536, 539, 542, 543, 568, 569, 576, 581 ], [ 537, 538, 596, 597 ],
[], [ 447, 457, 541 ], [ 600, 601 ], [ 443, 599 ], [ 542 ], [ 453,
544, 592, 593, 671 ], [], [ 545 ], [ 480, 539, 546 ], [ 461, 547 ],
[], [], [], [ 538, 543, 594, 595 ], [ 442, 446, 449, 451 ], [ 450,
456, 459, 464 ], [ 442, 446, 459, 464 ], [ 460, 463 ], [ 448, 460,
462, 465 ], [ 448, 452 ], [ 452, 455, 458, 463 ], [], [ 484, 538, 598
], [ 537, 543 ], [ 444, 445, 603 ], [ 449, 450, 451, 456 ], [ 579, 583
], [ 583, 584, 641, 680, 683 ], [ 579, 580 ], [ 546, 580, 583, 600 ],
[ 580, 584, 650, 674, 695, 706 ], [ 546, 579, 584, 601 ], [ 590, 591
], [ 588, 589 ], [ 587, 615, 618, 619, 678, 690 ], [ 582, 585 ], [
582, 587, 588, 590 ], [ 585, 589, 591, 615, 628, 675, 677, 713 ], [
643, 645, 707, 715 ], [ 571, 572, 573, 574 ], [ 646, 647, 709, 711 ],
[ 563, 564, 565, 566 ], [ 653, 655, 704, 717 ], [ 555, 556, 557, 558
], [ 616, 652, 708, 710 ], [ 548, 549, 550, 551 ], [ 544, 545, 582,
588, 591, 598, 600, 615 ], [], [], [], [], [ 478, 479, 544, 603, 636,
638 ], [ 476, 487, 592, 603 ], [ 630 ], [ 632, 634, 672, 673 ], [],
[], [], [ 635, 637, 686, 689 ], [ 470, 482, 586, 612, 627, 629 ], [
468, 485, 612, 614 ], [ 475, 483, 613, 614 ], [ 469, 486, 586, 613,
624, 625, 681, 682 ], [ 640, 705 ], [], [ 576, 577, 578, 581 ], [ 575,
581 ], [ 570, 576 ], [ 569, 577 ], [ 568, 578, 649, 651, 691, 692 ], [
568, 569, 570, 575, 654, 693, 694, 712 ], [ 657 ], [], [ 658, 716 ],
[], [ 477, 481, 617, 620 ], [ 466, 467 ], [ 471, 472, 621, 622 ], [
473, 474 ], [ 684, 685 ], [ 623, 631 ], [ 466, 473, 478, 482, 560,
561, 562, 567, 656, 659, 660, 661 ], [ 466, 479 ], [ 467, 478 ], [
470, 473 ], [ 474, 482 ], [ 467, 470, 474, 479, 552, 553, 554, 559,
639, 642, 644, 648 ], [ 471, 476, 477, 483 ], [ 471, 475 ], [], [ 472,
483 ], [ 476, 481 ], [ 477, 487 ], [ 472, 475, 481, 487 ], [ 594, 596
], [ 594, 597, 633, 687, 688 ], [ 595, 597, 626, 676, 679, 714 ], [
595, 596 ], [ 545, 585, 587, 589, 590, 592, 598, 601 ], [ 492, 493,
494, 497, 617, 621, 627, 636 ], [ 490, 496, 617, 621, 629, 638, 663,
665, 667, 668, 697, 698, 699, 700 ], [ 488, 491, 495, 498, 620, 622,
629, 638, 662, 664, 666, 669, 696, 701, 702, 703 ], [ 489, 499, 620,
622, 627, 636 ], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [ 718, 719 ], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [ 720, 721, 726, 727 ], [], [], [], [], [], [ 722, 723,
724, 725 ], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [], [],
[], [], [], [], [], [], [], [], [], [], [], [], [] ]; 
// The list of n so that X_n has genus zero and has rational points.  
genzeropoints :=
[ 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
20, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37,
38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 55, 56, 57, 58,
60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 73, 74, 75, 76, 77,
78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 89, 90, 91, 92, 93, 94, 95,
96, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110,
111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124,
177, 178, 181, 183, 185, 187, 188, 189, 190, 191, 192, 193, 194, 195,
196, 197, 199, 200, 202, 203, 204, 205, 207, 208, 209, 210, 211, 212,
213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226,
227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240,
241, 242, 243 ]; 
// The list of n so that X_n is an elliptic curve with positive rank.  
rankoneell := [ 150, 153, 155, 156, 165, 166,
167, 281, 284, 288, 289, 291, 295, 297, 302, 304, 308, 309, 312, 318,
320, 323, 324, 326, 328, 349, 350 ]; 
// The hard-coded list of which j-invariants lift to X_n for which the 
// genus of X_n > 1.
genustwoorhigher := [
<441,-18234932071051198464000/48661191875666868481>,
<441,-35817550197738955933474532061609984000/2301619141096101839813550846721>,
<556,78608>, <558,68769820673/16>, <563,2048>, <566,16974593/256>,
<619,-631595585199146625/218340105584896>,
<649,777228872334890625/60523872256>]; 
// Maps for X_n to the j-line for n in genzeropoints 
jmaps := [ <1, t>, <2, t^2 + 1728>, <3, -t^2 +
1728>, <4, -2*t^2 + 1728>, <5, 8*t^2 + 1728>, <6, t^3/(t + 16)>, <7,
(32*t - 4)/t^4>, <8, (64*t^6 + 576*t^4 + 1728*t^2 + 1728)/(t^4 - 2*t^2
+ 1)>, <9, (t^6 + 144*t^4 + 6912*t^2 + 110592)/(t^2 + 64)>, <10,
(1728*t^6 - 1728*t^4 + 576*t^2 - 64)/(t^6 + 2*t^4 + t^2)>, <11, (t^6 -
48*t^4 + 768*t^2 - 4096)/t^2>, <12, (t^6 + 48*t^4 + 768*t^2 +
4096)/t^2>, <13, (t^6 - 144*t^4 + 6912*t^2 - 110592)/(t^2 - 64)>, <14,
(4*t^6 + 96*t^4 + 768*t^2 + 2048)/t^2>, <15, (64*t^6 + 1152*t^4 +
6912*t^2 + 13824)/(t^2 + 8)>, <16, (4*t^6 - 288*t^4 + 6912*t^2 -
55296)/(t^2 - 32)>, <17, (1728*t^6 + 3456*t^4 + 2304*t^2 + 512)/(t^6 -
4*t^4 + 4*t^2)>, <18, (64*t^6 - 384*t^4 + 768*t^2 - 512)/t^2>, <19,
(1728*t^6 - 3456*t^4 + 2304*t^2 - 512)/(t^6 + 4*t^4 + 4*t^2)>, <20,
(-4*t^8 + 32*t^7 + 80*t^6 - 288*t^5 - 504*t^4 + 864*t^3 + 1296*t^2 -
864*t - 1188)/(t^4 + 4*t^3 + 6*t^2 + 4*t + 1)>, <22, (1728*t^8 -
1617408*t^6 - 35831808*t^5 - 360930816*t^4 - 2006581248*t^3 -
6352680960*t^2 - 10749542400*t - 7558272000)/(t^8 - 72*t^6 + 1944*t^4
- 23328*t^2 + 104976)>, <23, (-64*t^12 + 384*t^10 - 192*t^8 - 1792*t^6
+ 576*t^4 + 3456*t^2 + 1728)/(t^8 - 4*t^6 + 6*t^4 - 4*t^2 + 1)>, <24,
(256*t^12 + 768*t^10 + 1536*t^8 + 1792*t^6 + 1536*t^4 + 768*t^2 +
256)/(t^8 + 2*t^6 + t^4)>, <25, (256*t^12 - 768*t^10 + 1536*t^8 -
1792*t^6 + 1536*t^4 - 768*t^2 + 256)/(t^8 - 2*t^6 + t^4)>, <26,
(-64*t^12 - 384*t^10 - 192*t^8 + 1792*t^6 + 576*t^4 - 3456*t^2 +
1728)/(t^8 + 4*t^6 + 6*t^4 + 4*t^2 + 1)>, <27, (-64*t^12 + 672*t^10 -
2364*t^8 + 2828*t^6 - 591/4*t^4 + 21/8*t^2 - 1/64)/(t^10 + t^8 +
3/8*t^6 + 1/16*t^4 + 1/256*t^2)>, <28, (16*t^12 + 384*t^10 + 3648*t^8
+ 17408*t^6 + 43776*t^4 + 55296*t^2 + 27648)/(t^4 + 8*t^2 + 16)>, <29,
(1728*t^12 - 34560*t^10 + 251136*t^8 - 788480*t^6 + 1004544*t^4 -
552960*t^2 + 110592)/(t^12 + 4*t^10 - 4*t^8 - 32*t^6 - 16*t^4 + 64*t^2
+ 64)>, <30, (t^12 + 48*t^8 + 768*t^4 + 4096)/t^4>, <31, (1024*t^12 +
768*t^8 + 192*t^4 + 16)/t^8>, <32, (t^12 + 48*t^10 + 816*t^8 +
5632*t^6 + 13056*t^4 + 12288*t^2 + 4096)/(t^4 + 16*t^2)>, <33,
(16*t^12 + 384*t^10 + 3264*t^8 + 11264*t^6 + 13056*t^4 + 6144*t^2 +
1024)/(t^4 + 8*t^2)>, <34, (4096*t^12 - 24576*t^10 + 52224*t^8 -
45056*t^6 + 13056*t^4 - 1536*t^2 + 64)/(t^4 - 2*t^2)>, <35, (t^12 +
48*t^10 + 912*t^8 + 8704*t^6 + 43776*t^4 + 110592*t^2 + 110592)/(t^4 +
16*t^2 + 64)>, <36, (t^12 - 48*t^10 + 816*t^8 - 5632*t^6 + 13056*t^4 -
12288*t^2 + 4096)/(t^4 - 16*t^2)>, <37, (10976*t^12 + 28224*t^11 +
37128*t^10 + 32616*t^9 + 42297/2*t^8 + 10602*t^7 + 16807/4*t^6 +
5301/4*t^5 + 42297/128*t^4 + 4077/64*t^3 + 4641/512*t^2 + 441/512*t +
343/8192)/(t^12 + 5*t^11 + 11*t^10 + 113/8*t^9 + 759/64*t^8 +
221/32*t^7 + 23/8*t^6 + 221/256*t^5 + 759/4096*t^4 + 113/4096*t^3 +
11/4096*t^2 + 5/32768*t + 1/262144)>, <38, (64*t^12 - 384*t^10 +
1536*t^8 - 3584*t^6 + 6144*t^4 - 6144*t^2 + 4096)/(t^8 - 4*t^6 +
4*t^4)>, <39, (-t^12 + 48*t^10 - 192*t^8 - 14336*t^6 + 36864*t^4 +
1769472*t^2 + 7077888)/(t^8 - 32*t^6 + 384*t^4 - 2048*t^2 + 4096)>,
<40, (8000*t^12 + 9600*t^11 + 5040*t^10 + 2672*t^9 + 1587*t^8 +
636*t^7 + 413/2*t^6 + 159/2*t^5 + 1587/64*t^4 + 167/32*t^3 +
315/256*t^2 + 75/256*t + 125/4096)/(t^12 + 4*t^11 + 25/4*t^10 +
9/2*t^9 + 63/64*t^8 - 5/8*t^7 - 57/128*t^6 - 5/64*t^5 + 63/4096*t^4 +
9/1024*t^3 + 25/16384*t^2 + 1/8192*t + 1/262144)>, <41, (1728*t^12 +
34560*t^10 + 251136*t^8 + 788480*t^6 + 1004544*t^4 + 552960*t^2 +
110592)/(t^12 - 4*t^10 - 4*t^8 + 32*t^6 - 16*t^4 - 64*t^2 + 64)>, <42,
(128*t^12 - 3072*t^11 + 303104*t^9 + 417792*t^8 - 11403264*t^7 -
58720256*t^6 - 91226112*t^5 + 26738688*t^4 + 155189248*t^3 -
100663296*t + 33554432)/(t^12 + 16*t^11 + 112*t^10 + 384*t^9 + 192*t^8
- 4096*t^7 - 18432*t^6 - 32768*t^5 + 12288*t^4 + 196608*t^3 +
458752*t^2 + 524288*t + 262144)>, <43, (16*t^12 - 384*t^10 + 3648*t^8
- 17408*t^6 + 43776*t^4 - 55296*t^2 + 27648)/(t^4 - 8*t^2 + 16)>, <44,
(8*t^12 + 672*t^10 + 18912*t^8 + 180992*t^6 + 75648*t^4 + 10752*t^2 +
512)/(t^10 - 8*t^8 + 24*t^6 - 32*t^4 + 16*t^2)>, <45, (t^12 - 48*t^10
+ 912*t^8 - 8704*t^6 + 43776*t^4 - 110592*t^2 + 110592)/(t^4 - 16*t^2
+ 64)>, <46, (64*t^12 + 384*t^10 + 1536*t^8 + 3584*t^6 + 6144*t^4 +
6144*t^2 + 4096)/(t^8 + 4*t^6 + 4*t^4)>, <47, (-1024*t^12 + 768*t^8 -
192*t^4 + 16)/t^8>, <48, (-8*t^12 + 672*t^10 - 18912*t^8 + 180992*t^6
- 75648*t^4 + 10752*t^2 - 512)/(t^10 + 8*t^8 + 24*t^6 + 32*t^4 +
16*t^2)>, <49, (-t^12 - 48*t^10 - 192*t^8 + 14336*t^6 + 36864*t^4 -
1769472*t^2 + 7077888)/(t^8 + 32*t^6 + 384*t^4 + 2048*t^2 + 4096)>,
<50, (t^12 - 48*t^8 + 768*t^4 - 4096)/t^4>, <55, (-16384*t^13 +
8192*t^12 + 49152*t^11 - 73728*t^10 - 12288*t^9 + 116736*t^8 -
106496*t^7 + 4096*t^6 + 70656*t^5 - 70144*t^4 + 35840*t^3 - 10752*t^2
+ 1792*t - 128)/(t^16 - 4*t^14 + 7*t^12 - 7*t^10 + 35/8*t^8 - 7/4*t^6
+ 7/16*t^4 - 1/16*t^2 + 1/256)>, <56, (1728*t^16 - 1555200*t^14 -
35831808*t^13 - 403621056*t^12 - 2812796928*t^11 - 13331905344*t^10 -
45094330368*t^9 - 111483619704*t^8 - 202924486656*t^7 -
269971083216*t^6 - 256316120064*t^5 - 165509859276*t^4 -
66119763456*t^3 - 12914016300*t^2 + 1162261467/4)/(t^16 - 36*t^14 +
567*t^12 - 5103*t^10 + 229635/8*t^8 - 413343/4*t^6 + 3720087/16*t^4 -
4782969/16*t^2 + 43046721/256)>, <57, (-29000000*t^16 - 145280000*t^15
- 391283200*t^14 - 750631936*t^13 - 1117246208*t^12 - 1337748480*t^11
- 1309789696*t^10 - 1052867584*t^9 - 693335936*t^8 - 370770944*t^7 -
158256640*t^6 - 52466688*t^5 - 12899072*t^4 - 2180096*t^3 - 229888*t^2
- 13312*t - 320)/(t^16 - 16*t^15 + 104*t^14 - 336*t^13 + 476*t^12 +
112*t^11 - 1064*t^10 + 432*t^9 + 1222*t^8 - 432*t^7 - 1064*t^6 -
112*t^5 + 476*t^4 + 336*t^3 + 104*t^2 + 16*t + 1)>, <58, (16*t^24 +
672*t^20 + 9456*t^16 + 45248*t^12 + 9456*t^8 + 672*t^4 + 16)/(t^20 -
4*t^16 + 6*t^12 - 4*t^8 + t^4)>, <60, (-64*t^24 + 672*t^20 - 2364*t^16
+ 2828*t^12 - 591/4*t^8 + 21/8*t^4 - 1/64)/(t^20 + t^16 + 3/8*t^12 +
1/16*t^8 + 1/256*t^4)>, <61, (256*t^24 + 768*t^20 + 1536*t^16 +
1792*t^12 + 1536*t^8 + 768*t^4 + 256)/(t^16 + 2*t^12 + t^8)>, <62,
(4*t^24 + 672*t^20 + 37824*t^16 + 723968*t^12 + 605184*t^8 +
172032*t^4 + 16384)/(t^20 - 16*t^16 + 96*t^12 - 256*t^8 + 256*t^4)>,
<63, (1728*t^24 + 69120*t^22 + 1046016*t^20 + 7690240*t^18 +
32560128*t^16 + 104448000*t^14 + 234307584*t^12 + 417792000*t^10 +
520962048*t^8 + 492175360*t^6 + 267780096*t^4 + 70778880*t^2 +
7077888)/(t^24 - 8*t^22 + 8*t^20 + 96*t^18 - 272*t^16 - 256*t^14 +
1792*t^12 - 1024*t^10 - 4352*t^8 + 6144*t^6 + 2048*t^4 - 8192*t^2 +
4096)>, <64, (-64*t^24 + 192*t^22 + 504*t^20 - 1316*t^18 - 7407/4*t^16
+ 4899/2*t^14 + 50233/16*t^12 + 4899/32*t^10 - 7407/1024*t^8 -
329/1024*t^6 + 63/8192*t^4 + 3/16384*t^2 - 1/262144)/(t^20 - 2*t^18 +
7/4*t^16 - 7/8*t^14 + 35/128*t^12 - 7/128*t^10 + 7/1024*t^8 -
1/2048*t^6 + 1/65536*t^4)>, <65, (t^24 + 24*t^22 + 216*t^20 + 992*t^18
+ 3312*t^16 + 9984*t^14 + 19712*t^12 + 39936*t^10 + 52992*t^8 +
63488*t^6 + 55296*t^4 + 24576*t^2 + 4096)/(t^16 + 8*t^14 + 24*t^12 +
32*t^10 + 16*t^8)>, <66, (1728*t^24 + 82944*t^23 + 2101248*t^22 +
36495360*t^21 + 479453184*t^20 + 5008195584*t^19 + 42797891584*t^18 +
304316153856*t^17 + 1818588610560*t^16 + 9183889981440*t^15 +
39287173152768*t^14 + 142449234149376*t^13 + 437705016082432*t^12 +
1139593873195008*t^11 + 2514379081777152*t^10 + 4702151670497280*t^9 +
7448938948853760*t^8 + 9971831729553408*t^7 + 11219210491396096*t^6 +
10502947385376768*t^5 + 8043889629855744*t^4 + 4898324301742080*t^3 +
2256197860196352*t^2 + 712483534798848*t + 118747255799808)/(t^24 +
48*t^23 + 1024*t^22 + 12672*t^21 + 98176*t^20 + 457728*t^19 +
819200*t^18 - 4595712*t^17 - 40374272*t^16 - 133300224*t^15 -
90177536*t^14 + 1072693248*t^13 + 4899995648*t^12 + 8581545984*t^11 -
5771362304*t^10 - 68249714688*t^9 - 165373018112*t^8 -
150592290816*t^7 + 214748364800*t^6 + 959925190656*t^5 +
1647119958016*t^4 + 1700807049216*t^3 + 1099511627776*t^2 +
412316860416*t + 68719476736)>, <67, (16*t^24 + 192*t^20 + 1536*t^16 +
7168*t^12 + 24576*t^8 + 49152*t^4 + 65536)/(t^16 + 8*t^12 + 16*t^8)>,
<68, (1728*t^24 + 8640*t^22 + 16344*t^20 + 15020*t^18 + 31797/4*t^16 +
6375/2*t^14 + 14301/16*t^12 + 6375/32*t^10 + 31797/1024*t^8 +
3755/1024*t^6 + 2043/8192*t^4 + 135/16384*t^2 + 27/262144)/(t^24 -
t^22 + 1/8*t^20 + 3/16*t^18 - 17/256*t^16 - 1/128*t^14 + 7/1024*t^12 -
1/2048*t^10 - 17/65536*t^8 + 3/65536*t^6 + 1/524288*t^4 -
1/1048576*t^2 + 1/16777216)>, <69, (t^24 - 24*t^22 + 216*t^20 -
992*t^18 + 3312*t^16 - 9984*t^14 + 19712*t^12 - 39936*t^10 + 52992*t^8
- 63488*t^6 + 55296*t^4 - 24576*t^2 + 4096)/(t^16 - 8*t^14 + 24*t^12 -
32*t^10 + 16*t^8)>, <70, (1728*t^24 - 69120*t^22 + 1046016*t^20 -
7690240*t^18 + 32560128*t^16 - 104448000*t^14 + 234307584*t^12 -
417792000*t^10 + 520962048*t^8 - 492175360*t^6 + 267780096*t^4 -
70778880*t^2 + 7077888)/(t^24 + 8*t^22 + 8*t^20 - 96*t^18 - 272*t^16 +
256*t^14 + 1792*t^12 + 1024*t^10 - 4352*t^8 - 6144*t^6 + 2048*t^4 +
8192*t^2 + 4096)>, <71, (-t^24 - 24*t^22 + 504*t^20 + 10528*t^18 -
118512*t^16 - 1254144*t^14 + 12859648*t^12 - 5016576*t^10 -
1896192*t^8 + 673792*t^6 + 129024*t^4 - 24576*t^2 - 4096)/(t^20 +
16*t^18 + 112*t^16 + 448*t^14 + 1120*t^12 + 1792*t^10 + 1792*t^8 +
1024*t^6 + 256*t^4)>, <73, (1048576*t^21 + 56623104*t^20 +
1447034880*t^19 + 23177723904*t^18 + 260013293568*t^17 +
2164059537408*t^16 + 13818520403968*t^15 + 69095286374400*t^14 +
273881474531328*t^13 + 866290608635904*t^12 + 2191051796250624*t^11 +
4422098327961600*t^10 + 7075082446831616*t^9 + 8863987865223168*t^8 +
8520115603636224*t^7 + 6075901255090176*t^6 + 3034652092661760*t^5 +
949978046398464*t^4 + 140737488355328*t^3)/(t^24 + 64*t^23 + 1952*t^22
+ 37888*t^21 + 526976*t^20 + 5603328*t^19 + 47417344*t^18 +
328040448*t^17 + 1890185216*t^16 + 9190768640*t^15 + 38055182336*t^14
+ 134985285632*t^13 + 411573420032*t^12 + 1079882285056*t^11 +
2435531669504*t^10 + 4705673543680*t^9 + 7742198644736*t^8 +
10749229400064*t^7 + 12430172225536*t^6 + 11751030521856*t^5 +
8841190178816*t^4 + 5085241278464*t^3 + 2095944040448*t^2 +
549755813888*t + 68719476736)>, <74, (1728*t^24 - 8640*t^22 +
16344*t^20 - 15020*t^18 + 31797/4*t^16 - 6375/2*t^14 + 14301/16*t^12 -
6375/32*t^10 + 31797/1024*t^8 - 3755/1024*t^6 + 2043/8192*t^4 -
135/16384*t^2 + 27/262144)/(t^24 + t^22 + 1/8*t^20 - 3/16*t^18 -
17/256*t^16 + 1/128*t^14 + 7/1024*t^12 + 1/2048*t^10 - 17/65536*t^8 -
3/65536*t^6 + 1/524288*t^4 + 1/1048576*t^2 + 1/16777216)>, <75,
(-32*t^24 + 720*t^22 - 5601*t^20 + 66105/4*t^18 - 1522671/128*t^16 +
491085/128*t^14 - 1330727/2048*t^12 + 491085/8192*t^10 -
1522671/524288*t^8 + 66105/1048576*t^6 - 5601/16777216*t^4 +
45/67108864*t^2 - 1/2147483648)/(t^22 + 3/4*t^20 + 13/64*t^18 +
1/64*t^16 - 7/2048*t^14 - 7/8192*t^12 - 7/131072*t^10 + 1/262144*t^8 +
13/16777216*t^6 + 3/67108864*t^4 + 1/1073741824*t^2)>, <76,
(-8000/81*t^24 + 179200/27*t^23 - 3834880/27*t^22 + 67981312/81*t^21 +
50563072/27*t^20 + 186368000/9*t^19 - 1972748288/81*t^18 -
770244608/9*t^17 - 14969749504/27*t^16 + 172490752/81*t^15 +
47361556480/27*t^14 + 67366813696/27*t^13 + 121558269952/27*t^12 -
269467254784/27*t^11 + 757784903680/27*t^10 - 11039408128/81*t^9 -
3832255873024/27*t^8 + 788730478592/9*t^7 - 8080376987648/81*t^6 -
3053453312000/9*t^5 + 3313701486592/27*t^4 - 17820893052928/81*t^3 -
4021163130880/27*t^2 - 751619276800/27*t - 134217728000/81)/(t^24 +
16/3*t^23 - 16*t^22 + 3136/27*t^21 + 88096/81*t^20 - 159488/81*t^19 +
324352/81*t^18 + 2001920/27*t^17 - 6807808/81*t^16 + 7856128/81*t^15 +
49061888/27*t^14 - 76840960/81*t^13 + 314687488/81*t^12 +
307363840/81*t^11 + 784990208/27*t^10 - 502792192/81*t^9 -
1742798848/81*t^8 - 2049966080/27*t^7 + 1328545792/81*t^6 +
2613051392/81*t^5 + 5773459456/81*t^4 - 822083584/27*t^3 -
16777216*t^2 - 67108864/3*t + 16777216)>, <77, (1728*t^24 - 44928*t^22
+ 452448*t^20 - 2229344*t^18 + 5699796*t^16 - 7773168*t^14 +
5537396*t^12 - 1943292*t^10 + 1424949/4*t^8 - 69667/2*t^6 +
14139/8*t^4 - 351/8*t^2 + 27/64)/(t^24 - 2*t^22 - 15/2*t^20 + 3/2*t^18
+ 367/16*t^16 + 127/4*t^14 + 343/16*t^12 + 127/16*t^10 + 367/256*t^8 +
3/128*t^6 - 15/512*t^4 - 1/512*t^2 + 1/4096)>, <78, (2*t^24 + 720*t^22
+ 89616*t^20 + 4230720*t^18 + 48725472*t^16 + 251435520*t^14 +
681332224*t^12 + 1005742080*t^10 + 779607552*t^8 + 270766080*t^6 +
22941696*t^4 + 737280*t^2 + 8192)/(t^22 - 12*t^20 + 52*t^18 - 64*t^16
- 224*t^14 + 896*t^12 - 896*t^10 - 1024*t^8 + 3328*t^6 - 3072*t^4 +
1024*t^2)>, <79, (287496*t^24 + 3763584*t^23 + 22486464*t^22 +
79315200*t^21 + 177265728*t^20 + 250366464*t^19 + 239213312*t^18 +
381164544*t^17 + 1273038720*t^16 + 3023400960*t^15 + 4274632704*t^14 +
3569393664*t^13 + 2847586304*t^12 + 7138787328*t^11 + 17098530816*t^10
+ 24187207680*t^9 + 20368619520*t^8 + 12197265408*t^7 +
15309651968*t^6 + 32046907392*t^5 + 45380026368*t^4 + 40609382400*t^3
+ 23026139136*t^2 + 7707820032*t + 1177583616)/(t^24 + 36*t^23 +
596*t^22 + 6008*t^21 + 41112*t^20 + 201136*t^19 + 718608*t^18 +
1861536*t^17 + 3293680*t^16 + 2994816*t^15 - 2621824*t^14 -
15142144*t^13 - 28713728*t^12 - 30284288*t^11 - 10487296*t^10 +
23958528*t^9 + 52698880*t^8 + 59569152*t^7 + 45990912*t^6 +
25745408*t^5 + 10524672*t^4 + 3076096*t^3 + 610304*t^2 + 73728*t +
4096)>, <80, (1728*t^24 + 6048*t^22 + 6678*t^20 + 3913/2*t^18 -
16971/64*t^16 + 9597/64*t^14 - 13811/1024*t^12 + 9597/4096*t^10 -
16971/262144*t^8 + 3913/524288*t^6 + 3339/8388608*t^4 +
189/33554432*t^2 + 27/1073741824)/(t^24 - 5/2*t^22 + 65/32*t^20 -
49/128*t^18 - 785/4096*t^16 + 155/4096*t^14 + 839/65536*t^12 +
155/262144*t^10 - 785/16777216*t^8 - 49/33554432*t^6 +
65/536870912*t^4 - 5/2147483648*t^2 + 1/68719476736)>, <81, (1728*t^24
+ 179712*t^22 + 7239168*t^20 + 142678016*t^18 + 1459147776*t^16 +
7959724032*t^14 + 22681174016*t^12 + 31838896128*t^10 +
23346364416*t^8 + 9131393024*t^6 + 1853227008*t^4 + 184025088*t^2 +
7077888)/(t^24 + 8*t^22 - 120*t^20 - 96*t^18 + 5872*t^16 - 32512*t^14
+ 87808*t^12 - 130048*t^10 + 93952*t^8 - 6144*t^6 - 30720*t^4 +
8192*t^2 + 4096)>, <82, (2744000/9*t^24 + 13798400/3*t^23 +
103882240/3*t^22 + 1577867264/9*t^21 + 2032693760/3*t^20 +
2144727040*t^19 + 52258654208/9*t^18 + 13821583360*t^17 +
88310225920/3*t^16 + 510427824128/9*t^15 + 298660904960/3*t^14 +
479920455680/3*t^13 + 708001906688/3*t^12 + 959840911360/3*t^11 +
1194643619840/3*t^10 + 4083422593024/9*t^9 + 1412963614720/3*t^8 +
442290667520*t^7 + 3344553869312/9*t^6 + 274525061120*t^5 +
520369602560/3*t^4 + 807868039168/9*t^3 + 106375413760/3*t^2 +
28259123200/3*t + 11239424000/9)/(t^24 + 112/3*t^23 + 5848/9*t^22 +
63200/9*t^21 + 53000*t^20 + 2684096/9*t^19 + 11752928/9*t^18 +
13771136/3*t^17 + 119420528/9*t^16 + 289231360/9*t^15 +
595180288/9*t^14 + 1050446848/9*t^13 + 177612544*t^12 +
2100893696/9*t^11 + 2380721152/9*t^10 + 2313850880/9*t^9 +
1910728448/9*t^8 + 440676352/3*t^7 + 752187392/9*t^6 + 343564288/9*t^5
+ 13568000*t^4 + 32358400/9*t^3 + 5988352/9*t^2 + 229376/3*t + 4096)>,
<83, (1728*t^24 - 6048*t^22 + 6678*t^20 - 3913/2*t^18 - 16971/64*t^16
- 9597/64*t^14 - 13811/1024*t^12 - 9597/4096*t^10 - 16971/262144*t^8 -
3913/524288*t^6 + 3339/8388608*t^4 - 189/33554432*t^2 +
27/1073741824)/(t^24 + 5/2*t^22 + 65/32*t^20 + 49/128*t^18 -
785/4096*t^16 - 155/4096*t^14 + 839/65536*t^12 - 155/262144*t^10 -
785/16777216*t^8 + 49/33554432*t^6 + 65/536870912*t^4 +
5/2147483648*t^2 + 1/68719476736)>, <84, (-16*t^24 - 192*t^22 -
288*t^20 + 4160*t^18 + 13584*t^16 - 24960*t^14 - 128960*t^12 -
9600*t^10 + 382224*t^8 + 324160*t^6 - 37152*t^4 + 1344*t^2 - 16)/(t^20
+ 10*t^18 + 44*t^16 + 112*t^14 + 182*t^12 + 196*t^10 + 140*t^8 +
64*t^6 + 17*t^4 + 2*t^2)>, <85, (-4*t^24 + 96*t^22 - 288*t^20 -
8320*t^18 + 54336*t^16 + 199680*t^14 - 2063360*t^12 + 307200*t^10 +
24462336*t^8 - 41492480*t^6 - 9510912*t^4 - 688128*t^2 - 16384)/(t^20
- 20*t^18 + 176*t^16 - 896*t^14 + 2912*t^12 - 6272*t^10 + 8960*t^8 -
8192*t^6 + 4352*t^4 - 1024*t^2)>, <86, (-4*t^24 - 96*t^22 - 288*t^20 +
8320*t^18 + 54336*t^16 - 199680*t^14 - 2063360*t^12 - 307200*t^10 +
24462336*t^8 + 41492480*t^6 - 9510912*t^4 + 688128*t^2 - 16384)/(t^20
+ 20*t^18 + 176*t^16 + 896*t^14 + 2912*t^12 + 6272*t^10 + 8960*t^8 +
8192*t^6 + 4352*t^4 + 1024*t^2)>, <87, (4096*t^24 + 24576*t^22 +
64512*t^20 + 97280*t^18 + 93696*t^16 + 61440*t^14 + 29120*t^12 +
10560*t^10 + 2976*t^8 + 640*t^6 + 108*t^4 + 12*t^2 + 1)/(t^16 + 4*t^14
+ 13/2*t^12 + 11/2*t^10 + 41/16*t^8 + 5/8*t^6 + 1/16*t^4)>, <89, (t^24
- 24*t^22 + 1032*t^20 - 3072*t^19 - 9440*t^18 + 46080*t^17 +
217584*t^16 - 1867776*t^15 + 4066560*t^14 + 2605056*t^13 -
9840896*t^12 - 164069376*t^11 + 1131869184*t^10 - 3816325120*t^9 +
8412270336*t^8 - 13212057600*t^7 + 15286355968*t^6 - 13151502336*t^5 +
8350402560*t^4 - 3814457344*t^3 + 1189453824*t^2 - 227278848*t +
20123648)/(t^20 - 4*t^19 - 10*t^18 + 60*t^17 + 17*t^16 - 384*t^15 +
208*t^14 + 1344*t^13 - 1456*t^12 - 2688*t^11 + 4480*t^10 + 2688*t^9 -
7840*t^8 + 7936*t^6 - 3072*t^5 - 4096*t^4 + 3072*t^3 + 512*t^2 -
1024*t + 256)>, <90, (54000*t^24 + 1209600*t^23 + 13438080*t^22 +
98303488*t^21 + 531513216*t^20 + 2262789120*t^19 + 7893379584*t^18 +
23178921984*t^17 + 58389979392*t^16 + 127869853696*t^15 +
245678837760*t^14 + 416626950144*t^13 + 625734291456*t^12 +
833253900288*t^11 + 982715351040*t^10 + 1022958829568*t^9 +
934239670272*t^8 + 741725503488*t^7 + 505176293376*t^6 +
289637007360*t^5 + 136067383296*t^4 + 50331385856*t^3 +
13760593920*t^2 + 2477260800*t + 221184000)/(t^24 + 40*t^23 + 752*t^22
+ 8848*t^21 + 73208*t^20 + 453792*t^19 + 2192320*t^18 + 8477760*t^17 +
26738672*t^16 + 69711104*t^15 + 151659008*t^14 + 277058048*t^13 +
426557696*t^12 + 554116096*t^11 + 606636032*t^10 + 557688832*t^9 +
427818752*t^8 + 271288320*t^7 + 140308480*t^6 + 58085376*t^5 +
18741248*t^4 + 4530176*t^3 + 770048*t^2 + 81920*t + 4096)>, <91,
(-t^24 + 24*t^22 + 504*t^20 - 10528*t^18 - 118512*t^16 + 1254144*t^14
+ 12859648*t^12 + 5016576*t^10 - 1896192*t^8 - 673792*t^6 + 129024*t^4
+ 24576*t^2 - 4096)/(t^20 - 16*t^18 + 112*t^16 - 448*t^14 + 1120*t^12
- 1792*t^10 + 1792*t^8 - 1024*t^6 + 256*t^4)>, <92, (-4*t^24 +
720*t^22 - 44808*t^20 + 1057680*t^18 - 6090684*t^16 + 15714720*t^14 -
21291632*t^12 + 15714720*t^10 - 6090684*t^8 + 1057680*t^6 - 44808*t^4
+ 720*t^2 - 4)/(t^22 + 6*t^20 + 13*t^18 + 8*t^16 - 14*t^14 - 28*t^12 -
14*t^10 + 8*t^8 + 13*t^6 + 6*t^4 + t^2)>, <93, (t^24 - 24*t^22 +
312*t^20 + 192*t^19 - 2240*t^18 - 2880*t^17 + 9504*t^16 + 24576*t^15 -
70656*t^13 - 106496*t^12 + 24576*t^11 + 466944*t^10 + 1269760*t^9 +
2221056*t^8 + 2949120*t^7 + 3186688*t^6 + 2850816*t^5 + 2088960*t^4 +
1245184*t^3 + 589824*t^2 + 196608*t + 32768)/(t^16 + 8*t^15 + 20*t^14
- 8*t^13 - 130*t^12 - 200*t^11 + 108*t^10 + 648*t^9 + 577*t^8 -
384*t^7 - 1128*t^6 - 704*t^5 + 248*t^4 + 640*t^3 + 416*t^2 + 128*t +
16)>, <94, (1728*t^24 + 82944*t^23 + 1548288*t^22 + 12165120*t^21 -
9363456*t^20 - 919535616*t^19 - 6316097536*t^18 - 3336044544*t^17 +
178531860480*t^16 + 972784926720*t^15 + 946033655808*t^14 -
10871031988224*t^13 - 52593015390208*t^12 - 86968255905792*t^11 +
60546153971712*t^10 + 498065882480640*t^9 + 731266500526080*t^8 -
109315507617792*t^7 - 1655727072477184*t^6 - 1928405956165632*t^5 -
157092723818496*t^4 + 1632774767247360*t^3 + 1662461581197312*t^2 +
712483534798848*t + 118747255799808)/(t^24 + 48*t^23 + 1088*t^22 +
15488*t^21 + 154752*t^20 + 1143808*t^19 + 6377472*t^18 + 26468352*t^17
+ 75231232*t^16 + 90046464*t^15 - 408420352*t^14 - 3064987648*t^13 -
10964959232*t^12 - 24519901184*t^11 - 26138902528*t^10 +
46103789568*t^9 + 308147126272*t^8 + 867314958336*t^7 +
1671816019968*t^6 + 2398739234816*t^5 + 2596307730432*t^4 +
2078764171264*t^3 + 1168231104512*t^2 + 412316860416*t +
68719476736)>, <95, (-t^24 + 672*t^20 - 151296*t^16 + 11583488*t^12 -
38731776*t^8 + 44040192*t^4 - 16777216)/(t^20 + 64*t^16 + 1536*t^12 +
16384*t^8 + 65536*t^4)>, <96, (256*t^24 - 768*t^20 + 1536*t^16 -
1792*t^12 + 1536*t^8 - 768*t^4 + 256)/(t^16 - 2*t^12 + t^8)>, <97,
(1728*t^24 + 34560*t^23 + 256320*t^22 + 866240*t^21 + 1388376*t^20 +
1219920*t^19 + 320580*t^18 - 374220*t^17 - 1529163/4*t^16 - 51850*t^15
+ 182565/2*t^14 + 52275/2*t^13 - 279891/16*t^12 - 52275/8*t^11 +
182565/32*t^10 + 25925/32*t^9 - 1529163/1024*t^8 + 93555/256*t^7 +
80145/1024*t^6 - 76245/1024*t^5 + 173547/8192*t^4 - 13535/4096*t^3 +
4005/16384*t^2 - 135/16384*t + 27/262144)/(t^24 - 4*t^23 - t^22 +
23*t^21 - 143/8*t^20 - 195/4*t^19 + 1027/16*t^18 + 693/16*t^17 -
23185/256*t^16 - 301/32*t^15 + 7807/128*t^14 - 757/128*t^13 -
20521/1024*t^12 + 757/512*t^11 + 7807/2048*t^10 + 301/2048*t^9 -
23185/65536*t^8 - 693/16384*t^7 + 1027/65536*t^6 + 195/65536*t^5 -
143/524288*t^4 - 23/262144*t^3 - 1/1048576*t^2 + 1/1048576*t +
1/16777216)>, <98, (256*t^24 - 3072*t^22 + 16128*t^20 - 48640*t^18 +
93696*t^16 - 122880*t^14 + 116480*t^12 - 84480*t^10 + 47616*t^8 -
20480*t^6 + 6912*t^4 - 1536*t^2 + 256)/(t^16 - 8*t^14 + 26*t^12 -
44*t^10 + 41*t^8 - 20*t^6 + 4*t^4)>, <99, (256*t^24 + 3072*t^22 +
16128*t^20 + 48640*t^18 + 93696*t^16 + 122880*t^14 + 116480*t^12 +
84480*t^10 + 47616*t^8 + 20480*t^6 + 6912*t^4 + 1536*t^2 + 256)/(t^16
+ 8*t^14 + 26*t^12 + 44*t^10 + 41*t^8 + 20*t^6 + 4*t^4)>, <100,
(16*t^24 - 192*t^20 + 1536*t^16 - 7168*t^12 + 24576*t^8 - 49152*t^4 +
65536)/(t^16 - 8*t^12 + 16*t^8)>, <101, (1048576*t^24 - 1572864*t^22 +
1032192*t^20 - 389120*t^18 + 93696*t^16 - 15360*t^14 + 1820*t^12 -
165*t^10 + 93/8*t^8 - 5/8*t^6 + 27/1024*t^4 - 3/4096*t^2 +
1/65536)/(t^16 - t^14 + 13/32*t^12 - 11/128*t^10 + 41/4096*t^8 -
5/8192*t^6 + 1/65536*t^4)>, <102, (t^24 - 48*t^22 + 96*t^21 + 768*t^20
- 3264*t^19 - 736*t^18 + 30528*t^17 - 70608*t^16 + 8192*t^15 +
268800*t^14 - 574464*t^13 + 359424*t^12 + 694272*t^11 - 1944576*t^10 +
2091008*t^9 - 539904*t^8 - 1769472*t^7 + 3231744*t^6 - 3121152*t^5 +
2015232*t^4 - 901120*t^3 + 270336*t^2 - 49152*t + 4096)/(t^16 - 8*t^15
+ 12*t^14 + 104*t^13 - 650*t^12 + 1864*t^11 - 3332*t^10 + 4024*t^9 -
3359*t^8 + 1920*t^7 - 720*t^6 + 160*t^5 - 16*t^4)>, <103, (t^24 +
48*t^16 + 768*t^8 + 4096)/t^8>, <104, (256*t^24 + 768*t^16 + 768*t^8 +
256)/t^16>, <105, (-t^24 + 48*t^22 - 1008*t^20 + 12160*t^18 -
92352*t^16 + 448512*t^14 - 1304576*t^12 + 1622016*t^10 + 2002944*t^8 -
7929856*t^6 + 589824*t^4 + 14155776*t^2 + 7077888)/(t^16 - 32*t^14 +
416*t^12 - 2816*t^10 + 10624*t^8 - 22528*t^6 + 26624*t^4 - 16384*t^2 +
4096)>, <106, (-t^24 - 48*t^22 - 1008*t^20 - 12160*t^18 - 92352*t^16 -
448512*t^14 - 1304576*t^12 - 1622016*t^10 + 2002944*t^8 + 7929856*t^6
+ 589824*t^4 - 14155776*t^2 + 7077888)/(t^16 + 32*t^14 + 416*t^12 +
2816*t^10 + 10624*t^8 + 22528*t^6 + 26624*t^4 + 16384*t^2 + 4096)>,
<107, (-4096*t^24 + 24576*t^22 - 64512*t^20 + 97280*t^18 - 92352*t^16
+ 56064*t^14 - 20384*t^12 + 3168*t^10 + 489*t^8 - 242*t^6 + 9/4*t^4 +
27/4*t^2 + 27/64)/(t^16 - 4*t^14 + 13/2*t^12 - 11/2*t^10 + 83/32*t^8 -
11/16*t^6 + 13/128*t^4 - 1/128*t^2 + 1/4096)>, <108, (128*t^24 -
3072*t^23 - 6144*t^22 + 370688*t^21 + 1428480*t^20 - 14954496*t^19 -
136880128*t^18 - 391667712*t^17 + 33847296*t^16 + 3072098304*t^15 +
7585726464*t^14 + 3154182144*t^13 - 20746436608*t^12 -
47894102016*t^11 - 36310351872*t^10 + 27654356992*t^9 +
96373014528*t^8 + 114034999296*t^7 + 80912842752*t^6 + 37677957120*t^5
+ 11639980032*t^4 + 2324692992*t^3 + 283115520*t^2 + 18874368*t +
524288)/(t^24 + 16*t^23 + 104*t^22 + 256*t^21 - 872*t^20 - 10048*t^19
- 41504*t^18 - 84736*t^17 + 35056*t^16 + 952832*t^15 + 4054272*t^14 +
11264000*t^13 + 23835904*t^12 + 40491008*t^11 + 56503296*t^10 +
65413120*t^9 + 62951168*t^8 + 50155520*t^7 + 32753664*t^6 +
17235968*t^5 + 7116800*t^4 + 2211840*t^3 + 483328*t^2 + 65536*t +
4096)>, <109, (256*t^24 + 6144*t^22 + 64512*t^20 + 389120*t^18 +
1489152*t^16 + 3772416*t^14 + 6416384*t^12 + 7299072*t^10 +
5450496*t^8 + 2578432*t^6 + 728064*t^4 + 110592*t^2 + 6912)/(t^8 +
8*t^6 + 20*t^4 + 16*t^2 + 4)>, <110, (16777216*t^24 - 100663296*t^22 +
264241152*t^20 - 398458880*t^18 + 381222912*t^16 - 241434624*t^14 +
102662144*t^12 - 29196288*t^10 + 5450496*t^8 - 644608*t^6 + 45504*t^4
- 1728*t^2 + 27)/(t^8 - 2*t^6 + 5/4*t^4 - 1/4*t^2 + 1/64)>, <111,
(t^24 + 48*t^22 + 1008*t^20 + 12160*t^18 + 93072*t^16 + 471552*t^14 +
1604096*t^12 + 3649536*t^10 + 5450496*t^8 + 5156864*t^6 + 2912256*t^4
+ 884736*t^2 + 110592)/(t^8 + 16*t^6 + 80*t^4 + 128*t^2 + 64)>, <112,
(t^24 - 48*t^22 + 1008*t^20 - 12160*t^18 + 93072*t^16 - 471552*t^14 +
1604096*t^12 - 3649536*t^10 + 5450496*t^8 - 5156864*t^6 + 2912256*t^4
- 884736*t^2 + 110592)/(t^8 - 16*t^6 + 80*t^4 - 128*t^2 + 64)>, <113,
(t^24 - 48*t^16 + 768*t^8 - 4096)/t^8>, <114, (-256*t^24 + 768*t^16 -
768*t^8 + 256)/t^16>, <115, (64*t^24 + 768*t^20 + 3264*t^16 +
5632*t^12 + 3264*t^8 + 768*t^4 + 64)/(t^20 + 1/4*t^16)>, <116, (t^24 +
48*t^20 + 816*t^16 + 5632*t^12 + 13056*t^8 + 12288*t^4 + 4096)/(t^8 +
16*t^4)>, <117, (-64*t^24 + 768*t^20 - 3264*t^16 + 5632*t^12 -
3264*t^8 + 768*t^4 - 64)/(t^20 - 1/4*t^16)>, <118, (t^24 - 48*t^20 +
816*t^16 - 5632*t^12 + 13056*t^8 - 12288*t^4 + 4096)/(t^8 - 16*t^4)>,
<119, (t^24 + 48*t^22 + 1008*t^20 + 12160*t^18 + 92976*t^16 +
468480*t^14 + 1564160*t^12 + 3379200*t^10 + 4436736*t^8 + 3092480*t^6
+ 847872*t^4 + 98304*t^2 + 4096)/(t^8 + 16*t^6 + 80*t^4 + 128*t^2)>,
<120, (t^24 - 48*t^22 + 1008*t^20 - 12160*t^18 + 92976*t^16 -
468480*t^14 + 1564160*t^12 - 3379200*t^10 + 4436736*t^8 - 3092480*t^6
+ 847872*t^4 - 98304*t^2 + 4096)/(t^8 - 16*t^6 + 80*t^4 - 128*t^2)>,
<121, (256*t^24 + 6144*t^22 + 64512*t^20 + 389120*t^18 + 1487616*t^16
+ 3747840*t^14 + 6256640*t^12 + 6758400*t^10 + 4436736*t^8 +
1546240*t^6 + 211968*t^4 + 12288*t^2 + 256)/(t^8 + 8*t^6 + 20*t^4 +
16*t^2)>, <122, (16777216*t^24 - 100663296*t^22 + 264241152*t^20 -
398458880*t^18 + 380829696*t^16 - 239861760*t^14 + 100106240*t^12 -
27033600*t^10 + 4436736*t^8 - 386560*t^6 + 13248*t^4 - 192*t^2 +
1)/(t^8 - 2*t^6 + 5/4*t^4 - 1/4*t^2)>, <123, (10976*t^24 + 225792*t^23
+ 2338560*t^22 + 16484352*t^21 + 89505024*t^20 + 398039040*t^19 +
1497732096*t^18 + 4861280256*t^17 + 13780333056*t^16 +
34365390848*t^15 + 75629912064*t^14 + 146900189184*t^13 +
251211505664*t^12 + 376173428736*t^11 + 488976187392*t^10 +
545478672384*t^9 + 515466534912*t^8 + 406934126592*t^7 +
264375566336*t^6 + 138869735424*t^5 + 57616957440*t^4 +
18244698112*t^3 + 4167303168*t^2 + 616562688*t + 44957696)/(t^24 +
40*t^23 + 720*t^22 + 7760*t^21 + 56280*t^20 + 292768*t^19 +
1135808*t^18 + 3364160*t^17 + 7690480*t^16 + 13533440*t^15 +
17918464*t^14 + 16654848*t^13 + 8342784*t^12 - 2616320*t^11 -
8898560*t^10 - 7313408*t^9 - 1732864*t^8 + 1918976*t^7 + 1871872*t^6 +
430080*t^5 - 272384*t^4 - 188416*t^3 - 16384*t^2 + 16384*t + 4096)>,
<124, (-16*t^24 - 384*t^22 - 4032*t^20 - 24320*t^18 - 92352*t^16 -
224256*t^14 - 326144*t^12 - 202752*t^10 + 125184*t^8 + 247808*t^6 +
9216*t^4 - 110592*t^2 + 27648)/(t^16 + 16*t^14 + 104*t^12 + 352*t^10 +
664*t^8 + 704*t^6 + 416*t^4 + 128*t^2 + 16)>, <177, (1728*t^32 -
2709504*t^30 - 84934656*t^29 - 1352226816*t^28 - 14061404160*t^27 -
106521436160*t^26 - 625093902336*t^25 - 2956496744448*t^24 -
11585489731584*t^23 - 38383832104960*t^22 - 109169998823424*t^21 -
269665139326976*t^20 - 583591532691456*t^19 - 1113600197525504*t^18 -
1881802071343104*t^17 - 2823229083713536*t^16 - 3763604142686208*t^15
- 4454400790102016*t^14 - 4668732261531648*t^13 -
4314642229231616*t^12 - 3493439962349568*t^11 - 2456565254717440*t^10
- 1482942685642752*t^9 - 756863166578688*t^8 - 320048077996032*t^7 -
109077950627840*t^6 - 28797755719680*t^5 - 5538721038336*t^4 -
695784701952*t^3 - 44392513536*t^2 + 113246208)/(t^32 - 32*t^30 +
480*t^28 - 4480*t^26 + 29120*t^24 - 139776*t^22 + 512512*t^20 -
1464320*t^18 + 3294720*t^16 - 5857280*t^14 + 8200192*t^12 -
8945664*t^10 + 7454720*t^8 - 4587520*t^6 + 1966080*t^4 - 524288*t^2 +
65536)>, <178, (-23203748160*t^32 - 323005077504*t^31 -
2109977904128*t^30 - 8730811103232*t^29 - 26019722769920*t^28 -
59999818741760*t^27 - 111843664415744*t^26 - 173180712704000*t^25 -
226410818257664*t^24 - 252066767304704*t^23 - 239468220351488*t^22 -
193174781794304*t^21 - 130364955851264*t^20 - 71126597801984*t^19 -
28718444276736*t^18 - 5836526286848*t^17 + 2483144366208*t^16 +
3341506279424*t^15 + 1973119915008*t^14 + 763889752064*t^13 +
181814003200*t^12 + 5951166464*t^11 - 18634081280*t^10 -
10189539328*t^9 - 3149742848*t^8 - 306915328*t^7 - 62606336*t^6 +
101316608*t^5 - 1914368*t^4 + 8189952*t^3 - 1295360*t^2 + 153600*t -
72000)/(t^32 - 32*t^31 + 464*t^30 - 4000*t^29 + 22520*t^28 -
84896*t^27 + 206192*t^26 - 261664*t^25 - 81380*t^24 + 871520*t^23 -
1024816*t^22 - 704288*t^21 + 2465736*t^20 - 379680*t^19 - 3359760*t^18
+ 1376864*t^17 + 3607622*t^16 - 1376864*t^15 - 3359760*t^14 +
379680*t^13 + 2465736*t^12 + 704288*t^11 - 1024816*t^10 - 871520*t^9 -
81380*t^8 + 261664*t^7 + 206192*t^6 + 84896*t^5 + 22520*t^4 + 4000*t^3
+ 464*t^2 + 32*t + 1)>, <181, (4096*t^48 + 12288*t^46 + 14592*t^44 +
8512*t^42 + 3042*t^40 + 1545*t^38 + 14421/16*t^36 + 14751/64*t^34 +
171375/4096*t^32 + 70855/2048*t^30 + 235833/32768*t^28 -
79071/131072*t^26 + 3134807/4194304*t^24 - 79071/8388608*t^22 +
235833/134217728*t^20 + 70855/536870912*t^18 + 171375/68719476736*t^16
+ 14751/68719476736*t^14 + 14421/1099511627776*t^12 +
1545/4398046511104*t^10 + 1521/140737488355328*t^8 +
133/281474976710656*t^6 + 57/4503599627370496*t^4 +
3/18014398509481984*t^2 + 1/1152921504606846976)/(t^40 + 2*t^38 +
11/8*t^36 + 11/32*t^34 - 9/1024*t^32 - 39/2048*t^30 - 83/32768*t^28 +
27/131072*t^26 + 611/8388608*t^24 + 27/8388608*t^22 -
83/134217728*t^20 - 39/536870912*t^18 - 9/17179869184*t^16 +
11/34359738368*t^14 + 11/549755813888*t^12 + 1/2199023255552*t^10 +
1/281474976710656*t^8)>, <183, (16*t^48 + 672*t^40 + 9456*t^32 +
45248*t^24 + 9456*t^16 + 672*t^8 + 16)/(t^40 - 4*t^32 + 6*t^24 -
4*t^16 + t^8)>, <185, (16*t^48 + 720*t^44 + 11202*t^40 + 66105*t^36 +
1522671/16*t^32 + 491085/8*t^28 + 1330727/64*t^24 + 491085/128*t^20 +
1522671/4096*t^16 + 66105/4096*t^12 + 5601/32768*t^8 + 45/65536*t^4 +
1/1048576)/(t^44 - 3/2*t^40 + 13/16*t^36 - 1/8*t^32 - 7/128*t^28 +
7/256*t^24 - 7/2048*t^20 - 1/2048*t^16 + 13/65536*t^12 - 3/131072*t^8
+ 1/1048576*t^4)>, <187, (t^48 + 672*t^40 + 151296*t^32 +
11583488*t^24 + 38731776*t^16 + 44040192*t^8 + 16777216)/(t^40 -
64*t^32 + 1536*t^24 - 16384*t^16 + 65536*t^8)>, <188, (1/16*t^48 -
3*t^46 + 57*t^44 - 532*t^42 + 3042*t^40 - 24720*t^38 + 230736*t^36 -
944064*t^34 + 2742000*t^32 - 36277760*t^30 + 120746496*t^28 +
161937408*t^26 + 3210042368*t^24 + 647749632*t^22 + 1931943936*t^20 -
2321776640*t^18 + 701952000*t^16 - 966721536*t^14 + 945094656*t^12 -
405012480*t^10 + 199360512*t^8 - 139460608*t^6 + 59768832*t^4 -
12582912*t^2 + 1048576)/(t^40 - 32*t^38 + 352*t^36 - 1408*t^34 -
576*t^32 + 19968*t^30 - 42496*t^28 - 55296*t^26 + 312832*t^24 -
221184*t^22 - 679936*t^20 + 1277952*t^18 - 147456*t^16 - 1441792*t^14
+ 1441792*t^12 - 524288*t^10 + 65536*t^8)>, <189, (1/16*t^48 + 6*t^47
+ 276*t^46 + 8096*t^45 + 170688*t^44 + 2779392*t^43 + 36945664*t^42 +
421094400*t^41 + 4291398144*t^40 + 40195432448*t^39 +
349573398528*t^38 + 2824413511680*t^37 + 21238134865920*t^36 +
149572228743168*t^35 + 992610068987904*t^34 + 6211554303279104*t^33 +
36515979671371776*t^32 + 201026073608060928*t^31 +
1036249899261231104*t^30 + 5002031704131502080*t^29 +
22504629095513456640*t^28 + 93335740953009324032*t^27 +
351739640868526620672*t^26 + 1188716198729625894912*t^25 +
3571314952099434856448*t^24 + 9509729589837007159296*t^23 +
22511337015585703723008*t^22 + 47787899367940773904384*t^21 +
92178960775223118397440*t^20 + 163906574880981060157440*t^19 +
271646693591936166526976*t^18 + 421582232319292191277056*t^17 +
612636478398213302255616*t^16 + 833700705934744288755712*t^15 +
1065806945995837874896896*t^14 + 1284815661683475486867456*t^13 +
1459473514834619919237120*t^12 + 1552737748869902594211840*t^11 +
1537440065770838569254912*t^10 + 1414251051521943311220736*t^9 +
1207921192638552383422464*t^8 + 948220291463901295411200*t^7 +
665553914493422194917376*t^6 + 400552600816529203789824*t^5 +
196789865778333496639488*t^4 + 74672420010376264941568*t^3 +
20365205457375344984064*t^2 + 3541774862152233910272*t +
295147905179352825856)/(t^44 + 88*t^43 + 3704*t^42 + 99232*t^41 +
1898512*t^40 + 27566336*t^39 + 314791424*t^38 + 2886103040*t^37 +
21439795200*t^36 + 128638779392*t^35 + 609187790848*t^34 +
2115903684608*t^33 + 3916317851648*t^32 - 9702918324224*t^31 -
123082958176256*t^30 - 601728273612800*t^29 - 1826583388094464*t^28 -
2765498571816960*t^27 + 5288360750874624*t^26 + 51904453212635136*t^25
+ 190593985733984256*t^24 + 415235625701081088*t^23 +
338455088055975936*t^22 - 1415935268770283520*t^21 -
7481685557634924544*t^20 - 19717432069744230400*t^19 -
32265458988156452864*t^18 - 20348494569483010048*t^17 +
65704910521754451968*t^16 + 283991785214914330624*t^15 +
654110409703662026752*t^14 + 1104998700971997528064*t^13 +
1473331507471004467200*t^12 + 1586651925719831019520*t^11 +
1384467324048659972096*t^10 + 969904222949764759552*t^9 +
534383620984900943872*t^8 + 223450599111614529536*t^7 +
66725332079121268736*t^6 + 12682136550675316736*t^5 +
1152921504606846976*t^4)>, <190, (35152/9*t^48 + 346112*t^47 +
46418944/3*t^46 + 466157568*t^45 + 31983038464/3*t^44 +
197491163136*t^43 + 27762190385152/9*t^42 + 41719090380800*t^41 +
497622888087552*t^40 + 5303924948992000*t^39 +
153000248229232640/3*t^38 + 445529346581463040*t^37 +
10663191014426017792/3*t^36 + 25994298999995105280*t^35 +
174730763424732545024*t^34 + 1081459484909237174272*t^33 +
18508815280513614872576/3*t^32 + 32457234672745374023680*t^31 +
1417139570048807859126272/9*t^30 + 704173369159326766202880*t^29 +
8703838431108607963561984/3*t^28 + 11005426800090388713439232*t^27 +
115230418755305266453938176/3*t^26 + 123273769211818885332860928*t^25
+ 3273175754410138154018275328/9*t^24 +
986190153694551082662887424*t^23 + 7374746800339537053052043264/3*t^22
+ 5634778521646279021280886784*t^21 +
35650922213820858218749886464/3*t^20 +
23074352960612819474935971840*t^19 +
371494635450874687422797447168/9*t^18 +
68067754608417306624508559360*t^17 +
310526391865277507658020028416/3*t^16 +
145151034988568099743927894016*t^15 +
187615728628584809606231883776*t^14 +
223289328174848962675353845760*t^13 +
732768906847372970050466086912/3*t^12 +
244932348540881044048990699520*t^11 +
672902207922622566375159234560/3*t^10 +
186615268936573814092857344000*t^9 +
140068390835133076255499354112*t^8 + 93943039946603462035662438400*t^7
+ 500119161094237256108965101568/9*t^6 +
28461476118664173959975534592*t^5 +
36873932827813540031170084864/3*t^4 + 4299544677459428659842514944*t^3
+ 3425113520559422194912854016/3*t^2 + 204308463514872330525343744*t +
166000626605833768551841792/9)/(t^48 + 352/3*t^47 + 59776/9*t^46 +
724736/3*t^45 + 57089792/9*t^44 + 384133120/3*t^43 +
18588737536/9*t^42 + 27310080000*t^41 + 903358996480/3*t^40 +
8398010187776/3*t^39 + 198642338627584/9*t^38 + 442988531548160/3*t^37
+ 7493267477233664/9*t^36 + 3879404455329792*t^35 +
14203982142504960*t^34 + 104041100343246848/3*t^33 +
34709814290415616/9*t^32 - 1535482291504021504/3*t^31 -
9803533929980887040/3*t^30 - 38316960061591650304/3*t^29 -
297529548253897424896/9*t^28 - 111465409023142002688/3*t^27 +
435317943005289119744/3*t^26 + 1062498912435955564544*t^25 +
33784939610809965740032/9*t^24 + 8499991299487644516352*t^23 +
27860348352338503663616/3*t^22 - 57070289419848705376256/3*t^21 -
1218681029647963852374016/9*t^20 - 1255570147298235197161472/3*t^19 -
2569937598540909652213760/3*t^18 - 3220139758592241705156608/3*t^17 +
582334051670189519405056/9*t^16 + 13964160106690612081721344/3*t^15 +
15251409693756703679447040*t^14 + 33323830527196299068964864*t^13 +
514933420078384182884040704/9*t^12 +
243535520704308665468846080/3*t^11 +
873638244358585142776692736/9*t^10 + 295478715252512677672517632/3*t^9
+ 254272952495569575482490880/3*t^8 + 61496833055729217699840000*t^7 +
334864925761670207896551424/9*t^6 + 55359416834965312781680640/3*t^5 +
65820048890331935635668992/9*t^4 + 6684509756501982799986688/3*t^3 +
4410690295000248629592064/9*t^2 + 207784125246264389402624/3*t +
4722366482869645213696)>, <191, (1/4096*t^48 + 3/128*t^47 + 69/64*t^46
+ 509/16*t^45 + 10881/16*t^44 + 22551/2*t^43 + 151531*t^42 +
1702500*t^41 + 16362594*t^40 + 136928160*t^39 + 1012048320*t^38 +
6686135040*t^37 + 39899200768*t^36 + 217118115840*t^35 +
1086919446528*t^34 + 5046787096576*t^33 + 21895340027904*t^32 +
89321503457280*t^31 + 344373171060736*t^30 + 1259536876830720*t^29 +
4381918642569216*t^28 + 14528669942808576*t^27 +
45976791646470144*t^26 + 139030745258655744*t^25 +
402058899508690944*t^24 + 1112245962069245952*t^23 +
2942514665374089216*t^22 + 7438679010717990912*t^21 +
17948338759963508736*t^20 + 41272504379989032960*t^19 +
90275360554545577984*t^18 + 187320769618441666560*t^17 +
367342849041591435264*t^16 + 677368297802147299328*t^15 +
1167070869056045187072*t^14 + 1865030413803879137280*t^13 +
2741852198961569333248*t^12 + 3675741610680275435520*t^11 +
4451035582844664545280*t^10 + 4817731330879122309120*t^9 +
4605660765075919601664*t^8 + 3833689182799134720000*t^7 +
2729739820540314517504*t^6 + 1624970803149312884736*t^5 +
784058680726693871616*t^4 + 293418522922442555392*t^3 +
79551583817872441344*t^2 + 13835058055282163712*t +
1152921504606846976)/(t^40 + 64*t^39 + 1984*t^38 + 39936*t^37 +
589824*t^36 + 6840320*t^35 + 65032192*t^34 + 522027008*t^33 +
3613900800*t^32 + 21917597696*t^31 + 117836611584*t^30 +
566688219136*t^29 + 2454439591936*t^28 + 9623444652032*t^27 +
34284391890944*t^26 + 111262604197888*t^25 + 329402617430016*t^24 +
890100833583104*t^23 + 2194201081020416*t^22 + 4927203661840384*t^21 +
10053384568569856*t^20 + 18569239564648448*t^19 +
30890160707076096*t^18 + 45964533843361792*t^17 +
60631194324172800*t^16 + 70065278968397824*t^15 +
69827784456798208*t^14 + 58757901388349440*t^13 +
40532396646334464*t^12 + 21955048183431168*t^11 +
8725724278030336*t^10 + 2251799813685248*t^9 + 281474976710656*t^8)>,
<192, (1/4*t^48 + 696*t^46 + 654672*t^44 + 216040576*t^42 +
7561461888*t^40 + 45109340160*t^38 - 718836854784*t^36 -
1660941139968*t^34 + 32778430955520*t^32 - 276829437952000*t^30 +
2464198674087936*t^28 - 13354455546200064*t^26 +
53409700854628352*t^24 - 213671288739201024*t^22 +
630834860566511616*t^20 - 1133893377851392000*t^18 +
2148167251100958720*t^16 - 1741623016783085568*t^14 -
12060081181471801344*t^12 + 12108946295708712960*t^10 +
32476231518910414848*t^8 + 14846195336464039936*t^6 +
719819476379369472*t^4 + 12244161486913536*t^2 + 70368744177664)/(t^46
- 184*t^44 + 14704*t^42 - 662784*t^40 + 18203392*t^38 - 302012416*t^36
+ 2631094272*t^34 - 2874933248*t^32 - 150561226752*t^30 +
957929750528*t^28 + 1659893710848*t^26 - 26620442181632*t^24 +
26558299373568*t^22 + 245230016135168*t^20 - 616698784776192*t^18 -
188411625340928*t^16 + 2758902307356672*t^14 - 5066927537913856*t^12 +
4886435832266752*t^10 - 2846635604312064*t^8 + 1010451185926144*t^6 -
202310139510784*t^4 + 17592186044416*t^2)>, <193, (256*t^48 -
1536*t^46 + 3648*t^44 - 4256*t^42 + 3042*t^40 - 3090*t^38 +
14421/4*t^36 - 14751/8*t^34 + 171375/256*t^32 - 70855/64*t^30 +
235833/512*t^28 + 79071/1024*t^26 + 3134807/16384*t^24 +
79071/16384*t^22 + 235833/131072*t^20 - 70855/262144*t^18 +
171375/16777216*t^16 - 14751/8388608*t^14 + 14421/67108864*t^12 -
1545/134217728*t^10 + 1521/2147483648*t^8 - 133/2147483648*t^6 +
57/17179869184*t^4 - 3/34359738368*t^2 + 1/1099511627776)/(t^40 -
4*t^38 + 11/2*t^36 - 11/4*t^34 - 9/64*t^32 + 39/64*t^30 - 83/512*t^28
- 27/1024*t^26 + 611/32768*t^24 - 27/16384*t^22 - 83/131072*t^20 +
39/262144*t^18 - 9/4194304*t^16 - 11/4194304*t^14 + 11/33554432*t^12 -
1/67108864*t^10 + 1/4294967296*t^8)>, <194, (256*t^48 + 1536*t^46 +
3648*t^44 + 4256*t^42 + 3042*t^40 + 3090*t^38 + 14421/4*t^36 +
14751/8*t^34 + 171375/256*t^32 + 70855/64*t^30 + 235833/512*t^28 -
79071/1024*t^26 + 3134807/16384*t^24 - 79071/16384*t^22 +
235833/131072*t^20 + 70855/262144*t^18 + 171375/16777216*t^16 +
14751/8388608*t^14 + 14421/67108864*t^12 + 1545/134217728*t^10 +
1521/2147483648*t^8 + 133/2147483648*t^6 + 57/17179869184*t^4 +
3/34359738368*t^2 + 1/1099511627776)/(t^40 + 4*t^38 + 11/2*t^36 +
11/4*t^34 - 9/64*t^32 - 39/64*t^30 - 83/512*t^28 + 27/1024*t^26 +
611/32768*t^24 + 27/16384*t^22 - 83/131072*t^20 - 39/262144*t^18 -
9/4194304*t^16 + 11/4194304*t^14 + 11/33554432*t^12 + 1/67108864*t^10
+ 1/4294967296*t^8)>, <195, (-16*t^48 + 720*t^44 - 11202*t^40 +
66105*t^36 - 1522671/16*t^32 + 491085/8*t^28 - 1330727/64*t^24 +
491085/128*t^20 - 1522671/4096*t^16 + 66105/4096*t^12 - 5601/32768*t^8
+ 45/65536*t^4 - 1/1048576)/(t^44 + 3/2*t^40 + 13/16*t^36 + 1/8*t^32 -
7/128*t^28 - 7/256*t^24 - 7/2048*t^20 + 1/2048*t^16 + 13/65536*t^12 +
3/131072*t^8 + 1/1048576*t^4)>, <196, (-64*t^48 - 768*t^47 - 4416*t^46
- 16288*t^45 - 42804*t^44 - 83004*t^43 - 116971*t^42 - 212145/2*t^41 -
597537/32*t^40 + 2077515/16*t^39 + 16909305/64*t^38 +
37534965/128*t^37 + 188594087/1024*t^36 - 3179985/1024*t^35 -
657818793/4096*t^34 - 1755322159/8192*t^33 - 43901835759/262144*t^32 -
2532744975/32768*t^31 - 155342417/131072*t^30 + 9466285455/262144*t^29
+ 84988212627/2097152*t^28 + 61782097509/2097152*t^27 +
139192761153/8388608*t^26 + 128342978907/16777216*t^25 +
791625177369/268435456*t^24 + 128342978907/134217728*t^23 +
139192761153/536870912*t^22 + 61782097509/1073741824*t^21 +
84988212627/8589934592*t^20 + 9466285455/8589934592*t^19 -
155342417/34359738368*t^18 - 2532744975/68719476736*t^17 -
43901835759/4398046511104*t^16 - 1755322159/1099511627776*t^15 -
657818793/4398046511104*t^14 - 3179985/8796093022208*t^13 +
188594087/70368744177664*t^12 + 37534965/70368744177664*t^11 +
16909305/281474976710656*t^10 + 2077515/562949953421312*t^9 -
597537/9007199254740992*t^8 - 212145/4503599627370496*t^7 -
116971/18014398509481984*t^6 - 20751/36028797018963968*t^5 -
10701/288230376151711744*t^4 - 509/288230376151711744*t^3 -
69/1152921504606846976*t^2 - 3/2305843009213693952*t -
1/73786976294838206464)/(t^44 + 10*t^43 + 48*t^42 + 593/4*t^41 +
10669/32*t^40 + 18745/32*t^39 + 6735/8*t^38 + 261677/256*t^37 +
4397069/4096*t^36 + 508181/512*t^35 + 419267/512*t^34 +
2495337/4096*t^33 + 13502607/32768*t^32 + 8357861/32768*t^31 +
4758817/32768*t^30 + 20021681/262144*t^29 + 312225785/8388608*t^28 +
70670939/4194304*t^27 + 14886599/2097152*t^26 + 93504671/33554432*t^25
+ 273840151/268435456*t^24 + 93504671/268435456*t^23 +
14886599/134217728*t^22 + 70670939/2147483648*t^21 +
312225785/34359738368*t^20 + 20021681/8589934592*t^19 +
4758817/8589934592*t^18 + 8357861/68719476736*t^17 +
13502607/549755813888*t^16 + 2495337/549755813888*t^15 +
419267/549755813888*t^14 + 508181/4398046511104*t^13 +
4397069/281474976710656*t^12 + 261677/140737488355328*t^11 +
6735/35184372088832*t^10 + 18745/1125899906842624*t^9 +
10669/9007199254740992*t^8 + 593/9007199254740992*t^7 +
3/1125899906842624*t^6 + 5/72057594037927936*t^5 +
1/1152921504606846976*t^4)>, <197, (207646/6561*t^48 -
8341184/2187*t^47 + 290250944/2187*t^46 - 2371351808/6561*t^45 -
70659705728/2187*t^44 - 250785983488/2187*t^43 +
2369739971584/6561*t^42 - 6284946108416/2187*t^41 -
16727206700032/729*t^40 + 171162607304704/6561*t^39 -
186777349898240/2187*t^38 - 897267962937344/729*t^37 +
1017155880189952/729*t^36 - 9730939109507072/2187*t^35 -
57476450516992000/2187*t^34 + 591540222312841216/6561*t^33 -
125864743239811072/729*t^32 + 107591313204969472/2187*t^31 +
12838542402696249344/6561*t^30 - 6642977368124162048/2187*t^29 +
30732166651235532800/2187*t^28 + 12065016641537179648/6561*t^27 -
51779599482795065344/2187*t^26 + 81124937002299424768/2187*t^25 -
2397048714010386497536/6561*t^24 - 324499748009197699072/2187*t^23 -
828473591724721045504/2187*t^22 - 772161065058379497472/6561*t^21 +
7867434662716296396800/2187*t^20 + 6802408824959141937152/2187*t^19 +
52586669681443837313024/6561*t^18 - 1762776075550219829248/2187*t^17 -
8248671812964258414592/729*t^16 - 155068720037977447727104/6561*t^15 -
60268426577305403392000/2187*t^14 + 40814516830761950117888/2187*t^13
+ 17065043907616945733632/729*t^12 + 60214633696319259017216/729*t^11
- 50137663090405607997440/2187*t^10 -
183784450167948596740096/6561*t^9 - 71842805730069522153472/729*t^8 +
107974551971076761452544/2187*t^7 + 162847290847635989069824/6561*t^6
+ 68935526232073984540672/2187*t^5 - 77691168063166431100928/2187*t^4
+ 10429315545774562476032/6561*t^3 + 5106148606415369928704/2187*t^2 +
586958643034824114176/2187*t + 58447153014060875776/6561)/(t^48 +
40/3*t^47 + 544/9*t^46 - 7136/27*t^45 - 398528/81*t^44 -
6137216/243*t^43 - 10789376/729*t^42 + 1289818624/2187*t^41 +
24867701248/6561*t^40 + 63383455744/6561*t^39 - 91577491456/6561*t^38
- 1412684210176/6561*t^37 - 5783158636544/6561*t^36 -
431520907264/243*t^35 + 3353702170624/6561*t^34 +
111670095118336/6561*t^33 + 54439576010752/729*t^32 +
460330602856448/2187*t^31 + 1163342588674048/2187*t^30 +
6877045594259456/6561*t^29 + 15658048160792576/6561*t^28 +
22243325633363968/6561*t^27 + 2189051482341376/243*t^26 +
11111933818175488/2187*t^25 + 215647207435534336/6561*t^24 -
44447735272701952/2187*t^23 + 35024823717462016/243*t^22 -
1423572840535293952/6561*t^21 + 4008460329162899456/6561*t^20 -
7042094688521682944/6561*t^19 + 4765051243208900608/2187*t^18 -
7542056597200044032/2187*t^17 + 3567752053440643072/729*t^16 -
29273645414701072384/6561*t^15 + 3516611607264231424/6561*t^14 +
1809929867421024256/243*t^13 - 97025301607564181504/6561*t^12 +
94803632535648600064/6561*t^11 - 24582645678327463936/6561*t^10 -
68057467381985837056/6561*t^9 + 106805963586858385408/6561*t^8 -
22158915231406882816/2187*t^7 - 741440273027956736/729*t^6 +
1686985088543227904/243*t^5 - 438186169994313728/81*t^4 +
31384459903238144/27*t^3 + 9570149208162304/9*t^2 -
2814749767106560/3*t + 281474976710656)>, <199, (-1/2*t^48 + 696*t^46
- 327336*t^44 + 54010144*t^42 - 945182736*t^40 + 2819333760*t^38 +
22463651712*t^36 - 25952205312*t^34 - 256081491840*t^32 -
1081364992000*t^30 - 4812888035328*t^28 - 13041460494336*t^26 -
26078955495424*t^24 - 52165841977344*t^22 - 77006208565248*t^20 -
69207359488000*t^18 - 65556861911040*t^16 - 26575058239488*t^14 +
92011117412352*t^12 + 46191964323840*t^10 - 61943495786496*t^8 +
14158435188736*t^6 - 343236673536*t^4 + 2919235584*t^2 -
8388608)/(t^46 + 92*t^44 + 3676*t^42 + 82848*t^40 + 1137712*t^38 +
9437888*t^36 + 41110848*t^34 + 22460416*t^32 - 588129792*t^30 -
1870956544*t^28 + 1620989952*t^26 + 12998262784*t^24 + 6483959808*t^22
- 29935304704*t^20 - 37640306688*t^18 + 5749866496*t^16 +
42097508352*t^14 + 38657589248*t^12 + 18640273408*t^10 +
5429526528*t^8 + 963641344*t^6 + 96468992*t^4 + 4194304*t^2)>, <200,
(4*t^48 - 48*t^46 + 948*t^44 - 7732*t^42 + 144801/2*t^40 - 403665*t^38
+ 8282361/4*t^36 - 29311551/4*t^34 + 1105382895/64*t^32 -
223280215/8*t^30 + 1020068493/32*t^28 - 835340889/32*t^26 +
3942356327/256*t^24 - 835340889/128*t^22 + 1020068493/512*t^20 -
223280215/512*t^18 + 1105382895/16384*t^16 - 29311551/4096*t^14 +
8282361/16384*t^12 - 403665/16384*t^10 + 144801/131072*t^8 -
1933/65536*t^6 + 237/262144*t^4 - 3/262144*t^2 + 1/4194304)/(t^44 -
10*t^42 + 63/2*t^40 - 29/2*t^38 - 1299/16*t^36 + 119/2*t^34 +
909/8*t^32 - 301/8*t^30 - 9719/128*t^28 + 381/64*t^26 + 6669/256*t^24
+ 381/256*t^22 - 9719/2048*t^20 - 301/512*t^18 + 909/2048*t^16 +
119/2048*t^14 - 1299/65536*t^12 - 29/32768*t^10 + 63/131072*t^8 -
5/131072*t^6 + 1/1048576*t^4)>, <202, (2*t^48 + 696*t^46 + 81834*t^44
+ 3375634*t^42 + 59073921/4*t^40 + 22026045/2*t^38 - 175497279/8*t^36
- 50687901/8*t^34 + 2000636655/128*t^32 - 264005125/16*t^30 +
1175021493/64*t^28 - 795987579/64*t^26 + 3183466247/512*t^24 -
795987579/256*t^22 + 1175021493/1024*t^20 - 264005125/1024*t^18 +
2000636655/32768*t^16 - 50687901/8192*t^14 - 175497279/32768*t^12 +
22026045/32768*t^10 + 59073921/262144*t^8 + 1687817/131072*t^6 +
40917/524288*t^4 + 87/524288*t^2 + 1/8388608)/(t^46 - 23*t^44 +
919/4*t^42 - 2589/2*t^40 + 71107/16*t^38 - 147467/16*t^36 +
642357/64*t^34 - 10967/8*t^32 - 1148691/128*t^30 + 913553/128*t^28 +
791499/512*t^26 - 793351/256*t^24 + 791499/2048*t^22 +
913553/2048*t^20 - 1148691/8192*t^18 - 10967/2048*t^16 +
642357/65536*t^14 - 147467/65536*t^12 + 71107/262144*t^10 -
2589/131072*t^8 + 919/1048576*t^6 - 23/1048576*t^4 + 1/4194304*t^2)>,
<203, (64*t^48 + 192*t^46 + 948*t^44 + 1933*t^42 + 144801/32*t^40 +
403665/64*t^38 + 8282361/1024*t^36 + 29311551/4096*t^34 +
1105382895/262144*t^32 + 223280215/131072*t^30 +
1020068493/2097152*t^28 + 835340889/8388608*t^26 +
3942356327/268435456*t^24 + 835340889/536870912*t^22 +
1020068493/8589934592*t^20 + 223280215/34359738368*t^18 +
1105382895/4398046511104*t^16 + 29311551/4398046511104*t^14 +
8282361/70368744177664*t^12 + 403665/281474976710656*t^10 +
144801/9007199254740992*t^8 + 1933/18014398509481984*t^6 +
237/288230376151711744*t^4 + 3/1152921504606846976*t^2 +
1/73786976294838206464)/(t^44 + 5/2*t^42 + 63/32*t^40 + 29/128*t^38 -
1299/4096*t^36 - 119/2048*t^34 + 909/32768*t^32 + 301/131072*t^30 -
9719/8388608*t^28 - 381/16777216*t^26 + 6669/268435456*t^24 -
381/1073741824*t^22 - 9719/34359738368*t^20 + 301/34359738368*t^18 +
909/549755813888*t^16 - 119/2199023255552*t^14 -
1299/281474976710656*t^12 + 29/562949953421312*t^10 +
63/9007199254740992*t^8 + 5/36028797018963968*t^6 +
1/1152921504606846976*t^4)>, <204, (1556068/81*t^48 + 19781248/27*t^47
+ 385841792/27*t^46 + 14763951616/81*t^45 + 45949447936/27*t^44 +
334196074496/27*t^43 + 5945358026752/81*t^42 + 9855902359552/27*t^41 +
13976321951744/9*t^40 + 462910369595392/81*t^39 +
498430267064320/27*t^38 + 474534262734848/9*t^37 +
1209738445520896/9*t^36 + 8346214052921344/27*t^35 +
17428565755166720/27*t^34 + 98220515962912768/81*t^33 +
2084344786845696*t^32 + 84555054727561216/27*t^31 +
343618851587489792/81*t^30 + 145212682478288896/27*t^29 +
199732493916569600/27*t^28 + 597146982053576704/81*t^27 +
562602464528826368/27*t^26 + 183000203844911104/27*t^25 +
9718801131269783552/81*t^24 - 732000815379644416/27*t^23 +
9001639432461221888/27*t^22 - 38217406851428909056/81*t^21 +
51131518442641817600/27*t^20 - 148697786857767829504/27*t^19 +
1407462816102358188032/81*t^18 - 1385350016656362962944/27*t^17 +
136599619950719533056*t^16 - 25747918936581804654592/81*t^15 +
18275175765289698590720/27*t^14 - 35006558987024204824576/27*t^13 +
20296043204008304705536/9*t^12 - 31845455301213182492672/9*t^11 +
133796356023612520529920/27*t^10 - 497046224597870348075008/81*t^9 +
60027845701107370164224/9*t^8 - 169323113227380292845568/27*t^7 +
408561892606574929641472/81*t^6 - 91863117466361579700224/27*t^5 +
50521952295519923470336/27*t^4 - 64932545894857062744064/81*t^3 +
6787800588574861033472/27*t^2 - 1391981580026927644672/27*t +
437994204060197060608/81)/(t^48 - 16/3*t^47 - 96*t^46 + 8384/27*t^45 +
360256/81*t^44 + 13568/81*t^43 - 10435072/81*t^42 - 32914432/81*t^41 +
150462976/81*t^40 + 1009168384/81*t^39 + 624680960/81*t^38 -
12445171712/81*t^37 - 15975006208/27*t^36 + 2605907968/81*t^35 +
399232860160/81*t^34 + 1554796773376/81*t^33 + 1578545119232/81*t^32 -
5660949348352/81*t^31 - 30820177805312/81*t^30 -
99820433833984/81*t^29 - 1061687394304/81*t^28 - 7972499488768/9*t^27
+ 604673211891712/27*t^26 + 207366255542272/27*t^25 +
3665872497410048/27*t^24 - 829465022169088/27*t^23 +
9674771390267392/27*t^22 + 510239967281152/9*t^21 -
271791972941824/81*t^20 + 102216124245999616/81*t^19 -
126239448290557952/81*t^18 + 92748994123399168/81*t^17 +
103451532933988352/81*t^16 - 407580645359878144/81*t^15 +
418625995575132160/81*t^14 - 10929970213814272/81*t^13 -
268016129752956928/27*t^12 + 835181335877255168/81*t^11 +
167686518352117760/81*t^10 - 1083586301359292416/81*t^9 +
646233561178832896/81*t^8 + 565465636025663488/81*t^7 -
717092687542484992/81*t^6 - 3729543441416192/81*t^5 +
396105660976070656/81*t^4 - 36873221949095936/27*t^3 -
1688849860263936*t^2 + 1125899906842624/3*t + 281474976710656)>, <205,
(-4*t^48 + 720*t^44 - 44808*t^40 + 1057680*t^36 - 6090684*t^32 +
15714720*t^28 - 21291632*t^24 + 15714720*t^20 - 6090684*t^16 +
1057680*t^12 - 44808*t^8 + 720*t^4 - 4)/(t^44 + 6*t^40 + 13*t^36 +
8*t^32 - 14*t^28 - 28*t^24 - 14*t^20 + 8*t^16 + 13*t^12 + 6*t^8 +
t^4)>, <207, (-1/4*t^48 - 24*t^47 - 384*t^46 + 33856*t^45 +
1520448*t^44 + 7890432*t^43 - 879763456*t^42 - 24722903040*t^41 -
182706702336*t^40 + 4373903147008*t^39 + 145113654755328*t^38 +
2143959031480320*t^37 + 19625725978214400*t^36 +
110810322979258368*t^35 + 248374802497142784*t^34 -
1767275892473593856*t^33 - 20719039110482755584*t^32 -
96728221313591672832*t^31 - 150385089013681750016*t^30 +
960613058197774663680*t^29 + 7483241880019355566080*t^28 +
22713167653110632415232*t^27 + 10250232346133598830592*t^26 -
200181316734135878811648*t^25 - 879374520083920864673792*t^24 -
1601450533873087030493184*t^23 + 656014870152550325157888*t^22 +
11629141838392643796598784*t^21 + 30651358740559280398663680*t^20 +
31477368691024680179466240*t^19 - 39422548774402588676194304*t^18 -
202853782784241403862974464*t^17 - 347607794469017054707974144*t^16 -
237199755036978067347079168*t^15 + 266690413468921847680598016*t^14 +
951853426510223953788665856*t^13 + 1348669619787015303620198400*t^12 +
1178653942293991535054684160*t^11 + 638216603010220689155162112*t^10 +
153893035804842723331014656*t^9 - 51427364784906358191292416*t^8 -
55671028459230451382353920*t^7 - 15848409490463119013576704*t^6 +
1137131091679751599816704*t^5 + 1752957195836471270965248*t^4 +
312266483679755289755648*t^3 - 28334198897217871282176*t^2 -
14167099448608935641088*t - 1180591620717411303424)/(t^46 + 92*t^45 +
4148*t^44 + 122144*t^43 + 2640576*t^42 + 44656640*t^41 +
614579712*t^40 + 7067844608*t^39 + 69195886592*t^38 +
584389443584*t^37 + 4297081896960*t^36 + 27674980777984*t^35 +
156558882504704*t^34 + 777383359545344*t^33 + 3369004735397888*t^32 +
12563428551950336*t^31 + 39009211358642176*t^30 +
92240558445035520*t^29 + 109062204002664448*t^28 -
363777565619388416*t^27 - 3156028403170148352*t^26 -
13685448882480218112*t^25 - 43832546488905367552*t^24 -
109483591059841744896*t^23 - 201985817802889494528*t^22 -
186254113597126868992*t^21 + 446718787594913579008*t^20 +
3022538619126923919360*t^19 + 10226030702399894585344*t^18 +
26347419314579751043072*t^17 + 56522520150793212919808*t^16 +
104338628303183184658432*t^15 + 168103820064002561540096*t^14 +
237726274717739833622528*t^13 + 295293219450829469122560*t^12 +
321271494185077379694592*t^11 + 304326727608693652717568*t^10 +
248677674553916948217856*t^9 + 172988810122041671811072*t^8 +
100557813631809193246720*t^7 + 47568388358573899382784*t^6 +
17602805532337339629568*t^5 + 4782318401109201256448*t^4 +
848550227390639374336*t^3 + 73786976294838206464*t^2)>, <208,
(256*t^48 - 768*t^40 + 1536*t^32 - 1792*t^24 + 1536*t^16 - 768*t^8 +
256)/(t^32 - 2*t^24 + t^16)>, <209, (256*t^48 + 768*t^40 + 1536*t^32 +
1792*t^24 + 1536*t^16 + 768*t^8 + 256)/(t^32 + 2*t^24 + t^16)>, <210,
(t^48 + 48*t^40 + 1536*t^32 + 28672*t^24 + 393216*t^16 + 3145728*t^8 +
16777216)/(t^32 + 32*t^24 + 256*t^16)>, <211, (1/2*t^48 + 720*t^47 +
358512*t^46 + 67752000*t^45 + 3144241824*t^44 + 68431392000*t^43 +
848234961152*t^42 + 6547282191360*t^41 + 32628889518336*t^40 +
100882727055360*t^39 + 141061071360000*t^38 - 296892514713600*t^37 -
2301932208119808*t^36 - 6760806901678080*t^35 - 9692203365236736*t^34
+ 6621434000179200*t^33 + 67596846666055680*t^32 +
166310378251223040*t^31 + 214788884282736640*t^30 -
81128372054261760*t^29 - 1146989381186224128*t^28 -
1844274706513920000*t^27 - 1624814538580819968*t^26 -
1595614543101296640*t^25 + 16956476338758221824*t^24 +
6382458172405186560*t^23 - 25997032617293119488*t^22 +
118033581216890880000*t^21 - 293629281583673376768*t^20 +
83075452983564042240*t^19 + 879775270022089277440*t^18 -
2724829237268038287360*t^17 + 4430026943106625044480*t^16 -
1735769194542976204800*t^15 - 10163011835906475687936*t^14 +
28356879430935977656320*t^13 - 38620013872982972694528*t^12 +
19924119392532981350400*t^11 + 37865793014370140160000*t^10 -
108322003358516395376640*t^9 + 140140013386050312339456*t^8 -
112481451558297655050240*t^7 + 58290262679546727759872*t^6 -
18810277802224386048000*t^5 + 3457130446027619303424*t^4 -
297976447220318208000*t^3 + 6307009803155668992*t^2 -
50665495807918080*t + 140737488355328)/(t^47 - 48*t^46 + 916*t^45 -
7552*t^44 - 4080*t^43 + 604928*t^42 - 4292800*t^41 - 1216512*t^40 +
168931072*t^39 - 752128000*t^38 - 1207997440*t^37 + 19944996864*t^36 -
42148532224*t^35 - 158236344320*t^34 + 839134396416*t^33 -
350018863104*t^32 - 4874311172096*t^31 + 8519724040192*t^30 +
5636914413568*t^29 - 25109234450432*t^28 + 60715345903616*t^27 -
92765958438912*t^26 - 409268633206784*t^25 + 819491034365952*t^24 +
1637074532827136*t^23 - 1484255335022592*t^22 - 3885782137831424*t^21
- 6427964019310592*t^20 - 5772200359493632*t^19 +
34896789668626432*t^18 + 79860714243620864*t^17 -
22938836212383744*t^16 - 219974047214075904*t^15 -
165922832981688320*t^14 + 176783757301252096*t^13 +
334621520506650624*t^12 + 81067335913308160*t^11 -
201897822650368000*t^10 - 181388357379555328*t^9 -
5224879255191552*t^8 + 73749742433075200*t^7 + 41570335622955008*t^6 +
1121501860331520*t^5 - 8303511812964352*t^4 - 4028610604171264*t^3 -
844424930131968*t^2 - 70368744177664*t)>, <212, (-16*t^48 + 720*t^46 -
11205*t^44 + 264915/4*t^42 - 12405441/128*t^40 + 18096975/256*t^38 -
134396145/4096*t^36 + 182388465/16384*t^34 - 3121334511/1048576*t^32 +
341726265/524288*t^30 - 1006800165/8388608*t^28 +
631378215/33554432*t^26 - 2722943687/1073741824*t^24 +
631378215/2147483648*t^22 - 1006800165/34359738368*t^20 +
341726265/137438953472*t^18 - 3121334511/17592186044416*t^16 +
182388465/17592186044416*t^14 - 134396145/281474976710656*t^12 +
18096975/1125899906842624*t^10 - 12405441/36028797018963968*t^8 +
264915/72057594037927936*t^6 - 11205/1152921504606846976*t^4 +
45/4611686018427387904*t^2 - 1/295147905179352825856)/(t^46 + 3/2*t^44
+ 63/64*t^42 + 23/64*t^40 + 299/4096*t^38 + 39/8192*t^36 -
427/262144*t^34 - 35/65536*t^32 - 579/8388608*t^30 - 21/16777216*t^28
+ 611/536870912*t^26 + 117/536870912*t^24 + 611/34359738368*t^22 -
21/68719476736*t^20 - 579/2199023255552*t^18 - 35/1099511627776*t^16 -
427/281474976710656*t^14 + 39/562949953421312*t^12 +
299/18014398509481984*t^10 + 23/18014398509481984*t^8 +
63/1152921504606846976*t^6 + 3/2305843009213693952*t^4 +
1/73786976294838206464*t^2)>, <213, (t^48 + 24*t^46 + 228*t^44 +
1064*t^42 + 2322*t^40 + 840*t^38 - 5676*t^36 - 8712*t^34 + 1455*t^32 +
16240*t^30 + 14664*t^28 - 9456*t^26 - 21892*t^24 - 9456*t^22 +
14664*t^20 + 16240*t^18 + 1455*t^16 - 8712*t^14 - 5676*t^12 + 840*t^10
+ 2322*t^8 + 1064*t^6 + 228*t^4 + 24*t^2 + 1)/(t^32 + 8*t^30 + 12*t^28
- 8*t^26 - 26*t^24 - 8*t^22 + 12*t^20 + 8*t^18 + t^16)>, <214,
(1728*t^48 + 82944*t^47 + 2101248*t^46 + 36882432*t^45 +
496677888*t^44 + 5401718784*t^43 + 48880599040*t^42 +
374844973056*t^41 + 2463974184960*t^40 + 13980094152704*t^39 +
68734672109568*t^38 + 293400532451328*t^37 + 1088356500783104*t^36 +
3512936831975424*t^35 + 9900677217386496*t^34 + 24561181098901504*t^33
+ 54458357736357888*t^32 + 110498240078020608*t^31 +
210861290223042560*t^30 + 385828625763532800*t^29 +
678032230536708096*t^28 + 1131897098608836608*t^27 +
1781422056680718336*t^26 + 2658835509899427840*t^25 +
3818404670952439808*t^24 + 5317671019798855680*t^23 +
7125688226722873344*t^22 + 9055176788870692864*t^21 +
10848515688587329536*t^20 + 12346516024433049600*t^19 +
13495122574274723840*t^18 + 14143774729986637824*t^17 +
13941339580507619328*t^16 + 12575324722637570048*t^15 +
10138293470603771904*t^14 + 7194494631885668352*t^13 +
4457908227207593984*t^12 + 2403537161841278976*t^11 +
1126148867843162112*t^10 + 458099725195804672*t^9 +
161479012185538560*t^8 + 49131680308396032*t^7 + 12813755754741760*t^6
+ 2832056337825792*t^5 + 520804513087488*t^4 + 77348066033664*t^3 +
8813272891392*t^2 + 695784701952*t + 28991029248)/(t^48 + 48*t^47 +
1024*t^46 + 12512*t^45 + 91056*t^44 + 314688*t^43 - 880640*t^42 -
17552256*t^41 - 101767136*t^40 - 254255872*t^39 + 534986752*t^38 +
7699157504*t^37 + 33937231616*t^36 + 70155340800*t^35 -
71809302528*t^34 - 1139521587200*t^33 - 4650240250112*t^32 -
10913849548800*t^31 - 10381067550720*t^30 + 36062748262400*t^29 +
223002849140736*t^28 + 707251574964224*t^27 + 1653912291508224*t^26 +
3101952067829760*t^25 + 4806986446258176*t^24 + 6203904135659520*t^23
+ 6615649166032896*t^22 + 5658012599713792*t^21 +
3568045586251776*t^20 + 1154007944396800*t^19 - 664388323246080*t^18 -
1396972742246400*t^17 - 1190461504028672*t^16 - 583435052646400*t^15 -
73532725788672*t^14 + 143678137958400*t^13 + 139006900699136*t^12 +
63071498272768*t^11 + 8765222944768*t^10 - 8331456413696*t^9 -
6669411024896*t^8 - 2300609298432*t^7 - 230854492160*t^6 +
164987142144*t^5 + 95479136256*t^4 + 26239565824*t^3 + 4294967296*t^2
+ 402653184*t + 16777216)>, <215, (t^48 - 48*t^40 + 1536*t^32 -
28672*t^24 + 393216*t^16 - 3145728*t^8 + 16777216)/(t^32 - 32*t^24 +
256*t^16)>, <216, (1728*t^48 - 50112*t^46 + 583452*t^44 - 3529457*t^42
+ 390356571/32*t^40 - 1637898693/64*t^38 + 34767617387/1024*t^36 -
120539684859/4096*t^34 + 4511025549621/262144*t^32 -
913794639123/131072*t^30 + 4183447451367/2097152*t^28 -
3423154237197/8388608*t^26 + 16125200248573/268435456*t^24 -
3423154237197/536870912*t^22 + 4183447451367/8589934592*t^20 -
913794639123/34359738368*t^18 + 4511025549621/4398046511104*t^16 -
120539684859/4398046511104*t^14 + 34767617387/70368744177664*t^12 -
1637898693/281474976710656*t^10 + 390356571/9007199254740992*t^8 -
3529457/18014398509481984*t^6 + 145863/288230376151711744*t^4 -
783/1152921504606846976*t^2 + 27/73786976294838206464)/(t^48 - 5*t^46
+ 37/16*t^44 + 581/64*t^42 + 12097/2048*t^40 + 5289/4096*t^38 -
16943/65536*t^36 - 59561/262144*t^34 - 936721/16777216*t^32 -
27697/8388608*t^30 + 233605/134217728*t^28 + 316689/536870912*t^26 +
1652807/17179869184*t^24 + 316689/34359738368*t^22 +
233605/549755813888*t^20 - 27697/2199023255552*t^18 -
936721/281474976710656*t^16 - 59561/281474976710656*t^14 -
16943/4503599627370496*t^12 + 5289/18014398509481984*t^10 +
12097/576460752303423488*t^8 + 581/1152921504606846976*t^6 +
37/18446744073709551616*t^4 - 5/73786976294838206464*t^2 +
1/4722366482869645213696)>, <217, (-1/2*t^48 + 744*t^46 - 390696*t^44
+ 82703456*t^42 - 5836712976*t^40 + 208938610560*t^38 -
4467109918848*t^36 + 61759756288512*t^34 - 577063358855040*t^32 +
3742383726039040*t^30 - 17139292581285888*t^28 +
56086505494265856*t^26 - 132075245931126784*t^24 +
224346021977063424*t^22 - 274228681300574208*t^20 +
239512558466498560*t^18 - 147728219866890240*t^16 +
63241990439436288*t^14 - 18297282227601408*t^12 +
3423250195415040*t^10 - 382514821595136*t^8 + 21680214769664*t^6 -
409674448896*t^4 + 3120562176*t^2 - 8388608)/(t^46 + 4*t^44 - 164*t^42
- 1696*t^40 + 48*t^38 + 101184*t^36 + 822592*t^34 + 3344384*t^32 +
7002624*t^30 + 739328*t^28 - 41134080*t^26 - 125026304*t^24 -
164536320*t^22 + 11829248*t^20 + 448167936*t^18 + 856162304*t^16 +
842334208*t^14 + 414449664*t^12 + 786432*t^10 - 111149056*t^8 -
42991616*t^6 + 4194304*t^4 + 4194304*t^2)>, <218, (1728*t^48 +
50112*t^46 + 583452*t^44 + 3529457*t^42 + 390356571/32*t^40 +
1637898693/64*t^38 + 34767617387/1024*t^36 + 120539684859/4096*t^34 +
4511025549621/262144*t^32 + 913794639123/131072*t^30 +
4183447451367/2097152*t^28 + 3423154237197/8388608*t^26 +
16125200248573/268435456*t^24 + 3423154237197/536870912*t^22 +
4183447451367/8589934592*t^20 + 913794639123/34359738368*t^18 +
4511025549621/4398046511104*t^16 + 120539684859/4398046511104*t^14 +
34767617387/70368744177664*t^12 + 1637898693/281474976710656*t^10 +
390356571/9007199254740992*t^8 + 3529457/18014398509481984*t^6 +
145863/288230376151711744*t^4 + 783/1152921504606846976*t^2 +
27/73786976294838206464)/(t^48 + 5*t^46 + 37/16*t^44 - 581/64*t^42 +
12097/2048*t^40 - 5289/4096*t^38 - 16943/65536*t^36 +
59561/262144*t^34 - 936721/16777216*t^32 + 27697/8388608*t^30 +
233605/134217728*t^28 - 316689/536870912*t^26 +
1652807/17179869184*t^24 - 316689/34359738368*t^22 +
233605/549755813888*t^20 + 27697/2199023255552*t^18 -
936721/281474976710656*t^16 + 59561/281474976710656*t^14 -
16943/4503599627370496*t^12 - 5289/18014398509481984*t^10 +
12097/576460752303423488*t^8 - 581/1152921504606846976*t^6 +
37/18446744073709551616*t^4 + 5/73786976294838206464*t^2 +
1/4722366482869645213696)>, <219, (28756228/3*t^48 + 73902016*t^47 +
281705888*t^46 + 2897439808/3*t^45 + 3169096816*t^44 + 8673109952*t^43
+ 66414953248/3*t^42 + 53332609216*t^41 + 116262573912*t^40 +
732850865344/3*t^39 + 477988726240*t^38 + 872443052352*t^37 +
1553725783728*t^36 + 2527336257088*t^35 + 3983785870880*t^34 +
17903726640064/3*t^33 + 8137719659412*t^32 + 11251785561472*t^31 +
40386845682368/3*t^30 + 16056927660928*t^29 + 18619840183520*t^28 +
46671368714368/3*t^27 + 22234737701312*t^26 + 6637697341312*t^25 +
70777801907312/3*t^24 - 6637697341312*t^23 + 22234737701312*t^22 -
46671368714368/3*t^21 + 18619840183520*t^20 - 16056927660928*t^19 +
40386845682368/3*t^18 - 11251785561472*t^17 + 8137719659412*t^16 -
17903726640064/3*t^15 + 3983785870880*t^14 - 2527336257088*t^13 +
1553725783728*t^12 - 872443052352*t^11 + 477988726240*t^10 -
732850865344/3*t^9 + 116262573912*t^8 - 53332609216*t^7 +
66414953248/3*t^6 - 8673109952*t^5 + 3169096816*t^4 - 2897439808/3*t^3
+ 281705888*t^2 - 73902016*t + 28756228/3)/(t^48 - 88/3*t^47 +
1144/3*t^46 - 8504/3*t^45 + 38044/3*t^44 - 89560/3*t^43 - 18424/3*t^42
+ 301912*t^41 - 1038782*t^40 + 4598488/3*t^39 + 1501864/3*t^38 -
7065240*t^37 + 15438436*t^36 - 53412328/3*t^35 + 8935240*t^34 +
22126520/3*t^33 - 62868019/3*t^32 + 62271376/3*t^31 - 11142352/3*t^30
- 42475312/3*t^29 + 42928184/3*t^28 - 16297072/3*t^27 - 1521680*t^26 +
37718096/3*t^25 - 21428524/3*t^24 - 37718096/3*t^23 - 1521680*t^22 +
16297072/3*t^21 + 42928184/3*t^20 + 42475312/3*t^19 - 11142352/3*t^18
- 62271376/3*t^17 - 62868019/3*t^16 - 22126520/3*t^15 + 8935240*t^14 +
53412328/3*t^13 + 15438436*t^12 + 7065240*t^11 + 1501864/3*t^10 -
4598488/3*t^9 - 1038782*t^8 - 301912*t^7 - 18424/3*t^6 + 89560/3*t^5 +
38044/3*t^4 + 8504/3*t^3 + 1144/3*t^2 + 88/3*t + 1)>, <220,
(98611128000/289*t^48 + 1954805529600/289*t^47 +
19029301885440/289*t^46 + 120904714202112/289*t^45 +
563444118796032/289*t^44 + 2053520127507456/289*t^43 +
6095380896138752/289*t^42 + 15149404360559616/289*t^41 +
32163231653465472/289*t^40 + 59213193863496704/289*t^39 +
95647517938957824/289*t^38 + 136843130578818048/289*t^37 +
174757877046582016/289*t^36 + 200489288314954752/289*t^35 +
207707352293736960/289*t^34 + 11475286897955840/17*t^33 +
166629379497487680/289*t^32 + 129669586982983680/289*t^31 +
92251755809809408/289*t^30 + 60030879032162304/289*t^29 +
36394937464892928/289*t^28 + 19900852857952256/289*t^27 +
11854325560433664/289*t^26 + 4104201001433088/289*t^25 +
6251534073019648/289*t^24 - 4104201001433088/289*t^23 +
11854325560433664/289*t^22 - 19900852857952256/289*t^21 +
36394937464892928/289*t^20 - 60030879032162304/289*t^19 +
92251755809809408/289*t^18 - 129669586982983680/289*t^17 +
166629379497487680/289*t^16 - 11475286897955840/17*t^15 +
207707352293736960/289*t^14 - 200489288314954752/289*t^13 +
174757877046582016/289*t^12 - 136843130578818048/289*t^11 +
95647517938957824/289*t^10 - 59213193863496704/289*t^9 +
32163231653465472/289*t^8 - 15149404360559616/289*t^7 +
6095380896138752/289*t^6 - 2053520127507456/289*t^5 +
563444118796032/289*t^4 - 120904714202112/289*t^3 +
19029301885440/289*t^2 - 1954805529600/289*t + 98611128000/289)/(t^48
- 432/17*t^47 + 79160/289*t^46 - 452976/289*t^45 + 1326292/289*t^44 -
863408/289*t^43 - 5551672/289*t^42 + 6944272/289*t^41 +
50943810/289*t^40 - 169921616/289*t^39 + 90106216/289*t^38 +
328131952/289*t^37 - 152324476/289*t^36 - 1351993552/289*t^35 +
2185549720/289*t^34 - 123736976/17*t^33 + 5622566863/289*t^32 -
6739796192/289*t^31 - 5847056208/289*t^30 + 12004746144/289*t^29 +
2202060968/289*t^28 + 1711281440/289*t^27 - 12390842544/289*t^26 -
16787141472/289*t^25 + 21334898588/289*t^24 + 16787141472/289*t^23 -
12390842544/289*t^22 - 1711281440/289*t^21 + 2202060968/289*t^20 -
12004746144/289*t^19 - 5847056208/289*t^18 + 6739796192/289*t^17 +
5622566863/289*t^16 + 123736976/17*t^15 + 2185549720/289*t^14 +
1351993552/289*t^13 - 152324476/289*t^12 - 328131952/289*t^11 +
90106216/289*t^10 + 169921616/289*t^9 + 50943810/289*t^8 -
6944272/289*t^7 - 5551672/289*t^6 + 863408/289*t^5 + 1326292/289*t^4 +
452976/289*t^3 + 79160/289*t^2 + 432/17*t + 1)>, <221, (-t^48 +
672*t^40 - 151296*t^32 + 11583488*t^24 - 38731776*t^16 + 44040192*t^8
- 16777216)/(t^40 + 64*t^32 + 1536*t^24 + 16384*t^16 + 65536*t^8)>,
<222, (-16*t^48 + 672*t^40 - 9456*t^32 + 45248*t^24 - 9456*t^16 +
672*t^8 - 16)/(t^40 + 4*t^32 + 6*t^24 + 4*t^16 + t^8)>, <223,
(16777216*t^48 + 50331648*t^46 + 59768832*t^44 + 34865152*t^42 +
9510912*t^40 + 430080*t^38 - 363264*t^36 - 69696*t^34 + 1455*t^32 +
2030*t^30 + 1833/8*t^28 - 591/32*t^26 - 5473/1024*t^24 - 591/2048*t^22
+ 1833/32768*t^20 + 1015/131072*t^18 + 1455/16777216*t^16 -
1089/16777216*t^14 - 1419/268435456*t^12 + 105/1073741824*t^10 +
1161/34359738368*t^8 + 133/68719476736*t^6 + 57/1099511627776*t^4 +
3/4398046511104*t^2 + 1/281474976710656)/(t^32 + t^30 + 3/16*t^28 -
1/64*t^26 - 13/2048*t^24 - 1/4096*t^22 + 3/65536*t^20 + 1/262144*t^18
+ 1/16777216*t^16)>, <224, (1728*t^48 + 4752*t^46 + 11943/4*t^44 -
61417/64*t^42 - 3222309/8192*t^40 + 25070307/65536*t^38 -
556005173/4194304*t^36 + 1898188701/67108864*t^34 -
70087185099/17179869184*t^32 + 14269181637/34359738368*t^30 -
65471314233/2199023255552*t^28 + 53495774043/35184372088832*t^26 -
251655132547/4503599627370496*t^24 +
53495774043/36028797018963968*t^22 -
65471314233/2305843009213693952*t^20 +
14269181637/36893488147419103232*t^18 -
70087185099/18889465931478580854784*t^16 +
1898188701/75557863725914323419136*t^14 -
556005173/4835703278458516698824704*t^12 +
25070307/77371252455336267181195264*t^10 -
3222309/9903520314283042199192993792*t^8 -
61417/79228162514264337593543950336*t^6 +
11943/5070602400912917605986812821504*t^4 +
297/81129638414606681695789005144064*t^2 +
27/20769187434139310514121985316880384)/(t^48 - 13/4*t^46 +
1029/256*t^44 - 9331/4096*t^42 + 276929/524288*t^40 -
54127/4194304*t^38 - 1397711/268435456*t^36 + 812463/4294967296*t^34 +
27177199/1099511627776*t^32 - 1370009/2199023255552*t^30 -
9730203/140737488355328*t^28 + 1257657/2251799813685248*t^26 +
29945799/288230376151711744*t^24 + 1257657/2305843009213693952*t^22 -
9730203/147573952589676412928*t^20 -
1370009/2361183241434822606848*t^18 +
27177199/1208925819614629174706176*t^16 +
812463/4835703278458516698824704*t^14 -
1397711/309485009821345068724781056*t^12 -
54127/4951760157141521099596496896*t^10 +
276929/633825300114114700748351602688*t^8 -
9331/5070602400912917605986812821504*t^6 +
1029/324518553658426726783156020576256*t^4 -
13/5192296858534827628530496329220096*t^2 +
1/1329227995784915872903807060280344576)>, <225, (16*t^48 + 720*t^46 +
11205*t^44 + 264915/4*t^42 + 12405441/128*t^40 + 18096975/256*t^38 +
134396145/4096*t^36 + 182388465/16384*t^34 + 3121334511/1048576*t^32 +
341726265/524288*t^30 + 1006800165/8388608*t^28 +
631378215/33554432*t^26 + 2722943687/1073741824*t^24 +
631378215/2147483648*t^22 + 1006800165/34359738368*t^20 +
341726265/137438953472*t^18 + 3121334511/17592186044416*t^16 +
182388465/17592186044416*t^14 + 134396145/281474976710656*t^12 +
18096975/1125899906842624*t^10 + 12405441/36028797018963968*t^8 +
264915/72057594037927936*t^6 + 11205/1152921504606846976*t^4 +
45/4611686018427387904*t^2 + 1/295147905179352825856)/(t^46 - 3/2*t^44
+ 63/64*t^42 - 23/64*t^40 + 299/4096*t^38 - 39/8192*t^36 -
427/262144*t^34 + 35/65536*t^32 - 579/8388608*t^30 + 21/16777216*t^28
+ 611/536870912*t^26 - 117/536870912*t^24 + 611/34359738368*t^22 +
21/68719476736*t^20 - 579/2199023255552*t^18 + 35/1099511627776*t^16 -
427/281474976710656*t^14 - 39/562949953421312*t^12 +
299/18014398509481984*t^10 - 23/18014398509481984*t^8 +
63/1152921504606846976*t^6 - 3/2305843009213693952*t^4 +
1/73786976294838206464*t^2)>, <226, (287496*t^48 + 7527168*t^47 +
83673216*t^46 + 468163584*t^45 + 843290496*t^44 - 5687350272*t^43 -
38264757760*t^42 - 13513187328*t^41 + 871050481920*t^40 +
4862111223808*t^39 + 10123744352256*t^38 - 16325717213184*t^37 -
157673884116992*t^36 - 266851153723392*t^35 + 1380037527232512*t^34 +
10583702274572288*t^33 + 35225743045195776*t^32 +
61687139063955456*t^31 + 4322106343751680*t^30 -
239102320603299840*t^29 - 55428289519484928*t^28 +
4635419581605216256*t^27 + 28025334119030587392*t^26 +
105595237336476549120*t^25 + 305800453004913934336*t^24 +
730936706693301534720*t^23 + 1491813748814593916928*t^22 +
2649331951816511848448*t^21 + 4140576535677634609152*t^20 +
5734808613393876910080*t^19 + 7068704286616185733120*t^18 +
7771718366065745461248*t^17 + 7628083260805520818176*t^16 +
6681712766276627070976*t^15 + 5215979254174709710848*t^14 +
3620187882343265992704*t^13 + 2226404076148372799488*t^12 +
1207821326489235750912*t^11 + 574670692576231686144*t^10 +
238052595149469384704*t^9 + 85062639886050263040*t^8 +
25911041453293830144*t^7 + 6626239613511925760*t^6 +
1394023581502930944*t^5 + 234615730178359296*t^4 +
30330153879994368*t^3 + 2824030654562304*t^2 + 168379897872384*t +
4823382491136)/(t^48 + 72*t^47 + 2408*t^46 + 49520*t^45 + 697264*t^44
+ 7056096*t^43 + 51954336*t^42 + 269091904*t^41 + 820559904*t^40 -
388558976*t^39 - 21314538624*t^38 - 138961653504*t^37 -
534024704256*t^36 - 1197199166976*t^35 - 70070051328*t^34 +
12715812568064*t^33 + 65558903189248*t^32 + 213825795182592*t^31 +
524195693498368*t^30 + 1010773929648128*t^29 + 1539444624809984*t^28 +
1795505037295616*t^27 + 1435630168719360*t^26 + 405174100328448*t^25 -
846370904621056*t^24 - 1572944668065792*t^23 - 1319520693190656*t^22 -
359030186704896*t^21 + 525838155382784*t^20 + 748624701292544*t^19 +
381923295494144*t^18 - 68895981961216*t^17 - 232583301496832*t^16 -
134581618999296*t^15 + 2357711077376*t^14 + 47868235743232*t^13 +
25175369187328*t^12 - 761578651648*t^11 - 6503199145984*t^10 -
2506830643200*t^9 + 287764905984*t^8 + 505354911744*t^7 +
115251085312*t^6 - 26893877248*t^5 - 17800626176*t^4 - 1979711488*t^3
+ 637534208*t^2 + 201326592*t + 16777216)>, <227, (-8*t^48 + 720*t^46
- 22410*t^44 + 264915*t^42 - 12405441/16*t^40 + 18096975/16*t^38 -
134396145/128*t^36 + 182388465/256*t^34 - 3121334511/8192*t^32 +
341726265/2048*t^30 - 1006800165/16384*t^28 + 631378215/32768*t^26 -
2722943687/524288*t^24 + 631378215/524288*t^22 -
1006800165/4194304*t^20 + 341726265/8388608*t^18 -
3121334511/536870912*t^16 + 182388465/268435456*t^14 -
134396145/2147483648*t^12 + 18096975/4294967296*t^10 -
12405441/68719476736*t^8 + 264915/68719476736*t^6 -
11205/549755813888*t^4 + 45/1099511627776*t^2 -
1/35184372088832)/(t^46 + 3*t^44 + 63/16*t^42 + 23/8*t^40 +
299/256*t^38 + 39/256*t^36 - 427/4096*t^34 - 35/512*t^32 -
579/32768*t^30 - 21/32768*t^28 + 611/524288*t^26 + 117/262144*t^24 +
611/8388608*t^22 - 21/8388608*t^20 - 579/134217728*t^18 -
35/33554432*t^16 - 427/4294967296*t^14 + 39/4294967296*t^12 +
299/68719476736*t^10 + 23/34359738368*t^8 + 63/1099511627776*t^6 +
3/1099511627776*t^4 + 1/17592186044416*t^2)>, <228, (2048/3*t^48 +
98304*t^47 + 6619136*t^46 + 277610496*t^45 + 8143241216*t^44 +
177588928512*t^43 + 8946018418688/3*t^42 + 39280579706880*t^41 +
407718208733184*t^40 + 3287298846228480*t^39 + 19491843368550400*t^38
+ 68592565821112320*t^37 - 82995613020454912*t^36 -
3258181303350067200*t^35 - 26164394155334172672*t^34 -
123189167396021600256*t^33 - 288827323932376825856*t^32 +
684671096803672719360*t^31 + 31435412545303748804608/3*t^30 +
55177834661569395425280*t^29 + 168245063855472247832576*t^28 +
182916771235392236027904*t^27 - 1139976168827486361092096*t^26 -
8022124604092084056489984*t^25 - 85361921211841519737110528/3*t^24 -
64176996832736672451919872*t^23 - 72958474804959127109894144*t^22 +
93653386872520824846286848*t^21 + 689131781552014327122231296*t^20 +
1808067286190305949295575040*t^19 +
8240604786276105926635159552/3*t^18 +
1435859360004015850751262720*t^17 - 4845718400315455400780496896*t^16
- 16534170162105695425284538368*t^15 -
28093804304203453894364233728*t^14 -
27987564284654387726804582400*t^13 - 5703415098149210017320927232*t^12
+ 37709161849651814495063900160*t^11 +
85726033722038725557983641600*t^10 +
115661545772890968422728335360*t^9 +
114762473307683348149073608704*t^8 + 88452002065400917496420106240*t^7
+ 161157140867411482677704916992/3*t^6 +
25593261832696603285823422464*t^5 + 9388517915127210170768162816*t^4 +
2560504885943784592027680768*t^3 + 488406031124310186581295104*t^2 +
58028439341502200385896448*t + 9671406556917033397649408/3)/(t^48 +
320/3*t^47 + 5568*t^46 + 568832/3*t^45 + 14210816/3*t^44 +
92549120*t^43 + 4415565824/3*t^42 + 58733461504/3*t^41 +
222059700224*t^40 + 6535186743296/3*t^39 + 56032159858688/3*t^38 +
423097859571712/3*t^37 + 942785294237696*t^36 +
16787827572342784/3*t^35 + 88516659949600768/3*t^34 +
137825576093220864*t^33 + 566068561101979648*t^32 +
6032094497783939072/3*t^31 + 17854479321373081600/3*t^30 +
39724631409557504000/3*t^29 + 38026363828474740736/3*t^28 -
201982184191837601792/3*t^27 - 506670606740984168448*t^26 -
6384296640410439122944/3*t^25 - 6761453125853761241088*t^24 -
51074373123283512983552/3*t^23 - 32426918831422986780672*t^22 -
103414878306220852117504/3*t^21 + 155755986241432538054656/3*t^20 +
1301696722028380291072000/3*t^19 + 4680444627222025102950400/3*t^18 +
12650219040216583392722944/3*t^17 + 9497054520417110582099968*t^16 +
18498635683523220568276992*t^15 + 95044039908672076704120832/3*t^14 +
144206340788198662763184128/3*t^13 + 64787712094410265126240256*t^12 +
232600508143117261661536256/3*t^11 +
246432045176124356062871552/3*t^10 + 229936442046128678892470272/3*t^9
+ 62504248948925652946386944*t^8 + 132255997671796885780692992/3*t^7 +
79543762398385188494114816/3*t^6 + 13337733835054954200432640*t^5 +
16383955364411054716092416/3*t^4 + 5246549162468175832416256/3*t^3 +
410845884009659133591552*t^2 + 188894659314785808547840/3*t +
4722366482869645213696)>, <229, (-4*t^48 - 48*t^46 + 492*t^44 +
6668*t^42 - 36801/2*t^40 - 313665*t^38 + 396879/4*t^36 +
21425409/4*t^34 + 315735825/64*t^32 - 164014135/8*t^30 -
154003173/32*t^28 + 1226937111/32*t^26 - 8134644167/256*t^24 +
1226937111/128*t^22 - 154003173/512*t^20 - 164014135/512*t^18 +
315735825/16384*t^16 + 21425409/4096*t^14 + 396879/16384*t^12 -
313665/16384*t^10 - 36801/131072*t^8 + 1667/65536*t^6 + 123/262144*t^4
- 3/262144*t^2 - 1/4194304)/(t^44 + 10*t^42 + 87/2*t^40 + 221/2*t^38 +
2925/16*t^36 + 409/2*t^34 + 1205/8*t^32 + 477/8*t^30 - 1079/128*t^28 -
1885/64*t^26 - 5291/256*t^24 - 1885/256*t^22 - 1079/2048*t^20 +
477/512*t^18 + 1205/2048*t^16 + 409/2048*t^14 + 2925/65536*t^12 +
221/32768*t^10 + 87/131072*t^8 + 5/131072*t^6 + 1/1048576*t^4)>, <230,
(256*t^48 - 3072*t^46 + 14592*t^44 - 34048*t^42 + 37152*t^40 -
6720*t^38 - 22704*t^36 + 17424*t^34 + 1455*t^32 - 8120*t^30 +
3666*t^28 + 1182*t^26 - 5473/4*t^24 + 591/2*t^22 + 1833/8*t^20 -
1015/8*t^18 + 1455/256*t^16 + 1089/64*t^14 - 1419/256*t^12 -
105/256*t^10 + 1161/2048*t^8 - 133/1024*t^6 + 57/4096*t^4 - 3/4096*t^2
+ 1/65536)/(t^32 - 4*t^30 + 3*t^28 + t^26 - 13/8*t^24 + 1/4*t^22 +
3/16*t^20 - 1/16*t^18 + 1/256*t^16)>, <231, (1728*t^48 + 82944*t^47 +
1548288*t^46 + 11446272*t^45 - 41352192*t^44 - 1483739136*t^43 -
10697605120*t^42 - 3652952064*t^41 + 460075161600*t^40 +
3246081163264*t^39 + 6154897195008*t^38 - 47136833175552*t^37 -
389618214158336*t^36 - 1171068561063936*t^35 + 27707413364736*t^34 +
13766742092939264*t^33 + 53681798635143168*t^32 +
79928267521916928*t^31 - 115541902308147200*t^30 -
872839532799590400*t^29 - 2042310222562197504*t^28 -
1857858114412347392*t^27 + 3039113154580709376*t^26 +
14263446339248455680*t^25 + 26635840237761200128*t^24 +
28526892678496911360*t^23 + 12156452618322837504*t^22 -
14862864915298779136*t^21 - 32676963560995160064*t^20 -
27930865049586892800*t^19 - 7394681747721420800*t^18 +
10230818242805366784*t^17 + 13742540450596651008*t^16 +
7048571951584903168*t^15 + 28372391285489664*t^14 -
2398348413058940928*t^13 - 1595876205192544256*t^12 -
386144937374121984*t^11 + 100841835643011072*t^10 +
106367587557834752*t^9 + 30151485790617600*t^8 - 478799732932608*t^7 -
2804312996577280*t^6 - 777906624135168*t^5 - 43360916078592*t^4 +
24004572217344*t^3 + 6493990551552*t^2 + 695784701952*t +
28991029248)/(t^48 + 48*t^47 + 1088*t^46 + 15456*t^45 + 153328*t^44 +
1111616*t^43 + 5888768*t^42 + 20954240*t^41 + 26793504*t^40 -
248824576*t^39 - 2312692736*t^38 - 11607459328*t^37 - 40531873024*t^36
- 94111745024*t^35 - 62327926784*t^34 + 701978540032*t^33 +
4593616645888*t^32 + 18475556003840*t^31 + 58484604960768*t^30 +
156318117642240*t^29 + 363790825545728*t^28 + 749266964217856*t^27 +
1378652968058880*t^26 + 2278801771659264*t^25 + 3393647108538368*t^24
+ 4557603543318528*t^23 + 5514611872235520*t^22 +
5994135713742848*t^21 + 5820653208731648*t^20 + 5002179764551680*t^19
+ 3743014717489152*t^18 + 2364871168491520*t^17 +
1175965861347328*t^16 + 359413012496384*t^15 - 63823797026816*t^14 -
192740853809152*t^13 - 166018551906304*t^12 - 95088306814976*t^11 -
37891157786624*t^10 - 8153483706368*t^9 + 1755939078144*t^8 +
2746514145280*t^7 + 1543705198592*t^6 + 582806929408*t^5 +
160776060928*t^4 + 32413581312*t^3 + 4563402752*t^2 + 402653184*t +
16777216)>, <232, (-1481544000/83521*t^48 - 61382707200/83521*t^47 -
1195562833920/83521*t^46 - 15089082015744/83521*t^45 -
140621913427968/83521*t^44 - 1026816783187968/83521*t^43 -
6101318034685952/83521*t^42 - 30333922288730112/83521*t^41 -
128507963368439808/83521*t^40 - 473041394843779072/83521*t^39 -
1529606774431678464/83521*t^38 - 4388365077017788416/83521*t^37 -
11228242040637620224/83521*t^36 - 25618014970771931136/83521*t^35 -
52917696544175554560/83521*t^34 - 5817924155688878080/4913*t^33 -
169589457952985579520/83521*t^32 - 267006559262154424320/83521*t^31 -
403481828367467020288/83521*t^30 - 577250702549506326528/83521*t^29 -
566860353404909125632/83521*t^28 - 727386411321228525568/83521*t^27 +
480905330990090551296/83521*t^26 + 283592204392819851264/83521*t^25 +
6577186323848178434048/83521*t^24 - 1134368817571279405056/83521*t^23
+ 7694485295841448820736/83521*t^22 +
46552730324558625636352/83521*t^21 -
145116250471656736161792/83521*t^20 +
591104719410694478364672/83521*t^19 -
1652661568993144915099648/83521*t^18 +
4374635466951138088058880/83521*t^17 -
11114214716406862939422720/83521*t^16 +
1525133909868905255403520/4913*t^15 -
55488226571505426298306560/83521*t^14 +
107449742663968593851449344/83521*t^13 -
188378642016058132224016384/83521*t^12 +
294498195135936288390119424/83521*t^11 -
410600691995256749329219584/83521*t^10 +
507924330127063535822307328/83521*t^9 -
551937499943014973904519168/83521*t^8 +
521132816758005201641668608/83521*t^7 -
419279382743538519530012672/83521*t^6 +
282249248177679691564449792/83521*t^5 -
154615428934160847804039168/83521*t^4 +
66362484515105010798821376/83521*t^3 -
21032563802109867951390720/83521*t^2 +
4319424019889254091980800/83521*t -
417017562895812132864000/83521)/(t^48 - 96/17*t^47 - 25376/289*t^46 +
1095808/4913*t^45 + 409007424/83521*t^44 - 8332800/83521*t^43 -
690941440/4913*t^42 - 29139548160/83521*t^41 + 163814408704/83521*t^40
+ 924817842176/83521*t^39 + 693122473984/83521*t^38 -
11660372770816/83521*t^37 - 47533578502144/83521*t^36 -
26816031621120/83521*t^35 + 474786608578560/83521*t^34 +
90455928209408/4913*t^33 + 2213499320074240/83521*t^32 -
7948603024736256/83521*t^31 - 33031736984076288/83521*t^30 -
89128697117802496/83521*t^29 - 6530648726568960/83521*t^28 -
3611196624207872/4913*t^27 + 1802160242927075328/83521*t^26 +
18556615850459136/4913*t^25 + 11773728664275910656/83521*t^24 -
74226463401836544/4913*t^23 + 28834563886833205248/83521*t^22 +
231116583949303808/4913*t^21 - 1671846074001653760/83521*t^20 +
91267785848629755904/83521*t^19 - 135297994686776475648/83521*t^18 +
130229911957278818304/83521*t^17 + 145063891440385392640/83521*t^16 -
23712478844527050752/4913*t^15 + 497849842876872130560/83521*t^14 +
112474588692590100480/83521*t^13 - 797481113783426351104/83521*t^12 +
782514370465994113024/83521*t^11 + 186058647367743176704/83521*t^10 -
993015596725802369024/83521*t^9 + 703577527997257744384/83521*t^8 +
500613625469667901440/83521*t^7 - 47481134212018339840/4913*t^6 +
2290502622982963200/83521*t^5 + 449708418534708609024/83521*t^4 -
4819414551239852032/4913*t^3 - 446419313063100416/289*t^2 +
6755399441055744/17*t + 281474976710656)>, <233, (8*t^48 + 744*t^46 +
48837/2*t^44 + 2584483/8*t^42 + 364794561/256*t^40 +
1632332895/512*t^38 + 34899296241/8192*t^36 + 120624524001/32768*t^34
+ 4508307491055/2097152*t^32 + 913667901865/1048576*t^30 +
4184397602853/16777216*t^28 + 3423248626359/67108864*t^26 +
16122466544327/2147483648*t^24 + 3423248626359/4294967296*t^22 +
4184397602853/68719476736*t^20 + 913667901865/274877906944*t^18 +
4508307491055/35184372088832*t^16 + 120624524001/35184372088832*t^14 +
34899296241/562949953421312*t^12 + 1632332895/2251799813685248*t^10 +
364794561/72057594037927936*t^8 + 2584483/144115188075855872*t^6 +
48837/2305843009213693952*t^4 + 93/9223372036854775808*t^2 +
1/590295810358705651712)/(t^46 - 1/4*t^44 - 41/64*t^42 + 53/128*t^40 +
3/4096*t^38 - 1581/16384*t^36 + 12853/262144*t^34 - 1633/131072*t^32 +
13677/8388608*t^30 - 361/33554432*t^28 - 20085/536870912*t^26 +
7631/1073741824*t^24 - 20085/34359738368*t^22 - 361/137438953472*t^20
+ 13677/2199023255552*t^18 - 1633/2199023255552*t^16 +
12853/281474976710656*t^14 - 1581/1125899906842624*t^12 +
3/18014398509481984*t^10 + 53/36028797018963968*t^8 -
41/1152921504606846976*t^6 - 1/4611686018427387904*t^4 +
1/73786976294838206464*t^2)>, <234, (3065617154/9*t^48 +
40754362304/3*t^47 + 793377123136/3*t^46 + 30221912338688/9*t^45 +
93882494288768/3*t^44 + 684506046757888/3*t^43 +
12192930612601856/9*t^42 + 20201175942311936/3*t^41 +
28586228335567872*t^40 + 947220235131600896/9*t^39 +
1020199464513290240/3*t^38 + 973368291887611904*t^37 +
2486462218810195968*t^36 + 17109299548974743552/3*t^35 +
35420529001985474560/3*t^34 + 199457653690496712704/9*t^33 +
37877272358136184832*t^32 + 177487709287605075968/3*t^31 +
761857055777678688256/9*t^30 + 333964348407569973248/3*t^29 +
397921109630160732160/3*t^28 + 1316953433661287432192/9*t^27 +
400100454589240705024/3*t^26 + 328736393512104230912/3*t^25 +
1075179884621936459776/9*t^24 - 1314945574048416923648/3*t^23 +
6401607273427851280384/3*t^22 - 84285019754322395660288/9*t^21 +
101867804065321147432960/3*t^20 - 341979492769351652605952/3*t^19 +
3120566500465371907096576/9*t^18 - 2907958628968121564659712/3*t^17 +
2482324921262813009149952*t^16 - 52286627169041570255077376/9*t^15 +
37141116618785920972226560/3*t^14 - 71761603535462962779127808/3*t^13
+ 41715913720817920757465088*t^12 - 65321640322198050550317056*t^11 +
273857708467580883634749440/3*t^10 -
1017069982999914026111074304/9*t^9 + 122776915817252523828314112*t^8 -
347053560051886991001780224/3*t^7 + 837891811576355471822422016/9*t^6
- 188155589423320050561974272/3*t^5 +
103224894115114327073619968/3*t^4 - 132917376120057687600791552/9*t^3
+ 13957237953592053597208576/3*t^2 - 2867833295094009200377856/3*t +
862894517025937528193024/9)/(t^48 - 152/3*t^47 + 9760/9*t^46 -
109472/9*t^45 + 590144/9*t^44 + 168832/9*t^43 - 7345664/3*t^42 +
114873856/9*t^41 - 34388480/3*t^40 - 674084864/9*t^39 -
1213112320/9*t^38 + 12858400768/9*t^37 + 22557933568/3*t^36 -
575279693824/9*t^35 + 942628536320/9*t^34 + 455228391424/9*t^33 +
7143393001472/9*t^32 - 39637260173312/9*t^31 + 1902993473536/9*t^30 +
70765445644288/3*t^29 - 242265909886976/9*t^28 +
265481105702912/9*t^27 - 11892026245120*t^26 - 1882769982488576/3*t^25
+ 4819629696352256/9*t^24 + 7531079929954304/3*t^23 -
190272419921920*t^22 - 16990790764986368/9*t^21 -
62020072931065856/9*t^20 - 72463816339750912/3*t^19 +
7794661267603456/9*t^18 + 649416870679543808/9*t^17 +
468149403744468992/9*t^16 - 119335391441453056/9*t^15 +
988417660100280320/9*t^14 + 2412897920924778496/9*t^13 +
378459323983986688/3*t^12 - 862912668397207552/9*t^11 -
325642358798417920/9*t^10 + 723793111402151936/9*t^9 -
147697396959150080/3*t^8 - 1973517818741653504/9*t^7 -
504790186358472704/3*t^6 - 46408186785169408/9*t^5 +
648870190062239744/9*t^4 + 481462947663577088/9*t^3 +
171699735793500160/9*t^2 + 10696049115004928/3*t + 281474976710656)>,
<235, (t^48 - 24*t^46 + 228*t^44 - 1064*t^42 + 2322*t^40 - 840*t^38 -
5676*t^36 + 8712*t^34 + 1455*t^32 - 16240*t^30 + 14664*t^28 +
9456*t^26 - 21892*t^24 + 9456*t^22 + 14664*t^20 - 16240*t^18 +
1455*t^16 + 8712*t^14 - 5676*t^12 - 840*t^10 + 2322*t^8 - 1064*t^6 +
228*t^4 - 24*t^2 + 1)/(t^32 - 8*t^30 + 12*t^28 + 8*t^26 - 26*t^24 +
8*t^22 + 12*t^20 - 8*t^18 + t^16)>, <236, (-4*t^48 + 48*t^46 +
492*t^44 - 6668*t^42 - 36801/2*t^40 + 313665*t^38 + 396879/4*t^36 -
21425409/4*t^34 + 315735825/64*t^32 + 164014135/8*t^30 -
154003173/32*t^28 - 1226937111/32*t^26 - 8134644167/256*t^24 -
1226937111/128*t^22 - 154003173/512*t^20 + 164014135/512*t^18 +
315735825/16384*t^16 - 21425409/4096*t^14 + 396879/16384*t^12 +
313665/16384*t^10 - 36801/131072*t^8 - 1667/65536*t^6 + 123/262144*t^4
+ 3/262144*t^2 - 1/4194304)/(t^44 - 10*t^42 + 87/2*t^40 - 221/2*t^38 +
2925/16*t^36 - 409/2*t^34 + 1205/8*t^32 - 477/8*t^30 - 1079/128*t^28 +
1885/64*t^26 - 5291/256*t^24 + 1885/256*t^22 - 1079/2048*t^20 -
477/512*t^18 + 1205/2048*t^16 - 409/2048*t^14 + 2925/65536*t^12 -
221/32768*t^10 + 87/131072*t^8 - 5/131072*t^6 + 1/1048576*t^4)>, <237,
(1728*t^48 - 304128*t^46 + 12229632*t^44 + 251564032*t^42 -
6599288832*t^40 - 410751909888*t^38 - 9109588754432*t^36 -
124399694708736*t^34 - 1148308440662016*t^32 - 7481160702099456*t^30 -
34325824396591104*t^28 - 112188769525825536*t^26 -
263879532265603072*t^24 - 448755078103302144*t^22 -
549213190345457664*t^20 - 478794284934365184*t^18 -
293966960809476096*t^16 - 127385287381745664*t^14 -
37312875538153472*t^12 - 6729759291604992*t^10 - 432490992893952*t^8 +
65946001604608*t^6 + 12823698604032*t^4 - 1275605286912*t^2 +
28991029248)/(t^48 + 208*t^46 + 16464*t^44 + 597184*t^42 +
8861728*t^40 + 13856512*t^38 - 357814016*t^36 - 831962112*t^34 +
6957362944*t^32 + 11223113728*t^30 - 79709822976*t^28 -
41210904576*t^26 + 490631970816*t^24 - 164843618304*t^22 -
1275357167616*t^20 + 718279278592*t^18 + 1781084913664*t^16 -
851929202688*t^14 - 1465606209536*t^12 + 227025092608*t^10 +
580762206208*t^8 + 156548202496*t^6 + 17263755264*t^4 + 872415232*t^2
+ 16777216)>, <238, (128*t^48 - 6144*t^47 + 99072*t^46 - 495104*t^45 -
1521504*t^44 + 13826688*t^43 + 7616464*t^42 - 147260640*t^41 -
12706623*t^40 + 838349080*t^39 + 2857281*t^38 - 2824627026*t^37 +
66310513/8*t^36 + 12057817251/2*t^35 - 41813889/16*t^34 -
68328698097/8*t^33 - 1348732689/512*t^32 + 132638686977/16*t^31 +
64512695/128*t^30 - 360681559155/64*t^29 + 478507941/1024*t^28 +
697321363337/256*t^27 - 45010455/2048*t^26 - 967427479395/1024*t^25 -
1357387577/32768*t^24 + 967427479395/4096*t^23 - 45010455/32768*t^22 -
697321363337/16384*t^21 + 478507941/262144*t^20 +
360681559155/65536*t^19 + 64512695/524288*t^18 -
132638686977/262144*t^17 - 1348732689/33554432*t^16 +
68328698097/2097152*t^15 - 41813889/16777216*t^14 -
12057817251/8388608*t^13 + 66310513/134217728*t^12 +
1412313513/33554432*t^11 + 2857281/268435456*t^10 -
104793635/134217728*t^9 - 12706623/4294967296*t^8 +
4601895/536870912*t^7 + 476029/4294967296*t^6 - 108021/2147483648*t^5
- 47547/34359738368*t^4 + 967/8589934592*t^3 + 387/68719476736*t^2 +
3/34359738368*t + 1/2199023255552)/(t^48 - 8*t^47 + 6*t^46 + 38*t^45 +
69/4*t^44 + 167/2*t^43 + 253/8*t^42 - 13/8*t^41 + 5313/128*t^40 -
4255/32*t^39 + 5313/128*t^38 - 18915/128*t^37 + 33649/1024*t^36 -
33055/512*t^35 + 43263/2048*t^34 + 9477/2048*t^33 + 735471/65536*t^32
+ 85931/4096*t^31 + 81719/16384*t^30 + 197935/16384*t^29 +
245157/131072*t^28 + 231043/65536*t^27 + 156009/262144*t^26 +
105183/262144*t^25 + 676039/4194304*t^24 - 105183/1048576*t^23 +
156009/4194304*t^22 - 231043/4194304*t^21 + 245157/33554432*t^20 -
197935/16777216*t^19 + 81719/67108864*t^18 - 85931/67108864*t^17 +
735471/4294967296*t^16 - 9477/536870912*t^15 + 43263/2147483648*t^14 +
33055/2147483648*t^13 + 33649/17179869184*t^12 + 18915/8589934592*t^11
+ 5313/34359738368*t^10 + 4255/34359738368*t^9 + 5313/549755813888*t^8
+ 13/137438953472*t^7 + 253/549755813888*t^6 - 167/549755813888*t^5 +
69/4398046511104*t^4 - 19/2199023255552*t^3 + 3/8796093022208*t^2 +
1/8796093022208*t + 1/281474976710656)>, <239, (10976*t^48 +
602112*t^47 + 16353792*t^46 + 289933312*t^45 + 3740723712*t^44 +
37180637184*t^43 + 294649243648*t^42 + 1905230831616*t^41 +
10230984465408*t^40 + 46380313542656*t^39 + 180879749996544*t^38 +
621913202196480*t^37 + 1943263213527040*t^36 + 5684125326901248*t^35 +
15831960537759744*t^34 + 41960564183400448*t^33 +
104907804437323776*t^32 + 246878828022988800*t^31 +
551684842123952128*t^30 + 1183767035290583040*t^29 +
2442781973368012800*t^28 + 4803960642631368704*t^27 +
8922081089855422464*t^26 + 15649436532014579712*t^25 +
26108519091058245632*t^24 + 41562764783338389504*t^23 +
62534919245197737984*t^22 + 87160098345086615552*t^21 +
110152570235869200384*t^20 + 124253588313939640320*t^19 +
124073665913438601216*t^18 + 109484930661013782528*t^17 +
85636710120057470976*t^16 + 59735254821584764928*t^15 +
37427697037917290496*t^14 + 21200631002396884992*t^13 +
10904500494068088832*t^12 + 5104042899381682176*t^11 +
2175630045219913728*t^10 + 844213466880278528*t^9 +
297506070388015104*t^8 + 94682833213194240*t^7 + 26968595183435776*t^6
+ 6792823638589440*t^5 + 1485640461975552*t^4 + 272232207089664*t^3 +
38918846152704*t^2 + 3788161155072*t + 184146722816)/(t^48 + 16*t^47 -
96*t^46 - 3808*t^45 - 22736*t^44 + 176064*t^43 + 3299968*t^42 +
16105344*t^41 - 48149472*t^40 - 1144594688*t^39 - 7284023808*t^38 -
17991217664*t^37 + 82225391360*t^36 + 1142703629312*t^35 +
7012844812288*t^34 + 30416424454144*t^33 + 103293219499776*t^32 +
285589534580736*t^31 + 652728353832960*t^30 + 1233268689551360*t^29 +
1893026451054592*t^28 + 2238541400473600*t^27 + 1694340189585408*t^26
- 131452987834368*t^25 - 2928541823287296*t^24 - 5462878698209280*t^23
- 6097464857985024*t^22 - 3941577921921024*t^21 + 225792649199616*t^20
+ 4207653476106240*t^19 + 5846790535380992*t^18 +
4550747472527360*t^17 + 1578295924555776*t^16 - 1092531898023936*t^15
- 2221245011591168*t^14 - 1874693135007744*t^13 - 908929996423168*t^12
- 111241418768384*t^11 + 237160787607552*t^10 + 254222025097216*t^9 +
154624394788864*t^8 + 67171996663808*t^7 + 21956510351360*t^6 +
5440482050048*t^5 + 1006951727104*t^4 + 134553272320*t^3 +
12213813248*t^2 + 671088640*t + 16777216)>, <240, (t^48 - 48*t^40 +
816*t^32 - 5632*t^24 + 13056*t^16 - 12288*t^8 + 4096)/(t^16 -
16*t^8)>, <241, (16*t^48 + 768*t^40 + 13056*t^32 + 90112*t^24 +
208896*t^16 + 196608*t^8 + 65536)/(t^40 + t^32)>, <242, (t^48 +
48*t^40 + 816*t^32 + 5632*t^24 + 13056*t^16 + 12288*t^8 + 4096)/(t^16
+ 16*t^8)>, <243, (-16*t^48 + 768*t^40 - 13056*t^32 + 90112*t^24 -
208896*t^16 + 196608*t^8 - 65536)/(t^40 - t^32)> ]; 
// Models of the rank one elliptic curves X_n. These all have good reduction 
// away from 2, and there really aren't so many elliptic curves like that.
rankoneellmodels := [ <150, [ 0, 1, 0, -13, -21 ]>, <153, [ 0, 1, 0,
-3, 1 ]>, <155, [ 0, 0, 0, -2, 0 ]>, <156, [ 0, 0, 0, 8, 0 ]>, <165, [
0, 1, 0, -13, -21 ]>, <166, [ 0, 1, 0, -3, 1 ]>, <167, [ 0, 1, 0, 1, 1
]>, <281, [ 0, 1, 0, -13, -21 ]>, <284, [ 0, 0, 0, 8, 0 ]>, <288, [ 0,
1, 0, -3, 1 ]>, <289, [ 0, 1, 0, -13, -21 ]>, <291, [ 0, 0, 0, -2, 0
]>, <295, [ 0, 1, 0, -9, 7 ]>, <297, [ 0, 1, 0, -9, 7 ]>, <302, [ 0,
1, 0, -13, -21 ]>, <304, [ 0, 1, 0, -13, -21 ]>, <308, [ 0, 1, 0, -3,
1 ]>, <309, [ 0, 1, 0, -13, -21 ]>, <312, [ 0, 1, 0, -3, 1 ]>, <318, [
0, 0, 0, 8, 0 ]>, <320, [ 0, 1, 0, -13, -21 ]>, <323, [ 0, 1, 0, -3, 1
]>, <324, [ 0, 0, 0, -2, 0 ]>, <326, [ 0, 1, 0, -13, -21 ]>, <328, [
0, 0, 0, 8, 0 ]>, <349, [ 0, 1, 0, -13, -21 ]>, <350, [ 0, 0, 0, 8, 0
]> ]; 
// Maps from X_n -> X_m where X_n is an elliptic curve of positive rank 
// and X_m has genus zero.  
rankoneellmaps := <<150, [ a
- c, -c ]>, <153, [ 4*a + 4*c, -2*c ]>, <155, [ 4*a, -2*c ]>, <156, [
a, c ]>, <165, [ a - c, -c ]>, <166, [ 4*a + 4*c, -2*c ]>, <167, [ 4*a
+ 4*c, 2*c ]>, <281, [ -3*a*c + b*c - 5*c^2, 2*a*c + 4*c^2 ]>, <284, [
-3*a*c + b*c, 2*a*c - 2*c^2 ]>, <288, [ a - c, c ]>, <289, [ a + 3*c,
2*c ]>, <291, [ a, c ]>, <295, [ a + c, -2*c ]>, <297, [ a + c, 2*c
]>, <302, [ a*c + b*c + 3*c^2, 2*a*c + 4*c^2 ]>, <304, [ 4*a + 4*c,
-16*c ]>, <308, [ -3/16*a*c + 1/8*b*c + 1/8*c^2, 1/2*a*c - 5/8*c^2 ]>,
<309, [ 4*a + 4*c, 16*c ]>, <312, [ 4*a, -8*c ]>, <318, [ a, -2*c ]>,
<320, [ a + 3*c, 2*c ]>, <323, [ a - c, c ]>, <324, [ a, c ]>, <326, [
a*c + b*c + 3*c^2, 2*a*c + 4*c^2 ]>, <328, [ a, -2*c ]>, <349, [
-1/8*a*c - 1/4*c^2, 3/64*a*c - 1/64*b*c + 5/64*c^2 ]>, <350, [ -24*a*c
+ 8*b*c, 8*a*c - 8*c^2 ]>>; 
// If -I is not contained in a subgroup H, the corresponding modular curve 
// is a family E_t over P^1. Each curve is represented as 
// E_t : y^2 = x^3 + A(t) x + B(t), and an elliptic E corresponds to a point 
// on the modular curve if and only if E_t is isomorphic to E for some t 
// (including t = infinity, by the way).  
// This list takes up most of the file 

eqf := [ [ 0, 0, 0, -27*t^6 - 139968*t^4 - 241864704*t^2 - 139314069504, 
54*t^9 +
373248*t^7 + 967458816*t^5 + 1114512556032*t^3 + 481469424205824*t ],
[ 0, 0, 0, -108*t^6 - 559872*t^4 - 967458816*t^2 - 557256278016,
432*t^9 + 2985984*t^7 + 7739670528*t^5 + 8916100448256*t^3 +
3851755393646592*t ], [ 0, 0, 0, -108*t^6 + 559872*t^4 - 967458816*t^2
+ 557256278016, 432*t^9 - 2985984*t^7 + 7739670528*t^5 -
8916100448256*t^3 + 3851755393646592*t ], [ 0, 0, 0, -27*t^6 +
139968*t^4 - 241864704*t^2 + 139314069504, 54*t^9 - 373248*t^7 +
967458816*t^5 - 1114512556032*t^3 + 481469424205824*t ], [ 0, 0, 0,
-27*t^6 - 27*t^4 + 135*t^2 - 81, 54*t^9 - 648*t^7 + 1620*t^5 -
1512*t^3 + 486*t ], [ 0, 0, 0, -108*t^6 - 108*t^4 + 540*t^2 - 324,
432*t^9 - 5184*t^7 + 12960*t^5 - 12096*t^3 + 3888*t ], [ 0, 0, 0,
-27*t^2 - 81, 54*t^3 - 486*t ], [ 0, 0, 0, -108*t^2 - 324, 432*t^3 -
3888*t ], [ 0, 0, 0, -27*t^6 - 4752*t^4 - 276480*t^2 - 5308416, 54*t^9
+ 14256*t^7 + 1410048*t^5 + 61931520*t^3 + 1019215872*t ], [ 0, 0, 0,
-108*t^6 - 19008*t^4 - 1105920*t^2 - 21233664, 432*t^9 + 114048*t^7 +
11280384*t^5 + 495452160*t^3 + 8153726976*t ], [ 0, 0, 0, -27*t^2 -
1296, 54*t^3 + 3888*t ], [ 0, 0, 0, -108*t^2 - 5184, 432*t^3 + 31104*t
], [ 0, 0, 0, 324*t^4 - 108*t^2, 3888*t^5 + 432*t^3 ], [ 0, 0, 0,
81*t^8 + 135*t^6 + 27*t^4 - 27*t^2, 486*t^11 + 1512*t^9 + 1620*t^7 +
648*t^5 + 54*t^3 ], [ 0, 0, 0, 324*t^8 + 540*t^6 + 108*t^4 - 108*t^2,
3888*t^11 + 12096*t^9 + 12960*t^7 + 5184*t^5 + 432*t^3 ], [ 0, 0, 0,
81*t^4 - 27*t^2, 486*t^5 + 54*t^3 ], [ 0, 0, 0, -108*t^6 + 19008*t^4 -
1105920*t^2 + 21233664, 432*t^9 - 114048*t^7 + 11280384*t^5 -
495452160*t^3 + 8153726976*t ], [ 0, 0, 0, -27*t^6 + 4752*t^4 -
276480*t^2 + 5308416, 54*t^9 - 14256*t^7 + 1410048*t^5 - 61931520*t^3
+ 1019215872*t ], [ 0, 0, 0, -108*t^2 + 5184, 432*t^3 - 31104*t ], [
0, 0, 0, -27*t^2 + 1296, 54*t^3 - 3888*t ], [ 0, 0, 0, -108*t^4 +
1728*t^3 - 1728*t^2 - 82944*t + 331776, 432*t^6 - 10368*t^5 +
51840*t^4 + 525312*t^3 - 5971968*t^2 + 15925248*t ], [ 0, 0, 0,
-27*t^4 + 432*t^3 - 432*t^2 - 20736*t + 82944, -54*t^6 + 1296*t^5 -
6480*t^4 - 65664*t^3 + 746496*t^2 - 1990656*t ], [ 0, 0, 0, -108*t^4 +
1728*t^3 - 1728*t^2 - 82944*t + 331776, -432*t^6 + 10368*t^5 -
51840*t^4 - 525312*t^3 + 5971968*t^2 - 15925248*t ], [ 0, 0, 0,
-27*t^4 + 432*t^3 - 432*t^2 - 20736*t + 82944, 54*t^6 - 1296*t^5 +
6480*t^4 + 65664*t^3 - 746496*t^2 + 1990656*t ], [ 0, 0, 0, -108*t^8 +
2592*t^7 - 16848*t^6 - 9504*t^5 + 240408*t^4 + 318816*t^3 - 422928*t^2
- 940896*t - 431244, 432*t^12 - 15552*t^11 + 194400*t^10 - 782784*t^9
- 2010096*t^8 + 13779072*t^7 + 38574144*t^6 - 69579648*t^5 -
514588464*t^4 - 1123581888*t^3 - 1256096160*t^2 - 724489920*t -
170772624 ], [ 0, 0, 0, -27*t^8 + 648*t^7 - 4212*t^6 - 2376*t^5 +
60102*t^4 + 79704*t^3 - 105732*t^2 - 235224*t - 107811, 54*t^12 -
1944*t^11 + 24300*t^10 - 97848*t^9 - 251262*t^8 + 1722384*t^7 +
4821768*t^6 - 8697456*t^5 - 64323558*t^4 - 140447736*t^3 -
157012020*t^2 - 90561240*t - 21346578 ], [ 0, 0, 0, -108*t^10 -
324*t^8 - 432*t^6 - 324*t^4 - 108*t^2, 432*t^15 + 1944*t^13 +
2592*t^11 - 2592*t^7 - 1944*t^5 - 432*t^3 ], [ 0, 0, 0, -27*t^6 -
27*t^4 - 27*t^2, 54*t^9 + 81*t^7 - 81*t^5 - 54*t^3 ], [ 0, 0, 0,
-108*t^4 - 108*t^2 - 108, 432*t^6 + 648*t^4 - 648*t^2 - 432 ], [ 0, 0,
0, -27*t^8 - 81*t^6 - 108*t^4 - 81*t^2 - 27, 54*t^12 + 243*t^10 +
324*t^8 - 324*t^4 - 243*t^2 - 54 ], [ 0, 0, 0, -27*t^4 - 27*t^2 - 27,
54*t^6 + 81*t^4 - 81*t^2 - 54 ], [ 0, 0, 0, -108*t^6 - 108*t^4 -
108*t^2, 432*t^9 + 648*t^7 - 648*t^5 - 432*t^3 ], [ 0, 0, 0, -27*t^10
- 81*t^8 - 108*t^6 - 81*t^4 - 27*t^2, 54*t^15 + 243*t^13 + 324*t^11 -
324*t^7 - 243*t^5 - 54*t^3 ], [ 0, 0, 0, -108*t^8 - 324*t^6 - 432*t^4
- 324*t^2 - 108, 432*t^12 + 1944*t^10 + 2592*t^8 - 2592*t^4 - 1944*t^2
- 432 ], [ 0, 0, 0, -108*t^6 + 216*t^5 - 216*t^3 + 216*t - 108,
432*t^9 - 1296*t^8 + 648*t^7 + 1512*t^6 - 2592*t^5 + 2592*t^4 -
1512*t^3 - 648*t^2 + 1296*t - 432 ], [ 0, 0, 0, -108*t^6 - 216*t^5 +
216*t^3 - 216*t - 108, 432*t^9 + 1296*t^8 + 648*t^7 - 1512*t^6 -
2592*t^5 - 2592*t^4 - 1512*t^3 + 648*t^2 + 1296*t + 432 ], [ 0, 0, 0,
-108*t^6 + 108*t^4 - 108*t^2, 432*t^9 - 648*t^7 - 648*t^5 + 432*t^3 ],
[ 0, 0, 0, -108*t^10 + 324*t^8 - 432*t^6 + 324*t^4 - 108*t^2, 432*t^15
- 1944*t^13 + 2592*t^11 - 2592*t^7 + 1944*t^5 - 432*t^3 ], [ 0, 0, 0,
-108*t^4 + 108*t^2 - 108, 432*t^6 - 648*t^4 - 648*t^2 + 432 ], [ 0, 0,
0, -108*t^8 + 324*t^6 - 432*t^4 + 324*t^2 - 108, 432*t^12 - 1944*t^10
+ 2592*t^8 - 2592*t^4 + 1944*t^2 - 432 ], [ 0, 0, 0, -108*t^4 +
108*t^2 - 108, -432*t^6 + 648*t^4 + 648*t^2 - 432 ], [ 0, 0, 0,
-27*t^4 + 27*t^2 - 27, -54*t^6 + 81*t^4 + 81*t^2 - 54 ], [ 0, 0, 0,
-27*t^8 + 81*t^6 - 108*t^4 + 81*t^2 - 27, 54*t^12 - 243*t^10 + 324*t^8
- 324*t^4 + 243*t^2 - 54 ], [ 0, 0, 0, -27*t^6 + 54*t^5 - 54*t^3 +
54*t - 27, 54*t^9 - 162*t^8 + 81*t^7 + 189*t^6 - 324*t^5 + 324*t^4 -
189*t^3 - 81*t^2 + 162*t - 54 ], [ 0, 0, 0, -27*t^6 - 54*t^5 + 54*t^3
- 54*t - 27, 54*t^9 + 162*t^8 + 81*t^7 - 189*t^6 - 324*t^5 - 324*t^4 -
189*t^3 + 81*t^2 + 162*t + 54 ], [ 0, 0, 0, -27*t^6 + 27*t^4 - 27*t^2,
54*t^9 - 81*t^7 - 81*t^5 + 54*t^3 ], [ 0, 0, 0, -27*t^10 + 81*t^8 -
108*t^6 + 81*t^4 - 27*t^2, 54*t^15 - 243*t^13 + 324*t^11 - 324*t^7 +
243*t^5 - 54*t^3 ], [ 0, 0, 0, -27*t^4 + 27*t^2 - 27, 54*t^6 - 81*t^4
- 81*t^2 + 54 ], [ 0, 0, 0, -27648*t^10 + 82944*t^8 + 44928*t^6 +
5184*t^4 - 108*t^2, 1769472*t^15 + 15925248*t^13 + 7630848*t^11 -
476928*t^7 - 62208*t^5 - 432*t^3 ], [ 0, 0, 0, -6912*t^10 + 20736*t^8
+ 11232*t^6 + 1296*t^4 - 27*t^2, 221184*t^15 + 1990656*t^13 +
953856*t^11 - 59616*t^7 - 7776*t^5 - 54*t^3 ], [ 0, 0, 0, -27648*t^8 +
82944*t^6 + 44928*t^4 + 5184*t^2 - 108, 1769472*t^12 + 15925248*t^10 +
7630848*t^8 - 476928*t^4 - 62208*t^2 - 432 ], [ 0, 0, 0, -6912*t^8 +
20736*t^6 + 11232*t^4 + 1296*t^2 - 27, 221184*t^12 + 1990656*t^10 +
953856*t^8 - 59616*t^4 - 7776*t^2 - 54 ], [ 0, 0, 0, -1728*t^6 +
6048*t^4 - 108*t^2, 27648*t^9 + 228096*t^7 - 57024*t^5 - 432*t^3 ], [
0, 0, 0, -432*t^6 + 1512*t^4 - 27*t^2, 3456*t^9 + 28512*t^7 - 7128*t^5
- 54*t^3 ], [ 0, 0, 0, -1728*t^4 + 6048*t^2 - 108, 27648*t^6 +
228096*t^4 - 57024*t^2 - 432 ], [ 0, 0, 0, -432*t^4 + 1512*t^2 - 27,
3456*t^6 + 28512*t^4 - 7128*t^2 - 54 ], [ 0, 0, 0, -108*t^10 -
5184*t^8 - 84672*t^6 - 497664*t^4 - 442368*t^2, 432*t^15 + 31104*t^13
+ 881280*t^11 + 12192768*t^9 + 80953344*t^7 + 191102976*t^5 -
113246208*t^3 ], [ 0, 0, 0, -27*t^10 - 1296*t^8 - 21168*t^6 -
124416*t^4 - 110592*t^2, 54*t^15 + 3888*t^13 + 110160*t^11 +
1524096*t^9 + 10119168*t^7 + 23887872*t^5 - 14155776*t^3 ], [ 0, 0, 0,
-108*t^8 - 5184*t^6 - 84672*t^4 - 497664*t^2 - 442368, 432*t^12 +
31104*t^10 + 881280*t^8 + 12192768*t^6 + 80953344*t^4 + 191102976*t^2
- 113246208 ], [ 0, 0, 0, -108*t^8 - 5184*t^6 - 84672*t^4 - 497664*t^2
- 442368, -432*t^12 - 31104*t^10 - 881280*t^8 - 12192768*t^6 -
80953344*t^4 - 191102976*t^2 + 113246208 ], [ 0, 0, 0, -27*t^8 -
1296*t^6 - 21168*t^4 - 124416*t^2 - 110592, 54*t^12 + 3888*t^10 +
110160*t^8 + 1524096*t^6 + 10119168*t^4 + 23887872*t^2 - 14155776 ], [
0, 0, 0, -27*t^8 - 1296*t^6 - 21168*t^4 - 124416*t^2 - 110592,
-54*t^12 - 3888*t^10 - 110160*t^8 - 1524096*t^6 - 10119168*t^4 -
23887872*t^2 + 14155776 ], [ 0, 0, 0, -108*t^6 - 1728*t^4 - 1728*t^2,
432*t^9 + 10368*t^7 + 51840*t^5 - 27648*t^3 ], [ 0, 0, 0, -27*t^6 -
432*t^4 - 432*t^2, 54*t^9 + 1296*t^7 + 6480*t^5 - 3456*t^3 ], [ 0, 0,
0, -27*t^4 - 432*t^2 - 432, -54*t^6 - 1296*t^4 - 6480*t^2 + 3456 ], [
0, 0, 0, -108*t^4 - 1728*t^2 - 1728, 432*t^6 + 10368*t^4 + 51840*t^2 -
27648 ], [ 0, 0, 0, -108*t^4 - 1728*t^2 - 1728, -432*t^6 - 10368*t^4 -
51840*t^2 + 27648 ], [ 0, 0, 0, -27*t^4 - 432*t^2 - 432, 54*t^6 +
1296*t^4 + 6480*t^2 - 3456 ], [ 0, 0, 0, -108*t^8 - 2592*t^6 -
21168*t^4 - 62208*t^2 - 27648, 432*t^12 + 15552*t^10 + 220320*t^8 +
1524096*t^6 + 5059584*t^4 + 5971968*t^2 - 1769472 ], [ 0, 0, 0,
-108*t^8 - 2592*t^6 - 21168*t^4 - 62208*t^2 - 27648, -432*t^12 -
15552*t^10 - 220320*t^8 - 1524096*t^6 - 5059584*t^4 - 5971968*t^2 +
1769472 ], [ 0, 0, 0, -27*t^8 - 648*t^6 - 5292*t^4 - 15552*t^2 - 6912,
54*t^12 + 1944*t^10 + 27540*t^8 + 190512*t^6 + 632448*t^4 + 746496*t^2
- 221184 ], [ 0, 0, 0, -27*t^8 - 648*t^6 - 5292*t^4 - 15552*t^2 -
6912, -54*t^12 - 1944*t^10 - 27540*t^8 - 190512*t^6 - 632448*t^4 -
746496*t^2 + 221184 ], [ 0, 0, 0, -27*t^4 - 216*t^2 - 108, 54*t^6 +
648*t^4 + 1620*t^2 - 432 ], [ 0, 0, 0, -27*t^4 - 216*t^2 - 108,
-54*t^6 - 648*t^4 - 1620*t^2 + 432 ], [ 0, 0, 0, -108*t^4 - 864*t^2 -
432, -432*t^6 - 5184*t^4 - 12960*t^2 + 3456 ], [ 0, 0, 0, -108*t^4 -
864*t^2 - 432, 432*t^6 + 5184*t^4 + 12960*t^2 - 3456 ], [ 0, 0, 0,
-432*t^8 + 2592*t^6 - 5292*t^4 + 3888*t^2 - 432, 3456*t^12 -
31104*t^10 + 110160*t^8 - 190512*t^6 + 158112*t^4 - 46656*t^2 - 3456
], [ 0, 0, 0, -108*t^8 + 648*t^6 - 1323*t^4 + 972*t^2 - 108, 432*t^12
- 3888*t^10 + 13770*t^8 - 23814*t^6 + 19764*t^4 - 5832*t^2 - 432 ], [
0, 0, 0, -108*t^8 + 648*t^6 - 1323*t^4 + 972*t^2 - 108, -432*t^12 +
3888*t^10 - 13770*t^8 + 23814*t^6 - 19764*t^4 + 5832*t^2 + 432 ], [ 0,
0, 0, -432*t^8 + 2592*t^6 - 5292*t^4 + 3888*t^2 - 432, -3456*t^12 +
31104*t^10 - 110160*t^8 + 190512*t^6 - 158112*t^4 + 46656*t^2 + 3456
], [ 0, 0, 0, -432*t^4 + 864*t^2 - 108, -3456*t^6 + 10368*t^4 -
6480*t^2 - 432 ], [ 0, 0, 0, -432*t^4 + 864*t^2 - 108, 3456*t^6 -
10368*t^4 + 6480*t^2 + 432 ], [ 0, 0, 0, -108*t^4 + 216*t^2 - 27,
432*t^6 - 1296*t^4 + 810*t^2 + 54 ], [ 0, 0, 0, -108*t^4 + 216*t^2 -
27, -432*t^6 + 1296*t^4 - 810*t^2 - 54 ], [ 0, 0, 0, -27*t^8 + 216*t^7
- 3456*t^5 + 6480*t^4 + 3456*t^3 - 6912*t^2, 54*t^12 - 648*t^11 +
1296*t^10 + 12096*t^9 - 55728*t^8 + 5184*t^7 + 314496*t^6 - 456192*t^5
+ 165888*t^4 - 221184*t^3 ], [ 0, 0, 0, -27*t^6 - 216*t^5 + 3456*t^3 +
6480*t^2 - 3456*t - 6912, 54*t^9 + 648*t^8 + 1296*t^7 - 12096*t^6 -
55728*t^5 - 5184*t^4 + 314496*t^3 + 456192*t^2 + 165888*t + 221184 ],
[ 0, 0, 0, -108*t^8 + 5184*t^6 - 84672*t^4 + 497664*t^2 - 442368,
-432*t^12 + 31104*t^10 - 881280*t^8 + 12192768*t^6 - 80953344*t^4 +
191102976*t^2 + 113246208 ], [ 0, 0, 0, -108*t^10 + 5184*t^8 -
84672*t^6 + 497664*t^4 - 442368*t^2, 432*t^15 - 31104*t^13 +
881280*t^11 - 12192768*t^9 + 80953344*t^7 - 191102976*t^5 -
113246208*t^3 ], [ 0, 0, 0, -108*t^8 + 5184*t^6 - 84672*t^4 +
497664*t^2 - 442368, 432*t^12 - 31104*t^10 + 881280*t^8 - 12192768*t^6
+ 80953344*t^4 - 191102976*t^2 - 113246208 ], [ 0, 0, 0, -27*t^8 +
1296*t^6 - 21168*t^4 + 124416*t^2 - 110592, -54*t^12 + 3888*t^10 -
110160*t^8 + 1524096*t^6 - 10119168*t^4 + 23887872*t^2 + 14155776 ], [
0, 0, 0, -27*t^10 + 1296*t^8 - 21168*t^6 + 124416*t^4 - 110592*t^2,
54*t^15 - 3888*t^13 + 110160*t^11 - 1524096*t^9 + 10119168*t^7 -
23887872*t^5 - 14155776*t^3 ], [ 0, 0, 0, -27*t^8 + 1296*t^6 -
21168*t^4 + 124416*t^2 - 110592, 54*t^12 - 3888*t^10 + 110160*t^8 -
1524096*t^6 + 10119168*t^4 - 23887872*t^2 - 14155776 ], [ 0, 0, 0,
-108*t^4 + 1728*t^2 - 1728, -432*t^6 + 10368*t^4 - 51840*t^2 - 27648
], [ 0, 0, 0, -108*t^6 + 1728*t^4 - 1728*t^2, 432*t^9 - 10368*t^7 +
51840*t^5 + 27648*t^3 ], [ 0, 0, 0, -108*t^4 + 1728*t^2 - 1728,
432*t^6 - 10368*t^4 + 51840*t^2 + 27648 ], [ 0, 0, 0, -27*t^4 +
432*t^2 - 432, -54*t^6 + 1296*t^4 - 6480*t^2 - 3456 ], [ 0, 0, 0,
-27*t^6 + 432*t^4 - 432*t^2, 54*t^9 - 1296*t^7 + 6480*t^5 + 3456*t^3
], [ 0, 0, 0, -27*t^4 + 432*t^2 - 432, 54*t^6 - 1296*t^4 + 6480*t^2 +
3456 ], [ 0, 0, 0, -108*t^8 + 864*t^7 - 13824*t^5 + 25920*t^4 +
13824*t^3 - 27648*t^2, -432*t^12 + 5184*t^11 - 10368*t^10 - 96768*t^9
+ 445824*t^8 - 41472*t^7 - 2515968*t^6 + 3649536*t^5 - 1327104*t^4 +
1769472*t^3 ], [ 0, 0, 0, -108*t^6 - 864*t^5 + 13824*t^3 + 25920*t^2 -
13824*t - 27648, 432*t^9 + 5184*t^8 + 10368*t^7 - 96768*t^6 -
445824*t^5 - 41472*t^4 + 2515968*t^3 + 3649536*t^2 + 1327104*t +
1769472 ], [ 0, 0, 0, -108*t^8 + 864*t^7 - 13824*t^5 + 25920*t^4 +
13824*t^3 - 27648*t^2, 432*t^12 - 5184*t^11 + 10368*t^10 + 96768*t^9 -
445824*t^8 + 41472*t^7 + 2515968*t^6 - 3649536*t^5 + 1327104*t^4 -
1769472*t^3 ], [ 0, 0, 0, -108*t^6 + 864*t^5 - 13824*t^3 + 25920*t^2 +
13824*t - 27648, 432*t^9 - 5184*t^8 + 10368*t^7 + 96768*t^6 -
445824*t^5 + 41472*t^4 + 2515968*t^3 - 3649536*t^2 + 1327104*t -
1769472 ], [ 0, 0, 0, -27*t^8 + 216*t^7 - 3456*t^5 + 6480*t^4 +
3456*t^3 - 6912*t^2, -54*t^12 + 648*t^11 - 1296*t^10 - 12096*t^9 +
55728*t^8 - 5184*t^7 - 314496*t^6 + 456192*t^5 - 165888*t^4 +
221184*t^3 ], [ 0, 0, 0, -27*t^6 + 216*t^5 - 3456*t^3 + 6480*t^2 +
3456*t - 6912, 54*t^9 - 648*t^8 + 1296*t^7 + 12096*t^6 - 55728*t^5 +
5184*t^4 + 314496*t^3 - 456192*t^2 + 165888*t - 221184 ], [ 0, 0, 0,
-49545216*t^12 - 191102976*t^11 - 332660736*t^10 - 353009664*t^9 -
258453504*t^8 - 138682368*t^7 - 56180736*t^6 - 17335296*t^5 -
4038336*t^4 - 689472*t^3 - 81216*t^2 - 5832*t - 189, 123211874304*t^18
+ 684913065984*t^17 + 1747615481856*t^16 + 2723344809984*t^15 +
2897970462720*t^14 + 2216794521600*t^13 + 1236960018432*t^12 +
488968814592*t^11 + 117252292608*t^10 - 14656536576*t^8 -
7640137728*t^7 - 2415937536*t^6 - 541209600*t^5 - 88439040*t^4 -
10388736*t^3 - 833328*t^2 - 40824*t - 918 ], [ 0, 0, 0, -774144*t^8 -
2211840*t^7 - 2598912*t^6 - 1714176*t^5 - 736128*t^4 - 214272*t^3 -
40608*t^2 - 4320*t - 189, 240648192*t^12 + 976748544*t^11 +
1677459456*t^10 + 1583677440*t^9 + 875225088*t^8 + 253476864*t^7 -
31684608*t^5 - 13675392*t^4 - 3093120*t^3 - 409536*t^2 - 29808*t - 918
], [ 0, 0, 0, -49545216*t^12 - 191102976*t^11 - 332660736*t^10 -
353009664*t^9 - 258453504*t^8 - 138682368*t^7 - 56180736*t^6 -
17335296*t^5 - 4038336*t^4 - 689472*t^3 - 81216*t^2 - 5832*t - 189,
-123211874304*t^18 - 684913065984*t^17 - 1747615481856*t^16 -
2723344809984*t^15 - 2897970462720*t^14 - 2216794521600*t^13 -
1236960018432*t^12 - 488968814592*t^11 - 117252292608*t^10 +
14656536576*t^8 + 7640137728*t^7 + 2415937536*t^6 + 541209600*t^5 +
88439040*t^4 + 10388736*t^3 + 833328*t^2 + 40824*t + 918 ], [ 0, 0, 0,
-3096576*t^8 - 8847360*t^7 - 10395648*t^6 - 6856704*t^5 - 2944512*t^4
- 857088*t^3 - 162432*t^2 - 17280*t - 756, 1925185536*t^12 +
7813988352*t^11 + 13419675648*t^10 + 12669419520*t^9 + 7001800704*t^8
+ 2027814912*t^7 - 253476864*t^5 - 109403136*t^4 - 24744960*t^3 -
3276288*t^2 - 238464*t - 7344 ], [ 0, 0, 0, -27*t^8 + 162*t^6 -
432*t^4 + 648*t^2 - 432, 54*t^12 - 486*t^10 + 1296*t^8 - 5184*t^4 +
7776*t^2 - 3456 ], [ 0, 0, 0, -108*t^8 + 648*t^6 - 1728*t^4 + 2592*t^2
- 1728, 432*t^12 - 3888*t^10 + 10368*t^8 - 41472*t^4 + 62208*t^2 -
27648 ], [ 0, 0, 0, -27*t^4 + 54*t^2 - 108, -54*t^6 + 162*t^4 +
324*t^2 - 432 ], [ 0, 0, 0, -27*t^4 + 54*t^2 - 108, 54*t^6 - 162*t^4 -
324*t^2 + 432 ], [ 0, 0, 0, -17280*t^4 - 6912*t^3 - 864*t^2 - 864*t -
270, -774144*t^6 - 165888*t^5 + 373248*t^4 + 235008*t^3 + 46656*t^2 -
2592*t - 1512 ], [ 0, 0, 0, -17280*t^4 - 6912*t^3 - 864*t^2 - 864*t -
270, 774144*t^6 + 165888*t^5 - 373248*t^4 - 235008*t^3 - 46656*t^2 +
2592*t + 1512 ], [ 0, 0, 0, -70778880*t^12 - 169869312*t^11 -
130940928*t^10 - 21233664*t^9 + 15261696*t^8 + 5308416*t^7 +
552960*t^6 + 663552*t^5 + 238464*t^4 - 41472*t^3 - 31968*t^2 - 5184*t
- 270, -202937204736*t^18 - 652298158080*t^17 - 641426522112*t^16 +
97844723712*t^15 + 625119068160*t^14 + 411763212288*t^13 +
15174991872*t^12 - 102431195136*t^11 - 45312638976*t^10 +
5664079872*t^8 + 1600487424*t^7 - 29638656*t^6 - 100528128*t^5 -
19077120*t^4 - 373248*t^3 + 305856*t^2 + 38880*t + 1512 ], [ 0, 0, 0,
-70778880*t^12 - 169869312*t^11 - 130940928*t^10 - 21233664*t^9 +
15261696*t^8 + 5308416*t^7 + 552960*t^6 + 663552*t^5 + 238464*t^4 -
41472*t^3 - 31968*t^2 - 5184*t - 270, 202937204736*t^18 +
652298158080*t^17 + 641426522112*t^16 - 97844723712*t^15 -
625119068160*t^14 - 411763212288*t^13 - 15174991872*t^12 +
102431195136*t^11 + 45312638976*t^10 - 5664079872*t^8 - 1600487424*t^7
+ 29638656*t^6 + 100528128*t^5 + 19077120*t^4 + 373248*t^3 -
305856*t^2 - 38880*t - 1512 ], [ 0, 0, 0, 54*t^12 - 6048*t^10 -
41472*t^9 - 72576*t^8 + 331776*t^7 + 1880064*t^6 + 2654208*t^5 -
4644864*t^4 - 21233664*t^3 - 24772608*t^2 + 14155776, 540*t^18 +
11664*t^17 + 95904*t^16 + 311040*t^15 - 414720*t^14 - 6469632*t^13 -
15704064*t^12 + 19906560*t^11 + 145981440*t^10 - 1167851520*t^8 -
1274019840*t^7 + 8040480768*t^6 + 26499612672*t^5 + 13589544960*t^4 -
81537269760*t^3 - 201125265408*t^2 - 195689447424*t - 72477573120 ], [
0, 0, 0, 216*t^8 - 1728*t^7 - 17280*t^6 + 13824*t^5 + 248832*t^4 +
110592*t^3 - 1105920*t^2 - 884736*t + 884736, 4320*t^12 + 41472*t^11 -
41472*t^10 - 1105920*t^9 - 165888*t^8 + 13271040*t^7 - 106168320*t^5 +
10616832*t^4 + 566231040*t^3 + 169869312*t^2 - 1358954496*t -
1132462080 ], [ 0, 0, 0, 54*t^12 - 6048*t^10 - 41472*t^9 - 72576*t^8 +
331776*t^7 + 1880064*t^6 + 2654208*t^5 - 4644864*t^4 - 21233664*t^3 -
24772608*t^2 + 14155776, -540*t^18 - 11664*t^17 - 95904*t^16 -
311040*t^15 + 414720*t^14 + 6469632*t^13 + 15704064*t^12 -
19906560*t^11 - 145981440*t^10 + 1167851520*t^8 + 1274019840*t^7 -
8040480768*t^6 - 26499612672*t^5 - 13589544960*t^4 + 81537269760*t^3 +
201125265408*t^2 + 195689447424*t + 72477573120 ], [ 0, 0, 0, 54*t^8 -
432*t^7 - 4320*t^6 + 3456*t^5 + 62208*t^4 + 27648*t^3 - 276480*t^2 -
221184*t + 221184, 540*t^12 + 5184*t^11 - 5184*t^10 - 138240*t^9 -
20736*t^8 + 1658880*t^7 - 13271040*t^5 + 1327104*t^4 + 70778880*t^3 +
21233664*t^2 - 169869312*t - 141557760 ], [ 0, 0, 0, -108*t^8 -
2592*t^6 + 11232*t^4 - 10368*t^2 - 1728, 432*t^12 - 31104*t^10 +
119232*t^8 - 476928*t^4 + 497664*t^2 - 27648 ], [ 0, 0, 0, -27*t^8 -
648*t^6 + 2808*t^4 - 2592*t^2 - 432, 54*t^12 - 3888*t^10 + 14904*t^8 -
59616*t^4 + 62208*t^2 - 3456 ], [ 0, 0, 0, -27*t^4 - 756*t^2 - 108,
-54*t^6 + 3564*t^4 + 7128*t^2 - 432 ], [ 0, 0, 0, -27*t^4 - 756*t^2 -
108, 54*t^6 - 3564*t^4 - 7128*t^2 + 432 ], [ 0, 0, 0, -27*t^8 -
162*t^6 - 432*t^4 - 648*t^2 - 432, 54*t^12 + 486*t^10 + 1296*t^8 -
5184*t^4 - 7776*t^2 - 3456 ], [ 0, 0, 0, -108*t^8 - 648*t^6 - 1728*t^4
- 2592*t^2 - 1728, 432*t^12 + 3888*t^10 + 10368*t^8 - 41472*t^4 -
62208*t^2 - 27648 ], [ 0, 0, 0, -27*t^4 - 54*t^2 - 108, -54*t^6 -
162*t^4 + 324*t^2 + 432 ], [ 0, 0, 0, -27*t^4 - 54*t^2 - 108, 54*t^6 +
162*t^4 - 324*t^2 - 432 ], [ 0, 0, 0, -108*t^8 + 2592*t^6 + 11232*t^4
+ 10368*t^2 - 1728, 432*t^12 + 31104*t^10 + 119232*t^8 - 476928*t^4 -
497664*t^2 - 27648 ], [ 0, 0, 0, -27*t^8 + 648*t^6 + 2808*t^4 +
2592*t^2 - 432, 54*t^12 + 3888*t^10 + 14904*t^8 - 59616*t^4 -
62208*t^2 - 3456 ], [ 0, 0, 0, -27*t^4 + 756*t^2 - 108, -54*t^6 -
3564*t^4 + 7128*t^2 + 432 ], [ 0, 0, 0, -27*t^4 + 756*t^2 - 108,
54*t^6 + 3564*t^4 - 7128*t^2 - 432 ], [ 0, 0, 0, -32925150*t^20 +
16167384*t^19 + 101160468*t^18 + 230758632*t^17 + 336911994*t^16 +
315360*t^15 - 955654416*t^14 - 2522286432*t^13 - 4398721308*t^12 -
5861269296*t^11 - 6326184456*t^10 - 5664670416*t^9 - 4206560796*t^8 -
2560107168*t^7 - 1246787856*t^6 - 468036576*t^5 - 129264390*t^4 -
24870888*t^3 - 3100140*t^2 - 221400*t - 6750, -72719803296*t^30 +
53607412224*t^29 + 328107793824*t^28 + 684074101248*t^27 +
652759044192*t^26 - 1519490534400*t^25 - 6722743710816*t^24 -
13362326590464*t^23 - 15937371346464*t^22 - 3748273740288*t^21 +
34341605284896*t^20 + 102487396953600*t^19 + 193133174982624*t^18 +
285794906038272*t^17 + 353849310717984*t^16 + 376978275846144*t^15 +
350002399786272*t^14 + 284385380212224*t^13 + 202153174921440*t^12 +
125191778674176*t^11 + 66960218981664*t^10 + 30553319605248*t^9 +
11732785809120*t^8 + 3741616032768*t^7 + 977369185440*t^6 +
205582551552*t^5 + 33963564384*t^4 + 4237263360*t^3 + 374090400*t^2 +
20736000*t + 540000 ], [ 0, 0, 0, -32925150*t^20 + 16167384*t^19 +
101160468*t^18 + 230758632*t^17 + 336911994*t^16 + 315360*t^15 -
955654416*t^14 - 2522286432*t^13 - 4398721308*t^12 - 5861269296*t^11 -
6326184456*t^10 - 5664670416*t^9 - 4206560796*t^8 - 2560107168*t^7 -
1246787856*t^6 - 468036576*t^5 - 129264390*t^4 - 24870888*t^3 -
3100140*t^2 - 221400*t - 6750, 72719803296*t^30 - 53607412224*t^29 -
328107793824*t^28 - 684074101248*t^27 - 652759044192*t^26 +
1519490534400*t^25 + 6722743710816*t^24 + 13362326590464*t^23 +
15937371346464*t^22 + 3748273740288*t^21 - 34341605284896*t^20 -
102487396953600*t^19 - 193133174982624*t^18 - 285794906038272*t^17 -
353849310717984*t^16 - 376978275846144*t^15 - 350002399786272*t^14 -
284385380212224*t^13 - 202153174921440*t^12 - 125191778674176*t^11 -
66960218981664*t^10 - 30553319605248*t^9 - 11732785809120*t^8 -
3741616032768*t^7 - 977369185440*t^6 - 205582551552*t^5 -
33963564384*t^4 - 4237263360*t^3 - 374090400*t^2 - 20736000*t - 540000
], [ 0, 0, 0, -27*t^12 - 54*t^10 - 405*t^8 - 756*t^6 - 405*t^4 -
54*t^2 - 27, 54*t^18 + 162*t^16 - 1620*t^14 - 5292*t^12 - 7128*t^10 -
7128*t^8 - 5292*t^6 - 1620*t^4 + 162*t^2 + 54 ], [ 0, 0, 0, -108*t^10
- 1512*t^6 - 108*t^2, 432*t^15 - 14256*t^11 - 14256*t^7 + 432*t^3 ], [
0, 0, 0, -27*t^16 - 324*t^12 + 702*t^8 - 324*t^4 - 27, 54*t^24 -
1944*t^20 + 3726*t^16 - 3726*t^8 + 1944*t^4 - 54 ], [ 0, 0, 0,
-108*t^14 + 216*t^12 - 1620*t^10 + 3024*t^8 - 1620*t^6 + 216*t^4 -
108*t^2, 432*t^21 - 1296*t^19 - 12960*t^17 + 42336*t^15 - 57024*t^13 +
57024*t^11 - 42336*t^9 + 12960*t^7 + 1296*t^5 - 432*t^3 ], [ 0, 0, 0,
-108*t^8 - 1512*t^4 - 108, 432*t^12 - 14256*t^8 - 14256*t^4 + 432 ], [
0, 0, 0, -108*t^16 - 1296*t^12 + 2808*t^8 - 1296*t^4 - 108, 432*t^24 -
15552*t^20 + 29808*t^16 - 29808*t^8 + 15552*t^4 - 432 ], [ 0, 0, 0,
-27*t^18 - 324*t^14 + 702*t^10 - 324*t^6 - 27*t^2, 54*t^27 - 1944*t^23
+ 3726*t^19 - 3726*t^11 + 1944*t^7 - 54*t^3 ], [ 0, 0, 0, -27*t^12 -
54*t^10 - 405*t^8 - 756*t^6 - 405*t^4 - 54*t^2 - 27, -54*t^18 -
162*t^16 + 1620*t^14 + 5292*t^12 + 7128*t^10 + 7128*t^8 + 5292*t^6 +
1620*t^4 - 162*t^2 - 54 ], [ 0, 0, 0, -27*t^8 - 378*t^4 - 27, 54*t^12
- 1782*t^8 - 1782*t^4 + 54 ], [ 0, 0, 0, -27*t^14 + 54*t^12 - 405*t^10
+ 756*t^8 - 405*t^6 + 54*t^4 - 27*t^2, 54*t^21 - 162*t^19 - 1620*t^17
+ 5292*t^15 - 7128*t^13 + 7128*t^11 - 5292*t^9 + 1620*t^7 + 162*t^5 -
54*t^3 ], [ 0, 0, 0, -27*t^10 - 378*t^6 - 27*t^2, 54*t^15 - 1782*t^11
- 1782*t^7 + 54*t^3 ], [ 0, 0, 0, -27648*t^16 + 82944*t^12 + 44928*t^8
+ 5184*t^4 - 108, 1769472*t^24 + 15925248*t^20 + 7630848*t^16 -
476928*t^8 - 62208*t^4 - 432 ], [ 0, 0, 0, -6912*t^16 + 20736*t^12 +
11232*t^8 + 1296*t^4 - 27, 221184*t^24 + 1990656*t^20 + 953856*t^16 -
59616*t^8 - 7776*t^4 - 54 ], [ 0, 0, 0, -1728*t^8 + 6048*t^4 - 108,
27648*t^12 + 228096*t^8 - 57024*t^4 - 432 ], [ 0, 0, 0, -432*t^8 +
1512*t^4 - 27, 3456*t^12 + 28512*t^8 - 7128*t^4 - 54 ], [ 0, 0, 0,
-108*t^10 - 108*t^6 - 108*t^2, 432*t^15 + 648*t^11 - 648*t^7 - 432*t^3
], [ 0, 0, 0, -27*t^10 - 27*t^6 - 27*t^2, 54*t^15 + 81*t^11 - 81*t^7 -
54*t^3 ], [ 0, 0, 0, -108*t^16 - 324*t^12 - 432*t^8 - 324*t^4 - 108,
432*t^24 + 1944*t^20 + 2592*t^16 - 2592*t^8 - 1944*t^4 - 432 ], [ 0,
0, 0, -27*t^16 - 81*t^12 - 108*t^8 - 81*t^4 - 27, 54*t^24 + 243*t^20 +
324*t^16 - 324*t^8 - 243*t^4 - 54 ], [ 0, 0, 0, -108*t^18 - 324*t^14 -
432*t^10 - 324*t^6 - 108*t^2, 432*t^27 + 1944*t^23 + 2592*t^19 -
2592*t^11 - 1944*t^7 - 432*t^3 ], [ 0, 0, 0, -27*t^18 - 81*t^14 -
108*t^10 - 81*t^6 - 27*t^2, 54*t^27 + 243*t^23 + 324*t^19 - 324*t^11 -
243*t^7 - 54*t^3 ], [ 0, 0, 0, -108*t^8 - 108*t^4 - 108, 432*t^12 +
648*t^8 - 648*t^4 - 432 ], [ 0, 0, 0, -27*t^8 - 27*t^4 - 27, 54*t^12 +
81*t^8 - 81*t^4 - 54 ], [ 0, 0, 0, -27*t^8 - 1512*t^4 - 432, -54*t^12
+ 7128*t^8 + 28512*t^4 - 3456 ], [ 0, 0, 0, -27*t^8 - 1512*t^4 - 432,
54*t^12 - 7128*t^8 - 28512*t^4 + 3456 ], [ 0, 0, 0, -27*t^12 -
108*t^10 - 1620*t^8 - 6048*t^6 - 6480*t^4 - 1728*t^2 - 1728, -54*t^18
- 324*t^16 + 6480*t^14 + 42336*t^12 + 114048*t^10 + 228096*t^8 +
338688*t^6 + 207360*t^4 - 41472*t^2 - 27648 ], [ 0, 0, 0, -27*t^12 -
108*t^10 - 1620*t^8 - 6048*t^6 - 6480*t^4 - 1728*t^2 - 1728, 54*t^18 +
324*t^16 - 6480*t^14 - 42336*t^12 - 114048*t^10 - 228096*t^8 -
338688*t^6 - 207360*t^4 + 41472*t^2 + 27648 ], [ 0, 0, 0, -108*t^8 -
6048*t^4 - 1728, -432*t^12 + 57024*t^8 + 228096*t^4 - 27648 ], [ 0, 0,
0, -108*t^8 - 6048*t^4 - 1728, 432*t^12 - 57024*t^8 - 228096*t^4 +
27648 ], [ 0, 0, 0, -27*t^16 - 1296*t^12 + 11232*t^8 - 20736*t^4 -
6912, 54*t^24 - 7776*t^20 + 59616*t^16 - 953856*t^8 + 1990656*t^4 -
221184 ], [ 0, 0, 0, -27*t^12 + 108*t^10 - 1620*t^8 + 6048*t^6 -
6480*t^4 + 1728*t^2 - 1728, 54*t^18 - 324*t^16 - 6480*t^14 +
42336*t^12 - 114048*t^10 + 228096*t^8 - 338688*t^6 + 207360*t^4 +
41472*t^2 - 27648 ], [ 0, 0, 0, -27*t^12 + 108*t^10 - 1620*t^8 +
6048*t^6 - 6480*t^4 + 1728*t^2 - 1728, -54*t^18 + 324*t^16 + 6480*t^14
- 42336*t^12 + 114048*t^10 - 228096*t^8 + 338688*t^6 - 207360*t^4 -
41472*t^2 + 27648 ], [ 0, 0, 0, -108*t^16 - 5184*t^12 + 44928*t^8 -
82944*t^4 - 27648, 432*t^24 - 62208*t^20 + 476928*t^16 - 7630848*t^8 +
15925248*t^4 - 1769472 ], [ 0, 0, 0, -81*t^20 - 3240*t^19 - 61344*t^18
- 730944*t^17 - 6039360*t^16 - 35334144*t^15 - 140009472*t^14 -
298598400*t^13 + 301916160*t^12 + 5037686784*t^11 + 19782696960*t^10 +
40301494272*t^9 + 19322634240*t^8 - 152882380800*t^7 -
573478797312*t^6 - 1157829230592*t^5 - 1583181987840*t^4 -
1532900671488*t^3 - 1029181538304*t^2 - 434865438720*t - 86973087744,
3888*t^29 + 225504*t^28 + 6099840*t^27 + 102083328*t^26 +
1179712512*t^25 + 9913466880*t^24 + 61686448128*t^23 +
279105896448*t^22 + 825962987520*t^21 + 717177618432*t^20 -
7840360562688*t^19 - 52890933657600*t^18 - 184839894466560*t^17 -
363444226228224*t^16 + 2907553809825792*t^14 + 11829753245859840*t^13
+ 27080158032691200*t^12 + 32114116864770048*t^11 -
23500476200779776*t^10 - 216521241400442880*t^9 -
585327488947716096*t^8 - 1034926864516251648*t^7 -
1330563001236848640*t^6 - 1266706664430501888*t^5 -
876889110453682176*t^4 - 419177812973322240*t^3 -
123972135054999552*t^2 - 17099604835172352*t ], [ 0, 0, 0, -81*t^16 -
2592*t^15 - 38016*t^14 - 338688*t^13 - 1942272*t^12 - 6359040*t^11 -
2875392*t^10 + 73875456*t^9 + 334209024*t^8 + 591003648*t^7 -
184025088*t^6 - 3255828480*t^5 - 7955546112*t^4 - 11098128384*t^3 -
9965666304*t^2 - 5435817984*t - 1358954496, 3888*t^23 + 178848*t^22 +
3673728*t^21 + 44126208*t^20 + 337665024*t^19 + 1637922816*t^18 +
4167770112*t^17 - 3609722880*t^16 - 80372957184*t^15 -
350758895616*t^14 - 751756640256*t^13 + 6014053122048*t^11 +
22448569319424*t^10 + 41150954078208*t^9 + 14785424916480*t^8 -
136569491030016*t^7 - 429371638677504*t^6 - 708134880411648*t^5 -
740314922876928*t^4 - 493079425449984*t^3 - 192036577738752*t^2 -
33397665693696*t ], [ 0, 0, 0, -81*t^12 - 1944*t^11 - 25056*t^10 -
216000*t^9 - 1321920*t^8 - 5889024*t^7 - 19353600*t^6 - 47112192*t^5 -
84602880*t^4 - 110592000*t^3 - 102629376*t^2 - 63700992*t - 21233664,
3888*t^17 + 132192*t^16 + 2180736*t^15 + 23120640*t^14 +
176504832*t^13 + 1031546880*t^12 + 4790181888*t^11 + 18087100416*t^10
+ 56251514880*t^9 + 144696803328*t^8 + 306571640832*t^7 +
528152002560*t^6 + 722963791872*t^5 + 757617131520*t^4 +
571666857984*t^3 + 277226717184*t^2 + 65229815808*t ], [ 0, 0, 0,
-324*t^8 - 5184*t^7 - 48384*t^6 - 290304*t^5 - 1064448*t^4 -
2322432*t^3 - 3096576*t^2 - 2654208*t - 1327104, 31104*t^11 +
684288*t^10 + 6994944*t^9 + 43794432*t^8 + 189775872*t^7 +
613122048*t^6 + 1518206976*t^5 + 2802843648*t^4 + 3581411328*t^3 +
2802843648*t^2 + 1019215872*t ], [ 0, 0, 0, -81*t^8 - 1296*t^7 -
12096*t^6 - 72576*t^5 - 266112*t^4 - 580608*t^3 - 774144*t^2 -
663552*t - 331776, 3888*t^11 + 85536*t^10 + 874368*t^9 + 5474304*t^8 +
23721984*t^7 + 76640256*t^6 + 189775872*t^5 + 350355456*t^4 +
447676416*t^3 + 350355456*t^2 + 127401984*t ], [ 0, 0, 0, -324*t^16 -
10368*t^15 - 152064*t^14 - 1354752*t^13 - 7769088*t^12 - 25436160*t^11
- 11501568*t^10 + 295501824*t^9 + 1336836096*t^8 + 2364014592*t^7 -
736100352*t^6 - 13023313920*t^5 - 31822184448*t^4 - 44392513536*t^3 -
39862665216*t^2 - 21743271936*t - 5435817984, 31104*t^23 +
1430784*t^22 + 29389824*t^21 + 353009664*t^20 + 2701320192*t^19 +
13103382528*t^18 + 33342160896*t^17 - 28877783040*t^16 -
642983657472*t^15 - 2806071164928*t^14 - 6014053122048*t^13 +
48112424976384*t^11 + 179588554555392*t^10 + 329207632625664*t^9 +
118283399331840*t^8 - 1092555928240128*t^7 - 3434973109420032*t^6 -
5665079043293184*t^5 - 5922519383015424*t^4 - 3944635403599872*t^3 -
1536292621910016*t^2 - 267181325549568*t ], [ 0, 0, 0, -108*t^8 -
432*t^4 - 1728, 432*t^12 + 2592*t^8 - 10368*t^4 - 27648 ], [ 0, 0, 0,
-27*t^16 - 324*t^12 - 1728*t^8 - 5184*t^4 - 6912, 54*t^24 + 972*t^20 +
5184*t^16 - 82944*t^8 - 248832*t^4 - 221184 ], [ 0, 0, 0, -27*t^8 -
108*t^4 - 432, 54*t^12 + 324*t^8 - 1296*t^4 - 3456 ], [ 0, 0, 0,
-108*t^16 - 1296*t^12 - 6912*t^8 - 20736*t^4 - 27648, 432*t^24 +
7776*t^20 + 41472*t^16 - 663552*t^8 - 1990656*t^4 - 1769472 ], [ 0, 0,
0, -7077888*t^12 + 51314688*t^10 - 1658880*t^8 - 2045952*t^6 -
25920*t^4 + 12528*t^2 - 27, -7247757312*t^18 - 116870086656*t^16 +
74742497280*t^14 + 13872660480*t^12 - 2218917888*t^10 - 277364736*t^8
+ 27095040*t^6 + 2280960*t^4 - 55728*t^2 - 54 ], [ 0, 0, 0,
-1811939328*t^16 + 13589544960*t^14 - 3737124864*t^12 - 212336640*t^10
+ 117669888*t^8 - 3317760*t^6 - 912384*t^4 + 51840*t^2 - 108,
29686813949952*t^24 + 467567319711744*t^22 - 484266152558592*t^20 +
80363133075456*t^18 + 15111573995520*t^16 - 4337782751232*t^14 +
67777855488*t^10 - 3689349120*t^8 - 306561024*t^6 + 28864512*t^4 -
435456*t^2 - 432 ], [ 0, 0, 0, -7077888*t^12 + 51314688*t^10 -
1658880*t^8 - 2045952*t^6 - 25920*t^4 + 12528*t^2 - 27,
7247757312*t^18 + 116870086656*t^16 - 74742497280*t^14 -
13872660480*t^12 + 2218917888*t^10 + 277364736*t^8 - 27095040*t^6 -
2280960*t^4 + 55728*t^2 + 54 ], [ 0, 0, 0, -452984832*t^16 +
3397386240*t^14 - 934281216*t^12 - 53084160*t^10 + 29417472*t^8 -
829440*t^6 - 228096*t^4 + 12960*t^2 - 27, 3710851743744*t^24 +
58445914963968*t^22 - 60533269069824*t^20 + 10045391634432*t^18 +
1888946749440*t^16 - 542222843904*t^14 + 8472231936*t^10 -
461168640*t^8 - 38320128*t^6 + 3608064*t^4 - 54432*t^2 - 54 ], [ 0, 0,
0, -7077888*t^12 + 54853632*t^10 - 28200960*t^8 + 5363712*t^6 -
440640*t^4 + 13392*t^2 - 27, 7247757312*t^18 + 111434268672*t^16 -
160356630528*t^14 + 74516004864*t^12 - 16955080704*t^10 +
2119385088*t^8 - 145539072*t^6 + 4893696*t^4 - 53136*t^2 - 54 ], [ 0,
0, 0, -442368*t^8 + 3317760*t^6 - 926208*t^4 + 51840*t^2 - 108,
-113246208*t^12 - 1783627776*t^10 + 1842020352*t^8 - 390168576*t^6 +
28781568*t^4 - 435456*t^2 - 432 ], [ 0, 0, 0, -442368*t^8 +
3317760*t^6 - 926208*t^4 + 51840*t^2 - 108, 113246208*t^12 +
1783627776*t^10 - 1842020352*t^8 + 390168576*t^6 - 28781568*t^4 +
435456*t^2 + 432 ], [ 0, 0, 0, -7077888*t^12 + 54853632*t^10 -
28200960*t^8 + 5363712*t^6 - 440640*t^4 + 13392*t^2 - 27,
-7247757312*t^18 - 111434268672*t^16 + 160356630528*t^14 -
74516004864*t^12 + 16955080704*t^10 - 2119385088*t^8 + 145539072*t^6 -
4893696*t^4 + 53136*t^2 + 54 ], [ 0, 0, 0, -110592*t^8 + 829440*t^6 -
231552*t^4 + 12960*t^2 - 27, 14155776*t^12 + 222953472*t^10 -
230252544*t^8 + 48771072*t^6 - 3597696*t^4 + 54432*t^2 + 54 ], [ 0, 0,
0, -110592*t^8 + 829440*t^6 - 231552*t^4 + 12960*t^2 - 27,
-14155776*t^12 - 222953472*t^10 + 230252544*t^8 - 48771072*t^6 +
3597696*t^4 - 54432*t^2 - 54 ], [ 0, 0, 0, -2430*t^20 + 47952*t^19 +
227664*t^18 - 482112*t^17 + 3227040*t^16 + 13353984*t^15 +
3898368*t^14 + 67350528*t^13 - 210263040*t^12 + 1371561984*t^11 -
1515331584*t^10 - 5486247936*t^9 - 3364208640*t^8 - 4310433792*t^7 +
997982208*t^6 - 13674479616*t^5 + 13217955840*t^4 + 7898923008*t^3 +
14920187904*t^2 - 12570329088*t - 2548039680, -198288*t^30 -
933120*t^29 - 3810240*t^28 - 5197824*t^27 + 178972416*t^26 -
731566080*t^25 - 952777728*t^24 + 33570422784*t^23 - 32805015552*t^22
- 60436316160*t^21 + 1389027999744*t^20 - 91814363136*t^19 +
1315314008064*t^18 + 11691680071680*t^17 + 27920845504512*t^16 +
111683382018048*t^14 - 187066881146880*t^13 + 84180096516096*t^12 +
23504476962816*t^11 + 1422364671737856*t^10 + 247547150991360*t^9 -
537477374803968*t^8 - 2200071227572224*t^7 - 249764964728832*t^6 +
767102633902080*t^5 + 750664720318464*t^4 + 87205015977984*t^3 -
255700877967360*t^2 + 250482492702720*t - 212910118797312 ], [ 0, 0,
0, -270*t^12 + 3888*t^11 + 52272*t^10 + 129600*t^9 - 64800*t^8 -
870912*t^7 - 2225664*t^6 + 3483648*t^5 - 1036800*t^4 - 8294400*t^3 +
13381632*t^2 - 3981312*t - 1105920, -7344*t^18 - 93312*t^17 -
554688*t^16 - 1244160*t^15 + 12856320*t^14 + 90574848*t^13 +
98316288*t^12 - 258785280*t^11 - 80289792*t^10 - 321159168*t^8 +
4140564480*t^7 + 6292242432*t^6 - 23187161088*t^5 + 13164871680*t^4 +
5096079360*t^3 - 9088008192*t^2 + 6115295232*t - 1925185536 ], [ 0, 0,
0, -2430*t^20 + 47952*t^19 + 227664*t^18 - 482112*t^17 + 3227040*t^16
+ 13353984*t^15 + 3898368*t^14 + 67350528*t^13 - 210263040*t^12 +
1371561984*t^11 - 1515331584*t^10 - 5486247936*t^9 - 3364208640*t^8 -
4310433792*t^7 + 997982208*t^6 - 13674479616*t^5 + 13217955840*t^4 +
7898923008*t^3 + 14920187904*t^2 - 12570329088*t - 2548039680,
198288*t^30 + 933120*t^29 + 3810240*t^28 + 5197824*t^27 -
178972416*t^26 + 731566080*t^25 + 952777728*t^24 - 33570422784*t^23 +
32805015552*t^22 + 60436316160*t^21 - 1389027999744*t^20 +
91814363136*t^19 - 1315314008064*t^18 - 11691680071680*t^17 -
27920845504512*t^16 - 111683382018048*t^14 + 187066881146880*t^13 -
84180096516096*t^12 - 23504476962816*t^11 - 1422364671737856*t^10 -
247547150991360*t^9 + 537477374803968*t^8 + 2200071227572224*t^7 +
249764964728832*t^6 - 767102633902080*t^5 - 750664720318464*t^4 -
87205015977984*t^3 + 255700877967360*t^2 - 250482492702720*t +
212910118797312 ], [ 0, 0, 0, -270*t^12 + 3888*t^11 + 52272*t^10 +
129600*t^9 - 64800*t^8 - 870912*t^7 - 2225664*t^6 + 3483648*t^5 -
1036800*t^4 - 8294400*t^3 + 13381632*t^2 - 3981312*t - 1105920,
7344*t^18 + 93312*t^17 + 554688*t^16 + 1244160*t^15 - 12856320*t^14 -
90574848*t^13 - 98316288*t^12 + 258785280*t^11 + 80289792*t^10 +
321159168*t^8 - 4140564480*t^7 - 6292242432*t^6 + 23187161088*t^5 -
13164871680*t^4 - 5096079360*t^3 + 9088008192*t^2 - 6115295232*t +
1925185536 ], [ 0, 0, 0, -27*t^12 - 3132*t^10 - 1620*t^8 + 31968*t^6 -
6480*t^4 - 50112*t^2 - 1728, -54*t^18 + 13932*t^16 + 142560*t^14 -
423360*t^12 - 1083456*t^10 + 2166912*t^8 + 3386880*t^6 - 4561920*t^4 -
1783296*t^2 + 27648 ], [ 0, 0, 0, -27*t^12 - 3132*t^10 - 1620*t^8 +
31968*t^6 - 6480*t^4 - 50112*t^2 - 1728, 54*t^18 - 13932*t^16 -
142560*t^14 + 423360*t^12 + 1083456*t^10 - 2166912*t^8 - 3386880*t^6 +
4561920*t^4 + 1783296*t^2 - 27648 ], [ 0, 0, 0, -108*t^16 - 12960*t^14
- 57024*t^12 + 51840*t^10 + 459648*t^8 + 207360*t^6 - 912384*t^4 -
829440*t^2 - 27648, 432*t^24 - 108864*t^22 - 1804032*t^20 -
4790016*t^18 + 14411520*t^16 + 66189312*t^14 - 264757248*t^10 -
230584320*t^8 + 306561024*t^6 + 461832192*t^4 + 111476736*t^2 -
1769472 ], [ 0, 0, 0, -27*t^16 - 3240*t^14 - 14256*t^12 + 12960*t^10 +
114912*t^8 + 51840*t^6 - 228096*t^4 - 207360*t^2 - 6912, 54*t^24 -
13608*t^22 - 225504*t^20 - 598752*t^18 + 1801440*t^16 + 8273664*t^14 -
33094656*t^10 - 28823040*t^8 + 38320128*t^6 + 57729024*t^4 +
13934592*t^2 - 221184 ], [ 0, 0, 0, -27*t^12 - 3348*t^10 - 27540*t^8 -
83808*t^6 - 110160*t^4 - 53568*t^2 - 1728, -54*t^18 + 13284*t^16 +
305856*t^14 + 2274048*t^12 + 8278848*t^10 + 16557696*t^8 +
18192384*t^6 + 9787392*t^4 + 1700352*t^2 - 27648 ], [ 0, 0, 0,
-27*t^12 - 3348*t^10 - 27540*t^8 - 83808*t^6 - 110160*t^4 - 53568*t^2
- 1728, 54*t^18 - 13284*t^16 - 305856*t^14 - 2274048*t^12 -
8278848*t^10 - 16557696*t^8 - 18192384*t^6 - 9787392*t^4 - 1700352*t^2
+ 27648 ], [ 0, 0, 0, -108*t^8 - 12960*t^6 - 57888*t^4 - 51840*t^2 -
1728, -432*t^12 + 108864*t^10 + 1798848*t^8 + 6096384*t^6 +
7195392*t^4 + 1741824*t^2 - 27648 ], [ 0, 0, 0, -108*t^8 - 12960*t^6 -
57888*t^4 - 51840*t^2 - 1728, 432*t^12 - 108864*t^10 - 1798848*t^8 -
6096384*t^6 - 7195392*t^4 - 1741824*t^2 + 27648 ], [ 0, 0, 0, -27*t^8
- 3240*t^6 - 14472*t^4 - 12960*t^2 - 432, 54*t^12 - 13608*t^10 -
224856*t^8 - 762048*t^6 - 899424*t^4 - 217728*t^2 + 3456 ], [ 0, 0, 0,
-27*t^8 - 3240*t^6 - 14472*t^4 - 12960*t^2 - 432, -54*t^12 +
13608*t^10 + 224856*t^8 + 762048*t^6 + 899424*t^4 + 217728*t^2 - 3456
], [ 0, 0, 0, -891*t^12 - 7452*t^11 - 28944*t^10 - 65880*t^9 -
92340*t^8 - 80352*t^7 - 69120*t^6 - 160704*t^5 - 369360*t^4 -
527040*t^3 - 463104*t^2 - 238464*t - 57024, -10206*t^18 - 127332*t^17
- 749412*t^16 - 2666736*t^15 - 5957712*t^14 - 6676992*t^13 +
6537888*t^12 + 49180608*t^11 + 125846784*t^10 + 211545216*t^9 +
251693568*t^8 + 196722432*t^7 + 52303104*t^6 - 106831872*t^5 -
190646784*t^4 - 170671104*t^3 - 95924736*t^2 - 32596992*t - 5225472 ],
[ 0, 0, 0, -3564*t^8 - 15552*t^7 - 25056*t^6 - 10368*t^5 + 11232*t^4 -
20736*t^3 - 100224*t^2 - 124416*t - 57024, 81648*t^12 + 528768*t^11 +
1353024*t^10 + 1085184*t^9 - 3063744*t^8 - 12026880*t^7 - 21772800*t^6
- 24053760*t^5 - 12254976*t^4 + 8681472*t^3 + 21648384*t^2 +
16920576*t + 5225472 ], [ 0, 0, 0, -891*t^12 - 7452*t^11 - 28944*t^10
- 65880*t^9 - 92340*t^8 - 80352*t^7 - 69120*t^6 - 160704*t^5 -
369360*t^4 - 527040*t^3 - 463104*t^2 - 238464*t - 57024, 10206*t^18 +
127332*t^17 + 749412*t^16 + 2666736*t^15 + 5957712*t^14 + 6676992*t^13
- 6537888*t^12 - 49180608*t^11 - 125846784*t^10 - 211545216*t^9 -
251693568*t^8 - 196722432*t^7 - 52303104*t^6 + 106831872*t^5 +
190646784*t^4 + 170671104*t^3 + 95924736*t^2 + 32596992*t + 5225472 ],
[ 0, 0, 0, -3564*t^8 - 15552*t^7 - 25056*t^6 - 10368*t^5 + 11232*t^4 -
20736*t^3 - 100224*t^2 - 124416*t - 57024, -81648*t^12 - 528768*t^11 -
1353024*t^10 - 1085184*t^9 + 3063744*t^8 + 12026880*t^7 + 21772800*t^6
+ 24053760*t^5 + 12254976*t^4 - 8681472*t^3 - 21648384*t^2 -
16920576*t - 5225472 ], [ 0, 0, 0, -891*t^20 - 14580*t^19 -
102816*t^18 - 402408*t^17 - 899100*t^16 - 844992*t^15 + 1130112*t^14 +
4613760*t^13 + 4937760*t^12 - 2395008*t^11 - 11059200*t^10 -
4790016*t^9 + 19751040*t^8 + 36910080*t^7 + 18081792*t^6 -
27039744*t^5 - 57542400*t^4 - 51508224*t^3 - 26320896*t^2 - 7464960*t
- 912384, 10206*t^30 + 249804*t^29 + 2767284*t^28 + 18179856*t^27 +
76941576*t^26 + 205274736*t^25 + 252226224*t^24 - 462765312*t^23 -
3127592736*t^22 - 7745626944*t^21 - 9546330816*t^20 + 2319881472*t^19
+ 36116316288*t^18 + 78452064000*t^17 + 83647924992*t^16 -
167295849984*t^14 - 313808256000*t^13 - 288930530304*t^12 -
37118103552*t^11 + 305482586112*t^10 + 495720124416*t^9 +
400331870208*t^8 + 118467919872*t^7 - 129139826688*t^6 -
210201329664*t^5 - 157576347648*t^4 - 74464690176*t^3 -
22669590528*t^2 - 4092788736*t - 334430208 ], [ 0, 0, 0, -891*t^16 -
11016*t^15 - 51624*t^14 - 100656*t^13 + 8208*t^12 + 384480*t^11 +
538272*t^10 - 278208*t^9 - 1364256*t^8 - 556416*t^7 + 2153088*t^6 +
3075840*t^5 + 131328*t^4 - 3220992*t^3 - 3303936*t^2 - 1410048*t -
228096, 10206*t^24 + 188568*t^23 + 1452168*t^22 + 5746032*t^21 +
9924768*t^20 - 11205216*t^19 - 99946656*t^18 - 222201792*t^17 -
143589024*t^16 + 433057536*t^15 + 1337990400*t^14 + 1579378176*t^13 -
3158756352*t^11 - 5351961600*t^10 - 3464460288*t^9 + 2297424384*t^8 +
7110457344*t^7 + 6396585984*t^6 + 1434267648*t^5 - 2540740608*t^4 -
2941968384*t^3 - 1487020032*t^2 - 386187264*t - 41803776 ], [ 0, 0, 0,
-891*t^20 - 14580*t^19 - 102816*t^18 - 402408*t^17 - 899100*t^16 -
844992*t^15 + 1130112*t^14 + 4613760*t^13 + 4937760*t^12 -
2395008*t^11 - 11059200*t^10 - 4790016*t^9 + 19751040*t^8 +
36910080*t^7 + 18081792*t^6 - 27039744*t^5 - 57542400*t^4 -
51508224*t^3 - 26320896*t^2 - 7464960*t - 912384, -10206*t^30 -
249804*t^29 - 2767284*t^28 - 18179856*t^27 - 76941576*t^26 -
205274736*t^25 - 252226224*t^24 + 462765312*t^23 + 3127592736*t^22 +
7745626944*t^21 + 9546330816*t^20 - 2319881472*t^19 - 36116316288*t^18
- 78452064000*t^17 - 83647924992*t^16 + 167295849984*t^14 +
313808256000*t^13 + 288930530304*t^12 + 37118103552*t^11 -
305482586112*t^10 - 495720124416*t^9 - 400331870208*t^8 -
118467919872*t^7 + 129139826688*t^6 + 210201329664*t^5 +
157576347648*t^4 + 74464690176*t^3 + 22669590528*t^2 + 4092788736*t +
334430208 ], [ 0, 0, 0, -891*t^8 - 3888*t^7 - 6264*t^6 - 2592*t^5 +
2808*t^4 - 5184*t^3 - 25056*t^2 - 31104*t - 14256, 10206*t^12 +
66096*t^11 + 169128*t^10 + 135648*t^9 - 382968*t^8 - 1503360*t^7 -
2721600*t^6 - 3006720*t^5 - 1531872*t^4 + 1085184*t^3 + 2706048*t^2 +
2115072*t + 653184 ], [ 0, 0, 0, -891*t^8 - 3888*t^7 - 6264*t^6 -
2592*t^5 + 2808*t^4 - 5184*t^3 - 25056*t^2 - 31104*t - 14256,
-10206*t^12 - 66096*t^11 - 169128*t^10 - 135648*t^9 + 382968*t^8 +
1503360*t^7 + 2721600*t^6 + 3006720*t^5 + 1531872*t^4 - 1085184*t^3 -
2706048*t^2 - 2115072*t - 653184 ], [ 0, 0, 0, -3564*t^16 - 44064*t^15
- 206496*t^14 - 402624*t^13 + 32832*t^12 + 1537920*t^11 + 2153088*t^10
- 1112832*t^9 - 5457024*t^8 - 2225664*t^7 + 8612352*t^6 + 12303360*t^5
+ 525312*t^4 - 12883968*t^3 - 13215744*t^2 - 5640192*t - 912384,
81648*t^24 + 1508544*t^23 + 11617344*t^22 + 45968256*t^21 +
79398144*t^20 - 89641728*t^19 - 799573248*t^18 - 1777614336*t^17 -
1148712192*t^16 + 3464460288*t^15 + 10703923200*t^14 +
12635025408*t^13 - 25270050816*t^11 - 42815692800*t^10 -
27715682304*t^9 + 18379395072*t^8 + 56883658752*t^7 + 51172687872*t^6
+ 11474141184*t^5 - 20325924864*t^4 - 23535747072*t^3 -
11896160256*t^2 - 3089498112*t - 334430208 ], [ 0, 0, 0, -1890*t^12 -
24624*t^11 - 137592*t^10 - 453600*t^9 - 1046520*t^8 - 1897344*t^7 -
2894400*t^6 - 3794688*t^5 - 4186080*t^4 - 3628800*t^3 - 2201472*t^2 -
787968*t - 120960, -31536*t^18 - 614304*t^17 - 5393952*t^16 -
28646784*t^15 - 104302080*t^14 - 278256384*t^13 - 558157824*t^12 -
820772352*t^11 - 755910144*t^10 + 1511820288*t^8 + 3283089408*t^7 +
4465262592*t^6 + 4452102144*t^5 + 3337666560*t^4 + 1833394176*t^3 +
690425856*t^2 + 157261824*t + 16146432 ], [ 0, 0, 0, -17010*t^20 -
312336*t^19 - 2677320*t^18 - 14580000*t^17 - 57887784*t^16 -
181336320*t^15 - 470240640*t^14 - 1039993344*t^13 - 1999702080*t^12 -
3384737280*t^11 - 5078502144*t^10 - 6769474560*t^9 - 7998808320*t^8 -
8319946752*t^7 - 7523850240*t^6 - 5802762240*t^5 - 3704818176*t^4 -
1866240000*t^3 - 685393920*t^2 - 159916032*t - 17418240, 851472*t^30 +
23397984*t^29 + 306708768*t^28 + 2575694592*t^27 + 15696244800*t^26 +
74432151936*t^25 + 286950169728*t^24 + 925319614464*t^23 +
2541686936832*t^22 + 6010028969472*t^21 + 12279188224512*t^20 +
21583379460096*t^19 + 32068140143616*t^18 + 38352876779520*t^17 +
31244579629056*t^16 - 62489159258112*t^14 - 153411507118080*t^13 -
256545121148928*t^12 - 345334071361536*t^11 - 392934023184384*t^10 -
384641854046208*t^9 - 325335927914496*t^8 - 236881821302784*t^7 -
146918486900736*t^6 - 76218523582464*t^5 - 32145909350400*t^4 -
10550045048832*t^3 - 2512558227456*t^2 - 383352569856*t - 27901034496
], [ 0, 0, 0, -1890*t^12 - 24624*t^11 - 137592*t^10 - 453600*t^9 -
1046520*t^8 - 1897344*t^7 - 2894400*t^6 - 3794688*t^5 - 4186080*t^4 -
3628800*t^3 - 2201472*t^2 - 787968*t - 120960, 31536*t^18 +
614304*t^17 + 5393952*t^16 + 28646784*t^15 + 104302080*t^14 +
278256384*t^13 + 558157824*t^12 + 820772352*t^11 + 755910144*t^10 -
1511820288*t^8 - 3283089408*t^7 - 4465262592*t^6 - 4452102144*t^5 -
3337666560*t^4 - 1833394176*t^3 - 690425856*t^2 - 157261824*t -
16146432 ], [ 0, 0, 0, -17010*t^20 - 312336*t^19 - 2677320*t^18 -
14580000*t^17 - 57887784*t^16 - 181336320*t^15 - 470240640*t^14 -
1039993344*t^13 - 1999702080*t^12 - 3384737280*t^11 - 5078502144*t^10
- 6769474560*t^9 - 7998808320*t^8 - 8319946752*t^7 - 7523850240*t^6 -
5802762240*t^5 - 3704818176*t^4 - 1866240000*t^3 - 685393920*t^2 -
159916032*t - 17418240, -851472*t^30 - 23397984*t^29 - 306708768*t^28
- 2575694592*t^27 - 15696244800*t^26 - 74432151936*t^25 -
286950169728*t^24 - 925319614464*t^23 - 2541686936832*t^22 -
6010028969472*t^21 - 12279188224512*t^20 - 21583379460096*t^19 -
32068140143616*t^18 - 38352876779520*t^17 - 31244579629056*t^16 +
62489159258112*t^14 + 153411507118080*t^13 + 256545121148928*t^12 +
345334071361536*t^11 + 392934023184384*t^10 + 384641854046208*t^9 +
325335927914496*t^8 + 236881821302784*t^7 + 146918486900736*t^6 +
76218523582464*t^5 + 32145909350400*t^4 + 10550045048832*t^3 +
2512558227456*t^2 + 383352569856*t + 27901034496 ], [ 0, 0, 0,
-108*t^12 - 648*t^10 + 108*t^8 + 4752*t^6 + 7020*t^4 + 2808*t^2 - 108,
-432*t^18 - 3888*t^16 - 28512*t^14 - 127008*t^12 - 285120*t^10 -
300672*t^8 - 102816*t^6 + 44064*t^4 + 29808*t^2 + 432 ], [ 0, 0, 0,
-108*t^12 - 648*t^10 + 108*t^8 + 4752*t^6 + 7020*t^4 + 2808*t^2 - 108,
432*t^18 + 3888*t^16 + 28512*t^14 + 127008*t^12 + 285120*t^10 +
300672*t^8 + 102816*t^6 - 44064*t^4 - 29808*t^2 - 432 ], [ 0, 0, 0,
-27*t^12 - 162*t^10 + 27*t^8 + 1188*t^6 + 1755*t^4 + 702*t^2 - 27,
54*t^18 + 486*t^16 + 3564*t^14 + 15876*t^12 + 35640*t^10 + 37584*t^8 +
12852*t^6 - 5508*t^4 - 3726*t^2 - 54 ], [ 0, 0, 0, -27*t^12 - 162*t^10
+ 27*t^8 + 1188*t^6 + 1755*t^4 + 702*t^2 - 27, -54*t^18 - 486*t^16 -
3564*t^14 - 15876*t^12 - 35640*t^10 - 37584*t^8 - 12852*t^6 + 5508*t^4
+ 3726*t^2 + 54 ], [ 0, 0, 0, -108*t^16 - 1080*t^14 - 2916*t^12 +
2592*t^10 + 26460*t^8 + 49896*t^6 + 39204*t^4 + 10800*t^2 - 432,
432*t^24 + 6480*t^22 + 57024*t^20 + 348192*t^18 + 1420416*t^16 +
3763584*t^14 + 6344352*t^12 + 6461856*t^10 + 3344976*t^8 + 114480*t^6
- 712800*t^4 - 243648*t^2 - 3456 ], [ 0, 0, 0, -108*t^16 - 1080*t^14 -
2916*t^12 + 2592*t^10 + 26460*t^8 + 49896*t^6 + 39204*t^4 + 10800*t^2
- 432, -432*t^24 - 6480*t^22 - 57024*t^20 - 348192*t^18 - 1420416*t^16
- 3763584*t^14 - 6344352*t^12 - 6461856*t^10 - 3344976*t^8 -
114480*t^6 + 712800*t^4 + 243648*t^2 + 3456 ], [ 0, 0, 0, -27*t^16 -
270*t^14 - 729*t^12 + 648*t^10 + 6615*t^8 + 12474*t^6 + 9801*t^4 +
2700*t^2 - 108, -54*t^24 - 810*t^22 - 7128*t^20 - 43524*t^18 -
177552*t^16 - 470448*t^14 - 793044*t^12 - 807732*t^10 - 418122*t^8 -
14310*t^6 + 89100*t^4 + 30456*t^2 + 432 ], [ 0, 0, 0, -27*t^16 -
270*t^14 - 729*t^12 + 648*t^10 + 6615*t^8 + 12474*t^6 + 9801*t^4 +
2700*t^2 - 108, 54*t^24 + 810*t^22 + 7128*t^20 + 43524*t^18 +
177552*t^16 + 470448*t^14 + 793044*t^12 + 807732*t^10 + 418122*t^8 +
14310*t^6 - 89100*t^4 - 30456*t^2 - 432 ], [ 0, 0, 0, -108*t^8 -
432*t^6 + 1080*t^4 + 3024*t^2 - 108, -432*t^12 - 2592*t^10 - 19440*t^8
- 60480*t^6 - 42768*t^4 + 28512*t^2 + 432 ], [ 0, 0, 0, -108*t^8 -
432*t^6 + 1080*t^4 + 3024*t^2 - 108, 432*t^12 + 2592*t^10 + 19440*t^8
+ 60480*t^6 + 42768*t^4 - 28512*t^2 - 432 ], [ 0, 0, 0, -27*t^8 -
108*t^6 + 270*t^4 + 756*t^2 - 27, 54*t^12 + 324*t^10 + 2430*t^8 +
7560*t^6 + 5346*t^4 - 3564*t^2 - 54 ], [ 0, 0, 0, -27*t^8 - 108*t^6 +
270*t^4 + 756*t^2 - 27, -54*t^12 - 324*t^10 - 2430*t^8 - 7560*t^6 -
5346*t^4 + 3564*t^2 + 54 ], [ 0, 0, 0, -108*t^12 - 864*t^10 - 1080*t^8
+ 5616*t^6 + 16308*t^4 + 11664*t^2 - 432, 432*t^18 + 5184*t^16 +
40176*t^14 + 211680*t^12 + 659664*t^10 + 1109376*t^8 + 825552*t^6 -
2592*t^4 - 233280*t^2 - 3456 ], [ 0, 0, 0, -108*t^12 - 864*t^10 -
1080*t^8 + 5616*t^6 + 16308*t^4 + 11664*t^2 - 432, -432*t^18 -
5184*t^16 - 40176*t^14 - 211680*t^12 - 659664*t^10 - 1109376*t^8 -
825552*t^6 + 2592*t^4 + 233280*t^2 + 3456 ], [ 0, 0, 0, -27*t^12 -
216*t^10 - 270*t^8 + 1404*t^6 + 4077*t^4 + 2916*t^2 - 108, 54*t^18 +
648*t^16 + 5022*t^14 + 26460*t^12 + 82458*t^10 + 138672*t^8 +
103194*t^6 - 324*t^4 - 29160*t^2 - 432 ], [ 0, 0, 0, -27*t^12 -
216*t^10 - 270*t^8 + 1404*t^6 + 4077*t^4 + 2916*t^2 - 108, -54*t^18 -
648*t^16 - 5022*t^14 - 26460*t^12 - 82458*t^10 - 138672*t^8 -
103194*t^6 + 324*t^4 + 29160*t^2 + 432 ], [ 0, 0, 0, -27*t^12 +
324*t^10 + 108*t^8 - 9504*t^6 + 28080*t^4 - 22464*t^2 - 1728, -54*t^18
+ 972*t^16 - 14256*t^14 + 127008*t^12 - 570240*t^10 + 1202688*t^8 -
822528*t^6 - 705024*t^4 + 953856*t^2 - 27648 ], [ 0, 0, 0, -27*t^12 +
324*t^10 + 108*t^8 - 9504*t^6 + 28080*t^4 - 22464*t^2 - 1728, 54*t^18
- 972*t^16 + 14256*t^14 - 127008*t^12 + 570240*t^10 - 1202688*t^8 +
822528*t^6 + 705024*t^4 - 953856*t^2 + 27648 ], [ 0, 0, 0, -108*t^12 +
1296*t^10 + 432*t^8 - 38016*t^6 + 112320*t^4 - 89856*t^2 - 6912,
-432*t^18 + 7776*t^16 - 114048*t^14 + 1016064*t^12 - 4561920*t^10 +
9621504*t^8 - 6580224*t^6 - 5640192*t^4 + 7630848*t^2 - 221184 ], [ 0,
0, 0, -108*t^12 + 1296*t^10 + 432*t^8 - 38016*t^6 + 112320*t^4 -
89856*t^2 - 6912, 432*t^18 - 7776*t^16 + 114048*t^14 - 1016064*t^12 +
4561920*t^10 - 9621504*t^8 + 6580224*t^6 + 5640192*t^4 - 7630848*t^2 +
221184 ], [ 0, 0, 0, -27*t^16 + 540*t^14 - 2916*t^12 - 5184*t^10 +
105840*t^8 - 399168*t^6 + 627264*t^4 - 345600*t^2 - 27648, 54*t^24 -
1620*t^22 + 28512*t^20 - 348192*t^18 + 2840832*t^16 - 15054336*t^14 +
50754816*t^12 - 103389696*t^10 + 107039232*t^8 - 7326720*t^6 -
91238400*t^4 + 62373888*t^2 - 1769472 ], [ 0, 0, 0, -27*t^16 +
540*t^14 - 2916*t^12 - 5184*t^10 + 105840*t^8 - 399168*t^6 +
627264*t^4 - 345600*t^2 - 27648, -54*t^24 + 1620*t^22 - 28512*t^20 +
348192*t^18 - 2840832*t^16 + 15054336*t^14 - 50754816*t^12 +
103389696*t^10 - 107039232*t^8 + 7326720*t^6 + 91238400*t^4 -
62373888*t^2 + 1769472 ], [ 0, 0, 0, -108*t^16 + 2160*t^14 -
11664*t^12 - 20736*t^10 + 423360*t^8 - 1596672*t^6 + 2509056*t^4 -
1382400*t^2 - 110592, 432*t^24 - 12960*t^22 + 228096*t^20 -
2785536*t^18 + 22726656*t^16 - 120434688*t^14 + 406038528*t^12 -
827117568*t^10 + 856313856*t^8 - 58613760*t^6 - 729907200*t^4 +
498991104*t^2 - 14155776 ], [ 0, 0, 0, -108*t^16 + 2160*t^14 -
11664*t^12 - 20736*t^10 + 423360*t^8 - 1596672*t^6 + 2509056*t^4 -
1382400*t^2 - 110592, -432*t^24 + 12960*t^22 - 228096*t^20 +
2785536*t^18 - 22726656*t^16 + 120434688*t^14 - 406038528*t^12 +
827117568*t^10 - 856313856*t^8 + 58613760*t^6 + 729907200*t^4 -
498991104*t^2 + 14155776 ], [ 0, 0, 0, -108*t^8 + 864*t^6 + 4320*t^4 -
24192*t^2 - 1728, 432*t^12 - 5184*t^10 + 77760*t^8 - 483840*t^6 +
684288*t^4 + 912384*t^2 - 27648 ], [ 0, 0, 0, -108*t^8 + 864*t^6 +
4320*t^4 - 24192*t^2 - 1728, -432*t^12 + 5184*t^10 - 77760*t^8 +
483840*t^6 - 684288*t^4 - 912384*t^2 + 27648 ], [ 0, 0, 0, -27*t^8 +
216*t^6 + 1080*t^4 - 6048*t^2 - 432, 54*t^12 - 648*t^10 + 9720*t^8 -
60480*t^6 + 85536*t^4 + 114048*t^2 - 3456 ], [ 0, 0, 0, -27*t^8 +
216*t^6 + 1080*t^4 - 6048*t^2 - 432, -54*t^12 + 648*t^10 - 9720*t^8 +
60480*t^6 - 85536*t^4 - 114048*t^2 + 3456 ], [ 0, 0, 0, -27*t^12 +
432*t^10 - 1080*t^8 - 11232*t^6 + 65232*t^4 - 93312*t^2 - 6912,
54*t^18 - 1296*t^16 + 20088*t^14 - 211680*t^12 + 1319328*t^10 -
4437504*t^8 + 6604416*t^6 + 41472*t^4 - 7464960*t^2 + 221184 ], [ 0,
0, 0, -108*t^12 + 1728*t^10 - 4320*t^8 - 44928*t^6 + 260928*t^4 -
373248*t^2 - 27648, -432*t^18 + 10368*t^16 - 160704*t^14 +
1693440*t^12 - 10554624*t^10 + 35500032*t^8 - 52835328*t^6 -
331776*t^4 + 59719680*t^2 - 1769472 ], [ 0, 0, 0, -108*t^12 +
1728*t^10 - 4320*t^8 - 44928*t^6 + 260928*t^4 - 373248*t^2 - 27648,
432*t^18 - 10368*t^16 + 160704*t^14 - 1693440*t^12 + 10554624*t^10 -
35500032*t^8 + 52835328*t^6 + 331776*t^4 - 59719680*t^2 + 1769472 ], [
0, 0, 0, -27*t^12 + 432*t^10 - 1080*t^8 - 11232*t^6 + 65232*t^4 -
93312*t^2 - 6912, -54*t^18 + 1296*t^16 - 20088*t^14 + 211680*t^12 -
1319328*t^10 + 4437504*t^8 - 6604416*t^6 - 41472*t^4 + 7464960*t^2 -
221184 ], [ 0, 0, 0, -27*t^12 - 324*t^10 + 108*t^8 + 9504*t^6 +
28080*t^4 + 22464*t^2 - 1728, 54*t^18 + 972*t^16 + 14256*t^14 +
127008*t^12 + 570240*t^10 + 1202688*t^8 + 822528*t^6 - 705024*t^4 -
953856*t^2 - 27648 ], [ 0, 0, 0, -108*t^12 - 1296*t^10 + 432*t^8 +
38016*t^6 + 112320*t^4 + 89856*t^2 - 6912, -432*t^18 - 7776*t^16 -
114048*t^14 - 1016064*t^12 - 4561920*t^10 - 9621504*t^8 - 6580224*t^6
+ 5640192*t^4 + 7630848*t^2 + 221184 ], [ 0, 0, 0, -27*t^12 - 324*t^10
+ 108*t^8 + 9504*t^6 + 28080*t^4 + 22464*t^2 - 1728, -54*t^18 -
972*t^16 - 14256*t^14 - 127008*t^12 - 570240*t^10 - 1202688*t^8 -
822528*t^6 + 705024*t^4 + 953856*t^2 + 27648 ], [ 0, 0, 0, -108*t^12 -
1296*t^10 + 432*t^8 + 38016*t^6 + 112320*t^4 + 89856*t^2 - 6912,
432*t^18 + 7776*t^16 + 114048*t^14 + 1016064*t^12 + 4561920*t^10 +
9621504*t^8 + 6580224*t^6 - 5640192*t^4 - 7630848*t^2 - 221184 ], [ 0,
0, 0, -27*t^16 - 540*t^14 - 2916*t^12 + 5184*t^10 + 105840*t^8 +
399168*t^6 + 627264*t^4 + 345600*t^2 - 27648, -54*t^24 - 1620*t^22 -
28512*t^20 - 348192*t^18 - 2840832*t^16 - 15054336*t^14 -
50754816*t^12 - 103389696*t^10 - 107039232*t^8 - 7326720*t^6 +
91238400*t^4 + 62373888*t^2 + 1769472 ], [ 0, 0, 0, -108*t^16 -
2160*t^14 - 11664*t^12 + 20736*t^10 + 423360*t^8 + 1596672*t^6 +
2509056*t^4 + 1382400*t^2 - 110592, 432*t^24 + 12960*t^22 +
228096*t^20 + 2785536*t^18 + 22726656*t^16 + 120434688*t^14 +
406038528*t^12 + 827117568*t^10 + 856313856*t^8 + 58613760*t^6 -
729907200*t^4 - 498991104*t^2 - 14155776 ], [ 0, 0, 0, -27*t^16 -
540*t^14 - 2916*t^12 + 5184*t^10 + 105840*t^8 + 399168*t^6 +
627264*t^4 + 345600*t^2 - 27648, 54*t^24 + 1620*t^22 + 28512*t^20 +
348192*t^18 + 2840832*t^16 + 15054336*t^14 + 50754816*t^12 +
103389696*t^10 + 107039232*t^8 + 7326720*t^6 - 91238400*t^4 -
62373888*t^2 - 1769472 ], [ 0, 0, 0, -108*t^16 - 2160*t^14 -
11664*t^12 + 20736*t^10 + 423360*t^8 + 1596672*t^6 + 2509056*t^4 +
1382400*t^2 - 110592, -432*t^24 - 12960*t^22 - 228096*t^20 -
2785536*t^18 - 22726656*t^16 - 120434688*t^14 - 406038528*t^12 -
827117568*t^10 - 856313856*t^8 - 58613760*t^6 + 729907200*t^4 +
498991104*t^2 + 14155776 ], [ 0, 0, 0, -108*t^8 - 864*t^6 + 4320*t^4 +
24192*t^2 - 1728, 432*t^12 + 5184*t^10 + 77760*t^8 + 483840*t^6 +
684288*t^4 - 912384*t^2 - 27648 ], [ 0, 0, 0, -27*t^8 - 216*t^6 +
1080*t^4 + 6048*t^2 - 432, -54*t^12 - 648*t^10 - 9720*t^8 - 60480*t^6
- 85536*t^4 + 114048*t^2 + 3456 ], [ 0, 0, 0, -108*t^8 - 864*t^6 +
4320*t^4 + 24192*t^2 - 1728, -432*t^12 - 5184*t^10 - 77760*t^8 -
483840*t^6 - 684288*t^4 + 912384*t^2 + 27648 ], [ 0, 0, 0, -27*t^8 -
216*t^6 + 1080*t^4 + 6048*t^2 - 432, 54*t^12 + 648*t^10 + 9720*t^8 +
60480*t^6 + 85536*t^4 - 114048*t^2 - 3456 ], [ 0, 0, 0, -108*t^12 -
1728*t^10 - 4320*t^8 + 44928*t^6 + 260928*t^4 + 373248*t^2 - 27648,
-432*t^18 - 10368*t^16 - 160704*t^14 - 1693440*t^12 - 10554624*t^10 -
35500032*t^8 - 52835328*t^6 + 331776*t^4 + 59719680*t^2 + 1769472 ], [
0, 0, 0, -27*t^12 - 432*t^10 - 1080*t^8 + 11232*t^6 + 65232*t^4 +
93312*t^2 - 6912, 54*t^18 + 1296*t^16 + 20088*t^14 + 211680*t^12 +
1319328*t^10 + 4437504*t^8 + 6604416*t^6 - 41472*t^4 - 7464960*t^2 -
221184 ], [ 0, 0, 0, -27*t^12 - 432*t^10 - 1080*t^8 + 11232*t^6 +
65232*t^4 + 93312*t^2 - 6912, -54*t^18 - 1296*t^16 - 20088*t^14 -
211680*t^12 - 1319328*t^10 - 4437504*t^8 - 6604416*t^6 + 41472*t^4 +
7464960*t^2 + 221184 ], [ 0, 0, 0, -108*t^12 - 1728*t^10 - 4320*t^8 +
44928*t^6 + 260928*t^4 + 373248*t^2 - 27648, 432*t^18 + 10368*t^16 +
160704*t^14 + 1693440*t^12 + 10554624*t^10 + 35500032*t^8 +
52835328*t^6 - 331776*t^4 - 59719680*t^2 - 1769472 ], [ 0, 0, 0,
-6912*t^16 - 34560*t^14 - 72576*t^12 - 82944*t^10 - 56160*t^8 -
23328*t^6 - 6156*t^4 - 1080*t^2 - 108, 221184*t^24 + 1658880*t^22 +
5557248*t^20 + 10962432*t^18 + 14100480*t^16 + 12317184*t^14 +
7318080*t^12 + 2822688*t^10 + 593568*t^8 + 5616*t^6 - 29808*t^4 -
6480*t^2 - 432 ], [ 0, 0, 0, -6912*t^16 - 34560*t^14 - 72576*t^12 -
82944*t^10 - 56160*t^8 - 23328*t^6 - 6156*t^4 - 1080*t^2 - 108,
-221184*t^24 - 1658880*t^22 - 5557248*t^20 - 10962432*t^18 -
14100480*t^16 - 12317184*t^14 - 7318080*t^12 - 2822688*t^10 -
593568*t^8 - 5616*t^6 + 29808*t^4 + 6480*t^2 + 432 ], [ 0, 0, 0,
-432*t^8 - 864*t^6 - 540*t^4 - 108*t^2 - 27, 3456*t^12 + 10368*t^10 +
11664*t^8 + 6048*t^6 + 972*t^4 - 324*t^2 - 54 ], [ 0, 0, 0, -432*t^8 -
864*t^6 - 540*t^4 - 108*t^2 - 27, -3456*t^12 - 10368*t^10 - 11664*t^8
- 6048*t^6 - 972*t^4 + 324*t^2 + 54 ], [ 0, 0, 0, -1728*t^8 - 3456*t^6
- 2160*t^4 - 432*t^2 - 108, 27648*t^12 + 82944*t^10 + 93312*t^8 +
48384*t^6 + 7776*t^4 - 2592*t^2 - 432 ], [ 0, 0, 0, -1728*t^8 -
3456*t^6 - 2160*t^4 - 432*t^2 - 108, -27648*t^12 - 82944*t^10 -
93312*t^8 - 48384*t^6 - 7776*t^4 + 2592*t^2 + 432 ], [ 0, 0, 0,
-1728*t^16 - 8640*t^14 - 18144*t^12 - 20736*t^10 - 14040*t^8 -
5832*t^6 - 1539*t^4 - 270*t^2 - 27, 27648*t^24 + 207360*t^22 +
694656*t^20 + 1370304*t^18 + 1762560*t^16 + 1539648*t^14 + 914760*t^12
+ 352836*t^10 + 74196*t^8 + 702*t^6 - 3726*t^4 - 810*t^2 - 54 ], [ 0,
0, 0, -1728*t^16 - 8640*t^14 - 18144*t^12 - 20736*t^10 - 14040*t^8 -
5832*t^6 - 1539*t^4 - 270*t^2 - 27, -27648*t^24 - 207360*t^22 -
694656*t^20 - 1370304*t^18 - 1762560*t^16 - 1539648*t^14 - 914760*t^12
- 352836*t^10 - 74196*t^8 - 702*t^6 + 3726*t^4 + 810*t^2 + 54 ], [ 0,
0, 0, -6912*t^12 - 20736*t^10 - 24192*t^8 - 13824*t^6 - 4320*t^4 -
864*t^2 - 108, 221184*t^18 + 995328*t^16 + 1907712*t^14 + 2032128*t^12
+ 1285632*t^10 + 456192*t^8 + 60480*t^6 - 12960*t^4 - 5184*t^2 - 432
], [ 0, 0, 0, -6912*t^12 - 20736*t^10 - 24192*t^8 - 13824*t^6 -
4320*t^4 - 864*t^2 - 108, -221184*t^18 - 995328*t^16 - 1907712*t^14 -
2032128*t^12 - 1285632*t^10 - 456192*t^8 - 60480*t^6 + 12960*t^4 +
5184*t^2 + 432 ], [ 0, 0, 0, -432*t^12 - 1728*t^10 - 2700*t^8 -
2052*t^6 - 783*t^4 - 162*t^2 - 27, 3456*t^18 + 20736*t^16 + 53136*t^14
+ 75600*t^12 + 64476*t^10 + 32400*t^8 + 7938*t^6 - 162*t^4 - 486*t^2 -
54 ], [ 0, 0, 0, -432*t^12 - 1728*t^10 - 2700*t^8 - 2052*t^6 - 783*t^4
- 162*t^2 - 27, -3456*t^18 - 20736*t^16 - 53136*t^14 - 75600*t^12 -
64476*t^10 - 32400*t^8 - 7938*t^6 + 162*t^4 + 486*t^2 + 54 ], [ 0, 0,
0, -1728*t^12 - 5184*t^10 - 6048*t^8 - 3456*t^6 - 1080*t^4 - 216*t^2 -
27, -27648*t^18 - 124416*t^16 - 238464*t^14 - 254016*t^12 -
160704*t^10 - 57024*t^8 - 7560*t^6 + 1620*t^4 + 648*t^2 + 54 ], [ 0,
0, 0, -1728*t^12 - 6912*t^10 - 10800*t^8 - 8208*t^6 - 3132*t^4 -
648*t^2 - 108, 27648*t^18 + 165888*t^16 + 425088*t^14 + 604800*t^12 +
515808*t^10 + 259200*t^8 + 63504*t^6 - 1296*t^4 - 3888*t^2 - 432 ], [
0, 0, 0, -1728*t^12 - 6912*t^10 - 10800*t^8 - 8208*t^6 - 3132*t^4 -
648*t^2 - 108, -27648*t^18 - 165888*t^16 - 425088*t^14 - 604800*t^12 -
515808*t^10 - 259200*t^8 - 63504*t^6 + 1296*t^4 + 3888*t^2 + 432 ], [
0, 0, 0, -1728*t^12 - 5184*t^10 - 6048*t^8 - 3456*t^6 - 1080*t^4 -
216*t^2 - 27, 27648*t^18 + 124416*t^16 + 238464*t^14 + 254016*t^12 +
160704*t^10 + 57024*t^8 + 7560*t^6 - 1620*t^4 - 648*t^2 - 54 ], [ 0,
0, 0, -108*t^10 + 6480*t^8 - 14472*t^6 + 6480*t^4 - 108*t^2, 432*t^15
+ 54432*t^13 - 449712*t^11 + 762048*t^9 - 449712*t^7 + 54432*t^5 +
432*t^3 ], [ 0, 0, 0, -27*t^12 + 1566*t^10 - 405*t^8 - 3996*t^6 -
405*t^4 + 1566*t^2 - 27, 54*t^18 + 6966*t^16 - 35640*t^14 - 52920*t^12
+ 67716*t^10 + 67716*t^8 - 52920*t^6 - 35640*t^4 + 6966*t^2 + 54 ], [
0, 0, 0, -108*t^14 + 6696*t^12 - 27540*t^10 + 41904*t^8 - 27540*t^6 +
6696*t^4 - 108*t^2, 432*t^21 + 53136*t^19 - 611712*t^17 + 2274048*t^15
- 4139424*t^13 + 4139424*t^11 - 2274048*t^9 + 611712*t^7 - 53136*t^5 -
432*t^3 ], [ 0, 0, 0, -108*t^14 + 6264*t^12 - 1620*t^10 - 15984*t^8 -
1620*t^6 + 6264*t^4 - 108*t^2, 432*t^21 + 55728*t^19 - 285120*t^17 -
423360*t^15 + 541728*t^13 + 541728*t^11 - 423360*t^9 - 285120*t^7 +
55728*t^5 + 432*t^3 ], [ 0, 0, 0, -27*t^14 + 1674*t^12 - 6885*t^10 +
10476*t^8 - 6885*t^6 + 1674*t^4 - 27*t^2, 54*t^21 + 6642*t^19 -
76464*t^17 + 284256*t^15 - 517428*t^13 + 517428*t^11 - 284256*t^9 +
76464*t^7 - 6642*t^5 - 54*t^3 ], [ 0, 0, 0, -27*t^14 + 1566*t^12 -
405*t^10 - 3996*t^8 - 405*t^6 + 1566*t^4 - 27*t^2, 54*t^21 + 6966*t^19
- 35640*t^17 - 52920*t^15 + 67716*t^13 + 67716*t^11 - 52920*t^9 -
35640*t^7 + 6966*t^5 + 54*t^3 ], [ 0, 0, 0, -108*t^8 + 6480*t^6 -
14472*t^4 + 6480*t^2 - 108, 432*t^12 + 54432*t^10 - 449712*t^8 +
762048*t^6 - 449712*t^4 + 54432*t^2 + 432 ], [ 0, 0, 0, -27*t^8 +
1620*t^6 - 3618*t^4 + 1620*t^2 - 27, 54*t^12 + 6804*t^10 - 56214*t^8 +
95256*t^6 - 56214*t^4 + 6804*t^2 + 54 ], [ 0, 0, 0, -27*t^10 +
1620*t^8 - 3618*t^6 + 1620*t^4 - 27*t^2, 54*t^15 + 6804*t^13 -
56214*t^11 + 95256*t^9 - 56214*t^7 + 6804*t^5 + 54*t^3 ], [ 0, 0, 0,
-27*t^18 + 1620*t^16 - 3564*t^14 - 1620*t^12 + 7182*t^10 - 1620*t^8 -
3564*t^6 + 1620*t^4 - 27*t^2, 54*t^27 + 6804*t^25 - 56376*t^23 +
74844*t^21 + 112590*t^19 - 258552*t^17 + 258552*t^13 - 112590*t^11 -
74844*t^9 + 56376*t^7 - 6804*t^5 - 54*t^3 ], [ 0, 0, 0, -27*t^12 +
1566*t^10 - 405*t^8 - 3996*t^6 - 405*t^4 + 1566*t^2 - 27, -54*t^18 -
6966*t^16 + 35640*t^14 + 52920*t^12 - 67716*t^10 - 67716*t^8 +
52920*t^6 + 35640*t^4 - 6966*t^2 - 54 ], [ 0, 0, 0, 81*t^20 +
3240*t^19 + 52704*t^18 + 419904*t^17 + 1166400*t^16 - 7796736*t^15 -
88915968*t^14 - 351682560*t^13 - 308551680*t^12 + 3084189696*t^11 +
14686617600*t^10 + 24673517568*t^9 - 19747307520*t^8 -
180061470720*t^7 - 364199804928*t^6 - 255483445248*t^5 +
305764761600*t^4 + 880602513408*t^3 + 884226392064*t^2 +
434865438720*t + 86973087744, 3888*t^29 + 225504*t^28 + 5906304*t^27 +
91632384*t^26 + 921922560*t^25 + 6081454080*t^24 + 23728619520*t^23 +
20145438720*t^22 - 356136321024*t^21 - 2225808211968*t^20 -
5205347794944*t^19 + 6482467749888*t^18 + 83018190422016*t^17 +
226494227939328*t^16 - 1811953823514624*t^14 - 5313164187009024*t^13 -
3319023487942656*t^12 + 21321104568090624*t^11 +
72935283489767424*t^10 + 93358999738515456*t^9 - 42248047102525440*t^8
- 398100175068856320*t^7 - 816238949553930240*t^6 -
989906811161149440*t^5 - 787116185069027328*t^4 -
405878120323743744*t^3 - 123972135054999552*t^2 - 17099604835172352*t
], [ 0, 0, 0, 81*t^16 + 2592*t^15 + 29376*t^14 + 96768*t^13 -
718848*t^12 - 7188480*t^11 - 16478208*t^10 + 49987584*t^9 +
324919296*t^8 + 399900672*t^7 - 1054605312*t^6 - 3680501760*t^5 -
2944401408*t^4 + 3170893824*t^3 + 7700742144*t^2 + 5435817984*t +
1358954496, 3888*t^23 + 178848*t^22 + 3480192*t^21 + 35997696*t^20 +
191351808*t^19 + 196466688*t^18 - 3765657600*t^17 - 21297364992*t^16 -
19382796288*t^15 + 225650147328*t^14 + 854697443328*t^13 -
6837579546624*t^11 - 14441609428992*t^10 + 9923991699456*t^9 +
87234007007232*t^8 + 123393068236800*t^7 - 51502563459072*t^6 -
401293826850816*t^5 - 603941121294336*t^4 - 467103463243776*t^3 -
192036577738752*t^2 - 33397665693696*t ], [ 0, 0, 0, 81*t^12 +
1944*t^11 + 16416*t^10 + 43200*t^9 - 233280*t^8 - 2405376*t^7 -
9400320*t^6 - 19243008*t^5 - 14929920*t^4 + 22118400*t^3 +
67239936*t^2 + 63700992*t + 21233664, 3888*t^17 + 132192*t^16 +
1987200*t^15 + 17314560*t^14 + 95219712*t^13 + 327075840*t^12 +
572645376*t^11 - 306561024*t^10 - 3698196480*t^9 - 2452488192*t^8 +
36649304064*t^7 + 167462830080*t^6 + 390019940352*t^5 +
567363502080*t^4 + 520932556800*t^3 + 277226717184*t^2 + 65229815808*t
], [ 0, 0, 0, 324*t^8 + 5184*t^7 + 13824*t^6 - 124416*t^5 - 732672*t^4
- 995328*t^3 + 884736*t^2 + 2654208*t + 1327104, 31104*t^11 +
684288*t^10 + 5446656*t^9 + 15925248*t^8 - 14598144*t^7 -
167215104*t^6 - 116785152*t^5 + 1019215872*t^4 + 2788687872*t^3 +
2802843648*t^2 + 1019215872*t ], [ 0, 0, 0, 324*t^16 + 10368*t^15 +
117504*t^14 + 387072*t^13 - 2875392*t^12 - 28753920*t^11 -
65912832*t^10 + 199950336*t^9 + 1299677184*t^8 + 1599602688*t^7 -
4218421248*t^6 - 14722007040*t^5 - 11777605632*t^4 + 12683575296*t^3 +
30802968576*t^2 + 21743271936*t + 5435817984, 31104*t^23 +
1430784*t^22 + 27841536*t^21 + 287981568*t^20 + 1530814464*t^19 +
1571733504*t^18 - 30125260800*t^17 - 170378919936*t^16 -
155062370304*t^15 + 1805201178624*t^14 + 6837579546624*t^13 -
54700636372992*t^11 - 115532875431936*t^10 + 79391933595648*t^9 +
697872056057856*t^8 + 987144545894400*t^7 - 412020507672576*t^6 -
3210350614806528*t^5 - 4831528970354688*t^4 - 3736827705950208*t^3 -
1536292621910016*t^2 - 267181325549568*t ], [ 0, 0, 0, 81*t^8 +
1296*t^7 + 3456*t^6 - 31104*t^5 - 183168*t^4 - 248832*t^3 + 221184*t^2
+ 663552*t + 331776, 3888*t^11 + 85536*t^10 + 680832*t^9 + 1990656*t^8
- 1824768*t^7 - 20901888*t^6 - 14598144*t^5 + 127401984*t^4 +
348585984*t^3 + 350355456*t^2 + 127401984*t ], [ 0, 0, 0, -27*t^8 +
6048*t^4 - 6912, 54*t^12 + 28512*t^8 - 456192*t^4 - 221184 ], [ 0, 0,
0, -108*t^8 + 24192*t^4 - 27648, 432*t^12 + 228096*t^8 - 3649536*t^4 -
1769472 ], [ 0, 0, 0, -27*t^10 + 6048*t^6 - 6912*t^2, 54*t^15 +
28512*t^11 - 456192*t^7 - 221184*t^3 ], [ 0, 0, 0, -108*t^10 +
24192*t^6 - 27648*t^2, 432*t^15 + 228096*t^11 - 3649536*t^7 -
1769472*t^3 ], [ 0, 0, 0, -27*t^16 + 5184*t^12 + 179712*t^8 +
1327104*t^4 - 1769472, 54*t^24 + 31104*t^20 + 953856*t^16 -
244187136*t^8 - 2038431744*t^4 - 905969664 ], [ 0, 0, 0, -108*t^16 +
20736*t^12 + 718848*t^8 + 5308416*t^4 - 7077888, 432*t^24 +
248832*t^20 + 7630848*t^16 - 1953497088*t^8 - 16307453952*t^4 -
7247757312 ], [ 0, 0, 0, -27*t^18 + 5184*t^14 + 179712*t^10 +
1327104*t^6 - 1769472*t^2, 54*t^27 + 31104*t^23 + 953856*t^19 -
244187136*t^11 - 2038431744*t^7 - 905969664*t^3 ], [ 0, 0, 0,
-108*t^18 + 20736*t^14 + 718848*t^10 + 5308416*t^6 - 7077888*t^2,
432*t^27 + 248832*t^23 + 7630848*t^19 - 1953497088*t^11 -
16307453952*t^7 - 7247757312*t^3 ], [ 0, 0, 0, -108*t^16 + 324*t^12 -
432*t^8 + 324*t^4 - 108, 432*t^24 - 1944*t^20 + 2592*t^16 - 2592*t^8 +
1944*t^4 - 432 ], [ 0, 0, 0, -27*t^14 + 54*t^12 - 54*t^8 + 54*t^4 -
27*t^2, 54*t^21 - 162*t^19 + 81*t^17 + 189*t^15 - 324*t^13 + 324*t^11
- 189*t^9 - 81*t^7 + 162*t^5 - 54*t^3 ], [ 0, 0, 0, -108*t^12 +
216*t^10 - 216*t^6 + 216*t^2 - 108, 432*t^18 - 1296*t^16 + 648*t^14 +
1512*t^12 - 2592*t^10 + 2592*t^8 - 1512*t^6 - 648*t^4 + 1296*t^2 - 432
], [ 0, 0, 0, -27*t^18 + 81*t^14 - 108*t^10 + 81*t^6 - 27*t^2, 54*t^27
- 243*t^23 + 324*t^19 - 324*t^11 + 243*t^7 - 54*t^3 ], [ 0, 0, 0,
-27*t^12 + 54*t^10 - 54*t^6 + 54*t^2 - 27, 54*t^18 - 162*t^16 +
81*t^14 + 189*t^12 - 324*t^10 + 324*t^8 - 189*t^6 - 81*t^4 + 162*t^2 -
54 ], [ 0, 0, 0, -108*t^18 + 324*t^14 - 432*t^10 + 324*t^6 - 108*t^2,
432*t^27 - 1944*t^23 + 2592*t^19 - 2592*t^11 + 1944*t^7 - 432*t^3 ], [
0, 0, 0, -27*t^16 + 81*t^12 - 108*t^8 + 81*t^4 - 27, 54*t^24 -
243*t^20 + 324*t^16 - 324*t^8 + 243*t^4 - 54 ], [ 0, 0, 0, -108*t^14 +
216*t^12 - 216*t^8 + 216*t^4 - 108*t^2, 432*t^21 - 1296*t^19 +
648*t^17 + 1512*t^15 - 2592*t^13 + 2592*t^11 - 1512*t^9 - 648*t^7 +
1296*t^5 - 432*t^3 ], [ 0, 0, 0, -108*t^8 + 108*t^4 - 108, 432*t^12 -
648*t^8 - 648*t^4 + 432 ], [ 0, 0, 0, -27*t^14 - 54*t^12 + 54*t^8 -
54*t^4 - 27*t^2, 54*t^21 + 162*t^19 + 81*t^17 - 189*t^15 - 324*t^13 -
324*t^11 - 189*t^9 + 81*t^7 + 162*t^5 + 54*t^3 ], [ 0, 0, 0, -108*t^8
+ 108*t^4 - 108, -432*t^12 + 648*t^8 + 648*t^4 - 432 ], [ 0, 0, 0,
-108*t^12 - 216*t^10 + 216*t^6 - 216*t^2 - 108, 432*t^18 + 1296*t^16 +
648*t^14 - 1512*t^12 - 2592*t^10 - 2592*t^8 - 1512*t^6 + 648*t^4 +
1296*t^2 + 432 ], [ 0, 0, 0, -27*t^10 + 27*t^6 - 27*t^2, 54*t^15 -
81*t^11 - 81*t^7 + 54*t^3 ], [ 0, 0, 0, -108*t^12 - 216*t^10 + 216*t^6
- 216*t^2 - 108, -432*t^18 - 1296*t^16 - 648*t^14 + 1512*t^12 +
2592*t^10 + 2592*t^8 + 1512*t^6 - 648*t^4 - 1296*t^2 - 432 ], [ 0, 0,
0, -27*t^8 + 27*t^4 - 27, -54*t^12 + 81*t^8 + 81*t^4 - 54 ], [ 0, 0,
0, -108*t^14 - 216*t^12 + 216*t^8 - 216*t^4 - 108*t^2, 432*t^21 +
1296*t^19 + 648*t^17 - 1512*t^15 - 2592*t^13 - 2592*t^11 - 1512*t^9 +
648*t^7 + 1296*t^5 + 432*t^3 ], [ 0, 0, 0, -27*t^12 - 54*t^10 + 54*t^6
- 54*t^2 - 27, 54*t^18 + 162*t^16 + 81*t^14 - 189*t^12 - 324*t^10 -
324*t^8 - 189*t^6 + 81*t^4 + 162*t^2 + 54 ], [ 0, 0, 0, -108*t^10 +
108*t^6 - 108*t^2, 432*t^15 - 648*t^11 - 648*t^7 + 432*t^3 ], [ 0, 0,
0, -27*t^12 - 54*t^10 + 54*t^6 - 54*t^2 - 27, -54*t^18 - 162*t^16 -
81*t^14 + 189*t^12 + 324*t^10 + 324*t^8 + 189*t^6 - 81*t^4 - 162*t^2 -
54 ], [ 0, 0, 0, -27*t^8 + 27*t^4 - 27, 54*t^12 - 81*t^8 - 81*t^4 + 54
], [ 0, 0, 0, -108*t^16 + 1080*t^14 - 4536*t^12 + 10368*t^10 -
14040*t^8 + 11664*t^6 - 6156*t^4 + 2160*t^2 - 432, 432*t^24 -
6480*t^22 + 43416*t^20 - 171288*t^18 + 440640*t^16 - 769824*t^14 +
914760*t^12 - 705672*t^10 + 296784*t^8 - 5616*t^6 - 59616*t^4 +
25920*t^2 - 3456 ], [ 0, 0, 0, -108*t^8 + 432*t^6 - 540*t^4 + 216*t^2
- 108, 432*t^12 - 2592*t^10 + 5832*t^8 - 6048*t^6 + 1944*t^4 +
1296*t^2 - 432 ], [ 0, 0, 0, -108*t^8 + 432*t^6 - 540*t^4 + 216*t^2 -
108, -432*t^12 + 2592*t^10 - 5832*t^8 + 6048*t^6 - 1944*t^4 - 1296*t^2
+ 432 ], [ 0, 0, 0, -108*t^16 + 1080*t^14 - 4536*t^12 + 10368*t^10 -
14040*t^8 + 11664*t^6 - 6156*t^4 + 2160*t^2 - 432, -432*t^24 +
6480*t^22 - 43416*t^20 + 171288*t^18 - 440640*t^16 + 769824*t^14 -
914760*t^12 + 705672*t^10 - 296784*t^8 + 5616*t^6 + 59616*t^4 -
25920*t^2 + 3456 ], [ 0, 0, 0, -27*t^8 + 108*t^6 - 135*t^4 + 54*t^2 -
27, 54*t^12 - 324*t^10 + 729*t^8 - 756*t^6 + 243*t^4 + 162*t^2 - 54 ],
[ 0, 0, 0, -27*t^16 + 270*t^14 - 1134*t^12 + 2592*t^10 - 3510*t^8 +
2916*t^6 - 1539*t^4 + 540*t^2 - 108, 54*t^24 - 810*t^22 + 5427*t^20 -
21411*t^18 + 55080*t^16 - 96228*t^14 + 114345*t^12 - 88209*t^10 +
37098*t^8 - 702*t^6 - 7452*t^4 + 3240*t^2 - 432 ], [ 0, 0, 0, -27*t^16
+ 270*t^14 - 1134*t^12 + 2592*t^10 - 3510*t^8 + 2916*t^6 - 1539*t^4 +
540*t^2 - 108, -54*t^24 + 810*t^22 - 5427*t^20 + 21411*t^18 -
55080*t^16 + 96228*t^14 - 114345*t^12 + 88209*t^10 - 37098*t^8 +
702*t^6 + 7452*t^4 - 3240*t^2 + 432 ], [ 0, 0, 0, -27*t^8 + 108*t^6 -
135*t^4 + 54*t^2 - 27, -54*t^12 + 324*t^10 - 729*t^8 + 756*t^6 -
243*t^4 - 162*t^2 + 54 ], [ 0, 0, 0, -108*t^12 + 648*t^10 - 1512*t^8 +
1728*t^6 - 1080*t^4 + 432*t^2 - 108, 432*t^18 - 3888*t^16 + 14904*t^14
- 31752*t^12 + 40176*t^10 - 28512*t^8 + 7560*t^6 + 3240*t^4 - 2592*t^2
+ 432 ], [ 0, 0, 0, -108*t^12 + 864*t^10 - 2700*t^8 + 4104*t^6 -
3132*t^4 + 1296*t^2 - 432, 432*t^18 - 5184*t^16 + 26568*t^14 -
75600*t^12 + 128952*t^10 - 129600*t^8 + 63504*t^6 + 2592*t^4 -
15552*t^2 + 3456 ], [ 0, 0, 0, -108*t^12 + 864*t^10 - 2700*t^8 +
4104*t^6 - 3132*t^4 + 1296*t^2 - 432, -432*t^18 + 5184*t^16 -
26568*t^14 + 75600*t^12 - 128952*t^10 + 129600*t^8 - 63504*t^6 -
2592*t^4 + 15552*t^2 - 3456 ], [ 0, 0, 0, -108*t^12 + 648*t^10 -
1512*t^8 + 1728*t^6 - 1080*t^4 + 432*t^2 - 108, -432*t^18 + 3888*t^16
- 14904*t^14 + 31752*t^12 - 40176*t^10 + 28512*t^8 - 7560*t^6 -
3240*t^4 + 2592*t^2 - 432 ], [ 0, 0, 0, -27*t^12 + 216*t^10 - 675*t^8
+ 1026*t^6 - 783*t^4 + 324*t^2 - 108, -54*t^18 + 648*t^16 - 3321*t^14
+ 9450*t^12 - 16119*t^10 + 16200*t^8 - 7938*t^6 - 324*t^4 + 1944*t^2 -
432 ], [ 0, 0, 0, -27*t^12 + 216*t^10 - 675*t^8 + 1026*t^6 - 783*t^4 +
324*t^2 - 108, 54*t^18 - 648*t^16 + 3321*t^14 - 9450*t^12 + 16119*t^10
- 16200*t^8 + 7938*t^6 + 324*t^4 - 1944*t^2 + 432 ], [ 0, 0, 0,
-27*t^12 + 162*t^10 - 378*t^8 + 432*t^6 - 270*t^4 + 108*t^2 - 27,
54*t^18 - 486*t^16 + 1863*t^14 - 3969*t^12 + 5022*t^10 - 3564*t^8 +
945*t^6 + 405*t^4 - 324*t^2 + 54 ], [ 0, 0, 0, -27*t^12 + 162*t^10 -
378*t^8 + 432*t^6 - 270*t^4 + 108*t^2 - 27, -54*t^18 + 486*t^16 -
1863*t^14 + 3969*t^12 - 5022*t^10 + 3564*t^8 - 945*t^6 - 405*t^4 +
324*t^2 - 54 ], [ 0, 0, 0, -108*t^8 - 432*t^6 - 540*t^4 - 216*t^2 -
108, 432*t^12 + 2592*t^10 + 5832*t^8 + 6048*t^6 + 1944*t^4 - 1296*t^2
- 432 ], [ 0, 0, 0, -108*t^8 - 432*t^6 - 540*t^4 - 216*t^2 - 108,
-432*t^12 - 2592*t^10 - 5832*t^8 - 6048*t^6 - 1944*t^4 + 1296*t^2 +
432 ], [ 0, 0, 0, -108*t^16 - 1080*t^14 - 4536*t^12 - 10368*t^10 -
14040*t^8 - 11664*t^6 - 6156*t^4 - 2160*t^2 - 432, 432*t^24 +
6480*t^22 + 43416*t^20 + 171288*t^18 + 440640*t^16 + 769824*t^14 +
914760*t^12 + 705672*t^10 + 296784*t^8 + 5616*t^6 - 59616*t^4 -
25920*t^2 - 3456 ], [ 0, 0, 0, -108*t^16 - 1080*t^14 - 4536*t^12 -
10368*t^10 - 14040*t^8 - 11664*t^6 - 6156*t^4 - 2160*t^2 - 432,
-432*t^24 - 6480*t^22 - 43416*t^20 - 171288*t^18 - 440640*t^16 -
769824*t^14 - 914760*t^12 - 705672*t^10 - 296784*t^8 - 5616*t^6 +
59616*t^4 + 25920*t^2 + 3456 ], [ 0, 0, 0, -27*t^16 - 270*t^14 -
1134*t^12 - 2592*t^10 - 3510*t^8 - 2916*t^6 - 1539*t^4 - 540*t^2 -
108, 54*t^24 + 810*t^22 + 5427*t^20 + 21411*t^18 + 55080*t^16 +
96228*t^14 + 114345*t^12 + 88209*t^10 + 37098*t^8 + 702*t^6 - 7452*t^4
- 3240*t^2 - 432 ], [ 0, 0, 0, -27*t^16 - 270*t^14 - 1134*t^12 -
2592*t^10 - 3510*t^8 - 2916*t^6 - 1539*t^4 - 540*t^2 - 108, -54*t^24 -
810*t^22 - 5427*t^20 - 21411*t^18 - 55080*t^16 - 96228*t^14 -
114345*t^12 - 88209*t^10 - 37098*t^8 - 702*t^6 + 7452*t^4 + 3240*t^2 +
432 ], [ 0, 0, 0, -27*t^8 - 108*t^6 - 135*t^4 - 54*t^2 - 27, 54*t^12 +
324*t^10 + 729*t^8 + 756*t^6 + 243*t^4 - 162*t^2 - 54 ], [ 0, 0, 0,
-27*t^8 - 108*t^6 - 135*t^4 - 54*t^2 - 27, -54*t^12 - 324*t^10 -
729*t^8 - 756*t^6 - 243*t^4 + 162*t^2 + 54 ], [ 0, 0, 0, -108*t^12 -
864*t^10 - 2700*t^8 - 4104*t^6 - 3132*t^4 - 1296*t^2 - 432, 432*t^18 +
5184*t^16 + 26568*t^14 + 75600*t^12 + 128952*t^10 + 129600*t^8 +
63504*t^6 - 2592*t^4 - 15552*t^2 - 3456 ], [ 0, 0, 0, -108*t^12 -
864*t^10 - 2700*t^8 - 4104*t^6 - 3132*t^4 - 1296*t^2 - 432, -432*t^18
- 5184*t^16 - 26568*t^14 - 75600*t^12 - 128952*t^10 - 129600*t^8 -
63504*t^6 + 2592*t^4 + 15552*t^2 + 3456 ], [ 0, 0, 0, -108*t^12 -
648*t^10 - 1512*t^8 - 1728*t^6 - 1080*t^4 - 432*t^2 - 108, 432*t^18 +
3888*t^16 + 14904*t^14 + 31752*t^12 + 40176*t^10 + 28512*t^8 +
7560*t^6 - 3240*t^4 - 2592*t^2 - 432 ], [ 0, 0, 0, -108*t^12 -
648*t^10 - 1512*t^8 - 1728*t^6 - 1080*t^4 - 432*t^2 - 108, -432*t^18 -
3888*t^16 - 14904*t^14 - 31752*t^12 - 40176*t^10 - 28512*t^8 -
7560*t^6 + 3240*t^4 + 2592*t^2 + 432 ], [ 0, 0, 0, -27*t^12 - 216*t^10
- 675*t^8 - 1026*t^6 - 783*t^4 - 324*t^2 - 108, -54*t^18 - 648*t^16 -
3321*t^14 - 9450*t^12 - 16119*t^10 - 16200*t^8 - 7938*t^6 + 324*t^4 +
1944*t^2 + 432 ], [ 0, 0, 0, -27*t^12 - 162*t^10 - 378*t^8 - 432*t^6 -
270*t^4 - 108*t^2 - 27, 54*t^18 + 486*t^16 + 1863*t^14 + 3969*t^12 +
5022*t^10 + 3564*t^8 + 945*t^6 - 405*t^4 - 324*t^2 - 54 ], [ 0, 0, 0,
-27*t^12 - 162*t^10 - 378*t^8 - 432*t^6 - 270*t^4 - 108*t^2 - 27,
-54*t^18 - 486*t^16 - 1863*t^14 - 3969*t^12 - 5022*t^10 - 3564*t^8 -
945*t^6 + 405*t^4 + 324*t^2 + 54 ], [ 0, 0, 0, -27*t^12 - 216*t^10 -
675*t^8 - 1026*t^6 - 783*t^4 - 324*t^2 - 108, 54*t^18 + 648*t^16 +
3321*t^14 + 9450*t^12 + 16119*t^10 + 16200*t^8 + 7938*t^6 - 324*t^4 -
1944*t^2 - 432 ], [ 0, 0, 0, -27*t^16 + 324*t^12 - 1728*t^8 + 5184*t^4
- 6912, 54*t^24 - 972*t^20 + 5184*t^16 - 82944*t^8 + 248832*t^4 -
221184 ], [ 0, 0, 0, -27*t^12 - 108*t^10 + 432*t^6 - 1728*t^2 - 1728,
54*t^18 + 324*t^16 + 324*t^14 - 1512*t^12 - 5184*t^10 - 10368*t^8 -
12096*t^6 + 10368*t^4 + 41472*t^2 + 27648 ], [ 0, 0, 0, -27*t^12 -
108*t^10 + 432*t^6 - 1728*t^2 - 1728, -54*t^18 - 324*t^16 - 324*t^14 +
1512*t^12 + 5184*t^10 + 10368*t^8 + 12096*t^6 - 10368*t^4 - 41472*t^2
- 27648 ], [ 0, 0, 0, -108*t^16 + 1296*t^12 - 6912*t^8 + 20736*t^4 -
27648, 432*t^24 - 7776*t^20 + 41472*t^16 - 663552*t^8 + 1990656*t^4 -
1769472 ], [ 0, 0, 0, -27*t^8 + 108*t^4 - 432, 54*t^12 - 324*t^8 -
1296*t^4 + 3456 ], [ 0, 0, 0, -27*t^12 + 108*t^10 - 432*t^6 + 1728*t^2
- 1728, 54*t^18 - 324*t^16 + 324*t^14 + 1512*t^12 - 5184*t^10 +
10368*t^8 - 12096*t^6 - 10368*t^4 + 41472*t^2 - 27648 ], [ 0, 0, 0,
-27*t^12 + 108*t^10 - 432*t^6 + 1728*t^2 - 1728, -54*t^18 + 324*t^16 -
324*t^14 - 1512*t^12 + 5184*t^10 - 10368*t^8 + 12096*t^6 + 10368*t^4 -
41472*t^2 + 27648 ], [ 0, 0, 0, -27*t^8 + 108*t^4 - 432, -54*t^12 +
324*t^8 + 1296*t^4 - 3456 ], [ 0, 0, 0, -108*t^8 + 432*t^4 - 1728,
432*t^12 - 2592*t^8 - 10368*t^4 + 27648 ], [ 0, 0, 0, -108*t^8 +
432*t^4 - 1728, -432*t^12 + 2592*t^8 + 10368*t^4 - 27648 ], [ 0, 0, 0,
-452984832*t^16 + 566231040*t^14 - 297271296*t^12 + 84934656*t^10 -
14376960*t^8 + 1492992*t^6 - 98496*t^4 + 4320*t^2 - 108,
3710851743744*t^24 - 6957847019520*t^22 + 5827196878848*t^20 -
2873735774208*t^18 + 924089057280*t^16 - 201804742656*t^14 +
29974855680*t^12 - 2890432512*t^10 + 151953408*t^8 - 359424*t^6 -
476928*t^4 + 25920*t^2 - 432 ], [ 0, 0, 0, -110592*t^8 + 55296*t^6 -
8640*t^4 + 432*t^2 - 27, 14155776*t^12 - 10616832*t^10 + 2985984*t^8 -
387072*t^6 + 15552*t^4 + 1296*t^2 - 54 ], [ 0, 0, 0, -452984832*t^16 +
566231040*t^14 - 297271296*t^12 + 84934656*t^10 - 14376960*t^8 +
1492992*t^6 - 98496*t^4 + 4320*t^2 - 108, -3710851743744*t^24 +
6957847019520*t^22 - 5827196878848*t^20 + 2873735774208*t^18 -
924089057280*t^16 + 201804742656*t^14 - 29974855680*t^12 +
2890432512*t^10 - 151953408*t^8 + 359424*t^6 + 476928*t^4 - 25920*t^2
+ 432 ], [ 0, 0, 0, -110592*t^8 + 55296*t^6 - 8640*t^4 + 432*t^2 - 27,
-14155776*t^12 + 10616832*t^10 - 2985984*t^8 + 387072*t^6 - 15552*t^4
- 1296*t^2 + 54 ], [ 0, 0, 0, -442368*t^8 + 221184*t^6 - 34560*t^4 +
1728*t^2 - 108, 113246208*t^12 - 84934656*t^10 + 23887872*t^8 -
3096576*t^6 + 124416*t^4 + 10368*t^2 - 432 ], [ 0, 0, 0,
-113246208*t^16 + 141557760*t^14 - 74317824*t^12 + 21233664*t^10 -
3594240*t^8 + 373248*t^6 - 24624*t^4 + 1080*t^2 - 27,
463856467968*t^24 - 869730877440*t^22 + 728399609856*t^20 -
359216971776*t^18 + 115511132160*t^16 - 25225592832*t^14 +
3746856960*t^12 - 361304064*t^10 + 18994176*t^8 - 44928*t^6 -
59616*t^4 + 3240*t^2 - 54 ], [ 0, 0, 0, -442368*t^8 + 221184*t^6 -
34560*t^4 + 1728*t^2 - 108, -113246208*t^12 + 84934656*t^10 -
23887872*t^8 + 3096576*t^6 - 124416*t^4 - 10368*t^2 + 432 ], [ 0, 0,
0, -113246208*t^16 + 141557760*t^14 - 74317824*t^12 + 21233664*t^10 -
3594240*t^8 + 373248*t^6 - 24624*t^4 + 1080*t^2 - 27,
-463856467968*t^24 + 869730877440*t^22 - 728399609856*t^20 +
359216971776*t^18 - 115511132160*t^16 + 25225592832*t^14 -
3746856960*t^12 + 361304064*t^10 - 18994176*t^8 + 44928*t^6 +
59616*t^4 - 3240*t^2 + 54 ], [ 0, 0, 0, -28311552*t^12 + 21233664*t^10
- 6193152*t^8 + 884736*t^6 - 69120*t^4 + 3456*t^2 - 108,
57982058496*t^18 - 65229815808*t^16 + 31255953408*t^14 -
8323596288*t^12 + 1316487168*t^10 - 116785152*t^8 + 3870720*t^6 +
207360*t^4 - 20736*t^2 + 432 ], [ 0, 0, 0, -1769472*t^12 +
1769472*t^10 - 691200*t^8 + 131328*t^6 - 12528*t^4 + 648*t^2 - 27,
905969664*t^18 - 1358954496*t^16 + 870580224*t^14 - 309657600*t^12 +
66023424*t^10 - 8294400*t^8 + 508032*t^6 + 2592*t^4 - 1944*t^2 + 54 ],
[ 0, 0, 0, -28311552*t^12 + 21233664*t^10 - 6193152*t^8 + 884736*t^6 -
69120*t^4 + 3456*t^2 - 108, -57982058496*t^18 + 65229815808*t^16 -
31255953408*t^14 + 8323596288*t^12 - 1316487168*t^10 + 116785152*t^8 -
3870720*t^6 - 207360*t^4 + 20736*t^2 - 432 ], [ 0, 0, 0, -1769472*t^12
+ 1769472*t^10 - 691200*t^8 + 131328*t^6 - 12528*t^4 + 648*t^2 - 27,
-905969664*t^18 + 1358954496*t^16 - 870580224*t^14 + 309657600*t^12 -
66023424*t^10 + 8294400*t^8 - 508032*t^6 - 2592*t^4 + 1944*t^2 - 54 ],
[ 0, 0, 0, -7077888*t^12 + 5308416*t^10 - 1548288*t^8 + 221184*t^6 -
17280*t^4 + 864*t^2 - 27, -7247757312*t^18 + 8153726976*t^16 -
3906994176*t^14 + 1040449536*t^12 - 164560896*t^10 + 14598144*t^8 -
483840*t^6 - 25920*t^4 + 2592*t^2 - 54 ], [ 0, 0, 0, -7077888*t^12 +
7077888*t^10 - 2764800*t^8 + 525312*t^6 - 50112*t^4 + 2592*t^2 - 108,
7247757312*t^18 - 10871635968*t^16 + 6964641792*t^14 - 2477260800*t^12
+ 528187392*t^10 - 66355200*t^8 + 4064256*t^6 + 20736*t^4 - 15552*t^2
+ 432 ], [ 0, 0, 0, -7077888*t^12 + 5308416*t^10 - 1548288*t^8 +
221184*t^6 - 17280*t^4 + 864*t^2 - 27, 7247757312*t^18 -
8153726976*t^16 + 3906994176*t^14 - 1040449536*t^12 + 164560896*t^10 -
14598144*t^8 + 483840*t^6 + 25920*t^4 - 2592*t^2 + 54 ], [ 0, 0, 0,
-7077888*t^12 + 7077888*t^10 - 2764800*t^8 + 525312*t^6 - 50112*t^4 +
2592*t^2 - 108, -7247757312*t^18 + 10871635968*t^16 - 6964641792*t^14
+ 2477260800*t^12 - 528187392*t^10 + 66355200*t^8 - 4064256*t^6 -
20736*t^4 + 15552*t^2 - 432 ], [ 0, 0, 0, -108*t^14 - 648*t^13 +
2484*t^12 + 11232*t^11 - 42336*t^10 - 31104*t^9 + 319680*t^8 -
518400*t^7 + 84672*t^6 + 908928*t^5 - 1619136*t^4 + 1451520*t^3 -
760320*t^2 + 221184*t - 27648, -432*t^21 - 3888*t^20 + 9072*t^19 +
117936*t^18 - 217728*t^17 - 1529280*t^16 + 4527360*t^15 + 6013440*t^14
- 42705792*t^13 + 53578368*t^12 + 69973632*t^11 - 346488192*t^10 +
659349504*t^9 - 938552832*t^8 + 1203600384*t^7 - 1358760960*t^6 +
1218945024*t^5 - 806547456*t^4 + 374685696*t^3 - 115458048*t^2 +
21233664*t - 1769472 ], [ 0, 0, 0, -108*t^14 - 648*t^13 + 2484*t^12 +
11232*t^11 - 42336*t^10 - 31104*t^9 + 319680*t^8 - 518400*t^7 +
84672*t^6 + 908928*t^5 - 1619136*t^4 + 1451520*t^3 - 760320*t^2 +
221184*t - 27648, 432*t^21 + 3888*t^20 - 9072*t^19 - 117936*t^18 +
217728*t^17 + 1529280*t^16 - 4527360*t^15 - 6013440*t^14 +
42705792*t^13 - 53578368*t^12 - 69973632*t^11 + 346488192*t^10 -
659349504*t^9 + 938552832*t^8 - 1203600384*t^7 + 1358760960*t^6 -
1218945024*t^5 + 806547456*t^4 - 374685696*t^3 + 115458048*t^2 -
21233664*t + 1769472 ], [ 0, 0, 0, -27*t^14 - 162*t^13 + 621*t^12 +
2808*t^11 - 10584*t^10 - 7776*t^9 + 79920*t^8 - 129600*t^7 + 21168*t^6
+ 227232*t^5 - 404784*t^4 + 362880*t^3 - 190080*t^2 + 55296*t - 6912,
-54*t^21 - 486*t^20 + 1134*t^19 + 14742*t^18 - 27216*t^17 -
191160*t^16 + 565920*t^15 + 751680*t^14 - 5338224*t^13 + 6697296*t^12
+ 8746704*t^11 - 43311024*t^10 + 82418688*t^9 - 117319104*t^8 +
150450048*t^7 - 169845120*t^6 + 152368128*t^5 - 100818432*t^4 +
46835712*t^3 - 14432256*t^2 + 2654208*t - 221184 ], [ 0, 0, 0,
-27*t^14 - 162*t^13 + 621*t^12 + 2808*t^11 - 10584*t^10 - 7776*t^9 +
79920*t^8 - 129600*t^7 + 21168*t^6 + 227232*t^5 - 404784*t^4 +
362880*t^3 - 190080*t^2 + 55296*t - 6912, 54*t^21 + 486*t^20 -
1134*t^19 - 14742*t^18 + 27216*t^17 + 191160*t^16 - 565920*t^15 -
751680*t^14 + 5338224*t^13 - 6697296*t^12 - 8746704*t^11 +
43311024*t^10 - 82418688*t^9 + 117319104*t^8 - 150450048*t^7 +
169845120*t^6 - 152368128*t^5 + 100818432*t^4 - 46835712*t^3 +
14432256*t^2 - 2654208*t + 221184 ], [ 0, 0, 0, -108*t^10 + 216*t^9 +
1620*t^8 - 6912*t^7 + 8640*t^6 + 3456*t^5 - 24192*t^4 + 34560*t^3 -
25920*t^2 + 10368*t - 1728, -432*t^15 + 1296*t^14 + 9072*t^13 -
51408*t^12 + 51840*t^11 + 259200*t^10 - 1064448*t^9 + 2052864*t^8 -
2809728*t^7 + 3411072*t^6 - 3846528*t^5 + 3535488*t^4 - 2322432*t^3 +
995328*t^2 - 248832*t + 27648 ], [ 0, 0, 0, -108*t^10 + 216*t^9 +
1620*t^8 - 6912*t^7 + 8640*t^6 + 3456*t^5 - 24192*t^4 + 34560*t^3 -
25920*t^2 + 10368*t - 1728, 432*t^15 - 1296*t^14 - 9072*t^13 +
51408*t^12 - 51840*t^11 - 259200*t^10 + 1064448*t^9 - 2052864*t^8 +
2809728*t^7 - 3411072*t^6 + 3846528*t^5 - 3535488*t^4 + 2322432*t^3 -
995328*t^2 + 248832*t - 27648 ], [ 0, 0, 0, -27*t^10 + 54*t^9 +
405*t^8 - 1728*t^7 + 2160*t^6 + 864*t^5 - 6048*t^4 + 8640*t^3 -
6480*t^2 + 2592*t - 432, -54*t^15 + 162*t^14 + 1134*t^13 - 6426*t^12 +
6480*t^11 + 32400*t^10 - 133056*t^9 + 256608*t^8 - 351216*t^7 +
426384*t^6 - 480816*t^5 + 441936*t^4 - 290304*t^3 + 124416*t^2 -
31104*t + 3456 ], [ 0, 0, 0, -27*t^10 + 54*t^9 + 405*t^8 - 1728*t^7 +
2160*t^6 + 864*t^5 - 6048*t^4 + 8640*t^3 - 6480*t^2 + 2592*t - 432,
54*t^15 - 162*t^14 - 1134*t^13 + 6426*t^12 - 6480*t^11 - 32400*t^10 +
133056*t^9 - 256608*t^8 + 351216*t^7 - 426384*t^6 + 480816*t^5 -
441936*t^4 + 290304*t^3 - 124416*t^2 + 31104*t - 3456 ], [ 0, 0, 0,
-108*t^12 - 864*t^11 + 864*t^10 + 13824*t^9 - 15552*t^8 - 76032*t^7 +
183168*t^6 - 76032*t^5 - 250560*t^4 + 483840*t^3 - 400896*t^2 +
165888*t - 27648, 432*t^18 + 5184*t^17 + 5184*t^16 - 117504*t^15 -
145152*t^14 + 1451520*t^13 + 145152*t^12 - 10077696*t^11 +
13488768*t^10 + 17266176*t^9 - 68719104*t^8 + 102021120*t^7 -
129862656*t^6 + 174182400*t^5 - 189444096*t^4 + 138018816*t^3 -
62373888*t^2 + 15925248*t - 1769472 ], [ 0, 0, 0, -108*t^12 - 864*t^11
+ 864*t^10 + 13824*t^9 - 15552*t^8 - 76032*t^7 + 183168*t^6 -
76032*t^5 - 250560*t^4 + 483840*t^3 - 400896*t^2 + 165888*t - 27648,
-432*t^18 - 5184*t^17 - 5184*t^16 + 117504*t^15 + 145152*t^14 -
1451520*t^13 - 145152*t^12 + 10077696*t^11 - 13488768*t^10 -
17266176*t^9 + 68719104*t^8 - 102021120*t^7 + 129862656*t^6 -
174182400*t^5 + 189444096*t^4 - 138018816*t^3 + 62373888*t^2 -
15925248*t + 1769472 ], [ 0, 0, 0, -27*t^12 - 216*t^11 + 216*t^10 +
3456*t^9 - 3888*t^8 - 19008*t^7 + 45792*t^6 - 19008*t^5 - 62640*t^4 +
120960*t^3 - 100224*t^2 + 41472*t - 6912, 54*t^18 + 648*t^17 +
648*t^16 - 14688*t^15 - 18144*t^14 + 181440*t^13 + 18144*t^12 -
1259712*t^11 + 1686096*t^10 + 2158272*t^9 - 8589888*t^8 + 12752640*t^7
- 16232832*t^6 + 21772800*t^5 - 23680512*t^4 + 17252352*t^3 -
7796736*t^2 + 1990656*t - 221184 ], [ 0, 0, 0, -27*t^12 - 216*t^11 +
216*t^10 + 3456*t^9 - 3888*t^8 - 19008*t^7 + 45792*t^6 - 19008*t^5 -
62640*t^4 + 120960*t^3 - 100224*t^2 + 41472*t - 6912, -54*t^18 -
648*t^17 - 648*t^16 + 14688*t^15 + 18144*t^14 - 181440*t^13 -
18144*t^12 + 1259712*t^11 - 1686096*t^10 - 2158272*t^9 + 8589888*t^8 -
12752640*t^7 + 16232832*t^6 - 21772800*t^5 + 23680512*t^4 -
17252352*t^3 + 7796736*t^2 - 1990656*t + 221184 ], [ 0, 0, 0, -108*t^8
+ 1728*t^6 - 3456*t^5 + 6912*t^3 - 10368*t^2 + 6912*t - 1728, 432*t^12
- 10368*t^10 + 20736*t^9 + 41472*t^8 - 207360*t^7 + 338688*t^6 -
373248*t^5 + 466560*t^4 - 552960*t^3 + 414720*t^2 - 165888*t + 27648
], [ 0, 0, 0, -108*t^8 + 1728*t^6 - 3456*t^5 + 6912*t^3 - 10368*t^2 +
6912*t - 1728, -432*t^12 + 10368*t^10 - 20736*t^9 - 41472*t^8 +
207360*t^7 - 338688*t^6 + 373248*t^5 - 466560*t^4 + 552960*t^3 -
414720*t^2 + 165888*t - 27648 ], [ 0, 0, 0, -27*t^8 + 432*t^6 -
864*t^5 + 1728*t^3 - 2592*t^2 + 1728*t - 432, -54*t^12 + 1296*t^10 -
2592*t^9 - 5184*t^8 + 25920*t^7 - 42336*t^6 + 46656*t^5 - 58320*t^4 +
69120*t^3 - 51840*t^2 + 20736*t - 3456 ], [ 0, 0, 0, -27*t^8 + 432*t^6
- 864*t^5 + 1728*t^3 - 2592*t^2 + 1728*t - 432, 54*t^12 - 1296*t^10 +
2592*t^9 + 5184*t^8 - 25920*t^7 + 42336*t^6 - 46656*t^5 + 58320*t^4 -
69120*t^3 + 51840*t^2 - 20736*t + 3456 ], [ 0, 0, 0, 54*t^20 -
216*t^19 - 6912*t^18 - 24624*t^17 + 87480*t^16 + 808704*t^15 +
1689984*t^14 - 2778624*t^13 - 22814784*t^12 - 53450496*t^11 -
47402496*t^10 + 61309440*t^9 + 266727168*t^8 + 446570496*t^7 +
468633600*t^6 + 334430208*t^5 + 163524096*t^4 + 53250048*t^3 +
10838016*t^2 + 1216512*t + 55296, 540*t^30 + 8424*t^29 + 27864*t^28 -
229824*t^27 - 1932336*t^26 - 1796256*t^25 + 30878496*t^24 +
121678848*t^23 - 70279488*t^22 - 1630088064*t^21 - 3708270720*t^20 +
5403718656*t^19 + 49006487808*t^18 + 109890805248*t^17 +
9524169216*t^16 - 648125743104*t^15 - 2228816342016*t^14 -
4541579642880*t^13 - 6586419640320*t^12 - 7132383756288*t^11 -
5773489311744*t^10 - 3339683094528*t^9 - 1154721816576*t^8 +
26435911680*t^7 + 319026511872*t^6 + 220668198912*t^5 +
88345313280*t^4 + 23137615872*t^3 + 3922919424*t^2 + 392822784*t +
17694720 ], [ 0, 0, 0, 54*t^16 - 432*t^15 - 5616*t^14 + 864*t^13 +
132192*t^12 + 319680*t^11 - 630720*t^10 - 3874176*t^9 - 5358528*t^8 +
2744064*t^7 + 18005760*t^6 + 25698816*t^5 + 19367424*t^4 + 8487936*t^3
+ 2128896*t^2 + 276480*t + 13824, -540*t^24 - 5184*t^23 + 12960*t^22 +
262656*t^21 + 308448*t^20 - 4997376*t^19 - 15189120*t^18 +
39813120*t^17 + 247209408*t^16 + 86648832*t^15 - 1871133696*t^14 -
4676050944*t^13 + 223921152*t^12 + 23439974400*t^11 + 58742599680*t^10
+ 79136980992*t^9 + 65681611776*t^8 + 31456346112*t^7 + 3944816640*t^6
- 5584453632*t^5 - 4380106752*t^4 - 1633222656*t^3 - 353009664*t^2 -
42467328*t - 2211840 ], [ 0, 0, 0, 54*t^20 - 216*t^19 - 6912*t^18 -
24624*t^17 + 87480*t^16 + 808704*t^15 + 1689984*t^14 - 2778624*t^13 -
22814784*t^12 - 53450496*t^11 - 47402496*t^10 + 61309440*t^9 +
266727168*t^8 + 446570496*t^7 + 468633600*t^6 + 334430208*t^5 +
163524096*t^4 + 53250048*t^3 + 10838016*t^2 + 1216512*t + 55296,
-540*t^30 - 8424*t^29 - 27864*t^28 + 229824*t^27 + 1932336*t^26 +
1796256*t^25 - 30878496*t^24 - 121678848*t^23 + 70279488*t^22 +
1630088064*t^21 + 3708270720*t^20 - 5403718656*t^19 - 49006487808*t^18
- 109890805248*t^17 - 9524169216*t^16 + 648125743104*t^15 +
2228816342016*t^14 + 4541579642880*t^13 + 6586419640320*t^12 +
7132383756288*t^11 + 5773489311744*t^10 + 3339683094528*t^9 +
1154721816576*t^8 - 26435911680*t^7 - 319026511872*t^6 -
220668198912*t^5 - 88345313280*t^4 - 23137615872*t^3 - 3922919424*t^2
- 392822784*t - 17694720 ], [ 0, 0, 0, 54*t^16 - 432*t^15 - 5616*t^14
+ 864*t^13 + 132192*t^12 + 319680*t^11 - 630720*t^10 - 3874176*t^9 -
5358528*t^8 + 2744064*t^7 + 18005760*t^6 + 25698816*t^5 + 19367424*t^4
+ 8487936*t^3 + 2128896*t^2 + 276480*t + 13824, 540*t^24 + 5184*t^23 -
12960*t^22 - 262656*t^21 - 308448*t^20 + 4997376*t^19 + 15189120*t^18
- 39813120*t^17 - 247209408*t^16 - 86648832*t^15 + 1871133696*t^14 +
4676050944*t^13 - 223921152*t^12 - 23439974400*t^11 - 58742599680*t^10
- 79136980992*t^9 - 65681611776*t^8 - 31456346112*t^7 - 3944816640*t^6
+ 5584453632*t^5 + 4380106752*t^4 + 1633222656*t^3 + 353009664*t^2 +
42467328*t + 2211840 ], [ 0, 0, 0, 216*t^16 - 1728*t^15 - 22464*t^14 +
3456*t^13 + 528768*t^12 + 1278720*t^11 - 2522880*t^10 - 15496704*t^9 -
21434112*t^8 + 10976256*t^7 + 72023040*t^6 + 102795264*t^5 +
77469696*t^4 + 33951744*t^3 + 8515584*t^2 + 1105920*t + 55296,
-4320*t^24 - 41472*t^23 + 103680*t^22 + 2101248*t^21 + 2467584*t^20 -
39979008*t^19 - 121512960*t^18 + 318504960*t^17 + 1977675264*t^16 +
693190656*t^15 - 14969069568*t^14 - 37408407552*t^13 + 1791369216*t^12
+ 187519795200*t^11 + 469940797440*t^10 + 633095847936*t^9 +
525452894208*t^8 + 251650768896*t^7 + 31558533120*t^6 -
44675629056*t^5 - 35040854016*t^4 - 13065781248*t^3 - 2824077312*t^2 -
339738624*t - 17694720 ], [ 0, 0, 0, 216*t^16 - 1728*t^15 - 22464*t^14
+ 3456*t^13 + 528768*t^12 + 1278720*t^11 - 2522880*t^10 - 15496704*t^9
- 21434112*t^8 + 10976256*t^7 + 72023040*t^6 + 102795264*t^5 +
77469696*t^4 + 33951744*t^3 + 8515584*t^2 + 1105920*t + 55296,
4320*t^24 + 41472*t^23 - 103680*t^22 - 2101248*t^21 - 2467584*t^20 +
39979008*t^19 + 121512960*t^18 - 318504960*t^17 - 1977675264*t^16 -
693190656*t^15 + 14969069568*t^14 + 37408407552*t^13 - 1791369216*t^12
- 187519795200*t^11 - 469940797440*t^10 - 633095847936*t^9 -
525452894208*t^8 - 251650768896*t^7 - 31558533120*t^6 +
44675629056*t^5 + 35040854016*t^4 + 13065781248*t^3 + 2824077312*t^2 +
339738624*t + 17694720 ], [ 0, 0, 0, -432*t^16 - 1944*t^12 - 1323*t^8
- 324*t^4 - 27, -3456*t^24 + 23328*t^20 + 39528*t^16 + 23814*t^12 +
6885*t^8 + 972*t^4 + 54 ], [ 0, 0, 0, -1728*t^16 - 7776*t^12 -
5292*t^8 - 1296*t^4 - 108, -27648*t^24 + 186624*t^20 + 316224*t^16 +
190512*t^12 + 55080*t^8 + 7776*t^4 + 432 ], [ 0, 0, 0, -1728*t^16 -
7776*t^12 - 5292*t^8 - 1296*t^4 - 108, 27648*t^24 - 186624*t^20 -
316224*t^16 - 190512*t^12 - 55080*t^8 - 7776*t^4 - 432 ], [ 0, 0, 0,
-432*t^16 - 1944*t^12 - 1323*t^8 - 324*t^4 - 27, 3456*t^24 -
23328*t^20 - 39528*t^16 - 23814*t^12 - 6885*t^8 - 972*t^4 - 54 ], [ 0,
0, 0, -27*t^8 - 108*t^4 - 27, -54*t^12 + 405*t^8 + 324*t^4 + 54 ], [
0, 0, 0, -108*t^8 - 432*t^4 - 108, -432*t^12 + 3240*t^8 + 2592*t^4 +
432 ], [ 0, 0, 0, -27*t^8 - 108*t^4 - 27, 54*t^12 - 405*t^8 - 324*t^4
- 54 ], [ 0, 0, 0, -108*t^8 - 432*t^4 - 108, 432*t^12 - 3240*t^8 -
2592*t^4 - 432 ], [ 0, 0, 0, -108*t^18 - 5184*t^14 - 84672*t^10 -
497664*t^6 - 442368*t^2, 432*t^27 + 31104*t^23 + 881280*t^19 +
12192768*t^15 + 80953344*t^11 + 191102976*t^7 - 113246208*t^3 ], [ 0,
0, 0, -27*t^18 - 1296*t^14 - 21168*t^10 - 124416*t^6 - 110592*t^2,
54*t^27 + 3888*t^23 + 110160*t^19 + 1524096*t^15 + 10119168*t^11 +
23887872*t^7 - 14155776*t^3 ], [ 0, 0, 0, -108*t^16 - 5184*t^12 -
84672*t^8 - 497664*t^4 - 442368, 432*t^24 + 31104*t^20 + 881280*t^16 +
12192768*t^12 + 80953344*t^8 + 191102976*t^4 - 113246208 ], [ 0, 0, 0,
-108*t^16 - 5184*t^12 - 84672*t^8 - 497664*t^4 - 442368, -432*t^24 -
31104*t^20 - 881280*t^16 - 12192768*t^12 - 80953344*t^8 -
191102976*t^4 + 113246208 ], [ 0, 0, 0, -27*t^16 - 1296*t^12 -
21168*t^8 - 124416*t^4 - 110592, 54*t^24 + 3888*t^20 + 110160*t^16 +
1524096*t^12 + 10119168*t^8 + 23887872*t^4 - 14155776 ], [ 0, 0, 0,
-27*t^16 - 1296*t^12 - 21168*t^8 - 124416*t^4 - 110592, -54*t^24 -
3888*t^20 - 110160*t^16 - 1524096*t^12 - 10119168*t^8 - 23887872*t^4 +
14155776 ], [ 0, 0, 0, -108*t^10 - 1728*t^6 - 1728*t^2, 432*t^15 +
10368*t^11 + 51840*t^7 - 27648*t^3 ], [ 0, 0, 0, -27*t^10 - 432*t^6 -
432*t^2, 54*t^15 + 1296*t^11 + 6480*t^7 - 3456*t^3 ], [ 0, 0, 0,
-27*t^8 - 432*t^4 - 432, -54*t^12 - 1296*t^8 - 6480*t^4 + 3456 ], [ 0,
0, 0, -108*t^8 - 1728*t^4 - 1728, 432*t^12 + 10368*t^8 + 51840*t^4 -
27648 ], [ 0, 0, 0, -108*t^8 - 1728*t^4 - 1728, -432*t^12 - 10368*t^8
- 51840*t^4 + 27648 ], [ 0, 0, 0, -27*t^8 - 432*t^4 - 432, 54*t^12 +
1296*t^8 + 6480*t^4 - 3456 ], [ 0, 0, 0, -1728*t^16 + 7776*t^12 -
5292*t^8 + 1296*t^4 - 108, -27648*t^24 - 186624*t^20 + 316224*t^16 -
190512*t^12 + 55080*t^8 - 7776*t^4 + 432 ], [ 0, 0, 0, -432*t^16 +
1944*t^12 - 1323*t^8 + 324*t^4 - 27, -3456*t^24 - 23328*t^20 +
39528*t^16 - 23814*t^12 + 6885*t^8 - 972*t^4 + 54 ], [ 0, 0, 0,
-108*t^8 + 432*t^4 - 108, 432*t^12 + 3240*t^8 - 2592*t^4 + 432 ], [ 0,
0, 0, -27*t^8 + 108*t^4 - 27, 54*t^12 + 405*t^8 - 324*t^4 + 54 ], [ 0,
0, 0, -432*t^12 - 432*t^10 + 1620*t^8 + 1728*t^6 - 432*t^2 - 108,
3456*t^18 + 5184*t^16 + 28512*t^14 + 39312*t^12 - 1296*t^10 -
27864*t^8 - 12096*t^6 + 2592*t^4 + 2592*t^2 + 432 ], [ 0, 0, 0,
-108*t^12 - 108*t^10 + 405*t^8 + 432*t^6 - 108*t^2 - 27, 432*t^18 +
648*t^16 + 3564*t^14 + 4914*t^12 - 162*t^10 - 3483*t^8 - 1512*t^6 +
324*t^4 + 324*t^2 + 54 ], [ 0, 0, 0, -432*t^12 + 432*t^10 + 1620*t^8 -
1728*t^6 + 432*t^2 - 108, -3456*t^18 + 5184*t^16 - 28512*t^14 +
39312*t^12 + 1296*t^10 - 27864*t^8 + 12096*t^6 + 2592*t^4 - 2592*t^2 +
432 ], [ 0, 0, 0, -108*t^12 + 108*t^10 + 405*t^8 - 432*t^6 + 108*t^2 -
27, -432*t^18 + 648*t^16 - 3564*t^14 + 4914*t^12 + 162*t^10 - 3483*t^8
+ 1512*t^6 + 324*t^4 - 324*t^2 + 54 ], [ 0, 0, 0, -432*t^16 +
1944*t^12 - 1323*t^8 + 324*t^4 - 27, 3456*t^24 + 23328*t^20 -
39528*t^16 + 23814*t^12 - 6885*t^8 + 972*t^4 - 54 ], [ 0, 0, 0,
-1728*t^16 + 7776*t^12 - 5292*t^8 + 1296*t^4 - 108, 27648*t^24 +
186624*t^20 - 316224*t^16 + 190512*t^12 - 55080*t^8 + 7776*t^4 - 432
], [ 0, 0, 0, -27*t^8 + 108*t^4 - 27, -54*t^12 - 405*t^8 + 324*t^4 -
54 ], [ 0, 0, 0, -108*t^8 + 432*t^4 - 108, -432*t^12 - 3240*t^8 +
2592*t^4 - 432 ], [ 0, 0, 0, -432*t^12 + 432*t^10 + 1620*t^8 -
1728*t^6 + 432*t^2 - 108, 3456*t^18 - 5184*t^16 + 28512*t^14 -
39312*t^12 - 1296*t^10 + 27864*t^8 - 12096*t^6 - 2592*t^4 + 2592*t^2 -
432 ], [ 0, 0, 0, -108*t^12 - 108*t^10 + 405*t^8 + 432*t^6 - 108*t^2 -
27, -432*t^18 - 648*t^16 - 3564*t^14 - 4914*t^12 + 162*t^10 + 3483*t^8
+ 1512*t^6 - 324*t^4 - 324*t^2 - 54 ], [ 0, 0, 0, -432*t^12 - 432*t^10
+ 1620*t^8 + 1728*t^6 - 432*t^2 - 108, -3456*t^18 - 5184*t^16 -
28512*t^14 - 39312*t^12 + 1296*t^10 + 27864*t^8 + 12096*t^6 - 2592*t^4
- 2592*t^2 - 432 ], [ 0, 0, 0, -108*t^12 + 108*t^10 + 405*t^8 -
432*t^6 + 108*t^2 - 27, 432*t^18 - 648*t^16 + 3564*t^14 - 4914*t^12 -
162*t^10 + 3483*t^8 - 1512*t^6 - 324*t^4 + 324*t^2 - 54 ], [ 0, 0, 0,
-108*t^16 + 5184*t^12 - 84672*t^8 + 497664*t^4 - 442368, -432*t^24 +
31104*t^20 - 881280*t^16 + 12192768*t^12 - 80953344*t^8 +
191102976*t^4 + 113246208 ], [ 0, 0, 0, -108*t^14 + 864*t^12 -
13824*t^8 + 25920*t^6 + 13824*t^4 - 27648*t^2, 432*t^21 - 5184*t^19 +
10368*t^17 + 96768*t^15 - 445824*t^13 + 41472*t^11 + 2515968*t^9 -
3649536*t^7 + 1327104*t^5 - 1769472*t^3 ], [ 0, 0, 0, -27*t^16 +
1296*t^12 - 21168*t^8 + 124416*t^4 - 110592, -54*t^24 + 3888*t^20 -
110160*t^16 + 1524096*t^12 - 10119168*t^8 + 23887872*t^4 + 14155776 ],
[ 0, 0, 0, -27*t^14 + 216*t^12 - 3456*t^8 + 6480*t^6 + 3456*t^4 -
6912*t^2, 54*t^21 - 648*t^19 + 1296*t^17 + 12096*t^15 - 55728*t^13 +
5184*t^11 + 314496*t^9 - 456192*t^7 + 165888*t^5 - 221184*t^3 ], [ 0,
0, 0, -108*t^16 + 5184*t^12 - 84672*t^8 + 497664*t^4 - 442368,
432*t^24 - 31104*t^20 + 881280*t^16 - 12192768*t^12 + 80953344*t^8 -
191102976*t^4 - 113246208 ], [ 0, 0, 0, -27*t^16 + 1296*t^12 -
21168*t^8 + 124416*t^4 - 110592, 54*t^24 - 3888*t^20 + 110160*t^16 -
1524096*t^12 + 10119168*t^8 - 23887872*t^4 - 14155776 ], [ 0, 0, 0,
-108*t^8 + 1728*t^4 - 1728, -432*t^12 + 10368*t^8 - 51840*t^4 - 27648
], [ 0, 0, 0, -108*t^14 - 864*t^12 + 13824*t^8 + 25920*t^6 - 13824*t^4
- 27648*t^2, 432*t^21 + 5184*t^19 + 10368*t^17 - 96768*t^15 -
445824*t^13 - 41472*t^11 + 2515968*t^9 + 3649536*t^7 + 1327104*t^5 +
1769472*t^3 ], [ 0, 0, 0, -27*t^8 + 432*t^4 - 432, -54*t^12 + 1296*t^8
- 6480*t^4 - 3456 ], [ 0, 0, 0, -27*t^14 - 216*t^12 + 3456*t^8 +
6480*t^6 - 3456*t^4 - 6912*t^2, 54*t^21 + 648*t^19 + 1296*t^17 -
12096*t^15 - 55728*t^13 - 5184*t^11 + 314496*t^9 + 456192*t^7 +
165888*t^5 + 221184*t^3 ], [ 0, 0, 0, -108*t^8 + 1728*t^4 - 1728,
432*t^12 - 10368*t^8 + 51840*t^4 + 27648 ], [ 0, 0, 0, -27*t^8 +
432*t^4 - 432, 54*t^12 - 1296*t^8 + 6480*t^4 + 3456 ], [ 0, 0, 0,
-27*t^12 - 216*t^10 + 3456*t^6 + 6480*t^4 - 3456*t^2 - 6912, 54*t^18 +
648*t^16 + 1296*t^14 - 12096*t^12 - 55728*t^10 - 5184*t^8 + 314496*t^6
+ 456192*t^4 + 165888*t^2 + 221184 ], [ 0, 0, 0, -27*t^10 + 432*t^6 -
432*t^2, 54*t^15 - 1296*t^11 + 6480*t^7 + 3456*t^3 ], [ 0, 0, 0,
-108*t^12 + 864*t^10 - 13824*t^6 + 25920*t^4 + 13824*t^2 - 27648,
-432*t^18 + 5184*t^16 - 10368*t^14 - 96768*t^12 + 445824*t^10 -
41472*t^8 - 2515968*t^6 + 3649536*t^4 - 1327104*t^2 + 1769472 ], [ 0,
0, 0, -108*t^18 + 5184*t^14 - 84672*t^10 + 497664*t^6 - 442368*t^2,
432*t^27 - 31104*t^23 + 881280*t^19 - 12192768*t^15 + 80953344*t^11 -
191102976*t^7 - 113246208*t^3 ], [ 0, 0, 0, -27*t^12 + 216*t^10 -
3456*t^6 + 6480*t^4 + 3456*t^2 - 6912, -54*t^18 + 648*t^16 - 1296*t^14
- 12096*t^12 + 55728*t^10 - 5184*t^8 - 314496*t^6 + 456192*t^4 -
165888*t^2 + 221184 ], [ 0, 0, 0, -27*t^18 + 1296*t^14 - 21168*t^10 +
124416*t^6 - 110592*t^2, 54*t^27 - 3888*t^23 + 110160*t^19 -
1524096*t^15 + 10119168*t^11 - 23887872*t^7 - 14155776*t^3 ], [ 0, 0,
0, -108*t^12 + 864*t^10 - 13824*t^6 + 25920*t^4 + 13824*t^2 - 27648,
432*t^18 - 5184*t^16 + 10368*t^14 + 96768*t^12 - 445824*t^10 +
41472*t^8 + 2515968*t^6 - 3649536*t^4 + 1327104*t^2 - 1769472 ], [ 0,
0, 0, -27*t^12 + 216*t^10 - 3456*t^6 + 6480*t^4 + 3456*t^2 - 6912,
54*t^18 - 648*t^16 + 1296*t^14 + 12096*t^12 - 55728*t^10 + 5184*t^8 +
314496*t^6 - 456192*t^4 + 165888*t^2 - 221184 ], [ 0, 0, 0, -108*t^12
- 864*t^10 + 13824*t^6 + 25920*t^4 - 13824*t^2 - 27648, -432*t^18 -
5184*t^16 - 10368*t^14 + 96768*t^12 + 445824*t^10 + 41472*t^8 -
2515968*t^6 - 3649536*t^4 - 1327104*t^2 - 1769472 ], [ 0, 0, 0,
-108*t^10 + 1728*t^6 - 1728*t^2, 432*t^15 - 10368*t^11 + 51840*t^7 +
27648*t^3 ], [ 0, 0, 0, -27*t^12 - 216*t^10 + 3456*t^6 + 6480*t^4 -
3456*t^2 - 6912, -54*t^18 - 648*t^16 - 1296*t^14 + 12096*t^12 +
55728*t^10 + 5184*t^8 - 314496*t^6 - 456192*t^4 - 165888*t^2 - 221184
], [ 0, 0, 0, -108*t^12 - 864*t^10 + 13824*t^6 + 25920*t^4 - 13824*t^2
- 27648, 432*t^18 + 5184*t^16 + 10368*t^14 - 96768*t^12 - 445824*t^10
- 41472*t^8 + 2515968*t^6 + 3649536*t^4 + 1327104*t^2 + 1769472 ], [
0, 0, 0, -108*t^8 - 1728*t^6 - 8640*t^4 - 13824*t^2 - 1728, -432*t^12
- 10368*t^10 - 93312*t^8 - 387072*t^6 - 715392*t^4 - 414720*t^2 +
27648 ], [ 0, 0, 0, -27*t^8 - 432*t^6 - 2160*t^4 - 3456*t^2 - 432,
-54*t^12 - 1296*t^10 - 11664*t^8 - 48384*t^6 - 89424*t^4 - 51840*t^2 +
3456 ], [ 0, 0, 0, -108*t^8 - 1728*t^6 - 8640*t^4 - 13824*t^2 - 1728,
432*t^12 + 10368*t^10 + 93312*t^8 + 387072*t^6 + 715392*t^4 +
414720*t^2 - 27648 ], [ 0, 0, 0, -27*t^8 - 432*t^6 - 2160*t^4 -
3456*t^2 - 432, 54*t^12 + 1296*t^10 + 11664*t^8 + 48384*t^6 +
89424*t^4 + 51840*t^2 - 3456 ], [ 0, 0, 0, -108*t^12 - 3456*t^10 -
43200*t^8 - 262656*t^6 - 775872*t^4 - 912384*t^2 - 110592, -432*t^18 -
20736*t^16 - 425088*t^14 - 4838400*t^12 - 33229440*t^10 -
139677696*t^8 - 345461760*t^6 - 445243392*t^4 - 207028224*t^2 +
14155776 ], [ 0, 0, 0, -27*t^12 - 864*t^10 - 10800*t^8 - 65664*t^6 -
193968*t^4 - 228096*t^2 - 27648, -54*t^18 - 2592*t^16 - 53136*t^14 -
604800*t^12 - 4153680*t^10 - 17459712*t^8 - 43182720*t^6 -
55655424*t^4 - 25878528*t^2 + 1769472 ], [ 0, 0, 0, -108*t^12 -
3456*t^10 - 43200*t^8 - 262656*t^6 - 775872*t^4 - 912384*t^2 - 110592,
432*t^18 + 20736*t^16 + 425088*t^14 + 4838400*t^12 + 33229440*t^10 +
139677696*t^8 + 345461760*t^6 + 445243392*t^4 + 207028224*t^2 -
14155776 ], [ 0, 0, 0, -27*t^12 - 864*t^10 - 10800*t^8 - 65664*t^6 -
193968*t^4 - 228096*t^2 - 27648, 54*t^18 + 2592*t^16 + 53136*t^14 +
604800*t^12 + 4153680*t^10 + 17459712*t^8 + 43182720*t^6 +
55655424*t^4 + 25878528*t^2 - 1769472 ], [ 0, 0, 0, -108*t^12 -
2592*t^10 - 24192*t^8 - 110592*t^6 - 250560*t^4 - 235008*t^2 - 27648,
-432*t^18 - 15552*t^16 - 238464*t^14 - 2032128*t^12 - 10502784*t^10 -
33550848*t^8 - 64060416*t^6 - 65359872*t^4 - 25214976*t^2 + 1769472 ],
[ 0, 0, 0, -27*t^12 - 648*t^10 - 6048*t^8 - 27648*t^6 - 62640*t^4 -
58752*t^2 - 6912, -54*t^18 - 1944*t^16 - 29808*t^14 - 254016*t^12 -
1312848*t^10 - 4193856*t^8 - 8007552*t^6 - 8169984*t^4 - 3151872*t^2 +
221184 ], [ 0, 0, 0, -108*t^12 - 2592*t^10 - 24192*t^8 - 110592*t^6 -
250560*t^4 - 235008*t^2 - 27648, 432*t^18 + 15552*t^16 + 238464*t^14 +
2032128*t^12 + 10502784*t^10 + 33550848*t^8 + 64060416*t^6 +
65359872*t^4 + 25214976*t^2 - 1769472 ], [ 0, 0, 0, -27*t^12 -
648*t^10 - 6048*t^8 - 27648*t^6 - 62640*t^4 - 58752*t^2 - 6912,
54*t^18 + 1944*t^16 + 29808*t^14 + 254016*t^12 + 1312848*t^10 +
4193856*t^8 + 8007552*t^6 + 8169984*t^4 + 3151872*t^2 - 221184 ], [ 0,
0, 0, -27*t^16 - 1080*t^14 - 18144*t^12 - 165888*t^10 - 892080*t^8 -
2830464*t^6 - 4955904*t^4 - 3870720*t^2 - 442368, 54*t^24 + 3240*t^22
+ 86832*t^20 + 1370304*t^18 + 14127696*t^16 + 99734976*t^14 +
490783104*t^12 + 1677749760*t^10 + 3883935744*t^8 + 5743927296*t^6 +
4782882816*t^4 + 1571291136*t^2 - 113246208 ], [ 0, 0, 0, -108*t^16 -
4320*t^14 - 72576*t^12 - 663552*t^10 - 3568320*t^8 - 11321856*t^6 -
19823616*t^4 - 15482880*t^2 - 1769472, -432*t^24 - 25920*t^22 -
694656*t^20 - 10962432*t^18 - 113021568*t^16 - 797879808*t^14 -
3926264832*t^12 - 13421998080*t^10 - 31071485952*t^8 - 45951418368*t^6
- 38263062528*t^4 - 12570329088*t^2 + 905969664 ], [ 0, 0, 0, -27*t^16
- 1080*t^14 - 18144*t^12 - 165888*t^10 - 892080*t^8 - 2830464*t^6 -
4955904*t^4 - 3870720*t^2 - 442368, -54*t^24 - 3240*t^22 - 86832*t^20
- 1370304*t^18 - 14127696*t^16 - 99734976*t^14 - 490783104*t^12 -
1677749760*t^10 - 3883935744*t^8 - 5743927296*t^6 - 4782882816*t^4 -
1571291136*t^2 + 113246208 ], [ 0, 0, 0, -108*t^16 - 4320*t^14 -
72576*t^12 - 663552*t^10 - 3568320*t^8 - 11321856*t^6 - 19823616*t^4 -
15482880*t^2 - 1769472, 432*t^24 + 25920*t^22 + 694656*t^20 +
10962432*t^18 + 113021568*t^16 + 797879808*t^14 + 3926264832*t^12 +
13421998080*t^10 + 31071485952*t^8 + 45951418368*t^6 + 38263062528*t^4
+ 12570329088*t^2 - 905969664 ], [ 0, 0, 0, -108*t^8 + 1728*t^6 -
8640*t^4 + 13824*t^2 - 1728, -432*t^12 + 10368*t^10 - 93312*t^8 +
387072*t^6 - 715392*t^4 + 414720*t^2 + 27648 ], [ 0, 0, 0, -27*t^8 +
432*t^6 - 2160*t^4 + 3456*t^2 - 432, -54*t^12 + 1296*t^10 - 11664*t^8
+ 48384*t^6 - 89424*t^4 + 51840*t^2 + 3456 ], [ 0, 0, 0, -108*t^8 +
1728*t^6 - 8640*t^4 + 13824*t^2 - 1728, 432*t^12 - 10368*t^10 +
93312*t^8 - 387072*t^6 + 715392*t^4 - 414720*t^2 - 27648 ], [ 0, 0, 0,
-27*t^8 + 432*t^6 - 2160*t^4 + 3456*t^2 - 432, 54*t^12 - 1296*t^10 +
11664*t^8 - 48384*t^6 + 89424*t^4 - 51840*t^2 - 3456 ], [ 0, 0, 0,
-108*t^12 + 3456*t^10 - 43200*t^8 + 262656*t^6 - 775872*t^4 +
912384*t^2 - 110592, 432*t^18 - 20736*t^16 + 425088*t^14 -
4838400*t^12 + 33229440*t^10 - 139677696*t^8 + 345461760*t^6 -
445243392*t^4 + 207028224*t^2 + 14155776 ], [ 0, 0, 0, -27*t^12 +
864*t^10 - 10800*t^8 + 65664*t^6 - 193968*t^4 + 228096*t^2 - 27648,
54*t^18 - 2592*t^16 + 53136*t^14 - 604800*t^12 + 4153680*t^10 -
17459712*t^8 + 43182720*t^6 - 55655424*t^4 + 25878528*t^2 + 1769472 ],
[ 0, 0, 0, -108*t^12 + 3456*t^10 - 43200*t^8 + 262656*t^6 - 775872*t^4
+ 912384*t^2 - 110592, -432*t^18 + 20736*t^16 - 425088*t^14 +
4838400*t^12 - 33229440*t^10 + 139677696*t^8 - 345461760*t^6 +
445243392*t^4 - 207028224*t^2 - 14155776 ], [ 0, 0, 0, -27*t^12 +
864*t^10 - 10800*t^8 + 65664*t^6 - 193968*t^4 + 228096*t^2 - 27648,
-54*t^18 + 2592*t^16 - 53136*t^14 + 604800*t^12 - 4153680*t^10 +
17459712*t^8 - 43182720*t^6 + 55655424*t^4 - 25878528*t^2 - 1769472 ],
[ 0, 0, 0, -108*t^12 + 2592*t^10 - 24192*t^8 + 110592*t^6 - 250560*t^4
+ 235008*t^2 - 27648, 432*t^18 - 15552*t^16 + 238464*t^14 -
2032128*t^12 + 10502784*t^10 - 33550848*t^8 + 64060416*t^6 -
65359872*t^4 + 25214976*t^2 + 1769472 ], [ 0, 0, 0, -27*t^12 +
648*t^10 - 6048*t^8 + 27648*t^6 - 62640*t^4 + 58752*t^2 - 6912,
54*t^18 - 1944*t^16 + 29808*t^14 - 254016*t^12 + 1312848*t^10 -
4193856*t^8 + 8007552*t^6 - 8169984*t^4 + 3151872*t^2 + 221184 ], [ 0,
0, 0, -108*t^12 + 2592*t^10 - 24192*t^8 + 110592*t^6 - 250560*t^4 +
235008*t^2 - 27648, -432*t^18 + 15552*t^16 - 238464*t^14 +
2032128*t^12 - 10502784*t^10 + 33550848*t^8 - 64060416*t^6 +
65359872*t^4 - 25214976*t^2 - 1769472 ], [ 0, 0, 0, -27*t^12 +
648*t^10 - 6048*t^8 + 27648*t^6 - 62640*t^4 + 58752*t^2 - 6912,
-54*t^18 + 1944*t^16 - 29808*t^14 + 254016*t^12 - 1312848*t^10 +
4193856*t^8 - 8007552*t^6 + 8169984*t^4 - 3151872*t^2 - 221184 ], [ 0,
0, 0, -27*t^16 + 1080*t^14 - 18144*t^12 + 165888*t^10 - 892080*t^8 +
2830464*t^6 - 4955904*t^4 + 3870720*t^2 - 442368, 54*t^24 - 3240*t^22
+ 86832*t^20 - 1370304*t^18 + 14127696*t^16 - 99734976*t^14 +
490783104*t^12 - 1677749760*t^10 + 3883935744*t^8 - 5743927296*t^6 +
4782882816*t^4 - 1571291136*t^2 - 113246208 ], [ 0, 0, 0, -108*t^16 +
4320*t^14 - 72576*t^12 + 663552*t^10 - 3568320*t^8 + 11321856*t^6 -
19823616*t^4 + 15482880*t^2 - 1769472, -432*t^24 + 25920*t^22 -
694656*t^20 + 10962432*t^18 - 113021568*t^16 + 797879808*t^14 -
3926264832*t^12 + 13421998080*t^10 - 31071485952*t^8 + 45951418368*t^6
- 38263062528*t^4 + 12570329088*t^2 + 905969664 ], [ 0, 0, 0, -27*t^16
+ 1080*t^14 - 18144*t^12 + 165888*t^10 - 892080*t^8 + 2830464*t^6 -
4955904*t^4 + 3870720*t^2 - 442368, -54*t^24 + 3240*t^22 - 86832*t^20
+ 1370304*t^18 - 14127696*t^16 + 99734976*t^14 - 490783104*t^12 +
1677749760*t^10 - 3883935744*t^8 + 5743927296*t^6 - 4782882816*t^4 +
1571291136*t^2 + 113246208 ], [ 0, 0, 0, -108*t^16 + 4320*t^14 -
72576*t^12 + 663552*t^10 - 3568320*t^8 + 11321856*t^6 - 19823616*t^4 +
15482880*t^2 - 1769472, 432*t^24 - 25920*t^22 + 694656*t^20 -
10962432*t^18 + 113021568*t^16 - 797879808*t^14 + 3926264832*t^12 -
13421998080*t^10 + 31071485952*t^8 - 45951418368*t^6 + 38263062528*t^4
- 12570329088*t^2 - 905969664 ], [ 0, 0, 0, -108*t^8 - 864*t^6 -
2160*t^4 - 1728*t^2 - 108, -432*t^12 - 5184*t^10 - 23328*t^8 -
48384*t^6 - 44712*t^4 - 12960*t^2 + 432 ], [ 0, 0, 0, -108*t^8 -
864*t^6 - 2160*t^4 - 1728*t^2 - 108, 432*t^12 + 5184*t^10 + 23328*t^8
+ 48384*t^6 + 44712*t^4 + 12960*t^2 - 432 ], [ 0, 0, 0, -27*t^8 -
216*t^6 - 540*t^4 - 432*t^2 - 27, -54*t^12 - 648*t^10 - 2916*t^8 -
6048*t^6 - 5589*t^4 - 1620*t^2 + 54 ], [ 0, 0, 0, -27*t^8 - 216*t^6 -
540*t^4 - 432*t^2 - 27, 54*t^12 + 648*t^10 + 2916*t^8 + 6048*t^6 +
5589*t^4 + 1620*t^2 - 54 ], [ 0, 0, 0, -27*t^12 - 432*t^10 - 2700*t^8
- 8208*t^6 - 12123*t^4 - 7128*t^2 - 432, -54*t^18 - 1296*t^16 -
13284*t^14 - 75600*t^12 - 259605*t^10 - 545616*t^8 - 674730*t^6 -
434808*t^4 - 101088*t^2 + 3456 ], [ 0, 0, 0, -27*t^12 - 432*t^10 -
2700*t^8 - 8208*t^6 - 12123*t^4 - 7128*t^2 - 432, 54*t^18 + 1296*t^16
+ 13284*t^14 + 75600*t^12 + 259605*t^10 + 545616*t^8 + 674730*t^6 +
434808*t^4 + 101088*t^2 - 3456 ], [ 0, 0, 0, -108*t^12 - 1728*t^10 -
10800*t^8 - 32832*t^6 - 48492*t^4 - 28512*t^2 - 1728, -432*t^18 -
10368*t^16 - 106272*t^14 - 604800*t^12 - 2076840*t^10 - 4364928*t^8 -
5397840*t^6 - 3478464*t^4 - 808704*t^2 + 27648 ], [ 0, 0, 0, -108*t^12
- 1728*t^10 - 10800*t^8 - 32832*t^6 - 48492*t^4 - 28512*t^2 - 1728,
432*t^18 + 10368*t^16 + 106272*t^14 + 604800*t^12 + 2076840*t^10 +
4364928*t^8 + 5397840*t^6 + 3478464*t^4 + 808704*t^2 - 27648 ], [ 0,
0, 0, -27*t^12 - 324*t^10 - 1512*t^8 - 3456*t^6 - 3915*t^4 - 1836*t^2
- 108, -54*t^18 - 972*t^16 - 7452*t^14 - 31752*t^12 - 82053*t^10 -
131058*t^8 - 125118*t^6 - 63828*t^4 - 12312*t^2 + 432 ], [ 0, 0, 0,
-27*t^12 - 324*t^10 - 1512*t^8 - 3456*t^6 - 3915*t^4 - 1836*t^2 - 108,
54*t^18 + 972*t^16 + 7452*t^14 + 31752*t^12 + 82053*t^10 + 131058*t^8
+ 125118*t^6 + 63828*t^4 + 12312*t^2 - 432 ], [ 0, 0, 0, -108*t^12 -
1296*t^10 - 6048*t^8 - 13824*t^6 - 15660*t^4 - 7344*t^2 - 432,
-432*t^18 - 7776*t^16 - 59616*t^14 - 254016*t^12 - 656424*t^10 -
1048464*t^8 - 1000944*t^6 - 510624*t^4 - 98496*t^2 + 3456 ], [ 0, 0,
0, -108*t^12 - 1296*t^10 - 6048*t^8 - 13824*t^6 - 15660*t^4 - 7344*t^2
- 432, 432*t^18 + 7776*t^16 + 59616*t^14 + 254016*t^12 + 656424*t^10 +
1048464*t^8 + 1000944*t^6 + 510624*t^4 + 98496*t^2 - 3456 ], [ 0, 0,
0, -27*t^16 - 540*t^14 - 4536*t^12 - 20736*t^10 - 55755*t^8 -
88452*t^6 - 77436*t^4 - 30240*t^2 - 1728, 54*t^24 + 1620*t^22 +
21708*t^20 + 171288*t^18 + 882981*t^16 + 3116718*t^14 + 7668486*t^12 +
13107420*t^10 + 15171624*t^8 + 11218608*t^6 + 4670784*t^4 + 767232*t^2
- 27648 ], [ 0, 0, 0, -108*t^16 - 2160*t^14 - 18144*t^12 - 82944*t^10
- 223020*t^8 - 353808*t^6 - 309744*t^4 - 120960*t^2 - 6912, -432*t^24
- 12960*t^22 - 173664*t^20 - 1370304*t^18 - 7063848*t^16 -
24933744*t^14 - 61347888*t^12 - 104859360*t^10 - 121372992*t^8 -
89748864*t^6 - 37366272*t^4 - 6137856*t^2 + 221184 ], [ 0, 0, 0,
-108*t^16 - 2160*t^14 - 18144*t^12 - 82944*t^10 - 223020*t^8 -
353808*t^6 - 309744*t^4 - 120960*t^2 - 6912, 432*t^24 + 12960*t^22 +
173664*t^20 + 1370304*t^18 + 7063848*t^16 + 24933744*t^14 +
61347888*t^12 + 104859360*t^10 + 121372992*t^8 + 89748864*t^6 +
37366272*t^4 + 6137856*t^2 - 221184 ], [ 0, 0, 0, -27*t^16 - 540*t^14
- 4536*t^12 - 20736*t^10 - 55755*t^8 - 88452*t^6 - 77436*t^4 -
30240*t^2 - 1728, -54*t^24 - 1620*t^22 - 21708*t^20 - 171288*t^18 -
882981*t^16 - 3116718*t^14 - 7668486*t^12 - 13107420*t^10 -
15171624*t^8 - 11218608*t^6 - 4670784*t^4 - 767232*t^2 + 27648 ], [ 0,
0, 0, -27648*t^8 + 55296*t^6 - 34560*t^4 + 6912*t^2 - 108,
1769472*t^12 - 5308416*t^10 + 5971968*t^8 - 3096576*t^6 + 715392*t^4 -
51840*t^2 - 432 ], [ 0, 0, 0, -27648*t^8 + 55296*t^6 - 34560*t^4 +
6912*t^2 - 108, -1769472*t^12 + 5308416*t^10 - 5971968*t^8 +
3096576*t^6 - 715392*t^4 + 51840*t^2 + 432 ], [ 0, 0, 0, -6912*t^8 +
13824*t^6 - 8640*t^4 + 1728*t^2 - 27, -221184*t^12 + 663552*t^10 -
746496*t^8 + 387072*t^6 - 89424*t^4 + 6480*t^2 + 54 ], [ 0, 0, 0,
-6912*t^8 + 13824*t^6 - 8640*t^4 + 1728*t^2 - 27, 221184*t^12 -
663552*t^10 + 746496*t^8 - 387072*t^6 + 89424*t^4 - 6480*t^2 - 54 ], [
0, 0, 0, -6912*t^12 + 27648*t^10 - 43200*t^8 + 32832*t^6 - 12123*t^4 +
1782*t^2 - 27, -221184*t^18 + 1327104*t^16 - 3400704*t^14 +
4838400*t^12 - 4153680*t^10 + 2182464*t^8 - 674730*t^6 + 108702*t^4 -
6318*t^2 - 54 ], [ 0, 0, 0, -6912*t^12 + 27648*t^10 - 43200*t^8 +
32832*t^6 - 12123*t^4 + 1782*t^2 - 27, 221184*t^18 - 1327104*t^16 +
3400704*t^14 - 4838400*t^12 + 4153680*t^10 - 2182464*t^8 + 674730*t^6
- 108702*t^4 + 6318*t^2 + 54 ], [ 0, 0, 0, -27648*t^12 + 110592*t^10 -
172800*t^8 + 131328*t^6 - 48492*t^4 + 7128*t^2 - 108, 1769472*t^18 -
10616832*t^16 + 27205632*t^14 - 38707200*t^12 + 33229440*t^10 -
17459712*t^8 + 5397840*t^6 - 869616*t^4 + 50544*t^2 + 432 ], [ 0, 0,
0, -27648*t^12 + 110592*t^10 - 172800*t^8 + 131328*t^6 - 48492*t^4 +
7128*t^2 - 108, -1769472*t^18 + 10616832*t^16 - 27205632*t^14 +
38707200*t^12 - 33229440*t^10 + 17459712*t^8 - 5397840*t^6 +
869616*t^4 - 50544*t^2 - 432 ], [ 0, 0, 0, -110592*t^12 + 331776*t^10
- 387072*t^8 + 221184*t^6 - 62640*t^4 + 7344*t^2 - 108, -14155776*t^18
+ 63700992*t^16 - 122093568*t^14 + 130056192*t^12 - 84022272*t^10 +
33550848*t^8 - 8007552*t^6 + 1021248*t^4 - 49248*t^2 - 432 ], [ 0, 0,
0, -110592*t^12 + 331776*t^10 - 387072*t^8 + 221184*t^6 - 62640*t^4 +
7344*t^2 - 108, 14155776*t^18 - 63700992*t^16 + 122093568*t^14 -
130056192*t^12 + 84022272*t^10 - 33550848*t^8 + 8007552*t^6 -
1021248*t^4 + 49248*t^2 + 432 ], [ 0, 0, 0, -27648*t^12 + 82944*t^10 -
96768*t^8 + 55296*t^6 - 15660*t^4 + 1836*t^2 - 27, 1769472*t^18 -
7962624*t^16 + 15261696*t^14 - 16257024*t^12 + 10502784*t^10 -
4193856*t^8 + 1000944*t^6 - 127656*t^4 + 6156*t^2 + 54 ], [ 0, 0, 0,
-27648*t^12 + 82944*t^10 - 96768*t^8 + 55296*t^6 - 15660*t^4 +
1836*t^2 - 27, -1769472*t^18 + 7962624*t^16 - 15261696*t^14 +
16257024*t^12 - 10502784*t^10 + 4193856*t^8 - 1000944*t^6 + 127656*t^4
- 6156*t^2 - 54 ], [ 0, 0, 0, -110592*t^16 + 552960*t^14 -
1161216*t^12 + 1327104*t^10 - 892080*t^8 + 353808*t^6 - 77436*t^4 +
7560*t^2 - 108, 14155776*t^24 - 106168320*t^22 + 355663872*t^20 -
701595648*t^18 + 904172544*t^16 - 797879808*t^14 + 490783104*t^12 -
209718720*t^10 + 60686496*t^8 - 11218608*t^6 + 1167696*t^4 - 47952*t^2
- 432 ], [ 0, 0, 0, -27648*t^16 + 138240*t^14 - 290304*t^12 +
331776*t^10 - 223020*t^8 + 88452*t^6 - 19359*t^4 + 1890*t^2 - 27,
1769472*t^24 - 13271040*t^22 + 44457984*t^20 - 87699456*t^18 +
113021568*t^16 - 99734976*t^14 + 61347888*t^12 - 26214840*t^10 +
7585812*t^8 - 1402326*t^6 + 145962*t^4 - 5994*t^2 - 54 ], [ 0, 0, 0,
-27648*t^16 + 138240*t^14 - 290304*t^12 + 331776*t^10 - 223020*t^8 +
88452*t^6 - 19359*t^4 + 1890*t^2 - 27, -1769472*t^24 + 13271040*t^22 -
44457984*t^20 + 87699456*t^18 - 113021568*t^16 + 99734976*t^14 -
61347888*t^12 + 26214840*t^10 - 7585812*t^8 + 1402326*t^6 - 145962*t^4
+ 5994*t^2 + 54 ], [ 0, 0, 0, -110592*t^16 + 552960*t^14 -
1161216*t^12 + 1327104*t^10 - 892080*t^8 + 353808*t^6 - 77436*t^4 +
7560*t^2 - 108, -14155776*t^24 + 106168320*t^22 - 355663872*t^20 +
701595648*t^18 - 904172544*t^16 + 797879808*t^14 - 490783104*t^12 +
209718720*t^10 - 60686496*t^8 + 11218608*t^6 - 1167696*t^4 + 47952*t^2
+ 432 ], [ 0, 0, 0, -189*t^20 - 5076*t^19 - 60696*t^18 - 438696*t^17 -
2203524*t^16 - 8351424*t^15 - 25090560*t^14 - 61160832*t^13 -
121204512*t^12 - 192516480*t^11 - 238028544*t^10 - 216822528*t^9 -
126831744*t^8 - 20487168*t^7 + 40144896*t^6 + 41306112*t^5 +
16899840*t^4 + 248832*t^3 - 2930688*t^2 - 1271808*t - 193536,
-918*t^30 - 35316*t^29 - 612684*t^28 - 6326208*t^27 - 42797160*t^26 -
191791152*t^25 - 494829648*t^24 + 102269952*t^23 + 7988961312*t^22 +
44932459968*t^21 + 160949726784*t^20 + 432735851520*t^19 +
920784143232*t^18 + 1583438637312*t^17 + 2211355365120*t^16 +
2484482015232*t^15 + 2174846049792*t^14 + 1352611685376*t^13 +
387801603072*t^12 - 295069630464*t^11 - 504433391616*t^10 -
359680020480*t^9 - 123138994176*t^8 + 21467234304*t^7 +
49863942144*t^6 + 26113425408*t^5 + 4654153728*t^4 - 1833172992*t^3 -
1324449792*t^2 - 323813376*t - 30081024 ], [ 0, 0, 0, -189*t^16 -
4320*t^15 - 41904*t^14 - 235008*t^13 - 892944*t^12 - 2547072*t^11 -
5711040*t^10 - 9856512*t^9 - 12141792*t^8 - 9220608*t^7 - 2315520*t^6
+ 2764800*t^5 + 2965248*t^4 + 940032*t^3 - 193536*t^2 - 221184*t -
48384, 918*t^24 + 29808*t^23 + 417312*t^22 + 3256416*t^21 +
14760144*t^20 + 30165696*t^19 - 71798400*t^18 - 814271616*t^17 -
3389120352*t^16 - 9109559808*t^15 - 17488825344*t^14 -
24583357440*t^13 - 24853989888*t^12 - 16374638592*t^11 -
3728166912*t^10 + 5567311872*t^9 + 7500335616*t^8 + 4125634560*t^7 +
321822720*t^6 - 1083580416*t^5 - 680472576*t^4 - 100417536*t^3 +
61046784*t^2 + 29196288*t + 3760128 ], [ 0, 0, 0, -756*t^16 -
17280*t^15 - 167616*t^14 - 940032*t^13 - 3571776*t^12 - 10188288*t^11
- 22844160*t^10 - 39426048*t^9 - 48567168*t^8 - 36882432*t^7 -
9262080*t^6 + 11059200*t^5 + 11860992*t^4 + 3760128*t^3 - 774144*t^2 -
884736*t - 193536, 7344*t^24 + 238464*t^23 + 3338496*t^22 +
26051328*t^21 + 118081152*t^20 + 241325568*t^19 - 574387200*t^18 -
6514172928*t^17 - 27112962816*t^16 - 72876478464*t^15 -
139910602752*t^14 - 196666859520*t^13 - 198831919104*t^12 -
130997108736*t^11 - 29825335296*t^10 + 44538494976*t^9 +
60002684928*t^8 + 33005076480*t^7 + 2574581760*t^6 - 8668643328*t^5 -
5443780608*t^4 - 803340288*t^3 + 488374272*t^2 + 233570304*t +
30081024 ], [ 0, 0, 0, -756*t^16 - 17280*t^15 - 167616*t^14 -
940032*t^13 - 3571776*t^12 - 10188288*t^11 - 22844160*t^10 -
39426048*t^9 - 48567168*t^8 - 36882432*t^7 - 9262080*t^6 +
11059200*t^5 + 11860992*t^4 + 3760128*t^3 - 774144*t^2 - 884736*t -
193536, -7344*t^24 - 238464*t^23 - 3338496*t^22 - 26051328*t^21 -
118081152*t^20 - 241325568*t^19 + 574387200*t^18 + 6514172928*t^17 +
27112962816*t^16 + 72876478464*t^15 + 139910602752*t^14 +
196666859520*t^13 + 198831919104*t^12 + 130997108736*t^11 +
29825335296*t^10 - 44538494976*t^9 - 60002684928*t^8 - 33005076480*t^7
- 2574581760*t^6 + 8668643328*t^5 + 5443780608*t^4 + 803340288*t^3 -
488374272*t^2 - 233570304*t - 30081024 ], [ 0, 0, 0, -189*t^20 -
5076*t^19 - 60696*t^18 - 438696*t^17 - 2203524*t^16 - 8351424*t^15 -
25090560*t^14 - 61160832*t^13 - 121204512*t^12 - 192516480*t^11 -
238028544*t^10 - 216822528*t^9 - 126831744*t^8 - 20487168*t^7 +
40144896*t^6 + 41306112*t^5 + 16899840*t^4 + 248832*t^3 - 2930688*t^2
- 1271808*t - 193536, 918*t^30 + 35316*t^29 + 612684*t^28 +
6326208*t^27 + 42797160*t^26 + 191791152*t^25 + 494829648*t^24 -
102269952*t^23 - 7988961312*t^22 - 44932459968*t^21 -
160949726784*t^20 - 432735851520*t^19 - 920784143232*t^18 -
1583438637312*t^17 - 2211355365120*t^16 - 2484482015232*t^15 -
2174846049792*t^14 - 1352611685376*t^13 - 387801603072*t^12 +
295069630464*t^11 + 504433391616*t^10 + 359680020480*t^9 +
123138994176*t^8 - 21467234304*t^7 - 49863942144*t^6 - 26113425408*t^5
- 4654153728*t^4 + 1833172992*t^3 + 1324449792*t^2 + 323813376*t +
30081024 ], [ 0, 0, 0, -189*t^16 - 4320*t^15 - 41904*t^14 -
235008*t^13 - 892944*t^12 - 2547072*t^11 - 5711040*t^10 - 9856512*t^9
- 12141792*t^8 - 9220608*t^7 - 2315520*t^6 + 2764800*t^5 + 2965248*t^4
+ 940032*t^3 - 193536*t^2 - 221184*t - 48384, -918*t^24 - 29808*t^23 -
417312*t^22 - 3256416*t^21 - 14760144*t^20 - 30165696*t^19 +
71798400*t^18 + 814271616*t^17 + 3389120352*t^16 + 9109559808*t^15 +
17488825344*t^14 + 24583357440*t^13 + 24853989888*t^12 +
16374638592*t^11 + 3728166912*t^10 - 5567311872*t^9 - 7500335616*t^8 -
4125634560*t^7 - 321822720*t^6 + 1083580416*t^5 + 680472576*t^4 +
100417536*t^3 - 61046784*t^2 - 29196288*t - 3760128 ], [ 0, 0, 0,
-1855425871872*t^24 - 4638564679680*t^22 - 4232690270208*t^20 -
1637993152512*t^18 - 311200579584*t^16 - 147220070400*t^14 -
60671655936*t^12 - 2300313600*t^10 - 75976704*t^8 - 6248448*t^6 -
252288*t^4 - 4320*t^2 - 27, 972777519512027136*t^36 +
3647915698170101760*t^34 + 5608670385936531456*t^32 +
4499096027743125504*t^30 + 1838207519781027840*t^28 +
84785540641062912*t^26 - 322413642903453696*t^24 -
163604031678185472*t^22 - 29631730994380800*t^20 -
1613379768680448*t^18 - 462995796787200*t^16 - 39942390546432*t^14 -
1229910441984*t^12 + 5053612032*t^10 + 1711964160*t^8 + 65470464*t^6 +
1275264*t^4 + 12960*t^2 + 54 ], [ 0, 0, 0, -1855425871872*t^24 -
4638564679680*t^22 - 4232690270208*t^20 - 1637993152512*t^18 -
311200579584*t^16 - 147220070400*t^14 - 60671655936*t^12 -
2300313600*t^10 - 75976704*t^8 - 6248448*t^6 - 252288*t^4 - 4320*t^2 -
27, -972777519512027136*t^36 - 3647915698170101760*t^34 -
5608670385936531456*t^32 - 4499096027743125504*t^30 -
1838207519781027840*t^28 - 84785540641062912*t^26 +
322413642903453696*t^24 + 163604031678185472*t^22 +
29631730994380800*t^20 + 1613379768680448*t^18 + 462995796787200*t^16
+ 39942390546432*t^14 + 1229910441984*t^12 - 5053612032*t^10 -
1711964160*t^8 - 65470464*t^6 - 1275264*t^4 - 12960*t^2 - 54 ], [ 0,
0, 0, -7421703487488*t^24 - 18554258718720*t^22 - 16930761080832*t^20
- 6551972610048*t^18 - 1244802318336*t^16 - 588880281600*t^14 -
242686623744*t^12 - 9201254400*t^10 - 303906816*t^8 - 24993792*t^6 -
1009152*t^4 - 17280*t^2 - 108, 7782220156096217088*t^36 +
29183325585360814080*t^34 + 44869363087492251648*t^32 +
35992768221945004032*t^30 + 14705660158248222720*t^28 +
678284325128503296*t^26 - 2579309143227629568*t^24 -
1308832253425483776*t^22 - 237053847955046400*t^20 -
12907038149443584*t^18 - 3703966374297600*t^16 - 319539124371456*t^14
- 9839283535872*t^12 + 40428896256*t^10 + 13695713280*t^8 +
523763712*t^6 + 10202112*t^4 + 103680*t^2 + 432 ], [ 0, 0, 0,
-7421703487488*t^24 - 18554258718720*t^22 - 16930761080832*t^20 -
6551972610048*t^18 - 1244802318336*t^16 - 588880281600*t^14 -
242686623744*t^12 - 9201254400*t^10 - 303906816*t^8 - 24993792*t^6 -
1009152*t^4 - 17280*t^2 - 108, -7782220156096217088*t^36 -
29183325585360814080*t^34 - 44869363087492251648*t^32 -
35992768221945004032*t^30 - 14705660158248222720*t^28 -
678284325128503296*t^26 + 2579309143227629568*t^24 +
1308832253425483776*t^22 + 237053847955046400*t^20 +
12907038149443584*t^18 + 3703966374297600*t^16 + 319539124371456*t^14
+ 9839283535872*t^12 - 40428896256*t^10 - 13695713280*t^8 -
523763712*t^6 - 10202112*t^4 - 103680*t^2 - 432 ], [ 0, 0, 0,
-452984832*t^16 - 452984832*t^14 - 84934656*t^12 + 7077888*t^10 -
25436160*t^8 + 110592*t^6 - 20736*t^4 - 1728*t^2 - 27,
3710851743744*t^24 + 5566277615616*t^22 + 2435246456832*t^20 +
202937204736*t^18 - 508248981504*t^16 - 236458082304*t^14 +
14665383936*t^12 - 3694657536*t^10 - 124084224*t^8 + 774144*t^6 +
145152*t^4 + 5184*t^2 + 54 ], [ 0, 0, 0, -452984832*t^16 -
452984832*t^14 - 84934656*t^12 + 7077888*t^10 - 25436160*t^8 +
110592*t^6 - 20736*t^4 - 1728*t^2 - 27, -3710851743744*t^24 -
5566277615616*t^22 - 2435246456832*t^20 - 202937204736*t^18 +
508248981504*t^16 + 236458082304*t^14 - 14665383936*t^12 +
3694657536*t^10 + 124084224*t^8 - 774144*t^6 - 145152*t^4 - 5184*t^2 -
54 ], [ 0, 0, 0, -1811939328*t^16 - 1811939328*t^14 - 339738624*t^12 +
28311552*t^10 - 101744640*t^8 + 442368*t^6 - 82944*t^4 - 6912*t^2 -
108, -29686813949952*t^24 - 44530220924928*t^22 - 19481971654656*t^20
- 1623497637888*t^18 + 4065991852032*t^16 + 1891664658432*t^14 -
117323071488*t^12 + 29557260288*t^10 + 992673792*t^8 - 6193152*t^6 -
1161216*t^4 - 41472*t^2 - 432 ], [ 0, 0, 0, -1811939328*t^16 -
1811939328*t^14 - 339738624*t^12 + 28311552*t^10 - 101744640*t^8 +
442368*t^6 - 82944*t^4 - 6912*t^2 - 108, 29686813949952*t^24 +
44530220924928*t^22 + 19481971654656*t^20 + 1623497637888*t^18 -
4065991852032*t^16 - 1891664658432*t^14 + 117323071488*t^12 -
29557260288*t^10 - 992673792*t^8 + 6193152*t^6 + 1161216*t^4 +
41472*t^2 + 432 ], [ 0, 0, 0, -108*t^24 - 216*t^20 - 1620*t^16 -
3024*t^12 - 1620*t^8 - 216*t^4 - 108, 432*t^36 + 1296*t^32 -
12960*t^28 - 42336*t^24 - 57024*t^20 - 57024*t^16 - 42336*t^12 -
12960*t^8 + 1296*t^4 + 432 ], [ 0, 0, 0, -108*t^24 + 216*t^20 -
1620*t^16 + 3024*t^12 - 1620*t^8 + 216*t^4 - 108, 432*t^36 - 1296*t^32
- 12960*t^28 + 42336*t^24 - 57024*t^20 + 57024*t^16 - 42336*t^12 +
12960*t^8 + 1296*t^4 - 432 ], [ 0, 0, 0, -27*t^32 - 324*t^24 +
702*t^16 - 324*t^8 - 27, 54*t^48 - 1944*t^40 + 3726*t^32 - 3726*t^16 +
1944*t^8 - 54 ], [ 0, 0, 0, -27*t^16 - 378*t^8 - 27, 54*t^24 -
1782*t^16 - 1782*t^8 + 54 ], [ 0, 0, 0, -108*t^16 - 1512*t^8 - 108,
432*t^24 - 14256*t^16 - 14256*t^8 + 432 ], [ 0, 0, 0, -108*t^32 -
1296*t^24 + 2808*t^16 - 1296*t^8 - 108, 432*t^48 - 15552*t^40 +
29808*t^32 - 29808*t^16 + 15552*t^8 - 432 ], [ 0, 0, 0, -27*t^24 +
54*t^20 - 405*t^16 + 756*t^12 - 405*t^8 + 54*t^4 - 27, 54*t^36 -
162*t^32 - 1620*t^28 + 5292*t^24 - 7128*t^20 + 7128*t^16 - 5292*t^12 +
1620*t^8 + 162*t^4 - 54 ], [ 0, 0, 0, -27*t^24 - 54*t^20 - 405*t^16 -
756*t^12 - 405*t^8 - 54*t^4 - 27, 54*t^36 + 162*t^32 - 1620*t^28 -
5292*t^24 - 7128*t^20 - 7128*t^16 - 5292*t^12 - 1620*t^8 + 162*t^4 +
54 ], [ 0, 0, 0, -27*t^16 - 378*t^8 - 27, -54*t^24 + 1782*t^16 +
1782*t^8 - 54 ], [ 0, 0, 0, -108*t^24 - 216*t^20 - 1620*t^16 -
3024*t^12 - 1620*t^8 - 216*t^4 - 108, -432*t^36 - 1296*t^32 +
12960*t^28 + 42336*t^24 + 57024*t^20 + 57024*t^16 + 42336*t^12 +
12960*t^8 - 1296*t^4 - 432 ], [ 0, 0, 0, -108*t^16 - 1512*t^8 - 108,
-432*t^24 + 14256*t^16 + 14256*t^8 - 432 ], [ 0, 0, 0, -27*t^24 -
54*t^20 - 405*t^16 - 756*t^12 - 405*t^8 - 54*t^4 - 27, -54*t^36 -
162*t^32 + 1620*t^28 + 5292*t^24 + 7128*t^20 + 7128*t^16 + 5292*t^12 +
1620*t^8 - 162*t^4 - 54 ], [ 0, 0, 0, -442368*t^24 - 6414336*t^20 -
414720*t^16 + 1022976*t^12 - 25920*t^8 - 25056*t^4 - 108,
113246208*t^36 - 3652190208*t^32 - 4671406080*t^28 + 1734082560*t^24 +
554729472*t^20 - 138682368*t^16 - 27095040*t^12 + 4561920*t^8 +
222912*t^4 - 432 ], [ 0, 0, 0, -1769472*t^32 - 26542080*t^28 -
14598144*t^24 + 1658880*t^20 + 1838592*t^16 + 103680*t^12 - 57024*t^8
- 6480*t^4 - 27, 905969664*t^48 - 28538044416*t^44 - 59114520576*t^40
- 19619905536*t^36 + 7378698240*t^32 + 4236115968*t^28 -
264757248*t^20 - 28823040*t^16 + 4790016*t^12 + 902016*t^8 + 27216*t^4
- 54 ], [ 0, 0, 0, -7077888*t^32 - 106168320*t^28 - 58392576*t^24 +
6635520*t^20 + 7354368*t^16 + 414720*t^12 - 228096*t^8 - 25920*t^4 -
108, 7247757312*t^48 - 228304355328*t^44 - 472916164608*t^40 -
156959244288*t^36 + 59029585920*t^32 + 33888927744*t^28 -
2118057984*t^20 - 230584320*t^16 + 38320128*t^12 + 7216128*t^8 +
217728*t^4 - 432 ], [ 0, 0, 0, -110592*t^24 - 1603584*t^20 -
103680*t^16 + 255744*t^12 - 6480*t^8 - 6264*t^4 - 27, 14155776*t^36 -
456523776*t^32 - 583925760*t^28 + 216760320*t^24 + 69341184*t^20 -
17335296*t^16 - 3386880*t^12 + 570240*t^8 + 27864*t^4 - 54 ], [ 0, 0,
0, -442368*t^24 - 6856704*t^20 - 7050240*t^16 - 2681856*t^12 -
440640*t^8 - 26784*t^4 - 108, -113246208*t^36 + 3482320896*t^32 +
10022289408*t^28 + 9314500608*t^24 + 4238770176*t^20 + 1059692544*t^16
+ 145539072*t^12 + 9787392*t^8 + 212544*t^4 - 432 ], [ 0, 0, 0,
-442368*t^24 - 6856704*t^20 - 7050240*t^16 - 2681856*t^12 - 440640*t^8
- 26784*t^4 - 108, 113246208*t^36 - 3482320896*t^32 - 10022289408*t^28
- 9314500608*t^24 - 4238770176*t^20 - 1059692544*t^16 - 145539072*t^12
- 9787392*t^8 - 212544*t^4 + 432 ], [ 0, 0, 0, -6912*t^16 -
103680*t^12 - 57888*t^8 - 6480*t^4 - 27, -221184*t^24 + 6967296*t^20 +
14390784*t^16 + 6096384*t^12 + 899424*t^8 + 27216*t^4 - 54 ], [ 0, 0,
0, -6912*t^16 - 103680*t^12 - 57888*t^8 - 6480*t^4 - 27, 221184*t^24 -
6967296*t^20 - 14390784*t^16 - 6096384*t^12 - 899424*t^8 - 27216*t^4 +
54 ], [ 0, 0, 0, -110592*t^24 - 1714176*t^20 - 1762560*t^16 -
670464*t^12 - 110160*t^8 - 6696*t^4 - 27, 14155776*t^36 -
435290112*t^32 - 1252786176*t^28 - 1164312576*t^24 - 529846272*t^20 -
132461568*t^16 - 18192384*t^12 - 1223424*t^8 - 26568*t^4 + 54 ], [ 0,
0, 0, -27648*t^16 - 414720*t^12 - 231552*t^8 - 25920*t^4 - 108,
-1769472*t^24 + 55738368*t^20 + 115126272*t^16 + 48771072*t^12 +
7195392*t^8 + 217728*t^4 - 432 ], [ 0, 0, 0, -27648*t^16 - 414720*t^12
- 231552*t^8 - 25920*t^4 - 108, 1769472*t^24 - 55738368*t^20 -
115126272*t^16 - 48771072*t^12 - 7195392*t^8 - 217728*t^4 + 432 ], [
0, 0, 0, -110592*t^24 - 1714176*t^20 - 1762560*t^16 - 670464*t^12 -
110160*t^8 - 6696*t^4 - 27, -14155776*t^36 + 435290112*t^32 +
1252786176*t^28 + 1164312576*t^24 + 529846272*t^20 + 132461568*t^16 +
18192384*t^12 + 1223424*t^8 + 26568*t^4 - 54 ], [ 0, 0, 0, -108*t^24 -
864*t^20 - 25920*t^16 - 193536*t^12 - 414720*t^8 - 221184*t^4 -
442368, 432*t^36 + 5184*t^32 - 207360*t^28 - 2709504*t^24 -
14598144*t^20 - 58392576*t^16 - 173408256*t^12 - 212336640*t^8 +
84934656*t^4 + 113246208 ], [ 0, 0, 0, -108*t^24 + 864*t^20 -
25920*t^16 + 193536*t^12 - 414720*t^8 + 221184*t^4 - 442368, 432*t^36
- 5184*t^32 - 207360*t^28 + 2709504*t^24 - 14598144*t^20 +
58392576*t^16 - 173408256*t^12 + 212336640*t^8 + 84934656*t^4 -
113246208 ], [ 0, 0, 0, -27*t^32 - 5184*t^24 + 179712*t^16 -
1327104*t^8 - 1769472, 54*t^48 - 31104*t^40 + 953856*t^32 -
244187136*t^16 + 2038431744*t^8 - 905969664 ], [ 0, 0, 0, -27*t^16 -
6048*t^8 - 6912, 54*t^24 - 28512*t^16 - 456192*t^8 + 221184 ], [ 0, 0,
0, -108*t^16 - 24192*t^8 - 27648, 432*t^24 - 228096*t^16 - 3649536*t^8
+ 1769472 ], [ 0, 0, 0, -108*t^32 - 20736*t^24 + 718848*t^16 -
5308416*t^8 - 7077888, 432*t^48 - 248832*t^40 + 7630848*t^32 -
1953497088*t^16 + 16307453952*t^8 - 7247757312 ], [ 0, 0, 0, -27*t^24
+ 216*t^20 - 6480*t^16 + 48384*t^12 - 103680*t^8 + 55296*t^4 - 110592,
54*t^36 - 648*t^32 - 25920*t^28 + 338688*t^24 - 1824768*t^20 +
7299072*t^16 - 21676032*t^12 + 26542080*t^8 + 10616832*t^4 - 14155776
], [ 0, 0, 0, -27*t^24 - 216*t^20 - 6480*t^16 - 48384*t^12 -
103680*t^8 - 55296*t^4 - 110592, 54*t^36 + 648*t^32 - 25920*t^28 -
338688*t^24 - 1824768*t^20 - 7299072*t^16 - 21676032*t^12 -
26542080*t^8 + 10616832*t^4 + 14155776 ], [ 0, 0, 0, -108*t^16 -
24192*t^8 - 27648, -432*t^24 + 228096*t^16 + 3649536*t^8 - 1769472 ],
[ 0, 0, 0, -27*t^24 - 216*t^20 - 6480*t^16 - 48384*t^12 - 103680*t^8 -
55296*t^4 - 110592, -54*t^36 - 648*t^32 + 25920*t^28 + 338688*t^24 +
1824768*t^20 + 7299072*t^16 + 21676032*t^12 + 26542080*t^8 -
10616832*t^4 - 14155776 ], [ 0, 0, 0, -27*t^16 - 6048*t^8 - 6912,
-54*t^24 + 28512*t^16 + 456192*t^8 - 221184 ], [ 0, 0, 0, -108*t^24 -
864*t^20 - 25920*t^16 - 193536*t^12 - 414720*t^8 - 221184*t^4 -
442368, -432*t^36 - 5184*t^32 + 207360*t^28 + 2709504*t^24 +
14598144*t^20 + 58392576*t^16 + 173408256*t^12 + 212336640*t^8 -
84934656*t^4 - 113246208 ], [ 0, 0, 0, -27*t^24 + 1080*t^22 -
15768*t^20 + 97632*t^18 - 296784*t^16 + 2246400*t^14 - 14812416*t^12 +
8985600*t^10 - 4748544*t^8 + 6248448*t^6 - 4036608*t^4 + 1105920*t^2 -
110592, -54*t^36 + 3240*t^34 - 79704*t^32 + 1022976*t^30 -
6687360*t^28 + 4935168*t^26 + 300271104*t^24 - 2437890048*t^22 +
7064755200*t^20 - 6154555392*t^18 + 28259020800*t^16 -
39006240768*t^14 + 19217350656*t^12 + 1263403008*t^10 - 6847856640*t^8
+ 4190109696*t^6 - 1305870336*t^4 + 212336640*t^2 - 14155776 ], [ 0,
0, 0, -27*t^24 + 1080*t^22 - 15768*t^20 + 97632*t^18 - 296784*t^16 +
2246400*t^14 - 14812416*t^12 + 8985600*t^10 - 4748544*t^8 +
6248448*t^6 - 4036608*t^4 + 1105920*t^2 - 110592, 54*t^36 - 3240*t^34
+ 79704*t^32 - 1022976*t^30 + 6687360*t^28 - 4935168*t^26 -
300271104*t^24 + 2437890048*t^22 - 7064755200*t^20 + 6154555392*t^18 -
28259020800*t^16 + 39006240768*t^14 - 19217350656*t^12 -
1263403008*t^10 + 6847856640*t^8 - 4190109696*t^6 + 1305870336*t^4 -
212336640*t^2 + 14155776 ], [ 0, 0, 0, -27*t^16 + 432*t^14 - 1296*t^12
- 1728*t^10 - 99360*t^8 - 6912*t^6 - 20736*t^4 + 27648*t^2 - 6912,
54*t^24 - 1296*t^22 + 9072*t^20 - 12096*t^18 - 484704*t^16 +
3608064*t^14 + 3580416*t^12 + 14432256*t^10 - 7755264*t^8 - 774144*t^6
+ 2322432*t^4 - 1327104*t^2 + 221184 ], [ 0, 0, 0, -27*t^16 + 432*t^14
- 1296*t^12 - 1728*t^10 - 99360*t^8 - 6912*t^6 - 20736*t^4 + 27648*t^2
- 6912, -54*t^24 + 1296*t^22 - 9072*t^20 + 12096*t^18 + 484704*t^16 -
3608064*t^14 - 3580416*t^12 - 14432256*t^10 + 7755264*t^8 + 774144*t^6
- 2322432*t^4 + 1327104*t^2 - 221184 ], [ 0, 0, 0, -108*t^24 +
4320*t^22 - 63072*t^20 + 390528*t^18 - 1187136*t^16 + 8985600*t^14 -
59249664*t^12 + 35942400*t^10 - 18994176*t^8 + 24993792*t^6 -
16146432*t^4 + 4423680*t^2 - 442368, 432*t^36 - 25920*t^34 +
637632*t^32 - 8183808*t^30 + 53498880*t^28 - 39481344*t^26 -
2402168832*t^24 + 19503120384*t^22 - 56518041600*t^20 +
49236443136*t^18 - 226072166400*t^16 + 312049926144*t^14 -
153738805248*t^12 - 10107224064*t^10 + 54782853120*t^8 -
33520877568*t^6 + 10446962688*t^4 - 1698693120*t^2 + 113246208 ], [ 0,
0, 0, -108*t^24 + 4320*t^22 - 63072*t^20 + 390528*t^18 - 1187136*t^16
+ 8985600*t^14 - 59249664*t^12 + 35942400*t^10 - 18994176*t^8 +
24993792*t^6 - 16146432*t^4 + 4423680*t^2 - 442368, -432*t^36 +
25920*t^34 - 637632*t^32 + 8183808*t^30 - 53498880*t^28 +
39481344*t^26 + 2402168832*t^24 - 19503120384*t^22 + 56518041600*t^20
- 49236443136*t^18 + 226072166400*t^16 - 312049926144*t^14 +
153738805248*t^12 + 10107224064*t^10 - 54782853120*t^8 +
33520877568*t^6 - 10446962688*t^4 + 1698693120*t^2 - 113246208 ], [ 0,
0, 0, -108*t^16 + 1728*t^14 - 5184*t^12 - 6912*t^10 - 397440*t^8 -
27648*t^6 - 82944*t^4 + 110592*t^2 - 27648, 432*t^24 - 10368*t^22 +
72576*t^20 - 96768*t^18 - 3877632*t^16 + 28864512*t^14 + 28643328*t^12
+ 115458048*t^10 - 62042112*t^8 - 6193152*t^6 + 18579456*t^4 -
10616832*t^2 + 1769472 ], [ 0, 0, 0, -108*t^16 + 1728*t^14 - 5184*t^12
- 6912*t^10 - 397440*t^8 - 27648*t^6 - 82944*t^4 + 110592*t^2 - 27648,
-432*t^24 + 10368*t^22 - 72576*t^20 + 96768*t^18 + 3877632*t^16 -
28864512*t^14 - 28643328*t^12 - 115458048*t^10 + 62042112*t^8 +
6193152*t^6 - 18579456*t^4 + 10616832*t^2 - 1769472 ], [ 0, 0, 0,
-108*t^30 - 6480*t^29 - 178416*t^28 - 2975616*t^27 - 33723648*t^26 -
281788416*t^25 - 1914734592*t^24 - 11913854976*t^23 - 71244693504*t^22
- 373498380288*t^21 - 1456889462784*t^20 - 3127591305216*t^19 +
3507376619520*t^18 + 56277929558016*t^17 + 221308004597760*t^16 +
450223436464128*t^15 + 224472103649280*t^14 - 1601326748270592*t^13 -
5967419239563264*t^12 - 12238794925277184*t^11 -
18676368933912576*t^10 - 24985164790628352*t^9 - 32123915832655872*t^8
- 37821000972238848*t^7 - 36210491315453952*t^6 -
25560346810908672*t^5 - 12260654161330176*t^4 - 3562417673994240*t^3 -
474989023199232*t^2, 432*t^45 + 38880*t^44 + 1653696*t^43 +
44136576*t^42 + 823592448*t^41 + 11235926016*t^40 + 111950622720*t^39
+ 751866126336*t^38 + 2004964835328*t^37 - 25262826061824*t^36 -
438158780006400*t^35 - 3850950467911680*t^34 - 23333660799270912*t^33
- 101170906172227584*t^32 - 289184878040186880*t^31 -
288135579445493760*t^30 + 2067625456781230080*t^29 +
14064993346057666560*t^28 + 49907810542647508992*t^27 +
119739588348368388096*t^26 + 184523079839391940608*t^25 -
1476184638715135524864*t^23 - 7663333654295576838144*t^22 -
25552798997835524603904*t^21 - 57610212745452202229760*t^20 -
67751950967807347261440*t^19 + 75533013338159516221440*t^18 +
606464645351733995765760*t^17 + 1697366145767195377926144*t^16 +
3131790938400805865127936*t^15 + 4134926579549140754104320*t^14 +
3763755261165493341388800*t^13 + 1736048187841128865726464*t^12 -
1102241074862564535435264*t^11 - 3306742193749324090834944*t^10 -
3938912365356929557463040*t^9 - 3162632013676253854826496*t^8 -
1854565320958977301807104*t^7 - 795093868908038307446784*t^6 -
238322710060290552102912*t^5 - 44825588099114210426880*t^4 -
3984496719921263149056*t^3 ], [ 0, 0, 0, -108*t^22 - 4752*t^21 -
95472*t^20 - 1157760*t^19 - 9821952*t^18 - 68345856*t^17 -
455196672*t^16 - 2924937216*t^15 - 16131391488*t^14 - 69713657856*t^13
- 227720429568*t^12 - 557709262848*t^11 - 1032409055232*t^10 -
1497567854592*t^9 - 1864485568512*t^8 - 2239557009408*t^7 -
2574765785088*t^6 - 2427998699520*t^5 - 1601754365952*t^4 -
637802643456*t^3 - 115964116992*t^2, 432*t^33 + 28512*t^32 +
886464*t^31 + 17248896*t^30 + 231704064*t^29 + 2179270656*t^28 +
12972662784*t^27 + 17820352512*t^26 - 593210179584*t^25 -
8041241640960*t^24 - 63458482323456*t^23 - 368682528669696*t^22 -
1703216683155456*t^21 - 6555356633235456*t^20 - 21848898033156096*t^19
- 65009179579908096*t^18 - 174791184265248768*t^17 -
419542824527069184*t^16 - 872046941775593472*t^15 -
1510123637431074816*t^14 - 2079407548775006208*t^13 -
2107963248727818240*t^12 - 1244051914534944768*t^11 +
298975903289966592*t^10 + 1741161324978634752*t^9 +
2339974049163116544*t^8 + 1990322754460581888*t^7 +
1185335107393683456*t^6 + 487338737802412032*t^5 +
125397102124597248*t^4 + 15199648742375424*t^3 ], [ 0, 0, 0, -27*t^24
- 1296*t^23 - 27648*t^22 - 342144*t^21 - 2761344*t^20 - 16782336*t^19
- 99090432*t^18 - 629710848*t^17 - 3314110464*t^16 - 10273554432*t^15
- 1811939328*t^14 + 132526374912*t^13 + 579112796160*t^12 +
1060210999296*t^11 - 115964116992*t^10 - 5260059869184*t^9 -
13574596460544*t^8 - 20634365067264*t^7 - 25975962206208*t^6 -
35195109507072*t^5 - 46327664738304*t^4 - 45921790328832*t^3 -
29686813949952*t^2 - 11132555231232*t - 1855425871872, 54*t^36 +
3888*t^35 + 129600*t^34 + 2643840*t^33 + 36236160*t^32 +
333766656*t^31 + 1728995328*t^30 - 2402058240*t^29 - 143965569024*t^28
- 1516370264064*t^27 - 9489961451520*t^26 - 37456098361344*t^25 -
71662738341888*t^24 + 139076762075136*t^23 + 1527831771217920*t^22 +
5504425254715392*t^21 + 11289182890622976*t^20 +
13175466089250816*t^19 - 105403728714006528*t^17 -
722507704999870464*t^16 - 2818265730414280704*t^15 -
6257998934908600320*t^14 - 4557267339678056448*t^13 +
18785956879895887872*t^12 + 78551131590689292288*t^11 +
159215133103824568320*t^10 + 203523771649430126592*t^9 +
154581852677027659776*t^8 + 20633523167774638080*t^7 -
118815654219148689408*t^6 - 183490159617956118528*t^5 -
159368317063806320640*t^4 - 93021850303337594880*t^3 -
36479156981701017600*t^2 - 8754997675608244224*t - 972777519512027136
], [ 0, 0, 0, -27*t^16 - 864*t^15 - 12096*t^14 - 96768*t^13 -
580608*t^12 - 3870720*t^11 - 27095040*t^10 - 142884864*t^9 -
500539392*t^8 - 1143078912*t^7 - 1734082560*t^6 - 1981808640*t^5 -
2378170368*t^4 - 3170893824*t^3 - 3170893824*t^2 - 1811939328*t -
452984832, 54*t^24 + 2592*t^23 + 57024*t^22 + 760320*t^21 +
6386688*t^20 + 25546752*t^19 - 127733760*t^18 - 2934226944*t^17 -
24999321600*t^16 - 138195763200*t^15 - 563838713856*t^14 -
1862107398144*t^13 - 5461864611840*t^12 - 14896859185152*t^11 -
36085677686784*t^10 - 70756230758400*t^9 - 102397221273600*t^8 -
96148748500992*t^7 - 33484638781440*t^6 + 53575422050304*t^5 +
107150844100608*t^4 + 102048422952960*t^3 + 61229053771776*t^2 +
22265110462464*t + 3710851743744 ], [ 0, 0, 0, -27*t^16 - 864*t^15 -
12096*t^14 - 96768*t^13 - 580608*t^12 - 3870720*t^11 - 27095040*t^10 -
142884864*t^9 - 500539392*t^8 - 1143078912*t^7 - 1734082560*t^6 -
1981808640*t^5 - 2378170368*t^4 - 3170893824*t^3 - 3170893824*t^2 -
1811939328*t - 452984832, -54*t^24 - 2592*t^23 - 57024*t^22 -
760320*t^21 - 6386688*t^20 - 25546752*t^19 + 127733760*t^18 +
2934226944*t^17 + 24999321600*t^16 + 138195763200*t^15 +
563838713856*t^14 + 1862107398144*t^13 + 5461864611840*t^12 +
14896859185152*t^11 + 36085677686784*t^10 + 70756230758400*t^9 +
102397221273600*t^8 + 96148748500992*t^7 + 33484638781440*t^6 -
53575422050304*t^5 - 107150844100608*t^4 - 102048422952960*t^3 -
61229053771776*t^2 - 22265110462464*t - 3710851743744 ], [ 0, 0, 0,
-27*t^26 - 1188*t^25 - 23868*t^24 - 287712*t^23 - 2382912*t^22 -
15697152*t^21 - 97804800*t^20 - 602228736*t^19 - 3165585408*t^18 -
11728060416*t^17 - 20960722944*t^16 + 50033590272*t^15 +
492550225920*t^14 + 1704978284544*t^13 + 3005441114112*t^12 +
244158824448*t^11 - 14363696037888*t^10 - 45311166775296*t^9 -
85311539380224*t^8 - 122414620999680*t^7 - 158725885132800*t^6 -
202125455917056*t^5 - 230768592814080*t^4 - 203169132969984*t^3 -
122458107543552*t^2 - 44530220924928*t - 7421703487488, 54*t^39 +
3564*t^38 + 110808*t^37 + 2150928*t^36 + 28631232*t^35 +
262414080*t^34 + 1433465856*t^33 - 203461632*t^32 - 95798329344*t^31 -
1119878479872*t^30 - 7908547166208*t^29 - 38634407460864*t^28 -
126787892281344*t^27 - 201115922595840*t^26 + 518215553777664*t^25 +
5016170797203456*t^24 + 19872910248247296*t^23 +
51908887229497344*t^22 + 91108353660420096*t^21 +
51825871417245696*t^20 - 457619337935585280*t^19 -
2827144407067656192*t^18 - 10302681684658618368*t^17 -
25528863944714747904*t^16 - 36529876903334510592*t^15 +
11637127164532359168*t^14 + 230689789166449852416*t^13 +
723700103713558364160*t^12 + 1420493385482694033408*t^11 +
1956539635174559711232*t^10 + 1833396830954066018304*t^9 +
859300741913637814272*t^8 - 551975244079363522560*t^7 -
1637458159016104427520*t^6 - 1902570432380616572928*t^5 -
1459531070837857714176*t^4 - 792327289642546102272*t^3 -
297669920970680303616*t^2 - 70039981404865953792*t -
7782220156096217088 ], [ 0, 0, 0, -27*t^26 - 1188*t^25 - 23868*t^24 -
287712*t^23 - 2382912*t^22 - 15697152*t^21 - 97804800*t^20 -
602228736*t^19 - 3165585408*t^18 - 11728060416*t^17 - 20960722944*t^16
+ 50033590272*t^15 + 492550225920*t^14 + 1704978284544*t^13 +
3005441114112*t^12 + 244158824448*t^11 - 14363696037888*t^10 -
45311166775296*t^9 - 85311539380224*t^8 - 122414620999680*t^7 -
158725885132800*t^6 - 202125455917056*t^5 - 230768592814080*t^4 -
203169132969984*t^3 - 122458107543552*t^2 - 44530220924928*t -
7421703487488, -54*t^39 - 3564*t^38 - 110808*t^37 - 2150928*t^36 -
28631232*t^35 - 262414080*t^34 - 1433465856*t^33 + 203461632*t^32 +
95798329344*t^31 + 1119878479872*t^30 + 7908547166208*t^29 +
38634407460864*t^28 + 126787892281344*t^27 + 201115922595840*t^26 -
518215553777664*t^25 - 5016170797203456*t^24 - 19872910248247296*t^23
- 51908887229497344*t^22 - 91108353660420096*t^21 -
51825871417245696*t^20 + 457619337935585280*t^19 +
2827144407067656192*t^18 + 10302681684658618368*t^17 +
25528863944714747904*t^16 + 36529876903334510592*t^15 -
11637127164532359168*t^14 - 230689789166449852416*t^13 -
723700103713558364160*t^12 - 1420493385482694033408*t^11 -
1956539635174559711232*t^10 - 1833396830954066018304*t^9 -
859300741913637814272*t^8 + 551975244079363522560*t^7 +
1637458159016104427520*t^6 + 1902570432380616572928*t^5 +
1459531070837857714176*t^4 + 792327289642546102272*t^3 +
297669920970680303616*t^2 + 70039981404865953792*t +
7782220156096217088 ], [ 0, 0, 0, -108*t^24 - 5184*t^23 - 110592*t^22
- 1368576*t^21 - 11045376*t^20 - 67129344*t^19 - 396361728*t^18 -
2518843392*t^17 - 13256441856*t^16 - 41094217728*t^15 -
7247757312*t^14 + 530105499648*t^13 + 2316451184640*t^12 +
4240843997184*t^11 - 463856467968*t^10 - 21040239476736*t^9 -
54298385842176*t^8 - 82537460269056*t^7 - 103903848824832*t^6 -
140780438028288*t^5 - 185310658953216*t^4 - 183687161315328*t^3 -
118747255799808*t^2 - 44530220924928*t - 7421703487488, 432*t^36 +
31104*t^35 + 1036800*t^34 + 21150720*t^33 + 289889280*t^32 +
2670133248*t^31 + 13831962624*t^30 - 19216465920*t^29 -
1151724552192*t^28 - 12130962112512*t^27 - 75919691612160*t^26 -
299648786890752*t^25 - 573301906735104*t^24 + 1112614096601088*t^23 +
12222654169743360*t^22 + 44035402037723136*t^21 +
90313463124983808*t^20 + 105403728714006528*t^19 -
843229829712052224*t^17 - 5780061639998963712*t^16 -
22546125843314245632*t^15 - 50063991479268802560*t^14 -
36458138717424451584*t^13 + 150287655039167102976*t^12 +
628409052725514338304*t^11 + 1273721064830596546560*t^10 +
1628190173195441012736*t^9 + 1236654821416221278208*t^8 +
165068185342197104640*t^7 - 950525233753189515264*t^6 -
1467921276943648948224*t^5 - 1274946536510450565120*t^4 -
744174802426700759040*t^3 - 291833255853608140800*t^2 -
70039981404865953792*t - 7782220156096217088 ], [ 0, 0, 0, -108*t^16 -
3456*t^15 - 48384*t^14 - 387072*t^13 - 2322432*t^12 - 15482880*t^11 -
108380160*t^10 - 571539456*t^9 - 2002157568*t^8 - 4572315648*t^7 -
6936330240*t^6 - 7927234560*t^5 - 9512681472*t^4 - 12683575296*t^3 -
12683575296*t^2 - 7247757312*t - 1811939328, -432*t^24 - 20736*t^23 -
456192*t^22 - 6082560*t^21 - 51093504*t^20 - 204374016*t^19 +
1021870080*t^18 + 23473815552*t^17 + 199994572800*t^16 +
1105566105600*t^15 + 4510709710848*t^14 + 14896859185152*t^13 +
43694916894720*t^12 + 119174873481216*t^11 + 288685421494272*t^10 +
566049846067200*t^9 + 819177770188800*t^8 + 769189988007936*t^7 +
267877110251520*t^6 - 428603376402432*t^5 - 857206752804864*t^4 -
816387383623680*t^3 - 489832430174208*t^2 - 178120883699712*t -
29686813949952 ], [ 0, 0, 0, -27*t^30 - 1620*t^29 - 44604*t^28 -
743904*t^27 - 8430912*t^26 - 70447104*t^25 - 478683648*t^24 -
2978463744*t^23 - 17811173376*t^22 - 93374595072*t^21 -
364222365696*t^20 - 781897826304*t^19 + 876844154880*t^18 +
14069482389504*t^17 + 55327001149440*t^16 + 112555859116032*t^15 +
56118025912320*t^14 - 400331687067648*t^13 - 1491854809890816*t^12 -
3059698731319296*t^11 - 4669092233478144*t^10 - 6246291197657088*t^9 -
8030978958163968*t^8 - 9455250243059712*t^7 - 9052622828863488*t^6 -
6390086702727168*t^5 - 3065163540332544*t^4 - 890604418498560*t^3 -
118747255799808*t^2, 54*t^45 + 4860*t^44 + 206712*t^43 + 5517072*t^42
+ 102949056*t^41 + 1404490752*t^40 + 13993827840*t^39 +
93983265792*t^38 + 250620604416*t^37 - 3157853257728*t^36 -
54769847500800*t^35 - 481368808488960*t^34 - 2916707599908864*t^33 -
12646363271528448*t^32 - 36148109755023360*t^31 -
36016947430686720*t^30 + 258453182097653760*t^29 +
1758124168257208320*t^28 + 6238476317830938624*t^27 +
14967448543546048512*t^26 + 23065384979923992576*t^25 -
184523079839391940608*t^23 - 957916706786947104768*t^22 -
3194099874729440575488*t^21 - 7201276593181525278720*t^20 -
8468993870975918407680*t^19 + 9441626667269939527680*t^18 +
75808080668966749470720*t^17 + 212170768220899422240768*t^16 +
391473867300100733140992*t^15 + 516865822443642594263040*t^14 +
470469407645686667673600*t^13 + 217006023480141108215808*t^12 -
137780134357820566929408*t^11 - 413342774218665511354368*t^10 -
492364045669616194682880*t^9 - 395329001709531731853312*t^8 -
231820665119872162725888*t^7 - 99386733613504788430848*t^6 -
29790338757536319012864*t^5 - 5603198512389276303360*t^4 -
498062089990157893632*t^3 ], [ 0, 0, 0, -27*t^22 - 1188*t^21 -
23868*t^20 - 289440*t^19 - 2455488*t^18 - 17086464*t^17 -
113799168*t^16 - 731234304*t^15 - 4032847872*t^14 - 17428414464*t^13 -
56930107392*t^12 - 139427315712*t^11 - 258102263808*t^10 -
374391963648*t^9 - 466121392128*t^8 - 559889252352*t^7 -
643691446272*t^6 - 606999674880*t^5 - 400438591488*t^4 -
159450660864*t^3 - 28991029248*t^2, 54*t^33 + 3564*t^32 + 110808*t^31
+ 2156112*t^30 + 28963008*t^29 + 272408832*t^28 + 1621582848*t^27 +
2227544064*t^26 - 74151272448*t^25 - 1005155205120*t^24 -
7932310290432*t^23 - 46085316083712*t^22 - 212902085394432*t^21 -
819419579154432*t^20 - 2731112254144512*t^19 - 8126147447488512*t^18 -
21848898033156096*t^17 - 52442853065883648*t^16 -
109005867721949184*t^15 - 188765454678884352*t^14 -
259925943596875776*t^13 - 263495406090977280*t^12 -
155506489316868096*t^11 + 37371987911245824*t^10 +
217645165622329344*t^9 + 292496756145389568*t^8 +
248790344307572736*t^7 + 148166888424210432*t^6 +
60917342225301504*t^5 + 15674637765574656*t^4 + 1899956092796928*t^3
], [ 0, 0, 0, -108*t^16 - 3456*t^15 - 48384*t^14 - 387072*t^13 -
2322432*t^12 - 15482880*t^11 - 108380160*t^10 - 571539456*t^9 -
2002157568*t^8 - 4572315648*t^7 - 6936330240*t^6 - 7927234560*t^5 -
9512681472*t^4 - 12683575296*t^3 - 12683575296*t^2 - 7247757312*t -
1811939328, 432*t^24 + 20736*t^23 + 456192*t^22 + 6082560*t^21 +
51093504*t^20 + 204374016*t^19 - 1021870080*t^18 - 23473815552*t^17 -
199994572800*t^16 - 1105566105600*t^15 - 4510709710848*t^14 -
14896859185152*t^13 - 43694916894720*t^12 - 119174873481216*t^11 -
288685421494272*t^10 - 566049846067200*t^9 - 819177770188800*t^8 -
769189988007936*t^7 - 267877110251520*t^6 + 428603376402432*t^5 +
857206752804864*t^4 + 816387383623680*t^3 + 489832430174208*t^2 +
178120883699712*t + 29686813949952 ], [ 0, 0, 0, -27*t^28 - 1296*t^27
- 28512*t^26 - 378432*t^25 - 3438720*t^24 - 24095232*t^23 -
151372800*t^22 - 933949440*t^21 - 5207003136*t^20 - 22124593152*t^19 -
56108187648*t^18 + 7629963264*t^17 + 751360278528*t^16 +
3409956569088*t^15 + 7880803614720*t^14 + 6404299554816*t^13 -
21463780294656*t^12 - 96076270927872*t^11 - 207459805298688*t^10 -
315741299539968*t^9 - 410223063859200*t^8 - 526709019377664*t^7 -
639658069327872*t^6 - 617856815333376*t^5 - 410049117683712*t^4 -
163277476724736*t^3 - 29686813949952*t^2, -54*t^42 - 3888*t^41 -
132192*t^40 - 2814912*t^39 - 41482368*t^38 - 432594432*t^37 -
2978519040*t^36 - 8032296960*t^35 + 100035772416*t^34 +
1707483856896*t^33 + 14577356242944*t^32 + 84668003647488*t^31 +
345087100846080*t^30 + 880011707940864*t^29 + 355136937394176*t^28 -
8918426978353152*t^27 - 49861062392020992*t^26 -
160979401322790912*t^25 - 353393050883457024*t^24 -
457619337935585280*t^23 + 414606971337965568*t^22 +
5830934634266886144*t^21 + 26577350261502640128*t^20 +
81399440376820924416*t^19 + 164369884682762846208*t^18 +
135847098129491951616*t^17 - 421770659303711047680*t^16 -
2127147854988841058304*t^15 - 5185422392023414996992*t^14 -
8491737859434209083392*t^13 - 9619682893088868532224*t^12 -
6583211064702674141184*t^11 - 111854215095140745216*t^10 +
6304449506767508865024*t^9 + 9232874632068527554560*t^8 +
8058975360397388808192*t^7 + 4843459269650383110144*t^6 +
1996139470038679683072*t^5 + 513626530302350327808*t^4 +
62257761248769736704*t^3 ], [ 0, 0, 0, -27*t^28 - 1296*t^27 -
28512*t^26 - 378432*t^25 - 3438720*t^24 - 24095232*t^23 -
151372800*t^22 - 933949440*t^21 - 5207003136*t^20 - 22124593152*t^19 -
56108187648*t^18 + 7629963264*t^17 + 751360278528*t^16 +
3409956569088*t^15 + 7880803614720*t^14 + 6404299554816*t^13 -
21463780294656*t^12 - 96076270927872*t^11 - 207459805298688*t^10 -
315741299539968*t^9 - 410223063859200*t^8 - 526709019377664*t^7 -
639658069327872*t^6 - 617856815333376*t^5 - 410049117683712*t^4 -
163277476724736*t^3 - 29686813949952*t^2, 54*t^42 + 3888*t^41 +
132192*t^40 + 2814912*t^39 + 41482368*t^38 + 432594432*t^37 +
2978519040*t^36 + 8032296960*t^35 - 100035772416*t^34 -
1707483856896*t^33 - 14577356242944*t^32 - 84668003647488*t^31 -
345087100846080*t^30 - 880011707940864*t^29 - 355136937394176*t^28 +
8918426978353152*t^27 + 49861062392020992*t^26 +
160979401322790912*t^25 + 353393050883457024*t^24 +
457619337935585280*t^23 - 414606971337965568*t^22 -
5830934634266886144*t^21 - 26577350261502640128*t^20 -
81399440376820924416*t^19 - 164369884682762846208*t^18 -
135847098129491951616*t^17 + 421770659303711047680*t^16 +
2127147854988841058304*t^15 + 5185422392023414996992*t^14 +
8491737859434209083392*t^13 + 9619682893088868532224*t^12 +
6583211064702674141184*t^11 + 111854215095140745216*t^10 -
6304449506767508865024*t^9 - 9232874632068527554560*t^8 -
8058975360397388808192*t^7 - 4843459269650383110144*t^6 -
1996139470038679683072*t^5 - 513626530302350327808*t^4 -
62257761248769736704*t^3 ], [ 0, 0, 0, -3159*t^24 - 160704*t^23 -
4068576*t^22 - 68470272*t^21 - 863657856*t^20 - 8705470464*t^19 -
72648382464*t^18 - 510826217472*t^17 - 3046704648192*t^16 -
15428677533696*t^15 - 66259485720576*t^14 - 240950446129152*t^13 -
741147367440384*t^12 - 1927603569033216*t^11 - 4240607086116864*t^10 -
7899482897252352*t^9 - 12479302238994432*t^8 - 16738753494122496*t^7 -
19044337572642816*t^6 - 18256694794518528*t^5 - 14489774400208896*t^4
- 9189924343382016*t^3 - 4368600215322624*t^2 - 1380436848672768*t -
217084827009024, 51030*t^36 + 3312576*t^35 + 97689888*t^34 +
1600086528*t^33 + 11347019136*t^32 - 137995591680*t^31 -
5669491802112*t^30 - 100619809652736*t^29 - 1262385912840192*t^28 -
12491370505175040*t^27 - 102183265651654656*t^26 -
709551419585200128*t^25 - 4255210213459623936*t^24 -
22314137912271175680*t^23 - 103293962741474131968*t^22 -
425241710393699598336*t^21 - 1565939322271958040576*t^20 -
5179933655438123335680*t^19 - 15431571361347888218112*t^18 -
41439469243504986685440*t^17 - 100220116625405314596864*t^16 -
217723755721574194348032*t^15 - 423092071389078044540928*t^14 -
731189671109301884682240*t^13 - 1115477826197159657078784*t^12 -
1488037178685941618835456*t^11 - 1714350719423190921117696*t^10 -
1676563368810806111109120*t^9 - 1355476552644932778590208*t^8 -
864317583576492473843712*t^7 - 389604510000178299666432*t^6 -
75863878816994521251840*t^5 + 49904717922515124486144*t^4 +
56298039775479302455296*t^3 + 27497258949666593046528*t^2 +
7459258019618224078848*t + 919274755938865643520 ], [ 0, 0, 0,
-3159*t^24 - 160704*t^23 - 4068576*t^22 - 68470272*t^21 -
863657856*t^20 - 8705470464*t^19 - 72648382464*t^18 -
510826217472*t^17 - 3046704648192*t^16 - 15428677533696*t^15 -
66259485720576*t^14 - 240950446129152*t^13 - 741147367440384*t^12 -
1927603569033216*t^11 - 4240607086116864*t^10 - 7899482897252352*t^9 -
12479302238994432*t^8 - 16738753494122496*t^7 - 19044337572642816*t^6
- 18256694794518528*t^5 - 14489774400208896*t^4 - 9189924343382016*t^3
- 4368600215322624*t^2 - 1380436848672768*t - 217084827009024,
-51030*t^36 - 3312576*t^35 - 97689888*t^34 - 1600086528*t^33 -
11347019136*t^32 + 137995591680*t^31 + 5669491802112*t^30 +
100619809652736*t^29 + 1262385912840192*t^28 + 12491370505175040*t^27
+ 102183265651654656*t^26 + 709551419585200128*t^25 +
4255210213459623936*t^24 + 22314137912271175680*t^23 +
103293962741474131968*t^22 + 425241710393699598336*t^21 +
1565939322271958040576*t^20 + 5179933655438123335680*t^19 +
15431571361347888218112*t^18 + 41439469243504986685440*t^17 +
100220116625405314596864*t^16 + 217723755721574194348032*t^15 +
423092071389078044540928*t^14 + 731189671109301884682240*t^13 +
1115477826197159657078784*t^12 + 1488037178685941618835456*t^11 +
1714350719423190921117696*t^10 + 1676563368810806111109120*t^9 +
1355476552644932778590208*t^8 + 864317583576492473843712*t^7 +
389604510000178299666432*t^6 + 75863878816994521251840*t^5 -
49904717922515124486144*t^4 - 56298039775479302455296*t^3 -
27497258949666593046528*t^2 - 7459258019618224078848*t -
919274755938865643520 ], [ 0, 0, 0, -351*t^16 - 10368*t^15 -
157248*t^14 - 1658880*t^13 - 13706496*t^12 - 91570176*t^11 -
488927232*t^10 - 2043740160*t^9 - 6594600960*t^8 - 16349921280*t^7 -
31291342848*t^6 - 46883930112*t^5 - 56141807616*t^4 - 54358179840*t^3
- 41221619712*t^2 - 21743271936*t - 5888802816, 1890*t^24 + 62208*t^23
+ 710208*t^22 - 2239488*t^21 - 190003968*t^20 - 3218890752*t^19 -
34154459136*t^18 - 266174595072*t^17 - 1624896184320*t^16 -
8064035979264*t^15 - 33396264271872*t^14 - 117705164193792*t^13 -
357696245071872*t^12 - 941641313550336*t^11 - 2137360913399808*t^10 -
4128786421383168*t^9 - 6655574770974720*t^8 - 8722009131319296*t^7 -
8953386535747584*t^6 - 6750503178338304*t^5 - 3187737611993088*t^4 -
300578991243264*t^3 + 762580033339392*t^2 + 534362651099136*t +
129879811031040 ], [ 0, 0, 0, -351*t^16 - 10368*t^15 - 157248*t^14 -
1658880*t^13 - 13706496*t^12 - 91570176*t^11 - 488927232*t^10 -
2043740160*t^9 - 6594600960*t^8 - 16349921280*t^7 - 31291342848*t^6 -
46883930112*t^5 - 56141807616*t^4 - 54358179840*t^3 - 41221619712*t^2
- 21743271936*t - 5888802816, -1890*t^24 - 62208*t^23 - 710208*t^22 +
2239488*t^21 + 190003968*t^20 + 3218890752*t^19 + 34154459136*t^18 +
266174595072*t^17 + 1624896184320*t^16 + 8064035979264*t^15 +
33396264271872*t^14 + 117705164193792*t^13 + 357696245071872*t^12 +
941641313550336*t^11 + 2137360913399808*t^10 + 4128786421383168*t^9 +
6655574770974720*t^8 + 8722009131319296*t^7 + 8953386535747584*t^6 +
6750503178338304*t^5 + 3187737611993088*t^4 + 300578991243264*t^3 -
762580033339392*t^2 - 534362651099136*t - 129879811031040 ], [ 0, 0,
0, -12636*t^24 - 642816*t^23 - 16274304*t^22 - 273881088*t^21 -
3454631424*t^20 - 34821881856*t^19 - 290593529856*t^18 -
2043304869888*t^17 - 12186818592768*t^16 - 61714710134784*t^15 -
265037942882304*t^14 - 963801784516608*t^13 - 2964589469761536*t^12 -
7710414276132864*t^11 - 16962428344467456*t^10 - 31597931589009408*t^9
- 49917208955977728*t^8 - 66955013976489984*t^7 -
76177350290571264*t^6 - 73026779178074112*t^5 - 57959097600835584*t^4
- 36759697373528064*t^3 - 17474400861290496*t^2 - 5521747394691072*t -
868339308036096, 408240*t^36 + 26500608*t^35 + 781519104*t^34 +
12800692224*t^33 + 90776153088*t^32 - 1103964733440*t^31 -
45355934416896*t^30 - 804958477221888*t^29 - 10099087302721536*t^28 -
99930964041400320*t^27 - 817466125213237248*t^26 -
5676411356681601024*t^25 - 34041681707676991488*t^24 -
178513103298169405440*t^23 - 826351701931793055744*t^22 -
3401933683149596786688*t^21 - 12527514578175664324608*t^20 -
41439469243504986685440*t^19 - 123452570890783105744896*t^18 -
331515753948039893483520*t^17 - 801760933003242516774912*t^16 -
1741790045772593554784256*t^15 - 3384736571112624356327424*t^14 -
5849517368874415077457920*t^13 - 8923822609577277256630272*t^12 -
11904297429487532950683648*t^11 - 13714805755385527368941568*t^10 -
13412506950486448888872960*t^9 - 10843812421159462228721664*t^8 -
6914540668611939790749696*t^7 - 3116836080001426397331456*t^6 -
606911030535956170014720*t^5 + 399237743380120995889152*t^4 +
450384318203834419642368*t^3 + 219978071597332744372224*t^2 +
59674064156945792630784*t + 7354198047510925148160 ], [ 0, 0, 0,
-12636*t^24 - 642816*t^23 - 16274304*t^22 - 273881088*t^21 -
3454631424*t^20 - 34821881856*t^19 - 290593529856*t^18 -
2043304869888*t^17 - 12186818592768*t^16 - 61714710134784*t^15 -
265037942882304*t^14 - 963801784516608*t^13 - 2964589469761536*t^12 -
7710414276132864*t^11 - 16962428344467456*t^10 - 31597931589009408*t^9
- 49917208955977728*t^8 - 66955013976489984*t^7 -
76177350290571264*t^6 - 73026779178074112*t^5 - 57959097600835584*t^4
- 36759697373528064*t^3 - 17474400861290496*t^2 - 5521747394691072*t -
868339308036096, -408240*t^36 - 26500608*t^35 - 781519104*t^34 -
12800692224*t^33 - 90776153088*t^32 + 1103964733440*t^31 +
45355934416896*t^30 + 804958477221888*t^29 + 10099087302721536*t^28 +
99930964041400320*t^27 + 817466125213237248*t^26 +
5676411356681601024*t^25 + 34041681707676991488*t^24 +
178513103298169405440*t^23 + 826351701931793055744*t^22 +
3401933683149596786688*t^21 + 12527514578175664324608*t^20 +
41439469243504986685440*t^19 + 123452570890783105744896*t^18 +
331515753948039893483520*t^17 + 801760933003242516774912*t^16 +
1741790045772593554784256*t^15 + 3384736571112624356327424*t^14 +
5849517368874415077457920*t^13 + 8923822609577277256630272*t^12 +
11904297429487532950683648*t^11 + 13714805755385527368941568*t^10 +
13412506950486448888872960*t^9 + 10843812421159462228721664*t^8 +
6914540668611939790749696*t^7 + 3116836080001426397331456*t^6 +
606911030535956170014720*t^5 - 399237743380120995889152*t^4 -
450384318203834419642368*t^3 - 219978071597332744372224*t^2 -
59674064156945792630784*t - 7354198047510925148160 ], [ 0, 0, 0,
-1404*t^16 - 41472*t^15 - 628992*t^14 - 6635520*t^13 - 54825984*t^12 -
366280704*t^11 - 1955708928*t^10 - 8174960640*t^9 - 26378403840*t^8 -
65399685120*t^7 - 125165371392*t^6 - 187535720448*t^5 -
224567230464*t^4 - 217432719360*t^3 - 164886478848*t^2 - 86973087744*t
- 23555211264, -15120*t^24 - 497664*t^23 - 5681664*t^22 +
17915904*t^21 + 1520031744*t^20 + 25751126016*t^19 + 273235673088*t^18
+ 2129396760576*t^17 + 12999169474560*t^16 + 64512287834112*t^15 +
267170114174976*t^14 + 941641313550336*t^13 + 2861569960574976*t^12 +
7533130508402688*t^11 + 17098887307198464*t^10 + 33030291371065344*t^9
+ 53244598167797760*t^8 + 69776073050554368*t^7 +
71627092285980672*t^6 + 54004025426706432*t^5 + 25501900895944704*t^4
+ 2404631929946112*t^3 - 6100640266715136*t^2 - 4274901208793088*t -
1039038488248320 ], [ 0, 0, 0, -1404*t^16 - 41472*t^15 - 628992*t^14 -
6635520*t^13 - 54825984*t^12 - 366280704*t^11 - 1955708928*t^10 -
8174960640*t^9 - 26378403840*t^8 - 65399685120*t^7 - 125165371392*t^6
- 187535720448*t^5 - 224567230464*t^4 - 217432719360*t^3 -
164886478848*t^2 - 86973087744*t - 23555211264, 15120*t^24 +
497664*t^23 + 5681664*t^22 - 17915904*t^21 - 1520031744*t^20 -
25751126016*t^19 - 273235673088*t^18 - 2129396760576*t^17 -
12999169474560*t^16 - 64512287834112*t^15 - 267170114174976*t^14 -
941641313550336*t^13 - 2861569960574976*t^12 - 7533130508402688*t^11 -
17098887307198464*t^10 - 33030291371065344*t^9 - 53244598167797760*t^8
- 69776073050554368*t^7 - 71627092285980672*t^6 -
54004025426706432*t^5 - 25501900895944704*t^4 - 2404631929946112*t^3 +
6100640266715136*t^2 + 4274901208793088*t + 1039038488248320 ], [ 0,
0, 0, -27*t^26 + 216*t^25 - 25488*t^24 + 201312*t^23 - 719712*t^22 +
3328128*t^21 - 5412096*t^20 - 9580032*t^19 - 8729856*t^18 -
79958016*t^17 + 64806912*t^16 - 4866048*t^15 + 421576704*t^14 +
19464192*t^13 + 1036910592*t^12 + 5117313024*t^11 - 2234843136*t^10 +
9809952768*t^9 - 22167945216*t^8 - 54528049152*t^7 - 47167045632*t^6 -
52772732928*t^5 - 26726105088*t^4 - 905969664*t^3 - 452984832*t^2,
54*t^39 - 648*t^38 - 108864*t^37 + 1331424*t^36 - 15310944*t^35 +
132212736*t^34 - 525256704*t^33 + 1266057216*t^32 - 3700131840*t^31 +
4078854144*t^30 + 5130584064*t^29 - 35186835456*t^28 +
239444066304*t^27 - 63616057344*t^26 + 284488630272*t^25 +
747594842112*t^24 - 5970290540544*t^23 - 3552222117888*t^22 -
14208888471552*t^20 + 95524648648704*t^19 + 47846069895168*t^18 -
72829089349632*t^17 - 65142842720256*t^16 - 980762895581184*t^15 -
576501112111104*t^14 - 336237957218304*t^13 + 1069247140724736*t^12 +
3879869444259840*t^11 + 5310228845297664*t^10 + 8812345178456064*t^9 +
8872646519291904*t^8 + 4110000234430464*t^7 + 1429605634277376*t^6 +
467567319711744*t^5 - 11132555231232*t^4 - 3710851743744*t^3 ], [ 0,
0, 0, -27*t^26 - 216*t^25 - 25488*t^24 - 201312*t^23 - 719712*t^22 -
3328128*t^21 - 5412096*t^20 + 9580032*t^19 - 8729856*t^18 +
79958016*t^17 + 64806912*t^16 + 4866048*t^15 + 421576704*t^14 -
19464192*t^13 + 1036910592*t^12 - 5117313024*t^11 - 2234843136*t^10 -
9809952768*t^9 - 22167945216*t^8 + 54528049152*t^7 - 47167045632*t^6 +
52772732928*t^5 - 26726105088*t^4 + 905969664*t^3 - 452984832*t^2,
54*t^39 + 648*t^38 - 108864*t^37 - 1331424*t^36 - 15310944*t^35 -
132212736*t^34 - 525256704*t^33 - 1266057216*t^32 - 3700131840*t^31 -
4078854144*t^30 + 5130584064*t^29 + 35186835456*t^28 +
239444066304*t^27 + 63616057344*t^26 + 284488630272*t^25 -
747594842112*t^24 - 5970290540544*t^23 + 3552222117888*t^22 +
14208888471552*t^20 + 95524648648704*t^19 - 47846069895168*t^18 -
72829089349632*t^17 + 65142842720256*t^16 - 980762895581184*t^15 +
576501112111104*t^14 - 336237957218304*t^13 - 1069247140724736*t^12 +
3879869444259840*t^11 - 5310228845297664*t^10 + 8812345178456064*t^9 -
8872646519291904*t^8 + 4110000234430464*t^7 - 1429605634277376*t^6 +
467567319711744*t^5 + 11132555231232*t^4 - 3710851743744*t^3 ], [ 0,
0, 0, -108*t^26 - 864*t^25 - 101952*t^24 - 805248*t^23 - 2878848*t^22
- 13312512*t^21 - 21648384*t^20 + 38320128*t^19 - 34919424*t^18 +
319832064*t^17 + 259227648*t^16 + 19464192*t^15 + 1686306816*t^14 -
77856768*t^13 + 4147642368*t^12 - 20469252096*t^11 - 8939372544*t^10 -
39239811072*t^9 - 88671780864*t^8 + 218112196608*t^7 -
188668182528*t^6 + 211090931712*t^5 - 106904420352*t^4 +
3623878656*t^3 - 1811939328*t^2, 432*t^39 + 5184*t^38 - 870912*t^37 -
10651392*t^36 - 122487552*t^35 - 1057701888*t^34 - 4202053632*t^33 -
10128457728*t^32 - 29601054720*t^31 - 32630833152*t^30 +
41044672512*t^29 + 281494683648*t^28 + 1915552530432*t^27 +
508928458752*t^26 + 2275909042176*t^25 - 5980758736896*t^24 -
47762324324352*t^23 + 28417776943104*t^22 + 113671107772416*t^20 +
764197189189632*t^19 - 382768559161344*t^18 - 582632714797056*t^17 +
521142741762048*t^16 - 7846103164649472*t^15 + 4612008896888832*t^14 -
2689903657746432*t^13 - 8553977125797888*t^12 + 31038955554078720*t^11
- 42481830762381312*t^10 + 70498761427648512*t^9 -
70981172154335232*t^8 + 32880001875443712*t^7 - 11436845074219008*t^6
+ 3740538557693952*t^5 + 89060441849856*t^4 - 29686813949952*t^3 ], [
0, 0, 0, -108*t^26 + 864*t^25 - 101952*t^24 + 805248*t^23 -
2878848*t^22 + 13312512*t^21 - 21648384*t^20 - 38320128*t^19 -
34919424*t^18 - 319832064*t^17 + 259227648*t^16 - 19464192*t^15 +
1686306816*t^14 + 77856768*t^13 + 4147642368*t^12 + 20469252096*t^11 -
8939372544*t^10 + 39239811072*t^9 - 88671780864*t^8 - 218112196608*t^7
- 188668182528*t^6 - 211090931712*t^5 - 106904420352*t^4 -
3623878656*t^3 - 1811939328*t^2, 432*t^39 - 5184*t^38 - 870912*t^37 +
10651392*t^36 - 122487552*t^35 + 1057701888*t^34 - 4202053632*t^33 +
10128457728*t^32 - 29601054720*t^31 + 32630833152*t^30 +
41044672512*t^29 - 281494683648*t^28 + 1915552530432*t^27 -
508928458752*t^26 + 2275909042176*t^25 + 5980758736896*t^24 -
47762324324352*t^23 - 28417776943104*t^22 - 113671107772416*t^20 +
764197189189632*t^19 + 382768559161344*t^18 - 582632714797056*t^17 -
521142741762048*t^16 - 7846103164649472*t^15 - 4612008896888832*t^14 -
2689903657746432*t^13 + 8553977125797888*t^12 + 31038955554078720*t^11
+ 42481830762381312*t^10 + 70498761427648512*t^9 +
70981172154335232*t^8 + 32880001875443712*t^7 + 11436845074219008*t^6
+ 3740538557693952*t^5 - 89060441849856*t^4 - 29686813949952*t^3 ], [
0, 0, 0, -27*t^22 - 24840*t^20 - 116208*t^18 + 4188672*t^16 -
26445312*t^14 + 105172992*t^12 - 423124992*t^10 + 1072300032*t^8 -
475987968*t^6 - 1627914240*t^4 - 28311552*t^2, 54*t^33 - 112104*t^31 -
8639136*t^29 + 105591168*t^27 - 290345472*t^25 + 3155687424*t^23 -
67851067392*t^21 + 551817805824*t^19 - 2207271223296*t^17 +
4342468313088*t^15 - 3231423922176*t^13 + 4757020213248*t^11 -
27680091144192*t^9 + 36235162681344*t^7 + 7523172089856*t^5 -
57982058496*t^3 ], [ 0, 0, 0, -27*t^30 - 23544*t^28 + 1059696*t^26 -
5315328*t^24 - 279085824*t^22 + 4004149248*t^20 - 24796827648*t^18 +
106709778432*t^16 - 396749242368*t^14 + 1025062207488*t^12 -
1143135535104*t^10 - 348345335808*t^8 + 1111171792896*t^6 -
395002773504*t^4 - 7247757312*t^2, 54*t^45 - 115992*t^43 - 471744*t^41
+ 527641344*t^39 - 21426467328*t^37 + 347735918592*t^35 -
2757111644160*t^33 + 18883543302144*t^31 - 223568674947072*t^29 +
2333052597436416*t^27 - 15123131224031232*t^25 +
60492524896124928*t^23 - 149315366235930624*t^21 +
228934323145801728*t^19 - 309387973462327296*t^17 +
722760274846679040*t^15 - 1458510154294099968*t^13 +
1437905881915195392*t^11 - 566550579124371456*t^9 +
8104500208336896*t^7 + 31883638182248448*t^5 - 237494511599616*t^3 ],
[ 0, 0, 0, -27*t^16 - 25056*t^14 - 316224*t^12 + 2059776*t^10 -
4907520*t^8 + 32956416*t^6 - 80953344*t^4 - 102629376*t^2 - 1769472,
-54*t^24 + 111456*t^22 + 9979200*t^20 + 8805888*t^18 - 75852288*t^16 -
3849928704*t^14 + 25856409600*t^12 - 61598859264*t^10 -
19418185728*t^8 + 36068917248*t^6 + 653996851200*t^4 +
116870086656*t^2 - 905969664 ], [ 0, 0, 0, -27*t^24 - 23760*t^22 +
870048*t^20 + 2025216*t^18 - 276804864*t^16 + 1757306880*t^14 -
6309494784*t^12 + 28116910080*t^10 - 70862045184*t^8 + 8295284736*t^6
+ 57019465728*t^4 - 24914165760*t^2 - 452984832, 54*t^36 - 115344*t^34
- 1858464*t^32 + 510879744*t^30 - 15214086144*t^28 + 140525715456*t^26
- 307830620160*t^24 + 7470640005120*t^22 - 110151479328768*t^20 +
632942965555200*t^18 - 1762423669260288*t^16 + 1912483841310720*t^14 -
1260874220175360*t^12 + 9209493288124416*t^10 - 15953125592530944*t^8
+ 8571139815112704*t^6 - 498877631299584*t^4 - 495398707789824*t^2 +
3710851743744 ], [ 0, 0, 0, -27*t^16 - 25056*t^14 - 316224*t^12 +
2059776*t^10 - 4907520*t^8 + 32956416*t^6 - 80953344*t^4 -
102629376*t^2 - 1769472, 54*t^24 - 111456*t^22 - 9979200*t^20 -
8805888*t^18 + 75852288*t^16 + 3849928704*t^14 - 25856409600*t^12 +
61598859264*t^10 + 19418185728*t^8 - 36068917248*t^6 -
653996851200*t^4 - 116870086656*t^2 + 905969664 ], [ 0, 0, 0,
-108*t^22 - 99360*t^20 - 464832*t^18 + 16754688*t^16 - 105781248*t^14
+ 420691968*t^12 - 1692499968*t^10 + 4289200128*t^8 - 1903951872*t^6 -
6511656960*t^4 - 113246208*t^2, 432*t^33 - 896832*t^31 - 69113088*t^29
+ 844729344*t^27 - 2322763776*t^25 + 25245499392*t^23 -
542808539136*t^21 + 4414542446592*t^19 - 17658169786368*t^17 +
34739746504704*t^15 - 25851391377408*t^13 + 38056161705984*t^11 -
221440729153536*t^9 + 289881301450752*t^7 + 60185376718848*t^5 -
463856467968*t^3 ], [ 0, 0, 0, -108*t^30 - 94176*t^28 + 4238784*t^26 -
21261312*t^24 - 1116343296*t^22 + 16016596992*t^20 - 99187310592*t^18
+ 426839113728*t^16 - 1586996969472*t^14 + 4100248829952*t^12 -
4572542140416*t^10 - 1393381343232*t^8 + 4444687171584*t^6 -
1580011094016*t^4 - 28991029248*t^2, 432*t^45 - 927936*t^43 -
3773952*t^41 + 4221130752*t^39 - 171411738624*t^37 +
2781887348736*t^35 - 22056893153280*t^33 + 151068346417152*t^31 -
1788549399576576*t^29 + 18664420779491328*t^27 -
120985049792249856*t^25 + 483940199168999424*t^23 -
1194522929887444992*t^21 + 1831474585166413824*t^19 -
2475103787698618368*t^17 + 5782082198773432320*t^15 -
11668081234352799744*t^13 + 11503247055321563136*t^11 -
4532404632994971648*t^9 + 64836001666695168*t^7 +
255069105457987584*t^5 - 1899956092796928*t^3 ], [ 0, 0, 0, -108*t^16
- 100224*t^14 - 1264896*t^12 + 8239104*t^10 - 19630080*t^8 +
131825664*t^6 - 323813376*t^4 - 410517504*t^2 - 7077888, -432*t^24 +
891648*t^22 + 79833600*t^20 + 70447104*t^18 - 606818304*t^16 -
30799429632*t^14 + 206851276800*t^12 - 492790874112*t^10 -
155345485824*t^8 + 288551337984*t^6 + 5231974809600*t^4 +
934960693248*t^2 - 7247757312 ], [ 0, 0, 0, -108*t^24 - 95040*t^22 +
3480192*t^20 + 8100864*t^18 - 1107219456*t^16 + 7029227520*t^14 -
25237979136*t^12 + 112467640320*t^10 - 283448180736*t^8 +
33181138944*t^6 + 228077862912*t^4 - 99656663040*t^2 - 1811939328,
432*t^36 - 922752*t^34 - 14867712*t^32 + 4087037952*t^30 -
121712689152*t^28 + 1124205723648*t^26 - 2462644961280*t^24 +
59765120040960*t^22 - 881211834630144*t^20 + 5063543724441600*t^18 -
14099389354082304*t^16 + 15299870730485760*t^14 -
10086993761402880*t^12 + 73675946304995328*t^10 -
127625004740247552*t^8 + 68569118520901632*t^6 - 3991021050396672*t^4
- 3963189662318592*t^2 + 29686813949952 ], [ 0, 0, 0, -108*t^16 -
100224*t^14 - 1264896*t^12 + 8239104*t^10 - 19630080*t^8 +
131825664*t^6 - 323813376*t^4 - 410517504*t^2 - 7077888, 432*t^24 -
891648*t^22 - 79833600*t^20 - 70447104*t^18 + 606818304*t^16 +
30799429632*t^14 - 206851276800*t^12 + 492790874112*t^10 +
155345485824*t^8 - 288551337984*t^6 - 5231974809600*t^4 -
934960693248*t^2 + 7247757312 ], [ 0, 0, 0, -7077888*t^16 +
14155776*t^14 - 5308416*t^12 - 884736*t^10 - 6359040*t^8 - 55296*t^6 -
20736*t^4 + 3456*t^2 - 108, 7247757312*t^24 - 21743271936*t^22 +
19025362944*t^20 - 3170893824*t^18 - 15882780672*t^16 +
14778630144*t^14 + 1833172992*t^12 + 923664384*t^10 - 62042112*t^8 -
774144*t^6 + 290304*t^4 - 20736*t^2 + 432 ], [ 0, 0, 0, -28311552*t^22
+ 70778880*t^20 - 51314688*t^18 + 10616832*t^16 - 24993792*t^14 +
12275712*t^12 - 1562112*t^10 + 41472*t^8 - 12528*t^6 + 1080*t^4 -
27*t^2, 57982058496*t^33 - 217432719360*t^31 + 293534171136*t^29 -
173040205824*t^27 - 76780929024*t^25 + 206391214080*t^23 -
97434206208*t^21 + 20543569920*t^19 - 5135892480*t^17 +
1522409472*t^15 - 201553920*t^13 + 4686336*t^11 + 660096*t^9 -
69984*t^7 + 3240*t^5 - 54*t^3 ], [ 0, 0, 0, -1811939328*t^24 +
9059696640*t^22 - 16533946368*t^20 + 12796821504*t^18 -
4862509056*t^16 + 4600627200*t^14 - 3791978496*t^12 + 287539200*t^10 -
18994176*t^8 + 3124224*t^6 - 252288*t^4 + 8640*t^2 - 108,
29686813949952*t^36 - 222651104624640*t^34 + 684652146720768*t^32 -
1098412116148224*t^30 + 897562265518080*t^28 - 82798379532288*t^26 -
629714146295808*t^24 + 639078248742912*t^22 - 231497898393600*t^20 +
25209058885632*t^18 - 14468618649600*t^16 + 2496399409152*t^14 -
153738805248*t^12 - 1263403008*t^10 + 855982080*t^8 - 65470464*t^6 +
2550528*t^4 - 51840*t^2 + 432 ], [ 0, 0, 0, -7247757312*t^30 +
39862665216*t^28 - 84708163584*t^26 + 86520102912*t^24 -
49177165824*t^22 + 31326732288*t^20 - 25584795648*t^18 +
9884270592*t^16 - 1599049728*t^14 + 122370048*t^12 - 12006144*t^10 +
1320192*t^8 - 80784*t^6 + 2376*t^4 - 27*t^2, 237494511599616*t^45 -
1959329720696832*t^43 + 6857654022438912*t^41 - 13232897318191104*t^39
+ 14825780429193216*t^37 - 7780960321929216*t^35 -
3057277980377088*t^33 + 8654518015229952*t^31 - 6620674101608448*t^29
+ 2627991502848000*t^27 - 694134931193856*t^25 + 173533732798464*t^23
- 41062367232000*t^21 + 6465502052352*t^19 - 528229859328*t^17 +
11662589952*t^15 + 1855125504*t^13 - 220921344*t^11 + 12324096*t^9 -
399168*t^7 + 7128*t^5 - 54*t^3 ], [ 0, 0, 0, -452984832*t^26 +
905969664*t^25 + 452984832*t^24 - 1585446912*t^23 + 396361728*t^22 +
169869312*t^21 - 84934656*t^20 + 382205952*t^19 - 536150016*t^18 +
828112896*t^17 - 449445888*t^16 + 192872448*t^15 - 162349056*t^14 -
48218112*t^13 - 28090368*t^12 - 12939264*t^11 - 2094336*t^10 -
373248*t^9 - 20736*t^8 - 10368*t^7 + 6048*t^6 + 6048*t^5 + 432*t^4 -
216*t^3 - 27*t^2, 3710851743744*t^39 - 11132555231232*t^38 +
26903675142144*t^36 - 18786186952704*t^35 - 11132555231232*t^34 +
12987981103104*t^33 - 8349416423424*t^32 - 521838526464*t^31 +
24526410743808*t^30 - 18090402250752*t^29 - 8479876055040*t^28 +
12295820279808*t^27 - 13089449705472*t^26 + 13241652609024*t^25 -
5120540540928*t^24 + 3302599163904*t^23 - 1566534795264*t^22 -
391633698816*t^20 - 206412447744*t^19 - 80008445952*t^18 -
51725205504*t^17 - 12782665728*t^16 - 3001909248*t^15 - 517570560*t^14
+ 276037632*t^13 + 93560832*t^12 + 497664*t^11 - 1990656*t^10 -
774144*t^9 - 165888*t^8 + 69984*t^7 + 25056*t^6 - 648*t^4 - 54*t^3 ],
[ 0, 0, 0, -452984832*t^26 - 905969664*t^25 + 452984832*t^24 +
1585446912*t^23 + 396361728*t^22 - 169869312*t^21 - 84934656*t^20 -
382205952*t^19 - 536150016*t^18 - 828112896*t^17 - 449445888*t^16 -
192872448*t^15 - 162349056*t^14 + 48218112*t^13 - 28090368*t^12 +
12939264*t^11 - 2094336*t^10 + 373248*t^9 - 20736*t^8 + 10368*t^7 +
6048*t^6 - 6048*t^5 + 432*t^4 + 216*t^3 - 27*t^2, 3710851743744*t^39 +
11132555231232*t^38 - 26903675142144*t^36 - 18786186952704*t^35 +
11132555231232*t^34 + 12987981103104*t^33 + 8349416423424*t^32 -
521838526464*t^31 - 24526410743808*t^30 - 18090402250752*t^29 +
8479876055040*t^28 + 12295820279808*t^27 + 13089449705472*t^26 +
13241652609024*t^25 + 5120540540928*t^24 + 3302599163904*t^23 +
1566534795264*t^22 + 391633698816*t^20 - 206412447744*t^19 +
80008445952*t^18 - 51725205504*t^17 + 12782665728*t^16 -
3001909248*t^15 + 517570560*t^14 + 276037632*t^13 - 93560832*t^12 +
497664*t^11 + 1990656*t^10 - 774144*t^9 + 165888*t^8 + 69984*t^7 -
25056*t^6 + 648*t^4 - 54*t^3 ], [ 0, 0, 0, -452984832*t^24 +
2264924160*t^22 - 4133486592*t^20 + 3199205376*t^18 - 1215627264*t^16
+ 1150156800*t^14 - 947994624*t^12 + 71884800*t^10 - 4748544*t^8 +
781056*t^6 - 63072*t^4 + 2160*t^2 - 27, 3710851743744*t^36 -
27831388078080*t^34 + 85581518340096*t^32 - 137301514518528*t^30 +
112195283189760*t^28 - 10349797441536*t^26 - 78714268286976*t^24 +
79884781092864*t^22 - 28937237299200*t^20 + 3151132360704*t^18 -
1808577331200*t^16 + 312049926144*t^14 - 19217350656*t^12 -
157925376*t^10 + 106997760*t^8 - 8183808*t^6 + 318816*t^4 - 6480*t^2 +
54 ], [ 0, 0, 0, -28991029248*t^30 + 159450660864*t^28 -
338832654336*t^26 + 346080411648*t^24 - 196708663296*t^22 +
125306929152*t^20 - 102339182592*t^18 + 39537082368*t^16 -
6396198912*t^14 + 489480192*t^12 - 48024576*t^10 + 5280768*t^8 -
323136*t^6 + 9504*t^4 - 108*t^2, 1899956092796928*t^45 -
15674637765574656*t^43 + 54861232179511296*t^41 -
105863178545528832*t^39 + 118606243433545728*t^37 -
62247682575433728*t^35 - 24458223843016704*t^33 +
69236144121839616*t^31 - 52965392812867584*t^29 +
21023932022784000*t^27 - 5553079449550848*t^25 + 1388269862387712*t^23
- 328498937856000*t^21 + 51724016418816*t^19 - 4225838874624*t^17 +
93300719616*t^15 + 14841004032*t^13 - 1767370752*t^11 + 98592768*t^9 -
3193344*t^7 + 57024*t^5 - 432*t^3 ], [ 0, 0, 0, -1769472*t^16 +
3538944*t^14 - 1327104*t^12 - 221184*t^10 - 1589760*t^8 - 13824*t^6 -
5184*t^4 + 864*t^2 - 27, -905969664*t^24 + 2717908992*t^22 -
2378170368*t^20 + 396361728*t^18 + 1985347584*t^16 - 1847328768*t^14 -
229146624*t^12 - 115458048*t^10 + 7755264*t^8 + 96768*t^6 - 36288*t^4
+ 2592*t^2 - 54 ], [ 0, 0, 0, -113246208*t^22 + 283115520*t^20 -
205258752*t^18 + 42467328*t^16 - 99975168*t^14 + 49102848*t^12 -
6248448*t^10 + 165888*t^8 - 50112*t^6 + 4320*t^4 - 108*t^2,
463856467968*t^33 - 1739461754880*t^31 + 2348273369088*t^29 -
1384321646592*t^27 - 614247432192*t^25 + 1651129712640*t^23 -
779473649664*t^21 + 164348559360*t^19 - 41087139840*t^17 +
12179275776*t^15 - 1612431360*t^13 + 37490688*t^11 + 5280768*t^9 -
559872*t^7 + 25920*t^5 - 432*t^3 ], [ 0, 0, 0, -1811939328*t^26 +
3623878656*t^25 + 1811939328*t^24 - 6341787648*t^23 + 1585446912*t^22
+ 679477248*t^21 - 339738624*t^20 + 1528823808*t^19 - 2144600064*t^18
+ 3312451584*t^17 - 1797783552*t^16 + 771489792*t^15 - 649396224*t^14
- 192872448*t^13 - 112361472*t^12 - 51757056*t^11 - 8377344*t^10 -
1492992*t^9 - 82944*t^8 - 41472*t^7 + 24192*t^6 + 24192*t^5 + 1728*t^4
- 864*t^3 - 108*t^2, 29686813949952*t^39 - 89060441849856*t^38 +
215229401137152*t^36 - 150289495621632*t^35 - 89060441849856*t^34 +
103903848824832*t^33 - 66795331387392*t^32 - 4174708211712*t^31 +
196211285950464*t^30 - 144723218006016*t^29 - 67839008440320*t^28 +
98366562238464*t^27 - 104715597643776*t^26 + 105933220872192*t^25 -
40964324327424*t^24 + 26420793311232*t^23 - 12532278362112*t^22 -
3133069590528*t^20 - 1651299581952*t^19 - 640067567616*t^18 -
413801644032*t^17 - 102261325824*t^16 - 24015273984*t^15 -
4140564480*t^14 + 2208301056*t^13 + 748486656*t^12 + 3981312*t^11 -
15925248*t^10 - 6193152*t^9 - 1327104*t^8 + 559872*t^7 + 200448*t^6 -
5184*t^4 - 432*t^3 ], [ 0, 0, 0, -1811939328*t^26 - 3623878656*t^25 +
1811939328*t^24 + 6341787648*t^23 + 1585446912*t^22 - 679477248*t^21 -
339738624*t^20 - 1528823808*t^19 - 2144600064*t^18 - 3312451584*t^17 -
1797783552*t^16 - 771489792*t^15 - 649396224*t^14 + 192872448*t^13 -
112361472*t^12 + 51757056*t^11 - 8377344*t^10 + 1492992*t^9 -
82944*t^8 + 41472*t^7 + 24192*t^6 - 24192*t^5 + 1728*t^4 + 864*t^3 -
108*t^2, 29686813949952*t^39 + 89060441849856*t^38 -
215229401137152*t^36 - 150289495621632*t^35 + 89060441849856*t^34 +
103903848824832*t^33 + 66795331387392*t^32 - 4174708211712*t^31 -
196211285950464*t^30 - 144723218006016*t^29 + 67839008440320*t^28 +
98366562238464*t^27 + 104715597643776*t^26 + 105933220872192*t^25 +
40964324327424*t^24 + 26420793311232*t^23 + 12532278362112*t^22 +
3133069590528*t^20 - 1651299581952*t^19 + 640067567616*t^18 -
413801644032*t^17 + 102261325824*t^16 - 24015273984*t^15 +
4140564480*t^14 + 2208301056*t^13 - 748486656*t^12 + 3981312*t^11 +
15925248*t^10 - 6193152*t^9 + 1327104*t^8 + 559872*t^7 - 200448*t^6 +
5184*t^4 - 432*t^3 ], [ 0, 0, 0, -7077888*t^16 + 14155776*t^14 -
5308416*t^12 - 884736*t^10 - 6359040*t^8 - 55296*t^6 - 20736*t^4 +
3456*t^2 - 108, -7247757312*t^24 + 21743271936*t^22 - 19025362944*t^20
+ 3170893824*t^18 + 15882780672*t^16 - 14778630144*t^14 -
1833172992*t^12 - 923664384*t^10 + 62042112*t^8 + 774144*t^6 -
290304*t^4 + 20736*t^2 - 432 ], [ 0, 0, 0, -1769472*t^16 +
3538944*t^14 - 1327104*t^12 - 221184*t^10 - 1589760*t^8 - 13824*t^6 -
5184*t^4 + 864*t^2 - 27, 905969664*t^24 - 2717908992*t^22 +
2378170368*t^20 - 396361728*t^18 - 1985347584*t^16 + 1847328768*t^14 +
229146624*t^12 + 115458048*t^10 - 7755264*t^8 - 96768*t^6 + 36288*t^4
- 2592*t^2 + 54 ], [ 0, 0, 0, -7247757312*t^30 - 39862665216*t^28 -
84708163584*t^26 - 86520102912*t^24 - 49177165824*t^22 -
31326732288*t^20 - 25584795648*t^18 - 9884270592*t^16 -
1599049728*t^14 - 122370048*t^12 - 12006144*t^10 - 1320192*t^8 -
80784*t^6 - 2376*t^4 - 27*t^2, 237494511599616*t^45 +
1959329720696832*t^43 + 6857654022438912*t^41 + 13232897318191104*t^39
+ 14825780429193216*t^37 + 7780960321929216*t^35 -
3057277980377088*t^33 - 8654518015229952*t^31 - 6620674101608448*t^29
- 2627991502848000*t^27 - 694134931193856*t^25 - 173533732798464*t^23
- 41062367232000*t^21 - 6465502052352*t^19 - 528229859328*t^17 -
11662589952*t^15 + 1855125504*t^13 + 220921344*t^11 + 12324096*t^9 +
399168*t^7 + 7128*t^5 + 54*t^3 ], [ 0, 0, 0, -113246208*t^22 -
283115520*t^20 - 205258752*t^18 - 42467328*t^16 - 99975168*t^14 -
49102848*t^12 - 6248448*t^10 - 165888*t^8 - 50112*t^6 - 4320*t^4 -
108*t^2, 463856467968*t^33 + 1739461754880*t^31 + 2348273369088*t^29 +
1384321646592*t^27 - 614247432192*t^25 - 1651129712640*t^23 -
779473649664*t^21 - 164348559360*t^19 - 41087139840*t^17 -
12179275776*t^15 - 1612431360*t^13 - 37490688*t^11 + 5280768*t^9 +
559872*t^7 + 25920*t^5 + 432*t^3 ], [ 0, 0, 0, -28311552*t^22 -
70778880*t^20 - 51314688*t^18 - 10616832*t^16 - 24993792*t^14 -
12275712*t^12 - 1562112*t^10 - 41472*t^8 - 12528*t^6 - 1080*t^4 -
27*t^2, 57982058496*t^33 + 217432719360*t^31 + 293534171136*t^29 +
173040205824*t^27 - 76780929024*t^25 - 206391214080*t^23 -
97434206208*t^21 - 20543569920*t^19 - 5135892480*t^17 -
1522409472*t^15 - 201553920*t^13 - 4686336*t^11 + 660096*t^9 +
69984*t^7 + 3240*t^5 + 54*t^3 ], [ 0, 0, 0, -28991029248*t^30 -
159450660864*t^28 - 338832654336*t^26 - 346080411648*t^24 -
196708663296*t^22 - 125306929152*t^20 - 102339182592*t^18 -
39537082368*t^16 - 6396198912*t^14 - 489480192*t^12 - 48024576*t^10 -
5280768*t^8 - 323136*t^6 - 9504*t^4 - 108*t^2, 1899956092796928*t^45 +
15674637765574656*t^43 + 54861232179511296*t^41 +
105863178545528832*t^39 + 118606243433545728*t^37 +
62247682575433728*t^35 - 24458223843016704*t^33 -
69236144121839616*t^31 - 52965392812867584*t^29 -
21023932022784000*t^27 - 5553079449550848*t^25 - 1388269862387712*t^23
- 328498937856000*t^21 - 51724016418816*t^19 - 4225838874624*t^17 -
93300719616*t^15 + 14841004032*t^13 + 1767370752*t^11 + 98592768*t^9 +
3193344*t^7 + 57024*t^5 + 432*t^3 ], [ 0, 0, 0, -7077888*t^16 -
14155776*t^14 - 5308416*t^12 + 884736*t^10 - 6359040*t^8 + 55296*t^6 -
20736*t^4 - 3456*t^2 - 108, 7247757312*t^24 + 21743271936*t^22 +
19025362944*t^20 + 3170893824*t^18 - 15882780672*t^16 -
14778630144*t^14 + 1833172992*t^12 - 923664384*t^10 - 62042112*t^8 +
774144*t^6 + 290304*t^4 + 20736*t^2 + 432 ], [ 0, 0, 0, -7077888*t^16
- 14155776*t^14 - 5308416*t^12 + 884736*t^10 - 6359040*t^8 + 55296*t^6
- 20736*t^4 - 3456*t^2 - 108, -7247757312*t^24 - 21743271936*t^22 -
19025362944*t^20 - 3170893824*t^18 + 15882780672*t^16 +
14778630144*t^14 - 1833172992*t^12 + 923664384*t^10 + 62042112*t^8 -
774144*t^6 - 290304*t^4 - 20736*t^2 - 432 ], [ 0, 0, 0,
-452984832*t^24 - 2264924160*t^22 - 4133486592*t^20 - 3199205376*t^18
- 1215627264*t^16 - 1150156800*t^14 - 947994624*t^12 - 71884800*t^10 -
4748544*t^8 - 781056*t^6 - 63072*t^4 - 2160*t^2 - 27,
3710851743744*t^36 + 27831388078080*t^34 + 85581518340096*t^32 +
137301514518528*t^30 + 112195283189760*t^28 + 10349797441536*t^26 -
78714268286976*t^24 - 79884781092864*t^22 - 28937237299200*t^20 -
3151132360704*t^18 - 1808577331200*t^16 - 312049926144*t^14 -
19217350656*t^12 + 157925376*t^10 + 106997760*t^8 + 8183808*t^6 +
318816*t^4 + 6480*t^2 + 54 ], [ 0, 0, 0, -452984832*t^24 -
2264924160*t^22 - 4133486592*t^20 - 3199205376*t^18 - 1215627264*t^16
- 1150156800*t^14 - 947994624*t^12 - 71884800*t^10 - 4748544*t^8 -
781056*t^6 - 63072*t^4 - 2160*t^2 - 27, -3710851743744*t^36 -
27831388078080*t^34 - 85581518340096*t^32 - 137301514518528*t^30 -
112195283189760*t^28 - 10349797441536*t^26 + 78714268286976*t^24 +
79884781092864*t^22 + 28937237299200*t^20 + 3151132360704*t^18 +
1808577331200*t^16 + 312049926144*t^14 + 19217350656*t^12 -
157925376*t^10 - 106997760*t^8 - 8183808*t^6 - 318816*t^4 - 6480*t^2 -
54 ], [ 0, 0, 0, -1811939328*t^24 - 9059696640*t^22 - 16533946368*t^20
- 12796821504*t^18 - 4862509056*t^16 - 4600627200*t^14 -
3791978496*t^12 - 287539200*t^10 - 18994176*t^8 - 3124224*t^6 -
252288*t^4 - 8640*t^2 - 108, 29686813949952*t^36 +
222651104624640*t^34 + 684652146720768*t^32 + 1098412116148224*t^30 +
897562265518080*t^28 + 82798379532288*t^26 - 629714146295808*t^24 -
639078248742912*t^22 - 231497898393600*t^20 - 25209058885632*t^18 -
14468618649600*t^16 - 2496399409152*t^14 - 153738805248*t^12 +
1263403008*t^10 + 855982080*t^8 + 65470464*t^6 + 2550528*t^4 +
51840*t^2 + 432 ], [ 0, 0, 0, -1811939328*t^24 - 9059696640*t^22 -
16533946368*t^20 - 12796821504*t^18 - 4862509056*t^16 -
4600627200*t^14 - 3791978496*t^12 - 287539200*t^10 - 18994176*t^8 -
3124224*t^6 - 252288*t^4 - 8640*t^2 - 108, -29686813949952*t^36 -
222651104624640*t^34 - 684652146720768*t^32 - 1098412116148224*t^30 -
897562265518080*t^28 - 82798379532288*t^26 + 629714146295808*t^24 +
639078248742912*t^22 + 231497898393600*t^20 + 25209058885632*t^18 +
14468618649600*t^16 + 2496399409152*t^14 + 153738805248*t^12 -
1263403008*t^10 - 855982080*t^8 - 65470464*t^6 - 2550528*t^4 -
51840*t^2 - 432 ], [ 0, 0, 0, -1769472*t^16 - 3538944*t^14 -
1327104*t^12 + 221184*t^10 - 1589760*t^8 + 13824*t^6 - 5184*t^4 -
864*t^2 - 27, -905969664*t^24 - 2717908992*t^22 - 2378170368*t^20 -
396361728*t^18 + 1985347584*t^16 + 1847328768*t^14 - 229146624*t^12 +
115458048*t^10 + 7755264*t^8 - 96768*t^6 - 36288*t^4 - 2592*t^2 - 54
], [ 0, 0, 0, -1769472*t^16 - 3538944*t^14 - 1327104*t^12 +
221184*t^10 - 1589760*t^8 + 13824*t^6 - 5184*t^4 - 864*t^2 - 27,
905969664*t^24 + 2717908992*t^22 + 2378170368*t^20 + 396361728*t^18 -
1985347584*t^16 - 1847328768*t^14 + 229146624*t^12 - 115458048*t^10 -
7755264*t^8 + 96768*t^6 + 36288*t^4 + 2592*t^2 + 54 ], [ 0, 0, 0,
-442368*t^24 + 6414336*t^20 - 414720*t^16 - 1022976*t^12 - 25920*t^8 +
25056*t^4 - 108, 113246208*t^36 + 3652190208*t^32 - 4671406080*t^28 -
1734082560*t^24 + 554729472*t^20 + 138682368*t^16 - 27095040*t^12 -
4561920*t^8 + 222912*t^4 + 432 ], [ 0, 0, 0, -110592*t^24 +
1603584*t^20 - 103680*t^16 - 255744*t^12 - 6480*t^8 + 6264*t^4 - 27,
-14155776*t^36 - 456523776*t^32 + 583925760*t^28 + 216760320*t^24 -
69341184*t^20 - 17335296*t^16 + 3386880*t^12 + 570240*t^8 - 27864*t^4
- 54 ], [ 0, 0, 0, -1769472*t^32 + 26542080*t^28 - 14598144*t^24 -
1658880*t^20 + 1838592*t^16 - 103680*t^12 - 57024*t^8 + 6480*t^4 - 27,
905969664*t^48 + 28538044416*t^44 - 59114520576*t^40 +
19619905536*t^36 + 7378698240*t^32 - 4236115968*t^28 + 264757248*t^20
- 28823040*t^16 - 4790016*t^12 + 902016*t^8 - 27216*t^4 - 54 ], [ 0,
0, 0, -7077888*t^32 + 106168320*t^28 - 58392576*t^24 - 6635520*t^20 +
7354368*t^16 - 414720*t^12 - 228096*t^8 + 25920*t^4 - 108,
7247757312*t^48 + 228304355328*t^44 - 472916164608*t^40 +
156959244288*t^36 + 59029585920*t^32 - 33888927744*t^28 +
2118057984*t^20 - 230584320*t^16 - 38320128*t^12 + 7216128*t^8 -
217728*t^4 - 432 ], [ 0, 0, 0, -442368*t^24 + 6414336*t^20 -
414720*t^16 - 1022976*t^12 - 25920*t^8 + 25056*t^4 - 108,
-113246208*t^36 - 3652190208*t^32 + 4671406080*t^28 + 1734082560*t^24
- 554729472*t^20 - 138682368*t^16 + 27095040*t^12 + 4561920*t^8 -
222912*t^4 - 432 ], [ 0, 0, 0, -110592*t^24 + 1603584*t^20 -
103680*t^16 - 255744*t^12 - 6480*t^8 + 6264*t^4 - 27, 14155776*t^36 +
456523776*t^32 - 583925760*t^28 - 216760320*t^24 + 69341184*t^20 +
17335296*t^16 - 3386880*t^12 - 570240*t^8 + 27864*t^4 + 54 ], [ 0, 0,
0, -27648*t^16 + 414720*t^12 - 231552*t^8 + 25920*t^4 - 108,
1769472*t^24 + 55738368*t^20 - 115126272*t^16 + 48771072*t^12 -
7195392*t^8 + 217728*t^4 + 432 ], [ 0, 0, 0, -6912*t^16 + 103680*t^12
- 57888*t^8 + 6480*t^4 - 27, -221184*t^24 - 6967296*t^20 +
14390784*t^16 - 6096384*t^12 + 899424*t^8 - 27216*t^4 - 54 ], [ 0, 0,
0, -110592*t^24 + 1714176*t^20 - 1762560*t^16 + 670464*t^12 -
110160*t^8 + 6696*t^4 - 27, 14155776*t^36 + 435290112*t^32 -
1252786176*t^28 + 1164312576*t^24 - 529846272*t^20 + 132461568*t^16 -
18192384*t^12 + 1223424*t^8 - 26568*t^4 - 54 ], [ 0, 0, 0,
-442368*t^24 + 6856704*t^20 - 7050240*t^16 + 2681856*t^12 - 440640*t^8
+ 26784*t^4 - 108, 113246208*t^36 + 3482320896*t^32 - 10022289408*t^28
+ 9314500608*t^24 - 4238770176*t^20 + 1059692544*t^16 - 145539072*t^12
+ 9787392*t^8 - 212544*t^4 - 432 ], [ 0, 0, 0, -27648*t^16 +
414720*t^12 - 231552*t^8 + 25920*t^4 - 108, -1769472*t^24 -
55738368*t^20 + 115126272*t^16 - 48771072*t^12 + 7195392*t^8 -
217728*t^4 - 432 ], [ 0, 0, 0, -6912*t^16 + 103680*t^12 - 57888*t^8 +
6480*t^4 - 27, 221184*t^24 + 6967296*t^20 - 14390784*t^16 +
6096384*t^12 - 899424*t^8 + 27216*t^4 + 54 ], [ 0, 0, 0, 45684*t^24 -
1591488*t^23 - 18591552*t^22 - 72997632*t^21 - 227187072*t^20 -
690398208*t^19 - 3708232704*t^18 - 12310548480*t^17 - 18470384640*t^16
- 22398861312*t^15 + 104746549248*t^14 + 80245555200*t^13 +
1148205072384*t^12 - 320982220800*t^11 + 1675944787968*t^10 +
1433527123968*t^9 - 4728418467840*t^8 + 12606001643520*t^7 -
15188921155584*t^6 + 11311484239872*t^5 - 14888931950592*t^4 +
19135891243008*t^3 - 19494655229952*t^2 + 6675184484352*t +
766450335744, 27515376*t^36 + 437913216*t^35 + 1898494848*t^34 +
12171216384*t^33 + 154282952448*t^32 + 1181314990080*t^31 +
5686781755392*t^30 + 16898445213696*t^29 + 34579768983552*t^28 +
53143377149952*t^27 + 50195712835584*t^26 + 200541976657920*t^25 +
751461386158080*t^24 + 7040564880998400*t^23 + 20068358820986880*t^22
+ 59571855421341696*t^21 + 96668090797916160*t^20 +
133704509187686400*t^19 + 230698477556858880*t^18 -
534818036750745600*t^17 + 1546689452766658560*t^16 -
3812598746965868544*t^15 + 5137499858172641280*t^14 -
7209538438142361600*t^13 + 3077985837703495680*t^12 -
3285679745563361280*t^11 + 3289626236392833024*t^10 -
13931217459597017088*t^9 + 36259515841697021952*t^8 -
70877216353585987584*t^7 + 95408365855070748672*t^6 -
79276707010440069120*t^5 + 41415014693405196288*t^4 -
13068744080454844416*t^3 + 8153973283784491008*t^2 -
7523291764824735744*t + 1890842240914292736 ], [ 0, 0, 0, 45684*t^24 -
1591488*t^23 - 18591552*t^22 - 72997632*t^21 - 227187072*t^20 -
690398208*t^19 - 3708232704*t^18 - 12310548480*t^17 - 18470384640*t^16
- 22398861312*t^15 + 104746549248*t^14 + 80245555200*t^13 +
1148205072384*t^12 - 320982220800*t^11 + 1675944787968*t^10 +
1433527123968*t^9 - 4728418467840*t^8 + 12606001643520*t^7 -
15188921155584*t^6 + 11311484239872*t^5 - 14888931950592*t^4 +
19135891243008*t^3 - 19494655229952*t^2 + 6675184484352*t +
766450335744, -27515376*t^36 - 437913216*t^35 - 1898494848*t^34 -
12171216384*t^33 - 154282952448*t^32 - 1181314990080*t^31 -
5686781755392*t^30 - 16898445213696*t^29 - 34579768983552*t^28 -
53143377149952*t^27 - 50195712835584*t^26 - 200541976657920*t^25 -
751461386158080*t^24 - 7040564880998400*t^23 - 20068358820986880*t^22
- 59571855421341696*t^21 - 96668090797916160*t^20 -
133704509187686400*t^19 - 230698477556858880*t^18 +
534818036750745600*t^17 - 1546689452766658560*t^16 +
3812598746965868544*t^15 - 5137499858172641280*t^14 +
7209538438142361600*t^13 - 3077985837703495680*t^12 +
3285679745563361280*t^11 - 3289626236392833024*t^10 +
13931217459597017088*t^9 - 36259515841697021952*t^8 +
70877216353585987584*t^7 - 95408365855070748672*t^6 +
79276707010440069120*t^5 - 41415014693405196288*t^4 +
13068744080454844416*t^3 - 8153973283784491008*t^2 +
7523291764824735744*t - 1890842240914292736 ], [ 0, 0, 0, 11421*t^24 -
397872*t^23 - 4647888*t^22 - 18249408*t^21 - 56796768*t^20 -
172599552*t^19 - 927058176*t^18 - 3077637120*t^17 - 4617596160*t^16 -
5599715328*t^15 + 26186637312*t^14 + 20061388800*t^13 +
287051268096*t^12 - 80245555200*t^11 + 418986196992*t^10 +
358381780992*t^9 - 1182104616960*t^8 + 3151500410880*t^7 -
3797230288896*t^6 + 2827871059968*t^5 - 3722232987648*t^4 +
4783972810752*t^3 - 4873663807488*t^2 + 1668796121088*t +
191612583936, -3439422*t^36 - 54739152*t^35 - 237311856*t^34 -
1521402048*t^33 - 19285369056*t^32 - 147664373760*t^31 -
710847719424*t^30 - 2112305651712*t^29 - 4322471122944*t^28 -
6642922143744*t^27 - 6274464104448*t^26 - 25067747082240*t^25 -
93932673269760*t^24 - 880070610124800*t^23 - 2508544852623360*t^22 -
7446481927667712*t^21 - 12083511349739520*t^20 -
16713063648460800*t^19 - 28837309694607360*t^18 +
66852254593843200*t^17 - 193336181595832320*t^16 +
476574843370733568*t^15 - 642187482271580160*t^14 +
901192304767795200*t^13 - 384748229712936960*t^12 +
410709968195420160*t^11 - 411203279549104128*t^10 +
1741402182449627136*t^9 - 4532439480212127744*t^8 +
8859652044198248448*t^7 - 11926045731883843584*t^6 +
9909588376305008640*t^5 - 5176876836675649536*t^4 +
1633593010056855552*t^3 - 1019246660473061376*t^2 +
940411470603091968*t - 236355280114286592 ], [ 0, 0, 0, 11421*t^24 -
397872*t^23 - 4647888*t^22 - 18249408*t^21 - 56796768*t^20 -
172599552*t^19 - 927058176*t^18 - 3077637120*t^17 - 4617596160*t^16 -
5599715328*t^15 + 26186637312*t^14 + 20061388800*t^13 +
287051268096*t^12 - 80245555200*t^11 + 418986196992*t^10 +
358381780992*t^9 - 1182104616960*t^8 + 3151500410880*t^7 -
3797230288896*t^6 + 2827871059968*t^5 - 3722232987648*t^4 +
4783972810752*t^3 - 4873663807488*t^2 + 1668796121088*t +
191612583936, 3439422*t^36 + 54739152*t^35 + 237311856*t^34 +
1521402048*t^33 + 19285369056*t^32 + 147664373760*t^31 +
710847719424*t^30 + 2112305651712*t^29 + 4322471122944*t^28 +
6642922143744*t^27 + 6274464104448*t^26 + 25067747082240*t^25 +
93932673269760*t^24 + 880070610124800*t^23 + 2508544852623360*t^22 +
7446481927667712*t^21 + 12083511349739520*t^20 +
16713063648460800*t^19 + 28837309694607360*t^18 -
66852254593843200*t^17 + 193336181595832320*t^16 -
476574843370733568*t^15 + 642187482271580160*t^14 -
901192304767795200*t^13 + 384748229712936960*t^12 -
410709968195420160*t^11 + 411203279549104128*t^10 -
1741402182449627136*t^9 + 4532439480212127744*t^8 -
8859652044198248448*t^7 + 11926045731883843584*t^6 -
9909588376305008640*t^5 + 5176876836675649536*t^4 -
1633593010056855552*t^3 + 1019246660473061376*t^2 -
940411470603091968*t + 236355280114286592 ], [ 0, 0, 0, 5076*t^16 -
203904*t^15 - 1095552*t^14 + 2336256*t^13 - 8232192*t^12 -
55019520*t^11 + 45398016*t^10 - 270508032*t^9 + 370538496*t^8 +
1082032128*t^7 + 726368256*t^6 + 3521249280*t^5 - 2107441152*t^4 -
2392326144*t^3 - 4487380992*t^2 + 3340763136*t + 332660736,
1019088*t^24 + 8066304*t^23 - 40414464*t^22 + 291271680*t^21 +
4044225024*t^20 + 875556864*t^19 - 4345270272*t^18 + 82544541696*t^17
- 241048866816*t^16 + 126959616000*t^15 + 1119470616576*t^14 -
823144218624*t^13 + 14415230140416*t^12 + 3292576874496*t^11 +
17911529865216*t^10 - 8125415424000*t^9 - 61708509904896*t^8 -
84525610696704*t^7 - 17798227034112*t^6 - 14345123659776*t^5 +
265042331172864*t^4 - 76355123281920*t^3 - 42377637003264*t^2 -
33832531132416*t + 17097459499008 ], [ 0, 0, 0, 5076*t^16 -
203904*t^15 - 1095552*t^14 + 2336256*t^13 - 8232192*t^12 -
55019520*t^11 + 45398016*t^10 - 270508032*t^9 + 370538496*t^8 +
1082032128*t^7 + 726368256*t^6 + 3521249280*t^5 - 2107441152*t^4 -
2392326144*t^3 - 4487380992*t^2 + 3340763136*t + 332660736,
-1019088*t^24 - 8066304*t^23 + 40414464*t^22 - 291271680*t^21 -
4044225024*t^20 - 875556864*t^19 + 4345270272*t^18 - 82544541696*t^17
+ 241048866816*t^16 - 126959616000*t^15 - 1119470616576*t^14 +
823144218624*t^13 - 14415230140416*t^12 - 3292576874496*t^11 -
17911529865216*t^10 + 8125415424000*t^9 + 61708509904896*t^8 +
84525610696704*t^7 + 17798227034112*t^6 + 14345123659776*t^5 -
265042331172864*t^4 + 76355123281920*t^3 + 42377637003264*t^2 +
33832531132416*t - 17097459499008 ], [ 0, 0, 0, 1269*t^16 - 50976*t^15
- 273888*t^14 + 584064*t^13 - 2058048*t^12 - 13754880*t^11 +
11349504*t^10 - 67627008*t^9 + 92634624*t^8 + 270508032*t^7 +
181592064*t^6 + 880312320*t^5 - 526860288*t^4 - 598081536*t^3 -
1121845248*t^2 + 835190784*t + 83165184, -127386*t^24 - 1008288*t^23 +
5051808*t^22 - 36408960*t^21 - 505528128*t^20 - 109444608*t^19 +
543158784*t^18 - 10318067712*t^17 + 30131108352*t^16 -
15869952000*t^15 - 139933827072*t^14 + 102893027328*t^13 -
1801903767552*t^12 - 411572109312*t^11 - 2238941233152*t^10 +
1015676928000*t^9 + 7713563738112*t^8 + 10565701337088*t^7 +
2224778379264*t^6 + 1793140457472*t^5 - 33130291396608*t^4 +
9544390410240*t^3 + 5297204625408*t^2 + 4229066391552*t -
2137182437376 ], [ 0, 0, 0, 1269*t^16 - 50976*t^15 - 273888*t^14 +
584064*t^13 - 2058048*t^12 - 13754880*t^11 + 11349504*t^10 -
67627008*t^9 + 92634624*t^8 + 270508032*t^7 + 181592064*t^6 +
880312320*t^5 - 526860288*t^4 - 598081536*t^3 - 1121845248*t^2 +
835190784*t + 83165184, 127386*t^24 + 1008288*t^23 - 5051808*t^22 +
36408960*t^21 + 505528128*t^20 + 109444608*t^19 - 543158784*t^18 +
10318067712*t^17 - 30131108352*t^16 + 15869952000*t^15 +
139933827072*t^14 - 102893027328*t^13 + 1801903767552*t^12 +
411572109312*t^11 + 2238941233152*t^10 - 1015676928000*t^9 -
7713563738112*t^8 - 10565701337088*t^7 - 2224778379264*t^6 -
1793140457472*t^5 + 33130291396608*t^4 - 9544390410240*t^3 -
5297204625408*t^2 - 4229066391552*t + 2137182437376 ], [ 0, 0, 0,
-27*t^24 + 11880*t^22 + 217512*t^20 - 253152*t^18 - 17300304*t^16 -
54915840*t^14 - 98585856*t^12 - 219663360*t^10 - 276804864*t^8 -
16201728*t^6 + 55683072*t^4 + 12165120*t^2 - 110592, -54*t^36 -
57672*t^34 + 464616*t^32 + 63859968*t^30 + 950880384*t^28 +
4391428608*t^26 + 4809853440*t^24 + 58364375040*t^22 +
430279216128*t^20 + 1236216729600*t^18 + 1721116864512*t^16 +
933830000640*t^14 + 307830620160*t^12 + 1124205723648*t^10 +
973701513216*t^8 + 261570428928*t^6 + 7612268544*t^4 - 3779592192*t^2
- 14155776 ], [ 0, 0, 0, -27*t^24 + 11880*t^22 + 217512*t^20 -
253152*t^18 - 17300304*t^16 - 54915840*t^14 - 98585856*t^12 -
219663360*t^10 - 276804864*t^8 - 16201728*t^6 + 55683072*t^4 +
12165120*t^2 - 110592, 54*t^36 + 57672*t^34 - 464616*t^32 -
63859968*t^30 - 950880384*t^28 - 4391428608*t^26 - 4809853440*t^24 -
58364375040*t^22 - 430279216128*t^20 - 1236216729600*t^18 -
1721116864512*t^16 - 933830000640*t^14 - 307830620160*t^12 -
1124205723648*t^10 - 973701513216*t^8 - 261570428928*t^6 -
7612268544*t^4 + 3779592192*t^2 + 14155776 ], [ 0, 0, 0, -108*t^24 +
47520*t^22 + 870048*t^20 - 1012608*t^18 - 69201216*t^16 -
219663360*t^14 - 394343424*t^12 - 878653440*t^10 - 1107219456*t^8 -
64806912*t^6 + 222732288*t^4 + 48660480*t^2 - 442368, 432*t^36 +
461376*t^34 - 3716928*t^32 - 510879744*t^30 - 7607043072*t^28 -
35131428864*t^26 - 38478827520*t^24 - 466915000320*t^22 -
3442233729024*t^20 - 9889733836800*t^18 - 13768934916096*t^16 -
7470640005120*t^14 - 2462644961280*t^12 - 8993645789184*t^10 -
7789612105728*t^8 - 2092563431424*t^6 - 60898148352*t^4 +
30236737536*t^2 + 113246208 ], [ 0, 0, 0, -108*t^24 + 47520*t^22 +
870048*t^20 - 1012608*t^18 - 69201216*t^16 - 219663360*t^14 -
394343424*t^12 - 878653440*t^10 - 1107219456*t^8 - 64806912*t^6 +
222732288*t^4 + 48660480*t^2 - 442368, -432*t^36 - 461376*t^34 +
3716928*t^32 + 510879744*t^30 + 7607043072*t^28 + 35131428864*t^26 +
38478827520*t^24 + 466915000320*t^22 + 3442233729024*t^20 +
9889733836800*t^18 + 13768934916096*t^16 + 7470640005120*t^14 +
2462644961280*t^12 + 8993645789184*t^10 + 7789612105728*t^8 +
2092563431424*t^6 + 60898148352*t^4 - 30236737536*t^2 - 113246208 ], [
0, 0, 0, -27*t^16 + 12528*t^14 - 79056*t^12 - 257472*t^10 - 306720*t^8
- 1029888*t^6 - 1264896*t^4 + 801792*t^2 - 6912, -54*t^24 - 55728*t^22
+ 2494800*t^20 - 1100736*t^18 - 4740768*t^16 + 120310272*t^14 +
404006400*t^12 + 481241088*t^10 - 75852288*t^8 - 70447104*t^6 +
638668800*t^4 - 57065472*t^2 - 221184 ], [ 0, 0, 0, -27*t^16 +
12528*t^14 - 79056*t^12 - 257472*t^10 - 306720*t^8 - 1029888*t^6 -
1264896*t^4 + 801792*t^2 - 6912, 54*t^24 + 55728*t^22 - 2494800*t^20 +
1100736*t^18 + 4740768*t^16 - 120310272*t^14 - 404006400*t^12 -
481241088*t^10 + 75852288*t^8 + 70447104*t^6 - 638668800*t^4 +
57065472*t^2 + 221184 ], [ 0, 0, 0, -108*t^16 + 50112*t^14 -
316224*t^12 - 1029888*t^10 - 1226880*t^8 - 4119552*t^6 - 5059584*t^4 +
3207168*t^2 - 27648, -432*t^24 - 445824*t^22 + 19958400*t^20 -
8805888*t^18 - 37926144*t^16 + 962482176*t^14 + 3232051200*t^12 +
3849928704*t^10 - 606818304*t^8 - 563576832*t^6 + 5109350400*t^4 -
456523776*t^2 - 1769472 ], [ 0, 0, 0, -108*t^16 + 50112*t^14 -
316224*t^12 - 1029888*t^10 - 1226880*t^8 - 4119552*t^6 - 5059584*t^4 +
3207168*t^2 - 27648, 432*t^24 + 445824*t^22 - 19958400*t^20 +
8805888*t^18 + 37926144*t^16 - 962482176*t^14 - 3232051200*t^12 -
3849928704*t^10 + 606818304*t^8 + 563576832*t^6 - 5109350400*t^4 +
456523776*t^2 + 1769472 ], [ 0, 0, 0, -6912*t^16 + 27648*t^14 -
435456*t^12 + 822528*t^10 - 721440*t^8 + 205632*t^6 - 27216*t^4 +
432*t^2 - 27, -221184*t^24 + 1327104*t^22 + 25546752*t^20 -
110702592*t^18 + 314316288*t^16 - 432470016*t^14 + 285562368*t^12 -
108117504*t^10 + 19644768*t^8 - 1729728*t^6 + 99792*t^4 + 1296*t^2 -
54 ], [ 0, 0, 0, -6912*t^16 + 27648*t^14 - 435456*t^12 + 822528*t^10 -
721440*t^8 + 205632*t^6 - 27216*t^4 + 432*t^2 - 27, 221184*t^24 -
1327104*t^22 - 25546752*t^20 + 110702592*t^18 - 314316288*t^16 +
432470016*t^14 - 285562368*t^12 + 108117504*t^10 - 19644768*t^8 +
1729728*t^6 - 99792*t^4 - 1296*t^2 + 54 ], [ 0, 0, 0, -27648*t^16 +
110592*t^14 - 1741824*t^12 + 3290112*t^10 - 2885760*t^8 + 822528*t^6 -
108864*t^4 + 1728*t^2 - 108, -1769472*t^24 + 10616832*t^22 +
204374016*t^20 - 885620736*t^18 + 2514530304*t^16 - 3459760128*t^14 +
2284498944*t^12 - 864940032*t^10 + 157158144*t^8 - 13837824*t^6 +
798336*t^4 + 10368*t^2 - 432 ], [ 0, 0, 0, -27648*t^16 + 110592*t^14 -
1741824*t^12 + 3290112*t^10 - 2885760*t^8 + 822528*t^6 - 108864*t^4 +
1728*t^2 - 108, 1769472*t^24 - 10616832*t^22 - 204374016*t^20 +
885620736*t^18 - 2514530304*t^16 + 3459760128*t^14 - 2284498944*t^12 +
864940032*t^10 - 157158144*t^8 + 13837824*t^6 - 798336*t^4 - 10368*t^2
+ 432 ], [ 0, 0, 0, -110592*t^24 + 1105920*t^22 - 10672128*t^20 +
59332608*t^18 - 157365504*t^16 + 208051200*t^14 - 150011136*t^12 +
52012800*t^10 - 9835344*t^8 + 927072*t^6 - 41688*t^4 + 1080*t^2 - 27,
14155776*t^36 - 212336640*t^34 - 477757440*t^32 + 18997051392*t^30 -
126478319616*t^28 + 456237121536*t^26 - 1060075634688*t^24 +
1623202136064*t^22 - 1586118537216*t^20 + 982579286016*t^18 -
396529634304*t^16 + 101450133504*t^14 - 16563681792*t^12 +
1782176256*t^10 - 123513984*t^8 + 4637952*t^6 - 29160*t^4 - 3240*t^2 +
54 ], [ 0, 0, 0, -110592*t^24 + 1105920*t^22 - 10672128*t^20 +
59332608*t^18 - 157365504*t^16 + 208051200*t^14 - 150011136*t^12 +
52012800*t^10 - 9835344*t^8 + 927072*t^6 - 41688*t^4 + 1080*t^2 - 27,
-14155776*t^36 + 212336640*t^34 + 477757440*t^32 - 18997051392*t^30 +
126478319616*t^28 - 456237121536*t^26 + 1060075634688*t^24 -
1623202136064*t^22 + 1586118537216*t^20 - 982579286016*t^18 +
396529634304*t^16 - 101450133504*t^14 + 16563681792*t^12 -
1782176256*t^10 + 123513984*t^8 - 4637952*t^6 + 29160*t^4 + 3240*t^2 -
54 ], [ 0, 0, 0, -442368*t^24 + 4423680*t^22 - 42688512*t^20 +
237330432*t^18 - 629462016*t^16 + 832204800*t^14 - 600044544*t^12 +
208051200*t^10 - 39341376*t^8 + 3708288*t^6 - 166752*t^4 + 4320*t^2 -
108, -113246208*t^36 + 1698693120*t^34 + 3822059520*t^32 -
151976411136*t^30 + 1011826556928*t^28 - 3649896972288*t^26 +
8480605077504*t^24 - 12985617088512*t^22 + 12688948297728*t^20 -
7860634288128*t^18 + 3172237074432*t^16 - 811601068032*t^14 +
132509454336*t^12 - 14257410048*t^10 + 988111872*t^8 - 37103616*t^6 +
233280*t^4 + 25920*t^2 - 432 ], [ 0, 0, 0, -442368*t^24 + 4423680*t^22
- 42688512*t^20 + 237330432*t^18 - 629462016*t^16 + 832204800*t^14 -
600044544*t^12 + 208051200*t^10 - 39341376*t^8 + 3708288*t^6 -
166752*t^4 + 4320*t^2 - 108, 113246208*t^36 - 1698693120*t^34 -
3822059520*t^32 + 151976411136*t^30 - 1011826556928*t^28 +
3649896972288*t^26 - 8480605077504*t^24 + 12985617088512*t^22 -
12688948297728*t^20 + 7860634288128*t^18 - 3172237074432*t^16 +
811601068032*t^14 - 132509454336*t^12 + 14257410048*t^10 -
988111872*t^8 + 37103616*t^6 - 233280*t^4 - 25920*t^2 + 432 ], [ 0, 0,
0, -442368*t^24 - 48660480*t^22 + 222732288*t^20 + 64806912*t^18 -
1107219456*t^16 + 878653440*t^14 - 394343424*t^12 + 219663360*t^10 -
69201216*t^8 + 1012608*t^6 + 870048*t^4 - 47520*t^2 - 108,
113246208*t^36 - 30236737536*t^34 - 60898148352*t^32 +
2092563431424*t^30 - 7789612105728*t^28 + 8993645789184*t^26 -
2462644961280*t^24 + 7470640005120*t^22 - 13768934916096*t^20 +
9889733836800*t^18 - 3442233729024*t^16 + 466915000320*t^14 -
38478827520*t^12 + 35131428864*t^10 - 7607043072*t^8 + 510879744*t^6 -
3716928*t^4 - 461376*t^2 + 432 ], [ 0, 0, 0, -442368*t^24 -
48660480*t^22 + 222732288*t^20 + 64806912*t^18 - 1107219456*t^16 +
878653440*t^14 - 394343424*t^12 + 219663360*t^10 - 69201216*t^8 +
1012608*t^6 + 870048*t^4 - 47520*t^2 - 108, -113246208*t^36 +
30236737536*t^34 + 60898148352*t^32 - 2092563431424*t^30 +
7789612105728*t^28 - 8993645789184*t^26 + 2462644961280*t^24 -
7470640005120*t^22 + 13768934916096*t^20 - 9889733836800*t^18 +
3442233729024*t^16 - 466915000320*t^14 + 38478827520*t^12 -
35131428864*t^10 + 7607043072*t^8 - 510879744*t^6 + 3716928*t^4 +
461376*t^2 - 432 ], [ 0, 0, 0, -110592*t^24 - 12165120*t^22 +
55683072*t^20 + 16201728*t^18 - 276804864*t^16 + 219663360*t^14 -
98585856*t^12 + 54915840*t^10 - 17300304*t^8 + 253152*t^6 + 217512*t^4
- 11880*t^2 - 27, 14155776*t^36 - 3779592192*t^34 - 7612268544*t^32 +
261570428928*t^30 - 973701513216*t^28 + 1124205723648*t^26 -
307830620160*t^24 + 933830000640*t^22 - 1721116864512*t^20 +
1236216729600*t^18 - 430279216128*t^16 + 58364375040*t^14 -
4809853440*t^12 + 4391428608*t^10 - 950880384*t^8 + 63859968*t^6 -
464616*t^4 - 57672*t^2 + 54 ], [ 0, 0, 0, -110592*t^24 - 12165120*t^22
+ 55683072*t^20 + 16201728*t^18 - 276804864*t^16 + 219663360*t^14 -
98585856*t^12 + 54915840*t^10 - 17300304*t^8 + 253152*t^6 + 217512*t^4
- 11880*t^2 - 27, -14155776*t^36 + 3779592192*t^34 + 7612268544*t^32 -
261570428928*t^30 + 973701513216*t^28 - 1124205723648*t^26 +
307830620160*t^24 - 933830000640*t^22 + 1721116864512*t^20 -
1236216729600*t^18 + 430279216128*t^16 - 58364375040*t^14 +
4809853440*t^12 - 4391428608*t^10 + 950880384*t^8 - 63859968*t^6 +
464616*t^4 + 57672*t^2 - 54 ], [ 0, 0, 0, -27648*t^16 - 3207168*t^14 -
5059584*t^12 + 4119552*t^10 - 1226880*t^8 + 1029888*t^6 - 316224*t^4 -
50112*t^2 - 108, -1769472*t^24 + 456523776*t^22 + 5109350400*t^20 +
563576832*t^18 - 606818304*t^16 - 3849928704*t^14 + 3232051200*t^12 -
962482176*t^10 - 37926144*t^8 + 8805888*t^6 + 19958400*t^4 +
445824*t^2 - 432 ], [ 0, 0, 0, -27648*t^16 - 3207168*t^14 -
5059584*t^12 + 4119552*t^10 - 1226880*t^8 + 1029888*t^6 - 316224*t^4 -
50112*t^2 - 108, 1769472*t^24 - 456523776*t^22 - 5109350400*t^20 -
563576832*t^18 + 606818304*t^16 + 3849928704*t^14 - 3232051200*t^12 +
962482176*t^10 + 37926144*t^8 - 8805888*t^6 - 19958400*t^4 -
445824*t^2 + 432 ], [ 0, 0, 0, -6912*t^16 - 801792*t^14 - 1264896*t^12
+ 1029888*t^10 - 306720*t^8 + 257472*t^6 - 79056*t^4 - 12528*t^2 - 27,
221184*t^24 - 57065472*t^22 - 638668800*t^20 - 70447104*t^18 +
75852288*t^16 + 481241088*t^14 - 404006400*t^12 + 120310272*t^10 +
4740768*t^8 - 1100736*t^6 - 2494800*t^4 - 55728*t^2 + 54 ], [ 0, 0, 0,
-6912*t^16 - 801792*t^14 - 1264896*t^12 + 1029888*t^10 - 306720*t^8 +
257472*t^6 - 79056*t^4 - 12528*t^2 - 27, -221184*t^24 + 57065472*t^22
+ 638668800*t^20 + 70447104*t^18 - 75852288*t^16 - 481241088*t^14 +
404006400*t^12 - 120310272*t^10 - 4740768*t^8 + 1100736*t^6 +
2494800*t^4 + 55728*t^2 - 54 ], [ 0, 0, 0, -1811939328*t^16 -
1811939328*t^14 - 7134511104*t^12 - 3369074688*t^10 - 738754560*t^8 -
52641792*t^6 - 1741824*t^4 - 6912*t^2 - 108, -29686813949952*t^24 -
44530220924928*t^22 + 214301688201216*t^20 + 232160162217984*t^18 +
164792258002944*t^16 + 56684709937152*t^14 + 9357307674624*t^12 +
885698592768*t^10 + 40232484864*t^8 + 885620736*t^6 + 12773376*t^4 -
41472*t^2 - 432 ], [ 0, 0, 0, -1811939328*t^16 - 1811939328*t^14 -
7134511104*t^12 - 3369074688*t^10 - 738754560*t^8 - 52641792*t^6 -
1741824*t^4 - 6912*t^2 - 108, 29686813949952*t^24 +
44530220924928*t^22 - 214301688201216*t^20 - 232160162217984*t^18 -
164792258002944*t^16 - 56684709937152*t^14 - 9357307674624*t^12 -
885698592768*t^10 - 40232484864*t^8 - 885620736*t^6 - 12773376*t^4 +
41472*t^2 + 432 ], [ 0, 0, 0, -1855425871872*t^24 - 4638564679680*t^22
- 11190537289728*t^20 - 15553687191552*t^18 - 10313105670144*t^16 -
3408710860800*t^14 - 614445613056*t^12 - 53261107200*t^10 -
2517848064*t^8 - 59332608*t^6 - 667008*t^4 - 4320*t^2 - 27,
-972777519512027136*t^36 - 3647915698170101760*t^34 +
2051952580220682240*t^32 + 20397928612267819008*t^30 +
33951265400234704896*t^28 + 30617554940910895104*t^26 +
17785117899497668608*t^24 + 6808203212101779456*t^22 +
1663165831279804416*t^20 + 257577264353378304*t^18 +
25986966113746944*t^16 + 1662158987329536*t^14 + 67844840620032*t^12 +
1824948486144*t^10 + 31619579904*t^8 + 296828928*t^6 + 466560*t^4 -
12960*t^2 - 54 ], [ 0, 0, 0, -1855425871872*t^24 - 4638564679680*t^22
- 11190537289728*t^20 - 15553687191552*t^18 - 10313105670144*t^16 -
3408710860800*t^14 - 614445613056*t^12 - 53261107200*t^10 -
2517848064*t^8 - 59332608*t^6 - 667008*t^4 - 4320*t^2 - 27,
972777519512027136*t^36 + 3647915698170101760*t^34 -
2051952580220682240*t^32 - 20397928612267819008*t^30 -
33951265400234704896*t^28 - 30617554940910895104*t^26 -
17785117899497668608*t^24 - 6808203212101779456*t^22 -
1663165831279804416*t^20 - 257577264353378304*t^18 -
25986966113746944*t^16 - 1662158987329536*t^14 - 67844840620032*t^12 -
1824948486144*t^10 - 31619579904*t^8 - 296828928*t^6 - 466560*t^4 +
12960*t^2 + 54 ], [ 0, 0, 0, -452984832*t^16 - 452984832*t^14 -
1783627776*t^12 - 842268672*t^10 - 184688640*t^8 - 13160448*t^6 -
435456*t^4 - 1728*t^2 - 27, -3710851743744*t^24 - 5566277615616*t^22 +
26787711025152*t^20 + 29020020277248*t^18 + 20599032250368*t^16 +
7085588742144*t^14 + 1169663459328*t^12 + 110712324096*t^10 +
5029060608*t^8 + 110702592*t^6 + 1596672*t^4 - 5184*t^2 - 54 ], [ 0,
0, 0, -452984832*t^16 - 452984832*t^14 - 1783627776*t^12 -
842268672*t^10 - 184688640*t^8 - 13160448*t^6 - 435456*t^4 - 1728*t^2
- 27, 3710851743744*t^24 + 5566277615616*t^22 - 26787711025152*t^20 -
29020020277248*t^18 - 20599032250368*t^16 - 7085588742144*t^14 -
1169663459328*t^12 - 110712324096*t^10 - 5029060608*t^8 -
110702592*t^6 - 1596672*t^4 + 5184*t^2 + 54 ], [ 0, 0, 0,
-7421703487488*t^24 - 18554258718720*t^22 - 44762149158912*t^20 -
62214748766208*t^18 - 41252422680576*t^16 - 13634843443200*t^14 -
2457782452224*t^12 - 213044428800*t^10 - 10071392256*t^8 -
237330432*t^6 - 2668032*t^4 - 17280*t^2 - 108,
7782220156096217088*t^36 + 29183325585360814080*t^34 -
16415620641765457920*t^32 - 163183428898142552064*t^30 -
271610123201877639168*t^28 - 244940439527287160832*t^26 -
142280943195981348864*t^24 - 54465625696814235648*t^22 -
13305326650238435328*t^20 - 2060618114827026432*t^18 -
207895728909975552*t^16 - 13297271898636288*t^14 -
542758724960256*t^12 - 14599587889152*t^10 - 252956639232*t^8 -
2374631424*t^6 - 3732480*t^4 + 103680*t^2 + 432 ], [ 0, 0, 0,
-7421703487488*t^24 - 18554258718720*t^22 - 44762149158912*t^20 -
62214748766208*t^18 - 41252422680576*t^16 - 13634843443200*t^14 -
2457782452224*t^12 - 213044428800*t^10 - 10071392256*t^8 -
237330432*t^6 - 2668032*t^4 - 17280*t^2 - 108,
-7782220156096217088*t^36 - 29183325585360814080*t^34 +
16415620641765457920*t^32 + 163183428898142552064*t^30 +
271610123201877639168*t^28 + 244940439527287160832*t^26 +
142280943195981348864*t^24 + 54465625696814235648*t^22 +
13305326650238435328*t^20 + 2060618114827026432*t^18 +
207895728909975552*t^16 + 13297271898636288*t^14 +
542758724960256*t^12 + 14599587889152*t^10 + 252956639232*t^8 +
2374631424*t^6 + 3732480*t^4 - 103680*t^2 - 432 ], [ 0, 0, 0,
-17739*t^24 - 320112*t^23 - 3144528*t^22 - 18767808*t^21 -
84271968*t^20 - 275035392*t^19 - 733798656*t^18 - 1430369280*t^17 -
2579662080*t^16 - 3808124928*t^15 - 7866851328*t^14 - 5472092160*t^13
- 8468250624*t^12 + 21888368640*t^11 - 125869621248*t^10 +
243719995392*t^9 - 660393492480*t^8 + 1464698142720*t^7 -
3005639294976*t^6 + 4506179862528*t^5 - 5522847694848*t^4 +
4919868260352*t^3 - 3297276592128*t^2 + 1342647042048*t -
297611034624, 867510*t^36 + 25345872*t^35 + 349986096*t^34 +
3283257024*t^33 + 22604196960*t^32 + 121327994880*t^31 +
519456098304*t^30 + 1836846637056*t^29 + 5448260745216*t^28 +
14046538088448*t^27 + 31155737149440*t^26 + 61732891459584*t^25 +
103622008209408*t^24 + 177813547646976*t^23 + 273955331506176*t^22 +
509882161692672*t^21 + 580576552943616*t^20 + 510617469321216*t^19 +
787850868031488*t^18 - 2042469877284864*t^17 + 9289224847097856*t^16 -
32632458348331008*t^15 + 70132564865581056*t^14 -
182081072790503424*t^13 + 424435745625735168*t^12 -
1011431693673824256*t^11 + 2041822389825699840*t^10 -
3682215680658112512*t^9 + 5712915459175612416*t^8 -
7704293197190529024*t^7 + 8715027163763441664*t^6 -
8142183907794616320*t^5 + 6067767918471413760*t^4 -
3525370385610571776*t^3 + 1503178836374716416*t^2 -
435438765314408448*t + 59614833263247360 ], [ 0, 0, 0, -7884*t^16 -
100224*t^15 - 680832*t^14 - 2225664*t^13 - 7402752*t^12 -
13547520*t^11 - 14321664*t^10 - 24993792*t^9 - 74041344*t^8 +
99975168*t^7 - 229146624*t^6 + 867041280*t^5 - 1895104512*t^4 +
2279079936*t^3 - 2788687872*t^2 + 1642070016*t - 516685824,
-257040*t^24 - 5453568*t^23 - 48418560*t^22 - 308689920*t^21 -
1344397824*t^20 - 4317401088*t^19 - 11010650112*t^18 -
27196342272*t^17 - 40513499136*t^16 - 66762178560*t^15 -
51969392640*t^14 - 34780741632*t^13 - 401266704384*t^12 +
139122966528*t^11 - 831510282240*t^10 + 4272779427840*t^9 -
10371455778816*t^8 + 27849054486528*t^7 - 45099622858752*t^6 +
70736299425792*t^5 - 88106455793664*t^4 + 80921210388480*t^3 -
50770539970560*t^2 + 22873922076672*t - 4312415600640 ], [ 0, 0, 0,
-7884*t^16 - 100224*t^15 - 680832*t^14 - 2225664*t^13 - 7402752*t^12 -
13547520*t^11 - 14321664*t^10 - 24993792*t^9 - 74041344*t^8 +
99975168*t^7 - 229146624*t^6 + 867041280*t^5 - 1895104512*t^4 +
2279079936*t^3 - 2788687872*t^2 + 1642070016*t - 516685824,
257040*t^24 + 5453568*t^23 + 48418560*t^22 + 308689920*t^21 +
1344397824*t^20 + 4317401088*t^19 + 11010650112*t^18 +
27196342272*t^17 + 40513499136*t^16 + 66762178560*t^15 +
51969392640*t^14 + 34780741632*t^13 + 401266704384*t^12 -
139122966528*t^11 + 831510282240*t^10 - 4272779427840*t^9 +
10371455778816*t^8 - 27849054486528*t^7 + 45099622858752*t^6 -
70736299425792*t^5 + 88106455793664*t^4 - 80921210388480*t^3 +
50770539970560*t^2 - 22873922076672*t + 4312415600640 ], [ 0, 0, 0,
-17739*t^24 - 320112*t^23 - 3144528*t^22 - 18767808*t^21 -
84271968*t^20 - 275035392*t^19 - 733798656*t^18 - 1430369280*t^17 -
2579662080*t^16 - 3808124928*t^15 - 7866851328*t^14 - 5472092160*t^13
- 8468250624*t^12 + 21888368640*t^11 - 125869621248*t^10 +
243719995392*t^9 - 660393492480*t^8 + 1464698142720*t^7 -
3005639294976*t^6 + 4506179862528*t^5 - 5522847694848*t^4 +
4919868260352*t^3 - 3297276592128*t^2 + 1342647042048*t -
297611034624, -867510*t^36 - 25345872*t^35 - 349986096*t^34 -
3283257024*t^33 - 22604196960*t^32 - 121327994880*t^31 -
519456098304*t^30 - 1836846637056*t^29 - 5448260745216*t^28 -
14046538088448*t^27 - 31155737149440*t^26 - 61732891459584*t^25 -
103622008209408*t^24 - 177813547646976*t^23 - 273955331506176*t^22 -
509882161692672*t^21 - 580576552943616*t^20 - 510617469321216*t^19 -
787850868031488*t^18 + 2042469877284864*t^17 - 9289224847097856*t^16 +
32632458348331008*t^15 - 70132564865581056*t^14 +
182081072790503424*t^13 - 424435745625735168*t^12 +
1011431693673824256*t^11 - 2041822389825699840*t^10 +
3682215680658112512*t^9 - 5712915459175612416*t^8 +
7704293197190529024*t^7 - 8715027163763441664*t^6 +
8142183907794616320*t^5 - 6067767918471413760*t^4 +
3525370385610571776*t^3 - 1503178836374716416*t^2 +
435438765314408448*t - 59614833263247360 ], [ 0, 0, 0, -70956*t^24 -
1280448*t^23 - 12578112*t^22 - 75071232*t^21 - 337087872*t^20 -
1100141568*t^19 - 2935194624*t^18 - 5721477120*t^17 - 10318648320*t^16
- 15232499712*t^15 - 31467405312*t^14 - 21888368640*t^13 -
33873002496*t^12 + 87553474560*t^11 - 503478484992*t^10 +
974879981568*t^9 - 2641573969920*t^8 + 5858792570880*t^7 -
12022557179904*t^6 + 18024719450112*t^5 - 22091390779392*t^4 +
19679473041408*t^3 - 13189106368512*t^2 + 5370588168192*t -
1190444138496, 6940080*t^36 + 202766976*t^35 + 2799888768*t^34 +
26266056192*t^33 + 180833575680*t^32 + 970623959040*t^31 +
4155648786432*t^30 + 14694773096448*t^29 + 43586085961728*t^28 +
112372304707584*t^27 + 249245897195520*t^26 + 493863131676672*t^25 +
828976065675264*t^24 + 1422508381175808*t^23 + 2191642652049408*t^22 +
4079057293541376*t^21 + 4644612423548928*t^20 + 4084939754569728*t^19
+ 6302806944251904*t^18 - 16339759018278912*t^17 +
74313798776782848*t^16 - 261059666786648064*t^15 +
561060518924648448*t^14 - 1456648582324027392*t^13 +
3395485965005881344*t^12 - 8091453549390594048*t^11 +
16334579118605598720*t^10 - 29457725445264900096*t^9 +
45703323673404899328*t^8 - 61634345577524232192*t^7 +
69720217310107533312*t^6 - 65137471262356930560*t^5 +
48542143347771310080*t^4 - 28202963084884574208*t^3 +
12025430690997731328*t^2 - 3483510122515267584*t + 476918666105978880
], [ 0, 0, 0, -1971*t^16 - 25056*t^15 - 170208*t^14 - 556416*t^13 -
1850688*t^12 - 3386880*t^11 - 3580416*t^10 - 6248448*t^9 -
18510336*t^8 + 24993792*t^7 - 57286656*t^6 + 216760320*t^5 -
473776128*t^4 + 569769984*t^3 - 697171968*t^2 + 410517504*t -
129171456, -32130*t^24 - 681696*t^23 - 6052320*t^22 - 38586240*t^21 -
168049728*t^20 - 539675136*t^19 - 1376331264*t^18 - 3399542784*t^17 -
5064187392*t^16 - 8345272320*t^15 - 6496174080*t^14 - 4347592704*t^13
- 50158338048*t^12 + 17390370816*t^11 - 103938785280*t^10 +
534097428480*t^9 - 1296431972352*t^8 + 3481131810816*t^7 -
5637452857344*t^6 + 8842037428224*t^5 - 11013306974208*t^4 +
10115151298560*t^3 - 6346317496320*t^2 + 2859240259584*t -
539051950080 ], [ 0, 0, 0, -70956*t^24 - 1280448*t^23 - 12578112*t^22
- 75071232*t^21 - 337087872*t^20 - 1100141568*t^19 - 2935194624*t^18 -
5721477120*t^17 - 10318648320*t^16 - 15232499712*t^15 -
31467405312*t^14 - 21888368640*t^13 - 33873002496*t^12 +
87553474560*t^11 - 503478484992*t^10 + 974879981568*t^9 -
2641573969920*t^8 + 5858792570880*t^7 - 12022557179904*t^6 +
18024719450112*t^5 - 22091390779392*t^4 + 19679473041408*t^3 -
13189106368512*t^2 + 5370588168192*t - 1190444138496, -6940080*t^36 -
202766976*t^35 - 2799888768*t^34 - 26266056192*t^33 -
180833575680*t^32 - 970623959040*t^31 - 4155648786432*t^30 -
14694773096448*t^29 - 43586085961728*t^28 - 112372304707584*t^27 -
249245897195520*t^26 - 493863131676672*t^25 - 828976065675264*t^24 -
1422508381175808*t^23 - 2191642652049408*t^22 - 4079057293541376*t^21
- 4644612423548928*t^20 - 4084939754569728*t^19 -
6302806944251904*t^18 + 16339759018278912*t^17 -
74313798776782848*t^16 + 261059666786648064*t^15 -
561060518924648448*t^14 + 1456648582324027392*t^13 -
3395485965005881344*t^12 + 8091453549390594048*t^11 -
16334579118605598720*t^10 + 29457725445264900096*t^9 -
45703323673404899328*t^8 + 61634345577524232192*t^7 -
69720217310107533312*t^6 + 65137471262356930560*t^5 -
48542143347771310080*t^4 + 28202963084884574208*t^3 -
12025430690997731328*t^2 + 3483510122515267584*t - 476918666105978880
], [ 0, 0, 0, -1971*t^16 - 25056*t^15 - 170208*t^14 - 556416*t^13 -
1850688*t^12 - 3386880*t^11 - 3580416*t^10 - 6248448*t^9 -
18510336*t^8 + 24993792*t^7 - 57286656*t^6 + 216760320*t^5 -
473776128*t^4 + 569769984*t^3 - 697171968*t^2 + 410517504*t -
129171456, 32130*t^24 + 681696*t^23 + 6052320*t^22 + 38586240*t^21 +
168049728*t^20 + 539675136*t^19 + 1376331264*t^18 + 3399542784*t^17 +
5064187392*t^16 + 8345272320*t^15 + 6496174080*t^14 + 4347592704*t^13
+ 50158338048*t^12 - 17390370816*t^11 + 103938785280*t^10 -
534097428480*t^9 + 1296431972352*t^8 - 3481131810816*t^7 +
5637452857344*t^6 - 8842037428224*t^5 + 11013306974208*t^4 -
10115151298560*t^3 + 6346317496320*t^2 - 2859240259584*t +
539051950080 ], [ 0, 0, 0, -108*t^32 + 6480*t^28 - 14256*t^24 -
6480*t^20 + 28728*t^16 - 6480*t^12 - 14256*t^8 + 6480*t^4 - 108,
432*t^48 + 54432*t^44 - 451008*t^40 + 598752*t^36 + 900720*t^32 -
2068416*t^28 + 2068416*t^20 - 900720*t^16 - 598752*t^12 + 451008*t^8 -
54432*t^4 - 432 ], [ 0, 0, 0, -108*t^24 + 6264*t^20 - 1620*t^16 -
15984*t^12 - 1620*t^8 + 6264*t^4 - 108, 432*t^36 + 55728*t^32 -
285120*t^28 - 423360*t^24 + 541728*t^20 + 541728*t^16 - 423360*t^12 -
285120*t^8 + 55728*t^4 + 432 ], [ 0, 0, 0, -108*t^24 + 6264*t^20 -
1620*t^16 - 15984*t^12 - 1620*t^8 + 6264*t^4 - 108, -432*t^36 -
55728*t^32 + 285120*t^28 + 423360*t^24 - 541728*t^20 - 541728*t^16 +
423360*t^12 + 285120*t^8 - 55728*t^4 - 432 ], [ 0, 0, 0, -27*t^32 +
1620*t^28 - 3564*t^24 - 1620*t^20 + 7182*t^16 - 1620*t^12 - 3564*t^8 +
1620*t^4 - 27, 54*t^48 + 6804*t^44 - 56376*t^40 + 74844*t^36 +
112590*t^32 - 258552*t^28 + 258552*t^20 - 112590*t^16 - 74844*t^12 +
56376*t^8 - 6804*t^4 - 54 ], [ 0, 0, 0, -27*t^24 + 1566*t^20 -
405*t^16 - 3996*t^12 - 405*t^8 + 1566*t^4 - 27, -54*t^36 - 6966*t^32 +
35640*t^28 + 52920*t^24 - 67716*t^20 - 67716*t^16 + 52920*t^12 +
35640*t^8 - 6966*t^4 - 54 ], [ 0, 0, 0, -27*t^24 + 1566*t^20 -
405*t^16 - 3996*t^12 - 405*t^8 + 1566*t^4 - 27, 54*t^36 + 6966*t^32 -
35640*t^28 - 52920*t^24 + 67716*t^20 + 67716*t^16 - 52920*t^12 -
35640*t^8 + 6966*t^4 + 54 ], [ 0, 0, 0, -27*t^24 + 1674*t^20 -
6885*t^16 + 10476*t^12 - 6885*t^8 + 1674*t^4 - 27, 54*t^36 + 6642*t^32
- 76464*t^28 + 284256*t^24 - 517428*t^20 + 517428*t^16 - 284256*t^12 +
76464*t^8 - 6642*t^4 - 54 ], [ 0, 0, 0, -27*t^16 + 1620*t^12 -
3618*t^8 + 1620*t^4 - 27, -54*t^24 - 6804*t^20 + 56214*t^16 -
95256*t^12 + 56214*t^8 - 6804*t^4 - 54 ], [ 0, 0, 0, -27*t^16 +
1620*t^12 - 3618*t^8 + 1620*t^4 - 27, 54*t^24 + 6804*t^20 - 56214*t^16
+ 95256*t^12 - 56214*t^8 + 6804*t^4 + 54 ], [ 0, 0, 0, -108*t^16 +
6480*t^12 - 14472*t^8 + 6480*t^4 - 108, -432*t^24 - 54432*t^20 +
449712*t^16 - 762048*t^12 + 449712*t^8 - 54432*t^4 - 432 ], [ 0, 0, 0,
-108*t^24 + 6696*t^20 - 27540*t^16 + 41904*t^12 - 27540*t^8 + 6696*t^4
- 108, 432*t^36 + 53136*t^32 - 611712*t^28 + 2274048*t^24 -
4139424*t^20 + 4139424*t^16 - 2274048*t^12 + 611712*t^8 - 53136*t^4 -
432 ], [ 0, 0, 0, -108*t^16 + 6480*t^12 - 14472*t^8 + 6480*t^4 - 108,
432*t^24 + 54432*t^20 - 449712*t^16 + 762048*t^12 - 449712*t^8 +
54432*t^4 + 432 ], [ 0, 0, 0, -27*t^28 - 1296*t^27 - 2592*t^26 +
762048*t^25 + 18645120*t^24 + 218930688*t^23 + 1446128640*t^22 +
4473999360*t^21 - 8445136896*t^20 - 150641344512*t^19 -
654791344128*t^18 - 907540955136*t^17 + 4010303029248*t^16 +
24341253193728*t^15 + 51034403635200*t^14 - 7484215394304*t^13 -
317498427703296*t^12 - 814256542973952*t^11 - 761478374227968*t^10 +
846219152719872*t^9 + 3654899057295360*t^8 + 5345713865097216*t^7 +
4203003456258048*t^6 + 1608654230913024*t^5 + 35253091565568*t^4 -
163277476724736*t^3 - 29686813949952*t^2, 54*t^42 + 3888*t^41 +
241056*t^40 + 10217664*t^39 + 264871296*t^38 + 4295960064*t^37 +
42977765376*t^36 + 210655199232*t^35 - 735926943744*t^34 -
22524538060800*t^33 - 200608395558912*t^32 - 985373276110848*t^31 -
1978817014923264*t^30 + 8868427305320448*t^29 + 88665439982321664*t^28
+ 321310758015074304*t^27 + 279380339346898944*t^26 -
2805499175544815616*t^25 - 14689451244804636672*t^24 -
28818268225787658240*t^23 + 25235310639499444224*t^22 +
311922964830892326912*t^21 + 827318285604879335424*t^20 +
587831300812217253888*t^19 - 2843300115942438076416*t^18 -
10685903310986485432320*t^17 - 15984104276431000829952*t^16 -
554279653325414596608*t^15 + 49416991593669781880832*t^14 +
113736009507646627381248*t^13 + 135421954454387247022080*t^12 +
74294499884695516348416*t^11 - 35217023749080389517312*t^10 -
109638592697262164410368*t^9 - 106614713777859028058112*t^8 -
59538361692973864845312*t^7 - 20042135234506295083008*t^6 -
3957258949374926389248*t^5 - 513626530302350327808*t^4 -
62257761248769736704*t^3 ], [ 0, 0, 0, -27*t^28 - 1296*t^27 -
2592*t^26 + 762048*t^25 + 18645120*t^24 + 218930688*t^23 +
1446128640*t^22 + 4473999360*t^21 - 8445136896*t^20 -
150641344512*t^19 - 654791344128*t^18 - 907540955136*t^17 +
4010303029248*t^16 + 24341253193728*t^15 + 51034403635200*t^14 -
7484215394304*t^13 - 317498427703296*t^12 - 814256542973952*t^11 -
761478374227968*t^10 + 846219152719872*t^9 + 3654899057295360*t^8 +
5345713865097216*t^7 + 4203003456258048*t^6 + 1608654230913024*t^5 +
35253091565568*t^4 - 163277476724736*t^3 - 29686813949952*t^2,
-54*t^42 - 3888*t^41 - 241056*t^40 - 10217664*t^39 - 264871296*t^38 -
4295960064*t^37 - 42977765376*t^36 - 210655199232*t^35 +
735926943744*t^34 + 22524538060800*t^33 + 200608395558912*t^32 +
985373276110848*t^31 + 1978817014923264*t^30 - 8868427305320448*t^29 -
88665439982321664*t^28 - 321310758015074304*t^27 -
279380339346898944*t^26 + 2805499175544815616*t^25 +
14689451244804636672*t^24 + 28818268225787658240*t^23 -
25235310639499444224*t^22 - 311922964830892326912*t^21 -
827318285604879335424*t^20 - 587831300812217253888*t^19 +
2843300115942438076416*t^18 + 10685903310986485432320*t^17 +
15984104276431000829952*t^16 + 554279653325414596608*t^15 -
49416991593669781880832*t^14 - 113736009507646627381248*t^13 -
135421954454387247022080*t^12 - 74294499884695516348416*t^11 +
35217023749080389517312*t^10 + 109638592697262164410368*t^9 +
106614713777859028058112*t^8 + 59538361692973864845312*t^7 +
20042135234506295083008*t^6 + 3957258949374926389248*t^5 +
513626530302350327808*t^4 + 62257761248769736704*t^3 ], [ 0, 0, 0,
-108*t^22 - 4752*t^21 + 8208*t^20 + 2989440*t^19 + 62339328*t^18 +
638337024*t^17 + 3673755648*t^16 + 10080681984*t^15 - 12150079488*t^14
- 221109682176*t^13 - 894669815808*t^12 - 1768877457408*t^11 -
777605087232*t^10 + 5161309175808*t^9 + 15047703134208*t^8 +
20917027602432*t^7 + 16341880799232*t^6 + 6269310074880*t^5 +
137707388928*t^4 - 637802643456*t^3 - 115964116992*t^2, 432*t^33 +
28512*t^32 + 1757376*t^31 + 71245440*t^30 + 1694836224*t^29 +
24404944896*t^28 + 205952827392*t^27 + 662685401088*t^26 -
6586979844096*t^25 - 110131636469760*t^24 - 813424020553728*t^23 -
3662208293142528*t^22 - 9354237853040640*t^21 - 1998858309599232*t^20
+ 89587256867684352*t^19 + 381431083394138112*t^18 +
716698054941474816*t^17 - 127926931814350848*t^16 -
4789369780756807680*t^15 - 15000405168711794688*t^14 -
26654278305504559104*t^13 - 28870347710728765440*t^12 -
13813897954005614592*t^11 + 11118016114100011008*t^10 +
27642520567730405376*t^9 + 26204610047250530304*t^8 +
14558532308312260608*t^7 + 4895949356626083840*t^6 +
966127673187237888*t^5 + 125397102124597248*t^4 +
15199648742375424*t^3 ], [ 0, 0, 0, -108*t^30 - 6480*t^29 - 74736*t^28
+ 2830464*t^27 + 111428352*t^26 + 1831624704*t^25 + 17489184768*t^24 +
98262319104*t^23 + 227672211456*t^22 - 1027495231488*t^21 -
11106315730944*t^20 - 40916718452736*t^19 - 33829898158080*t^18 +
335189750317056*t^17 + 1579216105635840*t^16 + 2681518002536448*t^15 -
2165113482117120*t^14 - 20949359847800832*t^13 -
45491469233946624*t^12 - 33668963745398784*t^11 +
59682904199921664*t^10 + 206071019033591808*t^9 +
293419830516645888*t^8 + 245836506319552512*t^7 +
119645281921794048*t^6 + 24313500625010688*t^5 - 5135818813341696*t^4
- 3562417673994240*t^3 - 474989023199232*t^2, 432*t^45 + 38880*t^44 +
2524608*t^43 + 119035008*t^42 + 3749856768*t^41 + 79222800384*t^40 +
1133704581120*t^39 + 10574164942848*t^38 + 50047313412096*t^37 -
180547243868160*t^36 - 5547197342416896*t^35 - 51541099444961280*t^34
- 274005871363620864*t^33 - 699224096193380352*t^32 +
1539397909034827776*t^31 + 22876765719757848576*t^30 +
98016765829164564480*t^29 + 146610969832203485184*t^28 -
607190132200411496448*t^27 - 4235795678515314032640*t^26 -
10416869423878735134720*t^25 + 83334955391029881077760*t^23 +
271090923424980098088960*t^22 + 310881347686610686181376*t^21 -
600518532432705475313664*t^20 - 3211813382690064448880640*t^19 -
5997006872840201457106944*t^18 - 3228351403728207140093952*t^17 +
11731033694241119935660032*t^16 + 36776445513085454219476992*t^15 +
55341834128998112396574720*t^14 + 47650062340277363835666432*t^13 +
12407112124746939771125760*t^12 - 27513801517774654823989248*t^11 -
46505669234730873957384192*t^10 - 39888683820939501990051840*t^9 -
22299235893039349213691904*t^8 - 8443926771528766234558464*t^7 -
2144344070691376041295872*t^6 - 363834356737810341298176*t^5 -
44825588099114210426880*t^4 - 3984496719921263149056*t^3 ], [ 0, 0, 0,
-27*t^26 - 1188*t^25 + 2052*t^24 + 749088*t^23 + 15657408*t^22 +
159314688*t^21 + 871395840*t^20 + 1614034944*t^19 - 11619237888*t^18 -
99396550656*t^17 - 310057058304*t^16 - 58470432768*t^15 +
3189650227200*t^14 + 12170626596864*t^13 + 16041212116992*t^12 -
29041310564352*t^11 - 167626584096768*t^10 - 308513473560576*t^9 -
138365122904064*t^8 + 586416044113920*t^7 + 1516375784816640*t^6 +
1836523720802304*t^5 + 1251252822343680*t^4 + 409121404747776*t^3 -
11132555231232*t^2 - 44530220924928*t - 7421703487488, 54*t^39 +
3564*t^38 + 219672*t^37 + 8900496*t^36 + 211522752*t^35 +
3030172416*t^34 + 24928929792*t^33 + 64059393024*t^32 -
1081127264256*t^31 - 15765190410240*t^30 - 105924913201152*t^29 -
368185278726144*t^28 + 33037641842688*t^27 + 7621814859595776*t^26 +
40763486141153280*t^25 + 86770633421094912*t^24 -
143513501174857728*t^23 - 1615856026572029952*t^22 -
4873796325482692608*t^21 - 3154413829937430528*t^20 +
28818268225787658240*t^19 + 117515609958437093376*t^18 +
179551947234868199424*t^17 - 143042733745612259328*t^16 -
1316088864829744349184*t^15 - 2905389137340716285952*t^14 -
2324805007525923520512*t^13 + 4149880060480352944128*t^12 +
16531820293939336839168*t^11 + 26925203069642458791936*t^10 +
24185538582160814899200*t^9 + 6321564311251423592448*t^8 -
14476115062940869066752*t^7 - 23627276383370386341888*t^6 -
18893832171317316550656*t^5 - 9319330234115158966272*t^4 -
2876016736437308227584*t^3 - 542809855887711141888*t^2 -
70039981404865953792*t - 7782220156096217088 ], [ 0, 0, 0, -27*t^26 -
1188*t^25 + 2052*t^24 + 749088*t^23 + 15657408*t^22 + 159314688*t^21 +
871395840*t^20 + 1614034944*t^19 - 11619237888*t^18 - 99396550656*t^17
- 310057058304*t^16 - 58470432768*t^15 + 3189650227200*t^14 +
12170626596864*t^13 + 16041212116992*t^12 - 29041310564352*t^11 -
167626584096768*t^10 - 308513473560576*t^9 - 138365122904064*t^8 +
586416044113920*t^7 + 1516375784816640*t^6 + 1836523720802304*t^5 +
1251252822343680*t^4 + 409121404747776*t^3 - 11132555231232*t^2 -
44530220924928*t - 7421703487488, -54*t^39 - 3564*t^38 - 219672*t^37 -
8900496*t^36 - 211522752*t^35 - 3030172416*t^34 - 24928929792*t^33 -
64059393024*t^32 + 1081127264256*t^31 + 15765190410240*t^30 +
105924913201152*t^29 + 368185278726144*t^28 - 33037641842688*t^27 -
7621814859595776*t^26 - 40763486141153280*t^25 -
86770633421094912*t^24 + 143513501174857728*t^23 +
1615856026572029952*t^22 + 4873796325482692608*t^21 +
3154413829937430528*t^20 - 28818268225787658240*t^19 -
117515609958437093376*t^18 - 179551947234868199424*t^17 +
143042733745612259328*t^16 + 1316088864829744349184*t^15 +
2905389137340716285952*t^14 + 2324805007525923520512*t^13 -
4149880060480352944128*t^12 - 16531820293939336839168*t^11 -
26925203069642458791936*t^10 - 24185538582160814899200*t^9 -
6321564311251423592448*t^8 + 14476115062940869066752*t^7 +
23627276383370386341888*t^6 + 18893832171317316550656*t^5 +
9319330234115158966272*t^4 + 2876016736437308227584*t^3 +
542809855887711141888*t^2 + 70039981404865953792*t +
7782220156096217088 ], [ 0, 0, 0, -27*t^22 - 1188*t^21 + 2052*t^20 +
747360*t^19 + 15584832*t^18 + 159584256*t^17 + 918438912*t^16 +
2520170496*t^15 - 3037519872*t^14 - 55277420544*t^13 -
223667453952*t^12 - 442219364352*t^11 - 194401271808*t^10 +
1290327293952*t^9 + 3761925783552*t^8 + 5229256900608*t^7 +
4085470199808*t^6 + 1567327518720*t^5 + 34426847232*t^4 -
159450660864*t^3 - 28991029248*t^2, 54*t^33 + 3564*t^32 + 219672*t^31
+ 8905680*t^30 + 211854528*t^29 + 3050618112*t^28 + 25744103424*t^27 +
82835675136*t^26 - 823372480512*t^25 - 13766454558720*t^24 -
101678002569216*t^23 - 457776036642816*t^22 - 1169279731630080*t^21 -
249857288699904*t^20 + 11198407108460544*t^19 + 47678885424267264*t^18
+ 89587256867684352*t^17 - 15990866476793856*t^16 -
598671222594600960*t^15 - 1875050646088974336*t^14 -
3331784788188069888*t^13 - 3608793463841095680*t^12 -
1726737244250701824*t^11 + 1389752014262501376*t^10 +
3455315070966300672*t^9 + 3275576255906316288*t^8 +
1819816538539032576*t^7 + 611993669578260480*t^6 +
120765959148404736*t^5 + 15674637765574656*t^4 + 1899956092796928*t^3
], [ 0, 0, 0, -27*t^30 - 1620*t^29 - 18684*t^28 + 707616*t^27 +
27857088*t^26 + 457906176*t^25 + 4372296192*t^24 + 24565579776*t^23 +
56918052864*t^22 - 256873807872*t^21 - 2776578932736*t^20 -
10229179613184*t^19 - 8457474539520*t^18 + 83797437579264*t^17 +
394804026408960*t^16 + 670379500634112*t^15 - 541278370529280*t^14 -
5237339961950208*t^13 - 11372867308486656*t^12 - 8417240936349696*t^11
+ 14920726049980416*t^10 + 51517754758397952*t^9 +
73354957629161472*t^8 + 61459126579888128*t^7 + 29911320480448512*t^6
+ 6078375156252672*t^5 - 1283954703335424*t^4 - 890604418498560*t^3 -
118747255799808*t^2, 54*t^45 + 4860*t^44 + 315576*t^43 + 14879376*t^42
+ 468732096*t^41 + 9902850048*t^40 + 141713072640*t^39 +
1321770617856*t^38 + 6255914176512*t^37 - 22568405483520*t^36 -
693399667802112*t^35 - 6442637430620160*t^34 - 34250733920452608*t^33
- 87403012024172544*t^32 + 192424738629353472*t^31 +
2859595714969731072*t^30 + 12252095728645570560*t^29 +
18326371229025435648*t^28 - 75898766525051437056*t^27 -
529474459814414254080*t^26 - 1302108677984841891840*t^25 +
10416869423878735134720*t^23 + 33886365428122512261120*t^22 +
38860168460826335772672*t^21 - 75064816554088184414208*t^20 -
401476672836258056110080*t^19 - 749625859105025182138368*t^18 -
403543925466025892511744*t^17 + 1466379211780139991957504*t^16 +
4597055689135681777434624*t^15 + 6917729266124764049571840*t^14 +
5956257792534670479458304*t^13 + 1550889015593367471390720*t^12 -
3439225189721831852998656*t^11 - 5813208654341359244673024*t^10 -
4986085477617437748756480*t^9 - 2787404486629918651711488*t^8 -
1055490846441095779319808*t^7 - 268043008836422005161984*t^6 -
45479294592226292662272*t^5 - 5603198512389276303360*t^4 -
498062089990157893632*t^3 ], [ 0, 0, 0, -108*t^24 - 5184*t^23 -
6912*t^22 + 3193344*t^21 + 73972224*t^20 + 778899456*t^19 +
3989716992*t^18 + 2577235968*t^17 - 90228473856*t^16 -
511632211968*t^15 - 777605087232*t^14 + 3516408004608*t^13 +
19299984998400*t^12 + 28131264036864*t^11 - 49766725582848*t^10 -
261955692527616*t^9 - 369575828914176*t^8 + 84450868199424*t^7 +
1045880371150848*t^6 + 1633470551949312*t^5 + 1241047980048384*t^4 +
428603376402432*t^3 - 7421703487488*t^2 - 44530220924928*t -
7421703487488, 432*t^36 + 31104*t^35 + 1907712*t^34 + 80372736*t^33 +
2035196928*t^32 + 30985224192*t^31 + 264766095360*t^30 +
605499162624*t^29 - 13660083191808*t^28 - 183139842392064*t^27 -
1069050771800064*t^26 - 2199326520508416*t^25 + 12591141023121408*t^24
+ 112070128916496384*t^23 + 315454215872839680*t^22 -
292799364508680192*t^21 - 5051413852437086208*t^20 -
14527984924650110976*t^19 + 116223879397200887808*t^17 +
323290486555973517312*t^16 + 149913274628444258304*t^15 -
1292100468215151329280*t^14 - 3672313984335753510912*t^13 -
3300692072365138378752*t^12 + 4612322011137265631232*t^11 +
17935695713456382541824*t^10 + 24580613552140915310592*t^9 +
14667402642363663777792*t^8 - 5201198202450931089408*t^7 -
18194587530573077544960*t^6 - 17034307144175107178496*t^5 -
8950890748599978688512*t^4 - 2827864249221462884352*t^3 -
536973190770638979072*t^2 - 70039981404865953792*t -
7782220156096217088 ], [ 0, 0, 0, -108*t^16 - 3456*t^15 + 55296*t^14 +
2515968*t^13 + 29611008*t^12 + 147087360*t^11 + 123863040*t^10 -
2057895936*t^9 - 9911697408*t^8 - 16463167488*t^7 + 7927234560*t^6 +
75308728320*t^5 + 121286688768*t^4 + 82443239424*t^3 + 14495514624*t^2
- 7247757312*t - 1811939328, 432*t^24 + 20736*t^23 + 1327104*t^22 +
44402688*t^21 + 709502976*t^20 + 5081481216*t^19 - 2972712960*t^18 -
360579465216*t^17 - 3066284408832*t^16 - 12049906139136*t^15 -
13122064613376*t^14 + 91578584530944*t^13 + 457518757183488*t^12 +
732628676247552*t^11 - 839812135256064*t^10 - 6169551943237632*t^9 -
12559500938575872*t^8 - 11815467916197888*t^7 - 779278866186240*t^6 +
10656638495096832*t^5 + 11903484680994816*t^4 + 5959627900452864*t^3 +
1424967069597696*t^2 + 178120883699712*t + 29686813949952 ], [ 0, 0,
0, -27*t^24 - 1296*t^23 - 1728*t^22 + 798336*t^21 + 18493056*t^20 +
194724864*t^19 + 997429248*t^18 + 644308992*t^17 - 22557118464*t^16 -
127908052992*t^15 - 194401271808*t^14 + 879102001152*t^13 +
4824996249600*t^12 + 7032816009216*t^11 - 12441681395712*t^10 -
65488923131904*t^9 - 92393957228544*t^8 + 21112717049856*t^7 +
261470092787712*t^6 + 408367637987328*t^5 + 310261995012096*t^4 +
107150844100608*t^3 - 1855425871872*t^2 - 11132555231232*t -
1855425871872, 54*t^36 + 3888*t^35 + 238464*t^34 + 10046592*t^33 +
254399616*t^32 + 3873153024*t^31 + 33095761920*t^30 + 75687395328*t^29
- 1707510398976*t^28 - 22892480299008*t^27 - 133631346475008*t^26 -
274915815063552*t^25 + 1573892627890176*t^24 + 14008766114562048*t^23
+ 39431776984104960*t^22 - 36599920563585024*t^21 -
631426731554635776*t^20 - 1815998115581263872*t^19 +
14527984924650110976*t^17 + 40411310819496689664*t^16 +
18739159328555532288*t^15 - 161512558526893916160*t^14 -
459039248041969188864*t^13 - 412586509045642297344*t^12 +
576540251392158203904*t^11 + 2241961964182047817728*t^10 +
3072576694017614413824*t^9 + 1833425330295457972224*t^8 -
650149775306366386176*t^7 - 2274323441321634693120*t^6 -
2129288393021888397312*t^5 - 1118861343574997336064*t^4 -
353483031152682860544*t^3 - 67121648846329872384*t^2 -
8754997675608244224*t - 972777519512027136 ], [ 0, 0, 0, -27*t^16 -
864*t^15 + 13824*t^14 + 628992*t^13 + 7402752*t^12 + 36771840*t^11 +
30965760*t^10 - 514473984*t^9 - 2477924352*t^8 - 4115791872*t^7 +
1981808640*t^6 + 18827182080*t^5 + 30321672192*t^4 + 20610809856*t^3 +
3623878656*t^2 - 1811939328*t - 452984832, -54*t^24 - 2592*t^23 -
165888*t^22 - 5550336*t^21 - 88687872*t^20 - 635185152*t^19 +
371589120*t^18 + 45072433152*t^17 + 383285551104*t^16 +
1506238267392*t^15 + 1640258076672*t^14 - 11447323066368*t^13 -
57189844647936*t^12 - 91578584530944*t^11 + 104976516907008*t^10 +
771193992904704*t^9 + 1569937617321984*t^8 + 1476933489524736*t^7 +
97409858273280*t^6 - 1332079811887104*t^5 - 1487935585124352*t^4 -
744953487556608*t^3 - 178120883699712*t^2 - 22265110462464*t -
3710851743744 ], [ 0, 0, 0, -108*t^16 - 3456*t^15 + 55296*t^14 +
2515968*t^13 + 29611008*t^12 + 147087360*t^11 + 123863040*t^10 -
2057895936*t^9 - 9911697408*t^8 - 16463167488*t^7 + 7927234560*t^6 +
75308728320*t^5 + 121286688768*t^4 + 82443239424*t^3 + 14495514624*t^2
- 7247757312*t - 1811939328, -432*t^24 - 20736*t^23 - 1327104*t^22 -
44402688*t^21 - 709502976*t^20 - 5081481216*t^19 + 2972712960*t^18 +
360579465216*t^17 + 3066284408832*t^16 + 12049906139136*t^15 +
13122064613376*t^14 - 91578584530944*t^13 - 457518757183488*t^12 -
732628676247552*t^11 + 839812135256064*t^10 + 6169551943237632*t^9 +
12559500938575872*t^8 + 11815467916197888*t^7 + 779278866186240*t^6 -
10656638495096832*t^5 - 11903484680994816*t^4 - 5959627900452864*t^3 -
1424967069597696*t^2 - 178120883699712*t - 29686813949952 ], [ 0, 0,
0, -27*t^16 - 864*t^15 + 13824*t^14 + 628992*t^13 + 7402752*t^12 +
36771840*t^11 + 30965760*t^10 - 514473984*t^9 - 2477924352*t^8 -
4115791872*t^7 + 1981808640*t^6 + 18827182080*t^5 + 30321672192*t^4 +
20610809856*t^3 + 3623878656*t^2 - 1811939328*t - 452984832, 54*t^24 +
2592*t^23 + 165888*t^22 + 5550336*t^21 + 88687872*t^20 +
635185152*t^19 - 371589120*t^18 - 45072433152*t^17 - 383285551104*t^16
- 1506238267392*t^15 - 1640258076672*t^14 + 11447323066368*t^13 +
57189844647936*t^12 + 91578584530944*t^11 - 104976516907008*t^10 -
771193992904704*t^9 - 1569937617321984*t^8 - 1476933489524736*t^7 -
97409858273280*t^6 + 1332079811887104*t^5 + 1487935585124352*t^4 +
744953487556608*t^3 + 178120883699712*t^2 + 22265110462464*t +
3710851743744 ], [ 0, 0, 0, -27*t^16 + 27*t^8 - 27, -54*t^24 + 81*t^16
+ 81*t^8 - 54 ], [ 0, 0, 0, -27*t^24 - 54*t^20 + 54*t^12 - 54*t^4 -
27, -54*t^36 - 162*t^32 - 81*t^28 + 189*t^24 + 324*t^20 + 324*t^16 +
189*t^12 - 81*t^8 - 162*t^4 - 54 ], [ 0, 0, 0, -27*t^16 + 27*t^8 - 27,
54*t^24 - 81*t^16 - 81*t^8 + 54 ], [ 0, 0, 0, -27*t^24 - 54*t^20 +
54*t^12 - 54*t^4 - 27, 54*t^36 + 162*t^32 + 81*t^28 - 189*t^24 -
324*t^20 - 324*t^16 - 189*t^12 + 81*t^8 + 162*t^4 + 54 ], [ 0, 0, 0,
-27*t^32 + 81*t^24 - 108*t^16 + 81*t^8 - 27, 54*t^48 - 243*t^40 +
324*t^32 - 324*t^16 + 243*t^8 - 54 ], [ 0, 0, 0, -27*t^24 + 54*t^20 -
54*t^12 + 54*t^4 - 27, 54*t^36 - 162*t^32 + 81*t^28 + 189*t^24 -
324*t^20 + 324*t^16 - 189*t^12 - 81*t^8 + 162*t^4 - 54 ], [ 0, 0, 0,
-108*t^24 - 216*t^20 + 216*t^12 - 216*t^4 - 108, 432*t^36 + 1296*t^32
+ 648*t^28 - 1512*t^24 - 2592*t^20 - 2592*t^16 - 1512*t^12 + 648*t^8 +
1296*t^4 + 432 ], [ 0, 0, 0, -108*t^16 + 108*t^8 - 108, 432*t^24 -
648*t^16 - 648*t^8 + 432 ], [ 0, 0, 0, -108*t^24 - 216*t^20 + 216*t^12
- 216*t^4 - 108, -432*t^36 - 1296*t^32 - 648*t^28 + 1512*t^24 +
2592*t^20 + 2592*t^16 + 1512*t^12 - 648*t^8 - 1296*t^4 - 432 ], [ 0,
0, 0, -108*t^16 + 108*t^8 - 108, -432*t^24 + 648*t^16 + 648*t^8 - 432
], [ 0, 0, 0, -108*t^24 + 216*t^20 - 216*t^12 + 216*t^4 - 108,
432*t^36 - 1296*t^32 + 648*t^28 + 1512*t^24 - 2592*t^20 + 2592*t^16 -
1512*t^12 - 648*t^8 + 1296*t^4 - 432 ], [ 0, 0, 0, -108*t^32 +
324*t^24 - 432*t^16 + 324*t^8 - 108, 432*t^48 - 1944*t^40 + 2592*t^32
- 2592*t^16 + 1944*t^8 - 432 ], [ 0, 0, 0, -108*t^16 - 108*t^8 - 108,
432*t^24 + 648*t^16 - 648*t^8 - 432 ], [ 0, 0, 0, -27*t^16 - 27*t^8 -
27, 54*t^24 + 81*t^16 - 81*t^8 - 54 ], [ 0, 0, 0, -27*t^32 - 81*t^24 -
108*t^16 - 81*t^8 - 27, 54*t^48 + 243*t^40 + 324*t^32 - 324*t^16 -
243*t^8 - 54 ], [ 0, 0, 0, -108*t^32 - 324*t^24 - 432*t^16 - 324*t^8 -
108, 432*t^48 + 1944*t^40 + 2592*t^32 - 2592*t^16 - 1944*t^8 - 432 ],
[ 0, 0, 0, -108*t^16 - 1728*t^8 - 27648, 432*t^24 + 10368*t^16 -
165888*t^8 - 1769472 ], [ 0, 0, 0, -27*t^16 - 432*t^8 - 6912, 54*t^24
+ 1296*t^16 - 20736*t^8 - 221184 ], [ 0, 0, 0, -27*t^32 - 1296*t^24 -
27648*t^16 - 331776*t^8 - 1769472, 54*t^48 + 3888*t^40 + 82944*t^32 -
21233664*t^16 - 254803968*t^8 - 905969664 ], [ 0, 0, 0, -108*t^32 -
5184*t^24 - 110592*t^16 - 1327104*t^8 - 7077888, 432*t^48 + 31104*t^40
+ 663552*t^32 - 169869312*t^16 - 2038431744*t^8 - 7247757312 ], [ 0,
0, 0, -108*t^26 - 52704*t^25 - 1346112*t^24 - 12624768*t^23 -
59280768*t^22 - 161782272*t^21 - 280433664*t^20 - 210511872*t^19 +
1026763776*t^18 + 4221517824*t^17 + 4718297088*t^16 + 13078167552*t^15
- 3409772544*t^14 - 52312670208*t^13 + 75492753408*t^12 -
270177140736*t^11 + 262851526656*t^10 + 215564156928*t^9 -
1148656287744*t^8 + 2650640744448*t^7 - 3885024411648*t^6 +
3309507182592*t^5 - 1411500736512*t^4 + 221056598016*t^3 -
1811939328*t^2, 432*t^39 - 430272*t^38 - 33965568*t^37 -
771828480*t^36 - 8831607552*t^35 - 59945287680*t^34 -
263218249728*t^33 - 804177248256*t^32 - 1708217745408*t^31 -
1631653134336*t^30 + 4031911821312*t^29 + 22897894883328*t^28 +
64224805257216*t^27 + 95590557941760*t^26 + 44811864244224*t^25 -
112684506808320*t^24 - 874052862345216*t^23 - 1037364029816832*t^22 -
4149456119267328*t^20 + 13984845797523456*t^19 - 7211808435732480*t^18
- 11471837246521344*t^17 + 97884731332362240*t^16 -
263064802333556736*t^15 + 375159109768445952*t^14 -
264235373121503232*t^13 - 427728079247376384*t^12 +
1791196130608939008*t^11 - 3372963849069133824*t^10 +
4416069430828597248*t^9 - 4022860158357995520*t^8 +
2370716600434163712*t^7 - 828744519930347520*t^6 +
145881003750064128*t^5 - 7392016673538048*t^4 - 29686813949952*t^3 ],
[ 0, 0, 0, -27*t^26 - 12744*t^25 - 129168*t^24 + 564192*t^23 +
3427488*t^22 + 6023808*t^21 - 26977536*t^20 - 71788032*t^19 -
486487296*t^18 - 803229696*t^17 + 436396032*t^16 - 1836269568*t^15 +
13586448384*t^14 + 7345078272*t^13 + 6982336512*t^12 +
51406700544*t^11 - 124540747776*t^10 + 73510944768*t^9 -
110499987456*t^8 - 98694070272*t^7 + 224623853568*t^6 -
147899547648*t^5 - 135442464768*t^4 + 53452210176*t^3 - 452984832*t^2,
-54*t^39 + 55080*t^38 + 2939328*t^37 + 10208160*t^36 - 125776800*t^35
- 909066240*t^34 + 1368299520*t^33 + 9254559744*t^32 +
44472748032*t^31 + 86997639168*t^30 - 280366645248*t^29 -
1243674279936*t^28 - 4936181907456*t^27 - 11964277850112*t^26 +
4176364437504*t^25 - 6929648713728*t^24 + 154047068504064*t^23 +
271474517606400*t^22 + 1085898070425600*t^20 - 2464753096065024*t^19 -
443497517678592*t^18 - 1069149296001024*t^17 - 12251420518514688*t^16
+ 20218601092939776*t^15 - 20376359402471424*t^14 +
18374108462972928*t^13 + 22805909122056192*t^12 -
46633056240402432*t^11 + 38816436952498176*t^10 -
22956256599736320*t^9 - 61006402667151360*t^8 + 33762952662220800*t^7
+ 10960928338083840*t^6 - 12624317632217088*t^5 + 946267194654720*t^4
+ 3710851743744*t^3 ], [ 0, 0, 0, -108*t^22 - 51840*t^21 - 928800*t^20
- 3939840*t^19 - 464832*t^18 + 39813120*t^17 + 69838848*t^16 -
26542080*t^15 - 105781248*t^14 - 982056960*t^13 - 853327872*t^12 +
3928227840*t^11 - 1692499968*t^10 + 1698693120*t^9 + 17878745088*t^8 -
40768634880*t^7 - 1903951872*t^6 + 64550338560*t^5 - 60869836800*t^4 +
13589544960*t^3 - 113246208*t^2, 432*t^33 - 435456*t^32 -
28766016*t^31 - 400619520*t^30 - 2187171072*t^29 - 1860268032*t^28 +
23140076544*t^27 + 80374726656*t^26 + 51186069504*t^25 -
366646984704*t^24 - 1194755899392*t^23 - 568085446656*t^22 +
1483392614400*t^21 + 6344363999232*t^20 + 27701586690048*t^19 -
19691250647040*t^18 - 110806346760192*t^17 + 101509823987712*t^16 -
94937127321600*t^15 - 145429874343936*t^14 + 1223430040977408*t^13 -
1501786049347584*t^12 - 838632562753536*t^11 + 5267438086127616*t^10 -
6066032225550336*t^9 - 1950632411922432*t^8 + 9173660375973888*t^7 -
6721280220856320*t^6 + 1930454655565824*t^5 - 116891829927936*t^4 -
463856467968*t^3 ], [ 0, 0, 0, -108*t^24 - 51840*t^23 - 924480*t^22 -
1866240*t^21 + 36657792*t^20 + 183306240*t^19 - 164422656*t^18 -
2796871680*t^17 - 4929278976*t^16 + 2202992640*t^15 + 10001940480*t^14
+ 65718190080*t^13 + 80080994304*t^12 - 262872760320*t^11 +
160031047680*t^10 - 140991528960*t^9 - 1261895417856*t^8 +
2863996600320*t^7 - 673475198976*t^6 - 3003289436160*t^5 +
2402405056512*t^4 + 489223618560*t^3 - 969387540480*t^2 +
217432719360*t - 1811939328, 432*t^36 - 435456*t^35 - 28791936*t^34 -
374492160*t^33 - 460774656*t^32 + 21737963520*t^31 + 125373726720*t^30
- 211359891456*t^29 - 3510605463552*t^28 - 6612354072576*t^27 +
22499200991232*t^26 + 119390692442112*t^25 + 190112078757888*t^24 -
209126789480448*t^23 - 1352069013307392*t^22 - 1919012258709504*t^21 -
3093024429047808*t^20 + 4692168186789888*t^19 + 38308310729883648*t^18
- 18768672747159552*t^17 - 49488390864764928*t^16 +
122816784557408256*t^15 - 346129667406692352*t^14 +
214145832427978752*t^13 + 778699074592309248*t^12 -
1956097104971563008*t^11 + 1474507636161380352*t^10 +
1733388946001362944*t^9 - 3681136634549501952*t^8 +
886507638173466624*t^7 + 2103422093906411520*t^6 -
1458810037500641280*t^5 - 123688254896603136*t^4 +
402107894952099840*t^3 - 123660423508525056*t^2 + 7481077115387904*t +
29686813949952 ], [ 0, 0, 0, -27*t^26 - 13176*t^25 - 336528*t^24 -
3156192*t^23 - 14820192*t^22 - 40445568*t^21 - 70108416*t^20 -
52627968*t^19 + 256690944*t^18 + 1055379456*t^17 + 1179574272*t^16 +
3269541888*t^15 - 852443136*t^14 - 13078167552*t^13 + 18873188352*t^12
- 67544285184*t^11 + 65712881664*t^10 + 53891039232*t^9 -
287164071936*t^8 + 662660186112*t^7 - 971256102912*t^6 +
827376795648*t^5 - 352875184128*t^4 + 55264149504*t^3 - 452984832*t^2,
54*t^39 - 53784*t^38 - 4245696*t^37 - 96478560*t^36 - 1103950944*t^35
- 7493160960*t^34 - 32902281216*t^33 - 100522156032*t^32 -
213527218176*t^31 - 203956641792*t^30 + 503988977664*t^29 +
2862236860416*t^28 + 8028100657152*t^27 + 11948819742720*t^26 +
5601483030528*t^25 - 14085563351040*t^24 - 109256607793152*t^23 -
129670503727104*t^22 - 518682014908416*t^20 + 1748105724690432*t^19 -
901476054466560*t^18 - 1433979655815168*t^17 + 12235591416545280*t^16
- 32883100291694592*t^15 + 46894888721055744*t^14 -
33029421640187904*t^13 - 53466009905922048*t^12 +
223899516326117376*t^11 - 421620481133641728*t^10 +
552008678853574656*t^9 - 502857519794749440*t^8 +
296339575054270464*t^7 - 103593064991293440*t^6 +
18235125468758016*t^5 - 924002084192256*t^4 - 3710851743744*t^3 ], [
0, 0, 0, -108*t^26 - 50976*t^25 - 516672*t^24 + 2256768*t^23 +
13709952*t^22 + 24095232*t^21 - 107910144*t^20 - 287152128*t^19 -
1945949184*t^18 - 3212918784*t^17 + 1745584128*t^16 - 7345078272*t^15
+ 54345793536*t^14 + 29380313088*t^13 + 27929346048*t^12 +
205626802176*t^11 - 498162991104*t^10 + 294043779072*t^9 -
441999949824*t^8 - 394776281088*t^7 + 898495414272*t^6 -
591598190592*t^5 - 541769859072*t^4 + 213808840704*t^3 -
1811939328*t^2, -432*t^39 + 440640*t^38 + 23514624*t^37 +
81665280*t^36 - 1006214400*t^35 - 7272529920*t^34 + 10946396160*t^33 +
74036477952*t^32 + 355781984256*t^31 + 695981113344*t^30 -
2242933161984*t^29 - 9949394239488*t^28 - 39489455259648*t^27 -
95714222800896*t^26 + 33410915500032*t^25 - 55437189709824*t^24 +
1232376548032512*t^23 + 2171796140851200*t^22 + 8687184563404800*t^20
- 19718024768520192*t^19 - 3547980141428736*t^18 -
8553194368008192*t^17 - 98011364148117504*t^16 +
161748808743518208*t^15 - 163010875219771392*t^14 +
146992867703783424*t^13 + 182447272976449536*t^12 -
373064449923219456*t^11 + 310531495619985408*t^10 -
183650052797890560*t^9 - 488051221337210880*t^8 +
270103621297766400*t^7 + 87687426704670720*t^6 -
100994541057736704*t^5 + 7570137557237760*t^4 + 29686813949952*t^3 ],
[ 0, 0, 0, -27*t^22 - 12960*t^21 - 232200*t^20 - 984960*t^19 -
116208*t^18 + 9953280*t^17 + 17459712*t^16 - 6635520*t^15 -
26445312*t^14 - 245514240*t^13 - 213331968*t^12 + 982056960*t^11 -
423124992*t^10 + 424673280*t^9 + 4469686272*t^8 - 10192158720*t^7 -
475987968*t^6 + 16137584640*t^5 - 15217459200*t^4 + 3397386240*t^3 -
28311552*t^2, 54*t^33 - 54432*t^32 - 3595752*t^31 - 50077440*t^30 -
273396384*t^29 - 232533504*t^28 + 2892509568*t^27 + 10046840832*t^26 +
6398258688*t^25 - 45830873088*t^24 - 149344487424*t^23 -
71010680832*t^22 + 185424076800*t^21 + 793045499904*t^20 +
3462698336256*t^19 - 2461406330880*t^18 - 13850793345024*t^17 +
12688727998464*t^16 - 11867140915200*t^15 - 18178734292992*t^14 +
152928755122176*t^13 - 187723256168448*t^12 - 104829070344192*t^11 +
658429760765952*t^10 - 758254028193792*t^9 - 243829051490304*t^8 +
1146707546996736*t^7 - 840160027607040*t^6 + 241306831945728*t^5 -
14611478740992*t^4 - 57982058496*t^3 ], [ 0, 0, 0, -27*t^24 -
12960*t^23 - 231120*t^22 - 466560*t^21 + 9164448*t^20 + 45826560*t^19
- 41105664*t^18 - 699217920*t^17 - 1232319744*t^16 + 550748160*t^15 +
2500485120*t^14 + 16429547520*t^13 + 20020248576*t^12 -
65718190080*t^11 + 40007761920*t^10 - 35247882240*t^9 -
315473854464*t^8 + 715999150080*t^7 - 168368799744*t^6 -
750822359040*t^5 + 600601264128*t^4 + 122305904640*t^3 -
242346885120*t^2 + 54358179840*t - 452984832, 54*t^36 - 54432*t^35 -
3598992*t^34 - 46811520*t^33 - 57596832*t^32 + 2717245440*t^31 +
15671715840*t^30 - 26419986432*t^29 - 438825682944*t^28 -
826544259072*t^27 + 2812400123904*t^26 + 14923836555264*t^25 +
23764009844736*t^24 - 26140848685056*t^23 - 169008626663424*t^22 -
239876532338688*t^21 - 386628053630976*t^20 + 586521023348736*t^19 +
4788538841235456*t^18 - 2346084093394944*t^17 - 6186048858095616*t^16
+ 15352098069676032*t^15 - 43266208425836544*t^14 +
26768229053497344*t^13 + 97337384324038656*t^12 -
244512138121445376*t^11 + 184313454520172544*t^10 +
216673618250170368*t^9 - 460142079318687744*t^8 +
110813454771683328*t^7 + 262927761738301440*t^6 -
182351254687580160*t^5 - 15461031862075392*t^4 + 50263486869012480*t^3
- 15457552938565632*t^2 + 935134639423488*t + 3710851743744 ], [ 0, 0,
0, -108*t^26 - 50976*t^25 - 516672*t^24 + 2256768*t^23 + 13709952*t^22
+ 24095232*t^21 - 107910144*t^20 - 287152128*t^19 - 1945949184*t^18 -
3212918784*t^17 + 1745584128*t^16 - 7345078272*t^15 + 54345793536*t^14
+ 29380313088*t^13 + 27929346048*t^12 + 205626802176*t^11 -
498162991104*t^10 + 294043779072*t^9 - 441999949824*t^8 -
394776281088*t^7 + 898495414272*t^6 - 591598190592*t^5 -
541769859072*t^4 + 213808840704*t^3 - 1811939328*t^2, 432*t^39 -
440640*t^38 - 23514624*t^37 - 81665280*t^36 + 1006214400*t^35 +
7272529920*t^34 - 10946396160*t^33 - 74036477952*t^32 -
355781984256*t^31 - 695981113344*t^30 + 2242933161984*t^29 +
9949394239488*t^28 + 39489455259648*t^27 + 95714222800896*t^26 -
33410915500032*t^25 + 55437189709824*t^24 - 1232376548032512*t^23 -
2171796140851200*t^22 - 8687184563404800*t^20 + 19718024768520192*t^19
+ 3547980141428736*t^18 + 8553194368008192*t^17 +
98011364148117504*t^16 - 161748808743518208*t^15 +
163010875219771392*t^14 - 146992867703783424*t^13 -
182447272976449536*t^12 + 373064449923219456*t^11 -
310531495619985408*t^10 + 183650052797890560*t^9 +
488051221337210880*t^8 - 270103621297766400*t^7 -
87687426704670720*t^6 + 100994541057736704*t^5 - 7570137557237760*t^4
- 29686813949952*t^3 ], [ 0, 0, 0, -108*t^16 - 51840*t^15 -
929664*t^14 - 4354560*t^13 - 7900416*t^12 + 5806080*t^11 +
21510144*t^10 + 89579520*t^9 + 192706560*t^8 - 358318080*t^7 +
344162304*t^6 - 371589120*t^5 - 2022506496*t^4 + 4459069440*t^3 -
3807903744*t^2 + 849346560*t - 7077888, 432*t^24 - 435456*t^23 -
28760832*t^22 - 405844992*t^21 - 2532321792*t^20 - 6709506048*t^19 -
5867237376*t^18 + 19313344512*t^17 + 100489973760*t^16 +
161195360256*t^15 + 130682585088*t^14 + 9809952768*t^13 -
2147438297088*t^12 - 39239811072*t^11 + 2090921361408*t^10 -
10316503056384*t^9 + 25725433282560*t^8 - 19776864780288*t^7 -
24032204292096*t^6 + 109928547090432*t^5 - 165958240960512*t^4 +
106389829582848*t^3 - 30157918175232*t^2 + 1826434842624*t +
7247757312 ], [ 0, 0, 0, -108*t^30 - 51840*t^29 - 923616*t^28 -
1451520*t^27 + 44051904*t^26 + 197406720*t^25 - 472476672*t^24 -
4293181440*t^23 - 3027373056*t^22 + 27510865920*t^21 +
46805409792*t^20 + 3344302080*t^19 - 78802993152*t^18 -
753370398720*t^17 - 320585859072*t^16 + 3013481594880*t^15 -
1260847890432*t^14 - 214035333120*t^13 + 11982184906752*t^12 -
28171126702080*t^11 - 12400120037376*t^10 + 70339484712960*t^9 -
30964231176192*t^8 - 51748987207680*t^7 + 46191769288704*t^6 +
6088116142080*t^5 - 15495705133056*t^4 + 3478923509760*t^3 -
28991029248*t^2, 432*t^45 - 435456*t^44 - 28797120*t^43 -
369266688*t^42 - 115250688*t^41 + 26210967552*t^40 + 129520982016*t^39
- 490163208192*t^38 - 5035364683776*t^37 - 3008645627904*t^36 +
70673895014400*t^35 + 187202436857856*t^34 - 256411313897472*t^33 -
1945681061216256*t^32 - 2328773561155584*t^31 + 6744453112922112*t^30
+ 20817234647580672*t^29 + 10041225079947264*t^28 -
1641881800802304*t^27 - 153803164879945728*t^26 -
571120875365990400*t^25 + 696081715046645760*t^24 +
2284483501463961600*t^23 - 2460850638079131648*t^22 +
105080435251347456*t^21 + 2570553620466499584*t^20 -
21316848279122608128*t^19 + 27625279950528970752*t^18 +
38154626025973088256*t^17 - 127512154027868553216*t^16 +
67216687470338899968*t^15 + 196295982430663213056*t^14 -
296427800554477977600*t^13 - 50476697566801035264*t^12 +
337917603753926590464*t^11 - 131577184305442455552*t^10 -
139072095476131037184*t^9 + 112575248432357179392*t^8 +
1979991743205998592*t^7 - 25375813575395770368*t^6 +
7915692071615201280*t^5 - 478788935384825856*t^4 -
1899956092796928*t^3 ], [ 0, 0, 0, -27*t^26 - 12744*t^25 - 129168*t^24
+ 564192*t^23 + 3427488*t^22 + 6023808*t^21 - 26977536*t^20 -
71788032*t^19 - 486487296*t^18 - 803229696*t^17 + 436396032*t^16 -
1836269568*t^15 + 13586448384*t^14 + 7345078272*t^13 + 6982336512*t^12
+ 51406700544*t^11 - 124540747776*t^10 + 73510944768*t^9 -
110499987456*t^8 - 98694070272*t^7 + 224623853568*t^6 -
147899547648*t^5 - 135442464768*t^4 + 53452210176*t^3 - 452984832*t^2,
54*t^39 - 55080*t^38 - 2939328*t^37 - 10208160*t^36 + 125776800*t^35 +
909066240*t^34 - 1368299520*t^33 - 9254559744*t^32 - 44472748032*t^31
- 86997639168*t^30 + 280366645248*t^29 + 1243674279936*t^28 +
4936181907456*t^27 + 11964277850112*t^26 - 4176364437504*t^25 +
6929648713728*t^24 - 154047068504064*t^23 - 271474517606400*t^22 -
1085898070425600*t^20 + 2464753096065024*t^19 + 443497517678592*t^18 +
1069149296001024*t^17 + 12251420518514688*t^16 -
20218601092939776*t^15 + 20376359402471424*t^14 -
18374108462972928*t^13 - 22805909122056192*t^12 +
46633056240402432*t^11 - 38816436952498176*t^10 +
22956256599736320*t^9 + 61006402667151360*t^8 - 33762952662220800*t^7
- 10960928338083840*t^6 + 12624317632217088*t^5 - 946267194654720*t^4
- 3710851743744*t^3 ], [ 0, 0, 0, -27*t^16 - 12960*t^15 - 232416*t^14
- 1088640*t^13 - 1975104*t^12 + 1451520*t^11 + 5377536*t^10 +
22394880*t^9 + 48176640*t^8 - 89579520*t^7 + 86040576*t^6 -
92897280*t^5 - 505626624*t^4 + 1114767360*t^3 - 951975936*t^2 +
212336640*t - 1769472, 54*t^24 - 54432*t^23 - 3595104*t^22 -
50730624*t^21 - 316540224*t^20 - 838688256*t^19 - 733404672*t^18 +
2414168064*t^17 + 12561246720*t^16 + 20149420032*t^15 +
16335323136*t^14 + 1226244096*t^13 - 268429787136*t^12 -
4904976384*t^11 + 261365170176*t^10 - 1289562882048*t^9 +
3215679160320*t^8 - 2472108097536*t^7 - 3004025536512*t^6 +
13741068386304*t^5 - 20744780120064*t^4 + 13298728697856*t^3 -
3769739771904*t^2 + 228304355328*t + 905969664 ], [ 0, 0, 0, -27*t^30
- 12960*t^29 - 230904*t^28 - 362880*t^27 + 11012976*t^26 +
49351680*t^25 - 118119168*t^24 - 1073295360*t^23 - 756843264*t^22 +
6877716480*t^21 + 11701352448*t^20 + 836075520*t^19 - 19700748288*t^18
- 188342599680*t^17 - 80146464768*t^16 + 753370398720*t^15 -
315211972608*t^14 - 53508833280*t^13 + 2995546226688*t^12 -
7042781675520*t^11 - 3100030009344*t^10 + 17584871178240*t^9 -
7741057794048*t^8 - 12937246801920*t^7 + 11547942322176*t^6 +
1522029035520*t^5 - 3873926283264*t^4 + 869730877440*t^3 -
7247757312*t^2, 54*t^45 - 54432*t^44 - 3599640*t^43 - 46158336*t^42 -
14406336*t^41 + 3276370944*t^40 + 16190122752*t^39 - 61270401024*t^38
- 629420585472*t^37 - 376080703488*t^36 + 8834236876800*t^35 +
23400304607232*t^34 - 32051414237184*t^33 - 243210132652032*t^32 -
291096695144448*t^31 + 843056639115264*t^30 + 2602154330947584*t^29 +
1255153134993408*t^28 - 205235225100288*t^27 - 19225395609993216*t^26
- 71390109420748800*t^25 + 87010214380830720*t^24 +
285560437682995200*t^23 - 307606329759891456*t^22 +
13135054406418432*t^21 + 321319202558312448*t^20 -
2664606034890326016*t^19 + 3453159993816121344*t^18 +
4769328253246636032*t^17 - 15939019253483569152*t^16 +
8402085933792362496*t^15 + 24536997803832901632*t^14 -
37053475069309747200*t^13 - 6309587195850129408*t^12 +
42239700469240823808*t^11 - 16447148038180306944*t^10 -
17384011934516379648*t^9 + 14071906054044647424*t^8 +
247498967900749824*t^7 - 3171976696924471296*t^6 +
989461508951900160*t^5 - 59848616923103232*t^4 - 237494511599616*t^3
], [ 0, 0, 0, -108*t^26 - 52704*t^25 - 1346112*t^24 - 12624768*t^23 -
59280768*t^22 - 161782272*t^21 - 280433664*t^20 - 210511872*t^19 +
1026763776*t^18 + 4221517824*t^17 + 4718297088*t^16 + 13078167552*t^15
- 3409772544*t^14 - 52312670208*t^13 + 75492753408*t^12 -
270177140736*t^11 + 262851526656*t^10 + 215564156928*t^9 -
1148656287744*t^8 + 2650640744448*t^7 - 3885024411648*t^6 +
3309507182592*t^5 - 1411500736512*t^4 + 221056598016*t^3 -
1811939328*t^2, -432*t^39 + 430272*t^38 + 33965568*t^37 +
771828480*t^36 + 8831607552*t^35 + 59945287680*t^34 +
263218249728*t^33 + 804177248256*t^32 + 1708217745408*t^31 +
1631653134336*t^30 - 4031911821312*t^29 - 22897894883328*t^28 -
64224805257216*t^27 - 95590557941760*t^26 - 44811864244224*t^25 +
112684506808320*t^24 + 874052862345216*t^23 + 1037364029816832*t^22 +
4149456119267328*t^20 - 13984845797523456*t^19 + 7211808435732480*t^18
+ 11471837246521344*t^17 - 97884731332362240*t^16 +
263064802333556736*t^15 - 375159109768445952*t^14 +
264235373121503232*t^13 + 427728079247376384*t^12 -
1791196130608939008*t^11 + 3372963849069133824*t^10 -
4416069430828597248*t^9 + 4022860158357995520*t^8 -
2370716600434163712*t^7 + 828744519930347520*t^6 -
145881003750064128*t^5 + 7392016673538048*t^4 + 29686813949952*t^3 ],
[ 0, 0, 0, -108*t^22 - 51840*t^21 - 928800*t^20 - 3939840*t^19 -
464832*t^18 + 39813120*t^17 + 69838848*t^16 - 26542080*t^15 -
105781248*t^14 - 982056960*t^13 - 853327872*t^12 + 3928227840*t^11 -
1692499968*t^10 + 1698693120*t^9 + 17878745088*t^8 - 40768634880*t^7 -
1903951872*t^6 + 64550338560*t^5 - 60869836800*t^4 + 13589544960*t^3 -
113246208*t^2, -432*t^33 + 435456*t^32 + 28766016*t^31 +
400619520*t^30 + 2187171072*t^29 + 1860268032*t^28 - 23140076544*t^27
- 80374726656*t^26 - 51186069504*t^25 + 366646984704*t^24 +
1194755899392*t^23 + 568085446656*t^22 - 1483392614400*t^21 -
6344363999232*t^20 - 27701586690048*t^19 + 19691250647040*t^18 +
110806346760192*t^17 - 101509823987712*t^16 + 94937127321600*t^15 +
145429874343936*t^14 - 1223430040977408*t^13 + 1501786049347584*t^12 +
838632562753536*t^11 - 5267438086127616*t^10 + 6066032225550336*t^9 +
1950632411922432*t^8 - 9173660375973888*t^7 + 6721280220856320*t^6 -
1930454655565824*t^5 + 116891829927936*t^4 + 463856467968*t^3 ], [ 0,
0, 0, -27*t^26 - 13176*t^25 - 336528*t^24 - 3156192*t^23 -
14820192*t^22 - 40445568*t^21 - 70108416*t^20 - 52627968*t^19 +
256690944*t^18 + 1055379456*t^17 + 1179574272*t^16 + 3269541888*t^15 -
852443136*t^14 - 13078167552*t^13 + 18873188352*t^12 -
67544285184*t^11 + 65712881664*t^10 + 53891039232*t^9 -
287164071936*t^8 + 662660186112*t^7 - 971256102912*t^6 +
827376795648*t^5 - 352875184128*t^4 + 55264149504*t^3 - 452984832*t^2,
-54*t^39 + 53784*t^38 + 4245696*t^37 + 96478560*t^36 + 1103950944*t^35
+ 7493160960*t^34 + 32902281216*t^33 + 100522156032*t^32 +
213527218176*t^31 + 203956641792*t^30 - 503988977664*t^29 -
2862236860416*t^28 - 8028100657152*t^27 - 11948819742720*t^26 -
5601483030528*t^25 + 14085563351040*t^24 + 109256607793152*t^23 +
129670503727104*t^22 + 518682014908416*t^20 - 1748105724690432*t^19 +
901476054466560*t^18 + 1433979655815168*t^17 - 12235591416545280*t^16
+ 32883100291694592*t^15 - 46894888721055744*t^14 +
33029421640187904*t^13 + 53466009905922048*t^12 -
223899516326117376*t^11 + 421620481133641728*t^10 -
552008678853574656*t^9 + 502857519794749440*t^8 -
296339575054270464*t^7 + 103593064991293440*t^6 -
18235125468758016*t^5 + 924002084192256*t^4 + 3710851743744*t^3 ], [
0, 0, 0, -27*t^22 - 12960*t^21 - 232200*t^20 - 984960*t^19 -
116208*t^18 + 9953280*t^17 + 17459712*t^16 - 6635520*t^15 -
26445312*t^14 - 245514240*t^13 - 213331968*t^12 + 982056960*t^11 -
423124992*t^10 + 424673280*t^9 + 4469686272*t^8 - 10192158720*t^7 -
475987968*t^6 + 16137584640*t^5 - 15217459200*t^4 + 3397386240*t^3 -
28311552*t^2, -54*t^33 + 54432*t^32 + 3595752*t^31 + 50077440*t^30 +
273396384*t^29 + 232533504*t^28 - 2892509568*t^27 - 10046840832*t^26 -
6398258688*t^25 + 45830873088*t^24 + 149344487424*t^23 +
71010680832*t^22 - 185424076800*t^21 - 793045499904*t^20 -
3462698336256*t^19 + 2461406330880*t^18 + 13850793345024*t^17 -
12688727998464*t^16 + 11867140915200*t^15 + 18178734292992*t^14 -
152928755122176*t^13 + 187723256168448*t^12 + 104829070344192*t^11 -
658429760765952*t^10 + 758254028193792*t^9 + 243829051490304*t^8 -
1146707546996736*t^7 + 840160027607040*t^6 - 241306831945728*t^5 +
14611478740992*t^4 + 57982058496*t^3 ], [ 0, 0, 0, -27*t^16 -
12960*t^15 - 232416*t^14 - 1088640*t^13 - 1975104*t^12 + 1451520*t^11
+ 5377536*t^10 + 22394880*t^9 + 48176640*t^8 - 89579520*t^7 +
86040576*t^6 - 92897280*t^5 - 505626624*t^4 + 1114767360*t^3 -
951975936*t^2 + 212336640*t - 1769472, -54*t^24 + 54432*t^23 +
3595104*t^22 + 50730624*t^21 + 316540224*t^20 + 838688256*t^19 +
733404672*t^18 - 2414168064*t^17 - 12561246720*t^16 - 20149420032*t^15
- 16335323136*t^14 - 1226244096*t^13 + 268429787136*t^12 +
4904976384*t^11 - 261365170176*t^10 + 1289562882048*t^9 -
3215679160320*t^8 + 2472108097536*t^7 + 3004025536512*t^6 -
13741068386304*t^5 + 20744780120064*t^4 - 13298728697856*t^3 +
3769739771904*t^2 - 228304355328*t - 905969664 ], [ 0, 0, 0, -108*t^16
- 51840*t^15 - 929664*t^14 - 4354560*t^13 - 7900416*t^12 +
5806080*t^11 + 21510144*t^10 + 89579520*t^9 + 192706560*t^8 -
358318080*t^7 + 344162304*t^6 - 371589120*t^5 - 2022506496*t^4 +
4459069440*t^3 - 3807903744*t^2 + 849346560*t - 7077888, -432*t^24 +
435456*t^23 + 28760832*t^22 + 405844992*t^21 + 2532321792*t^20 +
6709506048*t^19 + 5867237376*t^18 - 19313344512*t^17 -
100489973760*t^16 - 161195360256*t^15 - 130682585088*t^14 -
9809952768*t^13 + 2147438297088*t^12 + 39239811072*t^11 -
2090921361408*t^10 + 10316503056384*t^9 - 25725433282560*t^8 +
19776864780288*t^7 + 24032204292096*t^6 - 109928547090432*t^5 +
165958240960512*t^4 - 106389829582848*t^3 + 30157918175232*t^2 -
1826434842624*t - 7247757312 ], [ 0, 0, 0, -1855425871872*t^24 +
27831388078080*t^22 - 15597173735424*t^20 + 2174327193600*t^18 -
6794772480*t^16 - 40768634880*t^14 + 7842299904*t^12 - 637009920*t^10
- 1658880*t^8 + 8294400*t^6 - 929664*t^4 + 25920*t^2 - 27,
972777519512027136*t^36 + 30642491864628854784*t^34 -
63245738417024139264*t^32 + 27769758252319899648*t^30 -
4944635731504005120*t^28 + 104735079615430656*t^26 +
108371714324299776*t^24 - 24079716965154816*t^22 +
2644329759768576*t^20 - 41317652496384*t^16 + 5878837149696*t^14 -
413405282304*t^12 - 6242697216*t^10 + 4605050880*t^8 - 404103168*t^6 +
14380416*t^4 - 108864*t^2 - 54 ], [ 0, 0, 0, -7599824371187712*t^32 +
113997365567815680*t^30 - 64123518131896320*t^28 +
12468461858979840*t^26 - 2026125052084224*t^24 + 139156940390400*t^22
+ 15655155793920*t^20 - 5653250703360*t^18 + 990224842752*t^16 -
88332042240*t^14 + 3822059520*t^12 + 530841600*t^10 - 120766464*t^8 +
11612160*t^6 - 933120*t^4 + 25920*t^2 - 27,
255007790074960841539584*t^48 + 8032745387361266508496896*t^46 -
16567537361432612173774848*t^44 + 7656210447328707140911104*t^42 -
2073183449584032232243200*t^40 + 374573820553223120879616*t^38 -
44492898187441097146368*t^36 + 337067410510917402624*t^34 +
1012251007295976112128*t^32 - 248012668529339793408*t^30 +
37525082810785726464*t^28 - 2977468691924385792*t^26 +
46522948311318528*t^22 - 9161397170601984*t^20 + 946093248479232*t^18
- 60334861713408*t^16 - 313918488576*t^14 + 647456882688*t^12 -
85168226304*t^10 + 7365427200*t^8 - 425005056*t^6 + 14370048*t^4 -
108864*t^2 - 54 ], [ 0, 0, 0, -30399297484750848*t^32 +
455989462271262720*t^30 - 256494072527585280*t^28 +
49873847435919360*t^26 - 8104500208336896*t^24 + 556627761561600*t^22
+ 62620623175680*t^20 - 22613002813440*t^18 + 3960899371008*t^16 -
353328168960*t^14 + 15288238080*t^12 + 2123366400*t^10 - 483065856*t^8
+ 46448640*t^6 - 3732480*t^4 + 103680*t^2 - 108,
2040062320599686732316672*t^48 + 64261963098890132067975168*t^46 -
132540298891460897390198784*t^44 + 61249683578629657127288832*t^42 -
16585467596672257857945600*t^40 + 2996590564425784967036928*t^38 -
355943185499528777170944*t^36 + 2696539284087339220992*t^34 +
8098008058367808897024*t^32 - 1984101348234718347264*t^30 +
300200662486285811712*t^28 - 23819749535395086336*t^26 +
372183586490548224*t^22 - 73291177364815872*t^20 +
7568745987833856*t^18 - 482678893707264*t^16 - 2511347908608*t^14 +
5179655061504*t^12 - 681345810432*t^10 + 58923417600*t^8 -
3400040448*t^6 + 114960384*t^4 - 870912*t^2 - 432 ], [ 0, 0, 0,
-7421703487488*t^24 + 111325552312320*t^22 - 62388694941696*t^20 +
8697308774400*t^18 - 27179089920*t^16 - 163074539520*t^14 +
31369199616*t^12 - 2548039680*t^10 - 6635520*t^8 + 33177600*t^6 -
3718656*t^4 + 103680*t^2 - 108, 7782220156096217088*t^36 +
245139934917030838272*t^34 - 505965907336193114112*t^32 +
222158066018559197184*t^30 - 39557085852032040960*t^28 +
837880636923445248*t^26 + 866973714594398208*t^24 -
192637735721238528*t^22 + 21154638078148608*t^20 -
330541219971072*t^16 + 47030697197568*t^14 - 3307242258432*t^12 -
49941577728*t^10 + 36840407040*t^8 - 3232825344*t^6 + 115043328*t^4 -
870912*t^2 - 432 ], [ 0, 0, 0, -7421703487488*t^24 +
111325552312320*t^22 - 62852551409664*t^20 + 15655155793920*t^18 -
3940968038400*t^16 + 597939978240*t^14 - 92522151936*t^12 +
9342812160*t^10 - 962150400*t^8 + 59719680*t^6 - 3746304*t^4 +
103680*t^2 - 108, -7782220156096217088*t^36 -
245139934917030838272*t^34 + 505236324196559093760*t^32 -
245139934917030838272*t^30 + 86957190455129800704*t^28 -
22742474430779228160*t^26 + 5063857976327012352*t^24 -
897729253846548480*t^22 + 144714868589592576*t^20 -
18790361660915712*t^18 + 2261169821712384*t^16 - 219172181114880*t^14
+ 19317085175808*t^12 - 1355557109760*t^10 + 80985194496*t^8 -
3567255552*t^6 + 114877440*t^4 - 870912*t^2 - 432 ], [ 0, 0, 0,
-1855425871872*t^24 + 27831388078080*t^22 - 15713137852416*t^20 +
3913788948480*t^18 - 985242009600*t^16 + 149484994560*t^14 -
23130537984*t^12 + 2335703040*t^10 - 240537600*t^8 + 14929920*t^6 -
936576*t^4 + 25920*t^2 - 27, -972777519512027136*t^36 -
30642491864628854784*t^34 + 63154540524569886720*t^32 -
30642491864628854784*t^30 + 10869648806891225088*t^28 -
2842809303847403520*t^26 + 632982247040876544*t^24 -
112216156730818560*t^22 + 18089358573699072*t^20 -
2348795207614464*t^18 + 282646227714048*t^16 - 27396522639360*t^14 +
2414635646976*t^12 - 169444638720*t^10 + 10123149312*t^8 -
445906944*t^6 + 14359680*t^4 - 108864*t^2 - 54 ], [ 0, 0, 0,
-1855425871872*t^24 + 27831388078080*t^22 - 15713137852416*t^20 +
3913788948480*t^18 - 985242009600*t^16 + 149484994560*t^14 -
23130537984*t^12 + 2335703040*t^10 - 240537600*t^8 + 14929920*t^6 -
936576*t^4 + 25920*t^2 - 27, 972777519512027136*t^36 +
30642491864628854784*t^34 - 63154540524569886720*t^32 +
30642491864628854784*t^30 - 10869648806891225088*t^28 +
2842809303847403520*t^26 - 632982247040876544*t^24 +
112216156730818560*t^22 - 18089358573699072*t^20 +
2348795207614464*t^18 - 282646227714048*t^16 + 27396522639360*t^14 -
2414635646976*t^12 + 169444638720*t^10 - 10123149312*t^8 +
445906944*t^6 - 14359680*t^4 + 108864*t^2 + 54 ], [ 0, 0, 0,
-452984832*t^16 + 6794772480*t^14 - 3822059520*t^12 + 743178240*t^10 -
120987648*t^8 + 11612160*t^6 - 933120*t^4 + 25920*t^2 - 27,
3710851743744*t^24 + 116891829927936*t^22 - 241089399226368*t^20 +
111412525400064*t^18 - 30166071902208*t^16 + 5536380616704*t^14 -
824036032512*t^12 + 86505947136*t^10 - 7364763648*t^8 + 425005056*t^6
- 14370048*t^4 + 108864*t^2 + 54 ], [ 0, 0, 0, -452984832*t^16 +
6794772480*t^14 - 3822059520*t^12 + 743178240*t^10 - 120987648*t^8 +
11612160*t^6 - 933120*t^4 + 25920*t^2 - 27, -3710851743744*t^24 -
116891829927936*t^22 + 241089399226368*t^20 - 111412525400064*t^18 +
30166071902208*t^16 - 5536380616704*t^14 + 824036032512*t^12 -
86505947136*t^10 + 7364763648*t^8 - 425005056*t^6 + 14370048*t^4 -
108864*t^2 - 54 ], [ 0, 0, 0, -1811939328*t^16 + 27179089920*t^14 -
15288238080*t^12 + 2972712960*t^10 - 483950592*t^8 + 46448640*t^6 -
3732480*t^4 + 103680*t^2 - 108, -29686813949952*t^24 -
935134639423488*t^22 + 1928715193810944*t^20 - 891300203200512*t^18 +
241328575217664*t^16 - 44291044933632*t^14 + 6592288260096*t^12 -
692047577088*t^10 + 58918109184*t^8 - 3400040448*t^6 + 114960384*t^4 -
870912*t^2 - 432 ], [ 0, 0, 0, -1811939328*t^16 + 27179089920*t^14 -
15288238080*t^12 + 2972712960*t^10 - 483950592*t^8 + 46448640*t^6 -
3732480*t^4 + 103680*t^2 - 108, 29686813949952*t^24 +
935134639423488*t^22 - 1928715193810944*t^20 + 891300203200512*t^18 -
241328575217664*t^16 + 44291044933632*t^14 - 6592288260096*t^12 +
692047577088*t^10 - 58918109184*t^8 + 3400040448*t^6 - 114960384*t^4 +
870912*t^2 + 432 ], [ 0, 0, 0, -7421703487488*t^24 +
111325552312320*t^22 - 62852551409664*t^20 + 15655155793920*t^18 -
3940968038400*t^16 + 597939978240*t^14 - 92522151936*t^12 +
9342812160*t^10 - 962150400*t^8 + 59719680*t^6 - 3746304*t^4 +
103680*t^2 - 108, 7782220156096217088*t^36 +
245139934917030838272*t^34 - 505236324196559093760*t^32 +
245139934917030838272*t^30 - 86957190455129800704*t^28 +
22742474430779228160*t^26 - 5063857976327012352*t^24 +
897729253846548480*t^22 - 144714868589592576*t^20 +
18790361660915712*t^18 - 2261169821712384*t^16 + 219172181114880*t^14
- 19317085175808*t^12 + 1355557109760*t^10 - 80985194496*t^8 +
3567255552*t^6 - 114877440*t^4 + 870912*t^2 + 432 ], [ 0, 0, 0,
-27*t^30 - 594*t^28 - 5049*t^26 - 20628*t^24 - 40419*t^22 - 28782*t^20
+ 17847*t^18 + 44712*t^16 + 17847*t^14 - 28782*t^12 - 40419*t^10 -
20628*t^8 - 5049*t^6 - 594*t^4 - 27*t^2, 54*t^45 + 1782*t^43 +
24948*t^41 + 192564*t^39 + 890190*t^37 + 2492046*t^35 + 3956688*t^33 +
2432592*t^31 - 2664900*t^29 - 6870852*t^27 - 7533000*t^25 -
7533000*t^23 - 6870852*t^21 - 2664900*t^19 + 2432592*t^17 +
3956688*t^15 + 2492046*t^13 + 890190*t^11 + 192564*t^9 + 24948*t^7 +
1782*t^5 + 54*t^3 ], [ 0, 0, 0, -108*t^30 - 2376*t^28 - 20196*t^26 -
82512*t^24 - 161676*t^22 - 115128*t^20 + 71388*t^18 + 178848*t^16 +
71388*t^14 - 115128*t^12 - 161676*t^10 - 82512*t^8 - 20196*t^6 -
2376*t^4 - 108*t^2, 432*t^45 + 14256*t^43 + 199584*t^41 + 1540512*t^39
+ 7121520*t^37 + 19936368*t^35 + 31653504*t^33 + 19460736*t^31 -
21319200*t^29 - 54966816*t^27 - 60264000*t^25 - 60264000*t^23 -
54966816*t^21 - 21319200*t^19 + 19460736*t^17 + 31653504*t^15 +
19936368*t^13 + 7121520*t^11 + 1540512*t^9 + 199584*t^7 + 14256*t^5 +
432*t^3 ], [ 0, 0, 0, -27*t^22 - 270*t^20 - 783*t^18 - 648*t^16 +
378*t^14 + 972*t^12 + 378*t^10 - 648*t^8 - 783*t^6 - 270*t^4 - 27*t^2,
54*t^33 + 810*t^31 + 4374*t^29 + 10314*t^27 + 8910*t^25 - 6318*t^23 -
17874*t^21 - 14094*t^19 - 14094*t^17 - 17874*t^15 - 6318*t^13 +
8910*t^11 + 10314*t^9 + 4374*t^7 + 810*t^5 + 54*t^3 ], [ 0, 0, 0,
-108*t^22 - 1080*t^20 - 3132*t^18 - 2592*t^16 + 1512*t^14 + 3888*t^12
+ 1512*t^10 - 2592*t^8 - 3132*t^6 - 1080*t^4 - 108*t^2, 432*t^33 +
6480*t^31 + 34992*t^29 + 82512*t^27 + 71280*t^25 - 50544*t^23 -
142992*t^21 - 112752*t^19 - 112752*t^17 - 142992*t^15 - 50544*t^13 +
71280*t^11 + 82512*t^9 + 34992*t^7 + 6480*t^5 + 432*t^3 ], [ 0, 0, 0,
-108*t^24 - 2160*t^22 - 15768*t^20 - 48816*t^18 - 48276*t^16 +
30240*t^14 + 59184*t^12 + 30240*t^10 - 48276*t^8 - 48816*t^6 -
15768*t^4 - 2160*t^2 - 108, -432*t^36 - 12960*t^34 - 159408*t^32 -
1022976*t^30 - 3561408*t^28 - 6023808*t^26 - 1874880*t^24 +
7796736*t^22 + 9577440*t^20 + 4719168*t^18 + 9577440*t^16 +
7796736*t^14 - 1874880*t^12 - 6023808*t^10 - 3561408*t^8 - 1022976*t^6
- 159408*t^4 - 12960*t^2 - 432 ], [ 0, 0, 0, -27*t^24 - 540*t^22 -
3942*t^20 - 12204*t^18 - 12069*t^16 + 7560*t^14 + 14796*t^12 +
7560*t^10 - 12069*t^8 - 12204*t^6 - 3942*t^4 - 540*t^2 - 27, -54*t^36
- 1620*t^34 - 19926*t^32 - 127872*t^30 - 445176*t^28 - 752976*t^26 -
234360*t^24 + 974592*t^22 + 1197180*t^20 + 589896*t^18 + 1197180*t^16
+ 974592*t^14 - 234360*t^12 - 752976*t^10 - 445176*t^8 - 127872*t^6 -
19926*t^4 - 1620*t^2 - 54 ], [ 0, 0, 0, -108*t^16 - 864*t^14 -
1296*t^12 + 864*t^10 + 1080*t^8 + 864*t^6 - 1296*t^4 - 864*t^2 - 108,
-432*t^24 - 5184*t^22 - 18144*t^20 - 12096*t^18 + 24624*t^16 +
31104*t^14 - 12096*t^12 + 31104*t^10 + 24624*t^8 - 12096*t^6 -
18144*t^4 - 5184*t^2 - 432 ], [ 0, 0, 0, -27*t^16 - 216*t^14 -
324*t^12 + 216*t^10 + 270*t^8 + 216*t^6 - 324*t^4 - 216*t^2 - 27,
-54*t^24 - 648*t^22 - 2268*t^20 - 1512*t^18 + 3078*t^16 + 3888*t^14 -
1512*t^12 + 3888*t^10 + 3078*t^8 - 1512*t^6 - 2268*t^4 - 648*t^2 - 54
], [ 0, 0, 0, -27*t^16 - 216*t^14 - 324*t^12 + 216*t^10 + 270*t^8 +
216*t^6 - 324*t^4 - 216*t^2 - 27, 54*t^24 + 648*t^22 + 2268*t^20 +
1512*t^18 - 3078*t^16 - 3888*t^14 + 1512*t^12 - 3888*t^10 - 3078*t^8 +
1512*t^6 + 2268*t^4 + 648*t^2 + 54 ], [ 0, 0, 0, -108*t^24 - 2160*t^22
- 15768*t^20 - 48816*t^18 - 48276*t^16 + 30240*t^14 + 59184*t^12 +
30240*t^10 - 48276*t^8 - 48816*t^6 - 15768*t^4 - 2160*t^2 - 108,
432*t^36 + 12960*t^34 + 159408*t^32 + 1022976*t^30 + 3561408*t^28 +
6023808*t^26 + 1874880*t^24 - 7796736*t^22 - 9577440*t^20 -
4719168*t^18 - 9577440*t^16 - 7796736*t^14 + 1874880*t^12 +
6023808*t^10 + 3561408*t^8 + 1022976*t^6 + 159408*t^4 + 12960*t^2 +
432 ], [ 0, 0, 0, -27*t^24 - 540*t^22 - 3942*t^20 - 12204*t^18 -
12069*t^16 + 7560*t^14 + 14796*t^12 + 7560*t^10 - 12069*t^8 -
12204*t^6 - 3942*t^4 - 540*t^2 - 27, 54*t^36 + 1620*t^34 + 19926*t^32
+ 127872*t^30 + 445176*t^28 + 752976*t^26 + 234360*t^24 - 974592*t^22
- 1197180*t^20 - 589896*t^18 - 1197180*t^16 - 974592*t^14 +
234360*t^12 + 752976*t^10 + 445176*t^8 + 127872*t^6 + 19926*t^4 +
1620*t^2 + 54 ], [ 0, 0, 0, -108*t^16 - 864*t^14 - 1296*t^12 +
864*t^10 + 1080*t^8 + 864*t^6 - 1296*t^4 - 864*t^2 - 108, 432*t^24 +
5184*t^22 + 18144*t^20 + 12096*t^18 - 24624*t^16 - 31104*t^14 +
12096*t^12 - 31104*t^10 - 24624*t^8 + 12096*t^6 + 18144*t^4 + 5184*t^2
+ 432 ], [ 0, 0, 0, -324*t^32 - 10368*t^31 - 152064*t^30 -
1347840*t^29 - 7572096*t^28 - 21192192*t^27 + 51480576*t^26 +
856507392*t^25 + 4249352448*t^24 + 10000668672*t^23 - 3253837824*t^22
- 114492690432*t^21 - 457246218240*t^20 - 1087813214208*t^19 -
1827992862720*t^18 - 2458090782720*t^17 - 3254846920704*t^16 -
4916181565440*t^15 - 7311971450880*t^14 - 8702505713664*t^13 -
7315939491840*t^12 - 3663766093824*t^11 - 208245620736*t^10 +
1280085590016*t^9 + 1087834226688*t^8 + 438531784704*t^7 +
52716109824*t^6 - 43401609216*t^5 - 31015305216*t^4 - 11041505280*t^3
- 2491416576*t^2 - 339738624*t - 21233664, 31104*t^47 + 1461888*t^46 +
30820608*t^45 + 378224640*t^44 + 2872723968*t^43 + 12224494080*t^42 +
4443724800*t^41 - 316891201536*t^40 - 2293834291200*t^39 -
8274204297216*t^38 - 10502321836032*t^37 + 60453069520896*t^36 +
460703815852032*t^35 + 1758470671687680*t^34 + 4395063003070464*t^33 +
5348086599647232*t^32 - 14820988081471488*t^31 -
124390992103538688*t^30 - 482825896433418240*t^29 -
1314635428745183232*t^28 - 2668828023882252288*t^27 -
3948895346718670848*t^26 - 3643573881914523648*t^25 +
7287147763829047296*t^23 + 15795581386874683392*t^22 +
21350624191058018304*t^21 + 21034166859922931712*t^20 +
15450428685869383680*t^19 + 7961023494626476032*t^18 +
1897086474428350464*t^17 - 1369110169509691392*t^16 -
2250272257572077568*t^15 - 1800673967808184320*t^14 -
943521414864961536*t^13 - 247615772757590016*t^12 +
86035020480774144*t^11 + 135564563205586944*t^10 +
75164362054041600*t^9 + 20767781783863296*t^8 - 582447896985600*t^7 -
3204577776107520*t^6 - 1506134703734784*t^5 - 396597280112640*t^4 -
64635499708416*t^3 - 6131602685952*t^2 - 260919263232*t ], [ 0, 0, 0,
-81*t^16 - 1296*t^15 - 12096*t^14 - 78624*t^13 - 341712*t^12 -
955584*t^11 - 1693440*t^10 - 1994112*t^9 - 2261088*t^8 - 3988224*t^7 -
6773760*t^6 - 7644672*t^5 - 5467392*t^4 - 2515968*t^3 - 774144*t^2 -
165888*t - 20736, 3888*t^23 + 89424*t^22 + 959904*t^21 + 6386688*t^20
+ 29937600*t^19 + 108706752*t^18 + 330625152*t^17 + 868589568*t^16 +
1891524096*t^15 + 3090700800*t^14 + 3078964224*t^13 - 6157928448*t^11
- 12362803200*t^10 - 15132192768*t^9 - 13897433088*t^8 -
10580004864*t^7 - 6957232128*t^6 - 3832012800*t^5 - 1634992128*t^4 -
491470848*t^3 - 91570176*t^2 - 7962624*t ], [ 0, 0, 0, -81*t^32 -
2592*t^31 - 38016*t^30 - 336960*t^29 - 1893024*t^28 - 5298048*t^27 +
12870144*t^26 + 214126848*t^25 + 1062338112*t^24 + 2500167168*t^23 -
813459456*t^22 - 28623172608*t^21 - 114311554560*t^20 -
271953303552*t^19 - 456998215680*t^18 - 614522695680*t^17 -
813711730176*t^16 - 1229045391360*t^15 - 1827992862720*t^14 -
2175626428416*t^13 - 1828984872960*t^12 - 915941523456*t^11 -
52061405184*t^10 + 320021397504*t^9 + 271958556672*t^8 +
109632946176*t^7 + 13179027456*t^6 - 10850402304*t^5 - 7753826304*t^4
- 2760376320*t^3 - 622854144*t^2 - 84934656*t - 5308416, 3888*t^47 +
182736*t^46 + 3852576*t^45 + 47278080*t^44 + 359090496*t^43 +
1528061760*t^42 + 555465600*t^41 - 39611400192*t^40 -
286729286400*t^39 - 1034275537152*t^38 - 1312790229504*t^37 +
7556633690112*t^36 + 57587976981504*t^35 + 219808833960960*t^34 +
549382875383808*t^33 + 668510824955904*t^32 - 1852623510183936*t^31 -
15548874012942336*t^30 - 60353237054177280*t^29 -
164329428593147904*t^28 - 333603502985281536*t^27 -
493611918339833856*t^26 - 455446735239315456*t^25 +
910893470478630912*t^23 + 1974447673359335424*t^22 +
2668828023882252288*t^21 + 2629270857490366464*t^20 +
1931303585733672960*t^19 + 995127936828309504*t^18 +
237135809303543808*t^17 - 171138771188711424*t^16 -
281284032196509696*t^15 - 225084245976023040*t^14 -
117940176858120192*t^13 - 30951971594698752*t^12 +
10754377560096768*t^11 + 16945570400698368*t^10 + 9395545256755200*t^9
+ 2595972722982912*t^8 - 72805987123200*t^7 - 400572222013440*t^6 -
188266837966848*t^5 - 49574660014080*t^4 - 8079437463552*t^3 -
766450335744*t^2 - 32614907904*t ], [ 0, 0, 0, -324*t^16 - 5184*t^15 -
48384*t^14 - 314496*t^13 - 1366848*t^12 - 3822336*t^11 - 6773760*t^10
- 7976448*t^9 - 9044352*t^8 - 15952896*t^7 - 27095040*t^6 -
30578688*t^5 - 21869568*t^4 - 10063872*t^3 - 3096576*t^2 - 663552*t -
82944, 31104*t^23 + 715392*t^22 + 7679232*t^21 + 51093504*t^20 +
239500800*t^19 + 869654016*t^18 + 2645001216*t^17 + 6948716544*t^16 +
15132192768*t^15 + 24725606400*t^14 + 24631713792*t^13 -
49263427584*t^11 - 98902425600*t^10 - 121057542144*t^9 -
111179464704*t^8 - 84640038912*t^7 - 55657857024*t^6 - 30656102400*t^5
- 13079937024*t^4 - 3931766784*t^3 - 732561408*t^2 - 63700992*t ], [
0, 0, 0, -108*t^24 - 864*t^20 + 13824*t^12 - 221184*t^4 - 442368,
432*t^36 + 5184*t^32 + 10368*t^28 - 96768*t^24 - 663552*t^20 -
2654208*t^16 - 6193152*t^12 + 10616832*t^8 + 84934656*t^4 + 113246208
], [ 0, 0, 0, -108*t^16 + 1728*t^8 - 27648, 432*t^24 - 10368*t^16 -
165888*t^8 + 1769472 ], [ 0, 0, 0, -27*t^16 + 432*t^8 - 6912, 54*t^24
- 1296*t^16 - 20736*t^8 + 221184 ], [ 0, 0, 0, -27*t^24 - 216*t^20 +
3456*t^12 - 55296*t^4 - 110592, 54*t^36 + 648*t^32 + 1296*t^28 -
12096*t^24 - 82944*t^20 - 331776*t^16 - 774144*t^12 + 1327104*t^8 +
10616832*t^4 + 14155776 ], [ 0, 0, 0, -108*t^24 + 864*t^20 -
13824*t^12 + 221184*t^4 - 442368, 432*t^36 - 5184*t^32 + 10368*t^28 +
96768*t^24 - 663552*t^20 + 2654208*t^16 - 6193152*t^12 - 10616832*t^8
+ 84934656*t^4 - 113246208 ], [ 0, 0, 0, -108*t^32 + 5184*t^24 -
110592*t^16 + 1327104*t^8 - 7077888, 432*t^48 - 31104*t^40 +
663552*t^32 - 169869312*t^16 + 2038431744*t^8 - 7247757312 ], [ 0, 0,
0, -27*t^32 + 1296*t^24 - 27648*t^16 + 331776*t^8 - 1769472, 54*t^48 -
3888*t^40 + 82944*t^32 - 21233664*t^16 + 254803968*t^8 - 905969664 ],
[ 0, 0, 0, -27*t^24 + 216*t^20 - 3456*t^12 + 55296*t^4 - 110592,
54*t^36 - 648*t^32 + 1296*t^28 + 12096*t^24 - 82944*t^20 + 331776*t^16
- 774144*t^12 - 1327104*t^8 + 10616832*t^4 - 14155776 ], [ 0, 0, 0,
-108*t^16 + 1728*t^8 - 27648, -432*t^24 + 10368*t^16 + 165888*t^8 -
1769472 ], [ 0, 0, 0, -108*t^24 - 864*t^20 + 13824*t^12 - 221184*t^4 -
442368, -432*t^36 - 5184*t^32 - 10368*t^28 + 96768*t^24 + 663552*t^20
+ 2654208*t^16 + 6193152*t^12 - 10616832*t^8 - 84934656*t^4 -
113246208 ], [ 0, 0, 0, -27*t^24 - 216*t^20 + 3456*t^12 - 55296*t^4 -
110592, -54*t^36 - 648*t^32 - 1296*t^28 + 12096*t^24 + 82944*t^20 +
331776*t^16 + 774144*t^12 - 1327104*t^8 - 10616832*t^4 - 14155776 ], [
0, 0, 0, -27*t^16 + 432*t^8 - 6912, -54*t^24 + 1296*t^16 + 20736*t^8 -
221184 ], [ 0, 0, 0, -108*t^16 + 53568*t^14 - 1560384*t^12 +
13886208*t^10 - 44357760*t^8 + 55544832*t^6 - 24966144*t^4 +
3428352*t^2 - 27648, 432*t^24 + 425088*t^22 - 37376640*t^20 +
928295424*t^18 - 10162195200*t^16 + 54276231168*t^14 -
149378052096*t^12 + 217104924672*t^10 - 162595123200*t^8 +
59410907136*t^6 - 9568419840*t^4 + 435290112*t^2 + 1769472 ], [ 0, 0,
0, -27*t^16 + 13392*t^14 - 390096*t^12 + 3471552*t^10 - 11089440*t^8 +
13886208*t^6 - 6241536*t^4 + 857088*t^2 - 6912, -54*t^24 - 53136*t^22
+ 4672080*t^20 - 116036928*t^18 + 1270274400*t^16 - 6784528896*t^14 +
18672256512*t^12 - 27138115584*t^10 + 20324390400*t^8 - 7426363392*t^6
+ 1196052480*t^4 - 54411264*t^2 - 221184 ], [ 0, 0, 0, -27*t^16 +
13392*t^14 - 390096*t^12 + 3471552*t^10 - 11089440*t^8 + 13886208*t^6
- 6241536*t^4 + 857088*t^2 - 6912, 54*t^24 + 53136*t^22 - 4672080*t^20
+ 116036928*t^18 - 1270274400*t^16 + 6784528896*t^14 -
18672256512*t^12 + 27138115584*t^10 - 20324390400*t^8 + 7426363392*t^6
- 1196052480*t^4 + 54411264*t^2 + 221184 ], [ 0, 0, 0, -108*t^16 +
53568*t^14 - 1560384*t^12 + 13886208*t^10 - 44357760*t^8 +
55544832*t^6 - 24966144*t^4 + 3428352*t^2 - 27648, -432*t^24 -
425088*t^22 + 37376640*t^20 - 928295424*t^18 + 10162195200*t^16 -
54276231168*t^14 + 149378052096*t^12 - 217104924672*t^10 +
162595123200*t^8 - 59410907136*t^6 + 9568419840*t^4 - 435290112*t^2 -
1769472 ], [ 0, 0, 0, -108*t^24 + 56160*t^22 - 2862432*t^20 +
59488128*t^18 - 619949376*t^16 + 3381488640*t^14 - 9458463744*t^12 +
13525954560*t^10 - 9919190016*t^8 + 3807240192*t^6 - 732782592*t^4 +
57507840*t^2 - 442368, 432*t^36 + 409536*t^34 - 52488000*t^32 +
2461722624*t^30 - 61032268800*t^28 + 908384440320*t^26 -
8553406353408*t^24 + 51850555785216*t^22 - 202308258521088*t^20 +
505760996818944*t^18 - 809233034084352*t^16 + 829608892563456*t^14 -
547418006618112*t^12 + 232546416721920*t^10 - 62497043251200*t^8 +
10083215867904*t^6 - 859963392000*t^4 + 26839351296*t^2 + 113246208 ],
[ 0, 0, 0, -27*t^24 + 14040*t^22 - 715608*t^20 + 14872032*t^18 -
154987344*t^16 + 845372160*t^14 - 2364615936*t^12 + 3381488640*t^10 -
2479797504*t^8 + 951810048*t^6 - 183195648*t^4 + 14376960*t^2 -
110592, -54*t^36 - 51192*t^34 + 6561000*t^32 - 307715328*t^30 +
7629033600*t^28 - 113548055040*t^26 + 1069175794176*t^24 -
6481319473152*t^22 + 25288532315136*t^20 - 63220124602368*t^18 +
101154129260544*t^16 - 103701111570432*t^14 + 68427250827264*t^12 -
29068302090240*t^10 + 7812130406400*t^8 - 1260401983488*t^6 +
107495424000*t^4 - 3354918912*t^2 - 14155776 ], [ 0, 0, 0, -108*t^24 +
56160*t^22 - 2862432*t^20 + 59488128*t^18 - 619949376*t^16 +
3381488640*t^14 - 9458463744*t^12 + 13525954560*t^10 - 9919190016*t^8
+ 3807240192*t^6 - 732782592*t^4 + 57507840*t^2 - 442368, -432*t^36 -
409536*t^34 + 52488000*t^32 - 2461722624*t^30 + 61032268800*t^28 -
908384440320*t^26 + 8553406353408*t^24 - 51850555785216*t^22 +
202308258521088*t^20 - 505760996818944*t^18 + 809233034084352*t^16 -
829608892563456*t^14 + 547418006618112*t^12 - 232546416721920*t^10 +
62497043251200*t^8 - 10083215867904*t^6 + 859963392000*t^4 -
26839351296*t^2 - 113246208 ], [ 0, 0, 0, -27*t^24 + 14040*t^22 -
715608*t^20 + 14872032*t^18 - 154987344*t^16 + 845372160*t^14 -
2364615936*t^12 + 3381488640*t^10 - 2479797504*t^8 + 951810048*t^6 -
183195648*t^4 + 14376960*t^2 - 110592, 54*t^36 + 51192*t^34 -
6561000*t^32 + 307715328*t^30 - 7629033600*t^28 + 113548055040*t^26 -
1069175794176*t^24 + 6481319473152*t^22 - 25288532315136*t^20 +
63220124602368*t^18 - 101154129260544*t^16 + 103701111570432*t^14 -
68427250827264*t^12 + 29068302090240*t^10 - 7812130406400*t^8 +
1260401983488*t^6 - 107495424000*t^4 + 3354918912*t^2 + 14155776 ], [
0, 0, 0, -46899*t^24 + 4536*t^23 - 99252*t^22 - 712584*t^21 -
141318*t^20 - 1824984*t^19 - 4402404*t^18 - 489240*t^17 -
19534365*t^16 + 6711984*t^15 - 45292392*t^14 + 6093360*t^13 -
53065044*t^12 - 6093360*t^11 - 45292392*t^10 - 6711984*t^9 -
19534365*t^8 + 489240*t^7 - 4402404*t^6 + 1824984*t^5 - 141318*t^4 +
712584*t^3 - 99252*t^2 - 4536*t - 46899, 3908898*t^36 - 554040*t^35 +
12195684*t^34 + 90945720*t^33 + 1694034*t^32 + 420634944*t^31 +
497906784*t^30 + 1595422656*t^29 + 1731744360*t^28 + 6348776544*t^27 +
9067830768*t^26 + 6801918624*t^25 + 48120356232*t^24 -
12379563072*t^23 + 126428068512*t^22 - 28502988480*t^21 +
206315835804*t^20 - 18027874512*t^19 + 240335075160*t^18 +
18027874512*t^17 + 206315835804*t^16 + 28502988480*t^15 +
126428068512*t^14 + 12379563072*t^13 + 48120356232*t^12 -
6801918624*t^11 + 9067830768*t^10 - 6348776544*t^9 + 1731744360*t^8 -
1595422656*t^7 + 497906784*t^6 - 420634944*t^5 + 1694034*t^4 -
90945720*t^3 + 12195684*t^2 + 554040*t + 3908898 ], [ 0, 0, 0,
-187596*t^24 + 18144*t^23 - 397008*t^22 - 2850336*t^21 - 565272*t^20 -
7299936*t^19 - 17609616*t^18 - 1956960*t^17 - 78137460*t^16 +
26847936*t^15 - 181169568*t^14 + 24373440*t^13 - 212260176*t^12 -
24373440*t^11 - 181169568*t^10 - 26847936*t^9 - 78137460*t^8 +
1956960*t^7 - 17609616*t^6 + 7299936*t^5 - 565272*t^4 + 2850336*t^3 -
397008*t^2 - 18144*t - 187596, -31271184*t^36 + 4432320*t^35 -
97565472*t^34 - 727565760*t^33 - 13552272*t^32 - 3365079552*t^31 -
3983254272*t^30 - 12763381248*t^29 - 13853954880*t^28 -
50790212352*t^27 - 72542646144*t^26 - 54415348992*t^25 -
384962849856*t^24 + 99036504576*t^23 - 1011424548096*t^22 +
228023907840*t^21 - 1650526686432*t^20 + 144222996096*t^19 -
1922680601280*t^18 - 144222996096*t^17 - 1650526686432*t^16 -
228023907840*t^15 - 1011424548096*t^14 - 99036504576*t^13 -
384962849856*t^12 + 54415348992*t^11 - 72542646144*t^10 +
50790212352*t^9 - 13853954880*t^8 + 12763381248*t^7 - 3983254272*t^6 +
3365079552*t^5 - 13552272*t^4 + 727565760*t^3 - 97565472*t^2 -
4432320*t - 31271184 ], [ 0, 0, 0, -5211*t^16 - 13392*t^15 -
16632*t^14 - 60048*t^13 - 102708*t^12 - 127440*t^11 - 289224*t^10 -
80784*t^9 - 506466*t^8 + 80784*t^7 - 289224*t^6 + 127440*t^5 -
102708*t^4 + 60048*t^3 - 16632*t^2 + 13392*t - 5211, 144774*t^24 +
558576*t^23 + 1045224*t^22 + 3299184*t^21 + 7268940*t^20 +
14034384*t^19 + 23074632*t^18 + 42888528*t^17 + 57334554*t^16 +
63764064*t^15 + 127667664*t^14 + 32169312*t^13 + 159922728*t^12 -
32169312*t^11 + 127667664*t^10 - 63764064*t^9 + 57334554*t^8 -
42888528*t^7 + 23074632*t^6 - 14034384*t^5 + 7268940*t^4 - 3299184*t^3
+ 1045224*t^2 - 558576*t + 144774 ], [ 0, 0, 0, -20844*t^16 -
53568*t^15 - 66528*t^14 - 240192*t^13 - 410832*t^12 - 509760*t^11 -
1156896*t^10 - 323136*t^9 - 2025864*t^8 + 323136*t^7 - 1156896*t^6 +
509760*t^5 - 410832*t^4 + 240192*t^3 - 66528*t^2 + 53568*t - 20844,
-1158192*t^24 - 4468608*t^23 - 8361792*t^22 - 26393472*t^21 -
58151520*t^20 - 112275072*t^19 - 184597056*t^18 - 343108224*t^17 -
458676432*t^16 - 510112512*t^15 - 1021341312*t^14 - 257354496*t^13 -
1279381824*t^12 + 257354496*t^11 - 1021341312*t^10 + 510112512*t^9 -
458676432*t^8 + 343108224*t^7 - 184597056*t^6 + 112275072*t^5 -
58151520*t^4 + 26393472*t^3 - 8361792*t^2 + 4468608*t - 1158192 ], [
0, 0, 0, -187596*t^24 + 18144*t^23 - 397008*t^22 - 2850336*t^21 -
565272*t^20 - 7299936*t^19 - 17609616*t^18 - 1956960*t^17 -
78137460*t^16 + 26847936*t^15 - 181169568*t^14 + 24373440*t^13 -
212260176*t^12 - 24373440*t^11 - 181169568*t^10 - 26847936*t^9 -
78137460*t^8 + 1956960*t^7 - 17609616*t^6 + 7299936*t^5 - 565272*t^4 +
2850336*t^3 - 397008*t^2 - 18144*t - 187596, 31271184*t^36 -
4432320*t^35 + 97565472*t^34 + 727565760*t^33 + 13552272*t^32 +
3365079552*t^31 + 3983254272*t^30 + 12763381248*t^29 +
13853954880*t^28 + 50790212352*t^27 + 72542646144*t^26 +
54415348992*t^25 + 384962849856*t^24 - 99036504576*t^23 +
1011424548096*t^22 - 228023907840*t^21 + 1650526686432*t^20 -
144222996096*t^19 + 1922680601280*t^18 + 144222996096*t^17 +
1650526686432*t^16 + 228023907840*t^15 + 1011424548096*t^14 +
99036504576*t^13 + 384962849856*t^12 - 54415348992*t^11 +
72542646144*t^10 - 50790212352*t^9 + 13853954880*t^8 - 12763381248*t^7
+ 3983254272*t^6 - 3365079552*t^5 + 13552272*t^4 - 727565760*t^3 +
97565472*t^2 + 4432320*t + 31271184 ], [ 0, 0, 0, -46899*t^24 +
4536*t^23 - 99252*t^22 - 712584*t^21 - 141318*t^20 - 1824984*t^19 -
4402404*t^18 - 489240*t^17 - 19534365*t^16 + 6711984*t^15 -
45292392*t^14 + 6093360*t^13 - 53065044*t^12 - 6093360*t^11 -
45292392*t^10 - 6711984*t^9 - 19534365*t^8 + 489240*t^7 - 4402404*t^6
+ 1824984*t^5 - 141318*t^4 + 712584*t^3 - 99252*t^2 - 4536*t - 46899,
-3908898*t^36 + 554040*t^35 - 12195684*t^34 - 90945720*t^33 -
1694034*t^32 - 420634944*t^31 - 497906784*t^30 - 1595422656*t^29 -
1731744360*t^28 - 6348776544*t^27 - 9067830768*t^26 - 6801918624*t^25
- 48120356232*t^24 + 12379563072*t^23 - 126428068512*t^22 +
28502988480*t^21 - 206315835804*t^20 + 18027874512*t^19 -
240335075160*t^18 - 18027874512*t^17 - 206315835804*t^16 -
28502988480*t^15 - 126428068512*t^14 - 12379563072*t^13 -
48120356232*t^12 + 6801918624*t^11 - 9067830768*t^10 + 6348776544*t^9
- 1731744360*t^8 + 1595422656*t^7 - 497906784*t^6 + 420634944*t^5 -
1694034*t^4 + 90945720*t^3 - 12195684*t^2 - 554040*t - 3908898 ], [ 0,
0, 0, -5211*t^16 - 13392*t^15 - 16632*t^14 - 60048*t^13 - 102708*t^12
- 127440*t^11 - 289224*t^10 - 80784*t^9 - 506466*t^8 + 80784*t^7 -
289224*t^6 + 127440*t^5 - 102708*t^4 + 60048*t^3 - 16632*t^2 + 13392*t
- 5211, -144774*t^24 - 558576*t^23 - 1045224*t^22 - 3299184*t^21 -
7268940*t^20 - 14034384*t^19 - 23074632*t^18 - 42888528*t^17 -
57334554*t^16 - 63764064*t^15 - 127667664*t^14 - 32169312*t^13 -
159922728*t^12 + 32169312*t^11 - 127667664*t^10 + 63764064*t^9 -
57334554*t^8 + 42888528*t^7 - 23074632*t^6 + 14034384*t^5 -
7268940*t^4 + 3299184*t^3 - 1045224*t^2 + 558576*t - 144774 ], [ 0, 0,
0, -20844*t^16 - 53568*t^15 - 66528*t^14 - 240192*t^13 - 410832*t^12 -
509760*t^11 - 1156896*t^10 - 323136*t^9 - 2025864*t^8 + 323136*t^7 -
1156896*t^6 + 509760*t^5 - 410832*t^4 + 240192*t^3 - 66528*t^2 +
53568*t - 20844, 1158192*t^24 + 4468608*t^23 + 8361792*t^22 +
26393472*t^21 + 58151520*t^20 + 112275072*t^19 + 184597056*t^18 +
343108224*t^17 + 458676432*t^16 + 510112512*t^15 + 1021341312*t^14 +
257354496*t^13 + 1279381824*t^12 - 257354496*t^11 + 1021341312*t^10 -
510112512*t^9 + 458676432*t^8 - 343108224*t^7 + 184597056*t^6 -
112275072*t^5 + 58151520*t^4 - 26393472*t^3 + 8361792*t^2 - 4468608*t
+ 1158192 ], [ 0, 0, 0, -27*t^16 + 6048*t^8 - 6912, 54*t^24 +
28512*t^16 - 456192*t^8 - 221184 ], [ 0, 0, 0, -108*t^16 + 24192*t^8 -
27648, 432*t^24 + 228096*t^16 - 3649536*t^8 - 1769472 ], [ 0, 0, 0,
-27*t^32 + 5184*t^24 + 179712*t^16 + 1327104*t^8 - 1769472, 54*t^48 +
31104*t^40 + 953856*t^32 - 244187136*t^16 - 2038431744*t^8 - 905969664
], [ 0, 0, 0, -108*t^32 + 20736*t^24 + 718848*t^16 + 5308416*t^8 -
7077888, 432*t^48 + 248832*t^40 + 7630848*t^32 - 1953497088*t^16 -
16307453952*t^8 - 7247757312 ], [ 0, 0, 0, -27*t^16 + 378*t^8 - 27,
54*t^24 + 1782*t^16 - 1782*t^8 - 54 ], [ 0, 0, 0, -108*t^16 + 1512*t^8
- 108, 432*t^24 + 14256*t^16 - 14256*t^8 - 432 ], [ 0, 0, 0, -27*t^32
+ 324*t^24 + 702*t^16 + 324*t^8 - 27, 54*t^48 + 1944*t^40 + 3726*t^32
- 3726*t^16 - 1944*t^8 - 54 ], [ 0, 0, 0, -108*t^32 + 1296*t^24 +
2808*t^16 + 1296*t^8 - 108, 432*t^48 + 15552*t^40 + 29808*t^32 -
29808*t^16 - 15552*t^8 - 432 ], [ 0, 0, 0, -7421703487488*t^24 -
18554258718720*t^22 - 16930761080832*t^20 - 6551972610048*t^18 -
809936879616*t^16 + 63417876480*t^14 + 15514730496*t^12 +
990904320*t^10 - 197738496*t^8 - 24993792*t^6 - 1009152*t^4 -
17280*t^2 - 108, 7782220156096217088*t^36 + 29183325585360814080*t^34
+ 44869363087492251648*t^32 + 35992768221945004032*t^30 +
15663238029017874432*t^28 + 3311623469745045504*t^26 +
128840772542791680*t^24 - 66973452271091712*t^22 -
10283697894850560*t^20 - 633396007010304*t^18 - 160682779607040*t^16 -
16350940495872*t^14 + 491488542720*t^12 + 197388140544*t^10 +
14587527168*t^8 + 523763712*t^6 + 10202112*t^4 + 103680*t^2 + 432 ], [
0, 0, 0, -1855425871872*t^24 - 4638564679680*t^22 - 4232690270208*t^20
- 1637993152512*t^18 - 202484219904*t^16 + 15854469120*t^14 +
3878682624*t^12 + 247726080*t^10 - 49434624*t^8 - 6248448*t^6 -
252288*t^4 - 4320*t^2 - 27, 972777519512027136*t^36 +
3647915698170101760*t^34 + 5608670385936531456*t^32 +
4499096027743125504*t^30 + 1957904753627234304*t^28 +
413952933718130688*t^26 + 16105096567848960*t^24 -
8371681533886464*t^22 - 1285462236856320*t^20 - 79174500876288*t^18 -
20085347450880*t^16 - 2043867561984*t^14 + 61436067840*t^12 +
24673517568*t^10 + 1823440896*t^8 + 65470464*t^6 + 1275264*t^4 +
12960*t^2 + 54 ], [ 0, 0, 0, -1811939328*t^16 - 1811939328*t^14 -
339738624*t^12 + 28311552*t^10 + 4423680*t^8 + 442368*t^6 - 82944*t^4
- 6912*t^2 - 108, 29686813949952*t^24 + 44530220924928*t^22 +
19481971654656*t^20 + 1623497637888*t^18 - 413122166784*t^16 -
65229815808*t^14 + 3170893824*t^12 - 1019215872*t^10 - 100859904*t^8 +
6193152*t^6 + 1161216*t^4 + 41472*t^2 + 432 ], [ 0, 0, 0,
-452984832*t^16 - 452984832*t^14 - 84934656*t^12 + 7077888*t^10 +
1105920*t^8 + 110592*t^6 - 20736*t^4 - 1728*t^2 - 27,
3710851743744*t^24 + 5566277615616*t^22 + 2435246456832*t^20 +
202937204736*t^18 - 51640270848*t^16 - 8153726976*t^14 +
396361728*t^12 - 127401984*t^10 - 12607488*t^8 + 774144*t^6 +
145152*t^4 + 5184*t^2 + 54 ], [ 0, 0, 0, -1811939328*t^16 -
1811939328*t^14 - 339738624*t^12 + 28311552*t^10 + 4423680*t^8 +
442368*t^6 - 82944*t^4 - 6912*t^2 - 108, -29686813949952*t^24 -
44530220924928*t^22 - 19481971654656*t^20 - 1623497637888*t^18 +
413122166784*t^16 + 65229815808*t^14 - 3170893824*t^12 +
1019215872*t^10 + 100859904*t^8 - 6193152*t^6 - 1161216*t^4 -
41472*t^2 - 432 ], [ 0, 0, 0, -1855425871872*t^24 - 4638564679680*t^22
- 4232690270208*t^20 - 1637993152512*t^18 - 202484219904*t^16 +
15854469120*t^14 + 3878682624*t^12 + 247726080*t^10 - 49434624*t^8 -
6248448*t^6 - 252288*t^4 - 4320*t^2 - 27, -972777519512027136*t^36 -
3647915698170101760*t^34 - 5608670385936531456*t^32 -
4499096027743125504*t^30 - 1957904753627234304*t^28 -
413952933718130688*t^26 - 16105096567848960*t^24 +
8371681533886464*t^22 + 1285462236856320*t^20 + 79174500876288*t^18 +
20085347450880*t^16 + 2043867561984*t^14 - 61436067840*t^12 -
24673517568*t^10 - 1823440896*t^8 - 65470464*t^6 - 1275264*t^4 -
12960*t^2 - 54 ], [ 0, 0, 0, -7421703487488*t^24 - 18554258718720*t^22
- 16930761080832*t^20 - 6551972610048*t^18 - 809936879616*t^16 +
63417876480*t^14 + 15514730496*t^12 + 990904320*t^10 - 197738496*t^8 -
24993792*t^6 - 1009152*t^4 - 17280*t^2 - 108,
-7782220156096217088*t^36 - 29183325585360814080*t^34 -
44869363087492251648*t^32 - 35992768221945004032*t^30 -
15663238029017874432*t^28 - 3311623469745045504*t^26 -
128840772542791680*t^24 + 66973452271091712*t^22 +
10283697894850560*t^20 + 633396007010304*t^18 + 160682779607040*t^16 +
16350940495872*t^14 - 491488542720*t^12 - 197388140544*t^10 -
14587527168*t^8 - 523763712*t^6 - 10202112*t^4 - 103680*t^2 - 432 ], [
0, 0, 0, -452984832*t^16 - 452984832*t^14 - 84934656*t^12 +
7077888*t^10 + 1105920*t^8 + 110592*t^6 - 20736*t^4 - 1728*t^2 - 27,
-3710851743744*t^24 - 5566277615616*t^22 - 2435246456832*t^20 -
202937204736*t^18 + 51640270848*t^16 + 8153726976*t^14 -
396361728*t^12 + 127401984*t^10 + 12607488*t^8 - 774144*t^6 -
145152*t^4 - 5184*t^2 - 54 ], [ 0, 0, 0, -30399297484750848*t^32 -
455989462271262720*t^30 - 256494072527585280*t^28 -
49873847435919360*t^26 - 8104500208336896*t^24 - 556627761561600*t^22
+ 62620623175680*t^20 + 22613002813440*t^18 + 3960899371008*t^16 +
353328168960*t^14 + 15288238080*t^12 - 2123366400*t^10 - 483065856*t^8
- 46448640*t^6 - 3732480*t^4 - 103680*t^2 - 108,
2040062320599686732316672*t^48 - 64261963098890132067975168*t^46 -
132540298891460897390198784*t^44 - 61249683578629657127288832*t^42 -
16585467596672257857945600*t^40 - 2996590564425784967036928*t^38 -
355943185499528777170944*t^36 - 2696539284087339220992*t^34 +
8098008058367808897024*t^32 + 1984101348234718347264*t^30 +
300200662486285811712*t^28 + 23819749535395086336*t^26 -
372183586490548224*t^22 - 73291177364815872*t^20 -
7568745987833856*t^18 - 482678893707264*t^16 + 2511347908608*t^14 +
5179655061504*t^12 + 681345810432*t^10 + 58923417600*t^8 +
3400040448*t^6 + 114960384*t^4 + 870912*t^2 - 432 ], [ 0, 0, 0,
-7599824371187712*t^32 - 113997365567815680*t^30 -
64123518131896320*t^28 - 12468461858979840*t^26 -
2026125052084224*t^24 - 139156940390400*t^22 + 15655155793920*t^20 +
5653250703360*t^18 + 990224842752*t^16 + 88332042240*t^14 +
3822059520*t^12 - 530841600*t^10 - 120766464*t^8 - 11612160*t^6 -
933120*t^4 - 25920*t^2 - 27, 255007790074960841539584*t^48 -
8032745387361266508496896*t^46 - 16567537361432612173774848*t^44 -
7656210447328707140911104*t^42 - 2073183449584032232243200*t^40 -
374573820553223120879616*t^38 - 44492898187441097146368*t^36 -
337067410510917402624*t^34 + 1012251007295976112128*t^32 +
248012668529339793408*t^30 + 37525082810785726464*t^28 +
2977468691924385792*t^26 - 46522948311318528*t^22 -
9161397170601984*t^20 - 946093248479232*t^18 - 60334861713408*t^16 +
313918488576*t^14 + 647456882688*t^12 + 85168226304*t^10 +
7365427200*t^8 + 425005056*t^6 + 14370048*t^4 + 108864*t^2 - 54 ], [
0, 0, 0, -1855425871872*t^24 - 27831388078080*t^22 -
15597173735424*t^20 - 2174327193600*t^18 - 6794772480*t^16 +
40768634880*t^14 + 7842299904*t^12 + 637009920*t^10 - 1658880*t^8 -
8294400*t^6 - 929664*t^4 - 25920*t^2 - 27, 972777519512027136*t^36 -
30642491864628854784*t^34 - 63245738417024139264*t^32 -
27769758252319899648*t^30 - 4944635731504005120*t^28 -
104735079615430656*t^26 + 108371714324299776*t^24 +
24079716965154816*t^22 + 2644329759768576*t^20 - 41317652496384*t^16 -
5878837149696*t^14 - 413405282304*t^12 + 6242697216*t^10 +
4605050880*t^8 + 404103168*t^6 + 14380416*t^4 + 108864*t^2 - 54 ], [
0, 0, 0, -7421703487488*t^24 - 111325552312320*t^22 -
62388694941696*t^20 - 8697308774400*t^18 - 27179089920*t^16 +
163074539520*t^14 + 31369199616*t^12 + 2548039680*t^10 - 6635520*t^8 -
33177600*t^6 - 3718656*t^4 - 103680*t^2 - 108,
7782220156096217088*t^36 - 245139934917030838272*t^34 -
505965907336193114112*t^32 - 222158066018559197184*t^30 -
39557085852032040960*t^28 - 837880636923445248*t^26 +
866973714594398208*t^24 + 192637735721238528*t^22 +
21154638078148608*t^20 - 330541219971072*t^16 - 47030697197568*t^14 -
3307242258432*t^12 + 49941577728*t^10 + 36840407040*t^8 +
3232825344*t^6 + 115043328*t^4 + 870912*t^2 - 432 ], [ 0, 0, 0,
-1811939328*t^16 - 27179089920*t^14 - 15288238080*t^12 -
2972712960*t^10 - 483950592*t^8 - 46448640*t^6 - 3732480*t^4 -
103680*t^2 - 108, 29686813949952*t^24 - 935134639423488*t^22 -
1928715193810944*t^20 - 891300203200512*t^18 - 241328575217664*t^16 -
44291044933632*t^14 - 6592288260096*t^12 - 692047577088*t^10 -
58918109184*t^8 - 3400040448*t^6 - 114960384*t^4 - 870912*t^2 + 432 ],
[ 0, 0, 0, -1811939328*t^16 - 27179089920*t^14 - 15288238080*t^12 -
2972712960*t^10 - 483950592*t^8 - 46448640*t^6 - 3732480*t^4 -
103680*t^2 - 108, -29686813949952*t^24 + 935134639423488*t^22 +
1928715193810944*t^20 + 891300203200512*t^18 + 241328575217664*t^16 +
44291044933632*t^14 + 6592288260096*t^12 + 692047577088*t^10 +
58918109184*t^8 + 3400040448*t^6 + 114960384*t^4 + 870912*t^2 - 432 ],
[ 0, 0, 0, -452984832*t^16 - 6794772480*t^14 - 3822059520*t^12 -
743178240*t^10 - 120987648*t^8 - 11612160*t^6 - 933120*t^4 - 25920*t^2
- 27, 3710851743744*t^24 - 116891829927936*t^22 - 241089399226368*t^20
- 111412525400064*t^18 - 30166071902208*t^16 - 5536380616704*t^14 -
824036032512*t^12 - 86505947136*t^10 - 7364763648*t^8 - 425005056*t^6
- 14370048*t^4 - 108864*t^2 + 54 ], [ 0, 0, 0, -452984832*t^16 -
6794772480*t^14 - 3822059520*t^12 - 743178240*t^10 - 120987648*t^8 -
11612160*t^6 - 933120*t^4 - 25920*t^2 - 27, -3710851743744*t^24 +
116891829927936*t^22 + 241089399226368*t^20 + 111412525400064*t^18 +
30166071902208*t^16 + 5536380616704*t^14 + 824036032512*t^12 +
86505947136*t^10 + 7364763648*t^8 + 425005056*t^6 + 14370048*t^4 +
108864*t^2 - 54 ], [ 0, 0, 0, -7421703487488*t^24 -
111325552312320*t^22 - 62852551409664*t^20 - 15655155793920*t^18 -
3940968038400*t^16 - 597939978240*t^14 - 92522151936*t^12 -
9342812160*t^10 - 962150400*t^8 - 59719680*t^6 - 3746304*t^4 -
103680*t^2 - 108, -7782220156096217088*t^36 +
245139934917030838272*t^34 + 505236324196559093760*t^32 +
245139934917030838272*t^30 + 86957190455129800704*t^28 +
22742474430779228160*t^26 + 5063857976327012352*t^24 +
897729253846548480*t^22 + 144714868589592576*t^20 +
18790361660915712*t^18 + 2261169821712384*t^16 + 219172181114880*t^14
+ 19317085175808*t^12 + 1355557109760*t^10 + 80985194496*t^8 +
3567255552*t^6 + 114877440*t^4 + 870912*t^2 - 432 ], [ 0, 0, 0,
-1855425871872*t^24 - 27831388078080*t^22 - 15713137852416*t^20 -
3913788948480*t^18 - 985242009600*t^16 - 149484994560*t^14 -
23130537984*t^12 - 2335703040*t^10 - 240537600*t^8 - 14929920*t^6 -
936576*t^4 - 25920*t^2 - 27, 972777519512027136*t^36 -
30642491864628854784*t^34 - 63154540524569886720*t^32 -
30642491864628854784*t^30 - 10869648806891225088*t^28 -
2842809303847403520*t^26 - 632982247040876544*t^24 -
112216156730818560*t^22 - 18089358573699072*t^20 -
2348795207614464*t^18 - 282646227714048*t^16 - 27396522639360*t^14 -
2414635646976*t^12 - 169444638720*t^10 - 10123149312*t^8 -
445906944*t^6 - 14359680*t^4 - 108864*t^2 + 54 ], [ 0, 0, 0,
-1855425871872*t^24 - 27831388078080*t^22 - 15713137852416*t^20 -
3913788948480*t^18 - 985242009600*t^16 - 149484994560*t^14 -
23130537984*t^12 - 2335703040*t^10 - 240537600*t^8 - 14929920*t^6 -
936576*t^4 - 25920*t^2 - 27, -972777519512027136*t^36 +
30642491864628854784*t^34 + 63154540524569886720*t^32 +
30642491864628854784*t^30 + 10869648806891225088*t^28 +
2842809303847403520*t^26 + 632982247040876544*t^24 +
112216156730818560*t^22 + 18089358573699072*t^20 +
2348795207614464*t^18 + 282646227714048*t^16 + 27396522639360*t^14 +
2414635646976*t^12 + 169444638720*t^10 + 10123149312*t^8 +
445906944*t^6 + 14359680*t^4 + 108864*t^2 - 54 ], [ 0, 0, 0,
-7421703487488*t^24 - 111325552312320*t^22 - 62852551409664*t^20 -
15655155793920*t^18 - 3940968038400*t^16 - 597939978240*t^14 -
92522151936*t^12 - 9342812160*t^10 - 962150400*t^8 - 59719680*t^6 -
3746304*t^4 - 103680*t^2 - 108, 7782220156096217088*t^36 -
245139934917030838272*t^34 - 505236324196559093760*t^32 -
245139934917030838272*t^30 - 86957190455129800704*t^28 -
22742474430779228160*t^26 - 5063857976327012352*t^24 -
897729253846548480*t^22 - 144714868589592576*t^20 -
18790361660915712*t^18 - 2261169821712384*t^16 - 219172181114880*t^14
- 19317085175808*t^12 - 1355557109760*t^10 - 80985194496*t^8 -
3567255552*t^6 - 114877440*t^4 - 870912*t^2 + 432 ], [ 0, 0, 0,
-3564*t^32 - 88128*t^31 - 800064*t^30 - 2229120*t^29 + 14097024*t^28 +
134638848*t^27 + 323847936*t^26 - 988789248*t^25 - 8550358272*t^24 -
18531652608*t^23 + 32398396416*t^22 + 313799436288*t^21 +
855828633600*t^20 + 527502938112*t^19 - 4596406456320*t^18 -
21383917608960*t^17 - 55056817096704*t^16 - 100407001006080*t^15 -
137527648174080*t^14 - 142248065531904*t^13 - 106908432629760*t^12 -
50300615983104*t^11 - 2987281022976*t^10 + 17578568318976*t^9 +
16055072391168*t^8 + 7268445978624*t^7 + 1212605005824*t^6 -
636033171456*t^5 - 533403795456*t^4 - 190678302720*t^3 -
39437991936*t^2 - 4586471424*t - 233570304, -81648*t^48 - 3017088*t^47
- 45567360*t^46 - 321898752*t^45 - 326778624*t^44 + 13165161984*t^43 +
112051911168*t^42 + 346729559040*t^41 - 718858022400*t^40 -
11477684938752*t^39 - 49025054656512*t^38 - 76075655196672*t^37 +
278725138698240*t^36 + 2290028887941120*t^35 + 8241350089236480*t^34 +
17222563095035904*t^33 + 3186296647446528*t^32 -
155189034483646464*t^31 - 798926464888012800*t^30 -
2429545667999956992*t^29 - 4577511419798224896*t^28 -
2139614524879405056*t^27 + 20891300786264604672*t^26 +
95301949961009627136*t^25 + 257483368125498654720*t^24 +
520343985360089382912*t^23 + 837405411543094394880*t^22 +
1095412709814630875136*t^21 + 1163787006690629517312*t^20 +
981330926754255077376*t^19 + 612062871778077179904*t^18 +
213458880970339909632*t^17 - 64494374282207428608*t^16 -
166414170138882342912*t^15 - 138462146838512271360*t^14 -
66761729722244136960*t^13 - 12513730463753306112*t^12 +
8824262876895117312*t^11 + 9449870858832052224*t^10 +
4482763270058409984*t^9 + 1067959050269884416*t^8 -
83592541668114432*t^7 - 187906454106144768*t^6 - 86200658008473600*t^5
- 23929753618612224*t^4 - 4449905556848640*t^3 - 546756316102656*t^2 -
40442485800960*t - 1369826131968 ], [ 0, 0, 0, -3564*t^32 - 88128*t^31
- 800064*t^30 - 2229120*t^29 + 14097024*t^28 + 134638848*t^27 +
323847936*t^26 - 988789248*t^25 - 8550358272*t^24 - 18531652608*t^23 +
32398396416*t^22 + 313799436288*t^21 + 855828633600*t^20 +
527502938112*t^19 - 4596406456320*t^18 - 21383917608960*t^17 -
55056817096704*t^16 - 100407001006080*t^15 - 137527648174080*t^14 -
142248065531904*t^13 - 106908432629760*t^12 - 50300615983104*t^11 -
2987281022976*t^10 + 17578568318976*t^9 + 16055072391168*t^8 +
7268445978624*t^7 + 1212605005824*t^6 - 636033171456*t^5 -
533403795456*t^4 - 190678302720*t^3 - 39437991936*t^2 - 4586471424*t -
233570304, 81648*t^48 + 3017088*t^47 + 45567360*t^46 + 321898752*t^45
+ 326778624*t^44 - 13165161984*t^43 - 112051911168*t^42 -
346729559040*t^41 + 718858022400*t^40 + 11477684938752*t^39 +
49025054656512*t^38 + 76075655196672*t^37 - 278725138698240*t^36 -
2290028887941120*t^35 - 8241350089236480*t^34 - 17222563095035904*t^33
- 3186296647446528*t^32 + 155189034483646464*t^31 +
798926464888012800*t^30 + 2429545667999956992*t^29 +
4577511419798224896*t^28 + 2139614524879405056*t^27 -
20891300786264604672*t^26 - 95301949961009627136*t^25 -
257483368125498654720*t^24 - 520343985360089382912*t^23 -
837405411543094394880*t^22 - 1095412709814630875136*t^21 -
1163787006690629517312*t^20 - 981330926754255077376*t^19 -
612062871778077179904*t^18 - 213458880970339909632*t^17 +
64494374282207428608*t^16 + 166414170138882342912*t^15 +
138462146838512271360*t^14 + 66761729722244136960*t^13 +
12513730463753306112*t^12 - 8824262876895117312*t^11 -
9449870858832052224*t^10 - 4482763270058409984*t^9 -
1067959050269884416*t^8 + 83592541668114432*t^7 +
187906454106144768*t^6 + 86200658008473600*t^5 + 23929753618612224*t^4
+ 4449905556848640*t^3 + 546756316102656*t^2 + 40442485800960*t +
1369826131968 ], [ 0, 0, 0, -891*t^16 - 7776*t^15 - 18576*t^14 +
38016*t^13 + 267408*t^12 + 107136*t^11 - 3378240*t^10 - 16561152*t^9 -
45469728*t^8 - 85584384*t^7 - 116156160*t^6 - 114849792*t^5 -
81983232*t^4 - 41084928*t^3 - 13630464*t^2 - 2654208*t - 228096,
-10206*t^24 - 132192*t^23 - 563760*t^22 + 470016*t^21 + 15939504*t^20
+ 86873472*t^19 + 291598272*t^18 + 703157760*t^17 + 998518752*t^16 -
1481518080*t^15 - 16926340608*t^14 - 69110931456*t^13 -
190561641984*t^12 - 394978332672*t^11 - 640645687296*t^10 -
828529164288*t^9 - 860825387520*t^8 - 718074077184*t^7 -
477110366208*t^6 - 248593121280*t^5 - 99072626688*t^4 -
29057384448*t^3 - 5888360448*t^2 - 732561408*t - 41803776 ], [ 0, 0,
0, -891*t^16 - 7776*t^15 - 18576*t^14 + 38016*t^13 + 267408*t^12 +
107136*t^11 - 3378240*t^10 - 16561152*t^9 - 45469728*t^8 -
85584384*t^7 - 116156160*t^6 - 114849792*t^5 - 81983232*t^4 -
41084928*t^3 - 13630464*t^2 - 2654208*t - 228096, 10206*t^24 +
132192*t^23 + 563760*t^22 - 470016*t^21 - 15939504*t^20 -
86873472*t^19 - 291598272*t^18 - 703157760*t^17 - 998518752*t^16 +
1481518080*t^15 + 16926340608*t^14 + 69110931456*t^13 +
190561641984*t^12 + 394978332672*t^11 + 640645687296*t^10 +
828529164288*t^9 + 860825387520*t^8 + 718074077184*t^7 +
477110366208*t^6 + 248593121280*t^5 + 99072626688*t^4 +
29057384448*t^3 + 5888360448*t^2 + 732561408*t + 41803776 ], [ 0, 0,
0, -3564*t^16 - 31104*t^15 - 74304*t^14 + 152064*t^13 + 1069632*t^12 +
428544*t^11 - 13512960*t^10 - 66244608*t^9 - 181878912*t^8 -
342337536*t^7 - 464624640*t^6 - 459399168*t^5 - 327932928*t^4 -
164339712*t^3 - 54521856*t^2 - 10616832*t - 912384, 81648*t^24 +
1057536*t^23 + 4510080*t^22 - 3760128*t^21 - 127516032*t^20 -
694987776*t^19 - 2332786176*t^18 - 5625262080*t^17 - 7988150016*t^16 +
11852144640*t^15 + 135410724864*t^14 + 552887451648*t^13 +
1524493135872*t^12 + 3159826661376*t^11 + 5125165498368*t^10 +
6628233314304*t^9 + 6886603100160*t^8 + 5744592617472*t^7 +
3816882929664*t^6 + 1988744970240*t^5 + 792581013504*t^4 +
232459075584*t^3 + 47106883584*t^2 + 5860491264*t + 334430208 ], [ 0,
0, 0, -891*t^32 - 22032*t^31 - 200016*t^30 - 557280*t^29 +
3524256*t^28 + 33659712*t^27 + 80961984*t^26 - 247197312*t^25 -
2137589568*t^24 - 4632913152*t^23 + 8099599104*t^22 + 78449859072*t^21
+ 213957158400*t^20 + 131875734528*t^19 - 1149101614080*t^18 -
5345979402240*t^17 - 13764204274176*t^16 - 25101750251520*t^15 -
34381912043520*t^14 - 35562016382976*t^13 - 26727108157440*t^12 -
12575153995776*t^11 - 746820255744*t^10 + 4394642079744*t^9 +
4013768097792*t^8 + 1817111494656*t^7 + 303151251456*t^6 -
159008292864*t^5 - 133350948864*t^4 - 47669575680*t^3 - 9859497984*t^2
- 1146617856*t - 58392576, -10206*t^48 - 377136*t^47 - 5695920*t^46 -
40237344*t^45 - 40847328*t^44 + 1645645248*t^43 + 14006488896*t^42 +
43341194880*t^41 - 89857252800*t^40 - 1434710617344*t^39 -
6128131832064*t^38 - 9509456899584*t^37 + 34840642337280*t^36 +
286253610992640*t^35 + 1030168761154560*t^34 + 2152820386879488*t^33 +
398287080930816*t^32 - 19398629310455808*t^31 - 99865808111001600*t^30
- 303693208499994624*t^29 - 572188927474778112*t^28 -
267451815609925632*t^27 + 2611412598283075584*t^26 +
11912743745126203392*t^25 + 32185421015687331840*t^24 +
65042998170011172864*t^23 + 104675676442886799360*t^22 +
136926588726828859392*t^21 + 145473375836328689664*t^20 +
122666365844281884672*t^19 + 76507858972259647488*t^18 +
26682360121292488704*t^17 - 8061796785275928576*t^16 -
20801771267360292864*t^15 - 17307768354814033920*t^14 -
8345216215280517120*t^13 - 1564216307969163264*t^12 +
1103032859611889664*t^11 + 1181233857354006528*t^10 +
560345408757301248*t^9 + 133494881283735552*t^8 -
10449067708514304*t^7 - 23488306763268096*t^6 - 10775082251059200*t^5
- 2991219202326528*t^4 - 556238194606080*t^3 - 68344539512832*t^2 -
5055310725120*t - 171228266496 ], [ 0, 0, 0, -891*t^32 - 22032*t^31 -
200016*t^30 - 557280*t^29 + 3524256*t^28 + 33659712*t^27 +
80961984*t^26 - 247197312*t^25 - 2137589568*t^24 - 4632913152*t^23 +
8099599104*t^22 + 78449859072*t^21 + 213957158400*t^20 +
131875734528*t^19 - 1149101614080*t^18 - 5345979402240*t^17 -
13764204274176*t^16 - 25101750251520*t^15 - 34381912043520*t^14 -
35562016382976*t^13 - 26727108157440*t^12 - 12575153995776*t^11 -
746820255744*t^10 + 4394642079744*t^9 + 4013768097792*t^8 +
1817111494656*t^7 + 303151251456*t^6 - 159008292864*t^5 -
133350948864*t^4 - 47669575680*t^3 - 9859497984*t^2 - 1146617856*t -
58392576, 10206*t^48 + 377136*t^47 + 5695920*t^46 + 40237344*t^45 +
40847328*t^44 - 1645645248*t^43 - 14006488896*t^42 - 43341194880*t^41
+ 89857252800*t^40 + 1434710617344*t^39 + 6128131832064*t^38 +
9509456899584*t^37 - 34840642337280*t^36 - 286253610992640*t^35 -
1030168761154560*t^34 - 2152820386879488*t^33 - 398287080930816*t^32 +
19398629310455808*t^31 + 99865808111001600*t^30 +
303693208499994624*t^29 + 572188927474778112*t^28 +
267451815609925632*t^27 - 2611412598283075584*t^26 -
11912743745126203392*t^25 - 32185421015687331840*t^24 -
65042998170011172864*t^23 - 104675676442886799360*t^22 -
136926588726828859392*t^21 - 145473375836328689664*t^20 -
122666365844281884672*t^19 - 76507858972259647488*t^18 -
26682360121292488704*t^17 + 8061796785275928576*t^16 +
20801771267360292864*t^15 + 17307768354814033920*t^14 +
8345216215280517120*t^13 + 1564216307969163264*t^12 -
1103032859611889664*t^11 - 1181233857354006528*t^10 -
560345408757301248*t^9 - 133494881283735552*t^8 +
10449067708514304*t^7 + 23488306763268096*t^6 + 10775082251059200*t^5
+ 2991219202326528*t^4 + 556238194606080*t^3 + 68344539512832*t^2 +
5055310725120*t + 171228266496 ], [ 0, 0, 0, -3564*t^16 - 31104*t^15 -
74304*t^14 + 152064*t^13 + 1069632*t^12 + 428544*t^11 - 13512960*t^10
- 66244608*t^9 - 181878912*t^8 - 342337536*t^7 - 464624640*t^6 -
459399168*t^5 - 327932928*t^4 - 164339712*t^3 - 54521856*t^2 -
10616832*t - 912384, -81648*t^24 - 1057536*t^23 - 4510080*t^22 +
3760128*t^21 + 127516032*t^20 + 694987776*t^19 + 2332786176*t^18 +
5625262080*t^17 + 7988150016*t^16 - 11852144640*t^15 -
135410724864*t^14 - 552887451648*t^13 - 1524493135872*t^12 -
3159826661376*t^11 - 5125165498368*t^10 - 6628233314304*t^9 -
6886603100160*t^8 - 5744592617472*t^7 - 3816882929664*t^6 -
1988744970240*t^5 - 792581013504*t^4 - 232459075584*t^3 -
47106883584*t^2 - 5860491264*t - 334430208 ], [ 0, 0, 0,
-452984832*t^24 + 13589544960*t^22 - 15231614976*t^20 +
4246732800*t^18 - 26542080*t^16 - 318504960*t^14 + 122535936*t^12 -
19906560*t^10 - 103680*t^8 + 1036800*t^6 - 232416*t^4 + 12960*t^2 -
27, 3710851743744*t^36 + 233783659855872*t^34 - 965053381607424*t^32 +
847465766977536*t^30 - 301796614471680*t^28 + 12785043898368*t^26 +
26457938067456*t^24 - 11757674299392*t^22 + 2582353281024*t^20 -
161397080064*t^16 + 45928415232*t^14 - 6459457536*t^12 -
195084288*t^10 + 287815680*t^8 - 50512896*t^6 + 3595104*t^4 -
54432*t^2 - 54 ], [ 0, 0, 0, -1811939328*t^24 + 54358179840*t^22 -
60926459904*t^20 + 16986931200*t^18 - 106168320*t^16 - 1274019840*t^14
+ 490143744*t^12 - 79626240*t^10 - 414720*t^8 + 4147200*t^6 -
929664*t^4 + 51840*t^2 - 108, 29686813949952*t^36 +
1870269278846976*t^34 - 7720427052859392*t^32 + 6779726135820288*t^30
- 2414372915773440*t^28 + 102280351186944*t^26 + 211663504539648*t^24
- 94061394395136*t^22 + 20658826248192*t^20 - 1291176640512*t^16 +
367427321856*t^14 - 51675660288*t^12 - 1560674304*t^10 +
2302525440*t^8 - 404103168*t^6 + 28760832*t^4 - 435456*t^2 - 432 ], [
0, 0, 0, -463856467968*t^32 + 13915694039040*t^30 -
15655155793920*t^28 + 6088116142080*t^26 - 1978637746176*t^24 +
271790899200*t^22 + 61152952320*t^20 - 44166021120*t^18 +
15472263168*t^16 - 2760376320*t^14 + 238878720*t^12 + 66355200*t^10 -
30191616*t^8 + 5806080*t^6 - 933120*t^4 + 51840*t^2 - 108,
121597189939003392*t^48 + 7660622966157213696*t^46 -
31600069735398506496*t^44 + 29206125058474377216*t^42 -
15817134472534425600*t^40 + 5715542916156358656*t^38 -
1357815496442904576*t^36 + 20572962067316736*t^34 +
123565796789059584*t^32 - 60549967902670848*t^30 +
18322794341203968*t^28 - 2907684269457408*t^26 + 181730266841088*t^22
- 71573415395328*t^20 + 14782707007488*t^18 - 1885464428544*t^16 -
19619905536*t^14 + 80932110336*t^12 - 21292056576*t^10 +
3682713600*t^8 - 425005056*t^6 + 28740096*t^4 - 435456*t^2 - 432 ], [
0, 0, 0, -115964116992*t^32 + 3478923509760*t^30 - 3913788948480*t^28
+ 1522029035520*t^26 - 494659436544*t^24 + 67947724800*t^22 +
15288238080*t^20 - 11041505280*t^18 + 3868065792*t^16 - 690094080*t^14
+ 59719680*t^12 + 16588800*t^10 - 7547904*t^8 + 1451520*t^6 -
233280*t^4 + 12960*t^2 - 27, 15199648742375424*t^48 +
957577870769651712*t^46 - 3950008716924813312*t^44 +
3650765632309297152*t^42 - 1977141809066803200*t^40 +
714442864519544832*t^38 - 169726937055363072*t^36 +
2571620258414592*t^34 + 15445724598632448*t^32 - 7568745987833856*t^30
+ 2290349292650496*t^28 - 363460533682176*t^26 + 22716283355136*t^22 -
8946676924416*t^20 + 1847838375936*t^18 - 235683053568*t^16 -
2452488192*t^14 + 10116513792*t^12 - 2661507072*t^10 + 460339200*t^8 -
53125632*t^6 + 3592512*t^4 - 54432*t^2 - 54 ], [ 0, 0, 0,
-1811939328*t^24 + 54358179840*t^22 - 61379444736*t^20 +
30576476160*t^18 - 15394406400*t^16 + 4671406080*t^14 -
1445658624*t^12 + 291962880*t^10 - 60134400*t^8 + 7464960*t^6 -
936576*t^4 + 51840*t^2 - 108, -29686813949952*t^36 -
1870269278846976*t^34 + 7709294497628160*t^32 - 7481077115387904*t^30
+ 5307445706489856*t^28 - 2776180960788480*t^26 +
1236293451251712*t^24 - 438344362229760*t^22 + 141323113857024*t^20 -
36699925118976*t^18 + 8832694616064*t^16 - 1712282664960*t^14 +
301829455872*t^12 - 42361159680*t^10 + 5061574656*t^8 - 445906944*t^6
+ 28719360*t^4 - 435456*t^2 - 432 ], [ 0, 0, 0, -452984832*t^24 +
13589544960*t^22 - 15344861184*t^20 + 7644119040*t^18 -
3848601600*t^16 + 1167851520*t^14 - 361414656*t^12 + 72990720*t^10 -
15033600*t^8 + 1866240*t^6 - 234144*t^4 + 12960*t^2 - 27,
3710851743744*t^36 + 233783659855872*t^34 - 963661812203520*t^32 +
935134639423488*t^30 - 663430713311232*t^28 + 347022620098560*t^26 -
154536681406464*t^24 + 54793045278720*t^22 - 17665389232128*t^20 +
4587490639872*t^18 - 1104086827008*t^16 + 214035333120*t^14 -
37728681984*t^12 + 5295144960*t^10 - 632696832*t^8 + 55738368*t^6 -
3589920*t^4 + 54432*t^2 + 54 ], [ 0, 0, 0, -1811939328*t^24 +
54358179840*t^22 - 61379444736*t^20 + 30576476160*t^18 -
15394406400*t^16 + 4671406080*t^14 - 1445658624*t^12 + 291962880*t^10
- 60134400*t^8 + 7464960*t^6 - 936576*t^4 + 51840*t^2 - 108,
29686813949952*t^36 + 1870269278846976*t^34 - 7709294497628160*t^32 +
7481077115387904*t^30 - 5307445706489856*t^28 + 2776180960788480*t^26
- 1236293451251712*t^24 + 438344362229760*t^22 - 141323113857024*t^20
+ 36699925118976*t^18 - 8832694616064*t^16 + 1712282664960*t^14 -
301829455872*t^12 + 42361159680*t^10 - 5061574656*t^8 + 445906944*t^6
- 28719360*t^4 + 435456*t^2 + 432 ], [ 0, 0, 0, -7077888*t^16 +
212336640*t^14 - 238878720*t^12 + 92897280*t^10 - 30246912*t^8 +
5806080*t^6 - 933120*t^4 + 51840*t^2 - 108, -7247757312*t^24 -
456608710656*t^22 + 1883510931456*t^20 - 1740820709376*t^18 +
942689746944*t^16 - 346023788544*t^14 + 103004504064*t^12 -
21626486784*t^10 + 3682381824*t^8 - 425005056*t^6 + 28740096*t^4 -
435456*t^2 - 432 ], [ 0, 0, 0, -1769472*t^16 + 53084160*t^14 -
59719680*t^12 + 23224320*t^10 - 7561728*t^8 + 1451520*t^6 - 233280*t^4
+ 12960*t^2 - 27, -905969664*t^24 - 57076088832*t^22 +
235438866432*t^20 - 217602588672*t^18 + 117836218368*t^16 -
43252973568*t^14 + 12875563008*t^12 - 2703310848*t^10 + 460297728*t^8
- 53125632*t^6 + 3592512*t^4 - 54432*t^2 - 54 ], [ 0, 0, 0,
-7077888*t^16 + 212336640*t^14 - 238878720*t^12 + 92897280*t^10 -
30246912*t^8 + 5806080*t^6 - 933120*t^4 + 51840*t^2 - 108,
7247757312*t^24 + 456608710656*t^22 - 1883510931456*t^20 +
1740820709376*t^18 - 942689746944*t^16 + 346023788544*t^14 -
103004504064*t^12 + 21626486784*t^10 - 3682381824*t^8 + 425005056*t^6
- 28740096*t^4 + 435456*t^2 + 432 ], [ 0, 0, 0, -1769472*t^16 +
53084160*t^14 - 59719680*t^12 + 23224320*t^10 - 7561728*t^8 +
1451520*t^6 - 233280*t^4 + 12960*t^2 - 27, 905969664*t^24 +
57076088832*t^22 - 235438866432*t^20 + 217602588672*t^18 -
117836218368*t^16 + 43252973568*t^14 - 12875563008*t^12 +
2703310848*t^10 - 460297728*t^8 + 53125632*t^6 - 3592512*t^4 +
54432*t^2 + 54 ], [ 0, 0, 0, -452984832*t^24 + 13589544960*t^22 -
15344861184*t^20 + 7644119040*t^18 - 3848601600*t^16 + 1167851520*t^14
- 361414656*t^12 + 72990720*t^10 - 15033600*t^8 + 1866240*t^6 -
234144*t^4 + 12960*t^2 - 27, -3710851743744*t^36 -
233783659855872*t^34 + 963661812203520*t^32 - 935134639423488*t^30 +
663430713311232*t^28 - 347022620098560*t^26 + 154536681406464*t^24 -
54793045278720*t^22 + 17665389232128*t^20 - 4587490639872*t^18 +
1104086827008*t^16 - 214035333120*t^14 + 37728681984*t^12 -
5295144960*t^10 + 632696832*t^8 - 55738368*t^6 + 3589920*t^4 -
54432*t^2 - 54 ], [ 0, 0, 0, 486*t^24 + 33696*t^23 + 1050624*t^22 +
19761408*t^21 + 250953984*t^20 + 2258896896*t^19 + 14508343296*t^18 +
62900748288*t^17 + 133924036608*t^16 - 402781372416*t^15 -
5273960841216*t^14 - 27155478085632*t^13 - 91803661369344*t^12 -
217243824685056*t^11 - 337533493837824*t^10 - 206224062676992*t^9 +
548552853946368*t^8 + 2061131719901184*t^7 + 3803275144986624*t^6 +
4737250143240192*t^5 + 4210309195628544*t^4 + 2652331283841024*t^3 +
1128098930098176*t^2 + 289446436012032*t + 33397665693696, -5103*t^36
- 373248*t^35 - 12204432*t^34 - 219691008*t^33 - 1792787904*t^32 +
15984304128*t^31 + 770855657472*t^30 + 13858803744768*t^29 +
169061744001024*t^28 + 1570120425013248*t^27 + 11594918710738944*t^26
+ 69335416427249664*t^25 + 336873109192704000*t^24 +
1314710158332395520*t^23 + 3972084924558606336*t^22 +
8325423280919937024*t^21 + 6618150161627480064*t^20 -
30199927916074106880*t^19 - 144044548694514597888*t^18 -
241599423328592855040*t^17 + 423561610344158724096*t^16 +
4262616719831007756288*t^15 + 16269659850992051552256*t^14 +
43080422468235936399360*t^13 + 88309264336212197376000*t^12 +
145406907231239487356928*t^11 + 194530455712508783099904*t^10 +
210737996131672516460544*t^9 + 181528665372280567627776*t^8 +
119046217690921782214656*t^7 + 52972797420461088571392*t^6 +
8787464125321958129664*t^5 - 7884764586336652886016*t^4 -
7729690170042567622656*t^3 - 3435242212966784827392*t^2 -
840479776858391445504*t - 91927475593886564352 ], [ 0, 0, 0, 54*t^16 +
2592*t^15 + 50112*t^14 + 518400*t^13 + 2985984*t^12 + 7133184*t^11 -
24440832*t^10 - 272056320*t^9 - 1063895040*t^8 - 2176450560*t^7 -
1564213248*t^6 + 3652190208*t^5 + 12230590464*t^4 + 16986931200*t^3 +
13136560128*t^2 + 5435817984*t + 905969664, 189*t^24 + 7776*t^23 +
111456*t^22 - 62208*t^21 - 26490240*t^20 - 470292480*t^19 -
4693248000*t^18 - 30986551296*t^17 - 138235465728*t^16 -
381505241088*t^15 - 330926653440*t^14 + 2136786075648*t^13 +
10246545211392*t^12 + 17094288605184*t^11 - 21179305820160*t^10 -
195330683437056*t^9 - 566212467621888*t^8 - 1015367312867328*t^7 -
1230306803712000*t^6 - 986274815016960*t^5 - 444432478371840*t^4 -
8349416423424*t^3 + 119674968735744*t^2 + 66795331387392*t +
12987981103104 ], [ 0, 0, 0, 216*t^16 + 10368*t^15 + 200448*t^14 +
2073600*t^13 + 11943936*t^12 + 28532736*t^11 - 97763328*t^10 -
1088225280*t^9 - 4255580160*t^8 - 8705802240*t^7 - 6256852992*t^6 +
14608760832*t^5 + 48922361856*t^4 + 67947724800*t^3 + 52546240512*t^2
+ 21743271936*t + 3623878656, -1512*t^24 - 62208*t^23 - 891648*t^22 +
497664*t^21 + 211921920*t^20 + 3762339840*t^19 + 37545984000*t^18 +
247892410368*t^17 + 1105883725824*t^16 + 3052041928704*t^15 +
2647413227520*t^14 - 17094288605184*t^13 - 81972361691136*t^12 -
136754308841472*t^11 + 169434446561280*t^10 + 1562645467496448*t^9 +
4529699740975104*t^8 + 8122938502938624*t^7 + 9842454429696000*t^6 +
7890198520135680*t^5 + 3555459826974720*t^4 + 66795331387392*t^3 -
957399749885952*t^2 - 534362651099136*t - 103903848824832 ], [ 0, 0,
0, 1944*t^24 + 134784*t^23 + 4202496*t^22 + 79045632*t^21 +
1003815936*t^20 + 9035587584*t^19 + 58033373184*t^18 +
251602993152*t^17 + 535696146432*t^16 - 1611125489664*t^15 -
21095843364864*t^14 - 108621912342528*t^13 - 367214645477376*t^12 -
868975298740224*t^11 - 1350133975351296*t^10 - 824896250707968*t^9 +
2194211415785472*t^8 + 8244526879604736*t^7 + 15213100579946496*t^6 +
18949000572960768*t^5 + 16841236782514176*t^4 + 10609325135364096*t^3
+ 4512395720392704*t^2 + 1157785744048128*t + 133590662774784,
40824*t^36 + 2985984*t^35 + 97635456*t^34 + 1757528064*t^33 +
14342303232*t^32 - 127874433024*t^31 - 6166845259776*t^30 -
110870429958144*t^29 - 1352493952008192*t^28 - 12560963400105984*t^27
- 92759349685911552*t^26 - 554683331417997312*t^25 -
2694984873541632000*t^24 - 10517681266659164160*t^23 -
31776679396468850688*t^22 - 66603386247359496192*t^21 -
52945201293019840512*t^20 + 241599423328592855040*t^19 +
1152356389556116783104*t^18 + 1932795386628742840320*t^17 -
3388492882753269792768*t^16 - 34100933758648062050304*t^15 -
130157278807936412418048*t^14 - 344643379745887491194880*t^13 -
706474114689697579008000*t^12 - 1163255257849915898855424*t^11 -
1556243645700070264799232*t^10 - 1685903969053380131684352*t^9 -
1452229322978244541022208*t^8 - 952369741527374257717248*t^7 -
423782379363688708571136*t^6 - 70299713002575665037312*t^5 +
63078116690693223088128*t^4 + 61837521360340540981248*t^3 +
27481937703734278619136*t^2 + 6723838214867131564032*t +
735419804751092514816 ], [ 0, 0, 0, 486*t^24 + 33696*t^23 +
1050624*t^22 + 19761408*t^21 + 250953984*t^20 + 2258896896*t^19 +
14508343296*t^18 + 62900748288*t^17 + 133924036608*t^16 -
402781372416*t^15 - 5273960841216*t^14 - 27155478085632*t^13 -
91803661369344*t^12 - 217243824685056*t^11 - 337533493837824*t^10 -
206224062676992*t^9 + 548552853946368*t^8 + 2061131719901184*t^7 +
3803275144986624*t^6 + 4737250143240192*t^5 + 4210309195628544*t^4 +
2652331283841024*t^3 + 1128098930098176*t^2 + 289446436012032*t +
33397665693696, 5103*t^36 + 373248*t^35 + 12204432*t^34 +
219691008*t^33 + 1792787904*t^32 - 15984304128*t^31 -
770855657472*t^30 - 13858803744768*t^29 - 169061744001024*t^28 -
1570120425013248*t^27 - 11594918710738944*t^26 -
69335416427249664*t^25 - 336873109192704000*t^24 -
1314710158332395520*t^23 - 3972084924558606336*t^22 -
8325423280919937024*t^21 - 6618150161627480064*t^20 +
30199927916074106880*t^19 + 144044548694514597888*t^18 +
241599423328592855040*t^17 - 423561610344158724096*t^16 -
4262616719831007756288*t^15 - 16269659850992051552256*t^14 -
43080422468235936399360*t^13 - 88309264336212197376000*t^12 -
145406907231239487356928*t^11 - 194530455712508783099904*t^10 -
210737996131672516460544*t^9 - 181528665372280567627776*t^8 -
119046217690921782214656*t^7 - 52972797420461088571392*t^6 -
8787464125321958129664*t^5 + 7884764586336652886016*t^4 +
7729690170042567622656*t^3 + 3435242212966784827392*t^2 +
840479776858391445504*t + 91927475593886564352 ], [ 0, 0, 0, 1944*t^24
+ 134784*t^23 + 4202496*t^22 + 79045632*t^21 + 1003815936*t^20 +
9035587584*t^19 + 58033373184*t^18 + 251602993152*t^17 +
535696146432*t^16 - 1611125489664*t^15 - 21095843364864*t^14 -
108621912342528*t^13 - 367214645477376*t^12 - 868975298740224*t^11 -
1350133975351296*t^10 - 824896250707968*t^9 + 2194211415785472*t^8 +
8244526879604736*t^7 + 15213100579946496*t^6 + 18949000572960768*t^5 +
16841236782514176*t^4 + 10609325135364096*t^3 + 4512395720392704*t^2 +
1157785744048128*t + 133590662774784, -40824*t^36 - 2985984*t^35 -
97635456*t^34 - 1757528064*t^33 - 14342303232*t^32 + 127874433024*t^31
+ 6166845259776*t^30 + 110870429958144*t^29 + 1352493952008192*t^28 +
12560963400105984*t^27 + 92759349685911552*t^26 +
554683331417997312*t^25 + 2694984873541632000*t^24 +
10517681266659164160*t^23 + 31776679396468850688*t^22 +
66603386247359496192*t^21 + 52945201293019840512*t^20 -
241599423328592855040*t^19 - 1152356389556116783104*t^18 -
1932795386628742840320*t^17 + 3388492882753269792768*t^16 +
34100933758648062050304*t^15 + 130157278807936412418048*t^14 +
344643379745887491194880*t^13 + 706474114689697579008000*t^12 +
1163255257849915898855424*t^11 + 1556243645700070264799232*t^10 +
1685903969053380131684352*t^9 + 1452229322978244541022208*t^8 +
952369741527374257717248*t^7 + 423782379363688708571136*t^6 +
70299713002575665037312*t^5 - 63078116690693223088128*t^4 -
61837521360340540981248*t^3 - 27481937703734278619136*t^2 -
6723838214867131564032*t - 735419804751092514816 ], [ 0, 0, 0,
216*t^16 + 10368*t^15 + 200448*t^14 + 2073600*t^13 + 11943936*t^12 +
28532736*t^11 - 97763328*t^10 - 1088225280*t^9 - 4255580160*t^8 -
8705802240*t^7 - 6256852992*t^6 + 14608760832*t^5 + 48922361856*t^4 +
67947724800*t^3 + 52546240512*t^2 + 21743271936*t + 3623878656,
1512*t^24 + 62208*t^23 + 891648*t^22 - 497664*t^21 - 211921920*t^20 -
3762339840*t^19 - 37545984000*t^18 - 247892410368*t^17 -
1105883725824*t^16 - 3052041928704*t^15 - 2647413227520*t^14 +
17094288605184*t^13 + 81972361691136*t^12 + 136754308841472*t^11 -
169434446561280*t^10 - 1562645467496448*t^9 - 4529699740975104*t^8 -
8122938502938624*t^7 - 9842454429696000*t^6 - 7890198520135680*t^5 -
3555459826974720*t^4 - 66795331387392*t^3 + 957399749885952*t^2 +
534362651099136*t + 103903848824832 ], [ 0, 0, 0, 54*t^16 + 2592*t^15
+ 50112*t^14 + 518400*t^13 + 2985984*t^12 + 7133184*t^11 -
24440832*t^10 - 272056320*t^9 - 1063895040*t^8 - 2176450560*t^7 -
1564213248*t^6 + 3652190208*t^5 + 12230590464*t^4 + 16986931200*t^3 +
13136560128*t^2 + 5435817984*t + 905969664, -189*t^24 - 7776*t^23 -
111456*t^22 + 62208*t^21 + 26490240*t^20 + 470292480*t^19 +
4693248000*t^18 + 30986551296*t^17 + 138235465728*t^16 +
381505241088*t^15 + 330926653440*t^14 - 2136786075648*t^13 -
10246545211392*t^12 - 17094288605184*t^11 + 21179305820160*t^10 +
195330683437056*t^9 + 566212467621888*t^8 + 1015367312867328*t^7 +
1230306803712000*t^6 + 986274815016960*t^5 + 444432478371840*t^4 +
8349416423424*t^3 - 119674968735744*t^2 - 66795331387392*t -
12987981103104 ], [ 0, 0, 0, -442368*t^24 - 4423680*t^22 +
10395648*t^20 + 187342848*t^18 + 485305344*t^16 + 123310080*t^14 -
527053824*t^12 + 30827520*t^10 + 30331584*t^8 + 2927232*t^6 +
40608*t^4 - 4320*t^2 - 108, -113246208*t^36 - 1698693120*t^34 -
24715984896*t^32 - 219018166272*t^30 - 893087907840*t^28 -
1158763511808*t^26 + 2157644611584*t^24 + 6587744256000*t^22 +
1604147576832*t^20 - 4765666738176*t^18 + 401036894208*t^16 +
411734016000*t^14 + 33713197056*t^12 - 4526419968*t^10 - 872156160*t^8
- 53471232*t^6 - 1508544*t^4 - 25920*t^2 - 432 ], [ 0, 0, 0,
-110592*t^24 - 1105920*t^22 + 2598912*t^20 + 46835712*t^18 +
121326336*t^16 + 30827520*t^14 - 131763456*t^12 + 7706880*t^10 +
7582896*t^8 + 731808*t^6 + 10152*t^4 - 1080*t^2 - 27, 14155776*t^36 +
212336640*t^34 + 3089498112*t^32 + 27377270784*t^30 +
111635988480*t^28 + 144845438976*t^26 - 269705576448*t^24 -
823468032000*t^22 - 200518447104*t^20 + 595708342272*t^18 -
50129611776*t^16 - 51466752000*t^14 - 4214149632*t^12 + 565802496*t^10
+ 109019520*t^8 + 6683904*t^6 + 188568*t^4 + 3240*t^2 + 54 ], [ 0, 0,
0, -6912*t^16 - 27648*t^14 + 393984*t^12 + 836352*t^10 - 1136160*t^8 +
209088*t^6 + 24624*t^4 - 432*t^2 - 27, -221184*t^24 - 1327104*t^22 -
30191616*t^20 - 112250880*t^18 + 147101184*t^16 + 488208384*t^14 -
515676672*t^12 + 122052096*t^10 + 9193824*t^8 - 1753920*t^6 -
117936*t^4 - 1296*t^2 - 54 ], [ 0, 0, 0, -27648*t^16 - 110592*t^14 +
1575936*t^12 + 3345408*t^10 - 4544640*t^8 + 836352*t^6 + 98496*t^4 -
1728*t^2 - 108, 1769472*t^24 + 10616832*t^22 + 241532928*t^20 +
898007040*t^18 - 1176809472*t^16 - 3905667072*t^14 + 4125413376*t^12 -
976416768*t^10 - 73550592*t^8 + 14031360*t^6 + 943488*t^4 + 10368*t^2
+ 432 ], [ 0, 0, 0, -110592*t^24 - 1105920*t^22 + 2598912*t^20 +
46835712*t^18 + 121326336*t^16 + 30827520*t^14 - 131763456*t^12 +
7706880*t^10 + 7582896*t^8 + 731808*t^6 + 10152*t^4 - 1080*t^2 - 27,
-14155776*t^36 - 212336640*t^34 - 3089498112*t^32 - 27377270784*t^30 -
111635988480*t^28 - 144845438976*t^26 + 269705576448*t^24 +
823468032000*t^22 + 200518447104*t^20 - 595708342272*t^18 +
50129611776*t^16 + 51466752000*t^14 + 4214149632*t^12 - 565802496*t^10
- 109019520*t^8 - 6683904*t^6 - 188568*t^4 - 3240*t^2 - 54 ], [ 0, 0,
0, -442368*t^24 - 4423680*t^22 + 10395648*t^20 + 187342848*t^18 +
485305344*t^16 + 123310080*t^14 - 527053824*t^12 + 30827520*t^10 +
30331584*t^8 + 2927232*t^6 + 40608*t^4 - 4320*t^2 - 108,
113246208*t^36 + 1698693120*t^34 + 24715984896*t^32 +
219018166272*t^30 + 893087907840*t^28 + 1158763511808*t^26 -
2157644611584*t^24 - 6587744256000*t^22 - 1604147576832*t^20 +
4765666738176*t^18 - 401036894208*t^16 - 411734016000*t^14 -
33713197056*t^12 + 4526419968*t^10 + 872156160*t^8 + 53471232*t^6 +
1508544*t^4 + 25920*t^2 + 432 ], [ 0, 0, 0, -6912*t^16 - 27648*t^14 +
393984*t^12 + 836352*t^10 - 1136160*t^8 + 209088*t^6 + 24624*t^4 -
432*t^2 - 27, 221184*t^24 + 1327104*t^22 + 30191616*t^20 +
112250880*t^18 - 147101184*t^16 - 488208384*t^14 + 515676672*t^12 -
122052096*t^10 - 9193824*t^8 + 1753920*t^6 + 117936*t^4 + 1296*t^2 +
54 ], [ 0, 0, 0, -27648*t^16 - 110592*t^14 + 1575936*t^12 +
3345408*t^10 - 4544640*t^8 + 836352*t^6 + 98496*t^4 - 1728*t^2 - 108,
-1769472*t^24 - 10616832*t^22 - 241532928*t^20 - 898007040*t^18 +
1176809472*t^16 + 3905667072*t^14 - 4125413376*t^12 + 976416768*t^10 +
73550592*t^8 - 14031360*t^6 - 943488*t^4 - 10368*t^2 - 432 ], [ 0, 0,
0, -27648*t^16 + 110592*t^14 - 82944*t^12 - 27648*t^10 + 17280*t^8 -
6912*t^6 - 5184*t^4 + 1728*t^2 - 108, 1769472*t^24 - 10616832*t^22 +
18579456*t^20 - 6193152*t^18 - 6303744*t^16 + 3981312*t^14 +
774144*t^12 + 995328*t^10 - 393984*t^8 - 96768*t^6 + 72576*t^4 -
10368*t^2 + 432 ], [ 0, 0, 0, -6912*t^16 + 27648*t^14 - 20736*t^12 -
6912*t^10 + 4320*t^8 - 1728*t^6 - 1296*t^4 + 432*t^2 - 27,
-221184*t^24 + 1327104*t^22 - 2322432*t^20 + 774144*t^18 + 787968*t^16
- 497664*t^14 - 96768*t^12 - 124416*t^10 + 49248*t^8 + 12096*t^6 -
9072*t^4 + 1296*t^2 - 54 ], [ 0, 0, 0, -442368*t^24 + 4423680*t^22 -
16146432*t^20 + 24993792*t^18 - 12358656*t^16 - 3870720*t^14 +
3787776*t^12 - 967680*t^10 - 772416*t^8 + 390528*t^6 - 63072*t^4 +
4320*t^2 - 108, 113246208*t^36 - 1698693120*t^34 + 10446962688*t^32 -
33520877568*t^30 + 58350108672*t^28 - 49347035136*t^26 +
7679508480*t^24 + 15967715328*t^22 - 9807298560*t^20 + 2416214016*t^18
- 2451824640*t^16 + 997982208*t^14 + 119992320*t^12 - 192761856*t^10 +
56982528*t^8 - 8183808*t^6 + 637632*t^4 - 25920*t^2 + 432 ], [ 0, 0,
0, -110592*t^24 + 1105920*t^22 - 4036608*t^20 + 6248448*t^18 -
3089664*t^16 - 967680*t^14 + 946944*t^12 - 241920*t^10 - 193104*t^8 +
97632*t^6 - 15768*t^4 + 1080*t^2 - 27, -14155776*t^36 + 212336640*t^34
- 1305870336*t^32 + 4190109696*t^30 - 7293763584*t^28 +
6168379392*t^26 - 959938560*t^24 - 1995964416*t^22 + 1225912320*t^20 -
302026752*t^18 + 306478080*t^16 - 124747776*t^14 - 14999040*t^12 +
24095232*t^10 - 7122816*t^8 + 1022976*t^6 - 79704*t^4 + 3240*t^2 - 54
], [ 0, 0, 0, -6912*t^16 + 27648*t^14 - 20736*t^12 - 6912*t^10 +
4320*t^8 - 1728*t^6 - 1296*t^4 + 432*t^2 - 27, 221184*t^24 -
1327104*t^22 + 2322432*t^20 - 774144*t^18 - 787968*t^16 + 497664*t^14
+ 96768*t^12 + 124416*t^10 - 49248*t^8 - 12096*t^6 + 9072*t^4 -
1296*t^2 + 54 ], [ 0, 0, 0, -27648*t^16 + 110592*t^14 - 82944*t^12 -
27648*t^10 + 17280*t^8 - 6912*t^6 - 5184*t^4 + 1728*t^2 - 108,
-1769472*t^24 + 10616832*t^22 - 18579456*t^20 + 6193152*t^18 +
6303744*t^16 - 3981312*t^14 - 774144*t^12 - 995328*t^10 + 393984*t^8 +
96768*t^6 - 72576*t^4 + 10368*t^2 - 432 ], [ 0, 0, 0, -442368*t^24 +
4423680*t^22 - 16146432*t^20 + 24993792*t^18 - 12358656*t^16 -
3870720*t^14 + 3787776*t^12 - 967680*t^10 - 772416*t^8 + 390528*t^6 -
63072*t^4 + 4320*t^2 - 108, -113246208*t^36 + 1698693120*t^34 -
10446962688*t^32 + 33520877568*t^30 - 58350108672*t^28 +
49347035136*t^26 - 7679508480*t^24 - 15967715328*t^22 +
9807298560*t^20 - 2416214016*t^18 + 2451824640*t^16 - 997982208*t^14 -
119992320*t^12 + 192761856*t^10 - 56982528*t^8 + 8183808*t^6 -
637632*t^4 + 25920*t^2 - 432 ], [ 0, 0, 0, -110592*t^24 + 1105920*t^22
- 4036608*t^20 + 6248448*t^18 - 3089664*t^16 - 967680*t^14 +
946944*t^12 - 241920*t^10 - 193104*t^8 + 97632*t^6 - 15768*t^4 +
1080*t^2 - 27, 14155776*t^36 - 212336640*t^34 + 1305870336*t^32 -
4190109696*t^30 + 7293763584*t^28 - 6168379392*t^26 + 959938560*t^24 +
1995964416*t^22 - 1225912320*t^20 + 302026752*t^18 - 306478080*t^16 +
124747776*t^14 + 14999040*t^12 - 24095232*t^10 + 7122816*t^8 -
1022976*t^6 + 79704*t^4 - 3240*t^2 + 54 ], [ 0, 0, 0, 81*t^16 +
1296*t^15 + 3456*t^14 - 42336*t^13 - 323568*t^12 - 737856*t^11 +
483840*t^10 + 5864832*t^9 + 12559968*t^8 + 11729664*t^7 + 1935360*t^6
- 5902848*t^5 - 5177088*t^4 - 1354752*t^3 + 221184*t^2 + 165888*t +
20736, 3888*t^23 + 89424*t^22 + 766368*t^21 + 2322432*t^20 -
6640704*t^19 - 71475264*t^18 - 165214080*t^17 + 315850752*t^16 +
2844495360*t^15 + 7593896448*t^14 + 9354175488*t^13 - 18708350976*t^11
- 30375585792*t^10 - 22755962880*t^9 - 5053612032*t^8 + 5286850560*t^7
+ 4574416896*t^6 + 850010112*t^5 - 594542592*t^4 - 392380416*t^3 -
91570176*t^2 - 7962624*t ], [ 0, 0, 0, 324*t^32 + 10368*t^31 +
117504*t^30 + 311040*t^29 - 5042304*t^28 - 51383808*t^27 -
145345536*t^26 + 552434688*t^25 + 5580893952*t^24 + 15302780928*t^23 -
11072802816*t^22 - 200693403648*t^21 - 601120880640*t^20 -
581783150592*t^19 + 1518611742720*t^18 + 6835446005760*t^17 +
12881597405184*t^16 + 13670892011520*t^15 + 6074446970880*t^14 -
4654265204736*t^13 - 9617934090240*t^12 - 6422188916736*t^11 -
708659380224*t^10 + 1958755958784*t^9 + 1428708851712*t^8 +
282846560256*t^7 - 148833828864*t^6 - 105234038784*t^5 -
20653277184*t^4 + 2548039680*t^3 + 1925185536*t^2 + 339738624*t +
21233664, 31104*t^47 + 1461888*t^46 + 29272320*t^45 + 308551680*t^44 +
1502489088*t^43 - 2755192320*t^42 - 86400525312*t^41 -
487524925440*t^40 - 389037496320*t^39 + 9962739431424*t^38 +
57512867622912*t^37 + 82008434147328*t^36 - 560259735330816*t^35 -
3424903253729280*t^34 - 6522010118111232*t^33 + 12232033673674752*t^32
+ 105581416085913600*t^31 + 269075419698364416*t^30 +
153703495558103040*t^29 - 1181467369902440448*t^28 -
4634555790565048320*t^27 - 9020880097990410240*t^26 -
9450729185915437056*t^25 + 18901458371830874112*t^23 +
36083520391961640960*t^22 + 37076446324520386560*t^21 +
18903477918439047168*t^20 - 4918511857859297280*t^19 -
17220826860695322624*t^18 - 13514421258996940800*t^17 -
3131400620460736512*t^16 + 3339269180472950784*t^15 +
3507100931818782720*t^14 + 1147411937957511168*t^13 -
335906546267455488*t^12 - 471145411566895104*t^11 -
163229522844450816*t^10 + 12747980679413760*t^9 +
31950433513635840*t^8 + 11324689653694464*t^7 + 722257135534080*t^6 -
787736998969344*t^5 - 323539886407680*t^4 - 61388504432640*t^3 -
6131602685952*t^2 - 260919263232*t ], [ 0, 0, 0, 324*t^16 + 5184*t^15
+ 13824*t^14 - 169344*t^13 - 1294272*t^12 - 2951424*t^11 +
1935360*t^10 + 23459328*t^9 + 50239872*t^8 + 46918656*t^7 +
7741440*t^6 - 23611392*t^5 - 20708352*t^4 - 5419008*t^3 + 884736*t^2 +
663552*t + 82944, 31104*t^23 + 715392*t^22 + 6130944*t^21 +
18579456*t^20 - 53125632*t^19 - 571802112*t^18 - 1321712640*t^17 +
2526806016*t^16 + 22755962880*t^15 + 60751171584*t^14 +
74833403904*t^13 - 149666807808*t^11 - 243004686336*t^10 -
182047703040*t^9 - 40428896256*t^8 + 42294804480*t^7 + 36595335168*t^6
+ 6800080896*t^5 - 4756340736*t^4 - 3139043328*t^3 - 732561408*t^2 -
63700992*t ], [ 0, 0, 0, 81*t^32 + 2592*t^31 + 29376*t^30 + 77760*t^29
- 1260576*t^28 - 12845952*t^27 - 36336384*t^26 + 138108672*t^25 +
1395223488*t^24 + 3825695232*t^23 - 2768200704*t^22 - 50173350912*t^21
- 150280220160*t^20 - 145445787648*t^19 + 379652935680*t^18 +
1708861501440*t^17 + 3220399351296*t^16 + 3417723002880*t^15 +
1518611742720*t^14 - 1163566301184*t^13 - 2404483522560*t^12 -
1605547229184*t^11 - 177164845056*t^10 + 489688989696*t^9 +
357177212928*t^8 + 70711640064*t^7 - 37208457216*t^6 - 26308509696*t^5
- 5163319296*t^4 + 637009920*t^3 + 481296384*t^2 + 84934656*t +
5308416, 3888*t^47 + 182736*t^46 + 3659040*t^45 + 38568960*t^44 +
187811136*t^43 - 344399040*t^42 - 10800065664*t^41 - 60940615680*t^40
- 48629687040*t^39 + 1245342428928*t^38 + 7189108452864*t^37 +
10251054268416*t^36 - 70032466916352*t^35 - 428112906716160*t^34 -
815251264763904*t^33 + 1529004209209344*t^32 + 13197677010739200*t^31
+ 33634427462295552*t^30 + 19212936944762880*t^29 -
147683421237805056*t^28 - 579319473820631040*t^27 -
1127610012248801280*t^26 - 1181341148239429632*t^25 +
2362682296478859264*t^23 + 4510440048995205120*t^22 +
4634555790565048320*t^21 + 2362934739804880896*t^20 -
614813982232412160*t^19 - 2152603357586915328*t^18 -
1689302657374617600*t^17 - 391425077557592064*t^16 +
417408647559118848*t^15 + 438387616477347840*t^14 +
143426492244688896*t^13 - 41988318283431936*t^12 -
58893176445861888*t^11 - 20403690355556352*t^10 + 1593497584926720*t^9
+ 3993804189204480*t^8 + 1415586206711808*t^7 + 90282141941760*t^6 -
98467124871168*t^5 - 40442485800960*t^4 - 7673563054080*t^3 -
766450335744*t^2 - 32614907904*t ], [ 0, 0, 0, -452984832*t^16 -
14042529792*t^14 - 25565331456*t^12 - 14219476992*t^10 -
2838896640*t^8 - 222179328*t^6 - 6241536*t^4 - 53568*t^2 - 27,
3710851743744*t^24 - 228217382240256*t^22 - 1254151925268480*t^20 -
1946776605032448*t^18 - 1331979249254400*t^16 - 444630885728256*t^14 -
76481562673152*t^12 - 6947357589504*t^10 - 325190246400*t^8 -
7426363392*t^6 - 74753280*t^4 - 212544*t^2 + 54 ], [ 0, 0, 0,
-452984832*t^16 - 14042529792*t^14 - 25565331456*t^12 -
14219476992*t^10 - 2838896640*t^8 - 222179328*t^6 - 6241536*t^4 -
53568*t^2 - 27, -3710851743744*t^24 + 228217382240256*t^22 +
1254151925268480*t^20 + 1946776605032448*t^18 + 1331979249254400*t^16
+ 444630885728256*t^14 + 76481562673152*t^12 + 6947357589504*t^10 +
325190246400*t^8 + 7426363392*t^6 + 74753280*t^4 + 212544*t^2 - 54 ],
[ 0, 0, 0, -1811939328*t^16 - 56170119168*t^14 - 102261325824*t^12 -
56877907968*t^10 - 11355586560*t^8 - 888717312*t^6 - 24966144*t^4 -
214272*t^2 - 108, 29686813949952*t^24 - 1825739057922048*t^22 -
10033215402147840*t^20 - 15574212840259584*t^18 -
10655833994035200*t^16 - 3557047085826048*t^14 - 611852501385216*t^12
- 55578860716032*t^10 - 2601521971200*t^8 - 59410907136*t^6 -
598026240*t^4 - 1700352*t^2 + 432 ], [ 0, 0, 0, -1811939328*t^16 -
56170119168*t^14 - 102261325824*t^12 - 56877907968*t^10 -
11355586560*t^8 - 888717312*t^6 - 24966144*t^4 - 214272*t^2 - 108,
-29686813949952*t^24 + 1825739057922048*t^22 + 10033215402147840*t^20
+ 15574212840259584*t^18 + 10655833994035200*t^16 +
3557047085826048*t^14 + 611852501385216*t^12 + 55578860716032*t^10 +
2601521971200*t^8 + 59410907136*t^6 + 598026240*t^4 + 1700352*t^2 -
432 ], [ 0, 0, 0, -1855425871872*t^24 - 60301340835840*t^22 -
192094559797248*t^20 - 249511293222912*t^18 - 162516009222144*t^16 -
55402309877760*t^14 - 9685466873856*t^12 - 865661091840*t^10 -
39676760064*t^8 - 951810048*t^6 - 11449728*t^4 - 56160*t^2 - 27,
972777519512027136*t^36 - 57637068031087607808*t^34 -
461689330549653504000*t^32 - 1353346324723623002112*t^30 -
2097052787973449318400*t^28 - 1950740731684831887360*t^26 -
1148018767415186817024*t^24 - 434953987064309219328*t^22 -
106067792243504185344*t^20 - 16572776343763156992*t^18 -
1657309253804752896*t^16 - 106189938248122368*t^14 -
4379344052944896*t^12 - 116273208360960*t^10 - 1953032601600*t^8 -
19693780992*t^6 - 104976000*t^4 - 204768*t^2 + 54 ], [ 0, 0, 0,
-1855425871872*t^24 - 60301340835840*t^22 - 192094559797248*t^20 -
249511293222912*t^18 - 162516009222144*t^16 - 55402309877760*t^14 -
9685466873856*t^12 - 865661091840*t^10 - 39676760064*t^8 -
951810048*t^6 - 11449728*t^4 - 56160*t^2 - 27,
-972777519512027136*t^36 + 57637068031087607808*t^34 +
461689330549653504000*t^32 + 1353346324723623002112*t^30 +
2097052787973449318400*t^28 + 1950740731684831887360*t^26 +
1148018767415186817024*t^24 + 434953987064309219328*t^22 +
106067792243504185344*t^20 + 16572776343763156992*t^18 +
1657309253804752896*t^16 + 106189938248122368*t^14 +
4379344052944896*t^12 + 116273208360960*t^10 + 1953032601600*t^8 +
19693780992*t^6 + 104976000*t^4 + 204768*t^2 - 54 ], [ 0, 0, 0,
-7421703487488*t^24 - 241205363343360*t^22 - 768378239188992*t^20 -
998045172891648*t^18 - 650064036888576*t^16 - 221609239511040*t^14 -
38741867495424*t^12 - 3462644367360*t^10 - 158707040256*t^8 -
3807240192*t^6 - 45798912*t^4 - 224640*t^2 - 108,
-7782220156096217088*t^36 + 461096544248700862464*t^34 +
3693514644397228032000*t^32 + 10826770597788984016896*t^30 +
16776422303787594547200*t^28 + 15605925853478655098880*t^26 +
9184150139321494536192*t^24 + 3479631896514473754624*t^22 +
848542337948033482752*t^20 + 132582210750105255936*t^18 +
13258474030438023168*t^16 + 849519505984978944*t^14 +
35034752423559168*t^12 + 930185666887680*t^10 + 15624260812800*t^8 +
157550247936*t^6 + 839808000*t^4 + 1638144*t^2 - 432 ], [ 0, 0, 0,
-7421703487488*t^24 - 241205363343360*t^22 - 768378239188992*t^20 -
998045172891648*t^18 - 650064036888576*t^16 - 221609239511040*t^14 -
38741867495424*t^12 - 3462644367360*t^10 - 158707040256*t^8 -
3807240192*t^6 - 45798912*t^4 - 224640*t^2 - 108,
7782220156096217088*t^36 - 461096544248700862464*t^34 -
3693514644397228032000*t^32 - 10826770597788984016896*t^30 -
16776422303787594547200*t^28 - 15605925853478655098880*t^26 -
9184150139321494536192*t^24 - 3479631896514473754624*t^22 -
848542337948033482752*t^20 - 132582210750105255936*t^18 -
13258474030438023168*t^16 - 849519505984978944*t^14 -
35034752423559168*t^12 - 930185666887680*t^10 - 15624260812800*t^8 -
157550247936*t^6 - 839808000*t^4 - 1638144*t^2 + 432 ], [ 0, 0, 0,
-31131*t^16 - 413856*t^15 - 2554848*t^14 - 9991296*t^13 -
26526528*t^12 - 52738560*t^11 - 84865536*t^10 - 77580288*t^9 -
169468416*t^8 + 310321152*t^7 - 1357848576*t^6 + 3375267840*t^5 -
6790791168*t^4 + 10231087104*t^3 - 10464657408*t^2 + 6780616704*t -
2040201216, -2114154*t^24 - 42158880*t^23 - 400357728*t^22 -
2437513344*t^21 - 10602010944*t^20 - 35081786880*t^19 -
91919826432*t^18 - 197139142656*t^17 - 344702449152*t^16 -
507370881024*t^15 - 640018464768*t^14 - 539658878976*t^13 -
1287773061120*t^12 + 2158635515904*t^11 - 10240295436288*t^10 +
32471736385536*t^9 - 88243826982912*t^8 + 201870482079744*t^7 -
376503609065472*t^6 + 574779996241920*t^5 - 694813389225984*t^4 +
638979498049536*t^3 - 419805504995328*t^2 + 176827159019520*t -
35469618315264 ], [ 0, 0, 0, -31131*t^16 - 413856*t^15 - 2554848*t^14
- 9991296*t^13 - 26526528*t^12 - 52738560*t^11 - 84865536*t^10 -
77580288*t^9 - 169468416*t^8 + 310321152*t^7 - 1357848576*t^6 +
3375267840*t^5 - 6790791168*t^4 + 10231087104*t^3 - 10464657408*t^2 +
6780616704*t - 2040201216, 2114154*t^24 + 42158880*t^23 +
400357728*t^22 + 2437513344*t^21 + 10602010944*t^20 + 35081786880*t^19
+ 91919826432*t^18 + 197139142656*t^17 + 344702449152*t^16 +
507370881024*t^15 + 640018464768*t^14 + 539658878976*t^13 +
1287773061120*t^12 - 2158635515904*t^11 + 10240295436288*t^10 -
32471736385536*t^9 + 88243826982912*t^8 - 201870482079744*t^7 +
376503609065472*t^6 - 574779996241920*t^5 + 694813389225984*t^4 -
638979498049536*t^3 + 419805504995328*t^2 - 176827159019520*t +
35469618315264 ], [ 0, 0, 0, -124524*t^16 - 1655424*t^15 -
10219392*t^14 - 39965184*t^13 - 106106112*t^12 - 210954240*t^11 -
339462144*t^10 - 310321152*t^9 - 677873664*t^8 + 1241284608*t^7 -
5431394304*t^6 + 13501071360*t^5 - 27163164672*t^4 + 40924348416*t^3 -
41858629632*t^2 + 27122466816*t - 8160804864, 16913232*t^24 +
337271040*t^23 + 3202861824*t^22 + 19500106752*t^21 + 84816087552*t^20
+ 280654295040*t^19 + 735358611456*t^18 + 1577113141248*t^17 +
2757619593216*t^16 + 4058967048192*t^15 + 5120147718144*t^14 +
4317271031808*t^13 + 10302184488960*t^12 - 17269084127232*t^11 +
81922363490304*t^10 - 259773891084288*t^9 + 705950615863296*t^8 -
1614963856637952*t^7 + 3012028872523776*t^6 - 4598239969935360*t^5 +
5558507113807872*t^4 - 5111835984396288*t^3 + 3358444039962624*t^2 -
1414617272156160*t + 283756946522112 ], [ 0, 0, 0, -124524*t^16 -
1655424*t^15 - 10219392*t^14 - 39965184*t^13 - 106106112*t^12 -
210954240*t^11 - 339462144*t^10 - 310321152*t^9 - 677873664*t^8 +
1241284608*t^7 - 5431394304*t^6 + 13501071360*t^5 - 27163164672*t^4 +
40924348416*t^3 - 41858629632*t^2 + 27122466816*t - 8160804864,
-16913232*t^24 - 337271040*t^23 - 3202861824*t^22 - 19500106752*t^21 -
84816087552*t^20 - 280654295040*t^19 - 735358611456*t^18 -
1577113141248*t^17 - 2757619593216*t^16 - 4058967048192*t^15 -
5120147718144*t^14 - 4317271031808*t^13 - 10302184488960*t^12 +
17269084127232*t^11 - 81922363490304*t^10 + 259773891084288*t^9 -
705950615863296*t^8 + 1614963856637952*t^7 - 3012028872523776*t^6 +
4598239969935360*t^5 - 5558507113807872*t^4 + 5111835984396288*t^3 -
3358444039962624*t^2 + 1414617272156160*t - 283756946522112 ], [ 0, 0,
0, -1120716*t^24 - 20875968*t^23 - 197335872*t^22 - 1218454272*t^21 -
5360591232*t^20 - 17753637888*t^19 - 45664625664*t^18 -
94683893760*t^17 - 167975285760*t^16 - 249280561152*t^15 -
401251663872*t^14 - 327865466880*t^13 - 1214252384256*t^12 +
1311461867520*t^11 - 6420026621952*t^10 + 15953955913728*t^9 -
43001673154560*t^8 + 96956307210240*t^7 - 187042306719744*t^6 +
290875603156992*t^5 - 351311706980352*t^4 + 319410476679168*t^3 -
206921659318272*t^2 + 87560156086272*t - 18802494406656,
456657264*t^36 + 12759576192*t^35 + 180029609856*t^34 +
1683639461376*t^33 + 11578847346432*t^32 + 61906260344832*t^31 +
266105578143744*t^30 + 942417547296768*t^29 + 2803299591831552*t^28 +
7138223020965888*t^27 + 15844713734209536*t^26 +
31288075397627904*t^25 + 56005863392673792*t^24 +
92221034006052864*t^23 + 143524338497224704*t^22 +
199191383922180096*t^21 + 300132618347741184*t^20 +
230696874896523264*t^19 + 781818113883635712*t^18 -
922787499586093056*t^17 + 4802121893563858944*t^16 -
12748248571019526144*t^15 + 36742230655289524224*t^14 -
94434338822198132736*t^13 + 229400016456391852032*t^12 -
512623827314735579136*t^11 + 1038399159285156151296*t^10 -
1871242335608081743872*t^9 + 2939472672804361469952*t^8 -
3952785688297023209472*t^7 + 4464510763322472136704*t^6 -
4154458806229923790848*t^5 + 3108173167393863892992*t^4 -
1807794106216243789824*t^3 + 773221286643159269376*t^2 -
219207849821840867328*t + 31381248229773410304 ], [ 0, 0, 0,
-280179*t^24 - 5218992*t^23 - 49333968*t^22 - 304613568*t^21 -
1340147808*t^20 - 4438409472*t^19 - 11416156416*t^18 -
23670973440*t^17 - 41993821440*t^16 - 62320140288*t^15 -
100312915968*t^14 - 81966366720*t^13 - 303563096064*t^12 +
327865466880*t^11 - 1605006655488*t^10 + 3988488978432*t^9 -
10750418288640*t^8 + 24239076802560*t^7 - 46760576679936*t^6 +
72718900789248*t^5 - 87827926745088*t^4 + 79852619169792*t^3 -
51730414829568*t^2 + 21890039021568*t - 4700623601664, 57082158*t^36 +
1594947024*t^35 + 22503701232*t^34 + 210454932672*t^33 +
1447355918304*t^32 + 7738282543104*t^31 + 33263197267968*t^30 +
117802193412096*t^29 + 350412448978944*t^28 + 892277877620736*t^27 +
1980589216776192*t^26 + 3911009424703488*t^25 + 7000732924084224*t^24
+ 11527629250756608*t^23 + 17940542312153088*t^22 +
24898922990272512*t^21 + 37516577293467648*t^20 +
28837109362065408*t^19 + 97727264235454464*t^18 -
115348437448261632*t^17 + 600265236695482368*t^16 -
1593531071377440768*t^15 + 4592778831911190528*t^14 -
11804292352774766592*t^13 + 28675002057048981504*t^12 -
64077978414341947392*t^11 + 129799894910644518912*t^10 -
233905291951010217984*t^9 + 367434084100545183744*t^8 -
494098211037127901184*t^7 + 558063845415309017088*t^6 -
519307350778740473856*t^5 + 388521645924232986624*t^4 -
225974263277030473728*t^3 + 96652660830394908672*t^2 -
27400981227730108416*t + 3922656028721676288 ], [ 0, 0, 0,
-280179*t^24 - 5218992*t^23 - 49333968*t^22 - 304613568*t^21 -
1340147808*t^20 - 4438409472*t^19 - 11416156416*t^18 -
23670973440*t^17 - 41993821440*t^16 - 62320140288*t^15 -
100312915968*t^14 - 81966366720*t^13 - 303563096064*t^12 +
327865466880*t^11 - 1605006655488*t^10 + 3988488978432*t^9 -
10750418288640*t^8 + 24239076802560*t^7 - 46760576679936*t^6 +
72718900789248*t^5 - 87827926745088*t^4 + 79852619169792*t^3 -
51730414829568*t^2 + 21890039021568*t - 4700623601664, -57082158*t^36
- 1594947024*t^35 - 22503701232*t^34 - 210454932672*t^33 -
1447355918304*t^32 - 7738282543104*t^31 - 33263197267968*t^30 -
117802193412096*t^29 - 350412448978944*t^28 - 892277877620736*t^27 -
1980589216776192*t^26 - 3911009424703488*t^25 - 7000732924084224*t^24
- 11527629250756608*t^23 - 17940542312153088*t^22 -
24898922990272512*t^21 - 37516577293467648*t^20 -
28837109362065408*t^19 - 97727264235454464*t^18 +
115348437448261632*t^17 - 600265236695482368*t^16 +
1593531071377440768*t^15 - 4592778831911190528*t^14 +
11804292352774766592*t^13 - 28675002057048981504*t^12 +
64077978414341947392*t^11 - 129799894910644518912*t^10 +
233905291951010217984*t^9 - 367434084100545183744*t^8 +
494098211037127901184*t^7 - 558063845415309017088*t^6 +
519307350778740473856*t^5 - 388521645924232986624*t^4 +
225974263277030473728*t^3 - 96652660830394908672*t^2 +
27400981227730108416*t - 3922656028721676288 ], [ 0, 0, 0,
-1120716*t^24 - 20875968*t^23 - 197335872*t^22 - 1218454272*t^21 -
5360591232*t^20 - 17753637888*t^19 - 45664625664*t^18 -
94683893760*t^17 - 167975285760*t^16 - 249280561152*t^15 -
401251663872*t^14 - 327865466880*t^13 - 1214252384256*t^12 +
1311461867520*t^11 - 6420026621952*t^10 + 15953955913728*t^9 -
43001673154560*t^8 + 96956307210240*t^7 - 187042306719744*t^6 +
290875603156992*t^5 - 351311706980352*t^4 + 319410476679168*t^3 -
206921659318272*t^2 + 87560156086272*t - 18802494406656,
-456657264*t^36 - 12759576192*t^35 - 180029609856*t^34 -
1683639461376*t^33 - 11578847346432*t^32 - 61906260344832*t^31 -
266105578143744*t^30 - 942417547296768*t^29 - 2803299591831552*t^28 -
7138223020965888*t^27 - 15844713734209536*t^26 -
31288075397627904*t^25 - 56005863392673792*t^24 -
92221034006052864*t^23 - 143524338497224704*t^22 -
199191383922180096*t^21 - 300132618347741184*t^20 -
230696874896523264*t^19 - 781818113883635712*t^18 +
922787499586093056*t^17 - 4802121893563858944*t^16 +
12748248571019526144*t^15 - 36742230655289524224*t^14 +
94434338822198132736*t^13 - 229400016456391852032*t^12 +
512623827314735579136*t^11 - 1038399159285156151296*t^10 +
1871242335608081743872*t^9 - 2939472672804361469952*t^8 +
3952785688297023209472*t^7 - 4464510763322472136704*t^6 +
4154458806229923790848*t^5 - 3108173167393863892992*t^4 +
1807794106216243789824*t^3 - 773221286643159269376*t^2 +
219207849821840867328*t - 31381248229773410304 ], [ 0, 0, 0, -108*t^24
+ 2160*t^22 - 15768*t^20 + 48816*t^18 - 48276*t^16 - 30240*t^14 +
59184*t^12 - 30240*t^10 - 48276*t^8 + 48816*t^6 - 15768*t^4 + 2160*t^2
- 108, -432*t^36 + 12960*t^34 - 159408*t^32 + 1022976*t^30 -
3561408*t^28 + 6023808*t^26 - 1874880*t^24 - 7796736*t^22 +
9577440*t^20 - 4719168*t^18 + 9577440*t^16 - 7796736*t^14 -
1874880*t^12 + 6023808*t^10 - 3561408*t^8 + 1022976*t^6 - 159408*t^4 +
12960*t^2 - 432 ], [ 0, 0, 0, -108*t^30 + 2376*t^28 - 20196*t^26 +
82512*t^24 - 161676*t^22 + 115128*t^20 + 71388*t^18 - 178848*t^16 +
71388*t^14 + 115128*t^12 - 161676*t^10 + 82512*t^8 - 20196*t^6 +
2376*t^4 - 108*t^2, 432*t^45 - 14256*t^43 + 199584*t^41 - 1540512*t^39
+ 7121520*t^37 - 19936368*t^35 + 31653504*t^33 - 19460736*t^31 -
21319200*t^29 + 54966816*t^27 - 60264000*t^25 + 60264000*t^23 -
54966816*t^21 + 21319200*t^19 + 19460736*t^17 - 31653504*t^15 +
19936368*t^13 - 7121520*t^11 + 1540512*t^9 - 199584*t^7 + 14256*t^5 -
432*t^3 ], [ 0, 0, 0, -27*t^24 + 540*t^22 - 3942*t^20 + 12204*t^18 -
12069*t^16 - 7560*t^14 + 14796*t^12 - 7560*t^10 - 12069*t^8 +
12204*t^6 - 3942*t^4 + 540*t^2 - 27, -54*t^36 + 1620*t^34 - 19926*t^32
+ 127872*t^30 - 445176*t^28 + 752976*t^26 - 234360*t^24 - 974592*t^22
+ 1197180*t^20 - 589896*t^18 + 1197180*t^16 - 974592*t^14 -
234360*t^12 + 752976*t^10 - 445176*t^8 + 127872*t^6 - 19926*t^4 +
1620*t^2 - 54 ], [ 0, 0, 0, -27*t^30 + 594*t^28 - 5049*t^26 +
20628*t^24 - 40419*t^22 + 28782*t^20 + 17847*t^18 - 44712*t^16 +
17847*t^14 + 28782*t^12 - 40419*t^10 + 20628*t^8 - 5049*t^6 + 594*t^4
- 27*t^2, 54*t^45 - 1782*t^43 + 24948*t^41 - 192564*t^39 + 890190*t^37
- 2492046*t^35 + 3956688*t^33 - 2432592*t^31 - 2664900*t^29 +
6870852*t^27 - 7533000*t^25 + 7533000*t^23 - 6870852*t^21 +
2664900*t^19 + 2432592*t^17 - 3956688*t^15 + 2492046*t^13 -
890190*t^11 + 192564*t^9 - 24948*t^7 + 1782*t^5 - 54*t^3 ], [ 0, 0, 0,
-108*t^24 + 2160*t^22 - 15768*t^20 + 48816*t^18 - 48276*t^16 -
30240*t^14 + 59184*t^12 - 30240*t^10 - 48276*t^8 + 48816*t^6 -
15768*t^4 + 2160*t^2 - 108, 432*t^36 - 12960*t^34 + 159408*t^32 -
1022976*t^30 + 3561408*t^28 - 6023808*t^26 + 1874880*t^24 +
7796736*t^22 - 9577440*t^20 + 4719168*t^18 - 9577440*t^16 +
7796736*t^14 + 1874880*t^12 - 6023808*t^10 + 3561408*t^8 - 1022976*t^6
+ 159408*t^4 - 12960*t^2 + 432 ], [ 0, 0, 0, -27*t^24 + 540*t^22 -
3942*t^20 + 12204*t^18 - 12069*t^16 - 7560*t^14 + 14796*t^12 -
7560*t^10 - 12069*t^8 + 12204*t^6 - 3942*t^4 + 540*t^2 - 27, 54*t^36 -
1620*t^34 + 19926*t^32 - 127872*t^30 + 445176*t^28 - 752976*t^26 +
234360*t^24 + 974592*t^22 - 1197180*t^20 + 589896*t^18 - 1197180*t^16
+ 974592*t^14 + 234360*t^12 - 752976*t^10 + 445176*t^8 - 127872*t^6 +
19926*t^4 - 1620*t^2 + 54 ], [ 0, 0, 0, -108*t^16 + 864*t^14 -
1296*t^12 - 864*t^10 + 1080*t^8 - 864*t^6 - 1296*t^4 + 864*t^2 - 108,
-432*t^24 + 5184*t^22 - 18144*t^20 + 12096*t^18 + 24624*t^16 -
31104*t^14 - 12096*t^12 - 31104*t^10 + 24624*t^8 + 12096*t^6 -
18144*t^4 + 5184*t^2 - 432 ], [ 0, 0, 0, -108*t^22 + 1080*t^20 -
3132*t^18 + 2592*t^16 + 1512*t^14 - 3888*t^12 + 1512*t^10 + 2592*t^8 -
3132*t^6 + 1080*t^4 - 108*t^2, 432*t^33 - 6480*t^31 + 34992*t^29 -
82512*t^27 + 71280*t^25 + 50544*t^23 - 142992*t^21 + 112752*t^19 -
112752*t^17 + 142992*t^15 - 50544*t^13 - 71280*t^11 + 82512*t^9 -
34992*t^7 + 6480*t^5 - 432*t^3 ], [ 0, 0, 0, -27*t^16 + 216*t^14 -
324*t^12 - 216*t^10 + 270*t^8 - 216*t^6 - 324*t^4 + 216*t^2 - 27,
-54*t^24 + 648*t^22 - 2268*t^20 + 1512*t^18 + 3078*t^16 - 3888*t^14 -
1512*t^12 - 3888*t^10 + 3078*t^8 + 1512*t^6 - 2268*t^4 + 648*t^2 - 54
], [ 0, 0, 0, -27*t^22 + 270*t^20 - 783*t^18 + 648*t^16 + 378*t^14 -
972*t^12 + 378*t^10 + 648*t^8 - 783*t^6 + 270*t^4 - 27*t^2, 54*t^33 -
810*t^31 + 4374*t^29 - 10314*t^27 + 8910*t^25 + 6318*t^23 - 17874*t^21
+ 14094*t^19 - 14094*t^17 + 17874*t^15 - 6318*t^13 - 8910*t^11 +
10314*t^9 - 4374*t^7 + 810*t^5 - 54*t^3 ], [ 0, 0, 0, -108*t^16 +
864*t^14 - 1296*t^12 - 864*t^10 + 1080*t^8 - 864*t^6 - 1296*t^4 +
864*t^2 - 108, 432*t^24 - 5184*t^22 + 18144*t^20 - 12096*t^18 -
24624*t^16 + 31104*t^14 + 12096*t^12 + 31104*t^10 - 24624*t^8 -
12096*t^6 + 18144*t^4 - 5184*t^2 + 432 ], [ 0, 0, 0, -27*t^16 +
216*t^14 - 324*t^12 - 216*t^10 + 270*t^8 - 216*t^6 - 324*t^4 + 216*t^2
- 27, 54*t^24 - 648*t^22 + 2268*t^20 - 1512*t^18 - 3078*t^16 +
3888*t^14 + 1512*t^12 + 3888*t^10 - 3078*t^8 - 1512*t^6 + 2268*t^4 -
648*t^2 + 54 ], [ 0, 0, 0, -27*t^28 + 108*t^27 + 162*t^26 - 972*t^25 +
135*t^24 + 1944*t^23 - 972*t^22 + 1512*t^21 - 675*t^20 - 6156*t^19 +
1566*t^18 + 1836*t^17 + 567*t^16 + 6480*t^15 - 1512*t^14 - 6480*t^13 +
567*t^12 - 1836*t^11 + 1566*t^10 + 6156*t^9 - 675*t^8 - 1512*t^7 -
972*t^6 - 1944*t^5 + 135*t^4 + 972*t^3 + 162*t^2 - 108*t - 27, 54*t^42
- 324*t^41 - 162*t^40 + 4104*t^39 - 4212*t^38 - 15552*t^37 +
25164*t^36 + 9720*t^35 - 24138*t^34 + 47412*t^33 - 86994*t^32 -
80352*t^31 + 162864*t^30 - 20736*t^29 + 97200*t^28 + 101088*t^27 -
288036*t^26 - 36936*t^25 - 84564*t^24 - 6480*t^23 + 202824*t^22 +
202824*t^20 + 6480*t^19 - 84564*t^18 + 36936*t^17 - 288036*t^16 -
101088*t^15 + 97200*t^14 + 20736*t^13 + 162864*t^12 + 80352*t^11 -
86994*t^10 - 47412*t^9 - 24138*t^8 - 9720*t^7 + 25164*t^6 + 15552*t^5
- 4212*t^4 - 4104*t^3 - 162*t^2 + 324*t + 54 ], [ 0, 0, 0, -27*t^26 -
108*t^25 + 108*t^24 + 756*t^23 + 378*t^22 - 324*t^21 - 324*t^20 -
2916*t^19 - 1701*t^18 + 648*t^17 - 1512*t^16 + 2376*t^15 - 756*t^14 -
2376*t^13 - 1512*t^12 - 648*t^11 - 1701*t^10 + 2916*t^9 - 324*t^8 +
324*t^7 + 378*t^6 - 756*t^5 + 108*t^4 + 108*t^3 - 27*t^2, 54*t^39 +
324*t^38 - 3132*t^36 - 4374*t^35 + 5184*t^34 + 12096*t^33 + 15552*t^32
+ 25272*t^31 - 19440*t^30 - 51840*t^29 - 19440*t^28 - 56376*t^27 +
36288*t^26 + 108864*t^25 + 46656*t^24 + 155844*t^23 + 48600*t^22 +
48600*t^20 - 155844*t^19 + 46656*t^18 - 108864*t^17 + 36288*t^16 +
56376*t^15 - 19440*t^14 + 51840*t^13 - 19440*t^12 - 25272*t^11 +
15552*t^10 - 12096*t^9 + 5184*t^8 + 4374*t^7 - 3132*t^6 + 324*t^4 -
54*t^3 ], [ 0, 0, 0, -108*t^28 + 432*t^27 + 648*t^26 - 3888*t^25 +
540*t^24 + 7776*t^23 - 3888*t^22 + 6048*t^21 - 2700*t^20 - 24624*t^19
+ 6264*t^18 + 7344*t^17 + 2268*t^16 + 25920*t^15 - 6048*t^14 -
25920*t^13 + 2268*t^12 - 7344*t^11 + 6264*t^10 + 24624*t^9 - 2700*t^8
- 6048*t^7 - 3888*t^6 - 7776*t^5 + 540*t^4 + 3888*t^3 + 648*t^2 -
432*t - 108, -432*t^42 + 2592*t^41 + 1296*t^40 - 32832*t^39 +
33696*t^38 + 124416*t^37 - 201312*t^36 - 77760*t^35 + 193104*t^34 -
379296*t^33 + 695952*t^32 + 642816*t^31 - 1302912*t^30 + 165888*t^29 -
777600*t^28 - 808704*t^27 + 2304288*t^26 + 295488*t^25 + 676512*t^24 +
51840*t^23 - 1622592*t^22 - 1622592*t^20 - 51840*t^19 + 676512*t^18 -
295488*t^17 + 2304288*t^16 + 808704*t^15 - 777600*t^14 - 165888*t^13 -
1302912*t^12 - 642816*t^11 + 695952*t^10 + 379296*t^9 + 193104*t^8 +
77760*t^7 - 201312*t^6 - 124416*t^5 + 33696*t^4 + 32832*t^3 + 1296*t^2
- 2592*t - 432 ], [ 0, 0, 0, -108*t^26 - 432*t^25 + 432*t^24 +
3024*t^23 + 1512*t^22 - 1296*t^21 - 1296*t^20 - 11664*t^19 - 6804*t^18
+ 2592*t^17 - 6048*t^16 + 9504*t^15 - 3024*t^14 - 9504*t^13 -
6048*t^12 - 2592*t^11 - 6804*t^10 + 11664*t^9 - 1296*t^8 + 1296*t^7 +
1512*t^6 - 3024*t^5 + 432*t^4 + 432*t^3 - 108*t^2, 432*t^39 +
2592*t^38 - 25056*t^36 - 34992*t^35 + 41472*t^34 + 96768*t^33 +
124416*t^32 + 202176*t^31 - 155520*t^30 - 414720*t^29 - 155520*t^28 -
451008*t^27 + 290304*t^26 + 870912*t^25 + 373248*t^24 + 1246752*t^23 +
388800*t^22 + 388800*t^20 - 1246752*t^19 + 373248*t^18 - 870912*t^17 +
290304*t^16 + 451008*t^15 - 155520*t^14 + 414720*t^13 - 155520*t^12 -
202176*t^11 + 124416*t^10 - 96768*t^9 + 41472*t^8 + 34992*t^7 -
25056*t^6 + 2592*t^4 - 432*t^3 ], [ 0, 0, 0, -27*t^28 + 108*t^27 +
162*t^26 - 972*t^25 + 135*t^24 + 1944*t^23 - 972*t^22 + 1512*t^21 -
675*t^20 - 6156*t^19 + 1566*t^18 + 1836*t^17 + 567*t^16 + 6480*t^15 -
1512*t^14 - 6480*t^13 + 567*t^12 - 1836*t^11 + 1566*t^10 + 6156*t^9 -
675*t^8 - 1512*t^7 - 972*t^6 - 1944*t^5 + 135*t^4 + 972*t^3 + 162*t^2
- 108*t - 27, -54*t^42 + 324*t^41 + 162*t^40 - 4104*t^39 + 4212*t^38 +
15552*t^37 - 25164*t^36 - 9720*t^35 + 24138*t^34 - 47412*t^33 +
86994*t^32 + 80352*t^31 - 162864*t^30 + 20736*t^29 - 97200*t^28 -
101088*t^27 + 288036*t^26 + 36936*t^25 + 84564*t^24 + 6480*t^23 -
202824*t^22 - 202824*t^20 - 6480*t^19 + 84564*t^18 - 36936*t^17 +
288036*t^16 + 101088*t^15 - 97200*t^14 - 20736*t^13 - 162864*t^12 -
80352*t^11 + 86994*t^10 + 47412*t^9 + 24138*t^8 + 9720*t^7 - 25164*t^6
- 15552*t^5 + 4212*t^4 + 4104*t^3 + 162*t^2 - 324*t - 54 ], [ 0, 0, 0,
-108*t^28 + 432*t^27 + 648*t^26 - 3888*t^25 + 540*t^24 + 7776*t^23 -
3888*t^22 + 6048*t^21 - 2700*t^20 - 24624*t^19 + 6264*t^18 + 7344*t^17
+ 2268*t^16 + 25920*t^15 - 6048*t^14 - 25920*t^13 + 2268*t^12 -
7344*t^11 + 6264*t^10 + 24624*t^9 - 2700*t^8 - 6048*t^7 - 3888*t^6 -
7776*t^5 + 540*t^4 + 3888*t^3 + 648*t^2 - 432*t - 108, 432*t^42 -
2592*t^41 - 1296*t^40 + 32832*t^39 - 33696*t^38 - 124416*t^37 +
201312*t^36 + 77760*t^35 - 193104*t^34 + 379296*t^33 - 695952*t^32 -
642816*t^31 + 1302912*t^30 - 165888*t^29 + 777600*t^28 + 808704*t^27 -
2304288*t^26 - 295488*t^25 - 676512*t^24 - 51840*t^23 + 1622592*t^22 +
1622592*t^20 + 51840*t^19 - 676512*t^18 + 295488*t^17 - 2304288*t^16 -
808704*t^15 + 777600*t^14 + 165888*t^13 + 1302912*t^12 + 642816*t^11 -
695952*t^10 - 379296*t^9 - 193104*t^8 - 77760*t^7 + 201312*t^6 +
124416*t^5 - 33696*t^4 - 32832*t^3 - 1296*t^2 + 2592*t + 432 ], [ 0,
0, 0, -108*t^26 + 432*t^25 + 432*t^24 - 3024*t^23 + 1512*t^22 +
1296*t^21 - 1296*t^20 + 11664*t^19 - 6804*t^18 - 2592*t^17 - 6048*t^16
- 9504*t^15 - 3024*t^14 + 9504*t^13 - 6048*t^12 + 2592*t^11 -
6804*t^10 - 11664*t^9 - 1296*t^8 - 1296*t^7 + 1512*t^6 + 3024*t^5 +
432*t^4 - 432*t^3 - 108*t^2, 432*t^39 - 2592*t^38 + 25056*t^36 -
34992*t^35 - 41472*t^34 + 96768*t^33 - 124416*t^32 + 202176*t^31 +
155520*t^30 - 414720*t^29 + 155520*t^28 - 451008*t^27 - 290304*t^26 +
870912*t^25 - 373248*t^24 + 1246752*t^23 - 388800*t^22 - 388800*t^20 -
1246752*t^19 - 373248*t^18 - 870912*t^17 - 290304*t^16 + 451008*t^15 +
155520*t^14 + 414720*t^13 + 155520*t^12 - 202176*t^11 - 124416*t^10 -
96768*t^9 - 41472*t^8 + 34992*t^7 + 25056*t^6 - 2592*t^4 - 432*t^3 ],
[ 0, 0, 0, -27*t^26 + 108*t^25 + 108*t^24 - 756*t^23 + 378*t^22 +
324*t^21 - 324*t^20 + 2916*t^19 - 1701*t^18 - 648*t^17 - 1512*t^16 -
2376*t^15 - 756*t^14 + 2376*t^13 - 1512*t^12 + 648*t^11 - 1701*t^10 -
2916*t^9 - 324*t^8 - 324*t^7 + 378*t^6 + 756*t^5 + 108*t^4 - 108*t^3 -
27*t^2, 54*t^39 - 324*t^38 + 3132*t^36 - 4374*t^35 - 5184*t^34 +
12096*t^33 - 15552*t^32 + 25272*t^31 + 19440*t^30 - 51840*t^29 +
19440*t^28 - 56376*t^27 - 36288*t^26 + 108864*t^25 - 46656*t^24 +
155844*t^23 - 48600*t^22 - 48600*t^20 - 155844*t^19 - 46656*t^18 -
108864*t^17 - 36288*t^16 + 56376*t^15 + 19440*t^14 + 51840*t^13 +
19440*t^12 - 25272*t^11 - 15552*t^10 - 12096*t^9 - 5184*t^8 + 4374*t^7
+ 3132*t^6 - 324*t^4 - 54*t^3 ], [ 0, 0, 0, -110592*t^24 +
1105920*t^22 + 2598912*t^20 - 46835712*t^18 + 121326336*t^16 -
30827520*t^14 - 131763456*t^12 - 7706880*t^10 + 7582896*t^8 -
731808*t^6 + 10152*t^4 + 1080*t^2 - 27, -14155776*t^36 +
212336640*t^34 - 3089498112*t^32 + 27377270784*t^30 -
111635988480*t^28 + 144845438976*t^26 + 269705576448*t^24 -
823468032000*t^22 + 200518447104*t^20 + 595708342272*t^18 +
50129611776*t^16 - 51466752000*t^14 + 4214149632*t^12 + 565802496*t^10
- 109019520*t^8 + 6683904*t^6 - 188568*t^4 + 3240*t^2 - 54 ], [ 0, 0,
0, -6912*t^16 + 27648*t^14 + 393984*t^12 - 836352*t^10 - 1136160*t^8 -
209088*t^6 + 24624*t^4 + 432*t^2 - 27, 221184*t^24 - 1327104*t^22 +
30191616*t^20 - 112250880*t^18 - 147101184*t^16 + 488208384*t^14 +
515676672*t^12 + 122052096*t^10 - 9193824*t^8 - 1753920*t^6 +
117936*t^4 - 1296*t^2 + 54 ], [ 0, 0, 0, -27648*t^16 + 110592*t^14 +
1575936*t^12 - 3345408*t^10 - 4544640*t^8 - 836352*t^6 + 98496*t^4 +
1728*t^2 - 108, 1769472*t^24 - 10616832*t^22 + 241532928*t^20 -
898007040*t^18 - 1176809472*t^16 + 3905667072*t^14 + 4125413376*t^12 +
976416768*t^10 - 73550592*t^8 - 14031360*t^6 + 943488*t^4 - 10368*t^2
+ 432 ], [ 0, 0, 0, -442368*t^24 + 4423680*t^22 + 10395648*t^20 -
187342848*t^18 + 485305344*t^16 - 123310080*t^14 - 527053824*t^12 -
30827520*t^10 + 30331584*t^8 - 2927232*t^6 + 40608*t^4 + 4320*t^2 -
108, -113246208*t^36 + 1698693120*t^34 - 24715984896*t^32 +
219018166272*t^30 - 893087907840*t^28 + 1158763511808*t^26 +
2157644611584*t^24 - 6587744256000*t^22 + 1604147576832*t^20 +
4765666738176*t^18 + 401036894208*t^16 - 411734016000*t^14 +
33713197056*t^12 + 4526419968*t^10 - 872156160*t^8 + 53471232*t^6 -
1508544*t^4 + 25920*t^2 - 432 ], [ 0, 0, 0, -442368*t^24 +
4423680*t^22 + 10395648*t^20 - 187342848*t^18 + 485305344*t^16 -
123310080*t^14 - 527053824*t^12 - 30827520*t^10 + 30331584*t^8 -
2927232*t^6 + 40608*t^4 + 4320*t^2 - 108, 113246208*t^36 -
1698693120*t^34 + 24715984896*t^32 - 219018166272*t^30 +
893087907840*t^28 - 1158763511808*t^26 - 2157644611584*t^24 +
6587744256000*t^22 - 1604147576832*t^20 - 4765666738176*t^18 -
401036894208*t^16 + 411734016000*t^14 - 33713197056*t^12 -
4526419968*t^10 + 872156160*t^8 - 53471232*t^6 + 1508544*t^4 -
25920*t^2 + 432 ], [ 0, 0, 0, -27648*t^16 + 110592*t^14 + 1575936*t^12
- 3345408*t^10 - 4544640*t^8 - 836352*t^6 + 98496*t^4 + 1728*t^2 -
108, -1769472*t^24 + 10616832*t^22 - 241532928*t^20 + 898007040*t^18 +
1176809472*t^16 - 3905667072*t^14 - 4125413376*t^12 - 976416768*t^10 +
73550592*t^8 + 14031360*t^6 - 943488*t^4 + 10368*t^2 - 432 ], [ 0, 0,
0, -110592*t^24 + 1105920*t^22 + 2598912*t^20 - 46835712*t^18 +
121326336*t^16 - 30827520*t^14 - 131763456*t^12 - 7706880*t^10 +
7582896*t^8 - 731808*t^6 + 10152*t^4 + 1080*t^2 - 27, 14155776*t^36 -
212336640*t^34 + 3089498112*t^32 - 27377270784*t^30 +
111635988480*t^28 - 144845438976*t^26 - 269705576448*t^24 +
823468032000*t^22 - 200518447104*t^20 - 595708342272*t^18 -
50129611776*t^16 + 51466752000*t^14 - 4214149632*t^12 - 565802496*t^10
+ 109019520*t^8 - 6683904*t^6 + 188568*t^4 - 3240*t^2 + 54 ], [ 0, 0,
0, -6912*t^16 + 27648*t^14 + 393984*t^12 - 836352*t^10 - 1136160*t^8 -
209088*t^6 + 24624*t^4 + 432*t^2 - 27, -221184*t^24 + 1327104*t^22 -
30191616*t^20 + 112250880*t^18 + 147101184*t^16 - 488208384*t^14 -
515676672*t^12 - 122052096*t^10 + 9193824*t^8 + 1753920*t^6 -
117936*t^4 + 1296*t^2 - 54 ], [ 0, 0, 0, 231928233984*t^32 -
5566277615616*t^31 + 30614526885888*t^30 + 37572373905408*t^29 -
517779782369280*t^28 - 7305739370496*t^27 + 3287988591132672*t^26 -
47400332820480*t^25 - 9525962987274240*t^24 - 16285710680064*t^23 +
14613511736918016*t^22 + 9126738395136*t^21 - 12784184586141696*t^20 +
6782541889536*t^19 + 6673860636180480*t^18 + 1039260450816*t^17 -
2127098747289600*t^16 - 259815112704*t^15 + 417116289761280*t^14 -
105977217024*t^13 - 49938221039616*t^12 - 8912830464*t^11 +
3567751888896*t^10 + 994000896*t^9 - 145354659840*t^8 + 180817920*t^7
+ 3135670272*t^6 + 1741824*t^5 - 30862080*t^4 - 559872*t^3 +
114048*t^2 + 5184*t + 54, -151996487423754240*t^48 +
2188749418902061056*t^47 - 3830311483078606848*t^46 -
80436541144650743808*t^45 + 369465461805290618880*t^44 +
695064737340085764096*t^43 - 5849558219117882769408*t^42 -
2021959873339789934592*t^41 + 44782252713077140094976*t^40 +
1732660208935926497280*t^39 - 194180518355667026706432*t^38 +
805124206411068211200*t^37 + 517071234623214194786304*t^36 -
1237789629566272143360*t^35 - 894895700817073636638720*t^34 -
221919339504180658176*t^33 + 1045933513852795748352000*t^32 +
335602522098260508672*t^31 - 847677266770429269245952*t^30 +
61961770725707612160*t^29 + 485184263414926641463296*t^28 -
41797859093099053056*t^27 - 198484929953037721534464*t^26 -
9477421538551529472*t^25 + 58430586996037936742400*t^24 +
2369355384637882368*t^23 - 12405308122064857595904*t^22 +
653091548329672704*t^21 + 1895251028964557193216*t^20 -
60509541724323840*t^19 - 206952457707624333312*t^18 -
20483552374161408*t^17 + 15959678861279232000*t^16 +
846555097595904*t^15 - 853439045731614720*t^14 + 295112044707840*t^13
+ 30819847263289344*t^12 - 11997285580800*t^11 - 723378801180672*t^10
- 1613665566720*t^9 + 10426680723456*t^8 + 117693554688*t^7 -
85122275328*t^6 - 2528630784*t^5 + 336026880*t^4 + 18289152*t^3 -
217728*t^2 - 31104*t - 540 ], [ 0, 0, 0, 927712935936*t^32 -
22265110462464*t^31 + 122458107543552*t^30 + 150289495621632*t^29 -
2071119129477120*t^28 - 29222957481984*t^27 + 13151954364530688*t^26 -
189601331281920*t^25 - 38103851949096960*t^24 - 65142842720256*t^23 +
58454046947672064*t^22 + 36506953580544*t^21 - 51136738344566784*t^20
+ 27130167558144*t^19 + 26695442544721920*t^18 + 4157041803264*t^17 -
8508394989158400*t^16 - 1039260450816*t^15 + 1668465159045120*t^14 -
423908868096*t^13 - 199752884158464*t^12 - 35651321856*t^11 +
14271007555584*t^10 + 3976003584*t^9 - 581418639360*t^8 +
723271680*t^7 + 12542681088*t^6 + 6967296*t^5 - 123448320*t^4 -
2239488*t^3 + 456192*t^2 + 20736*t + 216, 1215971899390033920*t^48 -
17509995351216488448*t^47 + 30642491864628854784*t^46 +
643492329157205950464*t^45 - 2955723694442324951040*t^44 -
5560517898720686112768*t^43 + 46796465752943062155264*t^42 +
16175678986718319476736*t^41 - 358258021704617120759808*t^40 -
13861281671487411978240*t^39 + 1553444146845336213651456*t^38 -
6440993651288545689600*t^37 - 4136569876985713558290432*t^36 +
9902317036530177146880*t^35 + 7159165606536589093109760*t^34 +
1775354716033445265408*t^33 - 8367468110822365986816000*t^32 -
2684820176786084069376*t^31 + 6781418134163434153967616*t^30 -
495694165805660897280*t^29 - 3881474107319413131706368*t^28 +
334382872744792424448*t^27 + 1587879439624301772275712*t^26 +
75819372308412235776*t^25 - 467444695968303493939200*t^24 -
18954843077103058944*t^23 + 99242464976518860767232*t^22 -
5224732386637381632*t^21 - 15162008231716457545728*t^20 +
484076333794590720*t^19 + 1655619661660994666496*t^18 +
163868418993291264*t^17 - 127677430890233856000*t^16 -
6772440780767232*t^15 + 6827512365852917760*t^14 -
2360896357662720*t^13 - 246558778106314752*t^12 + 95978284646400*t^11
+ 5787030409445376*t^10 + 12909324533760*t^9 - 83413445787648*t^8 -
941548437504*t^7 + 680978202624*t^6 + 20229046272*t^5 - 2688215040*t^4
- 146313216*t^3 + 1741824*t^2 + 248832*t + 4320 ], [ 0, 0, 0,
927712935936*t^32 - 22265110462464*t^31 + 122458107543552*t^30 +
150289495621632*t^29 - 2071119129477120*t^28 - 29222957481984*t^27 +
13151954364530688*t^26 - 189601331281920*t^25 - 38103851949096960*t^24
- 65142842720256*t^23 + 58454046947672064*t^22 + 36506953580544*t^21 -
51136738344566784*t^20 + 27130167558144*t^19 + 26695442544721920*t^18
+ 4157041803264*t^17 - 8508394989158400*t^16 - 1039260450816*t^15 +
1668465159045120*t^14 - 423908868096*t^13 - 199752884158464*t^12 -
35651321856*t^11 + 14271007555584*t^10 + 3976003584*t^9 -
581418639360*t^8 + 723271680*t^7 + 12542681088*t^6 + 6967296*t^5 -
123448320*t^4 - 2239488*t^3 + 456192*t^2 + 20736*t + 216,
-1215971899390033920*t^48 + 17509995351216488448*t^47 -
30642491864628854784*t^46 - 643492329157205950464*t^45 +
2955723694442324951040*t^44 + 5560517898720686112768*t^43 -
46796465752943062155264*t^42 - 16175678986718319476736*t^41 +
358258021704617120759808*t^40 + 13861281671487411978240*t^39 -
1553444146845336213651456*t^38 + 6440993651288545689600*t^37 +
4136569876985713558290432*t^36 - 9902317036530177146880*t^35 -
7159165606536589093109760*t^34 - 1775354716033445265408*t^33 +
8367468110822365986816000*t^32 + 2684820176786084069376*t^31 -
6781418134163434153967616*t^30 + 495694165805660897280*t^29 +
3881474107319413131706368*t^28 - 334382872744792424448*t^27 -
1587879439624301772275712*t^26 - 75819372308412235776*t^25 +
467444695968303493939200*t^24 + 18954843077103058944*t^23 -
99242464976518860767232*t^22 + 5224732386637381632*t^21 +
15162008231716457545728*t^20 - 484076333794590720*t^19 -
1655619661660994666496*t^18 - 163868418993291264*t^17 +
127677430890233856000*t^16 + 6772440780767232*t^15 -
6827512365852917760*t^14 + 2360896357662720*t^13 +
246558778106314752*t^12 - 95978284646400*t^11 - 5787030409445376*t^10
- 12909324533760*t^9 + 83413445787648*t^8 + 941548437504*t^7 -
680978202624*t^6 - 20229046272*t^5 + 2688215040*t^4 + 146313216*t^3 -
1741824*t^2 - 248832*t - 4320 ], [ 0, 0, 0, 231928233984*t^32 -
5566277615616*t^31 + 30614526885888*t^30 + 37572373905408*t^29 -
517779782369280*t^28 - 7305739370496*t^27 + 3287988591132672*t^26 -
47400332820480*t^25 - 9525962987274240*t^24 - 16285710680064*t^23 +
14613511736918016*t^22 + 9126738395136*t^21 - 12784184586141696*t^20 +
6782541889536*t^19 + 6673860636180480*t^18 + 1039260450816*t^17 -
2127098747289600*t^16 - 259815112704*t^15 + 417116289761280*t^14 -
105977217024*t^13 - 49938221039616*t^12 - 8912830464*t^11 +
3567751888896*t^10 + 994000896*t^9 - 145354659840*t^8 + 180817920*t^7
+ 3135670272*t^6 + 1741824*t^5 - 30862080*t^4 - 559872*t^3 +
114048*t^2 + 5184*t + 54, 151996487423754240*t^48 -
2188749418902061056*t^47 + 3830311483078606848*t^46 +
80436541144650743808*t^45 - 369465461805290618880*t^44 -
695064737340085764096*t^43 + 5849558219117882769408*t^42 +
2021959873339789934592*t^41 - 44782252713077140094976*t^40 -
1732660208935926497280*t^39 + 194180518355667026706432*t^38 -
805124206411068211200*t^37 - 517071234623214194786304*t^36 +
1237789629566272143360*t^35 + 894895700817073636638720*t^34 +
221919339504180658176*t^33 - 1045933513852795748352000*t^32 -
335602522098260508672*t^31 + 847677266770429269245952*t^30 -
61961770725707612160*t^29 - 485184263414926641463296*t^28 +
41797859093099053056*t^27 + 198484929953037721534464*t^26 +
9477421538551529472*t^25 - 58430586996037936742400*t^24 -
2369355384637882368*t^23 + 12405308122064857595904*t^22 -
653091548329672704*t^21 - 1895251028964557193216*t^20 +
60509541724323840*t^19 + 206952457707624333312*t^18 +
20483552374161408*t^17 - 15959678861279232000*t^16 -
846555097595904*t^15 + 853439045731614720*t^14 - 295112044707840*t^13
- 30819847263289344*t^12 + 11997285580800*t^11 + 723378801180672*t^10
+ 1613665566720*t^9 - 10426680723456*t^8 - 117693554688*t^7 +
85122275328*t^6 + 2528630784*t^5 - 336026880*t^4 - 18289152*t^3 +
217728*t^2 + 31104*t + 540 ], [ 0, 0, 0, -189*t^32 - 3456*t^31 -
9504*t^30 + 314496*t^29 + 4009824*t^28 + 16740864*t^27 - 57891456*t^26
- 1159156224*t^25 - 7549260480*t^24 - 30445811712*t^23 -
86535959040*t^22 - 190095316992*t^21 - 375246079488*t^20 -
787829317632*t^19 - 1697577854976*t^18 - 3094579814400*t^17 -
4053122330112*t^16 - 3030202220544*t^15 + 355184123904*t^14 +
4076722814976*t^13 + 5345880809472*t^12 + 3577847611392*t^11 +
770395373568*t^10 - 974770274304*t^9 - 1266722979840*t^8 -
866659074048*t^7 - 440531288064*t^6 - 183614570496*t^5 -
62395121664*t^4 - 16024338432*t^3 - 2788687872*t^2 - 283115520*t -
12386304, 918*t^48 + 28512*t^47 + 222912*t^46 - 2403648*t^45 -
57788640*t^44 - 371952000*t^43 + 833317632*t^42 + 31569585408*t^41 +
248514630336*t^40 + 843453319680*t^39 - 2101305203712*t^38 -
47805188041728*t^37 - 359010401637888*t^36 - 1833245944621056*t^35 -
7066909877760000*t^34 - 20756908241178624*t^33 -
43739276560233984*t^32 - 49053453009960960*t^31 +
63180001798225920*t^30 + 489459682266808320*t^29 +
1422505823720325120*t^28 + 2772097602494201856*t^27 +
3905829301768814592*t^26 + 3838006907502723072*t^25 +
2001035405994393600*t^24 - 1052720510068850688*t^23 -
3914126315685937152*t^22 - 5412319688935342080*t^21 -
5306570353426563072*t^20 - 3922969766893977600*t^19 -
1612418743976067072*t^18 + 1077987637582626816*t^17 +
3121731360134922240*t^16 + 3569569567315329024*t^15 +
2450904811265064960*t^14 + 778430446093467648*t^13 -
395078547957350400*t^12 - 740449871814721536*t^11 -
564186735307653120*t^10 - 282303556013260800*t^9 -
98734905835388928*t^8 - 22914840196546560*t^7 - 2500883958988800*t^6 +
423303453868032*t^5 + 252507334901760*t^4 + 55883832754176*t^3 +
7066563379200*t^2 + 500095254528*t + 15401484288 ], [ 0, 0, 0,
-756*t^32 - 13824*t^31 - 38016*t^30 + 1257984*t^29 + 16039296*t^28 +
66963456*t^27 - 231565824*t^26 - 4636624896*t^25 - 30197041920*t^24 -
121783246848*t^23 - 346143836160*t^22 - 760381267968*t^21 -
1500984317952*t^20 - 3151317270528*t^19 - 6790311419904*t^18 -
12378319257600*t^17 - 16212489320448*t^16 - 12120808882176*t^15 +
1420736495616*t^14 + 16306891259904*t^13 + 21383523237888*t^12 +
14311390445568*t^11 + 3081581494272*t^10 - 3899081097216*t^9 -
5066891919360*t^8 - 3466636296192*t^7 - 1762125152256*t^6 -
734458281984*t^5 - 249580486656*t^4 - 64097353728*t^3 -
11154751488*t^2 - 1132462080*t - 49545216, -7344*t^48 - 228096*t^47 -
1783296*t^46 + 19229184*t^45 + 462309120*t^44 + 2975616000*t^43 -
6666541056*t^42 - 252556683264*t^41 - 1988117042688*t^40 -
6747626557440*t^39 + 16810441629696*t^38 + 382441504333824*t^37 +
2872083213103104*t^36 + 14665967556968448*t^35 +
56535279022080000*t^34 + 166055265929428992*t^33 +
349914212481871872*t^32 + 392427624079687680*t^31 -
505440014385807360*t^30 - 3915677458134466560*t^29 -
11380046589762600960*t^28 - 22176780819953614848*t^27 -
31246634414150516736*t^26 - 30704055260021784576*t^25 -
16008283247955148800*t^24 + 8421764080550805504*t^23 +
31313010525487497216*t^22 + 43298557511482736640*t^21 +
42452562827412504576*t^20 + 31383758135151820800*t^19 +
12899349951808536576*t^18 - 8623901100661014528*t^17 -
24973850881079377920*t^16 - 28556556538522632192*t^15 -
19607238490120519680*t^14 - 6227443568747741184*t^13 +
3160628383658803200*t^12 + 5923598974517772288*t^11 +
4513493882461224960*t^10 + 2258428448106086400*t^9 +
789879246683111424*t^8 + 183318721572372480*t^7 +
20007071671910400*t^6 - 3386427630944256*t^5 - 2020058679214080*t^4 -
447070662033408*t^3 - 56532507033600*t^2 - 4000762036224*t -
123211874304 ], [ 0, 0, 0, -756*t^32 - 13824*t^31 - 38016*t^30 +
1257984*t^29 + 16039296*t^28 + 66963456*t^27 - 231565824*t^26 -
4636624896*t^25 - 30197041920*t^24 - 121783246848*t^23 -
346143836160*t^22 - 760381267968*t^21 - 1500984317952*t^20 -
3151317270528*t^19 - 6790311419904*t^18 - 12378319257600*t^17 -
16212489320448*t^16 - 12120808882176*t^15 + 1420736495616*t^14 +
16306891259904*t^13 + 21383523237888*t^12 + 14311390445568*t^11 +
3081581494272*t^10 - 3899081097216*t^9 - 5066891919360*t^8 -
3466636296192*t^7 - 1762125152256*t^6 - 734458281984*t^5 -
249580486656*t^4 - 64097353728*t^3 - 11154751488*t^2 - 1132462080*t -
49545216, 7344*t^48 + 228096*t^47 + 1783296*t^46 - 19229184*t^45 -
462309120*t^44 - 2975616000*t^43 + 6666541056*t^42 + 252556683264*t^41
+ 1988117042688*t^40 + 6747626557440*t^39 - 16810441629696*t^38 -
382441504333824*t^37 - 2872083213103104*t^36 - 14665967556968448*t^35
- 56535279022080000*t^34 - 166055265929428992*t^33 -
349914212481871872*t^32 - 392427624079687680*t^31 +
505440014385807360*t^30 + 3915677458134466560*t^29 +
11380046589762600960*t^28 + 22176780819953614848*t^27 +
31246634414150516736*t^26 + 30704055260021784576*t^25 +
16008283247955148800*t^24 - 8421764080550805504*t^23 -
31313010525487497216*t^22 - 43298557511482736640*t^21 -
42452562827412504576*t^20 - 31383758135151820800*t^19 -
12899349951808536576*t^18 + 8623901100661014528*t^17 +
24973850881079377920*t^16 + 28556556538522632192*t^15 +
19607238490120519680*t^14 + 6227443568747741184*t^13 -
3160628383658803200*t^12 - 5923598974517772288*t^11 -
4513493882461224960*t^10 - 2258428448106086400*t^9 -
789879246683111424*t^8 - 183318721572372480*t^7 -
20007071671910400*t^6 + 3386427630944256*t^5 + 2020058679214080*t^4 +
447070662033408*t^3 + 56532507033600*t^2 + 4000762036224*t +
123211874304 ], [ 0, 0, 0, -189*t^32 - 3456*t^31 - 9504*t^30 +
314496*t^29 + 4009824*t^28 + 16740864*t^27 - 57891456*t^26 -
1159156224*t^25 - 7549260480*t^24 - 30445811712*t^23 -
86535959040*t^22 - 190095316992*t^21 - 375246079488*t^20 -
787829317632*t^19 - 1697577854976*t^18 - 3094579814400*t^17 -
4053122330112*t^16 - 3030202220544*t^15 + 355184123904*t^14 +
4076722814976*t^13 + 5345880809472*t^12 + 3577847611392*t^11 +
770395373568*t^10 - 974770274304*t^9 - 1266722979840*t^8 -
866659074048*t^7 - 440531288064*t^6 - 183614570496*t^5 -
62395121664*t^4 - 16024338432*t^3 - 2788687872*t^2 - 283115520*t -
12386304, -918*t^48 - 28512*t^47 - 222912*t^46 + 2403648*t^45 +
57788640*t^44 + 371952000*t^43 - 833317632*t^42 - 31569585408*t^41 -
248514630336*t^40 - 843453319680*t^39 + 2101305203712*t^38 +
47805188041728*t^37 + 359010401637888*t^36 + 1833245944621056*t^35 +
7066909877760000*t^34 + 20756908241178624*t^33 +
43739276560233984*t^32 + 49053453009960960*t^31 -
63180001798225920*t^30 - 489459682266808320*t^29 -
1422505823720325120*t^28 - 2772097602494201856*t^27 -
3905829301768814592*t^26 - 3838006907502723072*t^25 -
2001035405994393600*t^24 + 1052720510068850688*t^23 +
3914126315685937152*t^22 + 5412319688935342080*t^21 +
5306570353426563072*t^20 + 3922969766893977600*t^19 +
1612418743976067072*t^18 - 1077987637582626816*t^17 -
3121731360134922240*t^16 - 3569569567315329024*t^15 -
2450904811265064960*t^14 - 778430446093467648*t^13 +
395078547957350400*t^12 + 740449871814721536*t^11 +
564186735307653120*t^10 + 282303556013260800*t^9 +
98734905835388928*t^8 + 22914840196546560*t^7 + 2500883958988800*t^6 -
423303453868032*t^5 - 252507334901760*t^4 - 55883832754176*t^3 -
7066563379200*t^2 - 500095254528*t - 15401484288 ], [ 0, 0, 0,
-108*t^24 + 864*t^20 - 13824*t^12 + 25920*t^8 + 13824*t^4 - 27648,
-432*t^36 + 5184*t^32 - 10368*t^28 - 96768*t^24 + 445824*t^20 -
41472*t^16 - 2515968*t^12 + 3649536*t^8 - 1327104*t^4 + 1769472 ], [
0, 0, 0, -27*t^24 + 216*t^20 - 3456*t^12 + 6480*t^8 + 3456*t^4 - 6912,
-54*t^36 + 648*t^32 - 1296*t^28 - 12096*t^24 + 55728*t^20 - 5184*t^16
- 314496*t^12 + 456192*t^8 - 165888*t^4 + 221184 ], [ 0, 0, 0,
-108*t^24 - 864*t^20 + 13824*t^12 + 25920*t^8 - 13824*t^4 - 27648,
-432*t^36 - 5184*t^32 - 10368*t^28 + 96768*t^24 + 445824*t^20 +
41472*t^16 - 2515968*t^12 - 3649536*t^8 - 1327104*t^4 - 1769472 ], [
0, 0, 0, -27*t^24 - 216*t^20 + 3456*t^12 + 6480*t^8 - 3456*t^4 - 6912,
-54*t^36 - 648*t^32 - 1296*t^28 + 12096*t^24 + 55728*t^20 + 5184*t^16
- 314496*t^12 - 456192*t^8 - 165888*t^4 - 221184 ], [ 0, 0, 0,
-108*t^32 + 5184*t^24 - 84672*t^16 + 497664*t^8 - 442368, -432*t^48 +
31104*t^40 - 881280*t^32 + 12192768*t^24 - 80953344*t^16 +
191102976*t^8 + 113246208 ], [ 0, 0, 0, -27*t^32 + 1296*t^24 -
21168*t^16 + 124416*t^8 - 110592, -54*t^48 + 3888*t^40 - 110160*t^32 +
1524096*t^24 - 10119168*t^16 + 23887872*t^8 + 14155776 ], [ 0, 0, 0,
-108*t^16 + 1728*t^8 - 1728, -432*t^24 + 10368*t^16 - 51840*t^8 -
27648 ], [ 0, 0, 0, -27*t^16 + 432*t^8 - 432, -54*t^24 + 1296*t^16 -
6480*t^8 - 3456 ], [ 0, 0, 0, -108*t^32 + 5184*t^24 - 84672*t^16 +
497664*t^8 - 442368, 432*t^48 - 31104*t^40 + 881280*t^32 -
12192768*t^24 + 80953344*t^16 - 191102976*t^8 - 113246208 ], [ 0, 0,
0, -27*t^32 + 1296*t^24 - 21168*t^16 + 124416*t^8 - 110592, 54*t^48 -
3888*t^40 + 110160*t^32 - 1524096*t^24 + 10119168*t^16 - 23887872*t^8
- 14155776 ], [ 0, 0, 0, -108*t^16 + 1728*t^8 - 1728, 432*t^24 -
10368*t^16 + 51840*t^8 + 27648 ], [ 0, 0, 0, -27*t^16 + 432*t^8 - 432,
54*t^24 - 1296*t^16 + 6480*t^8 + 3456 ], [ 0, 0, 0, -27*t^24 -
216*t^20 + 3456*t^12 + 6480*t^8 - 3456*t^4 - 6912, 54*t^36 + 648*t^32
+ 1296*t^28 - 12096*t^24 - 55728*t^20 - 5184*t^16 + 314496*t^12 +
456192*t^8 + 165888*t^4 + 221184 ], [ 0, 0, 0, -108*t^24 + 864*t^20 -
13824*t^12 + 25920*t^8 + 13824*t^4 - 27648, 432*t^36 - 5184*t^32 +
10368*t^28 + 96768*t^24 - 445824*t^20 + 41472*t^16 + 2515968*t^12 -
3649536*t^8 + 1327104*t^4 - 1769472 ], [ 0, 0, 0, -27*t^24 + 216*t^20
- 3456*t^12 + 6480*t^8 + 3456*t^4 - 6912, 54*t^36 - 648*t^32 +
1296*t^28 + 12096*t^24 - 55728*t^20 + 5184*t^16 + 314496*t^12 -
456192*t^8 + 165888*t^4 - 221184 ], [ 0, 0, 0, -108*t^24 - 864*t^20 +
13824*t^12 + 25920*t^8 - 13824*t^4 - 27648, 432*t^36 + 5184*t^32 +
10368*t^28 - 96768*t^24 - 445824*t^20 - 41472*t^16 + 2515968*t^12 +
3649536*t^8 + 1327104*t^4 + 1769472 ], [ 0, 0, 0, -108*t^16 - 1728*t^8
- 1728, -432*t^24 + 12960*t^16 + 41472*t^8 + 27648 ], [ 0, 0, 0,
-108*t^16 - 1728*t^8 - 1728, 432*t^24 - 12960*t^16 - 41472*t^8 - 27648
], [ 0, 0, 0, -27*t^16 - 432*t^8 - 432, 54*t^24 - 1620*t^16 - 5184*t^8
- 3456 ], [ 0, 0, 0, -27*t^16 - 432*t^8 - 432, -54*t^24 + 1620*t^16 +
5184*t^8 + 3456 ], [ 0, 0, 0, -27*t^32 - 486*t^24 - 1323*t^16 -
1296*t^8 - 432, -54*t^48 + 1458*t^40 + 9882*t^32 + 23814*t^24 +
27540*t^16 + 15552*t^8 + 3456 ], [ 0, 0, 0, -108*t^32 - 1944*t^24 -
5292*t^16 - 5184*t^8 - 1728, -432*t^48 + 11664*t^40 + 79056*t^32 +
190512*t^24 + 220320*t^16 + 124416*t^8 + 27648 ], [ 0, 0, 0, -108*t^32
- 1944*t^24 - 5292*t^16 - 5184*t^8 - 1728, 432*t^48 - 11664*t^40 -
79056*t^32 - 190512*t^24 - 220320*t^16 - 124416*t^8 - 27648 ], [ 0, 0,
0, -27*t^32 - 486*t^24 - 1323*t^16 - 1296*t^8 - 432, 54*t^48 -
1458*t^40 - 9882*t^32 - 23814*t^24 - 27540*t^16 - 15552*t^8 - 3456 ],
[ 0, 0, 0, -108*t^16 - 1728*t^8 - 1728, -432*t^24 - 10368*t^16 -
51840*t^8 + 27648 ], [ 0, 0, 0, -27*t^16 - 432*t^8 - 432, -54*t^24 -
1296*t^16 - 6480*t^8 + 3456 ], [ 0, 0, 0, -108*t^16 - 1728*t^8 - 1728,
432*t^24 + 10368*t^16 + 51840*t^8 - 27648 ], [ 0, 0, 0, -27*t^16 -
432*t^8 - 432, 54*t^24 + 1296*t^16 + 6480*t^8 - 3456 ], [ 0, 0, 0,
-27*t^32 - 1296*t^24 - 21168*t^16 - 124416*t^8 - 110592, -54*t^48 -
3888*t^40 - 110160*t^32 - 1524096*t^24 - 10119168*t^16 - 23887872*t^8
+ 14155776 ], [ 0, 0, 0, -108*t^32 - 5184*t^24 - 84672*t^16 -
497664*t^8 - 442368, -432*t^48 - 31104*t^40 - 881280*t^32 -
12192768*t^24 - 80953344*t^16 - 191102976*t^8 + 113246208 ], [ 0, 0,
0, -108*t^32 - 5184*t^24 - 84672*t^16 - 497664*t^8 - 442368, 432*t^48
+ 31104*t^40 + 881280*t^32 + 12192768*t^24 + 80953344*t^16 +
191102976*t^8 - 113246208 ], [ 0, 0, 0, -27*t^32 - 1296*t^24 -
21168*t^16 - 124416*t^8 - 110592, 54*t^48 + 3888*t^40 + 110160*t^32 +
1524096*t^24 + 10119168*t^16 + 23887872*t^8 - 14155776 ], [ 0, 0, 0,
-108*t^32 + 1944*t^24 - 5292*t^16 + 5184*t^8 - 1728, -432*t^48 -
11664*t^40 + 79056*t^32 - 190512*t^24 + 220320*t^16 - 124416*t^8 +
27648 ], [ 0, 0, 0, -27*t^32 + 486*t^24 - 1323*t^16 + 1296*t^8 - 432,
-54*t^48 - 1458*t^40 + 9882*t^32 - 23814*t^24 + 27540*t^16 - 15552*t^8
+ 3456 ], [ 0, 0, 0, -108*t^16 + 1728*t^8 - 1728, 432*t^24 +
12960*t^16 - 41472*t^8 + 27648 ], [ 0, 0, 0, -27*t^16 + 432*t^8 - 432,
54*t^24 + 1620*t^16 - 5184*t^8 + 3456 ], [ 0, 0, 0, -27*t^32 +
486*t^24 - 1323*t^16 + 1296*t^8 - 432, 54*t^48 + 1458*t^40 - 9882*t^32
+ 23814*t^24 - 27540*t^16 + 15552*t^8 - 3456 ], [ 0, 0, 0, -108*t^32 +
1944*t^24 - 5292*t^16 + 5184*t^8 - 1728, 432*t^48 + 11664*t^40 -
79056*t^32 + 190512*t^24 - 220320*t^16 + 124416*t^8 - 27648 ], [ 0, 0,
0, -27*t^16 + 432*t^8 - 432, -54*t^24 - 1620*t^16 + 5184*t^8 - 3456 ],
[ 0, 0, 0, -108*t^16 + 1728*t^8 - 1728, -432*t^24 - 12960*t^16 +
41472*t^8 - 27648 ], [ 0, 0, 0, -27*t^24 + 54*t^20 + 405*t^16 -
864*t^12 + 864*t^4 - 432, 54*t^36 - 162*t^32 + 1782*t^28 - 4914*t^24 -
324*t^20 + 13932*t^16 - 12096*t^12 - 5184*t^8 + 10368*t^4 - 3456 ], [
0, 0, 0, -108*t^24 + 216*t^20 + 1620*t^16 - 3456*t^12 + 3456*t^4 -
1728, 432*t^36 - 1296*t^32 + 14256*t^28 - 39312*t^24 - 2592*t^20 +
111456*t^16 - 96768*t^12 - 41472*t^8 + 82944*t^4 - 27648 ], [ 0, 0, 0,
-27*t^24 - 54*t^20 + 405*t^16 + 864*t^12 - 864*t^4 - 432, -54*t^36 -
162*t^32 - 1782*t^28 - 4914*t^24 + 324*t^20 + 13932*t^16 + 12096*t^12
- 5184*t^8 - 10368*t^4 - 3456 ], [ 0, 0, 0, -108*t^24 - 216*t^20 +
1620*t^16 + 3456*t^12 - 3456*t^4 - 1728, -432*t^36 - 1296*t^32 -
14256*t^28 - 39312*t^24 + 2592*t^20 + 111456*t^16 + 96768*t^12 -
41472*t^8 - 82944*t^4 - 27648 ], [ 0, 0, 0, -27*t^24 - 54*t^20 +
405*t^16 + 864*t^12 - 864*t^4 - 432, 54*t^36 + 162*t^32 + 1782*t^28 +
4914*t^24 - 324*t^20 - 13932*t^16 - 12096*t^12 + 5184*t^8 + 10368*t^4
+ 3456 ], [ 0, 0, 0, -108*t^24 + 216*t^20 + 1620*t^16 - 3456*t^12 +
3456*t^4 - 1728, -432*t^36 + 1296*t^32 - 14256*t^28 + 39312*t^24 +
2592*t^20 - 111456*t^16 + 96768*t^12 + 41472*t^8 - 82944*t^4 + 27648
], [ 0, 0, 0, -27*t^24 + 54*t^20 + 405*t^16 - 864*t^12 + 864*t^4 -
432, -54*t^36 + 162*t^32 - 1782*t^28 + 4914*t^24 + 324*t^20 -
13932*t^16 + 12096*t^12 + 5184*t^8 - 10368*t^4 + 3456 ], [ 0, 0, 0,
-108*t^24 - 216*t^20 + 1620*t^16 + 3456*t^12 - 3456*t^4 - 1728,
432*t^36 + 1296*t^32 + 14256*t^28 + 39312*t^24 - 2592*t^20 -
111456*t^16 - 96768*t^12 + 41472*t^8 + 82944*t^4 + 27648 ] ]; 
// For each subgroup X_n a model was constructed using a covering X_n -> X_m.
// The nth entry in this list is m.  
coverlist := [ 0, 1, 1, 1, 1, 1,
1, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 11, 8, 8, 11, 10, 11,
11, 12, 12, 13, 13, 13, 11, 13, 9, 8, 11, 12, 11, 10, 11, 13, 11, 8,
11, 13, 11, 11, 12, 12, 11, 11, 7, 22, 22, 23, 24, 23, 24, 25, 35, 23,
23, 24, 24, 23, 23, 45, 43, 30, 26, 26, 34, 23, 29, 33, 32, 23, 28,
35, 26, 36, 36, 36, 25, 30, 30, 28, 39, 27, 30, 27, 27, 25, 39, 25,
25, 25, 25, 36, 30, 30, 39, 39, 39, 42, 45, 45, 45, 45, 50, 50, 32,
32, 36, 36, 36, 36, 36, 36, 37, 39, 29, 30, 23, 28, 25, 45, 41, 24,
29, 27, 27, 23, 43, 41, 26, 26, 24, 25, 39, 39, 31, 23, 35, 23, 45,
45, 45, 50, 45, 50, 50, 50, 32, 32, 36, 39, 39, 30, 30, 30, 39, 39,
30, 36, 51, 50, 50, 30, 51, 30, 51, 51, 56, 56, 55, 55, 87, 87, 58,
86, 87, 58, 58, 98, 58, 99, 64, 85, 96, 96, 84, 64, 75, 62, 75, 62,
78, 78, 62, 62, 85, 84, 92, 96, 61, 61, 85, 86, 102, 66, 96, 77, 75,
81, 85, 81, 95, 95, 102, 80, 85, 79, 84, 84, 86, 102, 94, 83, 78, 78,
102, 85, 83, 108, 123, 118, 116, 116, 118, 93, 89, 62, 62, 58, 58, 69,
58, 69, 73, 61, 58, 66, 64, 69, 71, 70, 65, 69, 63, 68, 89, 72, 73,
99, 87, 87, 88, 98, 72, 87, 74, 73, 60, 60, 98, 90, 90, 90, 90, 90,
90, 91, 91, 91, 91, 91, 91, 89, 93, 89, 93, 93, 89, 93, 89, 91, 97,
63, 97, 97, 92, 92, 63, 97, 97, 97, 97, 97, 96, 96, 96, 96, 65, 65,
69, 69, 69, 69, 69, 69, 70, 70, 71, 71, 75, 75, 78, 78, 79, 79, 85,
84, 86, 102, 85, 102, 85, 102, 145, 85, 86, 84, 102, 73, 73, 73, 73,
91, 118, 118, 118, 118, 58, 124, 106, 106, 112, 107, 81, 81, 124, 80,
110, 79, 79, 83, 92, 107, 77, 107, 111, 94, 97, 94, 95, 97, 95, 109,
110, 111, 112, 96, 110, 105, 105, 105, 77, 61, 61, 105, 66, 112, 110,
66, 107, 80, 109, 83, 112, 96, 118, 118, 116, 116, 142, 142, 134, 134,
141, 125, 143, 131, 143, 131, 126, 126, 145, 145, 142, 142, 141, 125,
168, 168, 168, 168, 168, 168, 168, 157, 157, 157, 157, 168, 179, 179,
55, 185, 191, 196, 191, 182, 183, 181, 187, 181, 192, 183, 195, 183,
181, 185, 187, 183, 181, 186, 191, 188, 182, 184, 187, 199, 197, 194,
194, 193, 202, 202, 202, 202, 193, 205, 199, 205, 195, 262, 197, 192,
192, 276, 185, 185, 195, 227, 211, 211, 213, 213, 212, 227, 225, 235,
225, 212, 235, 189, 189, 189, 189, 192, 192, 192, 193, 193, 193, 193,
211, 211, 211, 211, 235, 235, 235, 235, 192, 216, 207, 211, 235, 235,
224, 224, 207, 211, 207, 211, 207, 211, 216, 235, 235, 249, 250, 253,
250, 248, 248, 250, 252, 259, 261, 262, 263, 299, 299, 299, 299, 340,
340, 340, 297, 297, 297, 297, 340, 335, 335, 335, 295, 295, 295, 295,
335, 252, 252, 250, 293, 293, 293, 293, 250, 252, 250, 250, 280, 282,
252, 289, 280, 281, 289, 249, 288, 287, 287, 286, 286, 259, 259, 267,
267, 253, 253, 276, 257, 256, 256, 251, 278, 244, 244, 244, 244, 245,
245, 245, 245, 246, 246, 249, 288, 298, 329, 288, 288, 329, 331, 331,
334, 316, 316, 350, 313, 291, 313, 307, 334, 308, 349, 308, 312, 305,
312, 305, 340, 317, 281, 340, 292, 340, 292, 294, 294, 340, 323, 284,
323, 298, 296, 324, 296, 335, 325, 327, 335, 335, 335, 355, 354, 355,
354, 355, 354, 354, 355, 251, 259, 308, 308, 284, 291, 350, 291, 288,
350, 281, 316, 316, 281, 333, 333, 312, 349, 349, 312, 288, 323, 323,
324, 324, 284, 355, 354, 354, 354, 354, 355, 355, 355, 296, 317, 284,
292, 298, 294, 298, 294, 324, 291, 350, 292, 327, 296, 439, 439, 490,
490, 496, 496, 496, 496, 490, 490 ];

function make_2adic_string(E);

if HasComplexMultiplication(E) then
  outstr := "inf inf [] CM";
else

done1 := false;
curgp := 1;
// Make a list of subgroups, and all the choices for the invariants
invarlist := [];
jval := jInvariant(E);
//printf "j-invariant is %o.\n",jval;
P1 := ProjectiveSpace(Rationals(),1);
Append(~invarlist,<1,[P1![jval,1]]>);

while done1 eq false do
  // For each maximal subgroup G of curgp, test to see if the points on X_curgp
  // lift to points on X_G.
  testmore := false;
  moregps := [];
  //printf "Current group is %o.\n",curgp;
  for ind in subdatamaxsub[curgp] do
    //printf "Working on X_%o.\n",ind;
    if subdata[ind][3] eq 0 then
      if ind in genzeropoints then
        jind := Index(genzeropoints,ind);
        jfunc := jmaps[jind][2];
        num := R!Numerator(jfunc);
        denom := R!Denominator(jfunc);
        poly := num - jval*denom;
        rtlist := Roots(poly);
        p1points := [ P1![ rtlist[i][1],1] : i in [1..#rtlist]];
        if Degree(num) eq Degree(denom) then
          if LeadingCoefficient(num)/LeadingCoefficient(denom) eq jval then
            Append(~p1points, P1![1,0]);
          end if;
        end if;
        if #p1points ge 1 then
          //printf "Point on X_%o lifts to a point on X_%o!\n",curgp,ind;
          testmore := true;
          Append(~invarlist,<ind,p1points>);
          Append(~moregps,ind);
        end if;                   
      end if;
    end if;
    if subdata[ind][3] eq 1 then
      if ind in rankoneell then
        // Is the cover of this group in the invarlist?
        ind2 := Index([ invarlist[i][1] : i in [1..#invarlist]],coverlist[ind]);
        if (ind2 ne 0) then
          p1points := invarlist[ind2][2];
        else
          coverind := coverlist[ind];
          jind := Index(genzeropoints,coverind);
          jfunc := jmaps[jind][2];
          num := R!Numerator(jfunc);
          denom := R!Denominator(jfunc);
          poly := num - jval*denom;
          rtlist := Roots(poly);
          p1points := [ P1![ rtlist[i][1],1] : i in [1..#rtlist]];
          if Degree(num) eq Degree(denom) then
            if LeadingCoefficient(num)/LeadingCoefficient(denom) eq jval then
              Append(~p1points, P1![1,0]);
            end if;
          end if;
        end if;
        r1ind := Index(rankoneell,ind); 
        EE<a,b,c> := EllipticCurve(rankoneellmodels[r1ind][2]);
        ellmap := map<EE -> P1 | rankoneellmaps[r1ind][2]>;
        ptset := { Codomain(ellmap)!Eltseq(p1points[i]) : i in [1..#p1points]};
        //printf "Computing preimage of map from X%o to X%o.\n",ind,coverind;
        //printf "Point set is %o.\n",ptset;
        addinvarlist := <>;
        for p in ptset do
          preims := RationalPoints(p@@ellmap);
          //printf "The preimages of %o are %o.\n",p,preims;
          for p2 in preims do
            //printf "The image of %o is %o.\n",p2,ellmap(p2);
            if ellmap(p2) eq p then
              Append(~addinvarlist,p2);
            end if;
          end for;
        end for;
        if #addinvarlist gt 0 then
          //printf "Point on X_%o lifts to a point on X_%o!\n",curgp,ind;
          testmore := true;
          Append(~moregps,ind);
        end if;                 
      end if;       
    end if;
    if subdata[ind][3] ge 2 then
      ind2 := Index( [ genustwoorhigher[i][2] : i in [1..#genustwoorhigher]],jval);
      if (ind2 ne 0) then
        if ind eq genustwoorhigher[ind2][1] then
          //printf "Point on X_%o lifts to a point on X_%o!\n",curgp,ind;
          testmore := true; 
          Append(~moregps,ind);       
        end if;
      end if;
    end if;
  end for;
  if #moregps gt 0 then
    curgp := moregps[#moregps];
  end if;
  if testmore eq false then
    done1 := true;
    minusonegp := curgp;
  end if;
end while;
//printf "The image for the subgroup containing -I is X%o.\n",minusonegp;
finesubs := [ i : i in [1..#finesubdata] | finesubdata[i][1] eq minusonegp];
index := subdata[minusonegp][1];
level := subdata[minusonegp][2];
subname := "X" cat IntegerToString(minusonegp);
gens := genlists[minusonegp];
if (minusonegp eq 1) then
  // For technical reasons our subgroup data had 
  // the level of GL_2(Z_2) as 2.
  // We fix that now.
  level := 1;
end if;

foundtimes := 0;
for n in [1..#finesubs] do
  foundhere := false;
  // If we get here, the genus of X_minusonegp is zero.
  ind := Index([ invarlist[i][1] : i in [1..#invarlist]],minusonegp);
  tvals1 := invarlist[ind][2];
  tvals2 := [];
  infinitetval := [];
  for m in [1..#tvals1] do
    if tvals1[m][2] eq 0 then
      // Point at infinity.
      Append(~infinitetval,true);
      Append(~tvals2,0);
    else
      Append(~tvals2,Rationals()!tvals1[m][1]);
      Append(~infinitetval,false);
    end if;
  end for;
  ainvars := eqf[finesubs[n]];
  A := ainvars[4];
  B := ainvars[5];
  for m in [1..#tvals2] do
    if infinitetval[m] eq false then
      Aspec := Evaluate(A,tvals2[m]);
      Bspec := Evaluate(B,tvals2[m]);
      E2 := EllipticCurve([0,0,0,Aspec,Bspec]);
      chk := IsIsomorphic(E,E2);
      if (chk eq true) then
        foundhere := true;
      end if;
    else
      // We do this when E might be isomorphic to E_t
      // when t = (1:0).
      degA := Degree(Numerator(A));
      degB := Degree(Numerator(B));
      powt := Max(Ceiling(degA/4),Ceiling(degB/6));
      A2 := Evaluate(A,1/t)*t^(4*powt);
      B2 := Evaluate(B,1/t)*t^(6*powt);
      Aspec := Evaluate(A2,0);
      Bspec := Evaluate(B2,0);
      //printf "Curve is y^2 = x^3 + %ox + %o.\n",Aspec,Bspec;
      if (-64*Aspec^3 - 432*Bspec^2) ne 0 then
        E2 := EllipticCurve([0,0,0,Aspec,Bspec]);
        chk := IsIsomorphic(E,E2);
        if (chk eq true) then
          foundhere := true;
        end if;
      end if;
    end if;
  end for;
  if foundhere eq true then
    foundtimes := foundtimes + 1;
    subname := finesubdata[finesubs[n]][4];
    //printf "The image should be %o.\n",subname;
    index := finesubdata[finesubs[n]][2];
    level := finesubdata[finesubs[n]][3];
    gens := finegenlists[finesubs[n]];
  end if;
end for;
if foundtimes gt 1 then
  printf "Error! The elliptic curve was found multiple times!\n";
  printf "Something is haywire!\n";
  printf "The curve is %o.\n",E;
  bad := 0;
  bad2 := 1/bad;
end if;
// The index of the 2-adic image is stored in index
// The smallest 2^r so that every matrix congruent to I (mod 2^r)
// is level.
// The name of the modular curve is subname
outstr := IntegerToString(index) cat " " cat IntegerToString(level) cat " " cat gens cat " " cat subname;
// End the non-CM case
end if;
return outstr;
end function;

procedure make_2adic_file(curvefile);

// (Nicolas) Form the output file name from the input one
// Assuming the input file is called allcurves.XXX
// The output file will be called 2adic.XXX

outputfilename := "2adic" cat Substring(curvefile,Position(curvefile,"."),#curvefile);
SetColumns(0);

FP := Open(curvefile,"r");
Rewind(FP);
done := false;
while (done eq false) do
 A := Gets(FP);
 if IsEof(A) then
  done := true;
 else
  startind := Position(A,"[");
  endind := Position(A,"]");
  prefix := Substring(A,1,endind) cat " ";
  ainvars := eval Substring(A,startind,endind-startind+1);
  Eline := prefix cat make_2adic_string(EllipticCurve(ainvars));
  PrintFile(outputfilename, Eline);
//print Eline;
  // End the case when we haven't gotten to the end of the file
 end if;
// End the while loop
end while;

end procedure;

if (assigned filename) then
  printf "Processing curves in %o...", filename;
  make_2adic_file(filename);
  printf ".finished\n";
  quit;
else
  if (assigned ainvs) then
  E := EllipticCurve(eval ainvs);
  res := make_2adic_string(E);
  print res ;
  quit;
else
  printf "This magma script may be called with filename as an argument.\n";
  printf "Something like:\n";
  printf "magma -b filename:=allcurves.00000-09999 2adic.m";
  printf "which will then process all curves, writing the output file 2adic.00000-09999";
  printf "";
  printf "Otherwise run make_2adic_file(filename) to process a file, or use the function make_2adic_string(E) on a single elliptic curve.";
end if;
end if;
