/*
*  LibRAL
*  file: vcard.c
*
*  the vcard import engine v0.5 
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <glib/gstdio.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "vcard.h"

#include "../../types.h"
#include "../../abook.h"
#include "../../card.h"
#include "../../personal.h"
#include "../../company.h"
#include "../../contact.h"
#include "../../work.h"
#include "../../notes.h"
#include "../../address.h"
#include "../../net.h"
#include "../../telephone.h"
#include "../../plugin.h"
#include "../../filter.h"
#include "../../lookup.h"


#define VCARD_TOKEN_DELIM ':'
#define VCARD_PARAM_DELIM ';'
#define VCARD_VALUE_DELIM '='


typedef enum {
  VCARD_START = 0,
  VCARD_BUILD_TOKEN,           // token:
  VCARD_BUILD_PARAM,           // token;param=value:
  VCARD_BUILD_PARAM_VALUES,     
  VCARD_BUILD_VALUES,     
  VCARD_BUILD_DATA,            // token:data;data;
  VCARD_END
} ParserStatus;


typedef enum {
  N = 1, 
  FN, 
  TZ, 
  END, 
  ADR, 
  TEL, 
  GEO, 
  ORG, 
  REV, 
  UID, 
  URL, 
  KEY, 
  BDAY, 
  ROLE, 
  LOGO, 
  NOTE, 
  BEGIN,
  PHOTO, 
  LABEL,
  EMAIL, 
  TITLE, 
  AGENT, 
  PROID, 
  SOUND, 
  CLASS,
  MAILER, 
  VERSION,
  NICKNAME, 
  CATEGORIES,                  // last vcard token

  KDE_EXT_Profession,          // KDE extensions
  KDE_EXT_Department,
  KDE_EXT_ManagersName,
  KDE_EXT_AssistantsName,
  KDE_EXT_Office,
  KDE_EXT_SpousesName,
  KDE_EXT_Anniversary,
  KDE_EXT_CUSTOM,

  RUBRICA_EXT_Card,            // Rubrica extensions
  RUBRICA_EXT_Group, 
  RUBRICA_EXT_Ref,
  RUBRICA_EXT_Url,
  RUBRICA_EXT_CompanyName,
  RUBRICA_EXT_Street,
  RUBRICA_EXT_StreetNumber,
  RUBRICA_EXT_ZipCode,
  RUBRICA_EXT_City,
  RUBRICA_EXT_Province,
  RUBRICA_EXT_Country,
  RUBRICA_EXT_Web,
  RUBRICA_EXT_Email,
  RUBRICA_EXT_Operator,
  RUBRICA_EXT_Fax,
  RUBRICA_EXT_Green,
  RUBRICA_EXT_CustomerCare,
  RUBRICA_EXT_Notes,
  RUBRICA_EXT_Department,
  RUBRICA_EXT_SubDepartment,
  RUBRICA_EXT_Manager,
  RUBRICA_EXT_ManagerTelephone,
  RUBRICA_EXT_Collaborator,
  RUBRICA_EXT_CollaboratorTelephone,
  RUBRICA_EXT_SpouseName,
  RUBRICA_EXT_Child,
  RUBRICA_EXT_Anniversary,
  RUBRICA_EXT_Hobbies,

  EVOLUTION_EXT_File_as,       // Evolution extensions
  EVOLUTION_EXT_Office,
  EVOLUTION_EXT_Manager,
  EVOLUTION_EXT_Assistant,
  EVOLUTION_EXT_Spouse,
  EVOLUTION_EXT_Anniversary,

  MOZILLA_EXT_Html,

  BAD_TOKEN,                   // this isn't a VCard token
} RVCardTokenType;


typedef enum {
  TYPE = 1, 
  VALUE, 
  ENCODING,

  RATE,                        // rubrica's param
  OWNER,

  BAD_PARAM,                   // this isn't a VCard token
} RVCardParamType;


typedef enum {
  VCARD = 1,
  BINARY,
  BASIC, 
  PUBLIC, 
  PRIVATE, 
  CONFIDENTIAL,
  HOME,  
  WORK, 
  POSTAL, 
  PARCEL, 
  DOM, 
  INTL, 
  PREF,
  VOICE, 
  FAX, 
  MSG, 
  CELL, 
  PAGER, 
  BBS, 
  MODEM, 
  CAR, 
  ISDN, 
  VIDEO,  
  AOL, 
  INTERNET,
  APPLELINK, 
  ATTMAIL, 
  CIS, 
  EWORLD, 
  IBMMail, 
  MCIMAIL, 
  POWERSHARE, 
  PRODIGY, 
  TLX, 
  X400, 
  PGP, 
  X509, 
  TEXT,
  URI,
  JPEG,
  QUOTED_PRINTABLE,  
  BAD_VALUE,                 // this isn't a vcard token
} RVCardValueType;


static RLookupTable tokens[] = {
  {"N",     "", N},          {"FN",     "", FN},      {"TZ",      "", TZ},
  {"END",   "", END},        {"ADR",    "", ADR},     {"TEL",     "", TEL},
  {"GEO",   "", GEO},        {"ORG",    "", ORG},     {"REV",     "", REV},
  {"UID",   "", UID},        {"URL",    "", URL},     {"KEY",     "", KEY},
  {"BDAY",  "", BDAY},       {"ROLE",   "", ROLE},    {"LOGO",    "", LOGO}, 
  {"NOTE",  "",  NOTE},      {"BEGIN",  "", BEGIN},   {"PHOTO",   "", PHOTO},  
  {"LABEL", "", LABEL},      {"EMAIL",  "", EMAIL},   {"TITLE",   "", TITLE}, 
  {"AGENT", "", AGENT},      {"PROID",  "", PROID},   {"SOUND",   "", SOUND},  
  {"CLASS", "", CLASS},      {"MAILER", "", MAILER},  {"VERSION", "", VERSION},
  {"NICKNAME", "", NICKNAME}, {"CATEGORIES", "", CATEGORIES},

  /* kde extension.
  */
  {"X-KADDRESSBOOK-X-Profession",     "", KDE_EXT_Profession},
  {"X-KADDRESSBOOK-X-Department",     "", KDE_EXT_Department},
  {"X-KADDRESSBOOK-X-ManagersName",   "", KDE_EXT_ManagersName},
  {"X-KADDRESSBOOK-X-AssistantsName", "", KDE_EXT_AssistantsName},
  {"X-KADDRESSBOOK-X-Office",         "", KDE_EXT_Office},
  {"X-KADDRESSBOOK-X-SpousesName",    "", KDE_EXT_SpousesName},
  {"X-KADDRESSBOOK-X-Anniversary",    "", KDE_EXT_Anniversary},
  {"X-KADDRESSBOOK-X-CUSTOM",         "", KDE_EXT_CUSTOM},
  
  /* rubrica extension. 
     So rubrica can import a previously 
     exported rubrica's addressbook to vcard format
  */
  {"X-RUBRICA-X-Card",                  "", RUBRICA_EXT_Card},
  {"X-RUBRICA-X-Group",                 "", RUBRICA_EXT_Group},
  {"X-RUBRICA-X-Ref",                   "", RUBRICA_EXT_Ref},
  {"X-RUBRICA-X-URL",                   "", RUBRICA_EXT_Url},
  {"X-RUBRICA-X-CompanyName",           "", RUBRICA_EXT_CompanyName},
  {"X-RUBRICA-X-Street",                "", RUBRICA_EXT_Street},
  {"X-RUBRICA-X-StreetNumber",          "", RUBRICA_EXT_StreetNumber},
  {"X-RUBRICA-X-ZipCode",               "", RUBRICA_EXT_ZipCode},
  {"X-RUBRICA-X-City",                  "", RUBRICA_EXT_City},
  {"X-RUBRICA-X-Province",              "", RUBRICA_EXT_Province},
  {"X-RUBRICA-X-Country",               "", RUBRICA_EXT_Country},
  {"X-RUBRICA-X-Web",                   "", RUBRICA_EXT_Web},
  {"X-RUBRICA-X-Email",                 "", RUBRICA_EXT_Email},
  {"X-RUBRICA-X-Operator",              "", RUBRICA_EXT_Operator},
  {"X-RUBRICA-X-Fax",                   "", RUBRICA_EXT_Fax},
  {"X-RUBRICA-X-Green",                 "", RUBRICA_EXT_Green},
  {"X-RUBRICA-X-CustomerCare",          "", RUBRICA_EXT_CustomerCare},
  {"X-RUBRICA-X-Department",            "", RUBRICA_EXT_Department},
  {"X-RUBRICA-X-SubDepartment",         "", RUBRICA_EXT_SubDepartment},
  {"X-RUBRICA-X-Manager",               "", RUBRICA_EXT_Manager},
  {"X-RUBRICA-X-ManagerTelephone",      "", RUBRICA_EXT_ManagerTelephone},
  {"X-RUBRICA-X-Collaborator",          "", RUBRICA_EXT_Collaborator},
  {"X-RUBRICA-X-CollaboratorTelephone", "", RUBRICA_EXT_CollaboratorTelephone},
  {"X-RUBRICA-X-SpouseName",            "", RUBRICA_EXT_SpouseName},
  {"X-RUBRICA-X-Child",                 "", RUBRICA_EXT_Child},
  {"X-RUBRICA-X-Anniversary",           "", RUBRICA_EXT_Anniversary},

  /* evolution extensions
  */
  {"X-EVOLUTION-FILE-AS",               "", EVOLUTION_EXT_File_as},  
  {"X-EVOLUTION-OFFICE",                "", EVOLUTION_EXT_Office},
  {"X-EVOLUTION-MANAGER",               "", EVOLUTION_EXT_Manager},  
  {"X-EVOLUTION-ASSISTANT",             "", EVOLUTION_EXT_Assistant},
  {"X-EVOLUTION-SPOUSE",                "", EVOLUTION_EXT_Spouse},  
  {"X-EVOLUTION-ANNIVERSARY",           "", EVOLUTION_EXT_Anniversary},

  {"X-MOZILLA-HTML",                    "", MOZILLA_EXT_Html},

  /*     TODO: add evolution extensions
  */
  {NULL,  "", BAD_TOKEN}
};
	
	
static RLookupTable params[] = {
  {"type",     "", TYPE},
  {"value",    "", VALUE},   
  {"encoding", "", ENCODING},
  
  {"rate",     "", RATE},      // rubrica's param
  {"owner",    "", OWNER},
  
  {NULL,       "", BAD_PARAM}
};		


static RLookupTable values[] = {
  {"home",     "", HOME},     {"work",       "", WORK},     
  {"cell",     "", CELL},     {"fax",        "", FAX},    
  {"postal",   "", POSTAL},   {"parcel",     "", PARCEL},  
  {"dom",      "", DOM},      {"intl",       "", INTL}, 
  {"pref",     "", PREF},     {"voice",      "", VOICE}, 
  {"msg",      "", MSG},      {"pager",      "", PAGER},   
  {"bbs",      "", BBS},      {"modem",      "", MODEM},   
  {"car",      "", CAR},      {"isdn",       "", ISDN}, 
  {"video",    "", VIDEO},    {"aol",        "", AOL},     
  {"internet", "", INTERNET}, {"applelink",  "", APPLELINK},
  {"attmail",  "", ATTMAIL},  {"cis",        "", CIS},    
  {"eworld",   "", EWORLD},   {"ibmmail",    "", IBMMail},  
  {"mcimail",  "", MCIMAIL},  {"powershare", "", POWERSHARE},
  {"prodigy",  "", PRODIGY},  {"tlx",        "", TLX},   
  {"x400",     "", X400},     {"vcard",      "", VCARD},
  {"pgp",      "", PGP},      {"x509",       "", X509},     
  {"b",        "", BINARY},   {"binary",     "", BINARY},  
  {"text",     "", TEXT},     {"uri",        "", URI},
  {"basic",    "", BASIC},    {"jpeg",       "", JPEG},    
  {"public",   "", PUBLIC},   {"private",    "", PRIVATE}, 
  {"confidential",     "", CONFIDENTIAL},  
  {"quoted-printable", "", QUOTED_PRINTABLE},  
  {NULL,               "", BAD_VALUE}
};

 
static RPairTable address_pairs[] = {
  {"home",  HOME,      R_ADDRESS_HOME},
  {"work",  WORK,      R_ADDRESS_WORK},
  {NULL,    BAD_VALUE, R_ADDRESS_OTHER},
};


static RPairTable phone_pairs[] = {
  {"home",      HOME,      R_TELEPHONE_HOME     },
  {"work",      WORK,      R_TELEPHONE_WORK     },
  {"fax",       FAX,       R_TELEPHONE_FAX      },
  {"cellphone", CELL,      R_TELEPHONE_CELLPHONE},
  {"pref",      PREF,      R_TELEPHONE_OTHER    },
  {"voice",     VOICE,     R_TELEPHONE_OTHER    },
  {"msg",       MSG,       R_TELEPHONE_OTHER    },
  {"pager",     PAGER,     R_TELEPHONE_OTHER    },
  {"bbs",       BBS,       R_TELEPHONE_OTHER    },
  {"modem",     MODEM,     R_TELEPHONE_OTHER    },
  {"car",       CAR,       R_TELEPHONE_OTHER    },
  {"isdn",      ISDN,      R_TELEPHONE_OTHER    },
  {"video",     VIDEO,     R_TELEPHONE_OTHER    },
  {NULL,        BAD_VALUE, R_TELEPHONE_OTHER    },
};


/*  property enumeration
*/
enum {
  VCARD_PROP_0,
  VCARD_TOKEN,
  VCARD_PARAM,
  VCARD_VALUE,
};


static void r_vcard_init         (RVCard* obj);
static void r_vcard_class_init   (RVCardClass* klass);

static void r_vcard_set_property (GObject* obj, guint property_id,
				  const GValue* value, GParamSpec* spec);
static void r_vcard_get_property (GObject* obj, guint property_id,
				  GValue* value, GParamSpec* spec);

static void r_vcard_dispose      (RVCard* obj);
static void r_vcard_finalize     (RVCard* obj);




/* private functions
*/
static RCompanyCard* r_vcard_company_new        (RVCard *vcard, RCard* card);
static RAddress*     r_vcard_company_address_new(RVCard *vcard);

static gchar*        r_vcard_get_data        (RVCard* vcard);
static gchar*        r_vcard_get_param_value (RVCard* vcard);

static void          r_vcard_set_param_value (RVCard* vcard, gchar *string);
static void          r_vcard_free_values     (RVCard* vcard);

static RDate*        r_vcard_decode_date     (gchar* buffer); 

static void          r_vcard_build_card      (RVCard *vcard, RAbook* abook);

static gboolean      r_vcard_read_file       (RVCard* vcard, RAbook* abook,
					      gchar* filename);

static gboolean      r_vcard_open_file       (RAbook* vcard, gchar* filename);
static gboolean      r_vcard_write_file      (RAbook* vcard, gchar* filename,
					      gint crate);
static gboolean      r_vcard_overwrite_file  (RAbook* abook, gint crate);



struct _RVCardPrivate {
  gboolean have_company;
  gboolean have_company_address;

  RVCardTokenType token; 
  RVCardParamType param; // type || value || encoding || rank || owner
  RVCardValueType value; // home || work || ... || internet || ...
  
  GString *data;         // buffered data

  GList* values;         // RParams list

  gboolean dispose_has_run;
};


typedef struct _RParams { 
  RVCardParamType param;
  RVCardValueType value;
  gchar* str;
} RParams;



RPersonalCard *card              = NULL;
RCompanyCard  *company           = NULL;
RContact      *contact           = NULL;
RWork         *work              = NULL;
RNotes        *notes             = NULL;
RAddress      *address           = NULL;
RTelephone    *tel               = NULL;
RNetAddress   *net               = NULL;
RAddress      *company_address   = NULL;
RTelephone    *company_telephone = NULL;
RNetAddress   *company_net       = NULL;


gchar** categories = NULL;
gchar** decoded;


GType
r_vcard_get_type()
{
  static GType r_vcard_type = 0;
  
  if (!r_vcard_type)
    {
      static const GTypeInfo r_vcard_info =
	{
	  sizeof(RVCardClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_vcard_class_init,
	  NULL,
	  NULL,
	  sizeof(RVCard),
	  0,
	  (GInstanceInitFunc) r_vcard_init
	};

      r_vcard_type = g_type_register_static (G_TYPE_OBJECT,
					     "RVCard", 
					     &r_vcard_info, 0);
    }
  
  return r_vcard_type;
}


static void
r_vcard_init(RVCard* vcard)
{
  g_return_if_fail(R_VCARD(vcard) != NULL);

  vcard->priv = g_new(RVCardPrivate, 1);

  vcard->fp    = NULL;

  vcard->priv->have_company         = FALSE;
  vcard->priv->have_company_address = FALSE;

  vcard->priv->token  = BAD_TOKEN;
  vcard->priv->param  = BAD_PARAM;
  vcard->priv->value  = BAD_VALUE;
  vcard->priv->data   = g_string_new(NULL);
  vcard->priv->values = NULL;
  
  vcard->priv->dispose_has_run = FALSE;
}


static void
r_vcard_class_init(RVCardClass* klass)
{
  GObjectClass *class;
  GParamSpec* pspec;
  
  class = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) r_vcard_dispose;
  class->finalize = (GObjectFinalizeFunc) r_vcard_finalize;
  
  class->set_property = r_vcard_set_property;
  class->get_property = r_vcard_get_property;

  pspec = g_param_spec_int("token",
			   "token",
			   "token",
			   N,
			   BAD_TOKEN,
			   BAD_TOKEN,
			   G_PARAM_READWRITE);
  g_object_class_install_property(class, VCARD_TOKEN, pspec);

  pspec = g_param_spec_int("param",
			   "param",
			   "param",
			   TYPE,
			   BAD_PARAM,
			   BAD_PARAM,
			   G_PARAM_READWRITE);
  g_object_class_install_property(class, VCARD_PARAM, pspec);

  pspec = g_param_spec_int("value",
			   "value",
			   "value",
			   HOME,
			   BAD_VALUE,
			   BAD_VALUE,
			   G_PARAM_READWRITE);
  g_object_class_install_property(class, VCARD_VALUE, pspec);
}


static void 
r_vcard_dispose (RVCard* vcard)
{
  g_return_if_fail(IS_R_VCARD(vcard));

  if (vcard->priv->dispose_has_run)
    return;

  vcard->priv->dispose_has_run = TRUE;
}


static void 
r_vcard_finalize (RVCard* vcard)
{
  g_return_if_fail(IS_R_VCARD(vcard));
  
  g_free(vcard->priv);
  vcard->priv = NULL;
}


static void 
r_vcard_set_property (GObject* obj, guint property_id,
		      const GValue* value, GParamSpec* spec)
{
  RVCard* self = (RVCard*) obj;

  switch(property_id)
    {
    case VCARD_TOKEN:
      self->priv->token = g_value_get_int(value);
      break;

    case VCARD_PARAM:
      self->priv->param = g_value_get_int(value);
      break;
      
    case VCARD_VALUE:
      self->priv->value = g_value_get_int(value);
      break;

    default:
      break;
    }
}

static void 
r_vcard_get_property (GObject* obj, guint property_id,
		      GValue* value, GParamSpec* spec)
{
  RVCard* self = (RVCard*) obj;

  switch(property_id)
    {
    case VCARD_TOKEN:
      g_value_set_int(value, self->priv->token);
      break;

    case VCARD_PARAM:
      g_value_set_int(value, self->priv->param);
      break;

    case VCARD_VALUE:
      g_value_set_int(value, self->priv->value);   
      break;
      
    default:
      break;
    }
}


static gchar* 
r_vcard_get_data(RVCard* vcard)
{
  g_return_val_if_fail(IS_R_VCARD(vcard), "");

  return vcard->priv->data->str;
}


static gchar* 
r_vcard_get_param_value(RVCard* vcard)
{
  RParams* rparam = NULL;
  
  g_return_val_if_fail(IS_R_VCARD(vcard), NULL);
  
  if (vcard->priv->values)
    rparam = vcard->priv->values->data;
  
  return rparam->str;
}


void 
r_vcard_set_param_value (RVCard* vcard, gchar *string)
{  
  RParams* rparam;
  gint value;
	      
  g_return_if_fail(IS_R_VCARD(vcard));

  value = r_lookup_table_str2enum(values, string);
  
  rparam = g_malloc0(sizeof(RParams));
  if (!rparam)
    g_error("not enough memory");
  
  if ((vcard->priv->param == RATE) || (vcard->priv->param == OWNER))
    rparam->str = g_strdup(string);
  else
    rparam->str = NULL;
    
  rparam->param = vcard->priv->param;
  rparam->value = value;
  
  vcard->priv->values = g_list_append(vcard->priv->values, rparam);
}



static RDate*
r_vcard_decode_date (gchar* buffer)
{
  g_return_val_if_fail(buffer!= NULL, NULL);

  /* date: 2001-10-21T15:53:29
           1973-03-18
  */
  RDate* date;
  struct tm tm;
  gchar *tmp;
  gboolean time2RData = TRUE;
  
  date = r_date_new();

  tmp = buffer;
  for (; *tmp; tmp++)
    if (*tmp == '-')
      time2RData = FALSE;
  
  if (!time2RData)
    {
      gchar *str, *tok;

      str = g_strdup(buffer);
      tok = strtok(str, "-");
      tm.tm_year = atoi(tok) - 1900;
      tok = strtok(NULL, "-");
      tm.tm_mon = atoi(tok) - 1;
      tok = strtok(NULL, "T");
      tm.tm_mday = atoi(tok);
      g_free(str);
      
      /*
	if (has_hour)
	{
	tok = strtok(NULL, ":");
	tm.tm_hour = atoi(tok);
	tok = strtok(NULL, ":");
	tm.tm_min = atoi(tok);
	tok = strtok(NULL, "");
	tm.tm_sec = atoi(tok);
	}
      */
    }
  else
    {
      time_t t;
      
      t = atoi(buffer);
      localtime_r (&t, &tm); 
    }

  g_object_set(date, "know", TRUE, "day", tm.tm_mday,
	       "month", tm.tm_mon, "year", tm.tm_year, NULL);
 
  return date;
}



static void 
r_vcard_free_values(RVCard* vcard)
{
  GList* l;
  g_return_if_fail(IS_R_VCARD(vcard));

  l = vcard->priv->values;
  for (; l; l = l->next)
    {
      RParams* rparam = (RParams*) l->data;

      g_free(rparam->str);
      g_free(rparam);
    }
  vcard->priv->values = NULL;
}


static RCompanyCard* 
r_vcard_company_new(RVCard *vcard, RCard* card)
{
  RRef* ref;
  RCompanyCard* company = NULL;
  gulong id;
  gchar* info;
      
  vcard->priv->have_company = TRUE;

  g_object_get(R_CARD(card), "card-id", &id, "card-name", &info, NULL);

  ref = r_ref_new(id);
  g_object_set(ref, "ref-info", info, NULL);

  company = r_company_card_new();
  r_card_add_ref(R_CARD(company), ref);	  
  
  return company;
}


static RAddress*
r_vcard_company_address_new(RVCard *vcard)
{
  g_return_val_if_fail(IS_R_VCARD(vcard), NULL);
  
  vcard->priv->have_company_address = TRUE;

  return r_address_new();
}


gchar*
r_vcard_validate_data(gchar* str, int n)
{
  gchar* tmp, *ret;
  
  tmp = str;
  for (; *tmp; tmp++)
    if (tmp && *tmp == ';')
      n--;      
  
  if (n > 0)
    {
      gchar* buf;
      
      buf = g_strnfill(n-1, ';');
      ret = g_strdup_printf("%s%s", str, buf);
      g_free(buf);
    }
  else
    ret = g_strdup(str);
  
  return ret;
}

void 
r_vcard_build_card(RVCard *vcard, RAbook* abook)
{
  GList* lst = NULL;
  RDate* date;
  RGroup* group;
  gint i=0, day, month, year;

  g_return_if_fail(IS_R_VCARD(vcard));

  switch(vcard->priv->token)
    {
    case BEGIN:
      card    = r_personal_card_new();
      contact = r_contact_new();
      work    = r_work_new();
      notes   = r_notes_new();
      break;
      
    case END:
      r_personal_card_set_contact(card, contact);
      r_personal_card_set_notes(card, notes);
      r_personal_card_set_work(card, work);     

      r_abook_add_loaded_card(R_ABOOK(abook), R_CARD(card));
      g_signal_emit_by_name(R_ABOOK(abook), "card_read", card, G_TYPE_POINTER);

      if (vcard->priv->have_company)
	{
	  r_card_add_address(R_CARD(company), company_address);
	  vcard->priv->have_company = FALSE;
	  vcard->priv->have_company_address = FALSE;

	  r_abook_add_loaded_card(R_ABOOK(abook), R_CARD(company));
	  g_signal_emit_by_name(R_ABOOK(abook), "card_read", company,
				G_TYPE_POINTER);
	}
      break;

    case FN:
      /* card name */
      g_object_set(R_CARD(card), "card-name", vcard->priv->data->str, NULL);
      break;
      
    case N:
      /* ord ==> surname (last name), (given) name, (middle) second name,
	         prefix, suffix (title) 
	         cognome, nome, secondo nome, prefisso, suffisso (titolo) 
      */
      if (vcard->priv->data->str &&
	  (g_ascii_strcasecmp(vcard->priv->data->str, "") != 0))
	{
	  gchar *data;
	  data = r_vcard_validate_data(vcard->priv->data->str, 5);
	  
	  decoded = g_strsplit(data, ";", 5);   
	  g_object_set(contact, 
		       "last-name",   decoded[0],
		       "first-name",  decoded[1], 
		       "middle-name", decoded[2], 
		       "prefix",      decoded[3], 
		       "title",       decoded[4], NULL);
	  g_strfreev(decoded);
	  g_free(data);
	}
      break;

    case TZ:
      break;

    case REV:
      break;

    case ADR:
      /* mail address + mail type
      */

      /* pbox ==> decoded[0], extadd ==> decoded[1] */ 
      for (lst = vcard->priv->values; lst; lst = lst->next)
	{	  
	  RParams* rparam = (RParams*) lst->data;
	  RAddressType addtype;
	  
	  if (vcard->priv->data->str &&
	      (g_ascii_strcasecmp(vcard->priv->data->str, "") != 0))
	    {
	      gchar *data;
	      
	      data    = r_vcard_validate_data(vcard->priv->data->str, 7);
	      decoded = g_strsplit(data, ";", 7);	      
	    }
	  else
	    break;
	  
	  address = r_address_new();
	  g_object_set(address, 
		       "street",   decoded[2], 
		       "city",     decoded[3],
		       "state",    decoded[4],
		       "province", decoded[4], 
		       "zip",      decoded[5], 
		       "country",  decoded[6], NULL);
	  
	  addtype = r_lookup_table_get_pair_right(address_pairs,
						  rparam->value);
	  
	  g_object_set(address, "address-type", addtype, NULL);
	  r_card_add_address(R_CARD(card), address);
	  g_strfreev(decoded);
	}
      break;
      
    case TEL:
      for (lst = vcard->priv->values; lst; lst = lst->next)
	{
	  RParams* rparam = (RParams*) lst->data;
	  RTelephoneType type;
	  
	  tel = r_telephone_new();

	  type = r_lookup_table_get_pair_right(phone_pairs, rparam->value);
	  
	  g_object_set(tel, 
		       "telephone-number", r_vcard_get_data(vcard), 
		       "telephone-type", type, 
		       NULL);
	  
	  r_card_add_telephone(R_CARD(card), tel);	  
	}
      break;
      
    case EMAIL:  
      net = r_net_address_new();
      
      g_object_set(net, 
		   "url",      r_vcard_get_data(vcard), 
		   "url-type", R_NET_ADDRESS_EMAIL, 
		   NULL);
      r_card_add_net_address(R_CARD(card), net);
      break;

    case URL:      
      net = r_net_address_new();
      
      g_object_set(net, 
		   "url",      r_vcard_get_data(vcard), 
		   "url-type", R_NET_ADDRESS_WEB,
		   NULL);
      r_card_add_net_address(R_CARD(card), net);
     break;

    case GEO:
      break;

    case BDAY:
      date = r_vcard_decode_date(r_vcard_get_data(vcard)); 
      g_object_get(date, "day",&day, "month", &month, "year", &year, NULL);
      r_contact_set_birthday(contact, day, month, year);
      break;
      
    case TITLE:
    case KDE_EXT_Profession:
      g_object_set(contact, "profession", r_vcard_get_data(vcard), NULL);
      break;

    case ORG:
      g_object_set(work, "organization", r_vcard_get_data(vcard), NULL);
      break;

    case CATEGORIES:
      categories = g_strsplit(r_vcard_get_data(vcard), ",", 0);
      for (i = 0; categories[i]; i++)
	{
	  RGroup* group;

	  group = r_group_new();
	  g_object_set(group, 
		       "group-name", categories[i], 
		       "group-owner", "user",
		       NULL);
	  
	  r_card_add_group(R_CARD(card), group);
	}
      g_strfreev(categories); 
      break;

    case RUBRICA_EXT_Group:
      categories = g_strsplit(r_vcard_get_data(vcard), ";", 0);
      
      group = r_group_new();
      g_object_set(group, 
		   "group-name",   categories[0], 
		   "group-pixmap", categories[1],
		   "group-owner",  r_vcard_get_param_value(vcard),
		   NULL);
      
      r_card_add_group(R_CARD(card), group);
      g_strfreev(categories);      
      break;
     
    case NOTE:
      g_object_set(notes, "other-notes", r_vcard_get_data(vcard), NULL);
      break;
      
    case KEY: 
      g_object_set(notes, "pubkey", r_vcard_get_data(vcard), NULL);
      break;

    case ROLE:
      g_object_set(work, "assignment", r_vcard_get_data(vcard), NULL);
      break;

      /*   TODO -- recuperare campi condivisi
       */
    case UID:
      break;

    case LOGO:
      break;

    case PHOTO:
      break;

    case LABEL:
      break;

    case AGENT:
      break;

    case PROID:
      break;

    case SOUND:
      break;

    case CLASS:
      break;

    case MAILER:
      break;

    case VERSION:
      break;

    case NICKNAME:
      g_object_set(contact, "nick-name", r_vcard_get_data(vcard), NULL);
      break;
      
    case RUBRICA_EXT_Card:      
      categories = g_strsplit(r_vcard_get_data(vcard), ";", 0);
      if (categories)
	{
	  gboolean locked = TRUE, deleted = FALSE;

	  if (g_ascii_strcasecmp(g_ascii_strdown(categories[0], -1), 
				 "true") == 0)
	    locked = TRUE;
	  
	  if (g_ascii_strcasecmp(g_ascii_strdown(categories[1], -1),
				 "true") == 0)
	    deleted = TRUE;

	  g_object_set(card, 
		       "card-rate", atoi(r_vcard_get_param_value(vcard)), 
		       "card-locked",  locked,
		       "card-deleted", deleted,
		       NULL);
	  
	  g_strfreev(categories); 
	}
      break;

    case RUBRICA_EXT_Ref:
      // ref id
      break;

    case RUBRICA_EXT_Url:
      // other url
      break;
      
    case RUBRICA_EXT_Department:
    case KDE_EXT_Department:
      g_object_set(work, "department", r_vcard_get_data(vcard), NULL);
      break;
      
    case RUBRICA_EXT_SubDepartment:
      g_object_set(work, "sub-department", r_vcard_get_data(vcard), NULL);
      break;

    case RUBRICA_EXT_Collaborator:
    case KDE_EXT_AssistantsName:
    case EVOLUTION_EXT_Assistant:
      g_object_set(work, "collaborator", r_vcard_get_data(vcard), NULL);
      break;
      
    case RUBRICA_EXT_CollaboratorTelephone:
      g_object_set(work, "collaborator-phone", r_vcard_get_data(vcard), NULL);
      break;
      
    case RUBRICA_EXT_SpouseName:
    case KDE_EXT_SpousesName:
    case EVOLUTION_EXT_Spouse:
      g_object_set(notes, "has-partner", TRUE, 
		   "partner-name", r_vcard_get_data(vcard), NULL);
      break;
      
    case RUBRICA_EXT_Child:
      g_object_set(notes, "children", r_vcard_get_data(vcard), NULL);
      break;

    case RUBRICA_EXT_Anniversary:
    case KDE_EXT_Anniversary:
    case EVOLUTION_EXT_Anniversary:
      date = r_vcard_decode_date(r_vcard_get_data(vcard));
      r_notes_set_know_anniversary(notes, TRUE);
      g_object_get(date, "day",&day, "month", &month, "year", &year, NULL);
      r_notes_set_anniversary(notes, day, month, year);
    break;
      
    case RUBRICA_EXT_Manager:
    case KDE_EXT_ManagersName:
    case EVOLUTION_EXT_Manager:
      g_object_set(work, "manager-name", r_vcard_get_data(vcard), NULL);
      break;

    case RUBRICA_EXT_ManagerTelephone:
      // todo
      break;

    case KDE_EXT_Office:
    case EVOLUTION_EXT_Office:
      break;

    case KDE_EXT_CUSTOM:  
      break;
      
      /* Company tokens 
       */
    case RUBRICA_EXT_CompanyName:
      if (!vcard->priv->have_company) 
	company = r_vcard_company_new(vcard, R_CARD(card));
      
      g_object_set(company, "company-name", r_vcard_get_data(vcard), NULL); 
      g_object_set(R_CARD(company), "card-name", r_vcard_get_data(vcard),
		   NULL);
      break;

    case RUBRICA_EXT_Street:
      if (!vcard->priv->have_company)
	company = r_vcard_company_new(vcard, R_CARD(card));
       
      if (!vcard->priv->have_company_address)
	company_address = r_vcard_company_address_new(vcard);
 
      g_object_set(company_address, "street", r_vcard_get_data(vcard), NULL);
      break;
      
    case RUBRICA_EXT_StreetNumber:
      if (!vcard->priv->have_company)
	company = r_vcard_company_new(vcard, R_CARD(card));
       
      if (!vcard->priv->have_company_address)
	company_address = r_vcard_company_address_new(vcard);
 
      g_object_set(company_address, "street-number", 
		   r_vcard_get_data(vcard), NULL);
      break;
      
    case RUBRICA_EXT_ZipCode:
      if (!vcard->priv->have_company) 
	company = r_vcard_company_new(vcard, R_CARD(card));
       
      if (!vcard->priv->have_company_address)
	company_address = r_vcard_company_address_new(vcard);
 
      g_object_set(company_address, "zip", r_vcard_get_data(vcard), NULL);
      break;
      
    case RUBRICA_EXT_City:
      if (!vcard->priv->have_company) 
	company = r_vcard_company_new(vcard, R_CARD(card));
 
      if (!vcard->priv->have_company_address)
	company_address = r_vcard_company_address_new(vcard);
      
      g_object_set(company_address, "city", r_vcard_get_data(vcard), NULL);
      break;
      
    case RUBRICA_EXT_Province:
      if (!vcard->priv->have_company)
	company = r_vcard_company_new(vcard, R_CARD(card));
 
      if (!vcard->priv->have_company_address)
	company_address = r_vcard_company_address_new(vcard);
 
      g_object_set(company_address, "province", r_vcard_get_data(vcard), NULL);
      break;
      
    case RUBRICA_EXT_Country:
      if (!vcard->priv->have_company) 
	company = r_vcard_company_new(vcard, R_CARD(card));
      
      if (!vcard->priv->have_company_address)
	company_address = r_vcard_company_address_new(vcard);
 
      g_object_set(company_address, "country", r_vcard_get_data(vcard), NULL);
      break;
      
    case RUBRICA_EXT_Web:
      if (!vcard->priv->have_company) 
	company = r_vcard_company_new(vcard, R_CARD(card));
      
      company_net = r_net_address_new();
      g_object_set(company_net, 
		   "url", r_vcard_get_data(vcard), 
		   "url-type", R_NET_ADDRESS_WEB, NULL);
      r_card_add_net_address(R_CARD(company), company_net);
      break;
      
    case RUBRICA_EXT_Email:
      if (!vcard->priv->have_company) 
	company = r_vcard_company_new(vcard, R_CARD(card));
      
      company_net = r_net_address_new();
      g_object_set(company_net, "url", r_vcard_get_data(vcard), 
		   "url-type", R_NET_ADDRESS_EMAIL, NULL);
      r_card_add_net_address(R_CARD(company), company_net);
      break;
      
    case RUBRICA_EXT_Operator:
      if (!vcard->priv->have_company) 
	company = r_vcard_company_new(vcard, R_CARD(card));
	  
      company_telephone = r_telephone_new();
      g_object_set(company_telephone, 
		   "telephone-number", r_vcard_get_data(vcard), 
		   "telephone-type", R_TELEPHONE_OPERATOR, NULL);
      r_card_add_telephone(R_CARD(company), company_telephone);
      break;

    case RUBRICA_EXT_Fax:
      if (!vcard->priv->have_company) 
	company = r_vcard_company_new(vcard, R_CARD(card));
 
      company_telephone = r_telephone_new();
      g_object_set(company_telephone, 
		   "telephone-number", r_vcard_get_data(vcard), 
		   "telephone-type", R_TELEPHONE_FAX, NULL);
      r_card_add_telephone(R_CARD(company), company_telephone);
      break;
      
    case RUBRICA_EXT_Green:
      if (!vcard->priv->have_company) 
	company = r_vcard_company_new(vcard, R_CARD(card));
 
      company_telephone = r_telephone_new();
      g_object_set(company_telephone, 
		   "telephone-number", r_vcard_get_data(vcard), 
		   "telephone-type", R_TELEPHONE_GREEN, NULL);      
      r_card_add_telephone(R_CARD(company), company_telephone);   
      break;
      
    case RUBRICA_EXT_CustomerCare:
      if (!vcard->priv->have_company) 
	company = r_vcard_company_new(vcard, R_CARD(card));
      
      company_telephone = r_telephone_new();
      g_object_set(company_telephone, 
		   "telephone-number", r_vcard_get_data(vcard), 
		   "telephone-type", R_TELEPHONE_CUSTOMER_CARE, NULL);
      r_card_add_telephone(R_CARD(company), company_telephone);
      break;

    case EVOLUTION_EXT_File_as:
      break;

    case BAD_TOKEN:   
    default:   
      break;
    }
}


/* Public functions
 */

RVCard* 
r_vcard_new(void)
{
  RVCard *vcard;

  vcard = g_object_new(r_vcard_get_type(), NULL);

  return (RVCard*) vcard;
}


void 
r_vcard_free (RVCard *vcard)
{
  g_return_if_fail(IS_R_VCARD(vcard));

  g_object_unref(vcard);
}



static gboolean 
r_vcard_open_file (RAbook* abook, gchar* filename)
{
  RVCard* vcard;
  
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  
  if (!filename)
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", 
			    NO_FILENAME, G_TYPE_INT);

      return FALSE;
    }

  if (!g_file_test(filename, G_FILE_TEST_EXISTS))
    {
      g_signal_emit_by_name(R_ABOOK(abook), "open_fail", 
			    FILE_NOT_EXIST, G_TYPE_INT);
      
      return FALSE;
    }
  
  vcard = (RVCard*) r_abook_get_plugin(abook);
  
  if (!r_vcard_read_file(vcard, abook, filename))
    return FALSE;
  else
    g_signal_emit_by_name(R_ABOOK(abook), "addressbook_read", 
			  NULL, G_TYPE_NONE);
  
  g_object_set(R_ABOOK(abook),
	       "addressbook-name", g_path_get_basename(filename),
	       "addressbook-path", g_path_get_dirname(filename),
	       NULL);

  return TRUE;
}

static void
write_card (RCard* card, FILE* fp)
{
  gpointer data;
  gchar *id, *name, *cardtype, *tmp, *type;
  gboolean locked, deleted;
  gint created, changed;
  RRate rate;
  gchar *day, *month, *year;
  gchar *first, *middle, *last, *nick, *prefix, *prof, *title, *photo;

  fprintf(fp, "\n");

  fprintf(fp, "BEGIN:VCARD\n");
  fprintf(fp, "VERSION:3.0\n");

  g_object_get(card, 
	       "card-id",      &id,      "card-type",    &cardtype,
	       "card-name",    &name,    "card-locked",  &locked, 
	       "card-deleted", &deleted, "card-rate",    &rate, 
	       "card-created", &created, "card-changed", &changed,
	       NULL);
  
  if (name)
    fprintf(fp, "FN:%s\n", name);

  /*      Card's data (rubrica's extensions)
  */
  tmp = g_strdup_printf("%d", rate);
  fprintf(fp, "X-RUBRICA-X-Card;RATE=%s:%s;%s\n", tmp,
	  locked  ? "true" : "false", deleted ? "true" : "false");
  g_free(tmp);
  
  /*      Groups (rubrica's extensions)
  */
  data = r_card_get_group(R_CARD(card));
  for (; data; data = r_card_get_next_group(R_CARD(card)))
    {
      gchar* owner;
      gchar* pixmap = NULL;

      g_object_get(R_GROUP(data), 
		   "group-name",   &name, 
		   "group-owner",  &owner, 
		   "group-pixmap", &pixmap, NULL);
      
      fprintf(fp, "X-RUBRICA-X-Group;OWNER=%s:%s;%s\n", 
	      owner, 
	      name, 
	      pixmap);
    }


  /*      References (rubrica's extensions)
  */
  data = r_card_get_ref(R_CARD(card));
  for (; data; data = r_card_get_next_ref(R_CARD(card)))
    {
      glong id;
      gchar *info, *tmp;

      g_object_get(R_REF(data), "ref-to", &id, "ref-info", &info, NULL);
      tmp = g_strdup_printf("%ld", id);
      
      fprintf(fp, "X-RUBRICA-X-Ref;ID=%s:%s\n", tmp, info);
      g_free(tmp);
    }  

  if (g_ascii_strcasecmp(cardtype, "personal") == 0)
    {
      /*      Personal data
       */
      data = r_personal_card_get_contact(R_PERSONAL_CARD(card));
      g_object_get(R_CONTACT(data), "first-name", &first, 
		   "middle-name", &middle, "last-name", &last, 
		   "nick-name", &nick, "profession", &prof, 
		   "prefix", &prefix, "title", &title, "photo", &photo, NULL);

      if (first || middle || last || prefix || title)
	fprintf(fp, "N:%s;%s;%s;%s;%s\n", 
		last   ? last   : "", 
		first  ? first  : "",
		middle ? middle : "",
		prefix ? prefix : "",
		title  ? title  : "");
    
      day   = r_contact_get_birth_day   (R_CONTACT(data));
      month = r_contact_get_birth_month (R_CONTACT(data));
      year  = r_contact_get_birth_year  (R_CONTACT(data)); 
      if ((day   && (g_ascii_strcasecmp(day, "BadDay")   != 0)) && 
	  (month && (g_ascii_strcasecmp(day, "BadMonth") != 0)) &&
	  (year  && (g_ascii_strcasecmp(day, "BadYear")  != 0)))	
	fprintf(fp, "BDAY:%s-%s-%s\n", day, month, year);      
    }
  else
    {
      /*    TODO ***** Company  ******
	    
      fprintf(fp, ":%s\n", );
      
      
      
      */
    } 
  
  /*      Addresses
  */
  data = r_card_get_address(R_CARD(card));
  for (; data; data = r_card_get_next_address(R_CARD(card)))
    {
      if (IS_R_ADDRESS(data))
	{
	  RAddressType adtype = R_ADDRESS_UNKNOWN;
	  gchar *street, *number, *city, *zip;
	  gchar *province, *state, *country;

	  g_object_get(R_ADDRESS(data), 
		       "address-type", &adtype, "street", &street, 
		       "street-number", &number,  "city", &city, 
		       "zip", &zip, "province", &province, 
		       "state", &state, "country", &country, NULL);
	  
	  type = r_address_lookup_enum2str(adtype);
	  if (street || city || state || zip || country)
	    fprintf(fp, "ADR;TYPE=%s:%s;%s;%s;%s;%s;%s;%s\n", type, "", "",
		    street  ? street  : "", 
		    city    ? city    : "", 
		    state   ? state   : "", 
		    zip     ? zip     : "", 
		    country ? country : "");
	}
    }
  
  
  /*      Net (emails, web urls)
  */
  data = r_card_get_net_address(R_CARD(card));
  for (; data; data = r_card_get_next_net_address(R_CARD(card)))
    if (IS_R_NET_ADDRESS(data))
      {
	RNetAddressType nettype;
	gchar *url;
	
	g_object_get(R_NET_ADDRESS(data), "url", &url, 
		     "url-type", &nettype, NULL);
	type = r_net_address_decode_type(nettype);
	
	switch (nettype)
	  {
	  case R_NET_ADDRESS_EMAIL: 
	    fprintf(fp, "EMAIL;INTERNET:%s\n", url);
	    break;

	  case R_NET_ADDRESS_WEB:
	    fprintf(fp, "URL:%s\n", url);
	    break;

	  case R_NET_ADDRESS_EKIGA:
	  case R_NET_ADDRESS_IRC:
	  case R_NET_ADDRESS_IRC_AIM:
	  case R_NET_ADDRESS_IRC_ICQ:
	  case R_NET_ADDRESS_IRC_JABBER:
	  case R_NET_ADDRESS_IRC_YAHOO:
	  case R_NET_ADDRESS_IRC_MSN:
	  case R_NET_ADDRESS_WORK_WEB:
	  case R_NET_ADDRESS_WORK_EMAIL:
	  case R_NET_ADDRESS_UNKNOWN:
	    fprintf(fp, "X-RUBRICA-X-URL;TYPE=%s:%s\n", type, url);
	    break;
	    
	  default:
	    break;	      
	  }
      }
     
  
  /*     Telephone
  */
  data = r_card_get_telephone(R_CARD(card));
  for (; data; data = r_card_get_next_telephone(R_CARD(card)))
    if (IS_R_TELEPHONE(data))
      {
	gchar *number, *type; 
	RTelephoneType ttype;
	
	g_object_get(R_TELEPHONE(data), "telephone-number", &number, 
		     "telephone-type", &ttype, NULL);
	type = r_telephone_lookup_enum2str(ttype);
	
	fprintf(fp, "TEL;TYPE=%s:%s\n", type, number);
      }


  /*      Work
  */
  if (g_ascii_strcasecmp(cardtype, "personal") == 0)
    {
      data = r_personal_card_get_work(R_PERSONAL_CARD(card));
      if (IS_R_WORK(data))
	{     
	  gchar *assignment, *org, *dep, *subdep;
	  gchar *manager, *mphone, *collaborator, *cphone;
	  
	  g_object_get(R_WORK(data), 
		       "assignment",     &assignment, 
		       "organization",   &org, 
		       "department",     &dep, 
		       "sub-department", &subdep, 
		       "manager-name",   &manager, 
		       "manager-phone",  &mphone, 
		       "collaborator",   &collaborator, 
		       "collaborator-phone", &cphone, NULL);
	  if (assignment)
	    fprintf(fp, "ROLE:%s\n", assignment);
	  if (org)
	    fprintf(fp, "ORG:%s\n", org);
	  if (dep)
	    fprintf(fp, "X-RUBRICA-Department:%s\n", dep);
	  if (subdep)
	    fprintf(fp, "X-RUBRICA-X-SubDepartment:%s\n", subdep);
	  if (manager)
	    fprintf(fp, "X-RUBRICA-X-Manager:%s\n", manager);
	  if (mphone)
	    fprintf(fp, "X-RUBRICA-X-ManagerTelephone:%s\n", mphone);
	  if (collaborator)
	    fprintf(fp, "X-RUBRICA-X-Collaborator:%s\n", collaborator);
	  if (cphone)
	    fprintf(fp, "X-RUBRICA-X-CollaboratorTelephone:%s\n", cphone);
	} 
    }
  
  
  /*      Notes
  */
  if (g_ascii_strcasecmp(cardtype, "personal") == 0)
    {
      data = r_personal_card_get_notes(R_PERSONAL_CARD(card));
      if (IS_R_NOTES(data))
	{
	  gchar *other, *pkey;
	  
	  g_object_get(R_NOTES(data), 
		       "other-notes", &other, 
		       "pubkey", &pkey, NULL);
	  
	  if (other)
	    fprintf(fp, "NOTE:%s\n", other);
	  
	  if (pkey)
	    fprintf(fp, "KEY:%s\n", pkey);  
	}
    }

  fprintf(fp, "END:VCARD\n\n");
}

static gboolean 
r_vcard_write_file (RAbook* abook, gchar* filename, gint crate)
{
  FILE* fp;

  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  g_return_val_if_fail(filename != NULL, FALSE);
  
  if (!(fp = fopen(filename, "w")))
    {
      g_warning("\nCan't write file: %s", filename);
      g_signal_emit_by_name(abook, "save_fail", WRITING_FILE, G_TYPE_INT);
      
      return FALSE;
    }
  
  r_abook_foreach_card (abook, (RFunc) write_card, fp);
  fflush(fp);

  g_signal_emit_by_name(abook, "addressbook_saved", NULL, G_TYPE_NONE);

  return TRUE;
}



gboolean 
r_vcard_read_file (RVCard* vcard, RAbook* abook, gchar* filename)
{
  ParserStatus status;
  GString *buffer;
  gunichar ch = EOF;
  gboolean read = FALSE;

  g_return_val_if_fail(IS_R_VCARD(vcard), FALSE);
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  
  vcard->fp = fopen(filename, "r");
  if (!vcard->fp)
    return FALSE;

  buffer = g_string_new(NULL);
  status = VCARD_START;
  do
    {
      ch = fgetc(vcard->fp);

      switch(status)
	{
	case VCARD_START:
	  // card_new 
	  if ((ch != '\n') && (ch != '\r') && (ch != ' '))
	    {
	      buffer = g_string_append_unichar(buffer, ch); 
	      status = VCARD_BUILD_TOKEN;
	    }
	  break;

	case VCARD_BUILD_TOKEN:          // building token (until ch != :)     
	  if (ch == VCARD_TOKEN_DELIM)   // :
	    {
	      gint token;
	      
	      token = r_lookup_table_str2enum(tokens, buffer->str);
	      //	      r_vcard_decode_token(vcard, buffer->str);
	      g_object_set(vcard, "token", token, NULL);
	      g_string_truncate(buffer, 0);
	      
	      status = VCARD_BUILD_DATA;
	      break;
	    }
	  
	  if (ch == VCARD_PARAM_DELIM)   // ;
	    {
	      gint token;
	      
	      token = r_lookup_table_str2enum(tokens, buffer->str);
	      g_object_set(vcard, "token", token, NULL);
	      g_string_truncate(buffer, 0);

	      status = VCARD_BUILD_PARAM;
	      break;
	    }
	  
	  if (ch == ' ')
	    break;
	  
	  if (g_unichar_isalpha(ch) || g_unichar_isgraph(ch))
	    buffer = g_string_append_unichar(buffer, ch); 	  
	  break;

	case VCARD_BUILD_PARAM:        // building parameters (until :)	  
	  if (ch == VCARD_VALUE_DELIM) // =  (TYPE=...;RATE=...;
	    {
	      gint param;
	      
	      param = r_lookup_table_str2enum(params, buffer->str); 
	      g_object_set(vcard, "param", param, NULL);
	      g_string_truncate(buffer, 0);

	      status = VCARD_BUILD_PARAM_VALUES;
	      break;
	    }
	  
	  if (ch == VCARD_PARAM_DELIM) // ; (WORK;VOICE;FAX;...)
	    {
	      g_object_set(vcard, "param", TYPE, NULL);
	      r_vcard_set_param_value(vcard, buffer->str);
	      g_string_truncate(buffer, 0);

	      status = VCARD_BUILD_PARAM;
	      break;
	    }
	   
	  if (ch == VCARD_TOKEN_DELIM) // : (EMAIL;INTERNET:)
	    {	      
	      g_object_set(vcard, "param", TYPE, NULL);
	      r_vcard_set_param_value(vcard, buffer->str);
	      g_string_truncate(buffer, 0);

	      status = VCARD_BUILD_DATA;
	      break;
	    }
	  
	  if (g_unichar_isalpha(ch))
	    {
	      buffer = g_string_append_unichar(buffer, ch);
	      status = VCARD_BUILD_PARAM;
	    }	  
	  break;

	case VCARD_BUILD_PARAM_VALUES:     //  (RATE=4: TYPE=home:)
	  if (ch == VCARD_TOKEN_DELIM)     // :
	    {
	      r_vcard_set_param_value(vcard, buffer->str);
	      g_string_truncate(buffer, 0);
	      
	      status = VCARD_BUILD_DATA;
	      break;
	    }
	  
	  if (ch == VCARD_PARAM_DELIM)    // ;  (TYPE=...;)
	    {
	      r_vcard_set_param_value(vcard, buffer->str);
	      g_string_truncate(buffer, 0);

	      status = VCARD_BUILD_PARAM;    
	      break;
	    }
	  
	  if (g_unichar_isalnum(ch))
	    {
	      buffer = g_string_append_unichar(buffer, ch);
	      status = VCARD_BUILD_PARAM_VALUES;
	    }	  	  
	  break;

	case VCARD_BUILD_VALUES:
	  g_print("\nVCARD_BUILD_VALUES");
	  if (ch == VCARD_PARAM_DELIM) // ;
	    {
	      gint value;
	      
	      value = r_lookup_table_str2enum(values, buffer->str);
	      g_object_set(vcard, "value", value, NULL);
	      g_string_truncate(buffer, 0);

	      status = VCARD_BUILD_PARAM; 
	      break;
	    }

	  if (ch == VCARD_TOKEN_DELIM) // :
	    {
	      gint value;
	      
	      value = r_lookup_table_str2enum(values, buffer->str);
	      g_object_set(vcard, "value", value, NULL);
	      g_string_truncate(buffer, 0);

	      status = VCARD_BUILD_DATA;
	      break;
	    }
	  
	  buffer = g_string_append_unichar(buffer, ch);
	  break;

	case VCARD_BUILD_DATA:
	  if (ch == '\t')
	    break;

	  if (ch == '\r')
	    {
	      buffer = g_string_append_unichar(buffer, '\0');
	      break;
	    }

	  if ((ch == '\n') || (ch == EOF))
	    {
	      if (g_utf8_validate(buffer->str, -1, NULL))
		{
		  if ((vcard->priv->param == ENCODING) && 
		      (vcard->priv->value == QUOTED_PRINTABLE))
		    {
		      gchar *p = NULL;
		      
		      if ((p = g_strrstr(buffer->str, "=0D=0A=")) ||
			  (p = g_strrstr(buffer->str, "=0A=0D=")) ||
			  (p = g_strrstr(buffer->str, "=0A=0D"))  ||
			  (p = g_strrstr(buffer->str, "=0A")))
			*p = '\0';
		    }
		  g_string_append(vcard->priv->data, buffer->str);
		  
		  r_vcard_build_card(vcard, abook);
		  g_string_truncate(buffer, 0);
		  g_string_truncate(vcard->priv->data, 0);
		  r_vcard_free_values(vcard);

		  status = VCARD_BUILD_TOKEN;
		  read = TRUE;
		}
	      break;
	    }

	  buffer = g_string_append_unichar(buffer, ch);
	  break;
	  
	case VCARD_END:
	  break;
	  
	default:
	  break;
	}

    } while(!feof(vcard->fp));

  fclose(vcard->fp);

  if (read)
    return TRUE;
  
  return FALSE;
}


static gboolean 
r_vcard_overwrite_file(RAbook* abook, gint compression_rate)
{
  gchar* file;
  gchar* path;
  gchar* filename;
  
  g_return_val_if_fail(IS_R_ABOOK(abook), FALSE);
  
  g_object_get(abook, "addressbook-path", &path, 
	       "addressbook-name", &file, NULL);
  filename = g_strdup_printf("%s%s%s", path, G_DIR_SEPARATOR_S, file);
  
  if (g_file_test(filename, G_FILE_TEST_EXISTS))
    g_remove(filename);

  if (!r_vcard_write_file(abook, filename, compression_rate))
    {
      g_signal_emit_by_name(abook, "save_fail", OVERWRITING, G_TYPE_INT);
      
      g_free(filename);
      return FALSE;
    }
    
  g_free(filename);
  
  return TRUE;
}


G_MODULE_EXPORT void
plugin_init (RPlugin* plugin, gchar* file)
{
  RFilter* filter;
  RPluginAction* action;

  g_return_if_fail(plugin != NULL);
  
  r_plugin_set_obj(plugin, r_vcard_new());
  g_object_set(plugin, 
	       "plugin-name", "vcard", 
	       "plugin-filename", file,
	       "plugin-info", "This plugin manages the vcard file format",
	       "plugin-configurable", FALSE, NULL);
  
  filter = r_filter_new();
  g_object_set(filter, 
	       "filter-name", "vcard", 
	       "filter-extension", "vcard",
	       "filter-mime", "text/x-vcard", NULL);
  r_filter_add_pattern(filter, "vcard");
  r_filter_add_pattern(filter, "vcr");
  r_filter_add_pattern(filter, "*.vcr");
  r_filter_add_pattern(filter, "*.vcard");

  r_plugin_add_filter (plugin, filter);
  
  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("read");
  action->handle = (gpointer) r_vcard_open_file;
  r_plugin_add_action(plugin, action);  
    
  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("write");
  action->handle = (gpointer) r_vcard_write_file;;
  r_plugin_add_action(plugin, action);  

  action = g_malloc(sizeof(RPluginAction));
  action->name   = g_strdup("overwrite");
  action->handle = (gpointer) r_vcard_overwrite_file;
  r_plugin_add_action(plugin, action);  
}


G_MODULE_EXPORT void     
plugin_fini (void)
{
  // TODO: liberare la memoria e rilasciare l'handler al plugin
}
