"""
Unit Test Case
"""
#  Copyright (C) 2004  Henning Jacobs <henning@srcco.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  $Id: testvcard.py 82 2004-07-11 13:01:44Z henning $

import unittest
from vcard import vCard, vCardList, vC_adr, vC_tel, vC_email

DEMONOTE = u"""This is a simple Test Case for my vCard Python Module.
Here comes german umlauts: \xe4\xc4\xfc\xdc\xf6\xd6\xdf (this string
is Unicode and will be written to disk as Latin-1)
Now some special vCard chars: \t (tab) ; : , ? (note the escaping!)
"""

def fillDemoCard(card):
    "Completely fill a vCard with demo values"
    card.fn.set("*Demo Card*")
    card.nickname.set("Nickname")
    card.bday.set("1983-08-19")
    card.n.prefixes.set("Prefixes")
    card.n.given.set("Given")
    card.n.additional.set("Additional")
    card.n.family.set("Family")
    card.n.suffixes.set("Suffixes")
    card.org.org.set("MyCompany Ltd.")
    card.org.units.set("My Division, My Office; Another Dep.")
    card.title.set("Title")
    card.role.set("Role")
    card.adr.append(vC_adr())
    card.adr[0].pobox.set("POBox")
    card.adr[0].extended.set("Extended")
    card.adr[0].street.set("Street")
    card.adr[0].postcode.set("Postal Code")
    card.adr[0].city.set("City")
    card.adr[0].region.set("Region")
    card.adr[0].country.set("Country")
    card.adr[0].params.get("type").add("pref")
    card.adr.append(vC_adr())
    card.adr[1].pobox.set("POBox2")
    card.adr[1].extended.set("Extended2")
    card.adr[1].street.set("Street2")
    card.adr[1].postcode.set("Postal Code2")
    card.adr[1].city.set("City2")
    card.adr[1].region.set("Region2")
    card.adr[1].country.set("Country2")
    card.label.set("My Address Label\nStreet Somewhere\nNoCity")
    card.tel.append(vC_tel())
    card.tel[0].set("02323-92348223-9 34")
    card.tel[0].params.get("type").extend(["isdn","home","msg"])
    card.tel.append(vC_tel())
    card.tel[1].set("01234-56789")
    card.tel[1].params.get("type").extend(["work","cell"])
    card.email.append(vC_email())
    card.email[0].set("email@my.domain.nowhere")
    card.email[0].params.get("type").add("pref")
    card.email.append(vC_email())
    card.email[1].set("email2@my.2nddomain.nowhere")
    card.note.set(DEMONOTE)
    card.categories.set("Demo, Business, Private, TestCase")
    card.url.set("http://www.mydomain.nowhere")
    card.uid.set("-//TEST//TestvCard@mydomain.nowhere//123456789")
    card.key.set("""-----BEGIN PGP PUBLIC KEY BLOCK-----
Version: GnuPG v1.0.4e (MingW32)
Comment: Weitere Infos: siehe http://www.gnupg.org

mQGiBD46zoIRBADiXv3YlcimwxS1H9THTuzkcP1Ozry88Xvja36ct9uMsjYTl8HK
1sYNBCEnjBKkqIEC4RW1iOfMgLhYeix4stSR7aa+09ZmOkSwE/nq/bNjKaGOjIRE
QHQEQQaXeD6DPm9kc9XHLMpYZoC4Ix8JKOrQeBUXekXgWJ0xaFXNEg7zewCgsowY
W/BC4Cl9AX8t6SBPWRr0RjkD/A3iZc8J8WMSrVa8XFbcpPh/RddiNCMc8u3V/Lll
Gym6LUvjcTDdoAuZLcT04/w3HcvJvroHhy4yLuo6wk8xODYEFsmQS12XsMQe5ifr
GlE3om9HboqfiLjeizU1vG/FP2WLI19fsKtVnfKbsNhkFR6ar9j1+kxYAFjWOLP7
LWiuBACClxFUDLl0SyXC0C2b3/Lv/VYpnwO6B7n5s1k559OfQHK1alEK88js4vEe
m0jD2n1vJZZlJZei8gzZFJeGh6XdMmcidRp8g5TnEvs84xM9QsU0Q64Uoqbcu6u0
rUp/zEnTANmghz2riBsIwj3Ne32FQyYCVx+d27hj/zqL41syfbQjSGVubmluZyBK
YWNvYnMgPGhlbm5pbmdAamFjb2JzMS5kZT6IVwQTEQIAFwUCPjrOggULBwoDBAMV
AwIDFgIBAheAAAoJEAiNyhwAo48FstgAn0tRSNClh76+Rg9SghQk6w+gixnpAKCb
lW0d9ncBts2z/YYT3Ve0xiPI6LkBDQQ+Os6XEAQAyVV8OTs2WbUv4/MtI9Gb5mY/
4uAHZU4+8fTToKb6P8IJzEe05hIMJcREvUOI8m6zsKxoGitC1KuNYATZsIQ3Ul7D
2Vf016zScaPwHtfkgMkE+6VOUP1lt+nZTLwa5NUxs236L4WA4cdRs+20EQtYk8A+
NXwZOJpbAP4Hcny8QA8ABA0EALJ9953d1Z654g1cWNM/qjLhWR8GzhYBFEI1hWGT
r4hS4NMhxBSWLTCW1COw+7Z8PhTnYZwyW/Zi4SIOmwkwpQkG3r3+Ao5jjmX/m4yX
SM+sfv6KArwDuXvGGtpZ7tH/MJcMZd7TJgGxxbVz18HYtr7u471V1XD3Z6dsRD1i
zxpuiEYEGBECAAYFAj46zpcACgkQCI3KHACjjwUtlQCfQAxUyDfEoQH5F2W8CRDz
iYZtIf4AoJtFvYGBxWRa3ZzkV9mOnlY1t/q0
=TPAr
-----END PGP PUBLIC KEY BLOCK-----""")
    card.mailer.set("Pine for Linux ver. 4.9")
    card.tz.set("+01:00")
    card.geo.set("53.5;10.0")

class vCardTestCase(unittest.TestCase):
    def setUp(self):
        self.cardlist = vCardList()
        self.democard = vCard()
        fillDemoCard(self.democard)
        
    def testReadWrite(self):
        "writing and reading vCard from stream"
        self.cardlist.add(self.democard)
        cardbefore = self.democard
        import StringIO
        stream = StringIO.StringIO()
        self.cardlist.SaveToStream(stream)
        stream.seek(0)
        self.cardlist.clear()
        self.cardlist.LoadFromStream(stream)
        handle = self.cardlist.sortedlist()[0]
        cardafter = self.cardlist[handle]
        self.assertEqual(cardbefore.VCF_repr(), cardafter.VCF_repr())

    def testSort(self):
        "vCardList sorting"
        card = vCard()
        card.n.family.set("Aaaa")
        hdl1 = self.cardlist.add(card)
        card = vCard()
        card.fn.set("Bbbb")
        card.sort_string.set("0000")
        hdl2 = self.cardlist.add(card)
        card = vCard()
        card.fn.set("Cccc")
        hdl3 = self.cardlist.add(card)
        self.assertEqual(self.cardlist.sortedlist(), [hdl2, hdl1, hdl3])
        
    def testFieldValue(self):
        "compare card values with our demo values"
        self.assertEqual(self.democard.getFieldValueStr("Note"), DEMONOTE)
        self.assertEqual(self.democard.getDisplayName(), "Family, Given")
        self.assertEqual(self.democard.getFieldValueStr("Street 1"), "Street")
        self.assertEqual(self.democard.getFieldValueStr("Phone 2"), "01234-56789 (cell, voice, work)")
        vals = []
        for i in range(2):
            vals.append(self.democard.getFieldValueStr("PostalCode"))
        self.assertEqual(tuple(vals), ("Postal Code", "Postal Code2"))

if __name__ == "__main__":
    unittest.main()

