///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __SCRIPT_EDITOR_H
#define __SCRIPT_EDITOR_H

#include <scripting_gui/ScriptingGUI.h>

// Include the Qscintilla headers.
#ifdef QT_NO_KEYWORDS
	#define signals Q_SIGNALS
	#define slots Q_SLOTS
#endif
#include <Qsci/qsciscintilla.h>
#include <Qsci/qscilexerpython.h>

namespace Scripting {

/**
 * \brief A window containing the script editor.
 *
 * \author Alexander Stukowski
 */
class ScriptEditor : public QMainWindow
{
	Q_OBJECT

public:

	/// \brief Default constructor.
	/// \param parent The parent window for the new window.
	ScriptEditor(QWidget* parent = NULL);

	/// This is the implementation of the "Save" action.
	/// Returns true, if the script has been saved.
	bool fileSave();

	/// This is the implementation of the "Save As" action.
	/// Returns true, if the script has been saved.
	bool fileSaveAs(QString filename = QString());

	/// If the script has been changed this will ask the user if he wants
	/// to save the changes.
	/// Returns false if the operation has been canceled by the user.
	bool askForSaveChanges();

	/// Loads the given script file or asks the user to select one.
	/// Returns true if the file has been successfully loaded.
	bool loadFile(QString filename = QString(), bool dontAskSaveOld = false);

	/// Returns the path where the current script is stored.
	const QString& filePath() const { return _filePath; }

	/// Sets the path where current script is stored.
	void setFilePath(const QString& path) { _filePath = path; }

	/// Executes the current script.
	bool executeScript();

protected Q_SLOTS:

	/// Handles the "Run" command event.
	void onRun();
	void onCloseEditor();
	void onOpen();
	void onSave();
	void onSaveAs();

	/// Writes some text to the output window.
	void writeOutput(const QString& str);

	/// Writes an error text to the output window.
	void writeError(const QString& str);

protected:

	/// Is called when the user closes the window.
	virtual void closeEvent(QCloseEvent* event);

	/// The main text editor component.
	QsciScintilla* _scriptInput;

	/// The text box that is used to the display the script's output.
	QTextEdit* _outputWindow;

	/// The file path where the current script is stored.
	QString _filePath;
};

};

#endif // __SCRIPT_EDITOR_H
