// -*- indent-tabs-mode: nil -*-

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <arc/StringConv.h>
#include <arc/UserConfig.h>
#include <arc/communication/ClientInterface.h>
#include <arc/delegation/DelegationInterface.h>
#include <arc/infosys/InformationInterface.h>
#include <arc/ws-addressing/WSA.h>

#include "JobStateARC1.h"
#include "JobStateBES.h"
#include "AREXClient.h"

#ifdef CPPUNITTEST
#include "../../libs/communication/test/SimulatorClasses.h"
#define DelegationProviderSOAP DelegationProviderSOAPTest
#endif

#define BES_FACTORY_ACTIONS_BASE_URL "http://schemas.ggf.org/bes/2006/08/bes-factory/BESFactoryPortType/"

namespace Arc {

  Logger AREXClient::logger(Logger::rootLogger, "A-REX-Client");

  static void set_bes_namespaces(NS& ns) {
    ns["bes-factory"] = "http://schemas.ggf.org/bes/2006/08/bes-factory";
    ns["wsa"] = "http://www.w3.org/2005/08/addressing";
    ns["jsdl"] = "http://schemas.ggf.org/jsdl/2005/11/jsdl";
    ns["jsdl-posix"] = "http://schemas.ggf.org/jsdl/2005/11/jsdl-posix";
    ns["jsdl-hpcpa"] = "http://schemas.ggf.org/jsdl/2006/07/jsdl-hpcpa";
  }

  static void set_arex_namespaces(NS& ns) {
    ns["a-rex"] = "http://www.nordugrid.org/schemas/a-rex";
    ns["glue"] = "http://schemas.ogf.org/glue/2008/05/spec_2.0_d41_r01";
    ns["glue2"] = "http://schemas.ogf.org/glue/2009/03/spec/2/0";
    ns["glue3"] = "http://schemas.ogf.org/glue/2009/03/spec_2.0_r1";
    ns["jsdl-arc"] = "http://www.nordugrid.org/ws/schemas/jsdl-arc";
    ns["rp"] = "http://docs.oasis-open.org/wsrf/rp-2";
  
    set_bes_namespaces(ns);
  }

  AREXClient::AREXClient(const URL& url,
                         const MCCConfig& cfg,
                         int timeout,
                         bool arex_extensions)
    : client(NULL),
      rurl(url),
      cfg(cfg),
      timeout(timeout),
      arex_enabled(arex_extensions) {

    logger.msg(DEBUG, "Creating an A-REX client");
    client = new ClientSOAP(cfg, url, timeout);
    if (!client) {
      logger.msg(VERBOSE, "Unable to create SOAP client used by AREXClient.");
    }
    if(arex_enabled) {
      set_arex_namespaces(arex_ns);
    } else {
      set_bes_namespaces(arex_ns);
    }
  }

  AREXClient::~AREXClient() {
    if (client)
      delete client;
  }

  bool AREXClient::delegation(XMLNode& op) {
    DelegationProviderSOAP* deleg;
    if (!cfg.credential.empty()) {
      deleg = new DelegationProviderSOAP(cfg.credential);
    }
    else {
      const std::string& cert = (!cfg.proxy.empty() ? cfg.proxy : cfg.cert);
      const std::string& key  = (!cfg.proxy.empty() ? cfg.proxy : cfg.key);

      if (key.empty() || cert.empty()) {
        logger.msg(VERBOSE, "Failed locating credentials.");
        error_description = "Failed locating credentials for delegation to "+rurl.str();
        return false;
      }
      deleg = new DelegationProviderSOAP(cert, key);
    }

    MCC_Status r = client->Load();
    if(!r) {
      logger.msg(VERBOSE, "Failed initiate client connection.");
      error_description = "Failed initiating communication to "+rurl.str()+" - "+(std::string)r;
      delete deleg;
      return false;
    }

    MCC* entry = client->GetEntry();
    if(!entry) {
      logger.msg(VERBOSE, "Client connection has no entry point.");
      error_description = "Internal error: failed to properly initiate communication object for "+rurl.str();
      delete deleg;
      return false;
    }

    /* TODO: Enable password typing in case of cert and key. Currently when
     * using cert and key, one have to type password multiple times, which is
     * impracticable and should coordinated across execution.
     *DelegationProviderSOAP deleg(cert, key, (!cfg.proxy.empty() ? NULL : &std::cin));
     */
    logger.msg(VERBOSE, "Initiating delegation procedure");
    if (!deleg->DelegateCredentialsInit(*entry,&(client->GetContext()))) {
      logger.msg(VERBOSE, "Failed to initiate delegation credentials");
      error_description = "Internal error: failed to initiate delagtion at "+rurl.str();
      // TODO: propagate error from DelegationProviderSOAP
      delete deleg;
      return false;
    }
    deleg->DelegatedToken(op);
    delete deleg;
    return true;
  }

  bool AREXClient::reconnect(void) {
    delete client; client = NULL;
    logger.msg(DEBUG, "Re-creating an A-REX client");
    client = new ClientSOAP(cfg, rurl, timeout);
    //if (!client) {
    //  logger.msg(VERBOSE, "Unable to create SOAP client used by AREXClient.");
    //  error_description = "Internal error: unable to create object to handle connection to "+rurl.str();
    //  return false;
    //}
    if(arex_enabled) {
      set_arex_namespaces(arex_ns);
    } else {
      set_bes_namespaces(arex_ns);
    }
    return true;
  }

  bool AREXClient::process(PayloadSOAP& req, bool delegate, XMLNode& response, bool retry) {
    error_description = "";
    if (!client) {
      logger.msg(VERBOSE, "AREXClient was not created properly."); // Should not happen. Happens if client = null (out of memory?)
      error_description = "Internal error: object is not in proper state.";
      return false;
    }

    logger.msg(VERBOSE, "Processing a %s request", req.Child(0).FullName());

    if (delegate) {
      XMLNode op = req.Child(0);
      if(!delegation(op)) {
        delete client; client = NULL;
        // TODO: better way to check of retriable.
        if(!retry) return false;
        if(!reconnect()) return false;
        if(!delegation(op)) {
          delete client; client = NULL;
          return false;
        }
      }
    }

    WSAHeader header(req);
    header.To(rurl.str());
    PayloadSOAP* resp = NULL;
    MCC_Status r;
    if (!(r = client->process(header.Action(), &req, &resp))) {
      error_description = (std::string)r;
      logger.msg(VERBOSE, "%s request failed", action);
      delete client; client = NULL;
      if(!retry) return false;
      if(!reconnect()) return false;
      return process(req,false,response,false);
    }

    if (resp == NULL) {
      logger.msg(VERBOSE, "No response from %s", rurl.str());
      error_description = "No or malformed response received from "+rurl.str();
      delete client; client = NULL;
      if(!retry) return false;
      if(!reconnect()) return false;
      return process(req,false,response,false);
    }

    if (resp->IsFault()) {
      logger.msg(VERBOSE, "%s request to %s failed with response: %s", action, rurl.str(), resp->Fault()->Reason());
      error_description = "Fault received from "+rurl.str()+": "+resp->Fault()->Reason();
      if(resp->Fault()->Code() != SOAPFault::Receiver) retry = false;
      std::string s;
      resp->GetXML(s);
      logger.msg(DEBUG, "XML response: %s", s);
      delete resp;
      delete client; client = NULL;
      if(!retry) return false;
      if(!reconnect()) return false;
      return process(req,false,response,false);
    }

    if (!(*resp)[action + "Response"]) {
      logger.msg(VERBOSE, "%s request to %s failed. No expected response.", action, rurl.str());
      error_description = "No expected response received from "+rurl.str();
      delete resp;
      return false;
    }

    (*resp)[action + "Response"].New(response);
    delete resp;
    return true;
  }

  bool AREXClient::submit(const std::string& jobdesc, std::string& jobid,
                          bool delegate) {
    action = "CreateActivity";
    logger.msg(VERBOSE, "Creating and sending submit request to %s", rurl.str());

    // Create job request
    /*
       bes-factory:CreateActivity
         bes-factory:ActivityDocument
           jsdl:JobDefinition
     */

    PayloadSOAP req(arex_ns);
    XMLNode op = req.NewChild("bes-factory:" + action);
    XMLNode act_doc = op.NewChild("bes-factory:ActivityDocument");
    WSAHeader(req).Action(BES_FACTORY_ACTIONS_BASE_URL + action);
    act_doc.NewChild(XMLNode(jobdesc));
    act_doc.Child(0).Namespaces(arex_ns); // Unify namespaces

    logger.msg(DEBUG, "Job description to be sent: %s", jobdesc);

    XMLNode response;
    if (!process(req, delegate, response))
      return false;

    XMLNode xmlJobId;
    response["ActivityIdentifier"].New(xmlJobId);
    xmlJobId.GetDoc(jobid);
    return true;
  }

  bool AREXClient::stat(const std::string& jobid, Job& job) {
    std::string faultstring;
    logger.msg(VERBOSE, "Creating and sending job information query request to %s", rurl.str());

    PayloadSOAP req(arex_ns);
    if(arex_enabled) {
      // TODO: use wsrf classes
      // AREX service
      //action = "QueryResourceProperties";
      //std::string xpathquery = "//glue:Services/glue:ComputingService/glue:ComputingEndpoint/glue:ComputingActivities/glue:ComputingActivity/glue:ID[contains(.,'" + (std::string)(XMLNode(jobid)["ReferenceParameters"]["JobID"]) + "')]/..";
      //req = *InformationRequest(XMLNode("<XPathQuery>" + xpathquery + "</XPathQuery>")).SOAP();
      // GetActivityStatuses
      //  ActivityIdentifier
      //  ActivityStatusVerbosity
      action = "GetActivityStatuses";
      XMLNode op = req.NewChild("bes-factory:" + action);
      op.NewChild(XMLNode(jobid));
      op.NewChild("a-rex:ActivityStatusVerbosity") = "Full";
      op.Namespaces(arex_ns); // Unify namespaces
      WSAHeader(req).Action(BES_FACTORY_ACTIONS_BASE_URL + action);
    } else {
      // Simple BES service
      // GetActivityStatuses
      //  ActivityIdentifier
      action = "GetActivityStatuses";
      XMLNode jobref =
        req.NewChild("bes-factory:" + action).
        NewChild(XMLNode(jobid));
      jobref.Child(0).Namespaces(arex_ns); // Unify namespaces
      WSAHeader(req).Action(BES_FACTORY_ACTIONS_BASE_URL + action);
    }

    XMLNode response;
    if (!process(req, false, response))
      return false;

    if (!arex_enabled) {
      XMLNode activity = response["Response"]["ActivityStatus"];
      if(activity) {
        NS ns("a-rex","http://www.nordugrid.org/schemas/a-rex");
        activity.Namespaces(ns);
        std::string state = activity.Attribute("state");
        if(!state.empty()) {
          job.State = JobStateBES(state);
        }
      }
      if (!job.State) {
        logger.msg(VERBOSE, "Unable to retrieve status of job (%s)", job.JobID);
        return false;
      }
      return true;
    }

    // A-REX publishes GLUE2 information which is parsed by the Job& operator=(XMLNode).
    // See the ARC1ClusterInfo.pm script for details on what is actually published.
    //job = response["ComputingActivity"];
    XMLNode activity = response["Response"]["ActivityStatus"];
    if(activity) {
      XMLNode gactivity = activity["ComputingActivity"];
      if(gactivity) {
        job.SetFromXML(gactivity);

        // Fetch the proper restart state.
        if (gactivity["RestartState"]) {
          for (XMLNode n = response["ComputingActivity"]["RestartState"]; n; ++n) {
            std::list<std::string> gluestate;
            tokenize((std::string)n, gluestate, ":");

            if (!gluestate.empty() && gluestate.front() == "nordugrid") {
              job.RestartState = JobStateARC1(((std::string)n).substr(10));
              break;
            }
          }
        }
      }

      // Fetch the proper state.
      if (activity["glue:State"]) {
        job.State = JobStateARC1((std::string)activity["glue:State"]);
      }
      else if (activity["a-rex:State"]) {
        if (activity["LRMSState"]) {
          job.State = JobStateARC1("INLRMS:" + (std::string)activity["LRMSState"]);
        }
        else {
          job.State = JobStateARC1((std::string)activity["a-rex:State"]);
        }
      }
    }
    if (!job.State) {
      logger.msg(VERBOSE, "Unable to retrieve status of job (%s)", job.JobID);
    }

    return (bool)job.State;
  }

  bool AREXClient::sstat(XMLNode& response) {

    if(arex_enabled) {
      action = "QueryResourceProperties";
      logger.msg(VERBOSE, "Creating and sending service information query request to %s", rurl.str());

      PayloadSOAP req(*InformationRequest(XMLNode("<XPathQuery>//glue:ComputingService | //glue2:ComputingService | //glue3:ComputingService</XPathQuery>")).SOAP());
      req.Child(0).Namespaces(arex_ns);
      if (!process(req, false, response)) return false;
    } else {
      // GetFactoryAttributesDocument
      PayloadSOAP req(arex_ns);
      action = "GetFactoryAttributesDocument";
      req.NewChild("bes-factory:" + action);
      WSAHeader(req).Action(BES_FACTORY_ACTIONS_BASE_URL + action);
      if (!process(req, false, response)) return false;
    }
    return true;
  }

  bool AREXClient::listServicesFromISIS(std::list< std::pair<URL, ServiceType> >& services) {
    if(!arex_enabled) return false;

    action = "Query";
    logger.msg(VERBOSE, "Creating and sending ISIS information query request to %s", rurl.str());

    PayloadSOAP req(NS("isis", "http://www.nordugrid.org/schemas/isis/2007/06"));
    req.NewChild("isis:" + action).NewChild("isis:QueryString") = "/RegEntry/SrcAdv[Type=\"org.nordugrid.execution.arex\"]";
    WSAHeader(req).Action("http://www.nordugrid.org/schemas/isis/2007/06/Query/QueryRequest");

    XMLNode response;
    if (!process(req, false, response))
      return false;

    if (XMLNode n = response["RegEntry"])
      for (; n; ++n) {
        if ((std::string)n["SrcAdv"]["Type"] == "org.nordugrid.execution.arex") {
          //This check is right now superfluos but in the future a wider query might be used
          services.push_back(std::pair<URL, ServiceType>(URL((std::string)n["SrcAdv"]["EPR"]["Address"]), COMPUTING));
        }
        else
          logger.msg(DEBUG, "Service %s of type %s ignored", (std::string)n["MetaSrcAdv"]["ServiceID"], (std::string)n["SrcAdv"]["Type"]);
      }
    else
      logger.msg(VERBOSE, "No execution services registered in the index service");
    return true;
  }

  bool AREXClient::kill(const std::string& jobid) {
    action = "TerminateActivities";
    logger.msg(VERBOSE, "Creating and sending terminate request to %s", rurl.str());

    PayloadSOAP req(arex_ns);
    XMLNode jobref = req.NewChild("bes-factory:" + action).NewChild(XMLNode(jobid));
    WSAHeader(req).Action(BES_FACTORY_ACTIONS_BASE_URL + action);

    XMLNode response;
    if (!process(req, false, response))
      return false;

    if ((std::string)response["Response"]["Terminated"] != "true") {
      logger.msg(ERROR, "Job termination failed");
      return false;
    }

    return true;
  }

  bool AREXClient::clean(const std::string& jobid) {
    if(!arex_enabled) return false;

    action = "ChangeActivityStatus";
    logger.msg(VERBOSE, "Creating and sending clean request to %s", rurl.str());

    PayloadSOAP req(arex_ns);
    XMLNode op = req.NewChild("a-rex:" + action);
    op.NewChild(XMLNode(jobid));
    XMLNode jobstate = op.NewChild("a-rex:NewStatus");
    jobstate.NewAttribute("bes-factory:state") = "Finished";
    jobstate.NewChild("a-rex:state") = "Deleted";

    // Send clean request
    XMLNode response;
    if (!process(req, false, response))
      return false;

/*
 * It is not clear how (or if) the response should be interpreted.
 * Currently response contains status of job before invoking requst. It is
 * unclear if this is the desired behaviour.
 * See trunk/src/services/a-rex/change_activity_status.cpp
    ????if ((std::string)response["NewStatus"]["state"] != "Deleted") {????
      logger.msg(VERBOSE, "Job cleaning failed: Wrong response???");
      return false;
    }
*/

    return true;
  }

  bool AREXClient::getdesc(const std::string& jobid, std::string& jobdesc) {
    action = "GetActivityDocuments";
    logger.msg(VERBOSE, "Creating and sending job description retrieval request to %s", rurl.str());

    PayloadSOAP req(arex_ns);
    req.NewChild("bes-factory:" + action).NewChild(XMLNode(jobid));
    WSAHeader(req).Action(BES_FACTORY_ACTIONS_BASE_URL + action);

    XMLNode response;
    if (!process(req, false, response))
      return false;

    XMLNode xmlJobDesc;
    response["Response"]["JobDefinition"].New(xmlJobDesc);
    xmlJobDesc.GetDoc(jobdesc);
    return true;
  }

  bool AREXClient::migrate(const std::string& jobid, const std::string& jobdesc, bool forcemigration, std::string& newjobid, bool delegate) {
    if(!arex_enabled) return false;

    action = "MigrateActivity";
    logger.msg(VERBOSE, "Creating and sending job migrate request to %s", rurl.str());

    // Create migrate request
    /*
       bes-factory:MigrateActivity
        bes-factory:ActivityIdentifier
        bes-factory:ActivityDocument
          jsdl:JobDefinition
     */

    PayloadSOAP req(arex_ns);
    XMLNode op = req.NewChild("a-rex:" + action);
    XMLNode act_doc = op.NewChild("bes-factory:ActivityDocument");
    op.NewChild(XMLNode(jobid));
    op.NewChild("a-rex:ForceMigration") = (forcemigration ? "true" : "false");
    act_doc.NewChild(XMLNode(jobdesc));
    act_doc.Child(0).Namespaces(arex_ns); // Unify namespaces

    logger.msg(DEBUG, "Job description to be sent: %s", jobdesc);

    XMLNode response;
    if (!process(req, delegate, response))
      return false;

    XMLNode xmlNewJobId;
    response["ActivityIdentifier"].New(xmlNewJobId);
    xmlNewJobId.GetDoc(newjobid);
    return true;
  }

  bool AREXClient::resume(const std::string& jobid) {
    if(!arex_enabled) return false;

    action = "ChangeActivityStatus";
    logger.msg(VERBOSE, "Creating and sending job resume request to %s", rurl.str());

    PayloadSOAP req(arex_ns);
    XMLNode op = req.NewChild("a-rex:" + action);
    op.NewChild(XMLNode(jobid));
    XMLNode jobstate = op.NewChild("a-rex:NewStatus");
    jobstate.NewAttribute("bes-factory:state") = "Running";
    // Not supporting resume into user-defined state
    jobstate.NewChild("a-rex:state") = "";

    XMLNode response;
    if (!process(req, true, response))
      return false;

/*
 * It is not clear how (or if) the response should be interpreted.
 * Currently response contains status of job before invoking requst. It is
 * unclear if this is the desired behaviour.
 * See trunk/src/services/a-rex/change_activity_status.cpp
    ????if ((std::string)response["NewStatus"]["state"] != "Running") {????
      logger.msg(VERBOSE, "Job resuming failed: Wrong response???");
      return false;
    }
*/

    return true;
  }

  void AREXClient::createActivityIdentifier(const URL& jobid, std::string& activityIdentifier) {
    PathIterator pi(jobid.Path(), true);
    URL url(jobid);
    url.ChangePath(*pi);
    NS ns;
    ns["a-rex"] = "http://www.nordugrid.org/schemas/a-rex";
    ns["bes-factory"] = "http://schemas.ggf.org/bes/2006/08/bes-factory";
    ns["wsa"] = "http://www.w3.org/2005/08/addressing";
    ns["jsdl"] = "http://schemas.ggf.org/jsdl/2005/11/jsdl";
    ns["jsdl-posix"] = "http://schemas.ggf.org/jsdl/2005/11/jsdl-posix";
    ns["jsdl-arc"] = "http://www.nordugrid.org/ws/schemas/jsdl-arc";
    ns["jsdl-hpcpa"] = "http://schemas.ggf.org/jsdl/2006/07/jsdl-hpcpa";
    XMLNode id(ns, "ActivityIdentifier");
    id.NewChild("wsa:Address") = url.str();
    id.NewChild("wsa:ReferenceParameters").NewChild("a-rex:JobID") = pi.Rest();
    id.GetXML(activityIdentifier);
  }

// -----------------------------------------------------------------------------

  // TODO: does it need locking?

  AREXClients::AREXClients(const UserConfig& usercfg):usercfg_(&usercfg) {
  }

  AREXClients::~AREXClients(void) {
    std::multimap<URL, AREXClient*>::iterator it;
    for (it = clients_.begin(); it != clients_.end(); ++it) {
      delete it->second;
    }
  }

  AREXClient* AREXClients::acquire(const URL& url, bool arex_features) {
    std::multimap<URL, AREXClient*>::iterator it = clients_.find(url);
    if ( it != clients_.end() ) {
      // If AREXClient is already existing for the
      // given URL then return with that
      AREXClient* client = it->second;
      client->arexFeatures(arex_features);
      clients_.erase(it);
      return client;
    }
    // Else create a new one and return with that
    MCCConfig cfg;
    if(usercfg_) usercfg_->ApplyToConfig(cfg);
    AREXClient* client = new AREXClient(url, cfg, usercfg_?usercfg_->Timeout():0, arex_features);
    return client;
  }

  void AREXClients::release(AREXClient* client) {
    if(!client) return;
    if(!*client) { delete client; return; }
    // TODO: maybe strip path from URL?
    clients_.insert(std::pair<URL, AREXClient*>(client->url(),client));
  }

  void AREXClients::SetUserConfig(const UserConfig& uc) {
    // Changing user configuration may change identity.
    // Hence all open connections become invalid.
    usercfg_ = &uc;
    while(true) {
      std::multimap<URL, AREXClient*>::iterator it = clients_.begin();
      if(it == clients_.end()) break;
      delete it->second;
      clients_.erase(it);
    }
  }

}

