# player.py - Basic music player based on musiclibrarian.
#
# Copyright 2004 Daniel Burrows

from musiclibrarian import libraryview

import ao
import gtk
import Queue
import threading

# Threads seem like a good idea here.  The model is to use a single
# thread that knows how to read data from an audio file and send it to
# the audio device.  It also provides some read-only values which the
# GUI can use (eg, in an idle callback or a timeout) to view its
# current status.

# The communication protocol is simple: the player thread passes 

def autodetect_device(*args, **kwargs):
    """Tries to choose a 'good' ao device.  Never chooses the null driver."""
    devices=['alsa', 'oss', 'irix', 'sun', 'esd', 'arts']

    for device in devices:
        try:
            return apply(ao.AudioDevice, (device,)+args, kwargs)
        except ao.aoError:
            pass

    return None

class PlayerPlayThread(threading.Thread):
    """A thread which continuously pulls tuples of
    (buf, amt, samplerate, bits) off a queue and plays them via ao."""
    def __init__(self, q, **kwargs):
        threading.Thread.__init__(self)
        self.setDaemon(True)

        self.q=q

        self.samplerate=None
        self.bits=None
        self.channels=None
        self.cancelled=threading.Event()

    def set_params(self, samplerate, bits, channels):
        if samplerate <> self.samplerate or bits <> self.bits or channels <> self.channels:
            self.samplerate=samplerate
            self.bits=bits
            self.channels=channels

            # Try to guess the device
            self.device=autodetect_device(bits=bits, rate=samplerate, channels=channels)

            if self.device == None:
                print 'Unable to detect which audio device should be used, giving up.'
    def run(self):
        while not self.cancelled.isSet():
            msg=self.q.get()

            if msg <> None:
                buf, amt, samplerate, bits, channels=msg
                self.set_params(samplerate, bits, channels)
                self.device.play(buf, amt)

    def cancel(self):
        self.cancelled.set()
        self.q.put(None)

class PlayerDecodeThread(threading.Thread):
    """A thread whose sole purpose in life is to decode a list of files
    and chain blocks from them onto a queue."""
    def __init__(self, files, q):
        threading.Thread.__init__(self)
        self.files=files
        self.q=q
        self.setDaemon(True)
        self.cancelled=threading.Event()

    def run(self):
        for curr in self.files:
            if not hasattr(curr, 'get_file'):
                continue

            print 'Playing %s'%curr.fn
            # Implicitly uses the fact that it is always safe to call
            # get_file() from any thread because the filename is invariant.
            #
            # In the future this thread should be given the file object
            # directly, this is just test code.
            f=curr.get_file()

            if f == None:
                continue

            buf,amt,rate,bits,channels=f.read(4096)

            while buf <> None and amt > 0:
                self.q.put((buf, amt, rate, bits, channels))
                buf,amt,rate,bits,channels=f.read(4096)

                if self.cancelled.isSet():
                    return

    def cancel(self):
        self.cancelled.set()

# This class is attached to a view.
class PlayerControl:
    def __init__(self):
        self.q=None
        self.play_thread=None
        self.decode_thread=None

    def play(self, files):
        if self.play_thread <> None:
            self.stop()

        lst=[]
        for obj in files:
            obj.add_underlying_files(lst)

        self.q=Queue.Queue(500)

        self.decode_thread=PlayerDecodeThread(lst, self.q)
        self.play_thread=PlayerPlayThread(self.q)

        self.decode_thread.start()
        self.play_thread.start()

    def stop(self):
        if self.decode_thread == None:
            return

        assert(self.q <> None)
        assert(self.play_thread <> None)

        # Note that it's important to stop the decoding thread first
        # to avoid a deadlock when it tries to write to a full queue.
        self.decode_thread.cancel()
        self.decode_thread.join()

        self.play_thread.cancel()
        self.play_thread.join()

        self.q=None
        self.play_thread=None
        self.decode_thread=None

def setup_view(view):
    control=PlayerControl()

    def play(*args):
        control.play(view.get_selected_objects())

        stopitem.set_sensitive(True)
        stopbutton.set_sensitive(True)

    def stop(*args):
        control.stop()

        stopitem.set_sensitive(False)
        stopbutton.set_sensitive(False)

    view.add_context_menu_item('Play', 'Start playing the selected files',
                               gtk.STOCK_GO_FORWARD,
                               play)

    playmenuitem=gtk.MenuItem('Play')
    menu=gtk.Menu()
    playmenuitem.set_submenu(menu)

    playitem=gtk.ImageMenuItem('Play')
    im=gtk.Image()
    im.set_from_stock(gtk.STOCK_GO_FORWARD, gtk.ICON_SIZE_MENU)
    playitem.set_image(im)
    playitem.connect('activate', play)

    stopitem=gtk.ImageMenuItem('Stop')
    im=gtk.Image()
    im.set_from_stock(gtk.STOCK_STOP, gtk.ICON_SIZE_MENU)
    stopitem.set_image(im)
    stopitem.connect('activate', stop)

    menu.append(playitem)
    menu.append(stopitem)

    playmenuitem.show_all()
    view.add_menubar_item(playmenuitem)

    im=gtk.Image()
    im.set_from_stock(gtk.STOCK_GO_FORWARD, gtk.ICON_SIZE_LARGE_TOOLBAR)
    playbutton=view.add_toolbar_item('Play', 'Play the selected files', None, im, play)

    im=gtk.Image()
    im.set_from_stock(gtk.STOCK_STOP, gtk.ICON_SIZE_LARGE_TOOLBAR)
    stopbutton=view.add_toolbar_item('Stop', 'Stop playing music', None, im, stop)

    def selection_changed(view):
        active=len(filter(lambda x:hasattr(x, 'get_file'), view.get_selected_files()))

        playitem.set_sensitive(active)
        playbutton.set_sensitive(active)

    view.add_selection_listener(selection_changed)

    selection_changed(view)

    stopitem.set_sensitive(False)
    stopbutton.set_sensitive(False)

libraryview.add_new_view_hook(setup_view)
