/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <globus_config.h>
#include <globus_io.h>
#include <globus_error_string.h>
#include <globus_usage.h>

#include "globus_rls_client.h"
#include "auth.h"
#include "conf.h"
#include "misc.h"
#include "globus_rls_rpc.h"
#include "db.h"
#include "event.h"
#include "lock.h"
#include "bloom.h"
#include "lrc.h"
#include "rli.h"
#include "update.h"
#include "usage.h"
#include "version.h"

#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <stdarg.h>
#include <sys/resource.h>
#include <signal.h>

#define PS(s)	((s) ? (s) : "NULL")

/*
 * Per connection data.
 */
typedef struct {
  globus_io_handle_t	handle;
  BUFFER		ibuf;		/* Input buffer			*/
  BUFFER		obuf;		/* Output buffer		*/
  time_t		lastcall;
  char			*dn;
  int			perms;
  int			numrecs;	/* Num recs written to client	*/
  char			errmsg[MAXERRMSG];
} CONNECTION;

/*
 * Following CONNECTION pointers are queued by parent thread to cause
 * some event to be executed by a worker thread.  Currently the only event
 * is C_EXIT, which causes the worker thread to exit.
 */
#define C_EXIT		((CONNECTION *) 0)

typedef struct {		/* Database handles			*/
  void			*lrc;
  void			*rli;
} DBH;

typedef struct {
  char			*method;
  void			(*fun)(CONNECTION *c, DBH *dbh, char **arglist);
  int			nargs;
  int			perms;
  int			flags;
#define MF_NEEDLRC		0x1
#define MF_NEEDRLI		0x2
#define MF_NTA			0x4	/* Flush null term args on err	*/
#define MF_NEEDLRCBF 	0x8
} METHOD;

static void		init();
static void		lrc_add(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_add_bulk(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_attr_add(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_attr_add_bulk(CONNECTION *c, DBH *dbh,
					  char **arglist);
static void		lrc_attr_create(CONNECTION *c,DBH *dbh,char **arglist);
static void		lrc_attr_delete(CONNECTION *c,DBH *dbh,char **arglist);
static void		lrc_attr_get(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_attr_modify(CONNECTION *c,DBH *dbh,char **arglist);
static void		lrc_attr_remove(CONNECTION *c,DBH *dbh,char **arglist);
static void		lrc_attr_remove_bulk(CONNECTION *c, DBH *dbh,
					     char **arglist);
static void		lrc_attr_search(CONNECTION *c,DBH *dbh,char **arglist);
static void		lrc_attr_value_get(CONNECTION *c, DBH *dbh,
					   char **arglist);
static void		lrc_attr_value_get_bulk(CONNECTION *c, DBH *dbh,
						char **arglist);
static void		lrc_clear(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_create(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_create_bulk(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_delete(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_delete_bulk(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_exists(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_exists_bulk(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_get_lfn(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_get_lfn_bulk(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_get_lfn_wc(CONNECTION *c, DBH *dbh,char **arglist);
static void		lrc_get_pfn(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_get_pfn_bulk(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_get_pfn_wc(CONNECTION *c, DBH *dbh,char **arglist);
static void		lrc_mapping_exists(CONNECTION *c, DBH *dbh,
					   char **arglist);
static void		lrc_renamelfn(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_renamelfn_bulk(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_renamepfn(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_renamepfn_bulk(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_rli_add(CONNECTION *c, DBH *dbh, char **arglist);
static void		lrc_rli_delete(CONNECTION *c,DBH *dbh,char **arglist);
static void		lrc_rli_get_part(CONNECTION *c, DBH *dbh,
					 char **arglist);
static void		lrc_rli_info(CONNECTION *c,DBH *dbh,char **arglist);
static void		lrc_rli_list(CONNECTION *c,DBH *dbh,char **arglist);
static void		rli_exists(CONNECTION *c, DBH *dbh, char **arglist);
static void		rli_exists_bulk(CONNECTION *c, DBH *dbh, char **arglist);
static void		rli_get_lrc(CONNECTION *c, DBH *dbh, char **arglist);
static void		rli_get_lrc_bulk(CONNECTION *c, DBH *dbh, char **arglist);
static void		rli_get_lrc_wc(CONNECTION *c, DBH *dbh,char **arglist);
static void		rli_mapping_exists(CONNECTION *c, DBH *dbh,
					   char **arglist);
static void		rli_rli_add(CONNECTION *c, DBH *dbh, char **arglist);
static void		rli_rli_delete(CONNECTION *c,DBH *dbh,char **arglist);
static void		rli_rli_get_part(CONNECTION *c, DBH *dbh,
					 char **arglist);
static void		rli_rli_list(CONNECTION *c,DBH *dbh,char **arglist);
static void		rli_recv1(CONNECTION *c, DBH *dbh, char **arglist);
static void		rli_recv2(CONNECTION *c, DBH *dbh, char **arglist);
static void		rli_recv(CONNECTION *c, DBH *dbh, char *lrcurl,
				 char *senderurl);
static void		rli_recvbf1(CONNECTION *c, DBH *dbh, char **arglist);
static void		rli_recvbf2(CONNECTION *c, DBH *dbh, char **arglist);
static void		rli_recvbf(CONNECTION *c, char *lrcurl,
				   char *senderurl, char *abfsize,
				   char *anumhash, char *aupdateinterval);
static void		rli_sender_list(CONNECTION *c,DBH *dbh,char **arglist);
static void		rls_admin(CONNECTION *c, DBH *dbh, char **arglist);
static void		rls_close(CONNECTION *c, DBH *dbh, char **arglist);
static void		rls_get_configuration(CONNECTION *c, DBH *dbh,
					      char **arglist);
static void		rls_set_configuration(CONNECTION *c, DBH *dbh,
					      char **arglist);
static void		rls_stats(CONNECTION *c, DBH *dbh, char **arglist);
static void		doquit(globus_io_handle_t *handlep);
static void		*checkidle(void *a);
static void		doimmediate(RLIOP op, RLI *rlilist, LOCK *rlilistlock,
				    char *lfn, char *lrc);
static int		str2cb(void *cv, char *s1, char *s2);
static int		str3cb(void *cv, char *s1, char *s2, char *s3);
static int		str2bulkcb(void *cv, char *s1, char *s2);
static int		str4bulkcb(void *cv, char *s1, char *s2, char *s3, char *s4);
static int		rlipartcb(void *cv, char *s1, char *s2, char *s3);
static void		endstrcb(CONNECTION *c, int pending);
static int		listresult(void *cv, char *s1, char *s2, char *s3,
				   char *s4, int irc);
static int		getreslimit(char *areslimit);

METHOD			methods[] = {
  { "admin", rls_admin, 1, 0, 0 },
  { "close", rls_close, 0, 0, 0 },
  { "get_configuration", rls_get_configuration, 1, P_ADMIN, 0 },
  { "lrc_add", lrc_add, 2, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_add_bulk", lrc_add_bulk, 0, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_attr_add", lrc_attr_add, 5, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_attr_add_bulk", lrc_attr_add_bulk, 0, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_attr_create", lrc_attr_create, 3, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_attr_delete", lrc_attr_delete, 3, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_attr_get", lrc_attr_get, 2, P_LRCREAD, MF_NEEDLRC },
  { "lrc_attr_modify", lrc_attr_modify, 5, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_attr_remove", lrc_attr_remove, 3, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_attr_remove_bulk", lrc_attr_remove_bulk, 0, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_attr_search", lrc_attr_search, 7, P_LRCREAD, MF_NEEDLRC },
  { "lrc_attr_value_get", lrc_attr_value_get, 3, P_LRCREAD, MF_NEEDLRC },
  { "lrc_attr_value_get_bulk", lrc_attr_value_get_bulk, 2, P_LRCREAD, MF_NEEDLRC },
  { "lrc_clear", lrc_clear, 0, P_ADMIN|P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_create", lrc_create, 2, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_create_bulk", lrc_create_bulk, 0, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_delete", lrc_delete, 2, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_delete_bulk", lrc_delete_bulk, 0, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_exists", lrc_exists, 2, P_LRCREAD, MF_NEEDLRC },
  { "lrc_exists_bulk", lrc_exists_bulk, 0, P_LRCREAD, MF_NEEDLRC },
  { "lrc_renamelfn", lrc_renamelfn, 2, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_renamelfn_bulk", lrc_renamelfn_bulk, 0, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_renamepfn", lrc_renamepfn, 2, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_renamepfn_bulk", lrc_renamepfn_bulk, 0, P_LRCUPDATE, MF_NEEDLRC | MF_NEEDLRCBF },
  { "lrc_get_lfn", lrc_get_lfn, 3, P_LRCREAD, MF_NEEDLRC },
  { "lrc_get_lfn_bulk", lrc_get_lfn_bulk, 0, P_LRCREAD, MF_NEEDLRC },
  { "lrc_get_lfn_wc", lrc_get_lfn_wc, 3, P_LRCREAD, MF_NEEDLRC },
  { "lrc_get_pfn", lrc_get_pfn, 3, P_LRCREAD, MF_NEEDLRC },
  { "lrc_get_pfn_bulk", lrc_get_pfn_bulk, 0, P_LRCREAD, MF_NEEDLRC },
  { "lrc_get_pfn_wc", lrc_get_pfn_wc, 3, P_LRCREAD, MF_NEEDLRC },
  { "lrc_mapping_exists", lrc_mapping_exists, 2, P_LRCREAD, MF_NEEDLRC },
  { "lrc_rli_add", lrc_rli_add, 3, P_ADMIN, MF_NEEDLRC },
  { "lrc_rli_delete", lrc_rli_delete, 2, P_ADMIN, MF_NEEDLRC },
  { "lrc_rli_get_part", lrc_rli_get_part, 2, P_LRCREAD, MF_NEEDLRC },
  { "lrc_rli_info", lrc_rli_info, 1, P_LRCREAD, MF_NEEDLRC },
  { "lrc_rli_list", lrc_rli_list, 0, P_LRCREAD, MF_NEEDLRC },
  { "rli_exists", rli_exists, 2, P_RLIREAD, MF_NEEDLRC },
  { "rli_exists_bulk", rli_exists_bulk, 0, P_RLIREAD, MF_NEEDRLI },
  { "rli_get_lrc", rli_get_lrc, 3, P_RLIREAD, MF_NEEDRLI },
  { "rli_get_lrc_bulk", rli_get_lrc_bulk, 0, P_RLIREAD, MF_NEEDRLI },
  { "rli_get_lrc_wc", rli_get_lrc_wc, 3, P_RLIREAD, MF_NEEDRLI },
  /* Deprecated, should use rli_sender_list */
  { "rli_lrc_list", rli_sender_list, 0, P_RLIREAD, MF_NEEDRLI },
  { "rli_mapping_exists", rli_mapping_exists, 2, P_RLIREAD, MF_NEEDRLI },
  { "rli_rli_add", rli_rli_add, 2, P_ADMIN, MF_NEEDRLI },
  { "rli_rli_delete", rli_rli_delete, 2, P_ADMIN, MF_NEEDRLI },
  { "rli_rli_get_part", rli_rli_get_part, 2, P_RLIREAD, MF_NEEDRLI },
  { "rli_rli_list", rli_rli_list, 0, P_RLIREAD, MF_NEEDRLI },
  { "rli_sender_list", rli_sender_list, 0, P_RLIREAD, MF_NEEDRLI },
  { "rli_update", rli_recv1, 1, P_RLIUPDATE, MF_NEEDRLI|MF_NTA },
  { "rli_update2", rli_recv2, 2, P_RLIUPDATE, MF_NEEDRLI|MF_NTA },
  { "rli_updatebf", rli_recvbf1, 4, P_RLIUPDATE, MF_NEEDRLI|MF_NTA },
  { "rli_updatebf2", rli_recvbf2, 5, P_RLIUPDATE, MF_NEEDRLI|MF_NTA },
  { "set_configuration", rls_set_configuration, 2, P_ADMIN, 0 },
  { "stats", rls_stats, 0, P_STATS, 0 },
  { NULL, NULL, 0 }
};

#define MAXARGS		10	/* Max args for a method	*/

static globus_result_t	startlistener(u_short port,
				      globus_io_handle_t *handlep);
static void		*procreq(void *a);
static int		getmethodargs(CONNECTION *c, int *mi, char **arglist);
static void		dolisten(globus_io_handle_t *handlep);
static void		doaccept(globus_io_handle_t *handlep,
				 CONNECTION *c);
static void		listencb(void *a, globus_io_handle_t *handlep,
				 globus_result_t r);
static void		acceptcb(void *a, globus_io_handle_t *handlep,
				 globus_result_t r);
globus_bool_t		authcb(void *a, globus_io_handle_t *handlep,
			       globus_result_t r, char *identity,
			       gss_ctx_id_t ch);
static void		readcb(void *c, globus_io_handle_t *handlep,
			       globus_result_t r, globus_byte_t *buf,
			       globus_size_t len);
static void		cancelcb(void *cv, globus_io_handle_t *handlep,
			       globus_result_t r);
static void		closecb(void *cv, globus_io_handle_t *handlep,
			       globus_result_t r);
static void		startcb(void *cv, globus_io_handle_t *handlep,
			       globus_result_t r);
static void		queueconn(CONNECTION *c);
static CONNECTION	*newconn();
static void		freeconn(CONNECTION *c, int locked);
static char		*canonurl(char *s, char *buf, int len);
static int		startthread();
static void		endthread(DBH *dbh);
static int		rrpc_rliinfo(CONNECTION *c, RLI *rli,
				     int first, int last, int termlist);
static void		rrpc_error(CONNECTION *c, int rc, char *fmt, ...);
static int		rrpc_success(CONNECTION *c);
static void		rrpc_flushio(CONNECTION *c);
static int		rrpc_readstr(CONNECTION *c, char *rbuf, int rbuflen);
static void		hexitsig(int s);
static void		hhupsig(int s);
static void		cleanexit(int s);
static void		usage(char *prog);
static void     usage_send_wrapper();

/*
 * Configurable options.
 */
extern ACL		*acllist;
extern int		authentication;
extern char		*rli_bloomfilter_dir;
extern int		lrc_bloomfilter_ratio;
extern int		rli_bloomfilter;
extern char		*db_pwd;
extern char		*db_user;
extern int		idletimeout;
extern int		loglevel;
extern char		*lrc_dbname;
extern int		lrc_server;
extern int		maxbackoff;
extern int		maxconnections;
extern int		maxfreethreads;
extern int		maxthreads;
extern char		*myurl;
extern int		update_buftime;
extern int		port;
extern char		*pidfile;
extern char		*rli_dbname;
extern int		update_bf_int;
extern int		update_immediate;
extern int		rli_server;
extern int		update_ll_int;
extern int		update_factor;
extern int		result_limit;
extern char		*rlscertfile;
extern char		*rlskeyfile;
extern int		startthreads;
extern int		timeout;
extern bloomfilter_t	bloomfilter;
extern TABLE		table[];

extern SENDER		*senderlist;
extern LOCK		senderlistlock;
extern RLI		*lrc_rlilist;
extern LOCK		lrc_rlilistlock;
extern RLI		*rli_rlilist;
extern LOCK		rli_rlilistlock;

/* this extern variable is set in the lrc_init function and indicates that the
lrc_init function succeeded in creating the lrc server. */

static char		*conffile = NULL;
static char		conffilebuf[BUFLEN];
int			rlsdebug = 0;
static int		numthreads = 0;
static int		freethreads = 0;
int			quitting = 0;
static int		delayquit = 0;
static int		needlisten = 1;
static int		needstart = 0;
static int		needreadconf = 0;
static globus_fifo_t	reqqueue;
static globus_mutex_t	reqqueuemtx;
static globus_cond_t	reqqueuecond;
globus_mutex_t		commonmtx;
globus_cond_t		commoncond;
static globus_list_t	*connlist;
static int		conncount = 0;
static globus_mutex_t	connlistmtx;
static int		exitsigs[] = {SIGINT, SIGQUIT, SIGTERM, 0};
static time_t		starttime;

#define DEFAULT_USAGE_STATS_INT (60*60*24) /* Default == 24 hour interval */
static int          usage_stats_int = DEFAULT_USAGE_STATS_INT;
static time_t       next_usage_stats_upd = 0;

static globus_module_descriptor_t	*modules[] = {
  GLOBUS_COMMON_MODULE,
  GLOBUS_IO_MODULE,
  GLOBUS_USAGE_MODULE,
  GLOBUS_RLS_CLIENT_MODULE,
};
#define NMODS	(sizeof(modules) / sizeof(globus_module_descriptor_t *))

int
main(int argc, char **argv)

{
  extern char		*optarg;
  extern int		optind;
  char			*prog;
  int			i;
  globus_result_t	r;
  globus_io_handle_t	handle;
  char			*s;
  int			n;
  int			oldlrc_server = 0;
  int			oldrli_server = 0;
  struct timespec	t;
  int			eventwaiting = 0;
  time_t		now;
  static int		backoff = 1;
  struct        sigaction	sa;

  /* use this structure to pass the db_name, db_user and db_passwd to the
   lrc_init function through the pthread_create() function call. */

  lrc_init_args_t lrc_init_params;

  /* the thread to create the lrc; */
  pthread_t lrc_thread;

  /* the return value from the pthread_create call. This indicates if the
   thread was created successfully or not. However, it does not indicate if
   the lrc_server was created successfully or not. The later is determined
   from the return value of is_lrc_init_done() */

  int lrc_thread_rv;

  if ((prog = strrchr(*argv, '/')) == NULL)
    prog = *argv;
  else
    prog++;
  while ((i = getopt(argc, argv, "B:b:C:c:de:F:f:h:I:i:K:L:l:M:m:Nn:o:r:S:s:t:u:U:v")) != -1)
    switch (i) {
      case 'B': conf_option("update_bf_int", optarg, 1);
		break;
      case 'b': conf_option("maxbackoff", optarg, 1);
		break;
      case 'C': conf_option("rlscertfile", optarg, 1);
		break;
      case 'c': conffile = optarg;
		break;
      case 'd':	rlsdebug++;
		break;
      case 'e': conf_option("rli_expire_int", optarg, 1);
		break;
      case 'F': conf_option("update_factor", optarg, 1);
		break;
      case 'f': conf_option("maxfreethreads", optarg, 1);
		break;
      case 'I': conf_option("update_immediate", optarg, 1);
		break;
      case 'i':	conf_option("idletimeout", optarg, 1);
		break;
      case 'K': conf_option("rlskeyfile", optarg, 1);
		break;
      case 'L': conf_option("loglevel", optarg, 1);
		break;
      case 'l':	conf_option("lrc_server", optarg, 1);
		break;
      case 'M': conf_option("maxconnections", optarg, 1);
		break;
      case 'm': conf_option("maxthreads", optarg, 1);
		break;
      case 'N': conf_option("authentication", "false", 1);
		break;
      case 'o': conf_option("update_buftime", optarg, 1);
		break;
      case 'p': conf_option("pidfile", optarg, 1);
		break;
      case 'r':	conf_option("rli_server", optarg, 1);
		break;
      case 'S': conf_option("rli_expire_stale", optarg, 1);
		break;
      case 's': conf_option("startthreads", optarg, 1);
		break;
      case 't': conf_option("timeout", optarg, 1);
		break;
      case 'U': conf_option("myurl", optarg, 1);
		break;
      case 'u': conf_option("update_ll_int", optarg, 1);
		break;
      case 'v': printf("Version: %d.%d\n", MAJOR, MINOR);
		exit(0);
      default:	usage(prog);
		break;
    }

  if (!conffile) {
    if ((s = getenv("GLOBUS_LOCATION")))
      sprintf(conffilebuf, "%s/etc/%s.conf", s, prog);
    else
      sprintf(conffilebuf, "/usr/local/etc/%s.conf", prog);
    conffile = conffilebuf;
  }

  if (!rlsdebug)
    detach(prog);

  /*
   * If we detach after activating globus modules we get mysterious hangs
   * in globus_libc, so delay activation until after detaching.
   */
  for (i = 0; i < NMODS; i++)
    if ((n = globus_module_activate(modules[i])) != GLOBUS_SUCCESS) {
      fprintf(stderr, "main: Unable to activate module %d error %d", i, n);
      cleanexit(1);
    }

  if (log_init() != GLOBUS_SUCCESS)
  {
      fprintf(stderr, "main: Unable to initialize log");
      cleanexit(1);
  }

  conf_read(prog, conffile);
  if (!rlsdebug)
    writepid(prog);

  sigemptyset(&sa.sa_mask);
  sa.sa_flags = SA_RESTART;
  sa.sa_handler = hexitsig;

  for (i = 0; exitsigs[i]; i++)
    sigaction(exitsigs[i], &sa, NULL);

  sa.sa_handler = hhupsig;
  sigaction(SIGHUP, &sa, NULL);

  init();
  if ((r = startlistener((u_short) port, &handle)) != GLOBUS_SUCCESS)
    cleanexit(1);

  for (i = 0; i < startthreads; i++)
    if (!startthread())
      cleanexit(1);

  /*  dolisten(&handle);*/
  t.tv_nsec = 0;
  while (1) {
    /*
     * If config changed we may need to start up or end RLI or LRC
     * service.
     */
    if (oldlrc_server != lrc_server) {
      if (lrc_server) {
        lrc_init_params.dbname = lrc_dbname;
        lrc_init_params.db_user = db_user;
        lrc_init_params.db_pwd = db_pwd;

        /* create the lrc server in a separate thread. */
        lrc_thread_rv = pthread_create(&lrc_thread, NULL, lrc_init, (void *) & lrc_init_params);

        /* note: only looks at return value of thread creation and not return value of lrc_init */
        if (!lrc_thread_rv) {
          oldlrc_server = lrc_server;
        }
      } else {
	lrc_end();
	oldlrc_server = lrc_server;
      }
    }
    if (oldrli_server != rli_server) {
      if (rli_server) {
	if (rli_init(rli_dbname, db_user, db_pwd) == GLOBUS_RLS_SUCCESS)
	  oldrli_server = rli_server;
      } else {
	rli_end();
	oldrli_server = rli_server;
      }

      /* Send usage stass at startup */
      usage_send_wrapper();
    }

    /*
     * Wait for something to happen.
     */
    globus_mutex_lock(&commonmtx);
    while (!quitting && !needstart && !needlisten && !needreadconf) {
      if ((i = event_when()) == -1)
	globus_cond_wait(&commoncond, &commonmtx);
      else {
	/*
	 * Want to wait until next event, a request comes in, or we're
	 * signaled with a quit signal.  Ideally the signal handler should
	 * do a globus_cond_signal() to wake us up, however it's not safe
	 * to call pthread_cond_signal on linux from an async signal handler.
	 * So we wake up every 5 seconds whether we need to or not to see
	 * if we should exit.
	 */
	now = time(0);
	if (i > now + 5)
	  t.tv_sec = now + 5;
	else
	  t.tv_sec = i;
	globus_cond_timedwait(&commoncond, &commonmtx, &t);
	if (i <= time(0)) {
	  eventwaiting = 1;
	  break;
	}
      }
    }
    globus_mutex_unlock(&commonmtx);

    if (quitting) {
	/*
	 * If got admin quit cmd sleep for a sec to give admin prog chance to
	 * close connection cleanly, otherwise we may hang while canceling io
	 * callbacks.  There's probably a better way to do this.
	 */
      if (delayquit)
	sleep(delayquit);
      doquit(&handle);
      break;
    }
    if (needstart) {
       needstart = 0;
       /*
	* Create a new listener, retry if we fail. Double backoff
	* interval between retries upto a maximum.
	*/
       while (!quitting) {
	  if (startlistener((u_short) port, &handle) == GLOBUS_SUCCESS) {
	     backoff = 1;
	     /*
	      * Start listening again with this listener
	      */
	     globus_mutex_lock(&commonmtx);
	     needlisten++;
	     globus_mutex_unlock(&commonmtx);
	     break;
	  }

	  if (backoff < maxbackoff)
	     backoff <<= 1;
	  sleep(backoff);
       }
    }
    if (needlisten) {
      needlisten = 0;
      dolisten(&handle);
    }
    if (needreadconf) {
      needreadconf = 0;
      conf_read(prog, conffile);
    }
    if (eventwaiting) {
      eventwaiting = 0;
      event_process();
    }

    /*
     * Adjust number of free threads (add if none, kill if above max).
     */
    if (!freethreads && numthreads < maxthreads)
      if (!startthread())
	break;
    n = freethreads - maxfreethreads;
    if (loglevel && n > 0)
      logit(LOG_DEBUG, "main: Killing %d excess free threads", n);
    for (i = 0; i < n; i++)
      queueconn(C_EXIT);
  }

  cleanexit(0);
  exit(0);	/* Stop compiler warning	*/
}

static void
doquit(globus_io_handle_t *handlep)

{
  int	i;

  /*
   * Cancel any callbacks on the handle we're listening on.
   * The cancel callback will close the handle.
   */
  globus_io_register_cancel(handlep, GLOBUS_FALSE, cancelcb, NULL);

  if (lrc_server)
    lrc_end();
  if (rli_server)
    rli_end();
  globus_mutex_lock(&commonmtx);
  for (i = 0; i < numthreads; i++)
    queueconn(C_EXIT);
  while (numthreads)
    globus_cond_wait(&commoncond, &commonmtx);
  globus_mutex_unlock(&commonmtx);

   if (loglevel)
     for (i = 0; i < T_MAX; i++)
       printf("%s: %d\n", table[i].name, table[i].count);
}

static void
dolisten(globus_io_handle_t *handlep)

{
  globus_result_t	r;
  char			errmsg[MAXERRMSG];

  if (loglevel > 3)
    logit(LOG_DEBUG, "dolisten:");
  
  if ((r = globus_io_tcp_register_listen(handlep, listencb,
		NULL)) != GLOBUS_SUCCESS) {
     logit(LOG_WARNING, "dolisten: globus_io_tcp_register_listen: %s",
		globuserr(r, errmsg));
     /*
      * Close the handle we listen on. The close callback will
      * recreate a listener and start listening again
      */
     globus_io_register_close(handlep, startcb, NULL);
  }
}

static void
doaccept(globus_io_handle_t *handlep, CONNECTION *c)

{
  globus_io_attr_t  attr;
  globus_io_secure_authorization_data_t ad;
  globus_result_t   r;
  int               authzdata_initialized = 0;
  int               attr_initialized = 0;
  char              errmsg[MAXERRMSG];

  if (authentication) {
    /*
     * Initialize attribute. If any part of the initialization
     * fails, clean up as necessary and report error.
     * Liberal use of goto here, seems to me, makes this
     * easier to read.
     * XIO note: Supposed to have an attribute copy function
     * which can probably be used here.
     */
    if ((r = globus_io_tcpattr_init(&attr)) != GLOBUS_SUCCESS)
        goto done;
    attr_initialized++;
    if ((r = globus_io_attr_set_secure_authentication_mode(&attr,
            GLOBUS_IO_SECURE_AUTHENTICATION_MODE_GSSAPI,
            GSS_C_NO_CREDENTIAL)) != GLOBUS_SUCCESS)
        goto done;
    if ((r = globus_io_secure_authorization_data_initialize(&ad)) !=
            GLOBUS_SUCCESS)
        goto done;
    authzdata_initialized++;
    if ((r = globus_io_secure_authorization_data_set_callback(&ad,
            authcb, (void *) c)) != GLOBUS_SUCCESS)
        goto done;
    if ((r = globus_io_attr_set_secure_authorization_mode(&attr,
            GLOBUS_IO_SECURE_AUTHORIZATION_MODE_CALLBACK, &ad)) !=
            GLOBUS_SUCCESS)
        goto done;
    if ((r = globus_io_attr_set_socket_reuseaddr(&attr, GLOBUS_TRUE)) !=
            GLOBUS_SUCCESS)
        goto done;
    r = globus_io_tcp_register_accept(handlep, &attr, &c->handle,
        acceptcb, (void *) c);
  } else {
    r = globus_io_tcp_register_accept(handlep, GLOBUS_NULL, &c->handle,
        acceptcb, (void *) c);
  }

done:
  if (authzdata_initialized)
    globus_io_secure_authorization_data_destroy(&ad);
  if (attr_initialized)
    globus_io_tcpattr_destroy(&attr);
  if (r != GLOBUS_SUCCESS) {
    logit(LOG_WARNING, "doaccept: %s", globuserr(r, errmsg));
    freeconn(c, 0);
  }
}

/*
 * startlistener - Create listen handle.  Attempt to set reuseaddr on
 *   socket but ignore errors if that fails.
 */
static globus_result_t
startlistener(u_short port, globus_io_handle_t *handlep)

{
  globus_result_t			r;
  globus_io_attr_t			attr;
  globus_io_secure_authorization_data_t	ad;
  int					havead = 0;
  char					errmsg[MAXERRMSG];

  if (loglevel)
    logit(LOG_DEBUG, "startlistener: Port %d", port);
  if ((r = globus_io_tcpattr_init(&attr)) != GLOBUS_SUCCESS) {
    logit(LOG_WARNING, "startlistener: tcpattr_init: %s", globuserr(r,errmsg));
    return r;
  }
  if (authentication) {
    if ((r = globus_io_attr_set_secure_authentication_mode(&attr,
		  GLOBUS_IO_SECURE_AUTHENTICATION_MODE_GSSAPI,
		  GSS_C_NO_CREDENTIAL)) != GLOBUS_SUCCESS) {
      logit(LOG_WARNING, "startlistener: set_auth_mode: %s",
	    globuserr(r, errmsg));
      goto done;
    }
    if ((r = globus_io_secure_authorization_data_initialize(&ad)) !=
		  GLOBUS_SUCCESS) {
      logit(LOG_WARNING, "startlistener: auth_data_init: %s",
	    globuserr(r, errmsg));
      goto done;
    }
    havead++;
    if ((r = globus_io_secure_authorization_data_set_callback(&ad, authcb,
		  NULL)) != GLOBUS_SUCCESS) {
      logit(LOG_WARNING, "startlistener: set_auth_callback: %s",
	    globuserr(r, errmsg));
      goto done;
    }
    if ((r = globus_io_attr_set_secure_authorization_mode(&attr,
	  GLOBUS_IO_SECURE_AUTHORIZATION_MODE_CALLBACK, &ad))!=GLOBUS_SUCCESS){
      logit(LOG_WARNING, "startlistener: set_authorization_mode: %s",
	    globuserr(r, errmsg));
      goto done;
    }
  }
  if ((r = globus_io_attr_set_socket_reuseaddr(&attr,
		GLOBUS_TRUE)) != GLOBUS_SUCCESS)
    logit(LOG_WARNING, "startlistener: set_reuseaddr: %s",
	  globuserr(r, errmsg));
  if ((r = globus_io_tcp_create_listener(&port, -1, &attr,
					 handlep)) != GLOBUS_SUCCESS)
    logit(LOG_WARNING, "startlistener: %s", globuserr(r, errmsg));

 done:
    if (havead)
      globus_io_secure_authorization_data_destroy(&ad);
    globus_io_tcpattr_destroy(&attr);
    return r;
}

static void
listencb(void *a, globus_io_handle_t *handlep, globus_result_t r)

{
  CONNECTION       *c;
  char             errmsg[MAXERRMSG];
  globus_object_t  *eo;

  if (loglevel > 3)
    logit(LOG_DEBUG, "listencb:");
  if (r != GLOBUS_SUCCESS) {
    eo = globus_error_get(r);
    if (!eo || !globus_object_type_match(globus_object_get_type(eo),
        GLOBUS_IO_ERROR_TYPE_IO_CANCELLED))
      logit(LOG_WARNING, "listencb: %s", globuserr(r, errmsg));
    if (eo)
      globus_object_free(eo);
  } else {
    if ((c = newconn()) != NULL)
      doaccept(handlep, c);
  }

  /* Tell listener it needs to listen again			*/
  globus_mutex_lock(&commonmtx);
  needlisten++;
  globus_cond_signal(&commoncond);
  globus_mutex_unlock(&commonmtx);
}

/*
 * New connection processing.  Sets up read callback for handle
 * and signals listener it needs to listen again.
 */
static void
acceptcb(void *cv, globus_io_handle_t *handlep, globus_result_t r)

{
  CONNECTION	*c = (CONNECTION *) cv;

  if (loglevel > 3)
    logit(LOG_DEBUG, "acceptcb: Connection %X", c);
  /* Check for error in accept					*/
  if (r != GLOBUS_SUCCESS) {
    logit(LOG_WARNING, "acceptcb: %s", globuserr(r, c->errmsg));
    freeconn(c, 0);
    return;
  }

  if (conncount >= maxconnections) {
    rrpc_error(c, GLOBUS_RLS_TOO_MANY_CONNECTIONS, "%d", conncount);
    freeconn(c, 0);
    return;
  }

  if (!c->perms) {	/* Verify client has privs		*/
    rrpc_error(c, GLOBUS_RLS_PERM, "%s", c->dn);
    freeconn(c, 0);
    return;
  }

  rrpc_success(c);
  if ((r = globus_io_register_read(&c->handle, c->ibuf.buf, RPCBSIZ, 1,
				   readcb, c)) != GLOBUS_SUCCESS) {
    logit(LOG_WARNING, "acceptcb: %s", globuserr(r, c->errmsg));
    freeconn(c, 0);
  }
}

/*
 * Authentication callback.
 */
globus_bool_t
authcb(void *cv, globus_io_handle_t *handlep, globus_result_t r,
       char *identity, gss_ctx_id_t ch)

{
  char		errmsg[MAXERRMSG];
  CONNECTION	*c = (CONNECTION *) cv;	

  if (r != GLOBUS_SUCCESS) {
    logit(LOG_WARNING, "authcb: %s", globuserr(r, errmsg));
    return GLOBUS_FALSE;
  }
  c->dn = globus_libc_strdup(identity);
  c->perms = auth_getperms(acllist, c->dn);
  if (loglevel)
    logit(LOG_DEBUG, "authcb: Accepted connection from %s", c->dn);
  return GLOBUS_TRUE;
}

static void
readcb(void *cv, globus_io_handle_t *handlep, globus_result_t r,
       globus_byte_t *buf, globus_size_t len)

{
  CONNECTION	*c = (CONNECTION *) cv;

  if (r != GLOBUS_SUCCESS) {
    logit(LOG_WARNING, "readcb: %s", globuserr(r, c->errmsg));
    freeconn(c, 0);
    return;
  }
  if (loglevel > 3)
    logit(LOG_DEBUG, "readcb:");
  c->ibuf.len = len;
  c->ibuf.idx = 0;
  queueconn(c);
}

static void
closecb(void *cv, globus_io_handle_t *handlep, globus_result_t r)

{
  CONNECTION *c = (CONNECTION *) cv;
  if (c == GLOBUS_NULL) {
     if (loglevel)
 	logit(LOG_DEBUG, "closecb: closed server handle");
     return;
  }
  if (loglevel > 3)
     logit(LOG_DEBUG, "closecb: freeing connection %X", c);
  if (c->dn)
     globus_libc_free(c->dn);
  globus_libc_free(c);
}

static void
startcb(void *a, globus_io_handle_t *handlep, globus_result_t r)

{
  if (loglevel > 3)
     logit(LOG_DEBUG, "startcb:");
  /*
   * We don't want to try recreating the listener here since
   * we're in a callback. Setting the needstart flag will
   * recreate the listener in main()
   */
   globus_mutex_lock(&commonmtx);
   needstart++;
   globus_mutex_unlock(&commonmtx);
}

static void
cancelcb(void *cv, globus_io_handle_t *handlep, globus_result_t r)

{
  globus_result_t 	s = GLOBUS_FAILURE;
  char			errmsg[MAXERRMSG];
  CONNECTION		*c = (CONNECTION *) cv;

  if (loglevel > 3) 
     logit(LOG_DEBUG, "cancelcb: connection %X", c);

  /*
   * If we're quitting, we don't call into globus_io since the
   * module might be deactivated
   */
  if (!quitting && *handlep) {
     if ((s = globus_io_register_close(handlep, closecb, c)) !=
		GLOBUS_SUCCESS)
	logit(LOG_WARNING, "cancelcb: %s", globuserr(s, errmsg));
  }
     
  if ((quitting || s != GLOBUS_SUCCESS) && c) {
     if (c->dn)
        globus_libc_free(c->dn);
     globus_libc_free(c);
  }
}

/*
 * queueconn - We have input for a CONNECTION, queue it up to be
 *   serviced by a worker thread.
 */
static void
queueconn(CONNECTION *c)

{
  globus_mutex_lock(&reqqueuemtx);
  globus_fifo_enqueue(&reqqueue, (void *) c);
  globus_cond_signal(&reqqueuecond);
  globus_mutex_unlock(&reqqueuemtx);
}

/*
 * procreq - Wait for incoming connection handle to be placed on queue, then
 *  process it.  If the handle on the queue is NULL we're exiting.  Input
 *  has already been read and stored in c->b by the read callback
 *  (readcb).
 */
static void *
procreq(void *a)

{
  globus_result_t	r;
  CONNECTION		*c;
  char			*arglist[MAXARGS];
  int			mi;
  int			i;
  DBH			dbh = { NULL, NULL };
  int			rc;
  sigset_t		ss;
  char			errbuf[MAXERRMSG];

  sigemptyset(&ss);
  sigaddset(&ss, SIGALRM);
  sigaddset(&ss, SIGHUP);
  for (i = 0; exitsigs[i]; i++)
    sigaddset(&ss, exitsigs[i]);
  pthread_sigmask(SIG_BLOCK, &ss, NULL);

  while (1) {
    globus_mutex_lock(&reqqueuemtx);
    freethreads++;
    while (globus_fifo_empty(&reqqueue))
      globus_cond_wait(&reqqueuecond, &reqqueuemtx);
    c = (CONNECTION *) globus_fifo_dequeue(&reqqueue);
    freethreads--;
    globus_mutex_unlock(&reqqueuemtx);

    if (c == C_EXIT)			/* Parent wants us to exit?	*/
      endthread(&dbh);

    /*
     * Reconfiguration (including enabling or disabling LRC or RLI service)
     * is allowed at any time, so check if database connections need to
     * be opened or closed before processing a request.
     */
    if (lrc_server) {
      if (dbh.lrc && db_is_alive(dbh.lrc) != GLOBUS_RLS_SUCCESS) {
        db_close(dbh.lrc);
        dbh.lrc = NULL;
      }
      if (!dbh.lrc) {
        if ((rc = db_open(lrc_dbname, db_user, db_pwd, 1, &dbh.lrc,
			  errbuf)) != GLOBUS_RLS_SUCCESS) {
          logit(LOG_WARNING, "procreq: %s: %s", lrc_dbname, errbuf);
          rrpc_error(c, GLOBUS_RLS_DBERROR, "%s: %s", lrc_dbname, errbuf);
          goto nextread;
        }
      }
    } else {
      if (dbh.lrc) {
        db_close(dbh.lrc);
        dbh.lrc = NULL;
      }
    }

    if (rli_server) {
      if (dbh.rli && db_is_alive(dbh.rli) != GLOBUS_RLS_SUCCESS) {
        db_close(dbh.rli);
        dbh.rli = NULL;
      }
      if (!dbh.rli) {
        if ((rc = db_open(rli_dbname, db_user, db_pwd, 0, &dbh.rli,
			  errbuf)) != GLOBUS_RLS_SUCCESS) {
          logit(LOG_WARNING, "procreq: %s: %s", rli_dbname, errbuf);
          rrpc_error(c, GLOBUS_RLS_DBERROR, "%s: %s", lrc_dbname, errbuf);
          goto nextread;
        }
      }
    } else {
      if (dbh.rli) {
        db_close(dbh.rli);
        dbh.rli = NULL;
      }
    }

    *c->errmsg = '\0';			/* Clear error msg buffer	*/
    c->lastcall = time(0);
    if ((rc = getmethodargs(c, &mi, arglist)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      freeconn(c, 0);
      continue;
    }
    if ((c->perms & methods[mi].perms) != methods[mi].perms) {
      rrpc_error(c, GLOBUS_RLS_PERM, "%s", c->dn);
      goto nextread;
    }
    if ((methods[mi].flags & MF_NEEDLRC) && !lrc_server) {
      if (methods[mi].flags & MF_NTA)
	rrpc_flushio(c);
      rrpc_error(c, GLOBUS_RLS_INVSERVER, "not LRC server");
      goto nextread;
    }
		if ((methods[mi].flags & MF_NEEDLRCBF) && !is_lrc_init_done()) {
    	if (methods[mi].flags & MF_NTA)
   			rrpc_flushio(c);
     	rrpc_error(c, GLOBUS_RLS_INITIALIZING, "LRC server not initialized");
     	goto nextread;
   	}
    if ((methods[mi].flags & MF_NEEDRLI) && !rli_server) {
      if (methods[mi].flags & MF_NTA)
	rrpc_flushio(c);
      rrpc_error(c, GLOBUS_RLS_INVSERVER, "not RLI server");
      goto nextread;
    }
    methods[mi].fun(c, &dbh, arglist);
    for (i = 0; i < methods[mi].nargs; i++)
      if (arglist[i])
	globus_libc_free(arglist[i]);
    if (methods[mi].fun == rls_close)
      continue;

   nextread:
    if ((r = globus_io_register_read(&c->handle, c->ibuf.buf, RPCBSIZ, 1,
				     readcb, c)) != GLOBUS_SUCCESS) {
      logit(LOG_WARNING, "procreq: %s", globuserr(r, c->errmsg));
      freeconn(c, 0);
    }
  }
}

/*
 * getmethodargs - Get method name and args from buf.  This will eventually
 *   be SOAP, for now method name and args are null terminated strings.
 *   The index into the methods[] array will be returned in *mi,
 *   pointers to arg strings in arglist[].  If an arg string is empty
 *   a NULL is returned in arglist[].
 */
static int
getmethodargs(CONNECTION *c, int *mi, char **arglist)

{
  int	i;
  int	ai;
  char	buf[BUFLEN];
  int	rc;
  int	ch;
  
  i = 0;
  while ((ch = NEXTC(&c->handle, &c->ibuf, &rc, c->errmsg)) != -1 && i<BUFLEN)
    if (!(buf[i++] = ch))
      break;
  if (ch == -1)
    return rc;
  if (i == BUFLEN && buf[BUFLEN-1] != '\0') {
    snprintf(c->errmsg, MAXERRMSG,
            "getmethodargs: bad method: overflow: %s", buf);
    return GLOBUS_RLS_BADMETHOD;
  }
  for (*mi = 0; methods[*mi].method; (*mi)++)
    if (strcasecmp(buf, methods[*mi].method) == 0)
      break;
  if (!methods[*mi].method) {
    strncpy(c->errmsg, buf, MAXERRMSG);
    return GLOBUS_RLS_BADMETHOD;
  }
  for (ai = 0; ai < methods[*mi].nargs; ai++) {
    i = 0;
    while ((ch = NEXTC(&c->handle, &c->ibuf, &rc, c->errmsg)) != -1)
      if (!(buf[i++] = ch))
	break;
    if (ch == -1)
      goto badarg;
    if (i == BUFLEN && buf[BUFLEN-1] != '\0') {
      snprintf(c->errmsg, MAXERRMSG,
               "getmethodargs: bad arg: overflow: %s", buf);
      rc = GLOBUS_RLS_BADARG;
      goto badarg;
    }
    arglist[ai] = *buf ? globus_libc_strdup(buf) : NULL;
  }
  return GLOBUS_RLS_SUCCESS;
 badarg:
  for (i = 0; i < ai; i++)
    if (arglist[i])
      globus_libc_free(arglist[i]);
  return rc;
}

/*
 * Allocate new CONNECTION struct.
 */
static CONNECTION *
newconn()

{
  CONNECTION    *c;
  globus_bool_t send_usage_stats = GLOBUS_FALSE;

  if ((c = (CONNECTION *) globus_libc_malloc(sizeof(CONNECTION))) == NULL) {
    logit(LOG_WARNING, "newconn: No memory for CONNECTION");
    return NULL;
  }
  if (loglevel > 3)
    logit(LOG_DEBUG, "newconn: Connection %X", c);
  c->lastcall = time(0);
  c->dn = NULL;
  c->numrecs = 0;
  c->perms = authentication ? 0 : -1;
  globus_mutex_lock(&connlistmtx);
  conncount++;
  globus_list_insert(&connlist, (void *) c);

  /* Synchronize to check/update usage stats update interval */
  if (next_usage_stats_upd < time(0)) {
    next_usage_stats_upd += usage_stats_int;
    send_usage_stats = GLOBUS_TRUE;
  }
  globus_mutex_unlock(&connlistmtx);

  if (send_usage_stats)
    usage_send_wrapper();
  return c;
}

/*
 * Free CONNECTION struct.
 */
static void
freeconn(CONNECTION *c, int locked)

{
  globus_list_t	*l;
  globus_result_t r = GLOBUS_FAILURE;
  char errmsg[MAXERRMSG];

  if (loglevel > 3)
    logit(LOG_DEBUG, "freeconn: Connection %X", c);
  if (!locked)
    globus_mutex_lock(&connlistmtx);
  if ((l = globus_list_search(connlist, (void *)c)) == NULL)
    logit(LOG_WARNING, "freeconn: Connection %X not on connqueue", c);
  else {
    globus_list_remove(&connlist, l);
    conncount--;
    /*
     * If the handle is initialized, the CONNECTION structure is freed up
     * in closecb. If not, we can free the struct right here.
     * If we're quitting, it's not reliable to call into globus_io,
     * since the module might be deactivated from under us.
     */
    if (c->handle && !quitting) {
       if ((r = globus_io_register_cancel(&c->handle, GLOBUS_FALSE,
		cancelcb, (void *) c)) != GLOBUS_SUCCESS)
	  logit(LOG_WARNING, "freeconn: %s", globuserr(r, errmsg));
    }
    if (quitting || r != GLOBUS_SUCCESS) {
       if (c->dn)
	  globus_libc_free(c->dn);
       globus_libc_free(c);
    }
  }
  if (!locked)
    globus_mutex_unlock(&connlistmtx);
}

/*
 * Check connlist for connections that should be timed out.
 */
static void *
checkidle(void *a)

{
  EVENTCONTROL	*ec = (EVENTCONTROL *) a;
  globus_list_t	*l;
  CONNECTION	*c;
  time_t	now;

  while (1) {
    globus_mutex_lock(&ec->mtx);
    while (!(ec->flags & (EF_EXIT|EF_RUN)))
      globus_cond_wait(&ec->cond, &ec->mtx);

    if (ec->flags & EF_EXIT) {
      ec->flags &= ~EF_RUNNING;
      globus_cond_signal(&ec->cond);
      globus_mutex_unlock(&ec->mtx);
      pthread_exit(0);
    } else {
      ec->flags &= ~EF_RUN;
      globus_mutex_unlock(&ec->mtx);
    }

    if (loglevel)
      logit(LOG_DEBUG, "checkidle:");
    now = time(0);
    globus_mutex_lock(&connlistmtx);
    l = connlist;
    while (!globus_list_empty(l)) {
      c = globus_list_first(l);
      l = globus_list_rest(l);
      if (c->lastcall + idletimeout < now) {
	if (loglevel)
	  logit(LOG_DEBUG, "checkidle: Timing out connection %X", c);
	/*
	 * freeconn will remove this connection from connlist and 
	 * the struct will eventually be freed
	 */
	freeconn(c, 1);
      }
    }
    globus_mutex_unlock(&connlistmtx);
  }
}

static void
init()

{
  int	i;

  /*
   * reqqueue, reqqueuemtx and reqqueuecond are used to synchonize access to
   * the request queue.
   */
  if ((i = globus_fifo_init(&reqqueue)) != 0) {
    logit(LOG_WARNING, "fifo_init: %d", i);
    cleanexit(1);
  }
  if ((i = globus_mutex_init(&reqqueuemtx, GLOBUS_NULL)) != 0) {
    logit(LOG_WARNING, "mutex_init: %d", i);
    cleanexit(1);
  }
  if ((i = globus_cond_init(&reqqueuecond, GLOBUS_NULL)) != 0) {
    logit(LOG_WARNING, "cond_init: %d", i);
    cleanexit(1);
  }

  /*
   * commonmtx and commoncond are used to wake up the parent when
   * events happen (new request, or quit signal received).  Also used
   * to protect numthreads counter.
   */
  if ((i = globus_mutex_init(&commonmtx, GLOBUS_NULL)) != 0) {
    logit(LOG_WARNING, "mutex_init: %d", i);
    cleanexit(1);
  }
  if ((i = globus_cond_init(&commoncond, GLOBUS_NULL)) != 0) {
    logit(LOG_WARNING, "cond_init: %d", i);
    cleanexit(1);
  }

  /*
   * connlist and connlistmtx are used to synchronize access to
   * the connection queue, which is used to timeout idle connections.
   */
  connlist = NULL;
  if ((i = globus_mutex_init(&connlistmtx, GLOBUS_NULL)) != 0) {
    logit(LOG_WARNING, "mutex_init: %d", i);
    cleanexit(1);
  }

  event_init();
  event_queue(checkidle, &idletimeout);

  update_init();

  db_init();

  /* Initialize usage stats support */
  next_usage_stats_upd = time(0) + usage_stats_int;
  usage_init();

  starttime = time(0);
}

static int
rrpc_rliinfo(CONNECTION *c, RLI *rli, int first, int last, int termlist)

{
  int	rc;
  char	lbuf[MAXERRMSG];
  int	ui;
  int	bff;

  if (first) {
    rrpc_initbuf(&c->obuf);
    if ((rc = rrpc_bufwrite(&c->handle, &c->obuf, "0", 2, 0,
			    c->errmsg)) != GLOBUS_RLS_SUCCESS) {
      logit(LOG_WARNING, "rrpc_rliinfo: rrpc_bufwrite: %s",
	    globus_rls_errmsg(rc, c->errmsg, lbuf, MAXERRMSG));
      return rc;
    }
  }
  if ((rc = rrpc_bufwrite(&c->handle, &c->obuf, rli->url, strlen(rli->url) + 1,
			  0, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    logit(LOG_WARNING, "rrpc_rliinfo: rrpc_bufwrite: %s",
	  globus_rls_errmsg(rc, c->errmsg, lbuf, MAXERRMSG));
    return rc;
  }
  if (rli->flags & FR_BLOOMFILTER) {
    if (update_immediate && rli->softstatestart == rli->lastupdate)
      ui = update_bf_int * update_factor;
    else
      ui = update_bf_int;
    bff = FRLI_BLOOMFILTER;
  } else {
    if (update_immediate && (rli->flags & FR_SYNCED))
      ui = update_ll_int * update_factor;
    else
      ui = update_ll_int;
    bff = 0;
  }
  if (last && termlist)
    rc = rrpc_bufprintf(&c->handle, &c->obuf, last, c->errmsg,
			"%u%c%u%c%u%c%c", ui, 0, bff, 0, rli->lastupdate, 0,0);
  else
    rc = rrpc_bufprintf(&c->handle, &c->obuf, last, c->errmsg,
			"%u%c%u%c%u%c", ui, 0, bff, 0, rli->lastupdate, 0);
  if (rc != GLOBUS_SUCCESS) {
    logit(LOG_WARNING, "rrpc_rliinfo: rrpc_bufprintf: %s",
	  globus_rls_errmsg(rc, c->errmsg, lbuf, MAXERRMSG));
    return rc;
  }

  return GLOBUS_RLS_SUCCESS;
}

/*
 * Return error code and error message to client.  Also logs it.
 */
static void
rrpc_error(CONNECTION *c, int rc, char *fmt, ...)

{
  va_list	ap;
  char		rcbuf[20];
  char		buf[BUFLEN];
  char		lbuf[MAXERRMSG];
  globus_size_t	len;
  struct iovec	iov[2];
  int		trc;

  va_start(ap, fmt);
  globus_libc_sprintf(rcbuf, "%d", rc);
  iov[0].iov_base = rcbuf;
  iov[0].iov_len = strlen(rcbuf) + 1;
  iov[1].iov_base = buf;
  iov[1].iov_len = vsnprintf(buf, BUFLEN, fmt, ap) + 1;
  if ((trc = rrpc_writev(&c->handle, iov, 2, &len,
			 c->errmsg)) != GLOBUS_RLS_SUCCESS)
    logit(LOG_WARNING, "rrpc_error: rrpc_writev: %s",
	  globus_rls_errmsg(trc, c->errmsg, lbuf, MAXERRMSG));
  if (loglevel > 1)
    logit(LOG_INFO, "%s", globus_rls_errmsg(rc, buf, lbuf, MAXERRMSG));
  va_end(ap);
  c->numrecs = 0;
}

static int
rrpc_success(CONNECTION *c)

{
  int		rc;
  globus_size_t	len;
  char		lbuf[MAXERRMSG];

  if ((rc = rrpc_write(&c->handle, (globus_byte_t *) "0", 2,
		       &len, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    logit(LOG_WARNING, "rrpc_success: rrpc_write: %s",
	  globus_rls_errmsg(rc, c->errmsg, lbuf, MAXERRMSG));
    return GLOBUS_FALSE;
  }
  return GLOBUS_TRUE;
}

/*
 * rrpc_flushio - In some cases a RPC call error can be detected before the
 *   entire request has been read (eg if an LRC -> RLI call is made, but
 *   the server side is not configured as an RLI server).  We need to
 *   flush the entire request.  Unfortunately globus_io does not provide
 *   this operation, and globus_io_try_read() does not work on TCP sockets
 *   with GSI.  For most requests the args are parsed at the same time
 *   as the method name, except when the args are a null string terminated
 *   list of arbitrary length (currently only rli_recv uses this type
 *   of arg list).  This function reads and discards the null string
 *   terminated arg list.
 */
static void
rrpc_flushio(CONNECTION *c)

{
  globus_bool_t	sawnull;
  int		ch;
  int		rc;
  char		errmsg[MAXERRMSG];

  sawnull = GLOBUS_FALSE;
  while ((ch = NEXTC(&c->handle, &c->ibuf, &rc, errmsg)) != -1)
    if (ch == '\0') {
      if (sawnull)
	return;
      sawnull = GLOBUS_TRUE;
    } else
      sawnull = GLOBUS_FALSE;
}

static int
rrpc_readstr(CONNECTION *c, char *rbuf, int rbuflen)

{
  int	i;
  int	rc;
  int	ch;

  i = 0;
  while (i < rbuflen && (ch = NEXTC(&c->handle, &c->ibuf, &rc, c->errmsg)) != -1){
    rbuf[i++] = ch;
    if (!ch)
      return GLOBUS_RLS_SUCCESS;
  }
  rrpc_flushio(c);
  if (ch == -1)
    return rc;
  return GLOBUS_RLS_OVERFLOW;
}

static void
lrc_add(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;

  if (!arglist[0] || !arglist[1]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if ((rc = db_lrc_add(dbh->lrc, arglist[0], arglist[1],
		       c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  rrpc_success(c);
}

static void
lrc_add_bulk(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	lfn[BUFLEN];
  char	pfn[BUFLEN];

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_add_bulk:");
  while (1) {
    if ((rc = rrpc_readstr(c, lfn, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if (!*lfn)
      break;
    if ((rc = rrpc_readstr(c, pfn, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if ((rc = db_lrc_add(dbh->lrc,lfn,pfn,c->errmsg)) != GLOBUS_RLS_SUCCESS)
      listresult(c, lfn, pfn, NULL, NULL, rc);
  }
  endstrcb(c, 0);
  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_add_bulk: end");
}

static void
lrc_attr_add(CONNECTION *c, DBH *dbh, char **arglist)

{
  int				rc;
  globus_rls_attr_type_t	atype;
  char				*val;

  if (!arglist[0] || !arglist[1] || !arglist[2] || !arglist[3]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }

  atype = atoi(arglist[2]);
  if (atype == globus_rls_attr_type_str && !arglist[4])
    val = "";
  else
    val = arglist[4];
  if ((rc = db_attr_add(dbh->lrc, arglist[0], atoi(arglist[1]), atype,
			arglist[3], val, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  rrpc_success(c);
}

static void
lrc_attr_add_bulk(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	key[BUFLEN];
  char	obuf[BUFLEN];
  char	tbuf[BUFLEN];
  char	name[BUFLEN];
  char	vbuf[BUFLEN];

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_attr_add_bulk:");
  while (1) {
    if ((rc = rrpc_readstr(c, key, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if (!*key)
      break;
    rc = rrpc_readstr(c, obuf, BUFLEN);
    if (rc == GLOBUS_RLS_SUCCESS)
      rc = rrpc_readstr(c, tbuf, BUFLEN);
    if (rc == GLOBUS_RLS_SUCCESS)
      rc = rrpc_readstr(c, name, BUFLEN);
    if (rc == GLOBUS_RLS_SUCCESS)
      rc = rrpc_readstr(c, vbuf, BUFLEN);
    if (rc != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if ((rc = db_attr_add(dbh->lrc, key, atoi(obuf), atoi(tbuf), name, vbuf,
			  c->errmsg)) != GLOBUS_RLS_SUCCESS)
      listresult(c, key, name, NULL, NULL, rc);
  }
  endstrcb(c, 0);
  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_attr_add_bulk: end");
}

static void
lrc_attr_create(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;

  if (!arglist[0] || !arglist[1] || !arglist[2]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }

  if ((rc = db_attr_create(dbh->lrc, arglist[0], atoi(arglist[1]),
		atoi(arglist[2]), c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }

  rrpc_success(c);
}

static void
lrc_attr_delete(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;

  if (!arglist[0] || !arglist[1] || !arglist[2]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }

  if ((rc = db_attr_delete(dbh->lrc, arglist[0], atoi(arglist[1]),
		atoi(arglist[2]), c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }

  rrpc_success(c);
}

static void
lrc_attr_get(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;

  if (!arglist[1]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }

  if ((rc = db_attr_get(dbh->lrc, arglist[0], atoi(arglist[1]),
			str2cb, c, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  if (c->numrecs == 0)
    rrpc_error(c, GLOBUS_RLS_ATTR_NEXIST, "%s", arglist[1]);
  else
    endstrcb(c, 0);
}

static void
lrc_attr_modify(CONNECTION *c, DBH *dbh, char **arglist)

{
  int				rc;
  globus_rls_attr_type_t	atype;
  char				*val;

  if (!arglist[0] || !arglist[1] || !arglist[2]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }

  atype = atoi(arglist[3]);
  if (atype == globus_rls_attr_type_str && !arglist[3])
    val = "";
  else
    val = arglist[4];
  if ((rc = db_attr_modify(dbh->lrc, arglist[0], arglist[1], atoi(arglist[2]),
			   atype, val, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  rrpc_success(c);
}

static void
lrc_attr_remove(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;

  if (!arglist[0] || !arglist[1] || !arglist[2]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }

  if ((rc = db_attr_remove(dbh->lrc, arglist[0], arglist[1], atoi(arglist[2]),
			   c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  rrpc_success(c);
}

static void
lrc_attr_remove_bulk(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	key[BUFLEN];
  char	obuf[BUFLEN];
  char	name[BUFLEN];

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_attr_remove_bulk:");
  while (1) {
    if ((rc = rrpc_readstr(c, key, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if (!*key)
      break;
    rc = rrpc_readstr(c, obuf, BUFLEN);
    if (rc == GLOBUS_RLS_SUCCESS)
      rc = rrpc_readstr(c, name, BUFLEN);
    if (rc != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if ((rc = db_attr_remove(dbh->lrc, key, name, atoi(obuf),
			     c->errmsg)) != GLOBUS_RLS_SUCCESS)
      listresult(c, key, name, NULL, NULL, rc);
  }
  endstrcb(c, 0);
  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_attr_remove_bulk: end");
}

static void
lrc_attr_search(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  int	op;
  int	offset;
  int	reslimit;

  if (!arglist[0] || !arglist[1] || !arglist[2]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  op = atoi(arglist[2]);
  if ((op != globus_rls_attr_op_all && !arglist[3]) ||
      (op == globus_rls_attr_op_btw && !arglist[4])) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  offset = arglist[5] ? atoi(arglist[5]) : 0;
  reslimit = getreslimit(arglist[6]);
  if ((rc = db_attr_search(dbh->lrc, arglist[0], atoi(arglist[1]),
			   op, arglist[3], arglist[4], offset, reslimit,
			   str3cb, c, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  if (c->numrecs == 0 && offset == 0)
    rrpc_error(c, GLOBUS_RLS_ATTR_VALUE_NEXIST, "%s", arglist[0]);
  else
    endstrcb(c, reslimit && c->numrecs == reslimit);
}

static void
lrc_attr_value_get(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;

  if (!arglist[0] || !arglist[2]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }

  if ((rc = db_attr_value_get(dbh->lrc, arglist[0], arglist[1], atoi(arglist[2]),
			      str3cb, c, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  if (c->numrecs == 0)
    rrpc_error(c, GLOBUS_RLS_ATTR_NEXIST, "%s", arglist[0]);
  else
    endstrcb(c, 0);
}

static void
lrc_attr_value_get_bulk(CONNECTION *c, DBH *dbh, char **arglist)

{
  int			rc;
  globus_rls_obj_type_t	objtype;
  char			key[BUFLEN];
  int			onumrecs;

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_attr_value_get_bulk:");
  objtype = atoi(arglist[1]);
  while (1) {
    if ((rc = rrpc_readstr(c, key, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if (!*key)
      break;
    onumrecs = c->numrecs;
    if ((rc = db_attr_value_get_bulk(dbh->lrc, key, arglist[0], objtype,
			str4bulkcb, c, c->errmsg)) != GLOBUS_RLS_SUCCESS)
      listresult(c, key, arglist[0] ? arglist[0] : "", NULL, NULL, rc);
    else if (onumrecs == c->numrecs)
      listresult(c, key, arglist[0] ? arglist[0] : "", NULL, NULL,
		 GLOBUS_RLS_ATTR_NEXIST);
  }
  if (c->numrecs == 0)
    rrpc_error(c, GLOBUS_RLS_ATTR_NEXIST, "", NULL);
  else
    endstrcb(c, 0);
  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_attr_value_get_bulk: end");
}

static void
lrc_clear(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;

  if ((rc = db_lrc_clear(dbh->lrc, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  if (bloomfilter.bfsize) {
    bf_free(&bloomfilter);
    bloomfilter.startlfns = table[T_LRCLFN].count;
    lrc_getbf(dbh->lrc);
  }
  
  rrpc_success(c);
}

static void
lrc_create(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;

  if (!arglist[0] || !arglist[1]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if ((rc = db_lrc_create(dbh->lrc, arglist[0], arglist[1],
			  c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  } else if (rc == GLOBUS_RLS_SUCCESS) {
      if (bloomfilter.bfsize) {
        globus_mutex_lock(&bloomfilter.mtx);
        bf_addlfn(&bloomfilter, arglist[0]);
        globus_mutex_unlock(&bloomfilter.mtx);
      }
  }
  rrpc_success(c);

  if (update_immediate)
    doimmediate(rliop_add, lrc_rlilist, &lrc_rlilistlock, arglist[0], NULL);
  /*
   * If we're using bloom filters to update, and the number of lfns has
   * doubled in the db, then resize the bloom filter.
   */
  if (bloomfilter.bfsize &&
      (table[T_LRCLFN].count >= bloomfilter.startlfns * 2) &&
      (table[T_LRCLFN].count * lrc_bloomfilter_ratio > bloomfilter.bfsize)) {
    if (loglevel)
      logit(LOG_DEBUG, "lrc_create: Resizing bloom filter, current %d new %d",
	    bloomfilter.bfsize, table[T_LRCLFN].count * lrc_bloomfilter_ratio);
    bf_free(&bloomfilter);
    bloomfilter.startlfns = table[T_LRCLFN].count;
    lrc_getbf(dbh->lrc);
  }
}

static void
lrc_create_bulk(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	lfn[BUFLEN];
  char	pfn[BUFLEN];

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_create_bulk:");
  while (1) {
    if ((rc = rrpc_readstr(c, lfn, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if (!*lfn)
      break;
    if ((rc = rrpc_readstr(c, pfn, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if ((rc = db_lrc_create(dbh->lrc,lfn,pfn,c->errmsg)) != GLOBUS_RLS_SUCCESS)
      listresult(c, lfn, pfn, NULL, NULL, rc);
    else if (update_immediate)
      doimmediate(rliop_add, lrc_rlilist, &lrc_rlilistlock, lfn, NULL);
  }
  endstrcb(c, 0);
  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_create_bulk: end");
}

static void
lrc_delete(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  int	lfndeleted;

  if (!arglist[0] || !arglist[1]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if ((rc = db_lrc_delete(dbh->lrc, arglist[0], arglist[1],
			  &lfndeleted, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  } else if (lfndeleted && rc == GLOBUS_RLS_SUCCESS) {
      if (bloomfilter.bfsize) {
        globus_mutex_lock(&bloomfilter.mtx);
        bf_deletelfn(&bloomfilter, arglist[0]);
        globus_mutex_unlock(&bloomfilter.mtx);
      }
  }
  rrpc_success(c);

  if (update_immediate && lfndeleted)
    doimmediate(rliop_delete, lrc_rlilist, &lrc_rlilistlock, arglist[0], NULL);
}

static void
lrc_delete_bulk(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	lfn[BUFLEN];
  char	pfn[BUFLEN];
  int	lfndeleted;

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_delete_bulk:");
  while (1) {
    if ((rc = rrpc_readstr(c, lfn, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if (!*lfn)
      break;
    if ((rc = rrpc_readstr(c, pfn, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if ((rc = db_lrc_delete(dbh->lrc, lfn, pfn, &lfndeleted,
			    c->errmsg)) != GLOBUS_RLS_SUCCESS)
      listresult(c, lfn, pfn, NULL, NULL, rc);
    else if (update_immediate && lfndeleted)
      doimmediate(rliop_delete, lrc_rlilist, &lrc_rlilistlock, lfn, NULL);
  }
  endstrcb(c, 0);
  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_delete_bulk: end");
}

static void
lrc_exists(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;

  if (!arglist[0] || !arglist[1]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if ((rc = db_exists(dbh->lrc, arglist[0], atoi(arglist[1]),
		      c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  rrpc_success(c);
}

static void
lrc_exists_bulk(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	lfn[BUFLEN];
  char  objtype[BUFLEN];
  int	onumrecs;

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_exists_bulk:");

  if ((rc = rrpc_readstr(c, objtype, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }

  while (1) {
    if ((rc = rrpc_readstr(c, lfn, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if (!*lfn)
      break;
    onumrecs = c->numrecs;
    if ((rc = db_exists(dbh->lrc, lfn, atoi(objtype),
		      c->errmsg)) != GLOBUS_RLS_SUCCESS)
      listresult(c, lfn, NULL, NULL, NULL, rc);
    else if (rc == GLOBUS_RLS_SUCCESS)
      str2bulkcb(c, lfn, "");
    else if (onumrecs == c->numrecs)
      listresult(c, lfn, NULL, NULL, NULL, GLOBUS_RLS_LFN_NEXIST);
  }
  if (c->numrecs == 0)
    rrpc_error(c, GLOBUS_RLS_LFN_NEXIST, "", NULL);
  else
    endstrcb(c, 0);
  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_exists_bulk: end");
}

static void
lrc_get_lfn(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  int	offset;
  int	reslimit;

  if (!arglist[0]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  offset = arglist[1] ? atoi(arglist[1]) : 0;
  reslimit = getreslimit(arglist[2]);
  if ((rc = db_lrc_getlfn(dbh->lrc, arglist[0], GLOBUS_FALSE, offset, reslimit,
		str2cb, c, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  if (c->numrecs == 0 && offset == 0)
    rrpc_error(c, GLOBUS_RLS_PFN_NEXIST, "%s", arglist[0]);
  else
    endstrcb(c, reslimit && c->numrecs == reslimit);
}

static void
lrc_get_lfn_bulk(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	pfn[BUFLEN];
  int	onumrecs;

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_get_lfn_bulk:");
  while (1) {
    if ((rc = rrpc_readstr(c, pfn, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if (!*pfn)
      break;
    onumrecs = c->numrecs;
    if ((rc = db_lrc_getlfn(dbh->lrc, pfn, GLOBUS_FALSE, 0, 0, str2bulkcb,
			    c, c->errmsg)) != GLOBUS_RLS_SUCCESS)
      listresult(c, pfn, NULL, NULL, NULL, rc);
    else if (onumrecs == c->numrecs)
      listresult(c, pfn, NULL, NULL, NULL, GLOBUS_RLS_PFN_NEXIST);
  }
  if (c->numrecs == 0)
    rrpc_error(c, GLOBUS_RLS_PFN_NEXIST, "", NULL);
  else
    endstrcb(c, 0);
  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_get_lfn_bulk: end");
}

static void
lrc_get_lfn_wc(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  int	offset;
  int	reslimit;

  if (!arglist[0]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  offset = arglist[1] ? atoi(arglist[1]) : 0;
  reslimit = getreslimit(arglist[2]);
  if ((rc = db_lrc_getlfn(dbh->lrc, arglist[0], GLOBUS_TRUE, offset, reslimit,
		str2cb, c, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  if (c->numrecs == 0 && offset == 0)
    rrpc_error(c, GLOBUS_RLS_PFN_NEXIST, "%s", arglist[0]);
  else
    endstrcb(c, reslimit && c->numrecs == reslimit);
}

static void
lrc_get_pfn(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  int	offset;
  int	reslimit;

  if (!arglist[0]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  offset = arglist[1] ? atoi(arglist[1]) : 0;
  reslimit = getreslimit(arglist[2]);
  if ((rc = db_lrc_getpfn(dbh->lrc, arglist[0], GLOBUS_FALSE, offset, reslimit,
			  str2cb, c, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  if (c->numrecs == 0 && offset == 0)
    rrpc_error(c, GLOBUS_RLS_LFN_NEXIST, "%s", arglist[0]);
  else
    endstrcb(c, reslimit && c->numrecs == reslimit);
}

static void
lrc_get_pfn_bulk(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	lfn[BUFLEN];
  int	onumrecs;

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_get_pfn_bulk:");
  while (1) {
    if ((rc = rrpc_readstr(c, lfn, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if (!*lfn)
      break;
    onumrecs = c->numrecs;
    if ((rc = db_lrc_getpfn(dbh->lrc, lfn, GLOBUS_FALSE, 0, 0, str2bulkcb,
			    c, c->errmsg)) != GLOBUS_RLS_SUCCESS)
      listresult(c, lfn, NULL, NULL, NULL, rc);
    else if (onumrecs == c->numrecs)
      listresult(c, lfn, NULL, NULL, NULL, GLOBUS_RLS_LFN_NEXIST);
  }
  if (c->numrecs == 0)
    rrpc_error(c, GLOBUS_RLS_LFN_NEXIST, "", NULL);
  else
    endstrcb(c, 0);
  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_get_pfn_bulk: end");
}

static void
lrc_get_pfn_wc(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  int	offset;
  int	reslimit;

  if (!arglist[0]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  offset = arglist[1] ? atoi(arglist[1]) : 0;
  reslimit = getreslimit(arglist[2]);
  if ((rc = db_lrc_getpfn(dbh->lrc, arglist[0], GLOBUS_TRUE, offset, reslimit,
			  str2cb, c, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  if (c->numrecs == 0 && offset == 0)
    rrpc_error(c, GLOBUS_RLS_LFN_NEXIST, "%s", arglist[0]);
  else
    endstrcb(c, reslimit && c->numrecs == reslimit);
}

static void
lrc_mapping_exists(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;

  if (!arglist[0] || !arglist[1]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if ((rc = db_lrc_mapping_exists(dbh->lrc, arglist[0], arglist[1],
				  c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  rrpc_success(c);
}

static void
lrc_renamelfn(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;

  if (!arglist[0] || !arglist[1]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if ((rc = db_lrc_renamelfn(dbh->lrc, arglist[0], arglist[1],
		       c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  } else if (rc == GLOBUS_RLS_SUCCESS) {
      if (bloomfilter.bfsize) {
        globus_mutex_lock(&bloomfilter.mtx);
        bf_deletelfn(&bloomfilter, arglist[0]);
        bf_addlfn(&bloomfilter, arglist[1]);
        globus_mutex_unlock(&bloomfilter.mtx);
      }
  }
  rrpc_success(c);

  if (update_immediate) {
    doimmediate(rliop_delete, lrc_rlilist, &lrc_rlilistlock, arglist[0], NULL);
    doimmediate(rliop_add, lrc_rlilist, &lrc_rlilistlock, arglist[1], NULL);
  }
}

static void
lrc_renamelfn_bulk(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	oldname[BUFLEN];
  char	newname[BUFLEN];

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_renamelfn_bulk:");
  while (1) {
    if ((rc = rrpc_readstr(c, oldname, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if (!*oldname)
      break;
    if ((rc = rrpc_readstr(c, newname, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if ((rc = db_lrc_renamelfn(dbh->lrc,oldname,newname,c->errmsg)) != GLOBUS_RLS_SUCCESS)
      listresult(c, oldname, newname, NULL, NULL, rc);
    else if (update_immediate) {
      doimmediate(rliop_delete, lrc_rlilist, &lrc_rlilistlock, oldname, NULL);
      doimmediate(rliop_add, lrc_rlilist, &lrc_rlilistlock, newname, NULL);
    }
  }
  endstrcb(c, 0);
  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_renamelfn_bulk: end");
}

static void
lrc_renamepfn(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;

  if (!arglist[0] || !arglist[1]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if ((rc = db_lrc_renamepfn(dbh->lrc, arglist[0], arglist[1],
		       c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  rrpc_success(c);
}

static void
lrc_renamepfn_bulk(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	oldname[BUFLEN];
  char	newname[BUFLEN];

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_renamepfn_bulk:");
  while (1) {
    if ((rc = rrpc_readstr(c, oldname, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if (!*oldname)
      break;
    if ((rc = rrpc_readstr(c, newname, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if ((rc = db_lrc_renamepfn(dbh->lrc,oldname,newname,c->errmsg)) != GLOBUS_RLS_SUCCESS)
      listresult(c, oldname, newname, NULL, NULL, rc);
  }
  endstrcb(c, 0);
  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_renamepfn_bulk: end");
}

static void
lrc_rli_add(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	buf[BUFLEN];
  int	flags;

  if (!arglist[0] || !arglist[1]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  flags = atoi(arglist[1]);
  if (arglist[2] && (flags & FRLI_BLOOMFILTER)) {
    rrpc_error(c, GLOBUS_RLS_UNSUPPORTED, "Paritions not supported with Bloom filters");
    return;
  }
  if ((rc = db_update_add(dbh->lrc, canonurl(arglist[0], buf, BUFLEN), flags,
			  arglist[2], c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  rrpc_success(c);
  update_readrli(dbh->lrc, &lrc_rlilistlock, &lrc_rlilist);
  if (!bloomfilter.bfsize && (flags & FRLI_BLOOMFILTER)) {
    bloomfilter.startlfns = table[T_LRCLFN].count;
    lrc_getbf(dbh->lrc);
  }
}

static void
lrc_rli_delete(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	buf[BUFLEN];
  int	bf;
  RLI	*rli;

  if (!arglist[0]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if ((rc = db_update_delete(dbh->lrc, canonurl(arglist[0], buf, BUFLEN),
			     arglist[1], c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  rrpc_success(c);

  /*
   * Reread rli table, check if one is updated by bloomfilters, if so
   * create bloom filter if we don't already have it, else discard
   * existing bloomfilter if we just deleted the last rli that uses it.
   */
  update_readrli(dbh->lrc, &lrc_rlilistlock, &lrc_rlilist);
  bf = 0;
  rls_lock_get(&lrc_rlilistlock, readlock);
  for (rli = lrc_rlilist; rli; rli = rli->nxt)
    if (rli->flags & FR_BLOOMFILTER) {
      bf = 1;
      break;
    }
  rls_lock_release(&lrc_rlilistlock, readlock);
  if (bf) {
    if (!bloomfilter.bfsize) {
      bloomfilter.startlfns = table[T_LRCLFN].count;
      lrc_getbf(dbh->lrc);
    }
  } else if (bloomfilter.bfsize)
    bf_free(&bloomfilter);
}

static void
lrc_rli_get_part(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	buf[BUFLEN];
  char	*url;

  url = arglist[0] ? canonurl(arglist[0], buf, BUFLEN) : NULL;
  if ((rc = db_update_get_part(dbh->lrc, url, arglist[1], rlipartcb, c,
			       c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  if (c->numrecs == 0)
    rrpc_error(c, GLOBUS_RLS_RLI_NEXIST, "%s,%s",
	       PS(arglist[0]), PS(arglist[1]));
  else
    endstrcb(c, 0);
}

static void
lrc_rli_info(CONNECTION *c, DBH *dbh, char **arglist)

{
  RLI		*rli;
  char		buf[BUFLEN];
  char		*url;

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_rli_info: %s", PS(arglist[0]));
  url = canonurl(arglist[0], buf, BUFLEN);
  rls_lock_get(&lrc_rlilistlock, readlock);
  for (rli = lrc_rlilist; rli; rli = rli->nxt)
    if (strcasecmp(url, rli->url) == 0)
      break;
  rls_lock_release(&lrc_rlilistlock, readlock);
  if (!rli) {
    rrpc_error(c, GLOBUS_RLS_RLI_NEXIST, "%s", url);
    return;
  }
  rrpc_rliinfo(c, rli, 1, 1, 0);
}

static void
lrc_rli_list(CONNECTION *c, DBH *dbh, char **arglist)

{
  RLI	*rli;
  int	first;

  if (loglevel > 1)
    logit(LOG_DEBUG, "lrc_rli_list:");

  if (!lrc_rlilist) {
    rrpc_error(c, GLOBUS_RLS_RLI_NEXIST, "LRC not updating any RLI servers");
    return;
  }

  first = 1;
  rls_lock_get(&lrc_rlilistlock, readlock);
  for (rli = lrc_rlilist; rli; rli = rli->nxt) {
    if (rrpc_rliinfo(c,rli,first,rli->nxt == NULL,1) != GLOBUS_RLS_SUCCESS) {
      rls_lock_release(&lrc_rlilistlock, readlock);
      return;
    }
    first = 0;
  }
  rls_lock_release(&lrc_rlilistlock, readlock);
}

static void
rli_exists(CONNECTION *c, DBH *dbh, char **arglist)

{
  int			rc;
  SENDER		*sender;
  globus_rls_obj_type_t	objtype;
  int			nferr;

  if (!arglist[0] || !arglist[1]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if ((objtype = atoi(arglist[1])) == globus_rls_obj_rli_lrc)
    nferr = GLOBUS_RLS_LRC_NEXIST;
  else
    nferr = GLOBUS_RLS_LFN_NEXIST;
  if (rli_bloomfilter) {
    rls_lock_get(&senderlistlock, readlock);
    for (sender = senderlist; sender; sender = sender->nxt)
      if (objtype == globus_rls_obj_rli_lrc) {
	if (strcasecmp(arglist[0], sender->lrcurl) == 0)
	  break;
      } else {
	if (bf_testlfn(&sender->bf, arglist[0]))
	  break;
      }
    rls_lock_release(&senderlistlock, readlock);
    if (!sender)
      rrpc_error(c, nferr, "%s", arglist[0]);
  } else {
    if ((rc = db_exists(dbh->rli, arglist[0], objtype,
			c->errmsg)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
  }
  rrpc_success(c);
}

static void
rli_exists_bulk(CONNECTION *c, DBH *dbh, char **arglist)

{
  int			rc;
  SENDER		*sender;
  globus_rls_obj_type_t	objtype;
  char			b[BUFLEN];
  char			val[BUFLEN];
  int			nferr;
  int			exists;
  
  if (loglevel > 1)
    logit(LOG_DEBUG, "rli_exists_bulk:");

  /* read objtype from input stream */
  if ((rc = rrpc_readstr(c, b, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  } else if (!b || !*b) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }

  /* convert objtype from char* to int value */
  if ((objtype = atoi(b)) == globus_rls_obj_rli_lrc)
    nferr = GLOBUS_RLS_LRC_NEXIST;
  else
    nferr = GLOBUS_RLS_LFN_NEXIST;

  while (1) {

    /* read next value from input stream */
    if ((rc = rrpc_readstr(c, val, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if (!*val)
      break;

    /* reset flag */
    exists = 0;

    /* check existence of value */
    if (rli_bloomfilter) {
      if (loglevel > 1)
	logit(LOG_DEBUG, "rli_exists_bulk: %s %d", val, objtype);
      rls_lock_get(&senderlistlock, readlock);
      for (sender = senderlist; sender; sender = sender->nxt)
        if (objtype == globus_rls_obj_rli_lrc) {
          if (strcasecmp(val, sender->lrcurl) == 0) {
	    exists = 1;
            break;
	  }
        } else {
          if (bf_testlfn(&sender->bf, val)) {
	    exists = 1;
            break;
	  }
        }
      rls_lock_release(&senderlistlock, readlock);
    } else {
      if ((rc = db_exists(dbh->rli, val, objtype,
			  c->errmsg)) == GLOBUS_RLS_SUCCESS) {
        exists = 1;
      }
    }
    
    /* add results to list */
    if (exists)
      str2bulkcb(c, val, "");
    else
      listresult(c, val, NULL, NULL, NULL, nferr);

  } /* endwhile */
  
  if (c->numrecs == 0)
    rrpc_error(c, nferr, "", NULL);
  else
    endstrcb(c, 0);
  if (loglevel > 1)
    logit(LOG_DEBUG, "rli_exists_bulk: end");
}

static void
rli_get_lrc(CONNECTION *c, DBH *dbh, char **arglist)

{
  int		rc;
  SENDER	*sender;
  int		offset;
  int		reslimit;
  int		i;
  int		n;

  if (!arglist[0]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  offset = arglist[1] ? atoi(arglist[1]) : 0;
  reslimit = getreslimit(arglist[2]);

  if (loglevel > 1)
    logit(LOG_DEBUG, "rli_get_lrc: %s %d %d", arglist[0], offset, reslimit);

  if (rli_bloomfilter) {
    rls_lock_get(&senderlistlock, readlock);
    i = 0;
    n = 0;
    for (sender = senderlist; sender; sender = sender->nxt)
      if (bf_testlfn(&sender->bf, arglist[0])) {
	if (i++ < offset)
	  continue;
	str2cb(c, arglist[0], sender->lrcurl);
        if (reslimit && ++n >= reslimit)
	  break;
      }
    rls_lock_release(&senderlistlock, readlock);
  } else {
    if ((rc = db_rli_getlrc(dbh->rli, arglist[0], GLOBUS_FALSE, offset,
		reslimit, str2cb, c, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
  }
  if (c->numrecs == 0 && offset == 0)
    rrpc_error(c, GLOBUS_RLS_LFN_NEXIST, "%s", arglist[0]);
  else
    endstrcb(c, reslimit && c->numrecs == reslimit);
}

static void
rli_get_lrc_bulk(CONNECTION *c, DBH *dbh, char **arglist)

{
  int		rc;
  char		lfn[BUFLEN];
  int		onumrecs;
  SENDER	*sender;

  if (loglevel > 1)
    logit(LOG_DEBUG, "rli_get_lrc_bulk:");
  while (1) {
    if ((rc = rrpc_readstr(c, lfn, BUFLEN)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
    if (!*lfn)
      break;
    if (rli_bloomfilter) {
      if (loglevel > 1)
        logit(LOG_DEBUG, "rli_get_lrc_bulk: %s", lfn); 
      onumrecs = c->numrecs;
      rls_lock_get(&senderlistlock, readlock);
      for (sender = senderlist; sender; sender = sender->nxt)
	if (bf_testlfn(&sender->bf, lfn))
	  str2bulkcb(c, lfn, sender->lrcurl);
      rls_lock_release(&senderlistlock, readlock);

      if (onumrecs == c->numrecs)
	  listresult(c, lfn, NULL, NULL, NULL, GLOBUS_RLS_LFN_NEXIST);
    } else {
      onumrecs = c->numrecs;
      if ((rc = db_rli_getlrc(dbh->rli, lfn, GLOBUS_FALSE, 0, 0, str2bulkcb,
			      c, c->errmsg)) != GLOBUS_RLS_SUCCESS)
	listresult(c, lfn, NULL, NULL, NULL, rc);
      else if (onumrecs == c->numrecs)
	listresult(c, lfn, NULL, NULL, NULL, GLOBUS_RLS_LFN_NEXIST);
    }
  }
  if (c->numrecs == 0)
    rrpc_error(c, GLOBUS_RLS_LFN_NEXIST, "", NULL);
  else
    endstrcb(c, 0);
  if (loglevel > 1)
    logit(LOG_DEBUG, "rli_get_lrc_bulk: end");
}

static void
rli_get_lrc_wc(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  int	offset;
  int	reslimit;

  if (!arglist[0]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if (rli_bloomfilter) {
    rrpc_error(c, GLOBUS_RLS_UNSUPPORTED, "Wildcard searches with Bloom filters enabled");
    return;
  }
  offset = arglist[1] ? atoi(arglist[1]) : 0;
  reslimit = getreslimit(arglist[2]);
  if ((rc = db_rli_getlrc(dbh->rli, arglist[0], GLOBUS_TRUE, offset,
		reslimit, str2cb, c, c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  if (c->numrecs == 0 && offset == 0)
    rrpc_error(c, GLOBUS_RLS_LFN_NEXIST, "%s", arglist[0]);
  else
    endstrcb(c, reslimit && c->numrecs == reslimit);
}

static void
rli_sender_list(CONNECTION *c, DBH *dbh, char **arglist)

{
  SENDER	*sender;
  SENDER	*p;
  int		rc;
  char		errbuf[MAXERRMSG];

  if (loglevel)
    logit(LOG_DEBUG, "rli_sender_list:");

  if (!senderlist) {
    rrpc_error(c, GLOBUS_RLS_LRC_NEXIST, "Not updated by any servers");
    return;
  }
  rrpc_initbuf(&c->obuf);
  if ((rc = rrpc_bufwrite(&c->handle, &c->obuf, "0", 2, 0,
			  c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    logit(LOG_WARNING, "rli_sender_list: rpc_bufwrite: %s",
	  globus_rls_errmsg(rc, c->errmsg, errbuf, MAXERRMSG));
    return;
  }
  rls_lock_get(&senderlistlock, readlock);
  for (sender = senderlist; sender; sender = sender->nxt) {
    /*
     * If an rli updates another rli it will send a bloomfilter for each
     * lrc that updates it, which may result in dups in the receiver rlis
     * senderlist, following loop eliminates reporting dups.
     */
    for (p = senderlist; p != sender; p = p->nxt)
      if (strcasecmp(p->senderurl, sender->senderurl) == 0)
	break;
    if (p != sender) {
      if (sender->nxt)
	continue;
      rc = rrpc_bufprintf(&c->handle, &c->obuf, 1, c->errmsg, "%c", 0);
      goto next;
    }
    if (sender->nxt)
      rc = rrpc_bufprintf(&c->handle, &c->obuf, 0, c->errmsg, "%s%c%u%c",
			  sender->senderurl, 0, sender->lastupdate, 0);
    else
      rc = rrpc_bufprintf(&c->handle, &c->obuf, 1, c->errmsg, "%s%c%u%c%c",
			  sender->senderurl, 0, sender->lastupdate, 0, 0);
   next:
    if (rc != GLOBUS_RLS_SUCCESS) {
      logit(LOG_WARNING, "rli_sender_list: rpc_bufprintf: %s",
	    globus_rls_errmsg(rc, c->errmsg, errbuf, MAXERRMSG));
      break;
    }
  }
  rls_lock_release(&senderlistlock, readlock);
}

static void
rli_mapping_exists(CONNECTION *c, DBH *dbh, char **arglist)

{
  int		rc;
  SENDER	*sender;

  if (!arglist[0] || !arglist[1]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if (rli_bloomfilter) {
    rls_lock_get(&senderlistlock, readlock);
    for (sender = senderlist; sender; sender = sender->nxt)
      if (strcasecmp(arglist[1], sender->lrcurl) == 0 &&
	  bf_testlfn(&sender->bf, arglist[0]))
	break;
    rls_lock_release(&senderlistlock, readlock);
    if (!sender)
      rrpc_error(c,GLOBUS_RLS_MAPPING_NEXIST, "%s,%s", arglist[0], arglist[1]);
  } else {
    if ((rc = db_rli_mapping_exists(dbh->rli, arglist[0], arglist[1],
				    c->errmsg)) != GLOBUS_RLS_SUCCESS) {
      rrpc_error(c, rc, "%s", c->errmsg);
      return;
    }
  }
  rrpc_success(c);
}

static void
rli_rli_add(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	buf[BUFLEN];

  if (!arglist[0]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if (arglist[1] && rli_bloomfilter) {
    rrpc_error(c, GLOBUS_RLS_UNSUPPORTED, "Paritions not supported with Bloom filters");
    return;
  }
  if ((rc = db_update_add(dbh->rli, canonurl(arglist[0],buf,BUFLEN), FRLI_RLI,
		          arglist[1], c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  rrpc_success(c);
  update_readrli(dbh->rli, &rli_rlilistlock, &rli_rlilist);
}

static void
rli_rli_delete(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	buf[BUFLEN];

  if (!arglist[0]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if ((rc = db_update_delete(dbh->rli, canonurl(arglist[0], buf, BUFLEN),
			     arglist[1], c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  rrpc_success(c);
  update_readrli(dbh->rli, &rli_rlilistlock, &rli_rlilist);
}

static void
rli_rli_get_part(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;
  char	buf[BUFLEN];
  char	*url;

  url = arglist[0] ? canonurl(arglist[0], buf, BUFLEN) : NULL;
  if ((rc = db_update_get_part(dbh->rli, url, arglist[1], rlipartcb, c,
			       c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", c->errmsg);
    return;
  }
  if (c->numrecs == 0)
    rrpc_error(c, GLOBUS_RLS_RLI_NEXIST, "%s,%s",
	       PS(arglist[0]), PS(arglist[1]));
  else
    endstrcb(c, 0);
}

static void
rli_rli_list(CONNECTION *c, DBH *dbh, char **arglist)

{
  RLI	*rli;
  int	first;

  if (loglevel > 1)
    logit(LOG_DEBUG, "rli_rli_list:");

  if (!rli_rlilist) {
    rrpc_error(c, GLOBUS_RLS_RLI_NEXIST, "RLI not updating any RLI servers");
    return;
  }

  first = 1;
  rls_lock_get(&rli_rlilistlock, readlock);
  for (rli = rli_rlilist; rli; rli = rli->nxt) {
    if (rrpc_rliinfo(c,rli,first,rli->nxt == NULL,1) != GLOBUS_RLS_SUCCESS) {
      rls_lock_release(&rli_rlilistlock, readlock);
      return;
    }
    first = 0;
  }
  rls_lock_release(&rli_rlilistlock, readlock);
}

static void
rli_recv1(CONNECTION *c, DBH *dbh, char **arglist)

{
  rli_recv(c, dbh, NULL, arglist[0]);
}

static void
rli_recv2(CONNECTION *c, DBH *dbh, char **arglist)

{
  rli_recv(c, dbh, arglist[0], arglist[1]);
}

/*
 * Server side rli update function, by sending list of all lfns to be
 * added/deleted.  The first character of the lfn argument should be
 * either RLI_ADD or RLI_DELETE to indicate which.
 */
static void
rli_recv(CONNECTION *c, DBH *dbh, char *lrcurl, char *senderurl)

{
  char			buf[BUFLEN+1];	/* +1 for op char		*/
  int			ch;
  int			i;
  int			rc;
  int			saverc;
  char			errbuf[MAXERRMSG];
  char			*surl;
  char			surlbuf[BUFLEN];
  char			*lurl;
  char			lurlbuf[BUFLEN];
  SENDER		*sender;
  char			ts[BUFLEN];
  int			lfnchanged;

  if (!senderurl) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if (rli_bloomfilter) {
    rrpc_error(c, GLOBUS_RLS_UNSUPPORTED, "LFNList update with Bloom filters");
    return;
  }
  surl = canonurl(senderurl, surlbuf, BUFLEN);
  lurl = lrcurl ? canonurl(lrcurl, lurlbuf, BUFLEN) : surl;

  if ((sender = rli_findsender(surl, lurl)) == NULL) {
    rrpc_error(c, GLOBUS_RLS_NOMEMORY, "");
    return;
  }
  mycftime(ts, BUFLEN, "%Y-%m-%d %H:%M:%S", time(0));

  saverc = GLOBUS_RLS_SUCCESS;
  while (1) {
    i = 0;
    while ((ch = NEXTC(&c->handle, &c->ibuf, &rc, c->errmsg)) != -1)
      if (!(buf[i++] = ch))
	break;
    if (ch == -1) {
      logit(LOG_WARNING, "rli_recv: %s",
	    globus_rls_errmsg(rc, c->errmsg, errbuf, MAXERRMSG));
      return;
    }
    if (i == 1)
      break;
    if (*buf == RLI_ADD) {
      rc = db_rli_update(dbh->rli, &buf[1], lurl, surl, ts, &lfnchanged,
			 c->errmsg);
      if (rc == GLOBUS_RLS_SUCCESS && update_immediate && lfnchanged)
	doimmediate(rliop_add, rli_rlilist, &rli_rlilistlock, &buf[1], lurl);
    } else if (*buf == RLI_DELETE) {
      rc = db_rli_delete(dbh->rli, &buf[1], lurl, surl, &lfnchanged,
			 c->errmsg);
      if (rc == GLOBUS_RLS_SUCCESS && update_immediate && lfnchanged)
	doimmediate(rliop_delete, rli_rlilist, &rli_rlilistlock, &buf[1],lurl);
    } else  {
      if (saverc == GLOBUS_RLS_SUCCESS) {
	snprintf(errbuf, MAXERRMSG, "rli_recv: Invalid command: %d", *buf);
	saverc = GLOBUS_RLS_BADARG;
      }
      continue;
    }
    if (rc != GLOBUS_RLS_SUCCESS && saverc == GLOBUS_RLS_SUCCESS) {
      saverc = rc;
      snprintf(errbuf, MAXERRMSG, "db_rli_%s: %s",
	       *buf == RLI_ADD ? "update" : "delete", c->errmsg);
    }
  }
  if (saverc != GLOBUS_RLS_SUCCESS)
    rrpc_error(c, saverc, "%s", errbuf);
  else {
    rrpc_success(c);
    sender->lastupdate = time(0);
  }
}

static void
rli_recvbf1(CONNECTION *c, DBH *dbh, char **arglist)

{
  rli_recvbf(c, arglist[0], arglist[0], arglist[1], arglist[2], arglist[3]);
}

static void
rli_recvbf2(CONNECTION *c, DBH *dbh, char **arglist)

{
  rli_recvbf(c, arglist[0], arglist[1], arglist[2], arglist[3], arglist[4]);
}

/*
 * Server (RLI) side rli update function via bloom filter.
 */
static void
rli_recvbf(CONNECTION *c, char *lrcurl, char *senderurl, char *abfsize,
	     char *anumhash, char *aupdateinterval)

{
  int			rc;
  char			*surl;
  char			surlbuf[BUFLEN];
  char			*lurl;
  char			lurlbuf[BUFLEN];
  char			errbuf[MAXERRMSG];
  SENDER		*sender;
  int			bfsize;
  int			numhash;
  globus_size_t		bytes;
  globus_size_t		len;
  int			i;

  if (loglevel)
    logit(LOG_DEBUG, "rli_recvbf: %s %s %s %s %s", PS(senderurl), PS(lrcurl),
	  PS(abfsize), PS(anumhash), PS(aupdateinterval));

  if (!senderurl || !lrcurl || !abfsize || !anumhash || !aupdateinterval) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  if (!rli_bloomfilter) {
    rrpc_error(c, GLOBUS_RLS_UNSUPPORTED, "Bloom filters not enabled");
    return;
  }
  surl = canonurl(senderurl, surlbuf, BUFLEN);
  lurl = canonurl(lrcurl, lurlbuf, BUFLEN);
  if ((sender = rli_findsender(surl, lurl)) == NULL) {
    rrpc_error(c, GLOBUS_RLS_NOMEMORY, "");
    return;
  }
  bfsize = atoi(abfsize);
  numhash = atoi(anumhash);
  sender->updateinterval = atoi(aupdateinterval);

  if (sender->bf.bfsize != bfsize || sender->bf.numhash != numhash) {
    bf_free(&sender->bf);
    if (bf_init(&sender->bf, bfsize, numhash, 0) != GLOBUS_RLS_SUCCESS) {
      logit(LOG_WARNING, "rli_recvbf(%s): No memory for bloom filter", surl);
      rrpc_error(c, GLOBUS_RLS_NOMEMORY, "");
      return;
    }
  }
  bytes = BITS2BYTES(bfsize);
  /*
   * Part of the bloom filter may already be read into c->b, it will
   * be more efficient to do an unbuffered read so copy whatever bytes
   * are in c->b into bf.bits, then do a read for the rest directly
   * into bf.bits.
   *
   * FIXME
   * A race condition in the LRC allows for c->ibuf.len to be greater
   * than bfsize. This will be fixed in the future. For the time being,
   * if this happens, we reject the bloomfilter and return 
   * GLOBUS_RLS_BADARG. The new bloomfilter will be read when the LRC
   * sends the next update.
   */
  if (c->ibuf.len > bfsize) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "Bloomfilter size mismatch");
    return;
  }
  for (i = 0; i < c->ibuf.len - c->ibuf.idx; i++)
    sender->bf.bits[i] = c->ibuf.buf[c->ibuf.idx + i];
  if (i < bytes)
    if ((rc = rrpc_read(&c->handle, &sender->bf.bits[i], bytes - i,
			bytes - i, &len, c->errmsg)) != GLOBUS_SUCCESS) {
      logit(LOG_WARNING, "rli_recvbf(%s): %s", surl,
	    globus_rls_errmsg(rc, c->errmsg, errbuf, MAXERRMSG));
      return;
    }
  rrpc_success(c);

  sender->lastupdate = time(0);
  if (rli_bloomfilter_dir)
    bf_save(sender->lrcurl, &sender->bf);
  globus_mutex_lock(&sender->bf.mtx);
  sender->bf.flags |= BF_NEEDUPDATE;	/* For immediate mode updates	*/
  globus_mutex_unlock(&sender->bf.mtx);
}

static void
rls_admin(CONNECTION *c, DBH *dbh, char **arglist)

{
  globus_rls_admin_cmd_t	cmd;
  RLI				*rli;

  if (loglevel > 1)
    logit(LOG_DEBUG, "rls_admin: %s", PS(arglist[0]));
  if (!arglist[0]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "");
    return;
  }
  cmd = (globus_rls_admin_cmd_t) atoi(arglist[0]);
  switch (cmd) {
    case globus_rls_admin_cmd_ping:
      rrpc_success(c);
      break;

    case globus_rls_admin_cmd_quit:
      if (!(c->perms & P_ADMIN)) {
	rrpc_error(c, GLOBUS_RLS_PERM, "%s", c->dn);
	break;
      }
      rrpc_success(c);
      quitting++;
      delayquit = 2;
      globus_mutex_lock(&commonmtx);
      globus_cond_signal(&commoncond);
      globus_mutex_unlock(&commonmtx);
      break;

    case globus_rls_admin_cmd_ssu:
      if (!(c->perms & P_ADMIN)) {
	rrpc_error(c, GLOBUS_RLS_PERM, "%s", c->dn);
	break;
      }
      rrpc_success(c);
      if (lrc_server) {
	rls_lock_get(&lrc_rlilistlock, readlock);
	for (rli = lrc_rlilist; rli; rli = rli->nxt)
	  rli->softstatestart = 0;
	rls_lock_release(&lrc_rlilistlock, readlock);
	lrc_updatenow();
      }
      if (rli_server) {
	rls_lock_get(&rli_rlilistlock, readlock);
	for (rli = rli_rlilist; rli; rli = rli->nxt)
	  rli->softstatestart = 0;
	rls_lock_release(&rli_rlilistlock, readlock);
	rli_updatenow();
      }
      globus_mutex_lock(&commonmtx);
      globus_cond_signal(&commoncond);
      globus_mutex_unlock(&commonmtx);
      break;

    default:
      rrpc_error(c, GLOBUS_RLS_BADARG, "%s", arglist[0]);
      break;
  }
}

static void
rls_close(CONNECTION *c, DBH *dbh, char **arglist)

{
  if (loglevel > 3)
    logit(LOG_DEBUG, "rls_close: %X", c);
  freeconn(c, 0);
}

static void
rls_get_configuration(CONNECTION *c, DBH *dbh, char **arglist)

{
  int		rc;
  char		errmsg[MAXERRMSG];
  int		ci;
  char		*name;
  char		*val;
  void		*si;
  char		buf[1000];

  if (loglevel)
    logit(LOG_DEBUG, "rls_get_configuration: %s", PS(arglist[0]));
  rrpc_initbuf(&c->obuf);
  if ((rc = rrpc_bufwrite(&c->handle, &c->obuf, "0", 2, 0,
			  c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    logit(LOG_WARNING, "rls_get_configuration: rrpc_bufwrite: %s",
	  globus_rls_errmsg(rc, c->errmsg, errmsg, MAXERRMSG));
    return;
  }
  if ((name = arglist[0]))
    ci = OPT_ONE;
  else
    ci = OPT_ALL;
  si = NULL;
  while ((ci = conf_get(ci, &si, &name, &val, buf)) != OPT_DONE) {
    if ((rc = rrpc_bufwrite(&c->handle, &c->obuf, name, strlen(name) + 1,
			    0, c->errmsg)) == GLOBUS_RLS_SUCCESS)
       rc = rrpc_bufwrite(&c->handle, &c->obuf, val, strlen(val) + 1,
			  0, c->errmsg);
    if (rc != GLOBUS_RLS_SUCCESS) {
      logit(LOG_WARNING, "rls_get_configuration: rrpc_bufwrite: %s",
	    globus_rls_errmsg(rc, c->errmsg, errmsg, MAXERRMSG));
      return;
    }
    if (arglist[0] && ci != OPT_ONE)
      break;
  }
  if ((rc = rrpc_bufwrite(&c->handle, &c->obuf, "", 1, 1,
			  c->errmsg)) != GLOBUS_RLS_SUCCESS) {
    logit(LOG_WARNING, "rls_get_configuration: rrpc_bufwrite: %s",
	  globus_rls_errmsg(rc, c->errmsg, errmsg, MAXERRMSG));
    return;
  }
}

static void
rls_set_configuration(CONNECTION *c, DBH *dbh, char **arglist)

{
  int	rc;

  if (loglevel > 3)
    logit(LOG_DEBUG, "rls_set_configuration: %s %s", PS(arglist[0]),
	  PS(arglist[1]));
  if (!arglist[0] || !arglist[1]) {
    rrpc_error(c, GLOBUS_RLS_BADARG, "NULL");
    return;
  }
  if ((rc = conf_option(arglist[0], arglist[1], 1)) != GLOBUS_RLS_SUCCESS) {
    rrpc_error(c, rc, "%s", arglist[0]);
    return;
  }
  rrpc_success(c);

  globus_mutex_lock(&commonmtx);
  globus_cond_signal(&commoncond);
  globus_mutex_unlock(&commonmtx);
}

static void
rls_stats(CONNECTION *c, DBH *dbh, char **arglist)

{
  int		rc;
  int		i, flags;
  char		errmsg[MAXERRMSG];

  if (loglevel)
    logit(LOG_DEBUG, "rls_stats:");
  flags = 0;
  if (lrc_server)
    flags |= RLS_LRCSERVER|RLS_SNDLFNLIST|RLS_SNDBLOOMFILTER;
  if (rli_server) {
    flags |= RLS_RLISERVER;
    if (rli_bloomfilter)
      flags |= RLS_RCVBLOOMFILTER;
    else
      flags |= RLS_RCVLFNLIST;
  }
  if (is_lrc_init_done()) {
    flags |= RLS_INITIALIZED;
  }
  rrpc_initbuf(&c->obuf);

  /* Send stats */
  rc = rrpc_bufprintf(&c->handle, &c->obuf, 0, c->errmsg,
      "0%c%d.%d%c%u%c%u%c%u%c%u%c", 0, MAJOR, MINOR, 0, 
      (int) (time(0) - starttime), 0, flags, 0, update_bf_int, 0,
      update_ll_int, 0);
  if (rc != GLOBUS_RLS_SUCCESS) {
    logit(LOG_WARNING, "rls_stats: rrpc_bufprintf: %s",
	  globus_rls_errmsg(rc, c->errmsg, errmsg, MAXERRMSG));
    return;
  }

  for (i = 0; i < T_MAX; i++)
    if ((rc = rrpc_bufprintf(&c->handle, &c->obuf, 1, c->errmsg,
			    "%u%c", table[i].count, 0)) != GLOBUS_RLS_SUCCESS)
      break;
  if (rc != GLOBUS_RLS_SUCCESS)
    logit(LOG_WARNING, "rls_stats: rrpc_bufprintf: %s",
	  globus_rls_errmsg(rc, c->errmsg, errmsg, MAXERRMSG));
}

/*
 * An add, create or delete operation was just done to the LRC database,
 * push this update to the RLI servers immediately.
 */
static void
doimmediate(RLIOP op, RLI *rlilist, LOCK *rlilistlock, char *lfn, char *lrc)

{
  UPDATE	*u;
  RLI		*rli;

  rls_lock_get(rlilistlock, readlock);

  u = NULL;  /* Delay allocating until we know we need to update an RLI	*/
  for (rli = rlilist; rli; rli = rli->nxt) {
    if (rli->flags & FR_BLOOMFILTER)
      continue;
    if (update_patternmatch(rli, lfn)) {
      if (!u)		/* Allocate UPDATE if haven't already	*/
	if ((u = update_new(lfn, lrc, op)) == NULL)
	  break;
      update_queue(rli, u);
    }
  }

  rls_lock_release(rlilistlock, readlock);
}

/*
 * Callback functions strXcb, which are invoked by the db functions that
 * return lists of tuples of strings and write the tuples to the client,
 * use this function to add the strings to an iovec array and call writev()
 * when it's full to write the results to the client.  If s2 and/or s3 are
 * NULL they are ignored (eg when called from str2cb or str3cb).  If irc
 * is not -1 it is included in the result, it is the result of an individual
 * query within a bulk query.
 */
static int
listresult(void *cv, char *s1, char *s2, char *s3, char *s4, int irc)

{
  CONNECTION	*c = (CONNECTION *) cv;
  int		rc;
  char		lbuf[MAXERRMSG];

  if (!c->numrecs) {		/* First time through want to send	*/
    rrpc_initbuf(&c->obuf);	/* result code of 0			*/
    if ((rc = rrpc_bufwrite(&c->handle, &c->obuf, "0", 2, 0,
			    c->errmsg)) != GLOBUS_RLS_SUCCESS) {
      logit(LOG_WARNING, "listresult: rrpc_bufwrite: %s",
	    globus_rls_errmsg(rc, c->errmsg, lbuf, MAXERRMSG));
      return GLOBUS_FALSE;
    }
  }

  if (irc != -1)		/* Include result code for each query	*/
    rc = rrpc_bufprintf(&c->handle, &c->obuf, 0, c->errmsg, "%d%c%s%c",
			irc, 0, s1, 0);
  else
    rc = rrpc_bufwrite(&c->handle, &c->obuf, s1, strlen(s1) + 1, 0, c->errmsg);
  if (rc == GLOBUS_RLS_SUCCESS && s2)
    rc = rrpc_bufwrite(&c->handle, &c->obuf, s2, strlen(s2) + 1, 0, c->errmsg);
  if (rc == GLOBUS_RLS_SUCCESS && s3)
    rc = rrpc_bufwrite(&c->handle, &c->obuf, s3, strlen(s3) + 1, 0, c->errmsg);
  if (rc == GLOBUS_RLS_SUCCESS && s4)
    rc = rrpc_bufwrite(&c->handle, &c->obuf, s4, strlen(s4) + 1, 0, c->errmsg);
  if (rc != GLOBUS_RLS_SUCCESS) {
    logit(LOG_WARNING, "listresult: rrpc_bufwrite: %s",
	  globus_rls_errmsg(rc, c->errmsg, lbuf, MAXERRMSG));
    return GLOBUS_FALSE;
  }
  c->numrecs++;
  return GLOBUS_TRUE;
}

static int
str2cb(void *cv, char *s1, char *s2)

{
  return listresult(cv, s1, s2, NULL, NULL, -1);
}

static int
str3cb(void *cv, char *s1, char *s2, char *s3)

{
  return listresult(cv, s1, s2, s3, NULL, -1);
}

static int
str2bulkcb(void *cv, char *s1, char *s2)

{
  return listresult(cv, s1, s2, NULL, NULL, GLOBUS_RLS_SUCCESS);
}

static int
str4bulkcb(void *cv, char *s1, char *s2, char *s3, char *s4)

{
  return listresult(cv, s1, s2, s3, s4, GLOBUS_RLS_SUCCESS);
}

/*
 * db_update_get_part invokes the callback with the RLI URL, RLI flags and
 * any patterns for the RLI partition.  However the flags are not returned
 * to the client, so this callback ignores the 2nd string (flags).
 */
static int
rlipartcb(void *cv, char *s1, char *s2, char *s3)

{
  return listresult(cv, s1, s3, NULL, NULL, -1);
}

/*
 * Write terminating null byte after last callback that returned a list
 * of results to the caller.  Also reset c->numrecs count.  If pending
 * is set terminate list with a byte set to 1 to indicate to the client
 * more results are available.
 */
static void
endstrcb(CONNECTION *c, int pending)

{
  int		rc;
  char		lbuf[MAXERRMSG];
  int		i;
  char		buf[4];

  if (!c->numrecs) {		/* If no recs include result code of 0	*/
    rrpc_initbuf(&c->obuf);
    buf[0] = '0';
    buf[1] = '\0';
    i = 2;
  } else
    i = 0;
  if (pending)
    buf[i++] = 1;
  buf[i++] = '\0';
  if ((rc = rrpc_bufwrite(&c->handle, &c->obuf, buf, i, 1,
			  c->errmsg)) != GLOBUS_RLS_SUCCESS)
    logit(LOG_WARNING, "endstrcb: rrpc_bufwrite: %s",
	  globus_rls_errmsg(rc, c->errmsg, lbuf, MAXERRMSG));
  c->numrecs = 0;
}

static char *
canonurl(char *s, char *buf, int len)

{
  int			r;
  globus_url_t		url;
  char			hostbuf[BUFLEN];
  struct hostent	he;

  if ((r = globus_url_parse(s, &url)) != GLOBUS_SUCCESS) {
    logit(LOG_WARNING, "canonurl(%s): Can't canonicalize: %d", s, r);
    return s;
  }
  if (url.host == NULL) {
    logit(LOG_WARNING, "canonurl(%s): No host", s);
    globus_url_destroy(&url);
    return s;
  }
  if (!globus_libc_gethostbyname_r(url.host, &he, hostbuf, BUFLEN, &r)) {
    logit(LOG_WARNING, "canonurl(%s): Can't lookup hostname %s: %d", s,
	  url.host, r);
    globus_url_destroy(&url);
    return s;
  }
  if (url.port == 0)
    url.port = GLOBUS_RLS_SERVER_DEFPORT;
  snprintf(buf, len, "%s://%s:%d", url.scheme, he.h_name, url.port);
  globus_url_destroy(&url);
  return buf;
}

/*
 * Parse reslimit arg, return minimum of server limit and client limit,
 * a value of 0 is infinite.
 */
static int
getreslimit(char *areslimit)

{
  int	reslimit;

  if (!areslimit)
    return result_limit;
  reslimit = atoi(areslimit);
  if (!result_limit)
    return reslimit;
  else if (!reslimit)
    return result_limit;
  if (result_limit < reslimit)
    return result_limit;
  else
    return reslimit;
}

/*
 * Start up thread to process requests.
 */
static int
startthread()

{
  int		rc;
  pthread_t	thr;

  if ((rc = pthread_create(&thr, GLOBUS_NULL, procreq, NULL)) != 0) {
    logit(LOG_WARNING, "pthread_create: %s",
	  globus_libc_system_error_string(rc));
    return 0;
  }
  if ((rc = pthread_detach(thr)) != 0)
    logit(LOG_WARNING, "pthread_detach: %s",
	  globus_libc_system_error_string(rc));
  globus_mutex_lock(&commonmtx);
  numthreads++;
  if (loglevel)
    logit(LOG_DEBUG, "startthread: id %lu num %d", thr, numthreads);
  globus_mutex_unlock(&commonmtx);
  return 1;
}

static void
endthread(DBH *dbh)

{
  if (dbh->lrc)
    db_close(dbh->lrc);
  if (dbh->rli)
    db_close(dbh->rli);
  globus_mutex_lock(&commonmtx);
  numthreads--;
  if (loglevel > 3)
    logit(LOG_DEBUG, "endthread: num %d", numthreads);
  if (quitting && numthreads == 0)	/* If last thread let parent	*/
    globus_cond_signal(&commoncond);	/* know they're all dead	*/
  globus_mutex_unlock(&commonmtx);
  pthread_exit(0);
}

static void
hexitsig(int s)

{
  if (quitting++)
    return;
  logit(LOG_INFO, "hexitsig: Received signal %d", s);
  /*
   * Would like to signal commoncond but can't call pthread_cond_signal
   * safely from async signal handler on linux.  Main loop wakes up
   * periodically to check if quitting is set.
   */
}

static void
hhupsig(int s)

{
  if (needreadconf++)
    return;
  logit(LOG_INFO, "hhupsig: Received HUP signal");
  /*
   * Would like to signal commoncond but can't call pthread_cond_signal
   * safely from async signal handler on linux.  Main loop wakes up periodically
   * to check if needreadconf is set.
   */
}

static void
cleanexit(int s)

{
  int		i;
  struct rusage	u;
  double	ut;
  double	st;
  double	tt;
#ifdef COUNTDB
  extern int	dbstatements;
#endif
#ifdef COUNTUPDATE
  extern int	updatebytes;
#endif
  char		buf[BUFLEN];

  globus_mutex_lock(&connlistmtx);
  i = globus_list_size(connlist);
  globus_mutex_unlock(&connlistmtx);
  /*
   * There might be some connections on connlist which
   * haven't been freed because their cancel and close
   * callbacks haven't fired yet.
   * Since we're exiting, we don't want to wait a long
   * time for them to complete. They'll eventually be
   * cleaned up by the OS.
   * XIO Note: Deactivating the io module should fire
   * callbacks on all active handles.
   */
  if (i) {
  	logit(LOG_WARNING, "cleanexit: %d connection(s) on connlist", i);
	logit(LOG_WARNING, "cleanexit: waiting 1 second for cleanup");
	sleep(1);
  }

  /* Send usage stass at exit */
  usage_send_wrapper();
  /* Destroy usage stats support */
  usage_destroy();

  /* Destroy log handle */
  log_destroy();

  globus_fifo_destroy(&reqqueue);
  globus_mutex_destroy(&reqqueuemtx);
  globus_cond_destroy(&reqqueuecond);
  globus_cond_destroy(&commoncond);
  globus_mutex_destroy(&commonmtx);
  globus_mutex_destroy(&connlistmtx);
  for (i = NMODS - 1; i >= 0; i--)
    globus_module_deactivate(modules[i]);
  if (loglevel) {
    getrusage(RUSAGE_SELF, &u);
    ut = (double) u.ru_utime.tv_sec + u.ru_utime.tv_usec / 1000000.0;
    st = (double) u.ru_stime.tv_sec + u.ru_stime.tv_usec / 1000000.0;
    tt = ut + st;
    sprintf(buf, "cleanexit: %.2fu %.2fs %.2ft", ut, st, tt);
#ifdef COUNTDB
    sprintf(buf, "%s %ddb", buf, dbstatements);
#endif
#ifdef COUNTUPDATE
    sprintf(buf, "%s %drb", buf, updatebytes);
#endif
    logit(LOG_INFO, "%s", buf);
  }
  if (!rlsdebug && pidfile)
    unlink(pidfile);
  exit(s);
}

static void
usage(char *prog)

{
  globus_libc_fprintf(stderr, "Usage: %s options...\n", prog);
  globus_libc_fprintf(stderr, "  [-B update_bf_int]\n");
  globus_libc_fprintf(stderr, "  [-b maxbackoff]\n");
  globus_libc_fprintf(stderr, "  [-C rlscertfile]\n");
  globus_libc_fprintf(stderr, "  [-c conffile]\n");
  globus_libc_fprintf(stderr, "  [-d]\n");
  globus_libc_fprintf(stderr, "  [-e rli_expire_int]\n");
  globus_libc_fprintf(stderr, "  [-F update_factor]\n");
  globus_libc_fprintf(stderr, "  [-f maxfreethreads]\n");
  globus_libc_fprintf(stderr, "  [-I true|false]\n");
  globus_libc_fprintf(stderr, "  [-i idletimeout]\n");
  globus_libc_fprintf(stderr, "  [-K rlskeyfile]\n");
  globus_libc_fprintf(stderr, "  [-L loglevel]\n");
  globus_libc_fprintf(stderr, "  [-l true|false]\n");
  globus_libc_fprintf(stderr, "  [-M maxconnections]\n");
  globus_libc_fprintf(stderr, "  [-m maxthreads]\n");
  globus_libc_fprintf(stderr, "  [-N]\n");
  globus_libc_fprintf(stderr, "  [-o update_buftime]\n");
  globus_libc_fprintf(stderr, "  [-p pidfile]\n");
  globus_libc_fprintf(stderr, "  [-r true|false]\n");
  globus_libc_fprintf(stderr, "  [-S rli_expire_stale]\n");
  globus_libc_fprintf(stderr, "  [-s startthreads]\n");
  globus_libc_fprintf(stderr, "  [-t timeout]\n");
  globus_libc_fprintf(stderr, "  [-U myurl]\n");
  globus_libc_fprintf(stderr, "  [-u update_ll_int]\n");
  globus_libc_fprintf(stderr, "  [-v]\n");
  exit(1);
}

/*
 * Wrapper to simplify the call to the usage_send() function.
 */
static void
usage_send_wrapper()
{
  usage_send(
      lrc_server,
      rli_server,
      (int) (time(0) - starttime),
      table[T_LRCLFN].count,
      table[T_LRCPFN].count,
      table[T_LRCMAP].count,
      table[T_RLILFN].count,
      table[T_RLILRC].count,
      table[T_RLISENDER].count,
      table[T_RLIMAP].count,
      numthreads,
      conncount);
}
