/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file sendfax.c
 * \brief Fax dialog and fax processing functions
 */

#include <ffgtk.h>
#include <faxophone/faxophone.h>
#include <faxophone/fax.h>

enum {
	CONTACT_NAME,
	CONTACT_NUMBER
};

void faxSetLogLevel( int nLevel );
void faxSetLogHandler( void (*log)( int nLevel, const char *pnText ) );

/**
 * \brief Send fax
 * \param pnFile file name to send
 * \param pnNumber fax number
 * \param bImprove improve image quality
 * \param pnReportDir directory where to store report or NULL
 */
static void sendFax( char *pnFile, char *pnNumber, gboolean bImprove, char *pnReportDir ) {
	gchar *pnOutput = NULL;
	gchar *pnOption = NULL;
	gchar *pnFaxFile;

#ifdef G_OS_WIN32
	pnFaxFile = g_strdup( pnFile );
#else
	gchar *apnArgs[ 11 ];

	Debug( KERN_DEBUG, "pnFile: %s\n", pnFile );

	pnFaxFile = g_strdup_printf( "%s.tif", pnFile );

	/* convert ps to tiffg3 */
	apnArgs[ 0 ] = "gs";
	apnArgs[ 1 ] = "-q";
	apnArgs[ 2 ] = "-dNOPAUSE";
	apnArgs[ 3 ] = "-dSAFER=true";
	apnArgs[ 4 ] = "-sDEVICE=tiffg3";
	apnArgs[ 5 ] = "-sPAPERSIZE=a4";
	apnArgs[ 6 ] = "-dFIXEDMEDIA";
	if ( bImprove == TRUE ) {
		apnArgs[ 7 ] = "-r204x196";
	} else {
		apnArgs[ 7 ] = "-r204x98";
	}
	pnOutput = g_strdup_printf( "-sOutputFile=%s", pnFaxFile );
	apnArgs[ 8 ] = pnOutput;
	apnArgs[ 9 ] = pnFile;
	apnArgs[ 10 ] = NULL;

	if ( g_spawn_sync( NULL, apnArgs, NULL, G_SPAWN_SEARCH_PATH, NULL, NULL, NULL, NULL, NULL, NULL ) == FALSE ) {
		Debug( KERN_WARNING, "Error occurred: %d (%s)\n", errno, strerror( errno ) );
		g_free( pnFaxFile );
		return;
	}
#endif

	faxSetLogLevel( debugGetFaxLevel( getActiveProfile() ) );
	faxophoneSendFax( pnFaxFile, faxGetBitrate( getActiveProfile() ), faxGetEcm( getActiveProfile() ), faxGetController( getActiveProfile() ), faxGetSenderMsn( getActiveProfile() ), pnNumber, faxGetSenderNumber( getActiveProfile() ), faxGetSenderName( getActiveProfile() ), 0, pnReportDir );

	if ( pnOutput != NULL ) {
		g_free( pnOutput );
	}
	if ( pnOption != NULL ) {
		g_free( pnOption );
	}
	g_free( pnFaxFile );
}

/**
 * \brief Fax dialog window response callback
 * \param psDialog dialog window pointer
 * \param nResponse response id
 * \param pUserData file name
 */
void faxDialogResponse( GtkDialog *psDialog, gint nResponse, gpointer pUserData ) {
	gchar *pnFile = ( gchar * ) pUserData;
	gchar *pnNumber = NULL;
	char *pnReportDir = NULL;
	GtkWidget *psNumberEntry = g_object_get_data( G_OBJECT( psDialog ), "number_entry" );
	GtkWidget *psReportToggle = g_object_get_data( G_OBJECT( psDialog ), "report_toggle" );
	GtkWidget *psReportButton = g_object_get_data( G_OBJECT( psDialog ), "report_button" );
	GtkWidget *psImproveToggle = g_object_get_data( G_OBJECT( psDialog ), "improve_toggle" );
	gboolean bImprove = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psImproveToggle ) );

	Debug( KERN_DEBUG, "pnFile: %s\n", pnFile );

	if ( psNumberEntry != NULL && gtk_entry_get_text( GTK_ENTRY( psNumberEntry ) ) != NULL ) {
		pnNumber = g_strdup( gtk_entry_get_text( GTK_ENTRY( psNumberEntry ) ) );
	}

	if ( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psReportToggle ) ) == true ) {
		pnReportDir = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER( psReportButton ) );
	}

	gtk_widget_destroy( GTK_WIDGET( psDialog ) );

	switch ( nResponse ) {
		case 2:
			if ( pnNumber != NULL ) {
				gchar *pnFixNumber = fixNumber( pnNumber );
				sendFax( pnFile, pnFixNumber, bImprove, pnReportDir );
				g_free( pnFixNumber );
			}
			break;
		default:
			unlink( pnFile );
			break;
	}

	g_free( pnFile );

	if ( pnNumber != NULL ) {
		g_free( pnNumber );
	}
}

/**
 * \brief Report toggle callback
 * \param psWidget file chooser
 * \param pUserData toggle button
 */
void reportToggleCallback( GtkWidget *psWidget, gpointer pUserData ) {
	gtk_widget_set_sensitive( GTK_WIDGET( pUserData ), gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psWidget ) ) );
}

/**
 * \brief Open addressbook in dial mode
 * \param psWidget button widget
 * \param pUserData UNUSED
 */
static void faxAddressBook( GtkWidget *psWidget, gpointer pUserData ) {
	GtkWidget *psDialog = AddressBook( 1 );

	g_object_set_data( G_OBJECT( psDialog ), "parent", pUserData );
}

/**
 * \brief Process incoming fax
 * \param pnFile file name
 */
void processFax( gchar *pnFile ) {
	GtkBuilder *psBuilder = NULL;
	GError *psError = NULL;
	GtkWidget *psDialog = NULL;
	GtkListStore *psModel = NULL;
	GtkTreeIter sIter;
	GtkEntryCompletion *psCompletion = NULL;
	GtkWidget *psFileNameLabel = NULL;
	GtkWidget *psNameEntry = NULL;
	GtkWidget *psNumberEntry;
	GtkWidget *psBookButton = NULL;
	GtkWidget *psImproveToggle = NULL;
	GtkWidget *psReportToggle = NULL;
	GtkWidget *psReportButton = NULL;
	GList *psList = psPersonsList;
	gchar *pnUiFile;
	gchar *pnTmpFile = NULL;

	psBuilder = gtk_builder_new();
	pnUiFile = getUiFile( "sendfax.ui" );
	if ( gtk_builder_add_from_file( psBuilder, pnUiFile, &psError ) == 0 ) {
	    Debug( KERN_WARNING, "Error: %s\n", psError -> message );
	    g_error_free( psError );
		g_free( pnUiFile );
	    return;
	}
	g_free( pnUiFile );

	pnTmpFile = g_strdup( pnFile );

	psDialog = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxDialog" ) );

	psFileNameLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFileNameLabel" ) );
	char *pnMarkup = g_markup_printf_escaped ("<b>%s</b>", pnTmpFile );
	gtk_label_set_markup( GTK_LABEL( psFileNameLabel ), pnMarkup );
	g_free( pnMarkup );

	psNameEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psNameEntry" ) );
	g_object_set_data( G_OBJECT( psDialog ), "name_entry", psNameEntry );

	psNumberEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psNumberEntry" ) );
	g_object_set_data( G_OBJECT( psDialog ), "number_entry", psNumberEntry );
	
	psCompletion = gtk_entry_completion_new();
	gtk_entry_completion_set_text_column( psCompletion, CONTACT_NAME );
	gtk_entry_set_completion( GTK_ENTRY( psNameEntry ), psCompletion );

	psModel = gtk_list_store_new( 1, G_TYPE_STRING );
	while ( psList != NULL ) {
		struct sPerson *psPerson = psList -> data;

		if ( psPerson != NULL ) {
			gtk_list_store_append( psModel, &sIter );
			gtk_list_store_set( psModel, &sIter, CONTACT_NAME, psPerson -> pnDisplayName, -1 ); 
		}

		psList = psList -> next;
	}
	gtk_entry_completion_set_model( psCompletion, GTK_TREE_MODEL( psModel ) );
	g_signal_connect( G_OBJECT( psCompletion ), "match-selected", G_CALLBACK( dialNameMatchSelect ), psNumberEntry );

	psBookButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psAddressBookButton" ) );
	g_signal_connect( G_OBJECT( psBookButton ), "clicked", G_CALLBACK( faxAddressBook ), psDialog );

	psReportToggle = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxReportCheckButton" ) );
	g_object_set_data( G_OBJECT( psDialog ), "report_toggle", psReportToggle );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psReportToggle ), faxGetReportToggle( getActiveProfile() ) );

	psImproveToggle = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxImproveCheckButton" ) );
	g_object_set_data( G_OBJECT( psDialog ), "improve_toggle", psImproveToggle );

	psReportButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxReportFileChooserButton" ) );
	g_object_set_data( G_OBJECT( psDialog ), "report_button", psReportButton );
	gtk_file_chooser_set_current_folder( GTK_FILE_CHOOSER( psReportButton ), faxGetReportDir( getActiveProfile() ) );
	g_signal_connect( G_OBJECT( psReportToggle ), "toggled", G_CALLBACK( reportToggleCallback ), psReportButton );

	g_signal_connect( G_OBJECT( psDialog ), "response", G_CALLBACK( faxDialogResponse ), pnTmpFile );

	gtk_builder_connect_signals( psBuilder, NULL );

	g_object_unref( G_OBJECT( psBuilder ) );

	gtk_widget_show_all( psDialog );
}
