/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2016 Univ. Grenoble Alpes, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// Local
#include "EditFrame.h"

// CamiTK
#include <InteractiveViewer.h>
#include <Log.h>
#include <Application.h>
#include <Component.h>
#include <SingleImageComponent.h>
using namespace camitk;

// Qt
#include <QFileDialog>
#include <QTextStream>

// QT DOM
#include <QDomDocument>
#include <QDomElement>
#include <QDomText>

// vtk
#include <vtkUnstructuredGrid.h>
#include <vtkPolyData.h>
#include <vtkPointData.h>
#include <vtkCellData.h>
#include <vtkAlgorithmOutput.h>
#include <vtkMatrix3x3.h>


// --------------- constructor -------------------
EditFrame::EditFrame(ActionExtension* extension) : Action(extension) {
    setName("Edit Frame");
    setDescription(tr("This action helps to modify a component's frame from its parent frame (or the world frame if it has no parent) by setting translations and rotation parameters."));
    setComponent("Component");

    // Setting classification family and tags
    setFamily("Frame");
    addTag(tr("Test"));
    addTag(tr("Edit"));
    addTag(tr("Move"));
    addTag(tr("Visualization"));

    dialog = NULL;
    inputFrame = NULL;
}


// --------------- init -------------------
void EditFrame::init() {
    dialog = new QDialog();

    //-- init user interface
    ui.setupUi(dialog);

    // sliders initial values
    double bounds[6];
    InteractiveViewer::get3DViewer()->getBounds(bounds);
    double xLength = bounds[1] - bounds[0];
    double yLength = bounds[3] - bounds[2];
    double zLength = bounds[5] - bounds[4];
    ui.tX->init(-xLength*2, + xLength*2, 0.0);
    ui.tY->init(-yLength*2, + yLength*2, 0.0);
    ui.tZ->init(-zLength*2, + zLength*2, 0.0);
    ui.rX->init(-180.0, 180.0, 0.0);
    ui.rY->init(-180.0, 180.0, 0.0);
    ui.rZ->init(-180.0, 180.0, 0.0);

    // initialize slider names
    ui.tX->setName("X");
    ui.tY->setName("Y");
    ui.tZ->setName("Z");
    ui.rX->setName(tr("Around X"));
    ui.rY->setName(tr("Around Y"));
    ui.rZ->setName(tr("Around Z"));

    // connect everything
    connect(ui.tX, SIGNAL(valueChanged()), SLOT(apply()));
    connect(ui.tY, SIGNAL(valueChanged()), SLOT(apply()));
    connect(ui.tZ, SIGNAL(valueChanged()), SLOT(apply()));
    connect(ui.rX, SIGNAL(valueChanged()), SLOT(apply()));
    connect(ui.rY, SIGNAL(valueChanged()), SLOT(apply()));
    connect(ui.rZ, SIGNAL(valueChanged()), SLOT(apply()));
    connect(ui.resetButton, SIGNAL(clicked()), SLOT(reset()));
    connect(ui.loadTransformButton, SIGNAL(clicked()), SLOT(load()));
    connect(ui.saveTransformButton, SIGNAL(clicked()), SLOT(save()));
    connect(ui.parentFramePushButton, SIGNAL(clicked()), SLOT(changeParent()));
}


// --------------- destructor -------------------
EditFrame::~EditFrame() {
    if (dialog)
        delete dialog;
}

// --------------- getWidget -------------------
QWidget * EditFrame::getWidget() {
    if (!dialog)
        init();

    // The input frame is the last one selected
    inputFrame = getTargets().last();

    // parent component list
    ui.parentFrameComboBox->clear();
    ui.parentFrameComboBox->addItem("World");
    foreach(Component* comp, Application::getAllComponents()) {
        if(comp != inputFrame)
            ui.parentFrameComboBox->addItem(comp->getName());
    }

    // get frame's rotation
    ui.rX->setValue(inputFrame->getTransform()->GetOrientation()[0]);
    ui.rY->setValue(inputFrame->getTransform()->GetOrientation()[1]);
    ui.rZ->setValue(inputFrame->getTransform()->GetOrientation()[2]);

    // read translation
    // compute translation in the rotated frame
    vtkSmartPointer<vtkMatrix3x3> rotationMatrix = vtkSmartPointer<vtkMatrix3x3>::New();
    rotationMatrix->Identity();
    // isolate the rotation matrix from the transfor one
    rotationMatrix->SetElement(0, 0, inputFrame->getTransform()->GetMatrix()->GetElement(0, 0));
    rotationMatrix->SetElement(0, 1, inputFrame->getTransform()->GetMatrix()->GetElement(0, 1));
    rotationMatrix->SetElement(0, 2, inputFrame->getTransform()->GetMatrix()->GetElement(0, 2));
    rotationMatrix->SetElement(1, 0, inputFrame->getTransform()->GetMatrix()->GetElement(1, 0));
    rotationMatrix->SetElement(1, 1, inputFrame->getTransform()->GetMatrix()->GetElement(1, 1));
    rotationMatrix->SetElement(1, 2, inputFrame->getTransform()->GetMatrix()->GetElement(1, 2));
    rotationMatrix->SetElement(2, 0, inputFrame->getTransform()->GetMatrix()->GetElement(2, 0));
    rotationMatrix->SetElement(2, 1, inputFrame->getTransform()->GetMatrix()->GetElement(2, 1));
    rotationMatrix->SetElement(2, 2, inputFrame->getTransform()->GetMatrix()->GetElement(2, 2));
    rotationMatrix->Invert(); // which is a transpose in this case ...
    double translation[3] = {0.0, 0.0, 0.0};
    rotationMatrix->MultiplyPoint(inputFrame->getTransform()->GetPosition(), translation);

    // set frame translation
    ui.tX->setValue(translation[0]);
    ui.tY->setValue(translation[1]);
    ui.tZ->setValue(translation[2]);

    apply();

    return dialog;
}

//--------------- changeParent -------------
void EditFrame::changeParent() {
    if (inputFrame != NULL) {
        QString parentName = ui.parentFrameComboBox->currentText();
        if(parentName == "World") { // no parent frame ...
            inputFrame->setParentFrame(NULL);
            return;
        }

        // Find the corresponding component:
        const ComponentList existingComponents = Application::getAllComponents();
        Component * newParent = NULL;
        int i = 0;
        while ((i < existingComponents.size()) && (parentName != existingComponents.at(i)->getName())) {
            i++;
        }
        if (i < existingComponents.size()) {
            newParent = existingComponents.at(i);
        }

        inputFrame->setParentFrame(newParent);
    }
    InteractiveViewer::get3DViewer()->refresh();
}

//--------------- reset -------------
void EditFrame::reset() {

    // reset all values to 0.0
    ui.tX->setValue(0.0);
    ui.tY->setValue(0.0);
    ui.tZ->setValue(0.0);
    ui.rX->setValue(0.0);
    ui.rY->setValue(0.0);
    ui.rZ->setValue(0.0);

    apply(); // to refresh
}


//--------------- apply ------------
Action::ApplyStatus EditFrame::apply() {
    inputFrame = dynamic_cast<Component *> (getTargets().last());

    inputFrame->setTransformRotationVTK(ui.rX->getValue(), ui.rY->getValue(), ui.rZ->getValue());
    inputFrame->setTransformTranslationVTK(ui.tX->getValue(), ui.tY->getValue(), ui.tZ->getValue());

    // Refresh only 3D viewer : there is only it displaying the frame.
    InteractiveViewer::get3DViewer()->refresh();

    return SUCCESS;
}

//--------------- save ------------
void EditFrame::save() {

	QString filename = QFileDialog::getSaveFileName(NULL, tr("Save Frame File"), "/home/myFrame.frame", tr("Frames (*.frame)"));
    QFile file(filename);
    if (file.open(QIODevice::WriteOnly)) {
        QTextStream stream(&file);
        QDomDocument doc("frame");
        QDomElement root = doc.createElement("frame");
        root.setAttribute("name", "created frame");

        QDomElement transform = doc.createElement("transform");
        transform.setAttribute("type", "matrix");
        vtkSmartPointer<vtkMatrix4x4> matrix = inputFrame->getTransform()->GetMatrix();
        for (int i = 0; i < 4; i++) {
            QDomElement line = doc.createElement("line");
            // x
            QDomElement xElmt = doc.createElement("x");
            double xVal = matrix->GetElement(i, 0);
            QDomText x = doc.createTextNode(QString::number(xVal));
            xElmt.appendChild(x);
            line.appendChild(xElmt);

            // y
            QDomElement yElmt = doc.createElement("y");
            double yVal = matrix->GetElement(i, 1);
            QDomText y = doc.createTextNode(QString::number(yVal));
            yElmt.appendChild(y);
            line.appendChild(yElmt);

            // z
            QDomElement zElmt = doc.createElement("z");
            double zVal = matrix->GetElement(i, 2);
            QDomText z = doc.createTextNode(QString::number(zVal));
            zElmt.appendChild(z);
            line.appendChild(zElmt);

            // t
            QDomElement tElmt = doc.createElement("t");
            double tVal = matrix->GetElement(i, 3);
            QDomText t = doc.createTextNode(QString::number(tVal));
            tElmt.appendChild(t);
            line.appendChild(tElmt);
            transform.appendChild(line);
        }

        root.appendChild(transform);
        doc.appendChild(root);
        doc.save(stream, 0);

        file.close();
    }
}

//--------------- load ------------
void EditFrame::load() {

    // choose the file to open
	QString fileName = QFileDialog::getOpenFileName(NULL, tr("Open Frame File"), "/home", tr("Frames (*.frame)"));
    QFile file(fileName);
    if(file.open(QIODevice::ReadOnly)) {
        QDomDocument doc("frame");
        if (!doc.setContent(&file)) {
            file.close();
            return;
        }

        // we read the file as a vtkTransform, which is better to process then
        vtkSmartPointer<vtkTransform> transformation = vtkSmartPointer<vtkTransform>::New();
        transformation->Identity();
        vtkSmartPointer<vtkMatrix4x4> matrix = transformation->GetMatrix();

        // QDomElement docElem = doc.documentElement();
        QDomNode transformNode = doc.firstChild();

        if(!transformNode.isNull()) {
            QDomNodeList lines = doc.elementsByTagName("line");
            if(lines.length() != 4) {
                CAMITK_ERROR("MoveFrame", "load", "Number of XML line elements should be exactly 4.");
                return;
            }

            for (int i = 0; i < 4; i++) {
                QDomNode line = lines.at(i);
                if(line.childNodes().length() != 4) {
                    CAMITK_ERROR("MoveFrame", "load", "XML line elements should be have exactly 4 children.");
                    return;
                }
                for(int j = 0; j < 4; j++) {
                    double value = line.childNodes().at(j).firstChild().nodeValue().toDouble();
                    matrix->SetElement(i, j, value);
                }
            }
        }

        file.close();

        // read rotation
        double * rotationAngles = transformation->GetOrientation();
        ui.rX->setValue(rotationAngles[0]);
        ui.rY->setValue(rotationAngles[1]);
        ui.rZ->setValue(rotationAngles[2]);

        // read translation
        // compute translation in the rotated frame
        vtkSmartPointer<vtkMatrix3x3> rotationMatrix = vtkSmartPointer<vtkMatrix3x3>::New();
        rotationMatrix->Identity();
        // isolate the rotation matrix from the transform one
        rotationMatrix->SetElement(0, 0, transformation->GetMatrix()->GetElement(0, 0));
        rotationMatrix->SetElement(0, 1, transformation->GetMatrix()->GetElement(0, 1));
        rotationMatrix->SetElement(0, 2, transformation->GetMatrix()->GetElement(0, 2));
        rotationMatrix->SetElement(1, 0, transformation->GetMatrix()->GetElement(1, 0));
        rotationMatrix->SetElement(1, 1, transformation->GetMatrix()->GetElement(1, 1));
        rotationMatrix->SetElement(1, 2, transformation->GetMatrix()->GetElement(1, 2));
        rotationMatrix->SetElement(2, 0, transformation->GetMatrix()->GetElement(2, 0));
        rotationMatrix->SetElement(2, 1, transformation->GetMatrix()->GetElement(2, 1));
        rotationMatrix->SetElement(2, 2, transformation->GetMatrix()->GetElement(2, 2));
        rotationMatrix->Invert(); // which is a transpose in this case ...
        double translation[3] = {0.0, 0.0, 0.0};
        rotationMatrix->MultiplyPoint(transformation->GetPosition(), translation);

        // get it back in the widget
        ui.tX->setValue(translation[0]);
        ui.tY->setValue(translation[1]);
        ui.tZ->setValue(translation[2]);

        // refresh the widget
        apply();
    }



}
